/* ************************************************************************
   FILE : CLIPSENV.C
      This file contains the driver and the support routines that handle 
   the definiton of new key-bindings and color-settings for the interface.
      Note : CLIPSENV.EXE is a separate image from CLIPS.EXE.  CLIPSENV
   generates binary data files that CLIPS can read.
   *********************************************************************** */

/******************************************************************************
 ==============================================================================
                                 HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "common.h"     /* Common constants and memory macros               */
#define _NO_STDIO
#include "curses.h"     /* Curses constants, macros and function prototypes */
#include "box.h"
#include "curproto.h"
#define CLPSUTIL_SOURCE
#include "clpsutil.h"   /* Utility interface routines                       */
#include "env.h"        /* Binary file manipulation routines                */
#include "color.h"      /* CLIPS Interface color definitions                */
#include "key.h"        /* CLIPS Interface key-bindings definitions         */
#if MOUSE
#include "mouse.h"
#endif

/******************************************************************************
 ==============================================================================
                                    CONSTANTS
 ==============================================================================
 ******************************************************************************/
#define DRIVE_ATTR  A_FGREEN|A_BBLACK

#define DRIVE_ROWS  5
#define DRIVE_COLS  23
/* DRIVE_Y,DRIVE_X defined in terms of # of lines/cols on screen */

#define COLOR_INDEX 1
#define BIND_INDEX  2
#define QUIT_INDEX  3

/******************************************************************************
 ==============================================================================
                                     MACROS
 ==============================================================================
 ******************************************************************************/
#define hide_cursor() mvcur(0,0,LINES,0)

/******************************************************************************
 ==============================================================================
                     EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
void main(void);                   /* Environment utility driver            */
void instruction_line(char *),     /* Writes bottom info line on screen     */
     clear_status_line(void),      /* Erases top info line                  */
     writestring(char *,int),      /* Writes top info line                  */
     check_for_save(int);          /* Prompts to save file if necessary     */

char *getstring(char *,char*,int); /* Gets a line of text from user         */


/* =======================================================================
   ***********************************************************************
                   INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
static void setup_stdscr(void);    /* Sets up standard screen for utilities */
static void driver_choose(void);   /* Main menu selection loop              */
static void restore_main_screen(void);
                                   /* Redraws main screen                   */

/******************************************************************************
 ==============================================================================
                     INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
static WINDOW *driver_win = NULL;       /* Main option menu                   */
static int driver_select = COLOR_INDEX; /* Current main selection             */
static void highlight(int);             /* (Un)highlights main-menu selection */

/******************************************************************************
 ==============================================================================
                         EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 NAME        : main
 PURPOSE     : Environment utility driver
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void main()
  {
FILE *fp; 
   if (initscr() == NULL)
     return;
   if ((driver_win = newwin(DRIVE_ROWS,DRIVE_COLS,
                            LINES/2-DRIVE_ROWS/2,COLS/2-DRIVE_COLS/2)) == NULL)
     {
      nl();
      echo();
      noraw();
      endwin();
      return;
     }
   nodelay(driver_win,TRUE);
   keypad(driver_win,TRUE);
   scrollok(driver_win,FALSE);
   xpaint(driver_win,DRIVE_ATTR);
   xbox(driver_win,SG_V_LIN,SG_H_LIN,DRIVE_ATTR);
   wattrset(driver_win,DRIVE_ATTR);
   mvwaddstr(driver_win,0,DRIVE_COLS/2-8,"Environment Menu");
   mvwaddstr(driver_win,COLOR_INDEX,1,"Change Color-Settings");
   mvwaddstr(driver_win,BIND_INDEX,1," Change Key-Bindings ");
   mvwaddstr(driver_win,QUIT_INDEX,1,"      Quit/Save      ");
   raw();
   noecho();
   nonl();
#if MOUSE
   init_mouse();
   mouse_region(LINES/2-DRIVE_ROWS/2+1,COLS/2-DRIVE_COLS/2+1,
                LINES/2-DRIVE_ROWS/2+DRIVE_ROWS-2,COLS/2-DRIVE_COLS/2+1);
   clear_mouse();
   hide_mouse();
#endif
   save_defaults();
   restore_main_screen();
   do
     {
      driver_choose();
      if (driver_select == COLOR_INDEX)
        {
         setup_stdscr();
         set_colors();
         restore_main_screen();
        }
      else if (driver_select == BIND_INDEX)
        {
         setup_stdscr();
         bind_keys();
         restore_main_screen();
        }
     }
   while (driver_select != QUIT_INDEX);
   check_for_save(COLOR_SETTINGS);
   check_for_save(KEY_BINDS);
   delwin(driver_win);
   trash_defaults();
   nl();
   echo();
   noraw();
   endwin();
  }

/******************************************************************************
 NAME        : instuction_line
 PURPOSE     : Displays an information line at the bottom of the screen
 DESCRIPTION : 
 INPUTS      : 1) The address of the instruction-string to be printed
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void instruction_line(instr)
  char *instr;
  {
   register int i;

   attrset(A_FGREEN|A_BBLACK);
   move(LINES-1,0);
   xwclrtoeol(stdscr);
   move(LINES-1,0);
   for (i = 0 ; instr[i] != EOS ; i++)
     xwaddch(stdscr,instr[i]);
  }

/******************************************************************************
 NAME        : clear_status_line
 PURPOSE     : Clears informational messages from line at the top of the screen
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void clear_status_line()
  {
   move(1,0);
   xwclrtoeol(stdscr);
   refresh();
   hide_cursor();
  }

/******************************************************************************
 NAME        : writestring
 PURPOSE     : Writes an informational message to the line at the top of screen
 DESCRIPTION : 
 INPUTS      : 1) The message to be written
               2) A code indicating whether to sound the terminal bell or not
                   TRUE (1) - Yes
                   FALSE (0) - No
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void writestring(msg,bell)
  char *msg;
  int bell;
  {
   move(1,0);
   xwclrtoeol(stdscr);
   mvaddstr(1,0,msg);
   refresh();
   hide_cursor();
   if (bell)
     beep();
  }

/******************************************************************************
 NAME        : getstring
 PURPOSE     : Gets a line of text from the user
 DESCRIPTION : Puts the text in user's buffer
 INPUTS      : 1) The prompt to be displayed for a text-query
               2) The buffer to store the text in
               3) The size of the buffer (including space for the NULL char)
 RETURNS     : The address of the static buffer where the text is stored.
 NOTES       : None
 ******************************************************************************/
char *getstring(prompt,inbuf,insize)
  char *prompt,*inbuf;
  int insize;
  {
   register int i = 0;
   int ch,maxlen,
       y,x,
       quit = FALSE;
   
   move(1,0);
   xwclrtoeol(stdscr);
   mvaddstr(1,0,prompt);
   refresh();
   if (insize < (maxlen = ALL_COLS-strlen(prompt)))
     maxlen = insize;
   inbuf[0] = EOS;
#if MOUSE
   save_mouse();
#endif
   while (! quit)
     {
      if ((ch = wgetch(stdscr)) != -1)
        {
         if ((ch != NEWLINE) ? (ch == CRGRTN) : TRUE)
           quit = TRUE;
         else if (ch == ESC)
           {
            inbuf[0] = EOS;
            quit = TRUE;
           }
         else if (ch == DELETE)
           {
            if (i != 0)
              {
               addstr(DEL_STR);
               refresh();
               inbuf[--i] = EOS;
              }
            else
              beep();
           }
         else if ((ch >= LOW_PRN_ASCII) && (ch <= HIGH_PRN_ASCII))
           {
            if (i != maxlen-1)
              {
               addch(ch);
               refresh();
               inbuf[i++] = ch;
               inbuf[i] = EOS;
              }
            else
              beep();
           }
         else
           beep();
       }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == RIGHT_BUTTON)
           {
            inbuf[0] = EOS;
            quit = TRUE;
           }
         else
           beep();
        }
#endif
     }
   hide_cursor();
#if MOUSE
   restore_mouse();
#endif
   if (inbuf[0] == EOS)
     clear_status_line();
   return(inbuf);
  }

/******************************************************************************
 NAME        : check_for_save
 PURPOSE     : Prompts the user to save current key-bindings before loading
                new ones
 DESCRIPTION : 
 INPUTS      : 1) The environment code to be checked :
                  KEY_BINDS (0) : Check for recent save of key-bindings
                  COLOR_SETTINGS (1) : Check for recent save of color settinsg
 RETURNS     : Nothing useful
 NOTES       : Uses getstring() and writestring()
 ******************************************************************************/
void check_for_save(code)
  int code;
  {
   int saved;
   char *prompt,*savemsg,
        buffer[50];
   int (*save_routine)(char *) = NULL;

   if (code == KEY_BINDS)
     {
      saved = binds_saved;
      prompt = "Save key-bindings? [y/n] ";
      savemsg = "Key-bindings saved.";
      save_routine = save_binds;
     }
   else
     {
      saved = colors_saved;
      prompt = "Save color-settings? [y/n] ";
      savemsg = "Color-settings saved.";
      save_routine = save_colors;
     }
   if (saved == FALSE)
     {
      getstring(prompt,buffer,2);
      if (((buffer[0] == 'y') || (buffer[0] == 'Y')) && (buffer[1] == EOS))
        {
         if (getstring("Save file : ",buffer,50)[0] == EOS)
           return;
         if (save_routine(buffer) == OK_DATA_FILE)
           writestring(savemsg,0);
         else
           writestring("Could not write to file!",1);        
        }
     }
  }

/* =======================================================================
   ***********************************************************************
                      INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : setup_stdscr
 PURPOSE     : Draws borders for color-setting/key-binding standard screen.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void setup_stdscr()
  {
   register int i;

   nodelay(stdscr,TRUE);
   keypad(stdscr,TRUE);
   attrset(A_FHI_WHITE|A_BBLACK);
   for (i = 0 ; i < COLS ; i++)
     {
      mvaddch(2,i,SG_H_LIN);
      mvaddch(LINES-2,i,SG_H_LIN);
     }
  }

/******************************************************************************
 NAME        : driver_choose
 PURPOSE     : Displays main-menu and allows user to make a selection
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Sets global variable driver_select to current selection
 ******************************************************************************/
static void driver_choose()
  {
   int quit = FALSE,
       ch,y,x,
       choice = driver_select;
   
   while (! quit)
     {
      if (choice != driver_select)
        {
         highlight(choice);
#if MOUSE
         set_mouse(LINES/2-DRIVE_ROWS/2+choice,COLS/2-DRIVE_COLS/2+1);
#endif
        }
      if ((ch = wgetch(driver_win)) != -1)
        {
         ch = key_map(ch);
         switch(ch)
           {
            case NEWLINE    : quit = TRUE;
                              break;
            case ESC        : highlight(QUIT_INDEX);
                              quit = TRUE;
                              break;
            case UP_ARROW   : choice = (choice+DRIVE_ROWS-4)%(DRIVE_ROWS-2)+1;
                              break;
            case DOWN_ARROW : choice = choice%(DRIVE_ROWS-2)+1;
                              break;
            default         : beep();
           }
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        choice = y - (LINES/2-DRIVE_ROWS/2);
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           quit = TRUE;
         else if (ch == RIGHT_BUTTON)
           {
            highlight(QUIT_INDEX);
            quit = TRUE;
           }
         else
           beep();
        }
#endif
     }
  }

/******************************************************************************
 NAME        : highlight
 PURPOSE     : (Un)highlights (old)new selection in main-menu
 DESCRIPTION : 
 INPUTS      : 1) The index (1..3) of the new selection
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void highlight(choice)
  int choice;
  {
   register int i;

   wattrset(driver_win,DRIVE_ATTR);
   wmove(driver_win,driver_select,1);
   for (i = 1 ; i < DRIVE_COLS-1 ; i++)
     waddch(driver_win,winch(driver_win) & A_CHARTEXT);

   wattrset(driver_win,flip_fgd_bgd(DRIVE_ATTR));
   wmove(driver_win,choice,1);
   for (i = 1 ; i < DRIVE_COLS-1 ; i++)
     waddch(driver_win,winch(driver_win) & A_CHARTEXT);

   driver_select = choice;

   wnoutrefresh(stdscr);
   wnoutrefresh(driver_win);
   doupdate();
   hide_cursor();
  }

/******************************************************************************
 NAME        : restore_main_screen
 PURPOSE     : Redraws main menu with banner and instruction line
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void restore_main_screen()
  {
   char *dirx;

   werase(stdscr);
   attrset(A_FYELLOW|A_BBLACK);
   mvaddstr(0,COLS/2-21,"CLIPS DOS Interface Environment Utility V1.01");
   dirx = "\030\031-Change selection  <CRT>-Activate selection  <ESC>-Quit";
   instruction_line(dirx);
   touchwin(stdscr);
   touchwin(driver_win);
   highlight(driver_select);
  }

