/******************************************************************************
 FILE CLPMENU.C :
   This file contains the routines for menu-interfaces CLIPS data such as
 rules, deffacts, and facts.
 ******************************************************************************/

/******************************************************************************
 ==============================================================================
                                  HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "constant.h"  /* CLIPS Constants                                    */
/* -- Eliminate Conflicts with COMMON.H */
#undef FALSE
#undef TRUE
#undef OFF
#undef ON
#undef EOS

#include "common.h"    /* Standard header files and useful constants/macros  */
#include "engine.h"   /* CLIPS Fact and Rule Structure Definitions           */
#include "curses.h"    /* Aspen Scientific's Curses V4.0 Package Definitions */
#include "xcurses.h"
#include "box.h"
#include "curproto.h"
#include "clipswin.h"  /* Insert/Delete Pop-up stack routines                */
#include "color.h"     /* Color definitions                                  */
#include "key.h"       /* Key-code and mapping definitions                   */
#include "pop.h"       /* Menu highlighting routine                          */
#include "alpha.h"     /* Alphabetizing routine for Rules/Deffacts Menus     */

#if MOUSE
#include "mouse.h"     /* Mouse Systems PC-Mouse Interface Routines Defns    */
#endif

/* --- CLIPS Data Structures and Routines --- */

extern void *get_next_rule(void *);
extern char *get_rule_name(void *);

#if BREAKPOINTS
extern int rule_has_break(char *);
#endif

#if DEFFACTS_CONSTRUCT

extern void *get_next_deffact(void *);
extern char *get_deffact_name(void *);

#endif

#if DEFTEMPLATES

extern void *get_next_deftemplate(void *);
extern char *get_deftemplate_name(void *);

#endif

extern struct fact *get_next_fact(struct fact *);
extern char *num_to_string(double);
extern char *symbol_string(struct draw *);

extern struct activation *get_next_activation(struct activation *act_ptr);

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/
#define MAX_ROWS 15
#define MAX_COLS 40

/******************************************************************************
 ==============================================================================
                                    MACROS
 ==============================================================================
 ******************************************************************************/
#define hide_cursor()  mvcur(0,0,LINES,0)

/******************************************************************************
 ==============================================================================
                            DATA TYPES AND STRUCTURES
 ==============================================================================
 ******************************************************************************/
/* Stack nodes for item-strings found in first pass of internal CLIPS lists --
   the stack is kept to prevent having to do all the comparisons twice */

typedef struct found_item
  {
   char *str;
   void *key;
   struct found_item *nxt;   
  } FOUNDITEM;

/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
char *rules_menu(char *prompt,char *query);
                                     /* Allows user to select rules from menu */
#if DEFFACTS_CONSTRUCT
char *deffacts_menu(char *prompt,char *query);   
                                  /* Allows user to select deffacts from menu */
#endif
#if DEFTEMPLATES
char *deftemplates_menu(char *prompt,char *query);   
                              /* Allows user to select deftemplates from menu */
#endif
#if BREAKPOINTS
char *breaks_menu(char *prompt);     
                           /* Allows user to select rules w/ breaks from menu */
char *nobreaks_menu(char *prompt);   
                          /* Allows user to select rules w/o breaks from menu */
#endif

long int facts_menu(char *prompt,char *query);
                                     /* Allows user to select facts from menu */
  
int agenda_menu(char *prompt,char *query);
                              /* Allows user to select agenda-items from menu */

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
static FOUNDITEM *menu_choose(char *prompt,        /* Menu setup routine for */
                         void *(*get)(void *),     /* rules and deffacts     */
                         char *(*name)(void *),
                         void *(*key)(void *),
                         int (*valid)(char *),
                         char *query,int sortp);
static FOUNDITEM *push_find(char *str,void *key);  /* Builds find-stack     */
static void pop_find(void);                        /* Unbuilds find-stack   */

/* Alphabetical Sort Routine Data Structure Access Functions */

static FOUNDITEM *next_find(FOUNDITEM *f);
static void set_next_find(FOUNDITEM *f,FOUNDITEM *n);
static char *get_find(FOUNDITEM *f);

static void show_menu(int offset);
                                                 /* Displays menu on screen */
static void select(int *offset,int *curr_select);/* Selection-loop for menu */
#if BREAKPOINTS
static int rule_has_no_break(char *rule);        /* menu_choose() stub      */
#endif
static char *get_fact_string(struct fact *fact_ptr);
                                                   /* Gets string for fact    */
static char *append_element_string(char *factstr,int *size,
                                   struct element *elem_ptr);
                                            /* Builds elements of fact-string */
static long int *get_fact_id(struct fact *fact_ptr);
                                            /* Gets a fact id                 */
static char *get_activation_string(struct activation *act_ptr);
                                          /* Gets a string for an agenda item */

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
static WINDOW *menu = NULL,
              *border = NULL;
static FOUNDITEM *finds = NULL,     /* Valid-node from CLIPS list stack */
                 *findsb = NULL;    /* Bottom of CLIPS list stack       */

static int clips_allocated = TRUE;  /* Indicates if strings on find-list
                                       were previously allocated by CLIPS
                                       or not.  Strings not allocated by
                                       CLIPS are deallocated by pop_find()
                                       when processing is finished        */

/******************************************************************************
 ==============================================================================
                          EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*****************************************************************************
 NAME        : rules_menu
 PURPOSE     : Pops up a menu of all rules
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) The query string
                  If the query string is non-NULL, it will be used
                  to double-check the user when they make a selection
 RETURNS     : 1) A pointer to the internal CLIPS string name of the rule
                  selected - DO NOT MODIFY THIS!!!
               2) or NULL if no rules could be found or there was an error
 NOTES       : Uses get_next_rule() and get_rule_name() from RULEMNGR.C
 *****************************************************************************/
char *rules_menu(char *prompt,char *query)
  {
   FOUNDITEM *found;
   char *str;

   found = menu_choose(prompt,get_next_rule,get_rule_name,NULL,NULL,query,TRUE);
   if (found == NULL)
     return(NULL);
   str = found->str;
   release(1,FOUNDITEM,found);
   return(str);
  }

#if DEFFACTS_CONSTRUCT

/*****************************************************************************
 NAME        : deffacts_menu
 PURPOSE     : Pops up a menu of all deffacts
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) The query string
                  If the query string is non-NULL, it will be used
                  to double-check the user when they make a selection
 RETURNS     : 1) A pointer to the internal CLIPS string name of the deffact
                  selected - DO NOT MODIFY THIS!!!
               2) or NULL if no deffacts could be found or there was an error
 NOTES       : Uses get_next_deffact() and get_deffact_name() from DEFFACTS.C
 *****************************************************************************/
char *deffacts_menu(char *prompt,char *query)
  {
   FOUNDITEM *found;
   char *str;

   found = menu_choose(prompt,get_next_deffact,
                       get_deffact_name,NULL,NULL,query,TRUE);
   if (found == NULL)
     return(NULL);
   str = found->str;
   release(1,FOUNDITEM,found);
   return(str);
  }

#endif

#if DEFTEMPLATES

/*****************************************************************************
 NAME        : deftemplates_menu
 PURPOSE     : Pops up a menu of all deftemplates
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) The query string
                  If the query string is non-NULL, it will be used
                  to double-check the user when they make a selection
 RETURNS     : 1) A pointer to the internal CLIPS string name of the deffact
                  selected - DO NOT MODIFY THIS!!!
               2) or NULL if no deftemplates could be found 
                  or there was an error
 NOTES       : Uses get_next_deftemplate() and get_deftemplate_name()
               from DEFFACTS.C
 *****************************************************************************/
char *deftemplates_menu(char *prompt,char *query)
  {
   FOUNDITEM *found;
   char *str;

   found = menu_choose(prompt,get_next_deftemplate,
                       get_deftemplate_name,NULL,NULL,query,TRUE);
   if (found == NULL)
     return(NULL);
   str = found->str;
   release(1,FOUNDITEM,found);
   return(str);
  }

#endif

#if BREAKPOINTS

/*****************************************************************************
 NAME        : Pops up a menu of CLIPS rules that have breaks
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
 RETURNS     : 1) A pointer to the internal CLIPS string name of the rule
                  selected - DO NOT MODIFY THIS!!!
               2) or NULL if no rule could be found or there was an error
 NOTES       : Uses get_next_rule() and get_rule_name() from RULEMNGR.C
               and rule_has_break() from INTREXEC.C
 *****************************************************************************/
char *breaks_menu(char *prompt)
  {
   FOUNDITEM *found;
   char *str;

   found = menu_choose(prompt,get_next_rule,get_rule_name,
                       NULL,rule_has_break,NULL,TRUE); 
   if (found == NULL)
     return(NULL);
   str = found->str;
   release(1,FOUNDITEM,found);
   return(str);
  }

/*****************************************************************************
 NAME        : Pops up a menu of CLIPS rules that do not have breaks
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
 RETURNS     : 1) A pointer to the internal CLIPS string name of the rule
                  selected - DO NOT MODIFY THIS!!!
               2) or NULL if no rule could be found or there was an error
 NOTES       : Uses get_next_rule(), and get_rule_name() from RULEMNGR.C,
               and rule_has_no_break() (this file)
 *****************************************************************************/
char *nobreaks_menu(char *prompt)
  {
   FOUNDITEM *found;
   char *str;

   found = menu_choose(prompt,get_next_rule,get_rule_name,
                       NULL,rule_has_no_break,NULL,TRUE);
   if (found == NULL)
     return(NULL);
   str = found->str;
   release(1,FOUNDITEM,found);
   return(str);
  }

#endif

/*****************************************************************************
 NAME        : facts_menu
 PURPOSE     : Forms a list of strings representing the current-facts
               and then passes that list to menu_choose() for selection
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) A query string (can be NULL)
 RETURNS     : 1) The Fact-ID Number of the fact selected 
               2) -1 if the menu was cancelled
 NOTES       : Follows the methodologies of show_elements() and
               print_elements() in FACTMNGR.C
 *****************************************************************************/
long int facts_menu(char *prompt,char *query)
  {
   FOUNDITEM *found;
   long int fid;

   clips_allocated = FALSE;
   found = menu_choose(prompt,(void *(*)()) get_next_fact,
                       (char *(*)()) get_fact_string,
                       (void *(*)()) get_fact_id,NULL,query,TRUE);
   clips_allocated = TRUE;
   if (found == NULL)
     return((long int) -1);
   fid = (long int) *((long int *) (found->key));
   release(strlen(found->str)+1,char,found->str);
   release(1,FOUNDITEM,found);
   return(fid);
  }

/*****************************************************************************
 NAME        : agenda_menu
 PURPOSE     : Forms a list of strings representing the current agenda
               and then passes that list to menu_choose() for selection
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) A query string (can be NULL)
 RETURNS     : 1) The index of the agenda-item selected in the agenda-list
               2) -1 if the menu was cancelled
 NOTES       : 
 *****************************************************************************/
int agenda_menu(char *prompt,char *query)
  {
   FOUNDITEM *found;
   int agenda_id;

   clips_allocated = FALSE;
   found = menu_choose(prompt,(void *(*)()) get_next_activation,
                       (char *(*)()) get_activation_string,
                       NULL,NULL,query,FALSE);
   clips_allocated = TRUE;
   if (found == NULL)
     return(-1);
   agenda_id = (int) found->key;
   release(strlen(found->str)+1,char,found->str);
   release(1,FOUNDITEM,found);
   return(agenda_id);
  }

/******************************************************************************
 ==============================================================================
                          INTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*****************************************************************************
 NAME        : menu_choose
 PURPOSE     : Sets up a menu for CLIPS rules/deffacts and lets the user
               pick one
 DESCRIPTION : 
 INPUTS      : 1) A prompt string
               2) A pointer to a structure accessing function which MUST
                  operate the following way : when the function is passed
                  NULL, it returns the address of the head of the list
                  (which contains the address of the next node).  When
                  the function is passed a node address it returns the 
                  address of the next node - the function returns NULL when
                  the list is empty.
               3) A pointer to a structure accessing function which MUST
                  operate in the following way : when passed a valid node
                  address returned from function 2) this routine returns a
                  a pointer to a valid string representation of that node.
               4) A pointer to a structure accessing function which will
                  return the address of a key for that structure which
                  will be meaningful to the caller if that item is the
                  one selected by the user. If the address is NULL, no key
                  will be stored.
               5) A pointer to a function which when passed the string
                  returned from function 3) returns TRUE (1) or FALSE (0)
                  depending on whether that node should be included in the
                  menu or not.  If this address is NULL, no validating
                  function will be called.
               6) Query string - if the argument is NULL the selection will
                  not be confirmed.
               7) A flag indicating whether to alphabetically sort the 
                  menu-item list or not (TRUE (1) or FALSE(0)).
 RETURNS     : 1) A pointer to a FOUNDITEM structure containing the string
                  selected and its key (if any).  It is the caller's
                  responsibility to deallocate this structure (and any internal
                  fields if they were not allocated by CLIPS).
               2) NULL if there was an error or the user cancelled the menu.
 NOTES       : Assumes that the keys of the items in the list (if any) do
               not need to be deallocated by pop_find().  pop_find() does
               however deallocate non-CLIPS allocated strings if the variable
               clips_allocated is set to FALSE.
                 Also, this routine keeps a track of the index of the items
               in the list it builds and stores this value as the key for
               each item if no key-accessing function is specified. (It is
               stored as a (long int *) though, so be careful.)
 *****************************************************************************/
static FOUNDITEM *menu_choose(char *prompt,
                         void *(*get)(void *),
                         char *(*name)(void *),
                         void *(*key)(void *),
                         int (*valid)(char *),
                         char *query,int sortp)
  {
   int item_cnt = 0;
   void *gen_ptr,*key_index;
   char *str;
   register int i,j;
   ch_attr attr;
   int choice = 0,
       offset = 0;
   FOUNDITEM *fptr,*found;
   int done = FALSE;
   int len = 0,
       maxlen = 0;
   int cnt,y,x,rows,cols;

   for (gen_ptr = (*get)(NULL) , cnt = 1 ; gen_ptr != NULL ; 
        gen_ptr = (*get)(gen_ptr) , cnt++)
     {
      str = (*name)(gen_ptr);
      if ((str != NULL) ? ((valid != NULL) ? (*valid)(str) : TRUE) : FALSE)
        {
         key_index = (key != NULL) ? (*key)(gen_ptr) : (long int *) cnt;
         if (push_find(str,key_index) == NULL)
           return(NULL);
         item_cnt++;
         if ((len = strlen(str)) > maxlen)
           maxlen = len;
        }
     }
   if (item_cnt == 0)
     {
      beep();
      return(NULL);
     }
   menu = newpad(item_cnt,MAX_COLS);
   if (menu == NULL)
     {
      while (finds != NULL)
        pop_find();      
      return(NULL);
     }
   border = newwin(MAX_ROWS+2,MAX_COLS+2,
                   (LINES-MAX_ROWS-2)/2,(COLS-MAX_COLS-2)/2);
   if (border == NULL)
     {
      while (finds != NULL)
        pop_find();      
      delwin(menu);
      menu = NULL;
      return(NULL);
     }
   xpaint(border,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(border,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wmove(border,0,1);
   for (j = 0 ; ((j < MAX_COLS) ? (prompt[j] != EOS) : FALSE) ; j++)
     waddch(border,prompt[j]);
   mvwprintw(border,MAX_ROWS+1,(MAX_COLS+2)/2-4,"%d Item(s)",item_cnt);
   attr = colors[POP_FILE_TEXT]|colors[POP_FILE_BGD];
   xpaint(menu,attr);
   keypad(menu,TRUE);
   nodelay(menu,TRUE);
   wattrset(menu,attr);
   touchwin(border);
   wnoutrefresh(border);
   doupdate();
   hide_cursor();
   if (sortp)
     {
      fptr = alpha_sort(finds,
                        (void *(*)()) next_find,
                        (void (*)()) set_next_find,
                        (char *(*)()) get_find,maxlen);

      /* ==================================
         findsb pointer may now be invalid!
         ================================== */

      if (fptr != NULL)
        finds = fptr;
     }
   i = 0;
   for (fptr = finds ; fptr != NULL ; fptr = fptr->nxt)
     {
      wmove(menu,i++,0);
      for (j = 0 ; ((j < MAX_COLS) ? ((fptr->str)[j] != EOS) : FALSE) ; j++)
        waddch(menu,(fptr->str)[j]);
     }
   pop_up_highlight(menu,choice,attr,choice,flip_fgd_bgd(attr));
   show_menu(offset);
#if MOUSE
   save_mouse();
   mouse_region((LINES-MAX_ROWS-2)/2,(COLS-MAX_COLS-2)/2+1,
                (LINES-MAX_ROWS-2)/2+MAX_ROWS+1,(COLS-MAX_COLS-2)/2+1);
#endif
   insert_pop_up(POP_WIN,border);
   while (! done)
     {
      select(&offset,&choice);
      fptr = NULL;
      if (choice == -1)
        done = TRUE;
      else
        {
         i = 0;
         fptr = finds;
         while (i != choice)
           {
            i++;
            fptr = fptr->nxt;
           }
         str = fptr->str;
         if (query != NULL)
           {
            getbegyx(border,y,x);
            getmaxrc(border,rows,cols);
            insert_pop_up(POP_PAD,menu,offset,0,y+1,x+1,y+rows-2,x+cols-2);
            if (pop_up_warning(LFT_JUST,SILENCE,2,query,str) == YES)
              done =TRUE;
            delete_pop_up();
           }
         else
           done = TRUE;
        }
     }

#if MOUSE
   restore_mouse();
#endif      
   delete_pop_up();
   delwin(border);
   border = NULL;
   delwin(menu);
   menu = NULL;
   if (fptr != NULL)
     {
      found = finds;
      if (found == fptr)
        {
         finds = finds->nxt;
         fptr->nxt = NULL;
        }
      else
        {
         while (found->nxt != fptr)
           found = found->nxt;
         found->nxt = fptr->nxt;
         fptr->nxt = NULL;
        }
     }
   while (finds != NULL)
     pop_find();
   return(fptr);
  }

/*****************************************************************************
 NAME        : push_find
 PURPOSE     : Pushes the address of an internal CLIPS string onto a stack
 DESCRIPTION : This stack is popped off from the top and written into the
               menu backwards - since the top of the stack was the last item
               in the CLIPS internal list.
 INPUTS      : 1) The address of the CLIPS internal string
               2) The address of a key for the item
 RETURNS     : 1) The address of the top of the stack
               2) or NULL if a new node could not be allocated
 NOTES       : In the event of an allocation error, any partial stack will
               be deallocated by this routine.
 *****************************************************************************/
static FOUNDITEM *push_find(char *str,void *key)
  {
   FOUNDITEM *fptr;

   if ((fptr = balloc(1,FOUNDITEM)) == NULL)
     {
      while (finds != NULL)
        {
         fptr = finds;
         finds = finds->nxt;
         if ((! clips_allocated) ? (fptr->str != NULL) : FALSE)
            release(strlen(fptr->str)+1,char,fptr->str);
         release(1,FOUNDITEM,fptr);
        }
      return(NULL);
     }
   fptr->str = str;
   fptr->key = key;
   fptr->nxt = NULL;
   if (finds == NULL)
     {
      finds = fptr;
      findsb = fptr;
     }
   else
     {
      findsb->nxt = fptr;
      findsb = fptr;
     }
   return(finds);
  }

/*****************************************************************************
 NAME        : pop_find
 PURPOSE     : Removes the top node from the CLIPS string stack
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void pop_find()
  {
   FOUNDITEM *fptr;

   if (finds != NULL)
     {
      fptr = finds;
      finds = finds->nxt;
      if ((! clips_allocated) ? (fptr->str != NULL) : FALSE)
         release(strlen(fptr->str)+1,char,fptr->str);
      release(1,FOUNDITEM,fptr);
     }
  }

/*****************************************************************************
 NAME        : next_find, set_next_find, get_find
 PURPOSE     : Data structure access functions for alpha_sort() in ALPHA.C
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 *****************************************************************************/
static FOUNDITEM *next_find(FOUNDITEM *f)
  {
   return(f->nxt);
  }

static void set_next_find(FOUNDITEM *f,FOUNDITEM *n)
  {
   f->nxt = n;
  }

static char *get_find(FOUNDITEM *f)
  {
   return(f->str);
  }

/*****************************************************************************
 NAME        : show_menu
 PURPOSE     : Displays section of menu on screen
 DESCRIPTION : 
 INPUTS      : 1) The display-start-row in the menu
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void show_menu(int offset)
  {
   int y,x,
       rows,cols,
       items;

   getbegyx(border,y,x);
   getmaxrc(border,rows,cols);
   getmaxr(menu,items);
   wattrset(border,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wmove(border,1,cols-1);
   xwaddch(border,(offset == 0) ? DB_V_LIN : ACS_UARROW);
   wmove(border,rows-2,cols-1);
   xwaddch(border,((items-offset) <= (rows-2)) ? DB_V_LIN : ACS_DARROW);
   wnoutrefresh(border);
   touchwin(menu);
   pnoutrefresh(menu,offset,0,y+1,x+1,y+rows-2,x+cols-2);
   doupdate();
   hide_cursor();
  }

/*****************************************************************************
 NAME        : select
 PURPOSE     : Allows user to move through menu, and cancel or select an item
 DESCRIPTION : 
 INPUTS      : 1) The address of the top-of-menu-display offset counter
               2) The address of the index of the current menu selection
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void select(int *offset,int *curr_select)
  {
   ch_attr attr;
   int done = FALSE,
       new_select;
   int max_items,
       maxlen,
       maxwid,
       ch;
#if MOUSE
   int y,x;
#endif
   
   getmaxr(menu,max_items);
   getmaxrc(border,maxlen,maxwid);
   maxlen -= 2;
   maxwid -= 2;
   attr = colors[POP_FILE_TEXT]|colors[POP_FILE_BGD];
   new_select = *curr_select;
   while (! done)
     {
      if (new_select != *curr_select)
        {
         pop_up_highlight(menu,*curr_select,attr,new_select,flip_fgd_bgd(attr));
         if (new_select < *offset)
           *offset = new_select;
         else if (new_select > (*offset+maxlen-1))
           *offset = new_select-maxlen+1;
         *curr_select = new_select;
#if MOUSE
         getbegyx(border,y,x);
         set_mouse(y+new_select-(*offset)+1,x+1);
#endif
         show_menu(*offset);
        }
      if ((ch = wgetch(menu)) != -1)
        {
         ch = key_map(ch);
         switch(ch)
           {
            case ESC        : *curr_select = -1;
            case MAIN_MENU  :
            case NEWLINE    : done = TRUE;
                              break;
            case UP_ARROW   : 
            case PAGE_UP    : 
            case BUFFER_TOP : if (*curr_select > 0)
                                {
                                 if (ch == UP_ARROW)
                                   new_select = (*curr_select)-1;
                                 else if ((ch == PAGE_UP) && 
                                          (*curr_select > maxlen-1))
                                   new_select = (*curr_select)-maxlen;
                                 else
                                   new_select = 0;
                                }
                              else
                                beep();
                              break;
            case DOWN_ARROW : 
            case PAGE_DOWN  : 
            case BUFFER_END : if (*curr_select < max_items-1)
                                {
                                 if (ch == DOWN_ARROW)
                                   new_select = (*curr_select)+1;
                                 else if ((ch == PAGE_DOWN) && 
                                          ((max_items-(*curr_select)) > maxlen))
                                   new_select = (*curr_select)+maxlen;
                                 else
                                   new_select = max_items-1;
                                }
                              else
                                beep();
                              break;
            default         : beep();
           }
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        {         
         y -= (LINES-maxlen-2)/2+1;
         if (y < 0)
           new_select = *offset-1;
         else if (y > maxlen-1)
           new_select = *offset+maxlen;
         else
           new_select = *offset+y;
         if (new_select > max_items-1)
           new_select = max_items-1;
         else if (new_select < 0)
           new_select = 0;
        }
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           done = TRUE;
         else if (ch == RIGHT_BUTTON)
           {
            done = TRUE;
            *curr_select = -1;
           }
         else
           beep();
        }
#endif
     }
  }

#if BREAKPOINTS

/*****************************************************************************
 NAME        : rule_has_no_break
 PURPOSE     : Indicates whether a rule has a break or not - used merely
               as a stub for menu_choose()
 DESCRIPTION : 
 INPUTS      : 1) The rule name
 RETURNS     : 1) TRUE (1) if the rule has no break
               2) FALSE (0) otherwise
 NOTES       : None
 *****************************************************************************/
static int rule_has_no_break(char *rule)
  {
   return(! rule_has_break(rule));
  }

#endif

/*****************************************************************************
 NAME        : get_fact_string
 PURPOSE     : Builds a string representing a fact in the CLIPS network
 DESCRIPTION : Uses the methods of the routines show_elements() and
               print_element() in FACTMNGR.C
 INPUTS      : 1) The address of the internal CLIPS fact
 RETURNS     : 1) The address of that fact's string represention
               2) NULL if there was an error
 NOTES       : It is the caller's responsibility to deallocate the space
               allocated by this routine for the string
 *****************************************************************************/
static char *get_fact_string(struct fact *fact_ptr)
  {
   struct element *sublist;
   int length, i, size = 0;
   char *factstr = NULL;
   
   sublist = fact_ptr->atoms;
   length = fact_ptr->fact_length;

   for (i = 0; i < length ; i++)
     {
      if ((factstr = append_element_string(factstr,&size,&sublist[i])) == NULL)
        break;
     }
   if (factstr != NULL)
     factstr[size-2] = ')';
   return(factstr);
  }

/*****************************************************************************
 NAME        : append_element_string
 PURPOSE     : Allocates memory for and appends string-representation of
               a particular fact-element to the fact-string
 DESCRIPTION : 
 INPUTS      : 1) The current address of the fact-string
               2) The address of a buffer holding the current fact-string
                  size (including space for the terminating NULL)
               3) A pointer to the fact-element CLIPS draw-structure
 RETURNS     : 1) The new address of the fact-string
               2) NULL if there was an error
 NOTES       : None
 *****************************************************************************/
static char *append_element_string(char *factstr,int *size,
                                   struct element *elem_ptr)
  {
   extern int print_num(char *,float);

   char *newfstr,*str;
   int add_size;

   if (elem_ptr->type == NUMBER) 
     {
      str = num_to_string(elem_ptr->val.fvalue);
      add_size = strlen(str)+1;
     }
   else if (elem_ptr->type == WORD)
     {
      str = symbol_string(elem_ptr->val.hvalue);
      add_size = strlen(str)+1;
     }
   else if (elem_ptr->type == STRING)
     {
      str = symbol_string(elem_ptr->val.hvalue);
      add_size = strlen(str)+3;
     }
   if (*size == 0)
     {
      if ((newfstr = balloc(add_size+2,char)) != NULL)
        {
         if (elem_ptr->type != STRING)
           sprintf(newfstr,"(%s ",str);
         else
           sprintf(newfstr,"(\"%s\" ",str);
         *size = add_size+2;
        }
     }
   else
     {
      if ((newfstr = ralloc(char,factstr,*size,*size+add_size)) != NULL)
        {
         if (elem_ptr->type != STRING)
           sprintf(newfstr+((*size)-1),"%s ",str);
         else
           sprintf(newfstr+((*size)-1),"\"%s\" ",str);
         *size += add_size;
        }
     }
   return(newfstr);
  }

/*****************************************************************************
 NAME        : get_fact_id
 PURPOSE     : Finds the addresss of the ID-Number of a CLIPS Fact
 DESCRIPTION : Called by menu-choose to establish a key for selected facts
               since the list reordered
 INPUTS      : 1) A pointer to the internal CLIPS fact
 RETURNS     : 1) A pointer to a long integer which is the fact-id
 NOTES       : None
 *****************************************************************************/
static long int *get_fact_id(struct fact *fact_ptr)
  {
   return(&(fact_ptr->ID));
  }

/*****************************************************************************
 NAME        : get_activation_string
 PURPOSE     : Builds a string representing an agenda-item in the CLIPS network
 DESCRIPTION : 
 INPUTS      : 1) The address of the internal CLIPS agenda-item
 RETURNS     : 1) The address of that agenda-items's string represention
               2) NULL if there was an error
 NOTES       : It is the caller's responsibility to deallocate the space
               allocated by this routine for the string
 *****************************************************************************/
char *get_activation_string(struct activation *act_ptr)
  {
   struct fbind *list;
   unsigned size,szdelta;
   char *buffer,app[15];

   size = strlen(act_ptr->rule)+10;
   if ((buffer = balloc(size,char)) == NULL)
     return(NULL);
   sprintf(buffer,"%-6d %s: ",act_ptr->salience,act_ptr->rule);
   
   list = act_ptr->basis->binds;
   while (list != NULL)
     {
      app[0] = EOS;
      if ((list->whoset >= 0) && (list->next != NULL))
        sprintf(app,"f-%ld,",list->whoset);
      else if (list->whoset >= 0)
        sprintf(app,"f-%ld",list->whoset);
      else if (list->next != NULL)
        strcpy(app,",");
      if (app[0] != EOS)
        {
         szdelta = strlen(app);          
         if ((buffer = ralloc(char,buffer,size,size+szdelta)) == NULL)
           return(NULL);
         strcat(buffer,app);
         size += szdelta;
        }
        
      list = list->next;
     }
   return(buffer);
  }


