/* ***********************************************************************
   FILE : CLIPSWIN.C
     This file contains all of the primary driving routines for
   the visual interface.
     Support Routines can be found in the files : ENV.C, POP.C, KEY.C,
   COLOR.C, MOUSE.C, DIRMENU.C, CLPMENU.C, ALPHA.C and HELP.C
     All of these source files have HEADER files which correspond to them.
   *********************************************************************** */
   
/* =======================================================================
   ***********************************************************************
                             HEADER FILES
   ***********************************************************************
   ======================================================================= */
#include "common.h"   /* Standard header files, constants, and macros      */
#include <stdarg.h>   /* Variable argument lists for insert_pop_up()       */
#define _NO_STDIO
#include "curses.h"   /* Aspen Scientific Curses V4.0 Header Files -       */
#include "xcurses.h"  /*    Screen interface function prototypes and       */
#include "box.h"      /*      constants                                    */
#include "curproto.h"
#define CLPWIN_SOURCE
#include "clipswin.h"
#include "env.h"
#include "pop.h"      /* Menu, pop-up warning, and pop-up text entry stuff */
#include "color.h"    /* Color constant and access routines defns          */
#include "key.h"      /* Key-binding maps and info                         */
#include "dirmenu.h"  /* Menu access routine to DOS file system            */
#include "clpmenu.h"  /* Menu access routines to CLIPS rules/deffacts      */
#if CLP_TEXTPRO
#include "help.h"     /* Interface on-line help routines                   */
#endif
#if MOUSE
#include "mouse.h"    /* Mouse Systems PC-Mouse function prototypes        */
#endif

/* =======================================================================
   ***********************************************************************
                                CONSTANTS
   ***********************************************************************
   ======================================================================= */

#define TERMENVSIZE          40       /* Size of terminal defn buffer    */

#define LOG_NAME_TABLE_SIZE  15       /* Size of logical name hash table */

   /* ------------------------------------------------------------------
        This hash table size has been chosen with extreme care ; use
      caution when changing it or the hash algorithm defined below.
      The two together have been designed so that there will be no
      collisions between the 8 standard CLIPS logical names, thus
      insuring that only one string-comparison per logical name will
      be performed.
        The 8 standard logical names are : wclips, wdialog, wdisplay,
      werror, wtrace, wagenda, stdin, and stdout.
        The string comparison is done as a double-check once the hash
      node is reached, thus preventing bad logical names from being
      recognized.  However, it is important that no two logical names
      have the same hash value (i.e don't collide) or looking up a
      logical name will be equivalent to a linear search. It is highly
      desirable to always have to do only one string-compare since these
      logical names are used quite frequently.  This hash table size
      is the smallest possible with the current hash function so that
      there are no collisions between the 8 standard logical names.
        Thus, if you wish to add more valid logical names, to maintain
      efficiency you should insure that the table size combined with
      the hash function provide for no collisions.  However, if you
      choose not to do this, the lookups will still work via chaining
      resolution on collisions.
      ------------------------------------------------------------------ */
   
/* Screen Dimensions */

/* Screen-Map (Not to scale)

          HEADER WIN (Top 3 lines)
   +--------+----------------------------+
   | AD-WIN |         MAIN_MENU_WIN      |
   +--------+----------------------------+
  L|                                     |R
  B|                                     |B
  O|                                     |O
  R|              CLIPS WIN              |R
  D|                                     |D
   |                                     |
   |                                     |
   +-------------------------------------+
              FOOTER (Bottom 1 line)

*/

#define HDR_LNS         3
#define HDR_COLS        80
#define HDR_Y           0
#define HDR_X           0

#define CLIPS_LOGO      "CLIPS"

#define AD_LNS          1
#define AD_COLS         7
#define AD_Y            1
#define AD_X            1

#define MN_MENU_LNS     1
#define MN_MENU_COLS    HDR_COLS-AD_COLS-3
#define MN_MENU_Y       1
#define MN_MENU_X       AD_COLS+2

/* LBORD_LNS is defined in terms of how many lines there are on the screen */
#define LBORD_COLS      1
#define LBORD_Y         3
#define LBORD_X         0

/* LBORD_LNS is defined in terms of how many lines there are on the screen */
#define RBORD_COLS      1
#define RBORD_Y         3
#define RBORD_X         79

#define FTR_LNS         1
#define FTR_COLS        80
/* FTR_Y is defined in terms of how many lines there are on the screen */
#define FTR_X           0

/* CLIPS_LNS is defined in terms of how many lines there are on the screen */
#define CLIPS_COLS      78
#define CLIPS_Y         3
#define CLIPS_X         1

/* CLIPS Window Scrolling Command Codes */
#define UP              0
#define DOWN            1
#define TOP             2
#define BOTTOM          3

/* -------------------------------------------------
   Keyboard Buffer Size for getc/ungetc simulations
   ------------------------------------------------- */
#define UNGET_BUF_SIZE     128

/* =======================================================================
   ***********************************************************************
                       INTERNALLY VISIBLE DATA TYPES
   ***********************************************************************
   ======================================================================= */

/* ---------------------------- 
   Logical name hash table node
   ---------------------------- */
typedef struct log_name
  {
   char *name;           /* Logical name                                  */
   int color;            /* Index into color array for display attribute  */
   struct log_name *nxt; /* Pointer to the next node in the chain at a
                            particular slot in the hash table             */
  } LOGNAME;

/* ----------------------------------------------------------------------
   Queue block structure for placing/removing windows on/from the
   physical screen -- a queue is formed of all pop-up windows
   currently on the screen (this does not include the main windows which
   are always present).  On a complete redraw of the screen, the
   main windows are first redrawn, and then this queue is followed
   in order.  Thus, "removing" a window from the screen becomes a simple
   task : simply remove it from the queue and then redraw the screen
     As expected, insertions/deletions can only occur at the end of the
   queue 
   ---------------------------------------------------------------------- */
typedef struct window_block
  {
   WINDOW *win;               /* Window in the queue                   */
   int py,px,ssy,ssx,sey,sex; /* Parameters needed for refreshing pads */
   struct window_block *prv,  /* Pointer to the previous window        */
                       *nxt;  /* Pointer to the next window            */
  } WINBLK;


/* =======================================================================
   ***********************************************************************
            EXTERN PROTOTYPES FOR CLIPS FUNCTIONS NOT IN THIS FILE
   ***********************************************************************
   ======================================================================= */
extern FILE *ld_input_fp;

extern int set_redraw_function(int (*)());
extern int set_pause_env_function(int (*)());
extern int set_cont_env_function(int (*)());
extern void set_mem_status_function(void (*)());
extern int set_watch_display_function(void (*)());
extern int set_dribble_status_function(void (*)());
extern int cl_print(char *,char *);
extern int cl_exit(int);
extern int add_router(char *,int,int (*)(),int (*)(),
                      int (*)(),int (*)(),int (*)());
extern int del_router(char *);
extern int set_facts_watch(int);
extern int get_facts_watch(void);
extern int set_rules_watch(int);
extern int get_rules_watch(void);
extern int set_activations_watch(int);
extern int get_activations_watch(void);
extern int set_compilations_watch(int);
extern int get_compilations_watch(void);
extern unsigned long actual_mem_reqd(unsigned long);
extern int set_execution_error(int);
extern int add_exec_function(char *,int (*)());
extern int remove_exec_function(char *);
extern int dribble_active(void);

extern int move_activation_to_top(int),
           delete_activation(int);

/* CLIPS Command-Line Parsing Stuff */

extern int print_prompt(void);
extern int expand_command_string(char);
extern int set_command_string(char *);
extern int append_command_string(char *);
extern char *get_command_string(void);
extern int (*set_event_function(int (*)()))();

/* =======================================================================
   ***********************************************************************
                   EXTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */

/* CLIPS Interface Router Functions */

void next_event(void);                  /* Interface key/mouse routine   */
int  win_query(char *);                 /* Interface router recognizer   */
void win_print(char *,char *);          /* Interface output routine      */
int  win_getc(char *);                  /* Interface input routine       */
void win_ungetc(char,char *);           /* Interface input routine       */
void init_interface(int mono);          /* Initial setup of screen       */
void stop_interface(int);               /* Interface cleanup routine     */
void check_for_break(void);             /* Checks for CTRL-C/CTRL-BREAKS */

/* Window Interface Routines */

void redisplay_wins(void);              /* Redraws entire screen         */
void memory_status(void);               /* Amount of RAM left            */
void watch_status(void);                /* Prints Watch On/Off Messages  */
void dribble_status(int);               /* Updates debug-menu            */
int  insert_pop_up(int,WINDOW *,...);   /* Puts window in pop-up queue   */
void delete_pop_up(void);               /* Takes last window from queue  */

/* CLIPS Menu Interface Functions */

#if CLP_TEXTPRO

int do_interface_help(int);

#if CLP_HELP

int do_clips_help(int),
    do_clips_help_path(int);

#endif
#endif

#if ! BLOAD_ONLY
int do_load(int),       /* File menu options */
    do_save(int);
#endif

#if SAVE_FACTS
int do_load_facts(int),
    do_save_facts(int);
#endif

#if BLOAD || BLOAD_AND_BSAVE || BLOAD_ONLY
int do_bload(int);
#endif

#if BLOAD_AND_BSAVE
int do_bsave(int);
#endif

int do_reset(int),      /* Execute menu options */
    do_run(int),
    do_step(int),
    do_batch(int); 

int do_watch(int),     /* Debug menu options */
    do_matches(int),
    do_dribble(int);

#if BREAKPOINTS
int do_break(int);
#endif

void do_watch_all_toggle(void);

int do_assert(int),      /* Action menu options */
    do_retract(int),
    do_excise(int),
    do_fire(int),
    do_activate_remove(int),
#if DEFFACTS_CONSTRUCT
    do_deffact_remove(int),
#endif
#if DEFTEMPLATES
    do_deftemplate_remove(int),
#endif
    do_clear(int),
    do_exit(int);

int do_facts(int),       /* Examine menu options */
    do_rules(int),
    do_agenda(int),
    do_pprule(int);

#if DEFFACTS_CONSTRUCT
int do_deffacts(int),
    do_ppdeffact(int);
#endif

#if DEFTEMPLATES
int do_deftemplates(int),
    do_ppdeftemplate(int);
#endif

int do_clear_window(int), /* Options menu options */
    clips_clear_window(void),
    do_spawn(int),
    do_rule_memory_update(int);

#if CLP_EDIT
int do_edit(int);
#endif

int do_toggle_more(int);

int do_system(char *);    /* system() interface function for command-line */

int do_load_env(int);     /* Environment menu options */

void pause_interface(void);   /* Sets terminal to normal state                */
void restart_interface(int); /* Resets terminal to valid state for interface */

/* =======================================================================
   ***********************************************************************
                   INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
/* Window Interface Routines */

static int get_terminal_type(void);     /* Determines terminal caps      */
static void get_environment(void);      /* Sets screen to uncooked state */
#if MOUSE
static void load_mouse(void);           /* Initializes mouse environment */
#endif
static int  allocate_windows(void);     /* Inits screen data structures  */
static int  allocate_menus(void);       /* Initializes menu structures   */
static void draw_border(void);          /* Displays the main border      */
static void draw_advertisement(void);   /* Displays CLIPS on the screen  */
static void draw_main_menu(void);       /* Displays the main CLIPS menu  */
static void draw_main_window(void);     /* Displays CLIPS window text    */
static void reset_colors(void);         /* Redraws screen with new colors*/
static void update_screen(void);        /* Flushes output to the screen  */
static void highlight_main_menu(int);   /* Highlights main menu selects  */
static int  main_pop_up_start(int);     /* Gets center screen box coord  */
static void deallocate_pop_up_queue(void);
                                        /* Frees pop-up-queue memory     */
static void deallocate_windows(void);   /* Frees interface memory        */
static void deallocate_menus(void);     /* Releases menu memory space    */
static void restore_environment(void);  /* Puts terminal in normal state */

/* Logical Name Hash Table Routines */

static int  init_log_name_table(void);  /* Inits logical name hash table */
static void deallocate_log_name_table(void);
                                        /* Releases hash table memory    */
static LOGNAME *log_insert(char *,int); /* Puts logical name in table    */
static LOGNAME *log_lookup(char *);     /* Searches table for a logname  */
static int  hash(char *,int);           /* Gets the hash value of a str  */

/* I/O Stuff */

static void append_file_clips_cmd_str(char *);
                                         /* Appends a file-name to CLIPS
                                              command-string             */

static int clips_addch(char);          /* Curses linefeed and scroll    
                                           glitches hack-fixes           */
static void clips_advance_row(void);   /* Moves cursor to new line       */
static void clips_addstr(char *);      /* Adds a whole string to window  */
static int  process_special(int,int *,int *);
                                       /* Handles special controls       */
static void pause_output(void);        /* --More-- Utility for output    */
static void describe_key(void);        /* Run-time display of key-binds  */

/* =======================================================================
   ***********************************************************************
                       INTERNALLY VISIBLE GLOBAL VARIABLES
   ***********************************************************************
   ======================================================================= */

static char termenv[TERMENVSIZE];    /* Holds defn for standard terminal */

/* ----------------------------------------------------------------
   Hash table containing entries for each logical name -- used for
   quick lookup for logical name recognition and establishing
   what color to use.
   --------------------------------------------------------------- */
static LOGNAME *log_name_table[LOG_NAME_TABLE_SIZE],
               *stdin_log = NULL;

static int ega_mode = FALSE;  /* Indicates current display mode of screen */

/* Screen Interface Windows */

static WINDOW
   *header_window = NULL,
   *advertisement_window = NULL,
   *main_menu_window = NULL,
   *left_border_window = NULL,
   *right_border_window = NULL,
   *footer_window = NULL,
   *clips_window = NULL;

static int clips_lns = 0,
           clips_row = 0,
           clips_col = 0;           /* Current cursor posn in the CLIPS pad */

static int main_menu_cell_width,    /* Width of a main menu selection cell  */
           main_menu_cell_offset,   /* # of blanks between left edge of     
                                       main menu and first cell             */
           main_menu_cell = 0;      /* Currently selected main menu cell    */

/* -------------------------------------------------------------
   List of Menu Commands and the functions associated with them
   ------------------------------------------------------------- */
static MENU *main_menu = NULL,
            *file_menu = NULL,
            *execute_menu = NULL,
            *debug_menu = NULL,
            *watch_menu = NULL,            
            *action_menu = NULL,
            *examine_menu = NULL,
#if CLP_HELP && CLP_TEXTPRO
            *help_menu = NULL,
#endif
            *options_menu = NULL,
            *env_menu = NULL;
#if CLP_EDIT
static MENU *edit_menu = NULL;
#endif

/* -----------------------------------------------------------
    Pop-Up Window Queue -- see note above for WINBLK data type
   ----------------------------------------------------------- */
static WINBLK *pop_up_queue_top = NULL,
              *pop_up_queue_bottom = NULL;

/* ------------------------------------------------------------------------
   Keyboard buffer-queue and unget buffer-stack for getc/ungetc simulations
   ------------------------------------------------------------------------ */
static char unget_buf[UNGET_BUF_SIZE];

/* --------------------------------------------------
   The unget buffer is a first-in/last-out stack
     (Thus requiring only one pointer)
   --------------------------------------------------- */
static int unget_buf_ptr = 0;   /* Pointer to char in unget buffer stack     */

/* ----------------------------------------------------------------------
   Number of input characters echoed to screen by win_getc()
   since last system output - this the maximum number of characters for
   which win_getc() will accept a DELETE.
   ---------------------------------------------------------------------- */
static int input_cnt = 0;

/* -----------------------------------------------------------------------
   Number of lines printed since last echoed input.  This variable is used
   to know when to halt output and query the user when to continue with
   the "--More--" message.
   ----------------------------------------------------------------------- */
static int output_lines = 0;

/* -- Flag to use --More-- facility or not */

static int use_more = TRUE;

/* Update memory-status between rule-firings or not */

static int rule_fire_memory = FALSE;

/* -------------------------------------------------------------------------
   The following map is used to keep track of what column position a newline
   occurred on for a particular line.  The map is used for deletions across
   newlines so as to know where to place the cursor. Besides, for such a
   nice feature, I figured 39 bytes wasn't a big deal.
     The nlmap_ptr index is used to keep track of which position in the map
   corresponds to line 0 -- thus when the window scrolls, it is only 
   necessary to shift this index, not the whole array.
   ------------------------------------------------------------------------- */
unsigned char newline_map[EGA_LINES-HDR_LNS-FTR_LNS];
static int nlmap_ptr = 0;

/* =======================================================================
   ***********************************************************************
                         EXTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : next_event
 PURPOSE     : Grabs characters form the keyboard for the command line
               as well as processing special function characters and mouse
               input.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : This function is substituted for the normal get_next_event()
               function in COMMLINE.C by the function set_event_function()
               in the same file.
 ******************************************************************************/
void next_event()
  {
   int ch[2],
       quit = FALSE;
   char *cmdstr;
   int mn_menu_sel,
       perform = FALSE,
       shift_menu = FALSE;
   MENU_ITEM *item;
   int item_rtn;
#if MOUSE
   int y,x;
#endif

   ch[1] = EOS;
   mn_menu_sel = main_menu_cell;
   while (! quit)
     {
      if (mn_menu_sel != main_menu_cell)
        {
         highlight_main_menu(mn_menu_sel);
         update_screen();
         if (shift_menu ? ((main_menu->items)[mn_menu_sel]->type == SUBMENU) 
                        : FALSE)
           perform = TRUE;
        }
      if (perform)
        {
         perform = FALSE;
         item = (main_menu->items)[main_menu_cell];
         if (item->type == ITEM)
           item_rtn = (*(item->desc.handler))(main_menu_cell+1);
         else
           item_rtn = activate_pop_up_menu(item->desc.submenu,CLIPS_Y-1,
                                            main_pop_up_start(main_menu_cell));
         if (item_rtn == MENU_ITEM_CLIPS_INPUT)
           {
            cl_print("stdin",get_command_string());
            quit = TRUE;
           }
         else if (item_rtn == MENU_LEFT_ABORT)
           {
            mn_menu_sel = (mn_menu_sel+main_menu->item_cnt-1) % 
                          main_menu->item_cnt;
            shift_menu = TRUE;
           }
         else if (item_rtn == MENU_RIGHT_ABORT)
           {
            mn_menu_sel = (mn_menu_sel+1) % main_menu->item_cnt;
            shift_menu = TRUE;
           }
         else
           shift_menu = FALSE;
        }

      /* =================================================
         Poll the keyboard and handle any characters there
         ================================================= */
      else if ((ch[0] = wgetch(clips_window)) != -1)
        {
         if ((ch[0] >= LOW_PRN_ASCII) && (ch[0] <= HIGH_PRN_ASCII))
           {
            cl_print("stdin",(char *) ch);
            expand_command_string(ch[0]);
           }
         else if ((ch[0] != NEWLINE) ? (ch[0] == CRGRTN) : TRUE)
           {
            ch[0] = NEWLINE;
            cl_print("stdin",(char *) ch);
            expand_command_string(ch[0]);
            quit = TRUE;
           }
         else if (ch[0] == DELETE)
           {
            cmdstr = get_command_string();
            if ((cmdstr != NULL) ? (cmdstr[0] != EOS) : FALSE)
              {
               cl_print("stdin",(char *) ch);
               expand_command_string(ch[0]);
              }
            else
              beep();
           }
         else 
           {
            quit = process_special(key_map(ch[0]),&mn_menu_sel,&perform);
            if (quit)
              cl_print("stdin",get_command_string());
           }
        }

#if MOUSE
      /* ==================================================
         If the mouse moves, move the highlight bar with it
         ================================================== */
      else if (mouse_moved_posn(&y,&x))     /* Main menu highlight change */
        {
         mn_menu_sel = (x-AD_COLS-AD_X-1) / main_menu_cell_width; 
         if (mn_menu_sel < 0)
           mn_menu_sel = 0;
         else if (mn_menu_sel > (main_menu->item_cnt - 1))
           mn_menu_sel = main_menu->item_cnt-1;
         shift_menu = FALSE;
        }

      /* =================================================================
         If the mouse is left-clicked, activate the current main menu item
         ================================================================= */
      else if (mouse_clicked(&(ch[0]),&y,&x))     /* Main menu selection   */
        {
         if (ch[0] == LEFT_BUTTON)
           perform = TRUE;
         else
           beep();
        }
#endif
     }
  }

/******************************************************************************
 NAME        : win_query
 PURPOSE     : Determines if a logical name is recognized by the interface
 DESCRIPTION : Captures all the standard CLIPS logical names so that the
               screen interface can hadle all CLIPS I/O
 INPUTS      : 1) A string representing the logical name to be checked
 RETURNS     : TRUE (1) if the logical name was recognized, FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
int win_query(log_name)
  char *log_name;
  {
   return((log_lookup(log_name) != NULL) ? TRUE : FALSE);
  }

/******************************************************************************
 NAME        : win_print
 PURPOSE     : Sends CLIPS output to the windowed interface
 DESCRIPTION : Captures CLIPS output and prints in appropriate color on screen
 INPUTS      : 1) The logical name of the output destination
               2) The string to be printed
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void win_print(log_name,str)
  char *log_name,*str;
  {
   LOGNAME *lptr;
   register int i;

   if ((lptr = log_lookup(log_name)) == NULL)
     return;
   wattrset(clips_window,colors[lptr->color]|colors[MAIN_BACKGD]);
   if (lptr == stdin_log)
     {
      for (i = 0 ; str[i] != EOS ; i++)
        {
         if (! clips_addch(str[i]))
           {
            beep();
            break;
           }
        }
      if (i > 0)
        {
         output_lines = 0;
         wrefresh(clips_window);
        }
     }
   else
     {
      clips_addstr(str);
      wrefresh(clips_window);
      input_cnt = 0;
     }
  }

/******************************************************************************
 NAME        : win_getc
 PURPOSE     : Captures keyboard input and processes it
 DESCRIPTION : Captures standard printable chars and buffers them for CLIPS.
 INPUTS      : 1) The logical name from which input is desired.
 RETURNS     : The ascii value of the printable character obtained.
 NOTES       : This routine will buffer printable input until it sees the first
               carriage return and then return the characters to CLIPS one
               at a time (as per a standard terminal).  However, the 
               processing loop will aslo look at control sequences and
               mouse inputs.
 ******************************************************************************/
int win_getc(log_name)
  char *log_name;
  {
   LOGNAME *lptr;
   int ch[2],
       quit = FALSE;
   
   /* =========================================================
      Return any characters in the unget buffer-stack first
      The unget buffer is an array of char which grows from
        smaller indices to larger ones -- thus unget_buf_ptr
        indicates the number of characters in the unget buffer.
        unget_buf_ptr == 0           --> Unget buffer is empty.
        unget_buf_ptr == UNGET_BUF_SIZE --> Unget buffer is full.
      The next character to be pulled from the unget buffer
        is at position unget_buf_ptr-1 in the buffer.
      ========================================================= */
   if (unget_buf_ptr != 0)
     return(unget_buf[--unget_buf_ptr]);

   if ((lptr = log_lookup(log_name)) == NULL)
     return(EOS);
   wattrset(clips_window,colors[lptr->color]|colors[MAIN_BACKGD]);

#if MOUSE
   save_mouse();
#endif
   wrefresh(clips_window);
   ch[1] = EOS;
   while (! quit)
     {
      if ((ch[0] = wgetch(clips_window)) != -1)
        {
         /* =====================================================
            Fill buffer with printable characters including crlfs
              Always leave one spot empty for the send-sequence
            ===================================================== */
         if ((ch[0] >= LOW_PRN_ASCII) && (ch[0] <= HIGH_PRN_ASCII))
           {
            cl_print("stdin",(char *) ch);
            input_cnt++;
            quit = TRUE;
           }
         else if ((ch[0] != NEWLINE) ? (ch[0] == CRGRTN) : TRUE)
           {
            ch[0] = NEWLINE;
            cl_print("stdin",(char *) ch);
            input_cnt++;
            quit = TRUE;
           }
         /* ==============================================================
            Deletions : only allows deletes if the keyboard buffer has
            stuff in it and if the character to be deleted has not already
            scrolled off the screen.
            ============================================================== */
         else if (ch[0] == DELETE)
           {
            if ((input_cnt != 0) && ((clips_row != 0) || (clips_col != 0)))
              {
               cl_print("stdin",(char *) ch);
               input_cnt--;
               quit = TRUE;
              }
            else
              beep();
           }
          else if (ch[0]  == CTRL_C)
            {
             ch[0] = NEWLINE;
             set_execution_error(TRUE);
             quit = TRUE;
            }
         else
           beep();
        }
     }
#if MOUSE
   restore_mouse();
#endif
   return(ch[0]);
  }

/******************************************************************************
 NAME        : win_ungetc
 PURPOSE     : Replaces a character previously read by CLIPS for another read
 DESCRIPTION : Pushes characters onto a simple stack which win_getc always
               looks at first before going onto anything else
 INPUTS      : 1) The character to be ungotten
               2) The logical name associated with the character
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void win_ungetc(ch,log_name)
  char ch,*log_name;
  {
   if (unget_buf_ptr < UNGET_BUF_SIZE)
     unget_buf[unget_buf_ptr++] = ch;
  }

/******************************************************************************
 NAME        : init_interface
 PURPOSE     : Sets up the initial window environment for CLIPS
 DESCRIPTION : Allocates the windows, draws the screen setup, and initializes
               the CLIPS router for capturing all input and output
 INPUTS      : 1) Whether the command-line monochrome flag is set or not
 RETURNS     : Nothing useful.
 NOTES       : This function must be called before any I/O with CLIPS occurs.
               Assumes interface is not already initialized.
 ******************************************************************************/
void init_interface(int mono)
  {
   /* ================================================================
      Load CLIPS path, names of color/bind dat files, and save default
      configuration to temporary file
      ================================================================ */
   if (! setup_config(mono))
     {
      cl_print("werror","Could not setup configuration properly!\n");
      stop_interface(0);
      cl_exit(0);
     }

   /* =================================================================
      Reset DOS Critical Error Handler to special routine for interface
      ================================================================= */
   io_error_clips();

   /* ======================================================
      A hash table is kept for quick lookup of logical names
      ====================================================== */
   if (! init_log_name_table())
      {
       cl_print("werror","Could not initialize logical name hash table!\n");
       stop_interface(0);
       cl_exit(0);
      }

   /* ==============================================================
      Initialize the screen environment :
        1) Establish the type of terminal being used
        2) Initialize screen windows and internal buffers
        3) Setup the main menu and submenus
        4) Draw the main screen including : border, menu, and text
        5) Flush everything to the visual screen
        6) Set screen states such as no-echo, unbuffered I/O, etc.
      ============================================================= */
   if (! get_terminal_type())
     {
      cl_print("werror","Could not allocate screen windows!\n");
      stop_interface(0);
      cl_exit(0);
     }
   if (! allocate_windows())
     {
      cl_print("werror","Could not allocate screen windows!\n");
      stop_interface(0);
      cl_exit(0);
     }
   if (! allocate_menus())
     {
      cl_print("werror","Could not allocate screen menus!\n");
      stop_interface(0);
      cl_exit(0);
     }
   draw_border();
   draw_advertisement();
   draw_main_menu();

   get_environment();
#if MOUSE
   load_mouse();
#endif

   /* ========================================================================
      draw_main_menu() redraws what is currently in the main window imposing a
      new background on the colors that are already there.  allocate_windows()
      xpaint-ed the main-window already as part of the initialization to 
      make sure that all cells had the STDIN foreground attribute.  This
      insures that the cursor will always be the same color (that's the
      good part), but this assumes that user-input will always be requested
      from the logical name STDIN-a fairly reasonable assumption (that's
      the pseudo-bad-part).  Clearing and scrolling the main CLIPS window
      also use this attribute so that the cursor will not change color
      (see do_clear_window() and clips_advance_row()).
         The bottom line is that we do not need to call draw_main_window()
      until such time as the background color changes .

      draw_main_window();
      ======================================================================= */
   watch_status();   /* watch_status() also calls update_screen() */

   /* ==================================================================
      This statment lets CLIPS know which routines to call to reestablish
      the interface windows if it has to disturb the screen (as is the
      case when the editor is called).
      ================================================================== */
   (void) set_redraw_function((int (*)()) redisplay_wins);
   (void) set_pause_env_function((int (*)()) pause_interface);
   (void) set_cont_env_function((int (*)()) restart_interface);

    /* =============================================================
       Lets print_prompt() in COMMLINE.C know which routine to call
          to update the amount of memory available display.
       Also, the add_exec_function() call forces CLIPS to update
          the memory display after every rule fires.
       ============================================================= */
    set_mem_status_function(memory_status);

    /* =============================================================
       Lets set_watch() in INTRBRWS.C know which routine to call
        to update the watch rules/facts/activations screen messages
       ============================================================= */
    set_watch_display_function(watch_status);

    /* ===============================================================
       Lets trace_on() and trace_off() in INTRFILE.C know which 
       routine to call to update the dribble-status in the debug-menu
       =============================================================== */
    set_dribble_status_function(dribble_status);

    /* =============================================================
      This statement redefines the keyboard processing routine that
      CLIPS uses to get the command line.
      ============================================================= */
   (void) set_event_function((int (*)()) next_event);

   /* ===============================================
      Make CLIPS check for interruption between rules
      =============================================== */
      (void) add_exec_function("stop-rule-firing",(int (*)()) check_for_break);

   /* ==================================================================
      Reroute all CLIPS I/O through the screen handler functions
      ================================================================== */
    if (! add_router("clipswin",10,win_query,(int (*)()) win_print,win_getc,
                     (int (*)()) win_ungetc, (int (*)()) stop_interface))
      {
       cl_print("werror","Could not allocate interface router!\n");
       stop_interface(0);
       cl_exit(0);
      }
  }


/******************************************************************************
 NAME        : stop_interface
 PURPOSE     : Cleans up the terminal environment and quits CLIPS
 DESCRIPTION : Releases screen data structures, resets the terminal to a
               normal state, and quits CLIPS
 INPUTS      : Ignored.
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void stop_interface(num)
  int num;
  {
#if CLP_TEXTPRO
   kill_on_line_help();
#endif
   remove_exec_function("stop-rule-firing");
   if (rule_fire_memory)
     {
      remove_exec_function("rule-fire-memory");
      rule_fire_memory = FALSE;
     }
   use_more = TRUE;
   del_router("clipswin");
   deallocate_log_name_table();
   restore_environment();
   deallocate_menus();
   deallocate_windows();
   deallocate_pop_up_queue();
   reset_cwd();
   io_error_dos();
  }

/******************************************************************************
 NAME        : check_for_break
 PURPOSE     : Used as an add_exec() function checked between rule boundaries
               by CLIPS to see if the user wishes to halt rule execution
 DESCRIPTION : Calls set_execution_error() in ENGINE.C to set the halt flag
               for CLIPS rule execution
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void check_for_break()
  {
   int ch;

   if ((ch = wgetch(clips_window)) != -1)
     {
      if (ch == CTRL_C)
        set_execution_error(TRUE);
     }
  }

/******************************************************************************
 NAME        : redisplay_wins
 PURPOSE     : Flushes interface to screen
 DESCRIPTION : This routine is used to redraw the interface windows on the
               screen after somethings has been done to disturb them --
               such as pop-up menus or the editor.
                 The routine works by un-optimizing all windows and then
               flushing them to the virtual screen.  The final step is to
               flush the virtual screen to the physical screen so it looks
               like the whole screen comes up at once.
 INPUTS      : None
 RETURNS     : Nothing useful.
 NOTES       : Assumes all windows are currently allocated.
               If there are no pop-up menus on the screen, the cursor is
                 left in the main CLIPS window, otherwise it is hidden.
 ******************************************************************************/
void redisplay_wins()
  {
   WINBLK *qptr;

   touchwin(header_window);
   wnoutrefresh(header_window);
   touchwin(advertisement_window);
   wnoutrefresh(advertisement_window);
   touchwin(main_menu_window);
   wnoutrefresh(main_menu_window);
   touchwin(left_border_window);
   wnoutrefresh(left_border_window);
   touchwin(right_border_window);
   wnoutrefresh(right_border_window);
   wattrset(footer_window,colors[WTRACE]|colors[MAIN_BACKGD]);
   touchwin(footer_window);
   wnoutrefresh(footer_window);
   touchwin(clips_window);
   wnoutrefresh(clips_window);
   if ((qptr = pop_up_queue_top) != NULL)
     {
      while (qptr != NULL)
        {
         touchwin(qptr->win);
         if (qptr->py == -1)
           wnoutrefresh(qptr->win);
         else
           pnoutrefresh(qptr->win,qptr->py,qptr->px,qptr->ssy,qptr->ssx,
                        qptr->sey,qptr->sex);
         qptr = qptr->nxt;
        }
      doupdate();
      hide_cursor();
     }
   else
     doupdate();
  }

/******************************************************************************
 NAME        : memory_status
 PURPOSE     : Writes number of bytes still available to screen
 DESCRIPTION : Called by print_prompt() in COMMLINE.C (CLIPS File) --
                 see also the routine set_mem_status_function() in COMMLINE.C
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Calls mem_left() routine in ENV.C
 ******************************************************************************/
void memory_status()
  {
   if (footer_window != NULL)
     {
      wattrset(footer_window,colors[MEMORY_MSG]|colors[MAIN_BACKGD]);
      mvwprintw(footer_window,0,COLS-18,"%06lu Bytes Left",mem_left());
      wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
      update_screen();
     }
  }

/******************************************************************************
 NAME        : watch_status
 PURPOSE     : Writes watch rules/facts/activations status (on/off) to screen
 DESCRIPTION : Called by do_watch and do_watch_all_toggle (in this file) and 
               set_watch (in INTRBRWS.C) -- see also the routine 
               set_watch_display_function in same file.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void watch_status()
  {
   short r,c,f,a;

   r = (get_rules_watch() == ON);
   c = (get_compilations_watch() == ON);
   f = (get_facts_watch() == ON);
   a = (get_activations_watch() == ON);
   if (footer_window != NULL)
     {
      wattrset(footer_window,colors[WATCH_MSG]|colors[MAIN_BACKGD]);
      mvwprintw(footer_window,0,1,"Watch R/F/A/C : ");
      wprintw(footer_window,r ? "On/" : "Off/");
      wprintw(footer_window,f ? "On/" : "Off/");
      wprintw(footer_window,a ? "On/" : "Off/");
      wprintw(footer_window,c ? "On" : "Off");

      /* ========================
         Take care of any overlap
         ======================== */
      wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
      waddch(footer_window,DB_H_LIN);
      waddch(footer_window,DB_H_LIN);
      waddch(footer_window,DB_H_LIN);
      waddch(footer_window,DB_H_LIN);
      update_screen();
     }
   if (watch_menu != NULL)
      {
       change_menu_item(watch_menu,1,r ? "+    Rules    " : "-    Rules    ");
       change_menu_item(watch_menu,2,f ? "+    Facts    " : "-    Facts    ");
       change_menu_item(watch_menu,3,a ? "+ Activations " : "- Activations ");
       change_menu_item(watch_menu,4,c ? "+ Compilations" : "- Compilations");
      }
  }

/******************************************************************************
 NAME        : dribble_status
 PURPOSE     : Changes debug-menu to reflect current status fo dribble
 DESCRIPTION : This function is called  by the trace functions in INTRFILE.C
 INPUTS      : 1) The current status of the dribble
                  1- Dribble On     2 - Dribble Off
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void dribble_status(on)
  int on;
  {
   char *stat;

   stat = on ? "Turn Dribble Off" : "Turn Dribble On ";
   change_menu_item(debug_menu,3,stat);
  }

/******************************************************************************
 NAME        : insert_pop_up
 PURPOSE     : Puts a new window in the pop-up queue
 DESCRIPTION : The window is put in a queue which is used by redisplay_wins()
               After the main screen has been redrawn, this queue is 
               followed in order from top to bottom, and all windows found
               therein are flushed to the physical screen.
                 This makes it easy to "remove" pop-ups from the screen.
               Insertions/Deletions take place ONLY at the end of the queue.
 INPUTS      : 1) A code indicating whether a pad is being inserted or a
                  window is (POP_WIN (0) or POP_PAD (1))
               2) The address of the window/pad to be put in the queue
                  The window MUST have been previously allocated and drawn.
               3-4) Req'd for pads only : Start row/col for the pad
               5-6) Req'd for pads only : Start row/col for the screen
               7-8) Req'd for pads only : End row/col for the screen
 RETURNS     : TRUE (1) if everything was OK, FALSE (0) otherwise
 NOTES       : This routine does not check to see if a window is already
                 in the queue.
               Be very careful to maintain the equanimity between the inserts
               and deletes or unpredictable results might occur - it
               probably won't crash, but your screen will most likely look
               very strange.
 ******************************************************************************/
int insert_pop_up(int code,WINDOW *win,...)
  {
   WINBLK *qptr;
   va_list ap;

   if ((qptr = balloc(1,WINBLK)) == NULL)
     return(FALSE);
   qptr->win = win;
   if (code == POP_PAD)
     {
      va_start(ap,win);
      qptr->py = va_arg(ap,int);
      qptr->px = va_arg(ap,int);
      qptr->ssy = va_arg(ap,int);
      qptr->ssx = va_arg(ap,int);
      qptr->sey = va_arg(ap,int);
      qptr->sex = va_arg(ap,int);
      va_end(ap);
     }
   else
     qptr->py = -1;
   qptr->prv = NULL;
   qptr->nxt = NULL;

   if (pop_up_queue_top == NULL)
     {
      pop_up_queue_top = qptr;
      pop_up_queue_bottom = qptr;
     }
   else
     {
      pop_up_queue_bottom->nxt = qptr;
      qptr->prv = pop_up_queue_bottom;
      pop_up_queue_bottom = qptr;
     }
   return(TRUE);
  }

/******************************************************************************
 NAME        : delete_pop_up
 PURPOSE     : Removes a window from the bottom of the pop-up queue
 DESCRIPTION : See above
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void delete_pop_up()
  {
   WINBLK *qptr;

   if (pop_up_queue_bottom != NULL)
     {
      qptr = pop_up_queue_bottom;
      pop_up_queue_bottom = pop_up_queue_bottom->prv;
      if (pop_up_queue_bottom == NULL)
        pop_up_queue_top = NULL;
      else
        pop_up_queue_bottom->nxt = NULL;
      release(1,WINBLK,qptr);
     }
  }

/* ****************************************************************************
   ============================================================================
                       CLIPS MENU INTERFACE FUNCTIONS
   ============================================================================
   ****************************************************************************/

#if CLP_TEXTPRO

/******************************************************************************
 NAME        : do_interface_help
 PURPOSE     : Provides an on-line help facility for the interface
 DESCRIPTION : 
 INPUTS      : 1) The index in the main menu from which this routine was called
 RETURNS     : Returns MENU_ITEM_ABORT
 NOTES       : Calls on_line_help() in HELP.C
 ******************************************************************************/
int do_interface_help(index)
  int index;
  {
   WINDOW *win;

   on_line_help();
   return(MENU_ITEM_COMPLETION);
  }

#if CLP_HELP

/******************************************************************************
 NAME        : do_clips_help
 PURPOSE     : Issues the CLIPS (help) command
 DESCRIPTION : 
 INPUTS      : Not used
 RETURNS     : 1) MENU_ITEM_CLIPS_INPUT
 NOTES       : None
 ******************************************************************************/
int do_clips_help(index)
  int index;
  {
   set_command_string("(help)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_clips_help_path
 PURPOSE     : Issues the CLIPS (help-path) command
 DESCRIPTION : 
 INPUTS      : Not used
 RETURNS     : 1) MENU_ITEM_CLIPS_INPUT
 NOTES       : None
 ******************************************************************************/
int do_clips_help_path(index)
  int index;
  {
   char *file;
   
   file = pop_up_files("Enter Path for CLIPS Help File :",NULL,NULL,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(help-path ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif
#endif

#if ! BLOAD_ONLY

/******************************************************************************
 NAME        : do_load
 PURPOSE     : Performs a load rules/deffacts command from a pop-up menu
 DESCRIPTION : 
 INPUTS      : 1) The index in the main menu from which this routine was called
                  (Not Used)
 RETURNS     : 1) MENU_ITEM_ABORT if the dialog box is cancelled,
               2) MENU_ITEM_CLIPS_INPUT if a file is entered to be loaded.
 NOTES       : This routine and all routines that return MENU_ITEM_CLIPS_INPUT
               echo the equivalent CLIPS command to the screen and reset the
               internal CLIPS command buffer.  Routines which operate in this
               way make CLIPS think the command was typed in normally at
               the keyboard.  Anything that was already in the command buffer
               is lost.
                 Some of the options, such as Watch/Unwatch All toggle
               (bound to keystroke Alt-W) directly call internal CLIPS routines
               to do the work and in no way interrupt the user's command
               buffer.
                 In general, only the commands which themselves produce
               immediate output will be treated the former way.
 ******************************************************************************/
int do_load(index)
  int index;
  {
   char *file;
   
   file = pop_up_files("Enter Rules/Deffacts File to Load :",NULL,NULL,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(load ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_save
 PURPOSE     : Performs a save rules/deffacts from a pop-up menu
 DESCRIPTION : 
 INPUTS      : 1) The index in the main-menu from which this routine was called
                  (Not Used)
 RETURNS     : 
 NOTES       : See note for do_load().
 ******************************************************************************/
int do_save(index)
  int index;
  {
   char *file;
 
   file = pop_up_files("Enter Save File for Rules/Deffacts :",NULL,NULL,WRITE);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(save ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

#if SAVE_FACTS

/******************************************************************************
 NAME        : do_load_facts
 PURPOSE     : Performs a load-facts from a pop-up menu
 DESCRIPTION : 
 INPUTS      : See above.
 RETURNS     :    "
 NOTES       : Echoes command to screen.
 ******************************************************************************/
int do_load_facts(index)
  int index;
  {
   char *file;
   
   file = pop_up_files("Enter Facts File to Load :",NULL,NULL,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(load-facts ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_save_facts
 PURPOSE     : Performs a save-facts from a pop-up-menu
 DESCRIPTION : 
 INPUTS      : See above
 RETURNS     :    "
 NOTES       : Echoes command to screen.
 ******************************************************************************/
int do_save_facts(index)
  int index;
  {
   char *file;
 
   file = pop_up_files("Enter Save File for Facts :",NULL,NULL,WRITE);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(save-facts ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

#if BLOAD_ONLY || BLOAD || BLOAD_AND_BSAVE

/******************************************************************************
 NAME        : do_bload
 PURPOSE     : Performs the CLIPS command (bload <filename>)
 DESCRIPTION : 
 INPUTS      : See above
 RETURNS     : "
 NOTES       : 
 ******************************************************************************/
int do_bload(index)
  int index;
  {
   char *file;
   
   if (pop_up_warning(MID_JUST,SILENCE,1,
                      "\nBload clears current environment!\nOK to proceed?\n")
                      == NO)
     return(MENU_ITEM_ABORT);
   file = pop_up_files("Enter Binary File to Load :",NULL,NULL,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(bload ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

#if BLOAD_AND_BSAVE

/******************************************************************************
 NAME        : do_bsave
 PURPOSE     : Performs the CLIPS command (bsave <filename>)
 DESCRIPTION : 
 INPUTS      : See above
 RETURNS     : "
 NOTES       : 
 ******************************************************************************/
int do_bsave(index)
  int index;
  {
   char *file;
 
   file = pop_up_files("Enter Binary File for Rules/Deffacts :",NULL,NULL,WRITE);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(bsave ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

/******************************************************************************
 NAME        : do_reset
 PURPOSE     : CLIPS (reset)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command to screen.
 ******************************************************************************/
int do_reset(index)
  int index;
  {
   set_command_string("(reset)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_run
 PURPOSE     : CLIPS (run)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command to screen.
 ******************************************************************************/
int do_run(index)
  int index;
  {
   set_command_string("(run)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_step
 PURPOSE     : CLIPS (run <limit>)
 DESCRIPTION : This function keeps access to a static step-increment variable
               which is the limit-number for the run.
 INPUTS      : 1) The index in the execute-menu from which this routine
                  was called : 3 - Step (Perform the (run <limit>) command)
                               4 - Set Step... (Set the <limit> parameter>)
               By default the run-limit is set to 1.
 RETURNS     : 1) MENU_ITEM_ABORT if option cancelled or there is an error
                  in a new step-increment
               2) MENU_ITEM_COMPLETION if a new increment is set successfully.
               3) MENU_ITEM_CLIPS_INPUT if the step-command is selected.
 NOTES       : Checks for invalid step-increments.
               In the case of the 3rd return-option, the command is echoed.
 ******************************************************************************/
int do_step(index)
  int index;
  {
   static int step_increment = 1;
   char num_limit[10];
   char prompt[30];
   int tmp;

   if (index == 4)
     {
      sprintf(prompt,"Step-Increment(%d) : ",step_increment);
      if (pop_up_text(prompt,num_limit,9) == NULL)
        return(MENU_ITEM_ABORT);
      else if ((tmp = atoi(num_limit)) <= 0)
        {
         beep();
         return(MENU_ITEM_ABORT);
        }
      else
        step_increment = tmp;
      return(MENU_ITEM_COMPLETION);
     }
   set_command_string("(run ");
   sprintf(num_limit,"%d",step_increment);
   append_command_string(num_limit);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_batch
 PURPOSE     : CLIPS (batch <filename>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : 1) MENU_ITEM_ABORT if menu is cancelled,
               2) MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_batch(index)
  int index;
  {
   char *file;
   
   file = pop_up_files("Enter File for Batch :",NULL,NULL,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(batch ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_watch
 PURPOSE     : CLIPS (watch/unwatch ...)
 DESCRIPTION : This routine determines which watch to set from the indices
               of the main debug menu and the watch sub-menu.
 INPUTS      : Ignored.
 RETURNS     : 1) MENU_ITEM_ABORT if menu is cancelled,
               2) MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_watch(index)
  int index;
  {
   switch(index)
     {
      case 1  : if (get_rules_watch() == ON)
                  set_command_string("(unwatch rules)\n"); 
                else
                  set_command_string("(watch rules)\n"); 
                break;
      case 2  : if (get_facts_watch() == ON)
                  set_command_string("(unwatch facts)\n"); 
                else
                  set_command_string("(watch facts)\n"); 
                break;
      case 3  : if (get_activations_watch() == ON)
                  set_command_string("(unwatch activations)\n"); 
                else
                  set_command_string("(watch activations)\n"); 
                break;
      case 4  : if (get_compilations_watch() == ON)
                  set_command_string("(unwatch compilations)\n"); 
                else
                  set_command_string("(watch compilations)\n"); 
                break;
      case 5  : set_command_string("(watch all)\n"); 
                break;
      case 6  : set_command_string("(unwatch all)\n"); 
                break;
     }
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_watch_all_toggle
 PURPOSE     : Allows user to toggle watch/unwatch all w/o output to screen
 DESCRIPTION : Toggles all 3 CLIPS watch flags
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void do_watch_all_toggle()
  {
   int flag;
   if ((get_facts_watch() == OFF) &&
       (get_rules_watch() == OFF) &&
       (get_compilations_watch() == OFF) &&
       (get_activations_watch() == OFF))
     flag = ON;
   else
     flag = OFF;
   set_facts_watch(flag);
   set_rules_watch(flag);
   set_compilations_watch(flag);
   set_activations_watch(flag);
   watch_status();
  }

/******************************************************************************
 NAME        : do_matches
 PURPOSE     : CLIPS (matches <rule-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : 1) MENU_ITEM_ABORT if pop-up-text window is cancelled,
               2) MENU_ITEM_CLIPS_INPUT otherwise
 NOTES       : Echoes command.
 ******************************************************************************/
int do_matches(index)
  {
   char *rule_name;

   rule_name = rules_menu("Select rule for fact matches :",NULL);
   redisplay_wins();
   if (rule_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(matches ");
   append_command_string(rule_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_dribble
 PURPOSE     : CLIPS (dribble-on <file) and (dribble-off)
 DESCRIPTION : 
 INPUTS      : 1) The index in the debug menu from which this routine was
                  called (Used to determine whether to call dribble-on/off)
 RETURNS     : 1) MENU_ITEM_CLIPS_INPUT if the command was built,
               2) MENU_ITEM_ABORT if the file request for dribble-on was
                  cancelled.
 NOTES       : Uses dribble_active() in INTRFILE.C
 ******************************************************************************/
int do_dribble(index)
  int index;
  {
   char *file;

   if (dribble_active())
     {
      set_command_string("(dribble-off)\n");
      return(MENU_ITEM_CLIPS_INPUT);
     }
   file = pop_up_files("Enter Dribble-File Name :",NULL,NULL,WRITE);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(dribble-on ");
   append_file_clips_cmd_str(file);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#if BREAKPOINTS

/******************************************************************************
 NAME        : do_break
 PURPOSE     : CLIPS (set-break <name>),(remove-break <name), & (show breaks)
 DESCRIPTION : 
 INPUTS      : 1) The index in the debug menu from which this routine was
                  called (Used to determine whether to call set-break,
                  remove-break, or show-breaks).
 RETURNS     : 1) MENU_ITEM_CLIPS_INPUT if the command was built,
               2) MENU_ITEM_ABORT if the file request for breaks was cancelled.
 NOTES       : None
 ******************************************************************************/
int do_break(index)
  int index;
  {
   char *rule_name;

   if (index == 6)
     {
      set_command_string("(show-breaks)\n");
      return(MENU_ITEM_CLIPS_INPUT);
     }
   else if (index == 5)
     {
      rule_name = breaks_menu("Remove break from which rule?");
      redisplay_wins();
      if (rule_name == NULL)
        return(MENU_ITEM_ABORT);
      set_command_string("(remove-break ");
     }
   else
     {
      rule_name = nobreaks_menu("Set break for which rule?");
      redisplay_wins();
      if (rule_name == NULL)
        return(MENU_ITEM_ABORT);
      set_command_string("(set-break ");
     }
   append_command_string(rule_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

/******************************************************************************
 NAME        : do_assert
 PURPOSE     : CLIPS (assert (<fact>))
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : 1) MENU_ITEM_ABORT if pop-up-text window cancelled,
               2) MENU_ITEM_CLIPS_INPUT otherwise.
 NOTES       : Echoes command.
 ******************************************************************************/
int do_assert(index)
  int index;
  {
   char fact[61];

   if (pop_up_text("Enter fact to assert : ",fact,60) == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(assert (");
   append_command_string(fact);
   append_command_string("))\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_retract
 PURPOSE     : CLIPS (retract <fact-num>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : See above.
 NOTES       : Echoes command.
 ******************************************************************************/
int do_retract(index)
  int index;
  {
   long int fact_num;
   char fnumstr[20];

   fact_num = facts_menu("Select fact to retract : ","Retract fact?\n\n");
   redisplay_wins();
   if (fact_num == (long int) -1)
     return(MENU_ITEM_ABORT);
   set_command_string("(retract ");
   sprintf(fnumstr,"%ld",fact_num);
   append_command_string(fnumstr);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_excise
 PURPOSE     : CLIPS (excise <rule-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_excise(index)
  int index;
  {
   char *rule_name;

   rule_name = rules_menu("Select rule for removal :","Excise rule?\n\n");
   redisplay_wins();
   if (rule_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(excise ");
   append_command_string(rule_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_fire
 PURPOSE     : Moves specified activation to top of agenda and executes
               (run 1)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_fire(index)
  int index;
  {
   int agendai;

   agendai = agenda_menu("Select rule to fire :",NULL);
   if (agendai == -1)
     return(MENU_ITEM_ABORT);
   move_activation_to_top(agendai);
   set_command_string("(run 1)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_activate_remove
 PURPOSE     : Removes specified rule activation from agenda
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_COMPLETION
 NOTES       : Echoes command.
 ******************************************************************************/
int do_activate_remove(index)
  int index;
  {
   int agendai;

   agendai = agenda_menu("Select activation to remove :",
                         "Remove activation?\n\n");
   if (agendai == -1)
     return(MENU_ITEM_ABORT);
   delete_activation(agendai);
   return(MENU_ITEM_COMPLETION);   
  }

#if DEFFACTS_CONSTRUCT

/******************************************************************************
 NAME        : do_deffact_remove
 PURPOSE     : CLIPS (undeffacts <deffact-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_deffact_remove(index)
  int index;
  {
   char *deffact_name;

   deffact_name = deffacts_menu("Select deffacts for removal :",
                                "Remove deffact?\n\n");
   redisplay_wins();
   if (deffact_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(undeffacts ");
   append_command_string(deffact_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

#if DEFTEMPLATES

/******************************************************************************
 NAME        : do_deftemplate_remove
 PURPOSE     : CLIPS (undeftemplate <deftemplate-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_deftemplate_remove(index)
  int index;
  {
   char *deftemplate_name;

   deftemplate_name = deftemplates_menu("Select deftemplate for removal :",
                                       "Remove deftemplate?\n\n");
   redisplay_wins();
   if (deftemplate_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(undeftemplate ");
   append_command_string(deftemplate_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

/******************************************************************************
 NAME        : do_clear
 PURPOSE     : CLIPS (clear)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_clear(index)
  int index;
  {
   if (pop_up_warning(MID_JUST,SILENCE,1,"\nOK to Clear All Rules and Facts?\n")
                      == NO)
     return(MENU_ITEM_ABORT);
   set_command_string("(clear)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_exit
 PURPOSE     : CLIPS (exit)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_COMPLETION
 NOTES       : None
 ******************************************************************************/
int do_exit(index)
  int index;
  {
   int query;

   query = pop_up_warning(MID_JUST,SILENCE,1,
                          "\nWARNING!\n\nDo you really want to\nExit CLIPS?");
   if (query == NO)
     return(MENU_ITEM_ABORT);
   cl_exit(1);
   return(MENU_ITEM_COMPLETION);
  }

/******************************************************************************
 NAME        : do_facts
 PURPOSE     : CLIPS (facts)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_facts(index)
  int index;
  {
   set_command_string("(facts)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_rules
 PURPOSE     : CLIPS (rules)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_rules(index)
  int index;
  {
   set_command_string("(rules)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_agenda
 PURPOSE     : CLIPS (agenda)
 DESCRIPTION : 
 INPUTS      : Ignored
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_agenda(index)
  int index;
  {
   set_command_string("(agenda)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_pprule
 PURPOSE     : CLIPS (pprule <rule-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_pprule(index)
  int index;
  {
   char *rule_name;

   rule_name = rules_menu("Select rule for display :",NULL);
   redisplay_wins();
   if (rule_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(pprule ");
   append_command_string(rule_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#if DEFFACTS_CONSTRUCT

/******************************************************************************
 NAME        : do_deffacts
 PURPOSE     : CLIPS (list-deffacts)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_deffacts(index)
  int index;
  {
   set_command_string("(list-deffacts)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_ppdeffact
 PURPOSE     : CLIPS (ppdeffact <deffact-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_ppdeffact(index)
  int index;
  {
   char *deffact_name;

   deffact_name = deffacts_menu("Select deffacts for display :",NULL);
   redisplay_wins();
   if (deffact_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(ppdeffact ");
   append_command_string(deffact_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

#if DEFTEMPLATES

/******************************************************************************
 NAME        : do_deftemplates
 PURPOSE     : CLIPS (list-deftemplates)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_deftemplates(index)
  int index;
  {
   set_command_string("(list-deftemplates)\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

/******************************************************************************
 NAME        : do_ppdeftemplate
 PURPOSE     : CLIPS (ppdeffact <deftemplate-name>)
 DESCRIPTION : 
 INPUTS      : Ignored.
 RETURNS     : MENU_ITEM_ABORT or MENU_ITEM_CLIPS_INPUT
 NOTES       : Echoes command.
 ******************************************************************************/
int do_ppdeftemplate(index)
  int index;
  {
   char *deftemplate_name;

   deftemplate_name = deftemplates_menu("Select deftemplate for display :",NULL);
   redisplay_wins();
   if (deftemplate_name == NULL)
     return(MENU_ITEM_ABORT);
   set_command_string("(ppdeftemplate ");
   append_command_string(deftemplate_name);
   append_command_string(")\n");
   return(MENU_ITEM_CLIPS_INPUT);
  }

#endif

/******************************************************************************
 NAME        : do_clear_window
 PURPOSE     : Erases the current main CLIPS-window.
 DESCRIPTION : This function has no effect on the current state of CLIPS.
 INPUTS      : 1) The index of the menu the function was called from
                  (This argument is not used.)
 RETURNS     : MENU_ITEM_COMPLETION (1)
 NOTES       : 
 ******************************************************************************/
int do_clear_window(index)
  int index;
  {
   xpaint(clips_window,colors[STDIN]|colors[MAIN_BACKGD]);
   clips_row = 0;
   clips_col = 0;
   wmove(clips_window,clips_row,clips_col);
   return(MENU_ITEM_COMPLETION);
  }


/******************************************************************************
 NAME        : clips_clear_window
 PURPOSE     : Allows the main CLIPS window to be cleared by a rule
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Used in a define_function() call in usrfuncs() above.
 ******************************************************************************/
int clips_clear_window()
  {
   do_clear_window(OPTS_INDEX);
   return(1);
  }

/******************************************************************************
 NAME        : do_rule_memory_update
 PURPOSE     : Instructs to update memory-count between rule-firings or not
 DESCRIPTION : 
 INPUTS      : 1) Index of the menu from which this routine was called
 RETURNS     : MENU_ITEM_COMPLETION
 NOTES       : When this optin is set, it significantly slows down execution
               due to the cumbersome method being used to calculate the CLIPS
               free pool size.
 ******************************************************************************/
int do_rule_memory_update(index)
  int index;
  {
   if (rule_fire_memory)
     {
      rule_fire_memory = FALSE;
      change_menu_item(options_menu,index,"Turn Rule-Fire Memory-Update On ");
      remove_exec_function("rule-fire-memory");
     }
   else
     {
      rule_fire_memory = TRUE;
      change_menu_item(options_menu,index,"Turn Rule-Fire Memory-Update Off");
      add_exec_function("rule-fire-memory",(int (*)()) memory_status);
     }   
   return(MENU_ITEM_COMPLETION);
  }

/******************************************************************************
 NAME        : do_spawn
 PURPOSE     : Executes a single DOS command or starts a new 
               COMMAND.COM shell under CLIPS
 DESCRIPTION : 
 INPUTS      : 1) The index in the options-menu from which this routine
                  was called - determines whether to execute a single command
                  or spawn a whole new shell.
 RETURNS     : MENU_ITEM_COMPLETION or MENU_ITEM_ABORT
 NOTES       : None
 ******************************************************************************/
int do_spawn(index)
  int index;
  {
   char cmdline[78];

   if (index == 2)
     {
      if (pop_up_text("Enter DOS Command : ",cmdline,77) == NULL)
         return(MENU_ITEM_ABORT);
     }
   else
     cmdline[0] = EOS;
   if (do_system(cmdline))
     return(MENU_ITEM_COMPLETION);
   beep();
   return(MENU_ITEM_ABORT);
  }

/******************************************************************************
 NAME        : do_system
 PURPOSE     : Executes a single DOS command or starts a whole new shell
 DESCRIPTION : Sets terminal environment to normal temporarily so that the
               standard interface will work properly.
 INPUTS      : 1) The command to execute
                  If the command is empty (cmd[0] = '\0'), then a new
                  COMMAND.COM shell is spawned.
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise.
 NOTES       : Uses dos_command() and spawn_shell() routines in ENV.C
               
               For the CLIPS Interface the command line function my_system()
               in the standard CLIPS file SYSDEP.C calls this function.
 ******************************************************************************/
int do_system(cmd)
  char *cmd;
  {
   int success;

   pause_interface();
   if (cmd[0] != EOS)
     success = dos_command(cmd);
   else
     success = spawn_shell();
   restart_interface(FALSE);
   return(success);
  }

#if CLP_EDIT
/******************************************************************************
 NAME        : do_edit
 PURPOSE     : Forms a command for the MicroEMACS editor
 DESCRIPTION : 
 INPUTS      : 1) The index from the menu the function was called from
 RETURNS     : 1) MENU_ITEM_CLIPS_INPUT 
               2) MENU_ITEM_ABORT (0) if the file-menu is cancelled
 NOTES       : None
 ******************************************************************************/
int do_edit(index)
  int index;
  {
   char *file;

   if (index == 1)
     set_command_string("(edit)\n");
   else
     {
      file = pop_up_files("Enter file to edit :",NULL,NULL,READ|WRITE);
      redisplay_wins();
      if (file == NULL)
        return(MENU_ITEM_ABORT);
      set_command_string("(edit ");
      append_file_clips_cmd_str(file);
      append_command_string(")\n");
     }
   return(MENU_ITEM_CLIPS_INPUT);
  }
#endif

/******************************************************************************
 NAME        : do_toggle_more
 PURPOSE     : Toggles the --More-- Facility to On/Off
 DESCRIPTION : 
 INPUTS      : 1) The index in the menu from which this routine was called
 RETURNS     : 1) MENU_ITEM_COMPLETION if all went well
               2) MENU_ITEM_ABORT otherwise
 NOTES       : Uses change_menu_item() in POPUP.C
 ******************************************************************************/
int do_toggle_more(index)
  int index;
  {
   char *msg;

   msg = (use_more) ? "Turn --More-- On " : "Turn --More-- Off";
   if (! change_menu_item(options_menu,index,msg))
     return(MENU_ITEM_ABORT);
   use_more = (use_more == TRUE) ? FALSE : TRUE;
   return(MENU_ITEM_COMPLETION);
  }

/******************************************************************************
 NAME        : do_load_env
 PURPOSE     : Loads new colors/key-bindings from binary data file
 DESCRIPTION : 
 INPUTS      : 1) The index in the environment menu from which this routine
                  was called :
                  1 - Load Colors  2 - Load Key-Bindings 3 - Load Both
 RETURNS     : 1) MENU_ITEM_COMPLETION (1)
               2) MENU_ITEM_ABORT (0) if the text-box is cancelled.
 NOTES       : Calls binary data file routines in ENV.C
 ******************************************************************************/
int do_load_env(index)
  int index;
  {
   char *file;
   int (*valid)(char *);

   if (index == 1)
     valid = valid_color_file;
   else if (index == 2)
     valid = valid_bind_file;
   else
     valid = valid_color_bind_file;
   file = pop_up_files("Enter load file : ",NULL,valid,READ);
   redisplay_wins();
   if (file == NULL)
     return(MENU_ITEM_ABORT);
   if (index != 2) 
     {
      if (load_colors(file) == OK_DATA_FILE)
        reset_colors();
      else
        beep();
     }
   if (index != 1)
     {
      if (load_binds(file) != OK_DATA_FILE)
        beep();
     }
   return(MENU_ITEM_COMPLETION);
  }

/******************************************************************************
 NAME        : pause_interface
 PURPOSE     : Resets the terminal to a normal state
 DESCRIPTION : Sets echo-on, newline-mapping, cooked state and no-ega-mode,
               and saves the mouse.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Used in conjunction with system spawns and the editor.
 ******************************************************************************/
void pause_interface()
  {
#if MOUSE
   save_mouse();
#endif
   restore_environment();
   if (ega_mode)
     ega_close();
   else
     {
      mvcur(0,0,LINES-1,0);
      printf("\n");
     }
   io_error_dos();
  }

/******************************************************************************
 NAME        : restart_interface
 PURPOSE     : Resets the terminal to a valid state for the interface
 DESCRIPTION : No local-echo, no newline mapping, uncooked state, ega-mode
               (if it was set before), and restores the mouse.
 INPUTS      : 1) A code indicating whether to prompt the user to return or not
                  FALSE (0) - No    TRUE (1) - Yes
 RETURNS     : Nothing useful.
 NOTES       : Used in conjunction with system spawns and the editor.
 ******************************************************************************/
void restart_interface(query)
  int query;
  {
   if (query)
     {
      printf("\nPress any key to return to CLIPS...");
      while (wgetch(clips_window) == -1) ;
     }
   if (ega_mode)
     ega_open();
   get_environment();
#if MOUSE
   restore_mouse();
#endif
   io_error_clips();
  }

/* =======================================================================
   ***********************************************************************
                         INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : get_terminal_type
 PURPOSE     : Determines the type of device the interface is running on
 DESCRIPTION : Curses makes use of a DOS environment variable called TERM.
               If the variable is not defined, the terminal type ibm-pc
               is assumed.  If the variable is not one of the valid types,
               the program is aborted.
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went OK, FALSE (0) otherwise.
 NOTES       : The valid terminal types are : ibm-pc, ega-pc, bios-pc,
                                              ansi-pc, and c_ansi-pc

               To set the terminal type would require the following at
                 the DOS command prompt :

               C:\> set TERM=ibm-pc

               (Note the lack of imbedded blanks in the definition)
 ******************************************************************************/
static int get_terminal_type()
  {
   char *tty;

   tty = getenv("TERM");
   if (tty != NULL)
     {
      strncpy(termenv,"TERM=",TERMENVSIZE-1);
      if (strcmp(tty,"ibm-pc") == 0)
        strncat(termenv,tty,TERMENVSIZE-strlen(termenv)-1);
      else if (strcmp(tty,"ega-pc") == 0)
        {
         strncat(termenv,"ibm-pc",TERMENVSIZE-strlen(termenv)-1);
         if (ega_exist())
           ega_mode = TRUE;
         else
           cl_print("werror","EGA not available on this machine!\n");
        }
      else if (strcmp(tty,"bios-pc") == 0)
        strncat(termenv,tty,TERMENVSIZE-strlen(termenv)-1);
      else if (strcmp(tty,"ansi-pc") == 0)
        strncat(termenv,tty,TERMENVSIZE-strlen(termenv)-1);
      else if (strcmp(tty,"c_ansi-pc") == 0)
        strncat(termenv,tty,TERMENVSIZE-strlen(termenv)-1);
      else
        { 
         cl_print("werror",tty);
         cl_print("werror"," is an invalid terminal type!\n");
         return(FALSE);
        }
     }
   else
     {
      strncpy(termenv,"TERM=ibm-pc",TERMENVSIZE-1);
      putenv(termenv);
     }
   return(TRUE);
  }

/******************************************************************************
 NAME        : get_environment
 PURPOSE     : Sets up the environment for the screen interface
 DESCRIPTION : CLIPS interface needs terminal I/O in unprocessed format,
               i.e. Uncooked state (unbuffered I/O), no newline mapping,
               and no local echoing of input characters.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void get_environment()
  {
   raw();
   nonl();
   noecho();
  }

#if MOUSE
/******************************************************************************
 NAME        : load_mouse
 PURPOSE     : Initializes mouse and defines its region to be in main-menu
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Prints a message if the mouse-driver is not loaded.
 ******************************************************************************/
static void load_mouse()
   {
   init_mouse();
   if (! MOUSE_LOADED)
     win_print("werror","Mouse is not installed.\n");
   mouse_region(MN_MENU_Y,AD_COLS+AD_X+1,MN_MENU_Y,HDR_COLS-2);
   set_mouse(MN_MENU_Y,AD_COLS+AD_X);
   clear_mouse();
  }
#endif

/******************************************************************************
 NAME        : allocate_windows
 PURPOSE     : Gets space for border, advertisement and main menu
 DESCRIPTION : To conserve space, separate segmented windows are used to
               cover the portion of the screen which is not covered by
               the main I/O window, i.e the border, advertisement, and
               main menu.
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went well, FALSE (0) otherwise
 NOTES       : Should be called before interfacing with any Curses stuff
               Assumes windows are not already allocated.
 ******************************************************************************/
static int allocate_windows()
  {
   if (initscr() == NULL)
     return(FALSE);
   delwin(stdscr);     /* Don't need stdscr, so don't waste the space */
   stdscr = NULL;
   if ((header_window = newwin(HDR_LNS,HDR_COLS,HDR_Y,HDR_X)) == NULL)
     return(FALSE);
   scrollok(header_window,FALSE);
   if ((advertisement_window = subwin(header_window,AD_LNS,AD_COLS,AD_Y,AD_X))
        == NULL)
     return(FALSE);
   scrollok(advertisement_window,FALSE);
   if ((main_menu_window = subwin(header_window,MN_MENU_LNS,MN_MENU_COLS,
                                           MN_MENU_Y,MN_MENU_X)) == NULL)
     return(FALSE);
   scrollok(main_menu_window,FALSE);
   if ((left_border_window = newwin(LINES-HDR_LNS-FTR_LNS,LBORD_COLS,
                               LBORD_Y,LBORD_X)) == NULL)
     return(FALSE);
   scrollok(left_border_window,FALSE);
   if ((right_border_window = newwin(LINES-HDR_LNS-FTR_LNS,RBORD_COLS,
                                RBORD_Y,RBORD_X)) == NULL)
     return(FALSE);
   scrollok(right_border_window,FALSE);
   if ((footer_window = newwin(FTR_LNS,FTR_COLS,LINES-1,FTR_X)) == NULL)
     return(FALSE);
   scrollok(footer_window,FALSE);
   clips_lns = LINES-HDR_LNS-FTR_LNS;
   if ((clips_window = newwin(clips_lns,CLIPS_COLS,CLIPS_Y,CLIPS_X)) == NULL)
     return(FALSE);
   scrollok(clips_window,FALSE);
   keypad(clips_window,TRUE);
   nodelay(clips_window,TRUE);
   xpaint(clips_window,colors[STDIN]|colors[MAIN_BACKGD]);
   return(TRUE);
  }

/******************************************************************************
 NAME        : allocate_menus
 PURPOSE     : Initializes main menu structures and routine handlers
 DESCRIPTION : Refer to the MENU data type in the file MENU.H
               and the calling format description for new_menu() in the
               file MENU.C
 INPUTS      : None
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise
 NOTES       : The handler routine for each menu item must take one intger
                 command line argument which indicates the index of the menu
                 item for which the handler routine corresponds.  Also, it must
                 return an integer status code indicating whether the routine
                 was completed normally or not :
                    MENU_ITEM_COMPLETION (1) or MENU_ITEM_ABORT (0)
               Assumes internal menus are not already allocated.
 ******************************************************************************/
static int allocate_menus()
  {
   int item_cnt = 2;

#if SAVE_FACTS
   item_cnt += 2;
#endif
#if BLOAD_ONLY
   item_cnt++;
#elif BLOAD
   item_cnt += 3;
#elif BLOAD_AND_BSAVE
   item_cnt += 4;
#endif
   file_menu = new_menu(VISUAL_POP_UP_MENU_YES,item_cnt,
#if ! BLOAD_ONLY
                        "Load Rules/Deffacts",ITEM,do_load,
                        "Save Rules/Deffacts",ITEM,do_save,
#endif
#if SAVE_FACTS
                        "Load Facts",ITEM,do_load_facts,
                        "Save Facts",ITEM,do_save_facts,
#endif
#if BLOAD_ONLY || BLOAD || BLOAD_AND_BSAVE
                        "Binary Load",ITEM,do_bload,
#endif
#if BLOAD_AND_BSAVE
                        "Binary Save",ITEM,do_bsave,
#endif
                        "Clear Rules/Facts",ITEM,do_clear,
                        "Exit CLIPS",ITEM,do_exit
                        );
   if (file_menu == NULL)
     return(FALSE);
   execute_menu = new_menu(VISUAL_POP_UP_MENU_YES,5,
                           "Reset",ITEM,do_reset,
                           "Run",ITEM,do_run,
                           "Step",ITEM,do_step,
                           "Set Step...",ITEM,do_step,
                           "Batch...",ITEM,do_batch
                          );
   if (execute_menu == NULL)
     return(FALSE);
   watch_menu = new_menu(VISUAL_POP_UP_MENU_YES,6,
                         "-    Facts    ",ITEM,do_watch,
                         "-    Rules    ",ITEM,do_watch,
                         "- Activations ",ITEM,do_watch,
                         "+ Compilations",ITEM,do_watch,
                         "All",ITEM,do_watch,
                         "None",ITEM,do_watch
                        );
   if (watch_menu == NULL)
     return(FALSE);
   debug_menu = new_menu(VISUAL_POP_UP_MENU_YES,
#if BREAKPOINTS
                         6,
#else
                         3,
#endif
                         "Watch/Unwatch",SUBMENU,watch_menu,
                         "Matches",ITEM,do_matches,
                         "Turn Dribble On ",ITEM,do_dribble
#if BREAKPOINTS
                         ,"Set Break",ITEM,do_break,
                         "Remove Break",ITEM,do_break,
                         "Show Breaks",ITEM,do_break
#endif
                         );
   if (debug_menu == NULL)
     return(FALSE);
   item_cnt = 5;
#if DEFFACTS_CONSTRUCT
   item_cnt++;
#endif
#if DEFTEMPLATES
   item_cnt++;
#endif
   action_menu = new_menu(VISUAL_POP_UP_MENU_YES,item_cnt,
                          "Assert a Fact",ITEM,do_assert,
                          "Retract a Fact",ITEM,do_retract,
                          "Excise a Rule",ITEM,do_excise,
                          "Fire Rule Activation",ITEM,do_fire,
                          "Remove Rule Activation",ITEM,do_activate_remove
#if DEFFACTS_CONSTRUCT
                          ,"Remove a Deffact",ITEM,do_deffact_remove
#endif
#if DEFTEMPLATES
                          ,"Remove a Deftemplate",ITEM,do_deftemplate_remove
#endif
                          );
   if (action_menu == NULL)
     return(FALSE);
   item_cnt = 4;
#if DEFFACTS_CONSTRUCT
   item_cnt += 2;
#endif
#if DEFTEMPLATES
   item_cnt += 2;
#endif
   examine_menu = new_menu(VISUAL_POP_UP_MENU_YES,item_cnt,
                           "Display Facts",ITEM,do_facts,
                           "List Rules",ITEM,do_rules,
                           "Display a Rule",ITEM,do_pprule,
                           "Display Agenda",ITEM,do_agenda
#if DEFFACTS_CONSTRUCT
                           ,"List Deffacts",ITEM,do_deffacts,
                           "Display a Deffact",ITEM,do_ppdeffact
#endif
#if DEFTEMPLATES
                           ,"List Deftemplates",ITEM,do_deftemplates,
                           "Display a Deftemplate",ITEM,do_ppdeftemplate
#endif
                           );
   if (examine_menu == NULL)
     return(FALSE);
   env_menu = new_menu(VISUAL_POP_UP_MENU_YES,3,
                       "Load New Colors",ITEM,do_load_env,
                       "Load New Key-Bindings",ITEM,do_load_env,
                       "Load Both",ITEM,do_load_env
                      );
   if (env_menu == NULL)
     return(FALSE);
#if CLP_EDIT
   edit_menu = new_menu(VISUAL_POP_UP_MENU_YES,2,
                       "Start/Resume Editor",ITEM,do_edit,
                       "Edit New/Old File",ITEM,do_edit
                      );
   if (edit_menu == NULL)
     return(FALSE);
#endif
#if CLP_HELP && CLP_TEXTPRO
   help_menu = new_menu(VISUAL_POP_UP_MENU_YES,3,
                        "PC-Interface Help",ITEM,do_interface_help,
                        "CLIPS Help",ITEM,do_clips_help,
                        "Set CLIPS Help Path",ITEM,do_clips_help_path);
   if (help_menu == NULL)
     return(FALSE);
#endif
   options_menu = new_menu(VISUAL_POP_UP_MENU_YES,
#if CLP_EDIT
                           7,
#else
                           6,
#endif
                           "Clear Window",ITEM,do_clear_window,
                           "DOS Command",ITEM,do_spawn,
                           "DOS Shell",ITEM,do_spawn,
                           "Change Colors/Key-Bindings",SUBMENU,env_menu,
                           "Turn Rule-Fire Memory-Update On ",ITEM,
                                                    do_rule_memory_update,
                           "Turn --More-- Off",ITEM,do_toggle_more
#if CLP_EDIT
                           ,"Edit",SUBMENU,edit_menu
#endif
                          );
   if (options_menu == NULL)
     return(FALSE);
   /* =================================================================
      The main menu physical representation is always on the screen and
      hanlde by special routines in this file - only the internals need
      to be initialized.
      ================================================================= */
   main_menu = new_menu(VISUAL_POP_UP_MENU_NO,
#if CLP_TEXTPRO
                        7,
#else
                        6,
#endif
                        "File",SUBMENU,file_menu,
                        "Execute",SUBMENU,execute_menu,
                        "Debug",SUBMENU,debug_menu,
                        "Action",SUBMENU,action_menu,
                        "eXamine",SUBMENU,examine_menu,
#if CLP_TEXTPRO
#if CLP_HELP
                        "Help",SUBMENU,help_menu,
#else
                        "Help",ITEM,do_interface_help,
#endif
#endif
                        "Options",SUBMENU,options_menu
                        );
   if (main_menu == NULL)
     return(FALSE);
   return(TRUE);
  }

/******************************************************************************
 NAME        : draw_border
 PURPOSE     : Draws the main border on the screen
 DESCRIPTION : The border is stored in 4 different windows :
                 header, left_border, right_border, and footer
               (note : the advertisement and main_menu windows are sub-windows
                       of the header window)
               This was done in this rather roundabout away to save memory.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : The border is drawn character by character using the extended
               ascii set. Curses xbox routine is not used since it does not
               provide all the stuff needed to be drawn.
                 This routine is non-portable.
               Uses the colors defined in the colors array in COLOR.C
               Assumes header-window, border-windows, and footer-window
                 have already been allocated.
 ******************************************************************************/
static void draw_border()
  {
   register int i;

   wattrset(header_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   mvwaddch(header_window,0,0,DB_UL);
   for (i = 1 ; i < AD_COLS+AD_X ; i++)
     mvwaddch(header_window,0,i,DB_H_LIN);
   mvwaddch(header_window,0,i++,DBSG_UP_T);
   for ( ; i < HDR_COLS-1 ; i++)
     mvwaddch(header_window,0,i,DB_H_LIN);
   mvwaddch(header_window,0,i,DB_UR);
   mvwaddch(header_window,1,0,DB_V_LIN);
   mvwaddch(header_window,1,AD_COLS+AD_X,SG_V_LIN);
   mvwaddch(header_window,1,HDR_COLS-1,DB_V_LIN);
   mvwaddch(header_window,2,0,DBSG_LT_T);
   for (i = 1 ; i < AD_COLS+AD_X ; i++)
     mvwaddch(header_window,2,i,SG_H_LIN);
   mvwaddch(header_window,2,i++,SGSG_LO_T);
   for ( ; i < HDR_COLS-1 ; i++)
     mvwaddch(header_window,2,i,SG_H_LIN);
   mvwaddch(header_window,2,i,DBSG_RT_T);

   xpaint(left_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(left_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (i = 0 ; i < LINES-HDR_LNS-FTR_LNS ; i++)
     mvwaddch(left_border_window,i,0,DB_V_LIN);

   xpaint(right_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(right_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (i = 0 ; i < LINES-HDR_LNS-FTR_LNS ; i++)
     mvwaddch(right_border_window,i,0,DB_V_LIN);

   xpaint(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   mvwaddch(footer_window,0,0,DB_LL);
   for (i = 1 ; i < FTR_COLS-1 ; i++)
     waddch(footer_window,DB_H_LIN);
   waddch(footer_window,DB_LR);
  }

/******************************************************************************
 NAME        : draw_advertisement
 PURPOSE     : Draws the "CLIPS" logo on the screen
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes advertisement-window has been allocated
 ******************************************************************************/
static void draw_advertisement()
  {
   xpaint(advertisement_window,colors[ADVERTISE]|colors[MAIN_BACKGD]);
   wattrset(advertisement_window,colors[ADVERTISE]|colors[MAIN_BACKGD]);
   mvwaddstr(advertisement_window,0,1,CLIPS_LOGO);
  }

/******************************************************************************
 NAME        : draw_main_menu
 PURPOSE     : To display the main menu and allocate underlying windows
 DESCRIPTION : Draws the following selections in the main menu window :
                "Run","rEset","Step","File",
                "Debug","Action","eXamine", and "Help"
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the colors defined in the colors array in COLOR.C
               This routine assumes the internal main-menu has already been
                 correctly allocated.  Unpredictable results will occur if
                 it has not.
 ******************************************************************************/
static void draw_main_menu()
  {
   register int i,j,k;           /* Item loop counters                  */
   int item_wid;                 /* String length of item to be printed */
   char *item;                   /* Item to be printed                  */

   main_menu_cell_width = (MN_MENU_COLS) / main_menu->item_cnt;
   main_menu_cell_offset = ((MN_MENU_COLS) % main_menu->item_cnt) / 2;

   /* =======================
      Draw all the selections 
      ======================= */
   xpaint(main_menu_window,colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]);
   wattrset(main_menu_window,colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]);
   for (i = 0 ; i < main_menu->item_cnt ; i++)
     {
      /* ===============================================================
         Move to posn necessary to center string in cell and don't write
         any more characters than the cell width 
         =============================================================== */
      item = (main_menu->items)[i]->name;
      item_wid = strlen(item);
      if (item_wid >= main_menu_cell_width)
        j = 0;
      else
        j = main_menu_cell_width/2 - item_wid/2;
      wmove(main_menu_window,0,
            main_menu_cell_offset+main_menu_cell_width*i+j);
      for (k = 0 ; (j < main_menu_cell_width) && (item[k] != EOS) ; j++ , k++)
        waddch(main_menu_window,item[k]);
     }

   /* =====================================
      highlight initial main menu selection
      ===================================== */
   highlight_main_menu(main_menu_cell);

  }

/******************************************************************************
 NAME        : draw_main_window
 PURPOSE     : Used to draw/redraw the main window with a new background color
 DESCRIPTION : The foreground colors of individual characters are left intact
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the colors defined in the colors array in COLOR.C
               Assumes CLIPS I/O pad is currently allocated.
 ******************************************************************************/
static void draw_main_window()
  {
   register int i,j;
   int ch,rows,cols;

   getmaxrc(clips_window,rows,cols);
   wattrset(clips_window,0x0000);
   for (i = 0 ; i < rows ; i++)
     {
      for (j = 0 ; j < cols ; j++)
        { 
         ch = mvwinch(clips_window,i,j);
         if ((ch & A_CHARTEXT) != BLANK)
           ch = (ch & BGD_MSK) | colors[MAIN_BACKGD];
         else
           ch = (ch & A_CHARTEXT) | colors[STDIN] | colors[MAIN_BACKGD];
         mvwaddch(clips_window,i,j,ch);
        }
     }
   wattrset(clips_window,colors[STDIN]|colors[MAIN_BACKGD]);
   wmove(clips_window,clips_row,clips_col);
  }

/******************************************************************************
 NAME        : reset_colors
 PURPOSE     : Redraws main-screen colors
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void reset_colors()
  {
   draw_border();
   draw_advertisement();
   draw_main_menu();
   highlight_main_menu(main_menu_cell);
   draw_main_window();
   draw_pop_up_menu(file_menu);
   draw_pop_up_menu(execute_menu);
   draw_pop_up_menu(debug_menu);
   draw_pop_up_menu(watch_menu);
   draw_pop_up_menu(action_menu);
   draw_pop_up_menu(examine_menu);
#if CLP_HELP && CLP_TEXTPRO
   draw_pop_up_menu(help_menu);
#endif
   draw_pop_up_menu(options_menu);
   draw_pop_up_menu(env_menu);
#if CLP_EDIT
   draw_pop_up_menu(edit_menu);
#endif
   redisplay_wins();
   watch_status();
   memory_status();
  }

/******************************************************************************
 NAME        : update_screen
 PURPOSE     : Flushes most current window outputs to the physical screen
 DESCRIPTION : Has the same functionality as redisplay_wins() except does
               not un-optimize windows and thus force complete redraws of
               each.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes all windows are currently allocated
 ******************************************************************************/
static void update_screen()
  {
   WINBLK *qptr;

   wnoutrefresh(header_window);
   wnoutrefresh(advertisement_window);
   wnoutrefresh(main_menu_window);
   wnoutrefresh(left_border_window);
   wnoutrefresh(right_border_window);
   wattrset(footer_window,colors[WTRACE]|colors[MAIN_BACKGD]);
   wnoutrefresh(footer_window);
   wnoutrefresh(clips_window);
   if ((qptr = pop_up_queue_top) != NULL)
     {
      while (qptr != NULL)
        {
         if (qptr->py == -1)
           wnoutrefresh(qptr->win);
         else
           pnoutrefresh(qptr->win,qptr->py,qptr->px,qptr->ssy,qptr->ssx,
                        qptr->sey,qptr->sex);
         qptr = qptr->nxt;
        }
      doupdate();
      hide_cursor();
     }
   else
     doupdate();
  }

/******************************************************************************
 NAME        : highlight_main_menu
 PURPOSE     : Outlines current main menu selection
 DESCRIPTION : The routine uses the current background and foreground
               colors to unhighlight the old selection and the flip of
               these to highlight the new selection.
                 This routine sets the main-menu index main_menu_cell to
               the value of the new selection index.
 INPUTS      : 
 RETURNS     : Nothing useful
 NOTES       : If the foreground color has a high-bit (i.e FYELLOW, FLT_RED,
               etc.) it will be lost when that color is flipped to a 
               background color since background colors can only have 3 bits
               as opposed to foreground colors which can have 4.
                 Example : yellow on black would be flipped to black on brown
 ******************************************************************************/
static void highlight_main_menu(select)
  int select;
  {
   ch_attr unhgh_attr,hgh_attr;
   register int i;

   unhgh_attr = colors[MAIN_MENU_TEXT] | colors[MAIN_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);

   /* =========================
      Unhighlight old selection
      ========================= */
   wattrset(main_menu_window,unhgh_attr);
   wmove(main_menu_window,0,main_menu_cell_offset+
                            main_menu_cell*main_menu_cell_width);
   for (i = 0 ; i < main_menu_cell_width ; i++)
     waddch(main_menu_window,winch(main_menu_window) & A_CHARTEXT);

   /* =========================
      Highlight new selection
      ========================= */
   wattrset(main_menu_window,hgh_attr);
   main_menu_cell = select;
   wmove(main_menu_window,0,main_menu_cell_offset+
                            main_menu_cell*main_menu_cell_width);
   for (i = 0 ; i < main_menu_cell_width ; i++)
     waddch(main_menu_window,winch(main_menu_window) & A_CHARTEXT);

#if MOUSE
   /* ==============================================
      Set the mouse at the new menu position as well
      ============================================== */
   set_mouse(MN_MENU_Y,AD_COLS+AD_X+main_menu_cell_offset+1+
            (main_menu_cell*main_menu_cell_width) + main_menu_cell_width/2);
#endif
  }

/******************************************************************************
 NAME        : main_pop_up_start
 PURPOSE     : Finds screen x-coordinate start of main menu selection box
 DESCRIPTION : Used to start pop-up menus underneath main-menu selections
 INPUTS      : 1) The index (0..to max-selections-1) in the main-menu
 RETURNS     : The screen coordinate of the center of the box
 NOTES       : None
 ******************************************************************************/
static int main_pop_up_start(index)
  int index;
  {
   return(AD_COLS+AD_X+1+main_menu_cell_offset+index*main_menu_cell_width);
  }

/******************************************************************************
 NAME        : deallocate_pop_up_queue
 PURPOSE     : Deletes pop_up_queue space
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_pop_up_queue()
  {
   WINBLK *qptr;

   while (pop_up_queue_top != NULL)
     {
      qptr = pop_up_queue_top;
      pop_up_queue_top = pop_up_queue_top->nxt;
      release(1,WINBLK,qptr);
     }
   pop_up_queue_bottom= NULL;
  }

/******************************************************************************
 NAME        : deallocate_windows
 PURPOSE     : Frees main screen memory
 DESCRIPTION : Uses Curses routine delwin to free main screen window space
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Should be called after all Curses stuff is finished
 ******************************************************************************/
static void deallocate_windows()
  {
   /* =====================================================================
      Since the advertisement and main_menu windows are sub-windows of the
      header window, they must be deallocated first.
      ===================================================================== */
   if (advertisement_window != NULL)
     {
      delwin(advertisement_window);
      advertisement_window = NULL;
     }
   if (main_menu_window != NULL)
     {
      delwin(main_menu_window);
      main_menu_window = NULL;
     }
   if (header_window != NULL)
     {
      delwin(header_window);
      header_window = NULL;
     }
   if (left_border_window != NULL)
     {
      delwin(left_border_window);
      left_border_window = NULL;
     }
   if (right_border_window != NULL)
     {
      delwin(right_border_window);
      right_border_window = NULL;
     }
   if (footer_window != NULL)
     {
      delwin(footer_window);
      footer_window = NULL;
     }
   if (clips_window != NULL)
     {
      delwin(clips_window);
      clips_window = NULL;
     }
   if (stdscr == NULL)
     stdscr = newwin(LINES,COLS,0,0);  /* Hack to prevent endwin from bombing */
   endwin();
  }

/******************************************************************************
 NAME        : deallocate_menus
 PURPOSE     : Releases menu memory space
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_menus()
  {
   if (main_menu != NULL)
     main_menu = kill_menu(main_menu);
   if (file_menu != NULL)
     file_menu = kill_menu(file_menu);
   if (execute_menu != NULL)
     execute_menu = kill_menu(execute_menu);
   if (debug_menu != NULL)
     debug_menu = kill_menu(debug_menu);
   if (watch_menu != NULL)
     watch_menu = kill_menu(watch_menu);
   if (action_menu != NULL)
     action_menu = kill_menu(action_menu);
   if (examine_menu != NULL)
     examine_menu = kill_menu(examine_menu);
   if (options_menu != NULL)
     options_menu = kill_menu(options_menu);
   if (env_menu != NULL)
     env_menu = kill_menu(env_menu);
#if CLP_EDIT
   if (edit_menu != NULL)
     edit_menu = kill_menu(edit_menu);
#endif
  }

/******************************************************************************
 NAME        : restore_environment
 PURPOSE     : Resets the terminal to a normal state
 DESCRIPTION : Used to restore original environment upon CLIPS exit, i.e
               Local echo, Cooked state (buffered I/O), and newline mapping.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void restore_environment()
  {
   noraw();
   echo();
   nl();
  }

/******************************************************************************
 NAME        : init_log_name_table
 PURPOSE     : Loads logical name hash table with initial values
 DESCRIPTION : Initializes all nodes in hash table to NULL, and then allocates
               a node for each logical name (see note near beginning of file)
               and assigns its name and default I/O screen color.
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went well, FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
static int init_log_name_table()
  {
   register int i;

   for (i = 0 ; i < LOG_NAME_TABLE_SIZE ; i++)
     log_name_table[i] = NULL;

   if (log_insert("wclips",WCLIPS) == NULL)
     return(FALSE);
   if (log_insert("wdialog",WDIALOG) == NULL)
     return(FALSE);
   if (log_insert("wdisplay",WDISPLAY) == NULL)
     return(FALSE);
   if (log_insert("werror",WERROR) == NULL)
     return(FALSE);
   if (log_insert("wtrace",WTRACE) == NULL)
     return(FALSE);
   if (log_insert("wagenda",WAGENDA) == NULL)
     return(FALSE);
   if ((stdin_log = log_insert("stdin",STDIN)) == NULL)
     return(FALSE);
   if (log_insert("stdout",STDOUT) == NULL)
     return(FALSE);
   return(TRUE);
  }

/******************************************************************************
 NAME        : deallocate_log_name_table()
 PURPOSE     : Releases dynamic memory used by hash table
 DESCRIPTION : Memory associated with internal node fields as well as the
               nodes themselves is given back to the operating system
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_log_name_table()
  {
   register int i;
   LOGNAME *tmp1,*tmp2;

   for (i = 0 ; i < LOG_NAME_TABLE_SIZE ; i++)
     {
      tmp1 = log_name_table[i];
      while (tmp1 != NULL)
        {
         tmp2 = tmp1;
         tmp1 = tmp1->nxt;
         release(strlen(tmp2->name)+1,char,tmp2->name);
         release(1,LOGNAME,tmp2);
        }
      log_name_table[i] = NULL;      
     }
  }

/******************************************************************************
 NAME        : log_insert
 PURPOSE     : Puts a new logical name in the hash table
 DESCRIPTION : 
 INPUTS      : 1) The new logical name to be inserted in the hash table
               2) The screen color associated with it
 RETURNS     : The address of the log-entry if the insert was successful
               NULL if the logical name was already in the table
 NOTES       : None
 ******************************************************************************/
static LOGNAME *log_insert(log_name,color_index)
  char *log_name;
  int color_index;
  {
   register int i;
   LOGNAME *lnode,*lptr;

   i = hash(log_name,LOG_NAME_TABLE_SIZE);
   if ((lnode = balloc(1,LOGNAME)) == NULL)
     return(NULL);
   if ((lnode->name = balloc(strlen(log_name)+1,char)) == NULL)
     {
      release(1,LOGNAME,lnode);
      return(NULL);
     }
   strcpy(lnode->name,log_name);
   lnode->color = color_index;
   lnode->nxt = NULL;
   if (log_name_table[i] == NULL)
     log_name_table[i] = lnode;
   else
     {
      lptr = log_name_table[i];
      while (lptr != NULL)
        {
         if (strcmp(lptr->name,log_name) == 0)
           {
            release(strlen(log_name)+1,char,lnode->name);
            release(1,LOGNAME,lnode);
            return(NULL);
           }
         if (lptr->nxt == NULL)
           break;
         lptr = lptr->nxt;
        }
      lptr->nxt = lnode;
     }
   return(lnode);
  }

/******************************************************************************
 NAME        : log_lookup
 PURPOSE     : Searches the logical name hash table
 DESCRIPTION : 
 INPUTS      : 1) The logical name which is the object of the lookup
 RETURNS     : The address of the logical name node in the hash table if
               the name was found, NULL otherwise.
 NOTES       : This function utilizes chaining for collision resolution
 ******************************************************************************/
static LOGNAME *log_lookup(log_name)
  char *log_name;
  {
   LOGNAME *lptr;

   lptr = log_name_table[hash(log_name,LOG_NAME_TABLE_SIZE)];
   while (lptr != NULL)
     {
      if (strcmp(lptr->name,log_name) == 0)
        return(lptr);
      lptr = lptr->nxt;
     }
   return(NULL);
  }

/******************************************************************************
 NAME        : hash
 PURPOSE     : Map a string into an integer index
 DESCRIPTION : Hash generates an index into the hash table for a string
               by the following algorithm :
                 For each character in the string, take its ascii value
               multiplied by its position in the string, and then add all
               these values together.  The hash index is this final sum
               modulo the hash table size.
 INPUTS      : 1) the string to be hashed
               2) the number of elements in the hash table
 RETURNS     : the hash value of the string
 NOTES       : This routine is used to quickly look up logical names
               to determine recognition and/or where I/O should go.
 ******************************************************************************/
static int hash(str,size)
  char *str;
  int size;
  {
   register unsigned i; /* Index into the string                            */
   unsigned hashval;    /* String hash value - needs to be unsigned in case
                           of overflow during the hash                      */

   for (i = 0 , hashval = 0 ; str[i] != EOS ; i++)
     hashval += ((unsigned) str[i])*(i+1);
   hashval %= (unsigned) size;
   return((int) hashval);
  }

/******************************************************************************
 NAME        : append_file_clips_cmd_str
 PURPOSE     : Writes a DOS file-name to the CLIPS command-string in
               the appropriate format for parsing by CLIPS
 DESCRIPTION : The file-name is surrounded by double-quotes and all backslashes
               are expaned to double-backslashes
 INPUTS      : 1) The file-name
 RETURNS     : Nothing useful
 NOTES       : Uses append_command_string() in COMMLINE.C
 ******************************************************************************/
static void append_file_clips_cmd_str(file)
  char *file;
  {
   register int i,j;

   append_command_string("\"");
   for (i = j = 0 ; file[i] != EOS ; i++)
     {
      if (file[i] == '\\')
        {
         file[i] = EOS;
         if (file[j] != EOS)
           append_command_string(file+j);
         append_command_string("\\\\");
         file[i] = '\\';
         j = i+1;
        }
     }
   if (file[j] != EOS)
     append_command_string(file+j);
   append_command_string("\"");
  }

/******************************************************************************
 NAME        : clips_addch
 PURPOSE     : Prints characters to the main CLIPS window
 DESCRIPTION : This routine directly handles all window output, especially
               wrapping and scrolling, since the normal Curses automatic
               stuff screws up non-black background colors.
 INPUTS      : 1) The character to be printed
 RETURNS     : TRUE (1) if character successfully added, FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
static int clips_addch(ch)
  char ch;
  {
   if (ch == DELETE)
     {
      if (clips_col != 0)
        {
         mvwaddstr(clips_window,clips_row,clips_col,DEL_STR);
         clips_col--;
        }
      else if (clips_row != 0)
        {
         clips_row--;
         clips_col = (int) newline_map[(nlmap_ptr + clips_row) % clips_lns];
         if (clips_col == CLIPS_COLS)
           {
            mvwaddstr(clips_window,clips_row,clips_col--,DEL_STR);
            wmove(clips_window,clips_row,clips_col);
           }
         else
           wmove(clips_window,clips_row,clips_col);
        }
      else
        return(FALSE);
     }
   else if ((ch != NEWLINE) ? (ch == CRGRTN) : TRUE)
     clips_advance_row();
   else if (clips_col == CLIPS_COLS-1)
     {
      if ((ch != NEWLINE) ? (ch != CRGRTN) : FALSE)
        mvwaddch(clips_window,clips_row,clips_col++,ch);
      clips_advance_row();
     }
   else
     mvwaddch(clips_window,clips_row,clips_col++,ch);
   return(TRUE);
  }

/******************************************************************************
 NAME        : clips_advance_row
 PURPOSE     : Moves cursor to a new line
 DESCRIPTION : Not only takes care of Curses newline glitches with non-black
               backgrounds but also handles scrolling the CLIPS window 
               (if necessary) so as to preserve the background color.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Only this routine is permitted to scroll the CLIPS pad
               via the Curses routines deleteln() and xwclrtoeol().
               Thus, other routines (namely clips_addch()) can place
               characters in the last row/last column position of the window.
 ******************************************************************************/
static void clips_advance_row()
  {
   newline_map[(nlmap_ptr + clips_row) % clips_lns] = 
                                                   (unsigned char) clips_col;
   clips_col = 0;
   if (clips_row == (clips_lns-1))
     {
      nlmap_ptr = (nlmap_ptr + 1) % clips_lns;
      xscrollup(clips_window,0,0,clips_lns-1,CLIPS_COLS-1,
                colors[STDIN]|colors[MAIN_BACKGD]);
      wmove(clips_window,clips_row,clips_col);
     }
   else
     wmove(clips_window,++clips_row,clips_col);
  }

/******************************************************************************
 NAME        : clips_addstr
 PURPOSE     : Writes a whole string to the main CLIPS window
 DESCRIPTION : Automatically pauses output if the top line of continuous output
               is about to scroll of the screen, e.g the "More" facility.
 INPUTS      : 1) The address of the string to be printed
 RETURNS     : Nothing useful
 NOTES       : Assumes no "Delete" characters are imbedded in the string.
 ******************************************************************************/
static void clips_addstr(str)
  char *str;
  {
   register int i,ch;

   for (i = 0 ; (ch = str[i]) != EOS ; i++)
     {
      if ((ch != NEWLINE) ? (ch == CRGRTN) : TRUE)
        {
         if (use_more ? 
             (((output_lines+1) % (LINES-HDR_LNS-FTR_LNS)) == 0) : FALSE)
           pause_output();
         clips_advance_row();
         output_lines++;
        }
      else if (clips_col == CLIPS_COLS-1)
        {
         if ((ch != NEWLINE) ? (ch != CRGRTN) : FALSE)
           mvwaddch(clips_window,clips_row,clips_col++,ch);
         if (use_more ? 
              (((output_lines+1) % (LINES-HDR_LNS-FTR_LNS)) == 0) : FALSE)
           pause_output();
         clips_advance_row();
         output_lines++;
        }
      else
        mvwaddch(clips_window,clips_row,clips_col++,ch);
     }
  }

/******************************************************************************
 NAME        : process_special
 PURPOSE     : Runs special interface routines based on control-sequences
 DESCRIPTION : 
 INPUTS      : 1) The CLIPS action code to be processed
 RETURNS     : TRUE (1) if a command to CLIPS has been issued,
               FALSE (0) otherwise.
 NOTES       : Called by the function next_event().
 ******************************************************************************/
static int process_special(ch,new_cell,perform)
  int ch,*new_cell,*perform;
  {
   MENU_ITEM *item;
   int (*handler)(int) = NULL;
   int main_select = main_menu_cell,
       handler_arg = 0,
       handler_rtn;

   switch(ch)
     {
      case LEFT_ARROW   : *new_cell = (main_menu_cell+main_menu->item_cnt-1)
                                       % main_menu->item_cnt;
                          break;
      case RIGHT_ARROW  : *new_cell = (main_menu_cell+1) % main_menu->item_cnt;
                          break;
      case DOWN_ARROW   : 
      case MAIN_MENU    : *perform = TRUE;
                          break;
      case REDRAW_CMD   : redisplay_wins();
                          break;
      case FILE_MENU    : *new_cell = FILE_INDEX;
                          *perform = TRUE;
                          break;
      case EXEC_MENU    : *new_cell = EXEC_INDEX;
                          *perform = TRUE;
                          break;
      case DBG_MENU     : *new_cell = DBG_INDEX;
                          *perform = TRUE;
                          break;
      case ACTION_MENU  : *new_cell = ACTN_INDEX;
                          *perform = TRUE;
                          break;
      case EXAM_MENU    : *new_cell = EXAM_INDEX;
                          *perform = TRUE;
                          break;
      case OPTION_MENU  : *new_cell = OPTS_INDEX;
                          *perform = TRUE;
                          break;
      case EXIT_TO_OS   : handler = do_exit;
                          main_select = FILE_INDEX;
                          break;
      case LOAD_CMD     : handler = do_load;
                          main_select = FILE_INDEX;
                          break;
      case RESET_CMD    : handler = do_reset;
                          main_select = EXEC_INDEX;
                          break;
      case RUN_CMD      : handler = do_run;
                          main_select = EXEC_INDEX;
                          break;
      case STEP_CMD     : handler = do_step;
                          main_select = EXEC_INDEX;
                          handler_arg = 3;
                          break;
      case DBG_TOGGLE   : highlight_main_menu(DBG_INDEX);
                          update_screen();
                          *new_cell = DBG_INDEX;
                          do_watch_all_toggle();
                          break;
      case HELP_CMD     : *new_cell = HELP_INDEX;
                          *perform = TRUE;
                          break;
      case CLEAR_WIN    : handler = do_clear_window;
                          main_select = OPTS_INDEX;
                          break;
      case DOS_CMD      : handler = do_spawn;
                          main_select = OPTS_INDEX;
                          handler_arg = 2;
                          break;
      case SPAWN_CLI    : handler = do_spawn;
                          main_select = OPTS_INDEX;
                          handler_arg = 3;
                          break;
      case TOGGLE_MORE  : handler = do_toggle_more;
                          main_select = OPTS_INDEX;
#if CLP_EDIT
                          handler_arg = 6;
#else
                          handler_arg = 5;
#endif
                          break;
      case DESCRIBE_KEY : describe_key();
                          break;
      case DO_NOTHING   : break;
      default           : beep();

     }
   if (handler != NULL)
     {
      *new_cell = main_select;
      highlight_main_menu(main_select);
      update_screen();
      handler_rtn = handler(handler_arg);
      redisplay_wins();
      if (handler_rtn == MENU_ITEM_CLIPS_INPUT)
        return(TRUE);
     }
   return(FALSE);
  }

/******************************************************************************
 NAME        : pause_output
 PURPOSE     : Holds output and waits for a user to press a key indicating
               to continue
 DESCRIPTION : Used when a stream of continuous output is about to have its
               top line scroll off the screen.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void pause_output()
  {
   int quit = FALSE,
       ch,y,x;

   wattrset(footer_window,colors[MORE_MSG]|colors[MAIN_BACKGD]);
   mvwaddstr(footer_window,0,FTR_COLS/2-6," -- More -- ");
   update_screen();
#if MOUSE
   save_mouse();
#endif
   while (! quit)
     {
      if (wgetch(clips_window) != -1)
        quit = TRUE;
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        quit = TRUE;
#endif
     }
#if MOUSE
   restore_mouse();
#endif
   wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (x = FTR_COLS/2-6 ; x < FTR_COLS/2+6 ; x++)
     mvwaddch(footer_window,0,x,DB_H_LIN);
   update_screen();
  }

/******************************************************************************
 NAME        : describe_key
 PURPOSE     : Prints out a run-time display of key-binds
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses key_action() in KEY.C
 ******************************************************************************/
static void describe_key()
  {
   WINDOW *win,*text = NULL;
   int ch, done = FALSE;
   char *action;
#if MOUSE
   int y,x;
#endif

   if ((win = newwin(4,40,LINES/2-2,COLS/2-20)) == NULL)
     {
      beep();
      return;
     }
   if ((text = subwin(win,2,38,LINES/2-1,COLS/2-19)) == NULL)
     {
      beep();
      delwin(win);
      return;
     }
   keypad(text,TRUE);
   nodelay(text,TRUE);
   xpaint(win,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   xbox(win,SG_V_LIN,SG_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   xpaint(text,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   wattrset(text,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   mvwaddstr(text,0,0,"Describe action for what key?");
   wmove(text,1,0);
   touchwin(win);
   wnoutrefresh(win);
   touchwin(text);
   wnoutrefresh(text);
   doupdate();
   while (! done)
     {
      if ((ch = wgetch(text)) != -1)
        {
         if (ch == ESC)
           done = TRUE;
         else
           {
            action = key_action(ch);
            if (action != NULL)
              {
               wmove(text,1,0);
               xwclrtoeol(text);
               mvwaddstr(text,1,0,action);
               wrefresh(text);
              }
            else
              beep();
           }
        }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        done = TRUE;
#endif
     }
   delwin(text);
   delwin(win);
   redisplay_wins();
  }

/******************************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
