   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*             FUZZY REASONING COMMANDS MODULE         */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*      Brian L. Donnell                                     */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                  (extensions to run command)              */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/


/******************************************************************
    Fuzzy Logic Module

    This file contains the following categories of functions:
    
    Fuzzy reasoning related commands (functions)

 ******************************************************************/
 
#define _FUZZYCOM_SOURCE_


#include "setup.h"


#if FUZZY_DEFTEMPLATES

#include <stdio.h>
#define _CLIPS_STDIO_

#include <string.h>

#include "extnfunc.h"
#include "evaluatn.h"
#include "prntutil.h"
#include "argacces.h"
#include "clipsmem.h"
#include "router.h"
#include "factrhs.h"
#include "factmngr.h"
#include "symbol.h"

#include "fuzzyval.h"
#include "fuzzylv.h"
#include "fuzzycom.h"



#define ONE_THIRD    (1.0/3.0)
#define TWO_THIRDS   (2.0/3.0)



/******************************************************************
    Global Internal Function Declarations
    
    Defined in fuzzycom.h
 ******************************************************************/



/******************************************************************
    Local Internal Function Declarations
 ******************************************************************/
 
#if ANSI_COMPILER
    static struct universe *getFuzzyUniversePtr(struct expr *theArgument, char *functionName);
    static struct fact   *getFuzzyFactPtr(struct expr *theArgument, char *functionName);
    static char          *u_to_string ( struct universe *up, int *length );
    static char          *fs_to_string ( struct fact *factPtr, int *length );
    static double         moment_defuzzification( struct fact *factPtr );  
    static VOID           assert_defuzzified_fact( char *name, double value);
    static VOID           get_moment_and_area ( double *moment_ptr, double *area_ptr, double x1,
                                                double y1, double x2, double y2 );
    static double         maximum_defuzzification( struct fact *factPtr );  
    static char          *fv_to_string ( struct fact *fp, int *length_ptr );    

#else
    static struct universe *getFuzzyUniversePtr();
    static struct fact   *getFuzzyFactPtr();
    static char          *u_to_string ();
    static char          *fs_to_string ();
    static double         moment_defuzzification();  
    static VOID           assert_defuzzified_fact();
    static VOID           get_moment_and_area ();
    static double         maximum_defuzzification();  
    static char          *fv_to_string();    

#endif



/******************************************************************
    Local Internal Variable Declarations
 ******************************************************************/
   
   
   
/******************************************************************
    Global External Variable Declarations
 ******************************************************************/



/******************************************************************
    Global Internal Variable Declarations
 ******************************************************************/


/*************************************************************

        DefuzzyCommands
        
    - initialize the commands related to fuzzy reasoning 

**************************************************************/


globle VOID DeffuzzyCommands()
{

#if ! RUN_TIME
 
 DefineFunction2("moment-defuzzify",'d',PTIF moment_defuzzify, "moment_defuzzify", "11z");
 DefineFunction2("maximum-defuzzify",'d',PTIF maximum_defuzzify, "maximum_defuzzify", "11z");
 DefineFunction2("get-u", 'w', PTIF getu, "getu", "11z");
 DefineFunction2("get-u-from", 'd', PTIF getu_from, "getu_from", "11z");
 DefineFunction2("get-u-to", 'd', PTIF getu_to, "getu_to", "11z");
 DefineFunction2("get-u-units", 'w', PTIF getu_units, "getu_units", "11z");
 DefineFunction2("get-fs", 'w', PTIF get_fs, "get_fs", "11z");
 DefineFunction2("get-fs-length", 'i', PTIF get_fs_length, "get_fs_length", "11z");
 DefineFunction2("get-fs-x", 'd', PTIF get_fs_x, "get_fs_x", "22*zi");
 DefineFunction2("get-fs-y", 'd', PTIF get_fs_y, "get_fs_y", "22*zi");

#endif

}




/********************************************************************
    Functions to access the universe of discourse 
 ********************************************************************/                                                                    


/************************************************************
    seeIfFuzzyFact():                                            

    given a ptr to a DATA_OBJECT of type integer or fact-address
    see if it is designates a fuzzy fact or not
    
    returns a ptr to a fuzzy fact or NULL if error occurred
************************************************************/
static struct fact *seeIfFuzzyFact( theResult, functionName )

  DATA_OBJECT  *theResult;
  char *functionName;
{
   long int factIndex;
   int found_fact;
   struct fact *factPtr;
   char tempBuffer[20];

   if (theResult->type == INTEGER)
     { 
       factIndex = ValueToLong(theResult->value); 
       if (factIndex < 0)
         {            
           ExpectedTypeError1(functionName,1,"fact-index must be positive");
           return(NULL);
         }
       found_fact = CLIPS_FALSE;
       factPtr = (struct fact *) GetNextFact(NULL);
       while (factPtr != NULL)
         {
           if (factPtr->factIndex == factIndex)
             {
               found_fact = CLIPS_TRUE;
               break;
             }
           factPtr = factPtr->nextFact;
         }
       
       if (found_fact == CLIPS_FALSE)
         {
           sprintf(tempBuffer,"f-%ld",factIndex);
           CantFindItemErrorMessage("fuzzy fact",tempBuffer);
           return(NULL);
         }
      }
    else
      { /* arg type is fact address */
        factPtr = (struct fact *) theResult->value; 
      }     
        
    /* could also check with (factPtr->whichDeftemplate->fuzzyTemplate != NULL) */
    if ( factPtr->theProposition.theFields[0].type == FUZZY_VALUE )
       return( factPtr );
    else 
      {
        sprintf(tempBuffer,"f-%ld",factPtr->factIndex);
        CantFindItemErrorMessage("fuzzy fact", tempBuffer);
        return( NULL );
      }
}

/************************************************************
    getFuzzyFactPtr():                                            

    given a ptr to an argument of a function that is expected
    to be a fact address or a fact index get a ptr to the fact
    
    returns a ptr to a fact or NULL if error occurred
************************************************************/
static struct fact *getFuzzyFactPtr(theArgument, functionName)

  struct expr *theArgument;
  char *functionName;
{
   DATA_OBJECT  theResult;
   
   EvaluateExpression(theArgument,&theResult);
      
   if ((theResult.type == INTEGER) || (theResult.type == FACT_ADDRESS))
     {
       return( seeIfFuzzyFact(&theResult, functionName) );
     }
   
   ExpectedTypeError1(functionName,1,"fact-index or fact-address of fuzzy fact");
   return( NULL );
}



/************************************************************
    getFuzzyUniversePtr():                                   

    given a ptr to an argument of a function that is expected
    to be a fact address or a fact index or the name of a fuzzy
    deftemplate get a ptr to the universe of discourse
    
    returns a ptr to the universe struct  or NULL if an 
    error occurred
************************************************************/
static struct universe *getFuzzyUniversePtr(theArgument, functionName)

  struct expr *theArgument;
  char *functionName;
{
   DATA_OBJECT  theResult;
   struct fact *factPtr;
   struct deftemplate *theDeftemplate;
   
   EvaluateExpression(theArgument,&theResult);
      
   if ((theResult.type == INTEGER) || (theResult.type == FACT_ADDRESS))
     { /* Fuzzy fact being referenced */
       factPtr = seeIfFuzzyFact(&theResult, functionName);
       if (factPtr == NULL)
         return( NULL );
       else
         return( factPtr->whichDeftemplate->fuzzyTemplate->u_ptr );
     }
   else if (theResult.type == SYMBOL)
     { /* fuzzy deftemplate being referenced */
       char * theName;

       theName = ValueToString(theResult.value);
       theDeftemplate = (struct deftemplate *)FindDeftemplate(theName);
       if (theDeftemplate != NULL)
         {
           if (theDeftemplate->fuzzyTemplate != NULL)
              return( theDeftemplate->fuzzyTemplate->u_ptr );
           else
	     {
               CantFindItemErrorMessage("fuzzy fact", theName);
               return( NULL );
             }
         }
     }
   
   ExpectedTypeError1(functionName,1,"fact-index/fact-address of fuzzy fact or fuzzy deftemplate name (symbol)");
   return( NULL );
}




/************************************************************
    getu():                                            

    returns the universe limits and units of a single fact in
    SYMBOL format
************************************************************/
globle VOID *getu()
  {
   struct universe *uPtr;
   struct expr *theArgument;
   VOID *hashresult;
   char *ustring;
   int ulength = 0;

   if (ArgCountCheck("get-u",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           uPtr = getFuzzyUniversePtr(theArgument, "get-u");
       
           if (uPtr != NULL)
             {
               ustring = u_to_string(uPtr, &ulength); 
               hashresult = (VOID *) AddSymbol(ustring);
               rm(ustring, ulength + 1);
               return(hashresult);
             }
          }
      }
        
    SetEvaluationError(CLIPS_TRUE);
    return((VOID *) AddSymbol(""));
  }
  
  

/************************************************************
    u_to_string():                                            

    given a fuzzy universe pointer returns the universe limits 
    and units of a single fact as a string
    
************************************************************/
static char *u_to_string ( up, length_ptr )
struct universe *up;
int *length_ptr;
{
    char *string = NULL;
    
    if (up->units != NULL)
    {
        *length_ptr = 36 + strlen(ValueToString(up->units));
        string = gm2( (*length_ptr + 1));
        sprintf( string, "%.2f - %.2f %s", up->from, up->to, ValueToString(up->units));
    }
    else    /* no units specified */
    {
        *length_ptr = 36;
        string = gm2((*length_ptr + 1));
        sprintf( string, "%.2f - %.2f", up->from, up->to);
    }

    return (string);
}


/************************************************************
    getu_from():                                            

    returns the universe minimum limit of a single fact in
    double format
************************************************************/
globle double getu_from()
  {
   struct universe *uPtr;
   struct expr *theArgument;

   if (ArgCountCheck("get-u-from",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           uPtr = getFuzzyUniversePtr(theArgument, "get-u-from");
       
           if (uPtr != NULL)
             {
               return(uPtr->from);            
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0.0);
}



 /************************************************************
    getu_to():                                            

    returns the universe maximum limit of a single fact in
    double format
************************************************************/
globle double getu_to()
  {
   struct universe *uPtr;
   struct expr *theArgument;

   if (ArgCountCheck("get-u-to",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           uPtr = getFuzzyUniversePtr(theArgument, "get-u-to");
       
           if (uPtr != NULL)
             {
               return(uPtr->to);            
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0.0);
}



/************************************************************
    getu_units():                                            

    returns the universe units of a single fact in
    word format

    returns "" if no units specified in deffuzzy
************************************************************/
globle VOID *getu_units()
  {
   struct universe *uPtr;
   struct expr *theArgument;
   struct symbolHashNode *hashPtr;
   
   if (ArgCountCheck("get-u-units",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           uPtr = getFuzzyUniversePtr(theArgument, "get-u-units");
       
           if (uPtr != NULL)
             {
               hashPtr = uPtr->units;
               if (hashPtr == NULL)
		 return((VOID *) AddSymbol(""));
               else
                 return((VOID *) AddSymbol(ValueToString(hashPtr)));
             }
          }
      }    
        
    SetEvaluationError(TRUE);
    return((VOID *) AddSymbol(""));
  }
  
  


/**********************************************************************
    FUNCTIONS TO ACCESS FUZZY SET ELEMENTS
 **********************************************************************/
 
 /************************************************************
    get_fs():                                            

    returns the fuzzy set array of a fact in string format

    if the fact is not fuzzy, a warning is printed and 
    the word "no fuzzy set" is returned.
************************************************************/
globle VOID *get_fs()
  {
   struct fact *factPtr;
   struct expr *theArgument;
   VOID *hashresult;
   char *ustring;
   int ulength = 0;

   if (ArgCountCheck("get-fs",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "get-fs");
       
           if (factPtr != NULL)
             {
               ustring = fv_to_string(factPtr, &ulength); 
               hashresult = (VOID *) AddSymbol(ustring);
               rm(ustring, ulength + 1);
               return(hashresult);
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return((VOID *) AddSymbol(""));
}



/************************************************************
    fv_to_string():                                            

    given a fuzzy fact pointer returns the fuzzy set of 
    a single fact as a string
    
************************************************************/

static char *fv_to_string ( fp, length_ptr )    
struct fact *fp;
int *length_ptr;
{
    struct fuzzy_value *fv_ptr;
    int i, num;
    char *string, *strindex;

    fv_ptr = (struct fuzzy_value *)
             ValueToFuzzyValue((fp->theProposition.theFields[0].value));
    num = fv_ptr->n;
    
    /* NOTE: We should really use a MAXDIGITS that tells max float digits */
    /* allows for 10 digits in front of decimal point plus space for ... to
       indicate that space exceeded and not all numbers printed */
       
    *length_ptr = num*33 + 50;  

    string = gm2(*length_ptr + 1);
    strindex = string;
    
    for (i=0; i < num; i++)    
         {
           if (strindex-string > *length_ptr-50)
             { /* we are in the string safety zone -- stop before overrun space */
               /* perhaps we should do this in a better way -- get each pair and
                  append them to each other as needed 
               */
               strcpy(strindex, " ... ");
               PrintCLIPS(WERROR,"Internal Problem *****\n");
               PrintCLIPS(WERROR,"String space exceeded in fs_to_string (FUZZYCOM.C)\n");
               break;
             }
             
           sprintf(strindex, "(%.3f, %.3f) ", fv_ptr->x[i], fv_ptr->y[i]);
           strindex = string + strlen(string);
         }
         
   return(string);
}



 /************************************************************
    get_fs_length():                                            

    returns the size of a fuzzy set array in
    INTEGER format

    if the fact is not fuzzy, a warning is printed and 
    a value of 0 is returned.
************************************************************/
globle int get_fs_length()
  {
   struct fact *factPtr;
   struct expr *theArgument;
   struct fuzzy_value *fv_ptr;

   if (ArgCountCheck("get-fs-length",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "get-fs-length");
       
           if (factPtr != NULL)
             {
               fv_ptr = (struct fuzzy_value *)
                        ValueToFuzzyValue((factPtr->theProposition.theFields[0].value));
               return(fv_ptr->n);            
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0);
}



/************************************************************************
    get_fs_x()

    syntax: (get-fs-x <?var> <integer expression>)

    Returns the x value of the ith point in the fuzzy set array.
    If the expression evaluates to a number which is not an integer
    value, the expression result is truncated to an integer.
 ************************************************************************/
globle double get_fs_x()
{
   struct fact *factPtr;
   struct expr *theArgument;
   struct fuzzy_value *fv_ptr;
   DATA_OBJECT result;
   int index;

   if (ArgCountCheck("get-fs-x",EXACTLY,2) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "get-fs-x");
       
           if (factPtr != NULL)
             {
               fv_ptr = (struct fuzzy_value *)
                        ValueToFuzzyValue((factPtr->theProposition.theFields[0].value));

               if (ArgTypeCheck("get-fs-x",2,INTEGER_OR_FLOAT,&result) == CLIPS_TRUE)
                  {
                    if (result.type == INTEGER)
                      { index = ValueToInteger(result.value); }
                    else
                      { index = (int) ValueToDouble(result.value); }
                      
                    if (index < 0 || index >= fv_ptr->n)
                      { /* error - out of range */
                        PrintCLIPS(WERROR,"Function get-fs-x, index out of range\n");
                      }
                    else
                      {
                        return( (double)(fv_ptr->x[index]) );
                      }
                  }
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0.0);
    
}




/************************************************************************
    get_fs_y()

    syntax: (get-fs-y <?var> <integer expression>)

    Returns the y value of the ith point in the fuzzy set array.
    If the expression evaluates to a number which is not an integer
    value, the expression result is truncated to an integer.
 ************************************************************************/
globle double get_fs_y()
{
   struct fact *factPtr;
   struct expr *theArgument;
   struct fuzzy_value *fv_ptr;
   DATA_OBJECT result;
   int index;

   if (ArgCountCheck("get-fs-y",EXACTLY,2) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "get-fs-y");
       
           if (factPtr != NULL)
             {
               fv_ptr = (struct fuzzy_value *)
                        ValueToFuzzyValue((factPtr->theProposition.theFields[0].value));

               if (ArgTypeCheck("get-fs-y",2,INTEGER_OR_FLOAT,&result) == CLIPS_TRUE)
                  {
                    if (result.type == INTEGER)
                      { index = ValueToInteger(result.value); }
                    else
                      { index = (int) ValueToDouble(result.value); }
                      
                    if (index < 0 || index >= fv_ptr->n)
                      { /* error - out of range */
                        PrintCLIPS(WERROR,"Function get-fs-y, index out of range\n");
                      }
                    else
                      {
                        return( (double)(fv_ptr->y[index]) );
                      }
                  }
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0.0);
    
}





/***********************************************************************
    DEFUZZIFYING FUNCTIONS
 ***********************************************************************/

/************************************************************/
/* moment-defuzzify function:                               */
/************************************************************/
globle double moment_defuzzify()
{
   struct fact *factPtr;
   struct expr *theArgument;

   if (ArgCountCheck("moment-defuzzify",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "moment-defuzzify");
       
           if (factPtr != NULL)
             {
               return(moment_defuzzification(factPtr));
             }
          }
      }
        
    SetEvaluationError(TRUE);
    return(0.0);
}



/**********************************************************************
    moment_defuzzification()

    Calculates the first moment of area of a fuzzy set about the
    y axis.  The set is subdivided into different shapes by partitioning
    vertically at each point in the set, resulting in rectangles, 
    triangles, and trapezoids.  The centre of gravity and area of
    each subdivision is calculated using the appropriate formulas
    for each shape.  The first moment of area of the whole set is
    then:

        sum of ( moment(i) * area(i) )          <-- top
        ------------------------------
            sum of (area(i))                    <-- bottom

 **********************************************************************/   
static double moment_defuzzification(factPtr)
  struct fact *factPtr;  
{
    int i,  num;
    struct fuzzy_value *fv;    
       
    double result, local_moment, local_area, xmin, xmax;
    double currentx, currenty, nextx, nexty;
    double top = 0.0, bottom = 0.0;
    
    double *fsx, *fsy; /* ptrs to fuzzy set x and y arrays */

    fv = (struct fuzzy_value *)
               ValueToFuzzyValue((factPtr->theProposition.theFields[0].value));

    /**********************************************
     Start calculating the c.o.g.
     **********************************************/

    xmin = factPtr->whichDeftemplate->fuzzyTemplate->u_ptr->from;
    xmax = factPtr->whichDeftemplate->fuzzyTemplate->u_ptr->to;
    
    num = fv->n;
    if ( num <= 1 )  /* single point is constant fuzzy set over universe */
      {
        result = 0.5 * (xmax - xmin) + xmin;
      }
    else
      {
        fsx = fv->x;
        fsy = fv->y;
    
        currentx = fsx[0];
        currenty = fsy[0];

        /* Check for open-ended set & add initial rectangle if needed */
        if ( currenty != 0.0 && currentx != xmin )    
          {
            local_moment = 0.5 * (currentx+xmin);
            local_area = (currentx - xmin) * currenty;
            top += local_moment * local_area;
            bottom += local_area;    
          }

        for ( i = 1; i < num; i++ )
          {
            nextx = fsx[i];
            nexty = fsy[i];

            get_moment_and_area ( &local_moment, &local_area, currentx, currenty, nextx, nexty );
            top += local_moment * local_area;
            bottom += local_area;

            currentx = nextx;
            currenty = nexty;
          }
    
        /* Check for open-ended set & add final rectangle if needed */
        if ( currenty != 0.0 && currentx < xmax )
          {
            local_moment = 0.5 * (currentx + xmax);
            local_area = (xmax - currentx) * currenty;
            top += local_moment * local_area;
            bottom += local_area;        
          }
      
        /*************************************************************
        Calculate the final result but check for all zero set.
        **************************************************************/
        if ( bottom == 0.0 )
            result = 0.5 * ( xmax - xmin );
        else
            result = top/bottom;
      }

   return( result );

}




/***********************************************************************
    get_moment_and_area(moment_ptr, area_ptr, x1, y1, x2, y2)

    Given a polygon defined by the vertices (x1, 0), (x1, y1),
    (x2, y2), (x2, 0), the first moment of area of the polygon is
    returned in moment_ptr, and the area of the polygon in area_ptr.

    Conditions: x2 > x1.
 ***********************************************************************/
static VOID get_moment_and_area ( moment_ptr, area_ptr, x1, y1, x2, y2 )
double *moment_ptr, *area_ptr, x1, y1, x2, y2;
{
    if ( y1 == 0.0 && y2 == 0.0 )         /* rectangle of zero height */
      {
        *moment_ptr = 0.0;
        *area_ptr = 0.0;
      }
    else if ( y1 == y2 )            /* rectangle */
      {
        *moment_ptr = 0.5 * ( x1 + x2 );
        *area_ptr = ( x2 - x1 ) * y1;
      }
    else if ( y1 == 0.0 && y2 != 0.0 )    /* triangle, height y2 */
      {
        *moment_ptr = TWO_THIRDS * ( x2 - x1 ) + x1;
        *area_ptr = 0.5 * ( x2 - x1 ) * y2;
      }
    else if ( y2 == 0.0 && y1 != 0.0 )    /* triangle, height y1 */
      {
        *moment_ptr = ONE_THIRD * ( x2 - x1 ) + x1;
        *area_ptr = 0.5 * (x2 - x1 ) * y1;
      }
    else                     /* trapezoid */
      {
        *moment_ptr = ( TWO_THIRDS * (x2-x1) * (y2+0.5*y1))/(y1+y2) + x1;
        *area_ptr = 0.5 * ( x2 - x1 ) * ( y1 + y2 );
      }
}


/************************************************************/
/* maximum-defuzzify function:                              */
/************************************************************/
globle double maximum_defuzzify()
{
   struct fact *factPtr;
   struct expr *theArgument;

   if (ArgCountCheck("maximum-defuzzify",EXACTLY,1) != -1)
     {     
       theArgument = GetFirstArgument();

       if (theArgument != NULL)
         {
           factPtr = getFuzzyFactPtr(theArgument, "maximum-defuzzify");
       
           if (factPtr != NULL)
             {
               return( maximum_defuzzification(factPtr) );
             }
          }
      }
        
    SetEvaluationError(CLIPS_TRUE);
    return(0.0);
}
  
  
  
/********************************************************************
    maximum_defuzzification()

    finds the mean of maxima of a fuzzy set
    
    NOTE: This really doesn't work well because there can be x ranges
          where the y value is constant at the max value and other 
          places where the max is only reached for a single x value.
          When this happens the single value gets too much of a say
          in the defuzzified value.
          
              /------\                   /\
             /        \                 /  \
          --/          \---------------/    \-------------
                  ^       ^
                  |       |
                  |       | gives this as the mean of maximum
                  | this is more reasonable???
                  
         Centre of gravity is likely more useful
 ********************************************************************/
 
static double maximum_defuzzification(factPtr)
  struct fact *factPtr;  
{
    int i,  num, count;
    struct fuzzy_value *fv;
    
       
    double result,  xmin, xmax;
    double maxy, sum;

    double *fsx, *fsy; /* ptrs to fuzzy set x and y arrays */

    fv = (struct fuzzy_value *)
              ValueToFuzzyValue((factPtr->theProposition.theFields[0].value));

    /******************************************
        Find Mean of Maxima
     ******************************************/

    maxy = 0.0;
    num = fv->n;
    
    fsx = fv->x;
    fsy = fv->y;
    
    for (i=0; i < num; i++ )
      {
        if ( fsy[i] > maxy )
            maxy = fsy[i];
      }

    xmin = factPtr->whichDeftemplate->fuzzyTemplate->u_ptr->from;
    xmax = factPtr->whichDeftemplate->fuzzyTemplate->u_ptr->to;
    
    count = 0;
    sum = 0.0;

    /* Check for a zero max value or a single point in the set */
    if (maxy == 0.0 || num == 1)
        result = (xmax - xmin) * 0.5 + xmin;
        
    else /* Set has at least two points */
      {    
        /* check for maximum at beginning of open-ended set */
        if ( fsy[0] == maxy)
          {
            sum += xmin;
            count++;
            if ( fsx[0] != xmin && fsy[1] != maxy )
              {
                sum += fsx[0];
                count++;
              }
          }
          
        for ( i = 1; i < num - 1; i++ )
          {
            if ( fsy[i] == maxy )
              {
                if ( fsy[i-1] != maxy || fsy[i+1] != maxy )
                  {
                    sum += fsx[i];
                    count++;
                  }
              }
          }
          
        /* check for maximum at end of open-ended set */
        if ( fsy[num-1] == maxy )
          {
            if ( fsx[num-1] != xmax && fsy[num-2] != maxy )
              {
                sum += fsx[num-1];
                count++;
              }
            sum += xmax;
            count++;
          }
        
        result = sum/count;
      }
    
   return( result );
}






#endif  /* FUZZY_DEFTEMPLATES */
