  /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 6.00  05/12/93            */
   /*                                                     */
   /*             CERTAINTY FACTORS MODULE                */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*      Bob Orchard (NRCC - Nat'l Research Council of Canada)*/
/*                  (Fuzzy reasoning extensions)             */
/*                  (certainty factors for facts and rules)  */
/*                  (extensions to run command)              */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _CFDEF_SOURCE_

#include "setup.h"


#if CERTAINTY_FACTORS



 
#include "cfdef.h"

#if FUZZY_DEFTEMPLATES
#include "fuzzyutl.h"
#include "fuzzypsr.h"
#include "fuzzyrhs.h"
#include "fuzzyval.h"
#endif  /* FUZZY_DEFTEMPLATES */

#include "prntutil.h"
#include "argacces.h"
#include "engine.h"
#include "router.h"
#include "symbol.h"
#include "extnfunc.h"
#include "exprnpsr.h"
#include "evaluatn.h"
#include "match.h"
#include "factmngr.h"


#include <math.h>
#include <stdio.h>
#define _CLIPS_STDIO_





/******************************************************************
    Local Global Variable Declarations
 ******************************************************************/

   globle double                Threshold_CF;
   

/******************************************************************
    Local Internal Function Declarations
 ******************************************************************/
 
#if ANSI_COMPILER
   static struct fact *getFactPtr(struct expr *theArgument, char *functionName);
#else
   static struct fact *getFactPtr();
#endif


globle VOID InitializeCF()
{
 /* Variables and functions for CF and threshold value of CF */
 Threshold_CF = 0.0;   /* by default set this to 0.0 */
 
#if ! RUN_TIME  
 DefineFunction2("get-cf", 'd', PTIF getcf, "getcf", "11z");
 DefineFunction2("threshold", 'v', PTIF threshold, "threshold", "11n");
 DefineFunction2("get-threshold", 'd', PTIF get_threshold, "get_threshold", "00");
#endif
}




/***********************************************************************
    FUNCTIONS FOR PARSING CERTAINTY FACTORS
 ***********************************************************************/

#if ! RUN_TIME

/****************************************************************/
/* ParseDeclareUncertainty     :                                */
/* Parses uncertainty factor for Declare of Rule                */
/*                                                              */
/****************************************************************/
globle struct expr *ParseDeclareUncertainty(readSource,ruleName,error,cfVALUE)
  char *readSource;
  char *ruleName;
  int *error;
  double *cfVALUE;
  {
   double cf;
   DATA_OBJECT cfValueDO;
   struct expr *cfExpression;

   /*======================================*/
   /* Get the certainty factor expression. */
   /*======================================*/

   SavePPBuffer(" ");

   cfExpression = ParseAtomOrExpression(readSource,NULL);
   if (cfExpression == NULL)
     {
      *error = CLIPS_TRUE;
      *cfVALUE = 1.0;
      return(NULL);
     }

   /*=====================================================================*/
   /* Evaluate the expression and determine if it is an integer or float. */
   /*=====================================================================*/

   SetEvaluationError(CLIPS_FALSE);
   if (EvaluateExpression(cfExpression,&cfValueDO))
     {
      cfInformationError(ruleName);
      *error = CLIPS_TRUE;
      *cfVALUE = 1.0;
      return(cfExpression);
     }

   if ( cfValueDO.type != INTEGER && cfValueDO.type != FLOAT)
     {
      cfNonNumberError();
      *error = CLIPS_TRUE;
      *cfVALUE = 1.0;
      return(cfExpression);
     }
     
   /*==========================================*/
   /* The expression is Integer or Float,      */
   /* check range (0 to 1) and if              */
   /* OK then set the value in cfVALUE         */
   /*==========================================*/

   cf = (cfValueDO.type == INTEGER) ? 
                   (double) ValueToLong(cfValueDO.value) :
                                ValueToDouble(cfValueDO.value);
       
   if ((cf > 1.0) || (cf < 0.0))
     {
       cfRangeError();
       *error = CLIPS_TRUE;
       *cfVALUE = 1.0;
     }
   else
       *cfVALUE = cf;
   
   return(cfExpression);
}


#endif /*  ! RUN_TIME  */


/*****************************************************************/
/* printCF: prints certainty factor                              */
/*****************************************************************/  
globle VOID printCF(logicalName,cf)
  char *logicalName;
  double cf;
  {
   char printSpace[20];
   
   sprintf(printSpace," CF %.2f ",cf);
   PrintCLIPS(logicalName,printSpace);
   
  }
  
  

/***********************************************************************
    FUNCTIONS FOR COMPUTING CF'S OF RHS
 ***********************************************************************/

#if FUZZY_DEFTEMPLATES

/* Similarity calcs only needed if there are fuzzy facts allowed */

/******************************************************************/
/* POSSIBILITY: possibility measure of two fuzzy sets             */
/*                                                                */
/* p(f1,f2) = max(min(u  (x),u  (x))                              */ 
/*             x       f1     f2                                  */
/******************************************************************/
globle double possibility(f1,f2)
  struct fuzzy_value *f1, *f2;
  {        
    return( max_of_min (  f1->x, f1->y, f1->n, f2->x, f2->y, f2->n) );
  }



/********************************************************************/
/* NECESSITY; necessity measure of two fuzzy sets                   */
/*                   _                                              */
/* n(f1,f2) = 1 - p(f1,f2)                                          */
/********************************************************************/
globle double necessity(f1,f2)
  struct fuzzy_value *f1, *f2;
  {
   struct fuzzy_value *fc;
   double nc;
   
   fc = CopyFuzzyValue(f1);
   fcompliment(fc);
   
   nc = 1.0 - possibility(fc,f2);
   rtnFuzzyValue(fc);
   
   return(nc);
  }

/*****************************************************************/
/* SIMILARITY: similarity measure of two fuzzy sets              */
/* m = if    n(f1,f2) > 0.5                                      */
/*     then  p(f1,f2)                                            */
/*     else  (n(f1,f2) + 0.5) * p(f1,f2)                         */       
/*****************************************************************/
globle double similarity(f1,f2)
  struct fuzzy_value *f1, *f2;
  {
   double nec, poss;

   nec = necessity(f1,f2);
   poss = possibility(f1,f2);
   
   if (nec > 0.5)
     return( poss );
   else
     return( (nec + 0.5) * poss );
     
  }

#endif  /* FUZZY_DEFTEMPLATES */


/****************************************************************
    computeStdConclCF: computes certainty factor for RHS for use
                       by thresholding and for use in calculating 
                       CF for fuzzy facts or crisp facts in 
                       CRISP_ rules (ie. no fuzzy patterns on LHS)            

    Given a rule of the form:

    If A1 and A2 and ... and An then C

    This function returns the certainty factor of the conclusion:
    CFconc = min (CFf1, CFf2, ..., CFfn) * CFrule
    
              - where CFfi are the CFs of the facts matching the
                patterns on the LHS

 ****************************************************************/
globle double computeStdConclCF(ruleCF, binds)
  double ruleCF;
  struct partialMatch *binds;
{
   double StdCF;
   struct genericMatch *antecedent_binds;
   double fact_cf;
   struct fact *tmpFact;
   int i;

   /* initialize StdCF to the maximum value -- 1.0 */
   StdCF = 1.0; 

   /* for each pattern in the list of patterns that matched facts 
      find the minimum CF
   */
   
   antecedent_binds = &(binds->binds[0]); 

   for (i=0; i<binds->bcount; i++)
     {
       /* find the fact that matched a pattern */
       tmpFact = (struct fact *)(antecedent_binds[i].gm.theMatch->matchingItem);

       if (tmpFact == NULL) /* will be NULL for NOT matches */
          continue;         /* just treat as 1.0 CF */

       /* only facts have CF's associated with them -- so if 
          anything else has been matched (e.g. Object Instance)
          then just treat as if it had a CF of 1.0)
          At some later time we may want to associate CFs with
          Object Instances and this code will change.
       */
       if (tmpFact->factHeader.theInfo->base.type != FACT_ADDRESS)
          continue;

       fact_cf = tmpFact->factCF;
                    
       if (fact_cf < StdCF) /* keep the minimum */
               StdCF = fact_cf;

     } /* end   for (i=0; i<binds->count; i++)  */
  
  return( StdCF * ruleCF );
}

 
/****************************************************************
    computeFuzzyCrispConclCF: computes certainty factors for RHS 
                              for use in calculating CF for crisp
                              facts in FUZZY_CRISP rules             

    Given a rule of the form:

    If A1 and A2 and ... and An then C  [C is crsip conclusion]

    This function returns the certainty factor of the conclusion:
    CFconc = min (CFf1, CFf2, ..., CFfn) * CFrule
    
    where - CFfi is the certainty of the fact matching pattern i
            if the fact is CRISP and CFfi is the certainty of the
            fact matching pattern i * the similarity of the fuzzy
            set of the pattern and the fuzzy set of the matching
            fact if the fact is FUZZY

 ****************************************************************/
#if FUZZY_DEFTEMPLATES

globle double computeFuzzyCrispConclCF(theRule, binds)
  struct defrule *theRule;
  struct partialMatch *binds;
{
   double FuzzyCrispCF;
   struct fuzzy_value *fact_fv, *antecedent_fv;
   struct genericMatch *antecedent_binds;
   double fact_cf;
   struct fact *tmpFact;
   int i;
   
   /* initialize FuzzyCrispCF to the maximum value -- 1.0 */
   FuzzyCrispCF = 1.0;
 
   /* for each pattern in the list of patterns that matched facts 
      find the minimum CF -- if fact is a fuzzy fact and
      FuzzyCrispCF_needed is TRUE then must also calculate
      similarity and multiply by fact's CF
   */
   
   antecedent_binds = &(binds->binds[0]); 

   for (i=0; i<binds->bcount; i++)
     {
       /* find the fact that matched a pattern */
       tmpFact = (struct fact *)(antecedent_binds[i].gm.theMatch->matchingItem);

       if (tmpFact == NULL) /* will be NULL for NOT matches */
          continue;         /* just treat as 1.0 CF */

       /* only facts have CF's associated with them -- so if 
          anything else has been matched (e.g. Object Instance)
          then just treat as if it had a CF of 1.0)
          At some later time we may want to associate CFs with
          Object Instances and this code will change.
       */
       if (tmpFact->factHeader.theInfo->base.type != FACT_ADDRESS)
          continue;

       fact_cf = tmpFact->factCF;
                    
       /* theRule->pattern_fv_arrayPtr is ptr to array of fuzzy value hashNode 
          ptrs. Each of these ptrs (if NON-NULL) points to a fuzzy value 
          connected to a fuzzy pattern in the rule's LHS - NOTE!! if there are
          no fuzzy patterns on the LHS of the rule then there are NO
          entries in this array and numberOfPatterns == 0
       */
           
       if (theRule->numberOfPatterns != 0)
         {
          VOID *patFvArrayPtri = (VOID *)(theRule->pattern_fv_arrayPtr)[i];

          if (patFvArrayPtri == NULL)
              antecedent_fv = NULL;
          else
             antecedent_fv = (struct fuzzy_value *) ValueToFuzzyValue(patFvArrayPtri); 
           /* check if antecedent is fuzzy (NULL if not fuzzy or if ? wildcard
              pattern for fuzzy variable)   */
           if (antecedent_fv != NULL)
             {
               /* find fuzzy set of the fact which matched the pattern */
               fact_fv = (struct fuzzy_value *)
                         ValueToFuzzyValue((tmpFact->theProposition.theFields[0].value));
               fact_cf = similarity(antecedent_fv,fact_fv) * fact_cf;
             }
         }

       if (fact_cf < FuzzyCrispCF) /* keep the minimum */
          FuzzyCrispCF = fact_cf;

     } /* end   for (i=0; i<binds->count; i++)  */

  return( FuzzyCrispCF * theRule->CF );
}

#endif /* FUZZY_DEFTEMPLATES */
 

/********************************************************************
    changeCFofNewFact()

    If the new fact is being asserted from the command line, then
    the cf is not altered (no rule is executing).

    If the new fact is being asserted from the RHS of a rule, then
    the cf of the new fact is multiplied by the cf calculated for
    the whole rule -- depends on whether the asserted fact is Crisp
    or fuzzy -- if crisp and LHS had Fuzzy patterns then use 
    FuzzyCrispConcludingCF of Executing Rule else use
    StdConcludingCF of Executing Rule (both values are stored in the 
    current rule activation - if their value is -1.0 then they have
    not yet been calculated and should now be calculated )
 ********************************************************************/
 
globle VOID changeCFofNewFact(newFact)
  struct fact *newFact;
  {
    double ConcludingCFofExecutingRule;

    if ((ExecutingRule != NULL) &&
        (ExecutingRule->executing)) 
      {
        /* Fuzzy facts always use StdConcludingCF -- Crisp Facts do too
           when NO fuzzy facts in rule
        */
#if FUZZY_DEFTEMPLATES
        if (newFact->whichDeftemplate->fuzzyTemplate != NULL)
           {
#endif
             ConcludingCFofExecutingRule = theCurrentActivation->StdConcludingCF;
             if (ConcludingCFofExecutingRule < 0) /* not yet calculated if -1.0 */
                theCurrentActivation->StdConcludingCF
                    = ConcludingCFofExecutingRule
                    = computeStdConclCF(theCurrentActivation->CF, theCurrentActivation->basis);
#if FUZZY_DEFTEMPLATES
           }
        else
           { 
             /* Crisp facts use StdConcludingCF if LHS of Rule is Crisp and
                use FuzzyCrispConcludingCF when fuzzy patterns on LHS of rule
             */
             if (ExecutingRule->lhsRuleType == FUZZY_LHS)
                {
                  ConcludingCFofExecutingRule = theCurrentActivation->FuzzyCrispConcludingCF;
                  if (ConcludingCFofExecutingRule < 0) /* not yet calculated if -1.0 */
                     theCurrentActivation->FuzzyCrispConcludingCF
                         = ConcludingCFofExecutingRule
                         = computeFuzzyCrispConclCF(ExecutingRule, theCurrentActivation->basis);
                }
             else
                {
                  ConcludingCFofExecutingRule = theCurrentActivation->StdConcludingCF;
                  if (ConcludingCFofExecutingRule < 0) /* not yet calculated if -1.0 */
                     theCurrentActivation->StdConcludingCF
                         = ConcludingCFofExecutingRule
                         = computeStdConclCF(theCurrentActivation->CF, theCurrentActivation->basis);
                }
           }
#endif
        newFact->factCF = newFact->factCF * ConcludingCFofExecutingRule;
      }

  }

/******************************************************************
    Functions for accessing cf of a fact
 ******************************************************************/


/************************************************************
    getFactPtr():                                            

    given a ptr to an argument of a function that is expected
    to be a fact address or a fact index get a ptr to the fact
    
    returns a ptr to a fact or NULL if error occurred
************************************************************/
static struct fact *getFactPtr(theArgument, functionName)

  struct expr *theArgument;
  char *functionName;
{
   long int factIndex;
   int found_fact;
   DATA_OBJECT  theResult;
   struct fact *factPtr;
   
   EvaluateExpression(theArgument,&theResult);
      
   if ((theResult.type == INTEGER) || (theResult.type == FACT_ADDRESS))
     {
       if (theArgument->type == INTEGER)
         { 
           factIndex = ValueToLong(theResult.value); 
           if (factIndex < 0)
             {            
               ExpectedTypeError1(functionName,1,"fact-index must be positive");
               return(NULL);
             }
           found_fact = CLIPS_FALSE;
           factPtr = (struct fact *) GetNextFact(NULL);
           while (factPtr != NULL)
             {
               if (factPtr->factIndex == factIndex)
                 {
                   found_fact = CLIPS_TRUE;
                   break;
                 }
               factPtr = factPtr->nextFact;
             }
           
           if (found_fact == CLIPS_FALSE)
             {
               char tempBuffer[20];
               sprintf(tempBuffer,"f-%ld",factIndex);
               CantFindItemErrorMessage("fact",tempBuffer);
               return(NULL);
             }
          }
        else
          { /* arg type is fact address */
            factPtr = (struct fact *) theResult.value; 
          }     
        return( factPtr );
     }
   
   ExpectedTypeError1(functionName,1,"fact-index or fact-address");
   return( NULL );
}


/************************************************************
    getcf():                                            

    returns the certainty factor of a single fact in
    NUMBER format; if the certainty factor is a TFN,
    the peak value is returned.
/************************************************************/
globle double getcf()
  {
    struct fact *factPtr;
    struct expr *theArgument;

    if (ArgCountCheck("get-cf",EXACTLY,1) != -1)
      {     
        theArgument = GetFirstArgument();

        if (theArgument != NULL)
          {
            factPtr = getFactPtr(theArgument, "get-cf");
       
            if (factPtr != NULL)
              {
                return(factPtr->factCF);
              }
          }
      }
        
    SetEvaluationError(CLIPS_TRUE);
    return(0.0);
  }



/*******************************************************************
    Functions for setting and accessing the threshold cf value
 *******************************************************************/

/*******************************************************************
    threshold()
    
    Sets the threshold cf to desired CRISP value and changes
    threshold_on to TRUE.
 *******************************************************************/
globle VOID threshold()
  {
    DATA_OBJECT theArgument;
    double theThreshold;

    if (ArgCountCheck("threshold",EXACTLY,1) != -1)
      {     
        if (ArgTypeCheck("get-cf<",1,INTEGER_OR_FLOAT,&theArgument) != 0)
          {
            if (GetType(theArgument) == INTEGER)
              {
                theThreshold = (double)DOToLong(theArgument);
              }
            else
              {
                theThreshold = DOToDouble(theArgument);
              }
            if (theThreshold < 0.0 || theThreshold > 1.0)
              {
                cfRangeError();
              }
            else
              {
                Threshold_CF = theThreshold;
                return;
              }
          }
      }
        
    SetEvaluationError(CLIPS_TRUE);
  }


/********************************************************************
    get_threshold()

    Returns the CRISP threshold value.
 ********************************************************************/
globle double get_threshold()
{
    if (ArgCountCheck("get-threshold",EXACTLY,0) == -1)
      {     
        SetEvaluationError(CLIPS_TRUE);
      }
        
    return( Threshold_CF );
}




/*******************************************************************
    FUNCTIONS FOR CHANGING CF'S DUE TO
    GLOBAL CONTRIBUTION
 *******************************************************************/

/*******************************************************************
    changeCFofExistingFact(newFact, oldFact)

    where: oldFact is the existing fact on the fact list
           newFact is the new version of fact2 being asserted

    The purpose of this function is to update the certainty factor
    value of the existing fact when an identical fact is asserted
    (global contribution).  

    The new certainty factor is the maximum cf of the two facts.


 *******************************************************************/
globle VOID changeCFofExistingFact(newFact,oldFact)
  struct fact *newFact;
  struct fact *oldFact;
  {
    if (oldFact->factCF < newFact->factCF)
      {
        oldFact->factCF = newFact->factCF;
           
        /* fact has changed - set flag to say so */
        SetFactListChanged(CLIPS_TRUE);

#if DEBUGGING_FUNCTIONS
        if (oldFact->whichDeftemplate->watch)
          {
            PrintCLIPS(WTRACE,"~CF ");
            PrintFactWithIdentifier(WTRACE,oldFact);
            PrintCLIPS(WTRACE,"\n");
          }
#endif
      }
  }






/*******************************************************************
    FUNCTIONS FOR Printing Error Messages for CFs
    
 *******************************************************************/

globle VOID   cfInformationError(name)

  char *name;
{
   PrintErrorID("Certainty Factors ",901,CLIPS_TRUE);
   PrintCLIPS(WERROR,"This error occurred while evaluating a Certainty Factor");
   if (name != NULL)
     {
      PrintCLIPS(WERROR," for rule ");
      PrintCLIPS(WERROR,name);
     }
   PrintCLIPS(WERROR,".\n");
}


globle VOID   cfRangeError()

{
   PrintErrorID("Certainty Factors ",902,CLIPS_TRUE);
   PrintCLIPS(WERROR,"Expected Value in Range 0.0 to 1.0");
   PrintCLIPS(WERROR,".\n");
}


globle VOID   cfNonNumberError()

{
   PrintErrorID("Certainty Factors ",903,CLIPS_TRUE);
   PrintCLIPS(WERROR,"Expected Integer or Float Value");
   PrintCLIPS(WERROR,".\n");
}






#endif  /* CERTAINTY_FACTORS */
