   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                  SYSTEM I/O MODULE                  */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Brian L. Donnell                                     */
/*      Gary D. Riley                                        */
/*      Bebe Ly                                              */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _SYSIO_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"

#include "utility.h"
#include "router.h"
#include "evaluatn.h"
#include "facts.h"
#include "scanner.h"
#include "constant.h"
#include "clipsmem.h"
#include "commline.h"

#define FORMAT_MAX 512
#define FLAG_MAX    80

/***************************************************/
/* GLOBAL AND LOCAL INTERNAL FUNCTION DEFINITIONS  */
/***************************************************/

#if ANSI_COMPILER  
   VOID                    IOFunctionDefinitions(void);
   VOID                    PrintoutFunction(void);
   VOID                    ReadFunction(DATA_OBJECT_PTR);
   int                     OpenFunction(void); 
   int                     CloseFunction(void);
   VOID                   *FormatFunction(void);
   VOID                    ReadlineFunction(DATA_OBJECT_PTR);
#if EXT_IO
   static char            *ControlStringCheck(int);
   static char             FindFormatFlag(char *,int *,char *,int *);
   static int              PrintFormatFlag (char *,int,int,char *,int);
   static char            *FillBuffer(char *,int *,int *);
#endif
#else
   VOID                    IOFunctionDefinitions();
   VOID                    PrintoutFunction();
   VOID                    ReadFunction(); 
   int                     OpenFunction();
   int                     CloseFunction();
   VOID                   *FormatFunction();
   VOID                    ReadlineFunction();
#if EXT_IO
   static char            *ControlStringCheck();
   static char             FindFormatFlag();
   static int              PrintFormatFlag ();
   static char            *FillBuffer();
#endif
#endif

#if ! RUN_TIME
/********************************************************************/
/* IOFunctionDefinitions                                            */
/********************************************************************/
globle VOID IOFunctionDefinitions()
  {
#if BASIC_IO
   DefineFunction("printout",   'v', PTIF PrintoutFunction, "PrintoutFunction");
   DefineFunction("fprintout",  'v', PTIF PrintoutFunction, "PrintoutFunction");
   DefineFunction("read",       'u', PTIF ReadFunction,  "ReadFunction");
   DefineFunction("open",       'b', OpenFunction,  "OpenFunction");
   DefineFunction("close",      'b', CloseFunction, "CloseFunction");
#endif

#if EXT_IO
   DefineFunction("format",     's', PTIF FormatFunction, "FormatFunction");
   DefineFunction("readline",   'u', PTIF ReadlineFunction,  "ReadlineFunction");
#endif
  }
#endif

#if BASIC_IO

/*********************************************************************/
/* PrintoutFunction command                                          */
/*********************************************************************/
globle VOID PrintoutFunction()
  {
   char *dummyid;
   int i, argc;
   DATA_OBJECT argPtr;
 
   /*==========================================================*/
   /* Printout requires at least one argument: a logical name. */
   /*==========================================================*/
 
   if ((argc = ArgCountCheck("printout",AT_LEAST,1)) == -1) return;
 
   dummyid = GetLogicalName(1,"stdout");
   if (dummyid == NULL)
     {
      PrintCLIPS(WERROR,"Illegal logical name used for printout function\n");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return;
     }

   if (QueryRouters(dummyid) == CLIPS_FALSE)
     {
      UnrecognizedRouterMessage(dummyid);
      return;
     }

   for (i = 2; i <= argc; i++)
     {
      RtnUnknown(i,&argPtr);
      switch(GetType(argPtr))
        {
         case SYMBOL:
           if (strcmp(DOToString(argPtr),"crlf") == 0)
             { PrintCLIPS(dummyid,"\n"); }               
           else if (strcmp(DOToString(argPtr),"tab") == 0)
             { PrintCLIPS(dummyid,"\t"); }                
           else if (strcmp(DOToString(argPtr),"vtab") == 0)
             { PrintCLIPS(dummyid,"\v"); }               
           else if (strcmp(DOToString(argPtr),"t") == 0)
             { PrintCLIPS(dummyid,"\n"); }
           else
             { PrintCLIPS(dummyid,DOToString(argPtr)); }
           break;
           
         case STRING:
           PrintCLIPS(dummyid,DOToString(argPtr));
           break;
           
         default:
           PrintDataObject(dummyid,&argPtr);
           break;
        }
     }
  }

/****************************************/
/* ReadFunction                                 */
/****************************************/
globle VOID ReadFunction(read_value)
  DATA_OBJECT_PTR read_value;
  {
   struct token inp_tkn;
   int arg_no;
   char *dummyid = NULL;
   char *read_str;
   int max_char;
   int inchar;
   
   /*===============================================*/
   /* Check for an appropriate number of arguments. */
   /*===============================================*/

   if ((arg_no = ArgCountCheck("read",NO_MORE_THAN,1)) == -1)
     {
      read_value->type = STRING;
      read_value->value = (VOID *) AddSymbol("*** READ ERROR ***");
      return; 
     }
     
   /*======================================================*/
   /* Determine the logical name from which input is read. */
   /*======================================================*/

   if (arg_no == 0) 
     { dummyid = "stdin"; }
   else if (arg_no == 1) 
     {
      dummyid = GetLogicalName(1,"stdin");
      if (dummyid == NULL)
        {
         PrintCLIPS(WERROR,"Illegal logical name used for read function\n");
         SetHaltExecution(CLIPS_TRUE);
         SetEvaluationError(CLIPS_TRUE);
         read_value->type = STRING;
         read_value->value = (VOID *) AddSymbol("*** READ ERROR ***");
         return;
        }
     }

   if (QueryRouters(dummyid) == CLIPS_FALSE) 
     {
      UnrecognizedRouterMessage(dummyid);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      read_value->type = STRING;
      read_value->value = (VOID *) AddSymbol("*** READ ERROR ***");
      return;
     }

   /*====================================================*/
   /* Collect input into string if read source is stdin, */
   /* else just get token.                               */
   /*====================================================*/
   
   if (strcmp(dummyid,"stdin") == 0)
     {
      inp_tkn.type = STOP;
      while (inp_tkn.type == STOP)
        {
         read_str = NULL;
         max_char = CLIPSInputCount = 0;
         inchar = GetcCLIPS("stdin");
         while ((inchar != '\n') && (inchar != '\r') && (inchar != EOF) && 
                (!GetHaltExecution()))
           {
            read_str = ExpandStringWithChar(inchar,read_str,&CLIPSInputCount,
                                              &max_char,max_char + 80);
            inchar = GetcCLIPS("stdin");
           }
          
         OpenStringSource("read",read_str,0);
         GetToken("read",&inp_tkn);
         CloseStringSource("read");
         if (max_char > 0) rm(read_str,max_char);
         
         if (GetHaltExecution())
           {
            inp_tkn.type = STRING;
            inp_tkn.value = (VOID *) AddSymbol("*** READ ERROR ***");
           }
         
         if ((inp_tkn.type == STOP) && (inchar == EOF))
           {
            inp_tkn.type = SYMBOL;
            inp_tkn.value = (VOID *) AddSymbol("EOF");
           }
        }
     }
   else
     { GetToken(dummyid,&inp_tkn); }

   CLIPSInputCount = -1;

   /*=======================*/
   /* Process return value. */
   /*=======================*/
   
   read_value->type = inp_tkn.type;
   if ((inp_tkn.type == FLOAT) || (inp_tkn.type == STRING) || 
#if OBJECT_SYSTEM
       (inp_tkn.type == INSTANCE_NAME) ||
#endif
       (inp_tkn.type == SYMBOL) || (inp_tkn.type == INTEGER))
     { read_value->value = inp_tkn.value; }
   else if (inp_tkn.type == STOP)
     {
      read_value->type = SYMBOL;
      read_value->value = (VOID *) AddSymbol("EOF");
     }
   else if (inp_tkn.type == KUNKNOWN)
     {
      read_value->type = STRING;
      read_value->value = (VOID *) AddSymbol("*** READ ERROR ***");
     }
   else
     {
      read_value->type = STRING;
      read_value->value = (VOID *) AddSymbol(inp_tkn.print_rep);
     }

   return;
  }

/*###################################*/
/*# FILE SYSTEM INTERFACE FUNCTIONS #*/
/*###################################*/

/**************************************************************/
/* OpenFunction: This function opens a file named by the user */
/*   and identifies it with a character string tag specified  */
/*   by the user.  This function returns a non-zero value if  */
/*   the file was successfully opened.                        */
/**************************************************************/
globle int OpenFunction() 
  {
   int arg_no, status; 
   char *newfilename, *newfileid, *newmode = NULL;
   DATA_OBJECT arg_ptr;

   /*======================================*/
   /* Check for valid number of arguments. */
   /*======================================*/

   if ((arg_no = ArgRangeCheck("open",2,3)) == -1) return(0);
   
   /*==========================*/
   /* Check for the file name. */
   /*==========================*/
   
   if ((newfilename = GetFileName("open",1)) == NULL) return(0);
   
   /*=============================*/
   /* Check for the logical name. */
   /*=============================*/
   
   newfileid = GetLogicalName(2,NULL);
   if (newfileid == NULL)
     {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      PrintCLIPS(WERROR,"Illegal logical name used for open function\n");
      return(0);
     }
   
   if (FindFile(newfileid))
     {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      PrintCLIPS(WERROR,"Logical name ");
      PrintCLIPS(WERROR,newfileid);
      PrintCLIPS(WERROR," already in use.\n");
      return(0);
     }
     
   /*===================================*/
   /* Check for valid file access mode. */
   /*===================================*/

   if (arg_no == 2)
     { newmode = "r"; }
   else if (arg_no == 3)
     { 
      if (ArgTypeCheck("open",3,STRING,&arg_ptr) == CLIPS_FALSE) return(0);
      newmode = DOToString(arg_ptr);
     }
     
   if ((strcmp(newmode,"r") != 0) &&
       (strcmp(newmode,"r+") != 0) &&
       (strcmp(newmode,"w") != 0) &&
       (strcmp(newmode,"a") != 0))
     {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      ExpectedTypeError("open",3,"string with value \"r\", \"r+\", \"w\", or \"a\"");
      return(0); 
     }                                     

   /*======================================================*/
   /* Open named file and store it with named tag on list. */
   /*======================================================*/

/* Mod:
 >In the CLIPS code you will have to rename the procedure
 >OpenFile() as it is already used by Windows.  I use
 >ClipsOpenFile().  ROUTER.C, ROUTER.H, SYSIO.C,
*/
   status = CLIPSOpenFile(newfilename,newmode,newfileid);
   return(status);
  }


/***********************************************************************/
/* CloseFunction:  This function closes the file stream with the file  */
/*   id specified by the user, if such a stream exists.  This function */
/*   returns a non-zero value if the file was successfully closed.     */
/***********************************************************************/
globle int CloseFunction()
  {
   int arg_no;
   char *fileid;
                                                                    
   /*======================================================*/
   /* Check for valid number of arguments and assignments. */
   /*======================================================*/

   if ((arg_no = ArgCountCheck("close",NO_MORE_THAN,1)) == -1) return(0);

   if (arg_no == 0) return(CloseAllFiles());

   fileid = GetLogicalName(1,NULL);
   if (fileid == NULL)
     {
      PrintCLIPS(WERROR,"Illegal logical name used for close function\n");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(0);
     }

   return(CloseFile(fileid));
  }    


#endif

#if EXT_IO
  
/**********************************************************/
/* FormatFunction:                                        */
/**********************************************************/
globle VOID *FormatFunction()
  {    
   int num_a, start_pos;
   char *form_str, *logicalName;
   char f_type;
   int  f_cur_arg = 3;
   int form_pos = 0;
   char buffer[FORMAT_MAX];
   char percentBuffer[FLAG_MAX];
   char *fstr = NULL;
   int fmax = 0, fpos = 0;
   VOID *hptr;
   int longFound;

   /*======================================*/
   /* Set default return value for errors. */
   /*======================================*/
   
   hptr = AddSymbol("");
   
   /*=========================================*/
   /* Format requires at least two arguments: */
   /* a logical name and a format string.     */
   /*=========================================*/

   if ((num_a = ArgCountCheck("format",AT_LEAST,2)) == -1) 
     { return(hptr); }

   /*========================================*/
   /* First argument must be a logical name. */
   /*========================================*/
     
   if ((logicalName = GetLogicalName(1,"stdout")) == NULL)
     {
      PrintCLIPS(WERROR,"Illegal logical name used for format function\n");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(hptr);
     }

   if (strcmp(logicalName,"nil") == 0)
     { /* do nothing */ }
   else if (QueryRouters(logicalName) == CLIPS_FALSE)
     {
      UnrecognizedRouterMessage(logicalName);
      return(hptr);
     }

   /*=====================================================*/
   /* Second argument must be a string.  The appropriate  */
   /* number of arguments specified by the string must be */
   /* present in the argument list.                       */
   /*=====================================================*/

   if ((form_str = ControlStringCheck (num_a)) == NULL)
     { return (hptr); }
     
   /*==============================================*/
   /* Locate a string of 80 character for scanning */
   /* sub_string from control_string               */
   /*==============================================*/

   /* Scanning and print the format */

   while (form_str[form_pos] != '\0')
     {
      if (form_str[form_pos] != '%')
        {
         start_pos = form_pos;
         while ((form_str[form_pos] != '%') && 
                (form_str[form_pos] != '\0') &&
                ((form_pos - start_pos) < FLAG_MAX))
           { form_pos++; }
         fstr = AppendNToString(&form_str[start_pos],fstr,form_pos-start_pos,&fpos,&fmax);
        }
      else
        {
         start_pos = form_pos;
         form_pos++;
         f_type = FindFormatFlag(form_str,&form_pos,buffer,&longFound);
         if (f_type != ' ')
           {
            strncpy(percentBuffer,&form_str[start_pos],form_pos-start_pos);
            percentBuffer[form_pos-start_pos] = EOS;
            if (PrintFormatFlag(percentBuffer,f_cur_arg,f_type,buffer,longFound) == 0)
              {
               if (fstr != NULL) rm(fstr,fmax);
               return (hptr); 
              }
            fstr = AppendToString(buffer,fstr,&fpos,&fmax);
            if (fstr == NULL) return(hptr);
            f_cur_arg++;
           }
         else
           {
            fstr = AppendToString(buffer,fstr,&fpos,&fmax);
            if (fstr == NULL) return(hptr);
           }
        }
     }
     
   if (fstr != NULL)
     {
      hptr = AddSymbol(fstr);
      if (strcmp(logicalName,"nil") != 0) PrintCLIPS(logicalName,fstr);
      rm(fstr,fmax);
     }
   else
     { hptr = AddSymbol(""); }

   return(hptr);
  }
      
/*********************************************************************/
/* ControlStringCheck:  Checks the 2nd parameter which is the format */
/*   control string to see if there are enough matching arguments.   */
/*********************************************************************/
static char *ControlStringCheck(num_a)
  int num_a;
  {
   DATA_OBJECT t_ptr; 
   char *str_array;
   char print_buff[10];
   int longFound;

   int i,per_count;
       
   if (ArgTypeCheck("format",2,STRING,&t_ptr) == CLIPS_FALSE) return(NULL);

   per_count = 0;
   str_array = ValueToString(t_ptr.value);
   for (i= 0 ; str_array[i] != '\0' ; )
     {         
      if (str_array[i] == '%')
        {
         i++;
         if (FindFormatFlag(str_array,&i,print_buff,&longFound) != ' ')
           { per_count++; }
        }
      else 
        { i++; }
     }
     
   if (per_count != (num_a - 2))    
     {     
      ExpectedCountError("format",EXACTLY,per_count+2);
      return (NULL);
     }

   return(str_array);
  }

/****************************************************************/
/* FindFormatFlag:  This function will scan a sub_string, which */
/*   is the format for an argument from the control_string      */
/****************************************************************/
static char FindFormatFlag(str_array,a,print_buff,longFound)
  char *str_array;
  int *a;
  char *print_buff;
  int *longFound;
  {
   int found_type = CLIPS_FALSE;
   char inchar, f_arg_type;
   int start_pos, copy_pos = 0;

   f_arg_type = ' ';
   *longFound = CLIPS_FALSE;
   
   if (str_array[*a] == 'n')
     {
      sprintf(print_buff,"\n");
      (*a)++;
      return(f_arg_type);
     }
   else if (str_array[*a] == 't')
     {
      sprintf(print_buff,"\t");
      (*a)++;
      return(f_arg_type);
     }
   else if (str_array[*a] == 'v')
     {
      sprintf(print_buff,"\v");
      (*a)++;
      return(f_arg_type);
     }
   else if (str_array[*a] == '%')
     {
      sprintf(print_buff,"%%");
      (*a)++;
      return(f_arg_type);
     }
     
   start_pos = *a;
   print_buff[copy_pos] = '\0';
   while ((str_array[*a] != '%') && 
          (str_array[*a] != '\0') &&
          ((*a - start_pos) < FLAG_MAX))
     {
      inchar = str_array[*a];
      print_buff[copy_pos++] = inchar;
      print_buff[copy_pos] = '\0';
      if ( (found_type == CLIPS_FALSE) &&
           ( (inchar == 'd') ||
             (inchar == 'o') ||
             (inchar == 'x') ||
             (inchar == 'u') ||
             (inchar == 'c') ||
             (inchar == 's') ||
             (inchar == 'e') ||
             (inchar == 'f') ||
             (inchar == 'g') ) )
        {
         found_type = CLIPS_TRUE;
         f_arg_type = inchar;
         if (str_array[(*a) - 1] == 'l') *longFound = CLIPS_TRUE;
         (*a)++;
         return(f_arg_type);
        }
      (*a)++;
     }

   return(f_arg_type);
  }

/**********************************************************************/
/* PrintFormatFlag:  Prints out part of the total format string along */
/*   with the argument for that part of the format string.            */
/**********************************************************************/
static int PrintFormatFlag (f_sub_str,cur_arg,f_arg_type,print_buff,longFound)
  char *f_sub_str;
  int cur_arg;
  int f_arg_type;
  char *print_buff;
  int longFound;
  { 
   DATA_OBJECT t_ptr;
   TYPE arg_type;

   RtnUnknown(cur_arg,&t_ptr);
   arg_type = GetType(t_ptr);

   /*=================*/
   /* String argument */
   /*=================*/
   
   if (f_arg_type == 's')
     {
      if ((arg_type != SYMBOL) && (arg_type != STRING))
        {
         ExpectedTypeError("format",cur_arg,"symbol or string");
         return(0);
        }
      sprintf(print_buff,f_sub_str,ValueToString(t_ptr.value));
     }

   /*====================*/
   /* Character argument */
   /*====================*/

   else if (f_arg_type == 'c')
     {
      if ((arg_type != SYMBOL) && (arg_type != STRING))
        {
         ExpectedTypeError("format",cur_arg,"symbol or string");
         return(0);
        }
      sprintf(print_buff,f_sub_str,(ValueToString(t_ptr.value))[0]);
     }

   /*==================*/
   /* Integer argument */
   /*==================*/

   else if ((f_arg_type == 'd') || (f_arg_type == 'x') || 
            (f_arg_type == 'o') || (f_arg_type == 'u'))
     {
      if ((arg_type != FLOAT) && (arg_type != INTEGER))
        {
         ExpectedTypeError("format",cur_arg,"number");
         return(0);
        }
      if (arg_type == FLOAT)
        { 
         if (longFound)
           { sprintf(print_buff,f_sub_str,(long) ValueToDouble(t_ptr.value)); }
         else
           { sprintf(print_buff,f_sub_str,(int) ValueToDouble(t_ptr.value)); }
        }
      else
        { 
         if (longFound)
           { sprintf(print_buff,f_sub_str,(long) ValueToLong(t_ptr.value)); }
         else
           { sprintf(print_buff,f_sub_str,(int) ValueToLong(t_ptr.value)); }
        }
     }
        
   /*================*/
   /* Float argument */
   /*================*/
  
   else if ((f_arg_type == 'f') || (f_arg_type == 'e') || 
            (f_arg_type == 'g')) 
     {
      if ((arg_type != FLOAT) && (arg_type != INTEGER))
        {
         ExpectedTypeError("format",cur_arg,"number");
         return(0);
        }
      if (arg_type == FLOAT)
        { sprintf(print_buff,f_sub_str,ValueToDouble(t_ptr.value)); }
      else
        { sprintf(print_buff,f_sub_str,(double) ValueToLong(t_ptr.value)); }
     }
     
   /*========*/
   /* ERROR! */
   /*========*/
  
   else
     {
      PrintCLIPS (WERROR," Error in format, the conversion character");
      PrintCLIPS (WERROR," for formatted output is not valid\n");
      return(0);
     }

   return(1);
  }

/****************************************************************/
/* ReadlineFunction:                                                    */
/****************************************************************/
globle VOID ReadlineFunction(rlnval)
  DATA_OBJECT_PTR rlnval;   
  {
   char *buffer;
   int line_max = 0;
   int num;
   
   char *logicalName;
     
   rlnval->type = STRING;
   
   if ((num = ArgCountCheck("readline",NO_MORE_THAN,1)) == -1) 
     { 
      rlnval->value = (VOID *) AddSymbol("*** READ ERROR ***");
      return;
     }
     
   if (num == 0 )
     { logicalName = "stdin"; }
   else
     {
      logicalName = GetLogicalName(1,"stdin");
      if (logicalName == NULL)
        {
         PrintCLIPS(WERROR,"Illegal logical name used for readline function\n");
         SetHaltExecution(CLIPS_TRUE);
         SetEvaluationError(CLIPS_TRUE);
         rlnval->value = (VOID *) AddSymbol("*** READ ERROR ***");
         return;
        }
     } 
      
   if (QueryRouters(logicalName) == CLIPS_FALSE) 
     {
      UnrecognizedRouterMessage(logicalName);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      rlnval->value = (VOID *) AddSymbol("*** READ ERROR ***");
      return;
     }
        
   CLIPSInputCount = 0;
   buffer = FillBuffer(logicalName,&CLIPSInputCount,&line_max);
   CLIPSInputCount = -1;
   
   if (GetHaltExecution())
     { 
      rlnval->value = (VOID *) AddSymbol("*** READ ERROR ***");
      if (buffer != NULL) rm(buffer,(int) sizeof (char) * line_max);
      return;
     }
     
   if (buffer == NULL)
     {
      rlnval->value = (VOID *) AddSymbol("EOF");
      rlnval->type = SYMBOL;
      return;
     }
   
   rlnval->value = (VOID *) AddSymbol(buffer);
   rm(buffer,(int) sizeof (char) * line_max);
   return;
  }

/*****************************************************/
/* FillBuffer                                       */
/*****************************************************/
static char *FillBuffer(logicalName,buf_pos,buf_max)
  char *logicalName;
  int *buf_pos, *buf_max;
  {
   int c;
   char *buf = NULL;
    
   /*================================*/
   /* Read until end of line or eof. */
   /*================================*/

   c = GetcCLIPS(logicalName);

   if (c == EOF)
     { return(NULL); }
       
   /*==================================*/
   /* Grab characters until cr or eof. */
   /*==================================*/
   
   while ((c != '\n') && (c != '\r') && (c != EOF) &&
          (! GetHaltExecution()))
     {
      buf = ExpandStringWithChar(c,buf,buf_pos,buf_max,*buf_max+80);
      c = GetcCLIPS(logicalName);
     } 
      
   /*==================*/
   /* Add closing EOS. */
   /*==================*/
   
   buf = ExpandStringWithChar(EOS,buf,buf_pos,buf_max,*buf_max+80);
   return (buf);
  }

#endif


