   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                  I/O ROUTER MODULE                  */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
   
#define _ROUTER_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"
#include "constant.h"
#include "evaluatn.h"
#include "router.h"
#include "clipsmem.h"
#include "utility.h"
#include "sysdep.h"

#define READ_STRING 0
#define WRITE_STRING 1

/************************************************************************/
/* FILE DESCRIPTOR LIST                                                 */
/*   This data structure houses the nodes which link the file id tags   */
/*   used by CLIPS external I/O functions with the appropriate streams. */
/************************************************************************/
typedef struct filelist
  {
   char *fileid;
   FILE *stream;
   struct filelist *next;
  } filelist;

struct router
  {
   char *name;
   int active;
   int priority;
#if ANSI_COMPILER
   int (*query)(char *);
   int (*printer)(char *,char *);
   int (*exiter)(int);
   int (*charget)(char *);
   int (*charunget)(int,char *);
#else
   int (*query)();
   int (*printer)();
   int (*exiter)();
   int (*charget)();
   int (*charunget)();
#endif
   struct router *next;
  };

struct stringRouter
  {
   char *name;
   char *str;
   int currentPosition;
   int maximumPosition;
   int readWriteType;
   struct stringRouter *next;
  };
  
typedef filelist *fileptr;

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER  
   static int                     FindString(char *);
   static VOID                    PrintString(char *,char *);
   static int                     GetcString(char *);
   static int                     UngetcString(int,char *);
   static struct stringRouter    *FindStringRouter(char *); 
   static VOID                    ExitFile(int);
   static VOID                    PrintFile(char *,char *);
   static int                     GetcFile(char *);
   static VOID                    UngetcFile(int,char *); 
   static int                     QueryRouter(char *,struct router *);
   static int                     CreateReadStringSource(char *,char *,int,int);
#else
   static int                     FindString();
   static VOID                    PrintString();
   static int                     GetcString();
   static int                     UngetcString();
   static struct stringRouter    *FindStringRouter();  
   static VOID                    ExitFile();
   static VOID                    PrintFile();
   static int                     GetcFile();
   static VOID                    UngetcFile(); 
   static int                     QueryRouter();
   static int                     CreateReadStringSource();
#endif

/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static struct router       *ListOfRouters = NULL;
   static struct stringRouter *ListOfStringRouters = NULL;
   static FILE                *FastLoadFilePtr = NULL;
   static FILE                *FastSaveFilePtr = NULL;
   static int                  Abort;
   static fileptr              ListOfFileRouters = NULL;
   
/****************************************/
/* GLOBAL INTERNAL VARIABLE DEFINITIONS */
/****************************************/

   globle char                *WERROR = "werror"; 
   globle char                *WTRACE = "wtrace";
   globle char                *WDIALOG = "wdialog";
   globle char                *WCLIPS  = "wclips";
   globle char                *WDISPLAY = "wdisplay"; 
   globle int                  CLIPSInputCount = -1; 
   
/*##############################*/
/*##############################*/
/*######                  ######*/
/*###### ROUTER FUNCTIONS ######*/
/*######                  ######*/
/*##############################*/
/*##############################*/

/**********************************************************/
/* InitializeDefaultRouters:  Initializes output streams. */
/**********************************************************/
globle VOID InitializeDefaultRouters()
  {
   AddRouter("string",0,FindString,(int (*)()) PrintString,GetcString,UngetcString,NULL);
   AddRouter("fileio",0,FindFile,(int (*)()) PrintFile,GetcFile,
             (int (*)()) UngetcFile,(int (*)()) ExitFile);
  }
  
/****************************************/
/* PrintCLIPS:  Generic print function. */
/****************************************/
globle int PrintCLIPS(logicalName, str)
  char *logicalName;
  char *str;
  {
   struct router *currentPtr;

   if (((char *) FastSaveFilePtr) == logicalName)
     { 
      fprintf(FastSaveFilePtr,"%s",str);
      return(2);
     }

   currentPtr = ListOfRouters;
   while (currentPtr != NULL)
     {
      if ((currentPtr->printer != NULL) ? QueryRouter(logicalName,currentPtr) : CLIPS_FALSE)         
        { 
         (*currentPtr->printer) (logicalName,str);
         return(1);
        }
      currentPtr = currentPtr->next;
     }

   if (strcmp(WERROR,logicalName) != 0) UnrecognizedRouterMessage(logicalName);
 
   return(0);
  }       

/***********************************************/
/* GetcCLIPS:  Generic get character function. */
/***********************************************/
globle int GetcCLIPS(logicalName)
  char *logicalName;
  {
   struct router *currentPtr;
   int inchar;

   if (((char *) FastLoadFilePtr) == logicalName)
     { 
      inchar = getc(FastLoadFilePtr);
      
      if (inchar == '\r') return('\n');
         
      if (inchar != '\b')
        { return(inchar); }
        
      return(inchar);
     }

   currentPtr = ListOfRouters;
   while (currentPtr != NULL)
     {
      if ((currentPtr->charget != NULL) ? QueryRouter(logicalName,currentPtr) : CLIPS_FALSE)
        { 
         inchar = (*currentPtr->charget) (logicalName);
         
         if (inchar == '\r') return('\n');
         
         if (inchar != '\b')
           { return(inchar); }
     
         return(inchar);
        }
      currentPtr = currentPtr->next;
     }
  
   UnrecognizedRouterMessage(logicalName);
   return(-1);
  }  

/***************************************************/
/* UngetcCLIPS:  Generic unget character function. */
/***************************************************/
globle int UngetcCLIPS(ch,logicalName) 
  int ch;
  char *logicalName;
  {
   struct router *currentPtr;

   if (((char *) FastLoadFilePtr) == logicalName)
     { return(ungetc(ch,FastLoadFilePtr)); }

   currentPtr = ListOfRouters;
   while (currentPtr != NULL)
     {
      if ((currentPtr->charunget != NULL) ? QueryRouter(logicalName,currentPtr) : CLIPS_FALSE)
        { return((*currentPtr->charunget) (ch,logicalName)); }
      currentPtr = currentPtr->next;
     }
     
   UnrecognizedRouterMessage(logicalName);
   return(-1);
  }  

/**************************************/
/* ExitCLIPS:  Generic exit function. */
/**************************************/
globle VOID ExitCLIPS(num)
  int num; 
  {
   struct router *currentPtr, *nextPtr;

   Abort = CLIPS_FALSE;
   currentPtr = ListOfRouters;
   while (currentPtr != NULL)
     {
      nextPtr = currentPtr->next;
      if (currentPtr->active == CLIPS_TRUE)
        { if (currentPtr->exiter != NULL) (*currentPtr->exiter) (num); }
      currentPtr = nextPtr;
     }

   if (Abort) return;
   genexit(num); 
  }

/********************************************/
/* AbortExit: Forces ExitCLIPS to terminate */
/*   after calling all closing routers.     */
/********************************************/
globle VOID AbortExit()
  {
   Abort = CLIPS_TRUE;
  }
  
/***********************************************************/
/* AddRouter:  Adds a routing structure to the route list. */
/***********************************************************/
globle BOOLEAN AddRouter(rname,p_num,query_fun,print_fun,getc_fun,ungetc_fun,exit_fun)
  char *rname;
  int p_num;
  int (*query_fun)(), (*print_fun)(), (*exit_fun)();
  int (*getc_fun)(), (*ungetc_fun)();
  {
   struct router *newPtr, *lastPtr, *currentPtr;

   newPtr = get_struct(router);

   newPtr->name = rname;
   newPtr->active = CLIPS_TRUE;
   newPtr->priority = p_num;
   newPtr->query = query_fun;
   newPtr->printer = print_fun;
   newPtr->exiter = exit_fun;
   newPtr->charget = getc_fun;
   newPtr->charunget = ungetc_fun;
   newPtr->next = NULL;

   if (ListOfRouters == NULL)
     { 
      ListOfRouters = newPtr;
      return(1);
     }

   lastPtr = NULL;
   currentPtr = ListOfRouters;
   while ((currentPtr != NULL) ? (p_num < currentPtr->priority) : CLIPS_FALSE)
     {
      lastPtr = currentPtr; 
      currentPtr = currentPtr->next;
     }

   if (lastPtr == NULL)
     { 
      newPtr->next = ListOfRouters;
      ListOfRouters = newPtr;
     }
   else
     {
      newPtr->next = currentPtr;
      lastPtr->next = newPtr;
     }
     
   return(1);
  }

/*************************************************************/
/* DeleteRouter:  Removes a router from the list of routers. */
/*************************************************************/
globle int DeleteRouter(rname)
  char *rname;
  {
   struct router *currentPtr, *lastPtr;

   currentPtr = ListOfRouters;
   lastPtr = NULL;

   while (currentPtr != NULL)
     {
      if (strcmp(currentPtr->name,rname) == 0)
        {
         if (lastPtr == NULL)
           {
            ListOfRouters = currentPtr->next;
            rm(currentPtr,(int) sizeof(struct router));
            return(1);
           }
         lastPtr->next = currentPtr->next;
         rm(currentPtr,(int) sizeof(struct router));
         return(1);
        }
      lastPtr = currentPtr;
      currentPtr = currentPtr->next;
     }

   return(0);
  }

/**********************************************************************/
/* QueryRouters:  Determines if any router recognizes a logical name. */
/**********************************************************************/
globle int QueryRouters(logicalName)
  char *logicalName;
  {
   struct router *currentPtr;

   currentPtr = ListOfRouters;
   while (currentPtr != NULL)
     {
      if (QueryRouter(logicalName,currentPtr) == CLIPS_TRUE) return(CLIPS_TRUE);
      currentPtr = currentPtr->next;
     }
     
   return(CLIPS_FALSE);
  }

/**************************************************************/
/* QueryRouter:  Determines if a specific router recognizes a */
/*    logical name.                                           */
/**************************************************************/
static int QueryRouter(logicalName,currentPtr)
  char *logicalName;
  struct router *currentPtr;
  {
   if (currentPtr->active == CLIPS_FALSE)
     { return(CLIPS_FALSE); }
   
   if (currentPtr->query == NULL) return(CLIPS_FALSE);
   
   if ( (*currentPtr->query) (logicalName) == CLIPS_TRUE )
     { return(CLIPS_TRUE); }

   return(CLIPS_FALSE);
  }

/*****************************************************/
/* DeactivateRouter:  Deactivates a specific router. */
/*****************************************************/
globle int DeactivateRouter(rname)
  char *rname;
  {
   struct router *currentPtr;

   currentPtr = ListOfRouters;

   while (currentPtr != NULL)
     {
      if (strcmp(currentPtr->name,rname) == 0)
        { 
         currentPtr->active = CLIPS_FALSE;
         return(CLIPS_TRUE); 
        }
      currentPtr = currentPtr->next;
     }

   return(CLIPS_FALSE);
  }

/*************************************************/
/* ActivateRouter:  Activates a specific router. */
/*************************************************/
globle int ActivateRouter(rname)
  char *rname;
  {
   struct router *currentPtr;

   currentPtr = ListOfRouters;

   while (currentPtr != NULL)
     {
      if (strcmp(currentPtr->name,rname) == 0)
        { 
         currentPtr->active = CLIPS_TRUE;
         return(CLIPS_TRUE); 
        }
      currentPtr = currentPtr->next;
     }

   return(CLIPS_FALSE);
  }
  
/********************************************************/
/* SetFastLoad: Used to bypass router system for loads. */
/********************************************************/
globle VOID SetFastLoad(filePtr)
  FILE *filePtr;
  { FastLoadFilePtr = filePtr; }

/********************************************************/
/* SetFastSave: Used to bypass router system for saves. */
/********************************************************/
globle VOID SetFastSave(filePtr)
  FILE *filePtr;
  { FastSaveFilePtr = filePtr; }
  
/********************************************************/
/* GetFastLoad: Used to bypass router system for loads. */
/********************************************************/
globle FILE *GetFastLoad()
  { return(FastLoadFilePtr); }

/********************************************************/
/* GetFastSave: Used to bypass router system for saves. */
/********************************************************/
globle FILE *GetFastSave()
  { return(FastSaveFilePtr); }

/*##################*/
/*# STRING ROUTERS #*/
/*##################*/

/*************************************************************/
/* FindString: Find routine for string router logical names. */
/*************************************************************/
static int FindString(fileid) 
  char *fileid;
  {
   struct stringRouter *head;
   
   head = ListOfStringRouters;
   while (head != NULL)
     {
      if (strcmp(head->name,fileid) == 0)
        { return(CLIPS_TRUE); }
      head = head->next;
     }
     
   return(CLIPS_FALSE);
  }
  
/***************************************************/
/* PrintString:  Print routine for string routers. */
/***************************************************/
static VOID PrintString(logicalName,str)
  char *logicalName, *str;
  {
   struct stringRouter *head;
   
   head = FindStringRouter(logicalName);
   if (head == NULL)
     {
      CLIPSSystemError("ROUTER",3);
      ExitCLIPS(5); 
     }
   
   if (head->readWriteType != WRITE_STRING) return;
   if (head->currentPosition >= (head->maximumPosition - 1)) return;
   
   strncpy(&head->str[head->currentPosition],
           str,(head->maximumPosition - head->currentPosition) - 1);
   
   head->currentPosition += strlen(str);
  }
  
/************************************************/
/* GetcString: Getc routine for string routers. */
/************************************************/
static int GetcString(logicalName)
  char *logicalName;
  {
   struct stringRouter *head;
   int rc;
   
   head = FindStringRouter(logicalName);
   if (head == NULL)
     {
      CLIPSSystemError("ROUTER",1);
      ExitCLIPS(5); 
     }
  
   if (head->readWriteType != READ_STRING) return(EOF);
   if (head->currentPosition >= head->maximumPosition)
     {
      head->currentPosition++;
      return(EOF);
     }

   rc = head->str[head->currentPosition];
   head->currentPosition++;

   return(rc);
  }

/****************************************************/
/* UngetcString: Ungetc routine for string routers. */
/****************************************************/
static int UngetcString(ch,logicalName)
  int ch;
  char *logicalName;
  {
   struct stringRouter *head;
#if MAC_MPW
#pragma unused(ch)
#endif
   
   head = FindStringRouter(logicalName);
   
   if (head == NULL) 
     {
      CLIPSSystemError("ROUTER",2);
      ExitCLIPS(5);
     }
   
   if (head->readWriteType != READ_STRING) return(0);
   if (head->currentPosition > 0)
     { head->currentPosition--; }

   return(1);
  }
  
/************************************************/
/* OpenStringSource: Opens a new string router. */
/************************************************/
globle int OpenStringSource(name,str,currentPosition)
  char *name;
  char *str;
  int currentPosition;
  {
   int maximumPosition;
   
   if (str == NULL) 
     {
      currentPosition = 0;
      maximumPosition = 0;
     }
   else
     { maximumPosition = strlen(str); }
   
   return(CreateReadStringSource(name,str,currentPosition,maximumPosition));
  }
  
/******************************************************/
/* OpenTextSource: Opens a new string router for text */
/*   (which is not NULL terminated).                  */
/******************************************************/
globle int OpenTextSource(name,str,currentPosition,maximumPosition)
  char *name;
  char *str;
  int currentPosition;
  int maximumPosition;
  {
   if (str == NULL) 
     {
      currentPosition = 0;
      maximumPosition = 0;
     }
   
   return(CreateReadStringSource(name,str,currentPosition,maximumPosition));
  }
  
/******************************************************************/
/* CreateReadStringSource: Creates a new string router for input. */
/******************************************************************/
static int CreateReadStringSource(name,str,currentPosition,maximumPosition)
  char *name;
  char *str;
  int currentPosition;
  int maximumPosition;
  {
   struct stringRouter *newStringRouter;
   
   if (FindStringRouter(name) != NULL) return(0);
   
   newStringRouter = get_struct(stringRouter);
   newStringRouter->name = (char *) gm1((int) strlen(name) + 1);
   strcpy(newStringRouter->name,name);
   newStringRouter->str = str;
   newStringRouter->currentPosition = currentPosition;
   newStringRouter->readWriteType = READ_STRING;
   newStringRouter->maximumPosition = maximumPosition;
   newStringRouter->next = ListOfStringRouters;
   ListOfStringRouters = newStringRouter;

   return(1);
  }

/**********************************************/
/* CloseStringSource: Closes a string router. */
/**********************************************/
globle int CloseStringSource(name)
  char *name;
  {
   struct stringRouter *head, *last;
   
   last = NULL;
   head = ListOfStringRouters;
   while (head != NULL)
     {
      if (strcmp(head->name,name) == 0)
        { 
         if (last == NULL)
           {
            ListOfStringRouters = head->next;
            rm(head->name,(int) strlen(head->name) + 1);
            rtn_struct(stringRouter,head);
            return(1);
           }
         else
           {
            last->next = head->next;
            rm(head->name,(int) strlen(head->name) + 1);
            rtn_struct(stringRouter,head);
            return(1);
           }
        }
      last = head;
      head = head->next;
     }
     
   return(0);
  }
  
/******************************************************************/
/* OpenStringDestination: Opens a new string router for printing. */
/******************************************************************/
globle int OpenStringDestination(name,str,maximumPosition)
  char *name;
  char *str;
  int maximumPosition;
  {
   struct stringRouter *newStringRouter;
   
   if (FindStringRouter(name) != NULL) return(0);
   
   newStringRouter = get_struct(stringRouter);
   newStringRouter->name = (char *) gm1((int) strlen(name) + 1);
   strcpy(newStringRouter->name,name);
   newStringRouter->str = str;
   newStringRouter->currentPosition = 0;
   newStringRouter->readWriteType = WRITE_STRING;
   newStringRouter->maximumPosition = maximumPosition;
   newStringRouter->next = ListOfStringRouters;
   ListOfStringRouters = newStringRouter;

   return(1);
  }

/***************************************************/
/* CloseStringDestination: Closes a string router. */
/***************************************************/
globle int CloseStringDestination(name)
  char *name;
  { 
   return(CloseStringSource(name));
  }

/*******************************************************************/
/* FindStringRouter: Returns a pointer to the named string router. */
/*******************************************************************/
static struct stringRouter *FindStringRouter(name)
  char *name;
  {
   struct stringRouter *head;
   
   head = ListOfStringRouters;
   while (head != NULL)
     {
      if (strcmp(head->name,name) == 0)
        { return(head); }
      head = head->next;
     }
     
   return(NULL);
  }
  
/*########################################*/
/*# STANDARD STREAM AND FILE I/O ROUTERS #*/
/*########################################*/

/*******************************************************/
/* FindFptr:  Returns a pointer to a file stream for a */
/*   given file id tag.                                */
/*******************************************************/
globle FILE *FindFptr(fileid) 
  char *fileid;
  {
   fileptr fptr;
                                                                 
   /*========================================================*/
   /* Check to see if standard input or output is requested. */
   /*========================================================*/

   if (strcmp(fileid,"stdout") == 0)
     { return(stdout); }
   else if (strcmp(fileid,"stdin") == 0)
     { return(stdin); }  
   else if (strcmp(fileid,WTRACE) == 0)
     { return(stdout); }    
   else if (strcmp(fileid,WDIALOG) == 0)
     { return(stdout); }  
   else if (strcmp(fileid,WCLIPS) == 0)
     { return(stdout); } 
   else if (strcmp(fileid,WDISPLAY) == 0)
     { return(stdout); }  
   else if (strcmp(fileid,WERROR) == 0)
     { return(stdout); }

   /*=========================================================*/
   /* Otherwise, look up the file id on the global file list. */
   /*=========================================================*/
           
   fptr = ListOfFileRouters;
   while ((fptr != NULL) ? (strcmp(fileid,fptr->fileid) != 0) : CLIPS_FALSE)
     { fptr = fptr->next; }

   if (fptr != NULL) return(fptr->stream);

   return(NULL); 
  }

/*******************************************************/
/* FindFile:  Returns a pointer to a file stream for a */
/*   given file id tag.                                */
/*******************************************************/
globle int FindFile(fileid) 
  char *fileid;
  {
   fileptr fptr;

   if ( (strcmp(fileid,"stdout") == 0) ||
        (strcmp(fileid,"stdin") == 0) ||
        (strcmp(fileid,WCLIPS) == 0) ||
        (strcmp(fileid,WTRACE) == 0) ||
        (strcmp(fileid,WERROR) == 0) ||
        (strcmp(fileid,WDISPLAY) == 0) ||
        (strcmp(fileid,WDIALOG) == 0) )
     { return(CLIPS_TRUE); }

   fptr = ListOfFileRouters;
   while ((fptr != NULL) ? (strcmp(fileid,fptr->fileid) != 0) : CLIPS_FALSE)
     { fptr = fptr->next; }

   if (fptr != NULL) return(CLIPS_TRUE);

   return(CLIPS_FALSE); 
  }

/********************************************/
/* ExitFile:  Exit routine for file router. */
/********************************************/
static VOID ExitFile(num)
  int num;
  {
#if MAC_MPW
#pragma unused(num)
#endif
#if BASIC_IO
   CloseAllFiles();       
#endif
  }

/**********************************************/
/* PrintFile:  Print routine for file router. */
/**********************************************/
static VOID PrintFile(logicalName,str)
  char *logicalName, *str;
  {
   FILE *fptr;

   fptr = FindFptr(logicalName);
   fprintf(fptr,"%s",str);
   fflush(fptr);
  }

/********************************************/
/* GetcFile:  Getc routine for file router. */
/********************************************/
static int GetcFile(logicalName)
  char *logicalName;
  {
   FILE *fptr;
   int theChar;

   fptr = FindFptr(logicalName);
   
   theChar = getc(fptr);
   
   /* The following code prevents Control-D on UNIX   */
   /* machines from terminating all input from stdin. */
   if ((fptr == stdin) && (theChar == EOF)) clearerr(stdin);
   
   return(theChar);
  }

/************************************************/
/* UngetcFile:  Ungetc routine for file router. */
/************************************************/
static VOID UngetcFile(ch,logicalName)
  int ch;
  char *logicalName;
  {
   FILE *fptr;

   fptr = FindFptr(logicalName);
   ungetc(ch,fptr);
  }
  
/* Mod:
 >In the CLIPS code you will have to rename the procedure
 >OpenFile() as it is already used by Windows.  I use
 >ClipsOpenFile().  ROUTER.C, ROUTER.H, SYSIO.C,
*/

/************************************************************************/
/* CLIPSOpenFile:Opens a file with the specified access mode and stores */
/*   the opened stream as well as the file id tag on the global file    */
/*   list. Returns a non-zero value if the file was succesfully opened. */
/************************************************************************/
globle int CLIPSOpenFile(fname,fmode,fid) 
  char *fname,*fmode,*fid; 
  {

   FILE *newstream;
   fileptr fptr, prev;

   newstream = fopen(fname,fmode);

   /*==================================================================*/
   /* Make sure the file can be opened with the specified access mode. */
   /*==================================================================*/

   if (newstream == NULL)  return(0);

   /*=====================================*/
   /* Add stream and file id tag to list. */
   /*=====================================*/

   if (ListOfFileRouters == NULL)
     {
      ListOfFileRouters = get_struct(filelist);
      ListOfFileRouters->fileid = (char *) gm2 ((int) strlen(fid) + 1);
      strcpy(ListOfFileRouters->fileid,fid);
      ListOfFileRouters->stream = newstream;
      ListOfFileRouters->next = NULL; 
     }
   else 
     {
      fptr = ListOfFileRouters; 
      prev = fptr;
      while (fptr != NULL)
        {
         prev = fptr;
         fptr = fptr->next; 
        }
      fptr = get_struct(filelist);
      fptr->fileid = (char *) gm2 ((int) strlen(fid) + 1);
      strcpy(fptr->fileid,fid);
      fptr->stream = newstream;
      fptr->next = NULL;
      prev->next = fptr;
     }
               
   return(1);
  }
          
/******************************************************************/
/* CloseFile:  Closes a file with the specified file id tag.      */
/*   Returns a non-zero value if the file was succesfully closed. */
/******************************************************************/
globle int CloseFile(fid) 
  char *fid;
  {
   fileptr fptr, prev;

   /*=====================================================*/
   /* Locate the file with the given id in the file list. */
   /*=====================================================*/

   fptr = ListOfFileRouters;
   prev = NULL;

   while (fptr != NULL)
     {
      if (strcmp(fptr->fileid,fid) == 0)
        { 
         fclose(fptr->stream);
         rm(fptr->fileid,(int) strlen(fptr->fileid) + 1);
         if (prev == NULL)
           { ListOfFileRouters = fptr->next; }
         else
           { prev->next = fptr->next; }
         rm(fptr,(int) sizeof(filelist));

         return(1);
        }
        
      prev = fptr;
      fptr = fptr->next;
     }
   
   return(0);
  }

/******************************************************************/
/* CloseAllFiles:  Closes all files opened the file io utilities. */
/******************************************************************/
globle int CloseAllFiles()
  {
   fileptr fptr, prev;

   if (ListOfFileRouters == NULL) return(0);
   
   fptr = ListOfFileRouters;

   while (fptr != NULL)
     {
      fclose(fptr->stream);
      prev = fptr;
      rm(fptr->fileid,(int) strlen(fptr->fileid) + 1);
      fptr = fptr->next;
      rm(prev,(int) sizeof(filelist));
     }

   ListOfFileRouters = NULL;

   return(1);
  }

/************************************************************/
/* UnrecognizedRouterMessage: Standard error message for an */
/*   unrecognized router name.                              */
/************************************************************/
globle VOID UnrecognizedRouterMessage(logicalName)
  char *logicalName;
  {
   PrintCLIPS(WERROR,"ERROR: Logical name ");
   PrintCLIPS(WERROR,logicalName);
   PrintCLIPS(WERROR," was not recognized by any routers\n");
  }
