   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                   RETRACT MODULE                    */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _RETRACT_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_

#include "setup.h"

#if DEFRULE_CONSTRUCT

#include "constant.h"
#include "clipsmem.h"
#include "symbol.h"
#include "evaluatn.h"
#include "router.h"
#include "network.h"
#include "engine.h"
#include "match.h"
#include "reteutil.h"
#include "utility.h"
#include "retract.h"
#include "drive.h"
#include "lgcldpnd.h"

#if ANSI_COMPILER  
   static VOID                    TopNegJoinRetract(struct joinNode *,long int);
   static VOID                    NegEntryRetract(struct joinNode *,long int);
   static VOID                    ReturnMarkers(struct multifieldMarker *);
   static VOID                    DriveRetractions(void);
#else
   static VOID                    TopNegJoinRetract();
   static VOID                    NegEntryRetract();
   static VOID                    ReturnMarkers();
   static VOID                    DriveRetractions();
#endif

/**************/
/* STRUCTURES */
/**************/

struct rdriveinfo
  {
   struct partialMatch *link;
   struct joinNode *jlist;
   struct rdriveinfo *next;
  };
  
/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static struct rdriveinfo   *DriveRetractionList = NULL;
   
/****************************************/
/* GLOBAL INTERNAL VARIABLE DEFINITIONS */
/****************************************/

   globle struct partialMatch *GarbagePartialMatches = NULL;
   globle struct alphaMatch   *GarbageAlphaMatches = NULL;
   
/************************************************************/
/* NetworkRetract:  Retracts a fact from the pattern and    */
/*   join networks given a pointer to the list of patterns  */
/*   which the fact matched. The fact is retracted from the */
/*   join network first through positive patterns, then     */
/*   through negated patterns, and then any new partial     */
/*   matches created by the retraction are driven through   */
/*   the join network. This ordering prevents partial       */
/*   matches being generated that contain the deleted fact. */
/************************************************************/
globle VOID NetworkRetract(listOfMatchedPatterns,factIndex)
  struct patternMatch *listOfMatchedPatterns;
  FACT_ID factIndex;
  {
   struct patternMatch *temp_match;
   struct joinNode *joinPtr;
   int found_id;

   /*===============================*/
   /* Delete for positive patterns. */
   /*===============================*/
   
   temp_match = listOfMatchedPatterns;
   while (listOfMatchedPatterns != NULL)
     {
      /*================================================*/
      /* Loop through the list of all joins attached to */
      /* this pattern.                                  */
      /*================================================*/

      joinPtr = listOfMatchedPatterns->matchingPattern->entry_join;
        
      while (joinPtr != NULL)
        {
         if (joinPtr->LHSPatternIsNegated == CLIPS_FALSE)
           { PosEntryRetract(RHS,joinPtr,factIndex,(int) joinPtr->depth - 1); }

         joinPtr = joinPtr->right_match_node;
        }
        
      listOfMatchedPatterns = listOfMatchedPatterns->next;
     } 
     
   /*===============================*/
   /* Delete for negative patterns. */
   /*===============================*/
   
   listOfMatchedPatterns = temp_match;
   while (listOfMatchedPatterns != NULL)
     {
      /*================================================*/
      /* Loop through the list of all joins attached to */
      /* this pattern.                                  */
      /*================================================*/

      joinPtr = listOfMatchedPatterns->matchingPattern->entry_join;
        
      while (joinPtr != NULL)
        {         
         /*========================================*/
         /* Handle retract based on logic of join. */
         /*========================================*/

         if (joinPtr->LHSPatternIsNegated == CLIPS_TRUE)
           {
            if (joinPtr->firstJoin == CLIPS_TRUE)
              { TopNegJoinRetract(joinPtr,factIndex); }
            else 
              { NegEntryRetract(joinPtr,factIndex); }
           }

         joinPtr = joinPtr->right_match_node;
        }
        
      listOfMatchedPatterns->matchingPattern->alpha = DeletePartialMatches(factIndex,listOfMatchedPatterns->matchingPattern->alpha,&found_id,0,0);
      temp_match = listOfMatchedPatterns->next;
      rtn_struct(patternMatch,listOfMatchedPatterns); 
      listOfMatchedPatterns = temp_match;
     } 
      
   /*========================================*/
   /* Drive new partial matches generated by */
   /* retraction through the join network.   */
   /*========================================*/
   
   DriveRetractions();
  }

/***************************************************************/
/* PosEntryRetract:  Handles retract for a join of a rule with */
/*    a positive pattern when the retraction is starting from  */
/*    the RHS of that join (empty or positive LHS entry,       */
/*    positive RHS entry), or the LHS of that join (positive   */
/*    LHS entry, negative or positive RHS entry).              */ 
/***************************************************************/
globle VOID PosEntryRetract(dir,join,factIndex,position)
  FACT_ID factIndex;
  int dir;
  struct joinNode *join;
  int position;
  {
   struct joinNode *joinPtr;   
   int found_id;
   
   while (join != NULL)
     {
      /*=========================================*/
      /* Remove the bindings from this join that */
      /* contain the fact to be retracted.       */
      /*=========================================*/

      if (dir == RHS)
        { found_id = CLIPS_TRUE; }
      else
        { 
         if (join->beta == NULL) return; /* optimize */
         join->beta = DeletePartialMatches(factIndex,join->beta,&found_id,position,1); 
        }

      /*===================================================*/
      /* If no facts were deleted at this join, then there */
      /* is no need to check joins at a lower level.       */
      /* Global Variable FOUND_ID is set by DeletePartialMatches.   */
      /*===================================================*/

      if (found_id == CLIPS_FALSE) return;

      /*==================================================*/
      /* If there is more than one join below this join,  */
      /* then recursively remove fact bindings from all   */
      /* but one of the lower joins.  Remove the bindings */
      /* from the other join through this loop.           */
      /*==================================================*/
      
      joinPtr = join->next_level;
      if (joinPtr == NULL) return;
      
      while (joinPtr->right_drive_node != NULL)
        {
         PosEntryRetract(LHS,joinPtr,factIndex,position);
         joinPtr = joinPtr->right_drive_node;
        }
  
      dir = LHS;
      join = joinPtr;      
     }
  }

/********************************************************************/
/* TopNegJoinRetract:  Handles retract for the first join of a rule */
/*    with a negated pattern (empty LHS entry, negative RHS entry,  */
/*    retraction starting from RHS).                                */ 
/********************************************************************/
static VOID TopNegJoinRetract(join,factIndex)
  struct joinNode *join;
  FACT_ID factIndex;
  {
   struct partialMatch *nlr;
   struct partialMatch *rhs_ptr;
   int join_test;
   struct rdriveinfo *temp_dr;
   
   /*===========================*/
   /* Network Assumption Check. */
   /*===========================*/

   if (join->eval != NULL)
     {
      CLIPSSystemError("RETRACT",1);
      ExitCLIPS(5);
     }

   /*====================================================*/
   /* Decrement the id count in the join by one for each */
   /* occurence of the fact in the alpha memory.         */
   /*====================================================*/

   rhs_ptr = join->entry_pat->alpha;
   while (rhs_ptr != NULL)
     {
      if (rhs_ptr->binds[0]->whoset == factIndex)
        { join->id--; }
      rhs_ptr = rhs_ptr->next;
     }

   if (join->id != 0) return;
   
   /*=====================================================*/
   /* Check test associated with not. Note that since the */
   /* not was the first pattern in the rule, no variables */
   /* can be accessed by the expression.                  */
   /*=====================================================*/
   
   if (join->not_eval != NULL)
     {
      join_test = EvaluateJoinExpression(join->not_eval,NULL,NULL,join);
      EvaluationError = CLIPS_FALSE;
      if (join_test == CLIPS_FALSE) return;
     }
     
   /*====================================================*/
   /* If the rhs is null, then the not pattern has been  */
   /* satisfied and a partial match needs to be created. */
   /*====================================================*/

   nlr = NewPseudoFactPartialMatch();
   join->beta = nlr;
   join->id = nlr->binds[0]->whoset;

   temp_dr = get_struct(rdriveinfo);
   nlr->bcount =  1;
   temp_dr->link = CopyPartialMatch(nlr);
   nlr->bcount =  0;
   temp_dr->jlist = join->next_level;
   temp_dr->next = DriveRetractionList;
   DriveRetractionList = temp_dr;
  }
  
/*****************************************************************/
/* NegEntryRetract:  Handles retract for a join of a rule with a */
/*    negated pattern when the retraction is starting from the   */
/*    RHS of that join (positve LHS entry, negative RHS entry,   */
/*    retraction starting from RHS).                             */ 
/*****************************************************************/
static VOID NegEntryRetract(join,factIndex)
  struct joinNode *join;
  FACT_ID factIndex;
  { 
   struct partialMatch *nlr, *rhs_ptr, *lhs_ptr;
   int join_test;
   struct rdriveinfo *temp_dr;
   FACT_ID temp_id;
   struct alphaMatch *tempAlpha;
         
   /*===============================================*/
   /* Loop through all rhs bindings looking for the */
   /* fact that is to be retracted in the bindings. */
   /*===============================================*/

   rhs_ptr = join->entry_pat->alpha;
   while (rhs_ptr != NULL) 
     {
      if (rhs_ptr->binds[0]->whoset == factIndex)
        {
         /*==================================================*/
         /* Fact was found in rhs bindings. Loop through all */ 
         /* lhs bindings checking for sets that satisfied    */
         /* the join expression.                             */
         /*==================================================*/

         lhs_ptr = join->beta;
         while (lhs_ptr != NULL)
           {
            /*================================================*/
            /* Evaluate join expression with this combination */
            /* of rhs and lhs fact bindings.                  */
            /*================================================*/

            if (join->eval == NULL)
              { join_test = CLIPS_TRUE; }
            else
              { 
               join_test = EvaluateJoinExpression(join->eval,lhs_ptr,rhs_ptr,join); 
               if (EvaluationError)
                 { 
                  join_test = CLIPS_TRUE;
                  EvaluationError = CLIPS_FALSE;
                 }
              }

            /*===========================*/
            /* Check for a system error. */
            /*===========================*/

            if ((join_test != CLIPS_FALSE) && (lhs_ptr->counterf == CLIPS_FALSE))
              {
               CLIPSSystemError("RETRACT",2);
               ExitCLIPS(5);
              }

            /*================================================*/
            /* If the join expression evaluated to true, then */
            /* the lhs bindings will have one less set of     */
            /* of bindings that did not conflict with it.     */
            /*================================================*/

            if (join_test != CLIPS_FALSE)
              { 
               lhs_ptr->binds[lhs_ptr->bcount] = 
                   (struct alphaMatch *) (((long int) lhs_ptr->binds[lhs_ptr->bcount]) - 1);
              }

            /*==================================================*/
            /* If the lhs bindings now has no rhs bindings that */
            /* do not conflict with it, then it satisfies the   */
            /* the conditions of the rhs not pattern. Create a  */
            /* partial match and send it to the joins below.    */
            /*==================================================*/
            /* NOTE - Does not have to be performed unless count is zero. */
/*
	Mod: Patch to CLIPS: && (...(join_test == CLIPS_TRUE))
*/
            if ((((long int) lhs_ptr->binds[lhs_ptr->bcount]) == 0L) &&
                (join_test == CLIPS_TRUE))
              {
               if (join->not_eval == NULL)
                 { join_test = CLIPS_TRUE; }
               else
                 { 
                  join_test = EvaluateJoinExpression(join->not_eval,lhs_ptr,NULL,join);
                  EvaluationError = CLIPS_FALSE;
                 }
	      }
                 
            if ((((long int) lhs_ptr->binds[lhs_ptr->bcount]) == 0L) && (join_test == CLIPS_TRUE))
              {
               temp_id = IncrementPseudoFactIndex();
               lhs_ptr->counterf = CLIPS_FALSE;
               nlr = AddSingleMatch(lhs_ptr,NULL);
               tempAlpha = get_struct(alphaMatch);
               tempAlpha->aorigin = tempAlpha;
               tempAlpha->whoset = temp_id;
               tempAlpha->origin = NULL;
               tempAlpha->marker = NULL;   
	       lhs_ptr->binds[nlr->bcount - 1] = tempAlpha;
               nlr->binds[nlr->bcount - 1] = tempAlpha;
      
               temp_dr = get_struct(rdriveinfo);
               temp_dr->link = nlr;
               temp_dr->jlist = join->next_level;
               temp_dr->next = DriveRetractionList;
               DriveRetractionList = temp_dr;
              }

            lhs_ptr = lhs_ptr->next;
           }
	}
      rhs_ptr = rhs_ptr->next;
     }
  }
  
/*************************************************************/
/* DeletePartialMatches: Searches through a list of partial  */
/*   matches and removes any partial match that contains the */
/*   specified fact identifier.                              */
/*************************************************************/
globle struct partialMatch *DeletePartialMatches(factIndex,list_of_binds,found_id,position,beta_delete)
  FACT_ID factIndex;                 
  struct partialMatch *list_of_binds;
  int *found_id;
  int position;
  int beta_delete;
  {
   struct partialMatch *head, *past_bind, *next_bind;

   past_bind = NULL;
   head = list_of_binds;

   *found_id = CLIPS_FALSE;
   while (list_of_binds != NULL)
     {
      if (list_of_binds->binds[position]->whoset == factIndex)
        {
         *found_id = CLIPS_TRUE;
         
         if ((list_of_binds->activationf) ? 
             (list_of_binds->binds[list_of_binds->bcount] != NULL) : FALSE)
           { RemoveActivation((struct activation *) list_of_binds->binds[list_of_binds->bcount],CLIPS_TRUE,CLIPS_TRUE); }     

         if (list_of_binds == head)
           {
            /* Delete bind at beginning of list. */
	    next_bind = list_of_binds->next;
           
            if (list_of_binds->dependents != NULL)
              {
               AddToDependencyList(list_of_binds);
              }
              
            if (beta_delete && (list_of_binds->notOriginf == CLIPS_FALSE))
              {
               ReturnPartialMatch(list_of_binds);
              }
            else
	      {
               list_of_binds->next = GarbagePartialMatches;
               GarbagePartialMatches = list_of_binds;
              }
            list_of_binds = next_bind;
            head = list_of_binds;
           }
         else
           {
            /* Delete bind after beginning of list. */
            past_bind->next = list_of_binds->next;
            
	    if (list_of_binds->dependents != NULL)
              {
               AddToDependencyList(list_of_binds);
              }
              
            if (beta_delete && (list_of_binds->notOriginf == CLIPS_FALSE))
              { ReturnPartialMatch(list_of_binds); }
            else
              {
               list_of_binds->next = GarbagePartialMatches;
               GarbagePartialMatches = list_of_binds;
              }
	    list_of_binds = past_bind->next;
           }
        }
      else
        {
         past_bind = list_of_binds;
         list_of_binds = list_of_binds->next;
        }
     }
   return(head);
  }

/***************************************************************/
/* ReturnPartialMatch:  Returns the data structures associated */
/*   with a partial match to the pool of free memory.          */
/***************************************************************/
globle VOID ReturnPartialMatch(waste)
  struct partialMatch *waste;
  {   
   if (waste->busy)
     {
      waste->next = GarbagePartialMatches;
      GarbagePartialMatches = waste;
      return;
     }
     
   if (waste->betaMemory == CLIPS_FALSE)
     {
      if (waste->binds[0]->marker != NULL)
        { ReturnMarkers(waste->binds[0]->marker); }
      rm(waste->binds[0],(int) sizeof(struct alphaMatch));
     }
   
   if (waste->dependents != NULL) RemovePMDependencies(waste);

   rtn_var_struct(partialMatch,(int) sizeof(struct alphaMatch *) * 
		  (waste->bcount + waste->notOriginf + waste->activationf - 1),
                  waste);
  
  }
  
/************************************************************/
/* ReturnMarkers                                            */
/************************************************************/
static VOID ReturnMarkers(waste)
  struct multifieldMarker *waste;
  {
   struct multifieldMarker *temp;

   while (waste != NULL)
     {
      temp = waste->next;
      rtn_struct(multifieldMarker,waste);
      waste = temp;
     }
  }

/***********************************/
/* DriveRetractions:              */
/***********************************/
static VOID DriveRetractions()
  {
   struct rdriveinfo *temp_dr;
   struct joinNode *joinPtr;
   
   while (DriveRetractionList != NULL)
     {
      joinPtr = DriveRetractionList->jlist;
      while (joinPtr->right_drive_node != NULL)
        {
         Drive(DriveRetractionList->link,joinPtr,LHS,CLIPS_FALSE);
         joinPtr = joinPtr->right_drive_node;
	}

      Drive(DriveRetractionList->link,joinPtr,LHS,CLIPS_TRUE);
      
      temp_dr = DriveRetractionList->next;
      rtn_struct(rdriveinfo,DriveRetractionList);
      DriveRetractionList = temp_dr;
     }
  }
   
/************************************************************************/
/* FlushGarbagePartialMatches:  Returns partial matches and associated  */
/*   structures that were removed as part of a retraction. It is        */
/*   necessary to postpone returning these structures to memory because */
/*   RHS actions retrieve their variable bindings directly from the     */
/*   fact data structure through the alpha memory bindings.             */
/************************************************************************/
globle VOID FlushGarbagePartialMatches()
  {
   struct partialMatch *pmPtr;
   struct alphaMatch *amPtr;

   /* The alpha matches here should only belong to the */
   /* fake matches created for negated pseudo facts.   */

   while (GarbageAlphaMatches != NULL)
     {
      amPtr = GarbageAlphaMatches->aorigin;
      
      rtn_struct(alphaMatch,GarbageAlphaMatches);
      
      GarbageAlphaMatches = amPtr;
     }
   
   while (GarbagePartialMatches != NULL)
     {
      pmPtr = GarbagePartialMatches->next;
      if ((GarbagePartialMatches->notOriginf) && 
          (GarbagePartialMatches->counterf == CLIPS_FALSE))
        { 
         if (GarbagePartialMatches->binds[GarbagePartialMatches->bcount] != NULL)
           {
            rtn_struct(alphaMatch,
                       GarbagePartialMatches->binds[GarbagePartialMatches->bcount]); 
	   }
        }
         
      GarbagePartialMatches->busy = CLIPS_FALSE;
      ReturnPartialMatch(GarbagePartialMatches);
      
      GarbagePartialMatches = pmPtr;
     }
  }    

#endif

