   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10 07/17/91             */
   /*                                                     */
   /*                  CONSTRUCT MODULE                   */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
  
#define _CONSTRCT_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"

#include "constant.h"
#include "clipsmem.h"
#include "symbol.h"
#include "scanner.h"
#include "router.h"
#include "expressn.h"
#include "evaluatn.h"
#include "constrct.h"
#include "utility.h"
#include "spclform.h"
#include "parsutil.h"

/*************************/
/* STRUCTURE DEFINITIONS */
/*************************/

struct construct
  { 
   struct symbolHashNode *constructName;  
#if ANSI_COMPILER
   int (*ip)(char *);
#else
   int (*ip)();
#endif
   struct construct *next;
  };
  
struct saveFunction
  {
   char *name;  
#if ANSI_COMPILER
   int (*ip)(char *);
#else
   int (*ip)();
#endif
   int priority;
   struct saveFunction *next;
  };
  
struct resetFunction
  {
   char *name;  
#if ANSI_COMPILER
   VOID (*ip)(void);
#else
   VOID (*ip)();
#endif
   int priority;
   struct resetFunction *next;
  };
  
struct clearFunction
  {
   char *name;  
#if ANSI_COMPILER
   VOID (*ip)(void);
#else
   VOID (*ip)();
#endif
   int priority;
   struct clearFunction *next;
  };
  
/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if (! RUN_TIME) && (! BLOAD_ONLY)
#if ANSI_COMPILER 
   static int                     ErrorAlignment(int,char *,struct token *);
#else
   static int                     ErrorAlignment();
#endif
#endif

/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

#if (! RUN_TIME) && (! BLOAD_ONLY)
   static struct construct     *ListOfConstructs = NULL;
   static struct saveFunction  *ListOfSaveFunctions = NULL;
   static BOOLEAN               PrintWhileLoading = CLIPS_FALSE;
   static BOOLEAN               WatchCompilations = ON;
#endif
   static struct resetFunction *ListOfResetFunctions = NULL;
   static struct clearFunction *ListOfClearFunctions = NULL;
   static int                   Executing = CLIPS_FALSE;   
#if ANSI_COMPILER
   static int                 (*BeforeResetFunction)(void) = NULL;
   static int                 (*BeforeClearFunction)(void) = NULL;
#else
   static int                 (*BeforeResetFunction)() = NULL;
   static int                 (*BeforeClearFunction)() = NULL;
#endif

/* Mod: Error Check for __DLL__ definition */
#if (! RUN_TIME) && (! BLOAD_ONLY)
#ifndef __DLL__
#error __DLL__ is not defined!
#endif

/*******************************************************/
/* LoadConstructs:  Loads a set of constructs into the */
/*   current CLIPS environment from a file.            */
/*******************************************************/
globle int LoadConstructs(fileName)
  char *fileName;
  {
   FILE *file_ptr;

   /*=======================================*/
   /* Open the file specified by file name. */
   /*=======================================*/
  
   if ((file_ptr = fopen(fileName,"r")) == NULL)
     { return(-1); }
   SetFastLoad(file_ptr);

   LoadConstructsFromLogicalName((char *) file_ptr);

   /*=================*/
   /* Close the file. */
   /*=================*/
   
   fclose(file_ptr);
   SetFastLoad(NULL);

   DestroyPPBuffer();
   return(0);
  }

/******************************************************************/
/* LoadConstructsFromLogicalName:  Loads a set of constructs into */
/*   the current CLIPS environment from a specified logical name. */
/******************************************************************/
globle VOID LoadConstructsFromLogicalName(readSource)
  char *readSource;
  {
   int error_flag, construct_flag;
   struct token inp_tkn;

   /*==================================================*/
   /* Parse the file until the end of file is reached. */
   /*==================================================*/

   if (CurrentEvaluationDepth == 0) SetHaltExecution(CLIPS_FALSE);
   SetEvaluationError(CLIPS_FALSE);
   error_flag = CLIPS_FALSE;
   GetToken(readSource,&inp_tkn);

   while ((inp_tkn.type != STOP) && (GetHaltExecution() == CLIPS_FALSE))
     {
      error_flag = ErrorAlignment(error_flag,readSource,&inp_tkn);

      if (inp_tkn.type == STOP) { return; }

      if ((inp_tkn.type == SYMBOL) && (error_flag == CLIPS_FALSE))
              
        {
         construct_flag = ParseConstruct(ValueToString(inp_tkn.value),readSource);
         if (construct_flag == 1)
            {
             PrintCLIPS(WERROR,"\nERROR:\n");
             PrintInChunks(WERROR,GetPPBuffer());
             PrintCLIPS(WERROR,"\n");
             error_flag = CLIPS_TRUE;
	    }
         else 
           { error_flag = CLIPS_FALSE; }
         FlushPPBuffer();        
        }
 
      GetToken(readSource,&inp_tkn);
     }

   /*==================================================*/
   /* Print a carriage return if a single character is */
   /* being printed to indicate constructs are being   */
   /* processed.                                       */
   /*==================================================*/
 
   if ((GetWatchItem("compilations") != CLIPS_TRUE) && GetPrintWhileLoading()) 
     { PrintCLIPS(WDIALOG,"\n"); }
  }

/*******************************************************************/
/* ErrorAlignment: Positions the parser at a token which indicates */
/*   the beginning of a valid construct. If called as the result   */
/*   of an error in a construct, this routine skips over tokens    */
/*   until it finds the beginning of a new construct. If an error  */
/*   hasn't occured, then this routine checks to see that the      */
/*   parser is currently at the beginning of a new construct (a    */
/*   left parenthesis followed by a constructs name).              */ 
/*******************************************************************/
static int ErrorAlignment(error_flag,readSource,inp_tkn)
  int error_flag;
  char *readSource;
  struct token *inp_tkn;
  {

   if (error_flag == CLIPS_FALSE)
     {
      if (inp_tkn->type != LPAREN)
        {
         PrintCLIPS(WERROR,"\nExpected left parenthesis to begin ");
         PrintCLIPS(WERROR,"new construct\n");
         error_flag = CLIPS_TRUE;
        }
      else
        {
	 GetToken(readSource,inp_tkn);
         
         if ((inp_tkn->type == SYMBOL) && (ValidConstruct(ValueToString(inp_tkn->value)) == 0))
           {
            PrintCLIPS(WERROR,"\nFound unrecognized construct ");
            PrintCLIPS(WERROR,ValueToString(inp_tkn->value));
            PrintCLIPS(WERROR,"\n");
            PrintCLIPS(WERROR,"Check that all constructs ");
            PrintCLIPS(WERROR,"have the proper number of matching parentheses\n");
            error_flag = CLIPS_TRUE;
           }
	 else
           {
            FlushPPBuffer(); 
            return(error_flag);
           }
        }
     }

   /* Error Correction */
   while ((error_flag == CLIPS_TRUE) && (inp_tkn->type != STOP))
     {
      while ((inp_tkn->type != LPAREN) && (inp_tkn->type != STOP))
        { GetToken(readSource,inp_tkn); }
      if (inp_tkn->type != STOP)
        { 
         GetToken(readSource,inp_tkn);
         if ((inp_tkn->type == SYMBOL) && (ValidConstruct(ValueToString(inp_tkn->value)) == 1))
           {
            FlushPPBuffer(); 
            return(CLIPS_FALSE);
           }
        }
     }
   if (inp_tkn->type == STOP) { return(-1); }
   return(error_flag);
  }

/******************************************************************/
/* InitializeIgnoredConstructs: Initializes some parsing routines */
/*   for skipping over CRSV constructs not handled by CLIPS.      */
/******************************************************************/
globle VOID InitializeIgnoredConstructs()
  {     
   AddConstruct("defrelation",ParseIgnoredConstruct);
   AddConstruct("defexternal",ParseIgnoredConstruct);
  }
 
/***************************************************************/
/* ParseIgnoredConstruct: Parsing routine for skipping certain */
/*   constructs not handled by CLIPS.                          */
/***************************************************************/
globle int ParseIgnoredConstruct(readSource)
  char *readSource;
  {
   struct token art_tkn;
   
   SetPPBufferStatus(OFF);
   FlushPPBuffer(); 
   
   /*===============================================================*/
   /* If watch rules is on, indicate art construct being processed. */
   /*===============================================================*/

   if ((GetWatchItem("compilations") == ON) && GetPrintWhileLoading())
     { PrintCLIPS(WDIALOG,"Warning: Construct not supported\n"); }
   else if (GetPrintWhileLoading())
     { PrintCLIPS(WDIALOG,"?"); }

   /*================================*/
   /* Check for closing parenthesis. */
   /*================================*/
   
   if (ReadUntilClosingParen(readSource,&art_tkn) == 0) return(1);
   else return(0);
  }
 
/*************************************************************/
/* AddConstruct: Adds a construct and its associated parsing */
/*   function to the ListOfConstructs.                       */
/*************************************************************/
globle VOID AddConstruct(name,functionPtr)
  char *name;
  int (*functionPtr)();
  {
   struct construct *newPtr;

   newPtr = get_struct(construct);

   newPtr->constructName = (struct symbolHashNode *) AddSymbol(name);
   IncrementSymbolCount(newPtr->constructName);
   newPtr->ip = functionPtr;
   newPtr->next = ListOfConstructs;
   ListOfConstructs = newPtr;
  }

/***********************************************************/
/* RemoveConstruct: Removes a construct and its associated */
/*   parsing function from the ListOfConstructs.           */
/***********************************************************/
globle int RemoveConstruct(name)
  char *name;
  {
   struct construct *currentPtr, *lastPtr;

   lastPtr = NULL;
   currentPtr = ListOfConstructs;
   
   while (currentPtr != NULL)
     {
      if (strcmp(name,ValueToString(currentPtr->constructName)) == 0)
        {
         if (lastPtr == NULL)
           { ListOfConstructs = currentPtr->next; }
         else
           { lastPtr->next = currentPtr->next; }
         DecrementSymbolCount(currentPtr->constructName);
         rtn_struct(construct,currentPtr);
         return(1);
        }
      lastPtr = currentPtr;
      currentPtr = currentPtr->next;
     }
     
   return(0);
  }

/******************************************************************************/
/* ValidConstruct: Determines whether a construct is in the ListOfConstructs. */
/******************************************************************************/
globle int ValidConstruct(name)
  char *name;
  {
   struct construct *currentPtr;
   
   currentPtr = ListOfConstructs;
   while (currentPtr != NULL)
     {
      if (strcmp(name,ValueToString(currentPtr->constructName)) == 0)
        { return(CLIPS_TRUE); }
      currentPtr = currentPtr->next;
     }
     
   return(CLIPS_FALSE);
  }
  
/***********************************************************/
/* ParseConstruct: Parses a construct. Returns an integer. */
/*   -1 if the construct name has no parsing function, 0   */
/*   if the construct was parsed successfully, and 1 if    */
/*   the construct was parsed unsuccessfully.              */
/***********************************************************/
globle int ParseConstruct(name,log_name)
  char *name, *log_name;
  {
   struct construct *currentPtr;
   int rv, ov;
   
   ov = GetHaltExecution();
   
   currentPtr = ListOfConstructs;
   while (currentPtr != NULL)
     {
      if (strcmp(name,ValueToString(currentPtr->constructName)) == 0)
        {  
	 SetEvaluationError(CLIPS_FALSE);
         SetHaltExecution(CLIPS_FALSE);
         ClearParsedBindNames();
         
         rv = (*currentPtr->ip)(log_name);
         
         ClearParsedBindNames();
         SetPPBufferStatus(OFF);
         SetHaltExecution(ov);
         return(rv);
        }
      currentPtr = currentPtr->next;
     }
     
   SetHaltExecution(ov);
   return(-1);
  }
 
/********************************************************/
/* SaveConstructs:  Saves the current set of constructs */
/*   into the specified file.                           */ 
/********************************************************/ 
globle int SaveConstructs(file_found)
  char *file_found;
  {
   struct saveFunction *save_ptr;
   FILE *file_ptr;

   
   if ((file_ptr = fopen(file_found,"w")) == NULL)
     { return(CLIPS_FALSE); }
   SetFastSave(file_ptr);
   
   /*==========================*/
   /* Save unusual constructs. */
   /*==========================*/
   
   save_ptr = ListOfSaveFunctions;
   while (save_ptr != NULL)
     {
      (*save_ptr->ip)((char *) file_ptr);
      save_ptr = save_ptr->next;
     }

   fclose(file_ptr);
   SetFastSave(NULL);
   
   return(CLIPS_TRUE);
  }
  
/****************************************************************/
/* AddSaveFunction: Adds a function to the ListOfSaveFunctions. */
/****************************************************************/
globle BOOLEAN AddSaveFunction(name,functionPtr,priority)
  char *name;
  int (*functionPtr)();
  int priority;
  {
   struct saveFunction *newPtr, *currentPtr, *lastPtr = NULL;

   newPtr = get_struct(saveFunction);

   newPtr->name = name;
   newPtr->ip = functionPtr;
   newPtr->priority = priority;
   
   if (ListOfSaveFunctions == NULL)
     {
      newPtr->next = NULL;
      ListOfSaveFunctions = newPtr;
      return(1);
     }
     
   currentPtr = ListOfSaveFunctions;
   while ((currentPtr != NULL) ? (priority < currentPtr->priority) : CLIPS_FALSE)
     {
      lastPtr = currentPtr; 
      currentPtr = currentPtr->next;
     }

   if (lastPtr == NULL)
     { 
      newPtr->next = ListOfSaveFunctions;
      ListOfSaveFunctions = newPtr;
     }
   else
     {
      newPtr->next = currentPtr;
      lastPtr->next = newPtr;
     }
   
   
   return(1);
  }
 
/************************************************************************/
/* RemoveSaveFunction: Removes a function from the ListOfSaveFunctions. */
/************************************************************************/
globle BOOLEAN RemoveSaveFunction(name)
  char *name;
  {
   struct saveFunction *currentPtr, *lastPtr;

   lastPtr = NULL;
   currentPtr = ListOfSaveFunctions;
   
   while (currentPtr != NULL)
     {
      if (strcmp(name,currentPtr->name) == 0)
        {
	 if (lastPtr == NULL)
           { ListOfSaveFunctions = currentPtr->next; }
         else
           { lastPtr->next = currentPtr->next; }
         rtn_struct(saveFunction,currentPtr);
         return(1);
        }
      lastPtr = currentPtr;
      currentPtr = currentPtr->next;
     }
     
   return(0);
  }
  
/**************************************************************/
/* SetCompilationsWatch: Sets the value of WatchCompilations. */
/**************************************************************/
globle VOID SetCompilationsWatch(value)
  int value;
  {
   WatchCompilations = value;
  }

/*****************************************************************/
/* GetCompilationsWatch: Returns the value of WatchCompilations. */
/*****************************************************************/
globle BOOLEAN GetCompilationsWatch()
  { return(WatchCompilations); }

/**************************************************************/
/* SetPrintWhileLoading: Sets the value of PrintWhileLoading. */      
/**************************************************************/
globle VOID SetPrintWhileLoading(value)
  BOOLEAN value;
  {
   PrintWhileLoading = value;
  }
  
/*****************************************************************/
/* GetPrintWhileLoading: Returns the value of PrintWhileLoading. */       
/*****************************************************************/
globle BOOLEAN GetPrintWhileLoading()
  {
   return(PrintWhileLoading);
  }
#endif

/************************************************************/
/* InitializeConstructs: Initializes the Construct Manager. */
/************************************************************/
globle VOID InitializeConstructs()
  {
#if DEBUGGING_FUNCTIONS && (! RUN_TIME) && (! BLOAD_ONLY)
   AddWatchItem("compilations",&WatchCompilations,30);
#endif
  }
  
/*********************************************/
/* ResetCLIPS: Resets the CLIPS environment. */
/*********************************************/
globle VOID ResetCLIPS()
  {
   static int resetting = CLIPS_FALSE;
   struct resetFunction *resetPtr;
   
   if (resetting) return;

   if (CurrentEvaluationDepth == 0) SetHaltExecution(CLIPS_FALSE);

   if ((BeforeResetFunction != NULL) ? ((*BeforeResetFunction)() == CLIPS_FALSE) : CLIPS_FALSE) return;
   
   resetting = CLIPS_TRUE;
   
   resetPtr = ListOfResetFunctions;
   while ((resetPtr != NULL) && (GetHaltExecution() == CLIPS_FALSE))
     {
      (*resetPtr->ip)();
      resetPtr = resetPtr->next;
     }
   
   resetting = CLIPS_FALSE;
  }
    
/**********************************************************/
/* SetBeforeResetFunction:                                */
/**********************************************************/
globle int (*SetBeforeResetFunction(fun_ptr))()
  int (*fun_ptr)();
  {
   int (*tmp_ptr)();
   
   tmp_ptr = BeforeResetFunction;
   BeforeResetFunction = fun_ptr;
   return(tmp_ptr);
  }
  
/*************************/
/* AddResetFunction:    */
/*************************/
globle BOOLEAN AddResetFunction(name,functionPtr,priority)
  char *name;
  VOID (*functionPtr)();
  int priority;
  {
   struct resetFunction *newPtr, *currentPtr, *lastPtr = NULL;

   newPtr = get_struct(resetFunction);

   newPtr->name = name;
   newPtr->ip = functionPtr;
   newPtr->priority = priority;
   
   if (ListOfResetFunctions == NULL)
     {
      newPtr->next = NULL;
      ListOfResetFunctions = newPtr;
      return(1);
     }
     
   currentPtr = ListOfResetFunctions;
   while ((currentPtr != NULL) ? (priority < currentPtr->priority) : CLIPS_FALSE)
     {
      lastPtr = currentPtr; 
      currentPtr = currentPtr->next;
     }

   if (lastPtr == NULL)
     { 
      newPtr->next = ListOfResetFunctions;
      ListOfResetFunctions = newPtr;
     }
   else
     {
      newPtr->next = currentPtr;
      lastPtr->next = newPtr;
     }
   
   return(1);
  }
 
/****************************/
/* RemoveResetFunction:    */
/****************************/
globle BOOLEAN RemoveResetFunction(name)
  char *name;
  {
   struct resetFunction *currentPtr, *lastPtr;

   lastPtr = NULL;
   currentPtr = ListOfResetFunctions;
   
   while (currentPtr != NULL)
     {
      if (strcmp(name,currentPtr->name) == 0)
	{
         if (lastPtr == NULL)
           { ListOfResetFunctions = currentPtr->next; }
         else
           { lastPtr->next = currentPtr->next; }
         rtn_struct(resetFunction,currentPtr);
         return(CLIPS_TRUE);
        }
      lastPtr = currentPtr;
      currentPtr = currentPtr->next;
     }

   return(CLIPS_FALSE);
  }
  
/****************************************************************/
/* ClearCLIPS: The purpose of this function is to clear the     */
/*   CLIPS environment.  All rules and facts are removed. The   */
/*   effect is as if CLIPS were completely restarted.           */
/*   Syntax:  (clear)                                           */
/****************************************************************/
globle VOID ClearCLIPS()
  {
   /*========================================*/
   /* Function not allowed from RHS of rule. */
   /*========================================*/
     
   if ((BeforeClearFunction != NULL) ? ((*BeforeClearFunction)() == CLIPS_FALSE) : CLIPS_FALSE) return;
   
   /*======================================*/
   /* Don't watch anything during a clear. */
   /*======================================*/

#if DEBUGGING_FUNCTIONS
   SetAllWatchItems(CLIPS_FALSE);
#endif

   /*============================*/
   /* Calls all clear functions. */
   /*============================*/

   CallClearFunctions();

   /*=======================*/
   /* Restore watch values. */
   /*=======================*/
   
#if DEBUGGING_FUNCTIONS
   RestoreAllWatchItems();
#endif
  }
  
/****************************/
/* CallClearFunctions:      */
/****************************/
globle VOID CallClearFunctions()
  {
   struct clearFunction *clear_ptr;
   
   clear_ptr = ListOfClearFunctions;
   while (clear_ptr != NULL)
     {
      (*clear_ptr->ip)();
      clear_ptr = clear_ptr->next;
     }
  } 

/**********************************************************/
/* SetBeforeClearFunction:                                */
/**********************************************************/
globle int (*SetBeforeClearFunction(fun_ptr))()
  int (*fun_ptr)();
  {
   int (*tmp_ptr)();
   
   tmp_ptr = BeforeClearFunction;
   BeforeClearFunction = fun_ptr;
   return(tmp_ptr);
  }

/*************************/
/* AddClearFunction:     */
/*************************/
globle BOOLEAN AddClearFunction(name,functionPtr,priority)
  char *name;
  VOID (*functionPtr)();
  int priority;
  {
   struct clearFunction *newPtr, *currentPtr, *lastPtr = NULL;

   newPtr = get_struct(clearFunction);

   newPtr->name = name;
   newPtr->ip = functionPtr;
   newPtr->priority = priority;
   
   if (ListOfClearFunctions == NULL)
     {
      newPtr->next = NULL;
      ListOfClearFunctions = newPtr;
      return(1);
     }
     
   currentPtr = ListOfClearFunctions;
   while ((currentPtr != NULL) ? (priority < currentPtr->priority) : CLIPS_FALSE)
     {
      lastPtr = currentPtr; 
      currentPtr = currentPtr->next;
     }

   if (lastPtr == NULL)
     { 
      newPtr->next = ListOfClearFunctions;
      ListOfClearFunctions = newPtr;
     }
   else
     {
      newPtr->next = currentPtr;
      lastPtr->next = newPtr;
     }
   
   
   return(1);
  }
   
/****************************/
/* RemoveClearFunction:    */
/****************************/
globle BOOLEAN RemoveClearFunction(name)
  char *name;
  {
   struct clearFunction *currentPtr, *lastPtr;

   lastPtr = NULL;
   currentPtr = ListOfClearFunctions;
   
   while (currentPtr != NULL)
     {
      if (strcmp(name,currentPtr->name) == 0)
        {
         if (lastPtr == NULL)
	   { ListOfClearFunctions = currentPtr->next; }
         else
           { lastPtr->next = currentPtr->next; }
         rtn_struct(clearFunction,currentPtr);
         return(1);
        }
      lastPtr = currentPtr;
      currentPtr = currentPtr->next;
     }
     
   return(0);
  }

/************************************************************/
/* ExecutingConstruct: Returns CLIPS_TRUE if a construct is */
/*   currently being executed, otherwise CLIPS_FALSE.       */
/************************************************************/
globle int ExecutingConstruct()
  { return(Executing); }

/************************************************************/
/* SetExecutingConstruct: Sets the value of the executing   */
/*   variable indicating that actions such as reset, clear, */
/*   etc should not be performed.                           */
/************************************************************/
globle VOID SetExecutingConstruct(value)
  int value;
  { 
   Executing = value; 
  }
