;;; -*- Mode:Lisp; Package: RPG; Syntax:Common-Lisp; Base:10;              -*-
;;;
;;; **************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; **************************************************************************
;;;
;;; Filename:   rpg-input
;;; Short Desc: Repertory Grid Acquisition / Data aquisition
;;; Version:    1.0b
;;; Status:     Beta testing
;;; Author:     ThE 
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;; --------------------------------------------------------------------------
;;; Last Modified By: Thomas E. Rothenfluh
;;; Last Modified On: Tue Feb 18 19:06:08 1992
;;; Update Count    : 2
;;; --------------------------------------------------------------------------
;;; Modification History
;;;   	        CLOSified and GINified version
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;
;;; This file contains the source code for the Acquisition phase of the
;;; Repertory Grid Knowledge Acquisition technique.
;;; 
;;; For more documentation, see the files rpg-desc.asc and ml.tex
;;; 

;;; ==========================================================================
;;; PACKAGE DECLARATIONS 
;;; ==========================================================================

(in-package :rpg)

(eval-when (compile eval load)
  (export '(elements constructs 
	    table-to-array 
	    listener-input
	    input-elements extract-element name-construct-poles
	    define-construct check-ratings)))

;;; ==========================================================================
;;; DATA STRUCTURES - Definitions and Access Functions
;;; ==========================================================================


;;; ELEMENTS is based on class TABLE with the slots 
;;; :attributes   which is a list of all ELEMENTS to be dealt with in RepGrid 
;;; :rows         which is a list of all RATINGS  for each attribute (ELEMENT)
;;;                and each CONSTRUCT

(defclass elements			; Define new class
    (table)				; Based on TABLE
    ((display   :initarg :display	; Store here the displays
		:initform nil
		:accessor display
		:type list)
     (tree      :initarg :tree  	; Store here the displays
		:initform nil
		:accessor tree 
		:type list))
  (:documentation "Class for ELEMENTS"))

(defmethod get-display (attribute (data elements))
  "Get the value in the display row."
  (nth (position attribute (attributes data)) (display data)))

(defmethod get-tree-disp (attribute (data elements))
  "Get the value in the tree row."
  (nth (position attribute (attributes data)) (tree-disp data)))

(defmethod pail-lib::change-type ((source elements) (target table))
  "Turn table across diagonal for ID3 input."
  (make-table
   :rows (mapcar #'(lambda (attr) (cons attr (get-values attr source)))
	  (cdr (attributes source)))
   :attributes (cons 'dump::elements
		     (get-values (car (attributes source)) source))))

;;; elements point to displays, which are structures.   When I copy
;;; one of these things, I really don't care to copy the graphic
;;; structures, so I get rid of them.
(defmethod pail-lib::clos-copy ((item elements))
  (prog (ans save)
    (setf save (display item))
    (setf (display item) nil)
    (setf ans (eval (database::get-fasd-form item :reset t)))
    (setf (display item) save)
    (return ans)))
  

(defmethod pail-lib::change-type ((source elements) (target elements))
  source)


;;; CONSTRUCTS is based on class TABLE with the slots
;;; :attributes   which is a list of all SCALE-VALUES for a CONSTRUCT
;;; :rows         which is a list of all LABELS for each SCALE-VALUE

(defclass constructs
    (table)				; Store here the numbers
    ((names   :initarg :names		; Store here the translations
	      :initform nil
	      :accessor names
	      :type list)
     (title   :initarg :title		; Store here the scale type
	      :initform nil
	      :accessor title
	      :type string)
     (scale   :initarg :scale		; Store here the scale type
	      :initform nil
	      :accessor scale
	      :type symbol)
     (display :initarg :display	        ; Store here the construct displays
	      :initform nil		; NOT USED YET, 21.11.91 ThE
	      :accessor display
	      :type list))
  (:documentation "Class for CONSTRUCTS"))


(defmethod print-construct ((construct constructs) &optional (stream t))
  (format-display stream "~% CONSTRUCT: ~a"            (title construct))
  (format-display stream "~%   Scale-Type~18T~a"       (scale construct))
  (format-display stream "~%   Labels    ~18T~{~15a~}" (names construct))
  (format-display stream "~%   Scores    ~18T~{~15a~}" (attributes construct))
  (format-display stream "~%   Elements  ~18T~{~15a~}" (rows construct)))


(defmethod print-object ((construct constructs) stream)
  (if *readable*
      (print-construct construct stream)
    (print-unreadable-object 
	(construct stream :type t :identity t))))


;;; Nice (?) support methods
(defmethod translate-score (value (data constructs))
  (nth value (names data)))


(defmethod translate-label (label (data constructs))
  (position label (names data)))


(defmethod table-to-array ((table table))
  "Transform the rows of a table into an array of appropriate dimensions."
  (make-array (list (number-of-rows table) 
		    (length (car (rows table))))
	      :initial-contents (rows table)))


;;; Converts a (Repertory Grid) table into a numerical array
;;; which can be used for cluster analysis.
;;; Uses the construct object when possible.

(defmethod fulltable-to-cleanarray ((table table) &optional (out nil))
  "Converts a table into a clean cluster array.
   Constructs of type binary or ordinal are converted into
   numerical values (0,1 or 0,1,2,3,4, respectively).
   Warns also about constructs with dontcare-items (stars) 
   which are NOT included for in the cluster array."
  (let ((new-table (get-table (attributes table) table)) ; a copy
	(new-rows nil)
	result construct values)
    (setf new-rows
      (dolist (x (rows table)(reverse result))
	(cond ((member '* x)		; DONTCARES?
	       (format-display out 
		      "~%WARNING: Your construct '~a' with values~
                       ~% ~a contains~
                       ~% 'dont care' asteriks and is therefore not included~
                       ~% in the statistical cluster analysis!" 
		       (car x)(cdr x)))
	      ((some #'symbolp (cdr x))	; SYMBOLS?
	       (format-display out 
		      "~%WARNING: Your construct '~a' with values~
                       ~% ~a contains literal symbols~
                       ~% which are now translated to integers~
                       ~% for statistical cluster analysis!" 
		       (car x) (setf values (cdr x)))
	       (if (stringp (car x))
		   (setf construct 
		     (eval (read-from-string (car x))))
		 (setf construct (car x)))
	       (cond ((and  
		       (not (equal (class-of construct)
				   (class-of 'anysymbol)))
		       (equal 'construct
			      (intern (class-name construct))))
		      ;; ----- Should be defined only for RepGrid constructs!
		      ;; ----- so we will use the defined labels
		      ;; ----- Translate into (0,1) or (0,1,2,3,4)
		      ;; ----- works for all ordinal scales
		      (setf x construct)
		      (push (cons (car x) 
				  (mapcar #'(lambda (y) 
					      (translate-label y construct)) 
					  values)) 
			    result))
		     (t			; ----- We have to construct them
		      (push 
		       (mapcar #'(lambda (x)
					 (position 
					  x 
					  (remove-duplicates values)))
				     values)
		       result))))
	      (t 
	       ;; ---------- Everything is just fine, include that construct
	       (push (cdr x) result) ))))
    (format-display out "~%~%Resulting array: ~a" result)
    (setf (rows new-table) new-rows)
    (make-array (list (number-of-rows new-table) 
		    (length (car (rows new-table))))
	      :initial-contents (rows new-table) )))


;;; ==========================================================================
;;; MAIN INPUT FUNCTIONS
;;; ==========================================================================


(defun input-elements (elements &optional (comment-disp t) 
					  (verbose-disp t))
  "Input of all ELEMENTS which are to be classified.
  Then check for sufficient and correct data and construct classes."
  (format-display comment-disp
		  "~%~%Please enter now all the ELEMENTS~
                   ~%you want to classify.~%")
  (make-instance 'elements 
    :attributes 
    (if elements 
	(check-input elements)
      (check-input 
       (input-list 
	"~&~%Please enter now all the ELEMENTS~
         ~%you want to classify.~%~
         ~%(Enter a word for each element):~%~%"
	:title "RPG:Element Input") 
       comment-disp verbose-disp))
    :rows () ))


;;; Extra function because one might define other criteria to check the input
(defun check-input (input &optional (comment-disp t) 
				    (verbose-disp t))
  "Check for sufficient elements and allows to complete the list."
  (let ((number (length input)))
    (when (< number 4)			; Minimal number of ELEMENTS
      (format-display comment-disp
		      "~%Sorry, you have to enter at least 4 ELEMENTS.~
	      ~%Otherwise, the TRIAD Technique cannot be applied.~
	      ~%Please enter at least ~R additional element(s)."
		      (- 4 number))
      (setf input 
	(append input 
		(input-list "Please type in the elements to classify: "
			    :title "RPG:Element Input")))
      (when (< (length input) 4)
	(error "Sorry, that's the end.")))
    (format-display verbose-disp
		    "~&The following ~R element~:P have been defined ~
                       by user input:~%~a" 
		    (length input) input)
    input))


;;; ==========================================================================
;;; SELECTION of one ELEMENT
;;; ==========================================================================


(defun extract-element (selection &optional (out t))
  "Allows the USER to select one ELEMENT from a list (usually a triad)."
  (let ((answer nil))
    (format-display out "~%~%The TRIAD ~s has been selected."
	    selection)
    (format-display out 
	       "~%Decide on a way that ~R of these elements are similar~
                 ~%and in the same way different from the remaining element."
	    (1- (length selection)))
    (format-display out 
	       "~%Please name the element which is different from the rest.")
    (setf answer (accept-menu-item selection))
    (list (list answer) (remove answer selection))))

;;; ==========================================================================
;;; CONSTRUCT functions
;;; ==========================================================================


(defun name-construct-poles (poles &optional (names nil) (out t) (in t))
  "Find names for the poles of a dimension [i.e., define a CONSTRUCT]."
  (let* ((pole (car poles))
	 (len (length pole))) 		; one extreme
    (cond ((null poles) 
	   (reverse names)) ;we are thru...
	  ((> len 1) 			; this is the similarity cluster
	   (format-display out
		   "~%~%Please supply a description which expresses ~
                    ~%the SIMILARITY between the elements ~a."
		   pole)
	   (format-display out
		   "~%>>>> Enter a single word, maybe use hyphens: ")
	   (push (list (read in) pole) names)
	   (name-construct-poles (cdr poles) names out in)) 
	  ((= len 1) 			; this is the dissimilar element
	   (format-display out
		   "~&What is it about ~s that makes it DIFFERENT ~
                      from the other elements?"
		   pole)
	   (format-display out
		   "~%>>>> Enter a single word, maybe use hyphens: " 
		   pole)
	   (push (list (read in) pole) names)
	   (name-construct-poles (cdr poles) names out in)) )))


(defun define-construct (names-elementlist elements &optional (out t) (in t))
  (let* ((pole-min (caar   names-elementlist))
	 (pole-max (caadr  names-elementlist))
	 (ele-min  (cadar  names-elementlist))
	 (ele-max  (cadadr names-elementlist))
	 (result nil)
	 (construct-name
	  (read-from-string 
	   (format-display nil "~a-~a" pole-min pole-max)))  ; Default name
	 (scale '(0 1))			             ; Default binary scale
	 (labels (list pole-min pole-max)))          ; Default labels
    ;; Inform the USER about defaults
    (format-display out
	    "~&The current scale for this construct is:~
	    ~%  Value = ~a, Name = ~a, example~:P = ~a~
	    ~%  Value = ~a, Name = ~a, example~:P = ~a~%~%" 
	    0 pole-min ele-min
	    (1- (length scale)) pole-max ele-max)
    ;; Ordinal scale wanted?
    (when (y-or-n-query 
	   "~&Do you want to enter a labelled 5-point rating-scale? ")
      (setf scale '(0 1 2 3 4))	       
      (setf labels nil)
      (format-display out
	      "~%Please enter the possible ratings (label names) for ~a.~%~
	      (in descending order from ~a down to ~a)~%~
	      [Use single words as labels, maybe use hyphens]."
	      construct-name pole-max pole-min)
      (format-display out
	      "~%    Numerical value 4 : ~a [Current example(s) = ~a]" 
	      pole-max ele-max)
      (when (y-or-n-query "~%>>>> Do you want to change this label? ")
	(format-display out "~%>>>> Please enter the new label name: ")
	(setf pole-max (read in)))
      (push pole-max labels)
      (format-display out "~&>>>> Numerical value 3 [new] : ")
      (push (read in) labels)
      (format-display out "~&>>>> Numerical value 2 [new] : ")
      (push (read in) labels)
      (format-display out "~&>>>> Numerical value 1 [new] : ")
      (push (read in) labels)
      (format-display out
	      "~&    Numerical value 0 : ~a [Current example(s) = ~a]~%" 
	      pole-min ele-min)
      (when (y-or-n-query "~%>>>> Do you want to change this label? ")
	(format-display out "~%>>>> Please enter the new label name: ")
	(setf pole-min (read in)))
      (push pole-min labels))
    ;; Allow renaming the construct
    (when (y-or-n-query 
	   "~%~%>>>> Do you want to enter a new name ~%for the CONSTRUCT ~a? "
	   construct-name)
      (format-display out "~%>>>> Please enter the new construct name: ")
      (setf construct-name (read in)))
    ;; Define the construct data structure
    (set construct-name
      (make-instance 'constructs 
	:attributes     scale
	:rows           (list ele-min ele-max)
	:names         labels))
    ;; Initialize the rating values 
    (dolist (ele (cdr (attributes elements)))
      (cond ((member ele ele-min)
	     (push 0 result))
	    ((member ele ele-max)
	     (push (1- (length scale)) result))
	    (t (push nil result))
      ))
    ;; USE GLOBAL VARIABLE??
    (append (list construct-name) (reverse result))))

;;; ==========================================================================
;;; RATING functions
;;; ==========================================================================


;;; This works still on lists, since TABLEs are not modifiable!!!
(defun get-rating-old (element-name elements ratings)
  (nth (position element-name (attributes elements)) ratings))

(defun get-rating (element-name elements ratings)
  (nth (position element-name (attributes elements)) ratings))

(defun modify-rating (element-name new elements ratings)
  (setf (nth (position element-name (attributes elements)) 
	     ratings) new))

;(untrace check-ratings)
(defun check-ratings (elements ratings &optional (out t))
  "Adds (or modifies) rating for ELEMENTs."
  (let ((construct (car ratings))
	(menu (names (eval (car ratings)))))
    (format-display out
	    "~%~%~D ELEMENTS are to be classified: " 
	    (cadr (attributes elements)))
    (mapc #'(lambda (x)
	      ;; (print (list x menu))
	      (let ((rat (get-rating x elements ratings)))
		(cond (rat 
		       (format-display out 
			       "~%~a --> ~a [~s]" 
			       x (translate-score rat (eval construct)) rat)
		       (when 
			   (y-or-n-query 
			    "~%>>>> Do you want to change this rating now? ")
			 (format-display out "~%")
			 (modify-rating x
					(translate-label
					 (accept-menu-item menu)
					 (eval construct))
					elements ratings) ))
		      (t
		       (format-display out
			       "~%>>>> Please enter a rating for element ~a:"
			       x)
		       (modify-rating x
				      (translate-label 
				       (accept-menu-item menu)
				       (eval construct))
				      elements ratings) ))))
	  (cdr (attributes elements)))
    (push ratings (rows elements))))


;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================

