;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:RPG; Base:10; -*-
;;;
;;; **************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH 
;;;
;;; **************************************************************************
;;;
;;; Filename:   rpg-cluster
;;; Short Desc: Statistics for Repertory Grid Analysis
;;; Version:    1.0b
;;; Status:     Beta testing
;;; Author:     ThE
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;; --------------------------------------------------------------------------
;;; Last Modified By: Thomas E. Rothenfluh
;;; Last Modified On: Thu Feb 20 12:56:28 1992
;;; Update Count    : 8
;;; --------------------------------------------------------------------------
;;; 19.11.91 --- ThE: Cluster-print output now also for gin-displays
;;;
;;; --------------------------------------------------------------------------
;;; EXERCISES:	Implement other clustering algorithms (e.g., Ward).
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;
;;; Compute similarity and distance measures for the Reperory Grid outputs,
;;; which can be used by clustering algorithms.
;;;
;;; Example data for testing are provided with the function 
;;; (init-test-arrays).

;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :rpg)

(eval-when (compile eval load)
  (export '(/-safe copy-array array-print
	    cluster-all cluster-it cluster-print
	    *cluster-display* *cluster-tree* *cluster-history*)))

;;; Some globals to keep track of all the matrix-wrangling:
(defvar *cluster-display* nil  
  "Data structure for drawing trees in text mode")
(defvar *cluster-tree*    nil  
  "Data structure for tree generation.")
(defvar *cluster-history* nil  
  "Starting clustering with nearest neighbor method.")
(defvar *cluster-to-element* '()  "Mapping of clusters to elements.")

;;; ==========================================================================
;;; DATA STRUCTURES - Definitions and Access Functions
;;; ==========================================================================

;;; CLUSTER-TREE is based on class TREE with the slots 
;;; :content      which is a list of all ELEMENTS to be dealt with in RepGrid 
;;; :descendants  which is a list of all RATINGS  for each attribute (ELEMENT)
;;;                and each CONSTRUCT

(defclass cluster-tree			; Define new class
    (tree)				; Based on TABLE
    ((distance  :initarg :distance	; Store here the cluster merge value
		:initform 0
		:accessor distance
		:type number))
  (:documentation "Class for CLUSTER-TREEs"))


(defmethod label ((tree cluster-tree))
  (if (or
       (> (length (descendants tree)) 1)
       (and (= (length (descendants tree)) 1)
	    (null (descendants (first (descendants tree))))))
      (symbol-name (content tree))
    (format nil "~a" (content tree))))

;;; ==========================================================================
;;; UTILITY FUNCTIONS (mainly for math and arrays)
;;; ==========================================================================

;;; --------------------------------------------------------------------------
;;; DIVISION BY ZERO (which is an error in pure COMMON LISP)
;;; --------------------------------------------------------------------------

;;; A safe way to make division-handling easier
;;; If any zeroes are detected in the (list of) divisor(s), 
;;; 0 is returned.
(defun /-safe (x &rest y)
  "Avoid zero division, just return 0 if ZeroDivide attempted."
  (cond ((notany #'zerop (if (listp y) y (list y)))
	 (apply #'/ (cons x y)))
	(t 
	 0)))

;;; Another recovery is to just remove all zeros from
;;; the list of divisors
(defun /-cerror (x &rest y)
  "Avoid zero division"
  (cond ((notany #'zerop (if (listp y) y (list y)))
	 (apply #'/ (cons x y)))
	(t 
	 (cerror "Skip all zeros."
		 "Attempt to divide ~a by some zero values in ~a.~%~
		 Division not done.~%"
		 x y)
	 (apply #'/ (cons x (remove 0 y)) ))))

;;; Example usage:
;;; (/-safe 100 10 2 0)

;;; --------------------------------------------------------------------------
;;; Some array-functions (which should make later porting to CLOS easier)
;;; --------------------------------------------------------------------------

;;; Implementation independent access functions for array-like structures:
;;; assumes a 2-dimensional array and returns "row" or "column" as a list
(defun get-array (&key array (column nil) (row nil))
  "Returns a list according to keyword arguments (row and/or column)
  as a selection from a two-dimensional array."
  ;; Argument checking (for most of the problematic cases)
  (when (and (typep array 'array) 	; is it an array
	     (= 2 (array-rank array)) 	; and of dimensionality 2
	     (or (null column) 		; and column either NIL
		 (and (numberp column) 	;     or any positive number
		      (>= column 0)))
	     (or (null row) 		; and row either NIL
		 (and (numberp row) 	;     or any positive number
		      (>= row 0))))
    ;; arguments ok, now get details of request
    (let ((result '()) 			; holds the resulting list
	  (rows (array-dimension array 0))  ; how many rows are there
	  (cols (array-dimension array 1))) ; and how many colums
      (cond ((and column (not row) 	; only a column wanted
		  (<= column (1- cols))) ; and requested column is available
	     ;; loop through whole array
	     (dotimes (n rows (reverse result))
	       (push (aref array n column) result)))
	    ((and row (not column) 	; only a row wanted
		  (<= row (1- rows))) 	; and requested row is available
	     ;; loop through whole array
	     (dotimes (n cols (reverse result))
	       (push (aref array row n) result)))
	    ((and column 		; column 
		  row 			; and row wanted
		  (<= column (1- cols)) ; and requested column is available
		  (<= row (1- rows))) 	; and requested row is available
	     ;; that is a simple aref
	     (push (aref array row column) result))))))

;;; Example usage:
;;; (get-array :array grid-array :column 1)
;;; (get-array :array grid-array :row 0)
;;; (get-array :array grid-array :column 0 :row 0)
;;; (get-array :array grid-array :column 5)  ; should return NIL (no such col)

;;; Return a copy all elements of an array to avoid surgery on the original
(defun copy-array (old-array)
  "Return a copy of all elements of an array."
  (let ((rows (array-dimension old-array 0))
	(cols (array-dimension old-array 1))
	(tmp-array nil))
    (setf tmp-array (make-array (list rows cols)))
    (dotimes (r rows tmp-array)
      (dotimes (c cols)
	(setf (aref tmp-array r c)
	      (aref old-array r c))))))


;;; Printing of arrays in readable format
(defun array-print (array &optional (out nil))
  "Prints a two-dimensional array."
  (when (<= 2 (array-rank array)) 	; check dimensionality
    (format-display out "~%")
    (dotimes (n (array-dimension array 0))
      (dotimes (m (array-dimension array 1))
	(format-display out "~2,8T~7,3F" (aref array n m)))
      (format-display out "~%")
      )))

;;; --------------------------------------------------------------------------
;;; STANDARDIZATION OF VALUES (Z-TRANSFORMATION)
;;; --------------------------------------------------------------------------


(defun z-transform (list)
  "Transform a list into Z-Values (Average=0, Std deviation=1)."
  ;; for efficiency, everything is done locally within this function.
  (when (listp list)
    (let ((n (length list)) 
	  (new '())
	  (sum 0) (sum2 0) 
	  (average 0) (stddev 0))
      ;; now loop trough all elements of a row
      (dolist (x list)
	(setf sum (+ sum x)) 		; summing up
	(setf sum2 (+ sum2 (* x x))) 	; quadratic sum
	)
      (setf stddev  (sqrt (/ (- sum2 (/ (* sum sum) n)) n)))
      (setf average (/ sum (float n)))
      (dolist (x list (reverse new))
	(push (/ (- x average) stddev) new)) )))

;;; Example usage:
;;; (setf l0 '(3.3 1.7 2.0 4.0 1.3 2.0 3.0 2.7 3.7 2.3 1.7 2.3))
;;; (setf l1 (z-transform l0))

;;; ==========================================================================
;;; PLAY DATA (used for testing and demonstration)
;;; ==========================================================================


;;; Make some arrays as test cases
(defun init-test-arrays ()
  "Function to initialize test case data."
  ;; the DECLARE SPECIAL IS JUST USED FOR THE TEST DATA
  (declare (special grid-array grid-array-2 grid-array-3
		    everitt-array grid-array-t))
  (setf grid-array 
    ;; A more or less realistic example from Repertory Grid Output
    (make-array '(4 5)			; create a 2-dimensional array
		:initial-contents
		'(( 0 1 1 1 0)
		  ( 1 0 1 1 1)
		  ( 1 0 1 1 0)
		  ( 0 1 1 1 0)
		  )))
  (setf grid-array-2
    (make-array '(4 5) 
		:initial-contents
		'(( 6.0 1.7 4.0 4.0 4.0)
		  ( 6.0 1.9 4.0 4.0 4.0)
		  ( 3.0 2.3 1.7 1.7 1.6)
		  ( 3.0 2.3 1.7 1.7 1.7
		   ))))
  (setf grid-array-3
    (make-array '( 2 4 ) 
		:initial-contents
		'(( 4 4 2 2 )
		  ( 0 1 0 0 )
		  )))
  (setf everitt-array
    (make-array '(5 3) 
		:initial-contents
		'((  66.0  72.0  70.0)
		  ( 120.0 130.0 150.0)
		  (    1     2     1 )
		  (    1     2     1 )
		  (    1     1     0 )
		  )))
  (setf grid-array-t
    ;; Test case for statistics (Bortz, 1977, p. 57ff)
    ;; Mean     : 2.5
    ;; Variance : 0.66
    ;; Stddev   : 0.81
    ;; Well, that's essentially just a vector
    (make-array '(1 12) 
		:initial-contents
		'(( 3.3 1.7 2.0 4.0
			1.3 2.0 3.0 2.7
			3.7 2.3 1.7 2.3))))
  'done)

;;; Evaluate before playing around with the examples below:
(eval-when (load eval)
  (init-test-arrays))

;;; ==========================================================================
;;; SIMILARITY AND DISTANCE MEASURES
;;; ==========================================================================
;;;
;;; Compute similarity measure for binary values [0, 1]
;;; Similarity = sum over scores / sum over weights 
;;;
;;;                              Value of variable k: X[k]
;;;
;;; Value of individual i (vi)        1  1  0  0      X[ik]
;;; Value of individual j (vj)        1  0  1  0      X[jk]
;;; --------------------------------------------------
;;; Weight W[ijk]                     1  1  1  0
;;;  (= validity of comparison)
;;; Score  S[ijk]                     1  0  0  0      (for binary data)
;;; Score  S[ijk]                     1  0  0  1      (for qualitative data)
;;; Score  S[ijk]               1-Abs(X[ik]-X[jk])/R  (for quantiative data)
;;;                                                    R=Range of variable k
;;;
;;;
;;;
;;; The distance measure can then be computed from those similarites by
;;;
;;;        D[ij] = SQRT(2*(1-S[ij]))
;;;
;;;
;;; see Everitt, 1974, p. 54ff (originally by Gower, 1971)
;;; ==========================================================================
;;; COMPUTE A SIMILARITY MATRIX
;;; ==========================================================================
;;;
;;; This is the main part: To compute a similarity matrix for all elements,
;;;                        which are obtained as output from the Rep Grid.
;;;
;;; The similarity is first computed for every variable resulting
;;; in a SCORE and a WEIGHT.  These values are then summed over all
;;; variables and divided to return the final similarity value.
;;;
;;; The function COMPUTE-SIMILARITY-MATRIX does the computation for
;;; all elements, whereas the function COMPUTE-SIMILARITY only computes
;;; the similarity between two specific elements.

(defun compute-similarity-matrix (array &optional (out nil))
  "Compute the matrix with the similarity measures for all individuals."
  (let ((k (1- (array-dimension array 0))) ; # of variables [k] (=rows)
	(n (1- (array-dimension array 1))) ; # of elements  [N] (=cols)
	)
    (format-display out "~&Computing similarities for ~a variables & ~a elements.~%"
	    (1+ k) (1+ n))
    (let ((s-array (make-array (list (1+ n)(1+ n)) :initial-element 0)))
      (do ((i 0 (1+ i)))
	  ((> i n) s-array)
	(do ((j i (incf j)))
	    ((> j n) nil)
	  ;; (format-display out "~%[~a,~a] " i j ) ; DEBUG only
	  (cond ((= i j)
		 (setf (aref s-array i j) 0.0)) ; distance is zero
		(t
		 ;; Similarity
		 (setf (aref s-array i j)
		   (compute-similarity array i j out))
		 ;; Distance
		 (setf (aref s-array j i)
		   (sqrt (* 2 (- 1 (aref s-array i j))))
		   )
		 #|
                 ;; DEBUG only
		 (format-display out "~110T~a ~130T~a ~150T~a" 
			 (aref s-array i j)
			 (aref s-array j i)
			 (- 1 (aref s-array i j))
			 )
                 |#
		 )
		)) ))))

;;; Example usage:
;;; (array-print (compute-similarity-matrix grid-array-3))
;;; (array-print (setf d-s-arr (compute-similarity-matrix everitt-array)))
;;; (array-print (compute-similarity-matrix everitt-array t))

;;; ==========================================================================
;;; COMPUTE THE APPROPRIATE SIMILARITY MEASURE FOR TWO INDIVIDUALS
;;; ==========================================================================

;;; This function determines the type of the scale in which the
;;; raw data are coded (binary, nominal/ordinal, interval) and
;;; dispatches the actual computation to a specialized function

(defun compute-similarity (array i j &optional (out nil))
  "Compute similarity between two individuals i and j,
  use appropriate similarity measure according to scale type
  [one of BINARY, QUALITATIVE/ORDINAL, QUANTITATIVE/INTERVAL]."
  (format-display out "~&Computing similarity between elements ~a & ~a." i j)
  (let ((k (1- (array-dimension array 0))) ; # of variables [k]
	(s-sum 0)			; Scores
	(w-sum 0)			; Weights
	(sim 0))			; Similarity
    (do ((row 0 (1+ row)))		; iterate through all variables
	((> row k) 
	 (/-safe (float s-sum) w-sum))	; return similarity value
      (let ((row-data (get-array :array array :row row)))
	;; The type checking is done with several assumptions:
	;; 1 - If all values of a variable are either 0's or 1's
	;;    then binary data are assumed
	;; 2 - If all values of a variable are integers
	;;    then nominal/ordinal data are assumed
	;; 3 - If all values of a variable are floating point numbers
	;;    then interval data are assumed (and z-transformed)
	(format-display out "~% Checking data type of row ~a = ~a:" row row-data)
	(cond ((every (function (lambda (x)
				  (or (= 0 x)
				      (= 1 x))))
		      row-data)
	       (setf sim 
		 (compute-binary-similarity
		  row-data i j out)))
	      ;; Qualitative data range from 0 to 4
	      ((every #'integerp row-data)
	       (if (> (apply #'max row-data) 4)
		   (setf sim		; out of range
		     (compute-quantitative-similarity
		      row-data i j out))
		 (setf sim		; that's ordinal
		   (compute-qualitative-similarity
		    row-data i j out))))
	      ;; ----- One float indicates quantitative
	      ((some #'floatp row-data)	
	       (setf sim 
		 (compute-quantitative-similarity
		  row-data i j out)))
	      (t 
	       (format-display out
		       "~&WARNING within function COMPUTE-SIMILARITY:~
		       ~%Scale type could not be determined.~
		       ~%Using qualitative data as default.")
	       (setf sim 
		 (compute-qualitative-similarity 
		  row-data i j out))))
	(setf s-sum (+ s-sum (car sim)))
	(setf w-sum (+ w-sum (cadr sim)))
	;; (format-display out "~70Ts-sum = ~a; w-sum = ~a; sim = ~a" s-sum w-sum sim)
	)) ))

;;; EXAMPLE usage:
;;; (array-print grid-array)
;;; (compute-similarity grid-array 1 2)
;;; (compute-similarity grid-array 3 4)
;;; (compute-similarity grid-array 2 3)

;;; The following examples are as used in Everitt, 1974 on page 55:
;;; (array-print everitt-array)
;;; (compute-similarity everitt-array 1 1 t) ; This is really 1.0
;;; (compute-similarity everitt-array 1 2)   ; Should be 0.333333
;;; (compute-similarity everitt-array 1 3)   ; Should be 0.466666
;;; (compute-similarity everitt-array 2 3 t) ; Should be 0.200000

;;; --------------------------------------------------------------------------
;;; SIMILARITY MEASURE FOR BINARY DATA FOR ONE ROW (= ONE VARIABLE)
;;; (usually from 0-1 / or yes-no scales)
;;; --------------------------------------------------------------------------
 

(defun compute-binary-similarity (data i j &optional (out nil))
  "Compute similarity measure for binary data (after Gower, 1971)."
  (format-display out "Binary")
  (let ((s-sum 0)			; Weight
	(w-sum 0)			; Score
	(vi (nth i data))		; Value of element i
	(vj (nth j data)))		; Value if element j
    (cond ((= vi vj)
	   (when (= vi 1) 	
	     ;; Score S[ijk] 
	     (incf s-sum)
	     ;; Weight W[ijk]
	     (incf w-sum))
	   ;; otherwise Score & Weight = 0, default from initialization
	   )
	  (t				; (/= vi vj)
	   ;; Score  S[ijk] always 0, from default
	   ;; Weight W[ijk] always 1, since (0 0) is excluded above
	   (incf w-sum))) 
    (list s-sum w-sum)))
#|
;;; DEBUG only
(if (zerop w-sum)
         (format-display out 
		 "~&-> Score = ~a, Weight = ~a, Similarity = 0" s-sum w-sum)
        (format-display out "~&-> Score = ~a, Weight = ~a, Similarity = ~a" 
                 s-sum w-sum (/-safe s-sum w-sum)))
|#

;;; --------------------------------------------------------------------------
;;; SIMILARITY MEASURE FOR QUALITATIVE DATA FOR ONE ROW (= ONE VARIABLE)
;;; (usually from nominal or ordinal scale)
;;; --------------------------------------------------------------------------


(defun compute-qualitative-similarity (data i j &optional (out nil))
  "Compute similarity measure for qualitative data (after Gower, 1971)."
  (format-display out "Qualitative")
  (let ((s-sum 0)			; Weight
	(w-sum 0)			; Score
	(vi (nth i data))
	(vj (nth j data)))
    (cond ((= vi vj)
	   ;; Equal integers: score S[ijk] is always 1 in the qualitative case
	   (incf s-sum)
	   ;; AND WE ASSUME HERE THAT MEASUREMENTS ARE ALL VALID!
	   (incf w-sum))
#|
           ;; IMPLEMENTATION NOTE:
           ;; If we would use 0 and 1 for the qualitative case, too
           ;; than the following would be correct:
	   (when (= vi 1) 	
	     ;; Weight W[ijk]
	     (incf w-sum))
	   ;; otherwise Weight = 0, default from initialization
	   )
	  (t				; (/= vi vj)
	   ;; Score  S[ijk] always 0, from default
	   ;; Weight W[ijk] always 1, since (0 0) is excluded above
	   (incf w-sum))
|#
          )
    (list s-sum w-sum)))

;;; --------------------------------------------------------------------------
;;; SIMILARITY MEASURE FOR QUANTITATIVE DATA FOR ONE ROW (= ONE VARIABLE)
;;; (values should have interval scale level)
;;; --------------------------------------------------------------------------


(defvar *z-flag* t) 			; suppress z-transformations if nil

(defun compute-quantitative-similarity (data i j &optional (out nil))
  "Compute similarity measure for quantitative data (after Gower, 1971).
  Return a list : (score weight)."
  (format-display out "Quantitative")
  (when *z-flag*
    (format-display out "~&[*z-flag*] : Z-Transforation on raw data is done.")
    (setf data (z-transform data)))
  (let ((s-sum 0) 
	(vi (nth i data))
	(vj (nth j data))
	(rk (abs (- (apply #'max data)
		    (apply #'min data)))))
    ;; Score S[ijk] is computed in this case as follows:
    (setf s-sum 
	  (- 1
	     (/-safe (float (abs (- vi vj)))
		     rk)))
    (list s-sum 1)))


;;; ==========================================================================
;;; COMPUTE THE NEAREST NEIGHBOUR
;;; ==========================================================================


(defun init-cluster-list (array)
  "Initialize the cluster variable from ARRAY."
  (let ((n (1- (array-dimension array 0)))
	(result nil))
    (do ((x 0 (1+ x)))
	((> x n) (reverse result))
      (push (list x (list x)) result))))

(defun find-clusters (array &optional (out nil))
  "Find the smallest distance value (lower left part of matrix)."
  (let* ((n (1- (array-dimension array 0)))
	 (min most-positive-fixnum) (min-i nil) (min-j nil) 
	 (max most-negative-fixnum) (max-i nil) (max-j nil)
	 ;; (mind most-positive-fixnum)  ; useful for WARD etc
	 ;; (maxd most-negative-fixnum) 
	 ;; (new-d 0) 
	 (cands nil)
	 (to-del nil)
	 (to-del-el nil)
	 (to-merge nil)
	 (to-merge-el nil)
	 (n-array (make-array (list (1+ n) (1+ n)) :initial-element 0)))
    (format-display out "~&Searching smallest distance in ~ax~a matrix.~%" 
	    (1+ n) (1+ n))
    (do ((i 1 (1+ i)))
	((> i n) nil)
      (do ((j 0 (incf j)))
	  ((= j i) nil)
	(let ((val (aref array i j)))
	  ;; (format-display out "~&[~a,~a] = ~a" i j val)
	  (when (< val min) (setf min val) (setf min-i i) (setf min-j j))
	  (when (> val max) (setf max val) (setf max-i i) (setf max-j j)))))
    (format-display out "~&Found smallest distance of ~a between i=~a and j=~a." 
	    min min-i min-j)
    (format-display out "~&Computing new distances.")
    (setf cands 
      (do ((x 0 (1+ x))(result nil))
	  ((> x n) (reverse result))
	(when (not (or (eq x min-i)(eq x min-j)))
	  (push x result))))
    (dolist (c1 cands nil)
      (dolist (c2 cands nil)
	(setf (aref n-array (max c1 c2)
		    (min c1 c2)) 
	  (aref array (max c1 c2)(min c1 c2)))))
    (dolist (c1 cands nil)
      (dolist (c2 (list (min min-i min-j)) nil)
	(setf (aref n-array (max c1 c2)
		    (min c1 c2))
	  (aref array (max c1 c2)(min c1 c2)))))
    (format-display out "~%Merging clusters ~a and ~a at distance ~a."
	    min-i min-j min)
    ;; Determine clusters and elements (unsorted)
    (setf to-del      min-i)
    (setf to-del-el   (cadr (nth to-del *cluster-to-element*)))
    (setf to-merge    min-j)
    (setf to-merge-el (cadr (nth to-merge *cluster-to-element*)))
    (push
     (list
      (append to-del-el to-merge-el)	; new set
      min				; merge distance
      to-del-el				; descendant-1
      to-merge-el			; descendant-2
      )
     *cluster-tree*)
    ;; Determine clusters and elements (sorted for text drawing)
    (setf to-del (max min-i min-j))
    (setf to-del-el (cadr (nth to-del *cluster-to-element*)))
    (setf to-merge (min min-i min-j))
    (setf to-merge-el (cadr (nth to-merge *cluster-to-element*)))
    ;; Update history and print-variables
    (setf *cluster-history* 
      (concatenate 'string
	*cluster-history*
	(format nil 
		"~%Merging clusters ~a [~a] with ~a [~a] at distance ~a."
		to-merge to-merge-el to-del to-del-el min)))
    (setf *cluster-display* 
      (append (list (append (list min )
			    (if (> (length to-merge-el)
				   (length to-del-el))
				(list to-merge-el to-del-el)
			      (list to-del-el to-merge-el))
			    
			    ))
	      *cluster-display*))
    (when out
      (dotimes (x (1+ n) *cluster-to-element*)
	(format-display out "~%Cluster: ~a, Elements: ~a" 
		x (cadr (nth x *cluster-to-element*)))
	(when (member x cands)
	  (format-display out "  untouched."))
	(when (= x to-del)
	  (format-display out "  to be deleted."))
	(when (= x to-merge)
	  (format-display out "  to be merged.") )))
    ;; Append the to be merged elements to new cluster
    (setf (cadr (nth to-merge *cluster-to-element*)) 
      (append to-del-el to-merge-el))
    ;; Remove the merged cluster
    (delete (nth to-del *cluster-to-element*) *cluster-to-element*)
    ;; Reduce the array
    (reduce-array n-array to-del out) ))


;;; EXAMPLE usage:
;;; (array-print (setf d-s-0 (compute-similarity-matrix grid-array t)))
;;; (setf *cluster-to-element* (init-cluster-list d-s-0))
;;; (setf *cluster-history* "start")
;;; (array-print d-s-0)
;;; (array-print (setq c-p-1 (find-clusters d-s-0 t)))
;;; (array-print (setq c-p-2 (find-clusters c-p-1 t)))
;;; (array-print (setq c-p-3 (find-clusters c-p-2 t)))
;;; (array-print (setq c-p-4 (find-clusters c-p-3 t)))
;;; *cluster-to-element*


(defun reduce-array (array rc-remove &optional (out nil))
  "Row and Col RC-REMOVE are removed from the array."
  (let* ((n (1- (array-dimension array 0)))
	 (row-put 0) (col-put 0)
	 (n-array (make-array (list n n) :initial-element 0)))
    (format-display out "~&Shrinking array by removing row and col ~a."
	    rc-remove)
    (do ((row 0 (1+ row)))	
	((> row n) nil)
      (unless (eq row rc-remove)
	(when (not (eq row row-put))
	  (setf *cluster-history* 
		(concatenate 'string
		  *cluster-history*
		  (format nil "~%Cluster ~a becomes new cluster ~a."
			  row row-put)))
	  (format-display out "~%Cluster ~a becomes new cluster ~a."
		  row row-put) )
	(setf col-put 0)
	(do ((col 0 (1+ col)))
	    ((> col n) nil)
	  (unless (eq col rc-remove)
	    (setf (aref n-array row-put col-put) (aref array row col))
	    (incf col-put)))
	(incf row-put)))
    n-array))

;;; ==========================================================================
;;; PRINTING FUNCTIONS
;;; ==========================================================================


(defun cluster-print (c-display &optional (out nil))
  "Nice printing routine for clusters.
   If out is t, the Lisp-Listener is used,
   If out is a gin-display, that is used."
  (declare (special *output-button*))
  (if (displayp out)
    (activate-display out))
  (let* ((rpg-result (get-table 
		      (cdr (attributes 
			    (start-object (button-value *output-button*)))) 
		      (start-object (button-value *output-button*))))
	 (order (append (nth 1 (car c-display))
			(nth 2 (car c-display))))
	 ;; character-length of element names (minium 5)
	 (len (max 5 (+ 2 (string-counter (attributes rpg-result) #'max))))
	 (inlist nil)(outlist nil)
	 ;; pictures for drawing the cluster dendrogram
	 (spd "  +--")		   
	 (dps "--+  ")
	 (ddd "-----")
	 (sss "     ")
	 (sls "  |  ")
	 (adj 0)
	 (outstring ""))
    (when (evenp len) (incf len))	; an odd length is required
    ;; big dendrograms hav to fit into 100 characters, 
    ;; which allows 14 elements to be displayed nicely
    (when (> (* len (length (attributes rpg-result))) 99) (setf len 7))
    (when (> len 5)			; adjust the subpart pictures
      (setf adj (floor (- len 5) 2))
      (setf spd (string-pad spd  :pad-char #\Space :before adj))
      (setf spd (string-pad spd  :pad-char #\-     :after  adj))
      (setf dps (string-pad dps  :pad-char #\-     :before adj))
      (setf dps (string-pad dps  :pad-char #\Space :after  adj))
      (setf ddd (string-pad ddd  :pad-char #\-     :before adj :after adj))
      (setf sss (string-pad sss  :pad-char #\Space :before adj :after adj))
      (setf sls (string-pad sls  :pad-char #\Space :before adj :after adj))
      )
    (format-display out "~%~% Cluster diagram:~%")
    (format-display out "~%   Elem.~10T~a" 
		    (translate-index-to-pad-string order len))
    (format-display out "~%   Elem.~10T~a" 
		    (translate-to-pad-string order len))
    (setf outstring (format nil "~%~%   Dist.~10T"))
    (dotimes (n (length order))
      (setf outstring (concatenate 'string outstring (format nil "~a" sls))))
    (format-display out outstring)
    (setf inlist order)
    ;; ---------- Now loop through the accumulated history trace
    (dolist (x (reverse c-display))	; start from unmerged
      (let ((distance (car x))
	    (g1 (nth 1 x)) 
	    (g2 (nth 2 x)) 
	    (g0 nil))
	(cond ((member (first g1) (cdr (member (first g2) order)))
	       (setf g0 g1)(setf g1 g2)(setf g2 g0)))
	(setf g0 (append g1 g2))
	(if (displayp out)		
	    ;; ---------- format-display does always add a newline
	    (setf outstring (format nil "~&~8,3f :" distance g0 ))
	  (setf outstring (format nil "~%~8,3f :" distance g0 )))
	(dolist (a order)
	  (cond ((= a (first g1))
		 (setf outstring (concatenate 'string outstring
					      (format nil "~a" spd)))
		 (push a inlist))
		((= a (first g2))
		 (setf outstring (concatenate 'string outstring
					      (format nil"~a" dps)))
		 (push a outlist))
		((member a g1)
		 (setf outstring (concatenate 'string outstring
					      (format nil"~a" ddd))))
		((member a outlist)
		 (setf outstring (concatenate 'string outstring
					      (format nil "~a" sss))))
		((member a inlist)
		 (setf outstring (concatenate 'string outstring
					      (format nil"~a" sls))))
		(t
		 (setf outstring (concatenate 'string outstring
					      (format nil "xxxxx"))))) )
	(format-display out outstring)
	(translate-to-names g1) (translate-to-names g2))
      (if (displayp out)		
	  ;; ---------- format-display does always add a newline
	  (setf outstring (format nil "~&~10T"))
	(setf outstring (format nil "~%~10T")))
      (dolist (a order)
	(cond ((member a outlist)
	       (setf outstring (concatenate 'string outstring
					    (format nil "~a" sss))))
	      ((member a inlist)
	       (setf outstring (concatenate 'string outstring
					    (format nil "~a" sls))))
	      (t
	       (setf outstring (concatenate 'string outstring
					    (format nil "xxxxx"))))))
      (format-display out outstring))
    (format-display out "~%   Dist.~%") 
    (format-display out "~%   Elem.~10T~a" 
		    (translate-to-pad-string order len))
    (format-display out "~%   Elem.~10T~a" 
		    (translate-index-to-pad-string order len))
    (format-display out "~%~%")
    t))

;;; EXAMPLE usage:
;;; (cluster-print *cluster-display* t)

(defun translate-index-to-list (list)
  (declare (special *output-button*))
  (let ((result nil))
    (dolist (el list (reverse result))
      (push 
       (nth el 
	    (attributes 
	     (get-table 
	      (cdr (attributes 
		    (start-object (button-value *output-button*)))) 
	      (start-object (button-value *output-button*))))) result))))



(defun translate-index-to-pad-string (str len)
  (declare (special *output-button*))
  (let ((result nil))
    (dolist (el str result)
      (let* ((item 
	      (nth el (attributes 
		       (get-table 
			(cdr (attributes 
			      (start-object (button-value *output-button*)))) 
			(start-object (button-value *output-button*))))))
	     (el-len    (cond ((> (string-length item) len)
			       (setf item (concatenate 'string
					   (subseq (string item) 0 (1- len))
					   " "))
			       len)
			      (t (string-length item))))
	     (addbefore (floor (- len el-len) 2))
	     (addafter  (- len el-len addbefore)))
	(setf result 
	  (concatenate 'string 
	    result
	    (string-pad item
			:before addbefore
			:after addafter)))))))


(defun translate-to-pad-string (str len)
  (let ((result nil))
    (dolist (el str result)
      (let* ((item      (format nil "~a" el))
	     (el-len    (string-length item))
	     (addbefore (floor (- len el-len) 2))
	     (addafter  (- len el-len addbefore)))
	(setf result 
	  (concatenate 'string 
	    result
	    (string-pad item
			:before addbefore
			:after addafter)))))))
;;; (print (translate-to-pad-string '(0 1 2 3 4 5 6 7 8 9 10 11) 9))


(defun translate-to-names (l)
  (declare (special *output-button*))
  (let ((result nil))
    (dolist (el (reverse l) result)
      (push 
       (nth el (attributes 
		(get-table 
		 (cdr (attributes 
		       (start-object (button-value *output-button*)))) 
		 (start-object (button-value *output-button*))))) result))))

; (format-display t "~a" (translate-to-names '(1 2)))

;;; ==========================================================================
;;; DEMO FUNCTIONS
;;; ==========================================================================


(defun cluster-all (matrix clusters &optional (out nil))
  (cond ((<= (length clusters) 1)
	 clusters)
	(t 
	 (cluster-all (find-clusters matrix out) clusters))))

(defun cluster-it (data-array &optional (out nil))
  (declare (special *cluster-to-element* 
		    *cluster-history* *cluster-display*))
  (let ((d-s-m (compute-similarity-matrix data-array out)))
    (format-display out "~% Initial matrix:~%~%")
    (array-print data-array out)
    (format-display out "~% Distance and Similarity matrix:~%~%")
    (array-print d-s-m out)
    (setf *cluster-to-element* (init-cluster-list d-s-m))
    (setf *cluster-history* "Nearest neighbor clustering starts.")
    (setf *cluster-display* '())
    (setf *cluster-tree*    '())
    (cluster-all d-s-m *cluster-to-element* out)
    (format-display out "~&History of clustering:~%~a" *cluster-history*)
    (cluster-print *cluster-display* out)
    ))

;;; EXAMPLE usage:
;;; (cluster-it grid-array-3 t)


;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================


