;;; -*- Mode: LISP; Package: PLANNING; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   graphics.cl
;;; Short Desc: Graphic support for planning module, 
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   16.1.92 DTA
;;; Author:     Hank Han
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
;;; Graphics for Planner module of PAIL

;; Main routine here is (draw-tree tree), draws an interactive tree
;; that allows scrolling up and down: left button redraws tree with
;; node selected at top.  Right button redraws tree with node selected
;; at bottom.  Middle button displays further information on the node.

(in-package :planning)

(export '(init-planner-window draw-tree))

(defparameter *default-planner-font* (open-font-named "5x8"))

(defvar *planner-window* nil)
(defvar *node-info-window* nil)
(defvar *move-window* nil)

(defvar *planner-active-regions* nil)	; ones currently under use
(defvar *planner-unused-active-regions* nil) ; free active region queue
(defvar *outplan* nil)
(defvar *animate*)
(defvar *blocksize*)
(defvar *default-display-host*)

(defun init-planner-window ()
  
  (let ((window-width (min 400 (round (width *root-window*) 2)))
	(window-height (min 300 (- (round (* (height *root-window*) .40)) 20))))
      (unless (and *planner-window*
	       (member (window *planner-window*) cw:*all-window-streams*))
    (setf *planner-window*
      (make-instance 'display :title "Search Tree"
		    :left 0
		    :bottom 1
		    :width window-width
		    :height window-height
		    :font (findfont (width *root-window*) 1140 9)
		    :flush-method '(lambda (&rest arg)
				    (declare (ignore arg))
				    (setf *planner-active-regions* nil))
		    :borders 1))
    (setf *planner-active-regions* nil)
    (setf *planner-unused-active-regions* nil)
    (init-node-info-window)
    
    #| (sleep 0.5) |#)
  #| (init-movie-window start-state) |#))

(defun init-movie-window (type  start-state)
  (cond ((equal "blocks" type)
	 (let (hand)
	  (setf *movie-window* (make-instance 'block-display
					     :title "Blocks World"
					     :width 400
					     :height 300
					     :borders 1
					     :left (- (width *root-window*) 400)
					     :bottom 1
					     )
		)
	  (setf planning::*blocksize* 40)
	  (setf (hand-home *movie-window*)  '(0 294))

	  (setf hand (make-instance 'hand :name 'hand
				    :position (list 0 (- (height *movie-window*) 6))
				    :display *movie-window*))
	  (setf (hand *movie-window*) hand)
	  (if (not *animate*) (deactivate-display *movie-window*)))
	 (if start-state (setf (start-state *movie-window*)
			  start-state)))
	((equal "maze" type)
	 (setf *movie-window* (make-instance 'maze-display
					     :title "Maze"
					     :width 200
					     :height 200
					     :borders 1
					     :left (- (width  *root-window*) 250)
					     :bottom 1
					     ))
	 (setf (start-state *movie-window*)
	   start-state))
	(T nil)))




(defun draw-tree (planning-tree )
  (init-planner-window  )
  (free-all-active-regions)
  (clear-display *planner-window*)
  (assign-node-positions planning-tree
			 (width *planner-window*)
			 (height *planner-window*)
			 10)
  (draw-tree-nodes planning-tree 10))

(defmacro mappend (fn l)
  `(apply #'append (mapcar ,fn ,l)))

(defun assign-node-positions (planning-tree width height &optional n-layers)
  (let* ((leafs (leafs planning-tree n-layers))
	 nodes (x 0)
	 (h-gap (/ width (+ (length leafs) 1.0)))
	 (v-gap (/ height (+ n-layers 1.0))))
    (dolist (l leafs)
      (setf (search-tree-x l) (floor (incf x h-gap)))
      (setf (search-tree-y l) t)	; mark node done
      (when (search-tree-parent l)
	(if (every #'(lambda (node) (eq (search-tree-y node) t))
		   (search-tree-children (search-tree-parent l)))
					; queue parent if all children done
	    (push (search-tree-parent l) nodes))))
    (do ((l (pop nodes) (pop nodes)))
	((null l))
      (setf (search-tree-x l) (floor
			       (/ (+ (search-tree-x
				      (first (search-tree-children l)))
				     (search-tree-x
				      (car (last
					    (search-tree-children l)))))
				  2.0)))
      (setf (search-tree-y l) t)	; mark node done
      (when (search-tree-parent l)
	(if (every #'(lambda (node) (eq (search-tree-y node) t))
		   (search-tree-children (search-tree-parent l)))
					; queue parent if all children done
	    (push (search-tree-parent l) nodes))))
    (do ((layer (list planning-tree) (mappend #'search-tree-children layer))
	 (y (- height v-gap) (- y v-gap))
	 (i 1 (+ i 1)))
	((or (null layer)
	     (> i n-layers)))
      (dolist (node layer)
	(setf (search-tree-y node) (floor y))))))

(defun leafs (planning-tree &optional (max-depth nil) (planning-depth 1))
  (if (and (or (not max-depth)
	       (<= planning-depth max-depth))
	   (search-tree-children planning-tree))
      (mapcan #'(lambda (node) (leafs node max-depth (1+ planning-depth)))
	      (search-tree-children planning-tree))
    (list planning-tree)))

(defun draw-tree-nodes (planning-tree max-layers &optional (planning-depth 1))
  (unless (> planning-depth max-layers)
    (let ((x (search-tree-x planning-tree))
	  (y (search-tree-y planning-tree)))
      (draw-rectangle *planner-window*
		      (- x 5) (- y 5) 10 10)
      (make-search-node-active-region planning-tree (- x 5) (- y 5) 10 10)
      (when (search-tree-n-traverse planning-tree)
	(write-display *planner-window*
		       (format nil "~d" (search-tree-n-traverse planning-tree))
		       (- x 4) (- y 3)))
      (when (search-tree-parent planning-tree)
	(draw-line *planner-window* x (+ y 6) x (+ y 7)))
      (when (search-tree-children planning-tree)
	(draw-line *planner-window* x (- y 6) x (- y 7))
	(unless (= planning-depth max-layers)
	  (dolist (node (search-tree-children planning-tree))
	    (draw-line *planner-window*
		       x (- y 7)
		       (search-tree-x node) (+ (search-tree-y node) 7))
	    (draw-tree-nodes node max-layers (1+ planning-depth))))))))

;;;; Active Regions

; have a queue of reusable active regions

(defun free-all-active-regions ()
  (do ((ar (pop *planner-active-regions*) (pop *planner-active-regions*)))
      ((null ar))
    (cw:deactivate (car ar))
    (push (car ar) *planner-unused-active-regions*)))

(defun make-search-node-active-region (node left bottom width height)
  (cond (*planner-unused-active-regions*
	 (let ((ar (pop *planner-unused-active-regions*)))
	   (setf (cw:active-region-left ar) left)
	   (setf (cw:active-region-bottom ar) bottom)
	   (setf (cw:active-region-width ar) width)
	   (setf (cw:active-region-height ar) height)
	   (push (cons ar node) *planner-active-regions*)
	   (cw:activate ar)))
	(t
	 (let ((ar (make-active-region *planner-window*
				       :left left
				       :bottom bottom
				       :width width
				       :height height)))
	   (push (cons ar node)
		 *planner-active-regions*)
	   (add-active-region-method ar :left-button-up
				     :after 'left-button-fn)
	   (add-active-region-method ar :middle-button-up
				     :after 'middle-button-fn)
	   (add-active-region-method ar :right-button-up
				     :after 'right-button-fn)
	   ))))

(defun scroll-down (ar)
  (declare (ignore ignore))
  (let ((node (cdr (assoc ar *planner-active-regions*))))
    (assert node (ar) "~%Planner active region queue ERROR!!!")
    (when (search-tree-n-traverse node)
      (draw-tree node))))

(defvar *planning-menu* (make-instance 'menu
			  :items '(("Scroll up"
				    scroll-up
				    "Scrolls plan tree up")
				   ("Scroll down"
				    scroll-down
				    "Scrolls plan tree down"))))

(defun left-button-fn (ar &rest ignore)
  (let ((node (cdr (assoc ar *planner-active-regions*))))
    (assert node (ar) "~%Planner active region queue ERROR!!!")
    (setf *outplan* (snlp-graph-node-node-content node))
    (documentation-print
      (format nil "The rank of the plan ~a~%~% is ~a"
	           (snlp-graph-node-node-content node)
		   (funcall (planner-default-rank-fn *which-planner*)
	           (snlp-graph-node-node-content node))))))


(defun right-button-fn (ar &rest ignore)
  (declare (ignore ignore))
  (let ((ans (accept-items *planning-menu*)))
    (if ans (apply ans (list ar)))))

(defun scroll-up (ar)
  (let ((node (cdr (assoc ar *planner-active-regions*))))
    (assert node (ar) "~%Planner active region queue ERROR!!!")
    (do ((i 1 (1+ i))
	 (current node (search-tree-parent current)))
	((or (null (search-tree-parent current))
	     (= i 10))
	 (draw-tree current )))))

(defvar *current-node* nil)		; for debugging purposes

(defun middle-button-fn (ar &rest ignore)
  (declare (ignore ignore))
  (let ((node (cdr (assoc ar *planner-active-regions*)))
	(top nil)
	(graph nil))
;    (setf rules::*backward-chainer* nil)
    (assert node (ar) "~%Planner active region queue ERROR!!!")
    (setf *current-node* node)
					;    (format t "~%~a" node)
    (setf graph (when (planner-graph-fn *current-planner*)
		  (funcall (planner-graph-fn *current-planner*)
			   (search-tree-node  node) *node-info-window* )))
    (setq *outplan* (search-tree-node  node))
    
    (cond
     ((or (and (equal *current-planner* strips::*strips-planner*)
           rules::*backward-chainer*)
	  (equal *current-planner* rules::*backchainer*))
      (setf top (loop for goal in (strips::plan-current-state
				   (search::search-tree-node
				    (loop for n = node then (search-tree-parent n)
					until (null (search-tree-parent n))
					finally (return n))))
		    collect goal)))
     ((equal *current-planner* strips::*strips-planner*)
      (progn (setf top (loop for goal in (strips::plan-current-state
					  (search::search-tree-node
					   (loop for n = node then (search-tree-parent n)
					       until (null (search-tree-parent n))
					       finally (return n))))
			   collect goal))
	     (animate-plan *world* top
			   (reverse (strips::plan-steps-taken-stack (search-tree-node node)))
			   *movie-window*)))
     (t (animate-plan *world*
		      (start-state *movie-window*)
		      (snlp::plan-steps (search-tree-node node))
		      *movie-window*))
     )



    graph
    
    ))

;;; for SNLP



(defun init-node-info-window ()
  (let ((window-width (min 600 (round (* (width *root-window*) .66))))
	(window-height (min 400 (- (round (* (height *root-window*) .60)) 20))))
    (unless (cw::common-windows-initialized-p)
    (init-window-system :host *default-display-host*))
  (unless (and *node-info-window*
	       (member (window *node-info-window*) cw:*all-window-streams*))
    (setf *node-info-window*
      (make-instance 'display 

		    :title "Node Info"
      		    :left 0
		    :width window-width
		    :height  window-height
		    :bottom (round (* (height *root-window*) .40))
		    :font *default-planner-font*
		    :borders 1)))
    ))

(defstruct (snlp-graph-node
	    (:print-function print-snlp-graph-node))
  node-content
  next-nodes
  prev-nodes
  (x nil)
  y
  y-space)

(defstruct (snlp-graph
	    (:print-function print-snlp-graph))
  start
  end
  nodes
  plan
  )

(defun print-snlp-graph (graph stream ignore)
  (declare (ignore ignore))
  (format stream "<graph for ~s>" (snlp-graph-plan graph)))

(defun print-snlp-graph-node (node stream ignore)
  (declare (ignore ignore))
  (format stream "<node for ~s>" (snlp-graph-node-node-content node)))

;(defun draw-snlp-plan (plan disp)
;  (init-snlp-plan-window)
;  (back-clear-display *node-info-window*)
;  (unless (snlp::snlp-plan-graphics plan)
;    (compute-snlp-graphics-data plan
;				(width *node-info-window*)
;				(height *node-info-window*)))
;  (draw-snlp-graph (snlp::snlp-plan-graphics plan)))

(defun draw-snlp-plan (plan disp)
;  (init-snlp-plan-window)
  (back-clear-display disp)
  (unless (snlp::snlp-plan-graphics plan)
    (compute-snlp-graphics-data plan
				(width disp)
				(height disp)))
  (draw-snlp-graph (snlp::snlp-plan-graphics plan)))

(defmacro snlp-trans (id)
  `(let ((trans-id ,id))
     (if (eq trans-id 'snlp::goal)
	 max-step
       trans-id)))

(defun add-order (n1 n2)
  (unless (find n2 (snlp-graph-node-next-nodes n1))
    (push n2 (snlp-graph-node-next-nodes n1)))
  (unless (find n1 (snlp-graph-node-next-nodes n2))
    (push n1 (snlp-graph-node-prev-nodes n2))))

(defmacro step-id-lookup (steps id step-node max-step)
  `(let ((trans-id ,id))
     
     
     (cdr (assoc (aref ,steps (if (eq trans-id 'snlp::goal)
				  ,max-step
				trans-id))
		 ,step-node))))

(defvar *h-gap* 50)



(defun compute-snlp-graphics-data (plan width height)
  (format t "...computing...") (setq *outplan* plan)
  (let* ((data (make-snlp-graph :plan plan))
	 (max-step (1+ (if (snlp::snlp-plan-high-step plan)
			   (snlp::snlp-plan-high-step plan)
			 0)))
	 (steps (make-array (1+ max-step)))
	 step-node			; step . node assoc list
	 goal-node			; goal . node assoc list
	 already-order
	 (gdata (list nil))

	 )
    (setf (snlp::snlp-plan-graphics plan) data)
    (dolist (step (snlp::snlp-plan-steps plan))
      (let ((node (make-snlp-graph-node :node-content step)))
	(push node (snlp-graph-nodes data))
	(push (cons step node) step-node))
      (cond ((eql (snlp::snlp-step-id step) :goal)
	     (setf (aref steps max-step) step))
	    (t (setf (aref steps (snlp::snlp-step-id step)) step))))
    (setf (snlp-graph-start data) (cdr (assoc (aref steps 0) step-node)))
    (setf (snlp-graph-end data) (cdr (assoc (aref steps max-step) step-node)))
    (dolist (link (snlp::snlp-plan-links plan))
      (let ((node (make-snlp-graph-node :node-content (second link))))
	(add-order node
		   (step-id-lookup steps (third link) step-node max-step))
	(add-order (step-id-lookup steps (first link) step-node max-step)
		   node)
	(push node (snlp-graph-nodes data))
	(push (cons (second link) node) goal-node)
	(push (list (first link) (third link)) already-order)))
    (dolist (open (snlp::snlp-plan-open plan))
      (let ((node (make-snlp-graph-node :node-content (first open)))
	    )
	(add-order node
		   (step-id-lookup steps (second open) step-node max-step))
	(push node (snlp-graph-nodes data))
	(push (cons (first open) node) goal-node)))
					; better placement alg can go here
    
    
    (setf gdata (list nil))
    (add-to-graph (snlp-graph-end data) 1 gdata goal-node)

    ;; For every second layer, jiggle it up or down.  This makes it less
    ;; likely that labels will overlap.  

    (loop for layer in (cdr gdata) as i from 0 do
	  (loop for node in layer as j from 0 do
		(setf (snlp-graph-node-x node) (- width (* (+ 2 i) *h-gap*))
		      (snlp-graph-node-y node)
		      (* (+  (* (if (oddp i) 1 -1) (/ (mod i 5) 40))  0.5 j) (/ height (length layer))))))



    (setf (snlp-graph-node-x (snlp-graph-start data)) *h-gap*
	  (snlp-graph-node-y (snlp-graph-start data)) (/ height 2))
    (setf (snlp-graph-node-x (snlp-graph-end data)) (- width *h-gap*)
	  (snlp-graph-node-y (snlp-graph-end data)) (/ height 2)
	  (snlp-graph-node-y-space (snlp-graph-end data)) height)

    #| (do* ((queue (list (snlp-graph-end data)) (cdr queue))
	  (node (car queue) (car queue)))
	((null queue))
      (cond ((snlp::snlp-step-p (snlp-graph-node-node-content node))
	     (let* ((step (snlp-graph-node-node-content node))
		    (x (- (snlp-graph-node-x node) *h-gap*))
		    (y-gap (max 20 (/ (snlp-graph-node-y-space node)
			      (length (snlp::snlp-step-precond step)))))
		    (y (- (+ (snlp-graph-node-y node)
			     (/ (snlp-graph-node-y-space node) 2))
			  (/ y-gap 2))))
	       (setf queue
		 (nconc queue
			(mapcar #'(lambda (precond)
				    (let ((goal (cdr (assoc precond goal-node
							    :test #'equal))))
				      (setf (snlp-graph-node-x goal) x
					    (snlp-graph-node-y goal) y
					    (snlp-graph-node-y-space goal)
					    y-gap)
				      (decf y y-gap)
				      goal))
				(snlp::snlp-step-precond step))))))
	    (t				; (car queue) is a goal node
	     (let* ((est (first (snlp-graph-node-prev-nodes node))))
	       (when (and est (not (snlp-graph-node-x est)))
		 (setf (snlp-graph-node-x est) (- (snlp-graph-node-x node)
						  *h-gap*)
		       (snlp-graph-node-y est) (snlp-graph-node-y node)
		       (snlp-graph-node-y-space est)
		       (snlp-graph-node-y-space node))
		 (setf queue (nconc queue (list est)))))))) |#
    (dolist (order (snlp::snlp-plan-ordering plan))
      (unless (find order already-order :test #'equal)
	(add-order (step-id-lookup steps (first order) step-node max-step)
		   (step-id-lookup steps (second order) step-node max-step))))
    )
  (format t "done"))

(defun add-to-graph (node level gdata goal-node)
  (progn (if (> level (length gdata))
	     (nconc gdata (list (list node)))
	   (setf (nth (- level 1) gdata)
	     (cons node (nth (- level 1) gdata)))
	   )
	 (if (snlp::snlp-step-p (snlp-graph-node-node-content node))
	     (loop for child in (snlp::snlp-step-precond (snlp-graph-node-node-content node))
		 do (add-to-graph (cdr (assoc child goal-node
					      :test #'equal)) (+ level 1) gdata goal-node))
	   (when (first (snlp-graph-node-prev-nodes node))
	     (add-to-graph (first (snlp-graph-node-prev-nodes node)) (+ level 1) gdata goal-node)))
	 gdata)
  )



(defun max-level (graph)
  (/ (- (width *node-info-window*) (loop for node in (snlp-graph-nodes graph)
				       when (and (snlp-graph-node-x node)
						 (not (and (snlp::snlp-step-p (snlp-graph-node-node-content node))
							   (not (snlp::snlp-step-action (snlp-graph-node-node-content node)))
							   (eql (snlp::snlp-step-id (snlp-graph-node-node-content node)) 0)
							   )))
				       minimize (snlp-graph-node-x node)))
     *h-gap*))





(defun draw-snlp-graph (graph)
  (format t "...drawing...")
  (setf (font *node-info-window*) *default-planner-font*)
  (setf snlp::*bindings* (snlp::snlp-plan-bindings (snlp-graph-plan graph)))
  (dolist (node (snlp-graph-nodes graph))
    (let ((x (snlp-graph-node-x node))
	  (y (snlp-graph-node-y node))
	  (node-content (snlp-graph-node-node-content node)))
      (if (null x) (progn (setf x 100) (setf y 100)))
      (cond ((snlp::snlp-step-p node-content)
	     (back-draw-filled-circle *node-info-window* x y 5)
	     (back-write-display
	      *node-info-window*
	      (cond ((snlp::snlp-step-action node-content)
		     (format nil "~a" (mapcar #'snlp::bind-variable
					      (snlp::snlp-step-action node-content))))
		    ((eql (snlp::snlp-step-id node-content) 0)
		     "start")
		    (t "end"))
	      (- x 20) (+ y 7)))
	    (t (when (snlp-graph-node-prev-nodes node) ; dot if goal achieved
		 (back-draw-line *node-info-window* x y x y))
	       (back-draw-circle *node-info-window* x y 5)
	       (back-write-display
		*node-info-window*
		(format nil "~a" (mapcar #'snlp::bind-variable node-content))
		(- x 20) (- y 12))))
      (dolist (next (snlp-graph-node-next-nodes node))
	(if (and (not (eql (snlp-graph-node-x next) (* 2 *h-gap*)))
		 (snlp::snlp-step-p node-content)
		 (not (snlp::snlp-step-action node-content))
		 (eql (snlp::snlp-step-id node-content) 0))
	    ; long lines back to the origin are hard to read, so I
	    ; will not draw them.
	    (let ((nx (snlp-graph-node-x next))
		  (ny (snlp-graph-node-y next)))
	      (if (null nx) (progn (setf nx 200) (setf ny 200)))
	      (back-draw-line *node-info-window* (- nx 5) ny (- nx 20) ny)
	      (back-draw-line *node-info-window* (- nx 20) ny (- nx 21) (+ 4 ny))
	      (back-draw-line *node-info-window* (- nx 21) (+ 4 ny)
			      (- nx 22) (- ny 3))
	      (back-draw-line *node-info-window* (- nx 22) (- ny 3)
			      (- nx 23) (+ ny 2))
	      (back-draw-line *node-info-window* (- nx 23) (+ ny 2)
			      (- nx 24) (- ny 1))
	      (back-draw-line *node-info-window* (- nx 24) (- ny 1)
			      (- nx 25) ny))
	    (back-draw-line *node-info-window*
			(+ x 5) y
			(- (snlp-graph-node-x next) 5)
			(snlp-graph-node-y next)))
	)))
  (show-back-buffer *node-info-window*)
  (format t "done"))
