;;; -*- Mode: LISP; Package: PAIL-LIB; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   unify.cl
;;; Short Desc: unification algorithm
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   29.5.91 SK/DTA
;;; Author:     Tanimoto
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------
;;; --------------------------------------------------------------------------
;;; Code from: The Elements of AI Using Common Lisp, STEVEN L. TANIMOTO, 1990
;;; p.230 ff.
;;; --------------------------------------------------------------------------

(in-package :pail-lib)

(export '(variablep unify has-variable))

;;; UNIFY
;;; with occurs check.

(defun variablep (v)
  (and (symbolp v)
       (char= (schar (symbol-name v) 0) #\?)))

(defun has-variable (v) 
  (cond ((variablep v) t)
	((null v) nil)
	((atom v) nil)
	((has-variable (car v)) t)
	((has-variable (cdr v)) t)))


;;; UNIFY is the top-level function.
;;; It finds a most general unifier for 2 literals.
(defun unify (literal1 literal2)
  (let ((u nil)) ; unifier is initially null.
    ; make sure predicate symbols match:
    (if (eq (car literal1) (car literal2))
      (catch 'unify
        (unify1 (cdr literal1) (cdr literal2) u) )
      'fail)))

;;; The recursive function UNIFY1 unifies the two lists
;;; of terms, termlist1 and TERMLIST2, and in the course
;;; of doing so adds more pairs onto the unifier U.
(defun unify1 (termlist1 termlist2 u)
  (cond
   ; If atomic and equal, no substitution necessary:
   ((eq termlist1 termlist2) u)
   ; Check for list length mismatch (a syntax error):
   ((or (null termlist1) (null termlist2))
    (throw 'unify 'fail))
   ; If EXPl is a variable, try to add a substitution:
   ((variablep termlist1) (add-pair termlist2 termlist1 u))
   ; Handle the case when EXP2 is a variable similarly:
   ((variablep termlist2) (add-pair termlist1 termlist2 u))
   ; Now, if either expression is atomic, it is a
   ; constant and there's no match since they're not EQ:
   ((or (atom termlist1) (atom termlist2)) 
    (throw 'unify 'fail))
   ; The expressions must be non-atomic; do recursively.
   ; Apply current substitutions before unifying the CARs.
   (t (setq u (unify1 (do-subst (car termlist1) u)
                      (do-subst (car termlist2) u)
                      u))
      ; Now unify the CDRs.
      (unify1 (cdr termlist1) (cdr termlist2) u) )))

;;; ADD-PAIR adds a (term variable) pair to the front of the
;;; substitution list, after substituting TERM for each
;;; occurrence of VAR in the terms of U. The new list
;;; of substitutions is returned.
(defun add-pair (term variable u)
  (cond ((occurs-in variable term)
         (throw 'unify 'fail) )
        (t (cons (list variable term)
                 (subst term variable u)))))


(defun do-subst (exp l)
  (cond ((null l) exp)
        (t (subst (caar l)
                  (cadar l)
                  (do-subst exp (cdr l))))))


;;; occurs check
(defun occurs-in (elt exp)
  (cond ((eq elt exp) t)
        ((atom exp) nil)
        (t (or (occurs-in elt (car exp))
               (occurs-in elt (cdr exp))))))


;;; *EOF*
