;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:GIN; Base:10; -*-
;;;
;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   gin.cl
;;; Short Desc: graphic interface to Common Windows.
;;; Version:    1.1
;;; Status:     Experimental
;;; Last Mod:   Jun 15 1991
;;; Author:     na

;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================

(in-package :gin)

(import '(cw:red	cw:green	cw:blue		    cw:black
	  cw:purple	cw:yellow	cw:turquoise	    cw:white	cw:50%-gray
	  cw:make-bitmap		cw:read-bitmap	    cw:make-bitmap-stream
	  cw:flush	cw:bitmap-bit
	  cw:font-string-width		cw:font-character-width	cw:font-character-height
	  cw:open-font-named		cw:open-font		cw:font-baseline
	  cw:documentation-print	cw:font-ascent		cw:make-color
	  cw:*root-window*			cw:*mouse-cursor-northwest-arrow*
	  cw:*mouse-cursor-left-top-corner*	cw:*mouse-cursor-timer*
	  cw:*grey-bitmap*	cw:*light-grey-bitmap*	cw:*dark-grey-bitmap*
	  pail-lib:add-path	pail-lib:add-subdir	pail-lib:*pail-path*))

;;; ==========================================================================
;;; This might be site-specific, please edit.
;;; ==========================================================================

(defparameter *default-icon-bitmap* (read-bitmap (add-path "pail-icon" (add-subdir *pail-path* "bitmaps"))))
(defparameter *pail-logo*	    (read-bitmap (add-path "pail-logo" (add-subdir *pail-path* "bitmaps"))))
(defparameter *idsia-logo*	    (read-bitmap (add-path "idsia-logo" (add-subdir *pail-path* "bitmaps"))))
(defparameter *up*		    (read-bitmap (add-path "up" (add-subdir *pail-path* "bitmaps"))))
(defparameter *down*		    (read-bitmap (add-path "down" (add-subdir *pail-path* "bitmaps"))))
(defparameter *help-file*	    (add-path "pail-main" (add-subdir *pail-path* "help")))

;;; ==========================================================================
;;; The rest isn't site-specific, please DO NOT edit.
;;; send bug reports or comments to almassy@ifi.unizh.ch
;;; Thank you.
;;; ==========================================================================

;;; This variable enables the programmer to conform with the previous
;;; syntax of the calls to make-display.  If the variable is set to
;;; NIL the call (make-display &keys ) should be used to open up a
;;; window. If the variable is set to T the call (make-instance
;;; 'display &keys ) should be used.
(defvar *use-new-version* nil)

(export '(50%-gray
	  *small-font-8*  *small-font-9*  *small-font-11*
	  *small-font*	  *default-font*  *default-read-font*
	  *normal-font*	  *bold-font*	  *italic-font*

	  *default-display-title*	*default-display-border*
	  *default-display-left*	*default-display-bottom*
	  *default-display-width*	*default-display-height*

	  *default-icon-bitmap*		*pail-logo* *idsia-logo*

	  *mouse-cursor-timer*		*mouse-cursor-northwest-arrow*	*mouse-cursor-left-top-corner*
	  *prompt-window*	*root-window*
	  activate-display
	  active
	  add-active-region-method
	  bitmap-bit
	  black
	  blue
	  borders
	  bottom
	  bury
	  clear-display
	  close-all-displays
	  close-display
	  copy-mask
	  deactivate-display
	  display
	  displayp
	  display-parent
	  display-position
	  status
	  display-wait-status
	  draw-arc
	  draw-circle
	  draw-filled-arc
	  draw-filled-circle
	  draw-filled-rectangle
	  draw-line
	  draw-rectangle
	  documentation-print
	  enable-scrollbar
	  expose
	  flush
	  font
	  font-ascent
	  font-baseline
	  font-character-height
	  font-character-width
	  font-string-width
	  force-window-output
	  get-display-position
	  get-position
	  green
	  height
	  icon
	  init-window-system
	  inner-height
	  inner-width
	  kill-window-system
	  left
	  make-active-region
	  make-bitmap
	  make-color
	  make-display
	  open-font
	  open-font-named
	  parent
	  read-bitmap
	  read-display
	  red
	  refresh-display
	  region
	  reshape-method
	  save-display
	  title
	  white
	  width
	  window
	  write-display
	  x-position
	  x-scrollbar
	  y-position
	  y-scrollbar
	  ))

(defparameter *normal-font*	(open-font :courier :roman 13 :weight :medium))
(defparameter *bold-font*	(open-font :courier :roman 13 :weight :bold))
(defparameter *italic-font*	(open-font :courier :italic 13 :weight :medium))
(defparameter *small-font-8*	(open-font :courier :roman 8 :weight :medium))
(defparameter *small-font-9*	(open-font :courier :roman 9 :weight :medium))
(defparameter *small-font-11*	(open-font :courier :roman 11 :weight :medium))
(defparameter *small-font*	(open-font-named "fixed"))
(defparameter *big-bold-font*	(open-font :courier :roman 22 :weight :bold))
(defparameter *big-normal-font*	(open-font :courier :roman 22 :weight :medium))

(defparameter *default-font*		*bold-font*)
(defparameter *default-read-font*	*normal-font*)

(defparameter *default-display-border*	1)
(defparameter *default-display-left*	10)
(defparameter *default-display-bottom*	10)
(defparameter *default-display-width*	400)
(defparameter *default-display-height*	300)
(defparameter *default-display-title*	"Portable AI Lab")

(defparameter *prompt-window*	nil)
(defvar *debug* nil)

(defclass display ()
	  ((window :initform nil
		   :initarg :window
		   :accessor window)
	   (parent :initform nil
		   :initarg :parent
		   :accessor parent)
	   (pattern :initform nil
		    :initarg :pattern
		    :accessor pattern)
	   (left :type integer
		 :initform 0
		 :initarg :left
		 :accessor left)
	   (bottom :type integer
		   :initform 0
		   :initarg :bottom
		   :accessor bottom)
	   (width :type integer
		  :initarg :width
		  :initform *default-display-width*
		  :accessor width)
	   (height :type integer
		   :initarg :height
		   :initform *default-display-height*
		   :accessor height)
	   (inner-width :type integer
			:initform 0
			:initarg :inner-width
			:accessor inner-width)
	   (inner-height :type integer
			 :initform 0
			 :initarg :inner-height
			 :accessor inner-height)
	   (x-position :type integer :initform 0
		       :initarg :x-position
		       :accessor x-position)
	   (y-position :type integer :initform 0
		       :initarg :y-position
		       :accessor y-position)
	   (active :initform t
		   :initarg :active
		   :accessor active)
	   (borders :initform *default-display-border*
		    :initarg :borders
		    :accessor borders)
	   (region :initform nil
		   :initarg :region
		   :accessor region)
	   (title :type string
		  :initform *default-display-title*
		  :initarg :title
		  :accessor title)
	   (frame-menu 
		  :initform t
		  :initarg :frame-menu
		  :accessor frame-menu)
	   (font :initarg :font
		 :initform *default-font*
		 :accessor font)
	   (icon :initarg :icon
		 :initform *default-icon-bitmap*
		 :accessor icon)
	   (x-scrollbar :initform nil
			:initarg :x-scrollbar
			:accessor x-scrollbar)
	   (y-scrollbar :initform nil
			:initarg :y-scrollbar
			:accessor y-scrollbar)
	   (reshape-method :initform nil
			   :initarg :reshape-method
			   :accessor reshape-method)
	   (flush-method :initform nil
			 :initarg :flush-method
			 :accessor flush-method))
  (:documentation "A Display that is refreshed automatically"))

(defmethod displayp ((d display)) t)
(defmethod displayp (something-else) (declare (ignore something-else)) nil)

(defmethod status ((d display))
  (equal (cw:window-stream-status (window d)) :active))

(defmethod status ((d cw:window-stream))
  (equal (cw:window-stream-status d) :active))

(defmethod status ((d cw:active-region))
  (equal (cw:active-region-status d) :active))

(defclass display-position ()
	  ((x-position :accessor x-position)
	   (y-position :accessor y-position)))


(defun init-window-system (&key (host (sys:getenv "DISPLAY"))
				(prompt-window t))
  (cw:initialize-common-windows :prompt-window nil :host host :force t)
  (if prompt-window
    (setf *prompt-window* 
     (cw:make-prompt-window 
       :left 10 :bottom 20
       :icon *default-icon-bitmap* :borders 1
       :title "PAIL Messages"))))

(defmacro kill-window-system ()
  `(cw:kill-common-windows))

;;; this is the NEW method that is still compatible with the old
;;; version.
;;; old keylist: (parent nil) font title active borders
(defmethod initialize-instance ;;; :after  OKKKKKKKKKKKIO
	   ((d display) &key left bottom
				       (width *default-display-width*)
				       (height *default-display-height*)
				       (borders *default-display-border*)
				       (icon *default-icon-bitmap*)
				       (title "A PAIL Display")
				       (active t)
				       (x-scrollbar nil) (y-scrollbar nil)
				       inner-width inner-height
				       (parent nil)
				       (frame-menu t)
				       (font *default-font*)
				       (display-type 'display)
				       reshape-method flush-method
				       button-region from-button
				       (filename *help-file*))
  (call-next-method)
  (if *debug* (format t "~%initialize-instance :after (~A display) ~A ~A \"~A\" ~A" d parent font title active))
  (if *use-new-version*
      (progn
	(if (not (cw:common-windows-initialized-p))
	    (init-window-system))
	(if (not (or left bottom))
	    (let* ((pos (if (not (equal parent nil))
			    (cw:make-position :x *default-display-left*
					      :y (+ height *default-display-bottom*))
			  (prog1
			      (progn (documentation-print "Press any mouse-button to place window")
				     (cw:get-position *root-window* :cursor *mouse-cursor-left-top-corner*))
			    (clear-display cw:*prompt-window*)))))
	      (setf left (cw:position-x pos))
	      (setf bottom (- (cw:position-y pos) height)))
	  (progn (if (not left) (setf left *default-display-left*))
		 (if (not bottom) (setf bottom *default-display-bottom*))))
	(if (not inner-width)
	    (setf (slot-value d 'inner-width) width)
	  (when (< inner-width width)
	    (warn "inner-width is less than width, set to ~A" width)
	    (setf (slot-value d 'inner-width) width)))
	(if (not inner-height)
	    (setf (slot-value d 'inner-height) height)
	  (when (< inner-height height)
	    (warn "inner-height is less than height, set to ~A" height)
	    (setf (slot-value d 'inner-height) height)))
	(if parent
	    (setf (window d) (cw:make-window-stream :left (left d) :bottom (bottom d)
						    :inner-width width :inner-height height
						    :font font
						    :title title
						    :parent (window parent)
						    :activate-p active
						    :border-width borders))
	  (setf (window d) (cw:make-window-stream :left left :bottom bottom
						  :inner-width width :inner-height height
						  :font font
						  :title title
						  :activate-p active
						  :border-width borders
						  :icon icon)))
	(setf (cw:window-stream-extent-width (window d)) (inner-width d))
	(setf (cw:window-stream-extent-height (window d)) (inner-height d))
	(if x-scrollbar
	    (if y-scrollbar
		(cw:enable-window-stream-extent-scrolling (window d)
							  :horizontal t
							  :vertical t)
	      (cw:enable-window-stream-extent-scrolling (window d)
							:horizontal t
							:vertical nil))
	  (if y-scrollbar
	      (cw:enable-window-stream-extent-scrolling (window d)
							:horizontal nil
							:vertical t)
	    (cw:enable-window-stream-extent-scrolling (window d)
						      :horizontal nil
						      :vertical nil)))
	(setf (pattern d) (make-bitmap-stream :font font
					      :foreground-color black
					      :background-color white
					      :width (inner-width d)
					      :height (inner-height d)))
	(cw:modify-window-stream-method (window d) :move-notify
					:after (function (lambda (&rest internal-data)
							   (declare (ignore internal-data ))
							   (if *debug* (format t "~%~A :move-notify to :left ~A :bottom ~A"
									       d
									       (cw:window-stream-left (window d))
									       (cw:window-stream-bottom (window d))))
							   (setf (slot-value d 'left) (cw:window-stream-left (window d)))
							   (setf (slot-value d 'bottom) (cw:window-stream-bottom (window d))))))
	(cw:modify-window-stream-method (window d) :move
					:after (function (lambda (&rest internal-data)
							   (declare (ignore internal-data))
							   (if *debug* (format t "~%~A :move" d))
							   (when (parent d)
							     (if (< (cw:window-stream-left (window d)) 0)
								 (setf (cw:window-stream-left (window d)) 0))
							     (if (< (cw:window-stream-bottom (window d)) 0)
								 (setf (cw:window-stream-bottom (window d)) 0))
							     (if (> (+ (cw:window-stream-bottom (window d)) (height d)
								       (cw:window-stream-title-height (window d))) (height (parent d)))
								 (setf (cw:window-stream-bottom (window d))
								   (max (- (height (parent d)) (cw:window-stream-title-height (window d))
									   (height d) 1) 0)))
							     (if (> (+ (cw:window-stream-left (window d)) (width d))
								    (width (parent d)))
								 (setf (cw:window-stream-left (window d))
								   (max (- (width (parent d)) (width d) 2) 0))))
							   (setf (slot-value d 'left) (cw:window-stream-left (window d)))
							   (setf (slot-value d 'bottom) (cw:window-stream-bottom (window d))))))
	(cw:modify-window-stream-method (window d) :repaint :before 'clear-if-area
					:after (function (lambda (&rest internal-data)
							   (declare (ignore internal-data))
							   (if *debug* (format t "~%~A :repaint" d))
							   (if (pattern d)
							       (cw:bitblt (pattern d) 0 0 (window d) 0 0)))))
	(cw:modify-window-stream-method (window d) :reshape-notify :after
					(function (lambda (&rest internal-data)
						    (declare (ignore internal-data))
						    (if *debug* (format t "~%~A :reshape-notify" d))
						    (setf (slot-value d 'width) (cw:window-stream-inner-width (window d)))
						    (setf (slot-value d 'height) (cw:window-stream-inner-height (window d)))
						    (setf (cw:window-stream-extent-width (window d)) (inner-width d))
						    (setf (cw:window-stream-extent-height (window d)) (inner-height d))
						    (if (or (> (cw:window-stream-inner-width (window d)) (inner-width d))
							    (> (cw:window-stream-inner-height (window d)) (inner-height d)))
							(let ((tmp-p (make-bitmap-stream :font (font d)
											 :width (max (cw:window-stream-inner-width (window d)) (inner-width d))
											 :height (max (cw:window-stream-inner-height (window d)) (inner-height d)))))
							  (cw:bitblt (pattern d) 0 0 tmp-p 0 0)
							  (setf (pattern d) tmp-p)
							  (if (> (cw:window-stream-inner-width (window d)) (inner-width d))
							      (setf (slot-value d 'inner-width) (cw:window-stream-inner-width (window d))))
							  (if (> (cw:window-stream-inner-height (window d)) (inner-height d))
							      (setf (slot-value d 'inner-height) (cw:window-stream-inner-height (window d))))))
						    (if (reshape-method d)
							(apply (reshape-method d) nil)))))
	(cw:modify-window-stream-method (window d) :flush :after
					(function (lambda (&rest internal-data)
						    (declare (ignore internal-data))
						    (if *debug* (format t "~%~A :flush" d))
						    (if (flush-method d)
							(apply (flush-method d) nil)))))
	)
    (progn
      (if (not (slot-boundp d 'inner-width))
	  (setf (slot-value d 'inner-width) (width d)))
      (if (not (slot-boundp d 'inner-height))
	  (setf (slot-value d 'inner-height) (height d)))
      (if parent
	  (setf (window d) (cw:make-window-stream :left (left d) :bottom (bottom d)
						  :inner-width (width d) :inner-height (height d)
						  :font font
						  :title title
						  :parent (window parent)
						  :activate-p active
						  :border-width borders))
	(setf (window d) (cw:make-window-stream :left (left d) :bottom (bottom d)
						:inner-width (width d) :inner-height (height d)
						:font font
						:title title
						:activate-p active
						:border-width borders
						:icon (icon d))))
      (unless (frame-menu d)
	(cw:modify-window-stream-method (window d) :frame-right-button-down :remove 'cw::window-frame-right-down))
      (setf (x-position d) (cw:window-stream-x-position (window d)))
      (setf (y-position d) (cw:window-stream-y-position (window d))))
    ))

(defun make-display (&key
		     left bottom
		     (width *default-display-width*)
		     (height *default-display-height*)
		     (borders *default-display-border*)
		     (icon *default-icon-bitmap*)
		     (title "A PAIL Display")
		     (active t)
		     (x-scrollbar nil) (y-scrollbar nil)
		     inner-width inner-height
		     (parent nil)
		     (frame-menu t)
		     (font *default-font*)
		     (display-type 'display)
		     reshape-method flush-method
		     button-region from-button
		     (filename *help-file*))
  (if (not (cw:common-windows-initialized-p))
      (init-window-system))
  (if (not (or left bottom))
      (let* ((pos (if (not (equal parent nil))
		     (cw:make-position :x *default-display-left*
				       :y (+ height *default-display-bottom*))
		   (prog1
		       (progn (documentation-print "Press any mouse-button to place window")
			      (cw:get-position *root-window* :cursor *mouse-cursor-left-top-corner*))
		     (clear-display cw:*prompt-window*)))))
	(setf left (cw:position-x pos))
	(setf bottom (- (cw:position-y pos) height)))
    (progn (if (not left) (setf left *default-display-left*))
	   (if (not bottom) (setf bottom *default-display-bottom*))))
  (if (not inner-width)
      (setf inner-width width)
    (when (< inner-width width)
      (warn "inner-width is less than width, set to ~A" width)
      (setf inner-width width)))
  (if (not inner-height)
      (setf inner-height height)
    (when (< inner-height height)
      (warn "inner-height is less than height, set to ~A" height)
      (setf inner-height height)))
  (let ((tmp-display (make-instance display-type
				:width width :height height
				:left left :bottom bottom
				:active active
				:borders borders :title title :icon icon
				:inner-width inner-width
				:inner-height inner-height
				:parent parent
				:frame-menu frame-menu
				:font font
				:reshape-method reshape-method
				:flush-method flush-method
				:button-region button-region
				:from-button from-button
				:filename filename)))
    (setf (cw:window-stream-extent-width (window tmp-display)) (inner-width tmp-display))
    (setf (cw:window-stream-extent-height (window tmp-display)) (inner-height tmp-display))
    (if x-scrollbar
	(if y-scrollbar
	    (cw:enable-window-stream-extent-scrolling (window tmp-display)
						      :horizontal t
						      :vertical t)
	  (cw:enable-window-stream-extent-scrolling (window tmp-display)
						    :horizontal t
						    :vertical nil))
      (if y-scrollbar
	  (cw:enable-window-stream-extent-scrolling (window tmp-display)
						    :horizontal nil
						    :vertical t)
	(cw:enable-window-stream-extent-scrolling (window tmp-display)
						  :horizontal nil
						  :vertical nil)))
    (setf (pattern tmp-display) (make-bitmap-stream :font (font tmp-display)
						    :foreground-color black
						    :background-color white
						    :width inner-width
						    :height inner-height))
    (cw:modify-window-stream-method (window tmp-display) :move-notify
				    :after (function (lambda (&rest internal-data)
						       (declare (ignore internal-data ))
						       (if *debug* (format t "~%~A :move-notify to :left ~A :bottom ~A"
									   tmp-display
									   (cw:window-stream-left (window tmp-display))
									   (cw:window-stream-bottom (window tmp-display))))
						       (setf (slot-value tmp-display 'left) (cw:window-stream-left (window tmp-display)))
						       (setf (slot-value tmp-display 'bottom) (cw:window-stream-bottom (window tmp-display))))))
    (cw:modify-window-stream-method (window tmp-display) :move
				    :after (function (lambda (&rest internal-data)
						       (declare (ignore internal-data))
						       (if *debug* (format t "~%~A :move" tmp-display))
						       (when (parent tmp-display)
							 (if (< (cw:window-stream-left (window tmp-display)) 0)
							     (setf (cw:window-stream-left (window tmp-display)) 0))
							 (if (< (cw:window-stream-bottom (window tmp-display)) 0)
							     (setf (cw:window-stream-bottom (window tmp-display)) 0))
							 (if (> (+ (cw:window-stream-bottom (window tmp-display)) (height tmp-display)
								   (cw:window-stream-title-height (window tmp-display))) (height (parent tmp-display)))
							     (setf (cw:window-stream-bottom (window tmp-display))
							       (max (- (height (parent tmp-display)) (cw:window-stream-title-height (window tmp-display))
								       (height tmp-display) 1) 0)))
							 (if (> (+ (cw:window-stream-left (window tmp-display)) (width tmp-display))
								(width (parent tmp-display)))
							     (setf (cw:window-stream-left (window tmp-display))
							       (max (- (width (parent tmp-display)) (width tmp-display) 2) 0))))
						       (setf (slot-value tmp-display 'left) (cw:window-stream-left (window tmp-display)))
						       (setf (slot-value tmp-display 'bottom) (cw:window-stream-bottom (window tmp-display))))))
    (cw:modify-window-stream-method (window tmp-display) :repaint :before 'clear-if-area
				    :after (function (lambda (&rest internal-data)
						       (declare (ignore internal-data))
						       (if *debug* (format t "~%~A :repaint" tmp-display))
						       (if (pattern tmp-display)
							   (cw:bitblt (pattern tmp-display) 0 0 (window tmp-display) 0 0)))))
    #| (cw:modify-window-stream-method (window tmp-display) :reshape :after
				    (function (lambda (&rest internal-data)
						(declare (ignore internal-data))
						(when (parent tmp-display)
						  (if *debug* (format t "~%~A :reshape" tmp-display))
						  (setf (slot-value tmp-display 'width) (cw:window-stream-inner-width (window tmp-display)))
						  (setf (slot-value tmp-display 'height) (cw:window-stream-inner-height (window tmp-display)))
						  (setf (cw:window-stream-extent-width (window tmp-display)) (inner-width tmp-display))
						  (setf (cw:window-stream-extent-height (window tmp-display)) (inner-height tmp-display))
						  (if (or (> (cw:window-stream-width (window tmp-display)) (inner-width tmp-display))
							  (> (cw:window-stream-height (window tmp-display)) (inner-height tmp-display)))
						      (let ((tmp-p (make-bitmap-stream :font (font tmp-display)
										       :width (cw:window-stream-width (window tmp-display))
										       :height (cw:window-stream-height (window tmp-display)))))
							(cw:bitblt (pattern tmp-display) 0 0 tmp-p 0 0)
							(setf (pattern tmp-display) tmp-p)
							(if (> (cw:window-stream-width (window tmp-display)) (inner-width tmp-display))
							    (setf (slot-value tmp-display 'inner-width) (cw:window-stream-inner-width (window tmp-display))))
							(if (> (cw:window-stream-height (window tmp-display)) (inner-height tmp-display))
							    (setf (slot-value tmp-display 'inner-height) (cw:window-stream-inner-height (window tmp-display))))))
						  (apply (reshape-method tmp-display) nil))))) |#
    (cw:modify-window-stream-method (window tmp-display) :reshape-notify :after
				    (function (lambda (&rest internal-data)
						(declare (ignore internal-data))
						(if *debug* (format t "~%~A :reshape-notify" tmp-display))
						(setf (slot-value tmp-display 'width) (cw:window-stream-inner-width (window tmp-display)))
						(setf (slot-value tmp-display 'height) (cw:window-stream-inner-height (window tmp-display)))
						(setf (cw:window-stream-extent-width (window tmp-display)) (inner-width tmp-display))
						(setf (cw:window-stream-extent-height (window tmp-display)) (inner-height tmp-display))
						(if (or (> (cw:window-stream-inner-width (window tmp-display)) (inner-width tmp-display))
							(> (cw:window-stream-inner-height (window tmp-display)) (inner-height tmp-display)))
						    (let ((tmp-p (make-bitmap-stream :font (font tmp-display)
										     :width (max (cw:window-stream-inner-width (window tmp-display)) (inner-width tmp-display))
										     :height (max (cw:window-stream-inner-height (window tmp-display)) (inner-height tmp-display)))))
						      (cw:bitblt (pattern tmp-display) 0 0 tmp-p 0 0)
						      (setf (pattern tmp-display) tmp-p)
						      (if (> (cw:window-stream-inner-width (window tmp-display)) (inner-width tmp-display))
							  (setf (slot-value tmp-display 'inner-width) (cw:window-stream-inner-width (window tmp-display))))
						      (if (> (cw:window-stream-inner-height (window tmp-display)) (inner-height tmp-display))
							  (setf (slot-value tmp-display 'inner-height) (cw:window-stream-inner-height (window tmp-display))))))
						(if (reshape-method tmp-display)
						    (apply (reshape-method tmp-display) nil)))))
    (cw:modify-window-stream-method (window tmp-display) :flush :after
				    (function (lambda (&rest internal-data)
						(declare (ignore internal-data))
						(if *debug* (format t "~%~A :flush" tmp-display))
						(if (flush-method tmp-display)
						    (apply (flush-method tmp-display) nil)))))
    #| (cw:modify-window-stream-method (window tmp-display) :flush-notify :after
				    (function (lambda (&rest internal-data)
						(declare (ignore internal-data))
						(if *debug* (format t "~%~A :flush-notify" tmp-display))
						(if (flush-method tmp-display)
						    (apply (flush-method tmp-display) nil))))) |#
    #| (when (equal display-type 'scroll-display)
      (set-button (exit-button tmp-display) tmp-display
	      :bottom 5 :left (- (width tmp-display) (width (exit-button tmp-display)) 5)
	      :action `(lambda nil
			 (close-display ,tmp-display)
			 (if ,(from-button tmp-display)
			     (reset-button ,(from-button tmp-display)))))
      (draw-line tmp-display 0 31 (width tmp-display) 31)) |#
    (when (and (or (eql (class-name (class-of tmp-display)) 'scroll-display)
		   (eql (class-name (class-of tmp-display)) 'help-display)) button-region)
      (set-button (exit-button tmp-display) tmp-display
		  :left (- (width tmp-display) (width (exit-button tmp-display)) 5)
		  :bottom 5
		  :action `(lambda nil
			     (close-display ,tmp-display)
			     (if (from-button ,tmp-display)
				 (reset-button (from-button ,tmp-display)))))
      )
    (mp:process-wait-with-timeout "waiting for display to complete" user::*server-delay* #'(lambda nil nil))
    tmp-display))

(defun get-display-position (width height)
  (let ((frame (make-bitmap :width width :height height)))
    (dotimes (i (1- width))
      (setf (cw:bitmap-bit frame i 0) 1)
      (setf (cw:bitmap-bit frame i (1- height)) 1))
    (dotimes (i (1- height))
      (setf (cw:bitmap-bit frame 0 i) 1)
      (setf (cw:bitmap-bit frame (1- width) i) 1))
    (cw::get-position *root-window* :cursor (cw:make-mouse-cursor :source-bitmap frame))))

(defmethod enable-scrollbar ((d display) &key
			     (vertical nil)
			     (horizontal nil))
  (cw:enable-window-stream-extent-scrolling (window d)
					    :horizontal horizontal
					    :vertical vertical))

(defmethod (setf x-scrollbar) :after (arg (d display))
  (if arg
      (cw:enable-window-stream-extent-scrolling (window d)
						:vertical (y-scrollbar d)
						:horizontal t)
    (cw:disable-window-stream-extent-scrolling (window d)
						:vertical (y-scrollbar d)
						:horizontal t)))

(defmethod (setf y-scrollbar) :after (arg (d display))
  (if arg
      (cw:enable-window-stream-extent-scrolling (window d)
						:horizontal (x-scrollbar d)
						:vertical t)
    (cw:disable-window-stream-extent-scrolling (window d)
					       :horizontal (x-scrollbar d)
					       :vertical t)))

(defmethod (setf parent) :after (arg (d display))
  (setf (cw:window-stream-parent (window d)) (window arg)))

(defmethod (setf font) :after (arg (d display))
  (setf (cw:window-stream-font (window d)) arg)
  (setf (cw:window-stream-font (pattern d)) arg))

(defmethod (setf left) :after (arg (d display))
  (setf (cw:window-stream-left (window d)) arg))

(defmethod (setf bottom) :after (arg (d display))
  (setf (cw:window-stream-bottom (window d)) arg))

(defmethod (setf width) :after (arg (d display))
  (if *debug* (format t "~%~A (setf width)" d))
  (setf (cw:window-stream-width (window d)) (+ arg (* 2 (borders d))))
  #| (setf (cw:window-stream-height (window d)) (+ (height d) (borders d)
						(cw:window-stream-title-height (window d)))) |#
  (if (> arg (inner-width d))
      (setf (inner-width d) arg)))

(defmethod (setf height) :after (arg (d display))
  (if *debug* (format t "~%~A (setf height)" d))
  (setf (cw:window-stream-height (window d)) (+ arg (borders d)
						(cw:window-stream-title-height (window d))))
  #| (setf (cw:window-stream-width (window d)) (+ (width d) (* 2 (borders d)))) |#
  (if (> arg (inner-height d))
      (setf (inner-height d) arg)))

(defmethod (setf inner-width) :before (arg (d display))
  (setf (cw:window-stream-extent-width (window d)) arg)
  (if (and (pattern d) (> arg (inner-width d)))
      (let ((tmp-p (make-bitmap-stream :font (font d)
				       :width arg
				       :height (inner-height d))))
	(cw:bitblt (pattern d) 0 0 tmp-p 0 0)
	(setf (pattern d) tmp-p))))

  #| (cw:enable-window-stream-extent-scrolling (window d)
						  :horizontal (> (inner-width d) (width d))
						  :vertical t) |#

(defmethod (setf inner-height) :before (arg (d display))
  (setf (cw:window-stream-extent-height (window d)) arg)
  (if (and (pattern d) (> arg (inner-height d)))
      (let ((tmp-p (make-bitmap-stream :font (font d)
				       :width (inner-width d)
				       :height arg)))
	(cw:bitblt (pattern d) 0 0 tmp-p 0 0)
	(setf (pattern d) tmp-p))))

(defmethod (setf borders) :after (arg (d display))
  (setf (cw:window-stream-borders (window d)) arg))

(defmethod (setf title) :after (arg (d display))
  (cw:write-title (window d) arg))

(defmethod (setf icon) :after (arg (d display))
  (cw:make-icon (window d) arg)
  (setf (cw:window-stream-left (window d)) (left d))
  (setf (cw:window-stream-bottom (window d)) (bottom d)))

(defmethod activate-display ((d display))
  (when (member (window d) cw:*all-window-streams*)
      (cw:activate (window d))
      (setf (active d) t)))

(defmethod deactivate-display ((d display))
  (when (member (window d) cw:*all-window-streams*)
      (cw::deactivate (window d))
      (setf (active d) nil)
      t))

(defmethod close-display ((d display))
  (cw:flush (window d))
  (setf (active d) nil))

(defmethod close-display (l)
  (dolist (i l) (close-display i)))

(defmethod save-display ((d display) filename &key
			 (left 0)
			 (bottom 0)
			 (width (width d))
			 (height (height d)))
  (let ((b (make-bitmap :width width :height height)))
    (cw:bitblt (window d) left bottom b 0 0 width height)
    (cw:save-bitmap b filename :if-exists :supersede :package (find-package 'user) :format :x11)))


(defmethod clear-display ((d display) &key
			  (left 0)
			  (bottom 0)
			  width
			  height)
  (if (status d)
    (progn
      (unless width (setf width (inner-width d)))
      (unless height (setf height (inner-height d)))
      (setf (cw:window-stream-x-position (pattern d)) 0)
      (setf (cw:window-stream-y-position (pattern d)) (- (height d) (font-character-height (font d))))
      (setf (cw:window-stream-x-position (window d)) 0)
      (setf (cw:window-stream-y-position (window d)) (- (height d) (font-character-height (font d))))
      (cw:clear-area (pattern d) (cw:make-region :left left :bottom bottom
						 :width width :height height))
      (cw:clear-area (window d) (cw:make-region :left left :bottom bottom
						:width width :height height)))
    (warn (format nil "~A has been closed" d))))

(defmethod clear-display ((d cw:window-stream) &key
			  (left 0)
			  (bottom 0)
			  width
			  height)
  (if (status d)
      (progn
	(unless width (setf width (cw:window-stream-width d)))
	(unless height (setf height (cw:window-stream-height d)))
      	(cw:clear-area d (cw:make-region :left left :bottom bottom
				       :width width :height height)))
    (warn (format nil "~A has been closed" d))))

(defmethod clear-display (something-else-than-a-window &key
			  left bottom width height)
  (declare (ignore something-else-than-a-window left bottom width height)))


(defun clear-if-area (wstream region)
  (if region (cw:clear-area wstream region)))

(defparameter arrow-head-angle  (* 20 (/ pi 180)))
(defparameter arrow-head-length 10)

(defmethod draw-line ((d display) x1 y1 x2 y2 &key
		      (color black)
		      (operation cw::boole-1)
		      (arrow nil))
  (cw:draw-line (pattern d) (cw:make-position :x x1 :y y1)
		(cw:make-position :x x2 :y y2) :color color :operation operation)
  (cw:draw-line (window d) (cw:make-position :x x1 :y y1)
		(cw:make-position :x x2 :y y2) :color color :operation operation)
  (if arrow
      (let ((delta1 (+ (if (equal x1 x2) (if (> y2 y1) (/ pi 2) (- (/ pi 2))) (atan (/ (- y2 y1) (- x2 x1)))) arrow-head-angle))
	    (delta2 (- (if (equal x1 x2) (if (> y2 y1) (/ pi 2) (- (/ pi 2))) (atan (/ (- y2 y1) (- x2 x1)))) arrow-head-angle)))
	(draw-line d x2 y2
		   (floor (if (> x1 x2) (+ x2 (* arrow-head-length (cos delta1))) (- x2 (* arrow-head-length (cos delta1)))))
		   (floor (if (> x1 x2) (+ y2 (* arrow-head-length (sin delta1))) (- y2 (* arrow-head-length (sin delta1)))))
		   :color color :operation operation)
	(draw-line d
		   x2 y2
		   (floor (if (> x1 x2) (+ x2 (* arrow-head-length (cos delta2))) (- x2 (* arrow-head-length (cos delta2)))))
		   (floor (if (> x1 x2) (+ y2 (* arrow-head-length (sin delta2))) (- y2 (* arrow-head-length (sin delta2)))))
		   :color color :operation operation)
	(draw-line d
		   (floor (if (> x1 x2) (+ x2 (* arrow-head-length (cos delta1))) (- x2 (* arrow-head-length (cos delta1)))))
		   (floor (if (> x1 x2) (+ y2 (* arrow-head-length (sin delta1))) (- y2 (* arrow-head-length (sin delta1)))))
		   (floor (if (> x1 x2) (+ x2 (* arrow-head-length (cos delta2))) (- x2 (* arrow-head-length (cos delta2)))))
		   (floor (if (> x1 x2) (+ y2 (* arrow-head-length (sin delta2))) (- y2 (* arrow-head-length (sin delta2)))))
		   :color color :operation operation))))

(defmethod draw-arc ((d display) x y width height &key
		     (color black)
		     (operation cw::boole-1)
		     brush-width
		     dashing
		     (start-angle 0.0)
		     (end-angle (* 2.0 lisp:pi)))
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-arc (window d) (cw:make-position :x x :y y)
	       width height
	       :color color :operation operation
;	       :brush-width brush-width :dashing dashing
	       :start-angle start-angle :end-angle end-angle)
  (cw:draw-arc (pattern d) (cw:make-position :x x :y y)
	       width height
	       :color color :operation operation
;	       :brush-width brush-width :dashing dashing
	       :start-angle start-angle :end-angle end-angle))

(defmethod draw-filled-arc ((d display) x y width height &key
		     (color black)
		     (operation cw::boole-1)
		     brush-width
		     dashing
		     (start-angle 0.0)
		     (end-angle (* 2.0 lisp:pi)))
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-filled-arc (window d) (cw:make-position :x x :y y)
	       width height
	       :color color :operation operation
;	       :brush-width brush-width :dashing dashing
	       :start-angle start-angle :end-angle end-angle)
  (cw:draw-filled-arc (pattern d) (cw:make-position :x x :y y)
	       width height
	       :color color :operation operation
;	       :brush-width brush-width :dashing dashing
	       :start-angle start-angle :end-angle end-angle))


(defmethod draw-circle ((d display) x y r &key
			(color black)
			(operation cw::boole-1)
			brush-width
			dashing)
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-circle (pattern d) (cw:make-position :x x :y y) r :color color :operation operation
;		  :brush-width brush-width :dashing dashing
		  )
  (cw:draw-circle (window d) (cw:make-position :x x :y y) r :color color :operation operation
;		  :brush-width brush-width :dashing dashing
		  ))

(defmethod draw-filled-circle ((d display) x y r &key
			       (color black)
			       (operation cw::boole-1) brush-width dashing)
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-filled-circle (pattern d) (cw:make-position :x x :y y) r :color color :operation operation
;			 :brush-width brush-width :dashing dashing
			 )
  (cw:draw-filled-circle (window d) (cw:make-position :x x :y y) r :color color :operation operation
;			 :brush-width brush-width :dashing dashing
			 ))

(defmethod draw-rectangle ((d display) x1 y1 x2 y2 &key
			   (color black)
			   (operation cw::boole-1) brush-width dashing)
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-rectangle-xy (pattern d) x1 y1 x2 y2 :color color :operation operation
;			:brush-width brush-width :dashing dashing
			)
  (cw:draw-rectangle-xy (window d) x1 y1 x2 y2 :color color :operation operation
;			:brush-width brush-width :dashing dashing
			))

(defmethod draw-rectangle (bitmap x1 y1 x2 y2 &key
			   (color black)
			   (operation cw::boole-1) brush-width dashing)
  (cw:draw-rectangle-xy bitmap x1 y1 x2 y2 :color color :operation operation
			:brush-width brush-width :dashing dashing))

(defmethod draw-filled-rectangle ((d display) x1 y1 x2 y2 &key
				  (color black)
				  (operation cw::boole-1) brush-width dashing)
  (unless brush-width
    (setf brush-width (cw:window-stream-brush-width (window d))))
  (unless dashing
    (setf dashing (cw:window-stream-dashing (window d))))
  (cw:draw-filled-rectangle-xy (pattern d) x1 y1 x2 y2 :color color :operation operation
;			       :brush-width brush-width :dashing dashing
			       )
  (cw:draw-filled-rectangle-xy (window d) x1 y1 x2 y2 :color color :operation operation
;			       :brush-width brush-width :dashing dashing
			       ))

(defmethod make-active-region ((d display) &key
			       (left 0) (bottom 0)
			       (width (width d)) (height (height d))
			       (active t))
  (cw:make-active-region :left left :bottom bottom :width width :height height
			 :parent (window d) :activate-p active))

(defun add-active-region-method (region what &key (after nil) (before nil))
  (if before
      (cw:modify-active-region-method region what :before before)
    (cw:modify-active-region-method region what :after after)))

(defmacro force-window-output ()
  (cw:force-graphics-output))

(defmethod write-display ((d display) string &optional
			  (x (cw:window-stream-x-position (pattern d)))
			  (y (cw:window-stream-y-position (pattern d))) &key
			  (operation cw::boole-1)
			  font)
  (setf (cw:window-stream-x-position (pattern d)) x)
  (setf (cw:window-stream-y-position (pattern d)) y)
  (setf (cw:window-stream-x-position (window d)) x)
  (setf (cw:window-stream-y-position (window d)) y)
  (let ((wstream-op (cw:window-stream-operation (window d)))
	(wstream-fn (cw:window-stream-font (pattern d))))
    (setf (cw:window-stream-operation (window d)) operation)
    (setf (cw:window-stream-operation (pattern d)) operation)
    (if font (setf (cw:window-stream-font (pattern d)) font))
    (format (pattern d) "~A" string) (refresh-display d)
    (setf (x-position d) (cw:window-stream-x-position (pattern d)))
    (setf (y-position d) (cw:window-stream-y-position (pattern d)))
    #| (cw:bitblt (window d) x (- y (font-baseline (font d))) (pattern d)
	       x (- y (font-baseline (font d)))
	       (font-string-width (font d) string)
	       (font-character-height (font d))) |#
    (if font (setf (cw:window-stream-font (pattern d)) font))
    (setf (cw:window-stream-operation (window d)) wstream-op))
  string)

(defmethod copy-mask ((d1 display) sx sy
		      (d2 display) tx ty &optional
		      (sw (- (width d1) sx))
		      (sh (- (height d1) sy))
		      (operation cw::boole-1)
		      (clip-region (cw:make-region :left tx :bottom ty
						   :width (- (width d1) sx)
						   :height (- (height d1) sy)))
		      (replicate nil))
  (cw:bitblt (window d1) sx sy (window d2) tx ty sw sh operation clip-region replicate)
  (cw:bitblt (pattern d1) sx sy (pattern d2) tx ty sw sh operation clip-region replicate))

(defmethod copy-mask (d1 sx sy
		      (d2 display) tx ty &optional
		      (sw (- (width d1) sx))
		      (sh (- (height d1) sy))
		      (operation cw::boole-1)
		      (clip-region (cw:make-region :left tx :bottom ty
						   :width (- (width d1) sx)
						   :height (- (height d1) sy)))
		      (replicate nil))
  (cw:bitblt d1 sx sy (window d2) tx ty sw sh operation clip-region replicate)
  (cw:bitblt d1 sx sy (pattern d2) tx ty sw sh operation clip-region replicate))

(defparameter *first-read-char* #\newline)

(defmethod read-display ((d display) &optional
			 (left (x-position d))
			 (bottom (y-position d)) &key
			 (remove nil)
			 (flush-input t)
			 (close-display nil))
  (setf (cw:window-stream-x-position (window d)) left)
  (setf (cw:window-stream-y-position (window d)) bottom)
  (cw:select (window d))
  (let ((cf (font d))
	(region (make-active-region d)) 
	(result (cw:rubout-handler :stream (window d)
				   :do-not-echo '(#\newline)
				   :body 
				   
       (progn
	 (cw::sleep-cursor (window d))
	 (draw-cursor d (cw:window-stream-x-position (window d))
		      (cw:window-stream-y-position (window d)))
	 (if flush-input (clear-input (window d)))
	 (do ((thechar (progn (setf *first-read-char* (read-char (window d) nil ""))
			      (unless (char=  *first-read-char* #\newline)
				(draw-cursor d (cw:window-stream-x-position (window d))
					     (cw:window-stream-y-position (window d))))
			      *first-read-char*)
							  
		       (progn (setf (x-position d) (cw:window-stream-x-position (window d)))
			      (setf (y-position d) (cw:window-stream-y-position (window d)))
			      (draw-cursor d (- (cw:window-stream-x-position (window d))
						(font-character-width *default-read-font*))
					   (cw:window-stream-y-position (window d)) :color white)
			      (read-char-display d))))
	     ((char= thechar #\newline)))
	 (cw:get-rubout-handler-buffer (window d)))

	 )))
    (draw-cursor d (cw:window-stream-x-position (window d)) (cw:window-stream-y-position (window d))
		      :color white)
    (if remove
	(write-display d result left bottom :operation boole-clr)
      (write-display d result left bottom))
    (setf (font d) cf)
    (cw:flush region)
    (if close-display (close-display d))
    result))

(defmethod draw-cursor ((d display) left bottom &key (color black))
  (cw:draw-line-xy (window d) left bottom (- left 5) (- bottom 5) :color color)
  (cw:draw-line-xy (window d) left bottom (+ left 5) (- bottom 5) :color color)
  (cw:draw-line-xy (window d) (- left 5) (- bottom 5) (+ left 5) (- bottom 5) :color color))

(defmethod read-char-display ((d display))
  (let ((result (read-char (window d) nil "")))
    (if (char= result #\newline)
	(draw-cursor d (x-position d) (y-position d) :color white)
      (draw-cursor d (cw:window-stream-x-position (window d))
		   (cw:window-stream-y-position (window d))))
    result))

(defun close-all-displays ()
  (mapcar #'cw:flush cw:*all-window-streams*))

(defmethod get-position ((d display))
  (let ((p (make-instance 'display-position))
	(cwpos (cw:get-position (window d))))
    (setf (x-position p) (cw:position-x cwpos))
    (setf (y-position p) (cw:position-y cwpos))
    p))

(defmethod display-wait-status ((d display) wait)
  (when (status d)
    (if wait (setf (cw:window-stream-mouse-cursor (window d)) *mouse-cursor-timer*)
      (setf (cw:window-stream-mouse-cursor (window d)) *mouse-cursor-northwest-arrow*))))

(defmethod display-wait-status (l wait)
  (dolist (i l) (display-wait-status i wait)))

(defmethod protect-display ((d display) status)
  (if status
      (setf (region d) (make-active-region d))
    (progn (if (region d) (cw:flush (region d)))
	   (setf (region d) nil))))

;;; ==========================================================================

;;; This method should not be needed, but just in case...
(defmethod refresh-display ((d display)) (cw:repaint (window d)))

(defmethod print-object ((p display-position) stream)
  (format stream "#(display-position :left ~A :bottom ~A)" (x-position p) (y-position p)))

(defmethod window ((w cw::root-window)) w)

(defmethod width ((w cw::bitmap)) (cw:bitmap-width w))
(defmethod height ((w cw::bitmap)) (cw:bitmap-height w))
(defmethod width ((w cw::bitmap-stream)) (cw::bitmap-stream-width w))
(defmethod height ((w cw::bitmap-stream)) (cw::bitmap-stream-height w))


(defmethod height ((w cw::root-window)) (cw:window-stream-height cw:*root-window*))
(defmethod width ((w cw::root-window)) (cw:window-stream-width cw:*root-window*))

(defmethod flush ((d display)) (close-display d))

(defmethod print-object ((d display) stream)
  (format stream "#<~A '~A' ~Ax~A>" (class-name (class-of d)) (title d) (width d) (height d)))

(defmethod expose ((d display))
  (cw:expose (window d)))

(defmethod bury ((d display))
  (cw:bury (window d)))

(defmethod display-parent ((d display))
  (if (eq (cw::window-stream-parent (window d))
	  cw:*root-window*)
      cw:*root-window*
    (parent d)))

(defun rad (d)
  (/ (* lisp:pi d) 180))

(defun deg (d)
  (* (/ lisp:pi d) 180))

(defmethod cw::text-scroll-hook (a b c)
  (declare (ignore a b c))
  nil)

;;;(defun text-scroll-hook ((bms cw::bitmap-stream) x y)
;;;  (format t "~%(text-scroll-hook ~a ~a ~a)" bms x y))


;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
