;;; -*- Mode: LISP; Package: EBG; Syntax:Common-Lisp; Base:10; -*-
;;; ************************************************************************
;;; PORTABLE AI LAB - UNI ZH
;;; ************************************************************************
;;;
;;; Filename:   ebg
;;; Short Desc: An implementation of Mitchells EBG-Method
;;; Version:    1.0
;;; Status:     passed preliminary tests
;;; Author:     Dean Allemang
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;; ------------------------------------------------------------------------
;;; Change History:
;;; 29.01.91 SK Some pail standards applied
;;; 25.05.91 SK Runnable
;;; 29.01.92 SK More readable trace
;;; 8.9.92   DTA Rewrittten to support suicide and kidnapping examples
;;; ========================================================================


;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================


;;; The Explanation Based Generalization algorithm.  
;;;
;;; This version was coded from the following article:
;;;    Tom M. MITCHELL, Richard M.Keller & Smadar T.Kedar-Cabelli (1986),
;;;    Explanation-Based Generaliazation: A Unifying View, in Machine 
;;;    Learning, No.1, 47-105.


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :ebg)


(eval-when (compile load eval)
  (export '(
            *verbose-stream*
            ebg-m
            ))
  )

;;; ==========================================================================
;;; GLOBALS
;;; ==========================================================================

;;; Streams as described in section IMPLEMENTATION NOTES:
(defvar *verbose-stream*    nil               "Stream for [verbose] comments")

(defparameter *var-counter* 0        "For new variables used in NEW-VARIABLE")



;;; ==========================================================================
;;; MAIN FN's
;;; ==========================================================================


(defmethod ebg-m ((ebg-tree ebg-tree) rule-set)
  "The driver fn of the EBG algorithm. Returns a new ebg rule."
  (let* ((gen-ebg-tree (generalize ebg-tree rule-set))
         (rule (regress gen-ebg-tree)))
    rule))


(defmethod generalize ((ebg-tree ebg-tree) rule-set)
  "Dummy fn here. Returns the generalized proof tree"
  (declare (ignore rule-set))
  (format-display *verbose-stream* 
                  "~%Phase 1: Get the generalized proof tree.")
  ;(print-ebg-tree ebg-tree *verbose-stream*)
  ebg-tree)



(defun get-ebg-rule (rule)
  "Get all leaves of the tree. Returns the new ebg rule."
  (format-display *verbose-stream* "~%Phase 3: Get the new ebg rule from the regressed proof tree.")
  (sexpr-to-rule 'rule-ebg rule))


;;; --------------------------------------------------------------------------
;;; REGRESS
;;;
;;; The goal concept is unified with the top of the tree.  The
;;; bindings are stored there.  Then they are applied backwards
;;; (rename-term) to produce the regressed goal.  This is repeated
;;; down the tree (depth-first).
;;;             
;;; --------------------------------------------------------------------------
(defvar *goal-list* nil)
;;; REGRESS -> a list of the goal-concept and the regressed leaves
(defmethod regress ((ebg-tree ebg-tree))
  "The main EBG fn. Returns the regressed proof tree"
  (format-display *verbose-stream* "~%Phase 2: Regress through the generalized proof tree.~%")
  ;; Initialize some counter and the first goal-concept(s)
  (setq *var-counter* 0)
  (let ((goal-concept (new-term (then-part (content ebg-tree))))
	tree
	)
    ;; root is level 1
    (setf *goal-list* nil)
    (setf tree (depth-first goal-concept ebg-tree ))
    (simplify-rule (make-instance 'rule
		     :if-part (expand-all  (reverse *goal-list*) tree)
		     :then-part (expand-all goal-concept tree) 
		     ))))




;;; applies all subsitutions in the tree, top-down.

(defmethod expand-all (goal-concept (ebg-tree ebg-tree) )
  (let ((answer (apply-sub
		 goal-concept (substitutions ebg-tree))))
    
    (loop for descendant in (descendants ebg-tree) do
	  (setf answer
	    (expand-all answer descendant )))
    answer))




;;; DEPTH-FIRST -> given a goal concept, and a tree, this unifies the
;;; goal with the appropriate then-clause at the top of the tree, and
;;; writes this unification in that node of the tree.  For the
;;; descendants of the tree, all the bindings that have happened
;;; before in its siblings and their descendants are applied to the
;;; goal before the recursion is done. 

;;; Each tree node has a rule, with several preconditions, and a
;;; number of descendants.  By convention, the descendants correspond
;;; to the rule preconditions in order.

(defmethod depth-first (goal-concept (ebg-tree ebg-tree) )
  (if (null (fact ebg-tree))
      (let* ((newgoal goal-concept)
	     (u (unify newgoal (unique (then-part (content ebg-tree)) ebg-tree) nil))
	     (newbind (if (eq u mgu::fail) nil u))
	     )
	(setf (substitutions ebg-tree)
	  newbind)
;	(format-display *verbose-stream* "Binding ~a to ~a - ~a"
;			newgoal (then-part (content ebg-tree)) newbind)
	
	(loop for descendant in (descendants ebg-tree)
	    as ifs in (unique (if-part (content ebg-tree)) ebg-tree)
	    do (depth-first (let ((answer ifs))
			      (loop for sofar in (descendants ebg-tree)
				  until (eq sofar descendant)
				  do (setf answer (expand-all answer sofar)))
			      answer)
			    descendant))
	)
  (push goal-concept *goal-list*))
  ebg-tree)





;;; Constructs a unique term from the uid of the rule instance. 

(defmethod unique (term (ebg-tree ebg-tree))
  (cond ((is-int-var term) (intern (format nil "~a~a" term (uid ebg-tree)) :dump))
	((atom term) term)
	(t (loop for t1 in term collect (unique t1 ebg-tree))))) 

(defun de-unique (var)
  (intern (subseq (symbol-name var) 0 (search "EBG" (symbol-name var)) ) :dump))

(defun collision (var1 var2)
   (and (not (equal var1 var2)) (equal (de-unique var1) (de-unique var2))))

(defmethod simplify-rule ((rule rule))
  (setf (if-part rule) (reverse (remove-duplicates (reverse (if-part rule)) :test #'equal)))
  (let* ((vars (remove-duplicates (find-all-vars (cons (then-part rule) (if-part rule)))
				  :test  #'equal))
	 (alist (remove-duplicates (loop for v in vars collect
					 (cons (de-unique v) 0))
				   :test #'equal))
	 (bindings (loop for vart on vars
		       collect
			 (let ((crash (find-if #'(lambda (var) (collision (car vart)
									  var)) (cdr vart))
				      ))
			   (if (not crash)
			       (cons (car vart) (de-unique (car vart)))
			     (prog1
				 (cons (car vart)
				       (intern
					(format nil
						"~a~a" (de-unique (car vart))
						(cdr (assoc (de-unique (car
									vart)) alist)))
					:dump)
				       )
			       (rplacd (assoc (de-unique (car
							  vart)) alist)
				       (1+ (cdr (assoc (de-unique (car
							      vart)) alist))))
			       )
			     )))))
	 (setf (if-part rule) (apply-sub (if-part rule) bindings))
	 (setf (then-part rule) (apply-sub (then-part rule) bindings))
	 rule))









(defun find-all-vars (l)
  (cond ((is-int-var l) (list l))
	((atom l) nil)
	(t (loop for item in l append (find-all-vars item)))))


;(regress *gen-ebg-tree*)



;;; If two variables bind to the same variable, they should be bound
;;; together.  This can be done entirely locally.  The new binding is
;;; simply added to the list of bindings.








;;; --------------------------------------------------------------------------
;;; NEW-TERM
;;; --------------------------------------------------------------------------

;;; This re-news all variables in term
(defun new-term (term)
  (labels ((new (term)
		(cond ((is-int-var term)
                       (new-variable))
                      ((atom term) term)
                      (t (cons (new (car term))
                               (new (cdr term)))))))
    (new term)))


(defun new-variable ()
  (intern (concatenate 'string
                       (symbol-name '?var)
                       (princ-to-string (incf *var-counter*))) :dump))


;(new-term '(isa ?p1 ((?y ?v) ?z)))
;-> (ISA ?VAR1 ((?VAR2 ?VAR3) ?VAR4))



;;; ==========================================================================
;;; * END OF FILE *
;;; ==========================================================================

