;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PAIL-LIB; Base:10; -*-
;;; **************************************************************************
;;;                       EXPLANATION BASED LEARNING BROWSER
;;; **************************************************************************
;;;
;;; Filename:   ebg-browser.cl
;;; Short Desc: Browsers to display EBG trees
;;; Version:    0.1
;;; Status:     Experimental
;;; Last Mod:   29.5.91 15:00:01 DTA
;;; Author:     DTA
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;




;;;
;;; ------------------------------------------------------------------------
;;; Change History:
;;; DTA  29.5.91  Added to full pAIl
;;; DTA  11.5.91  Corrected the order of browsed subs
;;; ------------------------------------------------------------------------
;;;
;;; ==========================================================================

;;; DESCRIPTION
;;; ------------------------------------------------------------------------
;;; Specializes the general browser class for use by EBG.  This
;;; includes defining the action of the buttons in an EBG browser, and
;;; various modes of operation, as well as some help windows. 
;;; ------------------------------------------------------------------------



(in-package :pail-lib)



(eval-when (load compile eval)
  (export '(generate-explanation ebg-browser highlight-rule
	    reset-highlights reshow-item)))


;;; Path where all help files will be found
(defparameter *ebg-help-path* (add-subdir *pail-path* "ebg"))



;;; ==========================================================================
;;; CLASSES AND STRUCTURES - Definitions and Access Functions
;;; ==========================================================================



;;; --------------------------------------------------------------------------


(defclass ebg-browser (browser)
	  ((left-menu :initarg left-menu
		      :initform (make-instance 'menu
				  :items '(("Show bindings" show-bindings "Shows the bindings unified by the EBG algorithm.")
					   ("Show fact" show-fact "Displays the conclusion of the rule.")
;;; DEADLY OKKIO			   ("Inspect" inspect-ebg "Calls the inspector on the selected item.")
					   ("Show rule" show-rule "Displays the rule associated with this conclusion.")
					   ("Help" help "Tell about the EBG Browser.")
					   ("Generalize" ebg-run "Run a generalization algorithm from this node.")
					   ("Put pool" pool-enter "Puts this explanation into the pool.")
					   ("Reset buttons" reset-ebg-buttons "Resets all buttons in the browser.")
					   ("Toggle Brief/Verbose" toggle-verbose "Changes browser mode from verbose to brief or back.")
;;; DEADLY OKKIO			   ("Close" close-browser "Closes the browser display")
;;; DEADLY OKKIO			   ("Recompute" recompute-me "Recomputes the browser")
					   ))
		      :accessor left-menu
		      :type menu)
	   (title :initarg :title
		   :initform "EBG Browser "
		   :accessor title
		   :type string)
	   (verbose :initarg :verbose	; In brief mode, one node is shown for each rule in an explanation.
		    :initform t		; in verbose mode, an extra node is placed for each precondition
		    :accessor verbose	; of a rule.  This allows one to see how conclusions were bound to premises.
		    :documentation "Determines whether intermediate conclusions will be shown in the browser")
	   (help-window :initarg :help-window ; Only one help window per browser
			:initform nil
			:accessor help-window
			:type help-stream
			:documentation "attached help window for this browser")
	   (ruleset :initarg :ruleset ;; the rule-set from which this explanation comes
		   :initform nil
		   :accessor ruleset
		   :type rule-set)
	   (wm :initarg :wm ;; the working memory for which this is an explanation
		   :initform nil
		   :accessor wm
		   :type working-memory)))






(defmethod browser-select-item ((b ebg-browser) item mousestate button)
	     (let ((ans (accept-items (left-menu b))))
	       (if ans (apply ans (list item b))))
	     (reset-button button)
	     )


(defmethod help-window ((b ebg-browser))
  (if (not (slot-value b 'help-window))
      (setf (help-window b) (make-instance 'help-display
			     :width 520
			     :button-region t
			     :title "Explanation browser help"
			     :filename (add-path "ebg-brs-hlp0.asc" *ebg-help-path*)))
    (slot-value b 'help-window)))

(defmethod help ((item  ebg-tree) (b ebg-browser))
  (if (verbose b) (setf (filename (help-window b))  (add-path "ebg-brs-hlp0.asc" *ebg-help-path*))
	(setf (filename (help-window b))  (add-path "ebg-brs-hlp1.asc" *ebg-help-path*))))

(defmethod show-rule ((item ebg-tree) (b ebg-browser))
  (when (content item)
    (documentation-print (print-rule (content item) nil))
    ))
  
(defmethod ebg-run ((item ebg-tree) (b ebg-browser))
  (ebg::demo-ebg item (ruleset b) rules::*verbose-disp*))

(defmethod pool-enter ((item ebg-tree) (b ebg-browser))
  (put-pool *pail-pool* item))

(defmethod reset-ebg-buttons ((item ebg-tree) (b ebg-browser))
  (mapcar 'reset-button (buttons b)))

(defmethod show-bindings ((item ebg-tree) (b ebg-browser))
  (documentation-print (format nil "~a~%" (substitutions item))))

(defmethod show-fact ((item ebg-tree) (b ebg-browser))
  (format  *prompt-window* "~a~%" (label item)))


(defmethod toggle-verbose  ((item ebg-tree) (b ebg-browser))
  (setf (verbose b) (not (verbose b)))
  (recompute-browser b))



  

(defmethod find-descendants ((b ebg-browser) (item ebg-tree))
  (if (verbose b)
      
      (if (null (content item)) nil
	(do ((desc (slot-value item 'descendants) (cdr desc))
	     (hyps (if-part (content item)) (cdr hyps))
	     (all nil))
	    ((null hyps) (reverse all))
	  (setq all (cons (make-instance 'hypothesis
			    :content (car hyps)
			    :descendants (list (car desc)))
			  all))))


    (slot-value item 'descendants)))



(defmethod find-descendants ((b ebg-browser) item)
  (descendants item))

(defclass hypothesis (tree)
	  ((predicate :accessor predicate
		      :initarg :predicate
		      :initform nil
		      :type list)
	   ))


(defmethod label ((h hypothesis)) (all-symbol-names (content h)))

;------------

(defmethod help ((item  hypothesis) (b ebg-browser))
  (setf (filename (help-window b)) (add-path "ebg-brs-hlp2.asc" *ebg-help-path*)))

(defmethod show-rule ((item hypothesis) (b ebg-browser))
  (when (content item)
    (print (content item) *prompt-window*)
    (format t "~%")))

  
(defmethod reset-ebg-buttons ((item hypothesis) (b ebg-browser))
  (mapcar 'reset-button (buttons b)))

(defmethod show-bindings ((item hypothesis) (b ebg-browser))
  nil)

(defmethod show-fact ((item hypothesis) (b ebg-browser))
  (format  t "~a~%" (label item)))

(defmethod inspect-ebg ((item hypothesis) (b ebg-browser))
  (inspect item))

(defmethod toggle-verbose  ((item hypothesis) (b ebg-browser))
  (setf (verbose b) (not (verbose b)))
  (recompute-browser b))


(defmethod highlight-item ((b ebg-browser) (item ebg-tree))
  (loop for i in (items b)
      as button in (buttons b) do
	(when (eq i item)
	  (setf (gin::border button) 1)
	  (set-button button (display b)))))

(defmethod highlight-item ((b ebg-browser) (item hypothesis))
  (loop for i in (items b)
      as button in (buttons b) do
	(when (and (eq (class-name (class-of i)) 'hypothesis)
		   (eq (content i) (content item)))
	  (setf (gin::border button) 1)
	  (set-button button (display b)))))


(defmethod highlight-rule ((b ebg-browser) rule)
  (loop for item in (items b) do
	(when (and (eq (class-name (class-of (content item))) 'rule)
		   (eq (name-part (content item)) rule))
	  (highlight-item b item)
	  (loop for d in (find-descendants b item) do
		(highlight-item b d)))))


(defmethod reset-highlights ((b ebg-browser))
  (loop for button in (cdr (reverse (buttons b))) do
	(when (gin::border button)
	  (setf (gin::border button) nil)
	  (unset-button button)
	  (set-button button (display b)))))



;;; For doing a fancy demo of the EBG algorithm, we want to be able to
;;; modify the way that we display an item in the browser.  Given an
;;; item, and a new name for it, this takes care of dealing with the
;;; buttons.  

(defmethod reshow-item ((b ebg-browser) (item tree) string)
  (let ((button (loop for but in (buttons b) as i in (items b)
		    thereis (if (or
				 (eq item i)
				 (equal (find-descendants b item)
				     (find-descendants b i)))
				but)))
	)
    
    (setf (width button) (+ 8 (font-string-width (font b) string)))
    (setf (label button) string)))


;-----------------

(defmethod label ((etree ebg-tree))
  (if (null (content etree))
      (all-symbol-names (fact etree))
    (all-symbol-names (then-part (content etree)))))


(defun all-symbol-names (list) (string-right-trim '(#\Space) (Rall-symbol-names list)))


(defun Rall-symbol-names (list)
  (concatenate 'string
    (cond ((null list) nil)
	((numberp list) (write-to-string list))
	((stringp list) list)
	((atom list) (symbol-name list))
	(t (let ((result nil))
	     (eval `(concatenate 'string
		     "("
		     ,@ (dolist (c list
				  (reverse (cons
					    (string-right-trim '(#\Space) (car result))
					    (cdr result))))
		       (push (Rall-symbol-names c) result))
		     ")"))
	     )))
    " "))


(defvar *gealist* (list (cons t t)))


(defmethod generate-explanation (fact   (wm working-memory))
  (setf *gealist* (list (cons t t)))
  (generate-explanation1 fact wm))

(defmethod generate-explanation1 (fact   (wm working-memory))
  (if fact
      (let* (
	     (which (do ((instances (trace-info wm) (cdr instances))
			 (which nil))
			((null instances) which)
		      (cond ((equal (string-upcase (symbol-name (car fact))) "LISP")
			     (setf which (make-instance 'ebg-tree
					   :fact (cdr fact)
					   :content nil
					   :descendants nil)))
	
			    ((member fact (conclusions (car instances))
				     :test 'equal)
			
			
			     (setf which (make-instance 'ebg-tree
					   :content (simplify-postcondition (rule (car instances)) fact)
					   :descendants (let ((result nil))
							  (dolist (fact (preconditions (car instances)) (reverse result))
							    (push  (generate-explanation1 fact wm) result))))
				   )
			     (if (assoc (car instances) *gealist*)
				 (setf (uid which) (cdr (assoc (car instances) *gealist*)))
			       (setf *gealist* (acons (car instances) (uid which) *gealist*)))
	  
	
			     )
			    ))))
	(if which which
	  (if (member fact (assertions wm) :test 'equal)
	      (make-instance 'ebg-tree
		:fact fact
		:content nil
		:descendants nil)
	    (make-instance 'ebg-tree
	      :fact (list fact "was not proven.")
	      :content nil
	      :descendants nil)
	    )))))


(defmethod simplify-postcondition ((rule rule) fact)
  (if (equal (car (then-part rule)) 'and)
      (make-instance 'rule
	:then-part (find-if #'(lambda (conc) (not (eq 'unify::fail (unify::unify fact conc nil))))
			    (cdr (then-part rule)))
	:if-part (if-part rule)
	:name-part (format nil "~a+" (name-part rule))
	:doc-part (doc-part rule))
    rule))


;;;--- Tests

;(mapcar #'(lambda (rule-obj) (name-part rule-obj)) 
;        (rule-set-part *rule-set*))

;(if-part (get-rule *rule-set* 'rule-a))




;;; ==========================================================================
;;; * END OF FILE *
;;; ==========================================================================
