;; -*- Mode: LISP; Package: common-lisp-user; Syntax: Common-lisp;      -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA
;;;
;;; ************************************************************************
;;;
;;; Filename:   csp.cl
;;; Short Desc: Algorithms for Constraint Propagation & Search
;;; Version:    2.0
;;; Status:     Review
;;; Last Mod:   25.2.93 - FB
;;; Author:     Fabio Baj
;;; Build csp
;;;

(in-package :csp)

;; NOTE
;; The code in this file is independent from the graphical interface
;; except for parts labelled with the following comment:
;;;**GIN**
;;  

;;-----------------------------------------------------------------
;; The MAIN FUNCTION 
;;-----------------------------------------------------------------

(defun csp-main (n)
  (csp-init)
  (message " Computing Node Consistency...")
  (node-consistency n) 
  (display n )(possible-solutions-message)
  (message  "--------->> The Network is Node Consistent")
  (fill-queue-with-arcs n)
 (message  " Propagating Constraints...")
  (propagate-constraints n)
  (message "--------->>  The Network is Arc Consistent")
  (if *stop-at-prop* (process-interrupts-for-stepper )) ;;**GIN** 
  (message " Combining Search and Constraint Propagation")
   (catch 'no-more-solutions (search-solutions n 0) )
   (number-of-solutions-message)
  )

;;-----------------------------------------------------------------
;; CONSTRAINT PROPAGATION ALGORITHM
;;-----------------------------------------------------------------
(defmethod propagate-constraints ((net constraint-network))
  (loop (if (empty-queue) (return net))
    (let* ((arc (dequeue))
	   (node1 (car arc)) (node2 (cdr arc))   
	   (old-num (length (possible-values node1))))
      (show-arc arc)
      (arc-consistency node1 node2)
      (if (impossible-p node1)
	  (progn (notify-failure node1)
		 (notify-reduction node1 node2 old-num)
		 (return nil)))
      (when (< (length (possible-values node1)) old-num)
	(notify-reduction node1 node2 old-num)
	(eventually-remove-from-other-domains node1 net)
	(enqueue node1)))))
	     	  
;;-----------------------------------------------------------------
;; COMBINING CONSTRAINT PROPAGATION AND SEARCH
;;-----------------------------------------------------------------
(defmethod search-solutions ((net constraint-network) level) 
  (let ((node (min-non-ambig-node net)))  
    (if (null node)(progn
		     (process-solution net level)
		     (if  (more-solutions) (list net)
		       (throw 'no-more-solutions nil)))		  
      (mapcan
       #'(lambda (n-value)
	   (let* ((network2 (make-copy net))
		  (n2 (find-node (name node) network2))
		  (old-num (number-of-values n2)))
	     (setf *searching-network* network2)
	     (init-queue)
	     (setf (possible-values n2) (list n-value))
	     (search-message n2 level)
	     (eventually-remove-from-other-domains n2 network2)
	     (notify-reduction n2 n2 old-num)
	     (enqueue n2)
	     (if (and (not (impossible-p network2))
		      (propagate-constraints network2))
		 (search-solutions network2 (1+ level) ))))
       (possible-values node)))))


;;-----------------------------------------------------------------
;; COMPUTATION OF NODE-CONSISTENCY 
;;-----------------------------------------------------------------

(defmethod node-consistency ((net constraint-network))
   (dolist (node (nodes net))
     (let ((oldnum (number-of-values node)))
       (node-consistency node)
       (when (> oldnum (number-of-values node))
	 (notify-reduction node node oldnum)
	 (eventually-remove-from-other-domains node net)))))

(defmethod  node-consistency ((nd node))
  (setf (possible-values nd)
    (remove-if-not #'(lambda (v)(csp-apply 'p-1 (list (name nd) v)))
		   (possible-values nd))))
  

;;-----------------------------------------------------------------
;;  COMPUTATION OF ARC-CONSISTENCY
;;-----------------------------------------------------------------
(defmethod arc-consistency ((nd1 node)(nd2 node))
  (trace-arc-consistency1 nd1 nd2)
  (if *stop-at-arc* (process-interrupts-for-stepper))       ;;**GIN**
  (setf (possible-values nd1)
    (remove-if
     #'(lambda (v1)
	 (null (remove-if-not
		#'(lambda (v2)
		    (process-interrupts)                    ;;**GIN**
		    (csp-apply 'p-2 
				(list (name nd1) v1 (name nd2) v2)))
		(possible-values nd2))))
     (possible-values nd1)))
  (trace-arc-consistency2 nd1 nd2))

;;-----------------------------------------------------------------
;; Activated when a value can occur once in a solution 
;; that is when *all-distinct-values* = t
;;-----------------------------------------------------------------
;;this is the default function to compare values within domains
;; The user can redefine it.
(defun equal-value(v1 v2)(equal v1 v2))

(defun eventually-remove-from-other-domains  (node1 net)
  (if *all-distinct-values* 
      (if (not (ambiguous-p node1))
	  (remove-value-from-other-domains node1 net))))

(defun remove-value-from-other-domains (node net)
  (let ((value-to-delete (car (possible-values node)))
	(node-list(remove node (nodes net))))
    (dolist (nd node-list) 
      (let ((old-num (length (possible-values nd))))
	(setf (possible-values nd)
	  (csp-apply 'remove
		     (list  value-to-delete (possible-values nd) :test #'equal-value)))
	(when (< (length (possible-values nd)) old-num)
	  (if (not (ambiguous-p nd))
	      (remove-value-from-other-domains nd net))
	  (notify-reduction nd nd old-num)
	  (enqueue  nd))))))

(defmethod all-distinct ((net constraint-network))
  (let ((values
	 (mapcar #'(lambda (nd) (car (possible-values nd)))
		 (nodes net))))
    (all-diff values)))

(defun  all-diff (l)
  (all-diff1 (car l)(cdr l)))
(defun all-diff1 (v l)
  (cond ((null l) t)
	( (member v l) nil)
	(t(all-diff1 (car l) (cdr l)))))


;;=========library==========================
;;(defun atom-conc (l-a)
;;;  (intern  
;;   (apply 'concatenate 'string (mapcar 'string l-a))
;;	   :csp))
;; ATTENZIONE MODIFICATA!
(defun atom-conc (l-a)
  (intern  
   (apply 'concatenate 'string (mapcar '(lambda (s)(format nil "~a" s)) l-a))
	   :csp))

(defun more-solutions()
  (if *stop-at-sol*
     
	(if (not  *csp-interface*)
	    (progn    (format t "More solutions? [y/n]  -> ") (eq 'y (read)))
	  (progn
	    (equal 'Yes (my-y-or-n-dialog "More Solutions?" ))))
    t))



;;================================================
;; An ARC is a cons of two nodes
;;================================================

;-----------------------------------------------------------------
; Computes the weight of an arc (nd1 . nd2)
(defun arc-wgt (arc)
  (let ((from-w  (number-of-values (car arc)))
	(to-w (number-of-values (cdr arc))))
    (+ (* from-w to-w) 
       (* (* from-w to-w)
	  (signum (- to-w from-w))
	  0.5))))

(defun cartesian (arc)
  (let ((from-w  (number-of-values(car arc)))
	(to-w  (number-of-values (cdr arc))))
    (* from-w to-w) ))

(defun arc< (a1 a2) 
  (< (arc-wgt a1)(arc-wgt a2)))

(defun same-arc (a1 a2)(eq a1 a2))

;;-----------------------------------------------------------------
;; Returns a list of all the  arcs in a network
;;-----------------------------------------------------------------
(defmethod all-arcs ((net constraint-network))
  (flatten (mapcar #'all-arcs (nodes net))))

(defmethod all-arcs ((nd node))
  (mapcar #'(lambda (nb) (cons nd nb)) 
	(neighbors nd)))

(defmethod fill-queue-with-arcs((n constraint-network))
  (setq *queue* (all-arcs n)))

;;-----------------------------------------------------------------
(defun show-q ()
   (mapcar 'show-arc
	   *queue*)
   (format *verbose* "--------------------- ~%"))
(defun show-arc (x)
  (format *verbose* "~A ~A  ~A ~A ~%" (name (car x))(name (cdr x))
	  (cartesian x)(arc-wgt x)))
;;-----------------------------------------------------------------
;; Returns the node with the smallest domain
;;-----------------------------------------------------------------
(defun min-non-ambig-node (net)
  (let* ((amb-nodes (remove-if-not #'ambiguous-p (nodes net)))
	 (res (car amb-nodes)))
    (dolist (n amb-nodes res)
      (if (< (number-of-values n)
	     (number-of-values res))
	  (setq res n)))))


;;-----------------------------------------------------------------

;;-----------------------------------------------------------------
;; Interface to the QUEUE data structure
;;-----------------------------------------------------------------
(defun empty-queue() (null *queue*))

(defun init-queue() (setq *queue* nil))

(defun dequeue()
  (setq *queue* (shift-minimal-arc *queue*))
  (pop *queue*))

(defun enqueue(node1)
  (mapc #'(lambda (nb)
	    (setq  *queue*  (add-arc (cons nb node1) *queue*)))
	(neighbors node1)))

;;---------------auxiliary functions for queues-----------
(defun shift-minimal-arc (queue)
  (shift-min-aux (car queue) (cdr queue) nil))
(defun shift-min-aux (arc queue temp)
  (cond ((null queue) (cons arc temp))
	((arc< (car queue) arc)  
	 (shift-min-aux (car queue) (cdr queue)(cons arc temp)))
	(t  (shift-min-aux arc (cdr queue)(cons (car queue) temp)))))

(defun add-arc (arc queue)
  (insert-arc arc  (remove arc  queue :test #'equal)))
(defun insert-arc (arc queue)
  (cons arc queue))


;;-----------------------------------------------------------------
;;  Auxilary Functions
;;-----------------------------------------------------------------
(defun find-node (name network)
  "Find the node in the given network with the given name."
  (find name (nodes network) :key #'name))

(defun n-d-neighbors (node-descriptor)
  "The neighboring node names in a node descriptor."
  (rest (rest node-descriptor)))

(defun construct-node (node-descriptor)
  "Build the node corresponding to the descriptor."
  ;; Descriptors are like: (x Domain y z)
   (make-instance 'node
    :name (first node-descriptor) 
    :domain (second node-descriptor)
    :possible-values (variable-domain (second node-descriptor))))

(defun construct-network (node-descriptors)
   (let ((network (make-instance 'constraint-network :main-networkp t)))
      (setf (nodes network) (mapcar #'construct-node node-descriptors))
      (dolist (n-d node-descriptors)
      (setf (neighbors (find-node (first n-d) network))
	    (mapcar #'(lambda (neighbor)
			(find-node neighbor network))
		    (n-d-neighbors n-d))))
      (if *graphics* (draw-graph network))
      (setq *main-network* network)
      network))

(defmethod process-solution ((net constraint-network) level)
  (push net *solutions*)  
  (let ((solmsg(format nil "~:R solution" (length *solutions*)))
	(fill   (make-string (* 2 level) :initial-element #\Space)))
      (if *graphics* 
	  (setf (title (window net))solmsg))
      (message (format nil "~A  ~A found" fill solmsg))))
	

(defun display-solutions()
 (dolist (s *solutions*)
    (display s)
    (safe-apply 'DOMAIN-DEPENDENT-SHOW-SOLUTION (list s))
    (if (not  (more-solutions))(return)))
 (number-of-solutions-message))
 


(defun csp-init () 
  (setq *open-display-nets* nil)
  (setq *solutions* nil))
;;-----------------------------------------------------------------
;; To trace  operations on screen and graphs
;;-----------------------------------------------------------------
(defun message(string)
  (if *csp-interface* (format-display-fill *out-win* string)
    (format t string)))

(defmethod trace-arc-consistency1((nd1 node)(nd2 node))
  (when *verbose* 
    (message "Computing arc consistency between nodes:")
    (message (display-string nd1))
    (message (display-string nd2))
    (message (format nil "~D Tests needed ..~%" 
		     (* (number-of-values nd1)(number-of-values nd2)))))
  (when *graphics* (highlight-node nd1)(highlight-node nd2)))

(defmethod search-message ((nd node) level)
  (let* ( (fill  (make-string (* 2 level) :initial-element #\.))
	  (msg 	 (format nil " Trying with ~A = ~A"
			 (name nd)(car (possible-values nd))))
	  (msg1 	 (format nil "~A~A" fill msg)))
    (message msg1)
    (if *graphics* (setf (title (display (domain-button nd))) msg))))

(defun trace-arc-consistency2 (node1 node2 )
  (if *verbose* (message (format nil "New situation :~%")))
  (if  *verbose* (display node1)) 
  (when *graphics* 
    (highlight-node node1)
    (highlight-node node2))
  (format *verbose*  "----------------------------~%"))

(defun notify-reduction(node1 node2 old-num)
  (if *verbose* (message (format nil "** Reduced ~A from ~A to ~a "
			       (name node1)
			       old-num  
			       (number-of-values node1))))
  (when *graphics* 
    (setf (label (domain-button node1))  
      (format nil "~D"  (number-of-values node1)))
    )
  (force-output))

(defun notify-failure (node1)
  (if *graphics* (close-display (window *searching-network*)))
  (message  (format nil "         *****FAILURE**** on ~A" (name node1))))
		 
(defun flatten (l)
  (let ((res nil)) (dolist (x l) (setq res (append x res)))  res))

(defun number-of-solutions-message()
    (message "------------------------------")
    (message (format nil "Number of solution~:p: ~D" (length *solutions*)))
    (message "------------------------------")
 )

(defun possible-solutions-message()
(message (format nil 
"There are ~:d Potential Solution~:p.~%" 
(reduce #'* (mapcar #'number-of-values (nodes *main-network*))))))
;;-----------------------------------------------------------
;; This functions is like apply, but issues a message in the
;; CSP output window, without crashing PAIL.
;;
(defun safe-apply(fn args)
  (let ((result))
    (multiple-value-bind
	(result error-condition)
	(ignore-errors (setq result (apply fn args)))
      (if error-condition
	  (progn
	    (message
	     (format nil 
		     "**CSP Error***
A problem has been detected while executing 
~A on arguments ~A~%~A
Execution aborted."
		     fn args
		     (apply 'format nil  (slot-value error-condition 'excl::format-control ) 
			    (slot-value  error-condition 'excl::FORMAT-ARGUMENTS  ))))
	    'csp-error)
	result))))

    
(defun csp-apply(fn args)
  (let ((result (safe-apply fn args) ))
  (if (eq 'csp-error result)
      (throw 'aborted 'aborted) result)))

    

