;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   atn.graph.cl
;;; Short Desc: dialog handling for Augmented Transition Networks
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Mike Lenz
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atn)

;;
;; M. Lenz
;; pAIL ATN module
;;
;; Graphical network-plotting routines
;; 27-8-91
;;

;; These commands must have been executed before
;; this file is loaded:
;;

;;
;; Warning: Most of these routines are quick last-minute jobs
;; to get the thing to work...
;;

(setq *idlewindows* nil)
(setq *usedwindows* nil)

(setq *titlefont* (open-font :courier :roman 18 :weight :bold))


(setq *initleft* 0)
(setq *initbottom* 0)
(setq *nextleft* 0)
(setq *nextbottom* 0)



(defun drawnode (x y str)
  (setf (font *window*) *nodefont*)
  (draw-circle *window* x y *nodesize*)
  (let ((chx (- x (floor (/ (font-string-width (font *window*) str) 2))))
	(chy (- y (floor (/ (font-character-height (font *window*)) 2)))))
    (write-display *window* str chx (+ chy 3)))
  )

;;

(defstruct atn-node
  window            ; the node's window
  x                 ; its x and y coordinates
  y
  name              ; the name of the node, as a string
  radpos            ; its position in radians on the loop
)

; not used:

(defstruct netwindow
  window
  nodelist
)

;;
;; Places num nodes around a circle centered at x, y with radius rad.
;;

(defun placenodes (num cx cy rad)
  (let ((step (/ (* 2 pi) num))
	(r (- pi))
	(x 0)
	(y 0))
    (dotimes (i num)
      (setq x (+ (* (cos r) rad) cx))
      (setq y (+ (* (sin r) rad) cy))
      (format t "x = ~a  ... y = ~a~%" (floor x) (floor y))
      (node (floor x) (floor y) "hey")
      (setq r (+ r step)))))

;;
;; These do the same thing as placenodes, except you must
;; call drawnextnode explicitly for each node to draw.
;; Remembers its "state" via global variables.
;;

(defun initdrawnodes (num cx cy rad window)
  (setq *step* (/ (* 2 pi) num))
  (setq *pos* (- pi))
  (setq *rad* rad)
  (setq *window* window)
  (setq *cx* cx)
  (setq *cy* cy))

(defun drawnextnode (str)
  (let ((x (+ (* (cos *pos*) *rad*) *cx*))
	(y (+ (* (sin *pos*) *rad*) *cy*)))
;;    (format t "x = ~a  ... y = ~a~%" (floor x) (floor y))
    (drawnode (floor x) (floor y) (format nil "~A"   str))
    (prog1 (make-atn-node :x (floor x) :y (floor y) :name str :radpos *pos*)
           (setq *pos* (- *pos* *step*)))))

;;

(defun initnodes (num window)
  (let ((w (/ (width window) 2))
	(h (/ (height window) 2)))
    (initdrawnodes num w h (- (min w h) *margin*) window)))

;;
;; This returns a nodelist to drawnetwork, which fits
;; it into the nwindow's nodelist slot.
;;

(defun drawsubnet (net subnet window)
  (let ((nodes (cdr (assoc subnet net)))
	(nodelist nil))
    (if (eql (caar nodes) 'regs)
	(setq nodes (cdr nodes)))
    (if (eql (caar nodes) 'init)
	(setq nodes (cdr nodes)))

    (clear-display window)
;    (setf (font window) *titlefont*)
;    (write-display window
;		   (format nil "~A"   subnet) 10 (- (height window) 30))

    (initnodes (length nodes) window)
    (dolist (anode nodes)
      (push (drawnextnode (car anode))
	    nodelist))

    ;; draw arrow to start node

    (let* ((startnode (car (last nodelist)))
	   (srcx (atn-node-x startnode))
	   (srcy (atn-node-y startnode)))
      (offsetarrow (- srcx *selfarrowlen*) srcy srcx srcy))

    ;; draw arrows for all other arcs, after collapsing
    ;; multiple source/target arcs to single arcs

    (dolist (anode nodes)
      (let* ((srcgraph (itsgraph (car anode) nodelist))
	     (srcx (atn-node-x srcgraph))
	     (srcy (atn-node-y srcgraph))
	     (newarclist nil))
	(dolist (anarc (cdr anode))
	  (let* ((tgtnode (arc-tgtnode anarc))
		 (entry (assoc tgtnode newarclist)))
	    (if entry
		(setf (cdr entry)
		      (concatenate 'string (cdr entry) "/" (makelabel anarc)))
	      (push (cons tgtnode (makelabel anarc)) newarclist))))
	(dolist (anarc newarclist)
          (let* ((tgtnode (car anarc))
		 (tgtgraph (itsgraph tgtnode nodelist)))
	    ;; special case if node points to itself
	    (if (eql (car anode) tgtnode)
		(self-arrow-label
		 srcx srcy (atn-node-radpos srcgraph) t (cdr anarc))
	      (if tgtgraph
		  (let ((tgtx (atn-node-x tgtgraph))
			(tgty (atn-node-y tgtgraph)))
		    (progn (offsetarrow srcx srcy tgtx tgty)
			   (placelabel srcx srcy tgtx tgty (cdr anarc))))
		;; otherwise it's a POP arc
		(progn (draw-circle window srcx srcy (- *nodesize* 3))
		       (self-arrow-label
			srcx srcy (atn-node-radpos srcgraph) nil (cdr anarc)))))
	    ))))
    nodelist))

;;
;; Used and idle window lists have elements of the form
;;  (<subnetname> <windowstruct> <nodelist>)
;;

(defun drawnetwork (net)
  (format-display *atn-output-window*  "Please wait a moment while drawing network...~%")
  (dolist (subnet net)
    (let ((sname (car subnet))
	  (sc 0)
	  ;;;;;;(sc (subnet-complexity subnet))
	  (nwindow (pop *idlewindows*)))
      (if (null nwindow)
	  (progn (setf nwindow
		   (list sname
			 (make-instance 'display :left *nextleft*
				       :bottom *nextbottom*
				     ;;  :width  (floor (scalew 1152) (length *net*))
				    ;;   :height (floor (scalew 1152) (length *net*))
				       :width  (floor (scalew 1152) 4)
				       :height (floor (scalew 1152) 4)
				       :font (my-findfont 10)
				       :title (format nil "~A"   sname))
			 nil))
		 
		 (setq *nextleft* (+ *nextleft* (floor (- (scalew 1152) (floor (scalew 1152) 4))
						       (1- (length *net*))))))
	(progn (setf (first nwindow) sname)
	       (setf (title (second nwindow)) (format nil "~A"   sname))))
      
      (setq *usedwindows* (append *usedwindows* (list nwindow)))
      (setf (third nwindow)
	(drawsubnet net (car subnet) (second nwindow)))))
  (setq *hilitenet* nil)
  (setq *hilitenode* nil)
  (dolist (w *idlewindows*)
    (close-display (second w))
    (setq *nextleft* (- *nextleft* *defwidth*)))
  (setq *idlewindows* nil))
  
;;
;; This should be called after the completion of one
;; parse with dancing highlit node.
;;

(defun donedrawnet ()
  (setq *idlewindows* (append *usedwindows* *idlewindows*))
  (setq *usedwindows* nil))

;;
;; This should be called after done with ALL graphics
;; windows (e.g. user turns off graphics mode).
;;

(defun donegraphics ()
  (dolist (w *idlewindows*)
    (close-display (second w)))
  (setq *idlewindows* nil)
  (setq *nextleft* *initleft*)
  (setq *nextbottom* *initbottom*))

;;
;;

(defun itsgraph (nodename nodelist)
  (dolist (agraph nodelist)
    (if (eql nodename (atn-node-name agraph))
	(return-from itsgraph agraph)))
  ;;  (format-display *atn-output-window*  "ERROR node ~a has no graph.~%" nodename)
  )


;;
;; Draws an arrow from node centered at (x1, y1) to that
;; at (x2, y2) -- i.e. it offsets each endpoint in the
;; proper direction by the radius of a node.
;;

(defun offsetarrow (x1 y1 x2 y2)
  (let* ((posangle (if (= x1 x2)
		       (if (> y1 y2) (- (/ pi 2)) (/ pi 2))
		     (atan (/ (- y2 y1) (- x2 x1)))))
	 (angle (if (> x1 x2)
		    (+ pi posangle) posangle))
	 (sx1 (+ x1 (* (cos angle) *nodesize*)))
	 (sx2 (- x2 (* (cos angle) *nodesize*)))
	 (sy1 (+ y1 (* (sin angle) *nodesize*)))
	 (sy2 (- y2 (* (sin angle) *nodesize*)))
	 )
;;    (format t "y1 ~a y2 ~a atan ~a angle ~a cos ~a sin ~a~%" y1 y2 posangle angle
;;	    (cos angle) (sin angle))

    (draw-line *window* sx1 sy1 sx2 sy2 :arrow t)))

;;
;; Places a label near the midpoint of the line from
;; (x1, y1) to (x2, y2).
;;

(defun placelabel (x1 y1 x2 y2 str)
  (setf (font *window*) *labelfont*)
  (let* ((cx (+ x1 (/ (- x2 x1) 2)))
	 (cy (+ y1 (/ (- y2 y1) 2)))
	 (w (+ (font-string-width *labelfont* str) 1))
	 (h (- (font-character-height *labelfont*) 1))
	 (anchorx (+ cx 2))
	 (anchory (+ cy 2)))
    (if (< y1 y2)
	(progn (setq anchorx (- cx w))
	       (if (< x2 x1)
		   (setq anchory (- cy h))))
      (if (< x2 x1)
	  (setq anchory (- cy h))))
    (write-display *window* str anchorx anchory)))

;;
;; Construct an arc's label
;;

(defun makelabel (arc)
  (let* ((type (arc-type arc))
	 (str (format nil "~A"   type)))
    (if (or (eql type 'cat)
	    (eql type 'vir)
	    (eql type 'push))
	(setq str (concatenate 'string str " "
			       (format nil "~A"   (arc-label arc)))))
    str))

;;
;; Draws an arrow of a default length to the node at (x, y)
;; in the direction radpos; if pointin is t, the arrow
;; points inward (used for nodes to point to themselves);
;; otherwise it points outward (used for pop arcs).
;;

(defun self-arrow-label (x y radpos pointin label)
  (let* ((itsrad (if pointin
		     (- radpos (/ pi 4))
		   (+ radpos (/ pi 4))))
	 (tgtx (+ x (* *selfarrowlen* (cos itsrad))))
	 (tgty (+ y (* *selfarrowlen* (sin itsrad)))))
    (if pointin
	(progn (offsetarrow tgtx tgty x y)
	       (placelabel tgtx tgty x y label))
      (progn (offsetarrow x y tgtx tgty)
	     (placelabel x y tgtx tgty label)))))

;;
;; These are called inside parsing routine
;; to move about the jumping node cursor.
;;

(defun move-cursor-to (subnet node)
 (if *hilitenet*
      (highlight-node *hilitenet* *hilitenode*))
  (if subnet
      (highlight-node subnet node))
  (setq *hilitenet* subnet)
  (setq *hilitenode* node))

;;

(defun highlight-node (subnet node)
  (let ((nwindow (assoc subnet *usedwindows*)))
    (if nwindow
	(let ((graph (itsgraph node (third nwindow))))
	  (expose  (second nwindow))
	  (if graph
	      (draw-filled-circle (second nwindow)
				  (atn-node-x graph)
				  (atn-node-y graph)
				  (1- *nodesize*)
				  :operation cw::boole-xor)
	    (format-display *atn-output-window*  "Graphics error: node ~a not found.~%" node)))
      (format-display *atn-output-window* "Graphics error: subnet ~a not found.~%" subnet))))

;;

;; evaluting the complexity of a network

(defun subnet-complexity (subnet)
  (- (* 30 (length (cdr subnet))) 100))


