;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   order.cl
;;; Short Desc: Lexicogrphic path ordering
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)



;;----------------------------------------------------------
;; lro          : TERM x TERM  --->  { = > < <> }
;; Behavior     : Computes lro between s and tt               
;; Side effects : None

(defun lro ( s tt)
  (cond 
   ($manual-orient$ '<>)
   ((equal s tt) '=)
   ((lro-test s tt) '>)
   ((lro-test tt s) '<)
   (t '<>)))


;;----------------------------------------------------------
;; lro-test     : TERM x TERM  --->  {t nil}
;; Behavior     : is s > t in the recursive lex. path ordering?            
;; 
;; s>t IF  t is a variable occurring in s
;;  "  "   t is an argument of s      
;;  "  "   functor(s) > functor(t) AND 
;;            s > all the arguments of t.
;;  "  "   functor(s) = functor(t) AND   
;;            s > all the arguments of t,
;;            after having skipped equal arguments
;;         OR there exists an argument si of s such that
;;         si > t
;;  "  "   functor(s) < functor(t) AND
;;            there exists an argument si of s such that
;;            si > t
;; Side effects : None

(defun lro-test ( s tt)
  (cond 
   ((and (is-int-var tt)) (occurs tt s))
   ((is-int-var s) nil)
   ((and (listp s)(member tt (cdr s) :test #'equal)))
   ((and (funct> s tt))   (lro-sons s tt))
   ((and (funct= s tt))   (case-equal s tt))
   ((and (funct< s tt))   (exist-greater s tt))))
	 
(defun lro-sons ( s tt)
  (cond ((atom tt) tt)
	(t (lro-sons1 s (cdr tt)))))
(defun lro-sons1 (s tis)
  (cond ((null tis) t)
	(t (and (eq '> (lro s (car tis)))
		(lro-sons1 s (cdr tis))))))
  
(defun case-equal (s tt)
  (case-equal1 s tt (cdr s)(cdr tt)))
(defun case-equal1 ( s tt  sis tis)
  (cond 
   ((equal (car sis) (car tis)) (case-equal1 s tt (cdr sis) (cdr tis)))
   ((eq '> (lro (car sis) (car tis )))(lro-sons1 s (cdr tis)))
   (t (exist-greater1 (cdr sis) tt))))

(defun exist-greater ( s tt)
  (cond ((atom s) nil)
	(t (exist-greater1 (cdr s) tt))))
(defun exist-greater1 (sis tt)
  (cond 
   ((null sis) nil)
   (t (let ((ord (lro (car sis) tt)))
	(cond ((or (eq '= ord)(eq '> ord)) t)
	      ( t (exist-greater1 (cdr sis) tt)))))))
	       
  
  

(defun functor (term)
  (cond ((atom term)term)
	(t (car term))))

(defun funct> ( s tt)
  (cond 
   ((user-funct> s tt) t)
   ((and (numberp s) (numberp tt))  (> s tt))
   ((or  (numberp s) (numberp tt)) (numberp tt))
    (t (string> (string (functor s))  (string (functor tt))))))

(defun funct< ( s tt)
  (cond  
   ((user-funct< s tt) t)
   ((and (numberp s) (numberp tt))  (< s tt))
   ((or  (numberp s) (numberp tt)) (numberp s))
   
   (t (string< (string (functor s))  (string (functor tt))))))

(defun funct= ( s tt)
  (equal (functor s) (functor tt)))

(defun user-funct< (s tt)
  (member (functor tt) (member (functor s)  $precedence-list$)))

(defun user-funct> (s tt)
  (member  (functor s) (member  (functor tt) $precedence-list$)))

;;-----------------------------------------------------
;; orient-equations  : CLAUSE --> CLAUSE
;; Behavior          : orients each equation occurring
;;                   : in rule, according with >RLO. If
;;                   : $prolog-mode$ or manual orientation
;;                   : are set, nothing is done.
;; Side effects : None

(defmethod orient-equations (null-clause) nil)
(defmethod orient-equations ((clause clause-class))
  (if (or $manual-orient$ $prolog-mode$ $equations-in-theorem$)
      clause
    (remake-clause
     (orient-st (positive-atoms  clause))
     (orient-st (negative-atoms  clause))
     clause)))

(defun orient-st (st)
  (cond ((null st ) nil)
	((is-equation (car st))
	 (cons (orient (car st))
	       (orient-st (cdr st))))
	(t (cons (car st) (orient-st (cdr st))))))


(defun orient (equation)
  (let ((ord (lro (cadr equation)(caddr equation))))
    (cond ((eq ord '<) (list '-> (caddr equation)(cadr equation)))
	  ((eq ord '>) (list '-> (cadr equation)(caddr equation)))
	  (t (list '= (cadr equation)(caddr equation))))))
