//   C++ Object-oriented Base neural network
//   and derived Back-propagation neural network
//   based on the equations in Rummelhart and
//   McClelland's "Parallel Distributed Processing"
//   and the shareware back-propagation neural
//   network written in C by Eberhart and Dobbins
//   published in Micro Cornucopia, #51: Jan-Feb 90.
//   This is part of the Expert's Toolbox column for
//   AI Expert magazine, February 1991.

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <conio.h>
#include <ctype.h>
#include <string.h>
#include <iostream.h>

#define ESC   27
#define ITEMS  8

// object types

class BaseNet {         // A basic Neural Network type
                        // Includes matrix methods
public:
   float Eta,           // default learning rate
	 Alpha,         // default momentum factor
	 ErrorLevel,    // acceptable error level
	 Error;         // latest sum squared error value
   char  KeyboardRequest; // true when key pressed
   int   ErrorFreq,     // error reporting frequency
	 nInputNodes,   // number of input nodes
	 nHiddenNodes,  // number of hidden nodes
	 nOutputNodes,  // number of output nodes
	 nIterations,   // number of iterations
	 nPatterns,     // number of patterns
	 nRuns,         // number of runs (or input lines)
	 H,             // index hidden layer
	 I,             // index input layer
	 J,             // index output layer
	 P,             // index pattern number
	 Q,             // index iteration number
	 R;             // index run number
   FILE  *RunFile,        // RUN file
	 *PatternFile,    // source pattern input file
	 *WeightsInFile,  // initial weight file
	 *WeightsOutFile, // final weight output file
	 *ResultsFile,    // results output file
	 *ErrorFile;      // error output file
   char  szResults[40];   // various filenames
   char  szError[40];
   char  szPattern[40];
   char  szWeights[40];
   char  szWeightsOut[40];

// Matrix
// typedefs and prototypes for dynamic storage of arrays
   typedef float *FLOATPTR;   // Pointer to a real
   typedef FLOATPTR VECTOR;   // A Vector: one column
   typedef FLOATPTR *MATRIX;  // A Matrix: two columns

// Network Layers
// Arrays for inputs, outputs, deltas, weights & target outputs
   MATRIX   Out0;        // input layer
   MATRIX   Out1;        // hidden layer
   MATRIX   Delta1;      // delta at hidden layer
   MATRIX   Delw1;       // change in weights input:hidden
   MATRIX   W1;          // weights input:hidden
   MATRIX   Out2;        // output layer
   MATRIX   Delta2;      // delta at output layer
   MATRIX   Delw2;       // change in weights hidden:output
   MATRIX   W2;          // weights hidden:output
   MATRIX   TargetOutput; // target output
   VECTOR   PatternID;   // identifier for each stored pattern

// Memory allocation methods
   void  AllocateVector(VECTOR *Vector, int nCols);
   void  AllocateColumns(FLOATPTR Matrix[], int nRows, int nCols);
   void  AllocateMatrix(MATRIX *pmatrix, int nRows, int nCols);
   void  FreeMatrix(MATRIX Matrix,  int nRows);

   BaseNet();         // constructor
  ~BaseNet() {};      // destructor
   virtual void Iterate(char Netname) {}; // abstract iteration loop
					  // for any network
};

class BackProp : public BaseNet {  // Back Propagation network
public:
   BackProp()  {};
   ~BackProp() {};
   void Iterate(char Netname);  // iteration loop for this network
};

// BaseNet constructor initializes default fields.

BaseNet::BaseNet() {
    Eta   =  0.15,        // default learning rate
    Alpha =  0.075;       // default momentum factor
    ErrorFreq = 100;      // error reporting frequency
    ErrorLevel = 0.04;    // acceptable error level
    KeyboardRequest = 0;  // true when key pressed
};

// BaseNet methods

// Implementation of array allocation routines
// Allocate space for a vector of float cells,
// a one dimensional dynamic vector[cols]

void BaseNet::AllocateVector(VECTOR *Vector, int nCols)
{
   if ((*Vector = (VECTOR) calloc(nCols, sizeof(float))) == NULL)
   {
      cout << " Not enough memory!\n"; // If not, abort.
      exit(1);
   }
}

// Allocate space for a dynamic two dimensional matrix[rows][cols]
void BaseNet::AllocateColumns(FLOATPTR Matrix[], int nRows, int nCols)
{
   int  i;
   for (i = 0;  i < nRows;  i++)
      AllocateVector(&Matrix[i], nCols);
}

void BaseNet::AllocateMatrix(MATRIX *Pmatrix, int nRows, int nCols)
{
   if ((*Pmatrix = (MATRIX) calloc(nRows, sizeof(FLOATPTR))) == NULL)
   {
      cout << "Not enough memory!\n";
      exit(1);
   }
   AllocateColumns(*Pmatrix, nRows, nCols);
};

// Free the memory used by the Matrix
void BaseNet::FreeMatrix(MATRIX Matrix, int nRows)
{
   int   i;
   for (i = 0;  i < nRows;  i++)
      free(Matrix[i]);
   free(Matrix);
}

// Specific implementation of iteration loop
// for a back-propagation network

void BackProp::Iterate(char Netname) {
   for (R = 0;   R < nRuns;   R++)
   {
      // Read and parse the run specification line
      // to obtain information about this network.
      fscanf(RunFile,
	  "%s %s %s %s %s %d %d %d %d %d %f %f",
	  szResults,    // output results file
	  szError,      // error output file
	  szPattern,    // pattern input file
	  szWeights,    // initial weights file
	  szWeightsOut, // final weights output file
	  &nPatterns,   // number of patterns to learn
	  &nIterations, // number of iterations through the data
	  &nInputNodes, // number of input nodes
	  &nHiddenNodes,// number of hidden nodes
	  &nOutputNodes,// number of output nodes
	  &Eta,         // learning rate
	  &Alpha);      // momentum factor

// Allocate dynamic storage for nodes and patterns.
      AllocateMatrix(&Out0,      nPatterns,    nInputNodes);
      AllocateMatrix(&Out1,      nPatterns,    nHiddenNodes);
      AllocateMatrix(&Out2,      nPatterns,    nOutputNodes);
      AllocateMatrix(&Delta2,    nPatterns,    nOutputNodes);
      AllocateMatrix(&Delw2,     nOutputNodes, nHiddenNodes + 1);
      AllocateMatrix(&W2,        nOutputNodes, nHiddenNodes + 1);
      AllocateMatrix(&Delta1,    nPatterns,    nHiddenNodes);
      AllocateMatrix(&Delw1,     nHiddenNodes, nInputNodes + 1);
      AllocateMatrix(&W1,        nHiddenNodes, nInputNodes + 1);
      AllocateMatrix(&TargetOutput,nPatterns,  nOutputNodes);
      AllocateVector(&PatternID, nPatterns);

//  Read the initial weight matrices.
      if ((WeightsInFile = fopen(szWeights,"r"))  ==  NULL)
      {
	 cout << " Can't open file \n" <<  Netname << szWeights;
	 exit(1);
      }

// read input:hidden weights
      for (H = 0;  H < nHiddenNodes;  H++)
	 for (I = 0;  I <= nInputNodes;  I++)
	 {
	    fscanf(WeightsInFile, "%f", &W1[H][I]);
	    Delw1[H][I] = 0.0;
	 }

// read hidden:out weights
      for (J = 0;  J < nOutputNodes;  J++)
	 for (H = 0;  H <= nHiddenNodes;  H++)
	 {
	    fscanf(WeightsInFile, "%f", &W2[J][H]);
	    Delw2[J][H] = 0.0;
	 }
      fclose(WeightsInFile);

// Read in all patterns to be learned.
      if ((PatternFile = fopen(szPattern, "r"))  ==  NULL)
      {
	 cout << " Can't open file \n" << Netname << szPattern;
	 exit(1);
      }

      for (P = 0;  P < nPatterns;  P++)
      {
	 for (I = 0; I < nInputNodes; I++)
	    if (fscanf(PatternFile,"%f", &Out0[P][I])!= 1)
	       goto AllPatternsRead;

// Read in target outputs for input patterns.
	 for (J = 0; J < nOutputNodes; J++)
	    fscanf(PatternFile, "%f", &TargetOutput[P][J]);

// Read in identifier for each pattern.
	 fscanf(PatternFile,  "%f ",   &PatternID[P]);
      }
      AllPatternsRead:      // Then, we're done.
      fclose(PatternFile);

      if (P < nPatterns)
      {
	 cout << " Can't open file \n" <<  Netname << P << nPatterns;
	 nPatterns = P;
      }

// Open error output file
      if ((ErrorFile = fopen(szError, "w"))  ==  NULL)
      {
	 cout << " Can't open file \n" << Netname << szError;
	 exit(1);
      }
       fprintf(stderr, nIterations > 1 ? "Training...\n" : "Testing\n");

//  Iteration loop
      for (Q = 0;  Q < nIterations;  Q++)
      {
	 for (P = 0;  P < nPatterns;  P++)
	 {

      // Hidden layer
      // Sum input to hidden layer over all
      // input-weight combinations
	    for (H = 0;  H < nHiddenNodes;  H++)
	    {
	       float Sum = W1[H][nInputNodes];  // Begin with bias
	       for (I = 0;  I < nInputNodes;  I++)
		  Sum   +=   W1[H][I]  *  Out0[P][I];

	       // Compute output using sigmoid function.
	       Out1[P][H] = 1.0 / (1.0 + exp(-Sum));
	    }

       // Output layer
	    for (J = 0;  J < nOutputNodes;  J++)
	    {
	       float  Sum = W2[J][nHiddenNodes];
	       for (H = 0;  H < nHiddenNodes;  H++)
		  Sum  +=   W2[J][H]  *  Out1[P][H];
	       // Compute output using sigmoid function.
	       Out2[P][J]  =  1.0  /  (1.0  +  exp(-Sum));
	    }

	  // Delta output
	  // Calculate deltas for each output unit for each pattern.
	    for (J = 0;  J < nOutputNodes;  J++)
	       Delta2[P][J] = (TargetOutput[P][J] - Out2[P][J])  *
			       Out2[P][J] * (1.0 - Out2[P][J]);

	  // Delta hidden
	    for (H = 0;  H < nHiddenNodes;  H++)
	    {
	       float  Sum = 0.0;
	       for (J = 0;  J < nOutputNodes;  J++)
		  Sum += Delta2[P][J] * W2[J][H];
	       // Compute output using sigmoid function.
	       Delta1[P][H] = Sum * Out1[P][H] * (1.0 - Out1[P][H]);
	    }
	 }

	  // Adapt weights hidden:output
	  for (J = 0;  J < nOutputNodes;  J++)
	 {
	    float  Dw;        // delta weight
	    float  Sum = 0.0;

	    // Sum of deltas for each output node for one epoch
	    for (P = 0;  P < nPatterns;  P++)
	       Sum  +=  Delta2[P][J];

	    // Calculate new bias weight for each output unit
	    Dw   =   Eta * Sum  +  Alpha * Delw2[J][nHiddenNodes];
	    W2[J][nHiddenNodes]   +=   Dw;
	    Delw2[J][nHiddenNodes] =   Dw;  // delta for bias

	    // Calculate new weights
	    for (H = 0;  H < nHiddenNodes;  H++)
	    {
	       float  Sum = 0.0;
	       for (P = 0;  P < nPatterns;  P++)
		  Sum  +=  Delta2[P][J] * Out1[P][H];
	       Dw          =  Eta * Sum  +  Alpha * Delw2[J][H];
	       W2[J][H]    += Dw;
	       Delw2[J][H] =  Dw;
	    }
	 }

	    // Adapt weights input:hidden
	 for (H = 0;  H < nHiddenNodes;  H++)
	 {
	    float  Dw;                  // delta weight
	    float  Sum = 0.0;
	    for (P = 0;  P < nPatterns;  P++)
	       Sum  +=  Delta1[P][H];

	    // Calculate new bias weight for each hidden unit
	    Dw   =   Eta * Sum  +  Alpha * Delw1[H][nInputNodes];
	    W1[H][nInputNodes]   +=   Dw;
	    Delw1[H][nInputNodes] =   Dw;

	    // Calculate new weights
	    for (I = 0;  I < nInputNodes;  I++)
	    {
	       float  Sum = 0.0;
	       for (P = 0;  P < nPatterns;  P++)
		  Sum  +=  Delta1[P][H] * Out0[P][I];
	       Dw           =   Eta * Sum  +  Alpha * Delw1[H][I];
	       W1[H][I]     +=  Dw;
	       Delw1[H][I]  =   Dw;
	    }
	 }

 // Watch for keyboard requests
	 if (kbhit())
	 {
	     int    c = getch();
	     if ((c = toupper(c))  == 'E')
		KeyboardRequest++;
	     else if (c == ESC)
		break;  // End if ESC request
	 }

// Sum Squared Error
	 if (KeyboardRequest || (Q % ErrorFreq == 0))
	 {
	    for (P = 0, Error = 0.0; P < nPatterns;   P++)
	    {
	       for (J = 0;  J < nOutputNodes;  J++)
	       {
		  float  Temp   =   TargetOutput[P][J] - Out2[P][J];
		  Error += Temp * Temp;
	       }
	    }

	    // Average error over all patterns
	    Error  /=  nPatterns * nOutputNodes;

	    // Print iteration number and  error value
	    fprintf(stderr,"Iteration %5d/%-5d  Error %f\r",
	       Q, nIterations, Error);
	    KeyboardRequest = 0;

	    if (Q % ErrorFreq == 0)
	       fprintf(ErrorFile, "%d %f\n", Q, Error); // to file
	    // Terminate when error satisfactory
	    if (Error < ErrorLevel)
	       break;
	 }
      }
      // End iterate loop

      // Display error, iterations, etc.
      for (P = 0, Error = 0.0; P < nPatterns;  P++)
      {
	 for (J = 0; J < nOutputNodes;  J++)
	 {
	   float Temp = TargetOutput[P][J] - Out2[P][J];
	   Error += Temp * Temp;
	 }
      }

      // Average error over all patterns
      Error  /=  nPatterns *nOutputNodes;

      // Print final iteration number and error value
      fprintf(stderr, "Iteration %5d/%-5d  Error %f\n", Q,
	      nIterations, Error); /* to screen */
      fclose(ErrorFile);

      // Print final weights
      if ((WeightsOutFile = fopen(szWeightsOut,"w"))  ==  NULL)
      {
	 cout << " Can't write file\n" << Netname << szWeightsOut;
	 exit(1);
      }

      for (H = 0;  H < nHiddenNodes; H++)
	 for (I = 0;  I <= nInputNodes; I++)
	    fprintf(WeightsOutFile, "%g%c", W1[H][I],
		    I%ITEMS==ITEMS-1 ? '\n':' ');

      for (J = 0;  J < nOutputNodes;  J++)
	 for (H = 0;  H <= nHiddenNodes;  H++)
	    fprintf(WeightsOutFile,  "%g%c", W2[J][H],
		    J%ITEMS==ITEMS-1 ? '\n':' ');

      fclose(WeightsOutFile);

      // Print final activation values
      if ((ResultsFile = fopen(szResults,"w"))  ==  NULL)
      {
	 cout << " Can't write file \n" << Netname << szResults;
	 ResultsFile = stderr;
      }

      // Print final output vector
      for (P = 0;  P < nPatterns;  P++)
      {
	 cout << ResultsFile << P;
	 for (J = 0;  J < nOutputNodes;  J++)
	     cout << ResultsFile << Out2[P][J];
	 cout << ResultsFile << "\n" << PatternID[P];
      }
      fclose(ResultsFile);

  // Free memory used for Matrix
      FreeMatrix(Out0, nPatterns);
      FreeMatrix(Out1, nPatterns);
      FreeMatrix(Delta1, nPatterns);
      FreeMatrix(Delw1, nHiddenNodes);
      FreeMatrix(W1, nHiddenNodes);
      FreeMatrix(Out2, nPatterns);
      FreeMatrix(Delta2, nPatterns);
      FreeMatrix(Delw2, nOutputNodes);
      FreeMatrix(W2, nOutputNodes);
      FreeMatrix(TargetOutput, nPatterns);
      free(PatternID);
    }
   fclose(RunFile);  // Close Run file
}


// Main program : creates and runs a BackProp Network

   void main(int argc, char *argv[]) {

     BackProp Bp;              // Instance of a BackProp network
     char *Netname  =  *argv;  // netname is read from argument list

   // Read arguments from DOS command line
   for (; argc > 1;  argc--)
   {
      char *arg = *++argv;
      if (*arg  !=  '-')
         break;
      switch (*++arg)
      {
	case 'e':   sscanf(++arg,  "%d",  &Bp.ErrorFreq);   break;
	case 'd':   sscanf(++arg,  "%f",  &Bp.ErrorLevel);  break;
	default:    break;
      }
   }
   if (argc < 2)
   {
      fprintf(stderr, "Usage:  %s {-en -df} runfilename\n",  Netname);
      fprintf(stderr, "   -en   =>  report error every n iterations\n");
      fprintf(stderr, "   -df   =>  done if sum squared error < f\n");
      exit(1);
   }

   // Open run file for reading
   if ((Bp.RunFile = fopen(*argv, "r"))  ==  NULL)
   {
      cout << " Can't open file \n" << Netname << *argv;
      exit(1);
   }

   // Read first line from Run file
   // which contains number of runs
   // and other information for the network
   // Run file in this example looks like this:
   // training.out training.err training.pat weight.wts } these 2 lines
   // training.wts 100 100 9 4 2 0.15 0.075             } are on 1 line
   //                                                   } in file
   // pattern, weight, and other input files
   // consist of numbers separated by spaces.

      fscanf(Bp.RunFile, "%d", &Bp.nRuns); // Scan for no. of runs
      Bp.Iterate(*Netname);                // Iterate a BackProp network.
};


