{.PW132}
{.HE OPT.PAS                                Page # }
{$V-,R+}
PROGRAM optimum ;

(* Copyright 1987 - Knowledge Garden Inc.
                    473A Malden Bridge Rd.
                    R.D. 2
                    Nassau, NY 12123       *)


(* This program implements the probabilistic inference technique described in the
   AI apprentice column in December 1987 issue of AI Expert Magazine.

   This program has been tested using Turbo ver 3.01A on an IBM PC/AT and
   two PC clones. It has  been run under DOS 3.2 .

   We would be pleased to hear your comments, good or bad, or any applications
   and modifications of the program. Contact us at:

     AI Expert
     CL Publications Inc.
     650 Fifth St.
     Suite 311
     San Francisco, CA 94107

   or on the AI Expert BBS. Our id is BillandBev Thompson ,[76703,4324].
   You can also contact us on BIX, our id is bbt.

   Bill and Bev Thompson    *)

 CONST
  no_of_constraints = 50 ;  (* Maximum number of constraint equations *)
  no_of_hyp = 30 ;          (* Maximum number of nodes *)
  penalty = 1.0E+04 ;       (* Penalty cost = Penalty * constraint value *)
  tol = 1.0E-06 ;           (* Tolerance for deciding if two solutions are close *)
  ctrl_c = ^C ;             (* enough to quit. Increasing gives more accuracy,*)
  bell = ^G ;               (* but slow program down. *)
  back_space = ^H ;
  tab = ^I ;
  return = ^M ;
  eof_mark = ^Z ;
  color_base = $B800 ;   (* Location of PC color screen memory map *)
  mono_base = $B000 ;    (* Location of PC mono screen memory map *)
  esc = #27 ;      (* These rest of these constants could have been defined in *)
  F10 = #68 ;      (* process_file, but we put them here for convenience *)
  left_arrow = #75 ;
  right_arrow = #77 ;
  up_arrow = #72 ;
  down_arrow = #80 ;
  PgUp = #73 ;
  PgDn = #81 ;
  del = #83 ;
  mark_char = '\' ;
  enter = #13 ;
  check_mark = #4 ;
  def_window_size_x = 65 ;
  def_window_size_y = 12 ;
  def_fore_color = white ;
  def_back_color = red ;


 TYPE
  counter = 0 .. maxint ;
  string15 = string[15] ;
  string80 = string[80] ;
  string255 = string[255] ;
  text_file = text[2048] ;
  char_set = SET OF char ;
  constraint_vector = ARRAY [1 .. no_of_constraints] OF real ;
  constraint_matrix = ARRAY [1 .. no_of_constraints,1 .. no_of_hyp] OF real ;
  name_vector = ARRAY [1 .. no_of_hyp] OF string80 ;
  hyp_vector = ARRAY [1 .. no_of_hyp] OF real ;
  char_ptr = ^char ;
  col_pos = 1 .. 80 ;      (* The PC screen is 80 by 25 *)
  row_pos = 1 .. 25 ;
  color = 0 .. 31 ;
  window_pos = RECORD           (* cursor location on screen *)
                x : col_pos ;
                y : row_pos ;
               END ;
  window_ptr = ^window_desc ;
  window_desc = RECORD                        (* Basic window description *)
                 next_window : window_ptr ;   (* windows are linked lists of *)
                 prev_window : window_ptr ;   (* these descriptors *)
                 abs_org     : window_pos ;   (* origin relative to upper left *)
                 window_size : window_pos ;   (* rows and columns in window *)
                 cursor_pos  : window_pos ;   (* saves current cursor location *)
                 has_frame   : boolean ;      (* size and org do not include frame *)
                 fore_color  : color ;
                 back_color  : color ;
                 scrn_area   : char_ptr ;      (* pointer to actual window data *)
                END ;
  string_ptr = ^string255 ;   (* we don't actually allocate space for 255 chars *)
  line_ptr = ^line_desc ;
  line_desc = RECORD                 (* text is stored as a linked list *)
               next_line : line_ptr ;
               prev_line : line_ptr ;
               txt       : string_ptr ; (* points to actual text data *)
              END ;
  mark_ptr = ^mark_desc ;
  mark_desc = RECORD                   (* marked text is also a linked list *)
               next_mark : mark_ptr ;
               prev_mark : mark_ptr ;
               mark_pos  : window_pos ;  (* location of start of mark in window *)
               mark_text : string_ptr ;  (* actual marked text *)
              END ;
  dos_rec = RECORD                       (* used for low-level functions *)
             CASE boolean OF
              true  : (ax,bx,cx,dx,bp,si,di,ds,es,flags : integer) ;
              false : (al,ah,bl,bh,cl,ch,dl,dh          : byte) ;
             END ;
  monitor_type = (color_monitor,mono_monitor,ega_monitor) ;


 VAR
  opt_value : real ;
  c : constraint_matrix ;
  b : constraint_vector ;
  hyp : name_vector ;
  probability : hyp_vector ;
  default : constraint_vector ;
  line,saved_line : string255 ;
  delim_set : char_set ;
  rule_count,hyp_count : counter ;
  fixed : ARRAY [1 .. no_of_hyp] OF boolean ;
  window_list,text_window,message_window,last_window : window_ptr ;
  screen_base : char_ptr ;
  monitor_kind : monitor_type ;
  main_file : text_file ;
  button_fore,button_back : color ;

(* Important variables:
   opt_value   - The value of the objective function.
   c           - constraint matrix
   b           - right hand side of constraints
                 constraints are  Sum j c[i,j] * probability[j] < b[i]
                 constraints 0 <= probability[i] <= 1 are not included here,
                 they are handled intrinsically
   hyp         - the names of the hypothesis
   probability - proabilities associated with each node
   defaults    - default values for hypothesis probabilites
   fixed       - array of booleans indicating that user has decided to fix a
                 value for probability i
   window_list - points to a linked list of window descriptors,
                 the top window is the currently active window.
                 To write in a window, bring it to the front of the list.
   last_window - points to end of window list
   text_window - the big window, that text initially appears in
   message_window - 2 line area at the bottom of the screen, available keys,
                    commands etc. appear here
   screen_base - points to actual memory location of screen, either
                 mono_base or color_base
   main_file - the original text file, the one we start the program with
   button_fore,
   button_back - the button is the large cursor which moves from mark to mark
                 on a color screen it is yellow on black, on a mono screen
                 the text is underlined. *)


 (* Note - In most cases this program uses the Turbo standard string
           functions. You can probably get better performance by turning
           off range checking and accessing the strings directly, but
           we didn't want to make this program even less portable than it
           already is. *)


(* \\\\\\\\\\\\\ Basic Utility Routines  \\\\\\\\\\\\\\\\\\\\\\ *)


 FUNCTION min(x,y : integer) : integer ;
  BEGIN
   IF x <= y
    THEN min := x
    ELSE min := y ;
  END ; (* min *)


 FUNCTION max(x,y : integer) : integer ;
  BEGIN
   IF x >= y
    THEN max := x
    ELSE max := y ;
  END ; (* max *)


 PROCEDURE makestr(VAR s : string255 ; len : byte) ;
  (* Fixes string "s" to length "len" - pads with blanks if necessary. *)
  VAR
   old_length : byte ;
  BEGIN
   old_length := length(s) ;
   (*$R- *)
   s[0] := chr(len) ;
   (*$R+ *)
   IF old_length < len
    THEN fillchar(s[old_length+1],len - old_length,' ') ;
  END ; (* makestr *)


 FUNCTION open(VAR f : text_file ; f_name : string80) : boolean ;
  (* open a file - returns true if the file exists and was opened properly
     f      - file pointer
     f_name - external name of the file *)
  BEGIN
   assign(f,f_name) ;
   (*$I- *)
   reset(f) ;
   (*$I+ *)
   open := (ioresult = 0) ;
  END ; (* open *)


 FUNCTION is_console(VAR f : text_file) : boolean ;
  (* return true if f is open on the system console
     for details of fibs and fib_ptrs see the Turbo Pascal ver 3.0 reference
     manual chapter 20. This should work under CP/M-86 or 80, but we haven't
     tried it. *)
  TYPE
   fib = ARRAY [0 .. 75] OF byte ;
  VAR
   fib_ptr : ^fib ;
   dev_type : byte ;
  BEGIN
   fib_ptr := addr(f) ;
   dev_type := fib_ptr^[2] AND $07 ;
   is_console := (dev_type = 1) OR (dev_type = 2) ;
  END ; (* is_console *)


 PROCEDURE strip_leading_blanks(VAR s : string255) ;
  BEGIN
   IF length(s) > 0
    THEN
     IF (s[1] = ' ') OR (s[1] = tab)
      THEN
       BEGIN
        delete(s,1,1) ;
        strip_leading_blanks(s) ;
       END ;
  END ; (* strip_leading_blanks *)


 PROCEDURE strip_trailing_blanks(VAR s : string255) ;
  BEGIN
   IF length(s) > 0
    THEN
     IF (s[length(s)] = ' ') OR (s[length(s)] = tab)
      THEN
       BEGIN
        delete(s,length(s),1) ;
        strip_trailing_blanks(s) ;
       END ;
  END ; (* strip_trailing_blanks *)


 FUNCTION toupper(s : string255) : string255 ;
  (* returns s converted to upper case *)
  VAR
   i : byte ;
  BEGIN
   IF length(s) > 0
    THEN
     FOR i := 1 TO length(s) DO
      s[i] := upcase(s[i]) ;
   toupper := s ;
  END ; (* toupper *)


 FUNCTION toreal(s : string255) : real ;
  (* Converts "s" to a real - ignores non-numeric characters. *)
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   IF s = ''
    THEN code := -1
   ELSE IF length(s) = 1
    THEN
     IF s[1] IN ['0' .. '9']
      THEN val(s,num,code)
      ELSE code := -1
   ELSE val(s,num,code) ;
   IF code = 0
    THEN toreal := num
    ELSE toreal := 0 ;
  END ; (* toreal *)


 FUNCTION tointeger(s : string255) : integer ;
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   val(s,num,code) ;
   IF code = 0
    THEN
     IF (num < -32768.0) OR (num > 32767.0)
      THEN tointeger := 0
      ELSE tointeger := trunc(num)
    ELSE tointeger := 0 ;
  END ; (* tointeger *)


 FUNCTION is_number(s : string255) : boolean ;
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   IF s = ''
    THEN code := -1
   ELSE IF length(s) = 1
    THEN
     IF S[1] IN ['0' ..'9']
      THEN code := 0
      ELSE code := -1
   ELSE val(s,num,code) ;
   is_number := (code = 0) ;
  END ; (* is_number *)


 FUNCTION to_string(r : real ; width : byte) : string80 ;
  VAR
   s : string80 ;
  BEGIN
   str(r : 3 : width,s) ;
   to_string := s ;
  END ; (* to_string *)


 FUNCTION sign(r : real) : integer ;
  BEGIN
   IF r >= 0.0
    THEN sign := 1
    ELSE sign := - 1 ;
  END ; (* sign *)

{$I window.inc }

 PROCEDURE read_kbd(VAR s : string255) ;
  (* Read a line from the keyboard. The number of unmatched parentheses are
     printed along with the prompt *)

  PROCEDURE read_line ;
   VAR
    done : boolean ;
    ch : char ;

   PROCEDURE echo ;
    BEGIN
     s := concat(s,ch) ;
     write(ch) ;
    END ; (* echo *)

   PROCEDURE delete_back ;
    BEGIN
     IF length(s) > 0
      THEN
       BEGIN
        delete(s,length(s),1) ;
        gotoxy(pred(wherex),wherey) ;
        write(' ') ;
        gotoxy(pred(wherex),wherey) ;
       END ;
    END ; (* delete_back *)

   BEGIN
    s := '' ;
    done := false ;
    WHILE NOT done DO
     BEGIN
      read(kbd,ch) ;
      CASE ch OF
       ' ' .. '~' : echo ;
       return     : BEGIN
                     done := true ;
                     writeln ;
                    END ;
       back_space : delete_back ;
       ctrl_c     : halt(4) ;
       eof_mark   : BEGIN
                     done := true ;
                     s := concat(s,eof_mark) ;
                    END ;
      END ;
     END ;
   END ; (* read_line *)

  BEGIN
   read_line ;
  END ; (* read_kbd *)


 PROCEDURE read_from_file(VAR f : text_file) ;
  (* Read a line from file f and store it in the global variable line.
     It ignores blank lines and when the end of file is reached an
     eof_mark is returned. *)

  PROCEDURE read_a_line ;
   BEGIN
    (*$I- *)
    readln(f,line) ;
    (*$I+ *)
    IF ioresult <> 0
     THEN line := eof_mark
    ELSE IF eof(f)
     THEN line := concat(line,eof_mark) ;
   END ; (* read_a_line *)

  BEGIN
   line := '' ;
   IF is_console(f)
    THEN read_kbd(line)
    ELSE read_a_line ;
   saved_line := line ;
   line := toupper(line) ;
  END ; (* read_from_file *)


 PROCEDURE get_token(VAR t_line : string255 ; VAR token : string255) ;
  (* Get a token from t_line. A token is a string of text surrounded by
     blanks or a delimeter. Comments begin with ; and extend to the
     end of the line *)

  PROCEDURE get_word ;
   VAR
    done : boolean ;
    cn : integer ;
    len : byte ;
   BEGIN
    cn := 1 ;
    len := length(t_line) ;
    done := false ;
    WHILE NOT done DO
     IF cn > len
      THEN done := true
     ELSE IF t_line[cn] IN delim_set
      THEN done := true
     ELSE cn := cn + 1 ;
    token := copy(t_line,1,cn-1) ;
    delete(t_line,1,cn-1) ;
   END ; (* get_word *)

  PROCEDURE comment ;
   BEGIN
    t_line := '' ;
    get_token(t_line,token) ;
   END ; (* comment *)

  PROCEDURE get_number ;

   PROCEDURE get_digits ;
    BEGIN
     WHILE is_number(copy(t_line,1,1)) DO
      BEGIN
       token := concat(token,t_line[1]) ;
       delete(t_line,1,1) ;
      END ;
    END ; (* get_digits *)

   PROCEDURE get_exponent ;
    BEGIN
     delete(t_line,1,1) ;
     IF length(t_line) > 0
      THEN
       BEGIN
        IF t_line[1] IN ['+','-']
         THEN
          BEGIN
           token := concat(token,'E',t_line[1]) ;
           delete(t_line,1,1) ;
          END
         ELSE token := concat(token,'E+') ;
        get_digits ;
       END
      ELSE token := concat(token,'E+00') ;
    END ; (* get_exponent *)

   BEGIN
    get_digits ;
    IF length(t_line) > 0
     THEN
      IF t_line[1] = '.'
       THEN
        IF is_number(copy(t_line,2,1))
         THEN
          BEGIN
           token := concat(token,t_line[1]) ;
           delete(t_line,1,1) ;
           get_digits ;
           IF toupper(copy(t_line,1,1)) = 'E'
            THEN get_exponent ;
          END ;
   END ; (* get_number *)

  PROCEDURE check_number ;
   VAR
    sgn : char ;
   BEGIN
    sgn := t_line[1] ;
    delete(t_line,1,1) ;
    IF length(t_line) > 0
     THEN
      IF t_line[1] IN ['0' .. '9']
       THEN
        BEGIN
         get_number ;
         token := concat(sgn,token) ;
        END
       ELSE token := sgn
     ELSE token := sgn ;
   END ; (* check_number *)

  BEGIN
   strip_leading_blanks(t_line) ;
   token := '' ;
   IF length(t_line) > 0
    THEN
     BEGIN
      IF t_line[1] = ';'
       THEN comment
      ELSE IF t_line[1] IN delim_set
       THEN
        BEGIN
         token := t_line[1] ;
         delete(t_line,1,1) ;
        END
      ELSE IF t_line[1] IN ['+','-']
       THEN check_number
      ELSE IF t_line[1] IN ['0' .. '9']
       THEN get_number
      ELSE get_word ;
     END ;
  END ; (* get_token *)


 PROCEDURE scan(VAR f : text_file ; VAR token : string255) ;
  (* Scan repeatedly calls get_token to retreive tokens. When the
     end of a line has been reached, read_from_file is called to
     get a new line. *)
  BEGIN
   IF length(line) > 0
    THEN
     BEGIN
      get_token(line,token) ;
      strip_trailing_blanks(token) ;
      IF token = ''
       THEN scan(f,token) ;
     END
    ELSE
     BEGIN
      read_from_file(f) ;
      scan(f,token) ;
     END ;
  END ; (* scan *)


 PROCEDURE initialize ;
  VAR
   i : 1 .. no_of_hyp ;
   j : 1 .. no_of_constraints ;
  BEGIN
   init_windows ;
   FOR i := 1 TO no_of_hyp DO
    BEGIN
     fixed[i] := false ;
     probability[i] := 0.0 ;
     default[i] := 0.0 ;
     b[i] := 0.0 ;
     hyp[i] := '' ;
     FOR j := 1 TO no_of_constraints DO
      c[j,i] := 0.0 ;
    END ;
   delim_set := ['.',',','='] ;
   rule_count := 0 ;
   hyp_count := 0 ;
   line := '' ;
  END ; (* initialize *)


 FUNCTION matrix_built : boolean ;
  (* Read the text file and build the contraint matrix and default vectors *)
  VAR
   token : string255 ;
   kb_file : text_file ;
   file_ok : boolean ;

  PROCEDURE mat_error(error_msg : string80) ;
   (* Inform user about some kind of error in the text file
      Skip ahead to next '.' before continuing processing. *)
   VAR
    error_window : window_Ptr ;

   PROCEDURE runout ;
    BEGIN
     WHILE (token <> '.') AND (token <> eof_mark) DO
      scan(kb_file,token) ;
    END ; (* runout *)

   BEGIN
    error_window := open_window(10,10,60,5,true,'Error',white,red,white) ;
    window_writeln(error_msg) ;
    window_writeln('') ;
    window_writeln(saved_line) ;
    gotoxy(length(saved_line) - length(line) + 1,wherey) ;
    window_write('^') ;
    wait ;
    runout ;
    file_ok := false ;
    close_window(error_window) ;
   END ; (* mat_error *)

  FUNCTION got_file : boolean ;
   (* Called from main program block to get the file name typed after *)
   (* the program at the DOS prompt *)
   (* If the file cannot be found, ask for it. *)
   VAR
    f_name : string80 ;
   BEGIN
    f_name := paramstr(1) ;
    IF f_name = ''
     THEN
      BEGIN
       window_write('Knowledge Base File: ') ;
       readln(f_name) ;
      END ;
    IF open(kb_file,f_name)
     THEN got_file := true
     ELSE
      BEGIN
       error(concat('Unable to open ',f_name)) ;
       got_file := false ;
      END ;
   END ; (* got_file *)

  FUNCTION to_prob(s : string80) : real ;
   (* convert a text string to a real number *)
   VAR
    p : real ;
   BEGIN
    p := toreal(s) ;
    IF p < 0.0
     THEN mat_error('Probabilities must be >= 0.0')
    ELSE IF p > 1.0
     THEN mat_error('Probabilities must be <= 1.0') ;
    to_prob := p ;
   END ; (* to_prob *)

  PROCEDURE read_rules ;
   (* read the rules and add to constraint matrix *)
   VAR
    prob : real ;
    cond_count : counter ;

   PROCEDURE insert_in_matrix(s : string80) ;
    (* Add new element to c matrix, also add it to list of names if it's
       not there already *)
    VAR
     i : counter ;
     found : boolean ;
    BEGIN
     i := 1 ;
     found := false ;
     WHILE (i <= hyp_count) AND (NOT found) DO
      IF s = hyp[i]
       THEN found := true
       ELSE i := succ(i) ;
     IF NOT found
      THEN
       BEGIN
        hyp_count := succ(hyp_count) ;
        IF hyp_count <= no_of_hyp
         THEN
          BEGIN
           hyp[hyp_count] := s ;
           c[rule_count,hyp_count] := prob ;
          END
         ELSE mat_error('Too many hypothesis.') ;
       END
      ELSE c[rule_count,i] := prob ;
    END ; (* insert_in_matrix *)

   PROCEDURE read_cond_list ;
    BEGIN
     insert_in_matrix(token) ;
     cond_count := succ(cond_count) ;
     scan(kb_file,token) ;
     IF token = ','
      THEN
       BEGIN
        scan(kb_file,token) ;
        read_cond_list ;
       END ;
    END ; (* read_cond_list *)

   PROCEDURE read_exclusion ;
    BEGIN
     prob := 1.0 ;
     rule_count := succ(rule_count) ;
     IF rule_count <= no_of_constraints
      THEN
       BEGIN
        scan(kb_file,token) ;
        cond_count := 0 ;
        read_cond_list ;
        b[rule_count] := prob ;
        IF token = '.'
         THEN scan(kb_file,token)
          ELSE mat_error('Missing ''.''') ;
       END
      ELSE mat_error('Too many rules.') ;
    END ; (* read_exclusion *)

   PROCEDURE read_a_rule ;
    BEGIN
     prob := -1.0 ;
     rule_count := succ(rule_count) ;
     IF rule_count <= no_of_constraints
      THEN
       BEGIN
        insert_in_matrix(token) ;
        scan(kb_file,token) ;
        IF token = ','
         THEN
          BEGIN
           scan(kb_file,token) ;
           prob := to_prob(token) ;
           scan(kb_file,token) ;
          END ;
        IF token = '='
         THEN
          BEGIN
           cond_count := 0 ;
           scan(kb_file,token) ;
           read_cond_list ;
           b[rule_count] := prob * (cond_count - 1) ;
          END
         ELSE mat_error('Missing ''='' sign.') ;
        IF token = '.'
         THEN scan(kb_file,token)
         ELSE mat_error('Missing ''.''') ;
       END
      ELSE mat_error('Too many rules.') ;
    END ; (* read_a_rule *)

   BEGIN
    scan(kb_file,token) ;
    WHILE (copy(token,1,1) <> '%') AND (token <> eof_mark) AND file_ok DO
     IF token = '='
      THEN read_exclusion
      ELSE read_a_rule ;
   END ; (* read_rules *)

  PROCEDURE read_aux_info ;
   (* Read the default values and add to default vector *)

   PROCEDURE read_defaults ;
    VAR
     s : string80 ;
     prob : real ;

    PROCEDURE insert_default ;
     VAR
      i : counter ;
      found : boolean ;
     BEGIN
      i := 1 ;
      found := false ;
      WHILE (i <= hyp_count) AND (NOT found) DO
       IF s = hyp[i]
        THEN found := true
        ELSE i := succ(i) ;
      IF NOT found
       THEN
        BEGIN
         hyp_count := succ(hyp_count) ;
         IF hyp_count <= no_of_hyp
          THEN
           BEGIN
            hyp[hyp_count] := s ;
            default[hyp_count] := prob ;
           END
          ELSE mat_error('Too many hypothesis.') ;
        END
       ELSE default[i] := prob ;
     END ; (* insert_default *)

    BEGIN
     scan(kb_file,token) ;
     WHILE (token <> eof_mark) AND (file_ok) DO
      BEGIN
       s := token ;
       scan(kb_file,token) ;
       IF token = '='
        THEN
         BEGIN
          scan(kb_file,token) ;
          prob := to_prob(token) ;
          insert_default ;
          scan(kb_file,token) ;
          IF token = '.'
           THEN scan(kb_file,token)
           ELSE mat_error('Missing ''.''') ;
         END
        ELSE mat_error('Missing ''='' .') ;
      END ;
    END ; (* read_defaults *)

   BEGIN
    IF token = '%DEFAULTS'
     THEN read_defaults
     ELSE mat_error('Unknown identifier.') ;
   END ; (* read_aux_info *)

  BEGIN
   IF got_file
    THEN
     BEGIN
      file_ok := true ;
      read_rules ;
      IF file_ok AND (token <> eof_mark)
       THEN read_aux_info ;
      matrix_built := file_ok ;
      close(kb_file) ;
     END
    ELSE matrix_built := false ;
  END ; (* matrix_built *)


 PROCEDURE process_probabilities ;
  (* The user interface routine *)
  (* Much of this routine was lifted from HYPE.PAS *)
  (* builds a linked list of names and probability values them one page *)
  (* at a time in text_window *)
  (* User can move through list of probabilities and change them. It will *)
  (* complain if the user tries to enter aprobability <0 or >1 *)
  (* first_line - start of list of lines *)
  (* last_line - last line *)
  (* mark_win_org,mark_win_size,mark_fore,mark_back - window parameters *)
  (*              for threaded text display *)
  VAR
   first_line,last_line : line_ptr ;
   mark_win_org,mark_win_size : window_pos ;
   mark_fore,mark_back : color ;

  PROCEDURE release_list(list : line_ptr) ;
   (* free memory used by line descriptors and text *)
   VAR
    p : line_ptr ;
   BEGIN
    WHILE list <> NIL DO
     BEGIN
      p := list ;
      list := list^.next_line ;
      freemem(p^.txt,length(p^.txt^) + 1) ;
      freemem(p,sizeof(line_desc)) ;
     END ;
   END ; (* release_list *)

  PROCEDURE build_prob_list(VAR first,last : line_ptr) ;
   (* Build the display list from the probability array *)
   (* probabilities are displayed as marked text *)
   (* first,last point to the start and end of the line list *)
   (* We only allocate enough storage for the actual characters in the line, *)
   (* not all 255 characters *)
   VAR
    s : string80 ;
    p : line_ptr ;
    i : 1 .. no_of_hyp ;

   PROCEDURE insert_line(lne : line_ptr) ;
    (* insert a line at the end of the line list *)
    BEGIN
     lne^.next_line := NIL ;
     lne^.prev_line := last ;
     IF last = NIL
      THEN first := lne
      ELSE last^.next_line := lne ;
     last := lne ;
    END ; (* insert_line *)

   PROCEDURE add_line(ss : string255) ;
    VAR
     p : line_ptr ;
    BEGIN
     getmem(p,sizeof(line_desc)) ;
     getmem(p^.txt,length(ss) + 1) ;
     p^.txt^ := ss ;
     insert_line(p) ;
    END ; (* add_line *)

   PROCEDURE check_feasibility ;
    CONST
     f_tol = 1.0E-04 ;
    VAR
     j : 1 .. no_of_constraints ;
     k : 1 .. no_of_hyp ;
     sum : real ;
     violations : string80 ;
    BEGIN
     violations := '' ;
     FOR j := 1 TO rule_count DO
      BEGIN
       sum := 0.0 ;
       FOR k := 1 TO hyp_count DO
        sum := sum  + c[j,k] * probability[k] ;
       IF b[j] - sum < (- f_tol)
        THEN violations := concat(violations,' ',to_string(j,0)) ;
      END ;
    IF violations <> ''
     THEN
      BEGIN
       add_line('The solution is infeasible.') ;
       add_line('') ;
       add_line('The following rules are violated:') ;
       add_line(violations) ;
       add_line('') ;
      END ;
   END ; (* check_feasibility *)

   BEGIN
    first := NIL ;
    last := NIL ;
    add_line('') ;
    add_line('Hypothesis Values  (scaled to 100).') ;
    add_line('') ;
    add_line(concat('Objective value = ',to_string(opt_value,6))) ;
    add_line('') ;
    check_feasibility ;
    FOR i := 1 TO hyp_count DO
     BEGIN
      s := hyp[i] ;
      makestr(s,60) ;
      s := concat(s,'   \',to_string(probability[i] * 100.0,0),'\') ;
      IF fixed[i]
       THEN s := concat(check_mark,' ',s)
       ELSE s := concat('  ',s) ;
      add_line(s) ;
     END ;
   END ; (* build_prob_list *)

  PROCEDURE display_list(first,last : line_ptr ; disp_window : window_ptr) ;
   (* display the list pointed to by first in disp_window *)
   (* read keyboard until F10 or Esc is pressed *)
   (* up,down arrows move among marked text, Enter fixes probability at *)
   (* displayed value *)
   (* Text is displayed one page at a time - PgUp and PgDn page *)
   (* mark_list is a linked list of highlighted text on the current page *)
   (*           of the disp_window *)
   (* mark is the current mark, i.e. the one with the button color *)
   (* top_of_page points to first line on the page *)
   VAR
    done : boolean ;
    top_of_page : line_ptr ;
    mark,mark_list,last_mark : mark_ptr ;

   PROCEDURE display_message ;
    (* display available keys at bottom of screen *)
    CONST
     disp_up_arrow = #24 ;
     disp_down_arrow = #25 ;
    BEGIN
     use_window(message_window) ;
     clrscr ;
     window_write(concat(disp_up_arrow,' ',disp_down_arrow,'   Select')) ;
     gotoxy(1,2) ;
     window_write('<Enter>  Fix Value') ;
     gotoxy(30,1) ;
     window_write('<Esc>   Continue') ;
     gotoxy(60,1) ;
     window_write('PgUp PgDn   Page') ;
     gotoxy(60,2) ;
     window_write('F10         Quit') ;
     use_window(disp_window) ;
    END ; (* display_message *)

   PROCEDURE move_to_mark(m_ptr : mark_ptr) ;
    (* move to the highlighted region of screen pointed to by m_ptr *)
    (* redisplay text in button colors so that user can see where we are *)
    VAR
     p : mark_ptr ;

    PROCEDURE remove_old_mark ;
     (* return previous marked text to reverse video *)
     BEGIN
      gotoxy(mark^.mark_pos.x,mark^.mark_pos.y) ;
      window_reverse ;
      window_write(mark^.mark_text^) ;
      window_normal ;
     END ; (* remove_old_mark *)

    BEGIN
     IF m_ptr <> NIL
      THEN
       BEGIN
        IF mark <> NIL
         THEN remove_old_mark ;
        p := mark_list ;
        WHILE (p <> NIL) AND (p <> m_ptr) DO
         p := p^.next_mark ;
        IF p <> NIL
         THEN
          BEGIN
           mark := p ;
           gotoxy(mark^.mark_pos.x,mark^.mark_pos.y) ;
           textcolor(button_fore) ;
           textbackground(button_back) ;
           window_write(mark^.mark_text^) ;
           window_normal ;
           gotoxy(mark^.mark_pos.x,mark^.mark_pos.y) ;
          END ;
       END ;
    END ; (* move_to_mark *)

   PROCEDURE display_page ;
    (* display a page of text in disp_window *)
    (* marked text is displayed inreverse video *)
    (* move mark to first item on mark list *)
    VAR
     line_cnt : counter ;
     p : line_ptr ;

    PROCEDURE release_marks ;
     (* release the old mark list - the mark list is rebuilt each *)
     (* time a page is displayed *)
     VAR
      m_ptr : mark_ptr ;
     BEGIN
      WHILE mark_list <> NIL DO
       BEGIN
        m_ptr := mark_list ;
        mark_list := mark_list^.next_mark ;
        freemem(m_ptr^.mark_text,length(m_ptr^.mark_text^) + 1) ;
        freemem(m_ptr,sizeof(mark_desc)) ;
       END ;
      mark := NIL ;
      last_mark := NIL ;
     END ; (* release_marks *)

    PROCEDURE write_the_line(s : string255) ;
     (* write the line on the screen *)
     (* if text is marked add it to list and display inreverse video *)
     VAR
      mark_loc : byte ;

     PROCEDURE add_mark ;
      (* add this text to list and save its co-ordinates *)
      VAR
       m_ptr : mark_ptr ;
       ps : integer ;
      BEGIN
       getmem(m_ptr,sizeof(mark_desc)) ;
       m_ptr^.mark_pos.x := wherex ;
       m_ptr^.mark_pos.y := wherey ;
       delete(s,1,1) ;
       ps := pred(pos(mark_char,s)) ;
       IF ps < 0
        THEN ps := length(s) ;
       getmem(m_ptr^.mark_text,ps + 1) ;
       m_ptr^.mark_text^ := copy(s,1,ps) ;
       window_reverse ;
       window_write(m_ptr^.mark_text^) ;
       window_normal ;
       delete(s,1,succ(ps)) ;
       m_ptr^.next_mark := NIL ;
       m_ptr^.prev_mark := last_mark ;
       IF last_mark = NIL
        THEN mark_list := m_ptr
        ELSE last_mark^.next_mark := m_ptr ;
       last_mark := m_ptr ;
      END ; (* add_mark *)

     BEGIN
      IF s <> ''
       THEN
        BEGIN
         mark_loc := pos(mark_char,s) ;
         IF mark_loc > 0
          THEN
           BEGIN
            window_write(copy(s,1,pred(mark_loc))) ;
            delete(s,1,pred(mark_loc)) ;
            add_mark ;
            write_the_line(s) ;
           END
         ELSE window_write(s) ;
        END ;
     END ; (* write_the_line *)

    BEGIN
     release_marks ;
     clrscr ;
     p := top_of_page ;
     line_cnt := 1 ;
     WHILE (p <> NIL) AND (line_cnt <= disp_window^.window_size.y) DO
      BEGIN
       gotoxy(1,line_cnt) ;
       IF copy(p^.txt^,1,2) <> '..'
        THEN
         BEGIN
          write_the_line(p^.txt^) ;
          line_cnt := succ(line_cnt) ;
         END ;
       p := p^.next_line ;
      END ;
     move_to_mark(mark_list)
    END ; (* display_page *)

   PROCEDURE handle_keys ;
    (* read the keyboard - ignore everything but keys displayed on bottom *)
    (* of screen *)
    VAR
     ch : char ;

    PROCEDURE exit_prog ;
     (* F10 - pressed erase screen and quit *)
     BEGIN
      finish_up ;
      halt(0) ;
     END ; (* exit_prog *)

    PROCEDURE page_forward ;
     (* display previous page *)
     (* count backwards until we get to it *)
     VAR
      p : line_ptr ;
      line_cnt : counter ;
     BEGIN
      p := top_of_page ;
      line_cnt := 1 ;
      WHILE (p <> NIL) AND (line_cnt < disp_window^.window_size.y) DO
       BEGIN
        p := p^.next_line ;
        line_cnt := succ(line_cnt) ;
       END ;
      IF p <> NIL
       THEN
        IF p^.next_line <> NIL
         THEN
          BEGIN
           top_of_page := p^.next_line ;
           display_page ;
          END ;
     END ; (* page_forward *)

    PROCEDURE page_back ;
     (* display next page *)
     (* count forwards until we get to it *)
     VAR
      p : line_ptr ;
      line_cnt : counter ;
     BEGIN
      p := top_of_page ;
      line_cnt := disp_window^.window_size.y ;
      WHILE (p <> NIL) AND (line_cnt >= 1) DO
       BEGIN
        p := p^.prev_line ;
        line_cnt := pred(line_cnt) ;
       END ;
      IF p <> NIL
       THEN
        BEGIN
         top_of_page := p ;
         display_page ;
        END ;
     END ; (* page_back *)

    PROCEDURE move_to_next_mark ;
     (* move to next mark on screen, if at end go back to first *)
     BEGIN
      IF mark_list <> NIL
       THEN
        BEGIN
         IF mark^.next_mark <> NIL
          THEN move_to_mark(mark^.next_mark)
          ELSE move_to_mark(mark_list) ;
        END ;
     END ; (* move_to_next_mark *)

    PROCEDURE move_to_prev_mark ;
     (* move to prev mark on screen, if at first go to end *)
     BEGIN
      IF mark_list <> NIL
       THEN
        BEGIN
         IF mark^.prev_mark <> NIL
          THEN move_to_mark(mark^.prev_mark)
          ELSE move_to_mark(last_mark) ;
        END ;
     END ; (* move_to_prev_mark *)

    PROCEDURE move_right ;
     (* move one character to the right in current mark *)
     BEGIN
      gotoxy(succ(wherex),wherey) ;
      IF wherex > mark^.mark_pos.x + length(mark^.mark_text^) - 1
       THEN move_to_mark(mark^.next_mark) ;
     END ; (* move_right *)

    PROCEDURE move_left ;
     (* move one character to the left in current mark *)
     BEGIN
      gotoxy(pred(wherex),wherey) ;
      IF wherex < mark^.mark_pos.x
      THEN move_to_mark(mark^.prev_mark) ;
     END ; (* move_right *)

    FUNCTION line_location : line_ptr ;
     (* find out what line in linked list cursor is on *)
     VAR
      q : line_ptr ;
      cnt,y_pos : row_pos ;
     BEGIN
      q := top_of_page ;
      y_pos := wherey ;
      cnt := 1 ;
      WHILE (cnt <> y_pos) DO
       BEGIN
        q := q^.next_line ;
        cnt := succ(cnt) ;
       END ;
      line_location := q ;
     END ; (* line_location *)

    PROCEDURE fix_value ;
     (* <Enter> pressed, set the probability to the curren value on screen
        Sets element in fixed array to true *)
     VAR
      p : line_ptr ;
      hyp_str : string80 ;
      i : counter ;
      found : boolean ;

     PROCEDURE fix_it ;
      BEGIN
       gotoxy(1,wherey) ;
       fixed[i] := NOT fixed[i] ;
       IF fixed[i]
        THEN
         IF (probability[i] > 1.00) OR (probability[i] < 0.0)
          THEN
           BEGIN
            error('Probabilities must be in the range 0 to 1.0') ;
            fixed[i] := false ;
           END
          ELSE window_write(check_mark)
        ELSE window_write(' ') ;
       move_to_next_mark ;
      END ; (* fix_it *)

     BEGIN
      p := line_location ;
      hyp_str := copy(p^.txt^,3,60) ;
      strip_trailing_blanks(hyp_str) ;
      i := 1 ;
      found := false ;
      WHILE (NOT found) AND (i <= hyp_count) DO
       IF hyp_str = hyp[i]
        THEN found := true
        ELSE i := succ(i) ;
      IF found
       THEN fix_it
       ELSE
        BEGIN
         error(concat('Could not find ',hyp_str)) ;
         move_to_mark(mark) ;
        END ;
     END ; (* fix_value *)

    PROCEDURE change_prob(mark_value : string15) ;
     (* Change the probability to the number on the screen *)
     VAR
      p : line_ptr ;
      hyp_str : string80 ;
      i : counter ;
      found : boolean ;
     BEGIN
      p := line_location ;
      hyp_str := copy(p^.txt^,3,60) ;
      strip_trailing_blanks(hyp_str) ;
      i := 1 ;
      found := false ;
      WHILE (NOT found) AND (i <= hyp_count) DO
       IF hyp_str = hyp[i]
        THEN found := true
        ELSE i := succ(i) ;
      IF found
       THEN probability[i] := (tointeger(mark_value) / 100.0)
       ELSE
        BEGIN
         error(concat('Could not find ',hyp_str)) ;
         move_to_mark(mark) ;
        END ;
     END ; (* change_prob *)

    PROCEDURE echo ;
     (* echo the number just typed *)
     VAR
      mark_value : string15 ;
      ch_pos : col_pos ;
     BEGIN
      WITH mark^ DO
       BEGIN
        mark_value := mark_text^ ;
        ch_pos := wherex - mark_pos.x + 1 ;
        mark_value[ch_pos] := ch ;
        mark_text^ := mark_value ;
        textcolor(button_fore) ;
        textbackground(button_back) ;
        window_write(ch) ;
        textcolor(disp_window^.fore_color) ;
        textbackground(disp_window^.back_color) ;
        change_prob(mark_value) ;
        IF wherex > mark_pos.x + length(mark_text^) - 1
         THEN move_to_mark(mark) ;
       END ;
     END ; (* echo *)

    PROCEDURE delete_cursor_char ;
     (* dekete the number under the cursor *)
     VAR
      mark_value : string15 ;
      x_pos,ch_pos : col_pos ;
     BEGIN
      WITH mark^ DO
       BEGIN
        mark_value := mark_text^ ;
        ch_pos := wherex - mark_pos.x + 1 ;
        delete(mark_value,ch_pos,1) ;
        makestr(mark_value,length(mark_text^)) ;
        mark_text^ := mark_value ;
        textcolor(button_fore) ;
        textbackground(button_back) ;
        x_pos := wherex ;
        window_write(copy(mark_value,ch_pos,255)) ;
        gotoxy(x_pos,wherey) ;
        textcolor(disp_window^.fore_color) ;
        textbackground(disp_window^.back_color) ;
        change_prob(mark_value) ;
        IF wherex > mark_pos.x + length(mark_text^) - 1
         THEN move_to_mark(mark) ;
       END ;
     END ; (* delete_cursor_char *)

    PROCEDURE delete_back ;
     BEGIN
      IF wherex > mark^.mark_pos.x
       THEN
        BEGIN
         move_left ;
         delete_cursor_char ;
        END ;
     END ; (* delete_back *)

    BEGIN
     read(kbd,ch) ;
     IF ch = enter
      THEN fix_value
     ELSE IF ch IN [' ','0' .. '9']
      THEN echo
     ELSE IF ch = back_space
      THEN delete_back
     ELSE IF ch = esc
      THEN
       IF keypressed
        THEN
         BEGIN
          read(kbd,ch) ;
          CASE ch OF
           left_arrow  : move_left ;
           right_arrow : move_right ;
           down_arrow  : move_to_next_mark ;
           up_arrow    : move_to_prev_mark ;
           PgUp        : page_back ;
           PgDn        : page_forward ;
           del         : delete_cursor_char ;
           F10         : exit_prog ;
          END ;
         END
        ELSE done := true
     ELSE write(bell) ;
    END ; (* handle_keys *)

   BEGIN
    done := false ;
    display_message ;
    mark := NIL ;
    mark_list := NIL ;
    last_mark := NIL ;
    top_of_page := first ;
    display_page ;
    WHILE NOT done DO
     handle_keys ;
   END ; (* display_list *)

  BEGIN
   build_prob_list(first_line,last_line) ;
   display_list(first_line,last_line,text_window) ;
   release_list(first_line) ;
  END ; (* process_probabilities *)


 FUNCTION optimize : real ;
  (* The gradient descent routine. This routine could use some improvement *)
  (* It could be made a lot faster and more accurate, but as usual we waited *)
  (* until the last minute to write it <sigh>. *)
  VAR
   gradient : hyp_vector ; (* the gradien vector *)
   value,prev_value,step : real ;
   dot_count : 0 .. 50 ;
   iter : integer ;
   cx : char ;

  FUNCTION constraint(i : counter ; pr : hyp_vector) : real ;
   (* calculate the value of constraint i, using vector pr *)
   (* returns o if the constraint was not violated *)
   VAR
    j : 1 .. no_of_hyp ;
    sum,constr : real ;
   BEGIN
    sum := 0.0 ;
    FOR j := 1 TO hyp_count DO
     sum := sum + c[i,j] * pr[j] ;
    constr := b[i] - sum ;
    IF constr > 0.0
     THEN constr := 0.0 ;
    constraint := constr ;
   END ; (* constraint *)

  FUNCTION objective(pr : hyp_vector) : real ;
   (* calculate objective function value for probability array pr *)
   (* handles 0 < pr < 1 constraints also *)
   VAR
    diff,constr : real ;
    i : counter ;
   BEGIN
    diff := 0.0 ;
    FOR i := 1 TO hyp_count DO
     diff := diff + sqr(pr[i] - default[i]) ;
    constr := 0.0 ;
    FOR i := 1 TO rule_count DO
     constr := constr + sqr(constraint(i,pr)) ;
    FOR i := 1 TO hyp_count DO
     IF pr[i] < 0.0
      THEN constr := constr + sqr(pr[i])
     ELSE IF pr[i] > 1.0
      THEN constr := constr + sqr(1.0 - pr[i]) ;
    objective := diff + penalty * constr ;
   END ; (* objective *)

  PROCEDURE compute_gradient ;
   (* calculate the gradient, use constraint matrix and intrinsic constraints *)
   VAR
    gr : real ;
    i : 1 .. no_of_hyp ;
    j : 1 .. no_of_constraints ;
    constr : constraint_vector ;
   BEGIN
    FOR j := 1 TO rule_count DO
     constr[j] := constraint(j,probability) ;
    FOR i := 1 TO hyp_count DO
     IF fixed[i]
      THEN gradient[i] := 0.0
      ELSE
       BEGIN
        gr := 0.0 ;
        FOR j := 1 TO rule_count DO
         gr := gr - constr[j] * c[j,i] ;
        IF probability[i] < 0.0
         THEN gr := gr + probability[i]
        ELSE IF probability[i] > 1.0
         THEN gr := gr - (1 - probability[i]) ;
        gradient[i] :=  - 2.0 * ((probability[i] - default[i]) + penalty * gr) ;
       END ;
   END ; (* compute_gradient *)

  PROCEDURE compute_new_probability(VAR new_pr,pr : hyp_vector ; h : real) ;
   (* calculate new_prob from pr *)
   VAR
    k : 1 .. no_of_hyp ;
   BEGIN
    FOR k := 1 TO hyp_count DO
     new_pr[k] := pr[k] + h * gradient[k] ;
   END ; (* compute_new_probability *)

  PROCEDURE search ;
   (* The gradient search routine. *)
   (* Move along gradient untile we find three points *)
   (* so that val0 > val1 < val2 and then perform quadratic *)
   (* interpolation to find the minimum in this direction *)
   CONST
    min_step = 1.0E-05 ;
   VAR
    p0,p1,p2 : hyp_vector ;
    val0,val1,val2,t,denom : real ;

   PROCEDURE get_starting_move ;
    (* If initial step is to big to move to lower value, cut it in half *)
    (* and try again *)
    BEGIN
     compute_new_probability(p1,p0,step) ;
     val1 := objective(p1) ;
     IF val1 > val0
      THEN
       BEGIN
        step := step / 2 ;
        IF step > min_step
         THEN get_starting_move ;
       END ;
    END ; (* get_starting_move *)

   BEGIN
    step := 0.1 ;
    val0 := value ;
    prev_value := value ;
    move(probability,p0,sizeof(probability)) ;
    get_starting_move ;
    compute_new_probability(p2,p1,step) ;
    val2 := objective(p2) ;
    WHILE val2 < val1 DO
     BEGIN
      move(p1,p0,sizeof(p1)) ;
      val0 := val1 ;
      move(p2,p1,sizeof(p2)) ;
      val1 := val2 ;
      compute_new_probability(p2,p1,step) ;
      val2 := objective(p2) ;
     END ;
    denom := val2 - 2 * val1 + val0 ;
    IF abs(denom) < tol
     THEN denom := sign(denom) * tol ;
    t := - (step / 2) * ((val2 - val0) / denom) ;
    compute_new_probability(p0,p1,t) ;
    val0 := objective(p0) ;
    IF val1 < val0
     THEN
      BEGIN
       move(p1,probability,sizeof(p1)) ;
       value := val1 ;
      END
     ELSE
      BEGIN
       move(p0,probability,sizeof(p0)) ;
       value := val0 ;
      END ;
   END ; (* search *)

  BEGIN
   clrscr ;
   use_window(message_window) ;
   clrscr ;
   window_write('Press any key to examine probabilities.') ;
   use_window(text_window) ;
   iter := 0 ;
   prev_value := 0.0 ;
   value := objective(probability) ;
   writeln('                      Objective        Diff from previous value') ;
   WHILE (abs(value - prev_value) > tol) AND (NOT keypressed) DO
    BEGIN
     iter := succ(iter) ;
     write('Iteration ',iter : 6,'  ',value : 14 : 8,'  ',value - prev_value : 14 : 8) ;
     gotoxy(1,2) ;
     compute_gradient ;
     search ;
    END ;
   IF keypressed
    THEN read(kbd,cx) ;
   optimize := value ;
  END ; (* optimize *)


 BEGIN
  initialize ;
  IF matrix_built
   THEN
    BEGIN
     REPEAT
      opt_value := optimize ;
      process_probabilities ;
     UNTIL false ;
    END ;
  finish_up ;
 END.
