{.PW132}
{.IN+}
{.HE VTLISP.PAS                                          Page #}
{$V-,R+,C- }
PROGRAM very_tiny_LISP_2 ;

(* Copyright (c) 1987 - Knowledge Garden Inc.
                        473A Malden Bridge Rd.
                        RD #2
                        Nassau, NY 12123       *)


(* VT-LISP is a simple functional variation of LISP as described in the
   April and May 1987 issues of AI Expert. The object-oriented features
   were described in the September 1987 issue of AI expert.

   This program has been tested using Turbo ver 3.02 on an IBM PC. It has
   been run under both DOS 3.2 and Concurrent 5.0

   We would be pleased to hear your comments, good or bad, or any applications
   and modifications of the program. Contact us at:

     AI Expert
     CL Publications Inc.
     500 Howard St.
     San Francisco, CA 94105

   or on the AI Expert BBS on Compuserv.
   Our id is BillandBev Thompson ,[76703,4324].
   You can also contact us on BIX, our id is bbt.

   Bill and Bev Thompson    *)

 CONST
  ctrl_c = ^C ;
  ctrl_s = ^S ;
  back_space = ^H ;
  tab = ^I ;
  eof_mark = ^Z ;
  quote_char = #39 ;
  left_arrow = #75 ;
  return = ^M ;
  bell = ^G ;

 TYPE
  counter = 0 .. maxint ;
  string132 = string[132] ;
  string80 = string[80] ;
  string255 = string[255] ;
  text_file = text ;
  char_set = SET OF char ;
  node_type = (cons_node,symbol,number,free_node) ;
  s_expr = ^node ;
  node = RECORD
          in_use : boolean ;
          CASE tag : node_type OF
           cons_node : (car_ptr : s_expr ;
                        cdr_ptr : s_expr) ;
           symbol    : (string_data : string80) ;
           number    : (num_data : real) ;
           free_node : (next_free : s_expr ;
                        block_cnt : counter) ;
          END ;

(* node is the basic allocation unit for lists. The fields are used as
   follows:

    in_use     - in_use = false tells the garbage collector that this node
                 is available for re-use.
    tag        - which kind of node this is.
    cons_node  - cons_nodes consist of two pointers. one to the head (first item)
                 the other to the rest of the list. They are the "glue" which
                 holds the list together. The list (A B C) would be stored as
                   -------         --------          --------
                   | .| . |----->  |  .| . |------> |  .| . |---> NIL
                   --|-----         --|------        --|-----
                     |                |                |
                     V                V                V
                     A                B                C

                 The boxes are the cons nodes, the first part of the box
                 holds the car pointer, then second contains the cdr pointer.
    symbol     - holds string values, we don't actually use the entire 80
                 characters in most cases.
    number     - used for storage of numbers. All numbers are implemented as
                 reals. This is inefficient, but relatively easy in Turbo
                 Pascal.
    free_node  - the garbage collector gathers all unused nodes and puts
                 them on a free list. It also compacts the free space into
                 contiguous blocks. next_free points to the next free block.
                 block_cnt contains a count of the number of contiguous 8 byte free
                 blocks which follow this one.



    Note: we allocate a new node for each atom, instead of a pointer to an
          existing string in the heap. This slows down comparisons,
          because you have to compare strings instead of pointers, but
          speeds up allocation. We've tried it both ways and there seems
          to be no effect on small programs, but if you decide to
          expand this program you should take a long hard look at all
          of the allocation routines and improve them. *)


 VAR
  total_free : real ;
  result,fn,free,initial_heap,saved_list,pending,lambda_node,
        letrec_node,quote_node,constant_list : s_expr ;
  g_name_list,g_value_list,obj_list : s_expr ;
  token : string80 ;
  line,saved_line : string255 ;
  delim_set : char_set ;
  paren_level : counter ;
  mark_stack : integer ;

(* Variables - These are the important global variables:
     total_free -  a count of the total amount of free memory on the
                   free list.
     result      - the S-expression returned by eval.
     fn          - S-expression read by get_expression.
     free        - a linked list of free nodes. Memory is allocated from
                   from here if possible before getting memory from
                   the heap. This list is built by the garbage collector.
     inital_heap - a pointer to the bottom of the heap
     saved_list  - a list of all nodes which must absolutely not be
                   reclaimed by the garbage collector.
     obj_list   -  list of objects. Each object is described by an object
                   structure: (name inheritance_list ((selector1 method1) ....))
     pending,
     lambda_node,
     letrec_node,
     quote_node  - utility S-expressions, save allocation space
     token       - the returned by get_token. This really shouldn't
                   be a global. It's just sloppy programming.
     line        - the input buffer for S-expressions
     delim_set   - set of token delimeters
     paren_level - the count of unmatched parentheses, used while reading
                   S-expressions *)



(* ----------------------------------------------------------------------
        Utility Routines
   ---------------------------------------------------------------------- *)


 FUNCTION open(VAR f : text_file ; f_name : string80) : boolean ;
  (* open a file - returns true if the file exists and was opened properly
     f      - file pointer
     f_name - external name of the file *)
  BEGIN
   assign(f,f_name) ;
   (*$I- *)
   reset(f) ;
   (*$I+ *)
   open := (ioresult = 0) ;
  END ; (* open *)


 FUNCTION is_console(VAR f : text_file) : boolean ;
  (* return true if f is open on the system console
     for details of fibs and fib_ptrs see the Turbo Pascal ver 3.0 reference
     manual chapter 20. This should work under CP/M-86 or 80, but we haven't
     tried it. *)
  TYPE
   fib = ARRAY [0 .. 75] OF byte ;
  VAR
   fib_ptr : ^fib ;
   dev_type : byte ;
  BEGIN
   fib_ptr := addr(f) ;
   dev_type := fib_ptr^[2] AND $07 ;
   is_console := (dev_type = 1) OR (dev_type = 2) ;
  END ; (* is_console *)


 PROCEDURE strip_leading_blanks(VAR s : string80) ;
  BEGIN
   IF length(s) > 0
    THEN
     IF (s[1] = ' ') OR (s[1] = tab)
      THEN
       BEGIN
        delete(s,1,1) ;
        strip_leading_blanks(s) ;
       END ;
  END ; (* strip_leading_blanks *)


 PROCEDURE strip_trailing_blanks(VAR s : string80) ;
  BEGIN
   IF length(s) > 0
    THEN
     IF (s[length(s)] = ' ') OR (s[length(s)] = tab)
      THEN
       BEGIN
        delete(s,length(s),1) ;
        strip_trailing_blanks(s) ;
       END ;
  END ; (* strip_trailing_blanks *)



 FUNCTION toupper(s : string80) : string80 ;
  (* returns s converted to upper case *)
  VAR
   i : byte ;
  BEGIN
   IF length(s) > 0
    THEN
     FOR i := 1 TO length(s) DO
      s[i] := upcase(s[i]) ;
   toupper := s ;
  END ; (* toupper *)


 FUNCTION toreal(s : string255) : real ;
  (* Converts "s" to a real - ignores non-numeric characters. *)
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   IF s = ''
    THEN code := -1
   ELSE IF length(s) = 1
    THEN
     IF s[1] IN ['0' .. '9']
      THEN val(s,num,code)
      ELSE code := -1
   ELSE val(s,num,code) ;
   IF code = 0
    THEN toreal := num
    ELSE toreal := 0 ;
  END ; (* toreal *)


 FUNCTION tointeger(s : string80) : integer ;
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   val(s,num,code) ;
   IF code = 0
    THEN
     IF (num < -32768.0) OR (num > 32767.0)
      THEN tointeger := 0
      ELSE tointeger := trunc(num)
    ELSE tointeger := 0 ;
  END ; (* tointeger *)


 FUNCTION is_number(s : string255) : boolean ;
  VAR
   num : real ;
   code : integer ;
  BEGIN
   strip_trailing_blanks(s) ;
   strip_leading_blanks(s) ;
   IF s = ''
    THEN code := -1
   ELSE IF length(s) = 1
    THEN
     IF S[1] IN ['0' ..'9']
      THEN code := 0
      ELSE code := -1
   ELSE val(s,num,code) ;
   is_number := (code = 0) ;
  END ; (* is_number *)


 FUNCTION cardinal(i : integer) : real ;
  VAR
   r : real ;
  BEGIN
   r := i ;
   IF r < 0
    THEN r := r + 65536.0 ;
   cardinal := r ;
  END ; (* cardinal *)


 FUNCTION tag_value(list : s_expr) : node_type ;
  (* returns the value of the tag for a node.     *)
  BEGIN
   IF list = NIL
    THEN tag_value := free_node
    ELSE tag_value := list^.tag ;
  END ; (* tag_value *)


 FUNCTION car(list : s_expr) : s_expr ;
  (* returns a pointer to the first item in the list.
     If the list is empty, it returns NIL.  *)
  BEGIN
   IF list = NIL
    THEN car := NIL
   ELSE IF tag_value(list) = cons_node
    THEN car := list^.car_ptr
   ELSE car := NIL ;
  END ; (* car *)


 FUNCTION cdr(list : s_expr) : s_expr ;
  (* returns a pointer to a list starting at the second item in the list.
     Note - cdr( (a b c) ) points to the list (b c), but
            cdr( ((a b) c d) ) points to the list (c d) .  *)
  BEGIN
   IF list = NIL
    THEN cdr := NIL
    ELSE
     CASE tag_value(list) OF
      cons_node : cdr := list^.cdr_ptr ;
      free_node : cdr := list^.next_free ;
      ELSE        cdr := NIL ;
     END ;
  END ; (* cdr *)


 FUNCTION atom(p : s_expr) : boolean ;
  (* Return true if p is a symbolic or numeric atom, otherwise
     it returns false *)
  BEGIN
   IF p = NIL
    THEN atom := false
   ELSE IF tag_value(p) IN [number,symbol]
    THEN atom := true
   ELSE atom := false ;
  END ; (* atom *)


 FUNCTION allocation_size(x : integer) : integer ;
  (* Turbo 3.0 allocates memory in 8 byte blocks, this routine calculates the
     actual number of bytes returned for a request of x bytes.  *)
  BEGIN
   allocation_size := (((x - 1) SHR 3) + 1) SHL 3 ;
  END ; (* allocation_size *)


 FUNCTION node_size : counter ;
  (* calculates the size of a cons node. *)
  BEGIN
   node_size := 2 * sizeof(s_expr) + sizeof(boolean) + sizeof(node_type) ;
  END ; (* node_size *)


 FUNCTION normalize(pt : s_expr) : s_expr ;
  (* returns a normalized pointer. Pointers are 32 bit addresses. The first
     16 bits contain the segment number and the second 16 bits contain the
     offset within the segment. Normalized pointers have offsets in the range
     $0 to $F (0 .. 15)    *)
  VAR
   pt_seg,pt_ofs : integer ;
  BEGIN
   pt_seg := seg(pt^) + (ofs(pt^) SHR 4) ;
   pt_ofs := ofs(pt^) AND $000F ;
   normalize := ptr(pt_seg,pt_ofs) ;
  END ; (* normalize *)


 FUNCTION string_val(list : s_expr) : string80 ;
  (* returns the string pointed to by list. If list points to a number
     node, it returns a string representing that number *)
  TYPE
   real_rec = RECORD
               CASE boolean OF
                true  : (p1 : real) ;
                false : (p2 : ARRAY [0 ..5] OF byte) ;
               END ;
  VAR
   s : string80 ;
   p : real_rec ;

  PROCEDURE strip_trailing_zeros(VAR ss : string80) ;
   BEGIN
    IF ss <> ''
     THEN
      IF ss[length(ss)] = '0'
       THEN
        BEGIN
         delete(ss,length(ss),1) ;
         strip_trailing_zeros(ss) ;
        END ;
   END ; (* strip_trailing_zeros *)

  BEGIN
   IF list = NIL
    THEN string_val := ''
   ELSE IF list^.tag = symbol
    THEN string_val := list^.string_data
   ELSE IF list^.tag = number
    THEN
     WITH list^ DO
      BEGIN
       p.p1 := abs(frac(num_data)) ;
       IF p.p2[0] = 0
        THEN str(num_data : 20 : 0,s)
       ELSE IF p.p2[0] < 112
        THEN str(num_data,s)
       ELSE
        BEGIN
         str(num_data : 20 : 10,s) ;
         strip_trailing_zeros(s) ;
        END ;
       strip_leading_blanks(s) ;
       string_val := s ;
      END
   ELSE string_val := '' ;
  END ; (* string_val *)


 FUNCTION num_val(list : s_expr) : real ;
  (* returns the number pointed to by list. If list points to a string,
     it returns the numerical value of the string.   *)
  BEGIN
   IF list = NIL
    THEN num_val := 0.0
   ELSE IF list^.tag = number
    THEN num_val := list^.num_data
   ELSE IF list^.tag = symbol
    THEN num_val := toreal(list^.string_data)
   ELSE num_val := 0.0 ;
  END ; (* num_val *)


 PROCEDURE get_memory(VAR pt : s_expr ; size : integer) ;
  (* On exit p contains a pointer to a block of allocation_size(size) bytes.
     If possible this routine tries to get memory from the free list before
     requesting it from the heap *)
  VAR
   blks : counter ;
   allocated : boolean ;

  PROCEDURE get_from_free ;
   VAR
    p,q : s_expr ;
   BEGIN
    p := free ;
    q := p ;
    WHILE (p <> NIL) AND (NOT allocated) DO
     IF p^.block_cnt >= (blks - 1)
      THEN
       BEGIN
        pt := normalize(ptr(seg(p^),ofs(p^) + (p^.block_cnt - blks + 1) SHL 3)) ;
        IF p^.block_cnt = (blks - 1)
         THEN
          IF p = free
           THEN free := free^.next_free
           ELSE q^.next_free := p^.next_free
         ELSE p^.block_cnt := p^.block_cnt - blks ;
        allocated := true ;
        total_free := total_free - (blks * 8.0) ;
       END
      ELSE
       BEGIN
        q := p ;
        p := p^.next_free ;
       END ;
   END ; (* get_from_free *)

  BEGIN
   blks := ((size - 1) SHR 3) + 1 ;
   allocated := false ;
   get_from_free ;
   IF NOT allocated
    THEN getmem(pt,blks SHL 3) ;
  END ; (* get_memory *)


 FUNCTION alloc_str(s : string80) : s_expr ;
  (* Allocate storage for a string and return a pointer to the new node.
     This routine only allocates enough storage for the actual number of
     characters in the string plus one for the length. Because of this,
     concatenating anything to the end of a string stored in a symbol node
     will lead to disaster. Copy the string to a new string do the
     concatenation and then allocate a new node.  *)
  VAR
   pt : s_expr ;
  BEGIN
   get_memory(pt,allocation_size(sizeof(node_type) + sizeof(boolean) +
                                 length(s) + 1)) ;
   pt^.tag := symbol ;
   pt^.string_data := s ;
   alloc_str := pt ;
  END ; (* alloc_str *)


 FUNCTION alloc_num(r : real) : s_expr ;
  (* Allocate storage for a number and return a pointer to the new node.
     All numbers are stored as reals. This isn't efficient, but it is
     easy. *)
  VAR
   pt : s_expr ;
  BEGIN
   get_memory(pt,allocation_size(sizeof(node_type) + sizeof(boolean) +
                                 sizeof(real))) ;
   pt^.tag := number ;
   pt^.num_data := r ;
   alloc_num := pt ;
  END ; (* alloc_num *)


 FUNCTION cons(new_node,list : s_expr) : s_expr ;
  (* Construct a list. This routine allocates storage for a new cons node.
     new_node points to the new car of the list. The cdr pointer of the
     new node points to list. This routine adds the new cons node to the
     beginning of the list and returns a pointer to it. The list described
     in the comments at the beginning of the program could be constructed
     as cons(alloc_str('A'),cons(alloc_str('B'),cons(alloc_str('C'),NIL))). *)
  VAR
   p : s_expr ;
  BEGIN
   get_memory(p,allocation_size(node_size)) ;
   p^.tag := cons_node ;
   p^.car_ptr := new_node ;
   p^.cdr_ptr := list ;
   cons := p ;
  END ; (* cons *)


 FUNCTION eq(item1,item2 : s_expr) : boolean ;
  (* test the equality of two atoms, if item1 and item2 are not atoms
      it returns false *)
  BEGIN
   IF (item1 = NIL) AND (item2 = NIL)
    THEN eq := true
   ELSE IF (tag_value(item1) IN [number,symbol]) AND
           (tag_value(item2) IN [number,symbol])
    THEN
     BEGIN
      IF (tag_value(item1) = number) AND (tag_value(item2) = number)
       THEN eq := (num_val(item1) = num_val(item2))
       ELSE eq := (string_val(item1) = string_val(item2)) ;
     END
   ELSE eq := false ;
  END ; (* eq *)


 FUNCTION lt(item1,item2 : s_expr) : boolean ;
  (* tests if item1 < item2, if item1 and item2 are not atoms
      it returns false *)
  BEGIN
   IF (item1 = NIL) AND (item2 = NIL)
    THEN lt := false
   ELSE IF (tag_value(item1) IN [number,symbol]) AND
           (tag_value(item2) IN [number,symbol])
    THEN
     BEGIN
      IF (tag_value(item1) = number) AND (tag_value(item2) = number)
       THEN lt := (num_val(item1) < num_val(item2))
       ELSE lt := (string_val(item1) < string_val(item2)) ;
     END
   ELSE lt := false ;
  END ; (* lt *)


 FUNCTION gt(item1,item2 : s_expr) : boolean ;
  (* tests if item1 > item2, if item1 and item2 are not atoms
      it returns false *)
  BEGIN
   IF (item1 = NIL) AND (item2 = NIL)
    THEN gt := false
   ELSE IF (tag_value(item1) IN [number,symbol]) AND
           (tag_value(item2) IN [number,symbol])
    THEN
     BEGIN
      IF (tag_value(item1) = number) AND (tag_value(item2) = number)
       THEN gt := (num_val(item1) > num_val(item2))
       ELSE gt := (string_val(item1) > string_val(item2)) ;
     END
   ELSE gt := false ;
  END ; (* gt *)


 FUNCTION add(item1,item2 : s_expr) : s_expr ;
  (* add the values of two atoms, if item1 and item2 are not atoms
      it returns 0 *)
  VAR
    r1,r2 : real ;
  BEGIN
   IF tag_value(item1) = number
    THEN r1 := num_val(item1)
    ELSE r1 := toreal(string_val(item1)) ;
   IF tag_value(item2) = number
    THEN r2 := num_val(item2)
    ELSE r2 := toreal(string_val(item2)) ;
   add := alloc_num(r1 + r2) ;
  END ; (* add *)


 FUNCTION sub(item1,item2 : s_expr) : s_expr ;
  (* finds the difference between the values of two atoms,
     if item1 and item2 are not atoms it returns 0 *)
  VAR
    r1,r2 : real ;
  BEGIN
   IF tag_value(item1) = number
    THEN r1 := num_val(item1)
    ELSE r1 := toreal(string_val(item1)) ;
   IF tag_value(item2) = number
    THEN r2 := num_val(item2)
    ELSE r2 := toreal(string_val(item2)) ;
   sub := alloc_num(r1 - r2) ;
  END ; (* sub *)


 FUNCTION mul(item1,item2 : s_expr) : s_expr ;
  (* finds the product of the values of two atoms,
     if item1 and item2 are not atoms it returns 0 *)
  VAR
    r1,r2 : real ;
  BEGIN
   IF tag_value(item1) = number
    THEN r1 := num_val(item1)
    ELSE r1 := toreal(string_val(item1)) ;
   IF tag_value(item2) = number
    THEN r2 := num_val(item2)
    ELSE r2 := toreal(string_val(item2)) ;
   mul := alloc_num(r1 * r2) ;
  END ; (* mul *)


 FUNCTION div_f(item1,item2 : s_expr) : s_expr ;
  (* divides item1 by item2,
     if item1 and item2 are not atoms it returns 0 *)
  VAR
    r1,r2 : real ;
  BEGIN
   IF tag_value(item1) = number
    THEN r1 := num_val(item1)
    ELSE r1 := toreal(string_val(item1)) ;
   IF tag_value(item2) = number
    THEN r2 := num_val(item2)
    ELSE r2 := toreal(string_val(item2)) ;
   IF abs(r2) <= 1.0E-20
    THEN div_f := alloc_num(0.0)
    ELSE div_f := alloc_num(r1 / r2) ;
  END ; (* div_f *)


 FUNCTION mod_f(item1,item2 : s_expr) : s_expr ;
  (* finds the remainder of item1 divided by item2,
     if item1 and item2 are not atoms it returns 0 *)
  VAR
    r1,r2 : integer ;
  BEGIN
   r1 := tointeger(string_val(item1)) ;
   r2 := tointeger(string_val(item2)) ;
   mod_f := alloc_num(r1 MOD r2) ;
  END ; (* mod_f *)


 FUNCTION member(p,list : s_expr) : boolean ;
  (* returns true if p points to a member of list *)
  BEGIN
   IF list = NIL
    THEN member := false
   ELSE IF eq(p,car(list))
    THEN member := true
   ELSE member := member(p,cdr(list)) ;
  END ; (* member *)


 FUNCTION locate(p,list1,list2 : s_expr) : s_expr ;
  (* finds p on list1 and returns a pointer to the corresponding
     element of list2 *)
  BEGIN
   IF list1 = NIL
    THEN locate := NIL
   ELSE IF eq(p,car(list1))
    THEN locate := car(list2)
   ELSE locate := locate(p,cdr(list1),cdr(list2)) ;
  END ; (* locate *)


 FUNCTION assoc(p,list1,list2 : s_expr) : s_expr ;
  (* search each sublist of list1 for p. If found, return pointer to
     corresponding element of list2 *)
  BEGIN
   IF list1 = NIL
    THEN assoc := NIL
   ELSE IF member(p,car(list1))
    THEN assoc := locate(p,car(list1),car(list2))
   ELSE assoc := assoc(p,cdr(list1),cdr(list2)) ;
  END ; (* assoc *)


 FUNCTION locate_obj(list,p : s_expr) : s_expr ;
  (* search list for obj p, returns a pointer to the
       structure of p *)
  VAR
   found : boolean ;
   name : string80 ;
   l : s_expr ;
  BEGIN
   name := string_val(p) ;
   found := false ;
   l := list ;
   WHILE (l <> NIL) AND (NOT found) DO
    IF string_val(car(car(l))) = name
     THEN found := true
     ELSE l := cdr(l) ;
   locate_obj := car(l) ;
  END ; (* locate_obj *)


 FUNCTION append_list(list1,list2 : s_expr) : s_expr ;
  (* attach list2 to the end of list1, return a pointer to the
     combined list *)
  BEGIN
   IF list1 = NIL
    THEN append_list := list2
    ELSE append_list := cons(car(list1),append_list(cdr(list1),list2)) ;
  END ; (* append_list *)


 FUNCTION tf_node(t : boolean) : s_expr ;
  (* allocates T or F nodes for boolean expressions *)
  BEGIN
   IF t
    THEN tf_node := alloc_str('T')
    ELSE tf_node := alloc_str('F') ;
  END ; (* tf_node *)


 FUNCTION rplaca(list : s_expr ; item : s_expr) : s_expr ;
  (* replace the car of list with item, return a pointer to the new list.
     This routine has serious side effects *)
  BEGIN
   IF list <> NIL
    THEN
     IF tag_value(list) = cons_node
      THEN list^.car_ptr := item ;
   rplaca := list ;
  END ; (* rplaca *)


 PROCEDURE collect_garbage ;
  (* This routine is specific to Turbo Pascal Ver 3.01
     It depends upon the fact that Turbo allocates memory in 8 byte blocks
     on the PC. If you recompile this program on another system be very
     careful with this routine.
     Garbage collection proceeds in three phases:
      unmark  - free all memory between the initial_heap^ and the current
                top of the heap.
      mark_mem    - mark everything on the saved_list as being in ues.
      release - gather all unmarked blocks and put them on the free list.
     The collector displays a '*' on the screen to let you know it is
      operating.  *)

  FUNCTION lower(p1,p2 : s_expr) : boolean ;
   (* returns true if p1 points to a lower memory address than p2 *)
   BEGIN
    p1 := normalize(p1) ;
    p2 := normalize(p2) ;
    lower := (cardinal(seg(p1^)) < cardinal(seg(p2^))) OR
              ((seg(p1^) = seg(p2^)) AND
              (cardinal(ofs(p1^)) < cardinal(ofs(p2^)))) ;
   END ; (* lower *)

  PROCEDURE mark_mem(list : s_expr) ;
   (* Mark the blocks on list as being in use. Since a node may be on several
      lists at one time, if it is already marked we don't continue processing
      the cdr of the list. *)
   BEGIN
    IF list <> NIL
     THEN
      BEGIN
       IF NOT list^.in_use
        THEN
         BEGIN
          list^.in_use := true ;
          IF list^.tag = cons_node
           THEN
            BEGIN
             mark_mem(car(list)) ;
             mark_mem(cdr(list)) ;
            END ;
         END ;
      END ;
   END ; (* mark_mem *)

  PROCEDURE unmark_mem ;
   (* Go through memory from initial_heap^ to HeapPtr^ and mark each node
      as not in use. The tricky part here is updating the pointer p to point
      to the next cell. *)
   VAR
    p : s_expr ;
    string_base,node_allocation,number_allocation : integer ;
   BEGIN
    string_base := sizeof(node_type) + sizeof(boolean) ;
    p := normalize(initial_heap) ;
    node_allocation := allocation_size(node_size) ;
    number_allocation := allocation_size(sizeof(node_type) + sizeof(boolean) +
                                         sizeof(real)) ;
    WHILE lower(p,HeapPtr) DO
     BEGIN
      p^.in_use := false ;
      CASE p^.tag OF
       cons_node : p := normalize(ptr(seg(p^),ofs(p^) + node_allocation)) ;
       free_node : p := normalize(ptr(seg(p^),ofs(p^) + ((p^.block_cnt + 1) SHL 3))) ;
       number    : p := normalize(ptr(seg(p^),ofs(p^) + number_allocation)) ;
       symbol    : p := normalize(ptr(seg(p^),
                                  ofs(p^) +
                                  allocation_size(string_base +
                                                  length(p^.string_data) + 1))) ;
      END ;
     END ;
   END ; (* unmark_mem *)

  PROCEDURE release_mem ;
   (* This procedure does the actual collection and compaction of nodes.
      This is the slow phase of garbage collection because of all the pointer
      manipulation.  *)
   VAR
    heap_top : s_expr ;
    string_base,node_allocation,string_allocation,block_allocation,
                number_allocation : integer ;

   PROCEDURE free_memory(pt : s_expr ; size : counter) ;
    (* return size bytes pointed to by pt to the free list. If pt points to
       a block next to the head of the free list combine it with the top
       free node. total_free keeps track of the total number of free bytes. *)
    VAR
     blks : counter ;
    BEGIN
     IF pt <> NIL
      THEN
       BEGIN
        blks := ((size - 1) SHR 3) + 1 ;
        pt^.tag := free_node ;
        IF normalize(ptr(seg(pt^),ofs(pt^) + (blks SHL 3))) = free
         THEN
          BEGIN
           pt^.next_free := free^.next_free ;
           pt^.block_cnt := free^.block_cnt + blks ;
           free := pt ;
          END
        ELSE IF normalize(ptr(seg(free^),ofs(free^) +
                      ((free^.block_cnt + 1) SHL 3))) = normalize(pt)
         THEN free^.block_cnt := free^.block_cnt + blks
        ELSE
         BEGIN
          pt^.next_free := free ;
          pt^.block_cnt := blks - 1 ;
          free := pt ;
         END ;
        total_free := total_free + (blks * 8.0) ;
       END ;
    END ; (* free_memory *)

   PROCEDURE do_release ;
    (* This routine sweeps through memory and checks for nodes with
       in_use = false. *)
    VAR
     p : s_expr ;
    BEGIN
     p := normalize(initial_heap) ;
     WHILE lower(p,heap_top) DO
      CASE p^.tag OF
       cons_node : BEGIN
                    IF NOT p^.in_use
                     THEN free_memory(p,node_size) ;
                    p := normalize(ptr(seg(p^),ofs(p^) + node_allocation)) ;
                   END ;
       free_node : BEGIN
                    block_allocation := (p^.block_cnt + 1) SHL 3 ;
                    free_memory(p,block_allocation) ;
                    p := normalize(ptr(seg(p^),ofs(p^) + block_allocation)) ;
                   END ;
       number    : BEGIN
                    IF NOT p^.in_use
                     THEN free_memory(p,number_allocation) ;
                    p := normalize(ptr(seg(p^),ofs(p^) + number_allocation)) ;
                   END ;
       symbol    : BEGIN
                    string_allocation := allocation_size(string_base +
                                                length(p^.string_data) + 1) ;
                    IF NOT p^.in_use
                     THEN free_memory(p,string_base + length(p^.string_data)
                                      + 1) ;
                    p := normalize(ptr(seg(p^),ofs(p^) + string_allocation)) ;
                   END ;
      END ;
    END ; (* do_release *)

   BEGIN
    free := NIL ;
    total_free := 0.0 ;
    heap_top := HeapPtr ;
    string_base := sizeof(node_type) + sizeof(boolean) ;
    node_allocation := allocation_size(node_size) ;
    number_allocation := allocation_size(sizeof(node_type) + sizeof(boolean) +
                                         sizeof(real)) ;
    do_release ;
   END ; (* release_mem *)

  BEGIN
   write('*') ;
   unmark_mem ;
   mark_mem(saved_list) ;
   release_mem ;
   write(back_space) ;
   clreol ;
  END ; (* collect_garbage *)


 PROCEDURE test_memory ;
  (* This routine activates the garbage collector, if the the total available
     memory (free_list + heap) is less than a specified amount. Lowering the
     minimum causes garbage collection to be called less often, but if you
     make it too small you may not have enough room left for recursion or any
     temporary lists you need. Using 10000 is probably being overly
     cautious.   *)
  BEGIN
   IF (memavail * 16.0) + total_free < 10000
    THEN collect_garbage ;
  END ; (* test_memory *)


 PROCEDURE wait ;
  (* Just like it says. It waits for the user to press a key before
     continuing. *)
  VAR
   ch : char ;
  BEGIN
   writeln ;
   writeln ;
   write('Press any key to continue. ') ;
   read(kbd,ch) ;
   write(return) ;
   clreol ;
  END ; (* wait *)


 PROCEDURE break ;
  (* An extremely dangerous routine. Retreive the value of the stack pointer
     saved in read_and_process, point sp to it and pop the stack. Has the
     effect of returning control to the routine which called read_and_process.
     It aborts the current routine and abandons the stack and local values.
     We don't recommend this sort of thing, but it was the only method we could
     think of that allowed an abort from within recursive routines. If you
     come up with a less messy way of doing this, let us know *)
  BEGIN
   writeln ;
   inline($3E/$8B/$26/mark_stack/      {mov sp,ds:mark_stack}
          $5D/                         {pop bp}
          $C3) ;                       {ret}
  END ; (* break *)


 PROCEDURE check_break ;
  (* check for <ctrl-break> or <crtl-s> *)
  VAR
   ch : char ;
  BEGIN
   IF keypressed
    THEN
     BEGIN
      read(kbd,ch) ;
      CASE ch OF
       ctrl_c : break ;
       ctrl_s : BEGIN
                 read(kbd,ch) ;
                 IF ch = ctrl_c
                  THEN break ;
                END ;
      END ;
     END ;
  END ; (* check_break *)


(* ------------------------------------------------------------------------
        End of utility routines
   ------------------------------------------------------------------------ *)

 PROCEDURE read_kbd(VAR s : string80) ;
  (* Read a line from the keyboard. The number of unmatched parentheses are
     printed along with the prompt *)

  PROCEDURE read_line ;
   VAR
    done : boolean ;
    ch : char ;

   PROCEDURE echo ;
    BEGIN
     s := concat(s,ch) ;
     write(ch) ;
    END ; (* echo *)

   PROCEDURE delete_back ;
    BEGIN
     IF length(s) > 0
      THEN
       BEGIN
        delete(s,length(s),1) ;
        gotoxy(pred(wherex),wherey) ;
        write(' ') ;
        gotoxy(pred(wherex),wherey) ;
       END ;
    END ; (* delete_back *)

   BEGIN
    s := '' ;
    done := false ;
    WHILE NOT done DO
     BEGIN
      read(kbd,ch) ;
      CASE ch OF
       ' ' .. '~' : echo ;
       return     : BEGIN
                     done := true ;
                     writeln ;
                    END ;
       back_space : delete_back ;
       ctrl_c     : break ;
       eof_mark   : BEGIN
                     done := true ;
                     s := concat(s,eof_mark) ;
                    END ;
      END ;
     END ;
   END ; (* read_line *)

  BEGIN
   IF paren_level > 0
    THEN write(paren_level,'>')
    ELSE write('->') ;
   read_line ;
  END ; (* read_kbd *)


 PROCEDURE read_from_file(VAR f : text_file) ;
  (* Read a line from file f and store it in the global variable line.
     It ignores blank lines and when the end of file is reached an
     eof_mark is returned. *)

  PROCEDURE read_a_line ;
   BEGIN
    (*$I- *)
    readln(f,line) ;
    (*$I+ *)
    IF ioresult <> 0
     THEN line := eof_mark
    ELSE IF eof(f)
     THEN line := concat(line,eof_mark) ;
   END ; (* read_a_line *)

  BEGIN
   line := '' ;
   IF is_console(f)
    THEN read_kbd(line)
    ELSE read_a_line ;
   saved_line := line ;
  END ; (* read_from_file *)


 PROCEDURE get_token(VAR t_line : string255 ; VAR token : string80) ;
  (* Get a token from t_line. A token is a string of text surrounded by
     blanks or a delimeter. Comments begin with ; and extend to the
     end of the line *)

  PROCEDURE get_word ;
   VAR
    done : boolean ;
    cn : integer ;
    len : byte ;
   BEGIN
    cn := 1 ;
    len := length(t_line) ;
    done := false ;
    WHILE NOT done DO
     IF cn > len
      THEN done := true
     ELSE IF t_line[cn] IN delim_set
      THEN done := true
     ELSE cn := cn + 1 ;
    token := copy(t_line,1,cn-1) ;
    delete(t_line,1,cn-1) ;
   END ; (* get_word *)

  PROCEDURE comment ;
   BEGIN
    t_line := '' ;
    get_token(t_line,token) ;
   END ; (* comment *)

  PROCEDURE get_number ;

   PROCEDURE get_digits ;
    BEGIN
     WHILE is_number(copy(t_line,1,1)) DO
      BEGIN
       token := concat(token,t_line[1]) ;
       delete(t_line,1,1) ;
      END ;
    END ; (* get_digits *)

   PROCEDURE get_exponent ;
    BEGIN
     delete(t_line,1,1) ;
     IF length(t_line) > 0
      THEN
       BEGIN
        IF t_line[1] IN ['+','-']
         THEN
          BEGIN
           token := concat(token,'E',t_line[1]) ;
           delete(t_line,1,1) ;
          END
         ELSE token := concat(token,'E+') ;
        get_digits ;
       END
      ELSE token := concat(token,'E+00') ;
    END ; (* get_exponent *)

   BEGIN
    get_digits ;
    IF length(t_line) > 0
     THEN
      IF t_line[1] = '.'
       THEN
        IF is_number(copy(t_line,2,1))
         THEN
          BEGIN
           token := concat(token,t_line[1]) ;
           delete(t_line,1,1) ;
           get_digits ;
           IF toupper(copy(t_line,1,1)) = 'E'
            THEN get_exponent ;
          END ;
   END ; (* get_number *)

  PROCEDURE check_number ;
   VAR
    sgn : char ;
   BEGIN
    sgn := t_line[1] ;
    delete(t_line,1,1) ;
    IF length(t_line) > 0
     THEN
      IF t_line[1] IN ['0' .. '9']
       THEN
        BEGIN
         get_number ;
         token := concat(sgn,token) ;
        END
       ELSE token := sgn
     ELSE token := sgn ;
   END ; (* check_number *)

  BEGIN
   strip_leading_blanks(t_line) ;
   token := '' ;
   IF length(t_line) > 0
    THEN
     BEGIN
      IF t_line[1] = ';'
       THEN comment
      ELSE IF t_line[1] IN delim_set
       THEN
        BEGIN
         token := t_line[1] ;
         delete(t_line,1,1) ;
        END
      ELSE IF t_line[1] IN ['+','-']
       THEN check_number
      ELSE IF t_line[1] IN ['0' .. '9']
       THEN get_number
      ELSE get_word ;
     END ;
  END ; (* get_token *)


 PROCEDURE scan(VAR f : text_file ; VAR token : string80) ;
  (* Scan repeatedly calls get_token to retreive tokens. When the
     end of a line has been reached, read_from_file is called to
     get a new line. *)
  BEGIN
   IF length(line) > 0
    THEN
     BEGIN
      get_token(line,token) ;
      IF token = ''
       THEN scan(f,token)
      ELSE IF token = eof_mark
       THEN token := ')' ;
     END
    ELSE
     BEGIN
      read_from_file(f) ;
      scan(f,token) ;
     END ;
  END ; (* scan *)


 PROCEDURE error(error_msg : string80) ;
  BEGIN
   writeln ;
   writeln(error_msg) ;
   wait ;
  END ; (* error *)


 FUNCTION get_expression_list(VAR f : text_file) : s_expr ; FORWARD ;


 FUNCTION get_expression(VAR f : text_file) : s_expr ;
  (* read an expression from f. This routine and get_expression_list
     work together to form a small recursive descent compiler for
     S-expressions. It follows the definition fo an S-expression
     from the article. It adds a quote to numbers and strings beginning with
     a ' mark *)
  BEGIN
   IF token = '('
    THEN
     BEGIN
      scan(f,token) ;
      IF token = ')'
       THEN get_expression := NIL
       ELSE
        BEGIN
         paren_level := paren_level + 1 ;
         get_expression := get_expression_list(f) ;
         IF token <> ')'
          THEN error('Missing '')''') ;
        END ;
     END
   ELSE IF token = quote_char
    THEN
     BEGIN
      scan(f,token) ;
      get_expression := cons(quote_node,cons(get_expression(f),NIL)) ;
     END
   ELSE IF toupper(token) = 'NIL'
    THEN get_expression := NIL
   ELSE IF is_number(token)
    THEN get_expression := alloc_num(toreal(token))
   ELSE get_expression := alloc_str(toupper(token)) ;
  END ; (* get_expression *)


 FUNCTION get_expression_list (* VAR f : text_file) : s_expr *) ;
  (* read an S-expression list or dotted pair *)
  VAR
   p : s_expr ;
  BEGIN
   p := get_expression(f) ;
   scan(f,token) ;
   IF token = '.'
    THEN
     BEGIN
      scan(f,token) ;
      get_expression_list := cons(p,get_expression(f)) ;
      scan(f,token) ;
     END
   ELSE IF token = ')'
    THEN
     BEGIN
      paren_level := paren_level - 1 ;
      get_expression_list := cons(p,NIL) ;
     END
   ELSE get_expression_list := cons(p,get_expression_list(f)) ;
  END ; (* get_expression_list *)


 PROCEDURE print_expression(l : s_expr) ;
  (* recursively traverses the list and prints its elements. This is
     not a pretty printer, so the lists may look a bit messy. This routine
     tries to print the minimum possible number of parentheses. *)

  PROCEDURE print_list(list : s_expr) ;
   VAR
    p : s_expr ;
   BEGIN
    check_break ;
    IF list <> NIL
     THEN
      CASE list^.tag OF
       number,
       symbol    : write(string_val(list),' ') ;
       cons_node : BEGIN
                    write('(') ;
                    p := list ;
                    WHILE tag_value(p) = cons_node DO
                     BEGIN
                      print_expression(car(p)) ;
                      p := cdr(p) ;
                     END ;
                    IF p <> NIL
                     THEN
                      BEGIN
                       write('.') ;
                       print_expression(p) ;
                      END ;
                    write(') ') ;
                   END ;
      END ;
   END ; (* print_list *)

  BEGIN
   IF l = NIL
    THEN write('nil ')
    ELSE print_list(l) ;
  END ; (* print_expression *)


 PROCEDURE display_list(msg : string80 ; list : s_expr) ;
  BEGIN
   write(msg,' : ') ;
   print_expression(list) ;
   writeln ;
  END ; (* display_list *)


 FUNCTION eval(expr_list,name_list,value_list : s_expr) : s_expr ;
  (* The main evaluation routine. This routine is explained in the articles.
     expr_list contains the S-expression to be evaluated. name_list is the
     list of active variable names. value_list is the list of corresponding
     values. expr_list,name_list and value_list are attached to saved_list
     at the start of the routine so that if garbage collection takes place
     they won't be reclaimed. They are removed at the end of this routine. *)
  VAR
   f_name : string80 ;

  FUNCTION vars(list : s_expr) : s_expr ;
   (* make a list of variables from list *)
   BEGIN
    IF list = NIL
     THEN vars := NIL
     ELSE vars := cons(car(car(list)),vars(cdr(list))) ;
   END ; (* vars *)

  FUNCTION exprs(list : s_expr) : s_expr ;
   (* make a list of expressions *)
   BEGIN
    IF list = NIL
     THEN exprs := NIL
     ELSE exprs := cons(car(cdr(car(list))),exprs(cdr(list))) ;
   END ; (* exprs *)

  FUNCTION eval_list(list,name,value : s_expr) : s_expr ;
   (* evaluate a list, one item at a time. It does this by calling eval
      for each element of list *)
   BEGIN
    IF list = NIL
     THEN eval_list := NIL
     ELSE
      BEGIN
       saved_list := cons(list,saved_list) ;
       eval_list := cons(eval(car(list),name,value),
                            eval_list(cdr(list),name,value)) ;
       saved_list := cdr(saved_list) ;
      END ;
   END ; (* eval_list *)

  FUNCTION eval_if : s_expr ;
   BEGIN
    IF string_val(eval(car(cdr(expr_list)),name_list,value_list)) = 'T'
     THEN eval_if := eval(car(cdr(cdr(expr_list))),name_list,value_list)
     ELSE eval_if := eval(car(cdr(cdr(cdr(expr_list)))),name_list,
                          value_list)
   END ; (* eval_if *)

  FUNCTION eval_let : s_expr ;
   VAR
    y,z : s_expr ;
   BEGIN
    z := eval_list(exprs(cdr(cdr(expr_list))),name_list,value_list) ;
    y := vars(cdr(cdr(expr_list))) ;
    eval_let := eval(car(cdr(expr_list)),cons(y,name_list),
                     cons(z,value_list)) ;
   END ; (* eval_let *)

  FUNCTION eval_letrec : s_expr ;
   VAR
    v,y,z : s_expr ;
   BEGIN
    v := cons(cons(pending,NIL),value_list) ;
    y := vars(cdr(cdr(expr_list))) ;
    z := eval_list(exprs(cdr(cdr(expr_list))),cons(y,name_list),v) ;
    eval_letrec := eval(car(cdr(expr_list)),cons(y,name_list),
                        rplaca(v,z)) ;
   END ; (* eval_letrec *)

  FUNCTION eval_read : s_expr ;
   (* read an expression from a file. The file must end in ".LSP".
      That's because we were too lazy to implement strings. The expression
      read from the file is evaluated. *)
   VAR
    f : text_file ;
    file_name : string80 ;
    old_line,old_saved_line : string255 ;
   BEGIN
    file_name := string_val(eval(car(cdr(expr_list)),name_list,value_list)) ;
    IF pos('.',file_name) = 0
     THEN file_name := concat(file_name,'.LSP') ;
    IF open(f,file_name)
     THEN
      BEGIN
       old_line := line ;
       old_saved_line := saved_line ;
       line := '' ;
       scan(f,token) ;
       eval_read := eval(get_expression(f),name_list,value_list) ;
       close(f) ;
       line := old_line ;
       saved_line := old_saved_line ;
      END
     ELSE
      BEGIN
       error(concat('Unable to read ',file_name)) ;
       eval_read := NIL ;
      END ;
   END ; (* eval_read *)

  FUNCTION eval_load : s_expr ;
   (* read an entire file. The file must end in ".LSP".
      That's because we were too lazy to implement strings. The expression
      read from the file is evaluated. *)
   VAR
    f : text_file ;
    file_name : string80 ;
    old_line,old_saved_line : string255 ;
   BEGIN
    file_name := string_val(eval(car(cdr(expr_list)),name_list,value_list)) ;
    IF pos('.',file_name) = 0
     THEN file_name := concat(file_name,'.LSP') ;
    IF open(f,file_name)
     THEN
      BEGIN
       old_line := line ;
       old_saved_line := saved_line ;
       line := '' ;
       scan(f,token) ;
       WHILE NOT eof(f) DO
        BEGIN
         eval_load := eval(get_expression(f),name_list,value_list) ;
         scan(f,token) ;
        END ;
       close(f) ;
       line := old_line ;
       saved_line := old_saved_line ;
      END
     ELSE
      BEGIN
       error(concat('Unable to read ',file_name)) ;
       eval_load := NIL ;
      END ;
   END ; (* eval_load *)

  FUNCTION eval_defun : s_expr ;
   VAR
    z,v,y : s_expr ;
   BEGIN
    v := cons(lambda_node,cons(car(cdr(cdr(expr_list))),
                     cons(car(cdr(cdr(cdr(expr_list)))),NIL))) ;
    y := cons(car(cdr(expr_list)),cons(v,NIL)) ;
    z := cons(letrec_node,cons(car(cdr(expr_list)),cons(y,NIL))) ;
    g_value_list := cons(cons(eval(z,name_list,value_list),NIL),g_value_list) ;
    g_name_list := cons(cons(car(cdr(expr_list)),NIL),g_name_list) ;
    eval_defun := car(cdr(expr_list)) ;
   END ; (* eval_defun *)

  FUNCTION eval_make : s_expr ;
   VAR
    name,inheritance,vars : s_expr ;

   FUNCTION var_list(list : s_expr) : s_expr ;
    BEGIN
     IF list = NIL
      THEN var_list := NIL
     ELSE IF tag_value(car(list)) = cons_node
      THEN var_list := cons(car(list),var_list(cdr(list)))
     ELSE var_list := cons(cons(car(list),NIL),var_list(cdr(list))) ;
    END ; (* var_list *)

   BEGIN
    name := car(cdr(expr_list)) ;
    inheritance := car(cdr(cdr(expr_list))) ;
    vars := var_list(cdr(cdr(cdr(expr_list)))) ;
    obj_list := cons(cons(name,cons(inheritance,cons(vars,NIL))),
                        obj_list) ;
    eval_make := name ;
   END ; (* eval_make *)

  FUNCTION eval_send : s_expr ;
   VAR
    object,message,local_vars,locn,inherited,
          eval_message,new_name_list,new_value_list,args : s_expr ;

   FUNCTION inherit(list : s_expr) : s_expr ;
    BEGIN
     IF list = NIL
      THEN inherit := NIL
      ELSE
       BEGIN
        locn := locate_obj(obj_list,car(list)) ;
        IF locn <> NIL
         THEN inherit := append_list(car(cdr(cdr(locn))),
                               inherit(append_list(cdr(list),car(cdr(locn)))))
         ELSE inherit := inherit(cdr(list)) ;
       END ;
    END ; (* inherit *)

   BEGIN
    message := car(cdr(cdr(expr_list))) ;
    args := (cdr(cdr(cdr(expr_list)))) ;
    IF toupper(string_val(car(cdr(expr_list)))) = 'SELF'
     THEN
      BEGIN
       eval_message := eval(message,name_list,value_list) ;
       IF args = NIL
        THEN eval_send := eval(eval_message,name_list,value_list)
        ELSE eval_send := eval(cons(eval_message,eval_list(args,name_list,value_list)),
                               name_list,value_list) ;
      END
     ELSE
      BEGIN
       object := locate_obj(obj_list,car(cdr(expr_list))) ;
       IF object = NIL
        THEN eval_send := NIL
        ELSE
         BEGIN
          local_vars := car(cdr(cdr(object))) ;
          inherited := inherit(car(cdr(object))) ;
          new_name_list := cons(vars(local_vars),cons(vars(inherited),name_list)) ;
          new_value_list := cons(exprs(local_vars),cons(exprs(inherited),value_list)) ;
          eval_message := eval(message,new_name_list,new_value_list) ;
          IF args = NIL
           THEN eval_send := eval(eval_message,new_name_list,new_value_list)
           ELSE eval_send := eval(cons(eval_message,
                                  eval_list(args,new_name_list,new_value_list)),
                                  new_name_list,new_value_list) ;
         END ;
      END ;
   END ; (* eval_send *)

  FUNCTION eval_f_call : s_expr ;
   (* evaluate a function call *)
   VAR
    closure,e_list : s_expr ;
   BEGIN
    closure := eval(car(expr_list),name_list,value_list) ;
    e_list := eval_list(cdr(expr_list),name_list,value_list) ;
    eval_f_call := eval(cdr(car(closure)),cons(car(car(closure)),car(cdr(closure))),
                        cons(e_list,cdr(cdr(closure)))) ;
   END ; (* eval_f_call *)

  BEGIN
   check_break ;
   saved_list := cons(expr_list,cons(name_list,cons(value_list,
                      cons(obj_list,saved_list)))) ;
   test_memory ;
   IF expr_list = NIL
    THEN eval := NIL
   ELSE IF tag_value(expr_list) = symbol
    THEN eval := assoc(expr_list,name_list,value_list)
   ELSE IF tag_value(expr_list) = number
    THEN eval := expr_list
   ELSE
    BEGIN
     f_name := string_val(car(expr_list)) ;
     IF f_name = 'QUOTE'
      THEN eval := car(cdr(expr_list))
     ELSE IF f_name = 'CAR'
      THEN eval := car(eval(car(cdr(expr_list)),name_list,value_list))
     ELSE IF f_name = 'CDR'
      THEN eval := cdr(eval(car(cdr(expr_list)),name_list,value_list))
     ELSE IF f_name = 'ATOM'
      THEN eval := tf_node(atom(eval(car(cdr(expr_list)),name_list,
                                     value_list)))
     ELSE IF f_name = 'CONS'
      THEN eval := cons(eval(car(cdr(expr_list)),name_list,value_list),
                        eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF (f_name = 'EQ') OR (f_name = '=')
      THEN eval := tf_node(eq(eval(car(cdr(expr_list)),name_list,value_list),
                           eval(car(cdr(cdr(expr_list))),name_list,value_list)))
     ELSE IF (f_name = 'LT') OR (f_name = '<')
      THEN eval := tf_node(lt(eval(car(cdr(expr_list)),name_list,value_list),
                           eval(car(cdr(cdr(expr_list))),name_list,value_list)))
     ELSE IF (f_name = 'GT') OR (f_name = '>')
      THEN eval := tf_node(gt(eval(car(cdr(expr_list)),name_list,value_list),
                           eval(car(cdr(cdr(expr_list))),name_list,value_list)))
     ELSE IF (f_name = 'LE') OR (f_name = '<=') OR (f_name = '=<')
      THEN eval := tf_node(NOT gt(eval(car(cdr(expr_list)),name_list,value_list),
                           eval(car(cdr(cdr(expr_list))),name_list,value_list)))
     ELSE IF (f_name = 'GE') OR (f_name = '>=') OR (f_name = '=>')
      THEN eval := tf_node(NOT lt(eval(car(cdr(expr_list)),name_list,value_list),
                           eval(car(cdr(cdr(expr_list))),name_list,value_list)))
     ELSE IF (f_name = 'NEQ') OR (f_name = '<>') OR (f_name = '><')
      THEN eval := tf_node(NOT eq(eval(car(cdr(expr_list)),name_list,value_list),
                                  eval(car(cdr(cdr(expr_list))),name_list,
                                       value_list)))
     ELSE IF (f_name = '+') OR (f_name = 'ADD')
      THEN eval := add(eval(car(cdr(expr_list)),name_list,value_list),
                       eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF (f_name = '-') OR (f_name = 'SUB')
      THEN eval := sub(eval(car(cdr(expr_list)),name_list,value_list),
                       eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF (f_name = '*') OR (f_name = 'MUL')
      THEN eval := mul(eval(car(cdr(expr_list)),name_list,value_list),
                       eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF (f_name = '/') OR (f_name = 'DIV')
      THEN eval := div_f(eval(car(cdr(expr_list)),name_list,value_list),
                       eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF f_name = 'MOD'
      THEN eval := mod_f(eval(car(cdr(expr_list)),name_list,value_list),
                         eval(car(cdr(cdr(expr_list))),name_list,value_list))
     ELSE IF f_name = 'IF'
      THEN eval := eval_if
     ELSE IF f_name = 'LAMBDA'
      THEN eval := cons(cons(car(cdr(expr_list)),car(cdr(cdr(expr_list)))),
                        cons(name_list,value_list))
     ELSE IF f_name = 'LET'
      THEN eval := eval_let
     ELSE IF f_name = 'LETREC'
      THEN eval := eval_letrec
     ELSE IF f_name = 'EXIT'
      THEN halt(0)
     ELSE IF f_name = 'READ'
      THEN eval := eval_read
     ELSE IF f_name = 'LOAD'
      THEN eval := eval_load
     ELSE IF f_name = 'DEFUN'
      THEN eval := eval_defun
     ELSE IF f_name = 'MAKE'
      THEN eval := eval_make
     ELSE IF f_name = 'SEND'
      THEN eval := eval_send
     ELSE eval := eval_f_call ;
    END ;
   saved_list := cdr(cdr(cdr(cdr(saved_list)))) ;
  END ; (* eval *)


 PROCEDURE read_and_process ;
  (* The main processing routine. The inline stuff saves the current
     stack pointer so that a process can be aborted by a <ctrl-break> *)
  BEGIN
   inline($3E/$89/$2E/mark_stack) ;  {mov ds:mark_stack,bp}
   saved_list := cons(g_name_list,cons(g_value_list,
                      cons(obj_list,constant_list))) ;
   paren_level := 0 ;
   scan(input,token) ;
   fn := get_expression(input) ;
   result := eval(fn,g_name_list,g_value_list) ;
   print_expression(result) ;
   writeln ;
  END ; (* read_and_process *)


 PROCEDURE initialize ;
  BEGIN
   line := '' ;
   saved_line := '' ;
   delim_set := ['.','(',')',' ',eof_mark,quote_char,';',ctrl_c] ;
   total_free := 0.0 ;
   free := NIL ;
   mark(initial_heap) ;
   pending := alloc_str('#PENDING') ;
   lambda_node := alloc_str('LAMBDA') ;
   letrec_node := alloc_str('LETREC') ;
   quote_node := alloc_str('QUOTE') ;
   constant_list := cons(pending,cons(lambda_node,cons(letrec_node,
                          cons(quote_node,NIL)))) ;
   clrscr ;
   writeln('VT-LISP 2.0 -  Copyright 1987 [c] Knowledge Garden Inc.') ;
   writeln ;
   g_name_list := NIL ;
   g_value_list := NIL ;
   obj_list := NIL ;
  END ; (* initialize *)


 BEGIN
  initialize ;
  REPEAT
   read_and_process ;
  UNTIL false ;
 END.
