;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.LOCAL]UNIT-MAPPING.LISP *-*
;;;; *-* Last-Edit: Tuesday, November 29, 1988  20:18:30 *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 308) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *          UNIT MAPPING AND ASSOCIATED STORAGE BUILDING FUNCTIONS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;     ...   Many changes.  (Gallagher)
;;;  08-25-86 Redo handling of unstructured spaces and unstructured unit
;;;           mappings.  (Gallagher)
;;;  01-16-87 Added UNIT-MAPPING-SPECIFIED-P.  (Gallagher)
;;;  03-19-87 Added more error checking in define-unit-mapping and
;;;           insert-unit-on-space-instance (in particular added bounds
;;;           checking).  (Gallagher)
;;;  06-05-87 Modified DEFINE-UNIT-MAPPING so that it doesn't depend on the
;;;           spaces being defined at macroexpansion time.  (Gallagher)
;;;  10-06-87 Added ADD-UNIT-TO-SPACE and DELETE-UNIT-FROM-SPACE.  (Gallagher)
;;;  08-08-88 Added an optional no-error-p argument to ADD-UNIT-TO-SPACE and
;;;           DELETE-UNIT-FROM-SPACE.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :USE '(lisp umass-extended-lisp))

(export '(define-unit-mapping
	  add-unit-to-space
	  delete-unit-from-space))

(use-package '(umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))



;;;; --------------------------------------------------------------------------
;;;;   DEFINE-UNIT-MAPPING
;;;; --------------------------------------------------------------------------


(defmacro DEFINE-UNIT-MAPPING (units spaces &rest args)
  
  "DEFINE-UNIT-MAPPING units spaces [documentation] &KEY indexes index-structure

   Defines how to store UNITS on SPACES.  INDEXES is a list of indexes
   from the unit.  INDEX-STRUCTURE defines the structure of the storage
   that is set up."
  
  ;; The spaces may not be defined during macroexpansion time so any
  ;; access to the space object (a structure of type space-type) must
  ;; be done at runtime.  This shouldn't be too wasteful because unit
  ;; mappings aren't defined frequently.  (At best they'll be defined
  ;; only once at load time; at worst, they'll be redefined for each
  ;; instantiate-bb-db.)

  (setf units (assure-list units))
  (setf spaces (assure-list spaces))

  (let ((documentation (if (stringp (car args))
			   (pop args)
			   "No Documentation Supplied.")))
    (with-keywords-bound ((indexes index-structure)
			  args
			  "~s is not a valid keyword for DEFINE-UNIT-MAPPING.")
      
      ;; All indexes are represented internally as keywords so
      ;; make them keywords here...
      (setf indexes (preprocess-indexes indexes))
      (setf index-structure (keyword-first-elements index-structure))

      ;; For those units that are defined, make sure that they have
      ;; all the requested indexes.
      (check-unit-indexes units indexes t)

      ;; HACK::
      ;; Allow BB/Space paths to be able to specify the mappings for
      ;; particular instances.
      ;; E.g.,  (define-unit-mapping (hyp) ((blackboard node-blackboards 0 hyp sl)))
      (when (consp (first spaces))
        (return-from define-unit-mapping
          (define-unit-mapping-for-space-path
            units spaces documentation indexes index-structure)))

      `(progn

	 ;; Make sure that each unit type has all the requested indexes
	 (check-unit-indexes ',units ',indexes)

	 ;; The two arguments, INDEXES and INDEX-STRUCTURE, must name the
	 ;; same set of indexes.
	 (check-indexes-arguments ',indexes ',index-structure)

	 ;; Make sure all the spaces have the same dimensionality.
	 (check-similar-spaces ',spaces ',units)

	 (let ((space-dimensions (space.dimensions (get-space ',(car spaces))))
	       (umi-list nil))

	   (dolist (index-spec ',index-structure)
	     (push (create-umi index-spec space-dimensions ',spaces)
		   umi-list))

	   (dolist (space ',spaces)
	     (check-and-add-unit-mapping
	       space ',units ',documentation ',indexes umi-list))

	   ',units)))))


(defun update-unit-mappings (dimension unit-mappings
			     &aux (dimension-name (dimension.name dimension)))

  ;; Can't clobber the existing mappings because they may be in use.

  (flet ((maybe-create-new-mapping (old-mapping)
	   (cond ((not (find dimension-name
			     (unit-mapping.indexes old-mapping)
			     :test #'member-eq))
		  old-mapping)
		 (t
		  (let* ((new-mapping (if (unit-mapping.in-use old-mapping)
					  (copy-unit-mapping old-mapping)
					  old-mapping))
			 (old-umi (find dimension-name
					(unit-mapping.umi-list new-mapping)
					:key #'umi.name :test #'string=))
			 (new-umi (create-umi-internal
				    dimension-name
				    (umi.bucket-specs old-umi)
				    (umi.type old-umi)
				    dimension)))
		    (setf (unit-mapping.umi-list new-mapping)
			  (substitute new-umi
				      old-umi
				      (unit-mapping.umi-list new-mapping)
				      :test #'eq))
		    new-mapping)))))

    (mapcar #'maybe-create-new-mapping unit-mappings)))



(defun create-umi (index-spec space-dimensions spaces)

  (let* ((index-name (first index-spec))
	 (mapping-type (second index-spec))
	 (bucket-specs (cddr index-spec))
	 (dimension (find index-name space-dimensions
			  :key #'dimension.name :test #'string=)))

    (error-unless dimension
       "~a is not a dimension of these spaces ~s." index-name spaces)

    (create-umi-internal index-name bucket-specs mapping-type dimension)))


(defun create-umi-internal (index-name bucket-specs mapping-type dimension)

  (let (umi)

    (check-dimension-compatibility index-name dimension mapping-type)
    (setf umi (make-unit-mapping-index :name index-name
				       :type mapping-type
				       :bucket-specs bucket-specs))

    (case mapping-type
      (:subranges
       (build-subrange-buckets umi bucket-specs dimension))
      (:groups
       (build-groups umi bucket-specs dimension))
      (t (error
	   "Unknown index structure type, ~s, while defining a unit-mapping.~@
            The index structure type must be one of :SUBRANGES or :GROUPS.~@
            The offending form is (~a ~s ~s)."
	   mapping-type index-name mapping-type bucket-specs)))

    umi))


(defun preprocess-indexes (indexes)
  ;; Make sure each element is a list (even if only a singleton
  ;; list) of keywords.
  (flet ((keyword-or-error (x)
           (error-unless (symbolp x)
              "Error in DEFINE-UNIT-MAPPING:~@
               Indexes must be symbols.  ~s is not a symbol.~@
               The :indexes argument is ~s."
              x indexes)
           (form-keyword x)))
    (mapcar #'(lambda (index)
                (if (listp index)
                    (mapcar #'keyword-or-error index)
                    (list (keyword-or-error index))))
            indexes)))


(defun check-similar-spaces (spaces units)
  "Signals an error unless all the spaces have the same dimensionality
   and you can store UNITS on all of them."
  (let ((space-dimensions (space.dimensions (get-space (car spaces))))
	the-space)
    (dolist (s spaces)
      (setf the-space (get-space s))
      (error-unless (equalp space-dimensions (space.dimensions the-space))
	 "All the spaces must have the same dimensions: ~s." spaces)
      (error-unless (subsetp units (space.units the-space)
			     :test #'x-unit-subtypep)
	 "The space, ~s, can't store these unit types: ~s.~@
          The units types that can be stored on ~s are ~s."
	  s
	  (set-difference units (space.units the-space) :test #'x-unit-subtypep)
	  s
	  (space.units the-space)))))


(defun check-and-add-unit-mapping (space units documentation indexes umi-list)
  (setf space (get-space space))
  (let* ((mappings (space.unit-mappings space))
	 (equal-count (count units mappings :test #'equal :key #'unit-mapping.units))
	 (intersect-count (count units mappings
				 :test #'overlapp :key #'unit-mapping.units))
	 (new-unit-mapping (make-unit-mapping
			     :units         units
			     :spaces        (list space) ; Must be a list
			     :documentation documentation
			     :indexes       indexes
			     :umi-list      umi-list)))
    (cond ((= 1 equal-count)
	   (setf (space.unit-mappings space)
		 (nsubstitute
		   new-unit-mapping units (space.unit-mappings space)
		   :test #'equal :key #'unit-mapping.units)))
	  ((> equal-count 1)
	   ;; This should never happen.
	   (error "Redundant unit mappings for space ~s,~% units = ~s."
		  (space.name space) units))
	  ((zerop intersect-count)
	   (push new-unit-mapping (space.unit-mappings space)))
	  (t
	   (error "Incompatible redefinition of a unit-mapping.~@
                   There are ~d unit-mappings for the space ~s:~%~{~s~^, ~}.~@
                   The new unit mapping is ~s."
		  (length mappings)
		  (space.name space)
		  (mapcar #'unit-mapping.units mappings)
		  units)))
    nil))


(defun check-unit-indexes (units indexes &optional ignore-undefined-units)

  "Ensure that all the indexes for a single array are all composite
   or all scalar (non-composite) indexes.  Both arguments are lists
   of symbols.  UNITS is a list of unit types.  INDEXES is a list of
   lists of index names (e.g., ((:time) (:x :y))).  If
   IGNORE-UNDEFINED-UNITS is true then no error will be signalled if
   a unit is not defined."

  (let (unit-index-list)

    (flet ((index-exists-p (unit index)
	     ;; Returns true if `index' is one of the indexes for `unit'.
	     (let ((unit-index (find index unit-index-list
				     :key #'unit-index.name :test #'eq)))
	       (error-unless unit-index
		  "~a is not a dimensional index in ~s.~@
                   This problem occurred in DEFINE-UNIT-MAPPING."
		  index unit)
	       t))
	   (composite-index-p (index)
	     ;; Returns true if `index' is a composite index (for unit
	     ;; which is the unit from the dolist below).
	     (composite-index-structure-p
		(unit-index.index-structure
		  (find index unit-index-list :key #'unit-index.name)))))

    (dolist (unit units t)
      (cond
	((unit-type-p unit)
	 (setf unit-index-list (unit.d-indexes (get-unit-description unit)))
	 (dolist (ii indexes)
	   ;; Check that all the requested indexes exist.
	   (mapc #'(lambda (index) (index-exists-p unit index)) ii)
	   (and (some #'composite-index-p ii)
		(not (every #'composite-index-p ii))
		(error "You can't mix composite and non-composite indexes~@
                        like this in define-unit-mapping.~@
                        For unit ~s, indexes ~s."
		       unit ii))))
	((not ignore-undefined-units)
	 (error "~s is not a unit type." unit))
	(t nil))))))


(defun check-indexes-arguments (indexes index-structure)

  (let ((indexes-1 (flatten indexes))
	(indexes-2 (mapcar #'car index-structure)))
    (error-unless (subsetp indexes-1 indexes-2)
       "The :INDEXES argument to DEFINE-UNIT-MAPPING has some~@
        indexes that don't appear in the :INDEX-STRUCTURE argument.~@
        :INDEXES         = ~s~@
        :INDEX-STRUCTURE = ~s"
       indexes index-structure)
    (error-unless (subsetp indexes-2 indexes-1)
       "The :INDEX-STRUCTURE argument to DEFINE-UNIT-MAPPING has some~@
        indexes that don't appear in the :INDEXES argument.~@
        :INDEXES         = ~s~@
        :INDEX-STRUCTURE = ~s"
       indexes index-structure)))


(defun check-dimension-compatibility (name dimension mapping-type)
  "Checks that the space dimension type and the mapping type are
   compatible.  (E.g. :enumerated doesn't work with :subranges.)
   Signals an error if an incompatability is found."
  (ecase (dimension.type dimension)
    (:ordered
     (unless (eq mapping-type :subranges)
       (error "The type of the ~s dimension is :ORDERED,~@
               which can't be mapped by ~s."
	      name mapping-type)))
    (:enumerated
     (unless (eq mapping-type :groups)
       (error "The type of the ~s dimension is :ENUMERATED,~@
               which can't be mapped by ~s."
	      name mapping-type)))))


;;; HACK::

(defun define-unit-mapping-for-space-path
       (units space-paths documentation indexes index-structure)

  (let ((the-spaces (remove-duplicates
                      (mapcar #'get-space-name-from-path space-paths))))
    (setf space-paths (mapcar #'add-replications space-paths))

    `(progn
       (check-unit-indexes ',units ',indexes)
       (check-indexes-arguments ',indexes ',index-structure)
       (check-similar-spaces ',the-spaces ',units)
       (let ((space-dimensions (space.dimensions (get-space ',(car the-spaces))))
             (umi-list nil))
         
         (dolist (index-spec ',index-structure)
           (push (create-umi index-spec space-dimensions ',the-spaces)
                 umi-list))
         
         (dolist (space-path ',space-paths)
           (check-and-add-unit-mapping-for-space-path
             space-path ',units ',documentation ',indexes umi-list))
         
         ',units))))
  
;;; HACK::

(defun check-and-add-unit-mapping-for-space-path
       (space-path units documentation indexes umi-list)
  
  (let* ((space (get-space-name-from-path space-path))
         (pair (assoc space-path *unit-mappings-for-space-paths*
                        :test #'equal))
         (mappings (cdr pair))
	 (equal-count (count units mappings :test #'equal :key #'unit-mapping.units))
	 (intersect-count (count units mappings
				 :test #'overlapp :key #'unit-mapping.units))
	 (new-unit-mapping (make-unit-mapping
			     :units         units
			     :spaces        (list space) ; Must be a list
			     :documentation documentation
			     :indexes       indexes
			     :umi-list      umi-list)))

    (cond ((= 1 equal-count)
	   (setf (cdr pair)
		 (nsubstitute
		   new-unit-mapping units (cdr pair)
		   :test #'equal :key #'unit-mapping.units)))
	  ((> equal-count 1)
	   ;; This should never happen.
	   (error "Redundant unit mappings for path ~s,~% units = ~s."
                  space-path units))
	  ((zerop intersect-count)
           (if pair
               (push new-unit-mapping (cdr pair))
               (push-acons *unit-mappings-for-space-paths*
                           space-path
                           (list new-unit-mapping))))
	  (t
	   (error "Incompatible redefinition of a unit-mapping.~@
                   There are ~d unit-mappings for the path ~s:~%~{~s~^, ~}.~@
                   The new unit mapping is ~s."
		  (length mappings)
                  space-path
		  (mapcar #'unit-mapping.units mappings)
		  units)))
    nil))


(defun unit-mapping-specified-p (units spaces)

  "UNIT-MAPPING-SPECIFIED-P units spaces

   Returns true if there is one unit mapping which maps
   UNITS onto SPACES."

  (let ((unit-mapping (find units (space.unit-mappings (get-space (first spaces)))
			    :key #'unit-mapping.units :test #'equal)))
    (and unit-mapping
	 (equal spaces (unit-mapping.spaces unit-mapping)))))

(defun delete-unit-mappings (spaces)

  "DELETE-UNIT-MAPPING units spaces

   Deletes all the unit mappings for SPACES, which may be a single space
   name or a list of space names."

  (dolist-or-atom (space spaces)
    (setf (space.unit-mappings (get-space space)) nil)))


(defun build-subrange-buckets (umi index-spec dimension)
  "Build the buckets for an index.  UMI is a unit-mapping-index. This
   function fills in the BUCKETS and ARRAY-SIZE slots of UMI.  INDEX-SPEC
   is the list of bucket descriptions from define-unit-mapping.
   DIMENSION is a space-dimesion."

  (let* ((buckets nil)
         (space-start (%range-start (odim.range dimension)))
         (space-end (%range-end (odim.range dimension)))
         (last space-start)
         (offset 0)
         start end options width)

    (flet ((add-bucket (b-start b-end b-width b-count)
            (push (make-bucket :start b-start :end b-end
                               :offset offset :width b-width :count b-count)
                   buckets)
             (setf last b-end)
             (incf offset b-count))
           (error! (format-string &rest args)
             (error "While building buckets for dimension ~a.~%~?~@
                     The entire index spec is ~s."
                  (dimension.name dimension) format-string args index-spec)))
                    
      (when index-spec

        ;; If the first bucket doesn't start at the beginning of the
        ;; space range then create an initial bucket.
        (setf start (caar index-spec))
        (cond ((or (eq start :start) (= start space-start))
               ;; No need to create the initial bucket:
               )
              ((not (<= space-start start space-end))
               (error! "~d is out of range.  The range for ~a is ~d - ~d."
                       start (dimension.name dimension) space-start space-end))
              (t (add-bucket space-start start (- start space-start) 1)))

        (dolist (bucket-spec index-spec)
          (setf start (first bucket-spec))
          (when (eq start :start) (setf start space-start))
          (setf end (second bucket-spec))
          (when (eq end :end) (setf end space-end))
          (unless (= last start)
            (error "Subranges are not contiguous in index spec: ~s." index-spec))
          (unless (< start end)
            (error "Ill-formed subrange: (~d ~s)~@ in~%~s." start end index-spec))
          (setf options (cddr bucket-spec))
          (setf width (cadr (assoc :width options :test #'eq)))
          (cond ((null width)
                 (add-bucket start end (- end start) 1))
                (t (multiple-value-bind (q r)
                       (truncate (- end start) width)
                     (add-bucket start (- end r) width q)
                     (unless (zerop r)
                       (add-bucket (- end r) end r 1))))))

        ;; If the last bucket doesn't end at the end of the space
        ;; range then create a final bucket.
        (cond ((= last space-end))
              ((> last space-end)
               (error! "~d is out of range.  The range for ~a is ~d - ~d."
                       start (dimension.name dimension) space-start space-end))
              (t (add-bucket last space-end (- space-end last) 1))))
          

      (setf (umi.buckets umi) (nreverse buckets))
      (setf (umi.array-size umi) offset))))


(defun build-groups (umi groups dimension)
  "UMI is a unit-mapping.index.  This function fills in the BUCKETS and
   ARRAY-SIZE slots of UMI.  GROUPS is either nil or a list of groups
   of labels.  Each group must be a list (i.e., groups will nil or a
   list of lists).  This is because a label itself can be a list."
  (let* ((labelset (edim.labelset dimension))
	 (label-test (edim.test dimension))
	 (remaining-labels (copy-list labelset))
	 (array-index 0)
	 (table (make-label-index-table labelset label-test)))
    (dolist (group groups)
      (setf group (assure-list group))
      (unless (subsetp group labelset :test label-test)
	(let ((difference (set-difference group labelset :test label-test)))
	  (error "~s ~[is not a label~:;are not labels~] for the ~s dimension.~@
                  The labels are ~s."
		 difference (1- (length difference)) (umi.name umi) labelset)))
      (unless (subsetp group remaining-labels :test label-test)
	(let ((difference (set-difference group remaining-labels :test label-test)))
	  (error "The ~[label ~s was~:;labels ~s were~] mentioned twice ~
                  for the dimension ~s in DEFINE-UNIT-MAPPING."
		 (1- (length difference)) difference (umi.name umi))))	  
      (setf remaining-labels
	    (nset-difference remaining-labels group :test label-test))
      (dolist (label group)
	(setf (get-label-index table label) array-index))
      (incf array-index))
    (dolist (label remaining-labels)
      (setf (get-label-index table label) array-index)
      (incf array-index))
    (setf (umi.buckets umi) table)
    (setf (umi.array-size umi) array-index)))


;;;; --------------------------------------------------------------------------
;;;;   Storing Units on Spaces
;;;; --------------------------------------------------------------------------


(defun add-unit-to-space (unit-instance path-structure &optional no-error-p)

  "ADD-UNIT-TO-SPACE unit-instance path-structure &optional no-error-p

   Adds UNIT-INSTANCE to the space(s) represented by PATH-STRUCTURE.
   If NO-ERROR-P is true then no error will be signalled if UNIT-INSTANCE
   is already on the space(s)."

  (dolist-or-atom (path path-structure)
    (let ((space-instance (get-space-instance-from-path-structure path)))
      (cond ((not (member space-instance (basic-unit.%%space-instances%% unit-instance)
                          :test #'eq))
             (insert-unit-on-space-instance unit-instance space-instance))
            (no-error-p nil)
            (t
             (cerror "Ignore this error and continue."
                     "~s is already stored on this space: ~s."
                     (-> unit-instance)
                     (get-path-from-path-structure path)))))))


(defun insert-unit-on-space-instance (unit space-instance)

  "Add UNIT to each of the appropriate arrays of SPACE-INSTANCE at
   the appropriate places in those arrays."

  (when (member space-instance
		(basic-unit.%%space-instances%% unit)
		:test #'eq)
    (cerror "Ignore this error and continue."
	    "Attempt to store ~s on the space ~s which it is already on."
	    (-> unit) (get-path-from-space-instance space-instance))
    (return-from insert-unit-on-space-instance unit))

  (let* ((start-runtime (gbb-runtime))
         (unit-type (unit-type-of unit))
         (description (get-unit-description unit-type))
         (unit-index-list (unit.d-indexes description))
         (array-data (space-instance.data space-instance))
         (storage-entry (find unit-type array-data
                              :key #'(lambda (entry)
                                       (unit-mapping.units
                                         (storage-entry.unit-mapping entry)))
                              :test #'member-eq))
         (path-struct (get-path-structure-from-space-instance space-instance))
         unit-mapping)

    (check-unit/space-compatibility
      unit unit-type space-instance array-data storage-entry)

    (setf unit-mapping (storage-entry.unit-mapping storage-entry))

    (cond ((unstructured-unit-mapping-p unit-mapping)
	   (push unit (cdr storage-entry)))
	  (t
	   (dolist (index-names (unit-mapping.indexes unit-mapping))
	     (insert-unit-onto-space-array
	       unit index-names unit-index-list
	       (unit-mapping.umi-list unit-mapping)
	       (cdr storage-entry)))))

    ;; Save the space instance with the unit instance.
    (push space-instance (basic-unit.%%space-instances%% unit))

    (update-runtime *insert-unit-on-space-meter* start-runtime)
    (update-count *insert-unit-on-space-count*)
    (dolist (fn *insert-unit-hook-functions*)
      (funcall fn unit space-instance))
    (dolist (fn (get-unit-events :add-to-space-events description))
      (funcall fn unit path-struct))
    unit))


(defun check-unit/space-compatibility (unit-instance unit-type space-instance
                                       array-data storage-entry)
  "Checks that `unit-instance' can be stored on the space
   and that the indexes are in bounds."

  (error-unless storage-entry
     "Attempt to store ~s (a unit of type ~s)~@
      on the space instance ~s.~@
      This space can only store these unit type(s):~{~{~%~8@t~s~}~}"
     (-> unit-instance) unit-type (get-path-from-space-instance space-instance)
     (mapcar #'(lambda (entry)
                 (unit-mapping.units (storage-entry.unit-mapping entry)))
             array-data))
  ;; Check that the indexes are within bounds
  (check-space-bounds unit-instance (space-instance.space space-instance)))


;;;; --------------------------------------------------------------------------
;;;;   Moving Units on Spaces
;;;; --------------------------------------------------------------------------

(defun move-unit-on-space-instance (unit space-instance changed-indexes
                                    old-index-data)

  "Alter the location of UNIT on SPACE-INSTANCE.

   Arguments:
     UNIT             - a unit instance
     SPACE-INSTANCE   - a space instance
     CHANGED-INDEXES  - a list of index names (keywords) that have changed
     OLD-INDEX-DATA   - the old index data"

  (let* ((start-runtime (gbb-runtime))
         (space (space-instance.space space-instance))
         (type (unit-type-of unit))
         (unit-index-list (unit.d-indexes (get-unit-description type)))
         (array-data (space-instance.data space-instance))
         (storage-entry (find type array-data
                              :key #'(lambda (entry)
                                       (unit-mapping.units
                                         (storage-entry.unit-mapping entry)))
                              :test #'member-eq))
         (unit-mapping (storage-entry.unit-mapping storage-entry)))

    (check-space-bounds unit space)

    ;; There is no need to do anything if the unit mapping
    ;; is an unstructured unit mapping.
    (unless (unstructured-unit-mapping-p unit-mapping)
      (dolist (index-names (unit-mapping.indexes unit-mapping))
        (when (intersectp index-names changed-indexes)
          (without-metering
            (move-unit-on-space-array
              unit index-names unit-index-list
              (unit-mapping.umi-list unit-mapping)
              (cdr storage-entry)
              old-index-data)))))

    (update-runtime *move-unit-on-space-meter* start-runtime)
    (update-count *move-unit-on-space-count*)
    (dolist (fn *move-unit-hook-functions*)
      (funcall fn unit space-instance))

    unit))


(defun move-unit-on-space-array (unit
				 index-names
				 unit-index-list
				 umi-list
				 index-array-alist
				 old-index-data)

  ;; Args:
  ;;  unit	 	- A unit instance
  ;;  index-names  	- List of index names (e.g. (time x y)) that are stored
  ;;                      together in the unit-mapping
  ;;  unit-index-list	- List of unit-indexes from the unit
  ;;  umi-list        	- List of unit-mapping-indexes
  ;;  index-array-alist - Alist of index names and the arrays that the index
  ;;			  refers to
  ;;  old-index-data    - Old index data for the unit -- this is tells us where
  ;;                      the unit is moving from
  ;;
  ;; The strategy:
  ;; INDEX-NAMES is a list of symbols.  (The fact that it is a list is ensured
  ;; by define-unit-mapping.)  It represents the space dimensions of one
  ;; array.  I build up a list (stored in the variable REGIONS) of the form
  ;; (((x1-min . x1-max) (x2-min . x2-max) ...)
  ;;  ((y1-min . y1-max) ...)
  ;;  ...)
  ;; which gives the limits of the subarrays that this unit falls into.  This
  ;; list can be thought of as a matrix where each sublist is column describing
  ;; one dimension:
  ;;
  ;;      --- X ---           --- Y ---
  ;;  (x1-min . x1-max)   (y1-min . y1-max)   ...
  ;;  (x2-min . x2-max)   (y2-min . y2-max)   ...
  ;;  ...                 ...
  ;;
  ;; Now, each row describes a single subarray which corresponds to the values
  ;; of one composite index element.

  (let ((array (get-space-array index-names index-array-alist))
        (new-regions nil)
	(new-index-data (basic-unit.%%indexes%% unit))
        (old-regions nil)
        unit-index umi new-slot old-slot)

    (dolist (idx-1 index-names)
      (setf unit-index (find idx-1 unit-index-list :key #'unit-index.name))
      (setf umi (find idx-1 umi-list :key #'umi.name))
      (setf new-slot (nth (unit-index.offset unit-index) new-index-data))
      (setf new-regions (compute-regions new-regions umi unit-index new-slot))
      (setf old-slot (nth (unit-index.offset unit-index) old-index-data))
      (setf old-regions (compute-regions old-regions umi unit-index old-slot)))

    ;; Quick and dirty.  Delete the unit from the array and then put
    ;; it back in the array.  If the old and new regions overlap this
    ;; is a loss.
    (dolist-slices (region old-regions)
      (do-array-region (subscripts region)
        (remove-from-array unit (apply #'aref array subscripts))))

    (dolist-slices (region new-regions)
      (do-array-region (subscripts region)
        (pushnew-onto-array unit (apply #'aref array subscripts))))))



(defun check-space-bounds (unit-instance space)

  "Signal an error if the indexes for `unit-instance' are beyond
   the bounds of the space dimensions."

  (let ((description (get-unit-description unit-instance))
	(space-dimensions (space.dimensions space))
	index-name space-dimension)
    (dolists ((unit-index (unit.d-indexes description))
	      (index-data (basic-unit.%%indexes%% unit-instance)))
      (setf index-name (unit-index.name unit-index))
      (setf space-dimension (find index-name space-dimensions
				  :key #'dimension.name :test #'eq))
      ;; Ignore indexes that aren't dimensions of this space.
      (when space-dimension
	(etypecase space-dimension
	  (ordered-dimension
	   (check-bounds-on-ordered-dimension
	     unit-instance unit-index index-data space-dimension space))
	  (enumerated-dimension
	    (check-bounds-on-enumerated-dimension
	      unit-instance unit-index index-data space-dimension space)))))))


(defun check-bounds-on-ordered-dimension
           (unit-instance unit-index index-data space-dimension space)

  (let ((index-structure (unit-index.index-structure unit-index))
        (space-start (%range-start (odim.range space-dimension)))
        (space-end (%range-end (odim.range space-dimension))))

    (macrolet ((error! (&body args)
                 `(error "Some value of the ~a index for ~s is invalid.~%~@?"
                         (unit-index.name unit-index) (-> unit-instance)
                         ,@args)))

      (flet ((check-data-point (index-data)
               ;; Check a single index data point.
               (ecase (index-element.element-type index-data)
                 (:point
                  (let ((value (index-element.point-value index-data)))
                    (unless (and (<= space-start value) (< value space-end))
                      (error! "~d is out of bounds for the space ~s.~@
                               The ~a dimension bounds are ~d to ~d."
                              value (space.name space)
                              (unit-index.name unit-index) space-start space-end))))
                 (:range
                  (let* ((range (index-element.value index-data))
                         (data-start (range-start range))
                         (data-end (range-end range)))
                    (unless (<= data-start data-end)
                      (error! "~d - ~d is not a valid range.~@
                               The lower bound must be less than or ~
                               equal to the upper bound."
                              data-start data-end))
                    (unless (<= space-start data-start data-end space-end)
                      (error! "~d - ~d is out of bounds for the space ~s.~@
                               The ~a dimension bounds are ~d to ~d."
                              data-start data-end (space.name space)
                              (unit-index.name unit-index) space-start space-end)))))))
                              
      (cond
        ;; Scalar index structure
        ((or (null index-structure)
                 (scalar-index-structure-p index-structure))
         (check-data-point index-data))
        ;; Composite index structure
        (t (mapc #'check-data-point index-data)))))))


(defun check-bounds-on-enumerated-dimension
           (unit-instance unit-index index-data space-dimension space)

  (let ((index-structure (unit-index.index-structure unit-index))
	(labels (edim.labelset space-dimension))
	(test (edim.test space-dimension)))

    (error-unless
      (cond ((or (null index-structure)
		 (scalar-index-structure-p index-structure))
	     (check-bounds-on-enumerated-data-point index-data labels test))
	    (t
	     ;; Composite index structure
	     (dolist (data index-data t)
	       (unless (check-bounds-on-enumerated-data-point data labels test)
		 (return nil)))))
       "Some value of the ~a index for ~s~@
        is not a member of the labeset for the space ~s.~@
        The labels of the ~a dimension are: ~s."
       (unit-index.name unit-index) (-> unit-instance) (space.name space)
       (unit-index.name unit-index) labels)
    t))

(defun check-bounds-on-enumerated-data-point (data labels test)

  (ecase (index-element.element-type data)
    (:label (member (index-element.point-value data)
		    labels
		    :test test))))


(defun insert-unit-onto-space-array
           (unit index-names unit-index-list umi-list index-array-alist)
  ;; Args:
  ;;  unit	 	- a unit instance
  ;;  index-names  	- list of index names (e.g. (time x y)) that are stored
  ;;                      together in the unit-mapping
  ;;  unit-index-list	- list of unit-indexes from the unit
  ;;  umi-list        	- list of unit-mapping-indexes
  ;;  index-array-alist - alist of index names and the arrays that the index
  ;;			  refers to
  ;;
  ;; The strategy:
  ;; INDEX-NAMES is a list of symbols.  (The fact that it is a list is ensured
  ;; by define-unit-mapping.)  It represents the space dimensions of one
  ;; array.  I build up a list (stored in the variable REGIONS) of the form
  ;; (((x1-min . x1-max) (x2-min . x2-max) ...)
  ;;  ((y1-min . y1-max) ...)
  ;;  ...)
  ;; which gives the limits of the subarrays that this unit falls into.  This
  ;; list can be thought of as a matrix where each sublist is column describing
  ;; one dimension:
  ;;
  ;;      --- X ---           --- Y ---
  ;;  (x1-min . x1-max)   (y1-min . y1-max)   ...
  ;;  (x2-min . x2-max)   (y2-min . y2-max)   ...
  ;;  ...                 ...
  ;;
  ;; Now, each row describes a single subarray which corresponds to the values
  ;; of one composite index element.

  (let ((array (get-space-array index-names index-array-alist))
        (regions nil)
	(index-data (basic-unit.%%indexes%% unit))
        unit-index slot umi)

    (dolist (idx-1 index-names)
      (setf unit-index (find idx-1 unit-index-list :key #'unit-index.name))
      (setf umi (find idx-1 umi-list :key #'umi.name))
      (setf slot (nth (unit-index.offset unit-index) index-data))
      (setf regions (compute-regions regions umi unit-index slot)))

    (dolist-slices (region regions)
      (do-array-region (subscripts region)
        (pushnew-onto-array unit (apply #'aref array subscripts))))))


(defun compute-regions (regions umi unit-index slot)
  ;; Compute the actual array indexes to access the array elements based
  ;; on the values of the unit indexes.  REGIONS is a list of previously
  ;; computed regions.  The new region pairs are stuck onto the end of
  ;; REGIONS (with NCONC) to maintain the order of array indexes.  SLOT is
  ;; the slot value from the unit (as it has been stored by
  ;; compute-indexes).
  (let ((element-type (unit-index.element-type unit-index))
	(index-structure (unit-index.index-structure unit-index))
	;; The buckets slot is either a list of buckets (for ordered
	;; dimensions) or a table of labels (for enumerated dimensions).
        (bucket-info (umi.buckets umi))
        (region-list nil))
    (ecase (umi.type umi)
       (:groups
	(cond ((null index-structure)
	       (push (get-label-index bucket-info
				      (index-element.value slot))
		     region-list))
	      ((composite-index-structure-p index-structure)
	       (dolist (element slot)
		 (push (get-label-index bucket-info
					(index-element.value element))
		       region-list)))
	      ((eq element-type :label)
	       (push (get-label-index bucket-info
				      (index-element.value slot))
		     region-list))
	      (t (error ":ENUMERATED dimensions must be :LABELs."))))
       (:subranges
	(cond ((null index-structure)
	       (push (compute-region slot bucket-info)
		     region-list))
	      ((composite-index-structure-p index-structure)
	       (dolist (element slot)
		 (push (compute-region element bucket-info)
		       region-list)))
	      (t (push (compute-region slot bucket-info)
		       region-list)))))

    (nconc regions (list (nreverse region-list)))))


;;; ---------------------------------------------------------------------------
;;;   Deleting Units from Spaces
;;; ---------------------------------------------------------------------------

(defun delete-unit-from-space (unit-instance path-structure &optional no-error-p)

  "DELETE-UNIT-FROM-SPACE unit-instance path-structure &optional no-error-p

   Removes UNIT-INSTANCE from the space(s) represented by PATH-STRUCTURE.
   If NO-ERROR-P is true then no error will be signalled if UNIT-INSTANCE
   isn't, in fact, on the space(s)."

  (dolist-or-atom (path path-structure)
    (let ((space-instance (get-space-instance-from-path-structure path)))
      (cond ((member space-instance (basic-unit.%%space-instances%% unit-instance)
		    :test #'eq)
	     (delete-unit-from-space-instance unit-instance space-instance))
            (no-error-p nil)
	    (t (cerror "Ignore this error and continue."
		       "~s is not stored on this space: ~s."
		       (-> unit-instance)
		       (get-path-from-path-structure path)))))))


(defun delete-unit-from-space-instance (unit space-instance)
  ;; Delete unit from each of the appropriate arrays of space-instance at the
  ;; appropriate places in those arrays.

  (let* ((start-runtime (gbb-runtime))
         (type (unit-type-of unit))
         (description (get-unit-description type))
         (unit-index-list (unit.d-indexes description))
         (array-data (space-instance.data space-instance))
         (storage-entry (find type array-data
                              :key #'(lambda (entry)
                                       (unit-mapping.units
                                         (storage-entry.unit-mapping entry)))
                              :test #'member-eq))
         (unit-mapping (storage-entry.unit-mapping storage-entry))
         (path-struct (get-path-structure-from-space-instance space-instance)))

    (dolist (fn (get-unit-events :delete-from-space-events description))
      (funcall fn unit path-struct))
    (cond ((unstructured-unit-mapping-p unit-mapping)
	   (setf (cdr storage-entry)
		 (delete unit (cdr storage-entry) :test #'eq)))
	  (t
	   (dolist (index-names (unit-mapping.indexes unit-mapping))
	     (delete-unit-from-space-array
	       unit index-names unit-index-list
	       (unit-mapping.umi-list unit-mapping)
	       (cdr storage-entry)))))
    ;; Remove the space instance from the list sotred with the unit instance.
    (setf (basic-unit.%%space-instances%% unit)
	  (delete space-instance (basic-unit.%%space-instances%% unit)
		  :test #'eq))
    (update-runtime *delete-unit-from-space-meter* start-runtime)
    (update-count *delete-unit-from-space-count*)
    (dolist (fn *delete-unit-hook-functions*)
      (funcall fn unit space-instance))
    unit))


(defun delete-unit-from-space-array
           (unit index-names unit-index-list umi-list index-array-alist)

  ;; This code is very similar to insert-unit-onto-space-array.

  (let ((array (get-space-array index-names index-array-alist))
        (regions nil)
	(index-data (basic-unit.%%indexes%% unit))
        unit-index slot umi)

    (dolist (idx-1 index-names)
      (setf unit-index (find idx-1 unit-index-list :key #'unit-index.name))
      (setf umi (find idx-1 umi-list :key #'umi.name))
      (setf slot (nth (unit-index.offset unit-index) index-data))
      (setf regions (compute-regions regions umi unit-index slot)))

    (do* ((len (length regions))
          (keep-track regions)
          (region (make-list len)))
         ((null (car keep-track)))
      (dotimes (i len)
        (setf (nth i region) (car (nth i keep-track)))
        (setf (nth i keep-track) (cdr (nth i keep-track))))
      (do-array-region (subscripts region)
        (remove-from-array unit (apply #'aref array subscripts))))))


(defun compute-region (element bucket-list)
  ;; Compute the array indexes for an element of an index structure.
  ;; ELEMENT is an index element (i.e., a list like (:range 5 10).
  (let ((type (index-element.element-type element))
	(data (index-element.value element))
	(first-point nil)
	(last-point nil))
    (ecase type
       (:point
	(setf first-point data))
       (:range
	(setf first-point (first data))
	(setf last-point (second data))))
    (bucket-indexes first-point last-point bucket-list)))


(defun get-space-array (index-names index-array-alist)
  "Check that INDEX-NAMES (which is a list of index names) all refer to
   the same array and return the array.  If this does get an error then
   GBB's data structures are corrupted somehow."
  (cdr
    (or (find index-names index-array-alist :key #'car :test #'subsetp)
	(error "Space data arrays don't match (~s ~s)."
	       index-names index-array-alist))))


(defun bucket-indexes (start end bucket-list &optional force-list)

  "BUCKET-INDEXES start end bucket-list

   Map START and END, which are values along some dimension, to array
   indexes.  If END is nil then just find the array index for START.
   Returns a cons whose car is the inclusive start index and whose cdr
   is the exclusive end index, or just a single number which is
   equivalent to (n .  n+1).  If FORCE-LIST is true then always return
   a cons."

  (flet ((array-index (n bucket)
           (+ (bucket.offset bucket)
              (floor (- n (bucket.start bucket))
                     (bucket.width bucket)))))

    (let ((start-index nil)
          (end-index nil)
          (first-bucket (first bucket-list)))
      
      (error-when (< (or end start) (bucket.start first-bucket))
         "Dimension value out of bounds.~@
           START and END are before the first bucket.")
      
      ;; Find the start index.
      (setf start-index
            (if (< start (bucket.start first-bucket))
                0
                (dolist (bucket bucket-list nil)
                  (when (> (bucket.end bucket) start)
                    (return (array-index start bucket))))))
      (error-unless start-index
         "Dimension value out of bounds.~@
           START and END are after the last bucket.")
      
      ;; Find the end index.
      (cond
        ((null end)
         (setf end-index (1+ start-index)))
        (t
         (setf end-index
               (dolist (bucket bucket-list nil)
                 (when (> (bucket.end bucket) end) 
                   (return (1+ (array-index end bucket))))))
         (unless end-index
           (let ((last-bucket (first (last bucket-list))))
             (setf end-index (+ (bucket.offset last-bucket)
                                (bucket.count last-bucket)))))))
      
      (if (and (not force-list) (= end-index (1+ start-index)))
          start-index
          (cons start-index end-index)))))


(defun unstructured-unit-mapping-p (unit-mapping)
  (null (unit-mapping.indexes unit-mapping)))

(defun make-unstructured-unit-mapping (units spaces
				       &optional
				       (doc "No documentation supplied."))
  (make-unit-mapping
    :documentation doc
    :units (assure-list units)
    :spaces (mapcar #'get-space (assure-list spaces))
    :indexes nil
    :umi-list nil))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
