;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL]METER.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  12:44:51 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                      GBB METER SUPPORT FUNCTIONS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  02-19-88 File Created.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :use '(lisp umass-extended-lisp))

(export '(reset-meters report-meters *metering-enabled*))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Metering Stuff
;;;; --------------------------------------------------------------------------


(defvar *metering-enabled* nil
  "This variable controls whether or not GBB metering information will
   will be recorded.")

(defvar *simple-meters* nil)
(defvar *simple-counters* nil)

(defmacro define-simple-meter (meter &optional (documentation "This is a meter."))
  `(progn
     (defvar ,meter 0 ,documentation)
     (pushnew ',meter *simple-meters*)
     ',meter))

(defmacro define-simple-counter (counter &optional (documentation "This is a counter."))
  `(progn
     (defvar ,counter 0 ,documentation)
     (pushnew ',counter *simple-counters*)
     ',counter))

(defmacro update-runtime (meter start-runtime &optional key)

  "UPDATE-RUNTIME meter start-runtime &optional key

   Increment METER (which may be any place form) by the difference
   between the current runtime and START-RUNTIME."

  (let ((runtime-form
          #+SYMBOLICS          `(time:time-difference (gbb-runtime) ,start-runtime)
          #+TI                 `(time:time-difference (gbb-runtime) ,start-runtime)
          #-(or SYMBOLICS TI)  `(- (gbb-runtime) ,start-runtime)
          ))
    `(when *metering-enabled*
       ,(if (null key)
            `(incf ,meter ,runtime-form)
            (let ((pair (gensym)))
              `(let ((,pair (assoc ,key ,meter :test #'eq)))
                 (if ,pair
                     (incf (cdr ,pair) ,runtime-form)
                     (push-acons ,meter ,key ,runtime-form))))))))


(defmacro continue-runtime (meter start-runtime &optional key)

  "CONTINUE-RUNTIME meter start-runtime &optional key

   Increment METER (which may be any place form) by the difference
   between the current runtime and START-RUNTIME.  In addition, 
   START-RUNTIME is set to the current runtime."

  (let ((runtime-form
          #+SYMBOLICS          `(time:time-difference (gbb-runtime) ,start-runtime)
          #+TI                 `(time:time-difference (gbb-runtime) ,start-runtime)
          #-(or SYMBOLICS TI)  `(- (gbb-runtime) ,start-runtime)
          ))
    `(when *metering-enabled*
       ,(if (null key)
            `(incf ,meter ,runtime-form)
            (let ((pair (gensym)))
              `(let ((,pair (assoc ,key ,meter :test #'eq)))
                 (if ,pair
                     (incf (cdr ,pair) ,runtime-form)
                     (push-acons ,meter ,key ,runtime-form)))))
       (setf ,start-runtime (gbb-runtime)))))


(defmacro update-count (meter &optional key)
  `(when *metering-enabled*
     ,(if (null key)
          `(incf ,meter)
          (let ((pair (gensym)))
            `(let ((,pair (assoc ,key ,meter :test #'eq)))
               (if ,pair
                   (incf (cdr ,pair))
                   (push-acons ,meter ,key 1)))))))

(defmacro without-metering (&body body)
  "This simply turns off the >>recording<< of meter information during
   the execution of body.  It does not run without accumulating runtime
   on any meters that are outside the body."
  `(let ((*metering-enabled* nil))
     ,@body))


;;; Timer Issues:
;;;
;;; On the Explorer the function TIME::FIXNUM-MICROSECOND-TIME wraps
;;; around every 17 seconds.  This is much too small for accurate timing.
;;; On the 3600 TIME::FIXNUM-MICROSECOND-TIME wraps around every half
;;; hour.  This is is sufficient, but on longer runs would create bignums.
;;; However if microsecond precision is necessary then TIME-DIFFERENCE
;;; (used in UPDATE-RUNTIME, above) will work with the values returned by
;;; either TI's or SYMBOLICS' TIME::FIXNUM-MICROSECOND-TIME

(defun gbb-runtime ()
  (cond (*metering-enabled*
         #+SYMBOLICS  (zl::time)
         #+TI         (system::time-in-60ths)
         #-(or SYMBOLICS TI)
                      (get-internal-run-time)
         )
        (t 0)))

(defparameter units/second
              #+SYMBOLICS  60
              #+TI         60
              #-(or SYMBOLICS TI)
                           internal-time-units-per-second
  "Number of units per second used by metering functions.")


(defun pretty-time (tics &optional brief)
  "If BRIEF is true then leading zero fields will be omitted."
  (multiple-value-bind (seconds remainder)
      (truncate tics units/second)
    (multiple-value-bind (minutes seconds)
        (truncate seconds 60)
      ;; We don't want to print something like 29:02.100
      (when (= 100 (round (* 100 remainder) units/second))
        (setf remainder 0)
        (cond ((= 59 seconds)
               (incf minutes)
               (setf seconds 0))
              (t (incf seconds))))
      (let ((hundreths (round (* 100 remainder) units/second)))
        (cond ((zerop tics)
               (if brief "        " " 0      "))
              #+IGNORE
              ((not (zerop minutes))
               (format nil "~2d:~2,'0d" minutes seconds))
              #+IGNORE
              ((not (zerop seconds))
               (format nil "0:~2,'0d.~2,'0d"
                       seconds hundreths))
              ((and brief (zerop minutes) (zerop seconds))
               (format nil "     .~2,'0d" hundreths))
              ((and brief (zerop minutes))
               (format nil "  :~2,'0d.~2,'0d" seconds hundreths))
              (t
               (format nil "~2d:~2,'0d.~2,'0d"
                       minutes seconds hundreths)))))))


(define-simple-counter *map-unit-types-count*)
(define-simple-counter *map-space-count*)
(define-simple-counter *find-units-all-count*)
(define-simple-counter *delete-unit-from-space-count*)
(define-simple-counter *move-unit-on-space-count*)
(define-simple-counter *insert-unit-on-space-count*)

(defvar *find-units-count* nil
        "Association list of space-instances and counts.")

(define-simple-meter *map-unit-types-meter*)
(define-simple-meter *map-space-meter*)
(define-simple-meter *find-units-all-meter*)
(define-simple-meter *delete-unit-from-space-meter*)
(define-simple-meter *move-unit-on-space-meter*)
(define-simple-meter *insert-unit-on-space-meter*)

(defvar *find-units-meter* nil
        "Association list of space-instances and runtimes.")

(defun report-meters (&optional total-minutes total-seconds)

  (let* ((find-units-meter-total (mapc-+ #'cdr *find-units-meter*))
         (read-total (+ find-units-meter-total
                        *find-units-all-meter*
                        *map-unit-types-meter*
                        *map-space-meter*))
         (write-total (+ *insert-unit-on-space-meter*
                         *move-unit-on-space-meter*))
         (misc-total *delete-unit-from-space-meter*)
         (bb-total (+ read-total write-total misc-total))
         (total-runtime (and total-minutes
                             (+ total-minutes
                                (/ (or total-seconds 0) 60.0)))))

    (when total-runtime
      ;; Convert to internal time units.
      (setf total-runtime (* total-runtime 60 units/second))
      (format t "~%Total Run Time~20t= ~a" (pretty-time total-runtime))
      (format t "~%Total Non-BB Time~20t= ~a" (pretty-time (- total-runtime bb-total))))
    (format t "~%Total BB Time~20t= ~a" (pretty-time bb-total))
    (format t "~%BB Read Time~20t= ~a" (pretty-time read-total))
    (format t "~%BB Write Time~20t= ~a" (pretty-time write-total))
    (format t "~%BB Misc Time ~20t= ~a~%" (pretty-time misc-total))

    (when total-runtime
      (format t "~%BB/Non-BB Ratio~20,10t= ~,2f"
              (/ bb-total (- total-runtime bb-total))))
    (if (zerop write-total)
        (format t "~%Read/Write Ratio~20t= n.a.~%")
        (format t "~%Read/Write Ratio~20t= ~,2f~%" (/ read-total write-total)))

    (show-simple-meters)
    (show-simple-counters)

    (flet ((sort-meter (meter)
             (sort meter
                   #'>
                   :key #'(lambda (pair)
                            (or (position (car pair) *space-instance-list*)
                                0)))))
      (setf *find-units-count* (sort-meter *find-units-count*))
      (setf *find-units-meter* (sort-meter *find-units-meter*))
      (show-find-meters))))


(defun show-simple-meters ()
  (dolist (m *simple-meters*)
    (format t "~%~a~40,10t= ~a" m (pretty-time (symbol-value m))))
  (terpri))

(defun show-simple-counters ()
  (dolist (m *simple-counters*)
    (format t "~%~a~40,10t= ~5d" m (symbol-value m)))
  (terpri))

(defun show-find-meters ()

  (let* ((find-units-meter-total (mapc-+ #'cdr *find-units-meter*))
         (find-units-count-total (mapc-+ #'cdr *find-units-count*)))

    (cond

      ((mismatch *find-units-count* *find-units-meter* :key #'car)
       ;; Can't do side by side
       (format t "~%Count of FINDS by space:")
       (dolist (pair *find-units-count*)
         (format t "~%  ~a~50,10t= ~6d  ~[~:;~:*(~2d%)~]"
                 (get-path-from-space-instance (car pair))
                 (cdr pair)
                 (round (* 100 (cdr pair)) find-units-count-total)))
       (format t "~2%  Total~50t= ~6d~%" find-units-count-total)
       (format t "~%Runtime of FINDS by space:")
       (dolist (pair *find-units-meter*)
         (format t "~%  ~a~50,10t= ~a  ~[~:;~:*(~2d%)~]"
                 (get-path-from-space-instance (car pair))
                 (pretty-time (cdr pair))
                 (round (* 100 (cdr pair)) find-units-meter-total)))
       (format t "~2%  Total~50t= ~a~%" (pretty-time find-units-meter-total)))

      (t
       (format t "~%Counts and Runtimes of FINDS by space:")
       (mapc #'(lambda (count meter)
                 (format t "~%  ~a~50,10t= ~6d  ~[~:;~:*(~2d%)~]~
                           ~70,10t~a  ~[~:;~:*(~2d%)~] "
                         (get-path-from-space-instance (car count))
                         (cdr count)
                         (round (* 100 (cdr count)) find-units-count-total)
                         (pretty-time (cdr meter))
                         (round (* 100 (cdr meter)) find-units-meter-total)))
             *find-units-count*
             *find-units-meter*)
       (format t "~2%  Total~50t= ~6d~70t~a~%"
               find-units-count-total (pretty-time find-units-meter-total))))))


(defun reset-meters ()
  (dolist (i *simple-meters*)
    (setf (symbol-value i) 0))
  (dolist (i *simple-counters*)
    (setf (symbol-value i) 0))
  (setf *find-units-count* nil)
  (setf *find-units-meter* nil))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
