;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL]INDEX-STRUCTURE.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  12:41:23 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                  INDEX STRUCTURE DEFINITION FUNCTIONS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  03-11-86 File Created.  (Gallagher)
;;;     ...   Many changes.  (Gallagher and Johnson)
;;;  01-08-87 Changed DEFINE-INDEX-STRUCTURE to put the index accessor function
;;;           definitions at `top level' in the macro expansion.  Also, allow
;;;           any data type as the element type for an index structure.
;;;           (Gallagher)
;;;  01-29-87 Put an eval-when around the code that define-index-structure
;;;           expands into so that it acts like other `def...' forms.
;;;           (Gallagher)
;;;  03-19-87 Added scalar-index-structure-p, series-index-structure-p, and
;;;           set-index-structure-p.  (Gallagher)
;;;  06-02-88 Reversed the order of functions in index paths to conform to
;;;           normal lisp order.  (Kept special treatment of defstructs for
;;;           now -- but that will be eliminated in the future.)   (Gallagher)
;;;  06-21-88 Modified MAKE-UNIT-INDEXES-1 to allow indexes to be based on
;;;           slots in *external-unit-slots*.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'gbb :USE '(lisp umass-extended-lisp))

(export '(define-index-structure))

(use-package '(umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Defining Unit Indexes
;;;; --------------------------------------------------------------------------

;;; MAKE-UNIT-D-INDEXES and MAKE-UNIT-P-INDEXES are called from define-unit
;;; to parse the index specifications.  They return a list of unit-indexes.

(defun make-unit-d-indexes (description)

  "MAKE-UNIT-D-INDEXES description

   Gets the information for indexing the unit.  DESCRIPTION is a
   unit-description.  Returns a list of unit-indexes."

  ;; Note that all the unit indexes including indexes inherited through an
  ;; included unit are returned.
  ;;
  ;; Each index-spec is a list of the form (index-name slot-name) where
  ;; index-name is the name of the index (i.e., the name of a space
  ;; dimension) and slot-name is the name of the slot that contains the
  ;; value of that index.
  (let ((result (nconc
		  (when (unit.included description)
                    (mapcar #'copy-unit-index
                            (unit.d-indexes
                              (get-unit-description (unit.included description)))))
		  (mapcar #'(lambda (index-spec)
			      (make-unit-indexes-1
				index-spec
				description))
			  (unit.d-index-list description))))
	(offset 0))
    (dolist (i result)
      (setf (unit-index.offset i) offset)
      (incf offset))
    result))


(defun make-unit-p-indexes (description)

  "MAKE-UNIT-P-INDEXES description

   Gets the information for indexing the unit.  DESCRIPTION is a
   unit-description.  Returns a list of unit-indexes."

  ;; Note that all the unit indexes including indexes inherited through an
  ;; included unit are returned.
  ;;
  ;; Each index-spec is a list of the form
  ;;
  ;;     (index-name slot-name [:type <type>])
  ;;
  ;; where index-name is the name of the index (i.e., the name of a space
  ;; dimension) and slot-name is the name of the slot that contains the
  ;; value of that index.

  (let ((result (nconc
		  (when (unit.included description)
                    (mapcar #'copy-unit-index
                            (unit.p-indexes
                              (get-unit-description (unit.included description)))))
		  (mapcar #'(lambda (index-spec)
			      (make-unit-indexes-1
				index-spec
				description))
			  (unit.p-index-list description))))
	(offset 0))
    (dolist (i result)
      (error-when (composite-index-structure-p (unit-index.index-structure i))
	 "Path Indexes may not be composite.~@
          The path index ~a refers to the index-structure ~s which is a composite."
	 (unit-index.name i) (index-structure.name (unit-index.index-structure i)))
      (setf (unit-index.offset i) offset)
      (incf offset))
    result))


;;; MAKE-UNIT-INDEXES-1 is called from MAKE-UNIT-D-INDEXES and
;;; MAKE-UNIT-P-INDEXES (above) to parse each index spec.  It
;;; returns a unit-index.  Mostly it does error checking.

(defun make-unit-indexes-1 (index-spec description)

  "MAKE-UNIT-INDEXES-1 index-spec description

   Builds a unit-index for this index.  INDEX is the name of the index
   (e.g., time).  DESCRIPTION is the unit-description."
 
  (let* ((index (car index-spec))
	 (index-type (getf (cddr index-spec) :type))
	 (index-index-structure (get-index-structure index-type nil))
	 (slot-name (second index-spec))
	 (slot (find-slot slot-name description))
	 (slot-type (and slot (slot.type slot)))
	 (slot-index-structure (get-index-structure slot-type nil))
	 (the-index-structure (or slot-index-structure index-index-structure))
	 (unit-name (unit.name description))
	 index-functions element-type)
    
    ;; Does the slot exist?
    (when (null slot)
      (if (member slot-name *external-unit-slots* :test #'string=)
          (setf slot-type t)
          (error
            "The index ~a refers to slot ~s which doesn't exist in unit ~s."
            index slot-name unit-name)))
    ;; If both the slot and the index have a :type option whose
    ;; value is an index structure, then they must be the same.
    (error-when (and slot-index-structure
		     index-index-structure
		     (not (eq slot-index-structure index-index-structure)) )
       "While defining unit ~s,~@
        the type of the index ~a is ~a~@
        but the type of the slot, ~a, on which ~a is based is ~a."
       unit-name index index-type slot-name index slot-type)
    ;; If an index-structure is given then get the index accessor function(s).
    (error-when (and the-index-structure
		     (null (setf index-functions
				 (get-index-function-quietly
				   index
				   the-index-structure))))
       "~s is not an index in the index structure ~s~%While defining unit ~s."
       index (index-structure.name the-index-structure) unit-name)

    (cond
      ;; If no type was declared for either the index or the slot
      ;; then we're stuck.  We need to know what sort of data
      ;; (i.e., the index element type) to expect.
      ((and (null index-type) (null the-index-structure))
       (if slot-type
	   (error "Bad type for the index ~A while defining the unit ~S.~@
                   There is no type specified for the index, ~A, itself; and the slot to which~@
                   it refers, ~A, is of type, ~S, which is not an index-structure."
		  index unit-name index slot-name slot-type)
	   (error "No type can be determined for the index ~A while defining the unit ~S.~@
                   There is no type specified for the index, ~A, itself; nor is a type~@
                   specified for the slot to which it refers, ~S."
		  index unit-name index slot-name)))
      ;; This case implies that the type of the slot an index structure name.
      ((null index-type)
       (setf element-type (index-structure-function.index-type index-functions)))
      ;; The type of the index itself is an index structure.  Check to
      ;; make sure that the slot type is a supertype of the index type.
      (index-index-structure
       ;; SUBTYPE should work with two identical types but
       ;; it doesn't always work in vaxlisp.  (sigh)
       (error-unless (or (eq index-type slot-type)
                         (subtypep index-type slot-type))
          "While defining unit ~s.~@
           The index ~a is based on the slot ~a;~@
           but the type of the index (~s)~@
           is not a subtype of the type of the slot (~s)."
          unit-name index slot-name index-type slot-type)
       (setf element-type (index-structure-function.index-type index-functions)))
      ;; 
      ((index-element-type-p index-type)
       (error-when (and slot-index-structure
			(not (eq index-type (index-structure-function.index-type
					      index-functions))))
	  "The index ~s is of type ~s in the unit ~s,~@
           but type ~s in the index-structure ~s."
	  index index-type unit-name
	  (index-structure-function.index-type index-functions)  slot-type)
       (setf element-type index-type))
      (t
       (error "~s is not an index-structure or an index element type.~@
               The index specification was ~s."
	      index-type index-spec)))

    (make-unit-index
      :name (form-keyword index)
      :package (package-name (symbol-package index))
      :datatype (if the-index-structure
		    (index-structure.name the-index-structure)
		    (element-type->datatype element-type))
      :element-type element-type
      :index-structure the-index-structure
      :slot-name slot-name
      :slot-accessor (form-symbol (unit.internal-conc-name description)
				  (string slot-name))
      :functions (index-structure-function.functions index-functions))))


;;;; -------------------------------------------------------------------------
;;;;   DEFINE-INDEX-STRUCTURE
;;;; -------------------------------------------------------------------------

;;; There are several types of composites:
;;;
;;;   - Sets are groups of elements with no order.
;;;   - Lists are groups of elements that are ordered by their position in
;;;     the list.  [Lists are not implemented yet.]
;;;   - Series are groups of elements that are ordered by some data value(s)
;;;     in the elements.  Currently series must be stored in order and there
;;;     can't be any wholes in the sequence.

(defconstant *index-structure-composite-types*
	     '(:set :series)
  "The available types of composites.")


(defmacro define-index-structure (name &body args) 

  "DEFINE-INDEX-STRUCTURE name [documentation]
                          &KEY composite-type composite-index element-type
                               type indexes)

   Defines an index structure.

   `Indexes' is an alist of index descriptors each of which specifies the
   index name, the element type for that index and information about how to
   access the values for the index.

   In addition to the indexes, a composite index type should include the
   `Composite-Type', `Composite-Index' and `Element-Type' arguments.  A
   non-composite index type should include just the `Indexes' and `Type'
   arguments." 
  
  (let ((documentation (if (stringp (car args)) (pop args) "No documentation supplied.")))
    (with-keywords-bound ((composite-type composite-index element-type type indexes)
			  args
			  "~s is not a valid keyword for DEFINE-INDEX-STRUCTURE.")

      (let ((compositep (or composite-type composite-index element-type))
	    (the-type (or type element-type))
	    (deftype-type (or type composite-type))
	    (the-composite-index nil)
	    index-structure-type index-structure definitions descriptions)
	
	(and type compositep
	     (error "For ~s, if :TYPE is given, none of~@
                 :COMPOSITE-TYPE :COMPOSITE-INDEX :ELEMENT-TYPE may be given."
		    name))
	(when compositep
	  (error-unless (and composite-type composite-index element-type)
	     "In the index-structure ~s, one or more of :COMPOSITE-TYPE~@
              :COMPOSITE-INDEX :ELEMENT-TYPE have been omitted."
	     name)
	  (error-unless (eq composite-type 'list)
	     ":COMPOSITE-TYPE must be LIST; it is ~s.~@
              While defining the index structure ~s."
	     composite-type name)
	  (cond ((eq composite-index :none)
		 (setf the-composite-index nil))
		(t 
		 (let ((index-spec (assoc composite-index indexes :test #'string=)))
		   (error-unless index-spec
		      "The :INDEXES argument to the index structure ~s~@
                       does not include its composite index, ~a."
		      name composite-index)
		   (error-unless (eq :point (second index-spec))
		      "The index element type of the composite index, ~a,~@
                       must be :POINT; it is ~s.~@
                       In the index-structure ~s."
		      composite-index (second index-spec) name)
		   (setf the-composite-index (make-keyword composite-index))))))

	(setf index-structure-type (cond ((not compositep) :scalar)
					 ((eq composite-index :none) :set)
					 (t :series)))
	(setf indexes (keyword-first-elements indexes))
	(multiple-value-setq (definitions descriptions)
	    (make-index-functions indexes the-type name))
	(setf index-structure (make-index-structure
				:name name
				:documentation documentation
				:type index-structure-type
				:element-type the-type
				:composite-type composite-type
				:composite-index the-composite-index
				:symbols (mapcar #'car indexes)
				:functions descriptions
				;; Save indexes for use by save-bb code.
				:index-list indexes))
	
	`(eval-when (load eval compile)
	   ;; Define it as a TYPE.
	   ,(unless (eq name deftype-type)
	      `(deftype ,name () ',deftype-type))
	   ,@definitions
	   ;; Save the indexes
	   (setf (gethash ',name *index-structure-hash-table*) ',index-structure)
	   ',name)))))


;;;; --------------------------------------------------------------------------
;;;;   Predicates on Index Structures
;;;; --------------------------------------------------------------------------


(defun index-structure-name-p (name)

  "INDEX-STRUCTURE-NAME-P name

   Returns true if name is a symbol which names an index-structure."

  (and (symbolp name)
       (gethash name *index-structure-hash-table*)))


;;; An index structure is either a scalar index structure or a composite
;;; index structure.  A scalar index structure is one that has only a
;;; single value for each index element.  That value may be a :point, a
;;; :range, or a :label.  A composite index structure has several values
;;; for each index element.  A composite can either be one of two types:
;;; series or set.  The difference is that series index structures are
;;; ordered.  The ordering is determined by the values of one of the
;;; indexes.  A set index structure is unordered.

(defun scalar-index-structure-p (obj)

  "SCALAR-INDEX-STRUCTURE-P obj
   Returns true if `obj' is a scalar index structure."

  (and (index-structure-p obj)
       (eq (index-structure.type obj) :scalar)))


(defun composite-index-structure-p (obj)

  "COMPOSITE-INDEX-STRUCTURE-P obj
   Returns true if `obj' is a composite index structure."

  (and (index-structure-p obj)
       (member (index-structure.type obj) *index-structure-composite-types*
	       :test #'eq)))

(defun series-index-structure-p (obj)

  "SERIES-INDEX-STRUCTURE-P obj
   Returns true if `obj' is a series index structure."

  (and (index-structure-p obj)
       (eq (index-structure.type obj) :series)))

(defun set-index-structure-p (obj)

  "SET-INDEX-STRUCTURE-P obj
   Returns true if `obj' is a set index structure."

  (and (index-structure-p obj)
       (eq (index-structure.type obj) :set)))



(defun get-index-structure (structure &optional (error-p t))

  "GET-INDEX-STRUCTURE symbol &optional (error-p t)

   Returns the index structure object for STRUCTURE.  STRUCTURE
   may be an index-structure, or a symbol.  The optional argument
   ERROR-P controls what happens when an index structure can't be
   found.  If ERROR-P is true then an error is signalled.
   Otherwise this functions simply returns nil."

  (cond ((index-structure-p structure) structure)
	((and (symbolp structure)
	      (gethash structure *index-structure-hash-table*)))
	((not error-p) nil)
	(t (error "~s is not an index-structure." structure))))


(defun get-index-function (index index-structure)

  "GET-INDEX-FUNCTION index index-structure

   Returns the information for accessing INDEX from a structure of
   type INDEX-STRUCTURE.  Specifically it returns a list of the form

      (index-name index-type . functions)

   Use the functions index-structure-function.index-name,
   index-structure-function.index-type, and index-structure-function.functions
   to access each field."

  (setf index-structure (get-index-structure index-structure))
  (find index (index-structure.functions index-structure)
	:key #'index-structure-function.index-name :test #'string=))


(defun get-index-function-quietly (index index-structure)

  "GET-INDEX-FUNCTION-QUIETLY index index-structure

   If the argument, INDEX-STRUCTURE, is the name of an index-structure
   then return the information for accessing INDEX from a structure of
   that type.  Otherwise return nil."

  (when (index-structure-name-p index-structure)
    (setf index-structure (get-index-structure index-structure nil)))
  (when (index-structure-p index-structure)
    (find index (index-structure.functions (get-index-structure index-structure))
	  :key #'index-structure-function.index-name :test #'string=)))



(defun make-index-functions (indexes the-type i-s-name)

  (let ((definitions nil)
	(descriptions nil)
	defs desc)
    (dolist (index indexes)
      (multiple-value-setq (defs desc)
	  (make-index-function index the-type i-s-name))
      (npush-list defs definitions)
      (push desc descriptions))
    (values definitions descriptions)))


(defun make-index-function (slot-spec data-type i-s-name)

  "MAKE-INDEX-FUNCTION slot-spec data-type i-s-name

   Create an accessor entry for an index slot.  SLOT-SPEC is a list
   in one of these formats:

	(index-slot-name :POINT . path)
        (index-slot-name :LABEL . path)
        (index-slot-name :RANGE (:MIN . path) (:MAX . path))

   Where path is

	(defstruct-slot {(defstruct-type defstruct-slot)}*

   This function returns a list of the index slot name, the type of data
   in the slot (:point or :range), and functions to acccess the slots."

  ;; Perhaps this should be a structure instead of a list

  (let ((index-slot-name (first slot-spec))
	(index-element-type (second slot-spec)))
    (ecase index-element-type
      ((:point :label)
       (let ((f1 (form-symbol "%%GBB-" i-s-name "." index-slot-name)))
	 (values
	   `((defun ,f1 (obj)
	      ,(make-index-path-accessor
		 'obj data-type (cddr slot-spec) index-slot-name i-s-name)))
	   `(,index-slot-name ,index-element-type ,f1))))
      (:range
       (let ((f1 (form-symbol "%%GBB-" i-s-name "." index-slot-name ".MIN"))
	     (f2 (form-symbol "%%GBB-" i-s-name "." index-slot-name ".MAX")))
	 (values
	   `((defun ,f1 (obj)
	       ,(make-index-path-accessor
		  'obj data-type (cdr (assoc :min (cddr slot-spec) :test #'eq))
		  index-slot-name i-s-name))
	     (defun ,f2 (obj)
	       ,(make-index-path-accessor
		  'obj data-type (cdr (assoc :max (cddr slot-spec) :test #'eq))
		  index-slot-name i-s-name)))
	   `(,index-slot-name :range ,f1 ,f2)))))
      ))


(defun make-index-path-accessor
       (arg data-type path index-slot-name i-s-name)

  "MAKE-INDEX-PATH-ACCESSOR arg data-type path
                            index-slot-name i-s-name

   Constructs a function calling sequence from path."

  (cond ((and
	   (defstruct-p data-type)
	   (structure-slot-p data-type (first path)))
	 (make-structure-index-path-accessor
	   (list (form-symbol (defstruct-conc-name data-type) (first path))
		 arg)
	   (rest path) index-slot-name i-s-name))
	(t
	 (make-other-index-path-accessor arg (reverse path)))))


(defun make-structure-index-path-accessor (arg path index-slot-name i-s-name)

  (when (null path)
    (return-from make-structure-index-path-accessor arg))

  (unless (listp (first path))
    (error "Bad format for index structure slot path.~@
	    While defining ~s in index structure ~s."
	   index-slot-name i-s-name))
  (let* ((step (first path))
	 (struct (first step))
	 (slot (second step)))
    (unless (defstruct-p struct)
      (error "~s is not a defstruct.~@
	      While defining ~s in index structure ~s."
	     struct index-slot-name i-s-name))
    (unless (structure-slot-p struct slot)
      (error "~s is not a slot in ~s.~@
	      While defining ~s in index structure ~s."
	     slot struct index-slot-name i-s-name))
    (make-structure-index-path-accessor
      (list (form-symbol (defstruct-conc-name struct) slot)
	    arg)
      (rest path)
      index-slot-name i-s-name)))


(defun make-other-index-path-accessor (arg path)

  ;; This one is easy -- no type checking.
  (cond ((null path) arg)
	(t (make-other-index-path-accessor
	     (list (first path) arg)
	     (rest path)))))



;;;; --------------------------------------------------------------------------
;;;;   Composite Indexes
;;;; --------------------------------------------------------------------------

;;; Code for identifying and distiguishing composite indexes, set indexes
;;; and scalar indexes.

(defun get-unit-composite-index (description pattern-object)

  "Information about unit's composite indexing used by find.
   Return values:
     T - if the pattern-object's composite index (`c-index') is part
       of a set type of index structure;
     Composite Index Symbol - (e.g., :time) if the c-index is part of
       a series type of index structure
     Nil - if the c-index isn't part of a composite index structure;
     Error - if the c-index isn't an index in the unit at all."
  
  ;; This function is called by check-composite-indexes in find-units.
  ;; More error checking should be done in check-find-reasonability
  ;; so that less needs to be done here.

  (let ((c-index (pattern-object.c-index pattern-object)))

    (when (null c-index)
      (return-from get-unit-composite-index t))

    (let* ((unit-index (find c-index (unit.d-indexes description)
			     :key #'unit-index.name :test #'eq))
	   ;; Unit index type is either an index-structure or nil.
	   (unit-index-type (and unit-index (unit-index.index-structure unit-index)))
	   (index-type (and unit-index-type (index-structure.type unit-index-type))))

      (cond ((null unit-index)
	     ;; Shouldn't happen
	     (error "The composite index ~a was expected to be one of the~@
                     indexes for ~s but it wasn't."
		    c-index (unit.name description)))
	    ;; Should this be an error?
	    ((null unit-index-type) nil)
	    ((eq index-type :scalar) nil)
	    ((eq index-type :set) t)
	    ((eq index-type :series)
	     (index-structure.composite-index unit-index-type))
	    (t (error "Internal GBB Error. Unexpected index structure type: ~s."
		      index-type))))))


(defun composite-unit-index-p (unit-index)

  "Returns true if UNIT-INDEX represents a composite."

  (composite-index-structure-p
    (unit-index.index-structure unit-index)))

(defun series-unit-index-p (unit-index)

  "Returns true if UNIT-INDEX represents a series."

  (series-index-structure-p
    (unit-index.index-structure unit-index)))

(defun set-unit-index-p (unit-index)

  "Returns true if UNIT-INDEX represents a set."

  (set-index-structure-p
    (unit-index.index-structure unit-index)))

(defun scalar-unit-index-p (unit-index)

  "Returns true if UNIT-INDEX represents a scalar index."

  (let ((index-structure (unit-index.index-structure unit-index)))
    ;; If unit-index represents an index that is not based
    ;; on an index-structure then its type slot will nil.
    ;; In this case, it must be a scalar.
    (or (null index-structure)
	(eq :scalar (index-structure.type index-structure))
	;; :SINGLE is the old keyword
	(eq :single (index-structure.type index-structure)))))


(defun related-indexes-p (indexes description)

  "Returns true if all the INDEXES are related to one another (i.e.,
   grouped together) in the unit represented by DESCRIPTION.
   ``Related'' means that each index is derived from the same slot
   in the unit.  INDEXES is a list of keywords and DESCRIPTION is a
   unit description."

  (let* ((unit-indexes (unit.d-indexes description)))

    (mapc-eq #'(lambda (index)
                 (unit-index.slot-name
                   (or (find index unit-indexes
                             :key #'unit-index.name :test #'eq)
                       (error "~a is not an index in ~s."
                              index (unit.name description)))))
             indexes)))


(defun element-type->datatype (element-type)

  "Returns a common lisp datatype corresponding to `element-type'."

  (ecase element-type
    (:point 'number)
    (:label t)))
    
;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
