;;;; -*- Mode: Common-Lisp; Package: GBB; Fonts: (MEDFNT); Base: 10 -*- 
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL.GRAPHICS]GRAPHICS-SUPPORT.LISP *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  14:06:29; Edited-By: Cork *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; ==========================================================================
;;;;
;;;;                     SUPPORT FUNCTIONS FOR GBB GRAPHICS                    
;;;;
;;;; ==========================================================================
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  05-01-87 File Created.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB)

(use-package '(lisp umass-extended-lisp))

(import '(space-has-dimension 
          get-space-dimension-info
          get-common-dimensions
          all-space-unit-types
	  dimension-info.dimension-name
	  dimension-info.dimension-type
	  dimension-info.dimension-arg
	  dimension-info.dimension-options
	  getopt
	  unit-has-dimension
	  get-unit-dimension-info
	  mutual-composite-dimensions-p
	  composite-dimension-p
	  describe-unit-instance
          build-find-pattern
          find-units-semi-internal)
	(find-package "GBB-GRAPHICS"))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ===========================================================================

(defun space-has-dimension (space dimension)

  "SPACE-HAS-DIMENSION space dimension

   Returns true if DIMENSION is one of the dimensions of SPACE.
   SPACE can be a space type or a space instance.  DIMENSION
   must be a symbol or a string."

  (when (space-instance-p space)
    (setf space (space-instance.space space)))
  (find dimension (space.dimensions space)
        :key #'dimension.name :test #'string=))

;;; -----------------------------------------------------------------------

(defun get-space-dimension-info (space-spec)

  "GET-SPACE-DIMENSION-INFO space-spec

   Returns information about the dimensions of a space.  The argument,
   SPACE-SPEC, may be a space name, a space instance, or a path-structure.
   An alist is constructed for each space which describes the dimensions of
   the space.  The alist is in the same format as the :dimensions argument
   to define-space.  Note that the name of each dimension is a keyword.

   SPACE-SPEC may also be a list of space names, space instances or
   path-structures, in which case a list of alists is returned.

   For example:

     (get-space-dimension-info 'space1)
     ==>
     ((:x :ordered (-100 100))
      (:y :ordered (-100 100))
      (:type :enumerated (lion tiger bear) :test eq))"

  (labels ((get-info (space)
	     (mapcar #'get-dimensions (space.dimensions space)))
	   (get-dimensions (dimension)
	      (cons (dimension.name dimension)
		    (ecase (dimension.type dimension)
		      ((:ordered)
		       `(:ordered (,(%range-start (odim.range dimension))
				   ,(%range-end (odim.range dimension)))))
		      ((:enumerated)
		       `(:enumerated ,(edim.labelset dimension)
			 :test ,(edim.test dimension)))))))

    (cond ((listp space-spec)
	   (mapcan #'get-space-dimension-info space-spec))
	  (t
	   (get-info
             (cond ((path-structure-p space-spec)
                    (gbb::space-instance.space
                      (get-space-instance-from-path-structure
                        space-spec)))
                   ((space-instance-p space-spec)
                    (gbb::space-instance.space space-spec))
                   ((space-name-p space-spec)
                    (get-space space-spec))
                   (t (error "~s doesn't specify a space." space-spec)))))
	  )))

;;; -----------------------------------------------------------------------

(defun get-common-dimensions (space-instances)

  "GET-COMMON-DIMENSIONS space-instances

   Returns information about the common dimensions of several space instances.
   The argument, SPACE-INSTANCES, is a list of space instances.  An alist is
   constructed which describes the common dimensions of the space instances.
   The alist is in the same format as the :dimensions argument to
   define-space.  See GET-SPACE-DIMENSION-INFO."

  (let ((dimension-info (get-space-dimension-info (first space-instances))))
    (dolist (si (rest space-instances))
      (setf dimension-info
            (delete-if-not #'(lambda (dim-info)
                               (find (first dim-info)
                                     (space.dimensions (space-instance.space si))
                                     :key #'dimension.name
                                     :test #'string=))
                           dimension-info)))
    dimension-info))

;;; -----------------------------------------------------------------------

(defun all-space-unit-types (space-instances)

  "ALL-SPACE-UNIT-TYPES space-instances

   Returns a list of unit types that can be stored on the space-instances."

  (let ((unit-types nil))
    (dolist-or-atom (si space-instances)
      (dolist (entry (space-instance.data si))
        (dolist (u (unit-mapping.units (storage-entry.unit-mapping entry)))
          (pushnew u unit-types :test #'eq))))
    unit-types))

;;; -----------------------------------------------------------------------

;;; Accessor functions for components of the data returned by 
;;; GET-SPACE-DIMENSION-INFO.

(defun dimension-info.dimension-name (entry)
  "Returns the name of the dimension (e.g., :time)."
  (first entry))

;;; -----------------------------------------------------------------------

(defun dimension-info.dimension-type (entry)
  "Returns the dimension type (e.g., :ordered)."
  (second entry))

;;; -----------------------------------------------------------------------

(defun dimension-info.dimension-arg (entry)
  "Returns the dimension `arguments'.  For :ordered dimensions
   this is the bounds; for :enumerated dimensions, this is the
   label set."
  (third entry))

;;; -----------------------------------------------------------------------

(defun dimension-info.dimension-options (entry)
  "Returns any options for the dimension.  The result is a plist."
  (cdddr entry))

;;; -----------------------------------------------------------------------

;;; The options above are stored as alternating keywords and values
;;; -- a plist.  Unfortunately, getf doesn't work because it tries
;;; to do a locf on (dimension-info.dimension-options ...) and can't.

(defun getopt (list option)
  (second (member option list :test #'eq)))

;;; -----------------------------------------------------------------------

(defun unit-has-dimension (unit dimension)
  
  "UNIT-HAS-DIMENSION unit dimension

   Returns true if DIMENSION is an index of UNIT.  UNIT can be a
   unit type or unit instance."

  (let* ((description (get-unit-description unit))
	 (unit-indexes (unit.d-indexes description)))
    (find dimension unit-indexes
	  :test #'string=
	  :key #'unit-index.name)))

;;; -----------------------------------------------------------------------

(defun get-unit-dimension-info (unit-instance mutually-composite-p &rest dimensions)

  "GET-UNIT-DIMENSION-INFO unit-instance dimensions

   Returns the value(s) of some dimensions for UNIT-INSTANCE.
   DIMENSIONS indicates which dimension's values are to be returned.
   If MUTUALLY-COMPOSITE-P is true then the dimensions are indexes
   in the same composite."

  (let* ((description (get-unit-description unit-instance))
	 (unit-indexes (unit.d-indexes description))
	 (the-indexes (basic-unit.%%indexes%% unit-instance)))

    (labels ((do-one-element (element)
	       (ecase (index-element.element-type element)
		 ((:point :label)
		  (index-element.point-value element))
		 ((:range)
		  (let ((value (index-element.value element)))
		    (cons (range-start value)
			  (range-end value))))))

	     (get-index-data (dimension)
	       (let ((unit-index (find dimension unit-indexes
				       :test #'string=
				       :key #'unit-index.name)))
		 (when (null unit-index)
		   (error "~a is not a dimension in ~s."
			  dimension (unit.name description)))
		 (elt the-indexes (unit-index.offset unit-index)))))

    (cond (mutually-composite-p
           (let ((data (mapcar #'get-index-data dimensions))
                 (result nil)
                 (temp nil))
             (dolist-slices (slice data)
               (setf temp nil)
               (dolist (d slice)
                 (push (do-one-element d) temp))
               (push (nreverse temp) result))
             (nreverse result)))
          (t ;; Dimensions returned separately
           (mapcar
	     #'(lambda (dimension-name)
		 (let* ((data (get-index-data dimension-name))
			(result nil))
		   (cond
		     ;; The dimension is composite
		     ((listp (first data))
		      (dolist (element data)
			(push (do-one-element element) result))
		      (nreverse result))
		     (t ;; Dim is scalar
		      (do-one-element data)))))
	     dimensions))))))

;;; -----------------------------------------------------------------------

(defun mutual-composite-dimensions-p (unit-type &rest dimension-names)

  "MUTUAL-COMPOSITE-DIMENSIONS-P unit-type &rest dimension-names

   Returns true if all the dimensions are composite in unit-type
   and they are ``grouped.''"

  (let* ((description (get-unit-description unit-type))
         (unit-indexes (unit.d-indexes description)))
    (mapc-eq #'(lambda (dim)
                 (let* ((unit-index (find dim unit-indexes
                                          :test #'string= :key #'unit-index.name))
                        (index-structure (unit-index.index-structure unit-index)))
                   (unless (and index-structure
                                (composite-index-structure-p index-structure))
                     (return-from mutual-composite-dimensions-p nil))
                   (unit-index.slot-name unit-index)))
             dimension-names)))

;;; -----------------------------------------------------------------------

(defun convert-dimension-info-to-mutual (info)
  (let ((result nil))
    (dolist-slices (slice info)
      (push (copy-list slice) result))
    (nreverse result)))

;;; -----------------------------------------------------------------------

(defun composite-dimension-p (unit-type dimension)

  "COMPOSITE-DIMENSION-P unit-type dimension

   Returns true if `dimension' is a composite dimension in `unit-type'."

  (let* ((description (get-unit-description unit-type))
         (unit-indexes (unit.d-indexes description))
         (unit-index (find dimension unit-indexes
                           :test #'string= :key #'unit-index.name))
         index-structure)
    (unless unit-index
      (error "~a is not a dimension in ~s." dimension (unit.name description)))
    (and (setf index-structure (unit-index.index-structure unit-index))
         (composite-index-structure-p index-structure))))

;;; -----------------------------------------------------------------------

(defun set-dimension-p (unit-type dimension)

  "SET-DIMENSION-P unit-type dimension

   Returns true if `dimension' is a set dimension in `unit-type'."

  (let* ((description (get-unit-description unit-type))
         (unit-indexes (unit.d-indexes description))
         (unit-index (find dimension unit-indexes
                           :test #'string= :key #'unit-index.name))
         index-structure)
    (unless unit-index
      (error "~a is not a dimension in ~s." dimension (unit.name description)))
    (and (setf index-structure (unit-index.index-structure unit-index))
         (set-index-structure-p index-structure))))

;;; -----------------------------------------------------------------------

(defun describe-unit-instance (unit-instance &optional (stream *standard-output*))

  "DESCRIBE-UNIT-INSTANCE unit-instance &optional (stream *standard-output*)

   Print a description of UNIT-INSTANCE."

  (let ((description (get-unit-description unit-instance)))
    (flet ((print-a-slot (slot)
	     (format stream "~%~a~24,8t~s"
		     slot (get-structure-slot unit-instance slot))))
      (format stream "Unit of type ~s.~2%" (unit.name description))
      (when (unit.documentation description)
	(format stream "~s~%" (unit.documentation description)))
      (print-a-slot 'name)
      (dolist (slot (unit.slots description))
	(print-a-slot (slot.name slot)))
      (dolist (link (unit.links description))
	(print-a-slot (link.name link)))
      (terpri stream)))
  unit-instance)

;;; -----------------------------------------------------------------------

(defun build-find-pattern (x-dim x-type x-pattern y-dim y-type y-pattern)

  "Build a pattern for find-units-semi-internal to retrieve all
   units which overlap with X and Y.  The arguments are:

   .-DIM      The name of the dimension/index.  This should be a
              keyword or NIL which means to omit the dimension.  At
              least one of X-DIM and Y-DIM must be non-nil.

   .-TYPE     Either :ordered or :enumerated.

   .-PATTERN  A list of index values.  An index value, for this
              function, is either a label, for enumerated dimensions,
              or a range (a list of (upper-bound lower-bound)), for
              ordered dimensions."

  (unless (or x-dim y-dim)
    (error "Both x-dim and y-dim can not be nil."))

  (if (null x-dim)
      (setf x-type nil x-pattern nil)
      (setf x-dim (form-keyword x-dim)))
  (if (null y-dim)
      (setf y-type nil y-pattern nil)
      (setf y-dim (form-keyword y-dim)))

  (labels
    ((replicate (left-data right-data right-p)
         (cond ((or (null left-data) (null right-data))
                (if right-p right-data left-data))
               (t
                (mapcan #'(lambda (x)
                            (mapcar #'(lambda (y) (if right-p y x))
                                    right-data))
                        left-data))))
     (index-data-type (dimension-type)
         (ecase dimension-type
           ((nil)        nil)
           (:ordered     '(:range))
           (:enumerated  '(:label)))))

    (let* ((length (* (if x-dim (length x-pattern) 1)
                      (if y-dim (length y-pattern) 1)))
           (indexes `(,@(and x-dim (list x-dim))
                      ,@(and y-dim (list y-dim))))
           (pattern-object
             (make-pattern-object
               :all-indexes indexes
               :length      length
               :c-index     nil
               :c-type      :set
               :c-indexes   indexes
               :c-types     `(,@(index-data-type x-type)
                              ,@(index-data-type y-type))
               :c-data      `(,@(and x-dim `(,(replicate x-pattern y-pattern nil)))
                              ,@(and y-dim `(,(replicate x-pattern y-pattern t))))
               :n-indexes   nil
               :n-types     nil
               :n-data      nil)))
      
      (make-search-pattern
        :match-function  (get-element-match-function :overlaps)
        :before-extras   :dont-care
        :after-extras    :dont-care
        :match           '(:count 1)
        :mismatch        '(:percentage 100)
        :skipped         '(:percentage 100)
        :n-match         1
        :n-mismatch      length
        :n-skipped       length
        :contiguous      nil
        :pattern-object  pattern-object))))

;;; -----------------------------------------------------------------------

(defconstant *null-find-options* (make-find-options
                                   :filter-before nil
                                   :filter-after  nil)
  "Empty find-unit options.")

(defun find-units-semi-internal (unit-types space-instances search-pattern
                                 &optional filter-before filter-after)

  "A version of find-units without the `user interface'.  The arguments
   are the unit types (or extended unit types), the space instances to
   search, and the search pattern (this is a structure -- it's not the
   same as the pattern argument to find-units)."

  (let ((result nil)
        (options (if (and (null filter-before) (null filter-after))
                     *null-find-options*
                     (make-find-options
                       :filter-before filter-before :filter-after filter-after)))
        all-unit-types)

     (cond ((eq unit-types t)
	    (setf all-unit-types t))
	   (t
	    (setf unit-types (if (or (atom unit-types)
				     (x-unit-type-of unit-types t))
				 (list unit-types)
				 unit-types))
	    (setf all-unit-types (expand-unit-type-list unit-types))))

     (dolist-or-atom (space-instance space-instances)
       (check-find-reasonability space-instance search-pattern unit-types)
       (setf result
	     (npush-list
	       (find-units-internal all-unit-types
				    space-instance
				    search-pattern
				    options)
	       result)))
     result))


;;; ---------------------------------------------------------------------------
;;;                                End of File
;;; ---------------------------------------------------------------------------
