;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB1; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.DEVELOPMENT.GBB1.EXAMPLES]PRESCRIPTION-KS.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Friday, September 16, 1988  10:33:40 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *       GBB1 KS Shell: Generic KSs for "Prescription" Meta-Strategy
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Philip Johnson
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  06-18-87 File Created.  (Johnson)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package "GBB1")

(use-package '(lisp umass-extended-lisp gbb))

(import '(gbb::dolist-or-atom))

(provide "GBB1-PRESCRIPTIONS")


;;;-----------------------------------------------------------------------------
;;;
;;;  The Prescription "Meta Control" Strategy
;;;
;;;-----------------------------------------------------------------------------

#||

 The BB1 control model specifies a control hierarchy consisting of
 strategies, foci, and heuristics.  The function of this hierarchy is the
 construction and maintenance of a "control plan," which is a network of
 instantiated strategies, focii, and heuristics which are used to decide
 which of the executable KSARs is to be executed next.

 The function of the following set of KSs is to implement a kind of "meta
 control" strategy, or in plain language, a method for setting up the
 control plan.  This involves the use of certain special "prescription"
 slots in the application control KSs, as well as the use of the following
 Prescription KSs.

 In general, the prescription metastrategy is activated by the addition of a
 strategy plan object to the control plan blackboard.  (This should normally
 be one of the results of execution of the initial KSs actions.) This
 strategy plan object has a slot called FUTURE-PRESCRIPTION, which should be
 initialized to a list of the sequence of foci which must be satisfied in
 order to complete this strategy.  When a strategy PO is added, the
 INITIALIZE-PRESCRIPTION KS is triggered and its action is to set the
 CURRENT-PRESCRIPTION link of this PO to the first element of its
 FUTURE-PRESCRIPTION slot.  This should trigger the appropriate foci KSs to
 create their POs and link up to this strategy.  Each focus PO, in turn, has
 a slot called HEURISTICS, which is a list of heuristic KS names.  When a
 focus PO is added to the control plan blackboard, the heuristics which
 implement it should be triggered and add their own POs to be linked up to
 the corresponding focus.

 Note that the CURRENT-PRESCRIPTION slot of a strategy is also the
 first element of the FUTURE-PRESCRIPTION slot.

 When the current prescription of a strategy has been satisfied, the
 UPDATE-PRESCRIPTION KS is triggered to update the CURRENT-PRESCRIPTION link
 of this PO with the next set of focii (or individual focus) from its
 FUTURE-PRESCRIPTION slot.  When all the foci's goals have been satisfied
 for a strategy PO, the TERMINATE-PRESCRIPTION KS is triggered which sets
 the status slot of the strategy to inoperative.  A similar KS called
 TERMINATE-FOCUS is triggered when a focus' goal is satisfied.

||#

;;;-----------------------------------------------------------------------------
;;;
;;;  Initialize Prescription
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining INITIALIZE-PRESCRIPTION")

(define-gbb1-control-ks (INITIALIZE-PRESCRIPTION (:export))

  "Initialize-prescription is triggered by the addition of a new strategy PO
   to the control blackboard.  It performs the following action:
     Sets the CURRENT-PRESCRIPTION link to the first element of the FUTURE-PRESCRIPTION slot;
   (Setting the CURRENT-PRESCRIPTION link generates a GBB1 event which should trigger those KSs.)"

  :CONTROL-TYPE        :META
  :TRIGGER-CONDITIONS  ((trigger-event-class-p :unit-creation)
		        (strategy-po-p (trigger-unit)))
  :CONTEXT-SLOTS       ((triggering-po) `((,(trigger-unit))))
  :ACTION-FUNCTION     #'init-prescription
  :FROM-BB             (make-paths :paths '(gbb1 control control-plan))
  :TO-BB               (make-paths :paths '(gbb1 control control-plan))
  :KSAR-PRINT-FUNCTION print-init-prescr
  :AUTHOR              "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun INIT-PRESCRIPTION  (init-prescription-ksar)
  
  "INIT-PRESCRIPTION init-prescription-ksar

   Initializes the strategy prescription"
  
  (let* ((strategy-po (initialize-prescription-ksar$triggering-po init-prescription-ksar))
	 (prescr (first (basic-strategy-po$future-prescription strategy-po))))
    
    (dolist-or-atom (ks prescr)
      (linkf (basic-strategy-po$current-prescription strategy-po)
	     (find-unit-by-name (string ks) ks)))))

;;;-----------------------------------------------------------------------------

(defun PRINT-INIT-PRESCR (object stream depth)

  "PRINT-INIT-PRESCR object stream depth

   The print function for the initialize-prescription ksar."

  (declare (ignore depth object))
  (format stream "INIT-PRESCRIPT"))

;;;-----------------------------------------------------------------------------
;;;
;;;  Update Prescription
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining UPDATE-PRESCRIPTION")

(define-gbb1-control-ks (UPDATE-PRESCRIPTION (:export))

  "Update-prescription is triggered by removal of the final
   CURRENT-PRESCRIPTION link.

   It performs the following actions:
     o Sets the PAST-PRESCRIPTION slot to the first element of
       FUTURE-PRESCRIPTION.
     o Pops the FUTURE-PRESCRIPTION slot.
     o Sets the CURRENT-PRESCRIPTION link to the (new) first element of
       the FUTURE-PRESCRIPTION slot.

   (Note that setting the CURRENT-PRESCRIPTION link generates a GBB1
   event which should trigger those KSs.)"

  :CONTROL-TYPE       :META
  :TRIGGER-CONDITIONS ((strategy-po-p (trigger-unit))
		       (trigger-event-class-p :unlink)
		       (trigger-event-slot-p 'current-prescription nil))
  :CONTEXT-SLOTS      ((triggering-po) `((,(trigger-unit))))
  :ACTION-FUNCTION    #'upd-prescription
  :FROM-BB            (make-paths :paths '(gbb1 control control-plan))
  :TO-BB              (make-paths :paths '(gbb1 control control-plan))
  :KSAR-PRINT-FUNCTION print-update-prescr
  :AUTHOR              "Philip Johnson")

 
;;;-----------------------------------------------------------------------------

(defun UPD-PRESCRIPTION  (update-prescr-ksar)
  
  "UPD-PRESCRIPTION update-prescr-ksar

   Update the prescription of the triggering strategy PO, if required."
  
  (let ((strategy-po (update-prescription-ksar$triggering-po update-prescr-ksar)))
    
    ;; update the past prescription with the (now-defunct) current-prescription.
    (setf (basic-strategy-po$past-prescription strategy-po)
	  (nconc (basic-strategy-po$past-prescription strategy-po)
		 (list
		   (first (basic-strategy-po$future-prescription strategy-po)))
		 ))
    
    ;; update the future prescription by popping off the current prescription
    (pop (basic-strategy-po$future-prescription strategy-po))
    
    ;; set up the current prescription list.
    (let ((presc (first (basic-strategy-po$future-prescription strategy-po))))
      (dolist-or-atom (ks presc)
	(linkf (basic-strategy-po$current-prescription strategy-po)
	       (find-unit-by-name (string ks) ks))))))

;;;-----------------------------------------------------------------------------

(defun PRINT-UPDATE-PRESCR (object stream depth)

  "PRINT-UPDATE-PRESCR object stream depth

   The print function for the update-prescription ksar."

  (declare (ignore object depth))
  (format stream "UPDATE-PRESCRIPT"))

;;;-----------------------------------------------------------------------------
;;;
;;;  Terminate Prescription
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining TERMINATE-PRESCRIPTION")

(define-gbb1-control-ks  (TERMINATE-PRESCRIPTION (:export))
                  
  "Terminates a currently active strategy or focus and its subordinate
   elements.  Terminate-prescription is triggered by the addition of a
   focus or strategy PO, and becomes executable when the goal function
   is satisfied.

   It performs the following actions:
     o Sets the status slot to inoperative.
     o If the PO is a focus, it is unlinked from its implementing
       strategies, and sets its heuristics to inoperative."

  :CONTROL-TYPE       :META
  :TRIGGER-CONDITIONS ((or (focus-po-p (trigger-unit)) (strategy-po-p (trigger-unit)))
		       (trigger-event-class-p :unit-creation))
  :PRECONDITIONS      ((funcall (basic-control-po$goal-function
				  (terminate-prescription-ksar$triggering-po *this-ksar*))
				(terminate-prescription-ksar$triggering-po *this-ksar*)))
  :CONTEXT-SLOTS      ((triggering-po) `((,(trigger-unit))))
  :ACTION-FUNCTION    #'term-prescription
  :FROM-BB             (make-paths :paths '(gbb1 control control-plan))
  :TO-BB               (make-paths :paths '(gbb1 control control-plan))
  :KSAR-PRINT-FUNCTION print-terminate-prescr
  :AUTHOR              "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun TERM-PRESCRIPTION  (terminate-prescr-ksar)
  
  "TERM-PRESCR terminate-prescr-ksar

   Terminate the triggering focus or strategy."


  (let ((prescr-po (terminate-prescription-ksar$triggering-po terminate-prescr-ksar)))

    (cond ((focus-po-p prescr-po) (term-focus prescr-po))
	  ((strategy-po-p prescr-po) (term-strategy prescr-po))
	  (t (error "Term-prescription executed without a triggering strategy or focus")))))

;;;-----------------------------------------------------------------------------

(defun PRINT-TERMINATE-PRESCR (object stream depth)

  "PRINT-TERMINATE-PRESCR object stream depth

   The print function for the terminate-prescription ksar."

  (declare (ignore object depth))
  (format stream "TERM-PRESCRIPT"))

;;;-----------------------------------------------------------------------------

(defun TERM-FOCUS (focus-po)

  "TERM-FOCUS focus-po

   Terminates FOCUS-PO."

  (let ((triggering-ks (basic-focus-po$ks focus-po))
	(implementing-strategies (basic-focus-po$implementor-of focus-po)))

    (setf (basic-focus-po$status focus-po) :inoperative)
    (setf (basic-focus-po$last-cycle focus-po) *execution-cycle*)
    
    (dolist (heuristic-po (basic-focus-po$implemented-by focus-po))
      (setf (basic-heuristic-po$status heuristic-po) :inoperative))

    (dolist (strategy-po implementing-strategies)
      (unlinkf (basic-focus-po$implementor-of focus-po) strategy-po)
      (unlinkf (basic-focus-ks$current-prescription-for triggering-ks) strategy-po))
    ))

;;;-----------------------------------------------------------------------------

(defun TERM-STRATEGY (strategy-po)

  "TERM-STRATEGY strategy-po

   Terminates strategy-po."

  (setf (basic-strategy-po$status strategy-po) :inoperative)
  (setf (basic-strategy-po$last-cycle strategy-po) *execution-cycle*))


;;;-----------------------------------------------------------------------------
;;;
;;;  Helper function for applications using the prescription "meta strategy"
;;;
;;;-----------------------------------------------------------------------------

(defun CURRENT-PRESCRIPTION-P (focus-unit-type strategy-unit-instance)

  "CURRENT-PRESCRIPTION-P focus-unit-type strategy-unit-instance

   Returns T if FOCUS-UNIT-TYPE is in the current prescription of STRATEGY-UNIT-INSTANCE"

  (let ((current-prescription (basic-strategy-po$current-prescription strategy-unit-instance))
	(focus-instance (find-unit-by-name (string focus-unit-type) focus-unit-type)))
    (if (listp current-prescription)
	(member focus-instance current-prescription)
	(eq focus-instance current-prescription))))

;;;-----------------------------------------------------------------------------
;;;                                  End of file
;;;-----------------------------------------------------------------------------


