;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB1; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL.GBB1]KSS-FUNS.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Friday, September 16, 1988  10:25:35 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *           GBB1 Execution Shell: Function Definitions
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Philip Johnson
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  03-05-87 File Created.  (Johnson)
;;;  05-19-87 Conversion to GBB1 Execution Shell Version 1.0  (Johnson)
;;;  06-22-87 KS Shell conversion  (Johnson)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *


(in-package "GBB1")

(use-package '(lisp umass-extended-lisp gbb))

(export '(reset-gbb1
	  sum-of-weights-times-ratings))


;;; ============================================================================
;;; 
;;;  KS Shell Rating Functions.
;;;
;;; ============================================================================

;In general, rating a KSAR appears rather straightforward.  The priority
;function uses the ratings given to the KSAR by the focii to produce a
;single overall rating for the KSAR.  Each focus rating is computed by
;combining the ratings given to the KSAR by all the heuristics implementing
;that focus. 

;The implementation of rating is not straightforward because recomputing
;every priority, heuristic, and integrated rating for each executable KSAR
;on every execution cycle is very expensive computationally.  For this
;reason, the priority, focus, and heuristic rating functions can be made
;"stable," which indicates that their values, once obtained, do not need to
;be recomputed as long as the parts of the control plan which they depend
;upon do not change.  In addition, every control plan object can be made
;"inoperative," which indicates that their ratings should not be used.
;Finally, the actions of KSs may add or delete focii or heuristics from the
;control plan blackboard.

;Given that the stability, operativeness, and even existance of control plan
;objects can change from cycle to cycle, determining the priority of a KSAR
;efficiently becomes a somewhat more subtle problem.  For example, consider
;the situation where the priority, the focii, and their implementing
;heuristics are all stable.  If the control plan doesn't change, then the
;priority of the KSARs, once computed, remains constant and no recomputation
;need take place from one execution cycle to the next.  However, different
;types of changes to the control plan can have very different ramifications
;for the rating process.  For example, if the action of a KS is to make one
;of the focii inoperative, then just the priorities of the KSARs must be
;recomputed.  If a new heuristic is added however, its rating, as well as
;the integrated rating of its implementing focus and the priority of every
;KSAR which had been previously rated by this focus must be recomputed.
;Note that these recomputations are necessary despite the fact that all the
;rating functions are "stable."	

;Parenthetical comment: BB1 deals with this by creating lots of lists.
;First, it creates a list of all the focii, and removes the inoperative ones
;to form a new list of operative focii.  From this list the stable focii are
;removed to form a new list of dynamic focii.  Next, BB1 creates new lists
;of all the changed focii and changed heuristics during the last execution
;cycle.  The changed focii and the operative focii are intersected to form a
;new list of changed-operative focii, and this is unioned with the dynamic
;focii to create a new list of the focii which all KSARs must be rerated by.
;The inoperative focii are determined by creating a list of the
;set-difference between the operative focii and the changed focii.  There's
;a bit more miscellaneous list-making going on, but this example
;demonstrates that getting the rating process to work correctly when the
;control plan is dynamically changing requires maintaining a fair amount of
;information, and that without some care, a significant amount of consing
;can result.

;In GBB1, we avoid much of this overhead by defining event handlers for each
;of the control plan blackboard events which impact on the rating process.
;These event handlers update the KSARs on the executable agenda to reflect
;the current state of the control plan.  This updating process consists of
;removing any ratings from the KSAR which have been made invalid as a result
;of the control plan blackboard event.  For the second example above, the event
;handler for the addition of a heuristic object would remove the affected
;integrated rating and the priority value for each KSAR on the executable
;agenda.  During the next rating process, we simply check to see whether or
;not a rating is present; if absent and the rating function is dynamic, we
;simply compute it.  If absent and the rating function is stable, we compute
;it and store it with the KSAR.  If the rating is present, we can use it
;immediately, since its presence proves that the rating is stable and that
;no events of the last execution cycle have invalidated it.

;For clarity, let's define some data objects to be used in the
;implementation.  Each KSAR has two slots used in rating: PRIORITY and
;RATINGS.  The PRIORITY slot is given the KSARs current priority (whether
;stable or dynamic), and is the "public" priority to be accessed by the
;recommendation function.  The RATINGS slot of the KSAR is filled by the
;structure KSAR-RATING, which has two fields, PRIORITY and FOCUS-RATINGS.
;The "private" PRIORITY slot holds stable priority ratings.  The
;FOCUS-RATINGS slot contains a hash table of FOCUS-RATING structures, whose
;keys are the corresponding focus instances on the control plan.  The
;FOCUS-RATING structure has two slots, INTEGRATED-RATING and
;HEURISTIC-RATINGS.  The INTEGRATED-RATING slot holds stable integrated
;ratings given to the KSAR by this focus instance.  The HEURISTIC-RATINGS
;slot holds a hash table of heuristic ratings, whose keys are the heuristic
;plan objects responsible for the rating.  HEURISTIC-RATINGS-SET contains a
;set of structures called HEURISTIC-RATINGS.  HEURISTIC-RATINGS are simply
;the ratings given to the KSAR by the heuristics.

;The use of hash tables has some overhead in the current implementation of
;DEFINE-UNIT (GBB V1.0) which doesn't reuse deleted unit instances.  Once
;they are reused, the overhead associated with hash tables will be
;minimized.

;Note that without two PRIORITY slots, it would be necessary for the
;recommendation function to decide whether or not to clear the top-most
;priority slot depending upon its stability, and the rating process could be
;undermined by any user manipulation of the value of the PRIORITY slot.

;;;-----------------------------------------------------------------
;;;
;;;  Prioritize the KSAR
;;;
;;;-----------------------------------------------------------------

(defun PRIORITIZE-KSAR (ksar)

  "PRIORITIZE-KSAR ksar

   This function sets the value of the PRIORITY slot in KSAR."

  (let ((ksar-rating (basic-control-ksar$ratings ksar)))
    
    (setf (basic-ksar$priority ksar)
	  (or (ksar-rating$priority ksar-rating)
	      (updated-priority ksar)))))

;;;-----------------------------------------------------------------
;;;
;;;  Update functions for the priority, focus, and heuristic ratings
;;;
;;;-----------------------------------------------------------------

(defun UPDATED-PRIORITY (ksar)

  "UPDATED-PRIORITY ksar

   This function returns the new priority of KSAR."

  ;; note that the focus ratings will be updated as a side effect
  ;; of mapping through them during the user's call to MAP-FOCUS-RATINGS
  ;; during the funcall of *priority-function*.

  (let ((priority (funcall *priority-function* ksar)))

    ;; save the priority value if the function is stable.
    (setf (ksar-rating$priority (basic-control-ksar$ratings ksar))
	  (if (eq *priority-fn-stability* :stable)
              priority
	      NIL))
     priority))

;;;-----------------------------------------------------------------

(defun UPDATED-INTEGRATED-RATING (ksar focus-instance focus-rating-table)

  "UPDATED-INTEGRATED-RATING ksar focus-instance focus-rating-table

   Returns the new integrated rating of KSAR by FOCUS-INSTANCE."

  (let ((integrated-rating
          (funcall (basic-focus-po$integrated-rating-function focus-instance)
                   ksar focus-instance)))

    ;; save the integrated rating if it's stable.
    ;; we call GET-FOCUS-RATING so that a new focus-rating
    ;; structure is created if necessary.
    (setf (focus-rating$integrated-rating
            (get-focus-rating focus-instance focus-rating-table))
	  (if (stable-p focus-instance)
	      integrated-rating
	      NIL))

    integrated-rating))

;;;-----------------------------------------------------------------

(defun UPDATED-HEURISTIC-RATING (ksar heuristic-instance heuristic-rating-table)

  "UPDATED-HEURISTIC-RATING ksar heuristic-instance heuristic-rating-table

   Returns the heuristic rating of KSAR by HEURISTIC-INSTANCE."

  (let ((heuristic-rating
          (funcall (basic-heuristic-po$rating-function heuristic-instance) ksar)))

    ;; save the heuristic rating if it's stable.
    (setf (gethash heuristic-instance heuristic-rating-table)
	  (if (stable-p heuristic-instance)
	      heuristic-rating
	      NIL))

    heuristic-rating))

;;;-----------------------------------------------------------------
;;;
;;;  Mapping functions for focus and heuristic ratings.
;;;  To be called by the application priority and integration functions.
;;;
;;;-----------------------------------------------------------------

(defun MAP-FOCUS-RATINGS (func ksar)

  "MAP-FOCUS-RATINGS func ksar

   Map FUNC over each of the integrated (focus) ratings of KSAR.
   FUNC is passed 2 arguments: the integrated rating and the focus plan
   object unit instance responsible for it.

   MAP-FOCUS-RATINGS returns nil."

  ;; the use of MAP-SPACE has a problem associated with it; the unit-types
  ;; argument doesn't use typep, so that I can't supply BASIC-FOCUS-PO as
  ;; the argument and get all the foci.  The alternatives are to rewrite
  ;; map-space to accept typep (not trivial- I just tried.), rewrite this
  ;; function as a call to "mapping" call to FIND-UNITS and either (best)
  ;; retrieve only foci or else filter them (this means recompilation), or
  ;; what I'm going to do temporarily; map through all the elements and
  ;; filter them myself.

  (gbb1-debug :focus-ratings "~2%> Mapping all FOCI for: ~a" (basic-ksar$name ksar))
  (let ((focus-rating-table (ksar-rating$focus-ratings (basic-ksar$ratings ksar))))
    (map-space
      #'(lambda (focus-instance)
          (when (and (operative-p focus-instance)
                     (typep focus-instance 'basic-focus-po))
            (gbb1-debug :focus-ratings
                        "~% > Using FOCUS ~a"
                        (basic-focus-po$name focus-instance))
            (let* ((focus-rating (get-focus-rating focus-instance focus-rating-table))
                   (integrated-rating
                     (or (focus-rating$integrated-rating focus-rating)
                         (updated-integrated-rating ksar focus-instance focus-rating-table))))
              (gbb1-debug :focus-ratings "~% > FOCUS ~a gave the integrated rating ~d"
                          (basic-focus-po$name focus-instance) integrated-rating)
              (funcall func integrated-rating focus-instance))))
      t
      *cpl-agenda-path*))

  ;; explicitly return...
  nil)

;;;-----------------------------------------------------------------

(defun MAP-HEURISTIC-RATINGS (fn ksar focus-instance)

  "MAP-HEURISTIC-RATINGS fn ksar focus-instance

   Map FN over each of the ratings given to KSAR by the heuristics
   implementing FOCUS-INSTANCE.  FN is passed 2 arguments: the heuristic
   rating and the heuristic plan object unit instance responsible for it.

   MAP-HEURISTIC-RATINGS returns NIL."

  ;; we iterate through all the (operative) heuristics that implement
  ;; FOCUS-INSTANCE, checking to see if we've stored the value for the
  ;; heuristic rating within KSAR.  If not, and the heuristic is stable, we
  ;; store it there.

  (gbb1-debug :heuristic-ratings
              "~%  >Mapping over all HEURISTICS of ~a in order to rate ~a"
	      (basic-focus-po$name focus-instance) (basic-ksar$name ksar))
  
  (let ((heuristic-rating-table
          (focus-rating$heuristic-ratings
            (get-focus-rating focus-instance
                              (ksar-rating$focus-ratings (basic-ksar$ratings ksar))))))
    (dolist (heuristic-instance (basic-focus-po$implemented-by focus-instance))
      (when (operative-p heuristic-instance)
	(let ((heuristic-rating
                (or (gethash heuristic-instance heuristic-rating-table)
                    (updated-heuristic-rating ksar heuristic-instance heuristic-rating-table))))
	(gbb1-debug :heuristic-ratings "~%    >HEURISTIC ~a gave a rating of: ~d"
		    (basic-heuristic-po$name heuristic-instance) heuristic-rating)
	(funcall fn heuristic-rating heuristic-instance)))))

  ;; explicitly return...
  nil)

;;;-----------------------------------------------------------------
;;;
;;;  Accessor function for the focus hash table.
;;;
;;;-----------------------------------------------------------------


(defun GET-FOCUS-RATING (focus-instance focus-ratings-set)
  
  "GET-FOCUS-RATING focus-instance focus-ratings-set

   To be used instead of GETHASH to retrieve the focus-ratings structure
   corresponding to FOCUS-INSTANCE from FOCUS-RATINGS-SET."
  
  ;; We need to create a new focus-ratings structure and return it if
  ;; one doesn't exist.
  
  (or (gethash focus-instance focus-ratings-set)
      (setf (gethash focus-instance focus-ratings-set)
	    (make-focus-rating :integrated-rating nil
				:heuristic-ratings (make-hash-table)))))

;;; -----------------------------------------------------------------
;;;
;;;  Utility Functions for scheduling.
;;;
;;; -----------------------------------------------------------------
  
(defun OPERATIVE-P (plan-object)

  "OPERATIVE-P plan-object

   Returns T if PLAN-OBJECT is operative, NIL if inoperative."

  (eq (basic-control-po$status plan-object) :operative))

;;;-----------------------------------------------------------------------------

(defun STABLE-P (plan-object)

  "STABLE-P plan-object

   Returns T if PLAN-OBJECT is stable, NIL if dynamic."

  (eq (basic-control-po$stability plan-object) :stable))

;;; ============================================================================
;;; 
;;;  KS Shell Initialization and setup functions
;;;
;;; ============================================================================

(defun RESET-GBB1 nil

  "RESET-GBB1 nil

   To be called before redefining a new GBB1 system."

  (reset-execution-shell)
  (reset-KS-shell))

;;; -----------------------------------------------------------------

(defun SUM-OF-WEIGHTS-TIMES-RATINGS (ksar focus-instance)

  "SUM-OF-WEIGHTS-TIMES-RATINGS ksar focus-instance

   The default integration rating function.
   Calculates and returns the integrated rating for KSAR as the sum of
   the product of each heuristic rating of KSAR multiplied
   by its weight."

  (let ((total 0))
    (map-heuristic-ratings
      #'(lambda (rating heuristic-po)
          (incf total (* (basic-heuristic-po$weight heuristic-po) rating)))
      ksar
      focus-instance)
  total))

;;; -----------------------------------------------------------------

(defun PREFER-CONTROL-KSARS (ksar)

  "PREFER-CONTROL-KSARS ksar

   The default heuristic rating function.
   Gives high ratings to control ksars, low ones to domain ksar."

  (if (eq (basic-ksar$ksar-type ksar) :control)
      1000
      0))

;;; -----------------------------------------------------------------
;;;
;;;   Default control rating objects.
;;;
;;; -----------------------------------------------------------------

(define-unit (default-focus-po (:include basic-focus-po)
			       (:export)))

(define-unit (default-heuristic-po (:include basic-heuristic-po)
				   (:export)))

(pushnew 'default-focus-po *po-units*)
(pushnew 'default-heuristic-po *po-units*)


(defun BUILD-DEFAULT-PO-INSTANTIATIONS ()

  "BUILD-DEFAULT-PO-INSTANTIATIONS nil

   Create the default control plan rating functions."

  (make-default-focus-po 
    :NAME                       "Default Focus" 
    :STABILITY                  :STABLE
    :STATUS                     :OPERATIVE
    :HEURISTICS                 '(default-heuristic-po)
    :GOAL-FUNCTION              #'(lambda () nil)
    :INTEGRATED-RATING-FUNCTION #'sum-of-weights-times-ratings
    :KS                         nil)
    
  (make-default-heuristic-po
    :NAME            "Default Heuristic"
    :STABILITY       :STABLE
    :STATUS          :OPERATIVE
    :WEIGHT          1
    :IMPLEMENTOR-OF  (find-unit-by-name "Default Focus" 'default-focus-po)
    :RATING-FUNCTION #'prefer-control-ksars
    :KS              nil))


;;; -----------------------------------------------------------------
;;;
;;;      KS Shell Blackboard Definition.
;;;
;;; -----------------------------------------------------------------


(define-space KSs
  :units ((basic-ks :plus-subtypes))
  :dimensions ((ks-phase :enumerated (:default-phase))))

(define-unit-mapping (basic-ks) (KSs)
  :indexes (ks-phase)
  :index-structure ((ks-phase :groups)))


(define-spaces (:triggered :obviated :executable :executed)
  :units ((basic-ksar :plus-subtypes))
  :dimensions ((ksar-phase :enumerated (:default-phase))))

(define-unit-mapping (basic-ksar) (:triggered :obviated :executable :executed)
  :indexes (ksar-phase)
  :index-structure ((ksar-phase :groups)))


(define-space events
  :units (gbb1-event)
  :dimensions ((execution-cycle :ordered (-1 30))
	       (triggering-space :enumerated (nil))))

(define-unit-mapping (gbb1-event) (events)
  :indexes (execution-cycle triggering-space)
  :index-structure ((execution-cycle :subranges (:start :end (:width 1)))
                    (triggering-space :groups)))


;; You would like to define the control plan space like this:
;;
;;    (define-space control-plan
;;      :units ((basic-control-po :plus-subtypes))
;;      :dimensions ((control-type :enumerated
;;                                 (:strategy :focus :heuristic))))
;;
;; Unfortunately, the BASIC-CONTROL-PO unit type doesn't have CONTROL-TYPE
;; as a dimensional index.  (This is because it is easier to initialize
;; the control type slot in the basic strategy, focus, and heuristic
;; units.)  The problem arises when you try to do something like
;;
;;    (find-units t *cpl-agenda-path* ...)
;;
;; You get an error that "(BASIC-CONTROL-PO :PLUS-SUBTYPES) does not have
;; CONTROL-TYPE as a dimensional index." 
;;
;; Anyway, the consequence of this is that all PO types must include one
;; of BASIC-STRATEGY-PO, BASIC-FOCUS-PO, or BASIC-HEURISTIC-PO and not
;; BASIC-CONTROL-PO.

(define-space control-plan
  :units ((basic-strategy-po :plus-subtypes)
          (basic-focus-po :plus-subtypes)
          (basic-heuristic-po :plus-subtypes))
  :dimensions ((control-type :enumerated (:strategy :focus :heuristic))))

(define-unit-mapping (basic-strategy-po basic-focus-po basic-heuristic-po)
                     (control-plan)
  :indexes (control-type)
  :index-structure ((control-type :groups)))


(define-blackboard agendas (:triggered :executable :obviated :executed)
  "Holds the 4 agendas.")

(define-blackboard control (control-plan agendas)
  "The control blackboard")

(define-blackboard KB (KSs)
  "The knowledge base blackboard")

(define-blackboard GBB1 (control KB)
  "The top-level control shell blackboard structure of GBB1.")


(defun instantiate-gbb1-blackboards ()
  "Instantiate all the GBB1 KS Shell blackboards.
   Set path variables."

  (update-space-dimension 'KSs 'ks-phase *ks-phases*)
  (update-space-dimension '(:triggered :obviated :executable :executed)
			  'ksar-phase
			  *ksar-phases*)
  (instantiate-blackboard-database 'gbb1 :mode :append)

  (update-space-dimension 'events
			  'triggering-space
			  (get-triggering-space-labels))
  (update-space-dimension 'events
			  'execution-cycle
			  `(-1 ,*max-execution-cycles*))
  (instantiate-blackboard-database 'events :mode :append)

  (setf *exe-agenda-path* (make-paths :paths '(gbb1 control agendas :executable))
	*tri-agenda-path* (make-paths :paths '(gbb1 control agendas :triggered))
	*obv-agenda-path* (make-paths :paths '(gbb1 control agendas :obviated))
	*exd-agenda-path* (make-paths :paths '(gbb1 control agendas :executed))
	*eve-agenda-path* (make-paths :paths '(events))
	*cpl-agenda-path* (make-paths :paths '(gbb1 control control-plan))
	*str-agenda-path* (make-paths :paths '(gbb1 control control-plan))
	*foc-agenda-path* (make-paths :paths '(gbb1 control control-plan))
	*heu-agenda-path* (make-paths :paths '(gbb1 control control-plan))
	*KSs-agenda-path* (make-paths :paths '(gbb1 KB KSs)))

  ;; Reset the execution cycle during default PO instantiation.
  (setf *execution-cycle* -1)
  (build-default-po-instantiations)

  ;; Make the KSs.  This has the side effect of also putting them on
  ;; the KS space.
  (setf *execution-cycle* 0)
  (mapc #'eval *make-KSs*)

  nil)


(defun gbb1-blackboards-instantiated-p ()
  "Returns true if the GBB1 KS Shell blackboards are instantiated."
  (member 'gbb1 gbb::*blackboard-database*
	  :test #'eq :key #'gbb::db-node.name))

;;; --------------------------------------------------------------------------
;;;				End of File
;;; --------------------------------------------------------------------------
