;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB1; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.LOCAL.GBB1]GBB1-FUNS.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Wednesday, February 8, 1989  16:20:38 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 308) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *           GBB1 Execution Shell: Function Definitions
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Philip Johnson
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  03-05-87 File Created.  (Johnson)
;;;  05-19-87 Conversion to GBB1 Execution Shell Version 1.0  (Johnson)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *


(in-package "GBB1")

(use-package '(lisp umass-extended-lisp gbb))

(export '(continue-gbb1
          current-ks-phase
          current-ksar-phase
          define-gbb1-output
          define-gbb1-parameters
          run-gbb1
          set-ks-phase
          set-ksar-phase
          trigger-event-class-p
          trigger-event-level-p
          trigger-event-slot-p
          trigger-event-type-p
          trigger-unit))

;;; -----------------------------------------------------------------
;;;
;;;                Condition testing functions

;;; This following implementation of conditions was chosen to satisfy 4 criterion:
;;;   1.  Support for recheck intervals:  Once a condition is satisfied, it is
;;;       no longer tested until the next recheck interval.
;;;   2.  Maintaining the ordering of the conditions: conditions are always
;;;       evaluated in the order defined (though there may be "gaps" in the ordering
;;;       due to recheck intervals/stability.
;;;   3.  Support for the stability attribute of conditions: if a condition is
;;;       defined as stable, it is never retested after it succeeds once.
;;;   4.  Avoidance of consing: this implementation does not cons.

;;; The preconditions and obviation conditions are each implemented by a
;;; set of 3 parallel vectors, called active-<condition type>,
;;; all-<condition type>, and dynamic-<condition type>.  The purpose of
;;; the active vector is to indicate the current set of conditions to be
;;; tested.  The purpose of the all vector is to indicate the set of
;;; conditions to be "restored" during a recheck interval.  The dynamic
;;; vector indicates which conditions are stable, and thus should not be
;;; restored once they succeed.

;;; Initially the active vector and the all vector each contain pointers
;;; to the conditions, and the dynamic vector contains a NIL if the
;;; corresponding condition in the all/active vectors is stable, non-NIL
;;; otherwise.  The conditions are tested by running through each element
;;; in the active vector and funcalling it if it's non-NIL.  If the
;;; funcall returns non-NIL, then that element is set to NIL, and
;;; (furthermore) if the corresponding element in the dynamic array is
;;; NIL, then in addition, the corresponding element in the active array
;;; is set to NIL.

;;; Finally, when a precondition interval comes around, the elements in
;;; the active array are replaced by those in the all array, which resets
;;; the active array to all of the dynamic conditions and all the stable
;;; conditions that have not yet succeeded.

;;; -----------------------------------------------------------------
;;;
;;;   Check the pre and obviation conditions of an entire agenda.
;;;
;;; -----------------------------------------------------------------

(defun CHECK-PRECONDITIONS (agenda-path &optional (recheck? nil))

  "CHECK-PRECONDITIONS agenda-path reset?

   Resets (if necessary) and checks the preconditions of all KSARs on AGENDA-PATH"

  (map-space #'(lambda (ksar)
		 (if *meter-gbb1* (incf *retrieved-ksars*))
		 (when recheck?
		   (replace (basic-ksar$active-preconditions ksar) (basic-ksar$all-preconditions ksar)))
		 
		 (if (preconditions-satisfied ksar)
		     (move-unit-to-agenda ksar *exe-agenda-path*)
		     (move-unit-to-agenda ksar *tri-agenda-path*)))
	     t agenda-path))

;;; -----------------------------------------------------------------

(defun CHECK-OBV-CONDITIONS (agenda-path &optional (recheck? nil))

  "CHECK-OBV-CONDITIONS agenda-path reset?

   Resets (if necessary) and checks the obviation conditions of all KSARs on AGENDA-PATH"

  (map-space #'(lambda (ksar)
		 (if *meter-gbb1* (incf *retrieved-ksars*))
		 (when recheck?
		   (replace (basic-ksar$active-obv-conditions ksar) (basic-ksar$all-obv-conditions ksar)))
		 (when (obv-conditions-satisfied ksar)
		   (move-unit-to-agenda ksar *obv-agenda-path*)))
	     t agenda-path))

;;; -----------------------------------------------------------------
;;;
;;;   Check the pre and obviation conditions of an individual KSAR.
;;;
;;; -----------------------------------------------------------------

(defun PRECONDITIONS-SATISFIED (ksar &optional all)
  
  "PRECONDITIONS-SATISFIED ksar

  Returns T if the preconditions of KSAR are satisfied, NIL otherwise."

  (let ((*this-ksar* ksar)) ;; declared special!

    ;; this dotimes returns T if it completes successfully (indicating all preconditions satisfied)
    (dotimes (index (basic-ksar$precondition-vector-size ksar) t)
      (let ((func (aref (if all
                            (basic-ksar$all-preconditions ksar)
                            (basic-ksar$active-preconditions ksar))
                        index)))
	(when func                 ;; may be NIL if previously satisfied
	  (cond ((funcall func)  ;; when condition satisfied
		 (if *meter-gbb1* (incf *tested-preconditions*))
		 (setf (aref (basic-ksar$active-preconditions ksar) index) nil)
		 (unless (aref (basic-ksar$dynamic-preconditions ksar) index)    ;; if stable
		   (setf (aref (basic-ksar$all-preconditions ksar) index) nil))) ;; remove it
		(t
		 ;; otherwise the condition wasn't satisfied, so return NIL and don't
		 ;; check any other preconditions.
		 (return NIL))))))))

;;; -----------------------------------------------------------------

(defun OBV-CONDITIONS-SATISFIED (ksar)
  
  "OBV-CONDITIONS-SATISFIED ksar

  Returns T if the obviation conditions of KSAR are satisfied, NIL otherwise."
  
  (let ((*this-ksar* ksar))
    
    (dotimes (index (basic-ksar$obviation-vector-size ksar) t)
      (let ((func (aref (basic-ksar$active-obv-conditions ksar) index)))
	(when func                 ;; may be NIL if previously satisfied
	  (cond ((funcall func)    ;; when condition satisfied
		 (if *meter-gbb1* (incf *tested-obviation-conditions*))
		 (setf (aref (basic-ksar$active-obv-conditions ksar) index) nil)
		 (unless (aref (basic-ksar$dynamic-obv-conditions ksar) index)    ;; if stable
		   (setf (aref (basic-ksar$all-obv-conditions ksar) index) nil))) ;; remove it
		(t ;; otherwise the condition wasn't satisfied, so return NIL and don't
                 ;; check any other preconditions.
		 (return NIL))))))))

;;; -----------------------------------------------------------------
;;;
;;;   Trigger condition handling
;;;
;;; -----------------------------------------------------------------

(defvar *trace-triggers* nil
  "If This flag is true then information about triggering will be printed.")


(defun CHECK-TRIGGER-CONDITIONS ()

  "CHECK-TRIGGER-CONDITIONS nil

   Checks the trigger conditions of all the KSs active during the current
   phase of problem solving."

  (find-units t
	      *KSs-agenda-path*
	      `(:pattern-object
		 (:index-type ks-phase-set
		  :index-object ,*current-ks-phase-list*))
	      :filter-after #'(lambda (ks)
				(when *meter-gbb1* (incf *retrieved-kss*))
				(when *trace-triggers*
				  (format *trace-stream* "~&Considering ~s. " ks))
				(trigger-conditions-satisfied ks)
				;; make sure to return NIL to avoid consing!
				NIL)))

;;; -----------------------------------------------------------------

(defun TRIGGER-CONDITIONS-SATISFIED (ks)

  "TRIGGER-CONDITIONS-SATISFIED ks

   Test the trigger conditions of KS against those events generated
   during the last execution cycle on the spaces in KS's FROM-BB slot.
   If the trigger conditions are satisfied, generate ksars."

  ;; if the FROM-BB slot is nil, we find all the events from the last cycle.

  (find-units t *eve-agenda-path*
	      `(
		:AND
		 (:element-match :exact
		  :pattern-object (:index-type execution-cycle-type
				   :index-object ,(- *execution-cycle* 1)))
		 ,@(when (basic-ks$from-bb ks)
		     `((:match (:count 1)
			:mismatch (:all-but 1)
			:pattern-object (:index-type triggering-space-set
					 :index-object ,(basic-ks$from-bb ks)))))
		 )
	      
	      :filter-after #'(lambda (event)
				(when *meter-gbb1* (incf *retrieved-events*))
				(when *trace-triggers*
				  (format *trace-stream* "~& With ~s. " event))
				(test-ks-against-event ks event)
				;; make sure to return NIL to avoid consing!
				NIL)
	      ))

;;; -----------------------------------------------------------------

(defun TEST-KS-AGAINST-EVENT (ks event)

  "TEST-KS-AGAINST-EVENT KS EVENT

   Tests the trigger conditions of the KS against EVENT.
   Generates KSAR(s) if successful."

  (let ((success nil))
    (when *trace-triggers*
      (format *trace-stream* "~&  Testing ~s against ~s... " ks event))

    ;; when all the trigger conditions are satisfied...
    (when (dolist (trigger-condition (basic-ks$trigger-conditions ks) t)
	    (if *meter-gbb1* (incf *tested-triggers*))
	    (setf *trigger-event* event
		  *this-ks*       ks)
	    (unless (funcall trigger-condition)
	      (return nil)))
      (setf success t)
      (when *trace-triggers*
	(format *trace-stream* " Succeeded.~%"))
      ;;generate ksars
      (generate-ksars KS event))

    (when (and *trace-triggers* (not success))
      (format *trace-stream* " Failed.~%"))

    nil))

;;; -----------------------------------------------------------------

(defun GENERATE-KSARS (KS event)

  "GENERATE-KSARS KS

   Generates one or more KSARs from the argument KS, depending upon the
   length of the list of context slot values.  The preconditions are checked
   immediately upon creation, and if they are satisfied, the KSAR will be
   placed on the executable rather than triggered agenda."
  
  (let ((context-slots (first (basic-ks$context-slots KS)))
	(context-slot-values-list (eval (second (basic-ks$context-slots KS)))))

    (if context-slots
	(mapc #'(lambda (context-slot-values)
		  (let ((make-name (or (basic-ks$ksar-make-name KS)
				       (setf (basic-ks$ksar-make-name KS)
					     (form-symbol "MAKE-" (basic-ks$ksar-unit-type KS)))))
			(ksar nil))
		    
		    (setf ksar
			  (apply make-name
				 `(:trigger-event            ,event
				   :ksar-num                 ,(incf *ksar-id*)
				   :triggering-KS            ,KS
				   :ksar-type                ,(basic-ks$ks-type ks)
				   :ks-unit-type             ,(type-of ks)
				   :cost                     ,(basic-ks$cost ks)
				   :reliability              ,(basic-ks$reliability ks)

				   :active-preconditions     ,(copy-vector (basic-ks$active-preconditions ks))
				   :all-preconditions        ,(copy-vector (basic-ks$all-preconditions ks))
				   :dynamic-preconditions    ,(copy-vector (basic-ks$dynamic-preconditions ks))
				   :precondition-vector-size ,(basic-ks$precondition-vector-size ks)

				   :active-obv-conditions    ,(copy-vector (basic-ks$active-obv-conditions ks))
				   :all-obv-conditions       ,(copy-vector (basic-ks$all-obv-conditions ks))
				   :dynamic-obv-conditions   ,(copy-vector (basic-ks$dynamic-obv-conditions ks))
				   :obviation-vector-size    ,(basic-ks$obviation-vector-size ks)
								  
				   :status                   :triggered
				   ,.(mapcan #'(lambda (slot val)
						 `(,(make-keyword slot) ,val))
					     context-slots context-slot-values)
				   )))

		    (when (preconditions-satisfied ksar)
		      (move-unit-to-agenda ksar *exe-agenda-path*))))
	      context-slot-values-list)

	;; if no context slots, make a single KSAR
	(let ((make-name (or (basic-ks$ksar-make-name KS)
			     (setf (basic-ks$ksar-make-name KS)
				   (form-symbol "MAKE-" (basic-ks$ksar-unit-type KS)))))
	      (ksar nil))
		    
	  (setf ksar (apply make-name
			    `(:trigger-event            ,event
			      :ksar-num                 ,(incf *ksar-id*)
			      :triggering-KS            ,KS
			      :ksar-type                ,(basic-ks$ks-type ks)
			      :ks-unit-type             ,(type-of ks)
								  
			      :active-preconditions     ,(copy-vector (basic-ks$active-preconditions ks))
			      :all-preconditions        ,(copy-vector (basic-ks$all-preconditions ks))
			      :dynamic-preconditions    ,(copy-vector (basic-ks$dynamic-preconditions ks))
			      :precondition-vector-size ,(basic-ks$precondition-vector-size ks)
								  
			      :active-obv-conditions    ,(copy-vector (basic-ks$active-obv-conditions ks))
			      :all-obv-conditions       ,(copy-vector (basic-ks$all-obv-conditions ks))
			      :dynamic-obv-conditions  ,(copy-vector (basic-ks$dynamic-obv-conditions ks))
			      :obviation-vector-size    ,(basic-ks$obviation-vector-size ks)
								  
			      :status                   :triggered)))
		    
	           (when (preconditions-satisfied ksar)
		      (move-unit-to-agenda ksar *exe-agenda-path*)))
	)))

;;; -----------------------------------------------------------------

(defun GET-TRIGGERING-SPACE-LABELS ()

  "GET-TRIGGERING-SPACE-LABELS nil

   This function returns a list of spaces indicated in the from-bb fields of DEFINE-GBB1-KS."

  ;; *TRIGGERING-SPACE-LABELS* has a list of calls to MAKE-PATHS.  We call
  ;; each of them, then remove duplicate spaces, then cons on a :DEFAULT-SPACE
  ;; to the end.  Events generated on a space not indicated will be put on the
  ;; :default-space dimension.  Similarly, KSs which don't specify a from-bb
  ;; will look on the :default-space.

  ;; NOTE WELL: As a side effect *FROM-BB-SPACE-INSTANCES* is set here.  It is
  ;; then used when making event instances.

  (let ((path-structures nil))
    (mapc #'(lambda (make-paths-call)
		(setf path-structures
		      (union (eval make-paths-call)
			      path-structures)))
	    *triggering-space-labels*)
    (setf *from-bb-space-instances* (mapcar #'get-space-instance-from-path-structure path-structures))
    
    (cons :default-space *from-bb-space-instances*)))

;;; -----------------------------------------------------------------
;;;
;;;      Setup and initialization functions.
;;;
;;; -----------------------------------------------------------------

(defun RESET-EXECUTION-SHELL ()

  "RESET-EXECUTION-SHELL nil

   Clears the values of internal variables useful to GBB1.  This function
   should be called before defining a new GBB1 system."

  ;; I need to reset the bb db because I append the control shell BBs, and need
  ;; to make sure the old bb structure is clobbered.
  (instantiate-blackboard-database :mode :overwrite)

  (setf  *ksar-units*              nil
	 *ks-units*                nil
	 *ksar-phases*             (list :default-phase)
	 *ks-phases*               (list :default-phase)
	 *current-ks-phase-list*   (list :default-phase)
	 *current-ksar-phase-list* (list :default-phase)
	 *triggering-space-labels* nil 
	 *make-kss*                nil))

;;; -----------------------------------------------------------------

(defun RESET-KS-SHELL ()

  "RESET-KS-SHELL nil

   Clears the internal variables for the KS shell"

  (setf *po-units* nil))

;;; -----------------------------------------------------------------


(defun RESET-GBB1-COUNTERS ()

  "RESET-GBB1-COUNTERS nil

   To be called before rerunning the currently defined GBB1 system."

  (setf *execution-cycle* 0)
  (when *meter-gbb1*
   (setf *retrieved-events* 0
	 *generated-events* 0
	 *tested-preconditions* 0
	 *tested-obviation-conditions* 0
	 *tested-triggers* 0
	 *retrieved-KSARs* 0
	 *retrieved-KSs* 0)) )

;;; -----------------------------------------------------------------
;;;
;;;      The Interpretor, Agenda Manager, and Scheduler.
;;;
;;; -----------------------------------------------------------------

(defun PROCESS-INITIAL-KS (ks-unit-type)

  "PROCESS-INITIAL-KS ks

   Checks the passed KS for validity, and generates a KSAR for it"

  (let* ((ks-instance (find-unit-by-name (string ks-unit-type) ks-unit-type))
         (ksar-unit-type (and ks-instance
                              (basic-ks$ksar-unit-type ks-instance))))

    (when (null ks-instance)
      (error "The initial KS could not be found."))

    ;; Now generate a single KSAR for this KS. This is normally done
    ;; by GENERATE-KSARs, but we don't want to check context slots, etc.
    ;; (at some future time GENERATE-KSARs should be made to handle this case.)

    (make-unit
      ksar-unit-type
      :ksar-num                 (incf *ksar-id*)
      :triggering-KS            ks-instance
      :active-preconditions     (copy-vector (basic-ks$active-preconditions ks-instance))
      :all-preconditions        (copy-vector (basic-ks$all-preconditions ks-instance))
      :dynamic-preconditions    (copy-vector (basic-ks$dynamic-preconditions ks-instance))
      :precondition-vector-size (basic-ks$precondition-vector-size ks-instance)
      
      :active-obv-conditions    (copy-vector (basic-ks$active-obv-conditions ks-instance))
      :all-obv-conditions       (copy-vector (basic-ks$all-obv-conditions ks-instance))
      :dynamic-obv-conditions   (copy-vector (basic-ks$dynamic-obv-conditions ks-instance))
      :obviation-vector-size    (basic-ks$obviation-vector-size ks-instance)
      :ks-unit-type             ks-unit-type
      :status                   :executable)))

;;; -----------------------------------------------------------------

(defun INTERPRET (ksar)
  
  "INTERPRET ksar

   Execute the actions of one KSAR."
  
  ;; Note that no explicit interpretation-time bindings of KS variables are
  ;; done in the execution shell; this must be done in the action-function.
  
  (cond ((obv-conditions-satisfied ksar)
         (move-unit-to-agenda ksar *obv-agenda-path*))
        ((preconditions-satisfied ksar)
         (let ((*this-ksar* ksar))
           (funcall (basic-ks$action-function (basic-ksar$triggering-ks ksar)) ksar))
         (setf (basic-ksar$executed-cycle ksar) *execution-cycle*)
         (move-unit-to-agenda ksar *exd-agenda-path*))
        (t
         (move-unit-to-agenda ksar *tri-agenda-path*))))

;;; -----------------------------------------------------------------

(defun UPDATE-AGENDAS (&optional force-recheck)
  
  "UPDATE-AGENDAS nil

   Trigger KSs, and check preconditions and obviation conditions of KSARs."
  
  ;; Optimization: Avoid immediately rechecking the preconditions of a
  ;; KSAR moved back to the triggered from the executable agenda since
  ;; it's preconditions were no longer satisfied.
  
  (flet ((recheck-necessary? (interval)
           ;; Returns true if a recheck is necessary for INTERVAL.
           (or force-recheck
               (and interval
                    (= 0 (mod *execution-cycle* interval))))))

    (let ((precondition-recheck
                (recheck-necessary? *precondition-recheck-interval*))
          (obviation-recheck
                (recheck-necessary? *obviation-recheck-interval*)))
    
      ;; First, recheck the preconditions of all executable KSARs, and move
      ;; them back to the triggered state if they are no longer satisfied.
      (when precondition-recheck
        (check-preconditions *exe-agenda-path* :recheck))
    
      ;; Now check the preconditions of triggered KSARs, and make them
      ;; executable if they're satisfied.
      (check-preconditions *tri-agenda-path*)
    
      ;; Now trigger any KSs
      (check-trigger-conditions)
    
      ;; Finally, obviate any KSARs.
      (when obviation-recheck
        (check-obv-conditions *tri-agenda-path* :recheck)
        (check-obv-conditions *exe-agenda-path* :recheck)))))

;;; -----------------------------------------------------------------

(defun HIGHEST-PRIORITY-KSARS ()

  "HIGHEST-PRIORITY-KSARS nil

   This function applies the priority function to all of the KSARs on the
   executable agenda and returns a list of those with the highest priority."

  ;; for MAXIMUM efficiency, we really should be defining this after we know
  ;; whether or not the priority function is stable.

  (let ((max-value -99999)
	(max-units  nil))

    ;; Define local filter functions:
    (flet ((throw-away-lower-prio-units (unit)
	     ;; when stable, use the prio slot value if available, otherwise set it.
	     (let ((unit-prio (if *priority-fn-stability*
				  (or (basic-ksar$priority unit)
				      (setf (basic-ksar$priority unit)
                                            (funcall *priority-function* unit)))
				  (funcall *priority-function* unit))))
	       (gbb1-debug :priority "~&> Priority for ~a is ~d"
                           (basic-ksar$name unit) unit-prio)
	       (>= unit-prio max-value)))

	   (update-max-units-list (unit)
             ;; If this function runs then the unit is >= what we've seen
	     ;; We push it on the list if it's =, replace the list if >
	     (let ((unit-prio (or (basic-ksar$priority unit)
                                  (funcall *priority-function* unit))))
	     (cond ((= max-value unit-prio)
		    (push unit max-units))
		   (t (setf max-units (list unit)
			    max-value unit-prio)))
	     ;; regardless, we return nil to avoid consing within the FIND function
	     nil)))

      (find-units
	t
	*exe-agenda-path*
	:ALL
	:filter-before #'throw-away-lower-prio-units
	:filter-after  #'update-max-units-list))

    ;; Return the list of units with the maximum slot value.
    max-units))

;;; -----------------------------------------------------------------

(defun SCHEDULE-KSAR ()

  "SCHEDULE-KSAR nil

   Prioritize and recommend a KSAR for execution from among those executable."

  (let ((high-priority-ksars (highest-priority-ksars))
        (selected-ksar nil))
    (loop
      (setf selected-ksar (funcall *recommendation-function* high-priority-ksars))
      (when (preconditions-satisfied selected-ksar t)
        (return selected-ksar))
      (setf high-priority-ksars
            (delete selected-ksar high-priority-ksars :test #'eq))
      (when (null high-priority-ksars)
        (return nil)))))

;;; -----------------------------------------------------------------
;;;
;;;      Generic (common to all layers) GBB1 user interface functions
;;;
;;; -----------------------------------------------------------------

(defun DEFINE-GBB1-PARAMETERS (&key priority-fn (priority-fn-stability :dynamic)
			            recommendation-fn termination-fn
				    precondition-recheck-interval
				    obviation-recheck-interval
				    (max-execution-cycles 30))

  "DEFINE-GBB1-PARAMETERS &key priority-fn priority-fn-stability
                               recommendation-fn termination-fn
                               precondition-recheck-interval
                               obviation-recheck-interval
                               (max-execution-cycles 30)

   PRIORITY-FN is passed a KSAR and returns its priority.
   PRIORITY-FN-STABILITY: :STABLE indicates the priority won't change, :DYNAMIC
     means to rerate the KSAR's priority every time.
   TERMINATION-FN returns T if the problem has been solved.
   RECOMMENDATION-FN is passed a list of the highest priority KSARs and
     returns the one chosen to execute.
   PRECONDITION-RECHECK-INTERVAL specifies how frequently dynamic preconditions
     must be rechecked (if NIL, are never rechecked).
   OBVIATION-RECHECK-INTERVAL specifies how frequently dynamic obviation conditions
     must be rechecked. (if NIL, are never rechecked).
   MAX-EXECUTION-CYCLES is the maximum number of execution cycles allowed."

  (setf *max-execution-cycles* max-execution-cycles)
  (when priority-fn
    (setf *priority-function* priority-fn))
  (when termination-fn
    (setf *termination-function* termination-fn))
  (case priority-fn-stability
    (:stable (setf *priority-fn-stability* t))
    (:dynamic (setf *priority-fn-stability* nil))
    (otherwise (error "Priority-fn-stability must be supplied :STABLE or :DYNAMIC.")))
  (when recommendation-fn
    (setf *recommendation-function* recommendation-fn))
  (when precondition-recheck-interval
    (setf *precondition-recheck-interval* precondition-recheck-interval))
  (when obviation-recheck-interval
    (setf *obviation-recheck-interval* obviation-recheck-interval)))

;;; -----------------------------------------------------------------

(defun DEFINE-GBB1-OUTPUT (&key (trace-fn            nil)
			        (trace-print-points  *trace-list*)
				(print-unit-width    *unit-width*)
				(output-stream       *standard-output*)
				)

  "DEFINE-GBB1-OUTPUT &key (trace-fn            nil)
			   (trace-print-points  '(:before-agenda-update :final-state))
			   (print-unit-width    *unit-width*)
                           (output-stream       *standard-output*)

   Determines the output produced by the execution of GBB1.

   TRACE-FN, if supplied, is called immediately after the execution of a
        KSAR, and after the execution cycle terminates.

   TRACE-PRINT-POINTS is a list of keywords indicating when to print
       trace information.  The possible keywords are: :ALL, :FOCUS-RATINGS,
       :HEURISTIC-RATINGS, :PRIORITY, :BEFORE-INTERPRETATION,
       :BEFORE-AGENDA-UPDATE, :BEFORE-SCHEDULING-KSAR, and :FINAL-STATE.
       If TRACE-PRINT-POINTS is nil, nothing is printed.

   PRINT-UNIT-WIDTH specifies the size of each column in the agenda display.

   OUTPUT-STREAM specifies where the output should go."

  (setf *trace-function* trace-fn
	*unit-width*     print-unit-width
	*trace-stream*   output-stream)
  (cond ((or (eq trace-print-points :all)
	     (equal trace-print-points '(:all)))
	 (setf *trace-list* *all-trace-points*))
	(t (mapc #'(lambda (x)
		     (unless (member x *all-trace-points*)
		       (error "~s is not a valid Trace Print Point." x)))
		 trace-print-points)
	   (setf *trace-list* trace-print-points)))
  nil)

;;; -----------------------------------------------------------------

(defun RUN-GBB1 (&key initial-ks)

  "RUN-GBB1 &key initial-ks

   Runs the execution or KS shell (whichever is loaded)."

  (reset-gbb1-counters)
  (unless (gbb1-blackboards-instantiated-p)
    (instantiate-gbb1-blackboards))

  (let ((selected-ksar nil))
    ;; generates a KSAR for the initial KS.
    (setf selected-ksar (process-initial-ks initial-ks))

    (loop

      (gbb1-debug :before-interpretation (basic-ksar$name selected-ksar))
      (INTERPRET selected-ksar)

      (gbb1-debug :before-agenda-update (basic-ksar$name selected-ksar))
      (when *trace-function* (funcall *trace-function*))
      (incf *execution-cycle*)
      (UPDATE-AGENDAS)

      (when (or (empty-executable-agenda-p)
		(funcall *termination-function*))
	(return))

      (gbb1-debug :before-scheduling-ksar)
      (setf selected-ksar (or (SCHEDULE-KSAR)
			      (error "Scheduler was unable to select a ksar."))))

    ;; END OF EXECUTION CYCLE
    (gbb1-debug :final-state)
    (when *trace-function* (funcall *trace-function*))))

;;; -----------------------------------------------------------------

(defun CONTINUE-GBB1 ()

  "CONTINUE-GBB1 nil

   Resumes a GBB1 session."

  (let ((selected-ksar nil))

    ;; Update the agendas on the theory that the user may have done
    ;; something to change the agendas.  (In fact that is probably
    ;; why the run was interrupted.)
    (UPDATE-AGENDAS t)

    (loop

      (gbb1-debug :before-scheduling-ksar)
      (setf selected-ksar (or (SCHEDULE-KSAR)
			      (error "Scheduler was unable to select a ksar.")))

      (gbb1-debug :before-interpretation (basic-ksar$name selected-ksar))
      (INTERPRET selected-ksar)

      (gbb1-debug :before-agenda-update (basic-ksar$name selected-ksar))
      (when *trace-function* (funcall *trace-function*))
      (incf *execution-cycle*)
      (UPDATE-AGENDAS)

      (when (or (empty-executable-agenda-p)
		(funcall *termination-function*))
	(return)))


    ;; END OF EXECUTION CYCLE
    (gbb1-debug :final-state)
    (when *trace-function* (funcall *trace-function*))))


;;; -----------------------------------------------------------------
;;;
;;;      Phase manipulation functions
;;;
;;; -----------------------------------------------------------------


(defun SET-KS-PHASE (phase)

  "SET-KS-PHASE phase

   Set the current phase(s) to be PHASE"

  (setf *current-ks-phase-list* (assure-list phase)))

;;; -----------------------------------------------------------------

(defun SET-KSAR-PHASE (phase)

  "SET-KSAR-PHASE phase

   Set the current phase(s) to be PHASE"

  (setf *current-ksar-phase-list* (assure-list phase)))

;;; -----------------------------------------------------------------

(defun CURRENT-KS-PHASE ()

  "CURRENT-KS-PHASE nil

   Return a list of the current ks phase(s)"

  (copy-list *current-ks-phase-list*))

;;; -----------------------------------------------------------------

(defun CURRENT-KSAR-PHASE ()

  "CURRENT-KSAR-PHASE nil

   Return a list of the current ks phase(s)"

  (copy-list *current-ksar-phase-list*))


;;; -----------------------------------------------------------------
;;;
;;;	Utility functions for the trigger conditions.
;;;
;;; -----------------------------------------------------------------

(defun TRIGGER-UNIT ()

  "TRIGGER-UNIT nil

   Returns the unit instance which caused the triggering event."

  (gbb1-event$unit-instance *trigger-event*))

;;; -----------------------------------------------------------------

(defun TRIGGER-EVENT-LEVEL-P (path-structure)

  "TRIGGER-EVENT-LEVEL-P path-structure

   Returns T if PATH-STRUCTURE corresponds to the space on 
   which the trigger event originated, NIL otherwise."

  (when (member (get-space-instance-from-path-structure path-structure)
	      (gbb1-event$triggering-space *trigger-event*))
      t))

;;; -----------------------------------------------------------------

(defun TRIGGER-EVENT-CLASS-P (event-class)

  "TRIGGER-EVENT-CLASS-P event-class

   Returns T if EVENT-CLASS = the event class of the trigger event."

  (eq (gbb1-event$event-type *trigger-event*) event-class))

;;; -----------------------------------------------------------------

(defun TRIGGER-EVENT-SLOT-P (slot-or-link &optional (new-value nil value-supplied-p))

  "TRIGGER-EVENT-SLOT-P slot-or-link &optional new-value

   Returns T if the trigger event involved a slot (or link) and
   its name = SLOT-OR-LINK, and if NEW-VALUE was supplied, if the 
   new value of the slot (or link) = VALUE."

  (and
    (eq (gbb1-event$slot-or-link *trigger-event*) slot-or-link)
    (if value-supplied-p
	(eq (gbb1-event$new-value *trigger-event*) new-value)
	t)))

;;; -----------------------------------------------------------------

(defun TRIGGER-EVENT-TYPE-P (unit-type)

  "TRIGGER-EVENT-TYPE-P unit-type

   Returns T if the unit type of the triggering event unit instance
   equals UNIT-TYPE, NIL otherwise."

  (eq (type-of (gbb1-event$unit-instance *trigger-event*)) unit-type))


;;; -----------------------------------------------------------------
;;;
;;;      Metering function
;;;
;;; -----------------------------------------------------------------

(defun DISPLAY-METER-INFO ()
  (format nil "~%Retrieved events: ~d
             ~%Generated events: ~d
             ~%Tested preconditions: ~d
             ~%Tested obvconditions: ~d
             ~%Tested triggers: ~d
             ~%Retrieved KSARs: ~d
             ~%Retrieved KSs: ~d"
	  *retrieved-events* *generated-events* *tested-preconditions*
	  *tested-obviation-conditions* *tested-triggers* *retrieved-KSARs* *retrieved-KSs*))

;;; --------------------------------------------------------------------------
;;;				End of File
;;; --------------------------------------------------------------------------

