;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:USER; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.DEVELOPMENT.GBB1.EXAMPLES]BUILD-HYP-PYRAMID-DEMO.LISP *-*
;;;; *-* Last-Edit: Friday, September 16, 1988  10:32:47; Edited-By: Cork *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                 Simple Execution Shell Example: 
;;;; *                 The Hypothesis Pyramid Builder
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Philip Johnson and Daniel Corkill
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  08-25-86 File Created.  (Cork)
;;;  11-18-86 Testing Completed.  (Cork)
;;;  02-17-87 Converted to GBB V1.00.  (Cork)
;;;  03-29-87 Converted to GBB1 Execution Shell (Johnson)
;;;  05-15-87 Converted to GBB1 ES Version 1.0  (Johnson)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *


#|| ==========================================================================

                          The Hypothesis Pyramid Builder

This file contains the code to illustrate a simplistic use of the GBB1
Execution Shell.  The application blackboard structure consists of the
spaces SPACE1 through SPACE4.  The initial action of the system is to create
two dozen HYPs on SPACE1, positioned on this space according to its VALUE
slot, which varies randomly from 0 to 29.  Once this initial data is
created, the system builds a "pyramid" of HYPs by locating a HYP with 2
adjacent HYPs on SPACE1, and creating a new HYP on SPACE2 based upon these
three HYPs.  This process continues until there is a HYP on SPACE2 with 2
adjacent HYPs, at which point a HYP on SPACE3 will be created, and so on.
The process terminates when a HYP is created on SPACE4.

The blackboard objects used to implement the pyramid builder are:

  HYP:
      A standard GBB unit which is stored on spaces SPACE1 - SPACE4.
      HYPs are created either by the INITIALIZE-PYRAMID-BASE ks or the
      PYR-BUILDER ks.

  PYR-BUILDER:
      A GBB1 KS triggered by the creation of any HYP. It generates one
      PYR-BUILDER-KSAR corresponding to each HYP.

  PYR-BUILDER-KSAR:
      Its precondition is the existance of HYPs on each "side" of the HYP
      corresponding to this ksar.  If this precondition can be satisfied
      (i.e.  two neighboring HYPs exist) then the action of this ksar is to
      create a new HYP on the space one higher than the HYP corresponding to
      this ksar.  (PYR-BUILDER-KSAR is defined implicitly by the
      PYR-BUILDER ks definition.)

  INITIALIZE-PYRAMID-BASE:
      the initial KS, which simply creates a bunch of HYPs on SPACE1.

The following initializations for the GBB1 control shell are made:

  TERMINATION CONDITION:
      when a HYP is created on SPACE4, or when the executable agenda is
      empty.  Note that since the initial placement of HYPs is random, it
      may not be possible to build a pyramid all the way to SPACE4.

  PRIORITY FUNCTION:
      The priority function rates the KSARs based upon what space they're
      on.  Rating upper spaces more highly finds a solution more quickly,
      while rating lower spaces more highly builds a "bigger" pyramid.
      Currently upper spaces are rated more highly.

  RECOMMENDATION FUNCTION:
      The recommendation function just picks out one of the most highly
      prioritized KSARs at random.

  TRACE FUNCTION:
      This function prints out a listing of the "values" of the HYPS stored
      on each of the spaces.

========================================================================== ||#



#+SYMBOLICS
(in-package "CL-USER")
#-SYMBOLICS
(in-package "USER")

(use-package '("LISP" "UMASS-EXTENDED-LISP" "GBB" "GBB1"))


;;; ---------------------------------------------------------------------------
;;;
;;;   Define the HYP blackboard structure
;;;
;;; ---------------------------------------------------------------------------

(format t "~%~a ~s" (short-date-and-time) "Defining Blackboards")

(define-spaces (SPACE1 SPACE2 SPACE3 SPACE4)
  "The spaces that the HYPs will be stored on."
  :UNITS      (hyp)
  :DIMENSIONS ((value :ORDERED (0 30))))


(define-blackboard BLACKBOARD (space1 space2 space3 space4)
  "The domain blackboard structure.")


;;; ---------------------------------------------------------------------------
;;;
;;; HYP Blackboard Object Definition
;;;
;;; ---------------------------------------------------------------------------

(format t "~%~a ~s" (short-date-and-time) "Defining HYP")

(define-gbb1-unit (HYP (:CONC-NAME "HYP$")
		       (:PRINT-FUNCTION hyp-printer))
             
  "HYP (Hypothesis) Unit.
   The data for the pyramid."

  :EVENT-CLASSES
  (:creation-events)
  
  :SLOTS
  ((value nil)
   (level 'space1)
   (response-frame nil)
   (author "Philip Johnson"))

  :LINKS
  ((supported-hyp (hyp supporting-hyps))
   (supporting-hyps (hyp supported-hyp))
   (triggered-ksar :singular (pyr-builder-ksar triggering-hyp :singular)))

  :DIMENSIONAL-INDEXES
  ((value value :TYPE :point))

  :PATH-INDEXES
  ((level level :type :label))

  :PATHS
  ((:paths `(blackboard ,level))))

;;; ---------------------------------------------------------------------------


(defun HYP-PRINTER (object stream depth)

  "HYP-PRINTER object stream depth

   Prints out a HYP unit."

  (declare (ignore depth))
  (format stream "<~a ~s>" (hyp$name object) (hyp$value object)))


;;; ---------------------------------------------------------------------------
;;;
;;; PYR-BUILDER KS and KSAR Definitions
;;;
;;; ---------------------------------------------------------------------------

(format t "~%~a ~s" (short-date-and-time) "Defining PYR-BUILDER KS and KSAR")

(define-gbb1-ks (PYR-BUILDER (:CONC-NAME "PYR-BUILDER$"))

  "This KS is triggered by the creation of a HYP, and generates one KSAR
   which will look for neighbors on either side of the triggering HYP."

  :TRIGGER-CONDITIONS ((trigger-event-class-p :unit-creation)
		       (trigger-event-type-p  'hyp))
  :PRECONDITIONS      ((:stable (lower-neighbor *this-ksar*))
		       (:stable (upper-neighbor *this-ksar*)))
  :CONTEXT-SLOTS     ((triggering-hyp) `((,(trigger-unit))))
  :KSAR-UNIT-TYPE     pyr-builder-ksar
  :ACTION-FUNCTION    #'synthesize-hyp
  :FROM-BB            (make-paths :paths '(blackboard))
  :TO-BB              (make-paths :paths '(blackboard))
  :KS-PHASES          '(:build-phase)
  :AUTHOR             "Philip Johnson")

;;; ---------------------------------------------------------------------------

(define-gbb1-ksar (PYR-BUILDER-KSAR (:CONC-NAME "PYR-BUILDER-KSAR$")
				    (:PRINT-FUNCTION pyr-builder-ksar-printer))

  "This KSAR looks for the occurance of HYPs on both sides of the HYP
   which triggered it.  If both neighbors can be found, it creates a 
   new HYP on the next higher space."

  :KS-UNIT-TYPE pyr-builder
  :SLOTS        ((neighboring-hyps nil)
		 (lower-neighbor nil)
		 (upper-neighbor nil))
  :LINKS        ((triggering-hyp :singular (hyp triggered-ksar :singular)))
  :AUTHOR       "Philip Johnson")

;;; --------------------------------------------------------------------------

(defun PYR-BUILDER-KSAR-PRINTER (object stream depth)

  "PYR-BUILDER-KSAR-PRINTER object stream depth

   Prints out a PYR-BUILDER-KSAR unit."

  (declare (ignore depth))
  (format stream "<PB-KSAR ~s>" (pyr-builder-ksar$triggering-hyp object)))

;;;; --------------------------------------------------------------------------
;;;;
;;;;  Precondition and Action functions used in PYR-BUILDER
;;;;
;;;; --------------------------------------------------------------------------

(defun LOWER-NEIGHBOR (pyr-builder-ksar-instance)

  "LOWER-NEIGHBOR pyr-builder-ksar-instance

   This precondition checks to see if there exists a neighboring HYP with a VALUE
   slot value 1 less than the HYP that triggered PYR-BUILDER-KSAR-INSTANCE.
   If so, it sets the lower-neighbor slot to one of these HYPs."

  (let* ((stimulus-hyp (pyr-builder-ksar$triggering-hyp pyr-builder-ksar-instance))
	 (stimulus-value (hyp$value stimulus-hyp))
         (lower-neighbors (get-neighbors stimulus-hyp stimulus-value -1)))
    (if lower-neighbors
	(setf (pyr-builder-ksar$lower-neighbor pyr-builder-ksar-instance)
	      (first lower-neighbors)))))

;;;; --------------------------------------------------------------------------

(defun UPPER-NEIGHBOR (pyr-builder-ksar-instance)

  "UPPER-NEIGHBOR pyr-builder-ksar-instance

   This precondition checks to see if there exists a neighboring HYP with a VALUE
   slot value 1 more than the HYP that triggered PYR-BUILDER-KSAR-INSTANCE.
   If so, it sets the upper-neighbor slot to one of these HYPs."

  (let* ((stimulus-hyp (pyr-builder-ksar$triggering-hyp pyr-builder-ksar-instance))
	 (stimulus-value (hyp$value stimulus-hyp))
         (upper-neighbors (get-neighbors stimulus-hyp stimulus-value 1)))
    (if upper-neighbors
	(setf (pyr-builder-ksar$upper-neighbor pyr-builder-ksar-instance)
	      (first upper-neighbors)))))

;;;; --------------------------------------------------------------------------

(defun get-neighbors (hyp-instance value displacement)

  "GET-NEIGHBORS hyp-instance value displacement

   Returns a list of neighboring hyps with their values displaced from
   VALUE by DISPLACEMENT."

  (find-units 'hyp (make-paths :unit-instances hyp-instance)
	      `(:ELEMENT-MATCH :exact
		:PATTERN-OBJECT
		    (:INDEX-TYPE (:DIMENSION value :TYPE :point)
		     :INDEX-OBJECT ,value
		     :DISPLACE ((value ,displacement))))))

;;; ---------------------------------------------------------------------------

(defun SYNTHESIZE-HYP (pyr-builder-ksar-instance)
  
  "SYNTHESIZE-HYP pyr-builder-ksar-instance

   This function is the action for PYR-BUILDERs.
   It creates a new HYP on the space 1 higher than the HYP that triggered the
   PYR-BUILDER-KSAR-INSTANCE."

    (make-hyp :LEVEL (next-higher-space (pyr-builder-ksar$triggering-hyp pyr-builder-ksar-instance))
              :SUPPORTING-HYPS (list (pyr-builder-ksar$lower-neighbor pyr-builder-ksar-instance)
				     (pyr-builder-ksar$upper-neighbor pyr-builder-ksar-instance))
              :VALUE (hyp$value (pyr-builder-ksar$triggering-hyp pyr-builder-ksar-instance))))

;;; ---------------------------------------------------------------------------

(defun NEXT-HIGHER-SPACE (hyp)

  "NEXT-HIGHER-SPACE hyp

   Returns the name of the space one higher than the space that HYP is on."

  (let ((hyp-space (get-space-name-from-path-structure (make-paths :unit-instances hyp))))
    (case hyp-space
       (space1 'space2)
       (space2 'space3)
       (space3 'space4)
       (otherwise (error "Bad space name in NEXT-HIGHER-SPACE")))))


;;;; --------------------------------------------------------------------------
;;;;
;;;;  INITIALIZE-PYRAMID-BASE KS Definition
;;;;
;;;; --------------------------------------------------------------------------

(format t "~%~a ~s" (short-date-and-time) "Defining INITIAL-KS")

(define-gbb1-ks INITIALIZE-PYRAMID-BASE
  
  "This KS simply creates some HYPs with random VALUEs on SPACE1.
   Since this is the initial KS, it doesn't require any trigger or
   preconditions."

  :ACTION-FUNCTION #'(lambda (ksar)
		       (declare (ignore ksar))
		       (dolist (hyp-value `(1  3  5  6  7  8  10 11 12 13
					    14 15 16 17 17 18 18 20 20 20))
			 (make-hyp :LEVEL 'space1
				   :VALUE hyp-value))
		       (set-ks-phase :build-phase))
  :KS-PHASES '(:initial-phase)
  :AUTHOR "Philip Johnson")

;;; ---------------------------------------------------------------------------
;;;
;;; Trace function
;;;
;;; ---------------------------------------------------------------------------

(defun print-hyps ()

  "PRINT-HYPS nil

   Prints out the values of the VALUE slots of the HYPs on spaces SPACE1 - SPACE4.
   This is the TRACE function for this example."

  (flet ((show-space (space)
           (let ((all-units (find-units 'hyp
                                        (make-paths :paths `(blackboard ,space))
                                        :all)))
             (format t "~%~:(~a~): ~{ ~2d~}"
                     space
                     (sort (mapcar #'hyp$value all-units) #'<)))))

    (format t "~2%~78,,,'-<-~>~@
                Values of the VALUE slot of the HYPs on their spaces")
    (show-space 'space1)
    (show-space 'space2)
    (show-space 'space3)
    (show-space 'space4)))

;;; ---------------------------------------------------------------------------
;;;
;;;   The Priority and Termination Functions.
;;;
;;; ---------------------------------------------------------------------------

(defun KSAR-PRIORITY (ksar)

  "KSAR-PRIORITY ksar

   This function returns the priority of the passed KSAR.
   It is simply a number from 1 to 4, based upon the location (SPACE1 through
   SPACE4) of the HYP which triggered this KSAR."

  (let ((space-name (get-space-name-from-path-structure
		      (make-paths :unit-instances (pyr-builder-ksar$triggering-hyp ksar)))))
    (case space-name
       (space1 1)
       (space2 2)
       (space3 3)
       (space4 4)
       (otherwise (error "Illegal space name passed in KSAR-PRIORITY.")))))

;;; ---------------------------------------------------------------------------

(defun SPACE4-HYP ()
  
  "SPACE4-HYP nil

   This is the termination function: it returns T if there is a HYP on SPACE4."
  
  (space-occupied-p (make-paths :paths '(blackboard space4))))

;;; --------------------------------------------------------------------------
;;;
;;; Set the control shell parameters
;;;
;;; --------------------------------------------------------------------------

(define-gbb1-parameters
  :priority-fn                   'ksar-priority
  :recommendation-fn             'first
  :termination-fn                'space4-hyp
  :precondition-recheck-interval nil      ;; since stable, don't need to be rechecked.
  :obviation-recheck-interval    nil      ;; since they don't exist, don't need to be rechecked.
  :max-execution-cycles          30)

(define-gbb1-output
  :trace-fn              'print-hyps
  :trace-print-points    '(:before-agenda-update :final-state)
  :print-unit-width      25
  :output-stream         *standard-output*)

;;; --------------------------------------------------------------------------
;;;
;;; Run the system.
;;;
;;; --------------------------------------------------------------------------


(defun BUILD-PYRAMID ()

  "BUILD-PYRAMID nil

   Builds a pyramid of HYPs on spaces SPACE1 through SPACE4."

  ;; create the domain blackboard to hold the HYPs.
  (instantiate-blackboard-database 'blackboard :mode :overwrite)

  ;; Specify various execution shell parameters.
  (run-gbb1 :initial-ks 'initialize-pyramid-base))


(format t "~&;;;~%;;; To run the demo use the function (BUILD-PYRAMID).~%;;;~%")


;;; --------------------------------------------------------------------------
;;;                               End of File
;;; --------------------------------------------------------------------------
