;;;; -*- Mode:Common-Lisp; Package:UMASS-EXTENDED-LISP; Fonts:(MEDFNT); Base:10 -*- 
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL]EXHIBIT.LISP *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  12:37:20; Edited-By: Cork *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                           EXHIBITOR
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  This file implements a simple data structure browser.  The top level
;;;  function is called EXHIBIT so it doesn't confilict with the Common Lisp
;;;  function INSPECT.  This file is written for VaxLisp because DEC hasn't
;;;  written an inpector yet.
;;;
;;;  01-23-86 File Created.  (Gallagher)
;;;  06-30-86 Fixed to work under VaxLisp 2.0.  (Gallagher)
;;;  06-23-88 Updated to VaxLisp V2.2  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#-(and VAX VMS DEC)
(eval-when (compile load eval)
  (error "This file only works with VaxLisp."))

(in-package "UMASS-EXTENDED-LISP")

(export '(exhibit))

(use-package '(lisp vax-lisp))

(proclaim '(optimize (speed 3) (safety 1)))

;;; ***************************************************************************
;;;
;;; The exhibitor is a convenient way to browse through lisp objects.  It is
;;; particularly useful for examining networks of structures, arrays, or
;;; lists.  The top level function is EXHIBIT.  Type `?' inside the exhibitor
;;; for help.
;;; 
;;; ***************************************************************************



;;; --------------------------------------------------
;;;   Random Macros
;;; --------------------------------------------------


(defmacro notify (format-string &rest args)

  "NOTIFY format-string &rest args

   Print a message on *error-output*."
  
  `(format *error-output* "~&~@?~&" ,format-string ,@args))


(defmacro do-array-region ((subscripts region) &body body)

  "DO-ARRAY-REGION (subscripts region) &body body

   This macro provides iteration over a logically contiguous region of
   an array (a subarray of the array).

   REGION is a list of the form ((x1 . x2) (y1 . y2) ...)
   which specifies the two end points of the subarray of the array.
   Specifically, REGION is a list of pairs, one pair for each dimension
   of the array, such that (mapcar #'car region) produces the ``lower
   left'' point of the subarray and (mapcar #'cdr region) produces the
   ``upper right'' point of the subarray.  Note that, unlike almost
   every other common lisp function, the upper bound is inclusive not
   exclusive.  Any pair of coordinates can be replaced by a single
   element list to indicate just one row (column, plane, whatever).

   SUBSCRIPTS is a symbol which is bound to successive lists of array
   indexes.

   BODY is executed once for each combination of array indexes.

   A typical call would be:

      (do-array-region (indexes array-region)
        (setf (apply #'aref <some-array> indexes) <some-value>))

   A block named DO-ARRAY-REGION surrounds the form so RETURN-FROM may
   be used to exit the loop early."


  (let* ((finish-state (gensym))
         (len (gensym))
         (i (gensym)))
    `(block do-array-region
       (do* ((,subscripts (mapcar #'car ,region))
             (,finish-state (mapcar #'(lambda (x) 
                                        (cond ((cdr x)) (t (car x))))
                                    ,region))
             (,len (1- (length ,subscripts))))
            (())
         (progn ,@body)
         (do ((,i ,len (1- ,i)))
             ((< ,i 0) (return-from do-array-region))
           (incf (nth ,i ,subscripts))
           (if (> (nth ,i ,subscripts) (nth ,i ,finish-state))
               (setf (nth ,i ,subscripts) (car (nth ,i ,region)))
               (return)))))))


;;; Functions for manipulating the exhibited objects stack.

(defvar *exhibit-stack* nil
        "Stack of things that have been exhibited during this call
         to exhibt.")

(defun current-object ()
  (car *exhibit-stack*))

(defun set-current-object (new-object)
  (cond ((eq new-object (car *exhibit-stack*))
         new-object)
        (t (push new-object *exhibit-stack*)
           new-object)))

(defsetf current-object set-current-object)

(defun pop-current-object ()
  (pop *exhibit-stack*))


;;; Structures and varaibles for commands.

(defstruct (exhibitor-command (:conc-name exc-))
   "Data about an exhibitor command."
   (name             "")              ; Either a string or a character
   (length           0)               ; Length of the name
   (function         #'exhibit-help)  ; Function to implement the command
   (documentation    ""))             ; Description of the command

(defvar *exhibit-commands* nil
        "List of exhibitor commands.")


;;; --------------------------------------------------
;;;   The Exhibitor
;;; --------------------------------------------------


(defun exhibit (object &key (level 2) (length 4) (array nil) (lines 4))

  "EXHIBIT object &key level length array lines

   An interactive data structure browser.  This is what CLTL calls inspect.
   The keyword arguments :LEVEL, :LENGTH, :ARRAY, and :LINES control the
   printed representation of objects during EXHIBIT.  They correspond to
   the global variables *PRINT-LEVEL*, *PRINT-LENGTH*, *PRINT-ARRAY*,
   and *PRINT-LINES*.

   Type `?' for a list of exhibitor commands.  You can hit any key to
   get back to the `Exhibit:' prompt, when a long object such as a
   vector or array is being exhibited.

   Exhibit returns the current object in case you want to do anything
   more with it."

  (clear-input)
  (setf *exhibit-stack* (list object))

  (let ((*print-length* length)
        (*print-level* level)
        (*print-lines* lines)
        (*print-array* array))

    (unwind-protect

        (catch 'exhibit-exit

          (exhibit-display (current-object))

          (do ((function nil) (command nil))
              ()
            (fresh-line)
            (setf command (read-command "Exhibit: "))
            (cond ((numberp command)
                   (exhibit-nth command (current-object)))
                  ((setf function (get-command-function command))
                   (funcall function))
                  (t (beep))))))))


(defun exhibit-display (object)

  "EXHIBIT-DISPLAY object

   Display an object for the exhibitor."

  (cond ((defstruct-object-p object)
         (exhibit-defstruct object))
        ((symbolp object)
         (exhibit-symbol object))
        ((consp object)
         (exhibit-cons object))
        ((vectorp object)
         (exhibit-vector object))
        ((arrayp object)
         (exhibit-array object))
        (t (notify "[ Can't exhibit a ~s. ]" (type-of object))
           (describe object))))


(defun exhibit-nth (offset object)

  "EXHIBIT-NTH offset object

   Exhibit the NTH part of the current object."

  (cond ((defstruct-object-p object)
         (exhibit-nth-defstruct offset object))
        ((symbolp object)
         (exhibit-nth-symbol offset object))
        ((consp object)
         (exhibit-nth-cons offset object))
        ((vectorp object)
         (exhibit-nth-vector offset object))
        ((arrayp object)
         (exhibit-nth-array offset object))
        (t (notify "Can't select an element from a ~s."
                   (type-of object))
           object)))


;;; --------------------------------------------------
;;;   Exhibitor Command Definitions
;;; --------------------------------------------------

(defmacro define-exhibit-command ((fn-name keys) args &body body)

  "DEFINE-EXHIBIT-COMMAND (fn-name keys) args &body body

   Define an exhibitor command."

  (let* ((documentation (if (stringp (car body))
                            (car body)
                            "No documentation available.")))
    `(progn
       (defun ,fn-name ,args ,@body)
       (dolist (key ',(if (listp keys) keys (list keys)))
         (let ((temp (find key *exhibit-commands*
                           :key #'exc-name :test #'string=)))
           (cond (temp
                  (setf (exc-function temp) ',fn-name)
                  (setf (exc-documentation temp) ',documentation))
                 (t (push (make-exhibitor-command
                            :name (string key)
                            :length (length (string key))
                            :function ',fn-name
                            :documentation ',documentation)
                          *exhibit-commands*)))))
       ',fn-name)))


(defun get-command-function (name)

  "GET-COMMAND-FUNCTION name

   Searches for a command named NAME in *exhibit-commands*.  Returns
   The function associated with that command if NAME is a unique
   prefix for one of the defined commands.  Returns nil otherwise."

  (when (or (stringp name) (symbolp name))
    (when (symbolp name) (setf name (symbol-name name)))
    (let ((command nil)
          (length (length name)))
      (dolist (c *exhibit-commands*)
        (when (string-equal name (exc-name c)
                            :end2 (if (> length (exc-length c))
                                      (exc-length c)
                                      length))
          (when command                    ; Match is not unique
            (return-from get-command-function nil))
          (setf command (exc-function c))))
      (if (null command)
          nil           ; No match
          command       ; Unique match
          ))))



(define-exhibit-command (exhibit-help (? HELP)) ()
  "Help on the exhibitor."
  (format t
          "~%All commands can be abbreviated to one character, but you must type~@
           a return for it to be read.~%~@
           Commands:~@
           Any Integer~16,8tExhibit that numbered part of the object.~%")
  (dolist (i *exhibit-commands*)
    (format t "~:(~a~)~16,8t~a~%" (exc-name i) (exc-documentation i))))

(define-exhibit-command (exhibit-pop UP) ()
  "Go up the stack of exhibited objects."
  (if (= (length *exhibit-stack*) 1)
      (notify "You are at the top of the stack.~@
               Use RETURN or QUIT to exit.")
      (progn (pop-current-object)
             (exhibit-display (current-object)))))

(define-exhibit-command (exhibit-current CURRENT) ()
  "Show the current object again."
  (exhibit-display (current-object)))

(define-exhibit-command (exhibit-pp PPRINT) ()
  "Pretty print the currently displayed object."
  (let ((*print-level* nil)
        (*print-length* nil))
    (pprint (current-object))))

(define-exhibit-command (exhibit-show-stack STACK) ()
  "Show the stack of exhibited objects."
  (format t "~%Exhibitor stack (most recent objects first):~%")
  (do ((i 0 (1+ i))
       (p *exhibit-stack*))
      ((endp p))
    (format t "~%~d.~4t~s" i (pop p)))
  (terpri))

(define-exhibit-command (exhibit-quit-return-nil (QUIT EXIT)) ()
  "Exit the exhibitor and return NIL."
  (throw 'exhibit-exit nil))

(define-exhibit-command (exhibit-quit-return-obj RETURN) ()
  "Exit the exhibitor and return the current object."
  (throw 'exhibit-exit (current-object)))



;;; Exhibitor functions for each type of object.


;;; Defstruct handlers

(defun exhibit-defstruct (object)
  (let* ((type (type-of object))
         (description (get type 'system::defstruct-description))
         (slot-data (system::defstruct-description-slot-data description)))
    (format t "~%It is a ~s." type)
    (do* ((slot-data-list slot-data (cdr slot-data-list))
          (slot (car slot-data-list) (car slot-data-list))
          (i 0 (1+ i)))
         ((or (listen) (endp slot)))
      (format t "~%[~d]~6t~s  ~25,5t ~@!~s~."
              i (first slot) (elt object (second slot))))
    (terpri)
    (setf (current-object) object)))

(defun exhibit-nth-defstruct (offset object)
  (let* ((type (type-of object))
         (description (get type 'system::defstruct-description))
         (slot-data (system::defstruct-description-slot-data description)))
    (cond ((< offset (length slot-data))
           (exhibit-display (elt object (1+ offset))))
          (t (notify "~d is out of range." offset)
             object))))


;;; Symbol handlers

(defun exhibit-symbol (symbol)
  (format t "~%It is the symbol ~s.~%" symbol)
  (format t "[0]~6tPackage:   ~s~%" (symbol-package symbol))
  (format t "[1]~6tValue:     ~:[Unbound~;~:*~s~]~%"
          (if (boundp symbol) (symbol-value symbol) nil))
  (when (constantp symbol)
    (format t "  It's a CONSTANT.~%"))
  (format t "[2]~6tFunction:  ~:[Undefined~;~:*~s~]~%"
          (if (fboundp symbol) (symbol-function symbol) nil))
  (if (symbol-plist symbol)
      (progn (format t "  Property List:")
             (exhibit-plist (symbol-plist symbol) 3))
      (format t "  Plist:~8tNIL"))
  (terpri)
  (setf (current-object) symbol))

(defun exhibit-nth-symbol (offset symbol)
  (case offset
    (0 (exhibit-display (symbol-package symbol)))
    (1 (if (boundp symbol)
           (exhibit-display (symbol-value symbol))
           (progn (notify "~s has no value." symbol)
                  symbol)))
    (2 (if (fboundp symbol)
           (exhibit-display (symbol-function symbol))
           (progn (notify "~s has no function definition." symbol)
                  symbol)))
    (t (exhibit-nth-plist offset (symbol-plist symbol) 3))))


;;; Plist handlers

(defun exhibit-plist (list &optional (start 0))
  ;; START is the number to start counting with.
  (do* ((p list (cddr p))
        (i start (1+ i)))
       ((or (listen) (endp p)))
    (format t "~%[~d]~6t~s  ~25,5t ~@!~s~."
            i (first p) (second p))))

(defun exhibit-nth-plist (offset list &optional (start 0))
  ;; OFFSET is a count in the caller's frame of reference.
  ;; Reduce OFFSET by START to get the real index into this list
  (let* ((index (1+ (* 2 (- offset start))))
         (length (or (list-length list) index)))
    (cond ((> index length)
           (notify "~d is out of range." offset))
          (t (exhibit-display (nth index list))))))


;;; Cons handlers

(defun exhibit-cons (cons)
  (cond ((list-length cons)
         (format t "~%It is a CONS.")
         (do ((p cons (cdr p))
              (i 0 (1+ i)))
             ((or (listen) (atom p))
              (unless (null p)
                (format t "~%.~%[~d]~6t~s" i p)))
           (format t "~%[~d]~6t~s" i (car p))))
        (t (format t "~%It is a circular list.~%")
           (dotimes (i 5)
             (format t "~%[~d]~6t~s" i (nth i cons))
             (when (listen)
               (return)))
           (format t "~%  ...")))
  (setf (current-object) cons))

(defun exhibit-nth-cons (offset cons)
  (let* ((length (list-length cons))
         (circularp (not length)))
    (cond ((or circularp (< offset length))
           (exhibit-display (nth offset cons)))
          ((= offset length)
           (let ((last (last cons)))
             (if (atom (cdr last))
                 (exhibit-display (cdr last))
                 (notify "~d is out of range." offset))))
          (t (notify "~d is out of range." offset)))))


;;; Vector handlers

(defun exhibit-vector (vector)
  (format t "~&Its a ~s." (type-of vector))
  (dotimes (i (array-dimension vector 0))
    (format t "~%[~d]~6t~s" i (aref vector i))
    (when (listen)
      (return)))
  (setf (current-object) vector))

(defun exhibit-nth-vector (offset vector)
  (if (>= offset (array-total-size vector))
      (notify "~d is out of range." offset)
      (exhibit-display (aref vector offset))))


;;; Array handlers

(defun exhibit-array (array)
  (format t "~&Its a ~s." (type-of array))
  (let* ((real-dimensions (mapcar #'1- (array-dimensions array)))
         (region (mapcar #'(lambda (x) (cons 0 x)) real-dimensions))
         (index-width (length (format nil "~s" real-dimensions)))
         (i 0))

    (do-array-region (subscripts region)
      (format t "~%[~d]~6t~v@s  ~s"
              i index-width subscripts (apply #'aref array subscripts))
      (setf i (1+ i))
      (when (listen)
        (return-from do-array-region)))

    (setf (current-object) array)))

(defun exhibit-nth-array (offset array)
  (if (>= offset (array-total-size array))
      (notify "~d is out of range." offset)
      (exhibit-display
          (apply #'aref 
                 array (inverse-array-row-major-index
                           array offset)))))

(defun inverse-array-row-major-index (array index)

  "INVERSE-ARRAY-ROW-MAJOR-INDEX array index

   Returns a list of subscripts that will access the INDEXth element
   of the array in the row major ordering of the elements."

  (do* ((dimensions (cdr (array-dimensions array))
                    (cdr dimensions))
        (result nil))
       ((null dimensions)
        (nreverse (cons index result)))
    (multiple-value-bind (value remainder)
         (floor index (apply #'* dimensions))
      (push value result)
      (setf index remainder))))


;;; --------------------------------------------------
;;;   Random Functions
;;; --------------------------------------------------

(defun defstruct-object-p (object)

  "DEFSTRUCT-OBJECT-P object

   True if OBJECT is a instance of a defstruct."

  (let ((type (type-of object)))
    (and (symbolp type)
         (get type 'system::defstruct-description))))


(defun beep (&optional (stream *error-output*))
  (write-char #\BELL stream))

(defun refresh-screen (&optional (stream *standard-output*))
  ;; This function is just a placeholder.
  (declare (ignore stream))
  nil)



(defun read-command (&optional (prompt ""))

  "READ-COMMAND nil

   Read a form from the terminal.  Returns the form when a space
   or return is hit."

  (let (line)
    (loop
      (write-string prompt)
      (setf line (read-line))
      ;; Only exit the loop if the user typed something.
      (unless (zerop (length line))
        (return t)))
    (read-from-string line)))


#+IGNORE
(defun simple-print-structure (object stream x buffer y)

  "SIMPLE-PRINT-STRUCTURE object stream x buffer y

   Print a structure in the format #<type address>.  OBJECT is the
   structure to print.  STREAM is the stream to print it to.  BUFFER
   is a string.  X and Y are integers.  This function returns NIL."

  ;; BUFFER seems to be a place to put any unprinted portion of the object.
  ;; If there something is left in BUFFER than the return value is an
  ;; integer that specifies how long it is.  X and Y might be left and
  ;; right margins.  

  (declare (ignore x buffer y))

  (format stream "#<~a ~x>" (type-of object) (system::address-of object))
  nil)


#+IGNORE
(defun simple-structure-format (object)

  "SIMPLE-STRUCTURE-FORMAT object

   Print a structure in the format #<type address> to 
   *standard-output*."

  ;; Where is pprint's argument stream?

  (format t "#<~a ~x>" (type-of object) (system::address-of object))
  nil)


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
