;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.DISTRIBUTION]DEFINE-UNIT.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Tuesday, April 11, 1989  14:35:20 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 416) *-*
;;;; *-* Software: TI Common Lisp System 4.105 *-*
;;;; *-* Lisp: TI Common Lisp System 4.105 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                      UNIT DEFINITION FUNCTIONS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;  05-16-86 Inserted maker-name into export call in define-unit.  (Cork)
;;;  06-11-86 Added Events.  (Gallagher)
;;;  06-27-86 Added :INCLUDE option for DEFINE-UNIT.  (Gallagher)
;;;  07-03-86 Handle :EXPORT in define-unit rather than passing it on to
;;;           defxstruct.  (Gallagher)
;;;  07-17-86 Added default pathnames to define-unit (Johnson)
;;;  08-26-86 Replaced ENABLE-UNIT-HASHING with TOGGLE-UNIT-HASHING. (Johnson)
;;;  10-02-86 Moved the functions unit-type-p, unit-instance-p, and
;;;           get-unit-description from this file to utilities.  (Gallagher)
;;;  10-24-86 Moved code that handles link definition, access, etc. to a
;;;           separate file, links.  (Gallagher)
;;;  10-27-86 Removed <unit>-HASH-TABLE as a separate variable.  The
;;;           unit's hash table is now in the unit-description.  (Gallagher)
;;;  10-28-86 Added compiler transforms for MAKE-<unit> and DELETE-<unit>.
;;;           (Gallagher)
;;;  10-30-86 Added slot event inheritance (Johnson)
;;;  11-03-86 Changed GET-UNIT-FROM-TABLE to FIND-UNIT-BY-NAME.  (Johnson)
;;;  01-13-87 Modified handling of unit construction function names -- the
;;;           names are in the unit-description now.  Deleted references to
;;;           UNIT.HASH-P because all units are now hashed.  Deleted the
;;;           functions TOGGLE-UNIT-HASHING and GET-UNIT-FROM-TABLE.
;;;           (Gallagher)
;;;  01-16-87 Reworked the path generation code.  (Gallagher)
;;;  01-20-87 Added dummy unit code. (Johnson)
;;;  02-06-87 Several changes because the name slot of a unit-instance is now a
;;;           slot in basic-unit.  (Gallagher)
;;;  02-24-87 Cleaned up all cases where pointers to structures were being 
;;;           used as constants in code.  The problem was that in several
;;;           places I had code like (unit.hash-table ',description) which
;;;           appeared in the expansion of DEFINE-UNIT as (unit.hash-table
;;;           '#<UNIT ...>).  This was causing seemingly random bugs because
;;;           the compiler would create a new copy of the unit description in
;;;           each file in which the pointer appeared.
;;;           This is a bad coding technique anyway.  (Gallagher)
;;;  06-19-87 Changed the order of arguments to the update event functions.
;;;           This was done to make update events consistent with 
;;;           initialization events.  (Gallagher)
;;;  04-28-88 Added slot specialization and private slot capabilities.
;;;           (Gallagher)
;;;  06-21-88 Modified RECORD-INDEX-SOURCE-SLOTS to allow indexes to be based
;;;           on slots in *external-unit-slots*.  (Gallagher)
;;;  07-01-88 Added generic slot access and update functions, and a generic
;;;           make-unit and delete-unit.  (Gallagher)
;;;  07-05-88 Rewrote unit hash table handling to use the new UNIT-INFO
;;;           structure. (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'GBB :use '(lisp umass-extended-lisp))

(export '(define-unit
	  find-unit-by-name
          clear-unit-hash-tables
          make-unit
          delete-unit
          get-unit-slot))

(use-package '(lisp umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Defining Units
;;;; --------------------------------------------------------------------------

;;; DEFINE-UNIT:
;;;
;;; Units are built on top of defstructs.  Each unit has a unit-description
;;; in the *unit-description-hash-table* where information about the unit,
;;; such as the indexes, links, etc., is stored.  (Use the function
;;; get-unit-description to get the unit-description of a unit.)
;;;
;;; The conc-name that the user supplies is not the one that I pass on to
;;; defstruct.  Instead, I generate another conc-name (called here the
;;; ``internal conc-name'') which I give to defstruct; then I define the
;;; user callable functions with the user supplied conc-name in terms of the
;;; internal accessor functions.  The ``external'' accessors include code to
;;; handle events, links, etc.  A similar strategy is used for the
;;; constructor function.
;;;
;;; The defstruct called basic-unit is included in all units.  It provides
;;; slots needed by every unit like the mark slot used in retrieval and the
;;; space instance(s) that the unit instance is placed on.  If the user
;;; includes another unit then that one will have basic-unit already
;;; included in it so we only need to include basic-unit in units that don't
;;; include another unit.

(defmacro define-unit (name-and-options &body args &aux documentation)

  "DEFINE-UNIT name-and-options [documentation]
	       &KEY slots dimensional-indexes links path-indexes paths

   DEFINE-UNIT defines a unit.  Slots are similar to slots in defstruct.
   Links define how this unit will be linked to other units.  Indexes define
   how to store and retrieve the unit from spaces.  The syntax is:

     SLOTS:   ( { (slot-name [init-form {keyword value}*]) }* )

     LINKS:   ( { (link-name [:singular]
                      {:reflexive |
                       (inverse-link-unit inverse-link-name [:singular])}
                      {keyword value}*) }* )

     DIMENSIONAL-INDEXES:
              ( { (index-name slot-name [:type index-element-type]) }* )

     PATH-INDEXES:
              ( { (index-name slot-name [:type index-element-type]) }* )

     PATHS:   ( { (:PATH {path-list}*) |
                  (:PATH-STRUCTURE path-structure {change-specs}*) }* )

   For example:

     (define-unit (HYP)
       :SLOTS                ((belief nil)
                              (node)
                              (time 0))
       :LINKS                ((creating-ksis (ksi creating-hyps)))
       :DIMENSIONAL-INDEXES  ((time time :type :point)))
       :PATH-INDEXES         ((node node :type :label))
       :PATHS                ((:path
                                 `(root 0 ,node 0 location 0))
                              (:path-structure *current-goal-level*
                                 '(:change-subpaths (goal) (hyp)))))"

  (setf documentation
	(if (stringp (car args)) (pop args) "No Documentation Supplied"))
  (with-keywords-bound ((slots dimensional-indexes links path-indexes paths)
			args
			"~s is not a valid keyword for DEFINE-UNIT.")
    (let* ((unit-type (if (symbolp name-and-options)
			  name-and-options
			  (first name-and-options)))
	   export-p exports
	   description accessor-names)
      (error-when (keywordp unit-type)
         "Unit Type can not be a keyword.~%~s." name-and-options)
      ;; Extract GBB specific information from name-and-options and
      ;; make the unit description.
      (multiple-value-setq (name-and-options description export-p exports)
	(parse-name-and-options name-and-options))
      (setf (unit.documentation description) documentation)
      ;; Save the slots, indexes, links, and paths.
      (add-unit-data description slots dimensional-indexes links path-indexes paths)
      ;; If there's an included unit, take care of event inheritance.
      (inherit-events description)
      ;; Make the slot, link, and index descriptions.
      (setf (unit.links description) (make-unit-links description)
            (unit.slots description) (make-unit-slots description)
	    (unit.d-indexes description) (make-unit-d-indexes description)
	    (unit.p-indexes description) (make-unit-p-indexes description))
      ;; List of public, external accessor names.
      (setf accessor-names (accessor-names-list description))
      (record-index-source-slots description)
      (check-define-unit-reasonablity description)

      `(progn
	 (export ',(if export-p
		       `(,@exports
			 ,@accessor-names
			 ,@(additional-unit-exports description))
		       nil))
	 (proclaim '(inline ,@accessor-names))
	 ;; Build the defstruct.
	 (defstruct (,@name-and-options
                     ,(build-defstruct-include-option description)
                     (:copier nil))
	   ,documentation
	   ,.(filter-slot-definitions description)
	   ,.(build-slots-from-links description))

	 ;; Order is important among these build-...  forms because the
	 ;; code generated by later ones depend on functions and macros
	 ;; defined by code generated by earlier ones.  For example, the
	 ;; construction and deletion functions must appear after the
	 ;; access and update functions.
	 ,.(build-access-forms description)
	 ,.(build-update-forms description)
	 ,.(build-link-access-forms description)
	 ,.(build-path-generation-forms description)
	 ,.(build-unit-construction-code description)
         ,.(build-unit-deletion-code description)
	 ,.(build-misc-code description)

	 (eval-when (load eval compile)
	   (pushnew ',unit-type *all-units*)
	   ;; Store the information needed to do update links (via linkf).
	   ,.(store-link-methods description)
	   ;; Save the unit description.
	   (let ((old-description (get-unit-description ',unit-type t)))
	     (check-unit-redefinition ',description old-description)
             (initialize-unit-subtypes ',description old-description)
	     (setf (get-unit-description ',unit-type) ',description))
	   (check-unit-links-if-possible ',description))
         (propagate-unit-subtypes-up ',description)
         (initialize-unit-info ',description)

	 ;; Return the unit-type.
	 ',unit-type))))


;;;; --------------------------------------------------------------------------
;;;;   Slot Parsing
;;;; --------------------------------------------------------------------------

(defun make-unit-slots (description)

  "MAKE-UNIT-SLOTS description

   Put the information about each slot in a structure and return a list of 
   them all."

  (setf description (get-unit-description description))

  (let ((included-slots nil)
        (new-slots nil))
    
    ;; When there is an included unit, copy the slot structures, updating as
    ;; necessary.  We don't need to recur through all levels of inclusion
    ;; here because the `slots' slot of the unit description contains a slot
    ;; descriptor for every slot in the unit -- including the included
    ;; slots.
    (when (unit.included description)
      (setf included-slots
            (mapcar #'(lambda (slot) (copy-included-slot slot description))
                    (unit.slots (get-included-unit-description description)))))
    
    ;; For the new slots...
    (dolist (slot-info (unit.slot-list description))
      (let* ((name (first slot-info))
             (initial-value (second slot-info))
             (initial-value-present-p (>= (length slot-info) 2))
             (options (cddr slot-info))
             (the-slot (find name included-slots
                             :test #'string= :key #'slot.name)))
        
        (macrolet ((error! (&body args)
                     `(error "Syntax error in slot ~a, unit ~s.~%~@?"
                             name (unit.name description) ,@args)))
          (when (oddp (length options))
            (error! "Odd number of options and values:~%~8t~s" slot-info))
          (when (check-options options *slot-option-keywords*)
            (error! "Unknown slot option keyword(s): ~s."
                    (check-options options *slot-option-keywords*))))
        
        (cond
          ;; If this repeats an included slot then modify the
          ;; included slot to reflect the new information.
          (the-slot
           (check-slot-specialization-consistency
             name options the-slot description)
           (setf (slot.initial-value the-slot) initial-value
                 (slot.initial-value-present-p the-slot) initial-value-present-p
                 (slot.specialised the-slot) t
                 ;; Slot can only go public to private, not back.
                 (slot.private the-slot) (or (slot.private the-slot)
                                             (getf options :private))
                 (slot.save-print-function the-slot)
                 (getf options :save-print-function
                       (slot.save-print-function the-slot))))
          ;; This is a brand new slot.
          (t 
           (push (make-slot
                   :name name
                   :type (getf options :type t)
                   :unit (unit.name description)
                   :inherited-p nil
                   :private (getf options :private)
                   :initial-value initial-value
                   :initial-value-present-p initial-value-present-p
                   :modifiable (slot-modifiable-status options name description)
                   :save-print-function (getf options :save-print-function)
                   :initialization-events
                   (filter-duplicate-event-names
                     (getf options :initialization-events)
                     (get-unit-events :slot-initialization-events description))
                   :access-events
                   (filter-duplicate-event-names
                     (getf options :access-events)
                     (get-unit-events :slot-access-events description))
                   :update-events
                   (filter-duplicate-event-names
                     (getf options :update-events)
                     (get-unit-events :slot-update-events description)))
                 new-slots)))))
    
    (nconc included-slots (nreverse new-slots))))


(defun slot-modifiable-status (options slot-name description)

  "Determine the modifiable attribute for this slot.  OPTIONS is a
   list of (keyword value ...).  SLOT-NAME is the name of the slot.
   DESCRIPTION is a unit description.  Returns one of the keywords
   :dynamic, :static, or :read-only."

  (let* ((result nil))
    (when (getf options :dynamic)
      (setf result (if result :error :dynamic)))
    (when (getf options :static)
      (setf result (if result :error :static)))
    (when (getf options :read-only)
      (setf result (if result :error :read-only)))
    (error-when (eq result :error)
       "Bad slot spec for slot ~a in unit ~s.~@
        Only one of :DYNAMIC, :STATIC, or :READ-ONLY may be specified."
       slot-name (unit.name description))
    (or result :dynamic)))


(defun copy-included-slot (slot-struct description)

  "Copy the slot description from an included unit."

  (let ((new-slot (copy-slot slot-struct)))
    (setf (slot.unit new-slot) (unit.name description)
          (slot.inherited-p new-slot) t
          (slot.specialised new-slot) nil
          (slot.initialization-events new-slot) (mixin-included-events
                                                  :slot-initialization-events
                                                  (slot.initialization-events new-slot)
                                                  description)
          (slot.access-events new-slot) (mixin-included-events
                                          :slot-access-events
                                          (slot.access-events new-slot)
                                          description)
          (slot.update-events new-slot) (mixin-included-events
                                          :slot-update-events
                                          (slot.update-events new-slot)
                                          description))
    new-slot))

(defun check-slot-specialization-consistency (name options included-slot description)

  "Check that a slot specialization is consistent with the
   original slot from the included unit."

  (flet ((find-source-unit (name)
           (search-included-units
             #'(lambda (descrip)
                 (let ((slot (find name (unit.slots descrip)
                                   :key #'slot.name :test #'string=)))
                   (and slot (not (slot.inherited-p slot)))))
             (unit.included description))))
    (macrolet ((error! (&body args)
                 `(error "Bad slot spec for slot ~a in unit ~s.~@
                          ~a is also a slot in the included unit ~s.~%~@?"
                         name (unit.name description) name
                         (unit.name (find-source-unit name)) ,@args)))

      (when (not (eq (slot-modifiable-status options name description)
                     (slot.modifiable included-slot)))
        (error! "This slot is ~s, but the included slot is ~s."
                (slot-modifiable-status options name description)
                (slot.modifiable included-slot)))

      (dolist-by-twos (key value options)
        (case key
          ((:save-print-function :dynamic :static :read-only)
           ;; These are ok, or are checked above.
           nil)
          (:type
           (when (not (same-type-p value (slot.type included-slot)))
             (error! "The :TYPE option specified here (~s) is incompatible~@
                      with that in the included unit (~s)."
                     value (slot.type included-slot))))
          (:private
           (when (and (null value) (slot.private included-slot))
             (error! "~a is private in ~s,~%you can not make it public in ~s."
                     name (unit.name (unit.included description))
                     (unit.name description))))
          (otherwise
           (error! "The ~s option is not allowed when specialising a slot."
                   key)))))))


(defun record-index-source-slots (description)

  "Record which slots are used by dimensional indexes."

  (let ((slots (unit.slots description)))
    (dolist (index (unit.d-indexes description))
      (let ((slot-instance (find (unit-index.slot-name index)
				 slots
				 :key #'slot.name
				 :test #'string=)))
        ;; Checking for the existance of the referenced slot is done in
        ;; make-unit-indexes-1, so if slot-instance is nil then the source
        ;; slot must be one of *external-unit-slots*.
        (when slot-instance
          (push (unit-index.name index)
                (slot.index-source-p slot-instance)))))))

(defun build-defstruct-include-option (description)
  "Return the :include option for inclusion in the defstruct 
   definition for a unit."
  (let* ((included-unit (unit.included description))
         (slot-modifiers nil))
    (when included-unit
      (dolist (slot (unit.slots description))
        (when (slot.specialised slot)
          (push `(,(slot.name slot)
                  ,@(when (slot.initial-value-present-p slot)
                      `(,(slot.initial-value slot))))
                slot-modifiers))))
    `(:include ,(or (unit.included description) 'basic-unit)
               ,@slot-modifiers)))


(defun build-slots-from-links (description)

  "Returns a list of slot specifications appropriate for DEFSTRUCT
   based on the links in DESCRIPTION."

  (mapcar #'(lambda (link)
	      `(,(first link)
                ,(link-initial-value link)
		:type
                ,(link-slot-type link)))
	  (unit.link-list description)))

(defun link-initial-value (link)

  "Return NIL or an array with appropriate dimensionality."

  (let ((dimensions (second (member :dimensions link))))
    (if dimensions
        `(make-array ',dimensions)
        nil)))

(defun link-slot-type (link)

  "Return:
   ARRAY if the link has dimensionality,
   ATOM  if either the link is singular, or the link
         is reflexive and the inverse link is singular,
   T     otherwise."

  (cond ((member :dimensions link)
         'array)
        ((or (member :singular link)
             (and (member :reflexive link)
                  (member :singular link :test #'(lambda (singular element)
                                                   (when (listp element)
                                                     (member singular element))))))
         'atom)
        (t t)))

(defun accessor-names-list (description &optional
			    (external-conc-name
			      (unit.external-conc-name description)))
  "Returns a list of the names of the public, external accessor
   functions for the slots and links."

  (flet ((accessor-name (basic-slot)
           (unless (and (basic-slot.inherited-p basic-slot)
                        (basic-slot.private basic-slot))
             (form-symbol external-conc-name (basic-slot.name basic-slot)))))
    (nconc (mapc-condcons #'accessor-name (unit.slots description))
	   (mapc-condcons #'accessor-name (unit.links description))
           (mapcar #'(lambda (name)
                       (form-symbol external-conc-name name))
                   *external-unit-slots*))))

(defun filter-slot-definitions (description)
  "Returns a list of defstruct slot definitions based on the slots and
   indexes in DESCRIPTION."
  ;; -> Removes any slots which are included from another unit.
  ;; -> Removes keywords like :EVENTS.

  (flet ((filter-one-slot (slot-instance)

	  ;; Helper function for filter-slot-definitions.  This
	  ;; filters a single slot spec.  If the slot was included,
	  ;; it returns nil.  Otherwise, it always returns a list
	  ;; even if SLOT is a symbol.
  
	  (unless (slot.inherited-p slot-instance)
	    `(,(slot.name slot-instance)
	      ,@(if (slot.initial-value-present-p slot-instance)
		    `(,(slot.initial-value slot-instance)
		      :type      ,(slot.type slot-instance))
		    nil)))))

    (mapc-condcons #'filter-one-slot (unit.slots description))))


(defun check-define-unit-reasonablity (description)
  "Check for errors in the unit definition.
   Signal an error if any are found."
  ;; Check for conflicts with GBB internal slots.
  (check-for-name-conflicts description)
  ;; Check that read-only slots don't have any update events.
  (check-slot-reasonability description)
  ;; If the unit is instantiable, it must have a paths specification.
  (error-when (and (unit-instantiable-p description)
		   (null (unit.paths description)))
     "No paths directly specified ~:[~;or included ~]for ~s."
     (unit.included description) (unit.name description)))


(defun check-for-name-conflicts (description)
  "Signal an error if any of the user's slots or links conflict
   with slots in basic-unit."
  (flet ((check-conflict (slot-name)
           (when (find-slot slot-name description)
	     (error "The ~a slot in ~s conflicts with a GBB slot of the same name."
		    slot-name (unit.name description)))
           (when (find-link slot-name description)
	     (error "The ~a link in ~s conflicts with a GBB slot of the same name."
		    slot-name (unit.name description)))))
    (mapc #'check-conflict *internal-unit-slots*)
    (mapc #'check-conflict *external-unit-slots*)
    t))

(defun check-slot-reasonability (description)
  "Warn if a read-only slot has update events.  If any
   such slots are found, set their update events to nil."
  (dolist (slot-instance (unit.slots description))
    (when (slot.update-events slot-instance)
      (let* ((slot-name (slot.name slot-instance)))
        (when (eq :read-only (slot.modifiable slot-instance))
          (gbb-warning
            "The ~s slot in ~s has been declared read-only.~@
             The update events defined for ~a:~%~s~%will be ignored."
             slot-name (unit.name description) slot-name
             (slot.update-events slot-instance))
          (setf (slot.update-events slot-instance) nil))))))

(defun check-unit-redefinition (new-description old-description)
  "Check that the new definition of a unit is compatible with
   the old definition.  Prints a warning message if any problems
   are found."

  (when (null old-description)
    (return-from check-unit-redefinition t))

  ;; Check inheritance consistancy.
  (when (and (unit.subtypes old-description)
	     (unit.included old-description)
	     (not (eq (unit.included old-description)
		      (unit.included new-description))))
    (gbb-warning "Unit ~s incompatibly-changed.~@
                  Things will probably break."))
  ;; Check slots.
  ;; Check links.
  ;; Check indexes.
  t)


;;;; --------------------------------------------------------------------------
;;;;   Unit Accessor Functions
;;;; --------------------------------------------------------------------------

;;; Functions for generating the public functions to access and update
;;; slots in units.  Remember that the accessor functions generated by
;;; defstruct aren't directly accessible to the user.  The net result of
;;; this code are functions of the form:
;;;
;;;   (defun hyp$belief (unit)
;;;      "Function to access the BELIEF slot of UNIT."
;;;      (access-event-1 unit 'belief (internal-hyp$belief unit))
;;;      ;; ... More events ...
;;;      (internal-hyp$belief unit))
;;;
;;;   (defsetf hyp$belief (unit) (new-value)
;;;      "SETF to change the belief slot of UNIT."
;;;      `(let ((old-value (internal-hyp$belief ,unit)))
;;;         (setf (internal-hyp$belief ,unit) ,new-value)
;;;         (update-event-1 ,unit 'belief new-value old-value)
;;;         ;; ... More events ...
;;;         ,new-value))
;;;
;;; The current value of the slot is computed for each event because the
;;; events might modify it.
;;;
;;; If there are no events for a slot then the public function just
;;; indirects to the internal function.


(defun build-access-forms (description &optional
			   (external-conc-name
			     (unit.external-conc-name description)))

  "Builds code to define the slot access forms."

  (let ((internal-conc-name (unit.internal-conc-name description))
	(unit-type (unit.name description))
	(declarations nil)
	(functions nil))

    (flet ((simple-accessor (external-name internal-name)
	     (push `(setf (symbol-function ',external-name)
			  (symbol-function ',internal-name))
		   functions)
	     (push `(proclaim '(function ,external-name (,unit-type) t))
		   declarations)))

      (dolist (slot-instance (unit.slots description))
	(let* ((slot-name (slot.name slot-instance))
	       (external-name (form-symbol external-conc-name slot-name))
	       (internal-name (form-symbol internal-conc-name slot-name))
	       (access-events (slot.access-events slot-instance)))
	  (cond ((and (slot.inherited-p slot-instance)
                      (slot.private slot-instance))
                 ;; Don't generate any access forms for private slots.
                 nil)
                (access-events
		 (push `(defun ,external-name (unit-instance)
			  (when %%run-events%%
			    ,.(mapcar #'(lambda (event)
                                          (build-event-call
                                            event
                                            'unit-instance
                                            `',slot-name
                                            `(,internal-name unit-instance)))
                                      access-events))
			  (,internal-name unit-instance))
		       functions))
		(t (simple-accessor external-name internal-name)))))

      ;; Slots from basic-unit.
      (dolist (slot *external-unit-slots*)
	(let* ((external-name (form-symbol external-conc-name slot))
	       (internal-name (form-gbb-symbol "BASIC-UNIT." slot)))
	  (simple-accessor external-name internal-name)))

      (nconc declarations functions))))


;;; Setf methods are generated for all slots (except private slots),
;;; including read-only slots and the external slots from basic-unit,
;;; but some of the setf methods are only there to generate errors.

(defun build-update-forms (description &optional
			   (external-conc-name
			     (unit.external-conc-name description)))
  "Builds code to define the setf methods for slots."
  (let ((internal-conc-name (unit.internal-conc-name description))
	(unit-type (unit.name description)))
    (mapc-condcons
      #'(lambda (slot-instance)
	  (let* ((slot-name (slot.name slot-instance))
		 (external-name (form-symbol external-conc-name slot-name))
		 (internal-name (form-symbol internal-conc-name slot-name))
		 (update-events (slot.update-events slot-instance)))
	    (cond ((and (slot.inherited-p slot-instance)
                        (slot.private slot-instance))
                   ;; Don't generate any update forms for private slots.
                   nil)
                  ((eq :read-only (slot.modifiable slot-instance))
		   `(defsetf ,external-name (unit) (new-value)
		      (error "The slot ~s is read only in the unit ~s.~@
                              Attempting to modify ~s with ~s."
			     ',slot-name ',unit-type unit new-value)))
		  (t
		   `(defsetf ,external-name (unit) (new-value)
		      (build-update-body
			',unit-type ',slot-name ',internal-name ',update-events
			unit new-value))))))
      (unit.slots description))))


(defun build-update-body (unit-type slot-name internal-accessor update-events
			  unit-instance new-value)
  "Builds code to actually do the SETF of a unit's slot."
  ;; Note that SETF takes care of order of evaluation and multiple
  ;; evaluation issues so I don't need to worry about that here.
  (let* ((slot (find-slot slot-name unit-type))
	 ;; Forms that must be included regardless of whether there
	 ;; are events or not.
	 (change-forms
	   `((setf (,internal-accessor ,unit-instance) ,new-value)
	     ,@(when (slot.index-source-p slot)
		 (ecase (slot.modifiable slot)
		   (:dynamic
		     `((update-canonical-index-and-move
			 ',unit-type ',slot-name ,unit-instance)))
		   (:static
		     `((check-canonical-index
			 ',unit-type ',slot-name ,unit-instance))))))))
    (if (null update-events)
	`(progn ,@change-forms ,new-value)
	(let ((old-value-v (newsym "OLD-VALUE-")))
	  `(let ((,old-value-v (,internal-accessor ,unit-instance)))
	     ,@change-forms
	     (when %%run-events%%
	       ,.(mapcar #'(lambda (event)
                             (build-event-call
                               event
                               unit-instance
                               `',slot-name
                               new-value
                               old-value-v))
			 update-events))
	     ,new-value)))))


;;;; --------------------------------------------------------------------------
;;;;   Generic Slot Access Functions
;;;; --------------------------------------------------------------------------

(defun get-unit-slot (unit-instance slot-name)

  "GET-UNIT-SLOT unit-instance slot-name

   Return the value of SLOT-NAME in UNIT-INSTANCE.  SLOT-NAME should be a
   symbol which names a slot in UNIT-INSTANCE.  This function will run any
   access-events defined for that slot.

   This function may be used as a place form for SETF."

  (let* ((description (get-unit-description unit-instance))
         (slot (find-slot slot-name description))
         external-accessor)
    (cond
      ;; Normal unit slot exists.
      (slot
       (when (slot.private slot)
         ;; This is somewhat of a hack.  If the slot is private
         ;; then find where it was declared private and execute
         ;; that accessor.
         (setf description (search-included-units
                             #'(lambda (descrip)
                                 (let ((slot (find slot-name (unit.slots descrip)
                                                   :key #'slot.name :test #'string=)))
                                   (and slot (not (slot.inherited-p slot)))))
                             description)))
       (setf external-accessor (form-symbol-in-package
                                 (unit.package description)
                                 (unit.external-conc-name description)
                                 (slot.name slot)))
       (funcall external-accessor unit-instance))
      ;; Special external slot included in all units.
      ((member slot-name *external-unit-slots* :test #'string=)
       (get-structure-slot unit-instance slot-name))
      ;; No slot exists.
      (t
       (error "~a is not the name of a slot in ~s."
              slot-name (-> unit-instance))))))

(defun set-unit-slot (unit-instance slot-name new-value)

  "SET-UNIT-SLOT unit-instance slot-name new-value

   Change the value of SLOT-NAME in UNIT-INSTANCE to NEW-VALUE.  SLOT-NAME
   should be a symbol which names a slot in UNIT-INSTANCE.  This function
   will run any update-events defined for that slot."

  (let* ((description (get-unit-description unit-instance))
         (slot (find-slot slot-name description))
         (modifiable (cond (slot (slot.modifiable slot))
                           ((member slot-name *external-unit-slots*
                                    :test #'string=)
                            :read-only)
                           (t :dynamic)))
         old-value)
    (cond
      ((eq modifiable :read-only)
       (error "~a is a read-only slot in ~s." slot-name (-> unit-instance)))
      ((null slot)
       (error "~a is not the name of a slot in ~s."
              slot-name (-> unit-instance)))
      (t
       (shiftf old-value
               (get-structure-slot unit-instance slot-name)
               new-value)
       (when (slot.index-source-p slot)
         (ecase modifiable
           (:dynamic
            (update-canonical-index-and-move description slot-name unit-instance))
           (:static
            (check-canonical-index description slot-name unit-instance))))
       (when %%run-events%%
         (dolist (event (slot.update-events slot))
           (funcall event unit-instance
                    slot-name
                    new-value
                    old-value)))
       new-value))))

(defsetf get-unit-slot set-unit-slot)


;;;; --------------------------------------------------------------------------
;;;;   Path Generation Code
;;;; --------------------------------------------------------------------------

(defun build-path-generation-forms (description)

  "Build code that will define a function to determine where to place
   a unit instance on the blackboard.  This function returns a list of
   forms the first of which is the definition of a function which will
   return a list of space-instances."

  ;; Only build the function if the unit can be instantiated.
  (unless (unit-instantiable-p description)
    (return-from build-path-generation-forms nil))

  (let* ((fn-name (form-symbol "%%GBB-"
			       (unit.name description)
			       "-PATH-GENERATION-FUNCTION"))
	 (paths (unit.paths description))
	 (result-v (newsym "RESULT"))
	 (unit-instance-v (newsym "UNIT"))
         (path-index-bindings
           (build-path-generation-binding-code description unit-instance-v)))
    (setf (unit.path-generation-fn description) fn-name)
    `((defun ,fn-name (,unit-instance-v)
       ;; Avoid compiler warnings about "bound but not used"
       ,unit-instance-v
       (let ((,result-v nil)
	     ,@path-index-bindings)
         ;; Avoid compiler warnings about "bound but not used"
         ,@(mapcar #'car path-index-bindings)
	 ,@(mapcar
	     #'(lambda (path-clause)
		 (case (first path-clause)
		    ((:path :paths)
		     `(npush-list (list ,@(rest path-clause)) ,result-v))
		    ((:path-structure :path-structures)
		     `(npush-list (change-paths ,@(rest path-clause)) ,result-v))
		    (otherwise
		     (error "Unknown keyword (~s) in :PATHS argument to define-unit.~@
                             The :PATHS argument is:~%~s"
			    (first path-clause) paths))))
	     paths)
	 ,result-v)))))


(defun build-path-generation-binding-code (description unit-instance-v)

  "Returns a list of variable/value pairs, suitable for use with
   LET, to bind the path index values to the path index symbols.
   The symbols will be in the current package."

  (mapcar #'(lambda (unit-index)
	      `(,(form-symbol-in-package
		   (unit-index.package unit-index)
		   (unit-index.name unit-index))
		,(if (unit-index.index-structure unit-index)
		     `(funcall ',(first (unit-index.functions unit-index))
			       (,(unit-index.slot-accessor unit-index) ,unit-instance-v))
		     `(,(unit-index.slot-accessor unit-index) ,unit-instance-v))))
	  (unit.p-indexes description)))


;;;; --------------------------------------------------------------------------
;;;;   Unit Creation Code
;;;; --------------------------------------------------------------------------

(defun build-unit-construction-code (description)
  (when (unit-instantiable-p description)
    (let ((constructor-name (unit.internal-constructor description))
	  (maker-name (unit.external-constructor description)))
      `(,(build-unit-construction-function description constructor-name maker-name)
	,(build-unit-construction-transform description constructor-name maker-name)
	,(build-dummy-unit-construction-function description constructor-name)
	))))

  
(defun build-unit-construction-function (description constructor-name maker-name)

  "Return a form that will define a function to create a unit.
   The created function's arguments are the 
   alternating keywords and values to initialize the slots of the unit."

  `(defun ,maker-name (&rest key-value-pairs)
     ,(construction-function-documentation description maker-name)
     ,(build-arglist-check-code description 'key-value-pairs) 
     (let* ((the-description (get-unit-description ',(unit.name description)))
	    (unit-instance
	      (apply #',constructor-name
		     (args-for-unit-construction key-value-pairs the-description)))
	    bb-space-paths)
       ,(build-link-initialization-code description 'unit-instance 'key-value-pairs)
       ;; Name the unit and insert it into the hash table.
       ,.(build-name-code-if-necessary description 'unit-instance)
       ;; Runtime deletion of dummy unit instance if we're creating the real one now.
       (when (dummy-unit-instance-exists-p unit-instance)
	  ,(build-delete-dummy-unit-instance-code 'the-description 'unit-instance))
       ;; Get the bb/space paths
       (setf bb-space-paths
	     (,(unit.path-generation-fn description) unit-instance))
       ;; Put the dimensional index information into canonical form.
       (setf (basic-unit.%%indexes%% unit-instance)
	     (compute-canonical-indexes the-description unit-instance))
       (dolist (path bb-space-paths)
         (when path
           (insert-unit-on-space-instance
             unit-instance
             (if (path-structure-p path)
		 (get-space-instance-from-path-structure path)
		 (get-space-instance-from-path path)))))
       ;; Run the unit's creation events and each slot's initialization events.
       ,@(build-initialization-event-code description 'unit-instance 'key-value-pairs)
       ,@(build-creation-event-code description 'unit-instance)
       unit-instance)))


(defun build-unit-construction-transform (description constructor-name maker-name)
  
  `(deftransform (,maker-name :gbb) (&rest key-value-pairs)
     (catch 'no-transform
       (let* ((the-description (get-unit-description ',(unit.name description)))
	      (constructor-args nil))
	 (dolist-by-twos (key value key-value-pairs)
	   (unless (keyword-or-quoted-keyword-p key)
	     (throw 'no-transform '%pass%))
	   (setf key (unquote key))
	   (case key
	      ((,@(mapcar #'make-keyword *external-unit-slots*)
		,@(mapcar #'make-keyword (slot-names description)))
	       (setf constructor-args (list* key value constructor-args)))
	      ((,@(mapcar #'make-keyword (link-names description)))
	       ;; Don't do anything with links for now...
	       nil)
	      (otherwise
	       (error "~s doesn't name a link or slot in ~s."
		      key ',(unit.name description)))))
	 ;; Must copy on lispm's because &rest args are on the stack.
	 #+LISPM (setf key-value-pairs (copy-list key-value-pairs))
	 (xform-build-transformed-construction-code
	   the-description ',constructor-name constructor-args key-value-pairs)))))


;;; XFORM-BUILD-TRANSFORMED-CONSTRUCTION-CODE returns a form that is
;;; functionally equivalent to make-<unit>.  The difference is that the
;;; transformed code can use linkf instead of add-unit-link to initialize
;;; the links of the new unit.

(defun xform-build-transformed-construction-code
          (description constructor-name constructor-args kvp)

  `(let* ((unit-instance (,constructor-name ,@constructor-args))
	  (the-description (get-unit-description ',(unit.name description)))
	  bb-space-paths)
     ,(xform-build-link-initialization-code description 'unit-instance kvp)
     ;; Name the unit and insert it into the hash table.
     ,.(build-name-code-if-necessary description 'unit-instance)
     ;; Runtime deletion of dummy unit instance if we're creating the real one now.
     (when (dummy-unit-instance-exists-p unit-instance)
       ,(build-delete-dummy-unit-instance-code 'the-description 'unit-instance))
     (setf bb-space-paths
	   (,(unit.path-generation-fn description) unit-instance))
     ;; Put the dimensional index information into canonical form.
     (setf (basic-unit.%%indexes%% unit-instance)
	   (compute-canonical-indexes the-description unit-instance))
     (dolist (path bb-space-paths)
       (when path
         (insert-unit-on-space-instance
           unit-instance
           (if (path-structure-p path)
               (get-space-instance-from-path-structure path)
               (get-space-instance-from-path path)))))
     ;; Run the initialization events and the unit's creation events.
     ,@(xform-build-initialization-event-code description 'unit-instance kvp)
     ,@(build-creation-event-code description 'unit-instance)
     unit-instance))



(defvar *construction-function-documentation-text*
	"Creates a unit of type ~s and inserts it on the appropriate~@
         space(s) as specified by PATHS in the unit definition.")

(defun construction-function-documentation (description maker-name)

  "CONSTRUCTION-FUNCTION-DOCUMENTATION description maker-name
   Return a documentation string for a unit construction function."

  (let* ((fn-name-length (length (string maker-name)))
	 (all-slots (nconc (if (anonymous-unit-type-p description)
                               (remove 'name *external-unit-slots*)
                               (copy-list *external-unit-slots*))
			   (mapcar #'slot.name (unit.slots description))
			   (mapcar #'link.name (unit.links description))))
	 (n-slots (length all-slots))
 	 (text (format nil *construction-function-documentation-text*
		       ;; Use string here to keep it uppercase if
		       ;; *print-case* is not :upcase.
		       (string (unit.name description)))))

    (cond ((or (> fn-name-length 40)
	       (> n-slots 10))
	   (format nil "~a~%  &key~{~<~%      ~0,80:; ~a~>~^~}~2%~a"
		   (string maker-name)
		   all-slots
		   text))
	  (t
	   (let ((fmt-string (format nil "~~a &key~~{~~<~~%~a~~0,80:; ~~a~~>~~^~~}~~2%~~a"
				     (make-string (+ fn-name-length 5)
						  :initial-element #\space))))
	     (format nil fmt-string
		     (string maker-name)
		     all-slots
		     text))))))



(defun build-dummy-unit-construction-function (description constructor-name)
  
  "BUILD-DUMMY-UNIT-CONSTRUCTION-FUNCTION description constructor-name maker-name

   Returns a form that will define a function to create a dummy unit for all
   named units.  Returns NIL for anonymous units. 
   The created function's arguments are the keyword :NAME and its value.
   The function returns either a pre-existing dummy-unit-instance or else creates
   a new one, puts it in the dummy-hash-table, and returns it."
  
  (unless (anonymous-unit-type-p description)
    (let* ((unit-name (unit.name description))
	   (dummy-maker-name (form-gbb-symbol "MAKE-DUMMY-" unit-name)))
      `(defun ,dummy-maker-name (&rest key-value-pairs)
	 "Internal GBB function for dummy unit type creation."
	 (let ((the-description (get-unit-description ',unit-name))
               (dummy-hash-table (unit-info.dummy-hash-table
                                   (get-unit-info ',unit-name))))
	   (or
	     (gethash (getf key-value-pairs :name) dummy-hash-table)
	     (let ((unit-instance (apply #',constructor-name
					 (args-for-unit-construction
					   key-value-pairs the-description))))
	       (setf (gethash (basic-unit.name unit-instance) dummy-hash-table)
		     unit-instance)
	       unit-instance)))))))


;;; The arguments `descrip-v' and `unit-v' are symbols that are
;;; bound by make-<unit>.

(defun build-delete-dummy-unit-instance-code (descrip-v unit-v)
  
  "BUILD-DELETE-DUMMY-UNIT-INSTANCE-CODE descrip-v unit-v

   For each link in the dummy unit instance, check to see if it has values, and
   if it does, call linkf on the actual unit instance for each of those values
   without triggering events.  Finally, delete the dummy unit instance from its
   hash table."
  
  `(let* ((unit-name (basic-unit.name ,unit-v))
          (dummy-hash-table (unit-info.dummy-hash-table (get-unit-info unit-name)))
	  (dummy-unit-instance (gethash unit-name dummy-hash-table)))
     (dolist (link-name (link-names ,descrip-v))
       (let ((link-value (get-structure-slot dummy-unit-instance link-name)))
	 (when link-value
	   (dolist-or-atom (linked-unit-instance link-value)
	     (with-events-disabled
	       (add-unit-link link-name ,unit-v linked-unit-instance))))))
     (remhash unit-name dummy-hash-table)))


(defun build-unit-deletion-code (description)
  ;; If you can't instantiate it, you can't delete it either.
  (when (unit-instantiable-p description)
    (let ((deleter-name (form-symbol "DELETE-" (unit.name description))))
      `(,(build-unit-deletion-function description deleter-name)
	,(build-unit-deletion-transform description deleter-name)))))


(defun build-unit-deletion-function (description deleter-name)

  "BUILD-UNIT-DELETION-FUNCTION description deleter-name
   Return a form to define a function to delete a unit instance or
   list of unit instances."
  
  (let* ((unit-type (unit.name description))
	 (unlinkf-all-calls
	   (mapcar #'(lambda (link)
		       `(delete-all-links ',(link.name link) unit-instance))
		   (unit.links description)))
	 (deletion-event-calls
	   (mapcar #'(lambda (deletion-event)
                       (build-event-call deletion-event 'unit-instance))
		   (deletion-event-names description))))
    `(defun ,deleter-name (unit-instance)
       ,(format nil "~a unit-instances ~
           ~%   Deletes a unit-instance from the blackboard."
		deleter-name)
       (unless (typep unit-instance ',unit-type)
	 (error "~s is not of type ~s."
		(-> unit-instance) ',unit-type))
       ,@(when deletion-event-calls
	   `((when %%run-events%%
	       ,@deletion-event-calls)))
       ;; This may leave links hanging.  It should really check the
       ;; actual type of the argument and delete all links based on that type.
       ,@unlinkf-all-calls
       (dolist (space-instance (basic-unit.%%space-instances%% unit-instance))
	 (delete-unit-from-space-instance unit-instance space-instance))
       ,@(unless (anonymous-unit-type-p description)
	   `((remhash (basic-unit.name unit-instance)
		      (unit-info.hash-table (get-unit-info unit-instance))))))))

(defun build-unit-deletion-transform (description deleter-name)
  (declare (ignore description deleter-name))
  ;; Don't worry about the deletion transform for now...
  nil)



;;; BUILD-NAME-CODE-IF-NECESSARY returns a list of forms that will be
;;; spliced in to make-<unit>.  The arguments `descrip-v' and
;;; `unit-instance-v' are symbols that are bound by make-<unit>.

(defun build-name-code-if-necessary (description unit-instance-v)
  "If there is a name slot in the unit description then return
   code to generate a name for the unit; otherwise return nil."
  (let ((unit-type (unit.name description)))
    (unless (anonymous-unit-type-p description)
      `((when (eq (basic-unit.name ,unit-instance-v) :anonymous)
	  (setf (basic-unit.name ,unit-instance-v)
		(,(unit.name-function description) ,unit-instance-v)))
	(when (find-unit-by-name (basic-unit.name ,unit-instance-v) ',unit-type)
	  (cerror "Delete the existing unit and replace it with the new unit."
		  "A ~s named ~s already exists."
		   ',unit-type (basic-unit.name ,unit-instance-v))
          (,(form-symbol "DELETE-" unit-type)
           (find-unit-by-name (basic-unit.name ,unit-instance-v) ',unit-type)))
	(setf (gethash (basic-unit.name ,unit-instance-v)
		       (unit-info.hash-table (get-unit-info ',unit-type)))
	      ,unit-instance-v)))))


(defun build-creation-event-code (description unit-instance-v)
  "Returns a list of forms to run the creation events for this unit."
  (when (creation-event-names description)
    `((when %%run-events%%
	,.(mapcar #'(lambda (event) (build-event-call event unit-instance-v))
		  (creation-event-names description))))))


(defun build-initialization-event-code (description unit-instance-v kvp-v)

  "Returns a list of forms to run the initialization events
   for the slots and links of this unit."

  (let ((conc-name (unit.internal-conc-name description))
	(temp-v (gensym)))

    (flet ((build-slot-clause (slot)
	     (when (slot.initialization-events slot)
	       (let* ((slot-name (slot.name slot))
		      (current-value-form `(,(form-symbol conc-name slot-name)
					    ,unit-instance-v)))
		 `(,(form-keyword slot-name)
		   ,.(mapcar #'(lambda (event)
                                 (build-event-call event
                                                   unit-instance-v
                                                   `',slot-name
                                                   current-value-form))
			     (slot.initialization-events slot))))))
	   (build-link-clause (link)
	     (let* ((link-name (link.name link))
		    (link-value-form `(,(form-symbol conc-name link-name)
				       ,unit-instance-v)))
	       `(,(form-keyword link-name)
		 ;; Initialization event calls::
		 ,.(mapcar		       
		     #'(lambda (event)
                         (build-event-call event
                                           unit-instance-v
                                           `',link-name
                                           link-value-form))
		     (link.initialization-events link))
		 ;; Update events for the back links::
		 ,(if (link.singular link)
		      `(do-link-update-events
			 (get-link-from-link-name
			   ',(link.i-link-name link)
			   ',(link.i-link-unit link))
			 ,link-value-form
			 ,unit-instance-v)
		      `(dolist (,temp-v ,link-value-form)
			 (do-link-update-events
			   (get-link-from-link-name
			     ',(link.i-link-name link)
			     ',(link.i-link-unit link))
			   ,temp-v
			   ,unit-instance-v)))))))

      `((when %%run-events%%
	  (dolist-by-twos (key value ,kvp-v)
	    ;; This case form will expand into something like this:
	    ;;  (case key
	    ;;    (:slot-1
	    ;;      (slot-1-init-event-1 unit-instance 'slot-1 <value>)
	    ;;      (slot-1-init-event-2 unit-instance 'slot-1 <value>)
	    ;;      ...)
	    ;;    (:slot-2 ...)
	    ;;    ...)
	    (case key
	       ;; Slots
	       ,.(mapc-condcons #'build-slot-clause (unit.slots description))
	       ;; Links
	       ,.(mapc-condcons #'build-link-clause (unit.links description))
	       ;; Ignore anything else.
	       (otherwise nil))))))))


(defun xform-build-initialization-event-code (description unit-instance-v kvp)

  "Returns a list of forms to run the initialization events
   for the slots and links of this unit."

  (let ((the-package (unit.package description))
	(conc-name (unit.internal-conc-name description))
	(forms nil)
	(temp-v (gensym))
	slot)

    (flet ((build-slot-event-calls (slot)
	     ;; BUILD-SLOT-EVENT-CALLS will return the code for
	     ;; executing the initialization events for one slot.
	     (let* ((slot-name (slot.name slot))
		    (current-value-form `(,(form-symbol-in-package
					     the-package
					     conc-name
					     slot-name)
					  ,unit-instance-v)))
	       (mapcar #'(lambda (event)
                           (build-event-call event
                                             unit-instance-v
                                             `',slot-name
                                             current-value-form))
		       (slot.initialization-events slot))))
	   (build-link-event-calls (link)
	     ;; BUILD-LINK-EVENT-CALLS will return the code for
	     ;; executing the initialization events for one link
	     ;; and the corresponding update events for the backlink.
	     (let* ((link-name (link.name link))
		    (link-value-form `(,(form-symbol-in-package
					     the-package
					     conc-name
					     link-name)
					  ,unit-instance-v))
		    i-link)
	       (unless (unit-type-p (link.i-link-unit link))
		 (throw 'no-transform '%pass%))
	       (setf i-link (get-inverse-link link))
	       `(,.(mapcar #'(lambda (event)
                               (build-event-call event
                                                 unit-instance-v
                                                 `',link-name
                                                 link-value-form))
			   (link.initialization-events link))
		 ,@(when (link.update-events i-link)
		     (if (link.singular link)
			 (build-link-update-event-code
			   (get-inverse-link link)
			   link-value-form
			   unit-instance-v)
			 `((dolist (,temp-v ,link-value-form)
			     ,@(build-link-update-event-code
				 (get-inverse-link link)
				 temp-v
				 unit-instance-v)))))))))

      (dolist-by-twos (key value kvp)
	(setf slot (or (find-slot key description)
		       (find-link key description)))
	(when slot
          (setf forms (nconc forms
                             (if (slot-p slot)
                                 (build-slot-event-calls slot)
                                 (build-link-event-calls slot))))))

      (when forms
	`((when %%run-events%% ,.forms))))))


;;; Handling initial slot values supplied to the make function:
;;;
;;; The problem is that if we just let the defstruct construction
;;; function initialize the links in the unit they won't have their
;;; corresponding back links set.  Instead, we just allow defstruct
;;; to initialize the slots.  The links are initialized separately.

(defun args-for-unit-construction (key-value-pairs description)
  "Return a list of keys and values omitting the key-value pairs
   which refer to links."
  (let ((result nil))
    (dolist-by-twos (key value key-value-pairs result)
      (unless (find-link key description)
	(setf result (list* key value result))))))


(defun build-arglist-check-code (description kvp-v)

  "Returns code to check that the arguments make sense."

  `(dolist-by-twos (key val ,kvp-v)
     (error-unless
       (or (member key *external-unit-slots* :test #'string=)
	   (member key '(,.(mapcar #'make-keyword (slot-names description))
			 ,.(mapcar #'make-keyword (link-names description)))
		   :test #'eq))
	"~s is an unknown keyword for ~s."
	key ',(unit.external-constructor description))))


;;; The arguments `unit-v' and `kvp-v' are symbols that will be bound by
;;; MAKE-<unit> when the code that this function returns actually runs.
;;; UNIT-V will be bound to the newly created unit-instance.  KVP-V will
;;; be bound to the key value pairs that the user has given to
;;; MAKE-<unit>.

(defun build-link-initialization-code (description unit-v kvp-v)
  "Returns code to initialize the links after the unit is
   created in the make function.  Does not run any events.
   The links are initialized separately so that the backlinks
   will be set properly."

  `(with-events-disabled
     (dolist-by-twos (key val ,kvp-v)
       (when (member key ',(mapcar #'make-keyword (link-names description))
		     :test #'eq)
         (add-unit-link key ,unit-v val)))))


(defun xform-build-link-initialization-code (description unit-v kvp)
  (let ((conc-name (unit.external-conc-name description))
	(forms nil)
	(temp-v (gensym))
	link linkf-accessor)
    (dolist-by-twos (key val kvp)
      (when (setf link (find-link key description))
	(unless (unit-type-p (link.i-link-unit link))
	  (throw 'no-transform '%pass%))
	(setf linkf-accessor (form-symbol-in-package
			       (unit.package description)
			       conc-name
			       key))
	(push `(let ((,temp-v ,val))
		 (if (listp ,temp-v)
		     (linkf-list (,linkf-accessor ,unit-v) ,temp-v)
		     (linkf (,linkf-accessor ,unit-v) ,temp-v)))
	      forms)))
    `(with-events-disabled
       ,.(nreverse forms))))


;;;; --------------------------------------------------------------------------
;;;;   Generic Creation and Deletion Functions
;;;; --------------------------------------------------------------------------

(defun make-unit (unit-type &rest args)

  "MAKE-UNIT unit-type &rest args

   Create a unit-instance of type UNIT-TYPE.  ARGS is a list of
   keyword/value pairs which specify initial values for slots
   and links in the new unit-instance."

  (let ((description (get-unit-description unit-type)))
    (apply (unit.external-constructor description) args)))

(defun delete-unit (unit-instance)

  "DELETE-UNIT unit-instance

   Deletes UNIT-INSTANCE.  This runs the deletion-events defined for
   this unit, unlinks it from all other units it is linked to, and
   removes it from any spaces it is on."

  (let* ((description (get-unit-description unit-instance))
         (deleter-name (form-symbol-in-package
                         (unit.package description)
                         "DELETE-"
                         (unit.name description))))
    (funcall deleter-name unit-instance)))

(defun copy-unit (unit-instance &rest key-value-pairs)

  "COPY-UNIT unit-instance &rest key-value-pairs

   Creates a copy of UNIT-INSTANCE."

  unit-instance key-value-pairs ;; ignore
  (error "COPY-UNIT is not implemented yet."))


;;; Try to be smart about redefinition of units.  That is, if a unit
;;; description already exists, the user may have some information in
;;; the hash tables that s/he would like to save.

(defun initialize-unit-info (description)
  "Initialize the dynamic unit info data structures.  This includes the
   unit hash table and the dummy unit hash table for DESCRIPTION."
  (let* ((unit-name (unit.name description))
         (info (or (get-unit-info unit-name t)
                   (setf (get-unit-info unit-name)
                         (make-unit-information :name unit-name)))))
    (when (and (not (anonymous-unit-type-p description))
               (or (null (unit-info.hash-table info))
                   (null (unit-info.dummy-hash-table info))))
      (setf (unit-info.hash-table info) (make-hash-table :test 'equal))
      (setf (unit-info.dummy-hash-table info) (make-hash-table :test 'equal)))))


(defun initialize-unit-subtypes (new-description &optional old-description)
  "Initialize the subtypes slot in NEW-DESCRIPTION.  If
   OLD-DESCRIPTION has subtypes then they are copied."
  (setf (unit.subtypes new-description)
	(if (and old-description (unit.subtypes old-description))
	    (copy-list (unit.subtypes old-description))
	    nil)))

(defun propagate-unit-subtypes-up (description
			      &optional (unit-type (unit.name description)))
  "Add UNIT-TYPE to the list of subtypes of all its ancestors."
  ;; Lucid leaves garbage in the subtypes slot when this is
  ;; compiled (speed 3) (safety 0).
  (declare (optimize (speed 0) (safety 3)))
  (unless (eq (unit.name description) unit-type)
    (pushnew unit-type (unit.subtypes description) :test #'eq))
  (let ((included-unit (unit.included description)))
    (when included-unit
      (propagate-unit-subtypes-up (get-unit-description included-unit) unit-type))))


(defun build-misc-code (description)
  "Return a list of forms to handle miscellaneous stuff:
     1. Build the definiton of FIND-<unit>, which is a shorthand for
        (find-unit-by-name ... '<unit>)."

  (let* ((unit-type (unit.name description))
	 (function-name (form-symbol "FIND-" unit-type)))
    `((defun ,function-name (name)
	,(format nil "~a name~2%Shorthand for (FIND-UNIT-BY-NAME name '~a)."
		 (symbol-name function-name) (symbol-name unit-type))
	(find-unit-by-name name ',unit-type)))))


(defun additional-unit-exports (description) 
  "This function returns a list of additional symbols associated
   with the unit that are to be exported.  The unit-name, the
   predicate, and the constructor are taken care of by PARSE-
   NAME-AND-OPTIONS.  This function takes care of DELETE-<unit-type>,
   FIND-<unit-type>.

   Obviously, this function will have to be changed if the
   conventions are changed for naming these functions."

  (let ((unit-type (unit.name description))
	(symbols nil))
    (when (unit-instantiable-p description)
      (push (form-symbol "DELETE-" unit-type) symbols)
      (push (form-symbol "FIND-" unit-type) symbols))
    symbols))



;;; This function takes the indexing information from a unit-instance and
;;; puts it into a canonical form.  This form is a list
;;;
;;;     (<index-1> <index-2> ...)
;;;
;;; The indexes are in the same order as they are in the indexes slot of
;;; the unit description.  Each index-n is either a list of values (in the
;;; case that the index is part of a composite) or simply the value (if the
;;; index is not composite).  The values are represented as a list
;;;
;;;     (<element-type> <i-s-type> . (<value-1> <value-2> ...))
;;; 
;;; where
;;;    <element-type> is an index element type (:point, :range, :label, etc.)
;;;    <i-s-type>     is an index structure type (:scalar :list, etc.)

(defun compute-canonical-indexes (description unit-instance)

  "Extracts the indexing information for UNIT-INSTANCE, puts it into
   a canonical form, and stores the canonical form on the %%INDEXES%%
   slot of the instance."

  (mapcar #'(lambda (unit-index)
	      (compute-canonical-index unit-index unit-instance))
	  (unit.d-indexes description)))


(defun compute-canonical-index (unit-index unit-instance)

  (let* ((element-type (unit-index.element-type unit-index))
	 (functions (unit-index.functions unit-index))
	 (n-functions (length functions))
	 (index-structure (unit-index.index-structure unit-index))
	 (i-s-type (and index-structure (index-structure.type index-structure)))
	 (datatype (unit-index.datatype unit-index))
	 (the-slot (funcall (unit-index.slot-accessor unit-index)
			    unit-instance)))

    (flet ((canonical-value (elt)
	     (list* element-type
		    i-s-type
		    (cond ((= 0 n-functions)
			   elt)
			  ((= 1 n-functions)
			   (funcall (first functions) elt))
			  (t
			   (mapcar #'(lambda (f)
				       (funcall f elt))
				   functions))))))

      (error-unless (typep the-slot datatype)
	 "The value of the ~a slot of ~s was expected to be a ~s~@
          but it is a ~s, ~s, instead."
	 (unit-index.slot-name unit-index) (-> unit-instance)
	 (unit-index.datatype unit-index) (type-of the-slot) the-slot)

      (if (composite-index-structure-p index-structure)
	  (mapcar #'canonical-value the-slot)
	  (canonical-value the-slot)))))


(defun update-canonical-index-and-move (unit-type slot-name unit-instance)

  "Updates the canonical indexes for unit-instance and adjusts
   its placement on its spaces to reflect new values for some
   of its indexes."

  (let* ((description (get-unit-description unit-type))
	 (unit-indexes (unit.d-indexes description))
         (new-index-data (basic-unit.%%indexes%% unit-instance))
	 (old-index-data (copy-list new-index-data))
	 (space-instances (basic-unit.%%space-instances%% unit-instance))
         (changed-indexes nil)
	 canonical-index)

    (dolist (unit-index unit-indexes)
      (when (string= slot-name (unit-index.slot-name unit-index))
        (push (unit-index.name unit-index) changed-indexes)
	(setq canonical-index (compute-canonical-index unit-index unit-instance))
	(setf (elt new-index-data (unit-index.offset unit-index))
	      canonical-index)))

    (dolist (space-instance space-instances)
      (move-unit-on-space-instance
        unit-instance space-instance changed-indexes old-index-data))))


(defun check-canonical-index (unit-type slot-name unit-instance)

  "Checks that the indexing information hasn't changed..."

  (let* ((description (get-unit-description unit-type))
	 (unit-indexes (unit.d-indexes description))
	 (index-data (basic-unit.%%indexes%% unit-instance))
	 canonical-index)
    (dolist (unit-index unit-indexes)
      (when (string= slot-name (unit-index.slot-name unit-index))
	(setq canonical-index (compute-canonical-index unit-index unit-instance))
	(error-unless (equal canonical-index
			     (elt index-data (unit-index.offset unit-index)))
	   "The new value of the slot ~s in ~s has changed the~@
            indexing data used to locate it on its space(s)."
	   slot-name (-> unit-instance))))))



(defun DEFAULT-UNIT-NAME-FUNCTION (unit)

  "DEFAULT-UNIT-NAME-FUNCTION unit

   Constructs a name based on the type of the unit. Returns a
   string of the form <unit-type>-<number> (e.g., \"HYP-36\")."

  (flet ((get-count (unit-type)
           (unless (get unit-type 'unit-generation-count)
             (setf (get unit-type 'unit-generation-count) 0))
           (incf (get unit-type 'unit-generation-count))))
    (let ((type (unit-type-of unit)))
      (format nil "~a-~d" (string type) (get-count type)))))


(defun FIND-UNIT-BY-NAME (name unit-type)

  "FIND-UNIT-BY-NAME name unit-type

   Find the unit named NAME.  UNIT-TYPE is name of the unit."

  (assert (unit-type-p unit-type)
	  (unit-type)
	  "~s is not a unit type." unit-type)
  (error-when (anonymous-unit-type-p unit-type)
     "Units of type ~s are unnamed." unit-type)
  (let ((table (unit-info.hash-table (get-unit-info unit-type))))
    ;; Only return one value.
    (values (gethash name table))))


(defun PRINT-UNIT-HASH-TABLE (unit-types)

  "PRINT-UNIT-HASH-TABLE unit-types

   Prints out the contents of the hash table for the unit types given by
   the argument `unit-types', which may be a symbol or a list."
  
  (dolist-or-atom (unit-type unit-types)
    (if (anonymous-unit-type-p unit-type)
        (format t "~2%Instances of ~s are anonymous." unit-type)
        (let ((hash-table (unit-info.hash-table (get-unit-info unit-type))))
          (format t "~2%Contents of the table for ~s:~%" unit-type)
          (maphash #'(lambda (key values)
                       (format t "~% key: ~s ~35t value: ~s" key values))
                   hash-table)))))


(defun CLEAR-UNIT-HASH-TABLES (&optional (units *all-units*))
  (dolist (unit-type units)
    (unless (anonymous-unit-type-p unit-type)
      (let ((info (get-unit-info unit-type)))
        (clrhash (unit-info.hash-table info))
        (clrhash (unit-info.dummy-hash-table info))))))


;;; --------------------------------------------------------------------------
;;;                                End of File
;;; --------------------------------------------------------------------------
