/* 
** innergraph_w.c: internal graph widget routines
**
** Written by:	Upinder S. Bhalla 	(bhalla@aurel.caltech.edu)
**		John Uhley 		(uhley@tybalt.caltech.edu)
**
** Copyright 1988 by the California Institute of Technology
**
** Permission to use, copy, modify, and distribute this
** software and its documentation for any purpose and without
** fee is hereby granted, provided that the above copyright
** notice appear in all copies and that both that copyright
** notice and this permission notice appear in supporting
** documentation, and that the name of the California Institute
** of Technology not be used in advertising or publicity pertaining 
** to distribution of the software without specific, written prior 
** permission.  The California Institute of Technology makes no 
** representations about the suitability of this software for any purpose.
** It is provided "as is" without express or implied warranty.
** 
** The Xgraph Widget library was written by Upinder S. Bhalla and
** John Uhley.  Portions of the labeling routines were taken from
** David Harrison's xgraph (X10V4) program with his permission.
**
** The X Window system is copyright 1985, 1986, 1987 By the
** Massachusetts Institute of Technology, without whom none of this
** code would have been possible.
**
*/
#include "widg_ext.h"
#ifdef X11R3
#include <X11/LabelP.h>
#else
#include <X11/Xaw/LabelP.h>
#endif X11R3
#include "LayoutP.h"
#include "innergraph_wP.h"

typedef struct gcoord_type {
	float x,y;
} gCoord;

extern char 	*g_copy();

/*
** some default ranges for our graph 
**
*/

static XtResource resources[] = {
	{XtNxmin, XtCGraphXmin, XtRString, sizeof(char * ),
		XtOffset(InnerGraphWidget, graph.xmin_str), XtRString,
			"0.00"
	},
	{XtNymin, XtCGraphYmin, XtRString, sizeof(char * ),
		XtOffset(InnerGraphWidget, graph.ymin_str), XtRString,
			"0.00"
	},
	{XtNxmax, XtCGraphXmax, XtRString, sizeof(char * ),
		XtOffset(InnerGraphWidget, graph.xmax_str), XtRString,
			"100.0"
	},
	{XtNymax, XtCGraphYmax, XtRString, sizeof(char * ),
		XtOffset(InnerGraphWidget, graph.ymax_str), XtRString,
			"100.0"
	},
};

extern char *GetArray();
static void FreeArray();
static void SetArray();
static void Destroy();
static void inner_graph_initialize();
static Boolean innergraph_set_values();
void inner_graph_display();
void inner_graph_rescale();
void inner_graph_reset();
static void InnerGraphPrint();

static char translations[] = "Ctrl<Key>P:InnerGraphPrint()";

static XtActionsRec actionsList[] = {
		{"InnerGraphPrint",  InnerGraphPrint}, 
};
InnerGraphClassRec innergraphClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"InnerGraph",
    /* widget_size		*/	sizeof(InnerGraphRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	inner_graph_initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actionsList,
    /* num_actions		*/	XtNumber(actionsList),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	inner_graph_rescale,
    /* expose			*/	inner_graph_display,
    /* set_values		*/	innergraph_set_values,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
};

WidgetClass innergraphWidgetClass = (WidgetClass)&innergraphClassRec;

/*
** inner_graph_initialize
**
**	Make sure that the requested graph is reasonably big
**
*/
static void 
inner_graph_initialize(request,new)
	InnerGraphWidget request,new;
{
	new->graph.graphInfo = NULL;
	if (new->core.width < MIN_GRAPH_WIDTH)
		new->core.width = MIN_GRAPH_WIDTH;
	if (new->core.height < MIN_GRAPH_HEIGHT)
		new->core.height = MIN_GRAPH_HEIGHT;
	sscanf(new->graph.xmin_str, "%f", &(new->graph.xmin));
	sscanf(new->graph.xmax_str, "%f", &(new->graph.xmax));
	sscanf(new->graph.ymin_str, "%f", &(new->graph.ymin));
	sscanf(new->graph.ymax_str, "%f", &(new->graph.ymax));
}

/*
** Destroy
**
**	Standard death of a widget
**
*/
static void Destroy(w)
		 Widget w;
{
	if (XtHasCallbacks(w, XtNcallback) == XtCallbackHasSome)
		XtRemoveAllCallbacks(w, XtNcallback);
}

/*
** GetArray:
**
** 	Data management routines for graph points and the like...
**
*/
char *
GetArray(rootal,n)
        ArrayLink       **rootal;
        int                     n;
{
        ArrayLink       *al;

        if (!(al = *rootal)) return(NULL);
        while (n >= al->max){
                if (!(al->next)) return(NULL);
                al = al->next;
        }
        while (n < al->min){
                if (!(al->last)) return(NULL);
                al = al->last;
        }
        if (n >= al->min && n < al->max) {
                /* *rootal = al; */
                return(al->val[n - al->min]);
        }
}

/*
** SetArray:
**
**	Set array values
**
*/
static void
SetArray(rootal,valsize,value,n)
        ArrayLink       **rootal;
        int             valsize;
        char            *value;
        int             n;
{
        ArrayLink       *al;
        int             i;
        int             vsize = valsize * sizeof(char);

        if (!(al = *rootal)) {
                al = *rootal = (ArrayLink *) calloc(1,sizeof(ArrayLink));
                for (i = 0 ; i < BLOCKSIZE ; i++)
                        al->val[i] = (char *) calloc(1,vsize);
                al->min = 0;
                al->max = BLOCKSIZE;
                al->last = NULL;
        }

        while (n >= al->max) {
                if (!(al->next)) {
                        al->next = (ArrayLink *) calloc(1,sizeof(ArrayLink));
                        for (i = 0 ; i < BLOCKSIZE ; i++)
                                al->next->val[i] = (char *) calloc(1,vsize);
                        al->next->last = al;
                        al->next->min = al->max;
                        al->next->max = al->max + BLOCKSIZE;
                }
                al = al->next;
        }
        while (n < al->min){
                if (!(al->last)) return;
                al = al->last;
        }
        if (n >= al->min && n < al->max) {
                /* 	*rootal = al; */
                bcopy(value,al->val[n - al->min],valsize);
        }
}

#ifdef HOSED
/*
** FreeArray:
**
**	When we're done with an array return some memory
**
*/
static void
FreeArray(rootal)
        ArrayLink       *rootal;
{
        int             i;
        ArrayLink       *al;

	al = rootal;

	if (al == NULL)
		return;
	for (;;) {
		al = rootal;
		while (al->next != NULL) 
			al = al->next;
		for (i = 0; i < BLOCKSIZE; i++) {
			if (al->val[i] != NULL)
				free(al->val[i]);
		}
		if (al != NULL)
			free(al);
		if (al == rootal)
			break;
	}
	free(rootal);
}
#endif
/*
** FreeArray:
** Matts modification 4/20/89
** and PLEASE John dont trash it this time!
**
*/
static void
FreeArray(rootal)
        ArrayLink       *rootal;
{
        int             i;
        ArrayLink       *al;

	for(al=rootal;al;al=al->next){
		for (i = 0; i < BLOCKSIZE; i++) {
			if (al->val[i] != NULL)
				free(al->val[i]);
		}
		free(al);
	}
}
		
/*
** innergraph_set_values:
**
**	Kludge to set values of internal xmin,xmax,ymin,ymax floats from
**	external strings set by user or the "graph_w" widget.
**
*/
static Boolean innergraph_set_values(current, request, new)
	InnerGraphWidget	current;
	InnerGraphWidget	request;
	InnerGraphWidget	new;
{
	if (strcmp(current->graph.xmin_str, request->graph.xmin_str) != 0 ||
	   strcmp(current->graph.xmax_str, request->graph.xmax_str) != 0 ||
	   strcmp(current->graph.ymin_str, request->graph.ymin_str) != 0 ||
	   strcmp(current->graph.ymax_str, request->graph.ymax_str) != 0 ) {
		sscanf(new->graph.xmin_str, "%f", &(new->graph.xmin));
		sscanf(new->graph.xmax_str, "%f", &(new->graph.xmax));
		sscanf(new->graph.ymin_str, "%f", &(new->graph.ymin));
		sscanf(new->graph.ymax_str, "%f", &(new->graph.ymax));
		inner_graph_rescale(new);
	}
	return(FALSE);
}

/*
** inner_graph_display
**
**	Redraw the current graph curves.
**
*/
void inner_graph_display(gw,event,region)
	InnerGraphWidget gw;
	XEvent	*event;
	Region	region;
{
	GraphInfo	*info, *start_info;
	GC		gc;
	XGCValues	values;	
	char		*GetArray();
	XPoint		*pts,*lastpts;
	Display		*disp;
	Window		window;
	int		npts;
	int		i;
	static XPoint	temppts[BLOCKSIZE];
	ArrayLink	*al;

	gc = XtGetGC(gw,GCForeground, &values);

	info = gw->graph.graphInfo;
	start_info = info;
	disp = XtDisplay(gw);
	window = XtWindow(gw);
	if (window == NULL)
		return;

	XSetLineAttributes(disp,gc,0,LineSolid,CapProjecting,JoinBevel);

	/*
	** if the widget is smaller than our defined mininum 
	** fill the graph in with BLUE...  It's a kludge, I know...
	**
	*/
	if (gw->core.width < MIN_GRAPH_WIDTH || gw->core.height < 
		MIN_GRAPH_HEIGHT) {
		XSetForeground(disp,gc,graph_fg_error);
		XFillRectangle(disp, window, gc, 0, 0,
			gw->core.width, gw->core.height);
		XSetForeground(disp,gc,graph_fg);
		return;
	}
	
	while (info != NULL) {
        if (XDisplayCells(disp,XDefaultScreen(disp)) > 254)
			XSetForeground(disp,gc,name_to_color(info->color));
        else
            XSetForeground(disp,gc,
                XBlackPixel(disp,XDefaultScreen(disp)));

		lastpts = pts = (XPoint *)GetArray(&(info->pts),0);
		al = info->pts;
		if (info->npts < 2) {
			XPSDrawPoint(disp, window, gc, pts->x, pts->y);
			info = info->next;
			continue;
		}
		/*
		** A loop hacked for speed which requires knowledge of the
		** ArrayLink structure and operation (as in GetArray)
		*/
		while (al && info->npts >= al->min) {
			pts = (XPoint *)(al->val[0]);
			hackXPSDrawLine(disp,window,gc,pts->x,pts->y,
				lastpts->x,lastpts->y);
			npts = (info->npts >= al->max) ? 
				BLOCKSIZE - 1 : info->npts - al->min;
			for (i = 0 ; i < npts; i++)  {
				temppts[i] = *(XPoint *)(al->val[i]);
			}
			hackXPSDrawLines(disp,window,gc,
				temppts,npts,
				 CoordModeOrigin);
			lastpts = (XPoint *)al->val[npts];
			al = al->next;
		}
		info = info->next;
	}
	XSetForeground(disp,gc,graph_fg);
}

/*
** inner_graph_rescale:	Called when the widget is resized.  Reconverts
**			raw data to the new scaled data.
**
*/
void inner_graph_rescale(gw) 
	InnerGraphWidget	gw;
{
	GraphInfo	*gi; 
	gCoord		*data;
	char		*GetArray();
	float		xrange, yrange;
	float		xmin, ymin;
	int		sx, sy;
	int		lastx, lasty; 
	int		i, npts;
	XPoint		pts;

	gi = gw->graph.graphInfo;

	/* 
	** get a handle on the user defined range that the data
	** falls into.  If it's too small add 1.0 to make it reasonable.
	**
	*/
	
	if (gw->graph.xmax < gw->graph.xmin) {
		fprintf(stderr,"graph '%s': illegal xrange xmax %f < xmin %f\n",
			gw->core.name, gw->graph.xmax, gw->graph.xmin);
		fprintf(stderr,"graph '%s': inverting xrange values\n");
		xmin = gw->graph.xmax;
		gw->graph.xmax = gw->graph.xmin;
		gw->graph.xmin = xmin;
	}
	if (gw->graph.ymax < gw->graph.ymin) {
		fprintf(stderr,"graph '%s': illegal yrange ymax %f < ymin %f\n",
			gw->core.name, gw->graph.ymax, gw->graph.ymin);
		fprintf(stderr,"graph '%s': inverting yrange values\n");
		ymin = gw->graph.ymax;
		gw->graph.ymax = gw->graph.ymin;
		gw->graph.ymin = ymin;
	}

	xrange = gw->graph.xmax - gw->graph.xmin;
	yrange = gw->graph.ymax - gw->graph.ymin;

	if (xrange < EPSILON) {
		fprintf(stderr,"graph '%s': xrange(%e) was < %e, resetting to 1.0\n",
			gw->core.name,xrange, EPSILON);
		xrange = 1.0;
		gw->graph.xmax = gw->graph.xmin + 1.0;
	}
	if (yrange < EPSILON) {
		fprintf(stderr,"in graph '%s': yrange(%e) was < %e, resetting to 1.0\n",
			gw->core.name,yrange, EPSILON);
		yrange = 1.0;
		gw->graph.ymax = gw->graph.ymin + 1.0;
	}


	while (gi != NULL) {
		lastx = lasty = -1;
		npts = 0;
		for (i = 0 ; i < gi->datasize ; i++) { 
			data = ((gCoord *)(GetArray(&(gi->data),i)));

			sx = gw->core.width * (data->x - gw->graph.xmin) /
				 xrange;
			sy = gw->core.height - 
				(gw->core.height * (data->y - gw->graph.ymin) /
				 yrange);

			if (lastx != sx || lasty != sy) {
				lastx = pts.x = sx;
				lasty = pts.y = sy;
				SetArray(&(gi->pts),sizeof(XPoint),&pts,npts);
				npts++;
			}
		}
		gi->npts = npts;
		gi = gi->next;
	}
	if (gw->core.visible && XtWindow(gw) != NULL)  
		XClearArea(XtDisplay(gw),XtWindow(gw),0,0,0,0,True);
}

/*
** graph_reset
**
**	Clear out the points stored with the specified graph
**	and clear the displayed graph window.
**
*/
void inner_graph_reset(gw)
	InnerGraphWidget	gw;
{
	GraphInfo	*gi;

	gi = gw->graph.graphInfo;

	if (gi != NULL) {
		while (gi != NULL) {
			gi->datasize = 0;
			gi->npts = 0;
			gi->name = NULL;
			FreeArray(gi->data);
			FreeArray(gi->pts);
			gi->data = NULL;
			gi->pts = NULL;
			free(gi);
			gi = gi->next;
		}
		gw->graph.graphInfo = NULL;
	}
	if (XtWindow(gw) == NULL)
		return;
	if (gw->core.visible)  {
		XClearArea(XtDisplay(gw),XtWindow(gw),0,0,0,0,True);
		XFlush(XtDisplay(gw));
	}
}


/*
** inner_graph_add_points:
**
**	Add points to the graph
**
** Returns:	0 if pts were added to an old curve
**		1 if they were added to a new curve.
*/
inner_graph_add_points(gw,plotname,colorname,x,y)
	InnerGraphWidget	gw;
	char		*plotname;
	char		*colorname;
	float		x,y;
{
	GraphInfo	*gi, *firstgi;
	gCoord		data;
	int		n,npts;
	int		lastcolor = 1,colorstep = 0;
	char		*lastcolorname;
	Position	sx,sy;
	Position	lastx,lasty;
	Position	screenx,screeny;
	GC		gc;
	XGCValues	values;
	char		*GetArray();
	XPoint		pts;
	int			new_curve;
	static char		*lastname;
	static GraphInfo	*lastgi = NULL;
	Display		*display;

	display = XtDisplay(gw);
	firstgi = gi = gw->graph.graphInfo;
	if (firstgi == NULL) {
		firstgi = gi = (GraphInfo *) calloc(1,sizeof(GraphInfo));
		gw->graph.graphInfo = gi;
		gi->name = NULL;
	}
	new_curve = 1;
/*
	if (colorname == NULL) {
		printf("COLORNAME Was null\n");
	} else {
		printf("COLORNAME Was %s\n",colorname);
	}
*/
	while (True) {
		if (gi->name == NULL) {
			gi->name = (char *) malloc((1 + strlen(plotname))
				* sizeof(char));
			strcpy(gi->name,plotname);
			if (colorname == NULL) {
				colorname = XGetDefault(display, "genesis",
					"graphForeground");
			}
			if (colorname != NULL) 
				gi->color = (char *)g_copy(colorname);
			else
				gi->color = (char *)g_copy("black");
			gi->datasize = 0;
			break;
		} else {
			if (strcmp(gi->name,plotname) == 0) {
				if (colorname != NULL) 
					if (strcmp(gi->color, colorname)!=0)
						gi->color = (char *)
							g_copy(colorname);
				new_curve = 0;
				break;
			}
			if (gi->next == NULL) {
				gi->next = 
				(GraphInfo *)calloc(1,sizeof(GraphInfo));
				gi->next->name = NULL;
			}
			gi = gi->next;
		}
	}
	lastgi = gi;

	n = gi->datasize;
	data.x = x; 
	data.y = y; 
	SetArray(&(gi->data),sizeof(gCoord),&data,n);

	(gi->datasize)++;

	/*
	** Store the scaled data into the graph -- need to insert
	** code here to deal with axis-scaled window size 
	**
	*/
	screenx = gw->core.width;
	screeny = gw->core.height;
	sx = (Position)((float)(screenx) * (x - gw->graph.xmin) /
		(gw->graph.xmax - gw->graph.xmin)) ;
	sy = (Position)((float)(screeny) * 
		(1.0 - (y - gw->graph.ymin)/
		(gw->graph.ymax - gw->graph.ymin))) ;

	if (n == 0) {
		npts = 0;
		pts.x = sx;
		pts.y = sy;
		SetArray(&(gi->pts),sizeof(XPoint),&pts,npts);
		gi->npts = 1;
	} else {
		npts = gi->npts;
		pts = *(XPoint *)GetArray(&(gi->pts),npts - 1);
		lastx = pts.x;
		lasty = pts.y;
		if (lastx != sx || lasty != sy) {
			pts.x = sx;
			pts.y = sy;
			SetArray(&(gi->pts),sizeof(XPoint),&pts,npts);
			gc = XtGetGC(gw,GCForeground,&values);

        	if (XDisplayCells(display,XDefaultScreen(display)) > 254)
            	XSetForeground(display,gc,name_to_color(gi->color));
        	else
            	XSetForeground(display,gc,
                	XBlackPixel(display,XDefaultScreen(display)));

			hackXPSDrawLine(display,XtWindow(gw),gc,
				lastx, lasty, sx, sy);
			XFlush(display);

			gi->npts = npts + 1;
		}
	}
	return(new_curve);
}


static void InnerGraphPrint(w, event)
	InnerGraphWidget	w;
	XEvent	*event;
{
	Widget parent = XtParent(w);
	if (parent != NULL) 
		GraphPrint(parent);
}
	
/*
** These are kludges to deal with the correct x,y offset for
** plotting to a poscript printer.
**
*/

hackXPSDrawLine(disp,window,gc,x1,y1,x2,y2)
Display	*disp;
Window	window;
GC	gc;
Position	x1,y1,x2,y2;
{
	InnerGraphWidget	inner;
	if (output_flag != XOUT) {
		inner = (InnerGraphWidget)XtWindowToWidget(disp, window);
		XPSDrawLine(disp,window, gc, x1 + inner->core.x, 
			y1 + inner->core.y, x2 + inner->core.x, 
			y2 + inner->core.y);
	} else 
		XPSDrawLine(disp,window, gc, x1, y1, x2, y2);
}
hackXPSDrawLines(disp,window,gc, temppts,npts, mode)
Display	*disp;
Window	window;
GC	gc;
XPoint	*temppts;
int	npts;
int	mode;
{
	int i;
	InnerGraphWidget	inner;
	if (output_flag != XOUT) {
		inner = (InnerGraphWidget)XtWindowToWidget(disp, window);
		for (i = 0; i < npts; i++) {
			temppts[i].x = temppts[i].x + inner->core.x;
			temppts[i].y = temppts[i].y + inner->core.y;
		}
	}
	XPSDrawLines(disp,window,gc, temppts,npts, mode);
}
