static char rcsid[] = "$Id: new_interp.c,v 1.1.1.1 1993/07/09 01:14:34 dhb Exp $";

/*
** $Log: new_interp.c,v $
 * Revision 1.1.1.1  1993/07/09  01:14:34  dhb
 * (1) cosmetic changes to call the second argument "gate" instead of "table"
 * (2) change in the way the index "j" is computed.  This makes ox and sx
 *     behave in a way which is consistent with setting these fields using
 *     scale_table.  (However, as before, ScaleTabchan modifies table values
 *     instead of changing xmin and xmax.)
 * (3) Return with error for sx <= 0.  (sx == 0 would cause a divide by zero.)
 * (4) Fixed bug which kept it from working with tabgates.  In order to use the
 *     same code for scaling, the tabgate "beta" table is temporarily changed
 *     to alpha + beta.  As before, it is changed back after scaling.
 *
 * Revision 1.1  1992/11/14  00:37:20  dhb
 * Initial revision
 *
*/

#include "olf_ext.h"
#define DEFAULT_XDIVS 3000
#define DEFAULT_XMIN -0.1
#define DEFAULT_XMAX 0.05
#define SETUP_ALPHA 0
#define SETUP_TAU 1

/* 
** Following calc_modes for interpolation : 
**	interpolation 
**		1 Linear (default, simpler to use and backwards compatible)
**		2 no interpolation. (more efficient)
**
**	extrapolation always uses
**		1 Truncation 
**
**	Special mode : Fixed value, which returns the y offset of the table
**
*/




/*
** The table consists of xdivs + 1 points. Table[0] and Table[xdivs]
** are used as truncation values.
*/

double TabInterp(ip,x)
	Interpol *ip;
	double x;
{
	register double lo;
	register int ilo;
	/*
	register double *table;
	*/

	switch(ip->calc_mode) {
		case NO_INTERP :
			/*
			if(table=ip->table) {
				ilo = ((x - ip->xmin)*ip->invdx);
				if (ilo<0) return(table[0]);
				if (ilo>ip->xdivs) return(table[ip->xdivs]);
				return(table[ilo]);
			} else {
				return(0.0);
			}
			*/
			if(ip->table) {
				ilo = ((x - ip->xmin)*ip->invdx);
				if (ilo<0) return(ip->table[0]);
				if (ilo>ip->xdivs) return(ip->table[ip->xdivs]);
				return(ip->table[ilo]);
			} else {
				return(0.0);
			}
			break;
		case LIN_INTERP :
			if(ip->table) { 
				ilo = lo = (x - ip->xmin)*ip->invdx;
				if (ilo < 0){
					return(ip->table[0]);
				} else if (ilo > ip->xdivs) {
					return(ip->table[ip->xdivs]);
				} else {
					x=ip->table[ilo];
					return((lo-(double)ilo)*(ip->table[ilo+1]-x)+x);
				}
			} else {
				return(0.0);
			}
			/*
			if((table = ip->table) == NULL) return(0.0);
			ilo = lo = (x - ip->xmin)*ip->invdx;
			if (ilo < 0){
				return(table[0]);
			} else if (ilo > ip->xdivs) {
				return(table[ip->xdivs]);
			} else {
				return((lo - (double)ilo)*(table[ilo+1] - table[ilo]) + table[ilo]);
			}
			*/
			break;
		case FIXED :
			return(ip->oy);
			break;
	}
}

Interpol *create_interpol(xdivs,xmin,xmax)
	int xdivs;
	float xmin,xmax;
{
	Interpol *ip;

	ip = (Interpol *) calloc(1,sizeof(Interpol));

	ip->xdivs = xdivs;
	ip->xmin = xmin;
	ip->xmax = xmax;
	ip->table = (double *)calloc(xdivs + 1,sizeof(double));
	ip->allocated = 1;
	ip->calc_mode = LIN_INTERP;
	ip->sx = ip->sy = 1.0;
	ip->ox = ip->oy = 0.0;

	ip->dx = (ip->xmax - ip->xmin) / (double)(ip->xdivs);
	ip->invdx=1.0/ip->dx;

	return(ip);
}

/*
** Special function for filling table for nointerp :
** fill_table(interpol, npts, mode)
** where interpol gives the existing interpol pointer, and mode
** specifies linear, cubic spline or B_spline fitting
** The function takes the existing table in the interpol and uses it
** to create an expanded table with npts data values, using the
** specified interpolation method. The original table is then tossed.
*/

int fill_table(ip,npts,fill_mode)
	Interpol	*ip;
	int			npts;
	short		fill_mode;
{
	double	*old_tab;
	double	*new_tab;
	double	lo,dlo;
	double	t,t1,t2,t2a,t3,t3a,t4;
	double	nc1,nc2,nc3,nc4;
	double	nsa,nsb;
	int		i,j;
	int		xxs;
	int		xdivs;
	int		bad_flag=0;

	if ((old_tab = ip->table) == NULL) {
		printf("Error : NULL table. Filling aborted\n");
		return(0);
	}
	if (npts < 3) {
		printf("Error : %d divisions for table is invalid. Filling aborted\n",npts);
		return(0);
	}

	new_tab = (double *)calloc(npts + 1, sizeof(double));
	xdivs = ip->xdivs;

	switch (fill_mode) {
		case LINEAR_FILL :
			dlo = (double)xdivs / (double)npts ;
			/* i is index for old array, j for new array */
			for (lo=0.0,i=0,j=0;j<=npts;lo+=dlo,i=(int)lo,j++) {
				if (i < xdivs)
					new_tab[j] = (lo - (double)i)*
						(old_tab[i+1] - old_tab[i]) + old_tab[i];
				else
					new_tab[j] = old_tab[xdivs];
			}
			break;
		case C_SPLINE_FILL :
			printf("C_SPLINE_FILL not yet implemented\n");
			bad_flag = 1;
			break;
		case B_SPLINE_FILL :
			xxs=xdivs-2;nsa=1.0/6.0;nsb=2.0/3.0;
			dlo = (double)xdivs / (double)npts ;
			/* i is index for old array, j for new array */
			/* filling up newtab till first element in oldtab */
			for (lo = 0.0,j=0,i=0;i<=1;lo+=dlo,i=(int)lo,j++) {
				new_tab[j] = (lo - (double)i)*
					(old_tab[i+1] - old_tab[i]) + old_tab[i];
			}
			/* lo,i, and j carry on from where they were last time.*/
			for(;i<=xxs;lo+=dlo,i=(int)lo,j++){
				t = lo - (double)i;
				t1=t/2.0;
				t2=t*t;
				t2a=t2/2.0;
				t3=t2*t;
				t3a=t3/2.0;
				nc1= -nsa*t3+t2a-t1+nsa;
				nc2=t3a-t2+nsb;
				nc3= -t3a+t2a+t1+nsa;
				nc4=nsa*t3;
				new_tab[j]=nc1*old_tab[i-1] + nc2*old_tab[i] +
					nc3*old_tab[i+1] + nc4*old_tab[i+2];
			}
			/* lo, i, and j carry on from where they were last time.*/
			for (; j <= npts ; lo += dlo,i=(int)lo , j++) {
				if (i < xdivs)
					new_tab[j] = (lo - (double)i)*
						(old_tab[i+1] - old_tab[i]) + old_tab[i];
				else
					new_tab[j] = old_tab[xdivs];
			}
			break;
		default :
			bad_flag = 1;
			break;
	}

	if (bad_flag) {
		free(new_tab);
		return(0);
	}
	ip->table = new_tab;
	ip->xdivs = npts;
	ip->dx = (ip->xmax - ip->xmin)/(double)npts;
	ip->invdx = 1.0/ip->dx;
	free(old_tab);
	return(1);
}

/*
** Special function for tweaking table :
** scale_table(interpol,sx,sy,ox,oy)
** where sx, sy are magnification factors on X and Y, and ox,oy are
** offsets. The new values are then installed in the interpol struct
*/

scale_table(ip,field,value)
	Interpol	*ip;
	char		*field;
	char		*value;
{
	double	*table;
	int		i;
	int		xdivs;
	double	osy,ooy;
	double	sx,sy,ox,oy;

	sx = ip->sx; sy = ip->sy;
	ox = ip->ox; oy = ip->oy;

	if (strcmp(field,"sx") == 0)
		sscanf(value,"%lf",&sx);
	else if (strcmp(field,"sy") == 0)
		sscanf(value,"%lf",&sy);
	else if (strcmp(field,"ox") == 0)
		sscanf(value,"%lf",&ox);
	else if (strcmp(field,"oy") == 0)
		sscanf(value,"%lf",&oy);
	else 
		return;

	xdivs = ip->xdivs;
	table = ip->table;
	osy = ip->sy;
	ooy = ip->oy;
	if (oy != ooy || osy != sy) {
		osy = sy / ip->sy;
		for (i = 0 ; i <= xdivs ; i++) {
			table[i] = oy + (table[i] - ooy) * osy;
		}
	}
	ip->xmin = ox + (ip->xmin - ip->ox) * sx / ip->sx;
	ip->xmax = ox + (ip->xmax - ip->ox) * sx / ip->sx;
        ip->dx = (ip->xmax - ip->xmin) / (double)(ip->xdivs);
        ip->invdx = 1.0/ip->dx;
	ip->ox = ox;
	ip->oy = oy;
	ip->sx = sx;
	ip->sy = sy;
}

/*
** Special function for finding min/max values in table, between
** lox and hix.
** get_table_min(table,lox,hix)
** get_table_max(table,lox,hix)
**
** Special function for finding roots (x value for given y)
*/

#ifdef MAYBE

free_interpol(ip)
	Interpol	*ip;
{
	if (ip->allocated)
		free(ip->table);
	free(ip);
}

reset_interpol(ip)
	Interpol	*ip;
{
	switch (ip->extrapolation_mode) {
		case TRUNC_EXTRAP :
			ip->table[0] = ip->table[1];
			ip->table[xdivs + 1] = ip->table[xdivs];
		break;
		case LINEAR_EXTRAP :
		default : /* linear extrapolation */
			break;
	}
}

do_create_interpol(argc,argv)
	int argc;
	char **argv;
{
	tab_gate_struct	*elm;

	if (argc < 6) {
		printf("usage : %s element interp xdivs xmin xmax\n",argv[0]);
		printf("Warning : interp must be an interpolation structure\n");
		printf("in the element.\n");
		return;
	}
	elm = (tab_gate_struct *)(GetElement(argv[1]));
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	elm
}
#endif

#ifdef OLD
double TabInterp(ip,x)
	Interpol *ip;
	double x;
{
	register double lo;
	register int L;
	register double *table;

	if((table = ip->table) == NULL) return(0);
	lo = (x - ip->xmin)/ip->dx;
	L = (int)lo;
	if (L < 0){
		return(table[0]);
	} else 
	if (L >= ip->xdivs -1) {
		return(table[ip->xdivs - 1]);
	} else {
		return((lo - (double)L)*(table[L+1] - table[L]) + table[L]);
	}
}
#endif


/* Set up a tabulated channel from alpha-beta rate constants */
SetupAlpha(argc,argv)
	int argc;
	char	**argv;
{
	setup_tab_values(argc,argv,SETUP_ALPHA);
}

/* Set up a tabulated channel from tau-minf rate constants */
SetupTau(argc,argv)
	int argc;
	char	**argv;
{
	setup_tab_values(argc,argv,SETUP_TAU);
}

/* Set up a tabulated channel from alpha-beta rate constants */
setup_tab_values(argc,argv,mode)
	int argc;
	char	**argv;
	int		mode;
{
	float args[15];
	Element	*elm;
	Interpol	*A,*B;
	float temp,x,dx,temp2;
	int i;
	int	tab_gate_flag=0;
	Interpol *create_interpol();

	A=B=NULL;
	if (argc!=13 && argc!= 16) {
		printf(
			"usage: %s chan gate AA AB AC AD AF BA BB BC BD BF [xdivs xmin xmax]\n",
			argv[0]);
		return;
	}
	elm = GetElement(argv[1]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	for(i=3;i<argc;i++) args[i-3]=Atof(argv[i]);
	if (argc==13) {
		args[10]=DEFAULT_XDIVS;
		args[11]=DEFAULT_XMIN;
		args[12]=DEFAULT_XMAX;
	}
	/* for(i=3;i<16;i++) printf("%d %f\n",i-3,args[i-3]); */

	if (strcmp(elm->object->name,"tabgate")==0) {
		A=((struct tab_gate_type *)elm)->alpha=
			create_interpol((int)args[10],args[11],args[12]);
		((struct tab_gate_type *)elm)->alpha_alloced=1;
		B=((struct tab_gate_type *)elm)->beta=
			create_interpol((int)args[10],args[11],args[12]);
		((struct tab_gate_type *)elm)->beta_alloced=1;
		tab_gate_flag=1;
	}
	if (strcmp(elm->object->name,"tabchannel")==0) {
		if (strcmp(argv[2],"X")==0) {
			A=((struct tab_channel_type *)elm)->X_A=
				create_interpol((int)args[10],args[11],args[12]);
			B=((struct tab_channel_type *)elm)->X_B=
				create_interpol((int)args[10],args[11],args[12]);
			((struct tab_channel_type *)elm)->X_alloced=1;
		} else if (strcmp(argv[2],"Y")==0) {
			A=((struct tab_channel_type *)elm)->Y_A=
				create_interpol((int)args[10],args[11],args[12]);
			B=((struct tab_channel_type *)elm)->Y_B=
				create_interpol((int)args[10],args[11],args[12]);
			((struct tab_channel_type *)elm)->Y_alloced=1;
		} else if (strcmp(argv[2],"Z")==0) {
			A=((struct tab_channel_type *)elm)->Z_A=
				create_interpol((int)args[10],args[11],args[12]);
			B=((struct tab_channel_type *)elm)->Z_B=
				create_interpol((int)args[10],args[11],args[12]);
			((struct tab_channel_type *)elm)->Z_alloced=1;
		} else {
			printf("Gate type %s not known in %s \n",argv[2],argv[0]);
			return;
		}
	}
	if (!A || !B) {
		printf("Error: Element %s should be a tabgate or tabchannel\n",
			argv[1]);
		return;
	}
	dx=(args[12]-args[11])/args[10];
	/* compensating for rounding errors, we take middle sample point */
	for(x=args[11]+dx/2.0,i=0;i<=args[10];i++,x+=dx) {
		/* Assorted ugly NaN cases */
		if (args[4]==0.0) {
			temp=A->table[i]=0.0;
		} else {
			temp2=args[2]+exp((x+args[3])/args[4]);
			if (temp2==0.0)
				temp=A->table[i]=A->table[i-1];
			else
				temp=A->table[i]= (args[0]+args[1]*x)/temp2;
		}

		if (args[9]==0.0) {
			B->table[i]=0.0;
		} else {
			temp2=args[7]+exp((x+args[8])/args[9]);
			if (temp2==0.0)
				B->table[i]=B->table[i-1];
			else
				B->table[i]= (args[5]+args[6]*x)/temp2;
		}
		if (!tab_gate_flag && mode==SETUP_ALPHA)
			B->table[i]+=temp;
	}
	if (mode==SETUP_TAU) {
		for(i=0;i<=args[10];i++) {
			temp=A->table[i];
			temp2=B->table[i];
			if (temp == 0.0) {
				A->table[i]=0;
				B->table[i]=0;
			} else {
				A->table[i]=temp2/temp;
				if (!tab_gate_flag)
					B->table[i]=1.0/temp;
				else 
					B->table[i]=temp-temp2;
			}
		}
	}
}

TweakAlpha(argc,argv)
	int argc;
	char	**argv;
{
	tweak_tab_values(argc,argv,SETUP_ALPHA);
}

TweakTau(argc,argv)
	int argc;
	char	**argv;
{
	tweak_tab_values(argc,argv,SETUP_TAU);
}

/* Convert a sparsely filled table to tabchannel/tabgate format
** starting from alpha-beta or tau-minf values. Optionally
** do the TABFILL, but this is not yet implemented */
tweak_tab_values(argc,argv,mode)
	int argc;
	char	**argv;
	int		mode;
{
	Element	*elm;
	Interpol	*A,*B;
	float temp,x,dx,temp2;
	int i;
	int	tab_gate_flag=0;
	Interpol *create_interpol();
	int xdivs;
	float xmin,xmax;

	A=B=NULL;
	if (argc!=3 && argc!= 6) {
		printf(
			"usage: %s chan table [xdivs xmin xmax]\n",
			argv[0]);
		return;
	}
	elm = GetElement(argv[1]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	if (strcmp(elm->object->name,"tabgate")==0) {
		A=((struct tab_gate_type *)elm)->alpha;
		B=((struct tab_gate_type *)elm)->beta;
		tab_gate_flag=1;
	}
	if (strcmp(elm->object->name,"tabchannel")==0) {
		if (strcmp(argv[2],"X")==0) {
			A=((struct tab_channel_type *)elm)->X_A;
			B=((struct tab_channel_type *)elm)->X_B;
		} else if (strcmp(argv[2],"Y")==0) {
			A=((struct tab_channel_type *)elm)->Y_A;
			B=((struct tab_channel_type *)elm)->Y_B;
		} else if (strcmp(argv[2],"Z")==0) {
			A=((struct tab_channel_type *)elm)->Z_A;
			B=((struct tab_channel_type *)elm)->Z_B;
		} else {
			printf("Gate type %s not known in %s \n",argv[2],argv[0]);
			return;
		}
	}
	if (!A || !B) {
		printf("Error: Element %s should be a tabgate or tabchannel\n",
			argv[1]);
		return;
	}
	if (mode==SETUP_ALPHA) {
		for(i=0;i<=A->xdivs;i++)
			B->table[i]+=A->table[i];
	} else if (mode==SETUP_TAU) {
		for(i=0;i<=A->xdivs;i++) {
			temp=A->table[i];
			temp2=B->table[i];
			A->table[i]=temp2/temp;
			B->table[i]=1.0/temp;
		}
	}
	if (argc==6) {
		printf("This feature not yet implemented, just use TABFILL \n");
	}
}

/*
** Sets up a single table with A B C D F values, not for tabchannels
*/
SetupGate(argc,argv)
	int argc;
	char	**argv;
{
	Element	*elm;
	Interpol	*ip;
	float temp,x,dx,temp2;
	int i;
	int	tab_gate_flag=0;
	Interpol *create_interpol();
	int xdivs;
	float xmin,xmax;
	float A,B,C,D,F;
	int	noalloc_flag=0;

	ip=NULL;
	if (argc<11) {
		printf(
			"usage: %s channel table xdivs xmin xmax A B C D F [noalloc]\n",
			argv[0]);
		return;
	}
	elm = GetElement(argv[1]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	xdivs=atoi(argv[3]);
	xmin=Atof(argv[4]);
	xmax=Atof(argv[5]);
	A=Atof(argv[6]);
	B=Atof(argv[7]);
	C=Atof(argv[8]);
	D=Atof(argv[9]);
	F=Atof(argv[10]);
	if (argc==12 && argv[11][0]=='n') noalloc_flag=1;
	if (xdivs<=9) {
		printf("Error : table %s should have at least 10 elements\n",
			argv[1]);
		return;
	}
	if (xmin >= xmax) {
		printf("Error : xmin=%f should be less than xmax=%f\n",
			xmin,xmax);
		return;
	}
	if (strcmp(elm->object->name,"tabgate")==0) {
		if (argv[2][0]=='a') { /* Alpha */
			if (!noalloc_flag) {
				((struct tab_gate_type *)elm)->alpha=
					create_interpol(xdivs,xmin,xmax);
				((struct tab_gate_type *)elm)->alpha_alloced=1;
			}
			ip=((struct tab_gate_type *)elm)->alpha;
		} else if (argv[2][0]=='b') { /* Beta */
			if (!noalloc_flag) {
				((struct tab_gate_type *)elm)->beta=
					create_interpol(xdivs,xmin,xmax);
				((struct tab_gate_type *)elm)->beta_alloced=1;
			}
			ip=((struct tab_gate_type *)elm)->beta;
		} else {
			printf("Error : table name %s invalid for tab_gates\n");
			return;
		}
		tab_gate_flag=1;
	}
	if (strcmp(elm->object->name,"tabchannel")==0) {
		printf("Error : Must set both A and B tables together\n");
		return;
	}
	if (strcmp(elm->object->name,"table")==0) {
		if (!noalloc_flag) {
			((struct table_type *)elm)->table=
				create_interpol(xdivs,xmin,xmax);
			((struct table_type *)elm)->alloced=1;
		}
			ip=((struct table_type *)elm)->table;
	}
	if (ip==NULL) {
		printf("Table not found in %s, type %s is not handled yet\n");
		return;
	}
	dx=(xmax-xmin)/(float)xdivs;
	for(x=xmin+dx/2.0,i=0;i<=xdivs;i++,x+=dx) {
		/* Assorted ugly NaN cases */
		if (F==0.0) {
			temp=ip->table[i]=0.0;
		} else {
			temp2=C+exp((x+D)/F);
			if (temp2==0.0)
				temp=ip->table[i]=ip->table[i-1];
			else
				temp=ip->table[i]= (A+B*x)/temp2;
		}
	}
}

/*
** ScaleTabchan(channel,gate,mode,sx,sy,ox,oy[-duplicate])
** Scales the values in the tables 
** according to the modes, so that the hh equations are still OK
** Modes : alpha,beta,tau,minf
** if the duplicate flag is set then it will copy it to 
** a new table before messing with it.
*/
ScaleTabchan(argc,argv)
	int argc;
	char	**argv;
{
	Element	*elm;
	Interpol	*A,*B;
	float temp,x,dx,temp2;
	int i,j;
	int	tab_gate_flag=0;
	Interpol *create_interpol();
	int xdivs;
	float sx,sy,ox,oy;
	float alpha,beta,tau,minf;
	float *holding1;
	float *holding2;
	float xmin,xmax;

	holding1=holding2=NULL;
	A=B=NULL;
	if (argc < 8) {
		printf(
			"usage: %s channel gate mode sx sy ox oy [d[uplicate]]\n",
			argv[0]);
		printf("Valid modes : a[lpha] b[eta] t[au] m[inf]\n");
		return;
	}
	elm = GetElement(argv[1]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	if (argc==9 && argv[8][0]=='d') /* do duplicate */
		DuplicateTable(3,argv);
	if (strcmp(elm->object->name,"tabgate")==0) {
		A=((struct tab_gate_type *)elm)->alpha;
		B=((struct tab_gate_type *)elm)->beta;
		tab_gate_flag=1;
	}
	if (strcmp(elm->object->name,"tabchannel")==0) {
		if (strcmp(argv[2],"X")==0) {
			A=((struct tab_channel_type *)elm)->X_A;
			B=((struct tab_channel_type *)elm)->X_B;
		} else if (strcmp(argv[2],"Y")==0) {
			A=((struct tab_channel_type *)elm)->Y_A;
			B=((struct tab_channel_type *)elm)->Y_B;
		} else if (strcmp(argv[2],"Z")==0) {
			A=((struct tab_channel_type *)elm)->Z_A;
			B=((struct tab_channel_type *)elm)->Z_B;
		} else {
			printf("Gate type %s not known in %s \n",argv[2],argv[0]);
			return;
		}
	}
	if (!A || !B) {
		printf("Error: Element %s should be a tabgate or tabchannel\n",
			argv[1]);
		return;
	}
		
	sx=Atof(argv[4]);
	sy=Atof(argv[5]);
	ox=Atof(argv[6]);
	oy=Atof(argv[7]);
	if (sx <= 0.0) {
		printf("Error: sx must be > 0\n");
		return;
	}
	xdivs=A->xdivs;
	xmin=A->xmin;
	xmax=A->xmax;
	dx=(xmax-xmin)/(float)xdivs;
	holding1=(float *)calloc(xdivs+2,sizeof(float));
	holding2=(float *)calloc(xdivs+2,sizeof(float));
	/* to use tabchannel code, temporarily change tabgate beta table */
	if (tab_gate_flag) {
		for(i=0;i<=xdivs;i++)
			B->table[i] += A->table[i];
	}
	switch(argv[3][0]) {
		case 'a' : /* alpha */
			for(i=0;i<=xdivs;i++) {
				/* find beta */
				holding2[i]=B->table[i]-A->table[i];
				j=(xmin*(1.0-sx)/sx - ox)/dx + (float)i/sx;
				if (j>xdivs) j=xdivs;
				if (j<0) j=0;
				alpha=oy+A->table[j]*sy;
				if (alpha<0.0)
					holding1[i]=0.0;
				else
					holding1[i]=alpha;
			}
			for(i=0;i<=xdivs;i++) {
				A->table[i]=holding1[i];
				B->table[i]=holding1[i]+holding2[i];
			}
			break;
		case 'b' : /* beta */
			for(i=0;i<=xdivs;i++) {
				j=(xmin*(1.0-sx)/sx - ox)/dx + (float)i/sx;
				if (j>xdivs) j=xdivs;
				if (j<0) j=0;
				alpha=A->table[j];
				beta=B->table[j];
				beta-=alpha;
				beta=oy+beta*sy;
				if (beta<0.0)
					holding1[i]=0.0;
				else
					holding1[i]=beta;
			}
			for(i=0;i<=xdivs;i++) B->table[i]=holding1[i]+A->table[i];
			break;
		case 't' : /* tau */
			for(i=0;i<=xdivs;i++) {
				holding2[i]=A->table[i]/B->table[i];
				j=(xmin*(1.0-sx)/sx - ox)/dx + (float)i/sx;
				if (j>xdivs) j=xdivs;
				if (j<0) j=0;
				tau=1.0/B->table[j];
				tau=oy+tau*sy;
				if (tau<0.0)
					holding1[i]=0.00000001; /* totally random number */
				else
					holding1[i]=tau;
			}
			for(i=0;i<=xdivs;i++) {
				A->table[i]=holding2[i]/holding1[i];
				B->table[i]=1.0/holding1[i];
			}
			break;
		case 'm' : /* minf */
			for(i=0;i<=xdivs;i++) {
				j=(xmin*(1.0-sx)/sx - ox)/dx + (float)i/sx;
				if (j>xdivs) j=xdivs;
				if (j<0) j=0;
				minf=A->table[j]/B->table[j];
				minf=oy+minf*sy;
				if (minf<0.0)
					holding1[i]=0.0;
				else
					holding1[i]=minf;
			}
			for(i=0;i<=xdivs;i++) {
				A->table[i]=holding1[i]*B->table[i];
			}
			break;
	}
	if (tab_gate_flag) {
		for(i=0;i<=xdivs;i++) B->table[i] -= A->table[i];
	}
	if (holding1) free(holding1);
	if (holding2) free(holding2);
}

/*
** makes a bcopy of the entire ip structure, so we can alter it without
** messing up all other ip structs.
*/
DuplicateTable(argc,argv)
	int argc;
	char	**argv;
{
	Element	*elm;

	if (argc < 3) {
		printf("usage: %s channel gate\n", argv[0]);
		return;
	}
	elm = GetElement(argv[1]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[1]);
		return;
	}
	if (strcmp(elm->object->name,"tabgate")==0) {
		tab_dup_func(&(((struct tab_gate_type *)elm)->alpha),
			&(((struct tab_gate_type *)elm)->beta));
	}
	if (strcmp(elm->object->name,"tabchannel")==0) {
		if (strcmp(argv[2],"X")==0) {
			tab_dup_func(&(((struct tab_channel_type *)elm)->X_A),
				&(((struct tab_channel_type *)elm)->X_B));
		}
		if (strcmp(argv[2],"Y")==0) {
			tab_dup_func(&(((struct tab_channel_type *)elm)->Y_A),
				&(((struct tab_channel_type *)elm)->Y_B));
		}
		if (strcmp(argv[2],"Z")==0) {
			tab_dup_func(&(((struct tab_channel_type *)elm)->Z_A),
				&(((struct tab_channel_type *)elm)->Z_B));
		}
	}
}

tab_dup_func(pa,pb)
	Interpol	**pa,**pb;
{
	double	*temp;
	Interpol	*A,*B;
	int	i;

	A=*pa;
	B=*pb;

	*pa=create_interpol(A->xdivs,A->xmin,A->xmax);
	temp=(*pa)->table;
	**pa=*A;
	(*pa)->table=temp;
	/* Note that there are xdivs+1 entries in each table */
	for(i=0;i<=A->xdivs;i++)
		temp[i] = A->table[i];

	*pb=create_interpol(B->xdivs,B->xmin,B->xmax);
	temp=(*pb)->table;
	**pb=*B;
	(*pb)->table=temp;
	/* Note that there are xdivs+1 entries in each table */
	for(i=0;i<=B->xdivs;i++)
		temp[i] = B->table[i];
}

FileToTab(argc,argv)
	int argc;
	char	**argv;
{
	FILE	*fp,*fopen();
	Interpol	*ip;
	Element	*elm;
	int i;
	float temp;

	if (argc < 4) {
		printf("usage: %s file element table\n", argv[0]);
		return;
	}
	fp=fopen(argv[1],"r");
	if (!fp) {
		printf("Could not open file %s\n",argv[1]);
		return;
	}
	elm = GetElement(argv[2]);
	if (!elm) {
		printf("Element '%s' not found\n",argv[2]);
		return;
	}
	if (strcmp(elm->object->name,"tabgate")==0) {
		if (argv[3][0]=='b')
			ip=((struct tab_gate_type *)elm)->beta;
		else if (argv[3][0]=='a')
			ip=((struct tab_gate_type *)elm)->alpha;
	} else
	if (strcmp(elm->object->name,"tabchannel")==0) {
		if (strcmp(argv[3],"X_A")==0)
			ip=((struct tab_channel_type *)elm)->X_A;
		else if (strcmp(argv[3],"X_B")==0)
			ip=((struct tab_channel_type *)elm)->X_B;
		else if (strcmp(argv[3],"Y_A")==0)
			ip=((struct tab_channel_type *)elm)->Y_A;
		else if (strcmp(argv[3],"Y_B")==0)
			ip=((struct tab_channel_type *)elm)->Y_B;
		else if (strcmp(argv[3],"Z_A")==0)
			ip=((struct tab_channel_type *)elm)->Z_A;
		else if (strcmp(argv[3],"Z_B")==0)
			ip=((struct tab_channel_type *)elm)->Z_B;
	} else
	if (strcmp(elm->object->name,"table")==0) {
		ip=((struct table_type *)elm)->table;
	}
	if (!ip) {
		printf("Error : %s %s not allocated\n",argv[2],argv[3]);
		return;
	}
	for(i=0;i<ip->xdivs;i++) {
		if (fscanf(fp,"%f",&temp) < 1)
			break;
		ip->table[i]=temp;
	}
	ip->xmax=i;
	fclose(fp);
}
