static char rcsid[] = "$Id: hines.c,v 1.1 1992/10/29 16:26:11 dhb Exp $";

/*
** $Log: hines.c,v $
 * Revision 1.1  1992/10/29  16:26:11  dhb
 * Initial revision
 *
*/

#include "hines_ext.h"
#include "seg_struct.h"
#include "olf_struct.h"
#include "dev_struct.h"


/* Upinder S. Bhalla Caltech May-December 1991
** Erik De Schutter December 1991-Februari 1992
**
** We would like to acknowlege the advice of Dr. Michael Mascagni
** during the development of this element.
** 
** Element for applying Hines numbering method to solving the cable
** properites of a cell in an implicit solution.
** The element sets up a data structure whereby the only change
** needed for other identical cells is to reassign pointers to
** the compartment and channel elements.
** This avoids having to reallocate and rederive the solving scheme.
** Solutions are done using gaussian forward  and backward elimination
** without pivoting. The numbering and evaluation sequence ensures that
** no new off-diagonal terms terms are formed which might mess up
** the sparse matrix representation.
** The sparse matrix is represented by a single 'values' array indexed
** using the ri and cip arrays.
** The actual solution system is stored as a linear sequence of
** operations on the 'values' array, in an attempt to speed things
** up a bit.
*/

int DEBUG_HinesSolver=0;

HinesSolver(hsolve,action)
struct hsolve_type *hsolve;
Action		*action;
{
	int i;
	int comptno,nop,nchip;
	Element *elm;

	if(Debug(DEBUG_HinesSolver) > 1){
		ActionHeader("HinesSolver",hsolve,action);
	}
	SELECT_ACTION(action){
		case PROCESS:
			/* Doing the t(n+1/2) evaluation of hh channels */
			switch (hsolve->chanmode) {
				case 1 :
					do_unrolled_hh_update(hsolve);
					do_hcalc(hsolve);
					if (hsolve->useconcen)
						do_conc_update(hsolve);
					if (hsolve->comptmode) {
						do_fast_hsolve(hsolve);
					} else {
						do_hsolve(hsolve);
					}
					do_compt_update(hsolve);
					break;
				case 2 :
					if (hsolve->ininfo)
						h_in_msgs(hsolve);
					do_chip_hh_update(hsolve);
					do_chip_hcalc(hsolve);
					if (hsolve->comptmode) {
						do_fast_hsolve(hsolve);
					} else {
						do_hsolve(hsolve);
					}
					do_chip_update(hsolve);
					break;
				case 3 :
					if (hsolve->ininfo)
						h_in_msgs(hsolve);
					do_chip_hh_update(hsolve);
					do_chip_hcalc(hsolve);
					if (hsolve->comptmode) {
						do_fast_hsolve(hsolve);
					} else {
						do_hsolve(hsolve);
					}
					do_chip_update(hsolve);
					if (hsolve->outinfo)
						h_out_msgs(hsolve);
					break;
				default :
					do_hh_update(hsolve);
					do_hcalc(hsolve);
					if (hsolve->useconcen)
						do_conc_update(hsolve);
					if (hsolve->comptmode) {
						do_fast_hsolve(hsolve);
					} else {
						do_hsolve(hsolve);
					}
					do_compt_update(hsolve);
					break;
			}
			break;
		case RESET:
			switch (hsolve->chanmode) {
				case 0 :
				case 1 :
					do_hreset(hsolve);
					break;
				case 2 :
				case 3 :
					do_chip_hreset(hsolve,"RESET");
					break;
				default :
					do_hreset(hsolve);
					break;
			}
			break;
		case SETUP:
			/* all sorts of memory allocation */
			hsolve->dt=Clockrate(hsolve);
			if (h_init(hsolve)) {
				h_failed(hsolve);
				return(ERR);
			}
			if (conc_init(hsolve)) {
				h_failed(hsolve);
				return(ERR);
			}
			switch(hsolve->comptmode) {
				case 0 :
					break;
				case 1 :
				case 2 :
					hsolve->nfuncs = 0;
					/* The first pass counts the number of funcs */
					if (do_fast_hsetup(hsolve)) {
						h_failed(hsolve);
						return(ERR);
					}
					/* The second pass assigns the funcs */
					if (do_fast_hsetup(hsolve)) {
						h_failed(hsolve);
						return(ERR);
					}
					break;
				default:
					break;
			}

			/* Set up tabchannels */
			switch (hsolve->chanmode) {
				case 0 :
				case 1 :
					if (h_hh_init(hsolve)) {
						h_failed(hsolve);
						return(ERR);
					}
					break;
				case 2 :
				case 3 :
					if (h2_init(hsolve)) {
						h_failed(hsolve);
						return(ERR);
					}
					if (h_hh_chip_init(hsolve,0)) {
						h_failed(hsolve);
						return(ERR);
					}
					break;
				default :
					if (h_hh_init(hsolve)) {
						h_failed(hsolve);
						return(ERR);
					}
					break;
			}
			break;

		case DELETE:
			if (IsSilent() <= 0)
				fprintf(stderr,"deleting hsolve element : re-enabling path\n");
			h_delete(hsolve);
			break;
		case DUPLICATE:
			if (do_duplicate(hsolve,action->argv[0])) {
				ErrorMessage("hsolve","Cell is not identical.",hsolve);
			}
			break;
		case HPUT:
		case HGET:
			if (action->argc < 1) {
				if (action->type == HPUT) {
					fprintf(stderr,"usage: HPUT element_path\n");
					fprintf(stderr,"updates any reference to element fields in hsolve\n");
				} else {
					fprintf(stderr,"usage: HGET element_path\n");
					fprintf(stderr,"updates element fields from hsolve arrays\n");
				}
				return(ERR);
			}
			if (hsolve->no_elminfo) {
				fprintf(stderr,"HPUT/HGET calls do not work if the no_elminfo field is set\n");
				return(ERR);
			}
			if (hsolve->chanmode < 2) {
				fprintf(stderr,"HPUT/HGET calls should only be used when chanmode >= 2\n");
				return(ERR);
			}
			if (hsolve->ncompts == 0) {
				fprintf(stderr,"no SETUP call\n");
				return(ERR);
			}
			elm = GetElement(action->argv[0]);
			if (!elm) {
				fprintf(stderr,"%s not found\n",action->argv[0]);
				return(ERR);
			}
			if (!IsHsolved(elm)) {
				fprintf(stderr,"%s is not handled by hsolve\n",action->argv[0]);
				return(ERR);
			}
			if (hfind_elm(hsolve,elm,&comptno,&nop,&nchip)) {
				fprintf(stderr,"Do not know how to find %s fields in %s\n",elm->name,hsolve->name);
				return(ERR);
			}
			if (action->type == HPUT) {
				hput_elm(hsolve,elm,comptno,nop,nchip);
			} else {
				hget_elm(hsolve,elm,comptno,nop,nchip);
			}
			break;
		case HRESTORE:
		case HSAVE:
			if (hsolve->chanmode < 2) {
				fprintf(stderr,"HRESTORE/HSAVE calls should only be used when chanmode >= 2\n");
				return(ERR);
			}
			if (action->type == HRESTORE) {
				do_chip_hreset(hsolve,"NONE");
			} else {
				do_chip_hsave(hsolve);
			}
			break;
	}
}

h_failed(hsolve)
	Hsolve	*hsolve;
{
	fprintf(stderr,"hsolve setup failed\n");
	fprintf(stderr,"deleting hsolve element : re-enabling path\n");
	h_delete(hsolve);
}

/* Simple function for putting data values back into elements */
do_hreset(hsolve)
	Hsolve	*hsolve;
{
	int i,j;
	struct compartment_type **compts;
	int ncompts = hsolve->ncompts;
	int	*elmnum;
    Element **concen;
	Element *c;
    Element **nernst;
	Element *n;
	double	*results;
	double	*flux;
	Action	*action;
	Cinfo	**hh,*hentry;

	compts = (struct compartment_type **)(hsolve->compts);
	hh = hsolve->hh;
	elmnum = hsolve->elmnum;
	results = hsolve->results;

	action = GetAction("RESET");

	for(i=0;i<ncompts;i++) {
		compts[elmnum[i]]->Vm = results[i] = compts[elmnum[i]]->Em;
		for(hentry=hh[i];hentry;hentry=hentry->next) {
			CallElement(hentry->chan,action);
		}
	}
	if (hsolve->useconcen) {
		concen = hsolve->concens;
		flux = hsolve->flux;
		for(i=0;i<ncompts;i++) {
			if((c=concen[i])) {
				CallElement(c,action);
			}
			flux[i]=0.0;
		}
	}
	if (hsolve->usenernst) {
		nernst = hsolve->nernsts;
		for(i=0;i<ncompts;i++) {
			if((n=nernst[i])) {
				CallElement(n,action);
			}
		}
	}
}

/* Function for putting data values from elements into chip array */
do_chip_hreset(hsolve,actionstr)
	Hsolve	*hsolve;
	char	*actionstr;
{
	int i,j,k;
	Element **compts,*compt;
	Element **concens;
	Element **nernsts;
	Element	*elm;
	struct tab_channel_type *chan;
	struct channelA_type    *achan;
	int ncompts = hsolve->ncompts;
	int	*elmnum;
    double  dt;
    double  tby2;
	Action	*action;
	MsgIn	*msgin,*cmsgin;
	double	*chip;
	int		*ops;
	int		nchip,nop;
	int		hasEk;
	int		Vm_flag;

	if (IsSilent() < 0)
		printf("transferring element field values into solve arrays\n");
	compts = hsolve->compts;
	elmnum = hsolve->elmnum;
	concens = hsolve->concens;
	nernsts = hsolve->nernsts;
	chip=hsolve->chip;
	ops=hsolve->ops;

	if (strcmp(actionstr,"NONE")==0)
		Vm_flag=1;
	else
		Vm_flag=hsolve->Vm_reset;
	action = GetAction(actionstr);
	dt = Clockrate(hsolve);
    if (dt != hsolve->dt) {
	/* user changed the clocks after SETUP call */
        for(i=0;i<hsolve->ntab;i++) {
            for(j=0,k=i;j<hsolve->xdivs+1;j++,k+=hsolve->ntab) {
				hsolve->tablist[k]=1.0+((hsolve->tablist[k]-1.0)*dt/hsolve->dt);
            }
            i++;
            for(j=0,k=i;j<hsolve->xdivs+1;j++,k+=hsolve->ntab) {
				hsolve->tablist[k]=hsolve->tablist[k]*dt/hsolve->dt;
            }
        }
        for(i=0;i<hsolve->cntab;i++) {
            for(j=0,k=i;j<hsolve->cxdivs+1;j++,k+=hsolve->cntab) {
				hsolve->ctablist[k]=1.0+((hsolve->ctablist[k]-1.0)*dt/hsolve->dt);
            }
            i++;
            for(j=0,k=i;j<hsolve->cxdivs+1;j++,k+=hsolve->cntab) {
				hsolve->ctablist[k]=hsolve->ctablist[k]*dt/hsolve->dt;
            }
        }
		hsolve->dt = dt;
	}
    tby2 = dt/2.0;

	nchip=nop=0;
	for(i=0;i<ncompts;i++) {
		compt = compts[elmnum[i]];
		chip_put_compt(hsolve,compt,nchip,i,Vm_flag);
		nchip+=2;
		nop++;
	    if (hsolve->useconcen && concens[i]) {
			CallElement(hsolve->concens[i],action);
			chip_put_concen1(hsolve,concens[i],nchip);
			nchip+=2;
			nop++;
		}
		if (hsolve->usenernst && nernsts[i]) {
			CallElement(hsolve->nernsts[i],action);
			chip_put_nernst(hsolve,nernsts[i],nchip,ops[nop]);
			nchip+=2;
			nop++;
		}
		for(msgin=compt->msg_in;msgin;msgin=msgin->next) {
			/* look for any channels */
			if (msgin->type == CHANNEL) {
				elm=msgin->src;
                hasEk=1;
                for (cmsgin=elm->msg_in;cmsgin;cmsgin=cmsgin->next) {
                    if (cmsgin->type == EK) {
                        hasEk=0;
                        break;
                    }
                }
                if (strcmp(elm->object->name,"tabchannel")==0) {
					CallElement(elm,action);
					chan=(struct tab_channel_type *)msgin->src;
					chip_put_tabchannel(hsolve,elm,nchip,hasEk);
					nchip++;
					nop++;
					if (chan->X_A || chan->X_B) {
						nchip++;
						nop+=3;
					}
					if (chan->Y_A || chan->Y_B) {
						nchip++;
						nop+=3;
					}
					if (chan->Z_A || chan->Z_B) {
						nchip++;
						nop+=3;
					}
                } else if (strcmp(elm->object->name,"channelC2")==0) {
					CallElement(elm,action);
					chip_put_channelc3(hsolve,elm,nchip,hasEk);
                    nchip+=3;
					nop+=3;
                } else if (strcmp(elm->object->name,"channelC3")==0) {
					CallElement(elm,action);
					chip_put_channelc3(hsolve,elm,nchip,hasEk);
                    nchip+=4;
					nop+=3;
				} else if (strcmp(msgin->src->object->name,"channelC")!=0){
					/* not HsolveBlocked */
					nop+=2;
					if (hasEk) {
						achan=(struct channelA_type *)elm;
						chip[nchip]=achan->Ek;
					}
				}
				if (hasEk) {
					nchip++;
				}
				nop++;
				if ((ops[nop]>=POS_FLUX_OP) && (ops[nop]<3400)) {
					nop++;
				}
				if (ops[nop]==STORE_CURR_OP) {
					nop++;
					nchip+=2;
				}
			}
		}
	    if (hsolve->useconcen && concens[i]) {
			chip_put_concen2(hsolve,concens[i],nchip);
			nchip+=2;
			nop++;
		}
	}
}
/* Function for putting data values back into elements from chip array */
do_chip_hsave(hsolve)
	Hsolve	*hsolve;
{
	int i,j,k;
	Element **compts,*compt;
	Element **concens;
	Element **nernsts;
	Element	*elm;
	struct tab_channel_type *chan;
	struct channelA_type    *achan;
	int ncompts = hsolve->ncompts;
	int	*elmnum;
	MsgIn	*msgin,*cmsgin;
	double	*chip;
	int		*ops;
	int		nchip,nop;
	int		hasEk;

	if (IsSilent() < 0)
		printf("restoring all solve computed fields\n");
	compts = hsolve->compts;
	elmnum = hsolve->elmnum;
	concens = hsolve->concens;
	nernsts = hsolve->nernsts;
	chip=hsolve->chip;
	ops=hsolve->ops;

	nchip=nop=0;
	for(i=0;i<ncompts;i++) {
		compt = compts[elmnum[i]];
		chip_get_compt(hsolve,compt,i);
		nchip+=2;
		nop++;
	    if (hsolve->useconcen && concens[i]) {
			chip_get_concen(hsolve,concens[i],nchip);
			nchip+=2;
			nop++;
		}
		if (hsolve->usenernst && nernsts[i]) {
			chip_get_nernst(hsolve,nernsts[i],nchip,ops[nop]);
			nchip+=2;
			nop++;
		}
		for(msgin=compt->msg_in;msgin;msgin=msgin->next) {
			/* look for any channels */
			if (msgin->type == CHANNEL) {
				elm=msgin->src;
                hasEk=1;
                for (cmsgin=elm->msg_in;cmsgin;cmsgin=cmsgin->next) {
                    if (cmsgin->type == EK) {
                        hasEk=0;
                        break;
                    }
                }
                if (strcmp(elm->object->name,"tabchannel")==0) {
					chan=(struct tab_channel_type *)msgin->src;
					nchip++;
					nop++;
					chip_get_tabchannel(hsolve,elm,nchip,0);
					if (chan->X_A || chan->X_B) {
						nchip++;
						nop+=3;
					}
					if (chan->Y_A || chan->Y_B) {
						nchip++;
						nop+=3;
					}
					if (chan->Z_A || chan->Z_B) {
						nchip++;
						nop+=3;
					}
                } else if (strcmp(elm->object->name,"channelC2")==0) {
					chip_get_channelc3(hsolve,elm,nchip+1,0);
                    nchip+=3;
					nop+=3;
                } else if (strcmp(elm->object->name,"channelC3")==0) {
					chip_get_channelc3(hsolve,elm,nchip+1,0);
                    nchip+=4;
					nop+=3;
				} else if (strcmp(msgin->src->object->name,"channelC")!=0){
					/* not HsolveBlocked */
					nop+=2;
				}
				if (hasEk) {
					nchip++;
				}
				nop++;
				if ((ops[nop]>=POS_FLUX_OP) && (ops[nop]<3400)) {
					nop++;
				}
				if (ops[nop]==STORE_CURR_OP) {
					nop++;
					nchip+=2;
				}
			}
		}
	    if (hsolve->useconcen && concens[i]) {
			nchip+=2;
			nop++;
		}
	}
}

h_delete(hsolve)
	Hsolve	*hsolve;
{
	int i;
	Cinfo	*hentry,*hlast,*centry,*clast;

	if (hsolve->ncompts > 0) {
		for(i=0;i<hsolve->ncompts;i++) {
			HsolveEnable(hsolve->compts[i]);
			if (hsolve->concens) {
				HsolveEnable(hsolve->concens[i]);
			}
			if (hsolve->nernsts) {
				HsolveEnable(hsolve->nernsts[i]);
			}
			if (hsolve->nkids[i] > 0 && hsolve->kids[i]) {
				free(hsolve->kids[i]);
				hsolve->nkids[i]=0;
			}
			if (hsolve->hh) {
				for(hentry=hsolve->hh[i];hentry;){
					HsolveEnable(hentry->chan);
					hlast=hentry;
					hentry=hentry->next;
					if (hlast) free(hlast);
				}
				hsolve->hh[i]=NULL;
			}
			if (hsolve->chan) {
				for(centry=hsolve->chan[i];centry;){
					clast=centry;
					centry=centry->next;
					if (clast) free(clast);
				}
			}
		}
		if (hsolve->compts) free(hsolve->compts);
			hsolve->compts=NULL;
		if (hsolve->parents) free(hsolve->parents);
			hsolve->parents=NULL;
		if (hsolve->kids) free(hsolve->kids);
			hsolve->kids=NULL;
		if (hsolve->nkids) free(hsolve->nkids);
			hsolve->nkids=NULL;
		if (hsolve->nernsts) free(hsolve->nernsts);
			hsolve->nernsts=NULL;
		if (hsolve->nernstmsg) free(hsolve->nernstmsg);
			hsolve->nernstmsg=NULL;
		if (hsolve->concens) free(hsolve->concens);
			hsolve->concens=NULL;
		if (hsolve->flux) free(hsolve->flux);
			hsolve->flux=NULL;
		if (hsolve->chanmode >= 2) {
			if (hsolve->comps) free(hsolve->comps);
            if (hsolve->vm) free(hsolve->vm);
			if (hsolve->ntab>0 && hsolve->tablist) {
				free(hsolve->tablist);
				hsolve->tablist=NULL;
				hsolve->ntab=0;
			}
			if (hsolve->cntab>0 && hsolve->ctablist) {
				free(hsolve->ctablist);
				hsolve->ctablist=NULL;
				hsolve->cntab=0;
			}
			if (hsolve->sntab>0 && hsolve->stablist) {
				free(hsolve->stablist);
				hsolve->stablist=NULL;
				hsolve->sntab=0;
			}
			if (hsolve->nchips>0 && hsolve->chip) {
				free(hsolve->chip);
				hsolve->nchips=0;
				hsolve->chip=NULL;
			}
			if (hsolve->chip_index) free(hsolve->chip_index);
				hsolve->chip_index=NULL;
			if (hsolve->ops_index) free(hsolve->ops_index);
				hsolve->ops_index=NULL;
			if (hsolve->comp_elms) free(hsolve->comp_elms);
				hsolve->comp_elms=NULL;
			for(i=0; i<hsolve->nelm_names; i++){
				if (hsolve->elm_names[i]) free(hsolve->elm_names[i]);
				hsolve->elm_names[i]=NULL;
			}
			if (hsolve->elm_index) free(hsolve->elm_index);
			hsolve->elm_index=NULL;
		} else {
			if (hsolve->hh) free(hsolve->hh);
			hsolve->hh=NULL;
		}
		if (hsolve->chan) free(hsolve->chan);
		hsolve->chan=NULL;
		if (hsolve->hnum) free(hsolve->hnum);
		hsolve->hnum=NULL;
		if (hsolve->elmnum) free(hsolve->elmnum);
		hsolve->elmnum=NULL;
		if (hsolve->nvalues > 0) {
			if (hsolve->results) free(hsolve->results);
			if (hsolve->values) free(hsolve->values);
			if (hsolve->ri) free(hsolve->ri);
			if (hsolve->cip) free(hsolve->cip);
			if (hsolve->diag) free(hsolve->diag);
			hsolve->nvalues=0;
		}
		if (hsolve->nfuncs > 0 && hsolve->funcs) {
			free(hsolve->funcs);
			hsolve->nfuncs=0;
		}
	}
}

int do_duplicate(hsolve,path)
	Hsolve	*hsolve;
	char	*path;
{
	ElementList *list;
	int 	ncompts;
	Element	**compts;
	int 	i;
	int		*elmnum;
	char	*CopyString();

	if (!(hsolve->path) || !(path)) {
		fprintf(stderr,"No path defined for hsolve : DUPLICATE failed\n");
		return(ERR);
	}
	hsolve->path=CopyString(path);

	list = WildcardGetElement(path,0);
	if (hsolve->ncompts != list->nelements)
		return(0);
	compts=hsolve->compts = list->element;
	ncompts = hsolve->ncompts;
	elmnum = hsolve->elmnum;
	for (i=0;i<ncompts;i++)
		HsolveBlock(compts[elmnum[i]]);

	if (conc_init(hsolve)) {
		return(ERR);
	}

	/* Set up tabchannels */
	switch (hsolve->chanmode) {
		case 0 :
		case 1 :
			if (h_hh_init(hsolve)) {
				return(ERR);
			}
			break;
		case 2 :
		case 3 :
			if (h2_init(hsolve)) {
				return(ERR);
			}
			if (h_hh_chip_init(hsolve,1)) {
				return(ERR);
			}
			break;
		default :
			if (h_hh_init(hsolve)) {
				return(ERR);
			}
			break;
	}

	return(0);
}
