static char rcsid[] = "$Id: hh_init.c,v 1.1 1992/10/29 16:48:07 dhb Exp $";

/*
** $Log: hh_init.c,v $
 * Revision 1.1  1992/10/29  16:48:07  dhb
 * Initial revision
 *
*/

#include "hines_ext.h"
#include "seg_struct.h"
#include "olf_struct.h"
#include "dev_struct.h"

/*
** Fills up hh array according to hines numbering of parent compts,
** and moves all vdep-channel tables into a colocated chunk so that
** page swaps are (hopefully) minimized.
*/

int conc_init(hsolve)
	struct hsolve_type	*hsolve;
{
    int     ncompts;
	int		i;

	ncompts = hsolve->ncompts;
    /* setup nernst arrays if appropriate */
	if (hsolve->usenernst) { 
        if (!hsolve->useconcen) hsolve->useconcen = 1;
        hsolve->nernsts = (Element **)calloc(ncompts,sizeof(Element *));
		for (i=0;i<ncompts;i++){
			hsolve->nernsts[i]=NULL;
		}
    }
	if ((hsolve->chanmode < 2) && (hsolve->usenernst == 1)) {
        hsolve->nernstmsg = (int *)calloc(ncompts,sizeof(int));
    }
    /* setup concentration arrays if appropriate */
    if (hsolve->useconcen) {
        hsolve->concens = (Element **)calloc(ncompts,sizeof(Element *));
		for (i=0;i<ncompts;i++){
			hsolve->concens[i]=NULL;
		}
        if (hsolve->chanmode < 2) {
			hsolve->flux = (double *)calloc(ncompts,sizeof(double));
        }
    }
	return(0);
}

int check_tab(ip,hsolve,tablist,must)
	Interpol	*ip;
	struct hsolve_type	*hsolve;
	double	**tablist;
	int		must;
{
	int		j;

	if (!ip) {
		fprintf(stderr,"Warning : missing tabchannel table\n");
		return(ERR);
	}

	if (must>0) {
		if (tablist[must]==ip->table) {
			/* it has already been defined */
			return(must);
		} else if (must<hsolve->ntab) {
			fprintf(stderr,"Error in check_tab: diff A-tables for same B-table on %s,%d. must=%d,ntab=%d\n",hsolve->parent->name,ip,must,hsolve->ntab);
			return(ERR);
		}
	} else {
		for(j=0;j< hsolve->ntab;j++) {
			if (tablist[j]==ip->table) {
				/* it has already been defined */
				return(j);
			}
		}
	}
	if (hsolve->ntab == MAXTAB) {
		fprintf(stderr,"Error: too many tabchannel tables\n");
		return(ERR);
	} else if (hsolve->ntab == 0) {
		hsolve->xdivs=ip->xdivs;
		hsolve->xmin=ip->xmin;
		hsolve->invdx=ip->invdx;
	}
	tablist[hsolve->ntab]=ip->table;

	if ((ip->xdivs) != hsolve->xdivs) {
		fprintf(stderr,"Warning : hines xdivs=%d is != ip xdivs=%d\n",
			hsolve->xdivs,ip->xdivs);
		return(ERR);
	}
	if ((ip->xmin) != hsolve->xmin) {
		fprintf(stderr,"Warning : hines xmin=%f is != ip xmin=%f\n",
			hsolve->xmin,ip->xmin);
		return(ERR);
	}
	if ((ip->invdx) != hsolve->invdx) {
		fprintf(stderr,"Warning : hines invdx=%f is != ip invdx=%f\n",
			hsolve->invdx,ip->invdx);
		return(ERR);
	}
	(hsolve->ntab)++;
	return(hsolve->ntab-1);
}

int check_chanX(chan,hsolve,tablist)
	struct tab_channel_type	*chan;
	struct hsolve_type	*hsolve;
	double	**tablist;
{
	int	ia,ib;

	ib=check_tab(chan->X_B,hsolve,tablist,-1);
	if (ib==ERR){
		return(ERR);
	}
	ia=check_tab(chan->X_A,hsolve,tablist,ib+1);
	if (ia==ERR){
		return(ERR);
	}
	return(0);
}

int check_chanY(chan,hsolve,tablist)
	struct tab_channel_type	*chan;
	struct hsolve_type	*hsolve;
	double	**tablist;
{
	int	ia,ib;

	ib=check_tab(chan->Y_B,hsolve,tablist,-1);
	if (ib==ERR){
		return(ERR);
	}
	ia=check_tab(chan->Y_A,hsolve,tablist,ib+1);
	if (ia==ERR){
		return(ERR);
	}
	return(0);
}

int check_ctab(ip,hsolve,tablist,must)
	Interpol	*ip;
	struct hsolve_type	*hsolve;
	double	**tablist;
	int		must;
{
	int		j;

	if (!ip) {
		fprintf(stderr,"Warning : missing tabchannel table\n");
		return(ERR);
	}

	if (must>0) {
		if (tablist[must]==ip->table) {
			/* it has already been defined */
			return(must);
		} else if (must<hsolve->cntab) {
			fprintf(stderr,"Error in check_ctab: different A-tables for same B-table\n");
			return(ERR);
		}
	} else {
		for(j=0;j< hsolve->cntab;j++) {
			if (tablist[j]==ip->table) {
				/* it has already been defined */
				return(j);
			}
		}
	}

	if (hsolve->cntab == MAXTAB) {
		fprintf(stderr,"Error: too many tabchannel tables\n");
		return(ERR);
	} else if (hsolve->cntab == 0) {
		hsolve->cxdivs=ip->xdivs;
		hsolve->cxmin=ip->xmin;
		hsolve->cinvdx=ip->invdx;
	}
	tablist[hsolve->cntab]=ip->table;

	if ((ip->xdivs) != hsolve->cxdivs) {
		fprintf(stderr,"Warning : hines xdivs=%d is != ip xdivs=%d\n",
			hsolve->cxdivs,ip->xdivs);
		return(ERR);
	}
	if ((ip->xmin) != hsolve->cxmin) {
		fprintf(stderr,"Warning : hines xmin=%f is != ip xmin=%f\n",
			hsolve->cxmin,ip->xmin);
		return(ERR);
	}
	if ((ip->invdx) != hsolve->cinvdx) {
		fprintf(stderr,"Warning : hines invdx=%f is != ip invdx=%f\n",
			hsolve->cinvdx,ip->invdx);
		return(ERR);
	}
	(hsolve->cntab)++;
	return(hsolve->cntab-1);
}

int check_chanZ(chan,hsolve,tablist)
	struct tab_channel_type	*chan;
	struct hsolve_type	*hsolve;
	double	**tablist;
{
	int	ia,ib;

	ib=check_ctab(chan->Z_B,hsolve,tablist,-1);
	if (ib==ERR){
		return(ERR);
	}
	ia=check_ctab(chan->Z_A,hsolve,tablist,ib+1);
	if (ia==ERR){
		return(ERR);
	}
	return(0);
}

int check_stab(schan,hsolve,tablist)
	struct channelC3_type *schan;
	struct hsolve_type	*hsolve;
	double	*tablist;
{
	int		j;

	if (!schan)
		return(ERR);

	for(j=0;j<hsolve->sntab*SYNSIZE;j+=SYNSIZE) {
		if ((fabs(tablist[j] - schan->xconst1) < VTINY) &&
		    (fabs(tablist[j+1] - schan->xconst2) < VTINY) &&
		    (fabs(tablist[j+2] - schan->yconst1) < VTINY) &&
		    (fabs(tablist[j+3] - schan->yconst2) < VTINY)) {
			/* it has already been entered */
			return(j);
		}
	}
	if (hsolve->sntab*SYNSIZE == MAXSYN) {
		fprintf(stderr,"Error: too many synaptic channel types\n");
		return(ERR);
	} 
	tablist[j] = schan->xconst1;
	tablist[j+1] = schan->xconst2;
	tablist[j+2] = schan->yconst1;
	tablist[j+3] = schan->yconst2;
	(hsolve->sntab)++;
	return(j);
}

int store_name(hsolve,elm,nnames,names)
	Hsolve	*hsolve;
	Element	*elm;
	int     *nnames;
	char    **names;
{
	int     i;

	if (!elm || hsolve->no_elminfo) {
		return(0);
	}

	for (i=0;i<*nnames;i++) {
		if (strcmp(elm->name,names[i])==0) {
			return(i);
		}
	}
	(*nnames)++;
	if (*nnames==MAXNAMES) {	/* only one error message! */
		fprintf(stderr,"Error: too many element names\n");
		fprintf(stderr,"SETUP will continue, but HSET/HPUT calls might fail\n");
		return(ERR);
	}
	if (*nnames<MAXNAMES) {
		names[i]=(char *)calloc(strlen(elm->name)+1,sizeof(char));
		strcpy(names[i],elm->name);
		return(i);
	} else {
		return(ERR);
	}
}

store_elm(hsolve,elm,nelm,nop,nchip)
	Hsolve	*hsolve;
	Element	*elm;
	int     *nelm,nop,nchip;
{
	int     i;
	char    **names;
	int     *elms;

	if (hsolve->no_elminfo) return;

	names = hsolve->elm_names;
	elms = hsolve->elm_index;
	for (i=0;i<hsolve->nelm_names;i++) {
		if (strcmp(elm->name,names[i])==0) {
			elms[*nelm]=i;
			elms[*nelm+1]=nop;
			elms[*nelm+2]=nchip;
			(*nelm)+=3;
			return;
		}
	}
}

int h_hh_chip_init(hsolve,duplicate_flag)
	Hsolve	*hsolve;
	int		duplicate_flag;
{
	int 	ncompts;
	Element	**compts;
	Element	*compt;
	Element	*elm;
	struct Ca_shell_type **concens;
	struct nernst_type **nernsts;
	struct tab_channel_type	*chan;
	struct channelC3_type *schan;
	int 	i,j,k,op,ch,nc;
	MsgIn	*msgin,*cmsgin;
	MsgOut	*msgout;
	int		*elmnum;
	Cinfo	**hh;
	Cinfo	*hentry;
	double  dt;
	double  tby2;
	double	*tablist[MAXTAB],*ctablist[MAXTAB];
	double	stablist[MAXSYN];
	short	dtlist[MAXTAB],dtclist[MAXTAB];
	int		ntab=0,cntab=0,nchip=0,nop=0,nelm=0,nnames=0;
	int		tabsize=0;
    Action  *action;
	double	*chip;
	int		*ops;
	float	flux_scale;
	int		hasEk;
	char	**names[MAXNAMES];

	ncompts = hsolve->ncompts;
	compts = hsolve->compts;
	elmnum = hsolve->elmnum;
    concens = (struct Ca_shell_type **)hsolve->concens;
	nernsts = (struct nernst_type **)hsolve->nernsts;
	hsolve->ntab=0;
	hsolve->cntab=0;
	hsolve->sntab=0;
	for(i=0;i<MAXTAB;i++) tablist[i]=ctablist[i]=(double *)NULL;
	for(i=0;i<MAXTAB;i++) dtlist[i]=dtclist[i]=0;
	for(i=0;i<MAXSYN;i++) stablist[i]=0.0;

	action = GetAction("RECALC");
	dt = hsolve->dt;
	tby2 = dt/2.0;

	/* CHANNELs go from channels to compts */
/* This initial loop finds the number of elements that the chip array will
** have to deal with.  */
	for (i=0;i<ncompts;i++){
		compt = compts[elmnum[i]];
		nchip+=2;	/* Em/Rm and inject */
		nop++;
		for(msgin=compt->msg_in;msgin;msgin=msgin->next) {
			/* look for any channels */
			if (msgin->type == CHANNEL) {
				elm=msgin->src;
				nchip+=2;	/* Gk and Ek */
				nop+=2;
				for (cmsgin=elm->msg_in;cmsgin;cmsgin=cmsgin->next) {
					if (cmsgin->type == EK) {
						nchip--;	/* no Ek */
						break;
					}
				}
				if (strcmp(elm->object->name,"tabchannel")==0) {
					chan=(struct tab_channel_type *)elm;
					if ((chan->Xpower>6)||(chan->Ypower>6)||(chan->Zpower>6)) {
						fprintf(stderr,"Error: [XYZ]power larger than 6 not supported for chanmode >= 2\n");
					}
					k=store_name(hsolve,elm,&nnames,names);
					if (k>=0) nelm++;
					if (chan->X_A || chan->X_B) {
						nchip++;	/* X */
						nop+=3;
						if (check_chanX(chan,hsolve,tablist)) {
							fprintf(stderr,"Failed to set up X tabchannel\n");
							return(ERR);
						}
					}
					if (chan->Y_A || chan->Y_B) {
						nchip++;	/* Y */
						nop+=3;
						if (check_chanY(chan,hsolve,tablist)) {
							fprintf(stderr,"Failed to set up Y tabchannel\n");
							return(ERR);
						}
					}
					if (chan->Z_A || chan->Z_B) {
						nchip++;	/* Z */
						nop+=3;
						if (chan->Zpower > 0.0) {
						   /* concentration indexed */
							if (check_chanZ(chan,hsolve,ctablist)) {
								fprintf(stderr,"Failed to set up Z tabchannel\n");
								return(ERR);
						    }
						} else {
							/* voltage indexed */
							fprintf(stderr,"voltage index on Z gate not supported\n");
							return(ERR);
						}
					}
					HsolveBlock(elm);
					if (h_conc_init(hsolve,elm,0,i)) {
						return(ERR);
					}
					if ((h_has_output(elm))&&(hsolve->chanmode>2)) {
						nop++;
						nchip+=2;
					}
				} else if ((strcmp(elm->object->name,"channelC2")==0) ||
				           (strcmp(elm->object->name,"channelC3")==0)) {
					schan=(struct channelC3_type *)elm;
				    CallElement(elm,action);
					k=check_stab(schan,hsolve,stablist);
					if (k<0) {
						return(ERR);
					} else {
						schan->activation=k;	/* temporary store for index */
					}
					k=store_name(hsolve,elm,&nnames,names);
					if (k>=0) nelm++;
					HsolveBlock(elm);
					if (strcmp(elm->object->name,"channelC3")==0) {
						nchip+=3;		/* freq, X and Y */
					} else {
						nchip+=2;		/* X and Y */
					}
					nop+=2;
					if ((h_has_output(elm))&&(hsolve->chanmode>2)) {
						nop++;
						nchip+=2;
					}
				} else {	/* any other channel type */
					if (strcmp(msgin->src->object->name,"channelC")==0){
						fprintf(stderr,"Warning: hsolver cannot handle 'channelC' element '%s'. Ignoring it.\n",msgin->src->name);
					} else {
						if (h_conc_init(hsolve,msgin->src,0,i)) {
							return(ERR);
						}
						nop++;
					}
				}
				if (hsolve->useconcen && hsolve->concens[i]) {
					/* check if this channel adds to flux */
				  for (msgout=elm->msg_out;msgout;msgout=msgout->next) {
					if ((strcmp(msgout->dst->object->name,"Ca_concen")==0)||
						(strcmp(msgout->dst->object->name,"Ca_shell")==0)) {
						nop++;
						break;
					}
				  }
				}
			}
		}
		if (hsolve->useconcen && concens[i]) {
			HsolveBlock(concens[i]);
			nchip+=4;	/* Ca, Ca_base, tau and B */
			nop+=2;
			k=store_name(hsolve,concens[i],&nnames,names);
			if (k>=0) nelm++;
		}
		if (hsolve->usenernst && nernsts[i]) {
			HsolveBlock(nernsts[i]);
			nchip+=2;	/* constant and Cin/Cout */
			nop++;
			k=store_name(hsolve,nernsts[i],&nnames,names);
			if (k>=0) nelm++;
		}
	}

	if (duplicate_flag) {
	} else {
		duplicate_tables(hsolve,dt,tby2,tablist,ctablist,stablist);
	}
	ntab=hsolve->ntab;
	cntab=hsolve->cntab;

/* This second loop fills up the arrays allocated above */

#ifdef OLD
/* Here we duplicate the interpolation tables so that
** far memory accesses are minimized, and cache use maximized.  */
	ntab=hsolve->ntab;
	if (ntab) {
		tabsize=hsolve->xdivs+1;
		hsolve->tablist=(double *)calloc(ntab*tabsize,sizeof(double));
		for(i=0;i<ntab;i++) {
			for(j=0,k=i;j<tabsize;j++,k+=ntab) {
				hsolve->tablist[k]=1.0+tby2*tablist[i][j];
			}
			i++;
			for(j=0,k=i;j<tabsize;j++,k+=ntab) {
				hsolve->tablist[k]=dt*tablist[i][j];
			}
		}
	}
	cntab=hsolve->cntab;
	if (cntab) {
		tabsize=hsolve->cxdivs+1;
		hsolve->ctablist=(float *)calloc(cntab*tabsize,sizeof(float));
		for(i=0;i<cntab;i++) {
			for(j=0,k=i;j<tabsize;j++,k+=cntab) {
				hsolve->ctablist[k]=1.0+tby2*ctablist[i][j];
			}
			i++;
			for(j=0,k=i;j<tabsize;j++,k+=cntab) {
				hsolve->ctablist[k]=dt*ctablist[i][j];
			}
		}
	}
	if (hsolve->sntab) {
		hsolve->stablist=(double *)calloc(hsolve->sntab*SYNSIZE,sizeof(double));
		for(i=0;i<hsolve->sntab*SYNSIZE;i++) {
			hsolve->stablist[i]=stablist[i];
		}
	}
#endif

	hsolve->nchips=nchip;
	chip = hsolve->chip=(double *)calloc(nchip,sizeof(double));
	hsolve->nops=nop;
	ops = hsolve->ops=(int *)calloc(nop,sizeof(int));
	if (!hsolve->no_elminfo) {
		hsolve->nelms=nelm;
		hsolve->nelm_names=nnames;
		hsolve->chip_index=(int *)calloc(ncompts,sizeof(int));
		hsolve->ops_index=(int *)calloc(ncompts,sizeof(int));
		hsolve->comp_elms=(int *)calloc(ncompts,sizeof(int));
		hsolve->elm_names=(char **)calloc(nnames,sizeof(char *));
		hsolve->elm_index=(int *)calloc(3*nelm,sizeof(int));
		for (i=0;i<nnames;i++){
			hsolve->elm_names[i]=(char *)names[i];
		}
	}

/* This second loop fills up the arrays allocated above */
	nelm=nop=nchip=0;
	for (i=0;i<ncompts;i++){
		compt = compts[elmnum[i]];
		/* do first passive compartment structure (is also the boundary) */
		if (i==0) {
			ops[nop]=FCOMPT_OP;
		} else {
			ops[nop]=COMPT_OP;
		}
		if (!hsolve->no_elminfo) {
			hsolve->ops_index[i]=nop;
			hsolve->chip_index[i]=nchip;
			hsolve->comp_elms[i]=nelm;
		}
		h_check_msgs(hsolve,compt,COMPT_OP,i,nop,nchip);
		if (h_has_output(compt)) {
			h_store(hsolve,compt,COMPT_OP,i,nchip);
		}
		nop++;
		nchip+=2;
		/* initialize concen if present */
		if (hsolve->useconcen && concens[i]) {
			ops[nop]=CONC_VAL_OP;
			h_check_msgs(hsolve,concens[i],CONC_OP,i,nop,nchip);
			if (h_has_output(concens[i])) {
				h_store(hsolve,concens[i],CONC_OP,i,nchip);
			}
			nc=nchip;
			nop++;
			nchip+=2;
		}
		/* do nernsts, channels may depend on these values */
		if (hsolve->usenernst && nernsts[i]) {
		/* ne STRUCT */
			/* determine type of nernst channel */
			for(msgin=nernsts[i]->msg_in;msgin;msgin=msgin->next) {
				if ((msgin->type == CIN) || (msgin->type == COUT)) {
					if (msgin->src != hsolve->concens[i]) {
						fprintf(stderr,"Error: nernst expects other concen\n");
						return(ERR);
					}
					if (msgin->type == CIN) {
						op=ops[nop]=NERNST_IN_OP;
					} else {
						op=ops[nop]=NERNST_OUT_OP;
					}
				}
			}
			h_check_msgs(hsolve,nernsts[i],NERNST_IN_OP,i,nop,nchip);
			if (h_has_output(nernsts[i])) {
				h_store(hsolve,nernsts[i],op,i,nchip);
			}
			store_elm(hsolve,nernsts[i],&nelm,nop,nchip);
			nop++;
			nchip+=2;
		}
		/* to see if any chans are added into this compt */
		for(msgin=compt->msg_in;msgin;msgin=msgin->next) {
			/* look for any channels */
			if (msgin->type == CHANNEL) {
				elm=msgin->src;
				op=ops[nop]=CHAN_OP;
				nop++;
				ch=++nchip;
				if (strcmp(elm->object->name,"tabchannel")==0) {
					chan=(struct tab_channel_type *)elm;
					store_elm(hsolve,elm,&nelm,nop-1,nchip-1);
					h_check_msgs(hsolve,elm,CHAN_OP,i,nop,nchip);
					if (chan->X_A || chan->X_B) {
						ops[nop]=XGATE_OP;
						nop++;
						for(j=0;j<ntab;j+=2) {
							if (tablist[j]==chan->X_B->table) {
								ops[nop]=j;
								break;
							}
						}
						nop++;
						ops[nop]=(int)(chan->Xpower+1e-6);
						nop++;
						nchip++;
					}
					if (chan->Y_A || chan->Y_B) {
						ops[nop]=XGATE_OP;
						nop++;
						for(j=0;j<ntab;j+=2) {
							if (tablist[j]==chan->Y_B->table) {
								ops[nop]=j;
								break;
							}
						}
						nop++;
						ops[nop]=(int)(chan->Ypower+1e-6);
						nop++;
						nchip++;
					}
					if (chan->Z_A || chan->Z_B) {
						ops[nop]=ZGATE_OP;
						nop++;
						for(j=0;j<cntab;j+=2) {
							if (ctablist[j]==chan->Z_B->table) {
								ops[nop]=j;
								break;
							}
						}
						nop++;
						ops[nop]=(int)(chan->Zpower+1e-6);
						nop++;
						nchip++;
					}
				} else if ((strcmp(elm->object->name,"channelC2")==0) ||
				           (strcmp(elm->object->name,"channelC3")==0)) {
					op=SYN2_OP;
					store_elm(hsolve,elm,&nelm,nop-1,nchip-1);
					if (strcmp(elm->object->name,"channelC2")==0) {
						h_check_msgs(hsolve,elm,SYN2_OP,i,nop+1,nchip);
						ops[nop]=SYN2_OP;
						nchip+=2;
					} else {
						h_check_msgs(hsolve,elm,SYN2_OP,i,nop+1,nchip+1);
						ops[nop]=SYN3_OP;
						nchip+=3;
					}
					schan=(struct channelC3_type *)msgin->src;
					ops[nop+1]=schan->activation;
					nop+=2;
				} else {
					op=ops[nop-1]=GCHAN_OP;
					ops[nop]=(int )elm;
					nop++;
					nchip--;
				}
				hasEk=1;
				for (cmsgin=elm->msg_in;cmsgin;cmsgin=cmsgin->next) {
					if (cmsgin->type == EK) {
						hasEk=0;
						break;
					}
				}
				if (hasEk) {
					ops[nop]=ADD_EK_CURR_OP;
					nchip++;
				} else {
					ops[nop]=ADD_NERNST_CURR_OP;
				}
				nop++;
			    flux_scale=0.0;
				if (hsolve->useconcen && hsolve->concens[i]) {
					/* check if this channel adds to flux */
				  for (msgout=elm->msg_out;msgout;msgout=msgout->next) {
					if ((strcmp(msgout->dst->object->name,"Ca_concen")==0)||
						(strcmp(msgout->dst->object->name,"Ca_shell")==0)) {
						cmsgin = msgout->msg_in;
						switch (cmsgin->type) {
							case I_Ca :
								flux_scale=1.0;
								break;
							case fI_Ca:
								flux_scale=MSGVALUE(cmsgin,1);
								break;
							}
						}
				  }
				  if (flux_scale > 0) {
					  ops[nop]=POS_FLUX_OP;
					  nop++;
				  } else if (flux_scale < 0) {
					  ops[nop]=NEG_FLUX_OP;
					  nop++;
				  }
				}
				if ((op!=GCHAN_OP)&&(h_has_output(elm))&&(hsolve->chanmode>2)) {
					h_store(hsolve,elm,op,ch,nchip);
					ops[nop]=STORE_CURR_OP;
					nop++;
					nchip+=2;
				}
			}
		}
		if (hsolve->useconcen && concens[i]) {
			ops[nop]=CONC_OP;
			store_elm(hsolve,concens[i],&nelm,nc,nchip);
			nop++;
			nchip+=2;
		}
	}
	return(0);
}

/* Fills up hh and chan linklists according to hines numbering of parent compts */

int h_hh_init(hsolve)
	Hsolve	*hsolve;
{
	int 	ncompts;
	Element	**compts;
	Element	*compt;
	int 	i,j,k;
	MsgIn	*msgin;
	int		*elmnum;
	Cinfo	**hh;
	Cinfo	*hentry;
	Cinfo	**chan;
	Cinfo	*centry;

	ncompts = hsolve->ncompts;
	compts = hsolve->compts;
	elmnum = hsolve->elmnum;
	hh = hsolve->hh = (Cinfo **) calloc(ncompts,sizeof(Cinfo *));
	chan = hsolve->chan = (Cinfo **) calloc(ncompts,sizeof(Cinfo *));

	/* CHANNELs go from channels to compts */
	for (i=0;i<ncompts;i++){
		compt = compts[elmnum[i]];
		for(msgin=compt->msg_in;msgin;msgin=msgin->next) {
			/* look for any channels */
			if (msgin->type == CHANNEL) {
				if (strcmp(msgin->src->object->name,"tabchannel")==0) {
					hentry = (Cinfo *)calloc(1,sizeof(Cinfo));
					hentry->next = hh[i];
					hh[i] = hentry;
					hentry->chan = msgin->src;
					hentry->flux_scale = 0.0;
					hentry->nernst = 0;
					HsolveBlock(hentry->chan);
					if (h_conc_init(hsolve,msgin->src,hentry,i)) {
						return(ERR);
					}
				} else {
					if (strcmp(msgin->src->object->name,"channelC")==0){
						fprintf(stderr,"Warning: hsolver cannot handle 'channelC' element '%s'. Ignoring it.\n",msgin->src->name);
						continue;
					}
					centry = (Cinfo *)calloc(1,sizeof(Cinfo));
					centry->next = chan[i];
					chan[i] = centry;
					centry->chan = msgin->src;
					centry->flux_scale = 0.0;
					centry->nernst = 0;
					if (h_conc_init(hsolve,msgin->src,centry,i)) {
						return(ERR);
					}
				}
			}
		}
		if (hsolve->usenernst && hsolve->nernsts[i]) {
			for(msgin=hsolve->nernsts[i]->msg_in;msgin;msgin=msgin->next) {
				if ((msgin->type == CIN) || (msgin->type == COUT)) {
					if (msgin->src != hsolve->concens[i]) {
						fprintf(stderr,"Error: nernst expects other concen\n");
						return(ERR);
					}
					if (hsolve->usenernst == 1) {
						hsolve->nernstmsg[i] = msgin->type;
					} else if (hsolve->usenernst != (msgin->type + 2)) {
						fprintf(stderr,"Error: nernst with wrong msg\n");
						return(ERR);
					}
				}
			}
			HsolveBlock(hsolve->nernsts[i]);
		}
		if (hsolve->useconcen && hsolve->concens)
			HsolveBlock(hsolve->concens[i]);
	}
	return(0);
}

/* Fills up conc and nernst arrays according to hines numbering of parent compts
** and the channels that connect to them.  If hentry !=0 called from h_hh_init,
** if (hentry == 0) called from h_hh_chip_init. */
int h_conc_init(hsolve,chan,hentry,comptno)
    Hsolve  *hsolve;
    Element	*chan;
    Cinfo   *hentry;
    int comptno;
{
    MsgIn   *msgin;
    MsgOut  *msgout;
	Element **concens;
	Element **nernsts;
    Element *conc1 = NULL,*conc2 = NULL;
    Element *nernst = NULL;
	double  flux_scale;
	Action  *action;
    struct tab_channel_type *tchan;

	if (!hsolve->useconcen)
		return(0);
    concens = hsolve->concens;
	nernsts = hsolve->nernsts;
	action = GetAction("RESET");

	for (msgin=chan->msg_in;msgin;msgin=msgin->next) {
		if ((strcmp(msgin->src->object->name,"Ca_concen")==0) ||
			(strcmp(msgin->src->object->name,"Ca_shell")==0)) {
			/* We have a conc as the source of the msg */
			conc1= msgin->src;
		}
		if (msgin->type == EK) {
			/* Make sure that we have a nernst element */
			if (strcmp(msgin->src->object->name,"nernst")==0) {
				nernst = msgin->src;
			} else {
				fprintf(stderr,"Error: EK message from unsupported object");
				return(ERR);
			}
		}
	}
	if (strcmp(chan->object->name,"tabchannel")==0) {
		tchan=(struct tab_channel_type *)chan;
		if ((fabs(tchan->Zpower) > TINY) && (!conc1)) {
			fprintf(stderr,"Error: missing CONC msg for tabchannel %s Z-gate\n", tchan->name);
			return(ERR);
		}
	}
	if (hsolve->usenernst && nernst) {
		if (nernsts[comptno]) {
			if (nernst != (nernsts[comptno])) {
				fprintf(stderr,"Error: more than one nernst element\n");
				return(ERR);
			}
		} else {
			nernsts[comptno]=nernst;
			CallElement(nernst,action);  /* we may need the nernst->constant */
		}
		if (hentry) {
			hentry->nernst = 1;
		}
	}
	for (msgout=chan->msg_out;msgout;msgout=msgout->next) {
		if ((strcmp(msgout->dst->object->name,"Ca_concen")==0) ||
			(strcmp(msgout->dst->object->name,"Ca_shell")==0)) {
			conc2 = msgout->dst;
			msgin = msgout->msg_in;
			switch (msgin->type) {
				case I_Ca :
					flux_scale = 1.0;
					break;
				case fI_Ca:
					flux_scale = MSGVALUE(msgin,1);
					break;
			}
			if (hentry) { /* h_hh_init */
				hentry->flux_scale = flux_scale;
			} else { /* h_hh_chip_init */
				if ((fabs(flux_scale) - 1.0) > TINY) {
					fprintf(stderr,"Error: fI_Ca scale value not supported\n",msgin->type,msgout->dst->name);
					return(ERR);
				}
			}
		}
	}
	if (   (concens[comptno]) &&
				((conc1 && (conc1 != concens[comptno])) ||
				 (conc2 && (conc2 != concens[comptno])))
		|| (conc1 && conc2 && (conc1 != conc2))) {
			fprintf(stderr,"Error: more than one concen element\n");
			return(ERR);
	}
	if (conc1) {
		concens[comptno]=conc1;
	} else if (conc2) {
		concens[comptno]=conc2;
	}
	return(0);
}


duplicate_tables(hsolve,dt,tby2,tablist,ctablist,stablist)
	Hsolve	*hsolve;
	double	dt,tby2;
	double	**tablist,**ctablist;
	double	*stablist;
{
	int i,j,k;
	int tabsize,ntab,cntab;


/* Here we duplicate the interpolation tables so that
** far memory accesses are minimized, and cache use maximized.  */
	ntab=hsolve->ntab;
	if (ntab>0) {
		tabsize=hsolve->xdivs+1;
#ifdef TAB_FLOAT
		hsolve->tablist=(float *)calloc(ntab*tabsize,sizeof(float));
#else
		hsolve->tablist=(double *)calloc(ntab*tabsize,sizeof(double));
#endif
		for(i=0;i<ntab;i++) {
			for(j=0,k=i;j<tabsize;j++,k+=ntab) {
				hsolve->tablist[k]=1.0+tby2*tablist[i][j];
			}
			i++;
			for(j=0,k=i;j<tabsize;j++,k+=ntab) {
				hsolve->tablist[k]=dt*tablist[i][j];
			}
		}
	}
	cntab=hsolve->cntab;
	if (cntab>0) {
		tabsize=hsolve->cxdivs+1;
#ifdef TAB_FLOAT
		hsolve->ctablist=(float *)calloc(cntab*tabsize,sizeof(float));
#else
		hsolve->ctablist=(double *)calloc(cntab*tabsize,sizeof(double));
#endif
		for(i=0;i<cntab;i++) {
			for(j=0,k=i;j<tabsize;j++,k+=cntab) {
				hsolve->ctablist[k]=1.0+tby2*ctablist[i][j];
			}
			i++;
			for(j=0,k=i;j<tabsize;j++,k+=cntab) {
				hsolve->ctablist[k]=dt*ctablist[i][j];
			}
		}
	}
	if (hsolve->sntab>0) {
		hsolve->stablist=(double *)calloc(hsolve->sntab*SYNSIZE,sizeof(double));
		for(i=0;i<hsolve->sntab*SYNSIZE;i++) {
			hsolve->stablist[i]=stablist[i];
		}
	}
}
