static char rcsid[] = "$Id: difpool.c,v 1.1 1992/12/11 19:02:39 dhb Exp $";

/*
** $Log: difpool.c,v $
** Revision 1.1  1992/12/11  19:02:39  dhb
** Initial revision
**
*/

#include "conc_ext.h"

/* Implementation of a concentration pool can have miscellaneous 
**	in/flows (controlled by	other process) and one-dimensional 
**	diffusion to other difpools.
** It makes assumptions about the shape of the difpools.  The
**  cross-section over which diffusion occurs should equal the
**  volume (Vol) of the pool divided by the length (Len) along 
**  the axis of diffusion.
** This scheme works fine for single step integration methods,
**	multi-step methods will not be as accurate because changes in
**	variables outside this object (e.g. Bfree in poolbuffers) are
**	invisible to the integrator till the next message */

/* 1/91 Erik De Schutter, Caltech */
/* Corrected 8/91, error in Ad calculation, EDS */

DifPool(pool,action)
register struct difpool_type *pool;
Action		*action;
{
double		A = 0.0;		/* integrate dCo/dt= A - B*Co */
double		B = 0.0;
double		dt;
double		lsum;	/* dum of pool lengths */
double		L;		/* ratio of lenghts */
double		Ad;		/* coupling coefficient */
double		DV;		/* composite diffusion coefficient */
MsgIn		*msg;
double		savedata[2];
int			n;

    if(debug > 1){
	ActionHeader("DifPool",pool,action);
    }

    SELECT_ACTION(action){
    case INIT:
	pool->previous_state = pool->Co;
	break;

    case PROCESS:
	dt = Clockrate(pool);
	/* Read the msgs.  */
	MSGLOOP(pool,msg){
	    case 0:		/* in/outflow, not time-dependent */
				/* 0 = dCo */
		A += MSGVALUE(msg,0);
		break;
	    case 1:		/* current in/outflow */
				/* 0 = charge */
				/* 1 = I */
		/* d[Co]/dt = I/(e*Faraday*Vol) */
		A += MSGVALUE(msg,1)/(MSGVALUE(msg,0)*96494*pool->Vol);
		break;
	    case 2:		/* diffusion */
				/* 0 = Len */
				/* 1 = Vol */
				/* 2 = Co */
		/* dCo/dt = -D/Vi*((AiLj+AjLi)/(Li+Lj)^2)*(Coi-Coj)
		** We assume that Ai=Vi/Li
		** L = L0/L1
		** Ad = (Vol0/L + Vol1*L) / (Len0 + Len1)^2
		** DV =  2 * D / Vol0 * Ad */
		lsum = pool->Len + MSGVALUE(msg,0);
		L = pool->Len / MSGVALUE(msg,0);
		Ad = (pool->Vol/L + MSGVALUE(msg,1)*L) / (lsum * lsum);
		DV = 2 * pool->D / pool->Vol * Ad;
		A += DV * MSGVALUE(msg,2);
		B += DV;
		break;
	    case 3:		/* interaction with buffer */
				/* 0 = kBf */
				/* 1 = kBb */
				/* 2 = Btot */
				/* 3 = Bfree */
		/* dCo/dt = - kBf*Co*Bfree + kBb*(Btot-Bfree) */
		/* DV = Bfree */
		DV = MSGVALUE(msg,3);
		A += MSGVALUE(msg,1)*(MSGVALUE(msg,2)-DV);
		B += MSGVALUE(msg,0)*DV;
		break;
	    case 4:		/* interaction with pump */
				/* 0 = kPump */
				/* 1 = Ceq */
		/* dCo/dt = -kPump*(Co-Ceq) */
		/* DV = kPump */
		DV = MSGVALUE(msg,0);
		A += DV*MSGVALUE(msg,1);
		B += DV;
		break;
	}
	
	/* Integrate to get the net pool concentration */ 
	pool->Co = IntegrateMethod(pool->object->method,pool,pool->Co,A,B,dt,"Co");
	/* Do not allow Co to get lower tham Cmin, note that this
	** limit is only felt indirectly by poolbuffers! */
	if(pool->Co < pool->Cmin) {
	    pool->Co = pool->Cmin;
	}
	break;

    case RESET:
	pool->Co = pool->Cmin;
	break;

    case CHECK:
	/* minimum concentration */
	if(pool->Cmin < 0.0){
	    ErrorMessage("Pool", "Invalid Cmin.", pool);
	}
	/* scaled diffusion constant */
	if(pool->D <= 0.0){
	    ErrorMessage("Pool", "Invalid D.", pool);
	}
	/* length */
	if(pool->Len <= 0.0){
	    ErrorMessage("Pool", "Invalid Len.", pool);
	}
	/* cross volume */
	if(pool->Vol <= 0.0){
	    ErrorMessage("Pool", "Invalid Vol.", pool);
	}
	MSGLOOP(pool,msg){
	    case 2:			/* incoming diffusion */
		if((MSGVALUE(msg,0) <= 0.0)||(MSGVALUE(msg,1) <= 0.0)){
	    	ErrorMessage("Pool","Incoming diffusion data invalid.",pool);
		}
		break;
	    case 4:			/* pump */
		if((MSGVALUE(msg,0) <= 0.0)||(MSGVALUE(msg,1) < 0.0)){
	    	ErrorMessage("Pool","Incoming pump data invalid.",pool);
		}
		break;
	}
	break;

    case SAVE2:
	savedata[0] = pool->Co;
	savedata[1] = pool->previous_state;
	/* action->data contains the file pointer */
	fwrite(savedata,sizeof(double),2,action->data);
	break;

    case RESTORE2:
	/* action->data contains the file pointer */
	fread(savedata,sizeof(double),2,action->data);
	pool->Co = savedata[0];
	pool->previous_state = savedata[1];
	break;

    case SETUP:
#ifdef LATER
	/* compute how many state variables are coupled */
	/* we assume that if kPump is not constant, it
	**  nevertheless changes very slowly */
	n = 0;
	MSGLOOP(pool,msg){
	    case 2:			/* diffusion coupling */
	    case 3:			/* buffer coupling */
		n++;
		break;
	}
	matrixmsg = SetupMatrixMsg(pool,matrixsolver,count);
	matrixmsg->svid = &Co;
	matrixmsg->variable_coeff = 1;
	MSGLOOP(pool,msg){
	    case 2:			/* diffusion coupling */
		AddSVToMsg(matrixmsg,MSGPTR(msg,2));
	    case 3:			/* buffer coupling */
		AddSVToMsg(matrixmsg,MSGPTR(msg,3));
	}
#endif
	break;
    case RESULTS:
	break;
    }
}

