
//genesis
/* channel.g


	functions defined in this script
=============================================================================
	FUNCTION NAME		ARGUMENTS
=============================================================================
	makechannel		(compartment,channel,Ek,tau1,tau2,gmax)
	make_hhNa		(compartment,channel,activearea,ENa,Erest)
	make_hhK		(compartment,channel,activearea,EK,Erest)
=============================================================================
*/


//=================================================
//       BASIC TRANSMITTER ACTIVATED CHANNEL
//=================================================
function makechannel(compartment,channel,Ek,tau1,tau2,gmax)
    str compartment
    str channel
    float Ek				// mV
    float tau1,tau2			// msec
    float gmax				// mS

    create	channelC2		{compartment}/{channel}
    set		^ \
		Ek			{Ek} \	
		tau1			{tau1} \ 
		tau2			{tau2} \ 
		gmax			{gmax}
    link_channel2 	{compartment}/{channel} 	{compartment}
end
//
// The Hodgkin-Huxley voltage activated channels are implemented with
// vdep_channel and vdep_gate objects using parameters taken from
// A.L.Hodgkin and A.F.Huxley, J.Physiol(Lond) 117, pp 500-544 (1952)
//
//=================================================
//                 HH NA CHANNEL
//=================================================
function make_hhNa(compartment,channel,activearea,ENa,Erest)
    str compartment
    str channel
    float activearea		// cm^2
    float ENa,Erest		// mV

    create	vdep_channel	{compartment}/{channel}
    set	{compartment}/{channel} \
	    Ek 			{ENa} \			// mV
	    gbar		{120.0*activearea} 	// mmhos/cm^2
    create	vdep_gate 	{compartment}/{channel}/m
	set		^	alpha_A		 {0.1*(25.0 + Erest)}  \
		            	alpha_B		 -0.1	\
				alpha_C		 -1.0	\
				alpha_D		 {-1.0*(25.0 + Erest)} \
				alpha_F		-10.0	\
		            	beta_A		  4.0	\
		            	beta_B		  0.0	\
				beta_C		  0.0	\
				beta_D		 {-1.0*(0.0 + Erest)} \
				beta_F		 18.0	\
				instantaneous	0

    create	vdep_gate 	{compartment}/{channel}/h
	set		^	alpha_A		  0.07	\
		            	alpha_B		  0.0	\
				alpha_C		  0.0	\
				alpha_D	  	{-1.0*(0.0 + Erest)} \
				alpha_F		 20.0	\
		            	beta_A		  1.0	\
            			beta_B		  0.0	\
				beta_C		  1.0	\
				beta_D		{-1.0*(30.0 + Erest)} \
				beta_F		-10.0	\
				instantaneous	0

    sendmsg {compartment}/{channel}/m  {compartment}/{channel}  MULTGATE m 3
    sendmsg {compartment}/{channel}/h  {compartment}/{channel}  MULTGATE m 1
//
// This comment in squid_funcs.g seems to imply that the order of these
// messages is important:
//    Gates need an initial voltage (RESET) and the current voltage (PROCESS):
//    We'll send them the compartment voltage in both cases,
//    But we'll have to make sure that the compartment is RESET
//    before the gates are RESET so that the initial value is correct.
//
// Actually, this is has to be accomplished by a kludged reset function,
// reset_kludge defined in my forms.g, similar to the one in squid_forms.g
//
    sendmsg {compartment} {compartment}/{channel}   VOLTAGE Vm
    sendmsg {compartment} {compartment}/{channel}/m VOLTAGE Vm
    sendmsg {compartment} {compartment}/{channel}/h VOLTAGE Vm
    sendmsg {compartment} {compartment}/{channel}/m EREST Vm
    sendmsg {compartment} {compartment}/{channel}/h EREST Vm
    sendmsg {compartment}/{channel} {compartment}   CHANNEL Gk Ek
end

//=================================================
//                  HH K CHANNEL
//=================================================
function make_hhK(compartment,channel,activearea,EK,Erest)
    str compartment
    str channel
    float activearea
    float EK,Erest

    create	vdep_channel	{compartment}/{channel}
    set		^ \
		Ek 		{EK} \			// mV
 		gbar		{ 36.0 * activearea }	// mmhos/cm^2 
    create	vdep_gate 	{compartment}/{channel}/n
	set		^	alpha_A		 {0.01*(10.0 + Erest)}	\
		            	alpha_B		 -0.01	\
				alpha_C		 -1.0	\
				alpha_D		 {-1.0*(10.0 + Erest)}	\
				alpha_F		-10.0	\
		            	beta_A		  0.125	\
		            	beta_B		  0.0	\
				beta_C		  0.0	\
				beta_D		 {-1.0*(0.0 + Erest)}	\
				beta_F		 80.0	\
				instantaneous	0
    sendmsg {compartment}/{channel}/n  {compartment}/{channel}  MULTGATE m 4
    sendmsg {compartment} {compartment}/{channel}   VOLTAGE Vm
    sendmsg {compartment} {compartment}/{channel}/n VOLTAGE Vm
    sendmsg {compartment} {compartment}/{channel}/n EREST Vm
    sendmsg {compartment}/{channel} {compartment}   CHANNEL Gk Ek
end
