// genesis
/* FILE INFORMATION
** Molluscan Voltage dependent Na, K, KA, Ca, slow Na/Ca channels
** The channels given below are representative of those found in
** molluscan soma described in the review paper by
** David J. Adams, Stephen J. Smith and Stuart H. Thompson
** (Ann. Rev. Neurosci. 3, p.141 (1980)) and in
** the references given below.
** Implemented by David Beeman - January 1991
** Converted to tabchans by Upinder S. Bhalla - Feb 1991
** All parameters are in SI (MKSA) units.
** Voltages are in mV, and are for the range -100 mV to +50 mV.
** The resting membrane potential is -0.040 V.
**
**
** Typical equilibrium potentials (specified within the functions) are:
**        ENa             =        0.050
**        EK              =       -0.060
**        EA              =       -0.063
**        ECa             =        0.064
**        EB              =        0.068  // +/- 0.059 V uncertainty!
**
** This file depends on functions and constants defined in defaults.g
*/

// CONSTANTS
float        ENa             =        0.050
float        EK              =       -0.060
float        EA              =       -0.063
float        ECa             =        0.064
float        EB              =        0.068  // +/- 0.059 V uncertainty!
float		 SOMA_A			 =		  1e-9		// sq m


/***********************************************************************
                            Na-Current
 Adams and Gage, J. Physiol. 289, p. 143 (1979)  (Aplysia R15)
 (time constant for inactivation was taken from Connor and Stevens)
***********************************************************************/

function make_Na_aplysia_ag	// I-current (Na)
    if({exists(Na_aplysia_ag)})
        return
    end

    create  tabchannel  Na_aplysia_ag
    set  Na_aplysia_ag  \
         Ek     {ENa}             \
         Gbar   {1200.0 * SOMA_A} \
         Ik     0                 \
         Gk     0					\
		 Xpower	3	\
		 Ypower	1	\
		 Zpower	0

    call    Na_aplysia_ag    TABCREATE X 30 -0.100 0.050
	set		Na_aplysia_ag	\
		X_A->table[0]	0.0		X_B->table[0]	82.0	\	// -0.1
		X_A->table[1]	0.0		X_B->table[1]	82.0	\
		X_A->table[2]	0.0		X_B->table[2]	82.0	\	// -0.09
		X_A->table[3]	0.0		X_B->table[3]	82.0	\
		X_A->table[4]	0.0		X_B->table[4]	82.0	\	// -0.08
		X_A->table[5]	0.0		X_B->table[5]	82.0	\
		X_A->table[6]	0.0		X_B->table[6]	82.0	\	// -0.07
		X_A->table[7]	0.0		X_B->table[7]	82.0	\
		X_A->table[8]	0.0		X_B->table[8]	82.0	\	// -0.06
		X_A->table[9]	0.0		X_B->table[9]	82.0	\
		X_A->table[10]	0.0		X_B->table[10]	82.0	\	// -0.05
		X_A->table[11]	0.0		X_B->table[11]	82.0	\
		X_A->table[12]	0.0		X_B->table[12]	82.0	\	// -0.04
		X_A->table[13]	0.0		X_B->table[13]	82.0	\
		X_A->table[14]	0.0		X_B->table[14]	82.0	\	// -0.030
		X_A->table[15]	26.3	X_B->table[15]	81.0	\
		X_A->table[16]	52.0	X_B->table[16]	78.0	\	// -0.020
		X_A->table[17]	90.0	X_B->table[17]	72.0	\
		X_A->table[18]	190.0	X_B->table[18]	60.0	\	// -0.010
		X_A->table[19]	400.0	X_B->table[19]	40.0	\
		X_A->table[20]	615.0	X_B->table[20]	22.0	\	// 0.00
		X_A->table[21]	807.0	X_B->table[21]	12.0	\
		X_A->table[22]	1004.0	X_B->table[22]	6.0		\	// 0.010
		X_A->table[23]	1315.0	X_B->table[23]	0.0		\
		X_A->table[24]	1429.0	X_B->table[24]	0.0		\	// 0.020
		X_A->table[25]	1923.0	X_B->table[25]	0.0		\
		X_A->table[26]	2000.0	X_B->table[26]	0.0		\	// 0.030
		X_A->table[27]	2083.0	X_B->table[27]	0.0		\
		X_A->table[28]	2128.0	X_B->table[28]	0.0		\	// 0.040
		X_A->table[29]	2128.0	X_B->table[29]	0.0		\
		X_A->table[30]	2128.0	X_B->table[30]	0.0			// 0.050

		/* Setting the calc_mode to NO_INTERP for speed */
		set Na_aplysia_ag X_A->calc_mode 0 X_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(Na_aplysia_ag,X)

		/* Filling the tables using B-SPLINE interpolation */
		call Na_aplysia_ag TABFILL X 3000 0

    call    Na_aplysia_ag    TABCREATE Y 30 -0.100 0.050
	set		Na_aplysia_ag	\
		Y_A->table[0]	12.5	Y_B->table[0]	0.0		\	// -0.1
		Y_A->table[1]	12.5	Y_B->table[1]	0.0		\
		Y_A->table[2]	12.5	Y_B->table[2]	0.0		\	// -0.09
		Y_A->table[3]	12.5	Y_B->table[3]	0.0		\
		Y_A->table[4]	12.5	Y_B->table[4]	0.0		\	// -0.08
		Y_A->table[5]	12.5	Y_B->table[5]	0.0		\
		Y_A->table[6]	12.5	Y_B->table[6]	0.0		\	// -0.07
		Y_A->table[7]	12.5	Y_B->table[7]	0.0		\
		Y_A->table[8]	12.5	Y_B->table[8]	0.0		\	// -0.06
		Y_A->table[9]	12.5	Y_B->table[9]	0.0		\
		Y_A->table[10]	12.5	Y_B->table[10]	0.025	\	// -0.05
		Y_A->table[11]	12.3	Y_B->table[11]	0.16	\
		Y_A->table[12]	11.7	Y_B->table[12]	0.81	\	// -0.04
		Y_A->table[13]	9.14	Y_B->table[13]	3.36	\
		Y_A->table[14]	4.52	Y_B->table[14]	8.81	\	// -0.030
		Y_A->table[15]	1.62	Y_B->table[15]	16.35	\
		Y_A->table[16]	0.64	Y_B->table[16]	35.1	\	// -0.020
		Y_A->table[17]	0.0		Y_B->table[17]	75.7	\
		Y_A->table[18]	0.0		Y_B->table[18]	160.0	\	// -0.010
		Y_A->table[19]	0.0		Y_B->table[19]	160.0	\
		Y_A->table[20]	0.0		Y_B->table[20]	160.0	\	// 0.00
		Y_A->table[21]	0.0		Y_B->table[21]	160.0	\
		Y_A->table[22]	0.0		Y_B->table[22]	160.0	\	// 0.010
		Y_A->table[23]	0.0		Y_B->table[23]	160.0	\
		Y_A->table[24]	0.0		Y_B->table[24]	160.0	\	// 0.020
		Y_A->table[25]	0.0		Y_B->table[25]	160.0	\
		Y_A->table[26]	0.0		Y_B->table[26]	160.0	\	// 0.030
		Y_A->table[27]	0.0		Y_B->table[27]	160.0	\
		Y_A->table[28]	0.0		Y_B->table[28]	160.0	\	// 0.040
		Y_A->table[29]	0.0		Y_B->table[29]	160.0	\
		Y_A->table[30]	0.0		Y_B->table[30]	160.0		// 0.050

		/* Setting the calc_mode to NO_INTERP for speed */
		set Na_aplysia_ag Y_A->calc_mode 0 Y_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(Na_aplysia_ag,Y)

		/* Filling the tables using B-SPLINE interpolation */
		call Na_aplysia_ag TABFILL Y 3000 0
end

/**********************************************************************
**                      Late potassium K-Current
**	K current activation from Thompson, J. Physiol 265, 465 (1977)
**	(Tritonia (LPl	2 and LPl 3 cells)
** Inactivation from RW Aldrich, PA Getting, and SH Thompson, 
** J. Physiol, 291, 507 (1979)
**
**********************************************************************/
function make_K_trit_agt  // K-current     
    if(exists(K_trit_agt))
        return
    end

    create  tabchannel  K_trit_agt
    set  K_trit_agt  \
         Ek     {EK}              \
         Gbar   {1200 * SOMA_A}   \
         Ik     0                 \
         Gk     0					\
		 Xpower	2					\
		 Ypower	1					\
		 Zpower	0

    call    K_trit_agt    TABCREATE X 30 -0.100 0.050
    settab2const(K_trit_agt, X_A, 0, 12, 0.0)	//-0.1 thru -0.045=>0.0
    set		K_trit_agt \
    	X_A->table[13] 	0.00 \       
    	X_A->table[14]	2.87 \	// -0.030
    	X_A->table[15]	4.68 \
    	X_A->table[16]	7.46 \	// -0.020
    	X_A->table[17]	10.07 \
    	X_A->table[18]	14.27 \	// -0.010
    	X_A->table[19]	17.87 \
    	X_A->table[20]	22.9 \	// 0.0
    	X_A->table[21]	33.6 \
    	X_A->table[22]	49.3 \	// 0.010
    	X_A->table[23]	65.6 \
    	X_A->table[24]	82.0 \	// 0.020
    	X_A->table[25]	110.0 \	     
    	X_A->table[26]	147.1 \	// 0.030
    	X_A->table[27]	147.1 \ 	     
    	X_A->table[28]	147.1 \	// 0.040
    	X_A->table[29]	147.1 \ 	     
    	X_A->table[30]	147.1	// 0.050

    settab2const(K_trit_agt, X_B, 0, 11, 16.6)	//-0.1 thru -0.045=>16.6
    set		K_trit_agt \
		X_B->table[12]	 	16.6	\	// -0.040 Volts
    	X_B->table[13] 		15.4	\
    	X_B->table[14]		13.5	\	// -0.030
    	X_B->table[15]		13.2	\
    	X_B->table[16]		11.9	\	// -0.020
    	X_B->table[17]		11.5	\
    	X_B->table[18]		10.75	\	// -0.010
    	X_B->table[19]		9.30	\
    	X_B->table[20]		8.30	\	// 0.00
    	X_B->table[21]		6.00	\
    	X_B->table[22]		5.10	\	// 0.010
    	X_B->table[23]		4.80	\
    	X_B->table[24]		3.20	\	// 0.020
    	X_B->table[25]		1.60	\
    	X_B->table[26]		0.00	\	// 0.030
    	X_B->table[27]		0.00	\
    	X_B->table[28]		0.00	\	// 0.040
    	X_B->table[29]		0.00	\
    	X_B->table[30]		0.00		// 0.050

		/* Setting the calc_mode to NO_INTERP for speed */
		set K_trit_agt X_A->calc_mode 0 X_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(K_trit_agt,X)

		/* Filling the tables using B-SPLINE interpolation */
		call K_trit_agt TABFILL X 3000 0


    call    K_trit_agt    TABCREATE Y 30 -0.100 0.050
    settab2const(K_trit_agt, Y_A, 0, 11, 1.0) //-0.1 thru -0.035 => 1.0
    set		K_trit_agt \
		Y_A->table[12]	1.00	\	// -0.040	Volts
		Y_A->table[13]	0.97	\	// 
		Y_A->table[14]	0.94	\	// -0.030	Volts
    	Y_A->table[15]	0.88	\
    	Y_A->table[16]	0.75	\	// -0.020
    	Y_A->table[17]	0.61	\
    	Y_A->table[18]	0.43	\	// -0.010
    	Y_A->table[19]	0.305	\ 
    	Y_A->table[20]	0.220	\ 	// 0.00
    	Y_A->table[21]	0.175	\
    	Y_A->table[22]	0.155	\	// 0.010
    	Y_A->table[23]	0.143	\
    	Y_A->table[24]	0.138	\	// 0.020
    	Y_A->table[25]	0.137	\ 	     
    	Y_A->table[26]	0.136	\	// 0.030
    	Y_A->table[27]	0.135	\ 	     
    	Y_A->table[28]	0.135	\	// 0.040
    	Y_A->table[29]	0.135	\ 	     
    	Y_A->table[30]	0.135		// 0.050

    settab2const(K_trit_agt, Y_B, 0, 11, 0.0) //-0.1 thru -0.045 => 0.0
    set		K_trit_agt	\
		Y_B->table[12]	0.0 	\	// -0.040	Volts
		Y_B->table[13]	0.03	\	//
		Y_B->table[14]	0.06	\	// -0.030	Volts
    	Y_B->table[15]	0.12	\ 
    	Y_B->table[16]	0.25	\ 	// -0.020
    	Y_B->table[17]	0.39	\ 
    	Y_B->table[18]	0.57	\ 	// -0.010
    	Y_B->table[19]	0.695	\ 
    	Y_B->table[20]	0.78	\ 	// 0.00
    	Y_B->table[21]	0.825	\ 
    	Y_B->table[22]	0.845	\	// 0.010
    	Y_B->table[23]	0.857	\
    	Y_B->table[24]	0.862	\	// 0.020
    	Y_B->table[25]	0.863	\      
    	Y_B->table[26]	0.864	\	// 0.030
    	Y_B->table[27]	0.865	\
    	Y_B->table[28]	0.865	\	// 0.040
    	Y_B->table[29]	0.865	\      
    	Y_B->table[30]	0.865		// 0.050

		/* Setting the calc_mode to NO_INTERP for speed */
		set K_trit_agt Y_A->calc_mode 0 Y_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(K_trit_agt,Y)

		/* Filling the tables using B-SPLINE interpolation */
		call K_trit_agt TABFILL Y 3000 0

end

/**********************************************************************
**                 Ca-current
**	Adams and Gage, J. Physiol 289 p143	(1979) (Aplysia R15)
**********************************************************************/

function make_Ca_aplysia_ag  // A-current
    if(exists(Ca_aplysia_ag))
        return
    end

    create  tabchannel  Ca_aplysia_ag
    set  Ca_aplysia_ag  \
         Ek     {ECa}              \
         Gbar   {1200 * SOMA_A} \
         Ik     0                 \
         Gk     0		\
		 Xpower	2		\
		 Ypower	1		\
		 Zpower	0

    call    Ca_aplysia_ag    TABCREATE X 30 -0.100 0.050
    settab2const(Ca_aplysia_ag, X_A, 0, 18, 0.0)//-0.1 thru -0.005 => 0.0
    set		Ca_aplysia_ag	\
		X_A->table[19]	0.0 	\	// -0.005 Volts
    	X_A->table[20]	39.2	\	// 0.0
    	X_A->table[21]	46.7	\   // 0.005
    	X_A->table[22]	54.2	\	// 0.010
    	X_A->table[23]	85.3	\   // 0.015
    	X_A->table[24]	111.9	\	// 0.020
    	X_A->table[25]	128.2	\	// 0.025
    	X_A->table[26]	144.6	\	// 0.030
    	X_A->table[27]	156.7	\	// 0.035
    	X_A->table[28]	166.7	\	// 0.040
    	X_A->table[29]	166.7	\	// 0.045
    	X_A->table[30]	166.7		// 0.050

    settab2const(Ca_aplysia_ag,X_B,0,17,250.0)	//-0.1 thru -0.01 =>250
    set		Ca_aplysia_ag	\
		X_B->table[18]	250.0 	\	// -0.010 Volts
		X_B->table[19]	217.0 	\	// -0.005 Volts
    	X_B->table[20]	147.0	\	// 0.0
    	X_B->table[21]	103.9	\   // 0.005
    	X_B->table[22]	64.4	\	// 0.010
    	X_B->table[23]	26.5	\   // 0.015
    	X_B->table[24]	8.55	\	// 0.020
    	X_B->table[25]	2.35	\	// 0.025
    	X_B->table[26]	0.58	\	// 0.030
    	X_B->table[27]	0.0		\	// 0.035
    	X_B->table[28]	0.0		\	// 0.040
    	X_B->table[29]	0.0		\	// 0.045
    	X_B->table[30]	0.0			// 0.050

		/* Setting the calc_mode to NO_INTERP for speed */
		set Ca_aplysia_ag X_A->calc_mode 0 X_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(Ca_aplysia_ag,X)

		/* Filling the tables using B-SPLINE interpolation */
		call Ca_aplysia_ag TABFILL X 3000 0


    call    Ca_aplysia_ag    TABCREATE Y 30 -0.100 0.050
    settab2const(Ca_aplysia_ag, Y_A, 0, 11, 10.0) // -0.1 to -0.04=>10.0
    set		Ca_aplysia_ag	\
		Y_A->table[12]		10.0	\ // -0.040 Volts
    	Y_A->table[13]		9.7		\
    	Y_A->table[14]		9.24	\ // -0.030
    	Y_A->table[15]		7.5		\
    	Y_A->table[16]		5.0		\ // -0.020
    	Y_A->table[17]		2.5		\
    	Y_A->table[18]		0.633	\ // -0.010
    	Y_A->table[19]		0.153	\
    	Y_A->table[20]		0.004	\	// 0.00
    	Y_A->table[21]		0.0	
    settab2const(Ca_aplysia_ag, Y_A, 22, 30, 0.0) // 0.01 thru 0.05 => 0.0

    settab2const(Ca_aplysia_ag, Y_B, 0, 11, 0.0) // -0.1 to -0.04=>00.0
    set		Ca_aplysia_ag	\
		Y_B->table[12]		0.0		\ // -0.040 Volts
    	Y_B->table[13]		0.3		\
    	Y_B->table[14]		0.76	\ // -0.030
    	Y_B->table[15]		2.5		\
    	Y_B->table[16]		5.0		\ // -0.020
    	Y_B->table[17]		6.5		\
    	Y_B->table[18]		7.7		\ // -0.010
    	Y_B->table[19]		6.51	\
    	Y_B->table[20]		5.68	\	// 0.00
    	Y_B->table[21]		4.44	\
    	Y_B->table[22]		3.85	\	// 0.010
    	Y_B->table[23]		5.26	\
    	Y_B->table[24]		8.33	\	// 0.020
    	Y_B->table[25]		11.11	\
    	Y_B->table[26]		14.29	\	// 0.030
    	Y_B->table[27]		16.67	\
    	Y_B->table[28]		18.18	\	// 0.040
    	Y_B->table[29]		18.18	\
    	Y_B->table[30]		18.18		//0.05

		/* Setting the calc_mode to NO_INTERP for speed */
		set Ca_aplysia_ag Y_A->calc_mode 0 Y_B->calc_mode 0

		/* tweaking the tables for the tabchan calculation */
		tweak_tabchan(Ca_aplysia_ag,Y)

		/* Filling the tables using B-SPLINE interpolation */
		call Ca_aplysia_ag TABFILL Y 3000 0
end

/**********************************************************************
**          B-current : Slow inward Na and Ca bursting conductance
**	SJ Smith and SH Thompson  J.Physiol 382 p425	(1987) (Tritonia)
**********************************************************************/

function make_B_trit_st
    str chanpath = "B_trit_st"                // compartment/channel
    if(exists({chanpath}))
        return
    end
    str mgate = {chanpath}+"/m"
    str hgate = {chanpath}+"/h"

    create  vdep_channel  {chanpath}
    set  {chanpath}  \
         Ek     {EB}               \
         gbar   {0.35*SOMA_A} \
         Ik     0                  \
         Gk     0

    create  tabgate  {mgate}
    call	{mgate}	TABCREATE alpha 30 -100 50
    settab2const({mgate}, alpha, 0, 8, 0.0)
    set	{mgate} alpha->table[9]		0.000000  //  -55 mV
    set	{mgate} alpha->table[10]	0.000006  //  -50 mV
    set	{mgate} alpha->table[11]	0.000015  //  -45 mV
    set	{mgate} alpha->table[12]	0.000030  //  -40 mV
    set	{mgate} alpha->table[13]	0.000058  //  -35 mV
    set	{mgate} alpha->table[14]	0.000118  //  -30 mV
    set	{mgate} alpha->table[15]	0.000216  //  -25 mV
    set	{mgate} alpha->table[16]	0.000342  //  -20 mV
    set	{mgate} alpha->table[17]	0.000460  //  -15 mV
    set	{mgate} alpha->table[18]	0.000584  //  -10 mV
    set	{mgate} alpha->table[19]	0.000738  //   -5 mV
    set	{mgate} alpha->table[20]	0.000925  //    0 mV
    set	{mgate} alpha->table[21]	0.001130  //    5 mV
    set	{mgate} alpha->table[22]	0.001292  //   10 mV
    set	{mgate} alpha->table[23]	0.001444  //   15 mV
    set	{mgate} alpha->table[24]	0.001594  //   20 mV
    set	{mgate} alpha->table[25]	0.001869  //   25 mV
    set	{mgate} alpha->table[26]	0.002299  //   30 mV
    set	{mgate} alpha->table[27]	0.002904  //   35 mV
    set	{mgate} alpha->table[28]	0.003626  //   40 mV
    set	{mgate} alpha->table[29]	0.004783  //   45 mV
    set	{mgate} alpha->table[30]	0.005625  //   50 mV

    call	{mgate}	TABCREATE beta 30 -100 50
    settab2const({mgate}, beta, 0, 7, 0.000441)
    set	{mgate} beta->table[8]	0.000441  //  -60 mV
    set	{mgate} beta->table[9]	0.000490  //  -55 mV
    set	{mgate} beta->table[10]	0.000549  //  -50 mV
    set	{mgate} beta->table[11]	0.000646  //  -45 mV
    set	{mgate} beta->table[12]	0.000789  //  -40 mV
    set	{mgate} beta->table[13]	0.000987  //  -35 mV
    set	{mgate} beta->table[14]	0.001331  //  -30 mV
    set	{mgate} beta->table[15]	0.001726  //  -25 mV
    set	{mgate} beta->table[16]	0.002222  //  -20 mV
    set	{mgate} beta->table[17]	0.002507  //  -15 mV
    set	{mgate} beta->table[18]	0.002642  //  -10 mV
    set	{mgate} beta->table[19]	0.002952  //   -5 mV
    set	{mgate} beta->table[20]	0.003242  //    0 mV
    set	{mgate} beta->table[21]	0.003500  //    5 mV
    set	{mgate} beta->table[22]	0.003493  //   10 mV
    set	{mgate} beta->table[23]	0.003434  //   15 mV
    set	{mgate} beta->table[24]	0.003356  //   20 mV
    set	{mgate} beta->table[25]	0.003366  //   25 mV
    set	{mgate} beta->table[26]	0.003448  //   30 mV
    set	{mgate} beta->table[27]	0.003465  //   35 mV
    set	{mgate} beta->table[28]	0.003568  //   40 mV
    set	{mgate} beta->table[29]	0.003913  //   45 mV
    set	{mgate} beta->table[30]	0.003990  //   50 mV

    set		{mgate}      instantaneous   0

    sendmsg {chanpath}/m  {chanpath}  MULTGATE  m 1
    set -env {chanpath}	\
	sendmsg1	"..	m	VOLTAGE	Vm"
end
