
//genesis
/*   inputs.g for cable tutorial
---------------------------------------------------------------------------
makeinputs
     used to provide pulsed current injection to a compartment and
     pulsed spike trains to the synapse

set_inj_timing(delay,width)
     used to set the injection timing parameters

add_injection(comp_no)
     checks for valid compartment no., deletes message to previous
     compartment, and sends message to inject {comp_no}

set_injection
     gets info from form defined in forms.g and uses above functions to
     set injection parameters
     
setspikes(delay,width,interval)
     used in makeinputs and forms.g to set the spike train parameters
     for a pulsegen object

set_chan_params(gysn,tsyn,Vsyn)
     
add_syn_input(comp_no)

set_inputs
     used for setting synaptically activated channel input parameters

---------------------------------------------------------------------------
*/

function set_inj_timing(delay,width)
    float delay, width
    set /input/injectpulse width1 {width}  delay1 {delay} \
                 baselevel 0.0 trig_mode 0 delay2 100000
// free run mode with very long delay for 2nd pulse (non-repetitive)
// level1 is set by the inj_toggle function
end

function inj_toggle // toggles current injection ON/OFF
    if (get(/output/inject_menu/injtoggle,state)==1)
        set /input/injectpulse level1 1.0	// ON
    else
        set /input/injectpulse level1 0.0	// OFF
    end
end

function add_injection(comp_no)
   str comp_no		// compartment # for injection - #0 = Soma
   if ({comp_no} > {NCableSects})
      echo "There are only "{NCableSects}" cable sections!"
      return
   end
   if ({getmsg(/input/injectpulse/injcurr,OUT,-count)} > 0)
      deletemsg(/input/injectpulse/injcurr,OUT,0) // only outgoing message
   end
   if ({comp_no} <= 0)			       // then inject soma
      sendmsg /input/injectpulse/injcurr /cell/soma INJECT output
      echo "Current injection is to the soma"
   else
      sendmsg /input/injectpulse/injcurr /cell/cable[{comp_no - 1}] \
      	INJECT output
      echo "Current injection is to cable section number "{comp_no}
   end
end

function set_injection  // gets info from form defined in forms.g
   str dialog ="/output/inject_menu"
   set_inj_timing({get({dialog}/delay,value)}, \
              {get({dialog}/width,value)})
   set /input/injectpulse/injcurr gain {get({dialog}/inject,value)}
   add_injection({get({dialog}/comp_no,value)})
   echo "Injection current = "{get({dialog}/inject,value)}" microamperes"
   echo "Injection pulse delay = "{get({dialog}/delay,value)}" millisec"
   echo "Injection pulse width = "{get({dialog}/width,value)}" millisec"
end


function setspikes(delay,width,interval) //this is more general than necessary
    str pulser = "/input/synpulse"
    str spiker = "/input/synpulse/spiketrain"
    float delay, width, interval, width2
    set {pulser} width1 {width}  delay1 {delay} \
		 baselevel 0.0 trig_mode 0 delay2 100000
// free run mode with very long delay for 2nd pulse (non-repetitive)
// level1 is set by the syn_toggle function 
    set {spiker} abs_refract {interval} output_amp 1 thresh 0.5
end

function syn_toggle // toggles synaptic input ON/OFF
    if (get(/output/synapse_menu/syntoggle,state)==1)
        set /input/synpulse level1 1.0	// ON
    else
        set /input/synpulse level1 0.0	// OFF
    end
end

function set_chan_params(gsyn,tsyn,Vsyn)
   str gsyn,tsyn,Vsyn
   set /cell/syn_channel Ek	{Vsyn}
   set /cell/syn_channel tau1	{tsyn}	tau2	{tsyn}
   set /cell/syn_channel gmax	{gsyn}
end

// links channel which computes channel current (channelC2) with compartment
function link_channel2(channel,compartment)
    sendmsg   {channel}   {compartment} CHANNEL Gk Ek
    sendmsg   {compartment}   {channel} VOLTAGE Vm
end

//modify spike output based on integration step to give constant impulse area
function adjustspike
    set /##[TYPE=spike] output_amp {1.0/getclock(0)}
end

function add_syn_input(comp_no)
   str comp_no		// compartment # for input - #0 = Soma
   if ({comp_no} > {NCableSects})
      echo "There are only "{NCableSects}" cable sections!"
      return
   end
   // NOTE: We could disconnect the channel from the old compartment by
   // deleting all its messages.  If the conductance (Gk) is plotted this
   // won't work, unless we are careful to keep the message numbering staight.
   // To be safe, we locate the right "OUT" message.
    int i, n, nmsg
    nmsg=getmsg(/cell/syn_channel,OUT,-count)
    for (i=0; i<nmsg; i=i+1)
        if({getmsg(/cell/syn_channel,OUT,{i},type)}=="CHANNEL")
            n=i
            break
        else
            n=-1
        end
    end
    if (n >= 0)
        deletemsg(/cell/syn_channel,OUT,{n})
    end
   if ({getmsg(/cell/syn_channel,IN,-count)} > 0)
       deletemsg(/cell/syn_channel,IN,0)  // input from axon doesn't use msgs
   end
   if ({comp_no} <= 0)			       // then input to soma
      link_channel2 /cell/syn_channel /cell/soma
      echo "Synaptic input is to the soma"
   else
      link_channel2 /cell/syn_channel /cell/cable[{comp_no - 1}]
      echo "Synaptic input is to cable section number "{comp_no}
   end
end

function set_inputs // set parameters for synaptic input from dialog boxes
   str dialog ="/output/synapse_menu"
   setspikes({get({dialog}/delay,value)},{synwidth},{syninterval})
   adjustspike  // set spike amplitudes to 1.0/dt
   add_syn_input({get({dialog}/comp_no,value)})
   set_chan_params({get({dialog}/gsyn,value)},{get({dialog}/tsyn,value)}, \
	{get({dialog}/vsyn,value)})
   reset	// after so many changes, we need to reset everything
   echo "Synaptic input delay = "{get({dialog}/delay,value)}
   echo "Maximum conductance = "{get({dialog}/gsyn,value)}
   echo "Time constant for conductance = "{get({dialog}/tsyn,value)}
   echo "Channel equilibrium potential = "{get({dialog}/vsyn,value)}
end

//=======================================================================
// makeinputs - the main function to set up injection and synaptic input
//=======================================================================

function makeinputs


/* Default values of parameters - (global variables defined in cable.g)
float injwidth  =   10  // msec
float injdelay  =    0
float injcurrent       = 2e-4  // uA (200 nA)

float syndelay	=  0	// msec
float synwidth 	= 1000	// choose width and interval to give a single spike
float syninterval	= 100000

float gsyn	= 1e-6	// milliSiemen
float tsyn	= 1		// milliseconds
float Vsyn	= 50		// mV above resting potential (defined as zero)
	 Vsyn = 125 ==> Na channel (+55 mV with Erest = -70)
	 Vsyn = -12 ==>  K channel (-82 mV with Erest = -70)
*/
create neutral /input
//==================================================
// Make a periodic injection current step
//==================================================
create pulsegen /input/injectpulse
set_inj_timing({injdelay},{injwidth})
// set the amplitude (current) with the gain of a differential amplifier
create diffamp /input/injectpulse/injcurr
set /input/injectpulse/injcurr saturation 1 gain {injcurrent}
sendmsg /input/injectpulse /input/injectpulse/injcurr PLUS output

//==================================================
// spike trains for synaptic input - could be simpler if only a single spike
//==================================================
// pulse to gate spike train
create pulsegen /input/synpulse
// Use pulse as input to a spike object
create spike /input/synpulse/spiketrain
setspikes({syndelay},{synwidth},{syninterval})
sendmsg  /input/synpulse  /input/synpulse/spiketrain INPUT output
// Now feed these spike trains to an "axon" to allow synaptic weights
create axon /input/synpulse/axon
sendmsg /input/synpulse/spiketrain /input/synpulse/axon BUFFER name

// Lastly, make connection to chemically activated dendrite channel
create channelC2 /cell/syn_channel
set_chan_params({gsyn},{tsyn},{Vsyn})
connect /input/synpulse/axon  /cell/syn_channel with synapse
set /input/synpulse/axon:0 delay 0 weight 1  // no delay, initial weights 1

// initially, make both injection and synaptic input to the soma
sendmsg /input/injectpulse/injcurr /cell/soma INJECT output
link_channel2 /cell/syn_channel /cell/soma
end  // makeinputs
