/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	utilities.c

	PURPOSE:	frequently used functions in all parts of the program

	FUNCTIONS:	in order of appearance

	manage_link_CB			- manage (make it visible) a widget given by pointer
	unmanage_link_CB		- unmanage (make it invisible) a widget given by pointer
	map_link_CB			- map a widget given by pointer
	unmap_link_CB			- unmap a widget given by pointer
	enable_link_CB			- enable a widget given by pointer
	disable_link_CB			- disable a widget given by pointer
	get_color			- load the desired color
	load_colors			- load all colors needed for BIOSIM
	create_information_box		- create a standard information box
	create_warning_box		- create a standard warning box
	create_error_box		- create a standard error box
	is_it_a_number			- look if the given number is a valid number (no exponent allowed)
	is_it_a_float_number		- look if the given number is a valid float number
	intpower			- calculates the integer power of a given number
	convert_number			- convert given number to a valid float number (substitute ',' through '.')
	random_number_between_zero_and_one	- generates a random number between zero and one
	uniform_random			- generates a random number between a lower limit and a upper limit
	normal_random			- generates a normal distributed random number with a standard deviation of sigma
	noiseevents			- generates a exponential distributed random noise event in a given interval and
					  given intensity
 	realloc_graph			- reallocate memory for given neuron graph
 	realloc_graphS			- reallocate memory for given synapse graph
	reset_extern_input_currents	- resets extern input currents of current neuron
	error				- error handler

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/******	global variables	******/

Boolean errorOccured;		/* flag, that indicates that an error has been occured */

/****** functions ******/

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: manage_link_CB (w, client_data, call_data)
 
	PURPOSE		: manage (make it visible) a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void manage_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XtManageChild ( *(Widget *) client_data );

} /* end of manage_link_CB */

/************************************************************************************************************************

 	FUNCTION	: unmanage_link_CB (w, client_data, call_data)
 
	PURPOSE		: unmanage (make it invisible) a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void unmanage_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XtUnmanageChild ( *(Widget *) client_data );

} /* end of unmanage_link_CB */

/************************************************************************************************************************

 	FUNCTION	: map_link_CB (w, client_data, call_data)
 
	PURPOSE		: map a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void map_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg             al[10];         /*  arg list            */
	register int    ac;             /*  arg count           */

	ac = 0;
	XtSetArg (al[ac], XmNmappedWhenManaged, TRUE); ac++;
	XtSetValues ( *(Widget *) client_data, al, ac );

} /* end of map_link_CB */

/************************************************************************************************************************

 	FUNCTION	: unmap_link_CB (w, client_data, call_data)
 
	PURPOSE		: unmap a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void unmap_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg             al[10];         /*  arg list            */
	register int    ac;             /*  arg count           */

	ac = 0;
	XtSetArg (al[ac], XmNmappedWhenManaged, FALSE); ac++;
	XtSetValues ( *(Widget *) client_data, al, ac );

} /* end of unmap_link_CB */

/************************************************************************************************************************

 	FUNCTION	: enable_link_CB (w, client_data, call_data)
 
	PURPOSE		: enable a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void enable_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg             al[10];         /*  arg list            */
	register int    ac;             /*  arg count           */

	ac = 0;
	XtSetArg (al[ac], XmNsensitive, TRUE); ac++;
	XtSetValues ( *(Widget *) client_data, al, ac );

} /* end of enable_link_CB */

/************************************************************************************************************************

 	FUNCTION	: disable_link_CB (w, client_data, call_data)
 
	PURPOSE		: disable a widget given by pointer

	RETURNS		: nothing 

************************************************************************************************************************/

void disable_link_CB (w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg             al[10];         /*  arg list            */
	register int    ac;             /*  arg count           */

	ac = 0;
	XtSetArg (al[ac], XmNsensitive, FALSE); ac++;
	XtSetValues ( *(Widget *) client_data, al, ac );

} /* end of disable_link_CB */

/************************************************************************************************************************

 	FUNCTION	: get_color(colorName)
 
	PURPOSE		: load the desired color

	RETURNS		: the color as an unsigned long value (RGB-value) 

************************************************************************************************************************/

unsigned long get_color(colorName)
char *colorName;	/* color name */
{
	XColor	color;	/* color information */
	Status	status; /* status information */
	int	screenNr = DefaultScreen(disp);	/* screen number of current display */

	/* look if colorname is in color database */

	status = XParseColor(disp, DefaultColormap(disp, screenNr), colorName, &color);
	if (status == 0)
	{
		error(111);
		return(BlackPixel(disp, screenNr));
	}

	/* load color if color could be allocated */

	status = XAllocColor(disp, DefaultColormap(disp, screenNr), &color);
	if (status == 0)
	{
		error(112);
		return(BlackPixel(disp, screenNr));
	}
	return(color.pixel);
}

/************************************************************************************************************************

 	FUNCTION	: load_colors()
 
	PURPOSE		: load all colors needed in BIOSIM

	RETURNS		: nothing

************************************************************************************************************************/

void load_colors()
{
        int     screenNr = DefaultScreen(disp); /* screen number of current display */
	int	i;				/* loop variable */

	if (DefaultVisual(disp,screenNr)->class == GrayScale || DefaultVisual(disp,screenNr)->class == StaticGray)
	{	/* load colors for a monochrome display */

		backgroundColor = WhitePixel(disp, screenNr);
		rubberBandColor = BlackPixel(disp, screenNr);
		neuronColor = BlackPixel(disp, screenNr);
		neuronSelectionColor = BlackPixel(disp, screenNr);
		neuronMovingColor = BlackPixel(disp, screenNr);
		inputNeuronColor = BlackPixel(disp, screenNr);
		outputNeuronColor = BlackPixel(disp, screenNr);
		inOutNeuronColor = BlackPixel(disp, screenNr);
		connectionLineColor = BlackPixel(disp, screenNr);
		axisColor = BlackPixel(disp, screenNr);
		graphZeroLineColor = BlackPixel(disp, screenNr);
		restingLineColor = BlackPixel(disp, screenNr);
		spikeThresholdLineColor = BlackPixel(disp, screenNr);
		somaLineColor = BlackPixel(disp, screenNr);
		dendritLineColor = BlackPixel(disp, screenNr);
		synapticCurrentLineColor = BlackPixel(disp, screenNr);
		totalCurrentLineColor = BlackPixel(disp, screenNr);
		nLineColor = BlackPixel(disp, screenNr);
		sodiumLineColor = BlackPixel(disp, screenNr);
		potassiumLineColor = BlackPixel(disp, screenNr);
		calciumLineColor = BlackPixel(disp, screenNr);
		calcium1LineColor = BlackPixel(disp, screenNr);
		calcium2LineColor = BlackPixel(disp, screenNr);
		kcaLineColor = BlackPixel(disp, screenNr);
		aLineColor = BlackPixel(disp, screenNr);
		irLineColor = BlackPixel(disp, screenNr);
		pLineColor = BlackPixel(disp, screenNr);
		textColor = BlackPixel(disp, screenNr);
		excitationSynapseColor = BlackPixel(disp, screenNr);
		inhibitionSynapseColor = BlackPixel(disp, screenNr);
		electricalSynapseColor = BlackPixel(disp, screenNr);
		axoSynapseColor = BlackPixel(disp, screenNr);
		otherSynapseColor = BlackPixel(disp, screenNr);
		selectedToggleButtonColor = BlackPixel(disp, screenNr);

		/* load colors for neuron activity */

		for (i = 0; i < numActivityColors; i++)
			activityColors[i] = BlackPixel(disp, screenNr);
	}
	else	/* load colors for a color display */
	{
		backgroundColor = get_color(resData.backgroundColor);
		rubberBandColor = get_color(resData.rubberBandColor);
		neuronColor = get_color(resData.neuronColor);
		neuronSelectionColor = get_color(resData.neuronSelectionColor);
		neuronMovingColor = get_color(resData.neuronMovingColor);
		inputNeuronColor = get_color(resData.inputNeuronColor);
		outputNeuronColor = get_color(resData.outputNeuronColor);
		inOutNeuronColor = get_color(resData.inOutNeuronColor);
		connectionLineColor = get_color(resData.connectionLineColor);
		axisColor = get_color(resData.axisColor);
		graphZeroLineColor = get_color(resData.graphZeroLineColor);
		restingLineColor = get_color(resData.restingLineColor);
		spikeThresholdLineColor = get_color(resData.spikeThresholdLineColor);
		somaLineColor = get_color(resData.somaLineColor);
		dendritLineColor = get_color(resData.dendritLineColor);
		synapticCurrentLineColor = get_color(resData.synapticCurrentLineColor);
		totalCurrentLineColor = get_color(resData.totalCurrentLineColor);
		nLineColor = get_color(resData.nLineColor);
		sodiumLineColor = get_color(resData.sodiumLineColor);
		potassiumLineColor = get_color(resData.potassiumLineColor);
		calciumLineColor = get_color(resData.calciumLineColor);
		calcium1LineColor = get_color(resData.calcium1LineColor);
		calcium2LineColor = get_color(resData.calcium2LineColor);
		kcaLineColor = get_color(resData.kcaLineColor);
		aLineColor = get_color(resData.aLineColor);
		irLineColor = get_color(resData.irLineColor);
		pLineColor = get_color(resData.pLineColor);
		textColor = get_color(resData.textColor);
		excitationSynapseColor = get_color(resData.excitationSynapseColor);
		inhibitionSynapseColor = get_color(resData.inhibitionSynapseColor);
		electricalSynapseColor = get_color(resData.electricalSynapseColor);
		axoSynapseColor = get_color(resData.axoSynapseColor);
		otherSynapseColor = get_color(resData.otherSynapseColor);
		selectedToggleButtonColor = get_color("red");

		/* load colors for neuron activity */

		for (i = 0; i < numActivityColors; i++)
			activityColors[i] = get_color(colorList[i]);
	}
} /* end of load_colors */

/************************************************************************************************************************

 	FUNCTION	: create_information_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard information box

	RETURNS		: widget id of information box 

************************************************************************************************************************/

Widget create_information_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		informationDialog,
			informationWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	informationDialog = XmCreateDialogShell(parent, "informationDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_INFORMATION); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR(informationString, defaultFont);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	informationWidget = XmCreateMessageBox(informationDialog, "information", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(informationWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(informationWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(informationWidget, args, n);

	/* set symbol pixmap for an information dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(informationDialog), "default_xm_information", fg, bg)); n++;
	XtSetValues(informationWidget, args, n);

	/* return message box */

	return(informationWidget);

} /* end of create_information_box */

/************************************************************************************************************************

 	FUNCTION	: create_warning_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard warning box

	RETURNS		: widget id of warning box 

************************************************************************************************************************/

Widget create_warning_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		warningDialog,
			warningWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	warningDialog = XmCreateDialogShell(parent, "warningDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_WARNING); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR(warningString, defaultFont);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	warningWidget = XmCreateWarningDialog(warningDialog, "warning", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(warningWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(warningWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(warningWidget, args, n);

	/* set symbol pixmap for a warning dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(warningDialog), "default_xm_warning", fg, bg)); n++;
	XtSetValues(warningWidget, args, n);

	/* return message box */

	return(warningWidget);

} /* end of create_warning_box */

/************************************************************************************************************************

 	FUNCTION	: create_error_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard error box

	RETURNS		: widget id of error box 

************************************************************************************************************************/

Widget create_error_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		errorDialog,
			errorWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	errorDialog = XmCreateDialogShell(parent, "errorDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_ERROR); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR(errorString, defaultFont);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	errorWidget = XmCreateErrorDialog(errorDialog, "error", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(errorWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(errorWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(errorWidget, args, n);

	/* set symbol pixmap for an error dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(errorDialog), "default_xm_error", fg, bg)); n++;
	XtSetValues(errorWidget, args, n);

	/* return message box */

	return(errorWidget);

} /* end of create_error_box */
#endif

/************************************************************************************************************************

 	FUNCTION	: is_it_a_number(numericString)
 
	PURPOSE		: look if the given number is a valid number (no exponent allowed)

	RETURNS		: TRUE, if the number is valid, else FALSE 

************************************************************************************************************************/

Boolean is_it_a_number(numericString)
char numericString[255];	/* numeric string containing the number */
{
	int 	 charPosition, offset;	/* character position and offset into string */
	Boolean isANum = FALSE;		/* flag if it's a number */
	Boolean dotUsed = FALSE;	/* flag if the decimal point is used */

	/* Compact string */

	charPosition = 0;
	offset = 0;
	while (numericString[charPosition++] != '\0')
		if (numericString[charPosition] == ' ')
		{
			offset++;
			numericString[charPosition-offset] = numericString[charPosition];
		}

	/* Check Number */

	charPosition = 0;
	while (numericString[charPosition] != '\0')
	{
		switch(numericString[charPosition])
		{

		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			isANum = TRUE;
			break;

		case '+':
			if (charPosition != 0)
				return FALSE;
			break;

		case '-':
			if (charPosition != 0)
				return FALSE;
			break;

		case '.':
			if(! dotUsed)
			{
				dotUsed = TRUE;
			}
			else return FALSE;
			break;

		case ' ':
			break;

		default:
			return FALSE;

		} /* End of switch */


		charPosition++;
	} /* End of While */


	if (isANum) return TRUE;
	else return FALSE;

} /* end of is_it_a_number */

/************************************************************************************************************************

 	FUNCTION	: is_it_a_float_number(numericString)
 
	PURPOSE		: look if the given number is a valid float number

	RETURNS		: TRUE, if the number is valid, else FALSE 

************************************************************************************************************************/

Boolean is_it_a_float_number(numericString)
char numericString[255];	/* numeric string containing the number */
{
	int 	 charPosition, offset;	/* character position and offset into string */
	Boolean isANum = FALSE;		/* flag if it's a number */
	Boolean dotUsed = FALSE;	/* flag if the decimal point is used */
	Boolean digitUsed = FALSE;	/* flag if a digit is used */
	Boolean minusUsed = FALSE;	/* flag if a minus sign is used */
	Boolean plusUsed = FALSE;	/* flag if a plus sign is used */
	Boolean eUsed = FALSE;		/* flag if an 'e' for exponential is used */


	/* Compact string */

	charPosition = 0;
	offset = 0;

	while (numericString[charPosition++] != EOS)
		if (numericString[charPosition] == ' ')
		{

			offset++;

			numericString[charPosition - offset] = numericString[charPosition];

		}


	/* Check Number */

	charPosition = 0;

	while (numericString[charPosition] != EOS)
	{

		switch(numericString[charPosition])
		{


		case '0':

		case '1':

		case '2':

		case '3':

		case '4':

		case '5':

		case '6':

		case '7':

		case '8':

		case '9':

			isANum = TRUE;

			digitUsed = TRUE;

			break;


		case '+':

			if (charPosition == 0 ||
			    (charPosition > 0 &&
			    (numericString[charPosition - 1] == 'e' ||
			    numericString[charPosition - 1] == 'E')) )
				plusUsed = TRUE;

			else return FALSE;

			break;


		case '-':

			if (charPosition == 0 ||
			    (charPosition > 0 &&
			    (numericString[charPosition - 1] == 'e' ||
			    numericString[charPosition - 1] == 'E')) )
				minusUsed = TRUE;

			else return FALSE;

			break;


		case 'e':

		case 'E':

			if (! eUsed)
			{

				eUsed = TRUE;

				isANum = FALSE;

				digitUsed = FALSE;

				minusUsed = FALSE;

				plusUsed = FALSE;

				dotUsed = FALSE;

			}
			else return FALSE;

			break;


		case '.':

			if(! dotUsed)
			{

				dotUsed = TRUE;

			}
			else return FALSE;

			break;


		case ' ':

			break;

		default:

			return FALSE;

		} /* End of switch */

		charPosition++;

	} /* End of While */

	if (isANum) return TRUE;

	else return FALSE;

} /* end of is_it_a_float_number */

/************************************************************************************************************************

 	FUNCTION	: intpower(base, exponent)
 
	PURPOSE		: calculates the integer power of a given number

	RETURNS		: base ^ exponent

************************************************************************************************************************/

double intpower(base, exponent)
double	base;		/* base of number */
int	exponent;	/* exponent of number */
{
	double result;

	if (exponent & ~1)
	{
		result = intpower(base, exponent >> 1);

		if (exponent & 1)
			return(result * result * base);
		else	return(result * result);
	}
	else
	{
		if (exponent & 1)
			return(base);
		else	return(1.0);
	}
} /* end of intpower */

/************************************************************************************************************************

 	FUNCTION	: convert_number(numToConvert)
 
	PURPOSE		: convert given number to a valid float number (substitute ',' through '.')

	RETURNS		: the pointer to converted number 

************************************************************************************************************************/

char *convert_number(numToConvert)
double numToConvert;	/* number to convert */
{
	char numstringToConvert[20];	/* number string to be converted */
	register char *replacePointer;	/* current pointer to string */

	sprintf(numstringToConvert, "%e", numToConvert);

	/* if decimal character is not the point sign replace it to that */

	if (asciiDefaults.decimalChar != '.')
	{
		replacePointer = numstringToConvert;
		while (*replacePointer != '.')
			replacePointer++;
		*replacePointer = asciiDefaults.decimalChar;
	}

	return (numstringToConvert);

} /* end of convert_number */

/************************************************************************************************************************

 	FUNCTION	: random_number_between_zero_and_one()
 
	PURPOSE		: generates a random number between zero and one

	RETURNS		: random number between zero and one

************************************************************************************************************************/

double random_number_between_zero_and_one()
{
	return (double) random() / (double) 0x7FFFFFFF;

} /* end of random_number_between_zero_and_one */

/************************************************************************************************************************

 	FUNCTION	: uniform_random(lower_limit, upper_limit)
 
	PURPOSE		: generates a random number between a lower limit and a upper limit

	RETURNS		: random number between a lower limit and a upper limit

************************************************************************************************************************/

double uniform_random(lower_limit, upper_limit)
double lower_limit;	/* lower limit of range for the random value */
double upper_limit;	/* upper limit of range for the random value */
{
	return lower_limit + (upper_limit - lower_limit) * random_number_between_zero_and_one();

} /* end of uniform_random */

/************************************************************************************************************************

 	FUNCTION	: normal_random(mean, sigma, lower_limit, upper_limit)
 
	PURPOSE		: generates a normal distributed random number with a standard deviation of sigma

	RETURNS		: normal distributed random number

************************************************************************************************************************/

double normal_random(mean, sigma, lower_limit, upper_limit)
double mean;		/* mean value for normal distribution */
double sigma;		/* standard deviation for normal distribution */
double lower_limit;	/* lower limit for the random value */
double upper_limit;	/* upper limit for the random value */
{
	static short int pending = 0;
	static double thependingnumber;
	double u1, u2, r, result;

	if (pending)
	{
		pending = 0;
		result = mean + sigma * thependingnumber;

		if (result < lower_limit)
			result = lower_limit;
		else if (result > upper_limit)
			result = upper_limit;

		return result;
	}

	u1 = random_number_between_zero_and_one();
	u2 = random_number_between_zero_and_one();

	r = sqrt(-2.0 * log(u1));

	pending = 1;
	thependingnumber = r * cos(2.0 * M_PI * u2);

	result = mean + sigma * r * sin(2.0 * M_PI * u2);

	if (result < lower_limit)
		result = lower_limit;
	else if (result > upper_limit)
		result = upper_limit;

	return result;

} /* end of normal_random */

/************************************************************************************************************************

 	FUNCTION	: noiseevents(interval, intensity)
 
	PURPOSE		: generates a exponential distributed random noise event in a given interval and given intensity

	RETURNS		: noise event

************************************************************************************************************************/

double noiseevents(interval, intensity)
double interval;	/* interval in wich the event can occur */
double intensity;	/* intensity for noise generation */
{
	int	no_of_events = 0;
	double	accum = uniform_random(0.0, 1.0);
	double	limit = exp(-interval * intensity);

	while (accum > limit)
	{
		accum *= uniform_random(0.0, 1.0);
		no_of_events++;
	}

	return no_of_events;
} /* end of noiseevents */

/************************************************************************************************************************

 	FUNCTION	: realloc_graph(graph, graphLength)
 
	PURPOSE		: reallocate memory for given neuron graph

	RETURNS		: nothing

************************************************************************************************************************/

void realloc_graph(graph, graphLength)
Graph		*graph;		/* current neuron graph */
unsigned int	graphLength;	/* size of one reallocation entry */
{
	if (graph != NULL)	/* check if neuron has a graph */
	{
		/* check if graph must be reallocated */

		if (graphLength > graph->graphSize)
		{
			graph->soma = (float *) realloc(graph->soma, (size_t) graphLength);

			if (! graph->soma)
			{
				error(127);
				return;
			}

			graph->dend = (float *) realloc(graph->dend, (size_t) graphLength);

			if (! graph->dend)
			{
				error(127);
				return;
			}

			graph->synC = (float *) realloc(graph->synC, (size_t) graphLength);

			if (! graph->synC)
			{
				error(127);
				return;
			}

			graph->totC = (float *) realloc(graph->totC, (size_t) graphLength);

			if (! graph->totC)
			{
				error(127);
				return;
			}

			graph->iN = (float *) realloc(graph->iN, (size_t) graphLength);

			if (! graph->iN)
			{
				error(127);
				return;
			}

			graph->iNA = (float *) realloc(graph->iNA, (size_t) graphLength);

			if (! graph->iNA)
			{
				error(127);
				return;
			}

			graph->iK = (float *) realloc(graph->iK, (size_t) graphLength);

			if (! graph->iK)
			{
				error(127);
				return;
			}

			graph->iCA = (float *) realloc(graph->iCA, (size_t) graphLength);

			if (! graph->iCA)
			{
				error(127);
				return;
			}

			graph->iCA1 = (float *) realloc(graph->iCA1, (size_t) graphLength);

			if (! graph->iCA1)
			{
				error(127);
				return;
			}

			graph->iCA2 = (float *) realloc(graph->iCA2, (size_t) graphLength);

			if (! graph->iCA2)
			{
				error(127);
				return;
			}

			graph->iKCA = (float *) realloc(graph->iKCA, (size_t) graphLength);

			if (! graph->iKCA)
			{
				error(127);
				return;
			}

			graph->iA = (float *) realloc(graph->iA, (size_t) graphLength);

			if (! graph->iA)
			{
				error(127);
				return;
			}

			graph->iIR = (float *) realloc(graph->iIR, (size_t) graphLength);

			if (! graph->iIR)
			{
				error(127);
				return;
			}

			graph->iP = (float *) realloc(graph->iP, (size_t) graphLength);

			if (! graph->iP)
			{
				error(127);
				return;
			}

			graph->graphSize = graphLength;
		}
	}
} /* end of realloc_graph */

/************************************************************************************************************************

 	FUNCTION	: realloc_graphS(graph, graphLength)
 
	PURPOSE		: reallocate memory for given synapse graph

	RETURNS		: nothing

************************************************************************************************************************/

void realloc_graphS(graph, graphLength)
SGraph		*graph;		/* current synapse graph */
unsigned int	graphLength;	/* size of one reallocation entry */
{
	if (graph != NULL)	/* check if synapse has a graph */
	{
		/* check if graph must be reallocated */

		if (graphLength > graph->graphSize)
		{
			graph->pot = (float *) realloc(graph->pot, (size_t) graphLength);

			if (! graph->pot)
			{
				error(127);
				return;
			}

			graph->mem = (float *) realloc(graph->mem, (size_t) graphLength);

			if (! graph->mem)
			{
				error(127);
				return;
			}

			graph->str = (float *) realloc(graph->str, (size_t) graphLength);

			if (! graph->str)
			{
				error(127);
				return;
			}

			graph->con = (float *) realloc(graph->con, (size_t) graphLength);

			if (! graph->con)
			{
				error(127);
				return;
			}

			graph->graphSize = graphLength;
		}
	}
} /* end of realloc_graphS */

/************************************************************************************************************************

 	FUNCTION	: reset_extern_input_currents(neuron)
 
	PURPOSE		: resets extern input currents of current neuron

	RETURNS		: nothing

************************************************************************************************************************/

void reset_extern_input_currents(neuron)
Neuron *neuron;
{
	int	i, j;		/* loop variables */

	currentNeuron = neuron;

	/* reset extern input currents into neuron */

	for (i = 0; i <= curNumOfDendSegs; i++)
	{
		if (ICUR[i].numEntries == 0)	/* not initialized */
		{
			ICUR[i].starttime = (double *) malloc((size_t) (sizeof(double) * 3));

			if (! ICUR[i].starttime)
			{
				error(114);
				return;
			}

			ICUR[i].current = (double *) malloc((size_t) (sizeof(double) * 3));

			if (! ICUR[i].current)
			{
				error(114);
				return;
			}
#ifndef BATCH
			currentNeuron->neuronSize -= currentNeuron->inpCurSize;
			currentNeuron->inpCurSize = (unsigned int) (sizeof(double) * 6);
			currentNeuron->neuronSize += currentNeuron->inpCurSize;
#endif

			ICUR[i].numEntries = 3;
			ICUR[i].index = 0;
			ICUR[i].starttime[0] = 0.0;
			ICUR[i].current[0] = 0.0;

			/* set first two values of array, don't remove this */

			for (j = 1; j <= 2; j++)
			{
				ICUR[i].starttime[j] = simulation.length;
				ICUR[i].current[j] = 0.0;
			}
		}
		else	/* initialized, so set only simulation length in last entrie to current value */
		{
			ICUR[i].starttime[ICUR[i].numEntries - 1] = simulation.length;
		}
	}
} /* end of reset_extern_input_currents */

/************************************************************************************************************************

 	FUNCTION	: error(number)
 
	PURPOSE		: error handler

	RETURNS		: nothing

************************************************************************************************************************/

void error(number)
int	number;		/* error number */
{
#ifndef BATCH
	XmString        text = NULL;            /* text for error messages */
#endif

	/* number < 100 causes to build an error box, number >= 100 a warning box */

	switch(number)
	{
	case 1:
#ifndef BATCH
			text = XmStringCreate(fileNotFoundString, defaultFont);
#else
			fprintf(stderr, fileNotFoundString);
#endif
			break;

#ifndef BATCH
	case 2:		text = XmStringCreate(cannotRemoveFileString, defaultFont);
			break;

	case 3:		text = XmStringCreate(printFailedString, defaultFont);
			break;

	case 4:		text = XmStringCreate(wrongExtensionString, defaultFont);
			break;

	case 5:		text = XmStringCreateLtoR(eNaEqString, defaultFont);
			break;

	case 6:		text = XmStringCreateLtoR(eNaConString, defaultFont);
			break;

	case 7:		text = XmStringCreateLtoR(eKEqString, defaultFont);
			break;

	case 8:		text = XmStringCreateLtoR(eKConString, defaultFont);
			break;

	case 9:		text = XmStringCreateLtoR(eCaEqString, defaultFont);
			break;

	case 10:	text = XmStringCreateLtoR(eCaConString, defaultFont);
			break;

	case 11:	text = XmStringCreateLtoR(eConString, defaultFont);
			break;

	case 12:	text = XmStringCreateLtoR(eLeakString, defaultFont);
			break;

	case 13:	text = XmStringCreateLtoR(eCapString, defaultFont);
			break;

	case 14:	text = XmStringCreateLtoR(eTodString, defaultFont);
			break;

	case 15:	text = XmStringCreateLtoR(ePotString, defaultFont);
			break;

	case 16:	text = XmStringCreateLtoR(eDConString, defaultFont);
			break;

	case 17:	text = XmStringCreateLtoR(eDPotString, defaultFont);
			break;

	case 18:	text = XmStringCreateLtoR(eDLeakString, defaultFont);
			break;

	case 19:	text = XmStringCreateLtoR(eDCapString, defaultFont);
			break;

	case 20:	text = XmStringCreateLtoR(eTosString, defaultFont);
			break;

	case 21:	text = XmStringCreateLtoR(eSCurString, defaultFont);
			break;

	case 22:	text = XmStringCreateLtoR(eSStartString, defaultFont);
			break;

	case 23:	text = XmStringCreateLtoR(eSEndString, defaultFont);
			break;

	case 24:	text = XmStringCreateLtoR(eDCurString, defaultFont);
			break;

	case 25:	text = XmStringCreateLtoR(eDStartString, defaultFont);
			break;

	case 26:	text = XmStringCreateLtoR(eDEndString, defaultFont);
			break;

	case 27:	text = XmStringCreateLtoR(eNotConnectedString, defaultFont);
			break;

	case 28:	text = XmStringCreateLtoR(eTTString, defaultFont);
			break;

	case 29:	text = XmStringCreateLtoR(eResString, defaultFont);
			break;

	case 30:	text = XmStringCreateLtoR(eNoTypeString, defaultFont);
			break;

	case 31:	text = XmStringCreateLtoR(eNoSelectionString, defaultFont);
			break;

	case 32:	text = XmStringCreateLtoR(eSEQString, defaultFont);
			break;

	case 33:	text = XmStringCreateLtoR(eSimLenString, defaultFont);
			break;

	case 34:	text = XmStringCreateLtoR(eSimSizeString, defaultFont);
			break;

	case 35:	text = XmStringCreateLtoR(eTRTString, defaultFont);
			break;

	case 36:	text = XmStringCreateLtoR(eTRSString, defaultFont);
			break;

	case 37:	text = XmStringCreateLtoR(eTRSToSmallString, defaultFont);
			break;

	case 38:	text = XmStringCreateLtoR(eNaDurString, defaultFont);
			break;

	case 39:	text = XmStringCreateLtoR(eKStartString, defaultFont);
			break;

	case 40:	text = XmStringCreateLtoR(eKDurString, defaultFont);
			break;

	case 41:	text = XmStringCreateLtoR(eCaStartString, defaultFont);
			break;

	case 42:	text = XmStringCreateLtoR(eCaDurString, defaultFont);
			break;

	case 43:	text = XmStringCreateLtoR(eSTHString, defaultFont);
			break;

	case 44:	text = XmStringCreateLtoR(selectedItemNotFoundString, defaultFont);
			break;

	case 45:	text = XmStringCreateLtoR(notANumberString, defaultFont);
			break;

	case 46:	text = XmStringCreateLtoR(eASCOUTString, defaultFont);
			break;

	case 47:	text = XmStringCreateLtoR(eXSSCString, defaultFont);
			break;

	case 48:	text = XmStringCreateLtoR(eXESCString, defaultFont);
			break;

	case 49:	text = XmStringCreateLtoR(eXSC1String, defaultFont);
			break;

	case 50:	text = XmStringCreateLtoR(eXSC2String, defaultFont);
			break;

	case 51:	text = XmStringCreateLtoR(eXSC3String, defaultFont);
			break;

	case 52:	text = XmStringCreateLtoR(eXSC4String, defaultFont);
			break;

	case 53:	text = XmStringCreateLtoR(eYSSCString, defaultFont);
			break;

	case 54:	text = XmStringCreateLtoR(eYESCString, defaultFont);
			break;

	case 55:	text = XmStringCreateLtoR(eYSC1String, defaultFont);
			break;

	case 56:	text = XmStringCreateLtoR(eYSC2String, defaultFont);
			break;

	case 57:	text = XmStringCreateLtoR(eYSC3String, defaultFont);
			break;

	case 58:	text = XmStringCreateLtoR(eYSC4String, defaultFont);
			break;

	case 59:	text = XmStringCreateLtoR(eXZOString, defaultFont);
			break;

	case 60:	text = XmStringCreateLtoR(eYZOString, defaultFont);
			break;

	case 61:	text = XmStringCreateLtoR(eSTRString, defaultFont);
			break;

	case 62:	text = XmStringCreateLtoR(eSLOPEString, defaultFont);
			break;

	case 63:	text = XmStringCreateLtoR(eTargetString, defaultFont);
			break;

	case 64:	text = XmStringCreateLtoR(eRegionString, defaultFont);
			break;

	case 65:	text = XmStringCreateLtoR(eSTMString, defaultFont);
			break;

	case 66:	text = XmStringCreateLtoR(eLTTString, defaultFont);
			break;

	case 67:	text = XmStringCreateLtoR(eLTMString, defaultFont);
			break;

	case 68:	text = XmStringCreateLtoR(eHIString, defaultFont);
			break;

	case 69:	text = XmStringCreateLtoR(eHLFString, defaultFont);
			break;

	case 70:	text = XmStringCreateLtoR(eKLFString, defaultFont);
			break;

	case 71:	text = XmStringCreateLtoR(eCLFString, defaultFont);
			break;

	case 72:	text = XmStringCreateLtoR(eMaxStrString, defaultFont);
			break;

	case 73:	text = XmStringCreateLtoR(eMaxInhString, defaultFont);
			break;

	case 74:	text = XmStringCreateLtoR(eDRegString, defaultFont);
			break;

	case 75:	text = XmStringCreateLtoR(eNoiseString, defaultFont);
			break;
#endif

	case 76:
#ifndef BATCH
			text = XmStringCreateLtoR(unknownIntegrationMethodString, defaultFont);
#else
			fprintf(stderr, unknownIntegrationMethodString);
#endif
			break;

#ifndef BATCH
	case 77:	text = XmStringCreateLtoR(wrongGraphExtensionString, defaultFont);
			break;

	case 78:	text = XmStringCreateLtoR(eCompLFString, defaultFont);
			break;
#endif

	case 100:
#ifndef BATCH
			text = XmStringCreate(cannotUseFileString, defaultFont);
#else
			fprintf(stderr, cannotUseFileString);
#endif
			break;

	case 101:
#ifndef BATCH
			text = XmStringCreate(cannotCloseFileString, defaultFont);
#else
			fprintf(stderr, cannotCloseFileString);
#endif
			break;

#ifndef BATCH
	case 102:	text = XmStringCreate(commandFileHasWrongExtensionString, defaultFont);
			break;

	case 103:	text = XmStringCreate(commandFileNotRemoveableString, defaultFont);
			break;
#endif

	case 104:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForProcessQueueString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForProcessQueueString);
#endif
			break;

	case 105:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForNeuronString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForNeuronString);
#endif
			break;

	case 106:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForNeuronSelectionBufferString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForNeuronSelectionBufferString);
#endif
			break;

	case 107:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForSynapseSelectionBufferString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForSynapseSelectionBufferString);
#endif
			break;

	case 108:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForConnectionString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForConnectionString);
#endif
			break;

	case 109:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForSenderListString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForSenderListString);
#endif
			break;

	case 110:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForPoolString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForPoolString);
#endif
			break;

#ifndef BATCH
	case 111:	text = XmStringCreateLtoR(unknownColorString, defaultFont);
			break;

	case 112:	text = XmStringCreateLtoR(cannotLoadColorString, defaultFont);
			break;
#endif

	case 113:
#ifndef BATCH
			text = XmStringCreate(cannotReadFileString, defaultFont);
#else
			fprintf(stderr, cannotReadFileString);
#endif
			break;

	case 114:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForCurrentEntriesString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForCurrentEntriesString);
#endif
			break;

	case 115:
#ifndef BATCH
			text = XmStringCreate(cannotOpenFileString, defaultFont);
#else
			fprintf(stderr, cannotOpenFileString);
#endif
			break;

	case 116:
#ifndef BATCH
			text = XmStringCreate(fileReadOnlyString, defaultFont);
#else
			fprintf(stderr, fileReadOnlyString);
#endif
			break;

#ifndef BATCH
	case 117:	text = XmStringCreate(cannotAllocateMemoryForPrintCommandString, defaultFont);
			break;
#endif

	case 118:
#ifndef BATCH
			text = XmStringCreate(cannotReadNeuronString, defaultFont);
#else
			fprintf(stderr, cannotReadNeuronString);
#endif
			break;

	case 119:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateSynapseTypesString, defaultFont);
#else
			fprintf(stderr, cannotAllocateSynapseTypesString);
#endif
			break;

	case 120:
#ifndef BATCH
			text = XmStringCreate(noEndOfFileString, defaultFont);
#else
			fprintf(stderr, noEndOfFileString);
#endif
			break;

	case 121:
#ifndef BATCH
			text = XmStringCreate(neuronNotFoundString, defaultFont);
#else
			fprintf(stderr, neuronNotFoundString);
#endif
			break;

	case 122:
#ifndef BATCH
			text = XmStringCreate(cannotCreateFileBufferString, defaultFont);
#else
			fprintf(stderr, cannotCreateFileBufferString);
#endif
			break;

	case 123:
#ifndef BATCH
			text = XmStringCreate(eofNotExpectedString, defaultFont);
#else
			fprintf(stderr, eofNotExpectedString);
#endif
			break;

	case 124:
#ifndef BATCH
			text = XmStringCreate(badNumberString, defaultFont);
#else
			fprintf(stderr, badNumberString);
#endif
			break;

	case 125:
#ifndef BATCH
			text = XmStringCreate(cannotReadChannelFileString, defaultFont);
#else
			fprintf(stderr, cannotReadChannelFileString);
#endif
			break;

	case 126:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForSynapseTypesString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForSynapseTypesString);
#endif
			break;

	case 127:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForGraphString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForGraphString);
#endif
			break;

	case 128:
#ifndef BATCH
			text = XmStringCreate(cannotAllocateMemoryForCalculationString, defaultFont);
#else
			fprintf(stderr, cannotAllocateMemoryForCalculationString);
#endif
			break;

#ifndef BATCH
	case 129:	text = XmStringCreate(cannotAllocateMemoryForItemTableEntryString, defaultFont);
			break;
#endif

	case 130:
#ifndef BATCH
			text = XmStringCreate(wrongFormTypeString, defaultFont);
#else
			fprintf(stderr, wrongFormTypeString);
#endif
			break;

	case 131:
#ifndef BATCH
			text = XmStringCreate(divisionByZeroGateString, defaultFont);
#else
			fprintf(stderr, divisionByZeroGateString);
#endif
			break;

	default: fprintf(stderr, "Unknown error has been encountered in error handler\n");
		break;
	}

#ifndef BATCH
	if (number < 100)
		XtManageChild(create_error_box(mainWindow, XmDIALOG_MODELESS, text));
	else	XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));

	XmStringFree(text);
#endif
	errorOccured = TRUE;

} /* end of error */
