/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	menu.c

	PURPOSE:	contains all functions that builds and handles the main menu

	FUNCTIONS:	in order of appearance

	create_menubar				- create the main menubar with all pulldown menus
	create_askModelPoint			- create a dialog box for asking the point model
	create_noiseDialog			- creates a dialog box for noise parameters
	create_stddevDialog			- creates a dialog box for standard deviation parameters
	generate_file_selection_box		- generates a file selection box
	set_max_scale				- sets the maximal scale widgets of add synapse target,
						  edit synapse target, edit current target
	dialog_accept_CB			- callback routine for ok button
	dialog_text_CB				- callback routine for text widgets
	dialog_cancel_CB			- callback routine for cancel button
	dialog_help_CB				- callback routine for help button
	dialog_option_button_CB			- callback routine for buttons in option menu
 	dialog_stochastic_button_CB		- callback routine for buttons in stochastic menu
	dialog_learn_button_CB			- callback routine for buttons in learn menu
	dialog_model_button_CB			- callback routine for buttons in model menu
	dialog_integrationMethods_button_CB	- callback routine for buttons in integration method menu
	edit_sim_pars_CB			- callback routine for edit simulation parameter mask

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"
#include "calculations.h"
#include "menu.h"

/****** global variables ******/

#ifndef BATCH
XtWorkProcId workProcId = (XtWorkProcId) NULL;	/* working procedure id for run_simulation */
#endif
double noiseIntensity = NOISE_INTENSITY;	/* noise intensity */
double noiseStrength = NOISE_STRENGTH / I_SCALE;/* noise strength in nA */
double noiseDecay = NOISE_DECAY / E_SCALE;	/* noise decay rate in ms */
double stdInitialPotential = LEAK_SWIM_DEF / E_SCALE; /* initial potential for standard deviation */
double stdInitialPotentialStd = IPOT_STD / E_SCALE;   /* standard deviation of initial potential */
double stdSynapticStrength = STR_DEF / S_SCALE;	      /* synaptic strength for standard deviation */
double stdSynapticStrengthStd = SYNG_STD / S_SCALE;   /* standard deviation of synaptic strength */
double stdAxoAxonicStrength = ASTR_DEF;		/* axo-axonic strength for standard deviation */
double stdAxoAxonicStrengthStd = AXOS_STD;	/* standard deviation of axo-axonic strength */
double stdSynapticTT = TT_DEF / T_SCALE;	/* synaptic transmission time for standard deviation */
double stdSynapticTTStd = SYNT_STD / T_SCALE;	/* standard deviation of synaptic transmission time */
int curNumOfDendSegs = 3;			/* current network model type */
Boolean autoReset = FALSE;			/* reset network every time a new simulation is started */
Boolean showProcessingInfo = FALSE;		/* show information about processing time of simulation */
Boolean hebbLearning = FALSE;			/* flag for hebbian learning */
Boolean competitiveLearning = FALSE;		/* flag for competitive learning */
Boolean conditionedLearning = FALSE;		/* flag for conditioned learning */
Boolean plasticityLearning = FALSE;		/* flag for short term plasticity learning */
#ifndef BATCH
Boolean boundCheckingOn = FALSE;		/* flag if bound checking should be done during graph output */
Boolean manualWindowUpdate = FALSE;		/* flag if manual window refresh should be done */
Boolean editNetwork = FALSE;			/* flag that indicates whether network is editable or not */
Boolean fileSaved = TRUE;			/* flag indicating file saved status */
Boolean showSimulationInfo = FALSE;		/* show information during simulation */
Boolean showOutputImmediately = FALSE;		/* flag for showing output immediately during simulation */
Widget overBiosimDialog;			/* dialog box for over biosim */
Widget simInfButton;				/* button in option menu */
Widget timeMessureButton;			/* button in option menu */
Widget graphDisplayButton;			/* button in option menu */
Widget autoResetButton;				/* button in option menu */
Widget neuronActivityColorsButton;		/* button in option menu */
Widget outputWatchButton;			/* button in option menu */
Widget manualUpdateButton;			/* button in option menu */
Widget hebbLearnButton;				/* button in learn menu */
Widget competitiveLearnButton;			/* button in learn menu */
Widget conditionedLearnButton;			/* button in learn menu */
Widget plasticityLearnButton;			/* button in learn menu */
Widget noiseButton;				/* button in option menu */
Widget initialPotentialsButton;			/* button in stochastic distribution menu */
Widget synapticStrengthButton;			/* button in stochastic distribution menu */
Widget synapticTransmissionTimeButton;		/* button in stochastic distribution menu */
Widget eulerButton;				/* button in integration methods menu */
Widget expEulerButton;				/* button in integration methods menu */
Widget trapezoidButton;				/* button in integration methods menu */
Widget rungeKuttaButton;			/* button in integration methods menu */
Widget networkOnOffButton;			/* button in network model menu */ 
Widget networkSWIMButton;			/* button in network model menu */ 
Widget networkHHButton;				/* button in network model menu */ 
Widget networkGBButton;				/* button in network model menu */ 
Widget networkComplexButton;			/* button in network model menu */ 
Widget networkType1Button;			/* button in network model menu */ 
Widget networkType2Button;			/* button in network model menu */ 
Widget networkType4Button;			/* button in network model menu */ 
Widget networkTypeNButton;			/* button in network model menu */ 

/****** local variables ******/

static Widget askModelPointBB;			/* widget id for ask model type dialog box */
static Widget noiseBB;				/* widget id for noise dialog box */
static Widget stddevBB;				/* widget id for standard deviation box */
static Widget noiseIntensityText;		/* text widget of noise parameter */
static Widget noiseStrengthText;		/* text widget of noise parameter */
static Widget noiseDecayText;			/* text widget of noise parameter */
static Widget stddevIPotStdText;		/* text widget of standdard deviation parameter */
static Widget stddevIPotText;			/* text widget of standdard deviation parameter */
static Widget stddevSynGText;			/* text widget of standdard deviation parameter */
static Widget stddevSynGStdText;		/* text widget of standdard deviation parameter */
static Widget stddevSynSText;			/* text widget of standdard deviation parameter */
static Widget stddevSynSStdText;		/* text widget of standdard deviation parameter */
static Widget stddevSynTText;			/* text widget of standdard deviation parameter */
static Widget stddevSynTStdText;		/* text widget of standdard deviation parameter */
static Widget askModelPointText;		/* text widget of ask model type */
static Widget openFileDialog;			/* widget id for open file dialog box */
static Widget deleteFileDialog;			/* widget id for delete file dialog box */
static Widget saveAsFileDialog;			/* widget id for save as file dialog box */
static Widget printFileDialog;			/* widget id for print file dialog box */
static Widget exitMessageBox;			/* widget id for exit dialog box */
static Widget newMessageBox;			/* widget id for new dialog box */
static Widget saveSimMessageBox;		/* widget id for save simulation dialog box */
static Widget saveSimCloseMessageBox;		/* widget id for save simulation and close dialog box */
static Widget modelForm;			/* widget id for network model form */
static Widget integrationMethodsForm;		/* widget id for integration methods box */
static Widget resetForm;			/* widget id for reset method box */
static Widget loadParameterButton;		/* widget id for load parameter button */
static Widget saveButton;			/* widget id for save button */
static Widget saveAsButton;			/* widget id for save as button */
static Widget windowUpdateButton;		/* widget id for update window button */
#endif

/****** functions ******/

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: create_menubar(parent)
 
	PURPOSE		: create the main menubar with all pulldown menus

	RETURNS		: widget id of menubar

************************************************************************************************************************/

Widget create_menubar(parent)
Widget parent;	/* widget id of parent widget */
{
	Arg		args[20];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	int		fg, bg;			/* foreground and background color */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget 		menubar,
			button,
			fileMenu,
			openMenu,
			saveMenu,
			saveAsMenu,
			printMenu,
			optionMenu,
			networkMenu,
			simulationMenu,
			helpMenu,
			learnMenu,
			stochasticMenu,
			modelDialog,
			integrationMethodsDialog,
			integrationMethodsRadioBox,
			resetDialog,
			resetRadioBox,
			exitDialog,
			saveSimDialog,
			saveSimCloseDialog,
			modelRadioBox,
			newDialog,
			separator;		/* widget ids for internal use */

	/* menu bar */

	n = 0;
	menubar = XmCreateMenuBar(parent, "menubar", args, n);

	/* menu point 'file' with pulldown menu */

	n = 0;
	fileMenu = XmCreatePulldownMenu(menubar, "filemenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(fileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, fileMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, fileMenu); n++;
	button = XmCreateCascadeButton(menubar, "file", args, n);
	XtAddCallback(button, XmNcascadingCallback, dialog_accept_CB, (caddr_t *) DIALOG_FILE);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'new' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(newString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, newMnemonic); n++;
	button = XmCreatePushButton(fileMenu, "new", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &newMessageBox);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* create question dialog for new */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	newDialog = XmCreateDialogShell(parent, "newDialog", args, n);

	xmstrings[0] = XmStringCreateLtoR(newString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(areYouShureString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(yesString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL); n++;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_QUESTION); n++;
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
	newMessageBox = XmCreateMessageBox(newDialog, "newBox", args, n);
	XtAddCallback(newMessageBox, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_NEW);
	XtAddCallback(newMessageBox, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &newMessageBox);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(newMessageBox, args, n);

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(newDialog), "default_xm_question", fg, bg)); n++;
	XtSetValues(newMessageBox, args, n);

	/* help button is not needed, remove him */

	button = XmMessageBoxGetChild(newMessageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	/* pulldown menu point 'open' */

	n = 0;
	openMenu = XmCreatePulldownMenu (fileMenu, "openMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadSimMnemonic); n++;
	button = XmCreatePushButton(openMenu, "openSimulation", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_OPEN_SIMULATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(configurationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadConMnemonic); n++;
	button = XmCreatePushButton(openMenu, "openConfig", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_OPEN_CONFIGURATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(parameterFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadParMnemonic); n++;
	loadParameterButton = XmCreatePushButton(openMenu, "openParameter", args, n);
	XtAddCallback(loadParameterButton, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_OPEN_PARAMETER);
	XtManageChild(loadParameterButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(openString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, openMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, openMenu); n++;
	button = XmCreateCascadeButton(fileMenu, "open", args, n);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'save' */

	n = 0;
	saveMenu = XmCreatePulldownMenu (fileMenu, "saveMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadSimMnemonic); n++;
	button = XmCreatePushButton(saveMenu, "saveSimulation", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_SIMULATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(configurationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadConMnemonic); n++;
	button = XmCreatePushButton(saveMenu, "saveConfig", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_CONFIGURATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(parameterFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadParMnemonic); n++;
	button = XmCreatePushButton(saveMenu, "saveParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_PARAMETER);
	XtManageChild(button);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(saveString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, saveMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, saveMenu); n++;
	saveButton = XmCreateCascadeButton(fileMenu, "save", args, n);
	XtManageChild(saveButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'save as ...' */

	n = 0;
	saveAsMenu = XmCreatePulldownMenu (fileMenu, "saveAsMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadSimMnemonic); n++;
	button = XmCreatePushButton(saveAsMenu, "saveAsSimulation", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_SIMULATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(configurationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadConMnemonic); n++;
	button = XmCreatePushButton(saveAsMenu, "saveAsConfig", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_CONFIGURATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(parameterFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadParMnemonic); n++;
	button = XmCreatePushButton(saveAsMenu, "saveAsParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_PARAMETER);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(saveAsString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, saveAsMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, saveAsMenu); n++;
	saveAsButton = XmCreateCascadeButton(fileMenu, "saveAs", args, n);
	XtManageChild(saveAsButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'close' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(closeString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, closeMnemonic); n++;
	button = XmCreatePushButton(fileMenu, "close", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_CLOSE);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'delete' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(removeString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, deleteMnemonic); n++;
	button = XmCreatePushButton(fileMenu, "delete", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &deleteFileDialog);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* create file selection box for delete file */

	xmstrings[0] = XmStringCreateLtoR(deleteFileString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(filesString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(filterString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(selectionString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(deleteFileDirectory, defaultFont);
	xmstrings[5] = XmStringCreateLtoR(okString, defaultFont);
	xmstrings[6] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[7] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNlistLabelString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNfilterLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNselectionLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNdirMask, xmstrings[4]); n++;
	XtSetArg(args[n], XmNlistVisibleItemCount, 10); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[5]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[6]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[7]); n++;
	deleteFileDialog = XmCreateFileSelectionDialog(parent, "delete file selection dialog", args, n);
	XtAddCallback(deleteFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FDSELECT);
	XtAddCallback(deleteFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &deleteFileDialog);
	XtAddCallback(deleteFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FDSELECT);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);
	XmStringFree(xmstrings[5]);
	XmStringFree(xmstrings[6]);
	XmStringFree(xmstrings[7]);

	/* pulldown menu point 'print' */

	n = 0;
	printMenu = XmCreatePulldownMenu (fileMenu, "printMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadSimMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printSimulation", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_SIMULATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(configurationFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadConMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printConfig", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_CONFIGURATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(parameterFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, loadParMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_PARAMETER);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(generalParamFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, printNeuMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printNeuParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_NEURON);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(graphOutputFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, printGraMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printGraParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_GRAFIK);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(otherFileString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, printOthMnemonic); n++;
	button = XmCreatePushButton(printMenu, "printOthParameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_PRINT_OTHER);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(printString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, printMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, printMenu); n++;
	button = XmCreateCascadeButton(fileMenu, "print", args, n);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* separator */

	n = 0;
	XtManageChild(XmCreateSeparator(fileMenu, "separator", args, n));

	/* pulldown menu point 'exit' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(exitString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, exitMnemonic); n++;
	XtSetArg(args[n], XmNaccelerator, "<Key>F10:"); n++;
	XtSetArg(args[n], XmNacceleratorText, XmStringCreateLtoR("F10", defaultFont)); n++;
	button = XmCreatePushButton(fileMenu, "exit", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_CHECK_EXIT);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* create question dialog for exit */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	exitDialog = XmCreateDialogShell(parent, "exitDialog", args, n);

	xmstrings[0] = XmStringCreateLtoR(exitString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(areYouShureString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(yesString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL); n++;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_QUESTION); n++;
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
	exitMessageBox = XmCreateMessageBox(exitDialog, "exitBox", args, n);
	XtAddCallback(exitMessageBox, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_EXIT);
	XtAddCallback(exitMessageBox, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &exitMessageBox);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(exitMessageBox, args, n);

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(exitDialog), "default_xm_question", fg, bg)); n++;
	XtSetValues(exitMessageBox, args, n);

	/* help button is not needed, remove him */

	button = XmMessageBoxGetChild(exitMessageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	/* create question dialog for save simulation */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	saveSimDialog = XmCreateDialogShell(parent, "saveSimDialog", args, n);

	xmstrings[0] = XmStringCreateLtoR(saveSimString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(saveSimQuestionString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(yesString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL); n++;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_QUESTION); n++;
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
	saveSimMessageBox = XmCreateMessageBox(saveSimDialog, "saveSimBox", args, n);
	XtAddCallback(saveSimMessageBox, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_SIM);
	XtAddCallback(saveSimMessageBox, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &saveSimMessageBox);
	XtAddCallback(saveSimMessageBox, XmNcancelCallback, manage_link_CB, (caddr_t *) &exitMessageBox);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(saveSimMessageBox, args, n);

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(saveSimDialog), "default_xm_question", fg, bg)); n++;
	XtSetValues(saveSimMessageBox, args, n);

	/* help button is not needed, remove him */

	button = XmMessageBoxGetChild(saveSimMessageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	/* create question dialog for save simulation and close file */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	saveSimCloseDialog = XmCreateDialogShell(parent, "saveSimCloseDialog", args, n);

	xmstrings[0] = XmStringCreateLtoR(saveSimString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(saveSimQuestionString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(yesString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL); n++;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_QUESTION); n++;
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
	saveSimCloseMessageBox = XmCreateMessageBox(saveSimCloseDialog, "saveSimCloseBox", args, n);
	XtAddCallback(saveSimCloseMessageBox, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_SIM_AND_CLOSE);
	XtAddCallback(saveSimCloseMessageBox, XmNcancelCallback, dialog_accept_CB, (caddr_t *) DIALOG_CLOSE);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(saveSimCloseMessageBox, args, n);

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(saveSimCloseDialog), "default_xm_question", fg, bg)); n++;
	XtSetValues(saveSimCloseMessageBox, args, n);

	/* help button is not needed, remove him */

	button = XmMessageBoxGetChild(saveSimCloseMessageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	/* menupoint 'options' mit pulldown menu */

	n = 0;
	optionMenu = XmCreatePulldownMenu(menubar, "optionsmenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(optionString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, optionsMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, optionMenu); n++;
	XtManageChild(XmCreateCascadeButton(menubar, "options", args, n));
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'simulation information' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationInfoString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, simInfMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	simInfButton = XmCreateToggleButtonGadget(optionMenu, "simulationInfoOption", args, n);
	XtAddCallback(simInfButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_SIMINF);
	XtManageChild(simInfButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'processor time' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationProcessingString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, timeMessureMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	timeMessureButton = XmCreateToggleButtonGadget(optionMenu, "simulationProcessorOption", args, n);
	XtAddCallback(timeMessureButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_PROCESS_INF);
	XtManageChild(timeMessureButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'show output immediately' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(showOutputImmediatelyString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, showOutputImmediatelyMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	graphDisplayButton = XmCreateToggleButtonGadget(optionMenu, "showOutputImmediatelyOption", args, n);
	XtAddCallback(graphDisplayButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_SHOW_OUTPUT);
	XtManageChild(graphDisplayButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'automatic network reset' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(autoResetString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, autoResetMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	autoResetButton = XmCreateToggleButtonGadget(optionMenu, "autoResetOption", args, n);
	XtAddCallback(autoResetButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_AUTO_RESET);
	XtManageChild(autoResetButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'neuron activity colors' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(neuronActivityColorsString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, neuronActivityColorsMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	neuronActivityColorsButton = XmCreateToggleButtonGadget(optionMenu, "neuronActivityColorsOption", args, n);
	XtAddCallback(neuronActivityColorsButton, XmNvalueChangedCallback, dialog_option_button_CB,
		(caddr_t *) DIALOG_NEURON_COLORS);
	XtManageChild(neuronActivityColorsButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'bound checking' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(boundCheckingString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, boundCheckingMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	outputWatchButton = XmCreateToggleButtonGadget(optionMenu, "boundCheckingOption", args, n);
	XtAddCallback(outputWatchButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_BOUND_CHECKING);
	XtManageChild(outputWatchButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'manual graph updating' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(manualUpdatingString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, manualUpdatingMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	manualUpdateButton = XmCreateToggleButtonGadget(optionMenu, "manualUpdatingOption", args, n);
	XtAddCallback(manualUpdateButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_MANUAL_UPDATE);
	XtManageChild(manualUpdateButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'learning' */

	n = 0;
	learnMenu = XmCreatePulldownMenu (optionMenu, "learnMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(shortTermPlasticityString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	plasticityLearnButton = XmCreateToggleButtonGadget(learnMenu, "plasticityLearning", args, n);
	XtAddCallback(plasticityLearnButton, XmNvalueChangedCallback, dialog_learn_button_CB, (caddr_t *) PLASTICITY_BUTTON);
	XtManageChild(plasticityLearnButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(conditionedLearnString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	conditionedLearnButton = XmCreateToggleButtonGadget(learnMenu, "conditonedLearning", args, n);
	XtAddCallback(conditionedLearnButton, XmNvalueChangedCallback, dialog_learn_button_CB, (caddr_t *) CONDITIONED_BUTTON);
	XtManageChild(conditionedLearnButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(hebbLearnString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	hebbLearnButton = XmCreateToggleButtonGadget(learnMenu, "hebbianLearning", args, n);
	XtAddCallback(hebbLearnButton, XmNvalueChangedCallback, dialog_learn_button_CB, (caddr_t *) HEBB_BUTTON);
	XtManageChild(hebbLearnButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(competitiveLearnString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	competitiveLearnButton = XmCreateToggleButtonGadget(learnMenu, "competitiveLearning", args, n);
	XtAddCallback(competitiveLearnButton, XmNvalueChangedCallback, dialog_learn_button_CB, (caddr_t *) COMPETITIVE_BUTTON);
	XtManageChild(competitiveLearnButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(learnString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, learnMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, learnMenu); n++;
	button = XmCreateCascadeButton(optionMenu, "learn", args, n);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'noise' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(noiseString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, noiseMnemonic); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	noiseButton = XmCreateToggleButtonGadget(optionMenu, "noiseOption", args, n);
	XtAddCallback(noiseButton, XmNvalueChangedCallback, dialog_option_button_CB, (caddr_t *) DIALOG_NOISE_ACTIVATE);
	XtManageChild(noiseButton);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'stochastic distribution' */

	n = 0;
	stochasticMenu = XmCreatePulldownMenu (optionMenu, "stochasticMenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(initialPotentialsString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	initialPotentialsButton = XmCreateToggleButtonGadget(stochasticMenu, "initialPotentials", args, n);
	XtAddCallback(initialPotentialsButton, XmNvalueChangedCallback, dialog_stochastic_button_CB,
		(caddr_t *) INITIAL_POTENTIALS_BUTTON);
	XtManageChild(initialPotentialsButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(synapticStrengthsString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	synapticStrengthButton = XmCreateToggleButtonGadget(stochasticMenu, "synapticStrength", args, n);
	XtAddCallback(synapticStrengthButton, XmNvalueChangedCallback, dialog_stochastic_button_CB,
		(caddr_t *) SYNAPTIC_STRENGTH_BUTTON);
	XtManageChild(synapticStrengthButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(synapticTransmissionTimeString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	synapticTransmissionTimeButton = XmCreateToggleButtonGadget(stochasticMenu, "synapticTransmissionTime", args, n);
	XtAddCallback(synapticTransmissionTimeButton, XmNvalueChangedCallback, dialog_stochastic_button_CB,
		(caddr_t *) SYNAPTIC_TRANSMISSION_TIME_BUTTON);
	XtManageChild(synapticTransmissionTimeButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(stochasticDistributionString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, stochasticDistributionMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, stochasticMenu); n++;
	button = XmCreateCascadeButton(optionMenu, "stochastic", args, n);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'integration methods' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(integrationMethodsString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, integrationMethodsMnemonic); n++;
	button = XmCreatePushButton(optionMenu, "integrationMethods", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &integrationMethodsForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* create dialog for integration methods */

	n = 0;
	XtSetArg(args[n], XmNtitle, integrationMethodsString); n++;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	integrationMethodsDialog = XmCreateDialogShell (parent, "integrationMethodsDialog", args, n);

	n = 0;
	integrationMethodsForm = XmCreateForm (integrationMethodsDialog, "integrationMethodsForm", args, n);

	n = 0;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	integrationMethodsRadioBox = XmCreateRadioBox(integrationMethodsForm, "integrationMethodsRadioBox", args, n);
	XtManageChild(integrationMethodsRadioBox);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(eulerString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	eulerButton = XmCreateToggleButtonGadget(integrationMethodsRadioBox, "eulerButton", args, n);
	XtAddCallback(eulerButton, XmNvalueChangedCallback, dialog_integrationMethods_button_CB, (caddr_t *) EULER_BUTTON);
	XtManageChild(eulerButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(expEulerString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	XtSetArg(args[n], XmNset, TRUE); n++;
	expEulerButton = XmCreateToggleButtonGadget(integrationMethodsRadioBox, "expEulerButton", args, n);
	XtAddCallback(expEulerButton, XmNvalueChangedCallback, dialog_integrationMethods_button_CB, (caddr_t *) EXPEULER_BUTTON);
	XtManageChild(expEulerButton);
	XmStringFree(xmstrings[0]);
	integrationMethod = 2;	/* default value for integrationMethod */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(trapezoidString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	trapezoidButton = XmCreateToggleButtonGadget(integrationMethodsRadioBox, "trapezoidButton", args, n);
	XtAddCallback(trapezoidButton, XmNvalueChangedCallback, dialog_integrationMethods_button_CB, (caddr_t *) TRAPEZOID_BUTTON);
	XtManageChild(trapezoidButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(rungeKuttaString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	rungeKuttaButton = XmCreateToggleButtonGadget(integrationMethodsRadioBox, "rungeKuttaButton", args, n);
	XtAddCallback(rungeKuttaButton, XmNvalueChangedCallback, dialog_integrationMethods_button_CB, (caddr_t *) RUNGEKUTTA_BUTTON);
	XtManageChild(rungeKuttaButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, integrationMethodsRadioBox); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNrightOffset, 10); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	separator = XmCreateSeparator(integrationMethodsForm, "integrationMethodsSeparator", args, n);
	XtManageChild(separator);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, separator); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	button = XmCreatePushButton(integrationMethodsForm, "integrationMethodsOkButton", args, n);
	XtAddCallback(button, XmNactivateCallback, unmanage_link_CB, (caddr_t *) &integrationMethodsForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'window update' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(windowUpdateString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, windowUpdateMnemonic); n++;
	windowUpdateButton = XmCreatePushButton(optionMenu, "windowUpdateButton", args, n);
	XtAddCallback(windowUpdateButton, XmNactivateCallback, dialog_option_button_CB, (caddr_t *) DIALOG_WINDOW_UPDATE);
	XtManageChild(windowUpdateButton);
	XmStringFree(xmstrings[0]);

	/* menu point 'network' with pulldown menu */

	n = 0;
	networkMenu = XmCreatePulldownMenu(menubar, "networkmenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(networkString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, networkMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, networkMenu); n++;
	XtManageChild(XmCreateCascadeButton(menubar, "network", args, n));
	XmStringFree(xmstrings[0]);

	/* pulldown menupoint 'model' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(modelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, modelMnemonic); n++;
	button = XmCreatePushButton(networkMenu, "model", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &modelForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* create dialog for model */

	n = 0;
	XtSetArg(args[n], XmNtitle, modelString); n++;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	modelDialog = XmCreateDialogShell (parent, "modelDialog", args, n);

	n = 0;
	modelForm = XmCreateForm (modelDialog, "modelForm", args, n);

	n = 0;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	modelRadioBox = XmCreateRadioBox(modelForm, "modelRadioBox", args, n);
	XtManageChild(modelRadioBox);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(onOffModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkOnOffButton = XmCreateToggleButtonGadget(modelRadioBox, "networkOnOffButton", args, n);
	XtAddCallback(networkOnOffButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) ONOFF_BUTTON);
	XtManageChild(networkOnOffButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(HHModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkHHButton = XmCreateToggleButtonGadget(modelRadioBox, "networkHHButton", args, n);
	XtAddCallback(networkHHButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) HH_BUTTON);
	XtManageChild(networkHHButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(SWIMModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNset, TRUE); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkSWIMButton = XmCreateToggleButtonGadget(modelRadioBox, "networkSWIMButton", args, n);
	XtAddCallback(networkSWIMButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) SWIM_BUTTON);
	XtManageChild(networkSWIMButton);
	XmStringFree(xmstrings[0]);
	networkmodel = 2;	/* the SWIM model is the default value for network model */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(GBModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkGBButton = XmCreateToggleButtonGadget(modelRadioBox, "networkGBButton", args, n);
	XtAddCallback(networkGBButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) GB_BUTTON);
	XtManageChild(networkGBButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, modelRadioBox); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNrightOffset, 10); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	separator = XmCreateSeparator(modelForm, "modelSeparator", args, n);
	XtManageChild(separator);

	n = 0;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, separator); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	xmstrings[0] = XmStringCreateLtoR(complexModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	XtSetArg(args[n], XmNset, FALSE); n++;
	networkComplexButton = XmCreateToggleButtonGadget(modelForm, "networkComplexButton", args, n);
	XtAddCallback(networkComplexButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) COMPLEX_BUTTON);
	XtManageChild(networkComplexButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, networkComplexButton); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNrightOffset, 10); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	separator = XmCreateSeparator(modelForm, "modelSeparator2", args, n);
	XtManageChild(separator);

	n = 0;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, separator); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	modelRadioBox = XmCreateRadioBox(modelForm, "modelRadioBox2", args, n);
	XtManageChild(modelRadioBox);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(point1String, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkType1Button = XmCreateToggleButtonGadget(modelRadioBox, "networkType1Button", args, n);
	XtAddCallback(networkType1Button, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) POINT1_BUTTON);
	XtManageChild(networkType1Button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(point2String, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkType2Button = XmCreateToggleButtonGadget(modelRadioBox, "networkType2Button", args, n);
	XtAddCallback(networkType2Button, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) POINT2_BUTTON);
	XtManageChild(networkType2Button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(point4String, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNset, TRUE); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkType4Button = XmCreateToggleButtonGadget(modelRadioBox, "networkType4Button", args, n);
	XtAddCallback(networkType4Button, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) POINT4_BUTTON);
	XtManageChild(networkType4Button);
	XmStringFree(xmstrings[0]);
	networktype = 3; /* default value for network type */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(pointNString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	networkTypeNButton = XmCreateToggleButtonGadget(modelRadioBox, "networkTypeNButton", args, n);
	XtAddCallback(networkTypeNButton, XmNvalueChangedCallback, dialog_model_button_CB, (caddr_t *) POINTN_BUTTON);
	XtManageChild(networkTypeNButton);
	XmStringFree(xmstrings[0]);

	n = 0;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, modelRadioBox); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNrightOffset, 10); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	separator = XmCreateSeparator(modelForm, "modelSeparator2", args, n);
	XtManageChild(separator);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, separator); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	button = XmCreatePushButton(modelForm, "modelOkButton", args, n);
	XtAddCallback(button, XmNactivateCallback, unmanage_link_CB, (caddr_t *) &modelForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'reset' */

	n = 0;
	XtSetArg(args[n], XmNtitle, resetString); n++;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	resetDialog = XmCreateDialogShell (parent, "resetDialog", args, n);

	n = 0;
	resetForm = XmCreateForm (resetDialog, "resetForm", args, n);

	n = 0;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	resetRadioBox = XmCreateRadioBox(resetForm, "resetRadioBox", args, n);
	XtManageChild(resetRadioBox);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(withoutLearnParameterString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	button = XmCreateToggleButtonGadget(resetRadioBox, "resetWithoutLearningButton", args, n);
	XtAddCallback(button, XmNvalueChangedCallback, dialog_accept_CB,
		(caddr_t *) DIALOG_WITHOUT_LEARNPARAMETER_BUTTON);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(inclLearnParameterString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNselectColor, selectedToggleButtonColor); n++;
	XtSetArg(args[n], XmNset, TRUE); n++;
	button = XmCreateToggleButtonGadget(resetRadioBox, "resetWithLearningButton", args, n);
	XtAddCallback(button, XmNvalueChangedCallback, dialog_accept_CB, (caddr_t *) DIALOG_INCL_LEARNPARAMETER_BUTTON);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);
	resetLearningParameter = TRUE;	/* default value for reset parameter method */

	n = 0;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, resetRadioBox); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNrightOffset, 10); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	separator = XmCreateSeparator(resetForm, "resetSeparator", args, n);
	XtManageChild(separator);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopOffset, 10); n++;
	XtSetArg(args[n], XmNtopWidget, separator); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNleftOffset, 10); n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE); n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
	button = XmCreatePushButton(resetForm, "resetOkButton", args, n);
	XtAddCallback(button, XmNactivateCallback, unmanage_link_CB, (caddr_t *) &resetForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(resetString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, resetMnemonic); n++;
	button = XmCreatePushButton(networkMenu, "reset", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &resetForm);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'clear' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(clearString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, clearMnemonic); n++;
	button = XmCreatePushButton(networkMenu, "clear", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_CLEAR_NETWORK);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* menu point 'simulation' with pulldown menu */

	n = 0;
	simulationMenu = XmCreatePulldownMenu(menubar, "simulationsmenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, simulationMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, simulationMenu); n++;
	XtManageChild(XmCreateCascadeButton(menubar, "simulation", args, n));
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'simulation parameter' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationParameterString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, simParsMnemonic); n++;
	button = XmCreatePushButton(simulationMenu, "simulation parameter", args, n);
	XtAddCallback(button, XmNactivateCallback, edit_sim_pars_CB, (caddr_t *) NULL);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'noise parameter' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(noiseParameterString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, noiseParsMnemonic); n++;
	button = XmCreatePushButton(simulationMenu, "noise parameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_NOISE_INIT);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'distribution parameter' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(distributionParameterString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, distributionParsMnemonic); n++;
	button = XmCreatePushButton(simulationMenu, "distribution parameter", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_STDDEV_INIT);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* pulldown menu point 'start simulation' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(simulationStartString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, simStartMnemonic); n++;
	button = XmCreatePushButton(simulationMenu, "simulation run", args, n);
	XtAddCallback(button, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_SIMULATION);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* menu point 'help' with pulldown menu */

	n = 0;
	helpMenu = XmCreatePulldownMenu(menubar, "helpmenu", args, n);

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(helpString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, helpMnemonic); n++;
	XtSetArg(args[n], XmNsubMenuId, helpMenu); n++;
	button = XmCreateCascadeButton(menubar, "help", args, n);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	XtSetArg(args[n], XmNmenuHelpWidget, button); n++;
	XtSetValues(menubar, args, n);

	/* submenu point 'help index' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(indexString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, indexMnemonic); n++;
	XtSetArg(args[n], XmNaccelerator, "<Key>F1:"); n++;
	XtSetArg(args[n], XmNacceleratorText, XmStringCreateLtoR("F1", defaultFont)); n++;
	button = XmCreatePushButton ( helpMenu, "helpIndexButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_INDEX);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help features' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(featuresString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, featuresMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpFeaturesButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_FEATURES);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help intro' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(introString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, introMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpIntroButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_INTRO);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help target' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(targetString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, targetMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpTargetButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_TARGET);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help cellModel' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(cellModelString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, cellModelMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpCellModelButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_CELLMODEL);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help context' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(contextString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, contextMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpContextButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_CONTEXT);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help mouse' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(mouseString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, mouseMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpMouseButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_MOUSE);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help keyboard' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(keyboardString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, keyboardMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpKeyboardButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_KEYBOARD);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* submenu point 'help glossary' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(glossaryString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, glossaryMnemonic); n++;
	button = XmCreatePushButton ( helpMenu, "helpGlossaryButton", args, n );
	XtAddCallback (button, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_HELP_GLOSSAR);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	n = 0;
	separator = XmCreateSeparator ( helpMenu, "helpSeparator", args, n );
	XtManageChild(separator);

	/* submenu point 'over biosim' */

	n = 0;
	xmstrings[0] = XmStringCreateLtoR(overBiosimString, defaultFont);
	XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmnemonic, aboutBiosimMnemonic); n++;
	button = XmCreatePushButton(helpMenu, "ueber Biosim", args, n);
	XtAddCallback(button, XmNactivateCallback, manage_link_CB, (caddr_t *) &overBiosimDialog);
	XtManageChild(button);
	XmStringFree(xmstrings[0]);

	/* information text over biosim */

	xmstrings[0] = XmStringCreateLtoR(overBiosimText1String, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(versionString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR("\n\n", defaultFont);
	xmstrings[3] = XmStringCreateLtoR(overBiosimText2String, defaultFont);
	xmstrings[4] = XmStringConcat(xmstrings[0], xmstrings[1]);
	xmstrings[4] = XmStringConcat(xmstrings[4], xmstrings[2]);
	xmstrings[4] = XmStringConcat(xmstrings[4], xmstrings[3]);

	/* create information message box */

	overBiosimDialog = create_information_box(mainWindow, XmDIALOG_MODELESS, xmstrings[4]);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);

	/* return widget id of menubar */

	return(menubar);

} /* end of create_menubar */

/************************************************************************************************************************

 	FUNCTION	: create_askModelPoint(parent)
 
	PURPOSE		: create a dialog box for asking the point model

	RETURNS		: nothing

************************************************************************************************************************/

void create_askModelPoint( parent )
Widget parent;
{
	Widget children[4];	/* Children to manage */
	Arg al[64];		/* Arg List */
	register int ac = 0;	/* Arg Count */
	char text[20];		/* used for number conversion */
	XmString xmstrings[10];	/* temporary storage for XmStrings */
	Widget	askModelPoint,
		askModelPointForm,
		askModelPointLabel,
		askModelPointSeparator,
		askModelPointButton;	/* used for internal use */

	XtSetArg(al[ac], XmNallowShellResize, TRUE); ac++;
	XtSetArg(al[ac], XmNtitle, modelString); ac++;
	askModelPoint = XmCreateDialogShell ( parent, "askModelPoint", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNmarginWidth, 10); ac++;
	XtSetArg(al[ac], XmNmarginHeight, 10); ac++;
	XtSetArg(al[ac], XmNautoUnmanage, TRUE); ac++;
	askModelPointBB = XmCreateBulletinBoard ( askModelPoint, "askModelPointBB", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	askModelPointForm = XmCreateForm ( askModelPointBB, "askModelPointForm", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(numberOfDendriticSegmentsString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	askModelPointLabel = XmCreateLabel ( askModelPointForm, "askModelPointLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	sprintf(text, "%d", curNumOfDendSegs);
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNvalue, text); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	askModelPointText = XmCreateText ( askModelPointForm, "askModelPointText", al, ac );

	ac = 0;
	askModelPointSeparator = XmCreateSeparator ( askModelPointForm, "askModelPointSeparator", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	askModelPointButton = XmCreatePushButton ( askModelPointForm, "askModelPointButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( askModelPointLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftOffset, 10); ac++;
	XtSetArg(al[ac], XmNleftWidget, askModelPointLabel); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( askModelPointText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, askModelPointText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
        XtSetValues ( askModelPointSeparator,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, askModelPointSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( askModelPointButton,al, ac );

	XmAddTabGroup ( askModelPointForm );
	XmAddTabGroup ( askModelPointText );
	XtAddCallback (askModelPointButton, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_MODEL_TYPE);

	ac = 0;
	children[ac++] = askModelPointLabel;
	children[ac++] = askModelPointText;
	children[ac++] = askModelPointSeparator;
	children[ac++] = askModelPointButton;
	XtManageChildren(children, ac);

	ac = 0;
	children[ac++] = askModelPointForm;
	XtManageChildren(children, ac);

} /* end of create_askModelPoint */

/************************************************************************************************************************

 	FUNCTION	: create_noiseDialog(parent)
 
	PURPOSE		: creates a dialog box for noise parameters

	RETURNS		: nothing

************************************************************************************************************************/

void create_noiseDialog( parent )
Widget parent;
{
	Widget children[10];      	/* Children to manage */
	Arg al[64];           		/* Arg List */
	register int ac = 0; 		/* Arg Count */
	XmString xmstrings[10]; 	/* temporary storage for XmStrings */
	Widget	noiseDialog,
		noiseForm,
		noiseIntensityLabel,
		noiseStrengthLabel,
		noiseDecayLabel,
		noiseSeparator,
		noiseOkButton,
		noiseNoButton,
		noiseHelpButton;	/* internal used widgets */

	XtSetArg(al[ac], XmNtitle, noiseParameterString); ac++;
	XtSetArg(al[ac], XmNallowShellResize, TRUE); ac++;
	noiseDialog = XmCreateDialogShell ( parent, "noiseDialog", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNautoUnmanage, TRUE); ac++;
	XtSetArg(al[ac], XmNmarginWidth, 10); ac++;
	XtSetArg(al[ac], XmNmarginHeight, 10); ac++;
	noiseBB = XmCreateBulletinBoard ( noiseDialog, "noiseBB", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	noiseForm = XmCreateForm ( noiseBB, "noiseForm", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(noiseIntensityString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseIntensityLabel = XmCreateLabel ( noiseForm, "noiseIntensityLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	noiseIntensityText = XmCreateText ( noiseForm, "noiseIntensityText", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(noiseStrengthString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseStrengthLabel = XmCreateLabel ( noiseForm, "noiseStrengthLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	noiseStrengthText = XmCreateText ( noiseForm, "noiseStrengthText", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(noiseDecayString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseDecayLabel = XmCreateLabel ( noiseForm, "noiseDecayLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	noiseDecayText = XmCreateText ( noiseForm, "noiseDecayText", al, ac );

	ac = 0;
	noiseSeparator = XmCreateSeparator ( noiseForm, "noiseSeparator", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseOkButton = XmCreatePushButton ( noiseForm, "noiseOkButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	xmstrings[0] = XmStringCreateLtoR(noString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseNoButton = XmCreatePushButton ( noiseForm, "noiseNoButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	xmstrings[0] = XmStringCreateLtoR(helpString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	noiseHelpButton = XmCreatePushButton ( noiseForm, "noiseHelpButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseIntensityLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftOffset, 60); ac++;
	XtSetArg(al[ac], XmNleftWidget, noiseIntensityLabel); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseIntensityText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseIntensityText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseStrengthLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseStrengthLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, noiseIntensityText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseStrengthText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseStrengthText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseDecayLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseDecayLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, noiseIntensityText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseDecayText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseDecayText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
        XtSetValues ( noiseSeparator,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( noiseOkButton,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
	XtSetArg(al[ac], XmNrightPosition, 58); ac++;
        XtSetValues ( noiseNoButton,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, noiseSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
        XtSetValues ( noiseHelpButton,al, ac );

	XmAddTabGroup ( noiseForm );
	XmAddTabGroup ( noiseIntensityText );
	XmAddTabGroup ( noiseStrengthText );
	XmAddTabGroup ( noiseDecayText );

	XtAddCallback (noiseIntensityText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_NOISE_INTENSITY);
	XtAddCallback (noiseStrengthText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_NOISE_STRENGTH);
	XtAddCallback (noiseDecayText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_NOISE_DECAY);
	XtAddCallback (noiseOkButton, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_NOISE);
	XtAddCallback (noiseNoButton, XmNactivateCallback, unmanage_link_CB, (caddr_t *) &noiseBB);
	XtAddCallback (noiseHelpButton, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_NOISE);

	ac = 0;
	children[ac++] = noiseIntensityLabel;
	children[ac++] = noiseIntensityText;
	children[ac++] = noiseStrengthLabel;
	children[ac++] = noiseStrengthText;
	children[ac++] = noiseDecayLabel;
	children[ac++] = noiseDecayText;
	children[ac++] = noiseSeparator;
	children[ac++] = noiseOkButton;
	children[ac++] = noiseNoButton;
	children[ac++] = noiseHelpButton;
	XtManageChildren(children, ac);

	ac = 0;
	children[ac++] = noiseForm;
	XtManageChildren(children, ac);

} /* end of create_noiseDialog */

/************************************************************************************************************************

 	FUNCTION	: create_stddevDialog(parent)
 
	PURPOSE		: creates a dialog box for standard deviation parameters

	RETURNS		: nothing

************************************************************************************************************************/

void create_stddevDialog( parent )
Widget parent;
{
	Widget children[17];      	/* Children to manage */
	Arg al[64];           		/* Arg List */
	register int ac = 0;      	/* Arg Count */
	XmString xmstrings[10]; 	/* temporary storage for XmStrings */
	Widget	stddevDialog,
		stddevForm,
		stddevSTDLabel,
		stddevIPotLabel,
		stddevSynGLabel,
		stddevSynSLabel,
		stddevSynTLabel,
		stddevSeparator,
		stddevOkButton,
		stddevNoButton,
		stddevHelpButton;	/* widgets used for internal use */

	XtSetArg(al[ac], XmNtitle, distributionParameterString); ac++;
	XtSetArg(al[ac], XmNallowShellResize, TRUE); ac++;
	stddevDialog = XmCreateDialogShell ( parent, "stddevDialog", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNautoUnmanage, TRUE); ac++;
	XtSetArg(al[ac], XmNmarginWidth, 10); ac++;
	XtSetArg(al[ac], XmNmarginHeight, 10); ac++;
	stddevBB = XmCreateBulletinBoard ( stddevDialog, "stddevBB", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevForm = XmCreateForm ( stddevBB, "stddevForm", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(standardDeviationString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	stddevSTDLabel = XmCreateLabel ( stddevForm, "stddevSTDLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(initialPotentialString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	stddevIPotLabel = XmCreateLabel ( stddevForm, "stddevIPotLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevIPotText = XmCreateText ( stddevForm, "stddevIPotText", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevIPotStdText = XmCreateText ( stddevForm, "stddevIPotStdText", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(synapticStrengthString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	stddevSynGLabel = XmCreateLabel ( stddevForm, "stddevSynGLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynGText = XmCreateText ( stddevForm, "stddevSynGText", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynGStdText = XmCreateText ( stddevForm, "stddevSynGStdText", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(axoAxonicStrengthString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	stddevSynSLabel = XmCreateLabel ( stddevForm, "stddevSynSLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynSText = XmCreateText ( stddevForm, "stddevSynSText", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynSStdText = XmCreateText ( stddevForm, "stddevSynSStdText", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(synapticTransmissionString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	stddevSynTLabel = XmCreateLabel ( stddevForm, "stddevSynTLabel", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynTText = XmCreateText ( stddevForm, "stddevSynTText", al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNcolumns, 10); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevSynTStdText = XmCreateText ( stddevForm, "stddevSynTStdText", al, ac );

	ac = 0;
	stddevSeparator = XmCreateSeparator ( stddevForm, "stddevSeparator", al, ac );

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(okString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevOkButton = XmCreatePushButton ( stddevForm, "stddevOkButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(noString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevNoButton = XmCreatePushButton ( stddevForm, "stddevNoButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	xmstrings[0] = XmStringCreateLtoR(helpString, defaultFont);
	XtSetArg(al[ac], XmNlabelString, xmstrings[0]); ac++;
	XtSetArg(al[ac], XmNtraversalOn, TRUE); ac++;
	stddevHelpButton = XmCreatePushButton ( stddevForm, "stddevHelpButton", al, ac );
	XmStringFree(xmstrings[0]);

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftOffset, 200); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotLabel); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSTDLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNtopOffset, 30); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevIPotLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevIPotLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftOffset, 110); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotLabel); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevIPotText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevIPotText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftOffset, 20); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevSTDLabel); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevIPotStdText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevIPotText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynGLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynGLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynGText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynGText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotStdText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynGStdText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynGText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynSLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynSLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynSText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynSText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotStdText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynSStdText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynSText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynTLabel,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynTLabel); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynTText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynTText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(al[ac], XmNleftWidget, stddevIPotStdText); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevSynTStdText,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSynTText); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
        XtSetValues ( stddevSeparator,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_NONE); ac++;
        XtSetValues ( stddevOkButton,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
	XtSetArg(al[ac], XmNrightPosition, 55); ac++;
        XtSetValues ( stddevNoButton,al, ac );

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(al[ac], XmNtopOffset, 10); ac++;
	XtSetArg(al[ac], XmNtopWidget, stddevSeparator); ac++;
	XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_NONE); ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
        XtSetValues ( stddevHelpButton,al, ac );

	XmAddTabGroup ( stddevForm );
	XmAddTabGroup ( stddevIPotText );
	XmAddTabGroup ( stddevIPotStdText );
	XmAddTabGroup ( stddevSynGText );
	XmAddTabGroup ( stddevSynGStdText );
	XmAddTabGroup ( stddevSynSText );
	XmAddTabGroup ( stddevSynSStdText );
	XmAddTabGroup ( stddevSynTText );
	XmAddTabGroup ( stddevSynTStdText );

	XtAddCallback (stddevIPotText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_IPOT);
	XtAddCallback (stddevIPotStdText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_IPOT_STDDEV);
	XtAddCallback (stddevSynGText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_SYNG);
	XtAddCallback (stddevSynGStdText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_SYNG_STDDEV);
	XtAddCallback (stddevSynSText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_AXOS);
	XtAddCallback (stddevSynSStdText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_AXOS_STDDEV);
	XtAddCallback (stddevSynTText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_SYNT);
	XtAddCallback (stddevSynTStdText, XmNvalueChangedCallback, dialog_text_CB, (caddr_t *) DIALOG_TEXT_SYNT_STDDEV);
	XtAddCallback (stddevOkButton, XmNactivateCallback, dialog_accept_CB, (caddr_t *) DIALOG_STDDEV);
	XtAddCallback (stddevNoButton, XmNactivateCallback, unmanage_link_CB, (caddr_t *) &stddevBB);
	XtAddCallback (stddevHelpButton, XmNactivateCallback, dialog_help_CB, (caddr_t *) DIALOG_STDDEV);

	ac = 0;
	children[ac++] = stddevSTDLabel;
	children[ac++] = stddevIPotLabel;
	children[ac++] = stddevIPotText;
	children[ac++] = stddevIPotStdText;
	children[ac++] = stddevSynGLabel;
	children[ac++] = stddevSynGText;
	children[ac++] = stddevSynGStdText;
	children[ac++] = stddevSynSLabel;
	children[ac++] = stddevSynSText;
	children[ac++] = stddevSynSStdText;
	children[ac++] = stddevSynTLabel;
	children[ac++] = stddevSynTText;
	children[ac++] = stddevSynTStdText;
	children[ac++] = stddevSeparator;
	children[ac++] = stddevOkButton;
	children[ac++] = stddevNoButton;
	children[ac++] = stddevHelpButton;
	XtManageChildren(children, ac);

	ac = 0;
	children[ac++] = stddevForm;
	XtManageChildren(children, ac);

} /* end of create_stddevDialog */

/************************************************************************************************************************

 	FUNCTION	: generate_file_selection_box(w, title, dirMask, visibleItemCount)
 
	PURPOSE		: generates a file selection box

	RETURNS		: widget id of file selection box

************************************************************************************************************************/

Widget generate_file_selection_box(w, title, dirMask, visibleItemCount)
Widget w;			/* widget id of parent widget */
XmString title, dirMask;	/* title and directory mask of file selection box */
int visibleItemCount;		/* number of visible items in mask */
{
	Arg		args[20];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		fileSelectionBox;	/* file selection box */

	xmstrings[0] = XmStringCreateLtoR(filesString, defaultFont);
	xmstrings[1] = XmStringCreateLtoR(filterString, defaultFont);
	xmstrings[2] = XmStringCreateLtoR(selectionString, defaultFont);
	xmstrings[3] = XmStringCreateLtoR(okString, defaultFont);
	xmstrings[4] = XmStringCreateLtoR(noString, defaultFont);
	xmstrings[5] = XmStringCreateLtoR(helpString, defaultFont);

	n = 0;
	XtSetArg(args[n], XmNdialogTitle, title); n++;
	XtSetArg(args[n], XmNlistLabelString, xmstrings[0]); n++;
	XtSetArg(args[n], XmNfilterLabelString, xmstrings[1]); n++;
	XtSetArg(args[n], XmNselectionLabelString, xmstrings[2]); n++;
	XtSetArg(args[n], XmNdirMask, dirMask); n++;
	XtSetArg(args[n], XmNlistVisibleItemCount, visibleItemCount); n++;
	XtSetArg(args[n], XmNokLabelString, xmstrings[3]); n++;
	XtSetArg(args[n], XmNcancelLabelString, xmstrings[4]); n++;
	XtSetArg(args[n], XmNhelpLabelString, xmstrings[5]); n++;

	fileSelectionBox = XmCreateFileSelectionDialog(XtParent(w), "fileSelectionBox", args, n);

	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);
	XmStringFree(xmstrings[2]);
	XmStringFree(xmstrings[3]);
	XmStringFree(xmstrings[4]);
	XmStringFree(xmstrings[5]);

	return(fileSelectionBox);

} /* end of generate_file_selection_box */

/************************************************************************************************************************

 	FUNCTION	: set_max_scale()
 
	PURPOSE		: sets the maximal scale widgets of add synapse target, edit synapse target, edit current target

	RETURNS		: nothing

************************************************************************************************************************/

void set_max_scale()
{
	Arg		args[20];			/* argument list for manipulating widgets */
	Cardinal	n;				/* used as argument counter for manipulating widgets */
	char		*value = NULL;			/* used for detection of valid number string */
	int		val;				/* number for ascii to integer conversion */
	short int	decimals;			/* number of decimals for scale widget */

	/* set scale bar of add synapse mask, edit synapse mask and edit current mask to sensitive */

	XtSetSensitive(addSynapseTargetScale, TRUE);
	XtSetSensitive(editSynapseLocationScale, TRUE);
	XtSetSensitive(editCurrentLocationScale, TRUE);

	/* set maximum scale widgets of add synapse, edit synapse, edit current mask to curNumOfDendSegs */

	value = XmTextGetString(addSynapseTargetText);
	val = atoi(value);

	if (val > curNumOfDendSegs)
	{
		sprintf(value, "%i", curNumOfDendSegs);
		XmTextSetString(addSynapseTargetText, value);
		XmScaleSetValue(addSynapseTargetScale, curNumOfDendSegs);
	}

	free(value);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(addSynapseTargetScale, args, n);

	n = 0;
	XtSetArg(args[n], XmNmaximum, (int) (curNumOfDendSegs * pow((double) 10, (double) decimals))); n++;
	XtSetValues(addSynapseTargetScale, args, n);

	value = XmTextGetString(editSynapseLocationText);
	val = atoi(value);

	if (val > curNumOfDendSegs)
	{
		sprintf(value, "%i", curNumOfDendSegs);
		XmTextSetString(editSynapseLocationText, value);
		XmScaleSetValue(editSynapseLocationScale, curNumOfDendSegs);
	}

	free(value);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(editSynapseLocationScale, args, n);

	n = 0;
	XtSetArg(args[n], XmNmaximum, (int) (curNumOfDendSegs * pow((double) 10, (double) decimals))); n++;
	XtSetValues(editSynapseLocationScale, args, n);

	value = XmTextGetString(editCurrentLocationText);
	val = atoi(value);

	if (val > curNumOfDendSegs)
	{
		sprintf(value, "%i", curNumOfDendSegs);
		XmTextSetString(editCurrentLocationText, value);
		XmScaleSetValue(editCurrentLocationScale, curNumOfDendSegs);
	}

	free(value);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(editCurrentLocationScale, args, n);

	n = 0;
	XtSetArg(args[n], XmNmaximum, (int) (curNumOfDendSegs * pow((double) 10, (double) decimals))); n++;
	XtSetValues(editCurrentLocationScale, args, n);

	show_normal_state();
	show_all_objects_in_drawing_area(FALSE);

} /* end of set_max_scale */

/************************************************************************************************************************

 	FUNCTION	: dialog_accept_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for ok button

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_accept_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg		args[20];			/* argument list for manipulating widgets */
	Cardinal	n;				/* used as argument counter for manipulating widgets */
	char 		text[20];			/* used for number conversion */
	char		header[128];			/* used for biosim header */
	char 		temp[128];			/* full pathname and filename */
	char 		*ptr;				/* pointer to look for extension in filename */
	char 		*filename = NULL; 		/* selected file name */
	char 		*command; 	 		/* command used in printing */
	char		*value = NULL;			/* used for detection of valid number string */
	int 		length;				/* length of a number string */
	int 		status;				/* status of deletion */
	int		i;				/* loop variable */
	int		oldNumOfDendSegs;		/* old number of dendritic segments */
	unsigned int	len;				/* length of memory to allocate for neuron entries */
	double		number;				/* number read from text widget */
	XmString 	xmstrings[10]; 			/* temporary storage for XmStrings */
	FILE		*fp = NULL;	 		/* pointer to open file */
	XmSelectionBoxCallbackStruct *select;		/* call data for selection box */
	XmFileSelectionBoxCallbackStruct *selection;	/* call data for file selection box */

	/* evaluate client data */

	switch((int) client_data)
	{
	case DIALOG_NOISE_INIT:	/* initialization of noise parameter */

		sprintf(text,"%f", noiseIntensity);
		XmTextSetString(noiseIntensityText, text);

		sprintf(text,"%f", noiseStrength * I_SCALE);
		XmTextSetString(noiseStrengthText, text);

		sprintf(text,"%f", noiseDecay * T_SCALE);
		XmTextSetString(noiseDecayText, text);

		XtManageChild(noiseBB);

		break;

	case DIALOG_STDDEV_INIT:/* initialization of standdard deviation parameter */

		sprintf(text,"%f", stdInitialPotential * E_SCALE);
		XmTextSetString(stddevIPotText, text);

		sprintf(text,"%f", stdInitialPotentialStd * E_SCALE);
		XmTextSetString(stddevIPotStdText, text);

		sprintf(text,"%f", stdSynapticStrength * S_SCALE);
		XmTextSetString(stddevSynGText, text);

		sprintf(text,"%f", stdSynapticStrengthStd * S_SCALE);
		XmTextSetString(stddevSynGStdText, text);

		sprintf(text,"%f", stdAxoAxonicStrength);
		XmTextSetString(stddevSynSText, text);

		sprintf(text,"%f", stdAxoAxonicStrengthStd);
		XmTextSetString(stddevSynSStdText, text);

		sprintf(text,"%f", stdSynapticTT * T_SCALE);
		XmTextSetString(stddevSynTText, text);

		sprintf(text,"%f", stdSynapticTTStd * T_SCALE);
		XmTextSetString(stddevSynTStdText, text);

		XtManageChild(stddevBB);

		break;

	case DIALOG_NOISE:	/* ok button of noise parameter dialog was selected */

		value = XmTextGetString(noiseIntensityText);
		noiseIntensity = strtod(value, (char **) NULL);

		value = XmTextGetString(noiseStrengthText);
		noiseStrength = strtod(value, (char **) NULL) / I_SCALE;

		value = XmTextGetString(noiseDecayText);
		noiseDecay = strtod(value, (char **) NULL) / T_SCALE;

		XtUnmanageChild(noiseBB);

		break;

	case DIALOG_STDDEV:	/* ok button of standdard deviation parameter dialog was selected */

		value = XmTextGetString(stddevIPotText);
		stdInitialPotential = strtod(value, (char **) NULL) / E_SCALE;

		value = XmTextGetString(stddevIPotStdText);
		stdInitialPotentialStd = strtod(value, (char **) NULL) / E_SCALE;

		value = XmTextGetString(stddevSynGText);
		stdSynapticStrength = strtod(value, (char **) NULL) / S_SCALE;

		value = XmTextGetString(stddevSynGStdText);
		stdSynapticStrengthStd = strtod(value, (char **) NULL) / S_SCALE;

		value = XmTextGetString(stddevSynSText);
		stdAxoAxonicStrength = strtod(value, (char **) NULL);

		value = XmTextGetString(stddevSynSStdText);
		stdAxoAxonicStrengthStd = strtod(value, (char **) NULL);

		value = XmTextGetString(stddevSynTText);
		stdSynapticTT = strtod(value, (char **) NULL) / T_SCALE;

		value = XmTextGetString(stddevSynTStdText);
		stdSynapticTTStd = strtod(value, (char **) NULL) / T_SCALE;

		XtUnmanageChild(stddevBB);

		break;

	case DIALOG_MODEL_TYPE:	/* network model type was selected */

		oldNumOfDendSegs = curNumOfDendSegs;

		value = XmTextGetString(askModelPointText);
		number = strtod(value, (char **) NULL);

		if ((! is_it_a_number(value)) || (number < TARGET_MIN))
		{
			error(74);
			free(value);
			break;
		}
		else curNumOfDendSegs = atoi(value);

		free(value);

		switch(curNumOfDendSegs)
		{
		case 0:	networktype = 1;
			break;

		case 1:	networktype = 2;
			break;

		case 3:	networktype = 3;
			break;
		}

		XtUnmanageChild(askModelPointBB);

		/* calculate new dendrite length and set number of dendritic regions for simulation */

		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs);

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;

			/* reallocate memory for input currents, potentials, ... */

			len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

			currentNeuron->neuronPotentials = (double *) realloc(currentNeuron->neuronPotentials, (size_t) len);

			if (! currentNeuron->neuronPotentials)
			{
				error(114);
				return;
			}

			currentNeuron->currentChemical = (double *) realloc(currentNeuron->currentChemical, (size_t) len);

			if (! currentNeuron->currentChemical)
			{
				error(114);
				return;
			}

			currentNeuron->currentElectrical = (double *) realloc(currentNeuron->currentElectrical, (size_t) len);

			if (! currentNeuron->currentElectrical)
			{
				error(114);
				return;
			}

			len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(InputCurrent));

			currentNeuron->inputCurrent = (InputCurrent *) realloc(currentNeuron->inputCurrent, (size_t) len);

			if (! currentNeuron->inputCurrent)
			{
				error(114);
				return;
			}

			/* initialize pointers in InputCurrent structure */

			for (i = oldNumOfDendSegs; i <= curNumOfDendSegs; i++)
			{
				currentNeuron->inputCurrent[i].starttime = (double *) NULL;
				currentNeuron->inputCurrent[i].current = (double *) NULL;
				currentNeuron->inputCurrent[i].numEntries = 0;
			}
		}

		/* set maximum scale widgets of add synapse, edit synapse and edit current mask to curNumOfDendSegs */

		if (curNumOfDendSegs == 0)
		{
			XmTextSetString(addSynapseTargetText, "0");
			XmTextSetString(editSynapseLocationText, "0");
			XmTextSetString(editCurrentLocationText, "0");
			XmScaleSetValue(addSynapseTargetScale, 0);
			XmScaleSetValue(editSynapseLocationScale, 0);
			XmScaleSetValue(editCurrentLocationScale, 0);

			XtSetSensitive(addSynapseTargetScale, FALSE);
			XtSetSensitive(editSynapseLocationScale, FALSE);
			XtSetSensitive(editCurrentLocationScale, FALSE);

			show_normal_state();
			show_all_objects_in_drawing_area(FALSE);
		}
		else	set_max_scale();

		break;

	case DIALOG_SIMULATION:	/* run simulation */

		/* if no simulation info should be showed run simulation else show dialog box */

		if (! showSimulationInfo)
			run_simulation();
		else
		{
			L = 0.0;
			sprintf(text, "%d", (int) (rint (LTSTOP * T_SCALE * 10.0)));
			length = (int) strlen(text);
			text[length+1] = '\0';
			text[length] = text[length-1];
			text[length-1] = '.';
			XmTextSetString(simInfLengthText, text);
			XmTextSetString(simInfCompletedText, "0.0");
			n = 0;
			xmstrings[0] = XmStringCreateLtoR(stopString, defaultFont);
			XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
			XtSetValues(simInfStopButton, args, n);
			XmStringFree(xmstrings[0]);

			XtManageChild(simInfBB);
		}
		break;
				
	case DIALOG_FILE:	/* check status of network and set sensitive state of menu points */

		if (neuronNumber > 0)	/* exists at least one neuron ? */
		{
			XtSetSensitive(saveButton, TRUE);
			XtSetSensitive(saveAsButton, TRUE);
			XtSetSensitive(loadParameterButton, TRUE);
		}
		else
		{
			XtSetSensitive(saveButton, FALSE);
			XtSetSensitive(saveAsButton, FALSE);
			XtSetSensitive(loadParameterButton, FALSE);
		}
		break;
				
	case DIALOG_OPEN_SIMULATION:	/* display a file selection box for selecting a simulation file */

		xmstrings[0] = XmStringCreateLtoR(openFileTitleSimulationString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(simulationFileDirectory, defaultFont);

		openFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(openFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FSIMSELECT);
		XtAddCallback(openFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &openFileDialog);
		XtAddCallback(openFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FSIMSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(openFileDialog);
		break;

	case DIALOG_OPEN_CONFIGURATION:	/* display a file selection box for selecting a configuration file */

		xmstrings[0] = XmStringCreateLtoR(openFileTitleSimulationString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(configurationFileDirectory, defaultFont);

		openFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(openFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FCONSELECT);
		XtAddCallback(openFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &openFileDialog);
		XtAddCallback(openFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FCONSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(openFileDialog);
		break;

	case DIALOG_OPEN_PARAMETER:	/* display a file selection box for selecting a parameter file */

		xmstrings[0] = XmStringCreateLtoR(openFileTitleSimulationString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(parameterFileDirectory, defaultFont);

		openFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(openFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPARSELECT);
		XtAddCallback(openFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &openFileDialog);
		XtAddCallback(openFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPARSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(openFileDialog);
		break;

	case DIALOG_FSIMSELECT:	/* open the simulation file and read it into the internal structure */

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultFont, &filename);
		strcpy(currentFilename, filename);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, filename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);

		/* read in the file */

		read_network(filename);

		XtUnmanageChild(openFileDialog);

		/* initialize file saved status to true and store filename as current file */

		fileSaved = TRUE;
		strcpy(currentFilename, filename);
		editNetwork = TRUE;
		break;

	case DIALOG_FCONSELECT:	/* open the configuration file and read it into the internal structure */

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultFont, &filename);
		strcpy(currentFilename, filename);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, filename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);

		/* read in the file */

		read_configuration(filename);

		XtUnmanageChild(openFileDialog);

		/* initialize file saved status to true and store filename as current file */

		fileSaved = TRUE;
		strcpy(currentFilename, filename);
		editNetwork = TRUE;
		break;

	case DIALOG_FPARSELECT:	/* open the parameter file and read it into the internal structure */

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultFont, &filename);
		strcpy(currentFilename, filename);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, filename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);

		/* read in the file */

		read_parameter(filename);

		XtUnmanageChild(openFileDialog);

		/* initialize file saved status to true and store filename as current file */

		fileSaved = TRUE;
		strcpy(currentFilename, filename);
		break;

	case DIALOG_FDSELECT:	/* delete the selected files */

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultFont, &filename);

		/* delete the file */

		status = remove(filename);
		if (status != 0)
			error(2);
		XtUnmanageChild(deleteFileDialog);
		break;

	case DIALOG_FPSELECT:	/* print a file selected by the user */

		/* check if selected file exists */

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultFont, &filename);
		if ((fp = fopen(filename, "r")) == NULL)
		{
			XtManageChild(printFileDialog);
			error(1);
			break;
		}

		/* close the file */

		if (fclose(fp) != 0)
			error(101);

		/* print the file */

		command = (char *) malloc((size_t) (sizeof(filename) + sizeof(resData.printCommand) + 2));
		if (! command)
		{
			error(117);
			return;
		}
		sprintf(command, "%s %s", resData.printCommand, filename);
		if (system(command) != 0)
			error(3);
		XtUnmanageChild(printFileDialog);
		break;

	case DIALOG_SAVE_SIMULATION:	/* save the current simulation file */

		strcpy(temp, currentFilename);
		if (! strchr(temp, (int) '.')) /* does no extension exist ? */
		{
			strcat(temp, ".sim");
		}
		else
		{
			temp[strlen(temp) - 4] = '\0'; /* cut extension */
			strcat(temp, ".sim");
		}
		save_network(temp);
		fileSaved = TRUE;
		break;

	case DIALOG_SAVE_CONFIGURATION:	/* save the current configuration file */

		strcpy(temp, currentFilename);
		if (! strchr(temp, (int) '.')) /* does no extension exist ? */
		{
			strcat(temp, ".con");
		}
		else
		{
			temp[strlen(temp) - 4] = '\0'; /* cut extension */
			strcat(temp, ".con");
		}
		save_configuration(temp);
		fileSaved = TRUE;
		break;

	case DIALOG_SAVE_PARAMETER:	/* save the current parameter file */

		strcpy(temp, currentFilename);
		if (! strchr(temp, (int) '.')) /* does no extension exist ? */
		{
			strcat(temp, ".par");
		}
		else
		{
			temp[strlen(temp) - 4] = '\0'; /* cut extension */
			strcat(temp, ".par");
		}
		save_parameter(temp);
		fileSaved = TRUE;
		break;

	case DIALOG_SAVE_AS_SIMULATION:	/* outputs the dialog box for a simulation file */

		xmstrings[0] = XmStringCreateLtoR(saveAsSimPromptString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(saveAsTitleString, defaultFont);
		xmstrings[2] = XmStringCreateLtoR(okString, defaultFont);
		xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
		xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

		n = 0;
		XtSetArg(args[n], XmNselectionLabelString, xmstrings[0]); n++;
		XtSetArg(args[n], XmNdialogTitle, xmstrings[1]); n++;
		XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
		XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
		XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
		saveAsFileDialog = XmCreatePromptDialog(XtParent(w), "saveAsFileDialog", args, n);
		XtAddCallback(saveAsFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_SIM);
		XtAddCallback(saveAsFileDialog, XmNcancelCallback, dialog_cancel_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_SIM);
		XtAddCallback(saveAsFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_SIM);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);
		XmStringFree(xmstrings[2]);
		XmStringFree(xmstrings[3]);
		XmStringFree(xmstrings[4]);

		XtManageChild(saveAsFileDialog);
		break;

	case DIALOG_SAVE_AS_CONFIGURATION:	/* outputs the dialog box for a configuration file */

		xmstrings[0] = XmStringCreateLtoR(saveAsConPromptString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(saveAsTitleString, defaultFont);
		xmstrings[2] = XmStringCreateLtoR(okString, defaultFont);
		xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
		xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

		n = 0;
		XtSetArg(args[n], XmNselectionLabelString, xmstrings[0]); n++;
		XtSetArg(args[n], XmNdialogTitle, xmstrings[1]); n++;
		XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
		XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
		XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
		saveAsFileDialog = XmCreatePromptDialog(XtParent(w), "saveAsFileDialog", args, n);
		XtAddCallback(saveAsFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_CON);
		XtAddCallback(saveAsFileDialog, XmNcancelCallback, dialog_cancel_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_CON);
		XtAddCallback(saveAsFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_CON);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);
		XmStringFree(xmstrings[2]);
		XmStringFree(xmstrings[3]);
		XmStringFree(xmstrings[4]);

		XtManageChild(saveAsFileDialog);
		break;

	case DIALOG_SAVE_AS_PARAMETER:	/* outputs the dialog box for a parameter file */

		xmstrings[0] = XmStringCreateLtoR(saveAsParPromptString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(saveAsTitleString, defaultFont);
		xmstrings[2] = XmStringCreateLtoR(okString, defaultFont);
		xmstrings[3] = XmStringCreateLtoR(noString, defaultFont);
		xmstrings[4] = XmStringCreateLtoR(helpString, defaultFont);

		n = 0;
		XtSetArg(args[n], XmNselectionLabelString, xmstrings[0]); n++;
		XtSetArg(args[n], XmNdialogTitle, xmstrings[1]); n++;
		XtSetArg(args[n], XmNokLabelString, xmstrings[2]); n++;
		XtSetArg(args[n], XmNcancelLabelString, xmstrings[3]); n++;
		XtSetArg(args[n], XmNhelpLabelString, xmstrings[4]); n++;
		saveAsFileDialog = XmCreatePromptDialog(XtParent(w), "saveAsFileDialog", args, n);
		XtAddCallback(saveAsFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_PAR);
		XtAddCallback(saveAsFileDialog, XmNcancelCallback, dialog_cancel_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_PAR);
		XtAddCallback(saveAsFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_SAVE_AS_FILE_PAR);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);
		XmStringFree(xmstrings[2]);
		XmStringFree(xmstrings[3]);
		XmStringFree(xmstrings[4]);

		XtManageChild(saveAsFileDialog);
		break;

	case DIALOG_SAVE_AS_FILE_SIM:	/* save simulation file */

		filename = NULL;
		select = (XmSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(select->value, defaultFont, &filename);
		strcpy(temp, filename);

		if (! strchr(filename, (int) '/')) /* contains the filename no subdirectory ? */
		{
			strcpy(temp, simulationFileDirectory);
			temp[strlen(temp) - 5] = '\0'; /* cut extension */
			strcat(temp, filename);
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".sim");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 's') || (*(ptr + 2) != 'i') || (*(ptr + 3) != 'm'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}
		else
		{
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".sim");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 's') || (*(ptr + 2) != 'i') || (*(ptr + 3) != 'm'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}

		/* save the network to the file */

		save_network(temp);
		fileSaved = TRUE;

		/* generate the new header for the main menu box */

		strcpy(currentFilename,temp);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, currentFilename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);
		break;

	case DIALOG_SAVE_AS_FILE_CON:	/* save configuration file */

		filename = NULL;
		select = (XmSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(select->value, defaultFont, &filename);
		strcpy(temp, filename);

		if (! strchr(filename, (int) '/')) /* contains the filename no subdirectory ? */
		{
			strcpy(temp, configurationFileDirectory);
			temp[strlen(temp) - 5] = '\0'; /* cut extension */
			strcat(temp, filename);
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".con");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 'c') || (*(ptr + 2) != 'o') || (*(ptr + 3) != 'n'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}
		else
		{
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".con");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 'c') || (*(ptr + 2) != 'o') || (*(ptr + 3) != 'n'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}

		/* save configuration to file */

		save_configuration(temp);
		fileSaved = TRUE;

		/* generate the new header for the main menu box */

		strcpy(currentFilename,temp);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, currentFilename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);
		break;

	case DIALOG_SAVE_AS_FILE_PAR:	/* save parameter file */

		filename = NULL;
		select = (XmSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(select->value, defaultFont, &filename);
		strcpy(temp, filename);

		if (! strchr(filename, (int) '/')) /* contains the filename no subdirectory ? */
		{
			strcpy(temp, parameterFileDirectory);
			temp[strlen(temp) - 5] = '\0'; /* cut extension */
			strcat(temp, filename);
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".par");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 'p') || (*(ptr + 2) != 'a') || (*(ptr + 3) != 'r'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}
		else
		{
			if (! strchr(temp, (int) '.')) /* does no extension exist ? */
			{
				strcat(temp, ".par");
			}
			else
			{
				ptr = strrchr(temp, (int) '.');
				if ((*(ptr + 1) != 'p') || (*(ptr + 2) != 'a') || (*(ptr + 3) != 'r'))
				{
					error(4);
					XtManageChild(saveAsFileDialog);
					break;
				}
			}
		}

		/* save parameters to file */

		save_parameter(temp);
		fileSaved = TRUE;

		/* generate the new header for the main menu box */

		strcpy(currentFilename,temp);
		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, currentFilename);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);
		break;

	case DIALOG_CLOSE:	/* close the current file */

		if (! fileSaved)
		{
			XtManageChild(saveSimCloseMessageBox);
			fileSaved = TRUE;
		}
		else
		{ 	/* clear network, unmanage open windows, free memory of graph windows */

			clear_network();

			/* generate the new header for the main menu box */

			strcpy(header, "Biosim ");
			strcat(header, versionString);
			strcat(header, " : ");

			n = 0;
			XtSetArg(args[n], XmNtitle, header); n++;
			XtSetValues(applShell, args, n);

			editNetwork = FALSE;
			fileSaved = TRUE;
			variables_initialized = FALSE;
		}
		break;

	case DIALOG_NEW:	/* open a new untitled file */

		/* clear network, unmanage open windows, free memory of graph windows */

		clear_network();

		/* generate the new header for the main menu box */

		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");
		strcat(header, unnamedFile);

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);

		strcpy(currentFilename, unnamedFile);

		fileSaved = TRUE;
		editNetwork = TRUE;
		variables_initialized = FALSE;

		/*	generate default synaptic types		*/

		synapseTypes = (Synapse *) malloc((size_t) (3 * sizeof(Synapse)));
		if (! synapseTypes)
		{
			error(126);
			break;
		}
		synapseTypes[numTypes].ionType = 2;
		synapseTypes[numTypes++].equilPotential = (double) KEQ_SWIM_DEF / E_SCALE;
		synapseTypes[numTypes].ionType = 1;
		synapseTypes[numTypes++].equilPotential = (double) NAEQ_SWIM_DEF / E_SCALE;
		synapseTypes[numTypes].ionType = 4;
		synapseTypes[numTypes++].equilPotential = (double) 0.0;
		break;

	case DIALOG_PRINT_SIMULATION:	/* outputs dialog box for printing a simulation file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(simulationFileDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_PRINT_CONFIGURATION:	/* outputs dialog box for printing a configuration file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(configurationFileDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_PRINT_PARAMETER:	/* outputs dialog box for printing a parameter file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(parameterFileDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_PRINT_NEURON:	/* outputs dialog box for printing a neuron file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(neuronParameterDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_PRINT_GRAFIK:	/* outputs dialog box for printing a graph file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(graphOutputDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_PRINT_OTHER:	/* outputs dialog box for printing a other file */

		xmstrings[0] = XmStringCreateLtoR(printTitleString, defaultFont);
		xmstrings[1] = XmStringCreateLtoR(otherFileDirectory, defaultFont);

		printFileDialog = generate_file_selection_box(w, xmstrings[0], xmstrings[1], 10);
		XtAddCallback(printFileDialog, XmNokCallback, dialog_accept_CB, (caddr_t *) DIALOG_FPSELECT);
		XtAddCallback(printFileDialog, XmNcancelCallback, unmanage_link_CB, (caddr_t *) &printFileDialog);
		XtAddCallback(printFileDialog, XmNhelpCallback, dialog_help_CB, (caddr_t *) DIALOG_FPSELECT);

		XmStringFree(xmstrings[0]);
		XmStringFree(xmstrings[1]);

		XtManageChild(printFileDialog);
		break;

	case DIALOG_CHECK_EXIT:	/* check if file should be saved before exiting the program */

		if (! fileSaved)
			XtManageChild(saveSimMessageBox);	/* ask if simulation should be saved */
		else 	XtManageChild(exitMessageBox);
		break;

	case DIALOG_SAVE_SIM:	/* save current network to current filename */

		strcpy(temp, currentFilename);
		if (! strchr(temp, (int) '.')) /* does no extension exist ? */
		{
			strcat(temp, ".sim");
		}
		else
		{
			temp[strlen(temp) - 4] = '\0'; /* cut extension */
			strcat(temp, ".sim");
		}
		save_network(temp);
		XtManageChild(exitMessageBox);
		break;

	case DIALOG_SAVE_SIM_AND_CLOSE:	/* save simulation and close current file */

		strcpy(temp, currentFilename);
		if (! strchr(temp, (int) '.')) /* does no extension exist ? */
		{
			strcat(temp, ".sim");
		}
		else
		{
			temp[strlen(temp) - 4] = '\0'; /* cut extension */
			strcat(temp, ".sim");
		}
		save_network(temp);

		/* clear network, unmanage open windows, free memory of graph windows */

		clear_network();

		/* generate the new header for the main menu box */

		strcpy(header, "Biosim ");
		strcat(header, versionString);
		strcat(header, " : ");

		n = 0;
		XtSetArg(args[n], XmNtitle, header); n++;
		XtSetValues(applShell, args, n);

		editNetwork = FALSE;
		fileSaved = TRUE;
		variables_initialized = FALSE;
		break;

	case DIALOG_CLEAR_NETWORK:	/* clear the current network */

		clear_network();

		/*	generate default synaptic types		*/

		synapseTypes = (Synapse *) malloc((size_t) (3 * sizeof(Synapse)));
		if (! synapseTypes)
		{
			error(126);
			break;
		}
		synapseTypes[numTypes].ionType = 2;
		synapseTypes[numTypes++].equilPotential = (double) KEQ_SWIM_DEF / E_SCALE;
		synapseTypes[numTypes].ionType = 1;
		synapseTypes[numTypes++].equilPotential = (double) NAEQ_SWIM_DEF / E_SCALE;
		synapseTypes[numTypes].ionType = 4;
		synapseTypes[numTypes++].equilPotential = (double) 0.0;

		editNetwork = TRUE;
		fileSaved = TRUE;
		variables_initialized = FALSE;
		break;

	case DIALOG_WITHOUT_LEARNPARAMETER_BUTTON:	/* reset network without learning state */

		reset_network(FALSE);
		break;

	case DIALOG_INCL_LEARNPARAMETER_BUTTON:	/* reset network with learning state */

		reset_network(TRUE);
		break;

	case DIALOG_EXIT:	/* exit from program */

		exit(GOOD);
		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in ACCEPT_DIALOG occured !\n");
		break;
	}
} /* end of dialog_accept_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_text_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for text widgets

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_text_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	char            *value = NULL;          /* used for detection of valid number string */
	double          number;                 /* number read from text widget */

	/* check if number beginns with a valid sign */

	value = XmTextGetString(w);

	if ((strlen(value) == 0) ||
	    (strcmp(value,"+") == 0) ||
	    (strcmp(value,"-") == 0) ||
	    (strcmp(value,".") == 0) ||
	    (strcmp(value," +") == 0) ||
	    (strcmp(value," -") == 0) ||
	    (strcmp(value," .") == 0) ||
	    (strcmp(value,"+ ") == 0) ||
	    (strcmp(value,"- ") == 0) ||
	    (strcmp(value," + ") == 0) ||
	    (strcmp(value," - ") == 0))
	{
		free(value);
		return;
	}
	number = strtod(value, (char **) NULL);

	/* check number dependend on the widget from which it comes */

	switch((int) client_data)
	{
	case DIALOG_TEXT_NOISE_INTENSITY:	/* noise intensity was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	noiseIntensity = number;
		free(value);

		break;

	case DIALOG_TEXT_NOISE_STRENGTH:	/* noise current strength was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	noiseStrength = number / I_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_NOISE_DECAY:		/* noise decay rate was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	noiseDecay = number / T_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_IPOT:			/* initial potential for standard deviation was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdInitialPotential = number / E_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_IPOT_STDDEV:		/* standard deviation of initial potential was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdInitialPotentialStd = number / E_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_SYNG:			/* synaptic strength for standard deviation was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdSynapticStrength = number / S_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_SYNG_STDDEV:		/* standard deviation of synaptic strength was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdSynapticStrengthStd = number / S_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_AXOS:			/* axo-axonic strength for standard deviation was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdAxoAxonicStrength = number;
		free(value);

		break;

	case DIALOG_TEXT_AXOS_STDDEV:		/* standard deviation of axo-axonic strength was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdAxoAxonicStrengthStd = number;
		free(value);

		break;

	case DIALOG_TEXT_SYNT:			/* synaptic transmission time for standard deviation was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdSynapticTT = number / T_SCALE;
		free(value);

		break;

	case DIALOG_TEXT_SYNT_STDDEV:		/* standard deviation of synaptic transmission time was changed */

		/* write out an error message if its not a number */

		if (! is_it_a_number(value))
			error(45);
		else	stdSynapticTTStd = number / T_SCALE;
		free(value);

		break;

	default:        /* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in TEXT_DIALOG occured !\n");
		break;
	}
} /* end of dialog_accept_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_cancel_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for cancel button

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_cancel_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	/* evaluate client data */

	switch((int) client_data)
	{
	case DIALOG_SIMINF:

		showSimulationInfo = FALSE;
		break;

	case DIALOG_PROCESS_INF:

		showProcessingInfo = FALSE;
		break;

	case DIALOG_EXIT:
	case DIALOG_SAVE_SIMULATION:
	case DIALOG_SAVE_CONFIGURATION:
	case DIALOG_SAVE_PARAMETER:
	case DIALOG_SAVE_AS_SIMULATION:
	case DIALOG_SAVE_AS_CONFIGURATION:
	case DIALOG_SAVE_AS_PARAMETER:
	case DIALOG_SAVE_AS_FILE_SIM:
	case DIALOG_SAVE_AS_FILE_CON:
	case DIALOG_SAVE_AS_FILE_PAR:
	case DIALOG_CLOSE:
	case DIALOG_NEW:
	case DIALOG_PRINT_SIMULATION:
	case DIALOG_PRINT_CONFIGURATION:
	case DIALOG_PRINT_PARAMETER:
	case DIALOG_PRINT_NEURON:
	case DIALOG_PRINT_GRAFIK:
	case DIALOG_PRINT_OTHER:
	case DIALOG_FPSELECT:

		/* there is nothing to do */
		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in CANCEL_DIALOG occured !\n");
		break;
	}

} /* end of dialog_cancel_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_help_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for help button

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_help_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	/* evaluate client data */

	switch((int) client_data)	/* no help available for now */
	{
	case DIALOG_FSIMSELECT:
	case DIALOG_FCONSELECT:
	case DIALOG_FPARSELECT:

		XtManageChild(openFileDialog);
		sendhelp(w, "helpserver", "biosim", HELP_OPENFILE);
		break;

	case DIALOG_FPSELECT:

		XtManageChild(printFileDialog);
		sendhelp(w, "helpserver", "biosim", HELP_PRINTFILE);
		break;

	case DIALOG_FDSELECT:

		XtManageChild(deleteFileDialog);
		sendhelp(w, "helpserver", "biosim", HELP_DELETEFILE);
		break;

	case DIALOG_SAVE_AS_FILE_SIM:
	case DIALOG_SAVE_AS_FILE_CON:
	case DIALOG_SAVE_AS_FILE_PAR:

		XtManageChild(saveAsFileDialog);
		sendhelp(w, "helpserver", "biosim", HELP_SAVEAS);
		break;

	case DIALOG_PRINT_SIMULATION:
	case DIALOG_PRINT_CONFIGURATION:
	case DIALOG_PRINT_PARAMETER:
	case DIALOG_PRINT_NEURON:
	case DIALOG_PRINT_GRAFIK:
	case DIALOG_PRINT_OTHER:

		sendhelp(w, "helpserver", "biosim", HELP_PRINTFILE);
		break;

	case DIALOG_NOISE:

		sendhelp(w, "helpserver", "biosim", HELP_NOISE);
		break;

	case DIALOG_STDDEV:

		sendhelp(w, "helpserver", "biosim", HELP_STDDEV);
		break;

	case DIALOG_HELP_INDEX:

		sendhelp(w, "helpserver", "biosim", HELP_INDEX);
		break;

	case DIALOG_HELP_FEATURES:

		sendhelp(w, "helpserver", "biosim", HELP_FEATURES);
		break;

	case DIALOG_HELP_INTRO:

		sendhelp(w, "helpserver", "biosim", HELP_INTRO);
		break;

	case DIALOG_HELP_TARGET:

		sendhelp(w, "helpserver", "biosim", HELP_TARGET);
		break;

	case DIALOG_HELP_CELLMODEL:

		sendhelp(w, "helpserver", "biosim", HELP_CELLMODEL);
		break;

	case DIALOG_HELP_CONTEXT:

		sendhelp(w, "helpserver", "biosim", HELP_CONTEXT);
		break;

	case DIALOG_HELP_MOUSE:

		sendhelp(w, "helpserver", "biosim", HELP_MOUSE);
		break;

	case DIALOG_HELP_KEYBOARD:

		sendhelp(w, "helpserver", "biosim", HELP_KEYBOARD);
		break;

	case DIALOG_HELP_GLOSSAR:

		sendhelp(w, "helpserver", "biosim", HELP_GLOSSAR);
		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in HELP_DIALOG occured !\n");
		break;
	}

} /* end of dialog_help_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_option_button_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for buttons in option menu

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_option_button_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XmToggleButtonCallbackStruct *button = (XmToggleButtonCallbackStruct *) call_data;
	Boolean set = button->set;	/* status of toggle button */

	/* evaluate client data */

	switch((int) client_data)
	{
	case DIALOG_SIMINF:

		showSimulationInfo = set;
		break;

	case DIALOG_PROCESS_INF:

		showProcessingInfo = set;
		break;

	case DIALOG_SHOW_OUTPUT:

		showOutputImmediately = set;
		break;

	case DIALOG_AUTO_RESET:

		autoReset = set;
		break;

	case DIALOG_NEURON_COLORS:

		showActivityColors = set;
		break;

	case DIALOG_BOUND_CHECKING:

		boundCheckingOn = set;
		break;

	case DIALOG_MANUAL_UPDATE:

		manualWindowUpdate = set;
		break;

	case DIALOG_NOISE_ACTIVATE:

		noiseDefined = set;
		break;

	case DIALOG_WINDOW_UPDATE:

		/* loop through all neurons and look for graph windows */
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->graph) /* graph window exist ? */
				draw_graph(currentNeuron);
			
			/* loop through all synapses out of this neuron */

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
				if (currentSynapse->graph) /* graph window exist ? */
					draw_graphS(currentSynapse);
		}

		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in OPTION_BUTTON occured !\n");
		break;
	}

} /* end of dialog_option_button_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_stochastic_button_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for buttons in stochastic menu

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_stochastic_button_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XmToggleButtonCallbackStruct *button = (XmToggleButtonCallbackStruct *) call_data;
	Boolean set = button->set;

	/* evaluate client data */

	switch((int) client_data)
	{
	case INITIAL_POTENTIALS_BUTTON:

		stochasticInitialPotentials = set;
		break;

	case SYNAPTIC_STRENGTH_BUTTON:

		stochasticSynapticStrengths = set;
		break;

	case SYNAPTIC_TRANSMISSION_TIME_BUTTON:

		stochasticTransmissionTimes = set;
		break;
	}

} /* end of dialog_stochastic_button_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_model_button_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for buttons in learn menu

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_model_button_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XmToggleButtonCallbackStruct *button = (XmToggleButtonCallbackStruct *) call_data;
	Boolean set = button->set;
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	int		i;			/* loop variable */
	unsigned int	len;			/* length of an reallocation entry */

	/* evaluate client data */

	switch((int) client_data)
	{
	case ONOFF_BUTTON:

		if (set)
		{
			networkmodel = 1;

			n = 0;
			XtSetArg(args[n], XmNsensitive, FALSE); n++;
			XtSetValues(addSynapseLearnButton, args, n);
			XtSetValues(axoSynapseLearnButton, args, n);

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
				read_on_off_channels(currentNeuron);
		}
		break;

	case SWIM_BUTTON:

		if (set)
		{
			networkmodel = 2;

			n = 0;
			XtSetArg(args[n], XmNsensitive, FALSE); n++;
			XtSetValues(addSynapseLearnButton, args, n);
			XtSetValues(axoSynapseLearnButton, args, n);

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
				read_swim_channels(currentNeuron);
		}
		break;

	case HH_BUTTON:

		if (set)
		{
			networkmodel = 3;

			n = 0;
			XtSetArg(args[n], XmNsensitive, FALSE); n++;
			XtSetValues(addSynapseLearnButton, args, n);
			XtSetValues(axoSynapseLearnButton, args, n);

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
				read_HH_channels(currentNeuron);
		}
		break;

	case GB_BUTTON:

		if (set)
		{
			networkmodel = 4;

			n = 0;
			XtSetArg(args[n], XmNsensitive, FALSE); n++;
			XtSetValues(addSynapseLearnButton, args, n);
			XtSetValues(axoSynapseLearnButton, args, n);

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
				read_golowasch_channels(currentNeuron);
		}
		break;

	case COMPLEX_BUTTON:

		complexModel = set;
		break;

	case POINT1_BUTTON:

		if (set)
		{
			networktype = 1;
			curNumOfDendSegs = 0;	/* zero dendritic segments */

			XmTextSetString(addSynapseTargetText, "0");
			XmTextSetString(editSynapseLocationText, "0");
			XmTextSetString(editCurrentLocationText, "0");
			XmScaleSetValue(addSynapseTargetScale, 0);
			XmScaleSetValue(editSynapseLocationScale, 0);
			XmScaleSetValue(editCurrentLocationScale, 0);

			XtSetSensitive(addSynapseTargetScale, FALSE);
			XtSetSensitive(editSynapseLocationScale, FALSE);
			XtSetSensitive(editCurrentLocationScale, FALSE);

			show_normal_state();
			show_all_objects_in_drawing_area(FALSE);
		}
		break;

	case POINT2_BUTTON:

		if (set)
		{
			networktype = 2;

			/* initialize pointers in InputCurrent structure */

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;

				/* reallocate memory for input currents, potentials, ... */

				len = (unsigned int) (MaxNumberOfDendritStems * 2 * sizeof(double));

				currentNeuron->neuronPotentials = (double *) realloc(currentNeuron->neuronPotentials, (size_t) len);

				if (! currentNeuron->neuronPotentials)
				{
					error(114);
					return;
				}

				currentNeuron->currentChemical = (double *) realloc(currentNeuron->currentChemical, (size_t) len);

				if (! currentNeuron->currentChemical)
				{
					error(114);
					return;
				}

				currentNeuron->currentElectrical = (double *) realloc(currentNeuron->currentElectrical,
					len);

				if (! currentNeuron->currentElectrical)
				{
					error(114);
					return;
				}

				len = (unsigned int) (MaxNumberOfDendritStems * 2 * sizeof(InputCurrent));

				currentNeuron->inputCurrent = (InputCurrent *) realloc(currentNeuron->inputCurrent, (size_t) len);

				if (! currentNeuron->inputCurrent)
				{
					error(114);
					return;
				}

				for (i = curNumOfDendSegs; i <= 1; i++)
				{
					currentNeuron->inputCurrent[i].starttime = (double *) NULL;
					currentNeuron->inputCurrent[i].current = (double *) NULL;
					currentNeuron->inputCurrent[i].numEntries = 0;
				}
			}
			curNumOfDendSegs = 1;	/* one dendritic segment */

			/* set scale bar of add synapse mask, edit synapse mask and edit current mask to sensitive */

			set_max_scale();
		}
		break;

	case POINT4_BUTTON:

		if (set)
		{
			networktype = 3;
			DendriteLength = defaultDendriteLength;	/* set original dendrit length */

			/* initialize pointers in InputCurrent structure */

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;

				/* reallocate memory for input currents, potentials, ... */

				len = (unsigned int) (MaxNumberOfDendritStems * 4 * sizeof(double));

				currentNeuron->neuronPotentials = (double *) realloc(currentNeuron->neuronPotentials, (size_t) len);

				if (! currentNeuron->neuronPotentials)
				{
					error(114);
					return;
				}

				currentNeuron->currentChemical = (double *) realloc(currentNeuron->currentChemical, (size_t) len);

				if (! currentNeuron->currentChemical)
				{
					error(114);
					return;
				}

				currentNeuron->currentElectrical = (double *) realloc(currentNeuron->currentElectrical, (size_t) len);

				if (! currentNeuron->currentElectrical)
				{
					error(114);
					return;
				}

				len = (unsigned int) (MaxNumberOfDendritStems * 4 * sizeof(InputCurrent));

				currentNeuron->inputCurrent = (InputCurrent *) realloc(currentNeuron->inputCurrent, (size_t) len);

				if (! currentNeuron->inputCurrent)
				{
					error(114);
					return;
				}

				for (i = curNumOfDendSegs; i <= 3; i++)
				{
					currentNeuron->inputCurrent[i].starttime = (double *) NULL;
					currentNeuron->inputCurrent[i].current = (double *) NULL;
					currentNeuron->inputCurrent[i].numEntries = 0;
				}
			}
			curNumOfDendSegs = 3;	/* three dendritic segments */

			/* set maximum scale widgets of add synapse, edit synapse, edit current mask to curNumOfDendSegs */

			set_max_scale();
		}
		break;

	case POINTN_BUTTON:

		if (set)
		{
			networktype = 4;
			XtManageChild(askModelPointBB);	/* ask for number of dendritic compartments */
		}
		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in MODEL_BUTTON occured !\n");
		break;
	}

} /* end of dialog_model_button_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_learn_button_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for buttons in learn menu

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_learn_button_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XmToggleButtonCallbackStruct *button = (XmToggleButtonCallbackStruct *) call_data;
	Boolean set = button->set;

	/* evaluate client data */

	switch((int) client_data)
	{
	case COMPETITIVE_BUTTON:

		competitiveLearning = set;
		break;

	case HEBB_BUTTON:

		hebbLearning = set;
		break;

	case CONDITIONED_BUTTON:

		conditionedLearning = set;
		break;

	case PLASTICITY_BUTTON:

		plasticityLearning = set;
		break;

	default:	/* unknown callback type */
		fprintf(stderr, "Warning: unknown callback in LEARN_BUTTON occured !\n");
		break;
	}

} /* end of dialog_learn_button_CB */

/************************************************************************************************************************

 	FUNCTION	: dialog_integrationMethods_button_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for buttons in integration methods menu

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_integrationMethods_button_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	XmToggleButtonCallbackStruct *button = (XmToggleButtonCallbackStruct *) call_data;
	Boolean set = button->set;

	/* evaluate client data */

	switch((int) client_data)
	{
	case EULER_BUTTON:

		if (set)
			integrationMethod = EULER;
		break;

	case EXPEULER_BUTTON:

		if (set)
			integrationMethod = EXPEULER;
		break;

	case TRAPEZOID_BUTTON:

		if (set)
			integrationMethod = TRAPEZOID;
		break;

	case RUNGEKUTTA_BUTTON:

		if (set)
			integrationMethod = RUNGEKUTTA;
		break;

	}

} /* end of dialog_integrationMethods_button_CB */

/************************************************************************************************************************

 	FUNCTION	: edit_sim_pars_CB(w, client_data, call_data)
 
	PURPOSE		: callback routine for edit simulation parameter mask

	RETURNS		: nothing

************************************************************************************************************************/

void edit_sim_pars_CB(w, client_data, call_data)
Widget          w;              /*  widget id           */
caddr_t         client_data;    /*  data from application   */
caddr_t         call_data;      /*  data from widget class  */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	char 		textnumber[20];		/* number as string */
	int 		scaleValue;		/* value of scale widget */
	int		sF;			/* scaling factor */
	short int 	decimals;		/* number of decimals for scale widget */

	sprintf(textnumber,"%f", simulation.length * T_SCALE);
	XmTextSetString(editSimParsSimLenText, textnumber);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(editSimParsSimLenScale, args, n);
	sF = (int) pow((double) 10, (double) decimals);
	scaleValue = (int) (simulation.length * T_SCALE * sF);
	XmScaleSetValue(editSimParsSimLenScale, scaleValue);

	sprintf(textnumber,"%f", simulation.step * T_SCALE);
	XmTextSetString(editSimParsStepSizeText, textnumber);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(editSimParsStepSizeScale, args, n);
	sF = (int) pow((double) 10, (double) decimals);
	scaleValue = (int) (simulation.step * T_SCALE * sF);
	XmScaleSetValue(editSimParsStepSizeScale, scaleValue);

	sprintf(textnumber,"%d", neuronActivityInterval);
	XmTextSetString(neuronActivityIntervalText, textnumber);

	n = 0;
	XtSetArg(args[n], XmNdecimalPoints, &decimals); n++;
	XtGetValues(neuronActivityIntervalScale, args, n);
	sF = (int) pow((double) 10, (double) decimals);
	scaleValue = (int) (neuronActivityInterval * sF);
	XmScaleSetValue(neuronActivityIntervalScale, scaleValue);

	XtManageChild(editSimParsBB);

} /* end of edit_sim_pars_CB */
#endif
