/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	graphic.c

	PURPOSE:	the file contains the stuff to handle the topological window for input the neural net

	FUNCTIONS:	in order of appearance

	add_neuron_to_neuron_list			- add a neuron to the neuron list
	remove_neuron_from_neuron_list			- remove a neuron from the neuron list
	add_neuron_to_selection_list			- add a neuron to the neuron selection list
	remove_neuron_from_selection_list		- remove a neuron from the neuron selection list
	add_synapse_to_selection_list			- add a synapse to the synapse selection list
	remove_synapse_from_selection_list		- remove a synapse from the synapse selection list
	remove_selected_objects_from_selectionList	- remove all selected objects from the selection lists
	clear_neuron					- resets a neuron to its standard values
	clear_neuron_no_channels			- resets a neuron to its standard values without reading in
							  the neuron channels
	clear_neurons					- resets all neurons in the net to the standard values
	clear_network					- delete current network
	reset_network					- reset the network to its initial state
	change_neuron_color				- change the color of a neuron
	set_neuron_color				- set the color of a neuron voltage dependend
	change_synapse_color				- change the color of a synapse
	show_all_objects_in_drawing_area		- redraw all objects in the net
	show_input_neurons				- highlight input neurons to the selected neuron
	show_output_neurons				- highlight output neurons to the selected neuron
	show_input_output_neurons			- highlight input and output neurons to the selected neuron
	show_topology					- refits the sizes of the network to fit into the window
	show_normal_state				- reuse the initial sizes and colors to display the network
	process_buttonevent				- processes button events in the drawing area for the network
	is_connected					- looks if neuron 1 is connected with neuron 2
	connection_exists				- looks if given connection already exists
	delete_graphical_connection			- deletes the graphical representation from the connection
	delete_connection				- deletes the entire connection
	change_synaptic_values				- changes the synaptic values
	connect_neurons_graphical			- draws a graphical representation for the connection
	connect_neurons					- connects two neurons
	draw_synapse					- draws the synapse
	calculate_graphical_synapse			- calculates the graphical position for the synapse
	select_all_neurons_CB				- callback function that selects all neurons in the net
	select_all_synapses_CB				- callback function that selects all synapses in the net
	select_all_axo_synapses_CB			- callback function that selects all axo-synapses in the net
	drawing_input_CB				- callback function for inputs into the drawing area
	drawing_expose_CB				- callback function for expose events into the drawing area
	drawing_resize_CB				- callback function for resize events into the drawing area
	remove_selected_objects_CB			- callback function to remove all selected objects
	remove_selected_objects_from_selectionList_CB	- callback function to remove all selected objects from the selection list 

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/****** global variables ******/

Neuron *neuronListHead = NULL;				/* start pointer of double chained neuron list */
Neuron *neuronListTail = NULL;				/* end pointer of double chained neuron list */
SelectionListNeurons *selectionListNeuronsHead = NULL;	/* start pointer of double chained neuron selection list */
SelectionListNeurons *selectionListNeuronsTail = NULL;	/* end pointer of double chained neuron selection list */
SelectionListSynapses *selectionListSynapsesHead = NULL;/* start pointer of double chained synapse selection list */
SelectionListSynapses *selectionListSynapsesTail = NULL;/* end pointer of double chained synapse selection list */
Neuron *currentNeuron = NULL;				/* current neuron */
Connection *currentSynapse = NULL;			/* current synapse */
int selectionListNeuronsCounter = 0;			/* number of selected neurons in drawing area */
int selectionListSynapsesCounter = 0;			/* number of selected synapses in drawing area */
int neuronNumber = 0;					/* number of neurons in net */
int synapseNumber = 0;					/* number of synapses in net */
int axoSynapsesNumber = 0;				/* number of axo-axonic synapses in the net */
int numTypes = 0;					/* number of synaptic types in net */
Synapse *synapseTypes = NULL;				/* array of synatic types */
#ifndef BATCH
unsigned int neuronDiameter = defaultNeuronDiameter;	/* diameter of the graphic circle for a neuron */
unsigned int synapseDiameter = defaultSynapseDiameter;	/* diameter of the graphic circle for a synapse */
int DendriteWidth = defaultDendriteWidth;		/* dendrite width */
int DendriteLength = defaultDendriteLength;		/* dendrite length */
int StemLength = defaultStemLength;			/* length of dendritic stem for 1-Point model */
int StemWidth = defaultStemWidth;			/* width of dendrite for 1-Point model */
int OwnConnectionRadius = defaultOwnConnectionRadius;	/* radius for connection with itself */
double scalingFactor = 1.0;				/* scaling factor for graphic objects */
#endif

/****** functions ******/

/************************************************************************************************************************

 	FUNCTION	: add_neuron_to_neuron_list(neuronId)
 
	PURPOSE		: add a neuron to the neuron list

	RETURNS		: nothing

************************************************************************************************************************/

void add_neuron_to_neuron_list(neuronId, channelsRead)
int neuronId;		/* neuron id the new neuron should have */
Boolean channelsRead;	/* flag indicating if the neuron channels should be read in */
{
	Neuron	*new;		/* pointer to new neuron instance */
	int	i;		/* loop variable */
	unsigned int length;	/* length of area to be allocated */

	/* allocate memory for a new neuron */

	new = (Neuron *) malloc((size_t) sizeof(Neuron));
	if (! new)
	{
		error(105);
		return;
	}
#ifndef BATCH
	new->neuronSize = (unsigned int) sizeof(Neuron);
#endif

	/* allocate memory for input currents, potentials, ... */

	length = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

	new->neuronPotentials = (double *) malloc((size_t) length);

	if (! new->neuronPotentials)
	{
		error(114);
		return;
	}

	new->currentChemical = (double *) malloc((size_t) length);

	if (! new->currentChemical)
	{
		error(114);
		return;
	}

	new->currentElectrical = (double *) malloc((size_t) length);

	if (! new->currentElectrical)
	{
		error(114);
		return;
	}
#ifndef BATCH
	new->neuronSize += (unsigned int) (3 * length);
#endif


	length = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(InputCurrent));

	new->inputCurrent = (InputCurrent *) malloc((size_t) length);

	if (! new->inputCurrent)
	{
		error(114);
		return;
	}
#ifndef BATCH
	new->neuronSize += (unsigned int) length;
	new->inpCurSize = 0;
#endif

	/* initialize pointers in InputCurrent structure */

	for (i = 0; i <= curNumOfDendSegs; i++)
	{
		new->inputCurrent[i].starttime = (double *) NULL;
		new->inputCurrent[i].current = (double *) NULL;
		new->inputCurrent[i].index = 0;
		new->inputCurrent[i].numEntries = 0;
	}

	/* initialize neuron */

	new->neuronId = neuronId;
	new->next = NULL;
	new->previous = neuronListTail;
	new->selected = FALSE;
#ifndef BATCH
	new->isInputNeuron = FALSE;
	new->isOutputNeuron = FALSE;
#endif
	new->connectionList = NULL;
	new->poolList = NULL;
	new->graph = NULL;
	new->onoffSpikeTimer = 0;
	new->onoffSpikeOccuring = FALSE;
	new->neuronStems = 1;
	new->neuronRegions = curNumOfDendSegs;
	new->usedDendSCLocation = 0;
	new->usedDendSCStem = 1;
	new->pseudoSomaCurrent = new->pseudoSCStart = new->pseudoSCEnd = 0.0;
	new->synapticCurrent = new->totalCurrent = 0.0;
	new->M = new->H = new->N = new->A = new->B1 = new->B2 = new->gamma = new->C = new->D = new->E = new->P = new->Q =
	new->R = new->S = 0.0;
	new->iNoise = new->iNA = new->iK = new->iCA = new->iA = new->iCA1 = new->iCA2 = new->iKCA = new->iIR = new->iP = 0.0;
	new->conc_ca = new->ca_e = 0.0;
	new->somaCurrent = new->somaSCStart = new->somaSCEnd = 0.0;
	new->dendCurrent = new->dendSCStart = new->dendSCEnd = 0.0;
	new->SLmSG_const = new->SGpDSmDG_const = new->DLmDG_const = new->SGdSC_const = new->SGpDSmDGdSC_const =
	new->SWIM_NAEmNAG_const = new->SWIM_KEmKG_const = new->SWIM_CAEmCAG_const = new->HH_NAEmNAG_const =
	new->HH_KEmKG_const = new->GB_NAEmNAG_const = new->GB_KEmKG_const = new->GB_AEmAG_const = new->GB_KCAEmKCAG_const =
	new->GB_IREmIRG_const = new->GB_PEmPG_const = 0.0;

	for (i = 0; i <= curNumOfDendSegs; i++)
	{
		new->currentChemical[i] = new->currentElectrical[i] = new->neuronPotentials[i] = 0.0;
	}

	for (i = 0; i < NEURON_LENGTH; i++)
	{
		new->neuronParameters[i] = 0.0;
	}

	/* insert neuron into the neuron list */

	if (neuronListHead == NULL)
		neuronListHead = neuronListTail = new;
	else
	{
		neuronListTail->next = new;
		neuronListTail = new;
	}

#ifndef BATCH
	fileSaved = FALSE;	/* set file saved status to false */
#endif

	/* set default values for standard neuron */

	if (channelsRead)
		clear_neuron(new);
	else	clear_neuron_no_channels(new);

	neuronNumber++;		/* update neuron number */

} /* end of add_neuron_to_neuron_list */

/************************************************************************************************************************

 	FUNCTION	: remove_neuron_from_neuron_list(nptr)
 
	PURPOSE		: remove a neuron from the neuron list

	RETURNS		: nothing

************************************************************************************************************************/

void remove_neuron_from_neuron_list(nptr)
Neuron *nptr;	/* neuron pointer to the removable neuron */
{
	Neuron			*actptr;	/* current neuron pointer */
	Connection		*connection;	/* current connection pointer */
	Connection		*nextConnection;/* next connection */
	ChemicalPool		*pool;		/* current pool poiner */
	ChemicalSenderList	*sender;	/* current sender pointer */

	/* if this neuron has a graph window destroy it */

	if (nptr->graph)
	{
#ifndef BATCH
		XtUnrealizeWidget(XtParent(nptr->graph->formWidget));	/* unrealize graph shell widget */
		XmUpdateDisplay(nptr->graph->formWidget);		/* process all pending exposure events */
		XtDestroyWidget(XtParent(nptr->graph->formWidget));	/* free graph shell widget */
#endif
		free(nptr->graph->soma);
		free(nptr->graph->dend);
		free(nptr->graph->synC);
		free(nptr->graph->totC);
		free(nptr->graph->iN);
		free(nptr->graph->iNA);
		free(nptr->graph->iK);
		free(nptr->graph->iCA);
		free(nptr->graph->iCA1);
		free(nptr->graph->iCA2);
		free(nptr->graph->iKCA);
		free(nptr->graph->iA);
		free(nptr->graph->iIR);
		free(nptr->graph->iP);
		free(nptr->graph);					/* free graph */

		graphNumber--;						/* update number of graphs in the net */
	}

	/* free dynamic allocated memory of neuron */

	free(nptr->neuronPotentials);
	free(nptr->currentChemical);
	free(nptr->currentElectrical);
	free(nptr->inputCurrent);

	/* clear all connections to and from this neuron */

	connection = nptr->connectionList;
	while (connection != NULL)
	{
		nextConnection = connection->nextConnection;
		delete_connection(nptr, connection);
		connection = nextConnection;
	}

	for (pool = nptr->poolList; pool != NULL; pool = pool->nextPool)
	{
		for (sender = pool->senderList; sender != NULL; sender = sender->nextSender)
		{
			delete_connection(sender->connection->parentNeuron, sender->connection);
		}
	}

	/* remove neuron from neuron list */

	actptr = neuronListHead;
	if (neuronListHead == neuronListTail) /* only one neuron exists */
	{
		neuronListHead = neuronListTail = NULL;
		free(actptr);
	}
	else if (nptr == neuronListHead) /* remove first entry in neuron list */
	{
		neuronListHead = neuronListHead->next;
		neuronListHead->previous = NULL;
		free(actptr);
	}
	else if (nptr == neuronListTail) /* remove last entry in neuron list */
	{
		neuronListTail = neuronListTail->previous;
		neuronListTail->next = NULL;
		free(nptr);
	}
	else	/* delete neuron in the middle of the list */
	{
		nptr->previous->next = nptr->next;
		nptr->next->previous = nptr->previous;
		free(nptr);
	}

	neuronNumber--;		/* correct the number of neurons in the net */
#ifndef BATCH
	fileSaved = FALSE;	/* set file saved status to false */
#endif

} /* end of remove_neuron_from_neuron_list */

/************************************************************************************************************************

 	FUNCTION	: add_neuron_to_selection_list(nptr)
 
	PURPOSE		: add a neuron to the neuron selection list

	RETURNS		: nothing

************************************************************************************************************************/

void add_neuron_to_selection_list(nptr)
Neuron *nptr;	/* pointer to neuron which should be inserted into the selecton list */
{
	SelectionListNeurons	*new;		/* pointer to new object */

	/* allocate memory for a new pointer object to a neuron */

	new = (SelectionListNeurons *) malloc((size_t) sizeof(SelectionListNeurons));
	if (! new)
	{
		error(106);
		return;
	}

	/* initialize new object */

	new->neuronPtr = nptr;
	new->next = NULL;

	/* insert object into the neuron selection list */

	if (selectionListNeuronsHead == NULL)
		selectionListNeuronsHead = selectionListNeuronsTail = new;
	else selectionListNeuronsTail = selectionListNeuronsTail->next = new;

	selectionListNeuronsCounter++;	/* update counter for selected neurons */

} /* end of add_neuron_to_selection_list */

/************************************************************************************************************************

 	FUNCTION	: remove_neuron_from_selection_list(nptr)
 
	PURPOSE		: remove a neuron from the neuron selection list

	RETURNS		: nothing

************************************************************************************************************************/

void remove_neuron_from_selection_list(nptr)
Neuron *nptr;	/* pointer to neuron which should be deleted from the selecton list */
{
	SelectionListNeurons	*actptr, *copy;	/* help variables */

	/* remove neuron from selection list */

	actptr = selectionListNeuronsHead;
	if (selectionListNeuronsHead == selectionListNeuronsTail) /* only one neuron in the selection list exists */
	{
		selectionListNeuronsHead = selectionListNeuronsTail = NULL;
		free(actptr);
	}
	else if (selectionListNeuronsHead->neuronPtr == nptr) /* remove first entry in selection list */
	{
		selectionListNeuronsHead = selectionListNeuronsHead->next;
		free(actptr);
	}
	else /* search for nptr in selectionListNeurons */
	{
		for (actptr = selectionListNeuronsHead; actptr->next != NULL; actptr = actptr->next)
		{
			if (actptr->next->neuronPtr == nptr)
			{
				if (actptr->next == selectionListNeuronsTail) selectionListNeuronsTail = actptr;
				copy = actptr->next;
				actptr->next = actptr->next->next;
				free(copy);
				break; /* exit for loop */
			}
		}
	}

	selectionListNeuronsCounter--;	/* update counter for selected neurons */

} /* end of remove_neuron_from_selection_list */

/************************************************************************************************************************

 	FUNCTION	: add_synapse_to_selection_list(synapse)
 
	PURPOSE		: add a synapse to the synapse selection list

	RETURNS		: nothing

************************************************************************************************************************/

void add_synapse_to_selection_list(synapse)
Connection *synapse;	/* pointer to synapse which should be added to the selection list */
{
	SelectionListSynapses	*new;		/* pointer to new object */

	/* allocate memory for a new pointer object to a synapse */

	new = (SelectionListSynapses *) malloc((size_t) sizeof(SelectionListSynapses));
	if (! new)
	{
		error(107);
		return;
	}

	/* initialize object */

	new->synapsePtr = synapse;
	new->next = NULL;

	/* insert object into synapses selection list */

	if (selectionListSynapsesHead == NULL)
		selectionListSynapsesHead = selectionListSynapsesTail = new;
	else selectionListSynapsesTail = selectionListSynapsesTail->next = new;

	selectionListSynapsesCounter++;	/* update counter for selected synapses */

} /* end of add_synapse_to_selection_list */

/************************************************************************************************************************

 	FUNCTION	: remove_synapse_from_selection_list(synapse)
 
	PURPOSE		: remove a synapse from the synapse selection list

	RETURNS		: nothing

************************************************************************************************************************/

void remove_synapse_from_selection_list(synapse)
Connection *synapse;	/* synapse to be removed */
{
	SelectionListSynapses	*actptr, *copy;	/* help variables */

	/* remove synapse from selection list */

	actptr = selectionListSynapsesHead;
	if (selectionListSynapsesHead == selectionListSynapsesTail) /* only one synapse in the selection list exists */
	{
		selectionListSynapsesHead = selectionListSynapsesTail = NULL;
		free(actptr);
	}
	else if (selectionListSynapsesHead->synapsePtr == synapse) /* remove first entry in selection list */
	{
		selectionListSynapsesHead = selectionListSynapsesHead->next;
		free(actptr);
	}
	else /* search for synapse in selectionListSynapses */
	{
		for (actptr = selectionListSynapsesHead; actptr->next != NULL; actptr = actptr->next)
		{
			if (actptr->next->synapsePtr == synapse)
			{
				if (actptr->next == selectionListSynapsesTail)
					selectionListSynapsesTail = actptr;
				copy = actptr->next;
				actptr->next = actptr->next->next;
				free(copy);
				break; /* exit for loop */
			}
		}
	}

	selectionListSynapsesCounter--;	/* update counter for selected synapses */

} /* end of remove_synapse_from_selection_list */

/************************************************************************************************************************

 	FUNCTION	: remove_selected_objects_from_selectionList(void)
 
	PURPOSE		: remove all selected objects from the selection lists

	RETURNS		: nothing

************************************************************************************************************************/

void remove_selected_objects_from_selectionList()
{
	SelectionListNeurons	*actptrN, *copyN;	/* help variables */
	SelectionListSynapses	*actptrS, *copyS;	/* help variables */

	/* check if an error has been occured, if yes don't remove objects from selection list */

	if (errorOccured)
	{
		errorOccured = FALSE;
		return;
	}

	actptrS = selectionListSynapsesHead;

	/* remove all synapses from selection list */

	while (actptrS != NULL)
	{
		actptrS->synapsePtr->selected = FALSE;	/* deselect synapse */

		/* restore original synapse color dependend on synaptic type */

#ifndef BATCH
		if (actptrS->synapsePtr->axoSynapseNumber > 0)	/* axo-axonic synapse */
			change_synapse_color(actptrS->synapsePtr, resData.axoSynapseColor);
		else
			switch (synapseTypes[actptrS->synapsePtr->synapticType].ionType)
			{
			case 1:
				change_synapse_color(actptrS->synapsePtr, resData.excitationSynapseColor);
				break;
			case 2:
				change_synapse_color(actptrS->synapsePtr, resData.inhibitionSynapseColor);
				break;
			case 3:
				change_synapse_color(actptrS->synapsePtr, resData.otherSynapseColor);
				break;
			case 4:
				change_synapse_color(actptrS->synapsePtr, resData.electricalSynapseColor);
				break;
			}
#endif

		/* remove synapse */

		copyS = actptrS;
		actptrS = actptrS->next;
		free(copyS);
	}

	/* initialize synapse selection list */

	selectionListSynapsesHead = selectionListSynapsesTail = NULL;
	selectionListSynapsesCounter = 0;

	actptrN = selectionListNeuronsHead;

	/* remove all neurons from selection list */

	while (actptrN != NULL)
	{
#ifndef BATCH
		/* restore original color */

		if (actptrN->neuronPtr->isInputNeuron && actptrN->neuronPtr->isOutputNeuron)
			change_neuron_color(actptrN->neuronPtr, resData.inOutNeuronColor, resData.textColor);
		else if (actptrN->neuronPtr->isInputNeuron)
			change_neuron_color(actptrN->neuronPtr, resData.inputNeuronColor, resData.textColor);
		else if (actptrN->neuronPtr->isOutputNeuron)
			change_neuron_color(actptrN->neuronPtr, resData.outputNeuronColor, resData.textColor);
		else change_neuron_color(actptrN->neuronPtr, resData.neuronColor, resData.textColor);
#endif

		/* reset neurons selected state variable */

		actptrN->neuronPtr->selected = FALSE;

		/* delete neuron from selection list */

		copyN = actptrN;
		actptrN = actptrN->next;
		free(copyN);
	}

	/* initialize neuron selection list */

	selectionListNeuronsHead = selectionListNeuronsTail = NULL;
	selectionListNeuronsCounter = 0;

} /* end of remove_selected_objects_from_selectionList */

/************************************************************************************************************************

 	FUNCTION	: clear_neuron(neuron)
 
	PURPOSE		: resets a neuron to its standard values

	RETURNS		: nothing

************************************************************************************************************************/

void clear_neuron(neuron)
Neuron *neuron;	/* pointer to neuron which should be cleared */
{
	currentNeuron = neuron;	/* set currentNeuron variable to neuron for using macros */

	SpikeCounter		= 0;
	SOMA_Gcore		= TOD_DEF / 1.0E9;
	DENDRITE_REST_Gm	= DCON_DEF / 1.0E9;
	DENDRITE_Cm		= DCAP_DEF / 1.0E12;
	DENDRITE_Gcore		= TOS_DEF / 1.0E9;
	ACTUAL_DENDRITIC_STEMS	= 1;
	ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;
	SomaCur			= SCUR_DEF / 1.0E9;
	SCSTRT			= SSTART_DEF / 1.0E3;
	SCSTP			= SEND_DEF / 1.0E3;
	DendCur			= DCUR_DEF / 1.0E9;
	SDSTRT			= DSTART_DEF / 1.0E3;
	SDSTP			= DEND_DEF / 1.0E3;
	ActualIRegion		= curNumOfDendSegs;
	ActualIStem		= 1;

	/* read in channel parameters depending on the current network model */

	switch(networkmodel)
	{
	case 1:	/* On-Off model */
		
		SOMA_Cm			= CAP_ONOFF_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_ONOFF_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_ONOFF_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_ONOFF_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_ONOFF_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_ONOFF_DEF / 1.0E3;

#ifndef BATCH
		read_on_off_channels(neuron);
#endif
		break;
	case 2: /* SWIM model */

		SOMA_Cm			= CAP_SWIM_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_SWIM_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_SWIM_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_SWIM_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_SWIM_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_SWIM_DEF / 1.0E3;

#ifndef BATCH
		read_swim_channels(neuron);
#endif
		break;
	case 3: /* Hodgkin-Huxley model */

		SOMA_Cm			= CAP_HH_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_HH_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_HH_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_HH_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_HH_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_HH_DEF / 1.0E3;

#ifndef BATCH
		read_HH_channels(neuron);
#endif
		break;
	case 4: /* Golowasch-Buchholz model */

		SOMA_Cm			= CAP_GB_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_GB_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_GB_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_GB_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_GB_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_GB_DEF / 1.0E3;

#ifndef BATCH
		read_golowasch_channels(neuron);
#endif
		break;
	
	default: fprintf(stderr, "Wrong model type during clear neuron\n");
		 break;
	}

	reset_extern_input_currents(currentNeuron);

} /* end of clear_neuron */

/************************************************************************************************************************

 	FUNCTION	: clear_neuron_no_channels(neuron)
 
	PURPOSE		: resets a neuron to its standard values without reading in the neuron channels

	RETURNS		: nothing

************************************************************************************************************************/

void clear_neuron_no_channels(neuron)
Neuron *neuron;	/* pointer to neuron which should be cleared */
{
	currentNeuron = neuron;	/* set currentNeuron variable to neuron for using macros */

	SpikeCounter		= 0;
	SOMA_Gcore		= TOD_DEF / 1.0E9;
	DENDRITE_REST_Gm	= DCON_DEF / 1.0E9;
	DENDRITE_Cm		= DCAP_DEF / 1.0E12;
	DENDRITE_Gcore		= TOS_DEF / 1.0E9;
	ACTUAL_DENDRITIC_STEMS	= 1;
	ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;
	SomaCur			= SCUR_DEF / 1.0E9;
	SCSTRT			= SSTART_DEF / 1.0E3;
	SCSTP			= SEND_DEF / 1.0E3;
	DendCur			= DCUR_DEF / 1.0E9;
	SDSTRT			= DSTART_DEF / 1.0E3;
	SDSTP			= DEND_DEF / 1.0E3;
	ActualIRegion		= curNumOfDendSegs;
	ActualIStem		= 1;

	switch(networkmodel)
	{
	case 1:	/* On-Off model */
		
		SOMA_Cm			= CAP_ONOFF_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_ONOFF_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_ONOFF_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_ONOFF_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_ONOFF_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_ONOFF_DEF / 1.0E3;
		break;

	case 2: /* SWIM model */

		SOMA_Cm			= CAP_SWIM_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_SWIM_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_SWIM_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_SWIM_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_SWIM_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_SWIM_DEF / 1.0E3;
		break;

	case 3: /* Hodgkin-Huxley model */

		SOMA_Cm			= CAP_HH_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_HH_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_HH_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_HH_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_HH_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_HH_DEF / 1.0E3;
		break;

	case 4: /* Golowasch-Buchholz model */

		SOMA_Cm			= CAP_GB_DEF / 1.0E12;
		SOMA_REST_Gm		= CON_GB_DEF / 1.0E9;
		SOMA_Eleak_INITIAL	= LEAK_GB_DEF / 1.0E3;
		SOMA_E_INITIAL		= POT_GB_DEF / 1.0E3;
		DENDRITE_Eleak_INITIAL	= DLEAK_GB_DEF / 1.0E3;
		DENDRITE_E_INITIAL	= DPOT_GB_DEF / 1.0E3;
		break;
	
	default: fprintf(stderr, "Wrong model type during clear neuron\n");
		 break;
	}

	reset_extern_input_currents(currentNeuron);

} /* end of clear_neuron_no_channels */

/************************************************************************************************************************

 	FUNCTION	: clear_neurons(void)
 
	PURPOSE		: resets all neurons in the net to the standard values

	RETURNS		: nothing

************************************************************************************************************************/

void clear_neurons()
{
	/* loop through all neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		currentNeuron->somaCurrent = 0.0;
		currentNeuron->dendCurrent = 0.0;
		clear_neuron(currentNeuron);	/* set standard values for the neuron */
	}
} /* end of clear_neurons */

/************************************************************************************************************************

 	FUNCTION	: clear_network(void)
 
	PURPOSE		: delete current network

	RETURNS		: nothing

************************************************************************************************************************/

void clear_network()
{
	Neuron 			*neuron;		/* current neuron */
	Neuron			*nextNeuron;		/* next neuron */
	SelectionListNeurons	*actptrN, *copyN;	/* help variables */
	SelectionListSynapses	*actptrS, *copyS;	/* help variables */

	/* remove all objects from synapse selection list */

	actptrS = selectionListSynapsesHead;
	while (actptrS != NULL)
	{
		copyS = actptrS;
		actptrS = actptrS->next;
		free(copyS);
	}

	/* remove all objects from neuron selection list */

	actptrN = selectionListNeuronsHead;
	while (actptrN != NULL)
	{
		copyN = actptrN;
		actptrN = actptrN->next;
		free(copyN);
	}

	/* remove all neurons and associated synapses */

	neuron = neuronListHead;
	while (neuron != NULL)
	{
		nextNeuron = neuron->next;
		remove_neuron_from_neuron_list(neuron);
		neuron = nextNeuron;
	}

	/* reinitialization */

	neuronListHead = neuronListTail = NULL;
	selectionListNeuronsHead = selectionListNeuronsTail = NULL;
	selectionListSynapsesHead = selectionListSynapsesTail = NULL;

	neuronNumber = 0;
	graphNumber = 0;
	graphNumberS = 0;
	graphNumberAS = 0;
	selectionListNeuronsCounter = 0;
	selectionListSynapsesCounter = 0;

	synapseNumber = 0;
	axoSynapsesNumber = 0;
	numTypes = 0;
	free(synapseTypes);
	synapseTypes = NULL;

#ifndef BATCH
	XClearWindow(disp, XtWindow(drawingArea));	/* clear drawing area */
#endif

} /* end of clear_network */

/************************************************************************************************************************

 	FUNCTION	: reset_network(inclLearnParameterFlag)
 
	PURPOSE		: reset the network to its initial state

	RETURNS		: nothing

************************************************************************************************************************/

void reset_network(inclLearnParameterFlag)
Boolean inclLearnParameterFlag;	/* if this flag is true reset the learning parameter, too */
{
	variables_initialized = FALSE;

	if (inclLearnParameterFlag)

		resetLearningParameter = TRUE;
	else
		resetLearningParameter = FALSE;

} /* end of reset_network */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: change_neuron_color(nptr, neuronColour, textColour)
 
	PURPOSE		: change the color of a neuron

	RETURNS		: nothing

************************************************************************************************************************/

void change_neuron_color(nptr, neuronColour, textColour)
Neuron *nptr;				/* pointer to neuron which color should be changed */
char *neuronColour, *textColour;	/* neuron and text color */
{
	Window		window = XtWindow(drawingArea);			/* window of drawing area */
	GC		textGC, neuronGC, marginGC;			/* graphical contexts */
	XGCValues	gcv;						/* values used in graphical contexts */
	unsigned long	mask;						/* masks used in graphical contexts */
	int neuronRadius = (int) rint((double) neuronDiameter / 2.0);	/* graphical neuron radius */
	int neuronStartX = (int) nptr->x + neuronRadius;		/* x-position of middle of neuron circle */
	int neuronStartY = (int) nptr->y + neuronRadius;		/* y-position of middle of neuron circle */
	char 		text[20];					/* neuron id for inscribing neuron */
	unsigned int 	textWidth;					/* width in pixel of text */
	int 		textX, textY;					/* position of text */
	int		i;						/* loop variable */
        int     	screenNr = DefaultScreen(disp);			/* screen number of current display */

	/* create GC for a circle */

	if (strcmp(neuronColour, resData.backgroundColor) == 0)
		gcv.foreground = backgroundColor;
	else if (strcmp(neuronColour, resData.neuronColor) == 0)
		gcv.foreground = neuronColor;
	else if (strcmp(neuronColour, resData.neuronSelectionColor) == 0)
		gcv.foreground = neuronSelectionColor;
	else if (strcmp(neuronColour, resData.neuronMovingColor) == 0)
		gcv.foreground = neuronMovingColor;
	else if (strcmp(neuronColour, resData.inputNeuronColor) == 0)
		gcv.foreground = inputNeuronColor;
	else if (strcmp(neuronColour, resData.outputNeuronColor) == 0)
		gcv.foreground = outputNeuronColor;
	else if (strcmp(neuronColour, resData.inOutNeuronColor) == 0)
		gcv.foreground = inOutNeuronColor;

	mask = GCForeground;
	neuronGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = BlackPixel(disp,screenNr);
	gcv.line_width = 3;
	mask = GCForeground | GCLineWidth;
	marginGC = XCreateGC(disp, window, mask, &gcv);

	/* draw neuron as a filled circle in the color determined by neuronColor */

	XFillArc(disp, window, neuronGC, nptr->x, nptr->y, neuronDiameter, neuronDiameter, 0, 360 * 64);
	XDrawArc(disp, window, marginGC, nptr->x, nptr->y, neuronDiameter, neuronDiameter, 0, 360 * 64);

	/* draw dendritic stem */

	switch(networktype)
	{
	case 1: /* 1-point model ... No Dendritic Stem */

		break;

	case 2: /* 2-Point model ... short and fat dendritic stem */

		XFillRectangle(disp, window, neuronGC, neuronStartX - StemWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * StemWidth + 1), (unsigned int) (StemLength + 1));
		XDrawRectangle(disp, window, marginGC, neuronStartX - StemWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * StemWidth + 1), (unsigned int) (StemLength + 1));
		break;

	case 3: /* 4-Point model ... long and skinny dendritic stem */
	default:

		XFillRectangle(disp, window, neuronGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * DendriteWidth + 1), (unsigned int) (DendriteLength + 1));
		XDrawRectangle(disp, window, marginGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * DendriteWidth + 1), (unsigned int) (DendriteLength + 1));
		for (i = 1; i < 3; i++)
			XDrawLine(disp, window, marginGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius +
			    (int) rint(DendriteLength * i / 3.0), neuronStartX + DendriteWidth + 1, neuronStartY +
			    neuronRadius + (int) rint(DendriteLength * i / 3.0));
		break;

	case 4: /* n-Point model ... long and skinny dendritic stem */

		XFillRectangle(disp, window, neuronGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * DendriteWidth + 1), (unsigned int) (DendriteLength + 1));
		XDrawRectangle(disp, window, marginGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius,
		    (unsigned int) (2 * DendriteWidth + 1), (unsigned int) (DendriteLength + 1));
		for (i = 1; i < curNumOfDendSegs; i++)
			XDrawLine(disp, window, marginGC, neuronStartX - DendriteWidth, neuronStartY + neuronRadius +
			    (int) rint((double) (DendriteLength * i / curNumOfDendSegs)), neuronStartX + DendriteWidth +
			    1, neuronStartY + neuronRadius + (int) rint((double) (DendriteLength * i /
			    curNumOfDendSegs)));
		break;
	}

	/* draw the neuron number above the neuron */

	sprintf(text, "%d", nptr->neuronId);
	textWidth = XTextWidth(font, text, strlen(text));
	textX = (int) nptr->x + neuronRadius - (int) (rint((double) textWidth / 2.0));
	textY = (int) nptr->y - (int) font->descent - textOffset;

	if (strcmp(textColour, resData.backgroundColor) == 0)
		gcv.foreground = backgroundColor;
	else if (strcmp(textColour, resData.textColor) == 0)
		gcv.foreground = textColor;

	mask = GCForeground;
	textGC = XCreateGC(disp, window, mask, &gcv);
	XDrawString(disp, window, textGC, textX, textY, text, strlen(text));

	/* free graphical contexts */

	XFreeGC(disp, textGC);
	XFreeGC(disp, neuronGC);
	XFreeGC(disp, marginGC);

} /* end of change_neuron_color */

/************************************************************************************************************************

 	FUNCTION	: set_neuron_color()
 
	PURPOSE		: set the color of a neuron, depending on its voltage

	RETURNS		: nothing

************************************************************************************************************************/

void set_neuron_color()
{
	Window		window = XtWindow(drawingArea);			/* window of drawing area */
	GC		neuronGC;					/* graphical context */
	XGCValues	gcv;						/* values used in graphical contexts */
	unsigned long	mask;						/* masks used in graphical contexts */
	int neuronRadius = (int) rint((double) neuronDiameter / 2.0);	/* graphical neuron radius */
	int neuronStartX = (int) currentNeuron->x + neuronRadius;	/* x-position of middle of neuron circle */
	int neuronStartY = (int) currentNeuron->y + neuronRadius;	/* y-position of middle of neuron circle */
	int		i;						/* loop variables */
	int		colorNumber;					/* color number for activity state of neuron */
        int     	screenNr = DefaultScreen(disp);			/* screen number of current display */

	/* compute the color of the neuron */

	for (i = 0; i <= curNumOfDendSegs; i++)
	{
		/* compute color value of current segment */

		colorNumber = (int) rint((E_POT[i] - bottomVoltage) / neuronRange * numActivityColors);
		if (colorNumber >= numActivityColors)
			colorNumber = numActivityColors - 1;
		else if (colorNumber < 0)
			colorNumber = 0;

		gcv.foreground = activityColors[colorNumber];
		mask = GCForeground;
		neuronGC = XCreateGC(disp, window, mask, &gcv);

		if (i == 0) /* soma ? */
		{
			/* draw neuron as a filled circle */

			XFillArc(disp, window, neuronGC, currentNeuron->x, currentNeuron->y,
				neuronDiameter, neuronDiameter, 0, 360 * 64);
		}
		else /* dendritic segment */
		{
			if (networktype == 2)	/* short and fat dendritic stem */
			{
				XFillRectangle(disp, window, neuronGC, neuronStartX - StemWidth, neuronStartY +
				    neuronRadius, (unsigned int) (2 * StemWidth + 1), (unsigned int) (StemLength + 1));
			}
			else	/* long and skinny dendritic stem */
			{
				XFillRectangle(disp, window, neuronGC, neuronStartX - DendriteWidth, neuronStartY +
				    neuronRadius + (int) rint((double) (DendriteLength * (i - 1) / curNumOfDendSegs)),
				    (unsigned int) (2 * DendriteWidth + 1),
				    (unsigned int) (DendriteLength / curNumOfDendSegs));
			}
		}
	}

	/* free graphical contexts */

	XFreeGC(disp, neuronGC);

} /* end of set_neuron_color */

/************************************************************************************************************************

 	FUNCTION	: change_synapse_color(synapse, synapseColor)
 
	PURPOSE		: change the color of a synapse

	RETURNS		: nothing

************************************************************************************************************************/

void change_synapse_color(synapse, synapseColor)
Connection *synapse;	/* pointer to synapse which color should be changed */
char *synapseColor;	/* synapse color */
{
	Window		window = XtWindow(drawingArea);	/* window of drawing area */
	GC		synapseGC;			/* graphical context */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* masks used in graphical contexts */
	int		connectionX, connectionY;	/* graphical position of synapse */

	/* create GC for a circle */

	if (strcmp(synapseColor, resData.backgroundColor) == 0)
		gcv.foreground = backgroundColor;
	else if (strcmp(synapseColor, resData.neuronSelectionColor) == 0)
		gcv.foreground = neuronSelectionColor;
	else if (strcmp(synapseColor, resData.excitationSynapseColor) == 0)
		gcv.foreground = excitationSynapseColor;
	else if (strcmp(synapseColor, resData.inhibitionSynapseColor) == 0)
		gcv.foreground = inhibitionSynapseColor;
	else if (strcmp(synapseColor, resData.electricalSynapseColor) == 0)
		gcv.foreground = electricalSynapseColor;
	else if (strcmp(synapseColor, resData.axoSynapseColor) == 0)
		gcv.foreground = axoSynapseColor;
	else if (strcmp(synapseColor, resData.otherSynapseColor) == 0)
		gcv.foreground = otherSynapseColor;

	mask = GCForeground;
	synapseGC = XCreateGC(disp, window, mask, &gcv);

	/* calculate position of synapse depending on current network model */

	switch(networktype)
	{
	case 1: /* 1-point model */

		connectionX = synapse->connectionX_1;
		connectionY = synapse->connectionY_1;
		break;

	case 2: /* 2-Point model */

		connectionX = synapse->connectionX_2;
		connectionY = synapse->connectionY_2;
		break;

	case 3: /* 4-Point model */
	default:

		connectionX = synapse->connectionX_4;
		connectionY = synapse->connectionY_4;
		break;

	case 4: /* n-Point model */

		connectionX = synapse->connectionX_N;
		connectionY = synapse->connectionY_N;
		break;
	}

	/* compute middle of the synapse */

	connectionX -= (int) rint(synapseDiameter * 0.5);
	connectionY -= (int) rint(synapseDiameter * 0.5);

	/* draw synapse in the color determined by synapseColor */

	XFillArc(disp, window, synapseGC, connectionX, connectionY, synapseDiameter, synapseDiameter, 0, 360 * 64);

	XFreeGC(disp, synapseGC);	/* free graphical context */

} /* end of change_synapse_color */

/************************************************************************************************************************

 	FUNCTION	: show_all_objects_in_drawing_area(computeSynapse)
 
	PURPOSE		: redraw all objects in the net

	RETURNS		: nothing

************************************************************************************************************************/

void show_all_objects_in_drawing_area(computeSynapse)
Boolean computeSynapse;	/* flag, if synapse position must be computed or not */
{
	Window		window = XtWindow(drawingArea);	/* window of drawing area */
	GC		neuronGC, marginGC;		/* graphical contexts */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* masks used in graphical contexts */
	Neuron		*nptr;				/* pointer to current neuron */
	Connection 	*connection;			/* pointer to current synapse */
        int     	screenNr = DefaultScreen(disp);	/* screen number of current display */

	XClearWindow(disp, window);	/* clear window of drawing area */

	/* set neuron color depended on its state */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		if (nptr->selected)
			change_neuron_color(nptr, resData.neuronSelectionColor, resData.textColor);
		else if (nptr->isInputNeuron && nptr->isOutputNeuron)
			change_neuron_color(nptr, resData.inOutNeuronColor, resData.textColor);
		else if (nptr->isInputNeuron)
			change_neuron_color(nptr, resData.inputNeuronColor, resData.textColor);
		else if (nptr->isOutputNeuron)
			change_neuron_color(nptr, resData.outputNeuronColor, resData.textColor);
		else change_neuron_color(nptr, resData.neuronColor, resData.textColor);
	}

	/* draw all graphical connections from all neurons */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		for (connection = nptr->connectionList; connection != NULL; connection = connection->nextConnection)
		{
			connect_neurons_graphical(connection, computeSynapse);
		}
	}

	/* set neuron color depended on its state */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		if (nptr->selected)
			gcv.foreground = neuronSelectionColor;
		else if (nptr->isInputNeuron && nptr->isOutputNeuron)
			gcv.foreground = inOutNeuronColor;
		else if (nptr->isInputNeuron)
			gcv.foreground = inputNeuronColor;
		else if (nptr->isOutputNeuron)
			gcv.foreground = outputNeuronColor;
		else	gcv.foreground = neuronColor;

		mask = GCForeground;
		neuronGC = XCreateGC(disp, window, mask, &gcv);

		gcv.foreground = BlackPixel(disp,screenNr);
		gcv.line_width = 3;
		mask = GCForeground | GCLineWidth;
		marginGC = XCreateGC(disp, window, mask, &gcv);

		/* draw neuron as a filled circle in the color determined by neuronColor */

		XFillArc(disp, window, neuronGC, nptr->x, nptr->y, neuronDiameter, neuronDiameter, 0, 360 * 64);
		XDrawArc(disp, window, marginGC, nptr->x, nptr->y, neuronDiameter, neuronDiameter, 0, 360 * 64);

		XFreeGC(disp, neuronGC);
		XFreeGC(disp, marginGC);
	}
} /* end of show_all_objects_in_drawing_area */

/************************************************************************************************************************

 	FUNCTION	: show_input_neurons(neuron)
 
	PURPOSE		: highlight input neurons to the selected neuron

	RETURNS		: nothing

************************************************************************************************************************/

void show_input_neurons(neuron)
Neuron *neuron;	/* pointer to selected neuron */
{
	Neuron		*nptr;		/* current neuron pointer */
	Connection	*connection;	/* current synapse pointer */
	XmString	text;		/* information text */

	/* find all input neurons to the selected neuron and highlight them */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
		for (connection = nptr->connectionList; connection != NULL; connection = connection->nextConnection)
			if (connection->targetNeuron == neuron) /* does connection from nptr to neuron exist */
			{
				change_neuron_color(nptr, resData.inputNeuronColor, resData.textColor);
				nptr->isInputNeuron = TRUE;
				break;	/* exit for loop */
			}

	/* inform user that the highlightend neurons are input neurons */

	text = XmStringCreateLtoR(showInputString, defaultFont);
	XtManageChild(create_information_box(mainWindow, XmDIALOG_APPLICATION_MODAL, text));
	XmStringFree(text);

} /* end of show_input_neurons */

/************************************************************************************************************************

 	FUNCTION	: show_output_neurons(neuron)
 
	PURPOSE		: highlight output neurons to the selected neuron

	RETURNS		: nothing

************************************************************************************************************************/

void show_output_neurons(neuron)
Neuron *neuron;
{
	Connection	*connection;	/* current synapse pointer */
	XmString	text;		/* information text */

	/* find all output neurons from the selected neuron and highlight them */

	for (connection = neuron->connectionList; connection != NULL; connection = connection->nextConnection)
	{
		change_neuron_color(connection->targetNeuron, resData.outputNeuronColor, resData.textColor);
		connection->targetNeuron->isOutputNeuron = TRUE;
	}

	/* inform user that the highlightend neurons are output neurons */

	text = XmStringCreateLtoR(showOutputString, defaultFont);
	XtManageChild(create_information_box(mainWindow, XmDIALOG_APPLICATION_MODAL, text));
	XmStringFree(text);

} /* end of show_output_neurons */

/************************************************************************************************************************

 	FUNCTION	: show_input_output_neurons(neuron)
 
	PURPOSE		: highlight input and output neurons to the selected neuron

	RETURNS		: nothing

************************************************************************************************************************/

void show_input_output_neurons(neuron)
Neuron *neuron;
{
	Neuron		*nptr;		/* current neuron pointer */
	Connection	*connection;	/* current synapse pointer */
	XmString	text;		/* information text */

	/* find all input neurons to the selected neuron and highlight them */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
		for (connection = nptr->connectionList; connection != NULL; connection = connection->nextConnection)
			if (connection->targetNeuron == neuron) /* does connection from nptr to neuron exist */
			{
				change_neuron_color(nptr, resData.inputNeuronColor, resData.textColor);
				nptr->isInputNeuron = TRUE;
				break;	/* exit for loop */
			}

	/* find all output neurons from the selected neuron and highlight them */

	for (connection = neuron->connectionList; connection != NULL; connection = connection->nextConnection)
	{
		if (connection->targetNeuron->isInputNeuron)
			change_neuron_color(connection->targetNeuron, resData.inOutNeuronColor, resData.textColor);
		else change_neuron_color(connection->targetNeuron, resData.outputNeuronColor, resData.textColor);
		connection->targetNeuron->isOutputNeuron = TRUE;
	}

	/* inform user that the highlightend neurons are input and output neurons */

	text = XmStringCreateLtoR(showInputOutputString, defaultFont);
	XtManageChild(create_information_box(mainWindow, XmDIALOG_APPLICATION_MODAL, text));
	XmStringFree(text);

} /* end of show_input_output_neurons */

/************************************************************************************************************************

 	FUNCTION	: show_topology()
 
	PURPOSE		: refits the sizes of the network to fit into the window

	RETURNS		: nothing

************************************************************************************************************************/

void show_topology()
{
	Arg			args[10];			/* argument list for manipulating widgets */
	Cardinal		n;				/* used as argument counter for manipulating widgets */
	int			sliderSize;			/* size of current slider */
	int			increment;			/* step increment of current slider */
	int			pageIncrement;			/* page increment of current slider */
	int			xmin, ymin;			/* top left point of all objects in the net */
	int			xmax, ymax;			/* bottom right point of all objects in the net */
	double			sF;				/* scaling factor */
	double  		sFWidth, sFHeight;		/* scaling factor in respect of width and height */
	Neuron			*nptr;				/* pointer to current neuron */
	Widget			hSB, vSB;			/* horizontal and vertical scroll bars */
	XWindowAttributes	attr;				/* variable for referencing window attributes */
	XRectangle 		rectangle;			/* used to create region for neuron */
	Region 			region;				/* graphic region of neuron */

	if (neuronListHead == NULL)	/* if no neurons exist return */
		return;

	/* search top left most and bottom right most point of objects */

	xmin = xmax = neuronListHead->x;
	ymin = ymax = neuronListHead->y;

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		if (nptr->x < xmin)
			xmin = nptr->x;
		else if (nptr->x > xmax)
			xmax = nptr->x;
		if (nptr->y < ymin)
			ymin = nptr->y;
		else if (nptr->y > ymax)
			ymax = nptr->y;
	}

	/* correct found positions with constants */

	xmin -= msSpace;
	ymin -= font->ascent + font->descent + textOffset + msSpace;
	xmax += neuronDiameter + msSpace;
	ymax += neuronDiameter + DendriteLength + msSpace;

	/* calculate local scaling factor */

	XGetWindowAttributes(disp, XtWindow(XtParent(drawingArea)), &attr);

	sFWidth = (double) ((double) attr.width / (xmax - xmin));
	sFHeight = (double) ((double) attr.height / (ymax - ymin));

	if (sFWidth < sFHeight)
		sF = sFWidth;
	else sF = sFHeight;
	if ((int) (rint(neuronDiameter * sF)) < 1)
		sF = 1.0 / ((double) neuronDiameter);

	/* scale top left position with scaling factor sF */

	xmin = (int) (rint(xmin * sF));
	ymin = (int) (rint(ymin * sF));

	/* correct top left position so its impossible to be negative */

	if (xmin < 0)
		xmin = 0;
	if (ymin < 0)
		ymin = 0;

	/* get scroll bars */

	n = 0;
	XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
	XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
	XtGetValues(mainWindow, args, n);

	/* get values of horizontal scroll bar */

	n = 0;
	XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
	XtSetArg(args[n], XmNincrement, &increment); n++;
	XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
	XtGetValues(hSB, args, n);

	/* set new position of horizontal scroll bar */

	XmScrollBarSetValues(hSB, xmin, sliderSize, increment, pageIncrement, TRUE);

	/* get values of vertical scroll bar */

	n = 0;
	XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
	XtSetArg(args[n], XmNincrement, &increment); n++;
	XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
	XtGetValues(vSB, args, n);

	/* set new position of vertical scroll bar */

	XmScrollBarSetValues(vSB, ymin, sliderSize, increment, pageIncrement, TRUE);

	/* move drawing area to top left position */

	XtMoveWidget(drawingArea, - xmin, - ymin);

	/* update global scaling factor with local computed scaling factor */

	scalingFactor *= sF;

	/* compute new sizes of graphical objects */

	neuronDiameter = (int) (rint(defaultNeuronDiameter * scalingFactor));
	synapseDiameter = (int) (rint(defaultSynapseDiameter * scalingFactor));
	DendriteWidth = (int) (rint(defaultDendriteWidth * scalingFactor));
	if (networktype == 4)
		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs * scalingFactor);
	else	DendriteLength = (int) (rint(defaultDendriteLength * scalingFactor));
	StemWidth = (int) (rint(defaultStemWidth * scalingFactor));
	StemLength = (int) (rint(defaultStemLength * scalingFactor));
	OwnConnectionRadius = (int) (rint(defaultOwnConnectionRadius * scalingFactor));

	/* compute new graphical regions for neurons */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		nptr->x = (int) (rint(nptr->xOriginal * scalingFactor));
		nptr->y = (int) (rint(nptr->yOriginal * scalingFactor));

		XDestroyRegion(nptr->region);
		region = XCreateRegion();

		rectangle.x = (short) nptr->x;
		rectangle.y = (short) nptr->y;
		rectangle.width = (unsigned short) neuronDiameter;
		rectangle.height = (unsigned short) neuronDiameter;
		XUnionRectWithRegion(&rectangle, region, region);

		nptr->region = region;	/* set new region of neuron */
	}

	/* redraw all objects in drawing area */

	show_all_objects_in_drawing_area(TRUE);

} /* end of show_topology */

/************************************************************************************************************************

 	FUNCTION	: show_normal_state(void)
 
	PURPOSE		: reuse the initial sizes and colors to display the network

	RETURNS		: nothing

************************************************************************************************************************/

void show_normal_state()
{
	Arg			args[10];			/* argument list for manipulating widgets */
	Cardinal		n;				/* used as argument counter for manipulating widgets */
	int			sliderSize;			/* size of current slider */
	int			increment;			/* step increment of current slider */
	int			pageIncrement;			/* page increment of current slider */
	Neuron			*nptr;				/* pointer to current neuron */
	Connection		*connection;			/* pointer to current synapse */
	Widget			hSB, vSB;			/* horizontal and vertical scroll bars */
	XRectangle 		rectangle;			/* used to create region for neuron */
	Region 			region;				/* graphic region of neuron */

	/* set graphic dependend parameters to original values */

	neuronDiameter = defaultNeuronDiameter;
	synapseDiameter = defaultSynapseDiameter;
	DendriteWidth = defaultDendriteWidth;
	if (networktype == 4)
		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs);
	else	DendriteLength = defaultDendriteLength;
	StemLength = defaultStemLength;
	StemWidth = defaultStemWidth;
	OwnConnectionRadius = defaultOwnConnectionRadius;
	scalingFactor = 1.0;

	XClearWindow(disp, XtWindow(drawingArea));	/* clear drawing area */

	/* get scroll bars */

	n = 0;
	XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
	XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
	XtGetValues(mainWindow, args, n);

	/* get values of horizontal scroll bar */

	n = 0;
	XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
	XtSetArg(args[n], XmNincrement, &increment); n++;
	XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
	XtGetValues(hSB, args, n);

	/* set horizontal scroll bar to position zero */

	XmScrollBarSetValues(hSB, 0, sliderSize, increment, pageIncrement, TRUE);

	/* get values of vertical scroll bar */

	n = 0;
	XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
	XtSetArg(args[n], XmNincrement, &increment); n++;
	XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
	XtGetValues(vSB, args, n);

	/* set vertical scroll bar to position zero */

	XmScrollBarSetValues(vSB, 0, sliderSize, increment, pageIncrement, TRUE);

	/* move drawing area to position 0,0 */

	XtMoveWidget(drawingArea, 0, 0);

	/* reset neurons */

	for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
	{
		/* reset special neuron conditions */

		nptr->isInputNeuron = FALSE;
		nptr->isOutputNeuron = FALSE;

		/* restore original neuron position */

		nptr->x = nptr->xOriginal;
		nptr->y = nptr->yOriginal;

		/* create new region for neuron */

		XDestroyRegion(nptr->region);
		region = XCreateRegion();
		rectangle.x = (short) nptr->x;
		rectangle.y = (short) nptr->y;
		rectangle.width = (unsigned short) neuronDiameter;
		rectangle.height = (unsigned short) neuronDiameter;
		XUnionRectWithRegion(&rectangle, region, region);
		nptr->region = region;

		/* reset extern input currents */

		reset_extern_input_currents(nptr);

		/* reset selected state of synapses */

		for (connection = nptr->connectionList; connection != NULL; connection = connection->nextConnection)
		{
			connection->selected = FALSE;			/* reset selection status */
		}
	}

	show_all_objects_in_drawing_area(TRUE);

} /* end of show_normal_state */

/************************************************************************************************************************

 	FUNCTION	: process_buttonevent(buttonEvp)
 
	PURPOSE		: processes button events in the drawing area for the network
			  Button1: if mousepointer is inside a neuron or synapse select or deselect object 
			           if a modifier key is pressed, select all objects within the rubber banded rectangle
			           else if the control key is pressed create a new neuron
			  Button2: display menu of all posibilities within the drawing area depending on selected objects
			  Button3: move neuron or zoom selected region

	RETURNS		: nothing

************************************************************************************************************************/

void process_buttonevent(buttonEvp)
XButtonEvent	*buttonEvp;	/* button event pointer */
{
	Arg			args[10];			/* argument list for manipulating widgets */
	Cardinal		n;				/* used as argument counter for manipulating widgets */
	Boolean			neuronSelected;			/* true if neuron is selected */
	Boolean			synapseSelected;		/* true if synapse is selected */
	Boolean			idFound;			/* used to determine next free neuron id */
	Boolean			moving;				/* true if selected mode is moving neurons */
	Boolean			firstTime;			/* used for rubber band drawing */
	int			rootX, rootY;			/* position of the pointer relative to root's origin */
	int			winX, winY;			/* position of the pointer relative to current window */
	int			startX, startY;			/* position of the pointer relative to current window */
	int			x, y;				/* current pointer position */
	int			old_x, old_y;			/* old pointer position */
	int			xnew, ynew;			/* new pointer position */
	int			widthnew, heightnew;		/* rescaled width and height of zoomed area */
	int			maximumH, maximumV;		/* horizontal and vertical sliders maximum value */
	int			sliderSize;			/* size of current slider */
	int			increment;			/* step increment of current slider */
	int			pageIncrement;			/* page increment of current slider */
	int			i;				/* help variable */
	unsigned int		state;				/* state of the mouse buttons */
	unsigned int		width, height;			/* width and height of rubber banded rectangle */
	unsigned int		old_width, old_height;		/* old width and height of rubber banded rectangle */
	double			sF;				/* scaling factor */
	double  		sFWidth, sFHeight;		/* scaling factor in respect of width and height */
	Neuron			*nptr;				/* pointer to current neuron */
	Connection		*connection;			/* pointer to current synapse */
	ChemicalPool		*pool;				/* pointer to current pool */
	ChemicalSenderList 	*sender;			/* pointer to current sender */
	AxoAxonicSenderList	*axoSender;			/* pointer to current axo-axonic sender */
	Window			window = buttonEvp->window;	/* window in which the button event has occured */
	Window			root;				/* root window, used for query pointer position */
	Window			child;				/* child window, used for query pointer position */
	Widget			hSB, vSB;			/* horizontal and vertical scroll bars */
	XWindowAttributes	attr;				/* variable for referencing window attributes */
	XRectangle 		rectangle;			/* used to create region for neuron */
	Region 			region;				/* graphic region of neuron */
	Region			intersectRegion;		/* used for testing of creating overlapping neurons */
	GC			rubberBandGC, backgroundGC;	/* graphical context for rubber band and background */
	XGCValues		gcv;				/* values used in graphical contexts */
	unsigned long		mask;				/* masks used in graphical contexts */
	XEvent			report;				/* result of event status */

	/* save cursor position and mouse button state */

	x = buttonEvp->x;
	y = buttonEvp->y;
	state = buttonEvp->state;

	/* evaluate button event */

	switch(buttonEvp->button)
	{
	case Button1:	/* if mousepointer is inside a neuron or synapse select object or deselect it if its selected
			   if a modifier key is pressed, select all objects within the rubber banded rectangle
			   else if the control key is pressed create a new neuron */

		/* if Mod-Key (Alt-Key or AltGr-Key) is pressed, select all objects in the rectangle */

		if (state == Mod1Mask || state == Mod2Mask)
		{
			/* grab the mouse pointer */

			XGrabPointer(disp, window, FALSE, ButtonReleaseMask, GrabModeSync, GrabModeSync, window,
				XCreateFontCursor(disp, XC_crosshair), CurrentTime);

			/* allows pointer motion until next ButtonRelease event */

			XAllowEvents(disp, SyncPointer, CurrentTime);

			/* query current cursor position */

			XQueryPointer(disp, window, &root, &child, &rootX, &rootY, &startX, &startY, &state);
			firstTime = TRUE;

			/* create graphical contexts for the rubber band and the inverse rubber band (for deletion) */

			gcv.foreground = rubberBandColor;
			gcv.line_style = LineOnOffDash;
			mask = GCForeground | GCLineStyle;
			rubberBandGC = XCreateGC(disp, window, mask, &gcv);
			gcv.foreground = backgroundColor;
			backgroundGC = XCreateGC(disp, window, mask, &gcv);

			/* follow the mouse pointer and draw the rubber band until mouse button is released */

			do
			{
				XQueryPointer(disp, window, &root, &child, &rootX, &rootY, &winX, &winY, &state);
				if (winX != startX && winY != startY)
				{
					/* clear old rubber band */

					if (! firstTime)
						XDrawRectangle(disp, window, backgroundGC, old_x, old_y, old_width,
						    old_height);

					/* draw the rubber band */

					if (winX < startX)
						x = winX;
					else	x = startX;
					if (winY < startY)
						y = winY;
					else	y = startY;
					width = (unsigned int) abs(winX - startX);
					height = (unsigned int) abs(winY - startY);
					XDrawRectangle(disp, window, rubberBandGC, x, y, width, height);

					/* save current cursor position and rectangle size */

					old_x = x;
					old_y = y;
					old_width = width;
					old_height = height;
					firstTime = FALSE;
				}
			} while (! XCheckTypedEvent(disp, ButtonRelease, &report));

			/* release mouse pointer, clear rubber band and free graphical contexts */

			XUngrabPointer(disp, CurrentTime);
			XDrawRectangle(disp, window, backgroundGC, x, y, width, height);
			XFreeGC(disp, backgroundGC);
			XFreeGC(disp, rubberBandGC);

			/* select objects within rubber banded area */

			remove_selected_objects_from_selectionList();

			if (state == Mod1Mask)	/* select neurons */
			{
				for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
				{
					/* check if current neuron lies within the rubber banded area */

					if (x <= (int) nptr->x && y <= (int) nptr->y &&
						(int) (x + (int) width) >= (int) ((int) nptr->x + neuronDiameter) &&
						(int) (y + (int) height) >= (int) ((int) nptr->y + neuronDiameter))
					{
						add_neuron_to_selection_list(nptr);
						nptr->selected = TRUE;
						change_neuron_color(nptr, resData.neuronSelectionColor,
							resData.textColor);
						neuronSelected = TRUE;
					}
				}
			}
			else	/* select synapses */
			{
				for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
				{
					for (currentSynapse = nptr->connectionList; currentSynapse != NULL;
					    currentSynapse = currentSynapse->nextConnection)
					{
						/* check if current synapse lies within the rubber banded area */

						switch (networktype)
						{
						case 1:
							if (x <= (int) currentSynapse->connectionX_1 &&
								y <= (int) currentSynapse->connectionY_1 &&
								(int) (x + (int) width) >= (int) ((int) currentSynapse->
									connectionX_1 + synapseDiameter) &&
								(int) (y + (int) height) >= (int) ((int) currentSynapse->
									connectionY_1 + synapseDiameter))
							{
								add_synapse_to_selection_list(currentSynapse);
								currentSynapse->selected = TRUE;
								change_synapse_color(currentSynapse,
									resData.neuronSelectionColor);
								synapseSelected = TRUE;
							}
							break;
						case 2:
							if (x <= (int) currentSynapse->connectionX_2 &&
								y <= (int) currentSynapse->connectionY_2 &&
								(int) (x + (int) width) >= (int) ((int) currentSynapse->
									connectionX_2 + synapseDiameter) &&
								(int) (y + (int) height) >= (int) ((int) currentSynapse->
									connectionY_2 + synapseDiameter))
							{
								add_synapse_to_selection_list(currentSynapse);
								currentSynapse->selected = TRUE;
								change_synapse_color(currentSynapse,
									resData.neuronSelectionColor);
								synapseSelected = TRUE;
							}
							break;
						case 3:
						default:
							if (x <= (int) currentSynapse->connectionX_4 &&
								y <= (int) currentSynapse->connectionY_4 &&
								(int) (x + (int) width) >= (int) ((int) currentSynapse->
									connectionX_4 + synapseDiameter) &&
								(int) (y + (int) height) >= (int) ((int) currentSynapse->
									connectionY_4 + synapseDiameter))
							{
								add_synapse_to_selection_list(currentSynapse);
								currentSynapse->selected = TRUE;
								change_synapse_color(currentSynapse,
									resData.neuronSelectionColor);
								synapseSelected = TRUE;
							}
							break;
						case 4:
							if (x <= (int) currentSynapse->connectionX_N &&
								y <= (int) currentSynapse->connectionY_N &&
								(int) (x + (int) width) >= (int) ((int) currentSynapse->
									connectionX_N + synapseDiameter) &&
								(int) (y + (int) height) >= (int) ((int) currentSynapse->
									connectionY_N + synapseDiameter))
							{
								add_synapse_to_selection_list(currentSynapse);
								currentSynapse->selected = TRUE;
								change_synapse_color(currentSynapse,
									resData.neuronSelectionColor);
								synapseSelected = TRUE;
							}
							break;
						}
					}
				}
			}
			show_all_objects_in_drawing_area(TRUE);
			break;	/* leave case statement */

		} /* end of modifier key pressed */

		/* check if cursor is inside of an object */

		neuronSelected = FALSE;

		if (selectionListSynapsesHead == NULL)	/* look for neurons only if no synapse is selected */
		{
			/* loop through all neurons in the net */

			for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
			{
				if (XPointInRegion(nptr->region, x, y))	/* lies cursor in region of current neuron ? */
				{
					if (nptr->selected) /* neuron is selected, deselect it */
					{
						remove_neuron_from_selection_list(nptr);
						nptr->selected = FALSE;
						if (nptr->isInputNeuron && nptr->isOutputNeuron)
							change_neuron_color(nptr, resData.inOutNeuronColor,
								resData.textColor);
						else if (nptr->isInputNeuron)
							change_neuron_color(nptr, resData.inputNeuronColor,
								resData.textColor);
						else if (nptr->isOutputNeuron)
							change_neuron_color(nptr, resData.outputNeuronColor,
								resData.textColor);
						else change_neuron_color(nptr, resData.neuronColor, resData.textColor);
						neuronSelected = TRUE;
						break;	/* leave for loop, because neuron was found */
					}
					else /* select neuron */
					{
						add_neuron_to_selection_list(nptr);
						nptr->selected = TRUE;
						change_neuron_color(nptr, resData.neuronSelectionColor,
							resData.textColor);
						neuronSelected = TRUE;
						break;	/* leave for loop, because neuron was found */
					}
				}
			}
		}

		if (neuronSelected)
			break; /* leave case statement, because the action was to select/deselect a neuron */

		synapseSelected = FALSE;

		/* look for synapses only if no neuron is selected or the network modeltype is complex */

		if ((selectionListNeuronsHead == NULL) ||
		   (complexModel && (selectionListSynapsesCounter == 0) && (selectionListNeuronsCounter < 2)))
		{
			/* loop through all neurons in the net */

			for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
			{
				/* loop through all synapses in the net */

				for (currentSynapse = nptr->connectionList; currentSynapse != NULL;
				    currentSynapse = currentSynapse->nextConnection)
				{
					if (XPointInRegion(currentSynapse->targetRegion, x, y))
					{
						if (currentSynapse->selected) /* synapse is selected, deselect it */
						{
							remove_synapse_from_selection_list(currentSynapse);
							currentSynapse->selected = FALSE;
							if (currentSynapse->axoSynapseNumber > 0) /* axo-axonic */
								change_synapse_color(currentSynapse,
									resData.axoSynapseColor);
							else
								switch (synapseTypes[currentSynapse->synapticType].ionType)
								{
								case 1:
									change_synapse_color(currentSynapse,
										resData.excitationSynapseColor);
									break;

								case 2:
									change_synapse_color(currentSynapse,
										resData.inhibitionSynapseColor);
									break;

								case 3:
									change_synapse_color(currentSynapse,
										resData.otherSynapseColor);
									break;

								case 4:
									change_synapse_color(currentSynapse,
										resData.electricalSynapseColor);
									break;
								}
							synapseSelected = TRUE;
							break;
						}
						else /* select synapse if allowed */
						{
							if ((currentSynapse->targetSynapse == NULL ||
							    selectionListNeuronsHead == NULL) &&
							    ((selectionListSynapsesHead != NULL &&
							    ((selectionListSynapsesHead->synapsePtr->axoSynapseNumber ==
							    currentSynapse->axoSynapseNumber) ||
							    (selectionListSynapsesHead->synapsePtr->axoSynapseNumber > 0 &&
							    currentSynapse->axoSynapseNumber > 0))) ||
							    selectionListSynapsesHead == NULL))
							{
								add_synapse_to_selection_list(currentSynapse);
								currentSynapse->selected = TRUE;
								change_synapse_color(currentSynapse,
									resData.neuronSelectionColor);
								synapseSelected = TRUE;
							}
							break;	/* leave for loop through synapses */
						}
					}
				}
				if (synapseSelected)
					break;	/* leave for loop through neurons */
			}
		}

		if (synapseSelected)
			break; /* leave case statement, because the action was to select/deselect a synapse */

		/* check if one neuron and one synapse are selected for creating an axo-axonic connection */

		if ((selectionListSynapsesCounter == 1) && (selectionListNeuronsCounter == 1))
		{
			/* loop through all neurons in the net */

			for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
			{
				if (XPointInRegion(nptr->region, x, y))	/* lies cursor in region of current neuron ? */
				{
					/* is cursor in selected neuron ? */

					if (nptr == selectionListNeuronsHead->neuronPtr)
					{
						remove_neuron_from_selection_list(nptr);
						nptr->selected = FALSE;
						if (nptr->isInputNeuron && nptr->isOutputNeuron)
							change_neuron_color(nptr, resData.inOutNeuronColor,
								resData.textColor);
						else if (nptr->isInputNeuron)
							change_neuron_color(nptr, resData.inputNeuronColor,
								resData.textColor);
						else if (nptr->isOutputNeuron)
							change_neuron_color(nptr, resData.outputNeuronColor,
								resData.textColor);
						else change_neuron_color(nptr, resData.neuronColor, resData.textColor);
						neuronSelected = TRUE;
						break;	/* leave for loop, because neuron was found */
					}
				}

				/* loop through all synapses in the net */

				for (currentSynapse = nptr->connectionList; currentSynapse != NULL;
				    currentSynapse = currentSynapse->nextConnection)
				{
					if (XPointInRegion(currentSynapse->targetRegion, x, y))
					{
						/* is cursor in selected synapse ? */

						if (currentSynapse == selectionListSynapsesHead->synapsePtr)
						{
							remove_synapse_from_selection_list(currentSynapse);
							currentSynapse->selected = FALSE;
							if (currentSynapse->axoSynapseNumber > 0) /* axo-axonic */
								change_synapse_color(currentSynapse,
									resData.axoSynapseColor);
							else
								switch (synapseTypes[currentSynapse->synapticType].ionType)
								{
								case 1:
									change_synapse_color(currentSynapse,
										resData.excitationSynapseColor);
									break;

								case 2:
									change_synapse_color(currentSynapse,
										resData.inhibitionSynapseColor);
									break;

								case 3:
									change_synapse_color(currentSynapse,
										resData.otherSynapseColor);
									break;

								case 4:
									change_synapse_color(currentSynapse,
										resData.electricalSynapseColor);
									break;
								}
							synapseSelected = TRUE;
							break;
						}
					}
				}
			}
		}

		/* if control-key is not pressed return without creating a neuron */

		if (state != ControlMask)
			break;

		/* create the region for a new neuron at current cursor position */

		region = XCreateRegion();	/* creates an empty region */
		rectangle.x = (short) x;
		rectangle.y = (short) y;
		rectangle.width = (unsigned short) neuronDiameter;
		rectangle.height = (unsigned short) neuronDiameter;
		XUnionRectWithRegion(&rectangle, region, region);

		/* check if another neuron is occured by this neuron */

		neuronSelected = FALSE;
		for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
		{
			intersectRegion = XCreateRegion(); /* create an empty intersection region */
			XIntersectRegion(region, nptr->region, intersectRegion);
			if ( ! XEmptyRegion(intersectRegion))
			{
				neuronSelected = TRUE;
				XDestroyRegion(intersectRegion);
				XDestroyRegion(region);
				break;
			}
			XDestroyRegion(intersectRegion);
		}
		if (neuronSelected)
			break;	/* leave case statement cause neuron intersects with an other */

		/* get scroll bars */

		n = 0;
		XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
		XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
		XtGetValues(mainWindow, args, n);

		/* get maximum values of scroll bars */

		n = 0;
		XtSetArg(args[n], XmNmaximum, &maximumH); n++;
		XtGetValues(hSB, args, n);
		n = 0;
		XtSetArg(args[n], XmNmaximum, &maximumV); n++;
		XtGetValues(vSB, args, n);

		/* scale cursor position by global scaling factor */

		x = (int) rint((double) x / scalingFactor);
		y = (int) rint((double) y / scalingFactor);

		/* check if position is valid */

		if ((x < msSpace) || (y < msSpace) || (x > (maximumH - msSpace)) ||
			(y > (maximumV - DendriteLength - msSpace)))
			break;	/* leave case statement cause position is not allowed */

		/* search the next free neuron number beginning with number 1 */

		for (i = 1; i <= neuronNumber; i++)
		{
			idFound = FALSE;
			for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
			{
				if (nptr->neuronId == i)
				{
					idFound = TRUE;
					break;
				}
			}
			if (! idFound)
				break;	/* leave for loop */
		}

		/* store neuron in neuron list */

		add_neuron_to_neuron_list(i, 1);
		neuronListTail->x = buttonEvp->x;
		neuronListTail->xOriginal = x;
		neuronListTail->y = buttonEvp->y;
		neuronListTail->yOriginal = y;
		neuronListTail->region = region;
		change_neuron_color(neuronListTail, resData.neuronColor, resData.textColor);

		break;

	case Button2:	/* display a menu of all posibilities within the drawing area depending on selected objects */

		if (selectionListSynapsesCounter == 0)	/* no synapses selected */
		{
			switch (selectionListNeuronsCounter)
			{
			case 0:	/* no neurons selected */

				XtSetSensitive(deleteObjectsButton, FALSE);
				XtSetSensitive(specifyObjectsButton, FALSE);
				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editChannelButton, FALSE);
				XtSetSensitive(editNeuronButton, FALSE);
				XtSetSensitive(editCurrentButton, FALSE);
				XtSetSensitive(editConnectionButton, FALSE);
				XtSetSensitive(attachWindowButton, FALSE);
				XtSetSensitive(attachWindowSButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(informationObjectButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			case 1:	/* one neuron selected */

				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowSButton, FALSE);
				XtSetSensitive(connectObjectsButton, TRUE);
				XtSetSensitive(editConnectionButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editChannelButton, TRUE);
				XtSetSensitive(editNeuronButton, TRUE);
				XtSetSensitive(editCurrentButton, TRUE);
				XtSetSensitive(informationObjectButton, TRUE);
				XtSetSensitive(attachWindowButton, TRUE);
				XtSetSensitive(showInputNeuronsButton, TRUE);
				XtSetSensitive(showOutputNeuronsButton, TRUE);
				XtSetSensitive(showInputOutputNeuronsButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			case 2:	/* two neurons selected */

				XtSetSensitive(informationObjectButton, FALSE);
				XtSetSensitive(attachWindowSButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editChannelButton, TRUE);
				XtSetSensitive(editNeuronButton, TRUE);
				XtSetSensitive(editCurrentButton, TRUE);
				XtSetSensitive(connectObjectsButton, TRUE);
				XtSetSensitive(editConnectionButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			default: /* more than two neurons selected */

				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editConnectionButton, FALSE);
				XtSetSensitive(informationObjectButton, FALSE);
				XtSetSensitive(attachWindowSButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editChannelButton, TRUE);
				XtSetSensitive(editNeuronButton, TRUE);
				XtSetSensitive(editCurrentButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;
			}
		}
		else	/* no neurons selected */
		{
			switch (selectionListSynapsesCounter)
			{
			case 0:	/* no synapses selected */

				XtSetSensitive(deleteObjectsButton, FALSE);
				XtSetSensitive(specifyObjectsButton, FALSE);
				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editChannelButton, FALSE);
				XtSetSensitive(editNeuronButton, FALSE);
				XtSetSensitive(editCurrentButton, FALSE);
				XtSetSensitive(editConnectionButton, FALSE);
				XtSetSensitive(attachWindowButton, FALSE);
				XtSetSensitive(attachWindowSButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(informationObjectButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			case 1:	/* one synapse selected */

				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editChannelButton, FALSE);
				XtSetSensitive(editNeuronButton, FALSE);
				XtSetSensitive(editCurrentButton, FALSE);
				XtSetSensitive(attachWindowButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowSButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editConnectionButton, TRUE);
				XtSetSensitive(informationObjectButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			case 2:	/* two synapses selected */

				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editChannelButton, FALSE);
				XtSetSensitive(editNeuronButton, FALSE);
				XtSetSensitive(editCurrentButton, FALSE);
				XtSetSensitive(informationObjectButton, FALSE);
				XtSetSensitive(attachWindowButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowSButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editConnectionButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;

			default: /* more than two synapses selected */

				XtSetSensitive(connectObjectsButton, FALSE);
				XtSetSensitive(editChannelButton, FALSE);
				XtSetSensitive(editNeuronButton, FALSE);
				XtSetSensitive(editCurrentButton, FALSE);
				XtSetSensitive(informationObjectButton, FALSE);
				XtSetSensitive(attachWindowButton, FALSE);
				XtSetSensitive(showInputNeuronsButton, FALSE);
				XtSetSensitive(showOutputNeuronsButton, FALSE);
				XtSetSensitive(showInputOutputNeuronsButton, FALSE);
				XtSetSensitive(editAxoConnectionButton, FALSE);
				XtSetSensitive(attachWindowSButton, TRUE);
				XtSetSensitive(deleteObjectsButton, TRUE);
				XtSetSensitive(specifyObjectsButton, TRUE);
				XtSetSensitive(editConnectionButton, TRUE);
				XtSetSensitive(showTopologyButton, TRUE);
				XtSetSensitive(showNormalStateButton, TRUE);
				XtSetSensitive(selectNeuronsButton, TRUE);
				XtSetSensitive(selectSynapsesButton, TRUE);
				XtSetSensitive(selectGraphsButton, TRUE);
				if (complexModel)
				{
					XtSetSensitive(selectAxoSynapsesButton, TRUE);
				}
				else
				{
					XtSetSensitive(selectAxoSynapsesButton, FALSE);
				}
				break;
			}
		}

		/* check if one neuron and one synapse were selected and modeltype is complex */

		if (complexModel && (selectionListNeuronsCounter == 1) && (selectionListSynapsesCounter == 1))
		{
			XtSetSensitive(editChannelButton, FALSE);
			XtSetSensitive(editNeuronButton, FALSE);
			XtSetSensitive(editCurrentButton, FALSE);
			XtSetSensitive(informationObjectButton, FALSE);
			XtSetSensitive(attachWindowButton, FALSE);
			XtSetSensitive(attachWindowSButton, FALSE);
			XtSetSensitive(showInputNeuronsButton, FALSE);
			XtSetSensitive(showOutputNeuronsButton, FALSE);
			XtSetSensitive(showInputOutputNeuronsButton, FALSE);
			XtSetSensitive(deleteObjectsButton, FALSE);
			XtSetSensitive(editConnectionButton, FALSE);
			XtSetSensitive(editAxoConnectionButton, FALSE);
			XtSetSensitive(specifyObjectsButton, FALSE);
			XtSetSensitive(connectObjectsButton, TRUE);
			XtSetSensitive(showTopologyButton, TRUE);
			XtSetSensitive(showNormalStateButton, TRUE);
			XtSetSensitive(selectNeuronsButton, TRUE);
			XtSetSensitive(selectSynapsesButton, TRUE);
			XtSetSensitive(selectGraphsButton, TRUE);
		}

		/* check if one axo-axonic synapse was selected */

		else if (selectionListSynapsesCounter == 1 && selectionListSynapsesHead->synapsePtr->axoSynapseNumber > 0)
		{
			XtSetSensitive(editChannelButton, FALSE);
			XtSetSensitive(editNeuronButton, FALSE);
			XtSetSensitive(editCurrentButton, FALSE);
			XtSetSensitive(attachWindowButton, FALSE);
			XtSetSensitive(showInputNeuronsButton, FALSE);
			XtSetSensitive(showOutputNeuronsButton, FALSE);
			XtSetSensitive(showInputOutputNeuronsButton, FALSE);
			XtSetSensitive(editConnectionButton, FALSE);
			XtSetSensitive(connectObjectsButton, FALSE);
			XtSetSensitive(attachWindowSButton, TRUE);
			XtSetSensitive(informationObjectButton, TRUE);
			XtSetSensitive(deleteObjectsButton, TRUE);
			XtSetSensitive(editAxoConnectionButton, TRUE);
			XtSetSensitive(specifyObjectsButton, TRUE);
			XtSetSensitive(showTopologyButton, TRUE);
			XtSetSensitive(showNormalStateButton, TRUE);
			XtSetSensitive(selectNeuronsButton, TRUE);
			XtSetSensitive(selectSynapsesButton, TRUE);
			XtSetSensitive(selectGraphsButton, TRUE);
		}

		/* check if more than one axo-axonic synapse were selected */

		else if (selectionListSynapsesCounter > 0 && selectionListSynapsesHead->synapsePtr->axoSynapseNumber > 0)
		{
			XtSetSensitive(editChannelButton, FALSE);
			XtSetSensitive(editNeuronButton, FALSE);
			XtSetSensitive(editCurrentButton, FALSE);
			XtSetSensitive(attachWindowButton, FALSE);
			XtSetSensitive(showInputNeuronsButton, FALSE);
			XtSetSensitive(showOutputNeuronsButton, FALSE);
			XtSetSensitive(showInputOutputNeuronsButton, FALSE);
			XtSetSensitive(editConnectionButton, FALSE);
			XtSetSensitive(connectObjectsButton, FALSE);
			XtSetSensitive(attachWindowSButton, TRUE);
			XtSetSensitive(informationObjectButton, TRUE);
			XtSetSensitive(deleteObjectsButton, TRUE);
			XtSetSensitive(editAxoConnectionButton, TRUE);
			XtSetSensitive(specifyObjectsButton, TRUE);
			XtSetSensitive(showTopologyButton, TRUE);
			XtSetSensitive(showNormalStateButton, TRUE);
			XtSetSensitive(selectNeuronsButton, TRUE);
			XtSetSensitive(selectSynapsesButton, TRUE);
			XtSetSensitive(selectGraphsButton, TRUE);
		}

		/* show popup menu */

		XmMenuPosition(popupMenu, buttonEvp);
		XtManageChild(popupMenu);
		break;

	case Button3:	/* move neuron or zoom selected region */

		moving = FALSE;

		/* loop through all neurons in the net */

		for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
		{
			/* check if cursor lies in current neurons region */

			if (XPointInRegion(nptr->region, x, y))
			{
				/* clear all graphical connections to and from this neuron */

				for (connection = nptr->connectionList; connection != NULL; connection = connection->
				    nextConnection)
				{
					delete_graphical_connection(connection);

					for (axoSender = connection->axoSenderList; axoSender != NULL;
					     axoSender = axoSender->nextSender)
					{
						delete_graphical_connection(axoSender->connection);
					}
				}
				for (pool = nptr->poolList; pool != NULL; pool = pool->nextPool)
				{
					for (sender = pool->senderList; sender != NULL; sender = sender->nextSender)
					{
						delete_graphical_connection(sender->connection);

						for (axoSender = sender->connection->axoSenderList; axoSender != NULL;
						     axoSender = axoSender->nextSender)
						{
							delete_graphical_connection(axoSender->connection);
						}
					}
				}

				/* get scroll bars */

				n = 0;
				XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
				XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
				XtGetValues(mainWindow, args, n);

				/* get maximum values of scroll bars */

				n = 0;
				XtSetArg(args[n], XmNmaximum, &maximumH); n++;
				XtGetValues(hSB, args, n);
				n = 0;
				XtSetArg(args[n], XmNmaximum, &maximumV); n++;
				XtGetValues(vSB, args, n);

				/* grab the mouse pointer */

				XGrabPointer(disp, window, FALSE, ButtonReleaseMask, GrabModeSync, GrabModeSync, window,
				    XCreateFontCursor(disp, XC_hand2), CurrentTime);

				/* allows pointer motion until next ButtonRelease event */

				XAllowEvents(disp, SyncPointer, CurrentTime);

				/* follow the mouse pointer and draw the neuron until mouse button is released */

				do
				{
					/* query mouse pointer */

					XQueryPointer(disp, window, &root, &child, &rootX, &rootY, &winX, &winY, &state);

					/* check bounderies */

					if (winX < (int) rint((double) msSpace * scalingFactor))
						winX = (int) rint((double) msSpace * scalingFactor);
					if (winY < (int) rint((double) msSpace * scalingFactor))
						winY = (int) rint((double) msSpace * scalingFactor);
					if (winX > (int) rint((double) (maximumH - msSpace) * scalingFactor))
						winX = (int) rint((double) (maximumH - msSpace) * scalingFactor);
					if (winY > (int) rint((double) (maximumV - DendriteLength - msSpace)
							* scalingFactor))
						winY = (int) rint((double) (maximumV - DendriteLength - msSpace) *
							scalingFactor);

					/* clear old neuron position and set neuron to new position */

					change_neuron_color(nptr, resData.backgroundColor, resData.backgroundColor);
					nptr->x = winX;
					nptr->y = winY;
					change_neuron_color(nptr, resData.neuronMovingColor, resData.textColor);

				} while (! XCheckTypedEvent(disp, ButtonRelease, &report));

				/* create new region for the neuron at the current position */

				XDestroyRegion(nptr->region);	/* deallocate old region */
				region = XCreateRegion();	/* creates an empty region */
				rectangle.x = (short) nptr->x;
				rectangle.y = (short) nptr->y;
				rectangle.width = (unsigned short) neuronDiameter;
				rectangle.height = (unsigned short) neuronDiameter;
				XUnionRectWithRegion(&rectangle, region, region);
				nptr->region = region;

				/* set the original neuron position to the current position */

				nptr->xOriginal = (int) rint((double) nptr->x / scalingFactor);
				nptr->yOriginal = (int) rint((double) nptr->y / scalingFactor);

				/* redraw the neuron in the color which represents its state */

				if (nptr->selected)
					change_neuron_color(nptr, resData.neuronSelectionColor, resData.textColor);
				else if (nptr->isInputNeuron && nptr->isOutputNeuron)
					change_neuron_color(nptr, resData.inOutNeuronColor, resData.textColor);
				else if (nptr->isInputNeuron)
					change_neuron_color(nptr, resData.inputNeuronColor, resData.textColor);
				else if (nptr->isOutputNeuron)
					change_neuron_color(nptr, resData.outputNeuronColor, resData.textColor);
				else change_neuron_color(nptr, resData.neuronColor, resData.textColor);

				/* draw all graphical connections from the neuron */

				for (connection = nptr->connectionList; connection != NULL; connection = connection->
				    nextConnection)
				{
					connection->targetRegion = connect_neurons_graphical(connection, TRUE);

					for (axoSender = connection->axoSenderList; axoSender != NULL;
					     axoSender = axoSender->nextSender)
					{
						axoSender->connection->targetRegion =
							connect_neurons_graphical(axoSender->connection, TRUE);
					}

				} /* end of for loop through all graphical outgoing connections */

				/* draw all graphical connections to the neuron */

				for (pool = nptr->poolList; pool != NULL; pool = pool->nextPool)
				{
					for (sender = pool->senderList; sender != NULL; sender = sender->nextSender)
					{
						connection = sender->connection;	 /* set connection */

						connection->targetRegion = connect_neurons_graphical(connection, TRUE);

						for (axoSender = connection->axoSenderList; axoSender != NULL;
						     axoSender = axoSender->nextSender)
						{
							axoSender->connection->targetRegion =
								connect_neurons_graphical(axoSender->connection, TRUE);
						}
					} /* end of for loop through senders */

				} /* end of for loop through all graphical incoming connections */

				/* release mouse pointer and redraw all objects in drawing area */

				XUngrabPointer(disp, CurrentTime);
				show_all_objects_in_drawing_area(TRUE);
				moving = TRUE;
				break; /* leave for loop */

			} /* end of if cursor lies in current neuron region */

		} /* end of moving for loop */

		/* when no neuron was moved cause mouse pointer was outside of a neuron select region for zooming */

		if (! moving)
		{
			/* grab the mouse pointer */

			XGrabPointer(disp, window, FALSE, ButtonReleaseMask, GrabModeSync, GrabModeSync, window,
				XCreateFontCursor(disp, XC_crosshair), CurrentTime);

			/* allows pointer motion until next ButtonRelease event */

			XAllowEvents(disp, SyncPointer, CurrentTime);

			/* query mouse pointer position */

			XQueryPointer(disp, window, &root, &child, &rootX, &rootY, &startX, &startY, &state);
			firstTime = TRUE;

			/* set graphical context for rubber band and background */

			gcv.foreground = rubberBandColor;
			gcv.line_style = LineOnOffDash;
			mask = GCForeground | GCLineStyle;
			rubberBandGC = XCreateGC(disp, window, mask, &gcv);
			gcv.foreground = backgroundColor;
			backgroundGC = XCreateGC(disp, window, mask, &gcv);

			/* follow the mouse pointer and draw the rubber band until mouse button is released */

			do
			{

				/* query mouse pointer position */

				XQueryPointer(disp, window, &root, &child, &rootX, &rootY, &winX, &winY, &state);

				/* check if mouse pointer has been moved */

				if (winX != startX && winY != startY)
				{
					/* clear old rubber band */

					if (! firstTime)
						XDrawRectangle(disp, window, backgroundGC, old_x, old_y, old_width,
						    old_height);

					/* check bounderies */

					if (winX < startX)
						x = winX;
					else	x = startX;
					if (winY < startY)
						y = winY;
					else	y = startY;

					/* calculate size of rectangle which should be zoomed */

					width = (unsigned int) abs(winX - startX);
					height = (unsigned int) abs(winY - startY);

					/* draw the rubber band */

					XDrawRectangle(disp, window, rubberBandGC, x, y, width, height);
					old_x = x;
					old_y = y;
					old_width = width;
					old_height = height;
					firstTime = FALSE;
				}
			} while (! XCheckTypedEvent(disp, ButtonRelease, &report));

			/* release the mouse pointer */

			XUngrabPointer(disp, CurrentTime);

			/* clear the rubber band and free the graphical contexts */

			XDrawRectangle(disp, window, backgroundGC, x, y, width, height);
			XFreeGC(disp, backgroundGC);
			XFreeGC(disp, rubberBandGC);

			/* zoom the content of the rectangle if the size of it is greater 1 and
			   the mouse pointer has been moved */

			if (width > 1 && height > 1 && !((startX == winX) && (startY == winY)))
			{
				/* get window attributes of the drawing area for scaling purposes */

				XGetWindowAttributes(disp, XtWindow(XtParent(drawingArea)), &attr);

				/* calculate the scaling factors for width and height */

				sFWidth = (double) ((double) attr.width / width);
				sFHeight = (double) ((double) attr.height / height);

				/* set the local scaling factor to the smalest possible scaling factor */

				if (sFWidth < sFHeight)
					sF = sFWidth;
				else sF = sFHeight;

				/* get scroll bars */

				n = 0;
				XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
				XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
				XtGetValues(mainWindow, args, n);

				/* get maximum values of scroll bars */

				n = 0;
				XtSetArg(args[n], XmNmaximum, &maximumH); n++;
				XtGetValues(hSB, args, n);
				n = 0;
				XtSetArg(args[n], XmNmaximum, &maximumV); n++;
				XtGetValues(vSB, args, n);

				/* calculate original (unscaled) position and rectangle size */

				xnew = (int) rint((double) x / scalingFactor);
				ynew = (int) rint((double) y / scalingFactor);
				widthnew = (int) rint((double) width / scalingFactor);
				heightnew = (int) rint((double) height / scalingFactor);

				/* if scaling factor was less than 1 its possible that the resulting area is to big 
				   in this case zooming is not possible, so redraw all objects and stay window as is */

				if (((xnew + widthnew) > maximumH) || ((ynew + heightnew) > maximumV))
					{
						show_all_objects_in_drawing_area(TRUE);
						break;	/* leave case statement */
					}

				/* calculate new local scaled position and size from current position and size */

				x = (int) (rint(x * sF));
				y = (int) (rint(y * sF));
				width = (int) rint((double) width * sF);
				height = (int) rint((double) height * sF);

				/* if local scaling factor was greater 1 its possible that the resulting area is to big
				   in this case zooming is not possible, so redraw all objects and stay window as is */

				if (((x + width) > maximumH) || ((y + height) > maximumV))
					{
						show_all_objects_in_drawing_area(TRUE);
						break;	/* leave case statement */
					}

				/* calculate new global scaling factor */

				scalingFactor *= sF;

				/* get horizontal slider values of drawing area */

				n = 0;
				XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
				XtSetArg(args[n], XmNincrement, &increment); n++;
				XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
				XtGetValues(hSB, args, n);

				/* set new position of horizontal slider */

				XmScrollBarSetValues(hSB, x, sliderSize, increment, pageIncrement, TRUE);

				/* get vertical slider values of drawing area */

				n = 0;
				XtSetArg(args[n], XmNsliderSize, &sliderSize); n++;
				XtSetArg(args[n], XmNincrement, &increment); n++;
				XtSetArg(args[n], XmNpageIncrement, &pageIncrement); n++;
				XtGetValues(vSB, args, n);

				/* set new position of horizontal slider */

				XmScrollBarSetValues(vSB, y, sliderSize, increment, pageIncrement, TRUE);

				/* move drawing area to new position */

				XtMoveWidget(drawingArea, - x, - y);

				/* since global scaling has been changed rescale graphical object sizes */

				neuronDiameter = (int) (rint(defaultNeuronDiameter * scalingFactor));
				synapseDiameter = (int) (rint(defaultSynapseDiameter * scalingFactor));
				DendriteWidth = (int) (rint(defaultDendriteWidth * scalingFactor));
				if (networktype == 4)
					DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs *
						scalingFactor);
				else	DendriteLength = (int) (rint(defaultDendriteLength * scalingFactor));
				StemWidth = (int) (rint(defaultStemWidth * scalingFactor));
				StemLength = (int) (rint(defaultStemLength * scalingFactor));
				OwnConnectionRadius = (int) (rint(defaultOwnConnectionRadius * scalingFactor));

				/* rescale all neuron sizes and neuron regions */

				for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
				{
					nptr->x = (int) (rint(nptr->xOriginal * scalingFactor));
					nptr->y = (int) (rint(nptr->yOriginal * scalingFactor));
					XDestroyRegion(nptr->region);
					region = XCreateRegion();
					rectangle.x = (short) nptr->x;
					rectangle.y = (short) nptr->y;
					rectangle.width = (unsigned short) neuronDiameter;
					rectangle.height = (unsigned short) neuronDiameter;
					XUnionRectWithRegion(&rectangle, region, region);
					nptr->region = region;
				}

				/* rescale all synapses sizes and synapses regions */

				for (nptr = neuronListHead; nptr != NULL; nptr = nptr->next)
				{
					for (connection = nptr->connectionList; connection != NULL;
						connection = connection-> nextConnection)
					{
						delete_graphical_connection(connection);
						connection->targetRegion = connect_neurons_graphical(connection, TRUE);

						for (axoSender = connection->axoSenderList; axoSender != NULL;
						     axoSender = axoSender->nextSender)
						{
							delete_graphical_connection(axoSender->connection);
							axoSender->connection->targetRegion =
								connect_neurons_graphical(axoSender->connection, TRUE);
						}
					} /* end of for through connections of current neuron */

				} /* end of for through rescale synapses */

			} /* end of zooming if */

			show_all_objects_in_drawing_area(TRUE);	/* redraw all objects in drawing area */

		} /* end of if not moving */

		break;

	} /* end of switch */

} /* end of process_buttonevent */

/************************************************************************************************************************

 	FUNCTION	: is_connected(nptr1, nptr2)
 
	PURPOSE		: looks if neuron 1 is connected with neuron 2

	RETURNS		: TRUE if neuron 1 is connected with neuron 2 else FALSE

************************************************************************************************************************/

Boolean is_connected(nptr1, nptr2)
Neuron *nptr1, *nptr2;	/* neuron pointer to parent and target neuron */
{
	Connection	*connection;		/* current synapse */
	Boolean		connected = FALSE;	/* help variable */

	for (connection = nptr1->connectionList; connection != NULL; connection = connection->nextConnection)
	{
		if (connection->targetNeuron == nptr2) /* does connection from nptr1 to nptr2 exist */
		{
			connected = TRUE;
			break; /* end of for loop */
		}
	}
	return(connected);
} /* end of is_connected */

/************************************************************************************************************************

 	FUNCTION	: connection_exists(nptr, synapse)
 
	PURPOSE		: looks if given connection already exists

	RETURNS		: TRUE if synapse exists else FALSE

************************************************************************************************************************/

Boolean connection_exists(nptr, synapse)
Neuron *nptr;		/* neuron pointer of parent neuron */
Connection *synapse;	/* synapse pointer */
{
	Connection	*connection;		/* current synapse */
	Boolean		connected = FALSE;	/* help variable */

	for (connection = nptr->connectionList; connection != NULL; connection = connection->nextConnection)
	{
		if (connection == synapse)
		{
			connected = TRUE;
			break;
		}
	}
	return(connected);

} /* end of connection_exists */

/************************************************************************************************************************

 	FUNCTION	: delete_graphical_connection(connection)
 
	PURPOSE		: deletes the graphical representation from the connection

	RETURNS		: nothing

************************************************************************************************************************/

void delete_graphical_connection(connection)
Connection	*connection;	/* pointer to synapse */
{
	/* clear connection line */

	XDestroyRegion(draw_synapse(connection, resData.backgroundColor, resData.backgroundColor));

} /* end of delete_graphical_connection */
#endif

/************************************************************************************************************************

 	FUNCTION	: delete_connection(nptr, connection)
 
	PURPOSE		: deletes the entire connection

	RETURNS		: nothing

************************************************************************************************************************/

void delete_connection(nptr, connection)
Neuron *nptr;			/* pointer to parent neuron */
Connection *connection;		/* pointer to synapse */
{
	ChemicalPool		*pool;		/* current pool */
	ChemicalSenderList	*sender;	/* current sender */
	AxoAxonicSenderList	*axoSender;	/* current axo-axonic sender */

	/* if this synapse has a graph window destroy it */

	if (connection->graph)
	{
#ifndef BATCH
		XtUnrealizeWidget(XtParent(connection->graph->formWidget));	/* unrealize graph shell widget */
		XmUpdateDisplay(connection->graph->formWidget);			/* process all pending exposure events */
		XtDestroyWidget(XtParent(connection->graph->formWidget));	/* free graph shell widget */
#endif
		free(connection->graph->pot);
		free(connection->graph->mem);
		free(connection->graph->str);
		free(connection->graph->con);
		free(connection->graph);					/* free graph */
		if (connection->axoSynapseNumber > 0)				/* axo-axonic synapse */
			graphNumberAS--;					/* update number of graphs in the net */
		else	graphNumberS--;
	}

	/* delete the connection graphical */

#ifndef BATCH
	delete_graphical_connection(connection);
#endif

	/* check if connection is axo-axonic or not */

	if (connection->axoSynapseNumber > 0)	/* axo-axonic connection ? */
	{
		/* delete sender in sender list */

		axoSender = connection->axoSenderList;

		free(axoSender->processQueue);	/* deallocate memory for process queue */

		if ((axoSender->prevSender == NULL) && (axoSender->nextSender == NULL))
		{
			connection->targetSynapse->axoSenderList = NULL;
		}
		else if (axoSender->prevSender == NULL)
		{
			connection->targetSynapse->axoSenderList = axoSender->nextSender;
			axoSender->nextSender->prevSender = NULL;
		}
		else if (axoSender->nextSender == NULL)
		{
			axoSender->prevSender->nextSender = NULL;
		}
		else
		{
			axoSender->prevSender->nextSender = axoSender->nextSender;
			axoSender->nextSender->prevSender = axoSender->prevSender;
		}

		axoSynapsesNumber--;	/* decrement number of axo-axonic synapses in the net */

	}
	else	/* not an axo-axonic connection */
	{
		/* delete sender in sender list */

		sender = connection->poolSenderEntry;
		pool = connection->connectionPool;

		pool->totalConductance -= connection->initialConductance;

		if ((sender->prevSender == NULL) && (sender->nextSender == NULL))
		{
			pool->senderList = NULL;
		}
		else if (sender->prevSender == NULL)
		{
			pool->senderList = sender->nextSender;
			sender->nextSender->prevSender = NULL;
		}
		else if (sender->nextSender == NULL)
		{
			sender->prevSender->nextSender = NULL;
		}
		else
		{
			sender->prevSender->nextSender = sender->nextSender;
			sender->nextSender->prevSender = sender->prevSender;
		}

		/* if sender list in pool is empty delete pool */

		if (pool->senderList == NULL)
		{
			free(pool->processQueue);	/* deallocate memory for process queue */

			if ((pool->prevPool == NULL) && (pool->nextPool == NULL))
			{
				connection->targetNeuron->poolList = NULL;
			}
			else if ((pool->prevPool == NULL) && (pool->nextPool != NULL))
			{
				connection->targetNeuron->poolList = pool->nextPool;
				pool->nextPool->prevPool = NULL;
			}
			else if ((pool->prevPool != NULL) && (pool->nextPool != NULL))
			{
				pool->nextPool->prevPool = pool->prevPool;
				pool->prevPool->nextPool = pool->nextPool;
			}
			else
			{
				pool->prevPool->nextPool = NULL;
			}
		}

		synapseNumber--;	/* decrement the number of synapses in the net */

		/* now check if the synapse has it's own axo-axonic sender list, if the list is not empty
		   delete all connections to this synapse */
		
		for (axoSender = connection->axoSenderList; axoSender != NULL; axoSender = axoSender->nextSender)
			delete_connection(axoSender->connection->parentNeuron, axoSender->connection);
	}

	/* delete connection */

	if ((connection->prevConnection == NULL) && (connection->nextConnection == NULL))
	{
		nptr->connectionList = NULL;
	}
	else if (connection->prevConnection == NULL)
	{
		connection->nextConnection->prevConnection = NULL;
		nptr->connectionList = connection->nextConnection;
	}
	else if (connection->nextConnection == NULL)
	{
		connection->prevConnection->nextConnection = NULL;
	}
	else
	{
		connection->nextConnection->prevConnection = connection->prevConnection;
		connection->prevConnection->nextConnection = connection->nextConnection;
	}

	free(connection);	/* deallocate memory for connection */

} /* end of delete_connection */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: change_synaptic_values(connection, type, conductance, deliveryTime, trt, trs, location)
 
	PURPOSE		: changes the synaptic values

	RETURNS		: nothing region of synapse

************************************************************************************************************************/

void change_synaptic_values(connection, type, conductance, deliveryTime, trt, trs, location)
Connection *connection;	/* synapse to be changed */
int	type;		/* synaptic type */
double	conductance;	/* synaptic conductance */
double	deliveryTime;	/* synaptic transmission time */
double	trt;		/* transmitter release threshold */
double	trs;		/* transmitter satiate */
int	location;	/* location of synapse at target neuron */
{
	ChemicalSenderList *sender;	/* entry in pool sender list of target neuron */
	ChemicalPool *pool;		/* chemical pool of target neuron */
	AxoAxonicSenderList *axoSender;	/* axo-axonic senders */
	int queuePlaces;		/* length of process queue */

	/* change total Conductance of pool */

	if (connection->axoSynapseNumber == 0)	/* not axo-axonic ? */
	{
		connection->connectionPool->totalConductance -= connection->initialConductance;
		connection->connectionPool->totalConductance += conductance;
	}
	connection->initialConductance = conductance;
	connection->initialInhibitionFactor = conductance;
	connection->transmitterReleaseThreshold = trt;
	connection->transmitterSatiate = trs;

	/* check if transmission time has been changed */

	if (connection->deliveryTime != deliveryTime)
	{
		connection->deliveryTime = deliveryTime;

		if (connection->axoSynapseNumber > 0)	/* axo-axonic ? */
		{
			queuePlaces = (int) rint(deliveryTime / simulation.step);

			/* reallocate memory for process queue depending on queuePlaces */

			connection->axoSenderList->processQueue = (ProcessQueue *)
				realloc(connection->axoSenderList->processQueue,
					(size_t) (queuePlaces * sizeof(ProcessQueue)));

			if (! connection->axoSenderList->processQueue)
			{
				/* allocation fails so print warning box */

				error(104);
				return;
			}
		}
		else	/* chemical synapse */
		{
			pool = connection->connectionPool;

			if (deliveryTime > pool->longestDeliveryTime)
				pool->longestDeliveryTime = deliveryTime;

			if (synapseTypes[pool->typeOfConnection].ionType == 4)	/* electrical connection ? */
				queuePlaces = 2;
			else queuePlaces = (int) rint(pool->longestDeliveryTime / simulation.step);

			/* reallocate memory for process queue depending on queuePlaces */

			pool->processQueue = (ProcessQueue *) realloc(pool->processQueue,
				(size_t) (queuePlaces * sizeof(ProcessQueue)));

			if (! pool->processQueue)
			{
				/* allocation fails so print warning box */

				error(104);
				return;
			}
		}
	}

	/* check if location has been changed */

	if ((connection->targetArea != location) || (connection->synapticType != type))
	{

		/* set synaptic type */

		connection->synapticType = type;

		/* delete sender from sender list */

		sender = connection->poolSenderEntry;
		pool = connection->connectionPool;

		if ((sender->prevSender == NULL) && (sender->nextSender == NULL))
		{
			pool->senderList = NULL;
		}
		else if (sender->prevSender == NULL)
		{
			pool->senderList = sender->nextSender;
			sender->nextSender->prevSender = NULL;
		}
		else if (sender->nextSender == NULL)
		{
			sender->prevSender->nextSender = NULL;
		}
		else
		{
			sender->prevSender->nextSender = sender->nextSender;
			sender->nextSender->prevSender = sender->prevSender;
		}

		/* if sender list in pool is empty delete pool */

		if (pool->senderList == NULL)
		{
			free(pool->processQueue);	/* deallocate memory for process queue */

			if ((pool->prevPool == NULL) && (pool->nextPool == NULL))
			{
				connection->targetNeuron->poolList = NULL;
			}
			else if ((pool->prevPool == NULL) && (pool->nextPool != NULL))
			{
				connection->targetNeuron->poolList = pool->nextPool;
				pool->nextPool->prevPool = NULL;
			}
			else if ((pool->prevPool != NULL) && (pool->nextPool != NULL))
			{
				pool->nextPool->prevPool = pool->prevPool;
				pool->prevPool->nextPool = pool->nextPool;
			}
			else
			{
				pool->prevPool->nextPool = NULL;
			}
		}

		/* insert connection into pool of target neuron if possible or create new pool */

		for (pool = connection->targetNeuron->poolList; pool != NULL; pool = pool->nextPool)
		{
			/* check if needed pool exists */

			if ((pool->typeOfConnection == type) && (pool->locationOfPool == location))
			{
				/* pool exists, so insert params in this pool */

				if (deliveryTime > pool->longestDeliveryTime)
					pool->longestDeliveryTime = deliveryTime;

				if (synapseTypes[pool->typeOfConnection].ionType == 4)	/* electrical connection ? */
					queuePlaces = 2;
				else queuePlaces = (int) rint(pool->longestDeliveryTime / simulation.step);

				/* reallocate memory for process queue depending on queuePlaces */

				pool->processQueue = (ProcessQueue *) realloc(pool->processQueue,
					(size_t) (queuePlaces * sizeof(ProcessQueue)));

				if (! pool->processQueue)
				{
					/* allocation fails so print warning box */

					error(104);
					return;
				}

				/* initialize sender */

				sender->prevSender = NULL;
				sender->nextSender = pool->senderList;
				if (pool->senderList != NULL)
					pool->senderList->prevSender = sender;
				pool->senderList = sender;

				/* set connection pool and sender */

				connection->connectionPool = pool;
				connection->poolSenderEntry = sender;

				break; /* end of for loop */

			} /* end of if pool exists */

		} /* end of loop through pools */

		 /* create new pool if no pool was found and insert connection */

		if (pool == NULL)
		{
			/* create new pool */

			pool = (ChemicalPool *) malloc((size_t) sizeof(ChemicalPool));
			if (! pool)
			{
				/* allocation fails so print warning box */

				error(110);
				return;
			}

			/* initialize pool */

			pool->prevPool = NULL;
			pool->nextPool = connection->targetNeuron->poolList;
			if (connection->targetNeuron->poolList != NULL)
				connection->targetNeuron->poolList->prevPool = pool;
			connection->targetNeuron->poolList = pool;
			pool->currentQueuePlace = 0;
			pool->typeOfConnection = type;
			pool->longestDeliveryTime = deliveryTime;

			if (synapseTypes[pool->typeOfConnection].ionType == 4)	/* is it a electrical synapse ? */
				queuePlaces = 2;
			else queuePlaces = (int) rint(pool->longestDeliveryTime / simulation.step);

			/* allocate memory for process queue */

			pool->processQueue = (ProcessQueue *) malloc((size_t) (queuePlaces * sizeof(ProcessQueue)));
			if (! pool->processQueue)
			{
				/* allocation fails so print warning box */

				error(104);
				return;
			}
			pool->locationOfPool = location;

			/* initialize sender */

			sender->prevSender = NULL;
			sender->nextSender = NULL;

			/* set sender in pool sender list */

			pool->senderList = sender;

			/* set pool and sender in connection */

			connection->connectionPool = pool;
			connection->poolSenderEntry = sender;
		}

		/* set new location of synapse and redraw all connections involved in this action */

		connection->targetArea = location;

		delete_graphical_connection(connection);
		connection->targetRegion = connect_neurons_graphical(connection, TRUE);

		for (axoSender = connection->axoSenderList; axoSender != NULL; axoSender = axoSender->nextSender)
		{
			delete_graphical_connection(axoSender->connection);
			axoSender->connection->targetRegion = connect_neurons_graphical(axoSender->connection, TRUE);
		}
	}
} /* end of change_synaptic_values */

/************************************************************************************************************************

 	FUNCTION	: connect_neurons_graphical(connection, computeSynapse)
 
	PURPOSE		: draws a graphical representation for the connection

	RETURNS		: graphical region of synapse

************************************************************************************************************************/

Region connect_neurons_graphical(connection, computeSynapse)
Connection *connection;	/* synapse */
Boolean	computeSynapse;	/* flag, if synapse position must be computed or not */
{
	int		type = connection->synapticType;	/* synaptic type */
	Region		region;					/* graphical region of synapse */

	/*** connect two neurons graphical ***/

	/* check if synapse position must be computed */

	if (computeSynapse)
		calculate_graphical_synapse(connection);

	/* draw connection line */

	if (connection->selected)
		region = draw_synapse(connection, resData.neuronSelectionColor, resData.connectionLineColor);
	else
	{
		/* draw synapse in the color depending on synaptic ion type */

		if (connection->axoSynapseNumber > 0) /* axo-axonic */
			region = draw_synapse(connection, resData.axoSynapseColor, resData.connectionLineColor);
		else
			switch (synapseTypes[type].ionType)
			{
			case 1:
				region = draw_synapse(connection, resData.excitationSynapseColor,
					resData.connectionLineColor);
				break;

			case 2:
				region = draw_synapse(connection, resData.inhibitionSynapseColor,
					resData.connectionLineColor);
				break;

			case 3:
				region = draw_synapse(connection, resData.otherSynapseColor,
					resData.connectionLineColor);
				break;

			case 4:
				region = draw_synapse(connection, resData.electricalSynapseColor,
					resData.connectionLineColor);
				break;
			}
	}

	return(region); /* return graphical region of synapse */

} /* end of connect_neurons_graphical */
#endif

/************************************************************************************************************************

 	FUNCTION	: connect_neurons(nptr1, nptr2, type, conductance, transmissionTime, trt, trs, location,
					  targetSynapse)
 
	PURPOSE		: connects two neurons

	RETURNS		: nothing

************************************************************************************************************************/

void connect_neurons(nptr1, nptr2, type, conductance, transmissionTime, trt, trs, location, targetSynapse)
Neuron	*nptr1, *nptr2;		/* neuron pointer to parent and target neuron */
int	type;			/* synaptic type */
double	conductance;		/* synaptic conductance */
double	transmissionTime;	/* delivery time */
double	trt;			/* transmitter release threshold */
double	trs;			/* transmitter satiate */
int	location;		/* location of synapse (compartment number) or -1 for axo-axonic connection */
Connection *targetSynapse;	/* pointer to target synapse in case of axo-axonic connection, else NULL */
{
	Connection		*connection;	/* current synapse */
	ChemicalSenderList	*sender;	/* current sender */
	ChemicalPool		*pool;		/* current pool */
	AxoAxonicSenderList	*axoSender;	/* current axo-axonic sender */
	int			queuePlaces;	/* number of entries in process queue */

	/* allocate memory for a new connection (synapse) */

	connection = (Connection *) malloc((size_t) sizeof(Connection));
	if (! connection)
	{
		/* allocation fails so print warning box */

		error(108);
		return;
	}

	/* initialize connection */

	connection->nextConnection = nptr1->connectionList;
	connection->prevConnection = NULL;
	if (nptr1->connectionList)
		nptr1->connectionList->prevConnection = connection;
	nptr1->connectionList = connection;
	connection->connectionPool = NULL;
	connection->poolSenderEntry = NULL;
	connection->axoSenderList = NULL;
	connection->graph = NULL;
	connection->synapseID = ++synapseNumber;
	connection->numberOfAxoSynapses = 0;
	connection->initialConductance = conductance;
	connection->initialInhibitionFactor = conductance;
	connection->deliveryTime = transmissionTime;
	connection->transmitterReleaseThreshold = trt;
	connection->transmitterSatiate = trs;
	connection->synapticType = type;
	connection->targetArea = location;
	connection->parentNeuron = nptr1;
	connection->targetNeuron = nptr2;
	connection->targetCode = nptr2->neuronId;
	connection->targetSynapse = targetSynapse;

	if (targetSynapse)	/* axo-axonic connection */
	{
		axoSynapsesNumber++;
		targetSynapse->numberOfAxoSynapses++;
		connection->axoSynapseNumber = targetSynapse->numberOfAxoSynapses;
	}
	else	connection->axoSynapseNumber = 0;

#ifndef BATCH
	connection->targetRegion = connect_neurons_graphical(connection, TRUE);
#endif
	connection->selected = FALSE;

	connection->hebbCnt = connection->stmCnt = connection->condCnt = connection->trainingFlag = 0;
	connection->trainingFlagSTP = connection->trainingFlagHebb = connection->trainingFlagCond =
	connection->shortTermLearningCondition = connection->conditioningLearningCondition =
	connection->hebbLearningCondition = connection->competitiveLearningCondition =
	connection->competitiveDone = FALSE;
	connection->potential = connection->currentConductance = connection->currentStrength =
	connection->currentInhibitionFactor = connection->synapticMemory = connection->synapticMemorySTP =
	connection->synapticMemoryHebb = connection->synapticMemoryCond = connection->hebbIntervallCounter =
	connection->hebbTimer = connection->LEXP_const = connection->LFCONST_const = connection->TSmTRT_const = 0.0;
	connection->conductance = connection->initialConductance;
	connection->maxConductance = connection->initialConductance * 10.0;
	connection->maxInhibitionFactor = connection->initialInhibitionFactor * 10.0;

	/* set default learn parameter */

	connection->slope = SLOPE_DEF;
	connection->shortTermMemory = STM_DEF;
	connection->longTermThreshold = LTT_DEF;
	connection->longTermMinimum = (double) (LTM_DEF / P_SCALE);
	connection->hebbIntervall = (double) (HI_DEF / T_SCALE);
	connection->hebbLearnFactor = HLF_DEF;
	connection->competitiveLearnFactor = COMPLF_DEF;
	connection->shortTermPlasticityLearnFactor = KLF_DEF;
	connection->conditioningLearnFactor = CLF_DEF;

	/* check if connection is axo-axonic */

	if (targetSynapse)
	{
		/* create new sender */

		axoSender = (AxoAxonicSenderList *) malloc((size_t) sizeof(AxoAxonicSenderList));
		if (! axoSender)
		{
			/* allocation fails so print warning box */

			error(109);
			return;
		}

		/* initialize sender */

		axoSender->prevSender = NULL;
		axoSender->nextSender = targetSynapse->axoSenderList;
		if (targetSynapse->axoSenderList)
			targetSynapse->axoSenderList->prevSender = axoSender;
		
		axoSender->connection = connection;
		
		axoSender->currentQueuePlace = 0;
		queuePlaces = (int) rint(transmissionTime / STEP);
		connection->TTdSS_const = queuePlaces;

		/* allocate memory for process queue depending on queuePlaces */

		axoSender->processQueue = (ProcessQueue *) malloc((size_t) (queuePlaces * sizeof(ProcessQueue)));
		if (! axoSender->processQueue)
		{
			/* allocation fails so print warning box */

			error(104);
			return;
		}

		targetSynapse->axoSenderList = axoSender;

		/* set connection axoSenderList */

		connection->axoSenderList = axoSender;

		return;		/* exit connect_neurons */

	} /* end of if axo-axonic connection */

	/* insert connection into pool of target neuron if possible or create new pool */

	for (pool = nptr2->poolList; pool != NULL; pool = pool->nextPool)
	{
		/* check if needed pool exists */

		if ((pool->typeOfConnection == type) && (pool->locationOfPool == location))
		{
			/* pool exists, so insert params in this pool */

			pool->totalConductance += conductance;

			if (transmissionTime > pool->longestDeliveryTime)
				pool->longestDeliveryTime = transmissionTime;

			if (synapseTypes[pool->typeOfConnection].ionType == 4)	/* is it a electrical connection ? */
				queuePlaces = 2;
			else queuePlaces = (int) rint(pool->longestDeliveryTime / STEP);
			connection->TTdSS_const = queuePlaces;

			/* reallocate memory for process queue depending on queuePlaces */

			pool->processQueue = (ProcessQueue *) realloc(pool->processQueue,
				(size_t) (queuePlaces * sizeof(ProcessQueue)));

			if (! pool->processQueue)
			{
				/* allocation fails so print warning box */

				error(104);
				return;
			}

			/* create new sender */

			sender = (ChemicalSenderList *) malloc((size_t) sizeof(ChemicalSenderList));
			if (! sender)
			{
				/* allocation fails so print warning box */

				error(109);
				return;
			}

			/* initialize sender */

			sender->prevSender = NULL;
			sender->nextSender = pool->senderList;
			if (pool->senderList)
			        pool->senderList->prevSender = sender;
			sender->connection = connection;
			pool->senderList = sender;

			/* set connection pool and sender */

			connection->connectionPool = pool;
			connection->poolSenderEntry = sender;

			break; /* end of for loop */

		} /* end of if pool exists */

	} /* end of loop through pools */

	 /* create new pool if no pool was found and insert connection */

	if (! pool)
	{
		/* create new pool */

		pool = (ChemicalPool *) malloc((size_t) sizeof(ChemicalPool));
		if (! pool)
		{
			/* allocation fails so print warning box */

			error(110);
			return;
		}

		/* initialize pool */

		pool->prevPool = NULL;
		pool->nextPool = nptr2->poolList;
		if (nptr2->poolList)
			nptr2->poolList->prevPool = pool;
		nptr2->poolList = pool;
		pool->currentQueuePlace = 0;
		pool->typeOfConnection = type;
		pool->longestDeliveryTime = transmissionTime;
		pool->totalConductance = conductance;
		pool->activeSynapses = 0;

		if (synapseTypes[pool->typeOfConnection].ionType == 4)	/* is it a electrical synapse ? */
			queuePlaces = 2;
		else queuePlaces = (int) rint(pool->longestDeliveryTime / STEP);
		connection->TTdSS_const = queuePlaces;

		/* allocate memory for process queue */

		pool->processQueue = (ProcessQueue *) malloc((size_t) (queuePlaces * sizeof(ProcessQueue)));
		if (! pool->processQueue)
		{
			/* allocation fails so print warning box */

			error(104);
			return;
		}
		pool->locationOfPool = location;

		/* create new sender */

		sender = (ChemicalSenderList *) malloc((size_t) sizeof(ChemicalSenderList));
		if (! sender)
		{
			/* allocation fails so print warning box */

			error(109);
			return;
		}

		/* initialize sender */

		sender->prevSender = NULL;
		sender->nextSender = NULL;
		sender->connection = connection;

		/* set sender in pool sender list */

		pool->senderList = sender;

		/* set pool and sender in connection */

		connection->connectionPool = pool;
		connection->poolSenderEntry = sender;
	}
} /* end of connect_neurons */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: draw_synapse(connection, synapseColor, lineColor)
 
	PURPOSE		: draws the synapse

	RETURNS		: graphical region of synapse

************************************************************************************************************************/

Region draw_synapse(connection, synapseColor, lineColor)
Connection *connection;	/* synapse */
char *synapseColor;	/* synapse color */
char *lineColor;	/* line color */
{
	Neuron		*nptr1 = connection->parentNeuron;	/* pointer to parent neuron */
	int 		ownRadius;				/* connection radius for self connection */
	int 		connectionX, connectionY;		/* current synapse position */
	int		senderX, senderY;			/* current position of sending neuron */
	int		targetX, targetY;			/* current position of target neuron */
	Window		window = XtWindow(drawingArea);		/* window of drawing area */
	GC		lineGC, kreisGC;			/* graphical contexts for line and circle */
	XRectangle 	rectangle;				/* used to create region for neuron */
	Region 		region;					/* graphic region of neuron */
	XGCValues	gcv;					/* values used in graphical contexts */
	unsigned long	mask;					/* masks used in graphical contexts */

	/* calculate start and end point of connection line and synapse position */

	if (networktype == 2)	/* 2-Point model */
	{
		connectionX = connection->connectionX_2;
		connectionY = connection->connectionY_2;
		senderX = connection->senderX_2;
		senderY = connection->senderY_2;
		targetX = connection->targetX_2;
		targetY = connection->targetY_2;
		ownRadius = 40;
	}
	else if (networktype == 3)	/* 4-Point model */
	{
		connectionX = connection->connectionX_4;
		connectionY = connection->connectionY_4;
		senderX = connection->senderX_4;
		senderY = connection->senderY_4;
		targetX = connection->targetX_4;
		targetY = connection->targetY_4;
		ownRadius = 50;
	}
	else if (networktype == 4)	/* n-Point model */
	{
		connectionX = connection->connectionX_N;
		connectionY = connection->connectionY_N;
		senderX = connection->senderX_N;
		senderY = connection->senderY_N;
		targetX = connection->targetX_N;
		targetY = connection->targetY_N;
		ownRadius = curNumOfDendSegs * 10 + 20;
	}
	else	/* 1-Point model */
	{
		connectionX = connection->connectionX_1;
		connectionY = connection->connectionY_1;
		senderX = connection->senderX_1;
		senderY = connection->senderY_1;
		targetX = connection->targetX_1;
		targetY = connection->targetY_1;
		ownRadius = 30;
	}

	/* create graphical region for synapse and draw synapse and connection line */

	if (nptr1->neuronId != connection->targetCode)	/* connection to a different neuron */
	{
		Vertex	line[2];

		/* draw connection line */

		if (strcmp(lineColor, resData.backgroundColor) == 0)
			gcv.foreground = backgroundColor;
		else if (strcmp(lineColor, resData.connectionLineColor) == 0)
			gcv.foreground = connectionLineColor;

		mask = GCForeground;
		lineGC = XCreateGC(disp, window, mask, &gcv);

		/* use old draw instruction XDraw insted of XDrawLine, because the XDrawLine instruction is buggy:
		   it doesn't draws a line if the end point coordinate is greater than the maximal number of pixels
		   of the screen, i.e. 1280. So the old routine is to be used, which works fine */

		line[0].x = senderX;
		line[0].y = senderY;
		line[0].flags = 0;
		line[1].x = targetX;
		line[1].y = targetY;
		line[1].flags = 0;
		XDraw(disp, window, lineGC, line, 2);
/*		XDrawLine(disp, window, lineGC, senderX, senderY, targetX, targetY);*/

		XFreeGC(disp, lineGC);

		/* correct synapse position by synapse radius to get synapse top-left */

		connectionX -= (int) rint(synapseDiameter * 0.5);
		connectionY -= (int) rint(synapseDiameter * 0.5);

		/* draw synapse */

		if (strcmp(synapseColor, resData.backgroundColor) == 0)
			gcv.foreground = backgroundColor;
		else if (strcmp(synapseColor, resData.neuronSelectionColor) == 0)
			gcv.foreground = neuronSelectionColor;
		else if (strcmp(synapseColor, resData.excitationSynapseColor) == 0)
			gcv.foreground = excitationSynapseColor;
		else if (strcmp(synapseColor, resData.inhibitionSynapseColor) == 0)
			gcv.foreground = inhibitionSynapseColor;
		else if (strcmp(synapseColor, resData.electricalSynapseColor) == 0)
			gcv.foreground = electricalSynapseColor;
		else if (strcmp(synapseColor, resData.axoSynapseColor) == 0)
			gcv.foreground = axoSynapseColor;
		else if (strcmp(synapseColor, resData.otherSynapseColor) == 0)
			gcv.foreground = otherSynapseColor;

		kreisGC = XCreateGC(disp, window, mask, &gcv);
		XFillArc(disp, window, kreisGC, connectionX, connectionY, synapseDiameter, synapseDiameter, 0, 360 * 64);
		XFreeGC(disp, kreisGC);

		/* create synaptic region */

		region = XCreateRegion();
		rectangle.x = (short) connectionX;
		rectangle.y = (short) connectionY;
		rectangle.width = (unsigned short) synapseDiameter;
		rectangle.height = (unsigned short) synapseDiameter;
		XUnionRectWithRegion(&rectangle, region, region);
	}
	else	/* target neuron equals parent neuron -> self connection */
	{
		if (connection->targetArea == 0 || ownRadius == 30)	/* location = soma or 1-Point model */
		{
			/* draw connection line as an arc */

			ownRadius = (int) rint(scalingFactor * (double) ownRadius);
			if (strcmp(lineColor, resData.backgroundColor) == 0)
				gcv.foreground = backgroundColor;
			else if (strcmp(lineColor, resData.connectionLineColor) == 0)
				gcv.foreground = connectionLineColor;
			mask = GCForeground;
			kreisGC = XCreateGC(disp, window, mask, &gcv);
			XDrawArc(disp, window, kreisGC, senderX - (int) rint(15.0 * scalingFactor), senderY,
			    (unsigned int) rint(30.0 * scalingFactor), (unsigned int) ownRadius, 90 * 64, 360 * 64);
			XFreeGC(disp, kreisGC);
		}
		else	/* location != soma or 2-Point, 4-Point or n-Point model */
		{
			/* draw connection line as an arc */

			if (strcmp(lineColor, resData.backgroundColor) == 0)
				gcv.foreground = backgroundColor;
			else if (strcmp(lineColor, resData.connectionLineColor) == 0)
				gcv.foreground = connectionLineColor;
			mask = GCForeground;
			kreisGC = XCreateGC(disp, window, mask, &gcv);
			XDrawArc(disp, window, kreisGC, senderX - (int) rint(15.0 * scalingFactor), senderY,
			    (unsigned int) rint(30.0 * scalingFactor), (unsigned int) abs(targetY - senderY),
			    90 * 64, 180 * 64);

			XFreeGC(disp, kreisGC);
		}

		/* correct synapse position by synapse radius to get synapse top-left */

		connectionX -= (int) rint(synapseDiameter * 0.5);
		connectionY -= (int) rint(synapseDiameter * 0.5);

		/* draw synapse */

		if (strcmp(synapseColor, resData.backgroundColor) == 0)
			gcv.foreground = backgroundColor;
		else if (strcmp(synapseColor, resData.neuronSelectionColor) == 0)
			gcv.foreground = neuronSelectionColor;
		else if (strcmp(synapseColor, resData.excitationSynapseColor) == 0)
			gcv.foreground = excitationSynapseColor;
		else if (strcmp(synapseColor, resData.inhibitionSynapseColor) == 0)
			gcv.foreground = inhibitionSynapseColor;
		else if (strcmp(synapseColor, resData.electricalSynapseColor) == 0)
			gcv.foreground = electricalSynapseColor;
		else if (strcmp(synapseColor, resData.axoSynapseColor) == 0)
			gcv.foreground = axoSynapseColor;
		else if (strcmp(synapseColor, resData.otherSynapseColor) == 0)
			gcv.foreground = otherSynapseColor;

		mask = GCForeground;
		kreisGC = XCreateGC(disp, window, mask, &gcv);
		XFillArc(disp, window, kreisGC, connectionX, connectionY, synapseDiameter, synapseDiameter, 0, 360 * 64);
		XFreeGC(disp, kreisGC);

		/* create synaptic region */

		region = XCreateRegion();
		rectangle.x = (short) connectionX;
		rectangle.y = (short) connectionY;
		rectangle.width = (unsigned short) synapseDiameter;
		rectangle.height = (unsigned short) synapseDiameter;
		XUnionRectWithRegion(&rectangle, region, region);
	}

	return(region);	/* return graphical region of synapse */

} /* end of draw_synapse */

/************************************************************************************************************************

 	FUNCTION	: calculate_graphical_synapse(synapse)
 
	PURPOSE		: calculates the graphical position for the synapse

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_graphical_synapse(synapse)
Connection *synapse;	/* current synapse */
{
	int senderXIn = synapse->parentNeuron->x;			/* x-position of parent neuron */
	int senderYIn = synapse->parentNeuron->y;			/* y-position of parent neuron */
	int targetXIn = synapse->targetNeuron->x;			/* x-position of target neuron */
	int targetYIn = synapse->targetNeuron->y;			/* y-position of target neuron */
	int connectArea = synapse->targetArea;				/* location of connection */
	double xSenderOffset, ySenderOffset;				/* offset to parent neuron edge */
	double xTargetOffset, yTargetOffset;				/* offset to target neuron edge or stem edge */
	double xSynapseOffset, ySynapseOffset;				/* offset to synapse center position */
	double piHalf = asin((double) 1.0);				/* value of pi / 2 */
	double alpha, beta;						/* angles in triangular for calculation */
	double dist;							/* distance between parent and target neuron */
	int xDiff, yDiff;						/* difference between target and parent neuron */
	int senderX, senderY;						/* center of parent neuron */
	int targetX, targetY;						/* center of target neuron */
	int xSynapse[5], ySynapse[5];					/* synapse center position */
	int xSource[5], ySource[5];					/* starting point of connection line */
	int xTarget[5], yTarget[5];					/* ending point of connection line */
	int modelType;							/* network model loop variable */
	int offsetRadius;						/* offset for synapse */
	int NeuronRadius = (int) rint((double) neuronDiameter / 2.0);	/* neuron radius */
	int SynapseRadius = (int) rint((double) synapseDiameter / 2.0);/* synapse radius */

	/* get central coordinates of sender and target neuron */

	senderX = senderXIn + NeuronRadius;
	senderY = senderYIn + NeuronRadius;
	targetX = targetXIn + NeuronRadius;
	targetY = targetYIn + NeuronRadius;

	/*** calculate synapse position ***/

	/* check if connection is axo-axonic */

	if (synapse->axoSynapseNumber != 0)
	{
		for (modelType = 1; modelType <= 4; modelType++)
		{
			offsetRadius = SynapseRadius;

			/* set target coordinates to target synapse */

			switch (modelType)
			{
			case 1:
				targetX = synapse->targetSynapse->connectionX_1;
				targetY = synapse->targetSynapse->connectionY_1;
				break;

			case 2:
				targetX = synapse->targetSynapse->connectionX_2;
				targetY = synapse->targetSynapse->connectionY_2;
				break;

			case 3:
				targetX = synapse->targetSynapse->connectionX_4;
				targetY = synapse->targetSynapse->connectionY_4;
				break;

			case 4:
				targetX = synapse->targetSynapse->connectionX_N;
				targetY = synapse->targetSynapse->connectionY_N;
				break;
			}

			/* calculate start and end point of connection line */

			xDiff = targetX - senderX;
			yDiff = targetY - senderY;

			dist = sqrt(xDiff * xDiff + yDiff * yDiff);

			alpha = asin(abs(yDiff) / dist);
			beta = piHalf - alpha;

			xSenderOffset = cos(alpha) * NeuronRadius;
			ySenderOffset = sin(alpha) * NeuronRadius;

			xTargetOffset = sin(beta) * offsetRadius;
			yTargetOffset = cos(beta) * offsetRadius;

			xSynapseOffset = sin(beta) * (offsetRadius + SynapseRadius);
			ySynapseOffset = cos(beta) * (offsetRadius + SynapseRadius);

			/* set proper signs by determining how the sender lies in relation to the target */

			if (targetX > senderX)
			{
				xTarget[modelType] = targetX - (int) rint(xTargetOffset);
				xSynapse[modelType] = targetX - (int) rint(xSynapseOffset);
				xSource[modelType] = senderX + (int) rint(xSenderOffset);
			}
			else if (targetX < senderX)
			{
				xTarget[modelType] = targetX + (int) rint(xTargetOffset);
				xSynapse[modelType] = targetX + (int) rint(xSynapseOffset);
				xSource[modelType] = senderX - (int) rint(xSenderOffset);
			}
			else /* targetX = senderX */
			{
				xTarget[modelType] = targetX;
				xSynapse[modelType] = targetX;
				xSource[modelType] = senderX;
			}

			if (targetY > senderY)
			{
				yTarget[modelType] = targetY - (int) rint(yTargetOffset);
				ySynapse[modelType] = targetY - (int) rint(ySynapseOffset);
				ySource[modelType] = senderY + (int) rint(ySenderOffset);
			}
			else if (targetY < senderY)
			{
				yTarget[modelType] = targetY + (int) rint(yTargetOffset);
				ySynapse[modelType] = targetY + (int) rint(ySynapseOffset);
				ySource[modelType] = senderY - (int) rint(ySenderOffset);
			}
			else /* targetY = senderY */
			{
				yTarget[modelType] = targetY;
				ySynapse[modelType] = targetY;
				ySource[modelType] = senderY;
			}
		}
		
	} /* end of IF connection is axo-axonic */
	else if (senderX == targetX && senderY == targetY)	/* connection to itself */
	{
		/* set starting x-coordinate of synapse to center of neuron */

		xSource[1] = senderX;
		xSource[2] = senderX;
		xSource[3] = senderX;
		xSource[4] = senderX;

		/* set ending x-coordinate of synapse to center of neuron */

		xTarget[1] = senderX;
		xTarget[2] = senderX;
		xTarget[3] = senderX;
		xTarget[4] = senderX;

		/* set starting y-coordinate of synapse to center of neuron */

		ySource[1] = senderY;
		ySource[2] = senderY;
		ySource[3] = senderY;
		ySource[4] = senderY;

		/* calculate the offset of the synapse */

		alpha = asin((double) NeuronRadius / (2.0 * (double) OwnConnectionRadius));
		alpha += asin((double) SynapseRadius / (2.0 * (double) OwnConnectionRadius));
		alpha *= 2.0;

		beta = piHalf - alpha;

		dist = 2.0 * (double) OwnConnectionRadius * sin(alpha / 2.0);

		xSynapseOffset = cos(beta) * dist;
		ySynapseOffset = sin(beta) * dist;

		/* set synapse position for 1-Point model */

		xSynapse[1] = senderX + (int) xSynapseOffset;
		ySynapse[1] = senderY + (int) ySynapseOffset;

		if (connectArea == 0)	/* synapse goes to soma */
		{
			/* set ending y-coordinate of synapse to center of neuron */

			yTarget[1] = senderY;
			yTarget[2] = senderY;
			yTarget[3] = senderY;
			yTarget[4] = senderY;

			/* set synapse position for 2-Point model */

			xSynapse[2] = senderX + (int) xSynapseOffset;
			ySynapse[2] = senderY + (int) ySynapseOffset;

			/* set synapse position for 4-Point model */

			xSynapse[3] = senderX + (int) xSynapseOffset;
			ySynapse[3] = senderY + (int) ySynapseOffset;

			/* set synapse position for n-Point model */

			xSynapse[4] = senderX + (int) xSynapseOffset;
			ySynapse[4] = senderY + (int) ySynapseOffset;
		}
		else	/* synapse goes to dendrite */
		{
			/* set ending y-coordinate of synapse to center of neuron */

			yTarget[1] = senderY;

			/* set synapse position for 2-Point model and ending y-coordinate of synapse */

			xSynapse[2] = senderX - SynapseRadius;
			ySynapse[2] = senderY + NeuronRadius + (int) rint((double) StemLength / 2.0);
			yTarget[2] = ySynapse[2];

			/* set synapse position for 4-Point model and ending y-coordinate of synapse */

			xSynapse[3] = senderX - SynapseRadius;

			if (connectArea > 3)
				ySynapse[3] = senderY + NeuronRadius + DendriteLength - (int) rint(DendriteLength / 6.0);
			else	ySynapse[3] = senderY + NeuronRadius + (int) rint((connectArea * DendriteLength) / 3.0) -
					(int) rint(DendriteLength / 6.0);
			yTarget[3] = ySynapse[3];

			/* set synapse position for n-Point model and ending y-coordinate of synapse */

			xSynapse[4] = senderX - SynapseRadius;

			if (connectArea > curNumOfDendSegs)
				ySynapse[4] = senderY + NeuronRadius + DendriteLength -
					(int) (DendriteLength / (2 * curNumOfDendSegs));
			else	ySynapse[4] = senderY + NeuronRadius + (int) ((connectArea * DendriteLength)
					/ curNumOfDendSegs) - (int) (DendriteLength / (2 * curNumOfDendSegs));
			yTarget[4] = ySynapse[4];
		}
	}
	else	/* connection to a different neuron */
	{
		/* loop through all model types */

		for (modelType = 1; modelType <= 4; modelType++)
		{
			if (modelType == 1 || connectArea == 0)	/* 1-Point model or synapse connects to soma */
			{
				offsetRadius = NeuronRadius;

				targetX = targetXIn + NeuronRadius;
				targetY = targetYIn + NeuronRadius;
			}
			else if (modelType == 2)	/* 2-Point model */
			{
				offsetRadius = StemWidth;

				targetX = targetXIn + NeuronRadius;
				targetY = targetYIn + neuronDiameter + (int) rint((double) StemLength / 2.0);
			}
			else if (modelType == 3)	/* 4-Point model */
			{
				offsetRadius = DendriteWidth;

				targetX = targetXIn + NeuronRadius;

				if (connectArea > 3)
					targetY = targetYIn + neuronDiameter + DendriteLength - (int) rint(DendriteLength
						/ 6.0);
				else	targetY = targetYIn + neuronDiameter + (int) rint((connectArea * DendriteLength)
						/ 3.0) - (int) rint(DendriteLength / 6.0);
			}
			else if (modelType == 4)	/* n-Point model */
			{
				offsetRadius = DendriteWidth;

				targetX = targetXIn + NeuronRadius;

				if (curNumOfDendSegs == 0)	/* no dendrit */
					targetY = targetYIn + NeuronRadius;
				else if (connectArea > curNumOfDendSegs)
					targetY = targetYIn + neuronDiameter + DendriteLength -
						(int) (DendriteLength / (2 * curNumOfDendSegs));
				else	targetY = targetYIn + neuronDiameter + (int) ((connectArea * DendriteLength)
						/ curNumOfDendSegs) - (int) (DendriteLength / (2 * curNumOfDendSegs));
			}

			/* calculate start and end point of connection line */

			xDiff = targetX - senderX;
			yDiff = targetY - senderY;

			dist = sqrt(xDiff * xDiff + yDiff * yDiff);

			alpha = asin(abs(yDiff) / dist);
			beta = piHalf - alpha;

			xSenderOffset = cos(alpha) * NeuronRadius;
			ySenderOffset = sin(alpha) * NeuronRadius;

			xTargetOffset = sin(beta) * offsetRadius;
			yTargetOffset = cos(beta) * offsetRadius;

			xSynapseOffset = sin(beta) * (offsetRadius + SynapseRadius);
			ySynapseOffset = cos(beta) * offsetRadius;

			/* set proper signs by determining how the sender lies in relation to the target */

			if (targetX > senderX)
			{
				xTarget[modelType] = targetX - (int) rint(xTargetOffset);
				xSynapse[modelType] = targetX - (int) rint(xSynapseOffset);
				xSource[modelType] = senderX + (int) rint(xSenderOffset);
			}
			else if (targetX < senderX)
			{
				xTarget[modelType] = targetX + (int) rint(xTargetOffset);
				xSynapse[modelType] = targetX + (int) rint(xSynapseOffset);
				xSource[modelType] = senderX - (int) rint(xSenderOffset);
			}
			else /* targetX = senderX */
			{
				xTarget[modelType] = targetX;
				xSynapse[modelType] = targetX;
				xSource[modelType] = senderX;
			}

			if (targetY > senderY)
			{
				ySynapse[modelType] = targetY - (int) rint(ySynapseOffset);
				ySource[modelType] = senderY + (int) rint(ySenderOffset);
				yTarget[modelType] = targetY - (int) rint(yTargetOffset);
			}
			else if (targetY < senderY)
			{
				ySynapse[modelType] = targetY + (int) rint(ySynapseOffset);
				ySource[modelType] = senderY - (int) rint(ySenderOffset);
				yTarget[modelType] = targetY + (int) rint(yTargetOffset);
			}
			else /* targetY = senderY */
			{
				ySynapse[modelType] = targetY;
				ySource[modelType] = senderY;
				yTarget[modelType] = targetY;
			}
		}
	}

	/* set the appropriate values in the synapse */

	synapse->senderX_N = xSource[4];
	synapse->senderY_N = ySource[4];
	synapse->senderX_4 = xSource[3];
	synapse->senderY_4 = ySource[3];
	synapse->senderX_2 = xSource[2];
	synapse->senderY_2 = ySource[2];
	synapse->senderX_1 = xSource[1];
	synapse->senderY_1 = ySource[1];

	synapse->targetX_N = xTarget[4];
	synapse->targetY_N = yTarget[4];
	synapse->targetX_4 = xTarget[3];
	synapse->targetY_4 = yTarget[3];
	synapse->targetX_2 = xTarget[2];
	synapse->targetY_2 = yTarget[2];
	synapse->targetX_1 = xTarget[1];
	synapse->targetY_1 = yTarget[1];

	synapse->connectionX_N = xSynapse[4];
	synapse->connectionY_N = ySynapse[4];
	synapse->connectionX_4 = xSynapse[3];
	synapse->connectionY_4 = ySynapse[3];
	synapse->connectionX_2 = xSynapse[2];
	synapse->connectionY_2 = ySynapse[2];
	synapse->connectionX_1 = xSynapse[1];
	synapse->connectionY_1 = ySynapse[1];

} /* end of calculate_graphical_synapse */

/************************************************************************************************************************

 	FUNCTION	: select_all_neurons_CB(w, client_data, call_data)
 
	PURPOSE		: callback function that selects all neurons in the net

	RETURNS		: nothing

************************************************************************************************************************/

void select_all_neurons_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	/* remove all actually selected objects from selection list */

	remove_selected_objects_from_selectionList();

	/* loop through all neurons and add each neuron to the neuron selection list */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		add_neuron_to_selection_list(currentNeuron);
		currentNeuron->selected = TRUE;
		change_neuron_color(currentNeuron, resData.neuronSelectionColor, resData.textColor);
	}

} /* end of select_all_neurons_CB */

/************************************************************************************************************************

 	FUNCTION	: select_all_synapses_CB(w, client_data, call_data)
 
	PURPOSE		: callback function that selects all synapses in the net

	RETURNS		: nothing

************************************************************************************************************************/

void select_all_synapses_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	/* remove all actually selected objects from selection list */

	remove_selected_objects_from_selectionList();

	/* loop through all neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* loop through all synapses and add each synapse to the synapse selection list */

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			currentSynapse = currentSynapse->nextConnection)
		{
			/* in case of complex networkmodel check if synapse is not axo-axonic */

			if ((complexModel && (currentSynapse->axoSynapseNumber == 0)) || (! complexModel))
			{
				add_synapse_to_selection_list(currentSynapse);
				currentSynapse->selected = TRUE;
				change_synapse_color(currentSynapse, resData.neuronSelectionColor);
			}
		}
	}

} /* end of select_all_synapses_CB */

/************************************************************************************************************************

 	FUNCTION	: select_all_axo_synapses_CB(w, client_data, call_data)
 
	PURPOSE		: callback function that selects all synapses in the net

	RETURNS		: nothing

************************************************************************************************************************/

void select_all_axo_synapses_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	/* remove all actually selected objects from selection list */

	remove_selected_objects_from_selectionList();

	/* loop through all neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* loop through all synapses and add each synapse to the synapse selection list */

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			currentSynapse = currentSynapse->nextConnection)
		{
			/* in case of complex networkmodel check if synapse is axo-axonic */

			if (complexModel && (currentSynapse->axoSynapseNumber > 0))
			{
				add_synapse_to_selection_list(currentSynapse);
				currentSynapse->selected = TRUE;
				change_synapse_color(currentSynapse, resData.neuronSelectionColor);
			}
		}
	}

} /* end of select_all_axo_synapses_CB */

/************************************************************************************************************************

 	FUNCTION	: drawing_input_CB(w, client_data, call_data)
 
	PURPOSE		: callback function for inputs into the drawing area

	RETURNS		: nothing

************************************************************************************************************************/

void drawing_input_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	XmDrawingAreaCallbackStruct *selection = (XmDrawingAreaCallbackStruct *) call_data; /* pointer to call data */
	XEvent	*event = selection->event;	/* pointer to occured event */

	/* if network can't be edited return without doing anything */

	if (! editNetwork)
		return;

	/* evaluate type of event and react on each event */

	switch(event->type)
	{
	case ButtonPress:
			process_buttonevent(&(event->xbutton));
			break;
	case ButtonRelease:
	case KeyPress:
	case KeyRelease:
		break;
	default:
		fprintf(stderr,"Warning: unknown event occured in DRAWING_INPUT. Eventnumber: %d\n", event->type);
		break;
	}

} /* end of drawing_input_CB */

/************************************************************************************************************************

 	FUNCTION	: drawing_expose_CB(w, client_data, call_data)
 
	PURPOSE		: callback function for expose events into the drawing area

	RETURNS		: nothing

************************************************************************************************************************/

void drawing_expose_CB(w, client_data, call_data)
Widget w;					/* widget that caused the callback */
caddr_t client_data;				/* client data */
XmDrawingAreaCallbackStruct	*call_data;	/* call data */
{
	/* check if more pending expose events are in the queue, if so, return without doing anything */

	XmUpdateDisplay(w);	/* processes all pending exposure events immediately */
	if (call_data->event->xexpose.count != 0)
		return;	/* there are more pending expose events, so return */
	else show_all_objects_in_drawing_area(FALSE);	/* redraw all objects in the drawing area */

} /* end of drawing_expose_CB */

/************************************************************************************************************************

 	FUNCTION	: drawing_resize_CB(w, client_data, call_data)
 
	PURPOSE		: callback function for resize events into the drawing area

	RETURNS		: nothing

************************************************************************************************************************/

void drawing_resize_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	/* if the widget is realized then clear the entire area -> this causes an expose event */

	if (XtIsRealized(w))
		XClearArea(disp, XtWindow(w), 0, 0, 0, 0, TRUE);	/* generates an expose event */

} /* end of drawing_resize_CB */

/************************************************************************************************************************

 	FUNCTION	: remove_selected_objects_CB(w, client_data, call_data)
 
	PURPOSE		: callback function to remove all selected objects

	RETURNS		: nothing

************************************************************************************************************************/

void remove_selected_objects_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	SelectionListSynapses	*actptrS, *copyS;	/* pointers to synapse selection list */
	SelectionListNeurons	*actptrN, *copyN;	/* pointers to neuron selection list */

	if (selectionListSynapsesHead != NULL)	/* synapses are selected */
	{
		actptrS = selectionListSynapsesHead;

		/* loop through all selected synapses */

		while (actptrS != NULL)
		{
			/* delete synapse and free memory */

			delete_connection(actptrS->synapsePtr->parentNeuron, actptrS->synapsePtr);
			copyS = actptrS;
			actptrS = actptrS->next;
			free(copyS);
		}

		/* reset synapse selection list and counter */

		selectionListSynapsesHead = selectionListSynapsesTail = NULL;
		selectionListSynapsesCounter = 0;
	}
	else	/* neurons are selected */
	{
		actptrN = selectionListNeuronsHead;

		/* loop through all selected neurons */

		while (actptrN != NULL)
		{
			/* delete neuron and free memory */

			change_neuron_color(actptrN->neuronPtr, resData.backgroundColor, resData.backgroundColor);
			remove_neuron_from_neuron_list(actptrN->neuronPtr);
			copyN = actptrN;
			actptrN = actptrN->next;
			free(copyN);
		}

		/* reset neuron selection list and counter */

		selectionListNeuronsHead = selectionListNeuronsTail = NULL;
		selectionListNeuronsCounter = 0;
	}

	/* refresh the drawing area */

	show_all_objects_in_drawing_area(FALSE);

} /* end of remove_selected_objects_CB */

/************************************************************************************************************************

 	FUNCTION	: remove_selected_objects_from_selectionList_CB(w, client_data, call_data)
 
	PURPOSE		: callback function to remove all selected objects from the selection list

	RETURNS		: nothing

************************************************************************************************************************/

void remove_selected_objects_from_selectionList_CB(w, client_data, call_data)
Widget w;			/* widget that caused the callback */
caddr_t client_data, call_data; /* client and call data */
{
	remove_selected_objects_from_selectionList();

} /* end of remove_selected_objects_from_selectionList_CB */
#endif
