/**************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	graph.c

	PURPOSE:	Handels all of the needs of the separate graph windows

	FUNCTIONS:	in order of appearance

	void update_graph();	- update graph of drawing area in current graph until actual simulation time
	void draw_graph();	- draw graph of drawing area in current graph from start to end of simulation
	void graph_output();	- writes out graph data to file
	void set_x_axis();	- sets the x-axis time interval
	void set_y_axis();	- sets the y-axis time interval
	void draw_lines();	- draws grid lines on current graph
	void draw_dots();	- draws grid dots on current graph
	void draw_axises();	- draws axis on current graph
	void update_graphS();	- update synapse graph of drawing area in current graph until actual simulation time
	void draw_graphS();	- draw synapse graph of drawing area in current graph from start to end of simulation
	void set_x_axisS();	- sets the x-axis time interval for synapses
	void set_y_axisS();	- sets the y-axis time interval for synapses
	void draw_linesS();	- draws grid lines on current synapse graph
	void draw_dotsS();	- draws grid dots on current synapse graph
	void draw_axisesS();	- draws axis on current synapse graph

**************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/****** functions ******/

#ifndef BATCH
/**************************************************************************************************************************

 	FUNCTION	: update_graph(neuron)
 
	PURPOSE		: update graph of drawing area in current graph until actual simulation time

	RETURNS		: nothing

**************************************************************************************************************************/

void update_graph(neuron)
Neuron	*neuron;	/* neuron for which the graph should be updated */
{
	int		currentX;			/* current x value, used as index to for-loop */
	int		startX, endX;			/* start and end values of x for plotting */
	int		firstX;				/* first graph value for plotting */
	int		oldPlotXVal, oldPlotYVal;	/* old plot values for x and y-axis */
	int		plotXVal, plotYVal;		/* plot values for x and y-axis */
	int		zeroYVal;			/* top plot value */
	double 		yConst;				/* top value of y-axis */
	double		xMult, yMult;			/* number of pixels per ms of x and y-axis */
	char		tempString[128];		/* used for out of bounds message */
	XmString	text = NULL;			/* used for error messages */
	Window 		window;				/* corresponding window to given neuron */
	GC		somaGC,
			dendGC,
			synCGC,
			totCGC,
			NaGC,
			KGC,
			CaGC,
			Ca1GC,
			Ca2GC,
			KCaGC,
			AGC,
			IRGC,
			PGC,
			NGC;				/* graphical contexts for traces */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* mask used in graphical contexts */

	currentNeuron = neuron;
	currentGraph = neuron->graph;	/* corresponding graph to given neuron */
	window = XtWindow(currentGraph->drawingArea);

	if ((currentGraph->firstX / E_SCALE > L - STEP * 0.5) ||
	    ((currentGraph->lastX / E_SCALE < L - STEP * 0.5) &&
	     (currentGraph->lastX / E_SCALE <= currentGraph->updateTime))) /* if out of visible area -> return */
		return;

	/* create graphical contexts for all needed traces */

	if (currentGraph->traceSoma)
	{
		gcv.foreground = somaLineColor;
		mask = GCForeground;
		somaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceDendrit)
	{
		gcv.foreground = dendritLineColor;
		mask = GCForeground;
		dendGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceSynapticCurrent)
	{
		gcv.foreground = synapticCurrentLineColor;
		mask = GCForeground;
		synCGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceTotalCurrent)
	{
		gcv.foreground = totalCurrentLineColor;
		mask = GCForeground;
		totCGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceN)
	{
		gcv.foreground = nLineColor;
		mask = GCForeground;
		NGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceNA)
	{
		gcv.foreground = sodiumLineColor;
		mask = GCForeground;
		NaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceK)
	{
		gcv.foreground = potassiumLineColor;
		mask = GCForeground;
		KGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA)
	{
		gcv.foreground = calciumLineColor;
		mask = GCForeground;
		CaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA1)
	{
		gcv.foreground = calcium1LineColor;
		mask = GCForeground;
		Ca1GC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA2)
	{
		gcv.foreground = calcium2LineColor;
		mask = GCForeground;
		Ca2GC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceKCA)
	{
		gcv.foreground = kcaLineColor;
		mask = GCForeground;
		KCaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceA)
	{
		gcv.foreground = aLineColor;
		mask = GCForeground;
		AGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceIR)
	{
		gcv.foreground = irLineColor;
		mask = GCForeground;
		IRGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceP)
	{
		gcv.foreground = pLineColor;
		mask = GCForeground;
		PGC = XCreateGC(disp, window, mask, &gcv);
	}

	/* calculate needed constants for plotting */

	xMult = currentGraph->xScale * STEP * T_SCALE;
	yConst = currentGraph->highestY * currentGraph->yScale;
	yMult = currentGraph->yScale;
	zeroYVal = (int) rint(yConst);

	/* correct the time the graph was updated if necessary */

	if (currentGraph->updateTime < currentGraph->firstX / E_SCALE)
		currentGraph->updateTime = currentGraph->firstX / E_SCALE;

	/* calculate start and end values for plotting */

	firstX = (int) rint(currentGraph->firstX / E_SCALE / STEP);
	startX = (int) rint(currentGraph->updateTime / STEP);
	if (((L + STEP * 0.5) > currentGraph->lastX / E_SCALE) &&
	    (currentGraph->updateTime < currentGraph->lastX / E_SCALE))
		endX = (int) rint(currentGraph->lastX / E_SCALE / STEP);
	else	endX = (int) rint(L / STEP);

	/* calculate first x value to plot */

	plotXVal = (int) rint((startX - firstX) * xMult);

	/* plot graph values for selected traces through for-loop */

	for(currentX = startX + 1; currentX <= endX; currentX++)
	{

		oldPlotXVal = plotXVal;
		plotXVal = (int) rint((currentX - firstX) * xMult);	/* current value for plot x-axis */

		/* soma trace */

		if (currentGraph->traceSoma)
		{
			oldPlotYVal = (int) rint(yConst - (SOMA[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (SOMA[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, somaGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->somaBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->somaBound = TRUE;
					strcpy(tempString, somaParString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* dendrit trace */

		if (currentGraph->traceDendrit)
		{
			oldPlotYVal = (int) rint(yConst - (DENDRITE[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (DENDRITE[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, dendGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->dendBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->dendBound = TRUE;
					strcpy(tempString, dendritParString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* synaptic current trace */

		if (currentGraph->traceSynapticCurrent)
		{
			oldPlotYVal = (int) rint(yConst - (SYNC[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (SYNC[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, synCGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->synCBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->synCBound = TRUE;
					strcpy(tempString, synapticCurrentString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* total current trace */

		if (currentGraph->traceTotalCurrent)
		{
			oldPlotYVal = (int) rint(yConst - (TOTC[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (TOTC[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, totCGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->totCBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->totCBound = TRUE;
					strcpy(tempString, totalCurrentString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* N-trace */

		if (currentGraph->traceN)
		{
			oldPlotYVal = (int) rint(yConst - (IN[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IN[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, NGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iNBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iNBound = TRUE;
					strcpy(tempString, nChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* sodium trace */

		if (currentGraph->traceNA)
		{
			oldPlotYVal = (int) rint(yConst - (INA[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (INA[currentX] * yMult));
			if (plotYVal > 0 && plotYVal<currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, NaGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iNABound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iNABound = TRUE;
					strcpy(tempString, naChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* potassium trace */

		if (currentGraph->traceK)
		{
			oldPlotYVal = (int) rint(yConst - (IK[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IK[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, KGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iKBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iKBound = TRUE;
					strcpy(tempString, kChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* calcium trace */

		if (currentGraph->traceCA)
		{
			oldPlotYVal = (int) rint(yConst - (ICA[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (ICA[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, CaGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iCABound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iCABound = TRUE;
					strcpy(tempString, caChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* calcium1 trace */

		if (currentGraph->traceCA1)
		{
			oldPlotYVal = (int) rint(yConst - (ICA1[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (ICA1[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, Ca1GC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iCA1Bound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iCA1Bound = TRUE;
					strcpy(tempString, ca1ChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* calcium2 trace */

		if (currentGraph->traceCA2)
		{
			oldPlotYVal = (int) rint(yConst - (ICA2[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (ICA2[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, Ca2GC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iCA2Bound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iCA2Bound = TRUE;
					strcpy(tempString, ca2ChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* calcium dependend potassium trace */

		if (currentGraph->traceKCA)
		{
			oldPlotYVal = (int) rint(yConst - (IKCA[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IKCA[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, KCaGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iKCABound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iKCABound = TRUE;
					strcpy(tempString, kcaChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* A-trace */

		if (currentGraph->traceA)
		{
			oldPlotYVal = (int) rint(yConst - (IA[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IA[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, AGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iABound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iABound = TRUE;
					strcpy(tempString, aChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* IR-trace */

		if (currentGraph->traceIR)
		{
			oldPlotYVal = (int) rint(yConst - (IIR[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IIR[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, IRGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iIRBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iIRBound = TRUE;
					strcpy(tempString, irChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* P-trace */

		if (currentGraph->traceP)
		{
			oldPlotYVal = (int) rint(yConst - (IP[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (IP[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraph->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, PGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraph->iPBound &&
				   (plotYVal < 0 || plotYVal > currentGraph->height))
				{
					currentGraph->iPBound = TRUE;
					strcpy(tempString, pChannelsString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}
	}

	/* set updated time for this graph */

	currentGraph->updateTime = L;

	XFlush(disp);	/* get shure data is drawn */

	/* free used graphical contexts */

	if (currentGraph->traceSoma)
		XFreeGC(disp, somaGC);
	if (currentGraph->traceDendrit)
		XFreeGC(disp, dendGC);
	if (currentGraph->traceSynapticCurrent)
		XFreeGC(disp, synCGC);
	if (currentGraph->traceTotalCurrent)
		XFreeGC(disp, totCGC);
	if (currentGraph->traceN)
		XFreeGC(disp, NGC);
	if (currentGraph->traceNA)
		XFreeGC(disp, NaGC);
	if (currentGraph->traceK)
		XFreeGC(disp, KGC);
	if (currentGraph->traceCA)
		XFreeGC(disp, CaGC);
	if (currentGraph->traceCA1)
		XFreeGC(disp, Ca1GC);
	if (currentGraph->traceCA2)
		XFreeGC(disp, Ca2GC);
	if (currentGraph->traceKCA)
		XFreeGC(disp, KCaGC);
	if (currentGraph->traceA)
		XFreeGC(disp, AGC);
	if (currentGraph->traceIR)
		XFreeGC(disp, IRGC);
	if (currentGraph->traceP)
		XFreeGC(disp, PGC);

} /* end of update_graph */

/**************************************************************************************************************************

 	FUNCTION	: draw_graph(neuron)
 
	PURPOSE		: draw graph of drawing area in current graph from start to end of simulation

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_graph(neuron)
Neuron	*neuron;	/* neuron for which the graph should be drawn */
{
	int		currentX;			/* current x value, used as index to for-loop */
	int		startX, endX;			/* start and end values of x for plotting */
	int		oldPlotXVal, oldPlotYVal;	/* old plot values for x and y-axis */
	int		plotXVal, plotYVal;		/* plot values for x and y-axis */
	int		zeroYVal;			/* top plot value */
	double 		yConst;				/* top value of y-axis */
	double		xMult, yMult;			/* number of pixels per ms of x and y-axis */
	char		tempString[128];		/* used for out of bounds message */
	XmString	text = NULL;			/* used for error messages */
	Window 		window;				/* corresponding window to given neuron */
	GC		lineGC,
			textGC,
			somaGC,
			dendGC,
			synCGC,
			totCGC,
			NaGC,
			KGC,
			CaGC,
			Ca1GC,
			Ca2GC,
			KCaGC,
			AGC,
			IRGC,
			PGC,
			NGC;				/* graphical contexts for traces */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* mask used in graphical contexts */

	currentNeuron = neuron;
	currentGraph = neuron->graph;	/* corresponding graph to given neuron */
	window = XtWindow(currentGraph->drawingArea);

	/* clear window of corresponding graph */

	XClearWindow(disp, window);

	/* create graphical contexts for lines and text */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = textColor;
	mask = GCForeground;
	textGC = XCreateGC(disp, window, mask, &gcv);

	/* create graphical contexts for all needed traces */

	if (currentGraph->traceSoma)
	{
		gcv.foreground = somaLineColor;
		mask = GCForeground;
		somaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceDendrit)
	{
		gcv.foreground = dendritLineColor;
		mask = GCForeground;
		dendGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceSynapticCurrent)
	{
		gcv.foreground = synapticCurrentLineColor;
		mask = GCForeground;
		synCGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceTotalCurrent)
	{
		gcv.foreground = totalCurrentLineColor;
		mask = GCForeground;
		totCGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceN)
	{
		gcv.foreground = nLineColor;
		mask = GCForeground;
		NGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceNA)
	{
		gcv.foreground = sodiumLineColor;
		mask = GCForeground;
		NaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceK)
	{
		gcv.foreground = potassiumLineColor;
		mask = GCForeground;
		KGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA)
	{
		gcv.foreground = calciumLineColor;
		mask = GCForeground;
		CaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA1)
	{
		gcv.foreground = calcium1LineColor;
		mask = GCForeground;
		Ca1GC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceCA2)
	{
		gcv.foreground = calcium2LineColor;
		mask = GCForeground;
		Ca2GC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceKCA)
	{
		gcv.foreground = kcaLineColor;
		mask = GCForeground;
		KCaGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceA)
	{
		gcv.foreground = aLineColor;
		mask = GCForeground;
		AGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceIR)
	{
		gcv.foreground = irLineColor;
		mask = GCForeground;
		IRGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraph->traceP)
	{
		gcv.foreground = pLineColor;
		mask = GCForeground;
		PGC = XCreateGC(disp, window, mask, &gcv);
	}

	/* calculate needed constants for plotting */

	xMult = currentGraph->xScale * STEP * T_SCALE;
	yConst = currentGraph->highestY * currentGraph->yScale;
	yMult = currentGraph->yScale;
	zeroYVal = (int) rint(yConst);

	/* draw axis of current graph */

	draw_axises(currentGraph, xMult, yMult, zeroYVal);

	/* check if lines or dots must be plotted */

	if (currentGraph->gridType == 1) 				/* dots */
		draw_dots(currentGraph, xMult, yMult, zeroYVal);
	else if (currentGraph->gridType == 2)				/* lines */
		draw_lines(currentGraph, xMult, yMult, zeroYVal);

	/* check if graph has values and can be plotted */

	if (currentGraph->hasValues == TRUE && graphSim == TRUE)
	{
		/* calculate start and end values for for plotting */

		startX = (int) rint(currentGraph->firstX / E_SCALE / STEP);

		if (simulation.length < currentGraph->lastX / E_SCALE)
			endX = (int) rint(simulation.length / STEP);
		else	endX = (int) rint(currentGraph->lastX / E_SCALE / STEP);

		if (endX > simulationTime)
			endX = simulationTime;

		plotXVal = 0;	/* first value for x-axis must be initialized */

		/* plot graph values for selected traces through for-loop */

		for(currentX = startX + 1; currentX <= endX; currentX++)
		{

			oldPlotXVal = plotXVal;
			plotXVal = (int) rint(((currentX - startX) * xMult));	/* current value for x */

			/* soma trace */

			if (currentGraph->traceSoma)
			{
				oldPlotYVal = (int) rint(yConst - (SOMA[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (SOMA[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, somaGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->somaBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->somaBound = TRUE;
						strcpy(tempString, somaParString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* dendrit trace */

			if (currentGraph->traceDendrit)
			{
				oldPlotYVal = (int) rint(yConst - (DENDRITE[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (DENDRITE[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, dendGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->dendBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->dendBound = TRUE;
						strcpy(tempString, dendritParString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* synaptic current trace */

			if (currentGraph->traceSynapticCurrent)
			{
				oldPlotYVal = (int) rint(yConst - (SYNC[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (SYNC[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, synCGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->synCBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->synCBound = TRUE;
						strcpy(tempString, synapticCurrentString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* total current trace */

			if (currentGraph->traceTotalCurrent)
			{
				oldPlotYVal = (int) rint(yConst - (TOTC[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (TOTC[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, totCGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->totCBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->totCBound = TRUE;
						strcpy(tempString, totalCurrentString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* sodium trace */

			if (currentGraph->traceNA)
			{
				oldPlotYVal = (int) rint(yConst - (INA[currentX -1 ] * yMult));
				plotYVal = (int) rint(yConst - (INA[currentX] * yMult));
				if (plotYVal > 0 && plotYVal<currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, NaGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iNABound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iNABound = TRUE;
						strcpy(tempString, naChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* noise trace */

			if (currentGraph->traceN)
			{
				oldPlotYVal = (int) rint(yConst - (IN[currentX -1 ] * yMult));
				plotYVal = (int) rint(yConst - (IN[currentX] * yMult));
				if (plotYVal > 0 && plotYVal<currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, NGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iNBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iNBound = TRUE;
						strcpy(tempString, nChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* potassium trace */

			if (currentGraph->traceK)
			{
				oldPlotYVal = (int) rint(yConst - (IK[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (IK[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, KGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->somaBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iKBound = TRUE;
						strcpy(tempString, kChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* calcium trace */

			if (currentGraph->traceCA)
			{
				oldPlotYVal = (int) rint(yConst - (ICA[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (ICA[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, CaGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iCABound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iCABound = TRUE;
						strcpy(tempString, caChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* calcium1 trace */

			if (currentGraph->traceCA1)
			{
				oldPlotYVal = (int) rint(yConst - (ICA1[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (ICA1[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, Ca1GC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iCA1Bound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iCA1Bound = TRUE;
						strcpy(tempString, ca1ChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* calcium2 trace */

			if (currentGraph->traceCA2)
			{
				oldPlotYVal = (int) rint(yConst - (ICA2[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (ICA2[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, Ca2GC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iCA2Bound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iCA2Bound = TRUE;
						strcpy(tempString, ca2ChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* calcium dependend potassium trace */

			if (currentGraph->traceKCA)
			{
				oldPlotYVal = (int) rint(yConst - (IKCA[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (IKCA[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, KCaGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iKCABound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iKCABound = TRUE;
						strcpy(tempString, kcaChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* A-trace */

			if (currentGraph->traceA)
			{
				oldPlotYVal = (int) rint(yConst - (IA[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (IA[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, AGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iABound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iABound = TRUE;
						strcpy(tempString, aChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* IR-trace */

			if (currentGraph->traceIR)
			{
				oldPlotYVal = (int) rint(yConst - (IIR[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (IIR[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, IRGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iIRBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iIRBound = TRUE;
						strcpy(tempString, irChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* P-trace */

			if (currentGraph->traceP)
			{
				oldPlotYVal = (int) rint(yConst - (IP[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (IP[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraph->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, PGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraph->iPBound &&
					   (plotYVal < 0 || plotYVal > currentGraph->height))
					{
						currentGraph->iPBound = TRUE;
						strcpy(tempString, pChannelsString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}
		}
	} /* End of graph has data */

	XFlush(disp);	/* get shure data is drawn */

	/* free used graphical contexts */

	XFreeGC(disp, lineGC);
	XFreeGC(disp, textGC);
	if (currentGraph->traceSoma)
		XFreeGC(disp, somaGC);
	if (currentGraph->traceDendrit)
		XFreeGC(disp, dendGC);
	if (currentGraph->traceSynapticCurrent)
		XFreeGC(disp, synCGC);
	if (currentGraph->traceTotalCurrent)
		XFreeGC(disp, totCGC);
	if (currentGraph->traceN)
		XFreeGC(disp, NGC);
	if (currentGraph->traceNA)
		XFreeGC(disp, NaGC);
	if (currentGraph->traceK)
		XFreeGC(disp, KGC);
	if (currentGraph->traceCA)
		XFreeGC(disp, CaGC);
	if (currentGraph->traceCA1)
		XFreeGC(disp, Ca1GC);
	if (currentGraph->traceCA2)
		XFreeGC(disp, Ca2GC);
	if (currentGraph->traceKCA)
		XFreeGC(disp, KCaGC);
	if (currentGraph->traceA)
		XFreeGC(disp, AGC);
	if (currentGraph->traceIR)
		XFreeGC(disp, IRGC);
	if (currentGraph->traceP)
		XFreeGC(disp, PGC);

} /* end of draw_graph */
#endif

/**************************************************************************************************************************

 	FUNCTION	: graph_output(simTime)
 
	PURPOSE		: writes out graph data to file

	RETURNS		: nothing

**************************************************************************************************************************/

void graph_output(simTime)
int simTime;	/* actual simulation time in simulation steps */
{
	/* for all neurons in the network store values at current time, referenced through simtime */

	if (graphNumber + graphNumberS + graphNumberAS > 0)	/* check if graphs exist to be updated */
	{
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->graph != NULL)		/* check if graph is available */
			{
				currentGraph = currentNeuron->graph;	/* set current graph */

				/* if out of visible area -> do not store any values */
				
				if ((unsigned int) rint(currentGraph->lastX / E_SCALE / STEP) >=
					simTime)
				{
					/* check if in GB-model the gate variables should be drawn */

					if (networkmodel > 3 && currentGraph->gateVariables)
					{
						/* store the gate variables */

						SOMA[simTime] = (float) A_CONST;
						DENDRITE[simTime] = (float) B1_CONST;
						SYNC[simTime] = (float) B2_CONST;
						TOTC[simTime] = (float) P_CONST;
						IN[simTime] = (float) GB_GAMMA;
						INA[simTime] = (float) Q_CONST;
						IK[simTime] = (float) R_CONST;
						ICA[simTime] = (float) E_CONST;
						ICA1[simTime] = (float) M_CONST;
						ICA2[simTime] = (float) H_CONST;
						IKCA[simTime] = (float) N_CONST;
						IA[simTime] = (float) C_CONST;
						IIR[simTime] = (float) D_CONST;
						IP[simTime] = (float) S_CONST;
					}
					else
					{
						/* store the computed values as scaled values */

						SOMA[simTime] = (float) (E_POT[0] * E_SCALE);
						DENDRITE[simTime] = (float) (E_POT[DSeg] * E_SCALE);
						SYNC[simTime] = (float) (SynCur * I_SCALE);
						TOTC[simTime] = (float) (TotCur * I_SCALE);
						IN[simTime] = (float) (I_NOISE * I_SCALE);
						INA[simTime] = (float) (I_NA * I_SCALE);
						IK[simTime] = (float) (I_K * I_SCALE);
						ICA[simTime] = (float) (I_CA * I_SCALE);
						ICA1[simTime] = (float) (GB_I_CA1 * I_SCALE);
						ICA2[simTime] = (float) (GB_I_CA2 * I_SCALE);
						IKCA[simTime] = (float) (GB_I_KCA * I_SCALE);
						IA[simTime] = (float) (GB_I_A * I_SCALE);
						IIR[simTime] = (float) (GB_I_IR * I_SCALE);
						IP[simTime] = (float) (GB_I_P * I_SCALE);
					}
				}
			}

			/* store values for synapses */

			if (graphNumberS + graphNumberAS > 0)
			{
				for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
				     currentSynapse = currentSynapse->nextConnection)
				{
					if (currentSynapse->graph != NULL)		/* check if graph is available */
					{
						currentGraphS = currentSynapse->graph;	/* set current graph */

						/* if out of visible area -> do not store any values */
						
						if ((unsigned int) rint(currentGraphS->lastX / E_SCALE / STEP) >= simTime)
						{
							/* store the computed values as scaled values */

							POT[simTime] = (float)
								(currentSynapse->potential * E_SCALE);
							MEM[simTime] = (float)
								(currentSynapse->synapticMemory * MEM_SCALE);

							if (currentSynapse->axoSynapseNumber > 0) /* axo-axonic synapse */
							{
								STR[simTime] = (float)
									(currentSynapse->currentStrength * STR_SCALE);
								CON[simTime] = 0.0;
							}
							else	/* chemical synapse */
							{
								CON[simTime] = (float)
									(currentSynapse->currentConductance * CON_SCALE);
								STR[simTime] = 0.0;
							}
						}
					}
				} /* End of for through synapses */
			}
		} /* End of for through neurons */
	}
} /* end of graph_output */

#ifndef BATCH
/**************************************************************************************************************************

 	FUNCTION	: set_x_axis(workingGraph)
 
	PURPOSE		: sets the x-axis time interval
			  the current scaling of the x-axis is used to determine a suitable inscription

	RETURNS		: nothing

**************************************************************************************************************************/

void set_x_axis(workingGraph)
Graph *workingGraph;	/* graph on which the x-axis should be set */
{
	double	msInterval;		/* time interval for x-axis */
	char	formatString[10];	/* format string for plotting inscription */

	msInterval = workingGraph->xScale;

	if ( rint(0.1 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.1;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(0.25 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.25;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.5 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.5;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(1 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(2 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 2;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(5 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 5;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(10 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 10;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(20 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 20;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(50 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 50;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(100 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 100;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(200 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 200;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(500 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 500;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(1000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(2000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 2000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(5000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 5000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(10000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 10000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(20000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 20000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(50000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 50000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(100000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 100000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(200000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 200000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(500000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 500000;
		sprintf(formatString, "%c8.0f", '%');
	}
	else if ( rint(1000000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1000000;
		sprintf(formatString, "%c8.0f", '%');
	}

	strcpy(workingGraph->msFormatString, formatString);	/* store the format string for the x-axis */

} /* end of set_x_axis */

/**************************************************************************************************************************

 	FUNCTION	: set_y_axis(workinggraph)
 
	PURPOSE		: sets the y-axis time interval
			  the current scaling of the y-axis is used to determine a suitable inscription

	RETURNS		: nothing

**************************************************************************************************************************/

void set_y_axis(workingGraph)
Graph *workingGraph;	/* graph on which the y-axis should be set */
{
	double	mvInterval;		/* volt interval for y-axis */
	char	formatString[10];	/* format string for plotting inscription */

	mvInterval = workingGraph->yScale;

	if ( rint(0.001 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.001;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.005 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.005;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.01 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.01;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.025 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.025;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.05 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.05;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.1 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.1;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(0.25 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.25;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.5 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.5;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(1 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 1;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(2 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 2;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(5 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 5;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(10 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 10;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(20 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 20;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(50 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 50;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(100 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 100;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(200 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 200;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(500 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 500;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(1000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 1000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(2000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 2000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(5000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 5000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(10000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 10000;
		sprintf(formatString, "%c6.0f", '%');
	}

	strcpy(workingGraph->mvFormatString, formatString);

} /* end of set_y_axis */

/**************************************************************************************************************************

 	FUNCTION	: draw_lines(graph, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws grid lines on current graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_lines(graph, xMult, yMult, zeroYVal)
Graph	*graph;		/* graph on which the lines should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msOffset, mvOffset;	/* offsets for x and y-axis */
	double 		msValue, mvValue;	/* current values for inscribing x and y-axis */
	Window		window = XtWindow(graph->drawingArea);	/* window corresponding to current graph */
	GC		lineGC;			/* graphical context for drawing lines */
	XGCValues	gcv;			/* values used in graphical contexts */
	unsigned long	mask;			/* mask used in graphical contexts */

	currentGraph = graph;

	/* create graphical context for lines */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	/* inscription of x-axis */

	if ((graph->axis == 1) || (graph->axis == 3))	/* is x-axis selected ? */
		for (msValue = graph->timeInterval; ; msValue += graph->timeInterval)
		{
			msOffset = (int) rint(msValue * xMult / E_SCALE / STEP) + mvSpace;
			if (msOffset >= graph->width + mvSpace)	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, (int) msOffset, graph->height, msOffset, 0);
		}


	/* inscription of y-axis */

	if ((graph->axis == 2) || (graph->axis == 3))	/* is y-axis selected ? */
	{
		/* inscribe positive voltage area */

		for (mvValue = graph->voltInterval; ; mvValue += graph->voltInterval)
		{
			mvOffset = (int) rint(mvValue * yMult);
			if ((zeroYVal - mvOffset) < (int) 0)	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset,
			    mvSpace + graph->width, zeroYVal - mvOffset);
		}

		/* inscribe negative voltage area */

		for (mvValue = - graph->voltInterval; ; mvValue -= graph->voltInterval)
		{
			mvOffset = (int) rint(mvValue * yMult);
			if ((zeroYVal - mvOffset) > (graph->height))	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset,
			    mvSpace + graph->width, zeroYVal - mvOffset);
		}
	}
	XFreeGC(disp, lineGC);	/* free graphical context for lines */

} /* end of draw_lines */

/**************************************************************************************************************************

 	FUNCTION	: draw_dots(graph, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws grid dots on current graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_dots(graph, xMult, yMult, zeroYVal)
Graph	*graph;		/* graph on which the dots should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msOffset, mvOffset;	/* offsets for x and y-axis */
	double 		msValue, mvValue;	/* current values for inscribing x and y-axis */
	Window		window = XtWindow(graph->drawingArea);	/* window corresponding to current graph */
	GC		pointGC;		/* graphical context for drawing points */
	XGCValues	gcv;			/* values used in graphical contexts */
	unsigned long	mask;			/* mask used in graphical contexts */

	currentGraph = graph;

	/* create graphical context for lines */

	gcv.foreground = axisColor;
	mask = GCForeground;
	pointGC = XCreateGC(disp, window, mask, &gcv);

	if (graph->axis > 0)	/* should a grid be drawn ? */
		for (msValue = graph->timeInterval; ; msValue += graph->timeInterval)
		{
			msOffset = (int) rint(msValue * xMult / E_SCALE / STEP) + mvSpace;
			if (msOffset >= graph->width + mvSpace)	/* if end is reached leave for-loop */
				break;

			/* draw grid points above the zero line of the y-axis */

			for (mvValue = graph->voltInterval; ; mvValue += graph->voltInterval)
			{
				mvOffset = (int) rint(mvValue * yMult);
				if ((zeroYVal - mvOffset) < (int) 0)	/* if end is reached leave for-loop */
					break;
				XDrawPoint(disp, window, pointGC, msOffset, zeroYVal - mvOffset);
			}

			/* draw grid points below the zero line of the y-axis */

			for (mvValue = - graph->voltInterval; ; mvValue -= graph->voltInterval)
			{
				mvOffset = (int) rint(mvValue * yMult);
				if ((zeroYVal - mvOffset) > (graph->height)) /* if end is reached leave for-loop */
					break;
				XDrawPoint(disp, window, pointGC, msOffset, zeroYVal - mvOffset);
			}
		}
	XFreeGC(disp, pointGC);	/* free graphical context for points */

} /* end of draw_dots */

/**************************************************************************************************************************

 	FUNCTION	: draw_axises(graph, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws axis on current graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_axises(graph, xMult, yMult, zeroYVal)
Graph	*graph;		/* graph on which the axis should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msStringLength;				/* string length of x-axis string */
	int 		msOffset, mvOffset;			/* offsets for x and y-axis */
	int 		yOffset;				/* offset needed for y-axis */
	int		plotYVal;				/* plot values for y-axis */
	double 		yConst;					/* top value of y-axis */
	double 		msValue, mvValue;			/* current values for inscribing x and y-axis */
	char 		axisString[15];				/* current value of x-axis during inscription */
	char		formatString[8];			/* format string for inscibing the axis */
	Window		window = XtWindow(graph->drawingArea);	/* window corresponding to current graph */
	GC		lineGC,
			zeroLineGC,
			restingLineGC,
			spikeThresholdLineGC,
			textGC;					/* graphical context for drawing lines and text */
	XGCValues	gcv;					/* values used in graphical contexts */
	unsigned long	mask;					/* mask used in graphical contexts */

	/* check if width and height satisfies minimal conditions */

	if (graph->width < 50 || graph->height < 50)
		return;		/* drawing area to small */

	currentGraph = graph;
	currentNeuron = graph->neuron;
	yConst = graph->highestY * graph->yScale;

	/* create graphical contexts for lines and text */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = graphZeroLineColor;
	mask = GCForeground;
	zeroLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = restingLineColor;
	mask = GCForeground;
	restingLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = spikeThresholdLineColor;
	mask = GCForeground;
	spikeThresholdLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = textColor;
	mask = GCForeground;
	textGC = XCreateGC(disp, window, mask, &gcv);

	yOffset = 3;	/* initialize offset for y-axis to zero */

	/* draw lines representing the x and y-axis */

	XDrawLine(disp,window, lineGC, mvSpace, graph->height, mvSpace + graph->width, graph->height);
	XDrawLine(disp, window, lineGC, mvSpace, 0, mvSpace, graph->height);

	strcpy(formatString, graph->msFormatString);	/* set format string for x-axis */

	/* intersect x-axis */

	XDrawLine(disp, window, lineGC, mvSpace, graph->height, mvSpace, graph->height + (1.5 * charWidth));

	/* inscribe x-axis */

	for (msValue = graph->firstX; ; msValue += graph->timeInterval)
	{
		if (msValue == 0)
			msValue += graph->timeInterval;

		msOffset = (int) rint((msValue - graph->firstX) * xMult / E_SCALE / STEP) + mvSpace;

		if (msOffset >= graph->width + mvSpace - 30)	/* if end is reached leave for-loop */
			break;

		/* mark the x-axis with a little line */

		XDrawLine(disp,window, lineGC, msOffset, graph->height, msOffset, graph->height + charWidth);

		sprintf(axisString, formatString, msValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, msOffset - (int) (rint(msStringLength * 0.5) * 8),
		    graph->height + (int) (3.5 * charWidth), axisString, msStringLength);
	}

	/* if graph starts at time 0 draw a zero character at the beginning of the x-axis */

	if (graph->firstX == 0)
	{
		XDrawString(disp, window, textGC, mvSpace - (int) charWidth,
		    graph->height + (int) (3.5 * charWidth), "0", 1);
	}

	/* mark the axis with its correct unit */

	XDrawString(disp, window, textGC, graph->width - 22 + mvSpace,
	    graph->height + (int) (3.5 * charWidth), "[ms]", 4);
	XDrawString(disp, window, textGC, 1, graph->height + 7, "[mV|nA]", 7);

	strcpy(formatString, graph->mvFormatString);	/* set format string for y-axis */

	/* draw zero line and inscribe it with a zero character */

	if (zeroYVal >= 0 && zeroYVal <= graph->height)
	{
		/* draw zero line */

		XDrawLine(disp, window, zeroLineGC, mvSpace, zeroYVal, graph->width + 1 + mvSpace, zeroYVal);

		if (zeroYVal <= 4)
			yOffset = 4 - zeroYVal;
		else if (zeroYVal >= (graph->height - 4))
			yOffset = - (4 - (graph->height - zeroYVal));

		/* draw 0 */

		sprintf(axisString, formatString, 0);
		msStringLength = strlen(axisString);
		XDrawString(disp, window, textGC, 0, zeroYVal + yOffset, axisString, msStringLength);
	}

	/* draw resting potential line */

	plotYVal = (int) rint(yConst - SOMA_Eleak_INITIAL * E_SCALE * yMult);

	if (plotYVal >= 0 && plotYVal <= graph->height)
	{
		XDrawLine(disp, window, restingLineGC, mvSpace, plotYVal, graph->width + 1 + mvSpace, plotYVal);
	}

	/* draw spike threshold line */

	switch(networkmodel)
	{
	case 1:	/* On/Off model */

		plotYVal = (int) rint(yConst - ONOFF_SPIKE_THRESHOLD * E_SCALE * yMult);
		break;

	case 2:	/* SWIM model */

		plotYVal = (int) rint(yConst - SWIM_SPIKE_THRESHOLD * E_SCALE * yMult);
		break;

	case 3:	/* HH model */

		plotYVal = (int) rint(yConst - HH_SPIKE_THRESHOLD * E_SCALE * yMult);
		break;

	case 4:	/* GB model */

		plotYVal = (int) rint(yConst - GB_SPIKE_THRESHOLD * E_SCALE * yMult);
		break;

	}
	if (plotYVal >= 0 && plotYVal <= graph->height)
	{
		XDrawLine(disp, window, spikeThresholdLineGC, mvSpace, plotYVal,
			graph->width + 1 + mvSpace, plotYVal);
	}

	/* inscribe the y-axis */

	for (mvValue = graph->voltInterval; ; mvValue += graph->voltInterval)
	{
		mvOffset = (int) rint(mvValue * yMult);
		if ((zeroYVal - mvOffset) < 0)	/* if end is reached leave for-loop */
			break;

		/* mark the y-axis with a little line */

		XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset, mvSpace - charWidth, zeroYVal - mvOffset);

		if ((zeroYVal - mvOffset) >= (graph->height - 10))
			yOffset = ((zeroYVal - mvOffset) - graph->height - 10);
		else yOffset = 3;

		sprintf(axisString, formatString, mvValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, 0, zeroYVal - mvOffset + yOffset, axisString, msStringLength);
	}

	XDrawLine(disp, window, lineGC, mvSpace, zeroYVal, mvSpace - charWidth, zeroYVal);

	for (mvValue = - graph->voltInterval; ; mvValue -= graph->voltInterval)
	{
		mvOffset = (int) rint(mvValue * yMult);
		if ((zeroYVal - mvOffset) > (graph->height - 5))	/* if end is reached leave for-loop */
			break;

		/* mark the y-axis with a little line */

		XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset, mvSpace - charWidth, zeroYVal - mvOffset);

		if ((zeroYVal - mvOffset) <= 13)
			yOffset = (13 - (zeroYVal - mvOffset));
		else yOffset = 3;

		sprintf(axisString, formatString, mvValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, 0, zeroYVal - mvOffset + yOffset, axisString, msStringLength);
	}

	/* free the graphical contexts for lines and text */

	XFreeGC(disp, lineGC);
	XFreeGC(disp, zeroLineGC);
	XFreeGC(disp, restingLineGC);
	XFreeGC(disp, spikeThresholdLineGC);
	XFreeGC(disp, textGC);

} /* end of draw_axises */

/**************************************************************************************************************************

 	FUNCTION	: update_graphS(synapse)
 
	PURPOSE		: update synapse graph of drawing area in current graph until actual simulation time

	RETURNS		: nothing

**************************************************************************************************************************/

void update_graphS(synapse)
Connection	*synapse;	/* synapse for which the graph should be updated */
{
	int		currentX;			/* current x value, used as index to for-loop */
	int		startX, endX;			/* start and end values of x for plotting */
	int		firstX;				/* first graph value for plotting */
	int		oldPlotXVal, oldPlotYVal;	/* old plot values for x and y-axis */
	int		plotXVal, plotYVal;		/* plot values for x and y-axis */
	int		zeroYVal;			/* top plot value */
	double 		yConst;				/* top value of y-axis */
	double		xMult, yMult;			/* number of pixels per ms of x and y-axis */
	char		tempString[128];		/* used for out of bounds message */
	XmString	text = NULL;			/* used for error messages */
	Window 		window;				/* corresponding window to given synapse */
	GC		potGC,
			memGC,
			conGC,
			strGC;				/* graphical contexts for traces */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* mask used in graphical contexts */

	currentSynapse = synapse;
	currentGraphS = synapse->graph;	/* corresponding graph to given synapse */
	window = XtWindow(currentGraphS->drawingArea);

	if ((currentGraphS->firstX / E_SCALE > L - STEP * 0.5) ||
	    ((currentGraphS->lastX / E_SCALE < L - STEP * 0.5) &&
	     (currentGraphS->lastX / E_SCALE <= currentGraphS->updateTime))) /* if out of visible area -> return */
		return;

	/* create graphical contexts for all needed traces */

	if (currentGraphS->tracePot)
	{
		gcv.foreground = somaLineColor;
		mask = GCForeground;
		potGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceMem)
	{
		gcv.foreground = sodiumLineColor;
		mask = GCForeground;
		memGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceCon)
	{
		gcv.foreground = synapticCurrentLineColor;
		mask = GCForeground;
		conGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceStr)
	{
		gcv.foreground = potassiumLineColor;
		mask = GCForeground;
		strGC = XCreateGC(disp, window, mask, &gcv);
	}

	/* calculate needed constants for plotting */

	xMult = currentGraphS->xScale * STEP * T_SCALE;
	yConst = currentGraphS->highestY * currentGraphS->yScale;
	yMult = currentGraphS->yScale;
	zeroYVal = (int) rint(yConst);

	/* correct the time the graph was updated if necessary */

	if (currentGraphS->updateTime < currentGraphS->firstX / E_SCALE)
		currentGraphS->updateTime = currentGraphS->firstX / E_SCALE;

	/* calculate start and end values for plotting */

	firstX = (int) rint(currentGraphS->firstX / E_SCALE / STEP);
	startX = (int) rint(currentGraphS->updateTime / STEP);
	if (((L + STEP * 0.5) > currentGraphS->lastX / E_SCALE) &&
	    (currentGraphS->updateTime < currentGraphS->lastX / E_SCALE))
		endX = (int) rint(currentGraphS->lastX / E_SCALE / STEP);
	else	endX = (int) rint(L / STEP);

	/* calculate first x value to plot */

	plotXVal = (int) rint((startX - firstX) * xMult);

	/* plot graph values for selected traces through for-loop */

	for(currentX = startX + 1; currentX <= endX; currentX++)
	{

		oldPlotXVal = plotXVal;
		plotXVal = (int) rint((currentX - firstX) * xMult);	/* current value for plot x-axis */

		/* synapse potential trace */

		if (currentGraphS->tracePot)
		{
			oldPlotYVal = (int) rint(yConst - (currentGraphS->pot[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (currentGraphS->pot[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraphS->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, potGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraphS->potBound &&
				   (plotYVal < 0 || plotYVal > currentGraphS->height))
				{
					currentGraphS->potBound = TRUE;
					strcpy(tempString, synPotString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* memory trace */

		if (currentGraphS->traceMem)
		{
			oldPlotYVal = (int) rint(yConst - (currentGraphS->mem[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (currentGraphS->mem[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraphS->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, memGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraphS->memBound &&
				   (plotYVal < 0 || plotYVal > currentGraphS->height))
				{
					currentGraphS->memBound = TRUE;
					strcpy(tempString, memString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* current conductance trace */

		if (currentGraphS->traceCon)
		{
			oldPlotYVal = (int) rint(yConst - (currentGraphS->con[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (currentGraphS->con[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraphS->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, conGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraphS->conBound &&
				   (plotYVal < 0 || plotYVal > currentGraphS->height))
				{
					currentGraphS->conBound = TRUE;
					strcpy(tempString, condString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}

		/* current inhibition trace */

		if (currentGraphS->traceStr)
		{
			oldPlotYVal = (int) rint(yConst - (currentGraphS->str[currentX - 1] * yMult));
			plotYVal = (int) rint(yConst - (currentGraphS->str[currentX] * yMult));
			if (plotYVal > 0 && plotYVal < currentGraphS->height
			   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
				XDrawLine(disp, window, strGC, oldPlotXVal + mvSpace, oldPlotYVal,
					  plotXVal + mvSpace, plotYVal);
			else	/* bound checking */
				if (boundCheckingOn && ! currentGraphS->strBound &&
				   (plotYVal < 0 || plotYVal > currentGraphS->height))
				{
					currentGraphS->strBound = TRUE;
					strcpy(tempString, strString);
					strcat(tempString, " ");
					strcat(tempString, outOfBoundsString);
					text = XmStringCreate(tempString, defaultFont);
					XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
					XmStringFree(text);
				}
		}
	}

	/* set updated time for this graph */

	currentGraphS->updateTime = L;

	XFlush(disp);	/* get shure data is drawn */

	/* free used graphical contexts */

	if (currentGraphS->tracePot)
		XFreeGC(disp, potGC);
	if (currentGraphS->traceMem)
		XFreeGC(disp, memGC);
	if (currentGraphS->traceCon)
		XFreeGC(disp, conGC);
	if (currentGraphS->traceStr)
		XFreeGC(disp, strGC);

} /* end of update_graphS */

/**************************************************************************************************************************

 	FUNCTION	: draw_graphS(synapse)
 
	PURPOSE		: draw synapse graph of drawing area in current graph from start to end of simulation

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_graphS(synapse)
Connection	*synapse;	/* synapse for which the graph should be drawn */
{
	int		currentX;			/* current x value, used as index to for-loop */
	int		startX, endX;			/* start and end values of x for plotting */
	int		oldPlotXVal, oldPlotYVal;	/* old plot values for x and y-axis */
	int		plotXVal, plotYVal;		/* plot values for x and y-axis */
	int		zeroYVal;			/* top plot value */
	double 		yConst;				/* top value of y-axis */
	double		xMult, yMult;			/* number of pixels per ms of x and y-axis */
	char		tempString[128];		/* used for out of bounds message */
	XmString	text = NULL;			/* used for error messages */
	Window 		window;				/* corresponding window to given synapse */
	GC		lineGC,
			textGC,
			potGC,
			memGC,
			conGC,
			strGC;				/* graphical contexts for traces */
	XGCValues	gcv;				/* values used in graphical contexts */
	unsigned long	mask;				/* mask used in graphical contexts */

	currentSynapse = synapse;
	currentGraphS = synapse->graph;	/* corresponding graph to given synapse */
	window = XtWindow(currentGraphS->drawingArea);

	/* clear window of corresponding graph */

	XClearWindow(disp, window);

	/* create graphical contexts for lines and text */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = textColor;
	mask = GCForeground;
	textGC = XCreateGC(disp, window, mask, &gcv);

	/* create graphical contexts for all needed traces */

	if (currentGraphS->tracePot)
	{
		gcv.foreground = somaLineColor;
		mask = GCForeground;
		potGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceMem)
	{
		gcv.foreground = sodiumLineColor;
		mask = GCForeground;
		memGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceCon)
	{
		gcv.foreground = synapticCurrentLineColor;
		mask = GCForeground;
		conGC = XCreateGC(disp, window, mask, &gcv);
	}
	if (currentGraphS->traceStr)
	{
		gcv.foreground = potassiumLineColor;
		mask = GCForeground;
		strGC = XCreateGC(disp, window, mask, &gcv);
	}

	/* calculate needed constants for plotting */

	xMult = currentGraphS->xScale * STEP * T_SCALE;
	yConst = currentGraphS->highestY * currentGraphS->yScale;
	yMult = currentGraphS->yScale;
	zeroYVal = (int) rint(yConst);

	/* draw axis of current graph */

	draw_axisesS(currentGraphS, xMult, yMult, zeroYVal);

	/* check if lines or dots must be plotted */

	if (currentGraphS->gridType == 1) 				/* dots */
		draw_dotsS(currentGraphS, xMult, yMult, zeroYVal);
	else if (currentGraphS->gridType == 2)				/* lines */
		draw_linesS(currentGraphS, xMult, yMult, zeroYVal);

	/* check if graph has values and can be plotted */

	if (currentGraphS->hasValues == TRUE && graphSim == TRUE)
	{
		/* calculate start and end values for for plotting */

		startX = (int) rint(currentGraphS->firstX / E_SCALE / STEP);

		if (simulation.length < currentGraphS->lastX / E_SCALE)
			endX = (int) rint(simulation.length / STEP);
		else	endX = (int) rint(currentGraphS->lastX / E_SCALE / STEP);

		if (endX > simulationTime)
			endX = simulationTime;

		plotXVal = 0;	/* first value for x-axis must be initialized */

		/* plot graph values for selected traces through for-loop */

		for(currentX = startX + 1; currentX <= endX; currentX++)
		{

			oldPlotXVal = plotXVal;
			plotXVal = (int) rint(((currentX - startX) * xMult));	/* current value for x */

			/* synapse potential trace */

			if (currentGraphS->tracePot)
			{
				oldPlotYVal = (int) rint(yConst - (currentGraphS->pot[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (currentGraphS->pot[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraphS->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, potGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraphS->potBound &&
					   (plotYVal < 0 || plotYVal > currentGraphS->height))
					{
						currentGraphS->potBound = TRUE;
						strcpy(tempString, synPotString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* memory trace */

			if (currentGraphS->traceMem)
			{
				oldPlotYVal = (int) rint(yConst - (currentGraphS->mem[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (currentGraphS->mem[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraphS->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, memGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraphS->memBound &&
					   (plotYVal < 0 || plotYVal > currentGraphS->height))
					{
						currentGraphS->memBound = TRUE;
						strcpy(tempString, memString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* current conductance trace */

			if (currentGraphS->traceCon)
			{
				oldPlotYVal = (int) rint(yConst - (currentGraphS->con[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (currentGraphS->con[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraphS->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, conGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraphS->conBound &&
					   (plotYVal < 0 || plotYVal > currentGraphS->height))
					{
						currentGraphS->conBound = TRUE;
						strcpy(tempString, condString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}

			/* current inhibition trace */

			if (currentGraphS->traceStr)
			{
				oldPlotYVal = (int) rint(yConst - (currentGraphS->str[currentX - 1] * yMult));
				plotYVal = (int) rint(yConst - (currentGraphS->str[currentX] * yMult));
				if (plotYVal > 0 && plotYVal < currentGraphS->height
				   && (oldPlotYVal != zeroYVal || plotYVal != zeroYVal))
					XDrawLine(disp, window, strGC, oldPlotXVal + mvSpace, oldPlotYVal,
						  plotXVal + mvSpace, plotYVal);
				else	/* bound checking */
					if (boundCheckingOn && ! currentGraphS->strBound &&
					   (plotYVal < 0 || plotYVal > currentGraphS->height))
					{
						currentGraphS->strBound = TRUE;
						strcpy(tempString, strString);
						strcat(tempString, " ");
						strcat(tempString, outOfBoundsString);
						text = XmStringCreate(tempString, defaultFont);
						XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
						XmStringFree(text);
					}
			}
		}
	} /* End of graph has data */

	XFlush(disp);	/* get shure data is drawn */

	/* free used graphical contexts */

	XFreeGC(disp, lineGC);
	XFreeGC(disp, textGC);
	if (currentGraphS->tracePot)
		XFreeGC(disp, potGC);
	if (currentGraphS->traceMem)
		XFreeGC(disp, memGC);
	if (currentGraphS->traceCon)
		XFreeGC(disp, conGC);
	if (currentGraphS->traceStr)
		XFreeGC(disp, strGC);

} /* end of draw_graphS */

/**************************************************************************************************************************

 	FUNCTION	: set_x_axisS(workingGraph)
 
	PURPOSE		: sets the x-axis time interval for synapses
			  the current scaling of the x-axis is used to determine a suitable inscription

	RETURNS		: nothing

**************************************************************************************************************************/

void set_x_axisS(workingGraph)
SGraph *workingGraph;	/* graph on which the x-axis should be set */
{
	double	msInterval;		/* time interval for x-axis */
	char	formatString[10];	/* format string for plotting inscription */

	msInterval = workingGraph->xScale;

	if ( rint(0.1 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.1;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(0.25 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.25;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.5 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 0.5;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(1 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(2 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 2;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(5 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 5;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(10 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 10;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(20 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 20;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(50 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 50;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(100 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 100;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(200 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 200;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(500 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 500;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(1000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(2000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 2000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(5000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 5000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(10000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 10000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(20000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 20000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(50000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 50000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(100000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 100000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(200000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 200000;
		sprintf(formatString, "%c7.0f", '%');
	}
	else if ( rint(500000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 500000;
		sprintf(formatString, "%c8.0f", '%');
	}
	else if ( rint(1000000 * msInterval) > MIN_MS_SPACE )
	{
		workingGraph->timeInterval = 1000000;
		sprintf(formatString, "%c8.0f", '%');
	}

	strcpy(workingGraph->msFormatString, formatString);	/* store the format string for the x-axis */

} /* end of set_x_axisS */

/**************************************************************************************************************************

 	FUNCTION	: set_y_axisS(workinggraph)
 
	PURPOSE		: sets the y-axis time interval for synapses
			  the current scaling of the y-axis is used to determine a suitable inscription

	RETURNS		: nothing

**************************************************************************************************************************/

void set_y_axisS(workingGraph)
SGraph *workingGraph;	/* graph on which the y-axis should be set */
{
	double	mvInterval;		/* volt interval for y-axis */
	char	formatString[10];	/* format string for plotting inscription */

	mvInterval = workingGraph->yScale;

	if ( rint(0.001 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.001;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.005 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.005;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.01 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.01;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.025 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.025;
		sprintf(formatString, "%c4.3f", '%');
	}
	else if ( rint(0.05 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.05;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.1 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.1;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(0.25 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.25;
		sprintf(formatString, "%c4.2f", '%');
	}
	else if ( rint(0.5 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 0.5;
		sprintf(formatString, "%c4.1f", '%');
	}
	else if ( rint(1 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 1;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(2 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 2;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(5 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 5;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(10 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 10;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(20 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 20;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(50 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 50;
		sprintf(formatString, "%c4.0f", '%');
	}
	else if ( rint(100 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 100;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(200 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 200;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(500 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 500;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(1000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 1000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(2000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 2000;
		sprintf(formatString, "%c5.0f", '%');
	}
	else if ( rint(5000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 5000;
		sprintf(formatString, "%c6.0f", '%');
	}
	else if ( rint(10000 * mvInterval) > MIN_MV_SPACE )
	{
		workingGraph->voltInterval = 10000;
		sprintf(formatString, "%c6.0f", '%');
	}

	strcpy(workingGraph->mvFormatString, formatString);

} /* end of set_y_axisS */

/**************************************************************************************************************************

 	FUNCTION	: draw_linesS(graphS, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws grid lines on current synapse graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_linesS(graphS, xMult, yMult, zeroYVal)
SGraph	*graphS;	/* graph on which the lines should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msOffset, mvOffset;	/* offsets for x and y-axis */
	double 		msValue, mvValue;	/* current values for inscribing x and y-axis */
	Window		window = XtWindow(graphS->drawingArea);	/* window corresponding to current graph */
	GC		lineGC;			/* graphical context for drawing lines */
	XGCValues	gcv;			/* values used in graphical contexts */
	unsigned long	mask;			/* mask used in graphical contexts */

	currentGraphS = graphS;

	/* create graphical context for lines */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	/* inscription of x-axis */

	if ((graphS->axis == 1) || (graphS->axis == 3))	/* is x-axis selected ? */
		for (msValue = graphS->timeInterval; ; msValue += graphS->timeInterval)
		{
			msOffset = (int) rint(msValue * xMult / E_SCALE / STEP) + mvSpace;
			if (msOffset >= graphS->width + mvSpace)	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, (int) msOffset, graphS->height, msOffset, 0);
		}


	/* inscription of y-axis */

	if ((graphS->axis == 2) || (graphS->axis == 3))	/* is y-axis selected ? */
	{
		/* inscribe positive voltage area */

		for (mvValue = graphS->voltInterval; ; mvValue += graphS->voltInterval)
		{
			mvOffset = (int) rint(mvValue * yMult);
			if ((zeroYVal - mvOffset) < (int) 0)	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset,
			    mvSpace + graphS->width, zeroYVal - mvOffset);
		}

		/* inscribe negative voltage area */

		for (mvValue = - graphS->voltInterval; ; mvValue -= graphS->voltInterval)
		{
			mvOffset = (int) rint(mvValue * yMult);
			if ((zeroYVal - mvOffset) > (graphS->height))	/* if end is reached leave for-loop */
				break;
			XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset,
			    mvSpace + graphS->width, zeroYVal - mvOffset);
		}
	}
	XFreeGC(disp, lineGC);	/* free graphical context for lines */

} /* end of draw_linesS */

/**************************************************************************************************************************

 	FUNCTION	: draw_dotsS(graphS, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws grid dots on current synapse graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_dotsS(graphS, xMult, yMult, zeroYVal)
SGraph	*graphS;	/* graph on which the dots should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msOffset, mvOffset;	/* offsets for x and y-axis */
	double 		msValue, mvValue;	/* current values for inscribing x and y-axis */
	Window		window = XtWindow(graphS->drawingArea);	/* window corresponding to current graph */
	GC		pointGC;		/* graphical context for drawing points */
	XGCValues	gcv;			/* values used in graphical contexts */
	unsigned long	mask;			/* mask used in graphical contexts */

	currentGraphS = graphS;

	/* create graphical context for lines */

	gcv.foreground = axisColor;
	mask = GCForeground;
	pointGC = XCreateGC(disp, window, mask, &gcv);

	if (graphS->axis > 0)	/* should a grid be drawn ? */
		for (msValue = graphS->timeInterval; ; msValue += graphS->timeInterval)
		{
			msOffset = (int) rint(msValue * xMult / E_SCALE / STEP) + mvSpace;
			if (msOffset >= graphS->width + mvSpace)	/* if end is reached leave for-loop */
				break;

			/* draw grid points above the zero line of the y-axis */

			for (mvValue = graphS->voltInterval; ; mvValue += graphS->voltInterval)
			{
				mvOffset = (int) rint(mvValue * yMult);
				if ((zeroYVal - mvOffset) < (int) 0)	/* if end is reached leave for-loop */
					break;
				XDrawPoint(disp, window, pointGC, msOffset, zeroYVal - mvOffset);
			}

			/* draw grid points below the zero line of the y-axis */

			for (mvValue = - graphS->voltInterval; ; mvValue -= graphS->voltInterval)
			{
				mvOffset = (int) rint(mvValue * yMult);
				if ((zeroYVal - mvOffset) > (graphS->height)) /* if end is reached leave for-loop */
					break;
				XDrawPoint(disp, window, pointGC, msOffset, zeroYVal - mvOffset);
			}
		}
	XFreeGC(disp, pointGC);	/* free graphical context for points */

} /* end of draw_dotsS */

/**************************************************************************************************************************

 	FUNCTION	: draw_axisesS(graphS, xMult, yMult, zeroYVal)
 
	PURPOSE		: draws axis on current synapse graph

	RETURNS		: nothing

**************************************************************************************************************************/

void draw_axisesS(graphS, xMult, yMult, zeroYVal)
SGraph	*graphS;	/* graph on which the axis should be drawn */
double	xMult, yMult;	/* number of pixels per ms of x and y-axis */
int	zeroYVal;	/* top plot value */
{
	int 		msStringLength;				/* string length of x-axis string */
	int 		msOffset, mvOffset;			/* offsets for x and y-axis */
	int 		yOffset;				/* offset needed for y-axis */
	int		plotYVal;				/* plot values for y-axis */
	double 		yConst;					/* top value of y-axis */
	double 		msValue, mvValue;			/* current values for inscribing x and y-axis */
	char 		axisString[15];				/* current value of x-axis during inscription */
	char		formatString[8];			/* format string for inscibing the axis */
	Window		window = XtWindow(graphS->drawingArea);	/* window corresponding to current graph */
	GC		lineGC,
			zeroLineGC,
			restingLineGC,
			trtLineGC,
			textGC;					/* graphical context for drawing lines and text */
	XGCValues	gcv;					/* values used in graphical contexts */
	unsigned long	mask;					/* mask used in graphical contexts */

	/* check if width and height satisfies minimal conditions */

	if (graphS->width < 50 || graphS->height < 50)
		return;		/* drawing area to small */

	currentGraphS = graphS;
	currentNeuron = graphS->synapse->parentNeuron;
	yConst = graphS->highestY * graphS->yScale;

	/* create graphical contexts for lines and text */

	gcv.foreground = axisColor;
	mask = GCForeground;
	lineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = graphZeroLineColor;
	mask = GCForeground;
	zeroLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = restingLineColor;
	mask = GCForeground;
	restingLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = spikeThresholdLineColor;
	mask = GCForeground;
	trtLineGC = XCreateGC(disp, window, mask, &gcv);

	gcv.foreground = textColor;
	mask = GCForeground;
	textGC = XCreateGC(disp, window, mask, &gcv);

	yOffset = 3;	/* initialize offset for y-axis to zero */

	/* draw lines representing the x and y-axis */

	XDrawLine(disp,window, lineGC, mvSpace, graphS->height, mvSpace + graphS->width, graphS->height);
	XDrawLine(disp, window, lineGC, mvSpace, 0, mvSpace, graphS->height);

	strcpy(formatString, graphS->msFormatString);	/* set format string for x-axis */

	/* intersect x-axis */

	XDrawLine(disp, window, lineGC, mvSpace, graphS->height, mvSpace, graphS->height + (1.5 * charWidth));

	/* inscribe x-axis */

	for (msValue = graphS->firstX; ; msValue += graphS->timeInterval)
	{
		if (msValue == 0)
			msValue += graphS->timeInterval;

		msOffset = (int) rint((msValue - graphS->firstX) * xMult / E_SCALE / STEP) + mvSpace;

		if (msOffset >= graphS->width + mvSpace - 30)	/* if end is reached leave for-loop */
			break;

		/* mark the x-axis with a little line */

		XDrawLine(disp,window, lineGC, msOffset, graphS->height, msOffset, graphS->height + charWidth);

		sprintf(axisString, formatString, msValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, msOffset - (int) (rint(msStringLength * 0.5) * 8),
		    graphS->height + (int) (3.5 * charWidth), axisString, msStringLength);
	}

	/* if graph starts at time 0 draw a zero character at the beginning of the x-axis */

	if (graphS->firstX == 0)
	{
		XDrawString(disp, window, textGC, mvSpace - (int) charWidth,
		    graphS->height + (int) (3.5 * charWidth), "0", 1);
	}

	/* mark the axis with its correct unit */

	XDrawString(disp, window, textGC, graphS->width - 22 + mvSpace,
	    graphS->height + (int) (3.5 * charWidth), "[ms]", 4);
	XDrawString(disp, window, textGC, 1, graphS->height + 7, "[mV|nS]", 7);

	strcpy(formatString, graphS->mvFormatString);	/* set format string for y-axis */

	/* draw zero line and inscribe it with a zero character */

	if (zeroYVal >= 0 && zeroYVal <= graphS->height)
	{
		/* draw zero line */

		XDrawLine(disp, window, zeroLineGC, mvSpace, zeroYVal, graphS->width + 1 + mvSpace, zeroYVal);

		if (zeroYVal <= 4)
			yOffset = 4 - zeroYVal;
		else if (zeroYVal >= (graphS->height - 4))
			yOffset = - (4 - (graphS->height - zeroYVal));

		/* draw 0 */

		sprintf(axisString, formatString, 0);
		msStringLength = strlen(axisString);
		XDrawString(disp, window, textGC, 0, zeroYVal + yOffset, axisString, msStringLength);
	}

	/* draw resting potential line */

	plotYVal = (int) rint(yConst - SOMA_Eleak_INITIAL * E_SCALE * yMult);

	if (plotYVal >= 0 && plotYVal <= graphS->height)
	{
		XDrawLine(disp, window, restingLineGC, mvSpace, plotYVal, graphS->width + 1 + mvSpace, plotYVal);
	}

	/* draw transmitter release threshold line */

	plotYVal = (int) rint(yConst - graphS->synapse->transmitterReleaseThreshold * E_SCALE * yMult);

	if (plotYVal >= 0 && plotYVal <= graphS->height)
	{
		XDrawLine(disp, window, trtLineGC, mvSpace, plotYVal, graphS->width + 1 + mvSpace, plotYVal);
	}

	/* inscribe the y-axis */

	for (mvValue = graphS->voltInterval; ; mvValue += graphS->voltInterval)
	{
		mvOffset = (int) rint(mvValue * yMult);
		if ((zeroYVal - mvOffset) < 0)	/* if end is reached leave for-loop */
			break;

		/* mark the y-axis with a little line */

		XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset, mvSpace - charWidth, zeroYVal - mvOffset);

		if ((zeroYVal - mvOffset) >= (graphS->height - 10))
			yOffset = ((zeroYVal - mvOffset) - graphS->height - 10);
		else yOffset = 3;

		sprintf(axisString, formatString, mvValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, 0, zeroYVal - mvOffset + yOffset, axisString, msStringLength);
	}
	for (mvValue = -graphS->voltInterval; ; mvValue -= graphS->voltInterval)
	{
		mvOffset = (int) rint(mvValue * yMult);
		if ((zeroYVal - mvOffset) > (graphS->height - 5))	/* if end is reached leave for-loop */
			break;

		/* mark the y-axis with a little line */

		XDrawLine(disp, window, lineGC, mvSpace, zeroYVal - mvOffset, mvSpace - charWidth, zeroYVal - mvOffset);

		if ((zeroYVal - mvOffset) <= 13)
			yOffset = (13 - (zeroYVal - mvOffset));
		else yOffset = 3;

		sprintf(axisString, formatString, mvValue);
		msStringLength = strlen(axisString);

		/* draw character string */

		XDrawString(disp, window, textGC, 0, zeroYVal - mvOffset + yOffset, axisString, msStringLength);
	}

	/* free the graphical contexts for lines and text */

	XFreeGC(disp, lineGC);
	XFreeGC(disp, zeroLineGC);
	XFreeGC(disp, restingLineGC);
	XFreeGC(disp, trtLineGC);
	XFreeGC(disp, textGC);

} /* end of draw_axisesS */
#endif
