/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	filesave.c

	PURPOSE:	contains all functions that handles file save operations

	FUNCTIONS:	in order of appearance

	save_network				- save a simulation file ( .sim ) containing all information
	save_parameter				- save a parameter file ( .par ) - network parameter
	save_configuration			- save a configuration file ( .con ) - network configuration
	save_neuron				- save neuron parameter
	save_general_neuron_parameters		- save a general neuron parameter file ( .gen )
	save_graph				- save a neuron graph file ( .asc ) - graph traces from graph window
	save_graphS				- save a synapse graph file ( .asc ) - graph traces from graph window

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/****** global variables ******/

AsciiDefaults asciiDefaults = { ';', '.' };	/* default values for graph file save */

/****** functions ******/

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: save_network(filename)
 
	PURPOSE		: save a simulation file ( .sim ) containing all information

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_network(filename)
char filename[128];	/* filename of simulation file */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	AxoAxonicSenderList *axoSender;		/* current axo-axonic synapse */
	int     	index;			/* loop variable */
	int		dendritSegment;		/* number of dendrit segment to be traced */
	Position 	x, y;			/* left-top window position */
	Dimension 	width, height;		/* width and height of window */
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* open simulation file and write out file type */

	if ((fp = fopen(filename,"w")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	fprintf(fp, "%i\n", NETWORK_FILE);

	/* query window position and height */

	n = 0;
	XtSetArg(args[n], XmNx, &x); n++;
	XtSetArg(args[n], XmNy, &y); n++;
	XtSetArg(args[n], XmNwidth, &width); n++;
	XtSetArg(args[n], XmNheight, &height); n++;
	XtGetValues(applShell, args, n);

	/* write out window position and height */

	fprintf(fp, "%i\n%i\n%i\n%i\n", x, y, (int) x + (int) width, (int) y + (int) height);

	/* write out number of neurons, synapses, ... */

	fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n", neuronNumber, synapseNumber - axoSynapsesNumber, axoSynapsesNumber,
		numTypes, neuronNumber, graphNumber, graphNumberS, graphNumberAS);

	/* write out network model, type and number of dendritic segments actually used */

	index = (networkmodel << 1) - (int) (! complexModel);
	fprintf(fp, "%i\n%i\n%i\n", index, networktype, curNumOfDendSegs);

	/* save simulation parameters */

	fprintf(fp, "%e\n%e\n", simulation.length, simulation.step);

	index = 0;
	if (autoReset)
		if (showActivityColors)
			index = 3;
		else	index = 1;
	else if (showActivityColors)
		index = 2;

	/* write out global menu options */

	fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n",
		(int) XmToggleButtonGadgetGetState(simInfButton),
		(int) XmToggleButtonGadgetGetState(timeMessureButton),
		(int) XmToggleButtonGadgetGetState(graphDisplayButton),
		index,
		(int) XmToggleButtonGadgetGetState(outputWatchButton),
		(int) XmToggleButtonGadgetGetState(manualUpdateButton),
		(int) XmToggleButtonGadgetGetState(noiseButton),
		(int) XmToggleButtonGadgetGetState(initialPotentialsButton),
		(int) XmToggleButtonGadgetGetState(synapticStrengthButton),
		(int) XmToggleButtonGadgetGetState(synapticTransmissionTimeButton),
		integrationMethod);

	/* save global parameters */

	fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n", noiseIntensity, noiseStrength, noiseDecay,
		stdInitialPotential, stdInitialPotentialStd, stdSynapticStrength, stdSynapticStrengthStd,
		stdAxoAxonicStrength, stdAxoAxonicStrengthStd, stdSynapticTT, stdSynapticTTStd);

	/* write out learning conditions */

	index = 0;
	if (hebbLearning)
		if (competitiveLearning)
			index = 3;
		else	index = 1;
	else if (competitiveLearning)
		index = 2;

	fprintf(fp, "%i\n%i\n%i\n", (int) plasticityLearning, (int) conditionedLearning, (int) index);

	/* save neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n%i\n%i\n", currentNeuron->neuronId, (int) (currentNeuron->x + (int) rint((double)
			neuronDiameter / 2.0)), (int) (currentNeuron->y + (int) rint((double) neuronDiameter / 2.0)));

		save_neuron(fp, currentNeuron);
	}

	/* save neuron graph window data */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		if (currentNeuron->graph != NULL)
		{
			currentGraph = currentNeuron->graph;
			n = 0;
			XtSetArg(args[n], XmNx, &x); n++;
			XtSetArg(args[n], XmNy, &y); n++;
			XtSetArg(args[n], XmNwidth, &width); n++;
			XtSetArg(args[n], XmNheight, &height); n++;
			XtGetValues(XtParent(currentGraph->formWidget), args, n);

			if (! currentGraph->traceDendrit)
				dendritSegment = 0;
			else	dendritSegment = currentGraph->dendritSegment;

			fprintf(fp,"%i\n%i\n%i\n%i\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%e\n",
			    currentNeuron->neuronId, (int) x, (int) y, (int) (x + width), (int) (y + height),
			    currentGraph->firstX, currentGraph->lastX, currentGraph->xScale, currentGraph->highestY,
			    currentGraph->lowestY, currentGraph->yScale, currentGraph->yStep, currentGraph->gridType,
			    currentGraph->axis, (int) currentGraph->traceSoma, dendritSegment,
			    (int) currentGraph->traceSynapticCurrent, (int) currentGraph->traceTotalCurrent,
			    (int) currentGraph->traceN, (int) currentGraph->traceNA, (int) currentGraph->traceK,
			    (int) currentGraph->traceCA, (int) currentGraph->traceCA1, (int) currentGraph->traceCA2,
			    (int) currentGraph->traceKCA, (int) currentGraph->traceA, (int) currentGraph->traceIR,
			    (int) currentGraph->traceP, currentGraph->asciiDataOutputInterval); 
		}
	}

	/* save synapse types */

	if (synapseTypes != NULL)
	{
		for (index = 0; index <= numTypes - 1; index++)
		{
			fprintf(fp, "%i\n%e\n", synapseTypes[index].ionType, synapseTypes[index].equilPotential);
		}
	}

	/* save synapses */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
		{
		     if (currentSynapse->axoSynapseNumber == 0)	/* synapse is not axo-axonic */
			fprintf(fp, "%i\n%i\n%i\n%i\n%e\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
				currentSynapse->synapseID, currentNeuron->neuronId, currentSynapse->targetCode,
				currentSynapse->targetArea, currentSynapse->deliveryTime, currentSynapse->synapticType,
				currentSynapse->initialConductance, currentSynapse->maxConductance,
				currentSynapse->transmitterReleaseThreshold, currentSynapse->transmitterSatiate,
				currentSynapse->slope, currentSynapse->shortTermMemory, currentSynapse->longTermThreshold,
				currentSynapse->longTermMinimum, currentSynapse->hebbIntervall,
				currentSynapse->hebbLearnFactor, currentSynapse->shortTermPlasticityLearnFactor,
				currentSynapse->competitiveLearnFactor);
		}/* end of going through synapses */
	} /* end of going through net and saving synapses */

	/* save axo-axonic synapses */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
	  for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
	        currentSynapse = currentSynapse->nextConnection)
	  {
	     if (currentSynapse->numberOfAxoSynapses > 0)	/* synapse has axo-axonic synapses */
	       for(axoSender = currentSynapse->axoSenderList; axoSender != NULL; axoSender = axoSender->nextSender)
		 {
		    fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%e\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
			axoSender->connection->synapseID, axoSender->connection->targetSynapse->synapseID,
			axoSender->connection->parentNeuron->neuronId,
			axoSender->connection->targetSynapse->parentNeuron->neuronId,
			axoSender->connection->targetSynapse->targetNeuron->neuronId,
			axoSender->connection->targetSynapse->targetArea, axoSender->connection->deliveryTime,
			axoSender->connection->synapticType, axoSender->connection->transmitterReleaseThreshold,
			axoSender->connection->transmitterSatiate, axoSender->connection->initialInhibitionFactor,
			axoSender->connection->maxInhibitionFactor, axoSender->connection->slope,
			axoSender->connection->shortTermMemory, axoSender->connection->longTermThreshold,
			axoSender->connection->longTermMinimum, axoSender->connection->hebbIntervall,
			axoSender->connection->hebbLearnFactor, axoSender->connection->shortTermPlasticityLearnFactor,
			axoSender->connection->conditioningLearnFactor);
		 }
	  }/* end of going through synapses */
	}

	/* save synapse graph window data */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	    for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		 currentSynapse = currentSynapse->nextConnection)
		{
			if ((currentSynapse->graph != NULL) && (currentSynapse->axoSynapseNumber == 0))
			{
				currentGraphS = currentSynapse->graph;
				n = 0;
				XtSetArg(args[n], XmNx, &x); n++;
				XtSetArg(args[n], XmNy, &y); n++;
				XtSetArg(args[n], XmNwidth, &width); n++;
				XtSetArg(args[n], XmNheight, &height); n++;
				XtGetValues(XtParent(currentGraphS->formWidget), args, n);
				fprintf(fp,"%i\n%i\n%i\n%i\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%i\n%i\n%i\n%i\n%i\n%i\n%e\n",
				    currentSynapse->synapseID, (int) x, (int) y, (int) (x + width), (int) (y + height),
				    currentGraphS->firstX, currentGraphS->lastX, currentGraphS->xScale,
				    currentGraphS->highestY, currentGraphS->lowestY, currentGraphS->yScale,
				    currentGraphS->yStep, currentGraphS->gridType, currentGraphS->axis,
				    (int) currentGraphS->tracePot, (int) currentGraphS->traceMem,
				    (int) currentGraphS->traceCon, (int) currentGraphS->traceStr,
				    currentGraphS->asciiDataOutputInterval); 
			}
		}

	/* save axo-axonic synapse graph window data */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	  for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
	        currentSynapse = currentSynapse->nextConnection)
	  {
	     if (currentSynapse->numberOfAxoSynapses > 0)	/* synapse has axo-axonic synapses */
	       for(axoSender = currentSynapse->axoSenderList; axoSender != NULL; axoSender = axoSender->nextSender)
	       {
		if (axoSender->connection->graph != NULL)
		{
			currentGraphS = axoSender->connection->graph;
			n = 0;
			XtSetArg(args[n], XmNx, &x); n++;
			XtSetArg(args[n], XmNy, &y); n++;
			XtSetArg(args[n], XmNwidth, &width); n++;
			XtSetArg(args[n], XmNheight, &height); n++;
			XtGetValues(XtParent(currentGraphS->formWidget), args, n);
			fprintf(fp,"%i\n%i\n%i\n%i\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%i\n%i\n%i\n%i\n%i\n%i\n%e\n",
			    axoSender->connection->synapseID, (int) x, (int) y, (int) (x + width), (int) (y + height),
			    currentGraphS->firstX, currentGraphS->lastX, currentGraphS->xScale, currentGraphS->highestY,
			    currentGraphS->lowestY, currentGraphS->yScale, currentGraphS->yStep, currentGraphS->gridType,
			    currentGraphS->axis, (int) currentGraphS->tracePot, (int) currentGraphS->traceMem,
			    (int) currentGraphS->traceCon, (int) currentGraphS->traceStr,
			    currentGraphS->asciiDataOutputInterval); 
		}
	       }
	  }

	/* save currents */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n%e\n%e\n%e\n%e\n%e\n%e\n%i\n%i\n", currentNeuron->neuronId, currentNeuron->somaCurrent,
		    currentNeuron->somaSCStart, currentNeuron->somaSCEnd, currentNeuron->dendCurrent,
		    currentNeuron->dendSCStart, currentNeuron->dendSCEnd, currentNeuron->dendSCLocation,
		    (int) currentNeuron->noise);
	}

	/* close simulation file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_network */

/************************************************************************************************************************

 	FUNCTION	: save_parameter(filename)
 
	PURPOSE		: save a parameter file ( .par ) - network parameter

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_parameter(filename)
char filename[128];	/* filename for parameter file */
{
	AxoAxonicSenderList *axoSender;         /* current axo-axonic sender */
	ChemicalSenderList *currentSender;      /* current sender */
	ChemicalPool 	*currentPool;           /* current pool */
	int		totalNeurons;		/* number of neurons in the net */
	int		totalCurrents;		/* number of input currents in the net */
	int		queuePlaces;		/* number of entries in a process queue */
	int		index;			/* loop variable */
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* open parameter file and write out file type */

	if ((fp = fopen(filename,"w")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	fprintf(fp, "%i\n", PARAMETER_FILE);

	/* initialize number of items in net and write them out */

	totalNeurons = neuronNumber;
	totalCurrents = neuronNumber;

	fprintf(fp, "%i\n%i\n", totalNeurons, totalCurrents);

	/* save simulation parameters */

	fprintf(fp, "%e\n%e\n%i\n", simulation.length, simulation.step, neuronActivityInterval);

	index = 0;
	if (autoReset)
		if (showActivityColors)
			index = 3;
		else	index = 1;
	else if (showActivityColors)
		index = 2;

	/* write out global menu options */

	fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n",
		(int) XmToggleButtonGadgetGetState(simInfButton),
		(int) XmToggleButtonGadgetGetState(timeMessureButton),
		(int) XmToggleButtonGadgetGetState(graphDisplayButton),
		index,
		(int) XmToggleButtonGadgetGetState(outputWatchButton),
		(int) XmToggleButtonGadgetGetState(manualUpdateButton),
		(int) XmToggleButtonGadgetGetState(noiseButton),
		(int) XmToggleButtonGadgetGetState(initialPotentialsButton),
		(int) XmToggleButtonGadgetGetState(synapticStrengthButton),
		(int) XmToggleButtonGadgetGetState(synapticTransmissionTimeButton),
		integrationMethod);

	/* save global parameters */

	fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n", noiseIntensity, noiseStrength, noiseDecay,
		stdInitialPotential, stdInitialPotentialStd, stdSynapticStrength, stdSynapticStrengthStd,
		stdAxoAxonicStrength, stdAxoAxonicStrengthStd, stdSynapticTT, stdSynapticTTStd);

	/* write out learning conditions */

	index = 0;
	if (hebbLearning)
		if (competitiveLearning)
			index = 3;
		else	index = 1;
	else if (competitiveLearning)
		index = 2;

	fprintf(fp, "%i\n%i\n%i\n", (int) plasticityLearning, (int) conditionedLearning, (int) index);


	/* write out networkmodel, networktype and number of dendritic segments */

	index = (networkmodel << 1) - (int) (! complexModel);
	fprintf(fp, "%i\n%i\n%i\n", index, networktype, curNumOfDendSegs);

	/* save neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n", currentNeuron->neuronId);
		save_neuron(fp, currentNeuron);

		/* save current neuron values */

		for (index = 0; index <= curNumOfDendSegs; index++)
			fprintf(fp, "%e\n", E_POT[index]);

		fprintf(fp, "%i\n%e\n%e\n%e\n%i\n%i\n", SpikeCounter, PseudoSomaCur, PseudoSCSTRT, PseudoSCSTP, IRegion,
			IStem);
		
		fprintf(fp, "%e\n%e\n%e\n%e\n%e\n", SLmSG, SGpDSmDG, DLmDG, SGdSC, SGpDSmDGdSC);

		switch(networkmodel)
		{
		case 1: /* ON/OFF model */

			fprintf(fp, "%e\n%i\n", SPIKE, (int) SPIKE_IN_PROCESS);
			break;
		
		case 2: /* SWIM model */

			fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n", M_CONST, N_CONST, H_CONST, C_CONST, SWIM_NAEmNAG,
				SWIM_KEmKG, SWIM_CAEmCAG);
			break;

		case 3: /* HH model */

			fprintf(fp, "%e\n%e\n%e\n%e\n%e\n", M_CONST, N_CONST, H_CONST, HH_NAEmNAG, HH_KEmKG);
			break;

		case 4: /* GB model */

			fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
				M_CONST, N_CONST, H_CONST, C_CONST, D_CONST, E_CONST, A_CONST, B1_CONST, B2_CONST,
				P_CONST, Q_CONST, R_CONST, S_CONST);
			fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
				GB_I_CA1, GB_I_CA2, GB_CONC_CA, GB_NAEmNAG, GB_KEmKG, GB_AEmAG, GB_KCAEmKCAG,
				GB_IREmIRG, GB_PEmPG);
			break;
		}
	}

	/* save current synaptic values */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n", currentNeuron->neuronId);

		/* loop through pools */

		for (currentPool = currentNeuron->poolList; currentPool != NULL; currentPool = currentPool->nextPool)
		{
			fprintf(fp, "%e\n", currentPool->longestDeliveryTime);

			/* loop through senders */

			for (currentSender = currentPool->senderList; currentSender != NULL;
			     currentSender = currentSender->nextSender)
			{
				currentSynapse = currentSender->connection;

				fprintf(fp, "%e\n%e\n%i\n", currentSynapse->conductance, TSmTRT, TTdSS);

				if (complexModel)
				{
					currentSynapse->trainingFlag = 0;
					if (currentSynapse->trainingFlagSTP)
						currentSynapse->trainingFlag += 1;
					if (currentSynapse->trainingFlagHebb)
						currentSynapse->trainingFlag += 2;
					if (currentSynapse->trainingFlagCond)
						currentSynapse->trainingFlag += 4;

					fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%i\n%i\n%i\n%i\n%i\n%e\n%e\n%i\n%i\n%i\n",
						currentSynapse->synapticMemorySTP, currentSynapse->synapticMemoryHebb,
						LEXP, SFCONST, LFCONST, (int) currentSynapse->trainingFlag,
						(int) currentSynapse->shortTermLearningCondition,
						(int) currentSynapse->conditioningLearningCondition,
						(int) currentSynapse->hebbLearningCondition,
						(int) currentSynapse->competitiveLearningCondition,
						currentSynapse->hebbIntervallCounter, currentSynapse->hebbTimer,
						currentSynapse->stmCnt, currentSynapse->condCnt, currentSynapse->hebbCnt);
					
					/* loop through axo-axonic connections */

					for (axoSender = currentSynapse->axoSenderList; axoSender != NULL;
					     axoSender = axoSender->nextSender)
					{
						currentSynapse->trainingFlag = 0;
						if (currentSynapse->trainingFlagSTP)
							currentSynapse->trainingFlag += 1;
						if (currentSynapse->trainingFlagCond)
							currentSynapse->trainingFlag += 2;

						fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%i\n%i\n%i\n%i\n%i\n%e\n%e\n%i\n%i\n%i\n",
							axoSender->connection->currentInhibitionFactor,
							axoSender->connection->synapticMemoryCond,
							axoSender->connection->LEXP_const, axoSender->connection->
							SFCONST_const, axoSender->connection->LFCONST_const,
							(int) axoSender->connection->trainingFlag,
							(int) axoSender->connection->shortTermLearningCondition,
							(int) axoSender->connection->conditioningLearningCondition,
							(int) axoSender->connection->hebbLearningCondition,
							(int) axoSender->connection->competitiveLearningCondition,
							axoSender->connection->hebbIntervallCounter,
							axoSender->connection->synapticMemorySTP,
							axoSender->connection->stmCnt,
							axoSender->connection->condCnt, axoSender->connection->hebbCnt);

						queuePlaces = (int) rint(axoSender->connection->deliveryTime / STEP);

						fprintf(fp, "%e\n%i\n%i\n%i\n", axoSender->connection->TSmTRT_const,
							axoSender->connection->TTdSS_const, queuePlaces,
							axoSender->currentQueuePlace);

						for (index = 0; index < queuePlaces; index++)
							fprintf(fp, "%e\n", axoSender->processQueue[index]);
					}
				}
			}

			fprintf(fp, "%i\n", LTTdSS);

			if (synapseTypes[currentPool->typeOfConnection].ionType == 4)   /* electrical synapse ? */
				queuePlaces = 2;        /* only 2 queue entries are needed */
			else queuePlaces = LTTdSS;      /* maximum number of posible queue entries */

			fprintf(fp, "%i\n%i\n", queuePlaces, currentPool->currentQueuePlace);

			for (index = 0; index < queuePlaces; index++)
				fprintf(fp, "%e\n", currentPool->processQueue[index]);
		} /* End of for through the pools */

	} /* End of for loop through neurons */

	/* save currents */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n%e\n%e\n%e\n%e\n%e\n%e\n%i\n%i\n", currentNeuron->neuronId, currentNeuron->somaCurrent,
		    currentNeuron->somaSCStart, currentNeuron->somaSCEnd, currentNeuron->dendCurrent,
		    currentNeuron->dendSCStart, currentNeuron->dendSCEnd, currentNeuron->dendSCLocation,
		    (int) currentNeuron->noise);
	}

	/* close parameter file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_parameter */

/************************************************************************************************************************

 	FUNCTION	: save_configuration(filename)
 
	PURPOSE		: save a configuration file ( .con ) - network configuration

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_configuration(filename)
char filename[128];	/* filename for configuration file */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	AxoAxonicSenderList *axoSender;		/* current axo-axonic synapse */
	int     	index;			/* loop variable */
	Position 	x, y;			/* left-top window position */
	Dimension 	width, height;		/* width and height of window */
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* open configuration file and write out file type */

	if ((fp = fopen(filename,"w")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	fprintf(fp, "%i\n", CONFIGURATION_FILE);

	/* query window position and height */

	n = 0;
	XtSetArg(args[n], XmNx, &x); n++;
	XtSetArg(args[n], XmNy, &y); n++;
	XtSetArg(args[n], XmNwidth, &width); n++;
	XtSetArg(args[n], XmNheight, &height); n++;
	XtGetValues(applShell, args, n);

	/* write out window position and height */

	fprintf(fp, "%i\n%i\n%i\n%i\n", x, y, (int) x + (int) width, (int) y + (int) height);

	/* write out number of neurons, synapses, ... */

	fprintf(fp, "%i\n%i\n%i\n%i\n", neuronNumber, synapseNumber - axoSynapsesNumber, axoSynapsesNumber, numTypes);

	/* write out network model, type and number of dendritic segments actually used */

	index = (networkmodel << 1) - (int) (! complexModel);
	fprintf(fp, "%i\n%i\n%i\n", index, networktype, curNumOfDendSegs);

	/* save simulation parameters */

	fprintf(fp, "%e\n%e\n", simulation.length, simulation.step);

	index = 0;
	if (autoReset)
		if (showActivityColors)
			index = 3;
		else	index = 1;
	else if (showActivityColors)
		index = 2;

	/* write out global menu options */

	fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n%i\n",
		(int) XmToggleButtonGadgetGetState(simInfButton),
		(int) XmToggleButtonGadgetGetState(timeMessureButton),
		(int) XmToggleButtonGadgetGetState(graphDisplayButton),
		index,
		(int) XmToggleButtonGadgetGetState(outputWatchButton),
		(int) XmToggleButtonGadgetGetState(manualUpdateButton),
		(int) XmToggleButtonGadgetGetState(noiseButton),
		(int) XmToggleButtonGadgetGetState(initialPotentialsButton),
		(int) XmToggleButtonGadgetGetState(synapticStrengthButton),
		(int) XmToggleButtonGadgetGetState(synapticTransmissionTimeButton),
		integrationMethod);

	/* save global parameters */

	fprintf(fp, "%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n", noiseIntensity, noiseStrength, noiseDecay,
		stdInitialPotential, stdInitialPotentialStd, stdSynapticStrength, stdSynapticStrengthStd,
		stdAxoAxonicStrength, stdAxoAxonicStrengthStd, stdSynapticTT, stdSynapticTTStd);

	/* write out learning conditions */

	index = 0;
	if (hebbLearning)
		if (competitiveLearning)
			index = 3;
		else	index = 1;
	else if (competitiveLearning)
		index = 2;

	fprintf(fp, "%i\n%i\n%i\n", (int) plasticityLearning, (int) conditionedLearning, (int) index);


	/* Save Neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		fprintf(fp, "%i\n%i\n%i\n", currentNeuron->neuronId, (int) (currentNeuron->x + (int) rint((double)
			neuronDiameter / 2.0)), (int) (currentNeuron->y + (int) rint((double) neuronDiameter / 2.0)));
	}

	/* save synapse types */

	if (synapseTypes != NULL)
	{
		for (index = 0;index <= numTypes - 1; index++)
		{
			fprintf(fp, "%i\n%e\n", synapseTypes[index].ionType, synapseTypes[index].equilPotential);
		}
	}

	/* save synapses */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
		{
		     if (currentSynapse->axoSynapseNumber == 0)	/* synapse is not axo-axonic */
			fprintf(fp, "%i\n%i\n%i\n%i\n%e\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
				currentSynapse->synapseID, currentNeuron->neuronId, currentSynapse->targetCode,
				currentSynapse->targetArea, currentSynapse->deliveryTime, currentSynapse->synapticType,
				currentSynapse->initialConductance, currentSynapse->maxConductance,
				currentSynapse->transmitterReleaseThreshold, currentSynapse->transmitterSatiate,
				currentSynapse->slope, currentSynapse->shortTermMemory, currentSynapse->longTermThreshold,
				currentSynapse->longTermMinimum, currentSynapse->hebbIntervall,
				currentSynapse->hebbLearnFactor, currentSynapse->shortTermPlasticityLearnFactor,
				currentSynapse->competitiveLearnFactor);
		}/* end of going through synapses */
	} /* end of going through net and saving synapses */

	/* save axo-axonic synapses */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
	  for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
	        currentSynapse = currentSynapse->nextConnection)
	  {
	     if (currentSynapse->numberOfAxoSynapses > 0)	/* synapse has axo-axonic synapses */
	       for(axoSender = currentSynapse->axoSenderList; axoSender != NULL; axoSender = axoSender->nextSender)
		 {
		    fprintf(fp, "%i\n%i\n%i\n%i\n%i\n%i\n%e\n%i\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n%e\n",
			axoSender->connection->synapseID, axoSender->connection->targetSynapse->synapseID,
			axoSender->connection->parentNeuron->neuronId,
			axoSender->connection->targetSynapse->parentNeuron->neuronId,
			axoSender->connection->targetSynapse->targetNeuron->neuronId,
			axoSender->connection->targetSynapse->targetArea, axoSender->connection->deliveryTime,
			axoSender->connection->synapticType, axoSender->connection->transmitterReleaseThreshold,
			axoSender->connection->transmitterSatiate, axoSender->connection->initialInhibitionFactor,
			axoSender->connection->maxInhibitionFactor, axoSender->connection->slope,
			axoSender->connection->shortTermMemory, axoSender->connection->longTermThreshold,
			axoSender->connection->longTermMinimum, axoSender->connection->hebbIntervall,
			axoSender->connection->hebbLearnFactor, axoSender->connection->shortTermPlasticityLearnFactor,
			axoSender->connection->conditioningLearnFactor);
		 }
	  }/* end of going through synapses */
	}

	/* close configuration file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_configuration */

/************************************************************************************************************************

 	FUNCTION	: save_neuron(fp, neuron)
 
	PURPOSE		: save neuron parameter

	RETURNS		: nothing

************************************************************************************************************************/

void save_neuron(fp, neuron)
FILE *fp;		/* filepointer to current file */
Neuron *neuron;		/* current neuron */
{
	int length;	/* length of channel specific variables */
	int start;	/* start point of channel specific variables */
	int index;	/* loop variable */

	/* write out soma and dendrite variables */

	currentNeuron = neuron;

	length = SOMA_PLACE + SOMA_VARS + DENDRITE_VARS;
	start = SOMA_PLACE;

	for (index = start; index < length; index++)
	{
		fprintf(fp, "%e\n", currentNeuron->neuronParameters[index]);
	}

	/* write out channel specific variables */

	switch(networkmodel)
	{
	/* ON/OFF model */

	case 1:	length = ONOFF_PLACE + ONOFF_VARS;
		start = ONOFF_PLACE;
		break;
	
	/* SWIM model */

	case 2: length = SWIM_PLACE + SWIM_VARS;
		start = SWIM_PLACE;
		break;
	
	/* Hodgkin-Huxley model */

	case 3: length = HH_PLACE + HH_VARS;
		start = HH_PLACE;
		break;
	
	/* Golowasch-Buchholz model */

	case 4: length = GB_PLACE + GB_VARS;
		start = GB_PLACE;
		break;
	}

	for (index = start; index < length; index++)
	{
		fprintf(fp, "%e\n", currentNeuron->neuronParameters[index]);
	}

} /* end of save_neuron */

/************************************************************************************************************************

 	FUNCTION	: save_general_neuron_parameters(filename, neuron)
 
	PURPOSE		: save a general neuron parameter file ( .gen )

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_general_neuron_parameters(filename, neuron)
char filename[128];	/* filename of general neuron parameter file */
Neuron *neuron;		/* neuron to be saved */
{
	FILE		*fp = NULL;		/* file pointer to simulation file */
	int		index;			/* loop variable */


	/* open general neuron parameter file and write out file type */

	if ((fp = fopen(filename,"w")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	fprintf(fp, "%i\n", NEURON_COMBINED_FILE);

	/* write out soma and dendrite variables */

	currentNeuron = neuron;

	for (index = SOMA_PLACE; index < (SOMA_PLACE + SOMA_VARS + DENDRITE_VARS); index++)
	{
		fprintf(fp, "%e\n", currentNeuron->neuronParameters[index]);
	}

	/* close general neuron parameter file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_general_neuron_parameters */
#endif

/************************************************************************************************************************

 	FUNCTION	: save_graph(filename, neuron)
 
	PURPOSE		: save a neuron graph file ( .asc ) - graph traces from graph window

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_graph(filename, neuron)
char filename[128];	/* filename of graph file */
Neuron *neuron;	/* current neuron */
{
	int 		I;					/* loop variable */
	int 		timer;					/* used to compute output interval */
	Boolean		gateVars;				/* indicates, if gate variables are stored */
	Boolean		minMaxValues;				/* TRUE, if minimum and maximum values for output */
	char 		sepChar[2];				/* separation character */
	char		outputHeader[512];			/* output header line */
	char		*outHeader = NULL;			/* pointer to outputHeader */
	char		numberStr[256];				/* numbers of output line */
	char		*numStr = NULL;				/* pointer to numberStr */
	char 		timestep[20];				/* contains current simulation timestep */
	float		somaMin,
			somaMax,
			dendMin,
			dendMax,
			synCMin,
			synCMax,
			totCMin,
			totCMax,
			iNMin,
			iNMax,
			iNAMin,
			iNAMax,
			iKMin,
			iKMax,
			iCAMin,
			iCAMax,
			iCA1Min,
			iCA1Max,
			iCA2Min,
			iCA2Max,
			iKCAMin,
			iKCAMax,
			iAMin,
			iAMax,
			iIRMin,
			iIRMax,
			iPMin,
			iPMax;					/* values used for min/max evaluation */
	FILE		*fp = NULL;				/* file pointer to simulation file */

	currentNeuron = neuron;
	currentGraph = neuron->graph;
	gateVars = currentGraph->gateVariables;

	/* open graph file in append mode */

	if ((fp = fopen(filename,"a")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	sprintf(sepChar, "%c", asciiDefaults.separatorChar);

	/* write file identification number */

	if (STEP * 1.01 >= currentGraph->asciiDataOutputInterval)
	{
		currentGraph->asciiDataOutputInterval = STEP;
		fprintf(fp, "%i\n", OUTPUT_FILE);		/* output each time step a value */
		minMaxValues = FALSE;
	}
	else
	{
		fprintf(fp, "%i\n", OUTPUT_MINMAX_FILE);        /* output only min and max within a time interval */
		minMaxValues = TRUE;
	}

	/* write out the word Schrittweite and the current time step */

	strcpy(timestep, convert_number(currentGraph->asciiDataOutputInterval));
	fprintf(fp, "Schrittweite: %s\n", timestep);

	/* write out the word NEURON and the neuron number of the current neuron */

	fprintf(fp, "NEURON: %i\n", neuron->neuronId);

	/* build output header */

	strcpy(outputHeader, "");
	strcpy(numberStr, "");
	if (currentGraph->traceSoma)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, somaParString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) SOMA[0]));
		else	strcat(numberStr, convert_number((double) SOMA[0] / E_SCALE));
	}
	if (currentGraph->traceDendrit)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, dendritParString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) DENDRITE[0]));
		else	strcat(numberStr, convert_number((double) DENDRITE[0] / E_SCALE));
	}
	if (currentGraph->traceSynapticCurrent)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, synapticCurrentString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) SYNC[0]));
		else	strcat(numberStr, convert_number((double) SYNC[0] / I_SCALE));
	}
	if (currentGraph->traceTotalCurrent)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, totalCurrentString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) TOTC[0]));
		else	strcat(numberStr, convert_number((double) TOTC[0] / I_SCALE));
	}
	if (currentGraph->traceN)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, nChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) IN[0]));
		else	strcat(numberStr, convert_number((double) IN[0] / I_SCALE));
	}
	if (currentGraph->traceNA)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, naChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) INA[0]));
		else	strcat(numberStr, convert_number((double) INA[0] / I_SCALE));
	}
	if (currentGraph->traceK)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, kChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) IK[0]));
		else	strcat(numberStr, convert_number((double) IK[0] / I_SCALE));
	}
	if (currentGraph->traceCA)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, caChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) ICA[0]));
		else	strcat(numberStr, convert_number((double) ICA[0] / I_SCALE));
	}
	if (currentGraph->traceCA1)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, ca1ChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars || networkmodel < 4)
			strcat(numberStr, convert_number((double) ICA1[0]));
		else	strcat(numberStr, convert_number((double) ICA1[0] / I_SCALE));
	}
	if (currentGraph->traceCA2)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, ca2ChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars || networkmodel < 4)
			strcat(numberStr, convert_number((double) ICA2[0]));
		else	strcat(numberStr, convert_number((double) ICA2[0] / I_SCALE));
	}
	if (currentGraph->traceKCA)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, kcaChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars || networkmodel < 4)
			strcat(numberStr, convert_number((double) IKCA[0]));
		else	strcat(numberStr, convert_number((double) IKCA[0] / I_SCALE));
	}
	if (currentGraph->traceA)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, aChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars || networkmodel < 4)
			strcat(numberStr, convert_number((double) IA[0]));
		else	strcat(numberStr, convert_number((double) IA[0] / I_SCALE));
	}
	if (currentGraph->traceIR)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, irChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) IIR[0]));
		else	strcat(numberStr, convert_number((double) IIR[0] / I_SCALE));
	}
	if (currentGraph->traceP)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, pChannelsString);
		strcat(numberStr, sepChar);
		if (gateVars)
			strcat(numberStr, convert_number((double) IP[0]));
		else	strcat(numberStr, convert_number((double) IP[0] / I_SCALE));
	}

	/* eliminates the leading separation character */

	numStr = numberStr;
	numStr++;

	outHeader = outputHeader;
	outHeader++;

	/* writes out values at time zero (beginning of simulation) */

	fprintf(fp, "%s\n", outHeader);
	if (!minMaxValues)
		fprintf(fp, "%s\n", numStr);

	/* initialize min/max values */

	somaMin = somaMax = SOMA[0];
	dendMin = dendMax = DENDRITE[0];
	synCMin = synCMax = SYNC[0];
	totCMin = totCMax = TOTC[0];
	iNMin = iNMax = IN[0];
	iNAMin = iNAMax = INA[0];
	iKMin = iKMax = IK[0];
	iCAMin = iCAMax = ICA[0];
	iCA1Min = iCA1Max = ICA1[0];
	iCA2Min = iCA2Max = ICA2[0];
	iKCAMin = iKCAMax = IKCA[0];
	iAMin = iAMax = IA[0];
	iIRMin = iIRMax = IIR[0];
	iPMin = iPMax = IP[0];

	/* writes out values from the first simulation step to the last simulation step with an interval given by
	   asciiDataOutputInterval */

	timer = 0;
	for (I = 1; I <= (simulationTime - 1); I++)
	{
		timer++;
		if (minMaxValues)	/* only minimum and maximum within a interval should be outputed */
		{
			if (STEP * ((double) timer + 0.1) >= currentGraph->asciiDataOutputInterval)
			{
				/* build output header */

				timer = 0;

				strcpy(numberStr, "");
				if (currentGraph->traceSoma)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) somaMin));
					else	strcat(numberStr, convert_number((double) somaMin / E_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) somaMax));
					else	strcat(numberStr, convert_number((double) somaMax / E_SCALE));
				}
				if (currentGraph->traceDendrit)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) dendMin));
					else	strcat(numberStr, convert_number((double) dendMin / E_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) dendMax));
					else	strcat(numberStr, convert_number((double) dendMax / E_SCALE));
				}
				if (currentGraph->traceSynapticCurrent)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) synCMin));
					else	strcat(numberStr, convert_number((double) synCMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) synCMax));
					else	strcat(numberStr, convert_number((double) synCMax / I_SCALE));
				}
				if (currentGraph->traceTotalCurrent)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) totCMin));
					else	strcat(numberStr, convert_number((double) totCMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) totCMax));
					else	strcat(numberStr, convert_number((double) totCMax / I_SCALE));
				}
				if (currentGraph->traceN)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iNMin));
					else	strcat(numberStr, convert_number((double) iNMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iNMax));
					else	strcat(numberStr, convert_number((double) iNMax / I_SCALE));
				}
				if (currentGraph->traceNA)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iNAMin));
					else	strcat(numberStr, convert_number((double) iNAMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iNAMax));
					else	strcat(numberStr, convert_number((double) iNAMax / I_SCALE));
				}
				if (currentGraph->traceK)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iKMin));
					else	strcat(numberStr, convert_number((double) iKMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iKMax));
					else	strcat(numberStr, convert_number((double) iKMax / I_SCALE));
				}
				if (currentGraph->traceCA)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iCAMin));
					else	strcat(numberStr, convert_number((double) iCAMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iCAMax));
					else	strcat(numberStr, convert_number((double) iCAMax / I_SCALE));
				}
				if (currentGraph->traceCA1)
				{
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iCA1Min));
					else	strcat(numberStr, convert_number((double) iCA1Min / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iCA1Max));
					else	strcat(numberStr, convert_number((double) iCA1Max / I_SCALE));
				}
				if (currentGraph->traceCA2)
				{
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iCA2Min));
					else	strcat(numberStr, convert_number((double) iCA2Min / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iCA2Max));
					else	strcat(numberStr, convert_number((double) iCA2Max / I_SCALE));
				}
				if (currentGraph->traceKCA)
				{
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iKCAMin));
					else	strcat(numberStr, convert_number((double) iKCAMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iKCAMax));
					else	strcat(numberStr, convert_number((double) iKCAMax / I_SCALE));
				}
				if (currentGraph->traceA)
				{
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iAMin));
					else	strcat(numberStr, convert_number((double) iAMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars || networkmodel < 4)
						strcat(numberStr, convert_number((double) iAMax));
					else	strcat(numberStr, convert_number((double) iAMax / I_SCALE));
				}
				if (currentGraph->traceIR)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iIRMin));
					else	strcat(numberStr, convert_number((double) iIRMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iIRMax));
					else	strcat(numberStr, convert_number((double) iIRMax / I_SCALE));
				}
				if (currentGraph->traceP)
				{
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iPMin));
					else	strcat(numberStr, convert_number((double) iPMin / I_SCALE));
					strcat(numberStr, sepChar);
					if (gateVars)
						strcat(numberStr, convert_number((double) iPMax));
					else	strcat(numberStr, convert_number((double) iPMax / I_SCALE));
				}

				/* eliminates the leading separation character */

				numStr = numberStr;
				numStr++;

				/* writes out values */

				fprintf(fp, "%s\n", numStr);

				/* initialize min/max values */

				somaMin = somaMax = SOMA[I];
				dendMin = dendMax = DENDRITE[I];
				synCMin = synCMax = SYNC[I];
				totCMin = totCMax = TOTC[I];
				iNMin = iNMax = IN[I];
				iNAMin = iNAMax = INA[I];
				iKMin = iKMax = IK[I];
				iCAMin = iCAMax = ICA[I];
				iCA1Min = iCA1Max = ICA1[I];
				iCA2Min = iCA2Max = ICA2[I];
				iKCAMin = iKCAMax = IKCA[I];
				iAMin = iAMax = IA[I];
				iIRMin = iIRMax = IIR[I];
				iPMin = iPMax = IP[I];
			}
			else	/* update min/max values */
			{
				if (currentGraph->traceSoma)
					if (SOMA[I] < somaMin)
						somaMin = SOMA[I];
					else if (SOMA[I] > somaMax)
						somaMax = SOMA[I];

				if (currentGraph->traceDendrit)
					if (DENDRITE[I] < dendMin)
						dendMin = DENDRITE[I];
					else if (DENDRITE[I] > dendMax)
						dendMax = DENDRITE[I];

				if (currentGraph->traceSynapticCurrent)
					if (SYNC[I] < synCMin)
						synCMin = SYNC[I];
					else if (SYNC[I] > synCMax)
						synCMax = SYNC[I];

				if (currentGraph->traceTotalCurrent)
					if (TOTC[I] < totCMin)
						totCMin = TOTC[I];
					else if (TOTC[I] > totCMax)
						totCMax = TOTC[I];

				if (currentGraph->traceN)
					if (IN[I] < iNMin)
						iNMin = IN[I];
					else if (IN[I] > iNMax)
						iNMax = IN[I];

				if (currentGraph->traceNA)
					if (INA[I] < iNAMin)
						iNAMin = INA[I];
					else if (INA[I] > iNAMax)
						iNAMax = INA[I];

				if (currentGraph->traceK)
					if (IK[I] < iKMin)
						iKMin = IK[I];
					else if (IK[I] > iKMax)
						iKMax = IK[I];

				if (currentGraph->traceCA)
					if (ICA[I] < iCAMin)
						iCAMin = ICA[I];
					else if (ICA[I] > iCAMax)
						iCAMax = ICA[I];

				if (currentGraph->traceCA1)
					if (ICA1[I] < iCA1Min)
						iCA1Min = ICA1[I];
					else if (ICA1[I] > iCA1Max)
						iCA1Max = ICA1[I];

				if (currentGraph->traceCA2)
					if (ICA1[I] < iCA2Min)
						iCA2Min = ICA2[I];
					else if (ICA1[I] > iCA2Max)
						iCA2Max = ICA2[I];

				if (currentGraph->traceKCA)
					if (IKCA[I] < iKCAMin)
						iKCAMin = IKCA[I];
					else if (IKCA[I] > iKCAMax)
						iKCAMax = IKCA[I];

				if (currentGraph->traceA)
					if (IA[I] < iAMin)
						iAMin = IA[I];
					else if (IA[I] > iAMax)
						iAMax = IA[I];

				if (currentGraph->traceIR)
					if (IIR[I] < iIRMin)
						iIRMin = IIR[I];
					else if (IIR[I] > iIRMax)
						iIRMax = IIR[I];

				if (currentGraph->traceP)
					if (IP[I] < iPMin)
						iPMin = IP[I];
					else if (IP[I] > iPMax)
						iPMax = IP[I];
			}
		}
		else	/* each time step should be written an output value */
		{
			/* build output header */

			strcpy(numberStr, "");
			if (currentGraph->traceSoma)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) SOMA[I]));
				else	strcat(numberStr, convert_number((double) SOMA[I] / E_SCALE));
			}
			if (currentGraph->traceDendrit)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) DENDRITE[I]));
				else	strcat(numberStr, convert_number((double) DENDRITE[I] / E_SCALE));
			}
			if (currentGraph->traceSynapticCurrent)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) SYNC[I]));
				else	strcat(numberStr, convert_number((double) SYNC[I] / I_SCALE));
			}
			if (currentGraph->traceTotalCurrent)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) TOTC[I]));
				else	strcat(numberStr, convert_number((double) TOTC[I] / I_SCALE));
			}
			if (currentGraph->traceN)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) IN[I]));
				else	strcat(numberStr, convert_number((double) IN[I] / I_SCALE));
			}
			if (currentGraph->traceNA)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) INA[I]));
				else	strcat(numberStr, convert_number((double) INA[I] / I_SCALE));
			}
			if (currentGraph->traceK)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) IK[I]));
				else	strcat(numberStr, convert_number((double) IK[I] / I_SCALE));
			}
			if (currentGraph->traceCA)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) ICA[I]));
				else	strcat(numberStr, convert_number((double) ICA[I] / I_SCALE));
			}
			if (currentGraph->traceCA1)
			{
				strcat(numberStr, sepChar);
				if (gateVars || networkmodel < 4)
					strcat(numberStr, convert_number((double) ICA1[I]));
				else	strcat(numberStr, convert_number((double) ICA1[I] / I_SCALE));
			}
			if (currentGraph->traceCA2)
			{
				strcat(numberStr, sepChar);
				if (gateVars || networkmodel < 4)
					strcat(numberStr, convert_number((double) ICA2[I]));
				else	strcat(numberStr, convert_number((double) ICA2[I] / I_SCALE));
			}
			if (currentGraph->traceKCA)
			{
				strcat(numberStr, sepChar);
				if (gateVars || networkmodel < 4)
					strcat(numberStr, convert_number((double) IKCA[I]));
				else	strcat(numberStr, convert_number((double) IKCA[I] / I_SCALE));
			}
			if (currentGraph->traceA)
			{
				strcat(numberStr, sepChar);
				if (gateVars || networkmodel < 4)
					strcat(numberStr, convert_number((double) IA[I]));
				else	strcat(numberStr, convert_number((double) IA[I] / I_SCALE));
			}
			if (currentGraph->traceIR)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) IIR[I]));
				else	strcat(numberStr, convert_number((double) IIR[I] / I_SCALE));
			}
			if (currentGraph->traceP)
			{
				strcat(numberStr, sepChar);
				if (gateVars)
					strcat(numberStr, convert_number((double) IP[I]));
				else	strcat(numberStr, convert_number((double) IP[I] / I_SCALE));
			}

			/* eliminates the leading separation character */

			numStr = numberStr;
			numStr++;

			/* writes out values */

			fprintf(fp, "%s\n", numStr);
		}
	}

	/* close graph file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_graph */

/************************************************************************************************************************

 	FUNCTION	: save_graphS(filename, synapse)
 
	PURPOSE		: save a synapse graph file ( .asc ) - graph traces from graph window

	RETURNS		: TRUE, if save operation produces no error, else FALSE

************************************************************************************************************************/

Boolean save_graphS(filename, synapse)
char filename[128];		/* filename of graph file */
Connection *synapse;		/* current synapse */
{
	int 		I;					/* loop variable */
	int 		timer;					/* used to compute output interval */
	Boolean		minMaxValues;				/* TRUE, if minimum and maximum values for output */
	char 		sepChar[2];				/* separation character */
	char		outputHeader[128];			/* output header line */
	char		*outHeader = NULL;			/* pointer to outputHeader */
	char		numberStr[256];				/* numbers of output line */
	char		*numStr = NULL;				/* pointer to numberStr */
	char 		timestep[20];				/* contains current simulation timestep */
	float		potMin,
			potMax,
			memMin,
			memMax,
			conMin,
			conMax,
			strMin,
			strMax;					/* values used for min/max evaluation */
	FILE		*fp = NULL;				/* file pointer to simulation file */

	currentSynapse = synapse;
	currentGraphS = synapse->graph;

	/* open graph file in write mode */

	if ((fp = fopen(filename,"a")) == NULL)
	{
		error(115);
		return(FALSE);
	}

	sprintf(sepChar, "%c", asciiDefaults.separatorChar);

	/* write file identification number */

	if (STEP * 1.01 >= currentGraphS->asciiDataOutputInterval)
	{
		currentGraphS->asciiDataOutputInterval = STEP;
		fprintf(fp, "%i\n", S_OUTPUT_FILE);		/* output each time step a value */
		minMaxValues = FALSE;
	}
	else
	{
		fprintf(fp, "%i\n", S_OUTPUT_MINMAX_FILE);        /* output only min and max within a time interval */
		minMaxValues = TRUE;
	}

	/* write out the word Schrittweite and the current time step */

	strcpy(timestep, convert_number(currentGraphS->asciiDataOutputInterval));
	fprintf(fp, "Schrittweite: %s\n", timestep);

	/* write out the word SYNAPSE and the synapse number of the current synapse */

	fprintf(fp, "SYNAPSE: %i\n", synapse->synapseID);

	/* build output header */

	strcpy(outputHeader, "");
	strcpy(numberStr, "");
	if (currentGraphS->tracePot)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, synPotString);
		strcat(numberStr, sepChar);
		strcat(numberStr, convert_number((double) POT[0] / E_SCALE));
	}
	if (currentGraphS->traceMem)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, memString);
		strcat(numberStr, sepChar);
		strcat(numberStr, convert_number((double) MEM[0] / MEM_SCALE));
	}
	if (currentGraphS->traceCon)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, condString);
		strcat(numberStr, sepChar);
		strcat(numberStr, convert_number((double) CON[0] / CON_SCALE));
	}
	if (currentGraphS->traceStr)
	{
		strcat(outputHeader, sepChar);
		strcat(outputHeader, strString);
		strcat(numberStr, sepChar);
		strcat(numberStr, convert_number((double) STR[0] / STR_SCALE));
	}

	/* eliminates the leading separation character */

	numStr = numberStr;
	numStr++;

	outHeader = outputHeader;
	outHeader++;

	/* writes out values at time zero (beginning of simulation) */

	fprintf(fp, "%s\n", outHeader);
	if (!minMaxValues)
		fprintf(fp, "%s\n", numStr);

	/* initialize min/max values */

	potMin = potMax = POT[0];
	memMin = memMax = MEM[0];
	conMin = conMax = CON[0];
	strMin = strMax = STR[0];

	/* writes out values from the first simulation step to the last simulation step with an interval given by
	   asciiDataOutputInterval */

	timer = 0;
	for (I = 1; I <= (simulationTime - 1); I++)
	{
		timer++;
		if (minMaxValues)	/* only minimum and maximum within a interval should be outputed */
		{
			if (STEP * ((double) timer + 0.1) >= currentGraphS->asciiDataOutputInterval)
			{
				/* build output header */

				timer = 0;

				strcpy(numberStr, "");
				if (currentGraphS->tracePot)
				{
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) potMin / E_SCALE));
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) potMax / E_SCALE));
				}
				if (currentGraphS->traceMem)
				{
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) memMin / MEM_SCALE));
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) memMax / MEM_SCALE));
				}
				if (currentGraphS->traceCon)
				{
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) conMin / CON_SCALE));
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) conMax / CON_SCALE));
				}
				if (currentGraphS->traceStr)
				{
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) strMin / STR_SCALE));
					strcat(numberStr, sepChar);
					strcat(numberStr, convert_number((double) strMax / STR_SCALE));
				}

				/* eliminates the leading separation character */

				numStr = numberStr;
				numStr++;

				/* writes out values */

				fprintf(fp, "%s\n", numStr);

				/* initialize min/max values */

				potMin = potMax = POT[I];
				memMin = memMax = MEM[I];
				conMin = conMax = CON[I];
				strMin = strMax = STR[I];
			}
			else	/* update min/max values */
			{
				if (currentGraphS->tracePot)
					if (POT[I] < potMin)
						potMin = POT[I];
					else if (POT[I] > potMax)
						potMax = POT[I];

				if (currentGraphS->traceMem)
					if (MEM[I] < memMin)
						memMin = MEM[I];
					else if (MEM[I] > memMax)
						memMax = MEM[I];

				if (currentGraphS->traceCon)
					if (CON[I] < conMin)
						conMin = CON[I];
					else if (CON[I] > conMax)
						conMax = CON[I];

				if (currentGraphS->traceStr)
					if (STR[I] < strMin)
						strMin = STR[I];
					else if (STR[I] > strMax)
						strMax = STR[I];
			}
		}
		else	/* each time step should be written an output value */
		{
			/* build output header */

			strcpy(numberStr, "");
			if (currentGraphS->tracePot)
			{
				strcat(numberStr, sepChar);
				strcat(numberStr, convert_number((double) POT[I] / E_SCALE));
			}
			if (currentGraphS->traceMem)
			{
				strcat(numberStr, sepChar);
				strcat(numberStr, convert_number((double) MEM[I] / MEM_SCALE));
			}
			if (currentGraphS->traceCon)
			{
				strcat(numberStr, sepChar);
				strcat(numberStr, convert_number((double) CON[I] / CON_SCALE));
			}
			if (currentGraphS->traceStr)
			{
				strcat(numberStr, sepChar);
				strcat(numberStr, convert_number((double) STR[I] / STR_SCALE));
			}

			/* eliminates the leading separation character */

			numStr = numberStr;
			numStr++;

			/* writes out values */

			fprintf(fp, "%s\n", numStr);
		}
	}

	/* close graph file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of save_graphS */
