/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	fileopen.c

	PURPOSE:	this file contains all functions that handles with file read operations

	FUNCTIONS:	in order of appearance

	read_input_currents			- reads input current file and stores values in neurons
	read_general_neuron_parameters		- reads a general neuron parameter file ( .gen )
	read_network				- reads a simulation file ( .sim ) containing all information
	read_parameter				- reads a parameter file ( .par ) - network parameter
	read_configuration			- reads a configuration file ( .con ) - network configuration
	readf					- general read function, reads buffer wise a file
	get_neuron				- reads the neuron parameter
	read_on_off_channels			- read in on/off channel parameters
	read_swim_channels			- read in SWIM channel parameters
	read_HH_channels			- read in Hodgkin-Huxley channel parameters
	read_golowasch_channels			- read in Golowasch-Buchholz channel parameters

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/****** global variables ******/

ProgramStates programStates = { 1, 0 };	/* program states for reading files */

/****** functions ******/

/************************************************************************************************************************

 	FUNCTION	: read_input_currents()
 
	PURPOSE		: reads input current file and stores values in neurons

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_input_currents()
{
	Boolean neuronFound;		/* status of searched neuron for current input */
	int	i, j, k;		/* help variables */
	int	fileType;		/* identification number that identifies the file type */
	int	neuronId;		/* neuron number */
	int	location;		/* compartment number - location on which the current should be injected */
	int	entries;		/* number of current entries in the file */
	int	listEntries;		/* number of current pulses for each current entrie */
	FILE	*fp;			/* file pointer to input current file */
	char	*buffer = NULL;		/* input buffer for current entries */
	char	*stringPointer = NULL;	/* pointer to an entrie in the input buffer */
	char	fileName[128];		/* filename of the corresponding network file */

		/* attempt to open input current file, if this fails return with FALSE, but no error message */

		if ((fp = fopen(inputCurrentsFile, "r")) == NULL)
			return(FALSE);

		/* read filetype and check if its the id for a current file */

		if (fscanf(fp, "%i", &fileType) == 0)
			return(FALSE);
		if (fileType != CURRENT_FILE)
		{
			error(100);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		/* read in filename of the correspondin network file to which this file belongs */

		if (fscanf(fp, "%s", fileName) == 0)
			return(FALSE);

		/* if the current filename is not the same as the filename just read than this file should not be used */

		if (strcmp(fileName, currentFilename) != 0)
			return(TRUE);	/* file is not relevant for the current simulation, so return TRUE */

		/* read in the number of current entries in the file */

		if (fscanf(fp, "%i", &entries) == 0)
			return(FALSE);
		if (entries < 1)
			return(FALSE);

		/* loop through all current entries in the file */

		for (k = 1; k <= entries; k++)
		{
			/* read in data for one entrie */

			if (fscanf(fp, "%i%i%i ", &neuronId, &location, &listEntries) != 3)
				return(FALSE);
			if (location > curNumOfDendSegs || location < 0)	/* validate current location */
			{
				error(113);
				return(FALSE);
			}

			/* search target neuron in the neuron list */

			neuronFound = FALSE;
			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
				if (currentNeuron->neuronId == neuronId)
				{
					neuronFound = TRUE;
					break;
				}
			
			/* if neuron could not be found print a warning message and ignore input */

			if (! neuronFound)
			{
				char tempString[128];	/* temporare string */
#ifndef BATCH
				XmString text;		/* text for error message */
#endif

				sprintf(tempString, "%s %d", wrongNeuronNumberString, neuronId);
#ifndef BATCH
				text = XmStringCreate(tempString, defaultFont);
				XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, text));
				XmStringFree(text);
#else
				fprintf(stderr, tempString);
#endif

				/* allocate memory for the input buffer */

				buffer = (char *) malloc((size_t) (sizeof(char) * listEntries * 2 * NumberSize));
				if (! buffer)
				{
					error(114);
					return(FALSE);
				}

				/* read in the rest of the current line, this contains the time and current entries */

				if (! fgets(buffer, listEntries * 2 * NumberSize, fp))
				{
					error(113);
					return(FALSE);
				}
				break;	/* leave for loop -> go to next entry */
			}

			/* initialize the current entries in the neuron */

			if (TotCur != 1234567890.0)	/* neuron was not previously initialized, so initialize it */
			{
				for (i = 0; i <= curNumOfDendSegs; i++)
				{
					if (ICUR[i].starttime != NULL)
						free(ICUR[i].starttime);
					if (ICUR[i].current != NULL)
						free(ICUR[i].current);
					ICUR[i].starttime = (double *) malloc((size_t) (sizeof(double) * (listEntries + 2)));

					if (! ICUR[i].starttime)
					{
						error(114);
						return(FALSE);
					}

					ICUR[i].current = (double *) malloc((size_t) (sizeof(double) * (listEntries + 2)));

				if (! ICUR[i].current)
				{
					error(114);
					return(FALSE);
				}

#ifndef BATCH
				currentNeuron->neuronSize -= currentNeuron->inpCurSize;
				currentNeuron->inpCurSize = (unsigned int) (2 * (sizeof(double) * (listEntries + 2)));
				currentNeuron->neuronSize += currentNeuron->inpCurSize;
#endif

				ICUR[i].numEntries = listEntries + 2;
				ICUR[i].index = 0;
				ICUR[i].starttime[0] = 0.0;
				ICUR[i].current[0] = 0.0;
				for (j = 1; j <= (listEntries + 1); j++)
				{
					ICUR[i].starttime[j] = simulation.length;
					ICUR[i].current[j] = 0.0;
				}
			}
			TotCur = 1234567890.0;	/* mark neuron as initialized */
		}

		/* allocate memory for the input buffer */

		buffer = (char *) malloc((size_t) (sizeof(char) * listEntries * 2 * NumberSize));
		if (! buffer)
		{
			error(114);
			return(FALSE);
		}

		/* read in the rest of the current line, this contains the time and current entries */

		if (! fgets(buffer, listEntries * 2 * NumberSize, fp))
		{
			error(113);
			return(FALSE);
		}

		/* store the tuples (current start time, current value) in the neuron */

		i = 0;
		stringPointer = buffer;
		for (j = 1; j <= listEntries; j++)
		{
			while (buffer[i] != ' ' && buffer[i] != '\n')
				i++;
			buffer[i] = EOS;
			ICUR[location].starttime[j] = (double) (atof(stringPointer) / 1e3);
			stringPointer = &buffer[++i];
			while (buffer[i] != ' ' && buffer[i] != '\n')
				i++;
			buffer[i] = EOS;
			ICUR[location].current[j] = (double) (atof(stringPointer) / 1e9);
			stringPointer = &buffer[++i];
		}
		free(buffer);
	}

	/* close the input current file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of read_input_currents */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: read_general_neuron_parameters(filename, neuron)
 
	PURPOSE		: reads a general neuron parameter file ( .gen )

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_general_neuron_parameters(filename, neuron)
char filename[128];	/* name of the general parameter file */
Neuron *neuron;		/* neuron for which the parameters are */
{
	char    	charRead[30];	/* current number read */
	int     	fileType;	/* identification number which identifies the file type */
	FILE		*fp = NULL;	/* file pointer to general parameter file */

	/* attempt to open general parameter file, if this fails return with FALSE */

	if ((fp = fopen(filename, "r")) == NULL)
	{
		error(115);
		return(FALSE);
	}
	programStates.newRead = TRUE;

	/* read filetype and check if its the id for a current file */

	if (! readf(fp, charRead))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	fileType = atoi(charRead);

	if (fileType != NEURON_COMBINED_FILE)
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	/* get neuron parameters */

	get_neuron(fp, neuron, -1, fileType);

	/* close the general parameter file */

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of read_general_neuron_parameters */
#endif

/************************************************************************************************************************

 	FUNCTION	: read_network(filename)
 
	PURPOSE		: reads a simulation file ( .sim ) containing all information

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_network(filename)
char filename[128];	/* filename of the simulation file */
{
	int		n;			/* used as argument counter for manipulating widgets */
#ifndef BATCH
	Arg		args[10];		/* argument list for manipulating widgets */
#endif
	char    	charRead[100];		/* current number read */
	char		val[40];		/* number as string */
	char		tmp[10];		/* temporary string */
	char		*value;			/* temporary character pointer */
	Neuron		*sourceNeuron;		/* source neuron */
	Neuron		*targetNeuron;		/* target neuron */
	Neuron		*sourceNeuronTargetSynapse; /* source neuron for axo-axonic connections */
	Neuron		*targetNeuronTargetSynapse; /* target neuron for axo-axonic connections */
	int		fileType;		/* identification number which identifies the file type */
	int     	j;			/* loop variable */
	int 		neuronId;		/* neuron number */
	int 		synapseId;		/* synapse number */
	int 		neuronX, neuronY;	/* neuron position */
	int		totalNeurons;		/* number of neurons in the net */
	int		totalSynapses;		/* number of synapses in the net */
	int		totalAxoSynapses;	/* number of axo-axonic synapses in the net */
	int		totalSynapseTypes;	/* number of synaptic types in the net */
	int		totalCurrents;		/* number of input currents in the net */
	int		totalGraphs;		/* number of neuron graphs in the net */
	int		totalSGraphs;		/* number of synapse graphs in the net */
	int		totalASGraphs;		/* number of axo-axonic synapse graphs in the net */
	int		highestNeuronID;	/* highest neuron number read */
	int		gridType;		/* flag for grid graph: 0 = no grid, 1 = dots, 2 = lines */
	int		axis;			/* flag for grid axis: 0 = no grid, 1 = x-axis, 2 = y-axis, 3 = both */
#ifndef BATCH
	int		simInfState,
			graphDisplayState,
			autoResetState,
			neuronActivityColorsState,
			outputWatchState,
			manualUpdateState,
			initialPotentialsState,
			synapticStrengthState,
			synapticTransmissionTimeState,
			noiseState;		/* state of the global menu option buttons */
#endif
	unsigned int	graphLength;		/* length of current graph */
	double		asciiInterval;		/* time interval for writing out graph data during graph save */
	double		trt;			/* transmitter release threshold */
	double		trs;			/* transmitter satiate */
#ifndef BATCH
	Position 	rc_left, rc_top;	/* left-top graph window position */
	Position 	x, y;			/* offset to drawing area of graph window */
	Dimension 	rc_right, rc_bottom;	/* right-bottom window position */
	Dimension 	width, height;		/* width and height of graph window */
	Dimension	screenWidth, screenHeight; /* width and height of screen */
	double		firstX;			/* begin of graph in ms */
#endif
	double		lastX;			/* end of graph in ms */
#ifndef BATCH
	double		xScale;			/* pixels per ms for x-axis */
	double		yScale;			/* pixels per ms for y-axis */
	double		highestY;		/* highest value of y-axis in mv */
	double		lowestY;		/* lowest value of y-axis in mv */
	double		yStep;			/* save as yScale, but not longer needed */
#endif
	int		traceSoma,
			traceDendrit,
			traceSynapticCurrent,
			traceTotalCurrent,
			traceN,
			traceNA,
			traceK,
			traceCA,
			traceCA1,
			traceCA2,
			traceKCA,
			traceA,
			traceIR,
			traceP,
			tracePot,
			traceMem,
			traceCon,
			traceStr;		/* flags for traces */
#ifndef BATCH
	Region		region;			/* graphical region for neurons */
	XRectangle	rectangle;		/* rectangle for neuron region */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
#endif
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* clear current network */

	clear_network();
	variables_initialized = FALSE;
	programStates.newRead = TRUE;

	/* open file given by filename */

	if ((fp = fopen(filename, "r+")) == NULL)
		if ((fp = fopen(filename, "r")) != NULL)     /* create warning dialog box, file is read only */
			error(116);
		else	/* create error dialog box, cause selected file cannot be opened */
		{
			error(1);
			return(FALSE);
		}

	/* read filetype and check if its the id for a simulation file */

	if (! readf(fp, charRead))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	fileType = atoi(charRead);

	if (fileType != NETWORK_FILE)
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	/* get screen size */

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	rc_left = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	rc_top = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	rc_right = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	rc_bottom = atoi(charRead);
#endif

#ifndef BATCH
	/* set screen size */

	n = 0;
	XtSetArg(args[n], XmNx, rc_left); n++;
	XtSetArg(args[n], XmNy, rc_top); n++;
	XtSetArg(args[n], XmNwidth, rc_right - (Dimension) rc_left); n++;
	XtSetArg(args[n], XmNheight, rc_bottom - (Dimension) rc_top); n++;
	XtSetValues(applShell, args, n);

	/* get number of items in net */
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalNeurons = atoi(charRead);
	highestNeuronID = totalNeurons;

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalSynapses = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalAxoSynapses = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalSynapseTypes = atoi(charRead);
	numTypes = totalSynapseTypes;

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalCurrents = atoi(charRead);

#ifndef BATCH
	DendriteLength = defaultDendriteLength;		/* set dendrite length to the default value */
#endif

	/* read number of neuron graphs */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalGraphs = atoi(charRead);

	/* read number of synapse graphs */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalSGraphs = atoi(charRead);

	/* read number of axo-axonic synapse graphs */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalASGraphs = atoi(charRead);

	/* read network model and type */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);
	complexModel = ((j & 1) == 0);
	networkmodel = (j >> 1) + (int) (! complexModel);

#ifndef BATCH
	n = 0;
	if (complexModel)	/* complex model ? */
	{
		XtSetArg(args[n], XmNsensitive, TRUE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}
	else
	{
		XtSetArg(args[n], XmNsensitive, FALSE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	networktype = atoi(charRead); /* 1, 2, 4, N- point model */

	switch(networktype)
	{
		case 1:	/* 1-point model */
			curNumOfDendSegs = 0;
			break;

		case 2:	/* 2-point model */
			curNumOfDendSegs = 1;
			break;

		case 3:	/* 4-point model */
			curNumOfDendSegs = 3;
			break;

		case 4:	/* n-point model */
			curNumOfDendSegs = 3;
			break;
	}

	/* read in number of dendritic segments currently used */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	curNumOfDendSegs = atoi(charRead);

#ifndef BATCH
	if (networktype == 4)
		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs);
	else	DendriteLength = defaultDendriteLength;

	/* set maximum scale widgets of add synapse, edit synapse, edit current mask to curNumOfDendSegs */

	if (networktype == 1)	/* 1-Point model */
	{
		/* set scale bar of add synapse mask, edit synapse mask and edit current mask to insensitive */

		XmTextSetString(addSynapseTargetText, "0");
		XmTextSetString(editSynapseLocationText, "0");
		XmTextSetString(editCurrentLocationText, "0");
		XmScaleSetValue(addSynapseTargetScale, 0);
		XmScaleSetValue(editSynapseLocationScale, 0);
		XmScaleSetValue(editCurrentLocationScale, 0);

		XtSetSensitive(addSynapseTargetScale, FALSE);
		XtSetSensitive(editSynapseLocationScale, FALSE);
		XtSetSensitive(editCurrentLocationScale, FALSE);

		show_normal_state();
		show_all_objects_in_drawing_area(FALSE);
	}
	else	set_max_scale();

	/* set buttons in network model mask */

	XmToggleButtonGadgetSetState(networkOnOffButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkSWIMButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkHHButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkGBButton, FALSE, FALSE);

	if (complexModel)	/* complex model ? */
		XmToggleButtonGadgetSetState(networkComplexButton, TRUE, FALSE);
	else	XmToggleButtonGadgetSetState(networkComplexButton, FALSE, FALSE);

	switch (networkmodel)
	{
	case 1: XmToggleButtonGadgetSetState(networkOnOffButton, TRUE, FALSE);
		break;

	case 2: XmToggleButtonGadgetSetState(networkSWIMButton, TRUE, FALSE);
		break;

	case 3: XmToggleButtonGadgetSetState(networkHHButton, TRUE, FALSE);
		break;

	case 4: XmToggleButtonGadgetSetState(networkGBButton, TRUE, FALSE);
		break;
	}

	XmToggleButtonGadgetSetState(networkType1Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType2Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType4Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkTypeNButton, FALSE, FALSE);

	switch (networktype)
	{
		case 1:	XmToggleButtonGadgetSetState(networkType1Button, TRUE, FALSE);
			break;
		case 2:	XmToggleButtonGadgetSetState(networkType2Button, TRUE, FALSE);
			break;
		case 3:	XmToggleButtonGadgetSetState(networkType4Button, TRUE, FALSE);
			break;
		case 4:	XmToggleButtonGadgetSetState(networkTypeNButton, TRUE, FALSE);
			break;
	}
#endif

	/* read simulation parameters */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.length = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.step = atof(charRead);

	/* read global menu options */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	simInfState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	showProcessingInfo = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	graphDisplayState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	j = atoi(charRead);

	switch(j)
	{
		case 0: autoResetState = FALSE;
			neuronActivityColorsState = FALSE;
			break;

		case 1:	autoResetState = TRUE;
			neuronActivityColorsState = FALSE;
			break;

		case 2:	autoResetState = FALSE;
			neuronActivityColorsState = TRUE;
			break;

		case 3:	autoResetState = TRUE;
			neuronActivityColorsState = TRUE;
			break;

	}
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	outputWatchState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	manualUpdateState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	noiseState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	initialPotentialsState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	synapticStrengthState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
#ifndef BATCH
	synapticTransmissionTimeState = atoi(charRead);
#endif

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	integrationMethod = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseIntensity = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseDecay = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdInitialPotential = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdInitialPotentialStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticStrengthStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdAxoAxonicStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdAxoAxonicStrengthStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticTT = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticTTStd = atof(charRead);

	/* read learning conditions */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	plasticityLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	conditionedLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);

	switch(j)
	{
		case 0: hebbLearning = FALSE;
			competitiveLearning = FALSE;
			break;

		case 1:	hebbLearning = TRUE;
			competitiveLearning = FALSE;
			break;

		case 2:	hebbLearning = FALSE;
			competitiveLearning = TRUE;
			break;

		case 3:	hebbLearning = TRUE;
			competitiveLearning = TRUE;
			break;

	}

#ifndef BATCH
	/* set toggle buttons */

	XmToggleButtonGadgetSetState(simInfButton, simInfState, TRUE);
	XmToggleButtonGadgetSetState(timeMessureButton, showProcessingInfo, TRUE);
	XmToggleButtonGadgetSetState(graphDisplayButton, graphDisplayState, TRUE);
	XmToggleButtonGadgetSetState(autoResetButton, autoResetState, TRUE);
	XmToggleButtonGadgetSetState(neuronActivityColorsButton, neuronActivityColorsState, TRUE);
	XmToggleButtonGadgetSetState(outputWatchButton, outputWatchState, TRUE);
	XmToggleButtonGadgetSetState(manualUpdateButton, manualUpdateState, TRUE);
	XmToggleButtonGadgetSetState(plasticityLearnButton, plasticityLearning, TRUE);
	XmToggleButtonGadgetSetState(conditionedLearnButton, conditionedLearning, TRUE);
	XmToggleButtonGadgetSetState(hebbLearnButton, hebbLearning, TRUE);
	XmToggleButtonGadgetSetState(competitiveLearnButton, competitiveLearning, TRUE);

	XmToggleButtonGadgetSetState(noiseButton, noiseState, TRUE);
	XmToggleButtonGadgetSetState(initialPotentialsButton, initialPotentialsState, TRUE);
	XmToggleButtonGadgetSetState(synapticStrengthButton, synapticStrengthState, TRUE);
	XmToggleButtonGadgetSetState(synapticTransmissionTimeButton, synapticTransmissionTimeState,	
		TRUE);

	XmToggleButtonGadgetSetState(eulerButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(expEulerButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(trapezoidButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(rungeKuttaButton, FALSE, FALSE);

	switch (integrationMethod)
	{
	case EULER: XmToggleButtonGadgetSetState(eulerButton, TRUE, FALSE);
		break;

	case EXPEULER: XmToggleButtonGadgetSetState(expEulerButton, TRUE, FALSE);
		break;

	case TRAPEZOID: XmToggleButtonGadgetSetState(trapezoidButton, TRUE, FALSE);
		break;

	case RUNGEKUTTA: XmToggleButtonGadgetSetState(rungeKuttaButton, TRUE, FALSE);
		break;
	
	default:	error(76);
			XmToggleButtonGadgetSetState(expEulerButton, TRUE, FALSE);
			break;
	}
#endif

	/* read neurons */

	for (j = 1; j <= totalNeurons; j++)
	{
		/* read neuron id */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronId = atoi(charRead);
		if (neuronId > highestNeuronID) highestNeuronID = neuronId;

		/* read neuron position */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		neuronX = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		neuronY = atoi(charRead);

		/* calculate top-left position of neuron */

		neuronX -= (int) rint((double) neuronDiameter / 2.0);
		neuronY -= (int) rint((double) neuronDiameter / 2.0);

		/* create graphical region for neuron */

		region = XCreateRegion();
		rectangle.x = (short) neuronX;
		rectangle.y = (short) neuronY;
		rectangle.width = (unsigned short) neuronDiameter;
		rectangle.height = (unsigned short) neuronDiameter;
		XUnionRectWithRegion(&rectangle, region, region);
#endif

		/* add neuron to neuron list and set neuron position */

		add_neuron_to_neuron_list(neuronId, 0);
		currentNeuron = neuronListTail;
#ifndef BATCH
		currentNeuron->x = currentNeuron->xOriginal = neuronX;
		currentNeuron->y = currentNeuron->yOriginal = neuronY;
		currentNeuron->region = region;
		change_neuron_color(currentNeuron, resData.neuronColor, resData.textColor);
#endif

		/* read in neuron parameters */

		ACTUAL_DENDRITIC_REGIONS = curNumOfDendSegs;
		if (! get_neuron(fp, currentNeuron, neuronId, fileType))
		{
			error(118);
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
	}

	/* read neuron graph windows */

	for (j = 1; j <= totalGraphs; j++)
	{
		/* read neuron id */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronId = atoi(charRead);

		/* read graph window position */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_left = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_top = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_right = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_bottom = atoi(charRead);
#endif

		/* read other graph relevant data */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		firstX = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		lastX = atof(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		xScale = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		highestY = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		lowestY = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		yScale = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		yStep = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		gridType = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		axis = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceSoma = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceDendrit = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceSynapticCurrent = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceTotalCurrent = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceN = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceNA = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceK = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceCA = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceCA1 = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceCA2 = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceKCA = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceA = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceIR = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceP = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		asciiInterval = atof(charRead);

		/* insert neuron with neuron id in selection list - used for create new graph */

		remove_selected_objects_from_selectionList();
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->neuronId == neuronId)
			{
				add_neuron_to_selection_list(currentNeuron);
				break;
			}
		}

		if (currentNeuron == NULL)
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		/* create a new graph for the neuron */

#ifndef BATCH
		create_newGraph(mainWindow);
#else
		create_newGraph();
#endif

		/* set graph window position and size */

		currentGraph = currentNeuron->graph;

#ifndef BATCH
		n = 0;
		XtSetArg(args[n], XmNx, &x); n++;
		XtSetArg(args[n], XmNy, &y); n++;
		XtGetValues(currentGraph->drawingArea, args, n);

		width = (Dimension) (rc_right - rc_left);
		height = (Dimension) (rc_bottom - rc_top);
		screenWidth = (Dimension) (WidthOfScreen(DefaultScreenOfDisplay(disp)));
		screenHeight = (Dimension) (HeightOfScreen(DefaultScreenOfDisplay(disp)));

		if (width > screenWidth)
			width = screenWidth;
		if (height > screenHeight)
			height = screenHeight;
		if (rc_left + width > screenWidth)
			rc_left = screenWidth - width - 12;
		if (rc_top + height > screenHeight)
		{
			rc_top = screenHeight - height;

			if (rc_top == 0)
				rc_top = 34;
		}

		XtConfigureWidget(XtParent(currentGraph->formWidget), rc_left, rc_top, width, height, 0);
		XtConfigureWidget(currentGraph->drawingArea, rc_left + x, rc_top + y, width - x, height - y, 0);

		/* initialize graph content */

		currentGraph->firstX = firstX;
		currentGraph->xScale = xScale;
		currentGraph->highestY = highestY;
		currentGraph->lowestY = lowestY;
		currentGraph->yScale = yScale;
		currentGraph->yStep = yStep;
		currentGraph->gridType = gridType;
		currentGraph->axis = axis;
#endif
		currentGraph->lastX = lastX;
		currentGraph->asciiDataOutputInterval = asciiInterval;
		currentGraph->traceSoma = traceSoma;
		if (traceDendrit)
		{
			currentGraph->traceDendrit = TRUE;
			currentGraph->dendritSegment = traceDendrit;
		}
		else	currentGraph->traceDendrit = FALSE;
		currentGraph->traceSynapticCurrent = traceSynapticCurrent;
		currentGraph->traceTotalCurrent = traceTotalCurrent;
		currentGraph->traceN = traceN;
		currentGraph->traceNA = traceNA;
		currentGraph->traceK = traceK;
		currentGraph->traceCA = traceCA;
		currentGraph->traceCA1 = traceCA1;
		currentGraph->traceCA2 = traceCA2;
		currentGraph->traceKCA = traceKCA;
		currentGraph->traceA = traceA;
		currentGraph->traceIR = traceIR;
		currentGraph->traceP = traceP;

		/* reallocate memory for graph traces */

		graphLength = (unsigned int) rint((lastX + 1) / STEP / T_SCALE * sizeof(float));

		realloc_graph(currentGraph, graphLength);

#ifndef BATCH
		/* set graph axis */

		set_x_axis(currentGraph);
		set_y_axis(currentGraph);

		/* draw graph */

		draw_graph(currentNeuron);
		XmUpdateDisplay(currentGraph->formWidget);	/* processes all pending exposure events immediately */

		/* set toggle button states */

		XmToggleButtonGadgetSetState(newGraphParameterSomaToggle, traceSoma, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterDendritToggle, traceDendrit, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterSynapticCurrentToggle, traceSynapticCurrent, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterTotalCurrentToggle, traceTotalCurrent, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterNToggle, traceN, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterNAToggle, traceNA, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterKToggle, traceK, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterCAToggle, traceCA, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterCA1Toggle, traceCA1, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterCA2Toggle, traceCA2, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterKCAToggle, traceKCA, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterAToggle, traceA, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterIRToggle, traceIR, FALSE);
		XmToggleButtonGadgetSetState(newGraphParameterPToggle, traceP, FALSE);

		/* set label of dendrit toggle */

		n = 0;
		value = strchr(dendritParString, (int) '/');
		strncpy(val, dendritParString, strlen(dendritParString) - strlen(value));
		val[strlen(dendritParString) - strlen(value)] = '\0';
		val[strlen(val) - 1] = '\0';
		sprintf(tmp, "%d",currentGraph->dendritSegment);
		strcat(val, tmp);
		strcat(val, value);
		xmstrings[0] = XmStringCreateLtoR(val, defaultFont);
		XtSetArg(args[n], XmNlabelString, xmstrings[0]); n++;
		XtSetValues(currentGraph->dendritToggle, args, n);
		XmStringFree(xmstrings[0]);
#endif
	}
	remove_selected_objects_from_selectionList();

	/* read synapse types */

	if (synapseTypes != NULL)	/* if synaptic types exist, remove them */
		free(synapseTypes);

	/* build read in synapse types */

	if (numTypes > 0)
	{
		/* allocate memory for synaptic types */

		synapseTypes = (Synapse *) malloc((size_t) (numTypes * sizeof(Synapse)));
		if (! synapseTypes)
		{
			error(119);
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		/* build the new types */

		for (j = 0; j <= numTypes - 1; j++)
		{
			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseTypes[j].ionType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseTypes[j].equilPotential = atof(charRead);
		}

		/* read synapses */

		for (j = 1; j <= totalSynapses; j++)
		{
			int	synapseID;			/* synapse identification number */
			int 	sourceCode, targetCode;		/* neuron number for parent and target neuron */
			int 	targetArea;			/* connection location */
			int 	synapticType;			/* synaptic type of connection */
			double 	deliveryTime;			/* transmission time */
			double 	conductance;			/* conductance of synapse */
			double	maxConductance;			/* maximum conductance of synapse */
			double	learnParam[8];			/* learn parameter */
			Boolean sourceFound, targetFound;	/* help flags */

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetArea = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			deliveryTime = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapticType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			conductance = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			maxConductance = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trt = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trs = atof(charRead);

			sourceFound = FALSE;
			targetFound = FALSE;

			/* search parent and target neuron of connection */

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				if (currentNeuron->neuronId == sourceCode)
				{
					sourceNeuron = currentNeuron;
					sourceFound = TRUE;
				}
				if (currentNeuron->neuronId == targetCode)
				{
					targetNeuron = currentNeuron;
					targetFound = TRUE;
				}

				if (sourceFound && targetFound)
					break;
			}

			if (currentNeuron == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			connect_neurons(sourceNeuron, targetNeuron, synapticType, conductance, deliveryTime,
				trt, trs, targetArea, NULL);
			
			sourceNeuron->connectionList->synapseID = synapseID;
			sourceNeuron->connectionList->maxConductance = maxConductance;

			for (n = 0; n < 8; n++)
			{
				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				learnParam[n] = atof(charRead);
			}

			sourceNeuron->connectionList->slope = learnParam[0];
			sourceNeuron->connectionList->shortTermMemory = learnParam[1];
			sourceNeuron->connectionList->longTermThreshold = learnParam[2];
			sourceNeuron->connectionList->longTermMinimum = learnParam[3];
			sourceNeuron->connectionList->hebbIntervall = learnParam[4];
			sourceNeuron->connectionList->hebbLearnFactor = learnParam[5];
			sourceNeuron->connectionList->shortTermPlasticityLearnFactor = learnParam[6];
			sourceNeuron->connectionList->competitiveLearnFactor = learnParam[7];
		}

		/* read axo-axonic synapses */

		for (j = 1; j <= totalAxoSynapses; j++)
		{
			int	synapseID;			/* synapse identification number of axo-synapse */
			int	targetSynapseID;		/* synapse identification number of target synapse */
			int 	sourceCode;			/* neuron number of axo-axonic sender */
			int	sourceCodeTargetSynapse;	/* neuron number of parent neuron from target synapse */
			int	targetCodeTargetSynapse;	/* neuron number of target neuron from target synapse */
			int 	targetArea;			/* connection location */
			int 	synapticType;			/* synaptic type of connection */
			double 	deliveryTime;			/* transmission time */
			double 	inhibitionFactor;		/* inhibition factor of synapse */
			double	maxInhibitionFactor;		/* maximal inhibition factor of synapse */
			double	learnParam[8];			/* learn parameter */
			Boolean sourceFound;			/* help flag */
			Boolean sourceFoundTargetSynapse;	/* help flag */
			Boolean	targetFoundTargetSynapse;	/* help flag */

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetSynapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCodeTargetSynapse = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetCodeTargetSynapse = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetArea = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			deliveryTime = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapticType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trt = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trs = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			inhibitionFactor = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			maxInhibitionFactor = atof(charRead);

			sourceFound = FALSE;
			sourceFoundTargetSynapse = FALSE;
			targetFoundTargetSynapse = FALSE;

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				if (currentNeuron->neuronId == sourceCode)
				{
					sourceNeuron = currentNeuron;
					sourceFound = TRUE;
				}
				if (currentNeuron->neuronId == sourceCodeTargetSynapse)
				{
					sourceNeuronTargetSynapse = currentNeuron;
					sourceFoundTargetSynapse = TRUE;
				}
				if (currentNeuron->neuronId == targetCodeTargetSynapse)
				{
					targetNeuronTargetSynapse = currentNeuron;
					targetFoundTargetSynapse = TRUE;
				}

				if (sourceFound && sourceFoundTargetSynapse && targetFoundTargetSynapse)
					break;
			}

			if (currentNeuron == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			for (currentSynapse = sourceNeuronTargetSynapse->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
			{
				if (currentSynapse->synapseID == targetSynapseID)
					break;
			}

			if (currentSynapse == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			connect_neurons(sourceNeuron, targetNeuronTargetSynapse, synapticType, inhibitionFactor,
				deliveryTime, trt, trs, -1, currentSynapse);

			sourceNeuron->connectionList->synapseID = synapseID;
			sourceNeuron->connectionList->maxInhibitionFactor = maxInhibitionFactor;

			for (n = 0; n < 8; n++)
			{
				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				learnParam[n] = atof(charRead);
			}

			sourceNeuron->connectionList->slope = learnParam[0];
			sourceNeuron->connectionList->shortTermMemory = learnParam[1];
			sourceNeuron->connectionList->longTermThreshold = learnParam[2];
			sourceNeuron->connectionList->longTermMinimum = learnParam[3];
			sourceNeuron->connectionList->hebbIntervall = learnParam[4];
			sourceNeuron->connectionList->hebbLearnFactor = learnParam[5];
			sourceNeuron->connectionList->shortTermPlasticityLearnFactor = learnParam[6];
			sourceNeuron->connectionList->conditioningLearnFactor = learnParam[7];
		}
	} /* end of numTypes > 0 */

	/* read synapse graph windows */

	for (j = 1; j <= totalSGraphs + totalASGraphs; j++)
	{
		/* read synapse id */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		synapseId = atoi(charRead);

		/* read graph window position */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_left = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_top = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_right = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		rc_bottom = atoi(charRead);
#endif

		/* read other graph relevant data */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		firstX = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		lastX = atof(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		xScale = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		highestY = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		lowestY = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		yScale = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		yStep = atof(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		gridType = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
#ifndef BATCH
		axis = atoi(charRead);
#endif

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		tracePot = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceMem = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceCon = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		traceStr = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		asciiInterval = atof(charRead);

		/* insert synapse with synapse id in selection list - used for create new graph */

		remove_selected_objects_from_selectionList();
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
		   for (currentSynapse = currentNeuron->connectionList; ((currentSynapse != NULL) &&
		       (currentSynapse->synapseID != synapseId)); currentSynapse = currentSynapse->nextConnection);

		   if (currentSynapse != NULL)
		   {
			   add_synapse_to_selection_list(currentSynapse);
			   break;
		   }
		}

		if (currentSynapse == NULL)
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		/* create a new graph for the synapse */

#ifndef BATCH
		create_newGraphS(mainWindow);
#else
		create_newGraphS();
#endif

		/* set graph window position and size */

		currentGraphS = currentSynapse->graph;

#ifndef BATCH
		n = 0;
		XtSetArg(args[n], XmNx, &x); n++;
		XtSetArg(args[n], XmNy, &y); n++;
		XtGetValues(currentGraphS->drawingArea, args, n);

		width = (Dimension) (rc_right - rc_left);
		height = (Dimension) (rc_bottom - rc_top);
		screenWidth = (Dimension) (WidthOfScreen(DefaultScreenOfDisplay(disp)));
		screenHeight = (Dimension) (HeightOfScreen(DefaultScreenOfDisplay(disp)));

		if (width > screenWidth)
			width = screenWidth;
		if (height > screenHeight)
			height = screenHeight;
		if (rc_left + width > screenWidth)
			rc_left = screenWidth - width - 12;
		if (rc_top + height > screenHeight)
		{
			rc_top = screenHeight - height;

			if (rc_top == 0)
				rc_top = 34;
		}

		XtConfigureWidget(XtParent(currentGraphS->formWidget), rc_left, rc_top, width, height, 0);
		XtConfigureWidget(currentGraphS->drawingArea, rc_left + x, rc_top + y, width, height, 0);

		/* initialize graph content */

		currentGraphS->firstX = firstX;
		currentGraphS->xScale = xScale;
		currentGraphS->highestY = highestY;
		currentGraphS->lowestY = lowestY;
		currentGraphS->yScale = yScale;
		currentGraphS->yStep = yStep;
		currentGraphS->gridType = gridType;
		currentGraphS->axis = axis;
#endif
		currentGraphS->lastX = lastX;
		currentGraphS->asciiDataOutputInterval = asciiInterval;
		currentGraphS->tracePot = tracePot;
		currentGraphS->traceMem = traceMem;
		currentGraphS->traceCon = traceCon;
		currentGraphS->traceStr = traceStr;

		/* reallocate memory for graph traces */

		graphLength = (unsigned int) rint((lastX + 1) / STEP / T_SCALE * sizeof(float));

		realloc_graphS(currentGraphS, graphLength);

#ifndef BATCH
		/* set graph axis */

		set_x_axisS(currentGraphS);
		set_y_axisS(currentGraphS);

		/* draw graph */

		draw_graphS(currentSynapse);
		XmUpdateDisplay(currentGraphS->formWidget); /* processes all pending exposure events immediately */

		/* set toggle button states */

		XmToggleButtonGadgetSetState(newGraphSParameterPotToggle, tracePot, TRUE);
		XmToggleButtonGadgetSetState(newGraphSParameterMemToggle, traceMem, TRUE);
		XmToggleButtonGadgetSetState(newGraphSParameterConToggle, traceCon, TRUE);
		XmToggleButtonGadgetSetState(newGraphSParameterStrToggle, traceStr, TRUE);
#endif
	}
	remove_selected_objects_from_selectionList();

	/* load currents */

	for (j = 1; j <= totalCurrents; j++)
	{
		int sourceCode;		/* neuron number of current read in neuron */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		sourceCode = atoi(charRead);

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->neuronId == sourceCode)
			{
				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaCurrent = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaSCStart = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaSCEnd = atof(charRead);

				if (!readf(fp,charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendCurrent = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCStart = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCEnd = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCLocation = atoi(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->noise = (Boolean) atoi(charRead);
				break;
			}
		}
	} /* end of adding currents */

	/* if the end of file is not reached yet output a warning message */

	if (programStates.eofReached != TRUE)
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}

#ifndef BATCH
	editNetwork = TRUE;
	show_normal_state();
	show_all_objects_in_drawing_area();
#endif
	return TRUE;

} /* end of read_network */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: read_parameter(filename)
 
	PURPOSE		: reads a parameter file ( .par ) - network parameter

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_parameter(filename)
char filename[128];	/* filename for parameter file */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	char    	charRead[100];		/* current number read */
	AxoAxonicSenderList *axoSender;         /* current axo-axonic sender */
	ChemicalSenderList *currentSender;      /* current sender */
	ChemicalPool    *currentPool;           /* current pool */
	int             queuePlaces;            /* number of entries in a process queue */
	int		fileType;		/* identification number which identifies the file type */
	int     	i, j;			/* loop variable */
	int		totalNeurons;		/* number of neurons in the net */
	int		totalCurrents;		/* number of input currents in the net */
	int		simInfState,
			graphDisplayState,
			autoResetState,
			neuronActivityColorsState,
			outputWatchState,
			manualUpdateState,
			initialPotentialsState,
			synapticStrengthState,
			synapticTransmissionTimeState,
			noiseState;		/* state of the global menu option buttons */
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* open parameter file */

	variables_initialized = TRUE;
	programStates.newRead = TRUE;

	if ((fp = fopen(filename, "r+")) == NULL)
		if ((fp = fopen(filename, "r")) != NULL)     /* create warning dialog box, file is read only */
		error(116);
		else	/* create error dialog box, cause selected file cannot be opened */
		{
			error(1);
			return(FALSE);
		}

	/* read filetype and check if its the id for a parameter file */

	if (! readf(fp, charRead))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	fileType = atoi(charRead);

	if ((fileType != PARAMETER_FILE))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	/* get number of items in net */

	if (! readf(fp, charRead))
	{
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalNeurons = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalCurrents = atoi(charRead);

	/* Read simulation parameters */

	if (! readf(fp, charRead))
	{
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.length = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.step = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	neuronActivityInterval = atoi(charRead);

	/* read global menu options */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simInfState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	showProcessingInfo = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	graphDisplayState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);

	switch(j)
	{
		case 0: autoResetState = FALSE;
			neuronActivityColorsState = FALSE;
			break;

		case 1:	autoResetState = TRUE;
			neuronActivityColorsState = FALSE;
			break;

		case 2:	autoResetState = FALSE;
			neuronActivityColorsState = TRUE;
			break;

		case 3:	autoResetState = TRUE;
			neuronActivityColorsState = TRUE;
			break;

	}

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	outputWatchState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	manualUpdateState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	initialPotentialsState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	synapticStrengthState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	synapticTransmissionTimeState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	integrationMethod = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseIntensity = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	noiseDecay = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdInitialPotential = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdInitialPotentialStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticStrengthStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdAxoAxonicStrength = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdAxoAxonicStrengthStd = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticTT = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	stdSynapticTTStd = atof(charRead);

	/* read learning conditions */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	plasticityLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	conditionedLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);

	switch(j)
	{
		case 0: hebbLearning = FALSE;
			competitiveLearning = FALSE;
			break;

		case 1:	hebbLearning = TRUE;
			competitiveLearning = FALSE;
			break;

		case 2:	hebbLearning = FALSE;
			competitiveLearning = TRUE;
			break;

		case 3:	hebbLearning = TRUE;
			competitiveLearning = TRUE;
			break;

	}

	/* set toggle buttons */

	XmToggleButtonGadgetSetState(simInfButton, simInfState, TRUE);
	XmToggleButtonGadgetSetState(timeMessureButton, showProcessingInfo, TRUE);
	XmToggleButtonGadgetSetState(graphDisplayButton, graphDisplayState, TRUE);
	XmToggleButtonGadgetSetState(autoResetButton, autoResetState, TRUE);
	XmToggleButtonGadgetSetState(neuronActivityColorsButton, neuronActivityColorsState, TRUE);
	XmToggleButtonGadgetSetState(outputWatchButton, outputWatchState, TRUE);
	XmToggleButtonGadgetSetState(manualUpdateButton, manualUpdateState, TRUE);
	XmToggleButtonGadgetSetState(noiseButton, noiseState, TRUE);
	XmToggleButtonGadgetSetState(initialPotentialsButton, initialPotentialsState, TRUE);
	XmToggleButtonGadgetSetState(synapticStrengthButton, synapticStrengthState, TRUE);
	XmToggleButtonGadgetSetState(synapticTransmissionTimeButton, synapticTransmissionTimeState, TRUE);
	XmToggleButtonGadgetSetState(plasticityLearnButton, plasticityLearning, TRUE);
	XmToggleButtonGadgetSetState(conditionedLearnButton, conditionedLearning, TRUE);
	XmToggleButtonGadgetSetState(hebbLearnButton, hebbLearning, TRUE);
	XmToggleButtonGadgetSetState(competitiveLearnButton, competitiveLearning, TRUE);

	XmToggleButtonGadgetSetState(eulerButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(expEulerButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(trapezoidButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(rungeKuttaButton, FALSE, FALSE);

	switch (integrationMethod)
	{
	case EULER: XmToggleButtonGadgetSetState(eulerButton, TRUE, FALSE);
		break;

	case EXPEULER: XmToggleButtonGadgetSetState(expEulerButton, TRUE, FALSE);
		break;

	case TRAPEZOID: XmToggleButtonGadgetSetState(trapezoidButton, TRUE, FALSE);
		break;

	case RUNGEKUTTA: XmToggleButtonGadgetSetState(rungeKuttaButton, TRUE, FALSE);
		break;
	
	default:	error(76);
			XmToggleButtonGadgetSetState(expEulerButton, TRUE, FALSE);
			break;
	}

	/* read network model and type */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);
	complexModel = ((j & 1) == 0);
	networkmodel = (j >> 1) + (int) (! complexModel);

	n = 0;
	if (complexModel)	/* complex model ? */
	{
		XtSetArg(args[n], XmNsensitive, TRUE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}
	else
	{
		XtSetArg(args[n], XmNsensitive, FALSE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	networktype = atoi(charRead); /* 1, 2, 4, N- point model */

	switch(networktype)
	{
		case 1:	/* 1-point model */
			curNumOfDendSegs = 0;
			break;

		case 2:	/* 2-point model */
			curNumOfDendSegs = 1;
			break;

		case 3:	/* 4-point model */
			curNumOfDendSegs = 3;
			break;

		case 4:	/* n-point model */
			curNumOfDendSegs = 3;
			break;
	}

	/* read in number of dendritic segments currently used */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	curNumOfDendSegs = atoi(charRead);

	if (networktype == 4)
		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs);
	else	DendriteLength = defaultDendriteLength;

	/* set maximum scale widgets of add synapse, edit synapse, edit current mask to curNumOfDendSegs */

	if (networktype == 1)	/* 1-Point model */
	{
		/* set scale bar of add synapse mask, edit synapse mask and edit current mask to insensitive */

		XmTextSetString(addSynapseTargetText, "0");
		XmTextSetString(editSynapseLocationText, "0");
		XmTextSetString(editCurrentLocationText, "0");
		XmScaleSetValue(addSynapseTargetScale, 0);
		XmScaleSetValue(editSynapseLocationScale, 0);
		XmScaleSetValue(editCurrentLocationScale, 0);

		XtSetSensitive(addSynapseTargetScale, FALSE);
		XtSetSensitive(editSynapseLocationScale, FALSE);
		XtSetSensitive(editCurrentLocationScale, FALSE);

		show_normal_state();
		show_all_objects_in_drawing_area(FALSE);
	}
	else	set_max_scale();

	/* set buttons in network model mask */

	XmToggleButtonGadgetSetState(networkOnOffButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkSWIMButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkHHButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkGBButton, FALSE, FALSE);

	if (complexModel)	/* complex model ? */
		XmToggleButtonGadgetSetState(networkComplexButton, TRUE, FALSE);
	else	XmToggleButtonGadgetSetState(networkComplexButton, FALSE, FALSE);

	switch (networkmodel)
	{
	case 1: XmToggleButtonGadgetSetState(networkOnOffButton, TRUE, FALSE);
		break;

	case 2: XmToggleButtonGadgetSetState(networkSWIMButton, TRUE, FALSE);
		break;

	case 3: XmToggleButtonGadgetSetState(networkHHButton, TRUE, FALSE);
		break;

	case 4: XmToggleButtonGadgetSetState(networkGBButton, TRUE, FALSE);
		break;
	}

	XmToggleButtonGadgetSetState(networkType1Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType2Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType4Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkTypeNButton, FALSE, FALSE);

	switch (networktype)
	{
		case 1:	XmToggleButtonGadgetSetState(networkType1Button, TRUE, FALSE);
			break;
		case 2:	XmToggleButtonGadgetSetState(networkType2Button, TRUE, FALSE);
			break;
		case 3:	XmToggleButtonGadgetSetState(networkType4Button, TRUE, FALSE);
			break;
		case 4:	XmToggleButtonGadgetSetState(networkTypeNButton, TRUE, FALSE);
			break;
	}

	/* read neurons */

	for (j = 1; j <= totalNeurons; j++)
	{
		int 	neuronId;		/* current neuron number */
		Boolean neuronFound = FALSE;	/* help flag */

		if (! readf(fp, charRead))
		{
			clear_neurons();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronId = atoi(charRead);

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->neuronId == neuronId)
			{
				neuronFound = TRUE;

				if (! get_neuron(fp, currentNeuron, neuronId, fileType))
				{
					error(118);
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}

				if (currentNeuron->graph != NULL)
				{
					currentNeuron->graph->somaBound = FALSE;
					currentNeuron->graph->dendBound = FALSE;
					currentNeuron->graph->synCBound = FALSE;
					currentNeuron->graph->totCBound = FALSE;
					currentNeuron->graph->iNBound = FALSE;
					currentNeuron->graph->iNABound = FALSE;
					currentNeuron->graph->iKBound = FALSE;
					currentNeuron->graph->iCABound = FALSE;
					currentNeuron->graph->iCA1Bound = FALSE;
					currentNeuron->graph->iCA2Bound = FALSE;
					currentNeuron->graph->iKCABound = FALSE;
					currentNeuron->graph->iABound = FALSE;
					currentNeuron->graph->iIRBound = FALSE;
					currentNeuron->graph->iPBound = FALSE;
				}

				switch(networktype)
				{
				case 1:
					PseudoSomaCur = DendCur;
					PseudoSCSTRT = SDSTRT;
					PseudoSCSTP = SDSTP;

					IRegion = 0;
					IStem = 1;
					DENDRITE_STEMS = 0;
					DENDRITE_REGIONS = 0;

					break;
				case 2:
					DENDRITE_STEMS = 1;
					DENDRITE_REGIONS = 1;
					IRegion = 1;
					IStem = 1;

					break;
				case 3:
					DENDRITE_STEMS = 1;
					DENDRITE_REGIONS = 3;
					if (IRegion > 3) IRegion = 3;
					IStem = 1;

					break;
				case 4:
					DENDRITE_STEMS = (int) ACTUAL_DENDRITIC_STEMS;
					DENDRITE_REGIONS = (int) ACTUAL_DENDRITIC_REGIONS;
					if (IRegion > curNumOfDendSegs) IRegion = curNumOfDendSegs;

					break;
				}

				if (DENDRITE_STEMS == 0 || DENDRITE_REGIONS == 0)
				{
					DENDRITE_STEMS = 0;
					DENDRITE_REGIONS = 0;
				}

				for (i = 0; i <= curNumOfDendSegs; i++)
				{
					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					E_POT[i] = atof(charRead);
				}

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				SpikeCounter = atoi(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				PseudoSomaCur = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				PseudoSCSTRT = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				PseudoSCSTP = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				IRegion = atoi(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				IStem = atoi(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				SLmSG = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				SGpDSmDG = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				DLmDG = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				SGdSC = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				SGpDSmDGdSC = atof(charRead);

				switch(networkmodel)
				{
				case 1: /* ON/OFF model */

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					SPIKE = atoi(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					SPIKE_IN_PROCESS = atoi(charRead);

					break;

				case 2: /* SWIM model */

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					M_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					N_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					H_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					C_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					SWIM_NAEmNAG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					SWIM_KEmKG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					SWIM_CAEmCAG = atof(charRead);

					break;

				case 3: /* HH model */

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					M_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					N_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					H_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					HH_NAEmNAG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					HH_KEmKG = atof(charRead);

					break;

				case 4: /* GB model */

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					M_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					N_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					H_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					C_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					D_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					E_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					A_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					B1_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					B2_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					P_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					Q_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					R_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					S_CONST = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_I_CA1 = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_I_CA2 = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_CONC_CA = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_NAEmNAG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_KEmKG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_AEmAG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_KCAEmKCAG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_IREmIRG = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					GB_PEmPG = atof(charRead);

					break;
				}
				break;
			}
		}
		if (! neuronFound)
		{
			error(121);
			clear_neurons();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
	}

	/* read synaptic values */

	for (j = 1; j <= totalNeurons; j++)
	{
		int 	neuronId;		/* current neuron number */
		Boolean neuronFound = FALSE;	/* help flag */

		if (! readf(fp, charRead))
		{
			clear_neurons();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronId = atoi(charRead);

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->neuronId == neuronId)
			{
				neuronFound = TRUE;

				for (currentPool = currentNeuron->poolList; currentPool != NULL;
				     currentPool = currentPool->nextPool)
				{
					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					currentPool->longestDeliveryTime = atof(charRead);

					for (currentSender = currentPool->senderList; currentSender != NULL;
					     currentSender = currentSender->nextSender)
					{
						currentSynapse = currentSender->connection;

						if (! readf(fp, charRead))
						{
							clear_neurons();
							if (fclose(fp) != 0)
								error(101);
							return(FALSE);
						}
						currentSynapse->conductance = atof(charRead);

						if (! readf(fp, charRead))
						{
							clear_neurons();
							if (fclose(fp) != 0)
								error(101);
							return(FALSE);
						}
						TSmTRT = atof(charRead);

						if (! readf(fp, charRead))
						{
							clear_neurons();
							if (fclose(fp) != 0)
								error(101);
							return(FALSE);
						}
						TTdSS = atof(charRead);

						if (complexModel)
						{
							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->synapticMemorySTP = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->synapticMemoryHebb = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							LEXP = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							SFCONST = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							LFCONST = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->trainingFlag = atoi(charRead);

							if (currentSynapse->trainingFlag & 1)
								currentSynapse->trainingFlagSTP = TRUE;
							else	currentSynapse->trainingFlagSTP = FALSE;
							if (currentSynapse->trainingFlag & 2)
								currentSynapse->trainingFlagHebb = TRUE;
							else	currentSynapse->trainingFlagHebb = FALSE;
							if (currentSynapse->trainingFlag & 4)
								currentSynapse->trainingFlagCond = TRUE;
							else	currentSynapse->trainingFlagCond = FALSE;

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->shortTermLearningCondition = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->conditioningLearningCondition = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->hebbLearningCondition = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->competitiveLearningCondition = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->hebbIntervallCounter = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->hebbTimer = atof(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->stmCnt = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->condCnt = atoi(charRead);

							if (! readf(fp, charRead))
							{
								clear_neurons();
								if (fclose(fp) != 0)
									error(101);
								return(FALSE);
							}
							currentSynapse->hebbCnt = atoi(charRead);

							for (axoSender = currentSynapse->axoSenderList; axoSender != NULL;
							     axoSender = axoSender->nextSender)
							{
								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->currentInhibitionFactor =
									atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->synapticMemoryCond = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->LEXP_const = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->SFCONST_const = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->LFCONST_const = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->trainingFlag = atoi(charRead);

								if (axoSender->connection->trainingFlag & 1)
									axoSender->connection->trainingFlagSTP = TRUE;
								else	axoSender->connection->trainingFlagSTP = FALSE;
								if (axoSender->connection->trainingFlag & 2)
									axoSender->connection->trainingFlagCond = TRUE;
								else	axoSender->connection->trainingFlagCond = FALSE;

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->shortTermLearningCondition =
									atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->conditioningLearningCondition =
									atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->hebbLearningCondition =
									atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->competitiveLearningCondition =
									atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->hebbIntervallCounter =
									atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->synapticMemorySTP =
									atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->stmCnt = atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->condCnt = atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->hebbCnt = atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->TSmTRT_const = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->connection->TTdSS_const = atof(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								queuePlaces = atoi(charRead);

								if (! readf(fp, charRead))
								{
									clear_neurons();
									if (fclose(fp) != 0)
										error(101);
									return(FALSE);
								}
								axoSender->currentQueuePlace = atoi(charRead);

								/* now resize queue to proper length for this simulation */

								axoSender->processQueue =
									(ProcessQueue *) realloc(axoSender->processQueue,
									(size_t) (queuePlaces * sizeof(ProcessQueue)));
								
								if (! axoSender->processQueue)
								{
									error(104);
									return(FALSE);
								}

								for (i = 0; i < queuePlaces; i++)
								{
									if (! readf(fp, charRead))
									{
										clear_neurons();
										if (fclose(fp) != 0)
											error(101);
										return(FALSE);
									}
									axoSender->processQueue[i] = atof(charRead);
								}
							} /* end of look through axo-sender */
						} /* end of IF complex network model */
					} /* end of look through senders */

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					LTTdSS = atof(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					queuePlaces = atoi(charRead);

					if (! readf(fp, charRead))
					{
						clear_neurons();
						if (fclose(fp) != 0)
							error(101);
						return(FALSE);
					}
					currentPool->currentQueuePlace = atoi(charRead);

					/* now resize queue to proper length for this simulation */

					currentPool->processQueue = (ProcessQueue *) realloc(currentPool->processQueue,
						(size_t) (queuePlaces * sizeof(ProcessQueue)));
					
					if (! currentPool->processQueue)
					{
						error(104);
						return(FALSE);
					}

					for (i = 0; i < queuePlaces; i++)
					{
						if (! readf(fp, charRead))
						{
							clear_neurons();
							if (fclose(fp) != 0)
								error(101);
							return(FALSE);
						}
						currentPool->processQueue[i] = atof(charRead);
					}
				} /* end of look through pools */
				break;
			} /* end of IF neuron found */
		} /* end of loop through neurons */

		if (! neuronFound)
		{
			error(121);
			clear_neurons();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
	} /* end of loop through neurons to be read in */

	/* load currents */

	for (j = 1; j <= totalCurrents; j++)
	{
		int sourceCode;		/* number of current neuron */

		if (! readf(fp, charRead))
		{
			clear_neurons();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		sourceCode = atoi(charRead);

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->neuronId == sourceCode)
			{
				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaCurrent = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaSCStart = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->somaSCEnd = atof(charRead);

				if (!readf(fp,charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendCurrent = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCStart = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCEnd = atof(charRead);

				if (! readf(fp, charRead))
				{
					clear_neurons();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->dendSCLocation = atoi(charRead);

				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				currentNeuron->noise = (Boolean) atoi(charRead);
				break;
			}
		}
	} /* end of adding currents */

	/* if the end of file is not reached yet output a warning message */

	if (programStates.eofReached != TRUE)
	{
		error(120);
		clear_neurons();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}
	return(TRUE);

} /* end of read_parameter */

/************************************************************************************************************************

 	FUNCTION	: read_configuration(filename)
 
	PURPOSE		: reads a configuration file ( .con ) - network configuration

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_configuration(filename)
char filename[128];	/* filename for configuration file */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	char    	charRead[100];		/* current number read */
	Neuron		*sourceNeuron;		/* source neuron */
	Neuron		*targetNeuron;		/* target neuron */
	Neuron		*sourceNeuronTargetSynapse; /* source neuron for axo-axonic connections */
	Neuron		*targetNeuronTargetSynapse; /* target neuron for axo-axonic connections */
	int		fileType;		/* identification number which identifies the file type */
	int     	j;			/* loop variable */
	int 		neuronX, neuronY;	/* neuron position */
	int		totalNeurons;		/* number of neurons in the net */
	int		totalSynapses;		/* number of synapses in the net */
	int		totalAxoSynapses;	/* number of axo-axonic synapses in the net */
	int		totalSynapseTypes;	/* number of synaptic types in the net */
	int		highestNeuronID;	/* highest neuron number read */
	int		simInfState,
			graphDisplayState,
			autoResetState,
			neuronActivityColorsState,
			outputWatchState,
			manualUpdateState;	/* state of the global menu option buttons */
	double		trt;			/* transmitter release threshold */
	double		trs;			/* transmitter satiate */
	Position 	rc_left, rc_top;	/* left-top window position */
	Dimension 	rc_right, rc_bottom;	/* right-bottom window position */
	Region		region;			/* graphical region for neurons */
	XRectangle	rectangle;		/* rectangle for neuron region */
	FILE		*fp = NULL;		/* file pointer to simulation file */

	/* clear network */

	clear_network();

	/* open configuration file */

	variables_initialized = FALSE;
	programStates.newRead = TRUE;

	if ((fp = fopen(filename, "r+")) == NULL)
		if ((fp = fopen(filename, "r")) != NULL)     /* create warning dialog box, file is read only */
			error(116);
		else	/* create error dialog box, cause selected file cannot be opened */
		{
			error(1);
			return(FALSE);
		}

	/* read filetype and check if its the id for a configuration file */

	if (! readf(fp, charRead))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	fileType = atoi(charRead);

	if ((fileType != CONFIGURATION_FILE))
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}

	/* get screen size */

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	rc_left = atoi(charRead);

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	rc_top = atoi(charRead);

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	rc_right = atoi(charRead);

	if (! readf(fp, charRead))
	{
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	rc_bottom = atoi(charRead);

	/* set window position */

	n = 0;
	XtSetArg(args[n], XmNx, rc_left); n++;
	XtSetArg(args[n], XmNy, rc_top); n++;
	XtSetArg(args[n], XmNwidth, rc_right - rc_left); n++;
	XtSetArg(args[n], XmNheight, rc_bottom - rc_top); n++;
	XtSetValues(applShell, args, n);

	/* get number of items in net */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalNeurons = atoi(charRead);
	highestNeuronID = totalNeurons;

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalSynapses = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalAxoSynapses = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	totalSynapseTypes = atoi(charRead);
	numTypes = totalSynapseTypes;

	/* read network model and type */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);
	complexModel = ((j & 1) == 0);
	networkmodel = (j >> 1) + (int) (! complexModel);

	n = 0;
	if (complexModel)	/* complex model ? */
	{
		XtSetArg(args[n], XmNsensitive, TRUE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}
	else
	{
		XtSetArg(args[n], XmNsensitive, FALSE); n++;
		XtSetValues(addSynapseLearnButton, args, n);
		XtSetValues(axoSynapseLearnButton, args, n);

	}

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	networktype = atoi(charRead); /* 1, 2, 4, N- point model */

	switch(networktype)
	{
		case 1:	/* 1-point model */
			curNumOfDendSegs = 0;
			break;

		case 2:	/* 2-point model */
			curNumOfDendSegs = 1;
			break;

		case 3:	/* 4-point model */
			curNumOfDendSegs = 3;
			break;

		case 4:	/* n-point model */
			curNumOfDendSegs = 3;
			break;
	}

	/* read in number of dendritic segments currently used */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	curNumOfDendSegs = atoi(charRead);

	if (networktype == 4)
		DendriteLength = (int) rint(defaultDendriteLength / 3.0 * curNumOfDendSegs);
	else	DendriteLength = defaultDendriteLength;

	/* set maximum scale widgets of add synapse, edit synapse, edit current mask to curNumOfDendSegs */

	if (networktype == 1)	/* 1-Point model */
	{
		/* set scale bar of add synapse mask, edit synapse mask and edit current mask to insensitive */

		XmTextSetString(addSynapseTargetText, "0");
		XmTextSetString(editSynapseLocationText, "0");
		XmTextSetString(editCurrentLocationText, "0");
		XmScaleSetValue(addSynapseTargetScale, 0);
		XmScaleSetValue(editSynapseLocationScale, 0);
		XmScaleSetValue(editCurrentLocationScale, 0);

		XtSetSensitive(addSynapseTargetScale, FALSE);
		XtSetSensitive(editSynapseLocationScale, FALSE);
		XtSetSensitive(editCurrentLocationScale, FALSE);

		show_normal_state();
		show_all_objects_in_drawing_area(FALSE);
	}
	else	set_max_scale();

	/* set buttons in network model mask */

	XmToggleButtonGadgetSetState(networkOnOffButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkSWIMButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkHHButton, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkGBButton, FALSE, FALSE);

	if (complexModel)	/* complex model ? */
		XmToggleButtonGadgetSetState(networkComplexButton, TRUE, FALSE);
	else	XmToggleButtonGadgetSetState(networkComplexButton, FALSE, FALSE);

	switch (networkmodel)
	{
	case 1: XmToggleButtonGadgetSetState(networkOnOffButton, TRUE, FALSE);
		break;

	case 2: XmToggleButtonGadgetSetState(networkSWIMButton, TRUE, FALSE);
		break;

	case 3: XmToggleButtonGadgetSetState(networkHHButton, TRUE, FALSE);
		break;

	case 4: XmToggleButtonGadgetSetState(networkGBButton, TRUE, FALSE);
		break;
	}

	XmToggleButtonGadgetSetState(networkType1Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType2Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkType4Button, FALSE, FALSE);
	XmToggleButtonGadgetSetState(networkTypeNButton, FALSE, FALSE);

	switch (networktype)
	{
		case 1:	XmToggleButtonGadgetSetState(networkType1Button, TRUE, FALSE);
			break;
		case 2:	XmToggleButtonGadgetSetState(networkType2Button, TRUE, FALSE);
			break;
		case 3:	XmToggleButtonGadgetSetState(networkType4Button, TRUE, FALSE);
			break;
		case 4:	XmToggleButtonGadgetSetState(networkTypeNButton, TRUE, FALSE);
			break;
	}

	/* read simulation parameters */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.length = atof(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simulation.step = atof(charRead);

	/* read global menu options */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	simInfState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	showProcessingInfo = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	graphDisplayState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);

	switch(j)
	{
		case 0: autoResetState = FALSE;
			neuronActivityColorsState = FALSE;
			break;

		case 1:	autoResetState = TRUE;
			neuronActivityColorsState = FALSE;
			break;

		case 2:	autoResetState = FALSE;
			neuronActivityColorsState = TRUE;
			break;

		case 3:	autoResetState = TRUE;
			neuronActivityColorsState = TRUE;
			break;

	}

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	outputWatchState = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	manualUpdateState = atoi(charRead);

	/* read learning conditions */

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	plasticityLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	conditionedLearning = atoi(charRead);

	if (! readf(fp, charRead))
	{
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	j = atoi(charRead);

	switch(j)
	{
		case 0: hebbLearning = FALSE;
			competitiveLearning = FALSE;
			break;

		case 1:	hebbLearning = TRUE;
			competitiveLearning = FALSE;
			break;

		case 2:	hebbLearning = FALSE;
			competitiveLearning = TRUE;
			break;

		case 3:	hebbLearning = TRUE;
			competitiveLearning = TRUE;
			break;

	}

	/* set toggle buttons */

	XmToggleButtonGadgetSetState(simInfButton, simInfState, TRUE);
	XmToggleButtonGadgetSetState(timeMessureButton, showProcessingInfo, TRUE);
	XmToggleButtonGadgetSetState(graphDisplayButton, graphDisplayState, TRUE);
	XmToggleButtonGadgetSetState(autoResetButton, autoResetState, TRUE);
	XmToggleButtonGadgetSetState(neuronActivityColorsButton, neuronActivityColorsState, TRUE);
	XmToggleButtonGadgetSetState(outputWatchButton, outputWatchState, TRUE);
	XmToggleButtonGadgetSetState(manualUpdateButton, manualUpdateState, TRUE);
	XmToggleButtonGadgetSetState(plasticityLearnButton, plasticityLearning, TRUE);
	XmToggleButtonGadgetSetState(conditionedLearnButton, conditionedLearning, TRUE);
	XmToggleButtonGadgetSetState(hebbLearnButton, hebbLearning, TRUE);
	XmToggleButtonGadgetSetState(competitiveLearnButton, competitiveLearning, TRUE);

	/* read neurons */

	for (j = 1; j <= totalNeurons; j++)
	{
		int neuronId;	/* neuron number of current neuron */

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronId = atoi(charRead);
		if (neuronId > highestNeuronID) highestNeuronID = neuronId;

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronX = atoi(charRead);

		if (! readf(fp, charRead))
		{
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		neuronY = atoi(charRead);

		/* calculate left-top position of neuron */

		neuronX -= (int) rint((double) neuronDiameter / 2.0);
		neuronY -= (int) rint((double) neuronDiameter / 2.0);

		/* create neuron */

		region = XCreateRegion();
		rectangle.x = (short) neuronX;
		rectangle.y = (short) neuronY;
		rectangle.width = (unsigned short) neuronDiameter;
		rectangle.height = (unsigned short) neuronDiameter;
		XUnionRectWithRegion(&rectangle, region, region);

		/* store neuron in neuron list */

		add_neuron_to_neuron_list(neuronId, 0);
		neuronListTail->x = neuronListTail->xOriginal = neuronX;
		neuronListTail->y = neuronListTail->yOriginal = neuronY;
		neuronListTail->region = region;
		change_neuron_color(neuronListTail, resData.neuronColor, resData.textColor);

	}

	/* read synapse types */

	if (synapseTypes != NULL)	/* if synaptic types exist, remove them */
		free(synapseTypes);

	/* build read in synapse types */

	if (numTypes > 0)
	{
		/* allocate memory for synaptic types */

		synapseTypes = (Synapse *) malloc((size_t) (numTypes * sizeof(Synapse)));
		if (! synapseTypes)
		{
			error(119);
			clear_network();
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		for (j = 0; j <= numTypes - 1; j++)
		{
			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseTypes[j].ionType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseTypes[j].equilPotential = atof(charRead);
		}

		/* read synapses */

		for (j = 1; j <= totalSynapses; j++)
		{
			int	synapseID;			/* synapse identification number */
			int 	sourceCode, targetCode;		/* neuron number for parent and target neuron */
			int 	targetArea;			/* connection location */
			int 	synapticType;			/* synaptic type of connection */
			double 	deliveryTime;			/* transmission time */
			double 	conductance;			/* conductance of synapse */
			double	maxConductance;			/* maximum conductance of synapse */
			double	learnParam[8];			/* learn parameter */
			Boolean sourceFound, targetFound;	/* help flags */

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetArea = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			deliveryTime = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapticType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			conductance = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			maxConductance = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trt = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trs = atof(charRead);

			sourceFound = FALSE;
			targetFound = FALSE;

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				if (currentNeuron->neuronId == sourceCode)
				{
					sourceNeuron = currentNeuron;
					sourceFound = TRUE;
				}
				if (currentNeuron->neuronId == targetCode)
				{
					targetNeuron = currentNeuron;
					targetFound = TRUE;
				}

				if (sourceFound && targetFound)
					break;
			}

			if (currentNeuron == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			connect_neurons(sourceNeuron, targetNeuron, synapticType, conductance, deliveryTime,
				trt, trs, targetArea, NULL);

			sourceNeuron->connectionList->synapseID = synapseID;
			sourceNeuron->connectionList->maxConductance = maxConductance;

			for (n = 0; n < 8; n++)
			{
				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				learnParam[n] = atof(charRead);
			}

			sourceNeuron->connectionList->slope = learnParam[0];
			sourceNeuron->connectionList->shortTermMemory = learnParam[1];
			sourceNeuron->connectionList->longTermThreshold = learnParam[2];
			sourceNeuron->connectionList->longTermMinimum = learnParam[3];
			sourceNeuron->connectionList->hebbIntervall = learnParam[4];
			sourceNeuron->connectionList->hebbLearnFactor = learnParam[5];
			sourceNeuron->connectionList->shortTermPlasticityLearnFactor = learnParam[6];
			sourceNeuron->connectionList->competitiveLearnFactor = learnParam[7];
		}

		/* read axo-axonic synapses */

		for (j = 1; j <= totalAxoSynapses; j++)
		{
			int	synapseID;			/* synapse identification number of axo-synapse */
			int	targetSynapseID;		/* synapse identification number of target synapse */
			int 	sourceCode;			/* neuron number of axo-axonic sender */
			int	sourceCodeTargetSynapse;	/* neuron number of parent neuron from target synapse */
			int	targetCodeTargetSynapse;	/* neuron number of target neuron from target synapse */
			int 	targetArea;			/* connection location */
			int 	synapticType;			/* synaptic type of connection */
			double 	deliveryTime;			/* transmission time */
			double 	inhibitionFactor;		/* inhibition factor of synapse */
			double	maxInhibitionFactor;		/* maximal inhibition factor of synapse */
			double	learnParam[8];			/* learn parameter */
			Boolean sourceFound;			/* help flag */
			Boolean sourceFoundTargetSynapse;	/* help flag */
			Boolean	targetFoundTargetSynapse;	/* help flag */

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetSynapseID = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCode = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			sourceCodeTargetSynapse = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetCodeTargetSynapse = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			targetArea = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			deliveryTime = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			synapticType = atoi(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trt = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			trs = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			inhibitionFactor = atof(charRead);

			if (! readf(fp, charRead))
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			maxInhibitionFactor = atof(charRead);

			sourceFound = FALSE;
			sourceFoundTargetSynapse = FALSE;
			targetFoundTargetSynapse = FALSE;

			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				if (currentNeuron->neuronId == sourceCode)
				{
					sourceNeuron = currentNeuron;
					sourceFound = TRUE;
				}
				if (currentNeuron->neuronId == sourceCodeTargetSynapse)
				{
					sourceNeuronTargetSynapse = currentNeuron;
					sourceFoundTargetSynapse = TRUE;
				}
				if (currentNeuron->neuronId == targetCodeTargetSynapse)
				{
					targetNeuronTargetSynapse = currentNeuron;
					targetFoundTargetSynapse = TRUE;
				}

				if (sourceFound && sourceFoundTargetSynapse && targetFoundTargetSynapse)
					break;
			}

			if (currentNeuron == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			for (currentSynapse = sourceNeuronTargetSynapse->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
			{
				if (currentSynapse->synapseID == targetSynapseID)
					break;
			}

			if (currentSynapse == NULL)
			{
				clear_network();
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}

			connect_neurons(sourceNeuron, targetNeuronTargetSynapse, synapticType, inhibitionFactor,
				deliveryTime, trt, trs, -1, currentSynapse);

			sourceNeuron->connectionList->synapseID = synapseID;
			sourceNeuron->connectionList->maxInhibitionFactor = maxInhibitionFactor;

			for (n = 0; n < 8; n++)
			{
				if (! readf(fp, charRead))
				{
					clear_network();
					if (fclose(fp) != 0)
						error(101);
					return(FALSE);
				}
				learnParam[n] = atof(charRead);
			}

			sourceNeuron->connectionList->slope = learnParam[0];
			sourceNeuron->connectionList->shortTermMemory = learnParam[1];
			sourceNeuron->connectionList->longTermThreshold = learnParam[2];
			sourceNeuron->connectionList->longTermMinimum = learnParam[3];
			sourceNeuron->connectionList->hebbIntervall = learnParam[4];
			sourceNeuron->connectionList->hebbLearnFactor = learnParam[5];
			sourceNeuron->connectionList->shortTermPlasticityLearnFactor = learnParam[6];
			sourceNeuron->connectionList->conditioningLearnFactor = learnParam[7];
		}
	} /* End of numTypes > 0 */
	if (programStates.eofReached != TRUE)
	{
		error(120);
		clear_network();
		if (fclose(fp) != 0)
			error(101);
		return(FALSE);
	}
	if (fclose(fp) != 0)
	{
		error(101);
		return(FALSE);
	}

	editNetwork = TRUE;
	show_normal_state();
	show_all_objects_in_drawing_area();
	return (TRUE);

} /* end of read_configuration */
#endif

/************************************************************************************************************************

 	FUNCTION	: readf(fp, returnString)
 
	PURPOSE		: general read function, reads buffer wise a file

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean readf(fp, returnString)
FILE *fp;		/* file pointer to current file */
char *returnString;	/* strint to be returned */
{
	char 		*stringPointer;		/* current string pointer */
	static int 	fileBufferPosition;	/* position in file buffer */
	static char  	*fileBuffer = NULL;	/* pointer to file buffer */
	int 		numberOfCharsRead;	/* number of characters yet read */

	returnString[0] = EOS;
	stringPointer = returnString;

	/* if file buffer doesn't exist create it */

	if (fileBuffer == NULL)
	{
		/* file buffer needs to be created */

		if (! (fileBuffer = (char *) malloc((size_t) (DefBufferSize * sizeof(char)))))
		{
			error(122);
			return(FALSE);
		}
		else	fileBufferPosition = DefBufferSize;
	}

	if (programStates.newRead)
	{
		programStates.eofReached = FALSE;
	}

	/* read in a junk from file if necessary */

	if ((DefBufferSize < (fileBufferPosition + NumberSize) || programStates.newRead == TRUE) &&
	    programStates.eofReached != TRUE)
	{
		if (programStates.newRead)
		{
			fileBufferPosition = 0;
			rewind(fp);
		}

		if ((DefBufferSize < (fileBufferPosition + NumberSize)) && (programStates.newRead != TRUE))
		{
			fseek(fp, (long) - (DefBufferSize - fileBufferPosition), 1);	/* seek from current position */
		}

		numberOfCharsRead = (int) fread((void *) fileBuffer, (size_t) sizeof(char), (size_t) DefBufferSize, fp);

		programStates.newRead = FALSE;

		if (numberOfCharsRead < DefBufferSize)
		{
			fileBuffer[numberOfCharsRead] = EOF;
			programStates.eofReached = TRUE;
		}

		if (feof(fp))
			programStates.eofReached = TRUE;

		fileBufferPosition = 0;
	}

	/* get next value */

	while (fileBuffer[fileBufferPosition] != EOF && fileBuffer[fileBufferPosition] != '\n')
	{
		*stringPointer++ = fileBuffer[fileBufferPosition++];
	}
	if (fileBuffer[fileBufferPosition] == EOF)
	{
		error(123);
		return(FALSE);
	}

	*stringPointer = EOS;
	fileBufferPosition++;

	/* check if return string consists of a float number */

	if (! is_it_a_float_number(returnString))
	{
		error(124);
		return(FALSE);
	}
	else
		return(TRUE);

} /* end of readf */

/************************************************************************************************************************

 	FUNCTION	: get_neuron(fp, neuron, neuronId, fileType)
 
	PURPOSE		: reads the neuron parameter

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean get_neuron(fp, neuron, neuronId, fileType)
FILE *fp;		/* file pointer to current file */
Neuron *neuron;		/* current neuron */
int neuronId;		/* neuron number of current neuron */
int fileType;		/* identification number which identifies the file type */
{
	int 	numToRead;	/* numbers to be read */
	int	startPlace;	/* offset where to be read */
	int 	j = 0;		/* loop variable */
	char 	charRead[30];	/* current number read */

	currentNeuron = neuron;

	/* read soma and dendrite parameters from file */

	numToRead = SOMA_PLACE + SOMA_VARS + DENDRITE_VARS;
	startPlace = SOMA_PLACE;

	for (j = startPlace; j < numToRead; j++)
	{
		/* read neuron parameter */

		if (! readf(fp, charRead))
		{
			clear_neuron_no_channels(currentNeuron);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}
		currentNeuron->neuronParameters[j] = atof(charRead);
	}

	/* read channel specific parameters if it's not a neuron combined file */

	if (fileType != NEURON_COMBINED_FILE)
	{
		switch(networkmodel)
		{
		/* ON/OFF model */

		case 1:	numToRead = ONOFF_PLACE + ONOFF_VARS;
			startPlace = ONOFF_PLACE;
			break;
		
		/* SWIM model */

		case 2: numToRead = SWIM_PLACE + SWIM_VARS;
			startPlace = SWIM_PLACE;
			break;
		
		/* Hodgkin-Huxley model */

		case 3: numToRead = HH_PLACE + HH_VARS;
			startPlace = HH_PLACE;
			break;
		
		/* Golowasch-Buchholz model */

		case 4: numToRead = GB_PLACE + GB_VARS;
			startPlace = GB_PLACE;
			break;
		}

		for (j = startPlace; j < numToRead; j++)
		{
			/* read neuron parameter */

			if (! readf(fp, charRead))
			{
				clear_neuron_no_channels(currentNeuron);
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			currentNeuron->neuronParameters[j] = atof(charRead);
		}

		/* set neuron id */

		if (neuronId > 0)
			currentNeuron->neuronId = neuronId;
		else	currentNeuron->neuronId = 0;	/* value zero is used, because the read in value was negative */
	}

	return (TRUE);

} /* end of get_neuron */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: read_on_off_channels(neuron)
 
	PURPOSE		: read in on-off channel parameters

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_on_off_channels(neuron)
Neuron *neuron;
{
	FILE	*fp;		/* filepointer to onoff.ch file */
	char	charRead[30];	/* current number read */
	int	i;		/* loop index */

	if (neuron == neuronListHead)	/* should the file onoff.ch be read in ? */
	{
		if ((fp = fopen(onoffChannelFile, "r")) == NULL)
		{
			error(115);
			return(FALSE);
		}

		programStates.newRead = TRUE;

		for (i = 0; i < ONOFF_VARS; i++)
		{
			if (! readf(fp, charRead))
			{
				error(125);
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			neuron->neuronParameters[ONOFF_PLACE+i] = atof(charRead);
		}

		/* if the end of file is not reached yet output a warning message */

		if (programStates.eofReached != TRUE)
		{
			error(120);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		if (fclose(fp) != 0)
		{
			error(101);
			return(FALSE);
		}
	}
	else    /* the values only have to be copied from the first neuron in the net */
	{
		for (i = 0; i < ONOFF_VARS; i++)
			neuron->neuronParameters[ONOFF_PLACE+i] = neuronListHead->neuronParameters[ONOFF_PLACE+i];
	}

	return(TRUE);

} /* end of read_on_off_channels */

/************************************************************************************************************************

 	FUNCTION	: read_swim_channels(neuron)
 
	PURPOSE		: read in SWIM channel parameters

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_swim_channels(neuron)
Neuron *neuron;
{
	FILE	*fp;		/* filepointer to swim.ch file */
	char	charRead[30];	/* current number read */
	int	i;		/* loop index */

	if (neuron == neuronListHead)	/* should the file swim.ch be read in ? */
	{
		if ((fp = fopen(swimChannelFile, "r")) == NULL)
		{
			error(115);
			return(FALSE);
		}

		programStates.newRead = TRUE;

		for (i = 0; i < SWIM_VARS; i++)
		{
			if (! readf(fp, charRead))
			{
				error(125);
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			neuron->neuronParameters[SWIM_PLACE+i] = atof(charRead);
		}

		/* if the end of file is not reached yet output a warning message */

		if (programStates.eofReached != TRUE)
		{
			error(120);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		if (fclose(fp) != 0)
		{
			error(101);
			return(FALSE);
		}
	}
	else    /* the values only have to be copied from the first neuron in the net */
	{
		for (i = 0; i < SWIM_VARS; i++)
			neuron->neuronParameters[SWIM_PLACE+i] = neuronListHead->neuronParameters[SWIM_PLACE+i];
	}

	return(TRUE);

} /* end of read_swim_channels */

/************************************************************************************************************************

 	FUNCTION	: read_HH_channels(neuron)
 
	PURPOSE		: read in Hodgkin-Huxley channel parameters

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_HH_channels(neuron)
Neuron *neuron;
{
	FILE	*fp;		/* filepointer to hodgkin_huxley.ch file */
	char	charRead[30];	/* current number read */
	int	i;		/* loop index */

	if (neuron == neuronListHead)	/* should the file hodgkin_huxley.ch be read in ? */
	{
		if ((fp = fopen(hhChannelFile, "r")) == NULL)
		{
			error(115);
			return(FALSE);
		}

		programStates.newRead = TRUE;

		for (i = 0; i < HH_VARS; i++)
		{
			if (! readf(fp, charRead))
			{
				error(125);
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			neuron->neuronParameters[HH_PLACE+i] = atof(charRead);
		}

		/* if the end of file is not reached yet output a warning message */

		if (programStates.eofReached != TRUE)
		{
			error(120);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		if (fclose(fp) != 0)
		{
			error(101);
			return(FALSE);
		}
	}
	else    /* the values only have to be copied from the first neuron in the net */
	{
		for (i = 0; i < HH_VARS; i++)
			neuron->neuronParameters[HH_PLACE+i] = neuronListHead->neuronParameters[HH_PLACE+i];
	}
	return(TRUE);

} /* end of read_HH_channels */

/************************************************************************************************************************

 	FUNCTION	: read_golowasch_channels(neuron)
 
	PURPOSE		: read in Golowasch-Buchholz channel parameters

	RETURNS		: TRUE if no error has been occured during reading, else FALSE

************************************************************************************************************************/

Boolean read_golowasch_channels(neuron)
Neuron *neuron;
{
	FILE	*fp;		/* filepointer to golowasch.ch file */
	char	charRead[30];	/* current number read */
	int	i;		/* loop index */

	if (neuron == neuronListHead)	/* should the file golowasch.ch be read in ? */
	{
		if ((fp = fopen(golowaschChannelFile, "r")) == NULL)
		{
			error(115);
			return(FALSE);
		}

		programStates.newRead = TRUE;

		for (i = 0; i < GB_VARS; i++)
		{
			if (! readf(fp, charRead))
			{
				error(125);
				if (fclose(fp) != 0)
					error(101);
				return(FALSE);
			}
			neuron->neuronParameters[GB_PLACE+i] = atof(charRead);
		}

		/* if the end of file is not reached yet output a warning message */

		if (programStates.eofReached != TRUE)
		{
			error(120);
			if (fclose(fp) != 0)
				error(101);
			return(FALSE);
		}

		if (fclose(fp) != 0)
		{
			error(101);
			return(FALSE);
		}
	}
	else    /* the values only have to be copied from the first neuron in the net */
	{
		for (i = 0; i < GB_VARS; i++)
			neuron->neuronParameters[GB_PLACE+i] = neuronListHead->neuronParameters[GB_PLACE+i];
	}
	return(TRUE);

} /* end of read_golowasch_channels */
#endif
