/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	calculations.c

	PURPOSE:	Functions for calculating the network action

	FUNCTIONS:	in order of appearance

	run_simulation				- central simulation function used for foreground processing
	run_simulation_step			- central simulation function used for background processing
	init_learn_parameter			- initializes necessary learning parameter
	initialize_variables			- initializes necessary neuronal variables
	init_pools				- initializes necessary synaptic variables
	integrate_method			- do the integration process
	calculate_form_parameter		- calculate the form parameter
	calculate_new_learning_values		- calculate new values needed for learning process
	calculate_synapses			- calculates new entrees into synaptic pools
	calculate_synaptic_influences		- calculates synaptical currents
	calculate_onoff_neuron			- calculates a On/Off neuron
	calculate_HH_neuron			- calculates a Hodgkin-Huxley neuron
	calculate_SWIM_neuron			- calculates a SWIM neuron
	calculate_GB_neuron			- calculates a Golowasch-Buchholz neuron

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "calculations.h"
#include "functions.h"

/****** global variables ******/

#ifndef BATCH
double neuronRange;			/* working range of a neuron */
double topVoltage;			/* top voltage of a neuron */
double bottomVoltage;			/* bottom voltage of a neuron */
int neuronActivityInterval = NACTINT_DEF;/* neuron activity interval */
Boolean showActivityColors;		/* option to give neurons colors durings simulation */
#endif
double L = 0.0;				/* actual simulation time in s */
Boolean noiseDefined = FALSE;		/* indicated if noise should be added to neurons */
Boolean stochasticInitialPotentials = FALSE;	/* indicates if initial potentials should be stochastic */
Boolean stochasticSynapticStrengths = FALSE;	/* indicates if synaptic strengths should be stochastic */
Boolean stochasticTransmissionTimes = FALSE;	/* indicates if syn. transmission times should be stochastic */
Boolean graphSim = FALSE;		/* indicates if a simulation has been run */
Boolean variables_initialized = FALSE;	/* indicates if simulation variables are initialized */
Boolean resetLearningParameter = TRUE;	/* indicates if learning parameter should be initialized */
SimulationParams simulation = 
	{ (double) (SIMLEN_DEF / T_SCALE), (double) (SIMSIZE_DEF / T_SCALE) };	/* simulation params during simulation */
int integrationMethod = EXPEULER;	/* integration method used in calculations
					   1 = Euler, 2 = exponential Euler, 3 = Trapezoid, 4 = Runge-Kutta */
int simulationTime;			/* time during simulation process in simulation steps */
int networkmodel = 2;			/* 1 = ON/OFF, 2 = SWIM, 3 = Hodgkin-Huxley, 4 = Golowasch-Buchholz */
Boolean complexModel = FALSE;		/* TRUE if network model type is complex */
int networktype;			/* 1 = 1 point, 2 = 2 point, 3 = 4 point, 4 = n point */
int longTermAdaptiveFactor = 40000;	/* adaption factor for long term forgetting curve */

/****** local variables ******/

static double HH_SLmSG;			/* hodgkin-huxley leak current */
static noise = 0.0;			/* used for calculation of noise for neurons */
static Boolean errorFlag;		/* signals, if an error has been encountered during simulation */

/****** functions ******/

/************************************************************************************************************************

 	FUNCTION	: run_simulation(void)
 
	PURPOSE		: central simulation function used for foreground processing

	RETURNS		: nothing

************************************************************************************************************************/

void run_simulation()
{
	char text[100];		/* text for output ellapsed time */
	int start, end;		/* start and end time of simulation in s */
	int processorTime;	/* processor time the calculation has needed */
#ifndef BATCH
	Arg args[10];           /* argument list for manipulating widgets */
	Cardinal n;             /* used as argument counter for manipulating widgets */
	Neuron 	*neuron;	/* current neuron of the graph */
	Connection *synapse;	/* current synapse of the graph */
	Dimension width, height;/* dimension of the graph */
	Dimension oldwidth, oldheight;	/* old width and height of the drawingArea */
	Position x, y;		/* position of the graph */
	double temp;		/* temporary variable */
	XmString text2;		/* text for output ellapsed time */
	double simLen;		/* simulation length in s */
	XEvent event;		/* event */


	/* compute neuron working range */

	switch(networkmodel)
	{
	case 1:	/* ON/OFF model */

		topVoltage = NAEQ_ONOFF_DEF / E_SCALE;
		bottomVoltage = KEQ_ONOFF_DEF / E_SCALE;
		break;
	
	case 2:	/* SWIM model */
	default:

		topVoltage = NAEQ_SWIM_DEF / E_SCALE;
		bottomVoltage = KEQ_SWIM_DEF / E_SCALE;
		break;
	
	case 3:	/* Hodgkin-Huxley model */

		topVoltage = NAEQ_HH_DEF / E_SCALE;
		bottomVoltage = KEQ_HH_DEF / E_SCALE;
		break;

	case 4:	/* Golowasch-Buchholz model */

		topVoltage = NAEQ_GB_DEF / E_SCALE;
		bottomVoltage = KEQ_GB_DEF / E_SCALE;
		break;
	}
	neuronRange = topVoltage - bottomVoltage;

	/* set cursor to busy cursor */

	cursor = XCreateFontCursor(disp, XC_watch);
	XDefineCursor(disp, XtWindow(mainWindow), cursor);

	/* if anything is selected desect it */

	remove_selected_objects_from_selectionList();
#endif

	/****** initialization part ******/

	L = 0.0;
	errorFlag = FALSE;
	if ((! variables_initialized) || autoReset)
	{
		initialize_variables();
		init_pools();
		variables_initialized = TRUE;
	}

	read_input_currents();	/* if there are a input current file for this simulation read it */

	/* for each neuron set type of neuron and initialize graph output if possible */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		switch(networkmodel)
		{
		case 1:	/* ON/OFF model */

			if (ONOFF_NA_CONDUCTANCE == 0.0 && ONOFF_K_CONDUCTANCE == 0.0)
				currentNeuron->neuronType = nonspiker;
			else	currentNeuron->neuronType = spiker;
			break;
		
		case 2:	/* SWIM model */
		default:

			if (SWIM_NA_G == 0.0 && SWIM_K_G == 0.0)
				currentNeuron->neuronType = nonspiker;
			else	currentNeuron->neuronType = spiker;
			break;
		
		case 3:	/* Hodgkin-Huxley model */

			if (HH_NA_G == 0.0 && HH_K_G == 0.0)
				currentNeuron->neuronType = nonspiker;
			else	currentNeuron->neuronType = spiker;
			break;

		case 4:	/* Golowasch-Buchholz model */

			if (GB_NA_G == 0.0 && GB_K_G == 0.0)
				currentNeuron->neuronType = nonspiker;
			else	currentNeuron->neuronType = spiker;
			break;
		}

#ifndef BATCH
		/* set cursor to busy */

		if (currentNeuron->graph)
			XDefineCursor(disp, XtWindow(currentNeuron->graph->formWidget), cursor);

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
			if (currentSynapse->graph)
				XDefineCursor(disp, XtWindow(currentSynapse->graph->formWidget), cursor);
#endif
	}

	graph_output(0);	/* initialize first neuron value for graphing */

	/* initialization in case of output is to be shown during simulation */

#ifndef BATCH
	if (showOutputImmediately)
	{
		graphSim = TRUE;	/* this is needed to show the output data in a graph immediately */

		/* initialize all existing graphs for the neurons */

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			/* initialize neuron graphs */

			if (currentNeuron->graph)
			{
				simLen = LTSTOP;		/* store simulation length */
				LTSTOP = 0.0;			/* set simulation length to zero */
				draw_graph(currentNeuron);	/* clear graph, but show axises and grids */
				LTSTOP = simLen;		/* restore simulation length */
				currentNeuron->graph->hasValues = TRUE;	/* mark graph has values */
				currentNeuron->graph->updateTime = 0.0;	/* initialize updated time to zero */
			}

			/* initialize synapse graphs */

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
				if (currentSynapse->graph)
				{
					simLen = LTSTOP;		/* store simulation length */
					LTSTOP = 0.0;			/* set simulation length to zero */
					draw_graphS(currentSynapse);	/* clear graph, but show axises and grids */
					LTSTOP = simLen;		/* restore simulation length */
					currentSynapse->graph->hasValues = TRUE; /* mark graph has values */
					currentSynapse->graph->updateTime = 0.0; /* initialize updated time to zero */
				}
		}
	}
	else	graphSim = FALSE;		/* simulation has not yet been run and nothing has to be plotted out */
#endif
	simulationTime = 0;			/* initialize simulated time in simulation steps */

	/* start of simulation */

#ifdef BATCH
	printf("Now starting simulation ... Please wait\n");
#endif
	start = clock();	/* take the start time of the simulation */

	/****** main loop through simulation ******/

	while ((L < LTSTOP) && ! errorFlag)
	{
		/* check, if a F11 function key was pressed then return */

#ifndef BATCH
		if (XCheckTypedEvent(disp, KeyPress, &event))
			if (XLookupKeysym(&(event.xkey), 0) == XK_F11)
				break;
#endif

		/* calculate pool influence & pop last values */

		calculate_synaptic_influences();

		/* calculate neurons */

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			switch (networkmodel)
			{
			case 1:
				calculate_onoff_neuron();
				break;

			case 2:
				calculate_SWIM_neuron();
				break;

			case 3:
				calculate_HH_neuron();
				break;
			
			case 4:
				calculate_GB_neuron();
				break;
			}
#ifndef BATCH
			if (showActivityColors && (simulationTime % neuronActivityInterval == 0))
				set_neuron_color();
#endif
		} /* End of for */

		/* calculate synaptic outputs */

		calculate_synapses();

		/* store computed values */

		graph_output(++simulationTime);

		/* if output is showed during simulation update graphs each 100 simulation steps */

#ifndef BATCH
		if (showOutputImmediately)
			for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
			{
				if (currentNeuron->graph != NULL && simulationTime % 100 == 0)
					update_graph(currentNeuron);
				
				for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
				     currentSynapse = currentSynapse->nextConnection)
				{
					if (currentSynapse->graph != NULL && simulationTime % 100 == 0)
						update_graphS(currentSynapse);
				}
			}
#endif

		L += STEP;		/* increment the simulation time */

	} /* END of while */

	if (L > LTSTOP)
		L = LTSTOP;

	/* if output is showed during simulation update graphs until simulation end */

#ifndef BATCH
	if (showOutputImmediately)
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->graph != NULL)
				update_graph(currentNeuron);

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
			{
				if (currentSynapse->graph != NULL && simulationTime % 100 == 0)
					update_graphS(currentSynapse);
			}
		}
#endif

	end = clock();				/* stop time of simulation */
	processorTime = end - start;		/* calculate processed time */

#ifndef BATCH
	XSync(disp, FALSE);			/* synchronize X-server */
	graphSim = TRUE;			/* simulation has now been run */

	/* show graph output if not yet done */

	if (! showOutputImmediately ||
	   ((commandMode == 1) && (strcmp(outputFilename, "/dev/null") != 0)))
		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			if (currentNeuron->graph != NULL)
			{
				currentNeuron->graph->hasValues = TRUE;
				draw_graph(currentNeuron);				/* output graph with traces */
				if (commandMode == 1)					/* not interactive */
					save_graph(outputFilename, currentNeuron);	/* save graph as file */
			}

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
			{
				if (currentSynapse->graph != NULL)
				{
					currentSynapse->graph->hasValues = TRUE;
					draw_graphS(currentSynapse);			/* output graph with traces */
					if (commandMode == 1)				/* not interactive */
						save_graphS(outputFilename, currentSynapse); /* save graph as file */
				}
			}
		}

	/* set cursor to normal */

	XUndefineCursor(disp, XtWindow(mainWindow));

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		if (currentNeuron->graph)
			XUndefineCursor(disp, XtWindow(currentNeuron->graph->formWidget));


		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
		{
			if (currentSynapse->graph)
				XUndefineCursor(disp, XtWindow(currentSynapse->graph->formWidget));
		}
	}
#endif

	/* show processing info is this option was selected */

	if (showProcessingInfo)
	{
		sprintf(text, "%s %f", ellapsedTimeString, (float) (processorTime / 1.0E6));
#ifndef BATCH
		text2 = XmStringCreateLtoR(text, defaultFont);
		XtManageChild(create_information_box(mainWindow, XmDIALOG_APPLICATION_MODAL, text2));
		XmStringFree(text2);
#else
		printf("%s\n", text);
#endif
	}

#ifndef BATCH
	if (commandMode == 1)	/* not interactive mode ? */
	{
		/* switch to interactive mode again */

		commandMode = 0;

		/* remove command file */

		if (remove(commandFile) != 0)
			error(103);

		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
	}
#else
	printf("\nNow saving data to file %s\n", outputFilename);

	/* save graph values to file */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		if (currentNeuron->graph)
			save_graph(outputFilename, currentNeuron);	/* save graph as file */
		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
			if (currentSynapse->graph)
				save_graphS(outputFilename, currentSynapse); /* save graph as file */
	}
#endif
} /* end of run_simulation */

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: run_simulation_step(steps)
 
	PURPOSE		: central simulation function used for background processing

	RETURNS		: TRUE, if function has to be removed from background processing, else FALSE

************************************************************************************************************************/

Boolean run_simulation_step(steps)
int steps;	/* number of simulation steps */
{
	Arg args[10];		/* argument list for manipulating widgets */
	Cardinal n;		/* used as argument counter for manipulating widgets */
	char text[20]; 		/* time in ms for simulation completed string */
	XmString text2;		/* text for quit string */
	int length;		/* help variable for updating simulation completed string */
	double simLen;		/* simulation length in s */
	double startTime;	/* start time of this routine because of background processing */

	/* if simulated time is greater than simulation length, return and remove function from background processing */

	if (L >= LTSTOP)
		return(TRUE);

	/****** initialization part ******/

	/* compute neuron working range */

	switch(networkmodel)
	{
	case 1:	/* ON/OFF model */

		topVoltage = NAEQ_ONOFF_DEF / E_SCALE;
		bottomVoltage = KEQ_ONOFF_DEF / E_SCALE;
		break;
	
	case 2:	/* SWIM model */
	default:

		topVoltage = NAEQ_SWIM_DEF / E_SCALE;
		bottomVoltage = KEQ_SWIM_DEF / E_SCALE;
		break;
	
	case 3:	/* Hodgkin-Huxley model */

		topVoltage = NAEQ_HH_DEF / E_SCALE;
		bottomVoltage = KEQ_HH_DEF / E_SCALE;
		break;

	case 4:	/* Golowasch-Buchholz model */

		topVoltage = NAEQ_GB_DEF / E_SCALE;
		bottomVoltage = KEQ_GB_DEF / E_SCALE;
		break;
	}
	neuronRange = topVoltage - bottomVoltage;

	/* set cursor to busy cursor */

	cursor = XCreateFontCursor(disp, XC_watch);
	XDefineCursor(disp, XtWindow(mainWindow), cursor);

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		if (currentNeuron->graph)
			XDefineCursor(disp, XtWindow(currentNeuron->graph->formWidget), cursor);

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
			if (currentSynapse->graph)
				XDefineCursor(disp, XtWindow(currentSynapse->graph->formWidget), cursor);
	}

	if (L == 0.0)	/* is it the first time this routine is called during simulation */
	{
		errorFlag = FALSE;
		remove_selected_objects_from_selectionList();	/* if anything is selected deselect it */

		if ((! variables_initialized) || autoReset)
		{
			initialize_variables();
			init_pools();
			variables_initialized = TRUE;
		}

		read_input_currents();	/* if there are a input current file for this simulation read it */

		graphSim = TRUE;	/* we want see the simulation results immediateley, so this must be true */

		/* for each neuron set type of neuron and initialize graph output if possible */

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			switch(networkmodel)
			{
			case 1:	/* ON/OFF model */

				if (ONOFF_NA_CONDUCTANCE == 0.0 && ONOFF_K_CONDUCTANCE == 0.0)
					currentNeuron->neuronType = nonspiker;
				else	currentNeuron->neuronType = spiker;
				break;
			
			case 2:	/* SWIM model */
			default:

				if (SWIM_NA_G == 0.0 && SWIM_K_G == 0.0)
					currentNeuron->neuronType = nonspiker;
				else	currentNeuron->neuronType = spiker;
				break;
			
			case 3:	/* Hodgkin-Huxley model */

				if (HH_NA_G == 0.0 && HH_K_G == 0.0)
					currentNeuron->neuronType = nonspiker;
				else	currentNeuron->neuronType = spiker;
				break;

			case 4:	/* Golowasch-Buchholz model */

				if (GB_NA_G == 0.0 && GB_K_G == 0.0)
					currentNeuron->neuronType = nonspiker;
				else	currentNeuron->neuronType = spiker;
				break;
			}

			/* initialize all existing graphs for the neurons */

			if (currentNeuron->graph)
			{
				simLen = LTSTOP;		/* store simulation length */
				LTSTOP = 0.0;			/* set simulation length to zero */
				draw_graph(currentNeuron);	/* clear graph, but show axises and grids */
				LTSTOP = simLen;		/* restore simulation length */
				currentNeuron->graph->hasValues = TRUE;	/* mark graph has values */
				currentNeuron->graph->updateTime = 0.0;	/* initialize updated time to zero */
			}

			/* initialize synapse graphs */

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
				if (currentSynapse->graph)
				{
					simLen = LTSTOP;		/* store simulation length */
					LTSTOP = 0.0;			/* set simulation length to zero */
					draw_graphS(currentSynapse);	/* clear graph, but show axises and grids */
					LTSTOP = simLen;		/* restore simulation length */
					currentSynapse->graph->hasValues = TRUE; /* mark graph has values */
					currentSynapse->graph->updateTime = 0.0; /* initialize updated time to zero */
				}
		}

		graph_output(0);			/* initialize first neuron value for graphing */

		simulationTime = 0;			/* initialize simulated time in simulation steps */
		XSync(disp, FALSE);			/* synchronize X-server */
	}

	/* start of simulation */

	startTime = L;	/* store start time of simulation */

	/****** main loop through simulation ******/

	while ((L < startTime + steps * STEP) && (L < LTSTOP) && ! errorFlag)
	{

		/* calculate pool influence & pop last values */

		calculate_synaptic_influences();

		/* calculate neurons */

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			switch (networkmodel)
			{
			case 1:
				calculate_onoff_neuron();
				break;

			case 2:
				calculate_SWIM_neuron();
				break;

			case 3:
				calculate_HH_neuron();
				break;
			
			case 4:
				calculate_GB_neuron();
				break;
			}

			if (showActivityColors && (simulationTime % neuronActivityInterval == 0))
				set_neuron_color();

		} /* End of for */

		/* calculate synaptic outputs */

		calculate_synapses();

		L += STEP;		/* increment the simulation time */

		/* store the computed values */

		graph_output(++simulationTime);
	}

	if (L > LTSTOP)
		L = LTSTOP;

	/* update the existing graphs to the current simulation time */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		if (currentNeuron->graph)
		{
			update_graph(currentNeuron);

			/* set cursor to normal */

			XUndefineCursor(disp, XtWindow(currentNeuron->graph->formWidget));
		}

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
		{
			if (currentSynapse->graph)
			{
				update_graphS(currentSynapse);

				/* set cursor to normal */

				XUndefineCursor(disp, XtWindow(currentSynapse->graph->formWidget));
			}
		}
	}

	XUndefineCursor(disp, XtWindow(mainWindow));

	/* update the text in the simulation information box */

	sprintf(text, "%d", (int) (rint(L * T_SCALE * 10.0)));
	length = (int) strlen(text);
	text[length+1] = '\0';
	text[length] = text[length-1];
	text[length-1] = '.';
	XmTextSetString(simInfCompletedText, text);

	/* if simulation was finished redefine stop button in simulation information box to quit button */

	if (L >= LTSTOP)
	{
		n = 0;
		text2 = XmStringCreateLtoR(quitString, defaultFont);
		XtSetArg(args[n], XmNlabelString, text2); n++;
		XtSetValues(simInfStopButton, args, n);
		XmStringFree(text2);

		XSync(disp, FALSE);			/* synchronize X-server */
		return(TRUE);				/* remove working procedure */
	}

	if (steps == 1)		/* single step mode ? */
		return(TRUE);	/* remove working procedure */
	else return(FALSE);	/* stay as a working procedure */

} /* of run_simulation_steps */
#endif

/************************************************************************************************************************

 	FUNCTION	: init_learn_parameter(synapse)
 
	PURPOSE		: initializes necessary learning parameters

	RETURNS		: nothing

************************************************************************************************************************/

void init_learn_parameter(synapse)
Connection *synapse;
{
	currentSynapse = synapse;
	synapse->currentInhibitionFactor = synapse->initialInhibitionFactor;
	synapse->synapticMemorySTP = 0.01;
	synapse->synapticMemoryHebb = 0.01;
	synapse->synapticMemoryCond = 0.01;
	LEXP = exp(- 400 * synapse->slope * STEP);
	SFCONST = STEP / synapse->shortTermMemory;
	LFCONST = STEP / longTermAdaptiveFactor;
	synapse->trainingFlagSTP = FALSE;
	synapse->trainingFlagHebb = FALSE;
	synapse->trainingFlagCond = FALSE;
	synapse->shortTermLearningCondition = FALSE;
	synapse->conditioningLearningCondition = FALSE;
	synapse->hebbLearningCondition = FALSE;
	synapse->competitiveLearningCondition = FALSE;
	synapse->hebbIntervallCounter = 0.0;
	synapse->hebbTimer = 0.0;
	synapse->stmCnt = 0;
	synapse->condCnt = 0;
	synapse->hebbCnt = 0;

} /* end of init_learn_parameter */

/************************************************************************************************************************

 	FUNCTION	: initialize_variables(void)
 
	PURPOSE		: initializes necessary neuronal variables

	RETURNS		: nothing

************************************************************************************************************************/

void initialize_variables()
{
	int I;			/* general help variable */

	noise = 0.0;	/* reset noise variable */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* sets general information */

		SpikeCounter = 0;
		PseudoSomaCur = 0.0;
		PseudoSCSTRT = 0.0;
		PseudoSCSTP = 0.0;
		IRegion = ActualIRegion;
		IStem = ActualIStem;
#ifndef BATCH
		if (currentNeuron->graph != NULL)
		{
			currentNeuron->graph->somaBound = FALSE;
			currentNeuron->graph->dendBound = FALSE;
			currentNeuron->graph->synCBound = FALSE;
			currentNeuron->graph->totCBound = FALSE;
			currentNeuron->graph->iNBound = FALSE;
			currentNeuron->graph->iNABound = FALSE;
			currentNeuron->graph->iKBound = FALSE;
			currentNeuron->graph->iCABound = FALSE;
			currentNeuron->graph->iCA1Bound = FALSE;
			currentNeuron->graph->iCA2Bound = FALSE;
			currentNeuron->graph->iKCABound = FALSE;
			currentNeuron->graph->iABound = FALSE;
			currentNeuron->graph->iIRBound = FALSE;
			currentNeuron->graph->iPBound = FALSE;
		}
#endif

		/* sets number of compartments dependend information */

		switch(networktype)
		{
		case 1:
			PseudoSomaCur = DendCur;
			PseudoSCSTRT = SDSTRT;
			PseudoSCSTP = SDSTP;

			IRegion = 0;
			IStem = 1;
			DENDRITE_STEMS = 0;
			DENDRITE_REGIONS = 0;

			break;
		case 2:
			DENDRITE_STEMS = 1;
			DENDRITE_REGIONS = 1;
			IRegion = 1;
			IStem = 1;

			break;
		case 3:
			DENDRITE_STEMS = 1;
			DENDRITE_REGIONS = 3;
			if (IRegion > 3) IRegion = 3;
			IStem = 1;

			break;
		case 4:
			DENDRITE_STEMS = (int) ACTUAL_DENDRITIC_STEMS;
			DENDRITE_REGIONS = (int) ACTUAL_DENDRITIC_REGIONS;
			if (IRegion > curNumOfDendSegs) IRegion = curNumOfDendSegs;

			break;
		}

		/* initialize model parameter */

		switch(networkmodel)
		{
		case 1:	/* ON/OFF model */

			/* calculate the length of a spike */

			SPIKE = ONOFF_NA_DURATION;

			if (SPIKE < ONOFF_K_DURATION + ONOFF_K_START)
				SPIKE = ONOFF_K_DURATION + ONOFF_K_START;

			if (SPIKE < ONOFF_CA_DURATION + ONOFF_CA_START)
				SPIKE = ONOFF_CA_DURATION + ONOFF_CA_START;

			SPIKE_IN_PROCESS = FALSE;

			break;
		
		case 2:	/* SWIM model */

			/* initialize gate variables */

			M_CONST = SWIM_M_INITIAL;
			N_CONST = SWIM_N_INITIAL;
			H_CONST = SWIM_H_INITIAL;
			C_CONST = SWIM_C_INITIAL;
			GB_I_CA1 = M_CONST / I_SCALE;
			GB_I_CA2 = H_CONST / I_SCALE;
			GB_I_KCA = N_CONST / I_SCALE;
			GB_I_A = C_CONST / I_SCALE;

			/* calculate constants for speed up SWIM simulation */

			SWIM_NAEmNAG = SWIM_NA_E * SWIM_NA_G;
			SWIM_KEmKG = SWIM_K_E * SWIM_K_G;
			SWIM_CAEmCAG = SWIM_CA_E * SWIM_CA_G;

			break;
		
		case 3:	/* HH model */

			/* initialize gate variables */

			M_CONST = HH_M_INITIAL;
			N_CONST = HH_N_INITIAL;
			H_CONST = HH_H_INITIAL;
			GB_I_CA1 = M_CONST / I_SCALE;
			GB_I_CA2 = H_CONST / I_SCALE;
			GB_I_KCA = N_CONST / I_SCALE;

			/* calculate constants for speed up Hodgkin-Huxley simulation */

			HH_NAEmNAG = HH_NA_E * HH_NA_G;
			HH_KEmKG = HH_K_E * HH_K_G;
			HH_SLmSG = -0.054402 * SOMA_REST_Gm;

			break;
		
		case 4:	/* GB model */

			/* initialize gate variables */

			M_CONST = GB_M_INITIAL;
			N_CONST = GB_N_INITIAL;
			H_CONST = GB_H_INITIAL;
			C_CONST = GB_C_INITIAL;
			D_CONST = GB_D_INITIAL;
			E_CONST = GB_E_INITIAL;
			A_CONST = GB_A_INITIAL;
			B1_CONST = GB_B1_INITIAL;
			B2_CONST = GB_B2_INITIAL;
			P_CONST = GB_P_INITIAL;
			Q_CONST = GB_Q_INITIAL;
			R_CONST = GB_R_INITIAL;
			S_CONST = GB_S_INITIAL;

			/* initialize calcium currents and calcium concenctration */

			GB_I_CA1 = 0.0;
			GB_I_CA2 = 0.0;
			GB_CONC_CA = GB_CA_INITIAL;

			/* calculate constants for speed up Golowasch-Buchholz simulation */

			GB_NAEmNAG = GB_NA_E * GB_NA_G;
			GB_KEmKG = GB_K_E * GB_K_G;
			GB_AEmAG = GB_A_E * GB_A_G;
			GB_KCAEmKCAG = GB_KCA_E * GB_KCA_G;
			GB_IREmIRG = GB_IR_E * GB_IR_G;
			GB_PEmPG = GB_P_E * GB_P_G;

			break;
		}

		if (DENDRITE_STEMS == 0 || DENDRITE_REGIONS == 0)
		{
			DENDRITE_STEMS = 0;
			DENDRITE_REGIONS = 0;
		}

		/* calculate constants for speed up simulation */

		SLmSG = SOMA_Eleak_INITIAL * SOMA_REST_Gm;
		SGpDSmDG = SOMA_REST_Gm + DENDRITE_STEMS * DENDRITE_Gcore;
		DLmDG = DENDRITE_Eleak_INITIAL * DENDRITE_REST_Gm;
		SGdSC = SOMA_REST_Gm / SOMA_Cm;
		SGpDSmDGdSC = (SOMA_REST_Gm + DENDRITE_STEMS * SOMA_Gcore) / SOMA_Cm;

		/* initialize compartment potentials */

		if (stochasticInitialPotentials)	/* initial potentials should be different */
		{
			switch(networkmodel)
			{
			case 1: /* On-Off model */

				E_POT[0] = normal_random(stdInitialPotential, stdInitialPotentialStd,
					ONOFF_K_E, ONOFF_NA_E);
				break;

			case 2: /* SWIM model */

				E_POT[0] = normal_random(stdInitialPotential, stdInitialPotentialStd,
					SWIM_K_E, SWIM_NA_E);
				break;

			case 3: /* Hodgkin-Huxley model */

				E_POT[0] = normal_random(stdInitialPotential, stdInitialPotentialStd,
					HH_K_E, HH_NA_E);
				break;

			case 4: /* Golowasch-Buchholz model */

				E_POT[0] = normal_random(stdInitialPotential, stdInitialPotentialStd,
					GB_K_E, GB_NA_E);
				break;

			default: fprintf(stderr, "Wrong model type during initialization of variables\n");
				 break;
			}

			for (I = 1; I <= curNumOfDendSegs; I++)
				E_POT[I] = E_POT[0];
		}
		else	/* all initial potentials should be the same */
		{
			E_POT[0] = SOMA_E_INITIAL;
			for (I = 1; I <= curNumOfDendSegs; I++)
				E_POT[I] = DENDRITE_E_INITIAL;
		}

	} /* End of for */

} /* of initialize_variables */

/************************************************************************************************************************

 	FUNCTION	: init_pools(void)
 
	PURPOSE		: initializes necessary synaptic variables

	RETURNS		: nothing

************************************************************************************************************************/

void init_pools()
{
	AxoAxonicSenderList *axoSender;		/* current axo-axonic sender */
	ChemicalSenderList *currentSender;	/* current sender */
	ChemicalPool *currentPool;		/* current pool */
	double oldConductance;			/* old synaptic conductance */
	int queuePlaces;			/* length of process queue */
	int I;					/* general help variable */

	/* loop through neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* loop through pools */

		for (currentPool = currentNeuron->poolList; currentPool != NULL; currentPool = currentPool->nextPool)
		{
			currentPool->longestDeliveryTime = 0;	/* initialize longest delivery time */

			/* loop through senders */

			for (currentSender = currentPool->senderList; currentSender != NULL;
			     currentSender = currentSender->nextSender)
			{
				/* calculate necessary synaptical constants */

				currentSynapse = currentSender->connection;

				/* check if synaptic strength should be normal distributed */

				if (stochasticSynapticStrengths)
				{
					oldConductance = currentSynapse->initialConductance;
					currentSynapse->initialConductance = normal_random(stdSynapticStrength,
						stdSynapticStrengthStd, 0.0, currentSynapse->maxConductance);

					/* change total Conductance of pool */

					currentSynapse->connectionPool->totalConductance -= oldConductance;
					currentSynapse->connectionPool->totalConductance += currentSynapse->initialConductance;
				}
				currentSynapse->conductance = currentSynapse->initialConductance;

				/* check if synaptic transmission time should be normal distributed */

				if (stochasticTransmissionTimes)
				{
					currentSynapse->deliveryTime = normal_random(stdSynapticTT, stdSynapticTTStd,
						STEP, TT_MAX / T_SCALE);
				}

				if (currentSynapse->deliveryTime > currentPool->longestDeliveryTime)
					currentPool->longestDeliveryTime = currentSynapse->deliveryTime;

				TSmTRT = currentSynapse->transmitterSatiate -
					 currentSynapse->transmitterReleaseThreshold;
				TTdSS = (int) rint(currentSynapse->deliveryTime / STEP);

				/* if comlex network model: initialize all axo-axonic synapses to the currentSynapse */

				if (complexModel)
				{
					/* init learn parameter */

					if (resetLearningParameter)
						init_learn_parameter(currentSynapse);

					/* loop through axo-axonic connections */

					for (axoSender = currentSynapse->axoSenderList; axoSender != NULL;
					     axoSender = axoSender->nextSender)
					{
						/* check if axo-axonic strength should be normal distributed */

						if (stochasticSynapticStrengths)
						{
							axoSender->connection->initialInhibitionFactor = normal_random(
								stdAxoAxonicStrength, stdAxoAxonicStrengthStd, 0.0, 1.0);
						}

						/* init learn parameter */

						if (resetLearningParameter)
							init_learn_parameter(axoSender->connection);

						/* initialize constants */

						/* check if synaptic transmission time should be normal distributed */

						if (stochasticTransmissionTimes)
						{
							axoSender->connection->deliveryTime = normal_random(
								stdSynapticTT, stdSynapticTTStd, STEP, TT_MAX / T_SCALE);
						}

						axoSender->connection->TSmTRT_const =
							axoSender->connection->transmitterSatiate -
							axoSender->connection->transmitterReleaseThreshold;
						axoSender->connection->TTdSS_const =
							(int) rint(axoSender->connection->deliveryTime / STEP);

						queuePlaces = (int) rint(axoSender->connection->deliveryTime / STEP);

						/* now resize queue to proper length for this simulation */

						axoSender->processQueue = (ProcessQueue *)
							realloc(axoSender->processQueue,
							(size_t) (queuePlaces * sizeof(ProcessQueue)));

						if (! axoSender->processQueue)
						{
							error(104);
							return;
						}

						/* initialize process queue */

						axoSender->currentQueuePlace = 0;

						for (I = 0; I < queuePlaces; I++)
							axoSender->processQueue[I] = 0.0;
					}
				}
			}

			/* initialize pool constants */

			LTTdSS = (int) rint(currentPool->longestDeliveryTime / STEP);

			/* initialize pool variables starting with the queue length */

			if (synapseTypes[currentPool->typeOfConnection].ionType == 4)	/* electrical synapse ? */
				queuePlaces = 2;	/* only 2 queue entries are needed */
			else queuePlaces = LTTdSS;	/* maximum number of posible queue entries */
			currentPool->currentQueuePlace = 0;

			/* now resize queue to proper length for this simulation */

			currentPool->processQueue = (ProcessQueue *) realloc(currentPool->processQueue,
				(size_t) (queuePlaces * sizeof(ProcessQueue)));

			/* if allocation has failed -> output an error message and return */

			if (! currentPool->processQueue)
			{
				error(104);
				return;
			}

			/* initialize process queue */

			for (I = 0; I < queuePlaces; I++)
				currentPool->processQueue[I] = 0.0;

		} /* End of for through the pools */

	} /* End of for loop through neurons */

} /* end of init_pools */

/************************************************************************************************************************

 	FUNCTION	: integrate_method(method, state, A, B, dt)
 
	PURPOSE		: do the integration process

	RETURNS		: the result of the integration

************************************************************************************************************************/

double integrate_method(method, state, A, B, dt)
int method;	/* integration method */
double state;	/* current state variable */
double A;	/* derivation term depending on state */
double B;	/* derivation term depending not on state */
double dt;	/* integration time step */
{
	double D;		/* help variable */
	double k1, k2, k3, k4;	/* variables for the Runge-Kutta integration */
	double result;		/* result of integration */

	switch(method)
	{
		case EULER:	/* the normal foward euler integration method */

			result = Euler(state, B - state * A);
			break;

		case EXPEULER:	/* exponential euler method */
		default:

			D = exp(- A * dt);
			result = state * D + (B / A) * (1 - D);
			break;

		case TRAPEZOID:	/* trapezoid method */

			result = Trapezoid(state, B, A);
			break;

		case RUNGEKUTTA: /* Runge-Kutta method */

			k1 = (B - A * state) * dt;
			k2 = (B - A * (state + k1 * 0.5)) * dt;
			k3 = (B - A * (state + k2 * 0.5)) * dt;
			k4 = (B - A * (state + k3)) * dt;

			result = state + (k1 + 2.0 * k2 + 2.0 * k3 + k4) / 6.0;
			break;
	}
	return(result);

} /* end of integrate_method */

/************************************************************************************************************************

 	FUNCTION	: calculate_form_parameter(form, A, V0, B, W)
 
	PURPOSE		: calculate the form parameter

	RETURNS		: the result of the form parameter

************************************************************************************************************************/

double calculate_form_parameter(form, A, V0, B, W)
int	form;		/* describes the functional form of the parameter (exponential, linoid, sigmoid or constant) */
double A, V0, B, W;	/* parameter */
{
	double 	result;	/* result of form evaluation */
	double	term1;	/* nominator of functional form */
	double	term2;	/* denominator of functional form */
	double	epsilon;/* epsilon environment for fault detection */
	double	U;	/* help variable */

	switch(form)
	{
		/*** exponential forms ***/

		case 1:
		case 6:
/*		       		result = MUL_FORM(E_POT[0], A, 0.0, W, -V0, B, 1);*/

				result = A * (W + exp((E_POT[0] - V0) / B));
				break;

		case 2:
		case 3:
/*				result = MUL_FORM(E_POT[0], A, 0.0, W, -V0, -B, 1);*/

				result = A * (W + exp((E_POT[0] - V0) / -B));
				break;

		case 4:
/*				result = MUL_FORM(E_POT[0], 1.0 / A, 0.0, W, -V0, B, 1);*/

				result = (W + exp((E_POT[0] - V0) / B)) / A;
				break;

		case 5:
/*				result = MUL_FORM(E_POT[0], 1.0 / A, 0.0, W, -V0, -B, 1);*/

				result = (W + exp((E_POT[0] - V0) / -B)) / A;
				break;

		/*** linoid forms ***/

		case 11:
/*				result = DIV_FORM(E_POT[0], -A * V0, A, W, -V0, B, -1);*/
/*		        	term1 = TERM1(E_POT[0], -A * V0, A);*/
/*				term2 = TERM2(E_POT[0], W, -V0, B, -1);*/

				term1 = A * (E_POT[0] - V0);
				term2 = W - exp((E_POT[0] - V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					U = -A * V0;
					if (U < 0.0)
						epsilon = -PRECISION * U;
					else	epsilon = PRECISION * U;

					/* beware of precision problems */

					if (term1 <= epsilon && term1 >= - epsilon)
					{
						result = A * B / W;	/* use the limit */
					}
					else
					{
						result = A * B / W;
						if (! errorFlag)
							error(131);
						errorFlag = TRUE;
					}
				}
				else	result = term1 / term2;
				break;

		case 12:
/*				result = DIV_FORM(E_POT[0], -A * V0, A, W, -V0, -B, -1);*/
/*				term1 = TERM1(E_POT[0], -A * V0, A);*/
/*				term2 = TERM2(E_POT[0], W, -V0, -B, -1);*/

				term1 = A * (E_POT[0] - V0);
				term2 = W - exp((E_POT[0] - V0) / -B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					U = -A * V0;
					if (U < 0.0)
						epsilon = -PRECISION * U;
					else	epsilon = PRECISION * U;

					/* beware of precision problems */

					if (term1 <= epsilon && term1 >= - epsilon)
					{
						result = -A * B / W;	/* use the limit */
					}
					else
					{
						result = -A * B / W;
						if (! errorFlag)
							error(131);
						errorFlag = TRUE;
					}
				}
				else	result = term1 / term2;
				break;

		case 13:
/*				result = DIV_FORM(E_POT[0], A * V0, -A, W, -V0, B, -1);*/
/*				term1 = TERM1(E_POT[0], A * V0, -A);*/
/*				term2 = TERM2(E_POT[0], W, -V0, B, -1);*/

				term1 = A * (V0 - E_POT[0]);
				term2 = W - exp((E_POT[0] - V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					U = A * V0;
					if (U < 0.0)
						epsilon = -PRECISION * U;
					else	epsilon = PRECISION * U;

					/* beware of precision problems */

					if (term1 <= epsilon && term1 >= - epsilon)
					{
						result = -A * B / W;	/* use the limit */
					}
					else
					{
						result = -A * B / W;
						if (! errorFlag)
							error(131);
						errorFlag = TRUE;
					}
				}
				else	result = term1 / term2;
				break;

		case 14:
/*				result = DIV_FORM(E_POT[0], A * V0, -A, W, -V0, -B, -1);*/
/*				term1 = TERM1(E_POT[0], A * V0, -A);*/
/*				term2 = TERM2(E_POT[0], W, -V0, -B, -1);*/

				term1 = A * (V0 - E_POT[0]);
				term2 = W - exp((E_POT[0] - V0) / -B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 15:
/*				result = DIV_FORM(E_POT[0], A, A / V0, W, V0, B, 1);*/
/*				term1 = TERM1(E_POT[0], A, A / V0);*/
/*				term2 = TERM2(E_POT[0], W, V0, B, 1);*/

				term1 = A + A * E_POT[0] / V0;
				term2 = W + exp((E_POT[0] + V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		/*** sigmoid forms ***/

		case 21:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, -V0, B, 1.0);*/
/*		        	term1 = TERM1(E_POT[0], A, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, -V0, B, 1);*/

				term1 = A;
				term2 = W + exp((E_POT[0] - V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 22:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, -V0, -B, -1);*/
				term1 = TERM1(E_POT[0], A, 0.0);
				term2 = TERM2(E_POT[0], W, -V0, -B, -1);

				term1 = A;
				term2 = W - exp((E_POT[0] - V0) / -B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 23:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, -V0, -B, 1);*/
/*				term1 = TERM1(E_POT[0], A, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, -V0, -B, 1);*/

				term1 = A;
				term2 = W + exp((E_POT[0] - V0) / -B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 24:
/*				result = DIV_FORM(E_POT[0], 1.0, 0.0, W, -V0, B, 1);*/
/*				term1 = TERM1(E_POT[0], 1.0, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, -V0, B, 1);*/

				term1 = 1.0;
				term2 = W + exp((E_POT[0] - V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 25:
/*				result = DIV_FORM(E_POT[0], 1.0, 0.0, W, -V0, -B, 1);*/
/*				term1 = TERM1(E_POT[0], 1.0, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, -V0, -B, 1);*/

				term1 = 1.0;
				term2 = W + exp((E_POT[0] - V0) / -B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 26:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, -V0, B, -1);*/
/*				term1 = TERM1(E_POT[0], A, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, -V0, B, -1);*/

				term1 = A;
				term2 = W - exp((E_POT[0] - V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		case 27:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, 0.0, 1.0, 0);*/

				result = A / W;
				break;

		case 28:
/*				result = DIV_FORM(E_POT[0], A, 0.0, W, V0, B, 1);*/
/*				term1 = TERM1(E_POT[0], A, 0.0);*/
/*				term2 = TERM2(E_POT[0], W, V0, B, 1);*/

				term1 = A;
				term2 = W + exp((E_POT[0] + V0) / B);

				/* check for singularity */

				if (term2 == 0.0)
				{
					/* no correction is possible, since result reachs plus or minum infinity */

					result = 0.0;
					if (! errorFlag)
						error(131);
					errorFlag = TRUE;
				}
				else	result = term1 / term2;
				break;

		/*** constant forms ***/

		case 31:
/*				result = DIV_FORM(E_POT[0], 1.0, 0.0, W, 0.0, 1.0, 0);*/

				result = 1.0 / W;
				break;

		default:	if (! errorFlag)
					error(130);
				errorFlag = TRUE;
				result = 0.0;
				break;
	}

	return result;
} /* end of calculate_form_parameter */

/************************************************************************************************************************

 	FUNCTION	: calculate_new_learning_values(synapse)
 
	PURPOSE		: calculate new values needed for learning process

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_new_learning_values(synapse)
Connection *synapse;
{
	double spikeThreshold;		/* spike threshold of post synaptic neuron, needed for hebbian learning */
	double synMemSTP = synapse->synapticMemorySTP;		/* copy of synaptic memory to speed up simulation */
	double synMemHebb = synapse->synapticMemoryHebb;	/* copy of synaptic memory to speed up simulation */
	double synMemCond = synapse->synapticMemoryCond;	/* copy of synaptic memory to speed up simulation */

	currentSynapse = synapse;

	/* check if connection is axo-axonic */

	if (conditionedLearning && (synapse->axoSynapseNumber > 0))

		/* check for conditioning */

		if ((synapse->potential >= synapse->transmitterReleaseThreshold) &&
		    (synapse->targetSynapse->potential >= synapse->transmitterReleaseThreshold))
		{
			synapse->conditioningLearningCondition = TRUE;
			synapse->condCnt++;
		}
		else	synapse->conditioningLearningCondition = FALSE;

	else	/* not axo-axonic connection */
	{
		/* check for hebbian learning or competitive learning */

		if (hebbLearning || competitiveLearning)
		{
			switch(networkmodel)
			{
			case 1:	/* ON/OFF model */

				spikeThreshold = ONOFF_SPIKE_THRESHOLD;
				break;

			case 2:	/* SWIM model */

				spikeThreshold = SWIM_SPIKE_THRESHOLD;
				break;

			case 3:	/* HH model */

				spikeThreshold = HH_SPIKE_THRESHOLD;
				break;

			case 4:	/* GB model */

				spikeThreshold = GB_SPIKE_THRESHOLD;
				break;
			}

			/* hebb interval counter is greater zero if synapse was active in the last hebb interval */

			if (synapse->potential >= synapse->transmitterReleaseThreshold)
				synapse->hebbIntervallCounter = synapse->hebbIntervall;
			else	synapse->hebbIntervallCounter -= STEP;

			/* check hebb condition */

			if ((synapse->hebbIntervallCounter > 0.0) &&
			    (synapse->targetNeuron->neuronPotentials[0] >= spikeThreshold))
				if (synapse->hebbLearningCondition)	/* was hebb condition previously true */
				{
					synapse->hebbTimer += STEP;

					/* reset hebb condition if it was true a full hebb interval */

					if (synapse->hebbTimer >= synapse->hebbIntervall)
						synapse->hebbLearningCondition =
							synapse->competitiveLearningCondition = FALSE;
				}
				else
				{
					synapse->hebbLearningCondition = synapse->competitiveLearningCondition = TRUE;
					synapse->hebbTimer = 0.0;	/* start of hebb conditition is true */
					synapse->hebbCnt++;	/* count hebb event */
				}
			else	synapse->hebbLearningCondition = synapse->competitiveLearningCondition = FALSE;

		}
	}

	/* check for short term plasticity */

	if (plasticityLearning)
		if (synapse->potential >= synapse->transmitterReleaseThreshold)
		{
			synapse->shortTermLearningCondition = TRUE;
			synapse->stmCnt++;
		}
		else	synapse->shortTermLearningCondition = FALSE;

	/* calculate new synaptic memory and training state if at least one type of learning is set */

	if (plasticityLearning || conditionedLearning || hebbLearning)
	{
		/* check if synapse mets short term plasticity learning condition */

		if (synapse->shortTermLearningCondition)

		       synMemSTP = 1.0 / (1.0 + (1.0 / synMemSTP - 1.0) * LEXP);

		else if (synapse->trainingFlagSTP)	/* long term forgetting */

			synMemSTP = exp(log(synMemSTP - synapse->longTermMinimum) - LFCONST) + synapse->longTermMinimum;

		else 	/* short term forgetting */

			synMemSTP = synapse->longTermThreshold * (1.0 - sqrt(intpower(1.0 - 2.0 * synMemSTP, 2) + SFCONST));


		if (synMemSTP > synapse->longTermThreshold)

			synapse->trainingFlagSTP = TRUE;


		/* check if synaptic memory is in valid range (0.01,1.01) */

		if (synMemSTP < 0.01)
			synMemSTP = 0.01;
		else if (synMemSTP > 1.01)
			synMemSTP = 1.01;

		/* check if synapse mets Hebb learning condition */

		if (synapse->hebbLearningCondition)

		       synMemHebb = 1.0 / (1.0 + (1.0 / synMemHebb - 1.0) * LEXP);

		else if (synapse->trainingFlagHebb)	/* long term forgetting */

			synMemHebb = exp(log(synMemHebb - synapse->longTermMinimum) - LFCONST) + synapse->longTermMinimum;

		else 	/* short term forgetting */

			synMemHebb = synapse->longTermThreshold * (1.0 - sqrt(intpower(1.0 - 2.0 * synMemHebb, 2) + SFCONST));


		if (synMemHebb > synapse->longTermThreshold)

			synapse->trainingFlagHebb = TRUE;


		/* check if synaptic memory is in valid range (0.01,1.01) */

		if (synMemHebb < 0.01)
			synMemHebb = 0.01;
		else if (synMemHebb > 1.01)
			synMemHebb = 1.01;

		/* check if synapse mets conditioned learning condition */

		if (synapse->conditioningLearningCondition)

		       synMemCond = 1.0 / (1.0 + (1.0 / synMemCond - 1.0) * LEXP);

		else if (synapse->trainingFlagCond)	/* long term forgetting */

			synMemCond = exp(log(synMemCond - synapse->longTermMinimum) - LFCONST) + synapse->longTermMinimum;

		else 	/* short term forgetting */

			synMemCond = synapse->longTermThreshold * (1.0 - sqrt(intpower(1.0 - 2.0 * synMemCond, 2) + SFCONST));


		if (synMemCond > synapse->longTermThreshold)

			synapse->trainingFlagCond = TRUE;


		/* check if synaptic memory is in valid range (0.01,1.01) */

		if (synMemCond < 0.01)
			synMemCond = 0.01;
		else if (synMemCond > 1.01)
			synMemCond = 1.01;
	}

	synapse->synapticMemorySTP = synMemSTP;
	synapse->synapticMemoryHebb = synMemHebb;
	synapse->synapticMemoryCond = synMemCond;
	synapse->synapticMemory = synMemSTP + synMemHebb + synMemCond - 0.03;

} /* end of calculate_new_learning_values */

/************************************************************************************************************************

 	FUNCTION	: calculate_synapses(void)
 
	PURPOSE		: calculates new entrees into synaptic pools

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_synapses()
{
	Neuron *targetNeuron;			/* target neuron */
	Connection *conn;			/* connection */
	ChemicalPool *currentPool, *pool;	/* current pool */
	ChemicalSenderList *sender;		/* sender in pool */
	AxoAxonicSenderList *axoSender;		/* current axo-axonic sender */
	ProcessQueue *currentQueue;		/* current process queue */
	int queuePlaces;			/* length of process queue */
	double active;				/* flag if current connection is active */
	double weight;				/* weight of synaptic connection */
	double temp;				/* help variable for speed up simulation */

	/* main loop through neurons */

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* loop through synapses */

		for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
		     currentSynapse = currentSynapse->nextConnection)
		{
			/* update synapse potential from parent neuron */

			currentSynapse->potential = currentNeuron->neuronPotentials[0];

			/* check if current synapse is axo-axonic and networkmodel is of type complex */

			if (currentSynapse->axoSynapseNumber > 0 && complexModel)
			{
				/* set current axo-axonic sender and process queue */

				axoSender = currentSynapse->axoSenderList;

				/* calculate the open factor of a synapse */

				temp = (currentSynapse->potential - currentSynapse->transmitterReleaseThreshold) / TSmTRT;

				/* correct open factor, if he is out of range (0,1) */

				if (temp < 0.0)
					temp = 0.0;
				else if (temp > 1.0)
					temp = 1.0;

				/* determine current synapse strength depending on learning state and condition */

				currentSynapse->currentInhibitionFactor = currentSynapse->initialInhibitionFactor;

				if (plasticityLearning && currentSynapse->shortTermLearningCondition)
				{
					currentSynapse->currentInhibitionFactor *= (1.0 +
						currentSynapse->shortTermPlasticityLearnFactor *
						(currentSynapse->synapticMemorySTP - 0.01));

					/* check if current inhibition factor is in valid range */

					if (currentSynapse->currentInhibitionFactor < 0.0)
						currentSynapse->currentInhibitionFactor = 0.0;
					else if (currentSynapse->currentInhibitionFactor >
						 currentSynapse->maxInhibitionFactor)
						currentSynapse->currentInhibitionFactor =
							currentSynapse->maxInhibitionFactor;
				}

				if (conditionedLearning && currentSynapse->conditioningLearningCondition)
				{
					/* change target synapse's process queue entry (conductance) */

					currentPool = currentSynapse->targetSynapse->connectionPool;
					currentQueue = currentPool->processQueue;

					queuePlaces = currentPool->currentQueuePlace + TTdSS;

					if (queuePlaces >= LTTdSS)
						queuePlaces -= LTTdSS;

					currentQueue[queuePlaces] *= (1.0 + currentSynapse->conditioningLearnFactor *
						(currentSynapse->synapticMemoryCond - 0.01));

					/* check if target synapses conductance is in valid range */

					if (currentQueue[queuePlaces] > currentSynapse->targetSynapse->maxConductance)
						currentQueue[queuePlaces] = currentSynapse->targetSynapse->maxConductance;
					else if (currentQueue[queuePlaces] < 0.0)
						currentQueue[queuePlaces] = 0.0;
				}

				temp *= currentSynapse->currentInhibitionFactor;

				/* calculate entry point in process queue of receiving pool */

				queuePlaces = axoSender->currentQueuePlace + TTdSS;

				/* if queue entry lies out of range, correct it because its a circular queue */

				if (queuePlaces >= TTdSS)
					queuePlaces -= TTdSS;

				/* set inhibition factor in process queue */

				axoSender->processQueue[queuePlaces] = temp;

				/* compute new learning values */

				if (plasticityLearning || conditionedLearning)
					calculate_new_learning_values(currentSynapse);
			}
			else if (currentSynapse->axoSynapseNumber == 0)	/* not axo-axonic */
			{
				/* set current pool and process queue */

				currentPool = currentSynapse->connectionPool;
				currentQueue = currentPool->processQueue;

				switch(synapseTypes[currentPool->typeOfConnection].ionType)
				{

				/* calculcate all different types of chemical synapses */

				case	1:
				case	2:
				case	3:

					/* calculate the open factor of a synapse */

					temp = (currentSynapse->potential - currentSynapse->transmitterReleaseThreshold)
						/ TSmTRT;

					/* correct open factor, if he is out of range (0,1) */

					if (temp < 0.0)
						temp = 0.0;
					else if (temp > 1.0)
						temp = 1.0;

					/* determine current synapse conductance depending on learning state */

					temp *= currentSynapse->conductance;

					if (complexModel)	/* complex model ? */
					{
						temp *= (1.0 + (int) plasticityLearning *
								currentSynapse->shortTermPlasticityLearnFactor *
								currentSynapse->synapticMemorySTP +
								(int) hebbLearning *
								currentSynapse->hebbLearnFactor *
								currentSynapse->synapticMemoryHebb);

						if (competitiveLearning && currentSynapse->competitiveLearningCondition
						    && ! currentSynapse->competitiveDone
						    && currentSynapse->hebbTimer == 0.0)
						{
							/* count all synapses in the pool for which the competitive
							   learning condition mets */
							
							pool = currentSynapse->connectionPool;
							pool->activeSynapses = 0;
							for (sender = pool->senderList; sender != NULL;
							     sender = sender->nextSender)
								if (sender->connection->competitiveLearningCondition)
									pool->activeSynapses++;

							/* distribute synaptic weights in target neuron */

							for (sender = pool->senderList; sender != NULL;
							     sender = sender->nextSender)
							{
								conn = sender->connection;
								if (conn->competitiveLearningCondition)
									active = 1.0;
								else	active = 0.0;
								weight = conn->conductance /
									 pool->totalConductance;
								weight += conn->competitiveLearnFactor *
								   ((active / pool->activeSynapses)
								    - weight);
								weight *= pool->totalConductance;
								/* check if current conductance is
								   in valid range (0, maxConductance) */

								if (weight < 0.0)
									weight = 0.0;
								else if (weight > conn->maxConductance)
									weight = conn->maxConductance;

								conn->conductance = weight;
								conn->competitiveDone = TRUE;
							}
						}
						/* check if current conductance is in valid range (0, maxConductance) */

						if (temp < 0.0)
							temp = 0.0;
						else if (temp > currentSynapse->maxConductance)
							temp = currentSynapse->maxConductance;

						/* compute new learning values */

						if (plasticityLearning || hebbLearning || competitiveLearning)
							calculate_new_learning_values(currentSynapse);
					}

					/* calculate entry point in process queue of receiving pool */

					queuePlaces = currentPool->currentQueuePlace + TTdSS;

					/* if queue entry lies out of range, correct it because its a circular queue */

					if (queuePlaces >= LTTdSS)
						queuePlaces -= LTTdSS;

					/* add conductance to queue entry */

					currentSynapse->currentConductance = temp;
					currentQueue[queuePlaces] += temp;

					break;

				/* calculate electrical synapses */

				case	4:

					/* the current conductance is always the same as the initial conductance */

					temp = currentSynapse->conductance;
					currentSynapse->currentConductance = temp;
					targetNeuron = currentSynapse->targetNeuron;

					if (currentSynapse->potential > targetNeuron->neuronPotentials[0])
					{
						/* only the first 2 entries in the process queue are needed */

						currentQueue[0] += currentSynapse->potential * temp;
						currentQueue[1] += targetNeuron->neuronPotentials[0] * temp;
					}

					break;
				}
			} /* end of not an axo-axonic connection */
		} /* End of for through the synapses */
	} /* End of for loop through neurons */

} /* end of calculate_synapses */

/************************************************************************************************************************

 	FUNCTION	: calculate_synaptic_influences(void)
 
	PURPOSE		: calculates synaptical currents

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_synaptic_influences()
{
	ChemicalPool *currentPool;		/* current pool */
	AxoAxonicSenderList *axoSender;		/* current axo-axonic sender */
	ProcessQueue *currentQueue;		/* current process queue */
	int queuePlaces;			/* length of process queue */
	int poolLocation;			/* location of the pool */
	double iChem;				/* chemical current */
	double iElec;				/* electrical current */

	/***	calculate the synaptic influences of the axo-axonic synapses to the chemical synapses first	***/

	if (complexModel)	/* comlex network model */

		/* calculate synaptic influence of axo-axonic synapses to normal chemical synapses */

		for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
		{
			/* loop through all outgoing synapses */

			for (currentSynapse = currentNeuron->connectionList; currentSynapse != NULL;
			     currentSynapse = currentSynapse->nextConnection)
			{
				/* check if axo-axonic connections to this synapse exists */

				if (currentSynapse->axoSynapseNumber == 0) /* chemical connection */
					for (axoSender = currentSynapse->axoSenderList; axoSender != NULL;
					     axoSender = axoSender->nextSender)
					{
						/* update queue pointer */

						queuePlaces = axoSender->connection->TTdSS_const;
						axoSender->currentQueuePlace++;

						if (axoSender->currentQueuePlace >= queuePlaces)
							axoSender->currentQueuePlace -= queuePlaces;

						/* calculate new chemical conductance */

						currentPool = currentSynapse->connectionPool;
						currentQueue = currentPool->processQueue;

						queuePlaces = currentPool->currentQueuePlace + 1;

						if (queuePlaces == LTTdSS)
							queuePlaces = 0;

						/* set current strenght of sender synapses */

						axoSender->connection->currentStrength =
							axoSender->processQueue[axoSender->currentQueuePlace];

						currentQueue[queuePlaces] *=
							(1.0 - axoSender->connection->currentStrength);
					}
				/* reset flag for competitive learning */

				currentSynapse->competitiveDone = FALSE;

			} /* end of loop through synapses */
		} /* end of loop through neurons */

	/***	calculate the influences of the chemical synapses to the neurons	***/

	for (currentNeuron = neuronListHead; currentNeuron != NULL; currentNeuron = currentNeuron->next)
	{
		/* for every compartment initialize currents to zero */

		for (poolLocation = 0; poolLocation <= curNumOfDendSegs; poolLocation++)
		{
			ICH[poolLocation] = 0.0;
			IEL[poolLocation] = 0.0;
		}

		/* loop through all pools at current neuron */

		for (currentPool = currentNeuron->poolList; currentPool != NULL; currentPool = currentPool->nextPool)
		{
			currentQueue = currentPool->processQueue;

			switch(synapseTypes[currentPool->typeOfConnection].ionType)
			{
			/* calculate influence of chemical synapses */

			case 1:
			case 2:
			case 3:
				/* update queue pointer */

				queuePlaces = LTTdSS;

				currentPool->currentQueuePlace++;

				if (currentPool->currentQueuePlace >= queuePlaces)
					currentPool->currentQueuePlace -= queuePlaces;

				/* update chemical current in neuron dependend on network model */

				switch(networktype)
				{
				case 1: /* 1-Point */

					/* calculate chemical current */

					iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
						E_POT[0]) * currentQueue[currentPool->currentQueuePlace];

					ICH[0] += iChem;
					break;

				case 2: /* 2-Point */

					if (currentPool->locationOfPool >= 1)
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[1]) * currentQueue[currentPool->currentQueuePlace];

						ICH[1] += iChem;
					}
					else
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[0]) * currentQueue[currentPool->currentQueuePlace];

						ICH[0] += iChem;
					}
					break;

				case 3: /* 4-Point */

					if (currentPool->locationOfPool > 3)
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[3]) * currentQueue[currentPool->currentQueuePlace];

						ICH[3] += iChem;
					}
					else
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[currentPool->locationOfPool]) *
							currentQueue[currentPool->currentQueuePlace];

						ICH[currentPool->locationOfPool] += iChem;
					}
					break;

				case 4: /* N-Point */

					if (currentPool->locationOfPool > curNumOfDendSegs)
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[curNumOfDendSegs]) *
							currentQueue[currentPool->currentQueuePlace];

						ICH[curNumOfDendSegs] += iChem;
					}
					else
					{
						/* calculate chemical current */

						iChem = (synapseTypes[currentPool->typeOfConnection].equilPotential -
							E_POT[currentPool->locationOfPool]) *
							currentQueue[currentPool->currentQueuePlace];

						ICH[currentPool->locationOfPool] += iChem;
					}
					break;
				}

				/* reset entry in process queue at current position */

				currentQueue[currentPool->currentQueuePlace] = 0.0;
				break;

			case 4:
				currentPool->currentQueuePlace = 0;

				/* calculate electrical current */

				iElec = currentQueue[0] - currentQueue[1];

				/* reset entries in process queue */

				currentQueue[0] = 0.0;
				currentQueue[1] = 0.0;

				/* update electrical current in neuron dependend on network model */

				switch(networktype)
				{
				default:
				case 4: /* N-Point */

					if (currentPool->locationOfPool > curNumOfDendSegs)
						IEL[curNumOfDendSegs] += iElec;
					else	IEL[currentPool->locationOfPool] += iElec;
					break;

				case 3: /* 4-Point */

					if (currentPool->locationOfPool > 3)
						IEL[3] += iElec;
					else
						IEL[currentPool->locationOfPool] += iElec;
					break;

				case 2: /* 2-Point */

					if (currentPool->locationOfPool == 0)
						IEL[0] += iElec;
					else 
						IEL[1] += iElec;
					break;

				case 1: /* 1-Point */

					IEL[0] += iElec;
					break;
				}
				break;
			}
		} /* End of for loop through pool */
	} /* End of for loop through neurons */

} /* end of calculate_synaptic_influences */

/************************************************************************************************************************

 	FUNCTION	: calculate_onoff_neuron(void)
 
	PURPOSE		: calculates a On/Off neuron

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_onoff_neuron()
{
	int	I;					/* general help variables */
	unsigned int len;				/* length of allocation area */
	double	delta;					/* time to add because of rounding problems */
	double  tempExp;				/* temporare exponential expression */
	double  iExtS, iExtD;				/* extern currents for soma and dendrite */
	double	iExtSpseudo;				/* extern pseudo current for soma */
	double  eNeighbors;				/* potential of the neighbors of a current compartment */
	double  numNeighbors;				/* number of neighbors a current compartment has */
	double  oldEPot;				/* old soma potential, used to determine a new spike */
	double	*inpCur;				/* input current into neuron */
	double	*eTempA;				/* A value of differential equation for compartment potentials */
	double  *eTempB;				/* B value of differential equation for compartment potentials */

	/* allocate memory for input currents, potentials, ... */

	len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

	inpCur = (double *) malloc((size_t) len);

	if (! inpCur)
	{
		error(128);
		return;
	}

	eTempA = (double *) malloc((size_t) len);

	if (! eTempA)
	{
		error(128);
		return;
	}

	eTempB = (double *) malloc((size_t) len);

	if (! eTempB)
	{
		error(128);
		return;
	}

	delta = STEP / 10.0;	/* used because of rounding problems */

	/* calculates active sodium, potassium, calcium channels in soma */

	SPIKE += STEP;

	if (E_POT[0] > ONOFF_SPIKE_THRESHOLD && SPIKE_IN_PROCESS == FALSE)
	{
		SPIKE = 0.0;
		SPIKE_IN_PROCESS = TRUE;
	}
	else if (E_POT[0] < ONOFF_SPIKE_THRESHOLD &&  SPIKE_IN_PROCESS == TRUE)
		SPIKE_IN_PROCESS = FALSE;

	if (SPIKE < ONOFF_NA_DURATION + delta && SPIKE_IN_PROCESS == TRUE)
		I_NA = (ONOFF_NA_E - E_POT[0]) * ONOFF_NA_CONDUCTANCE;
	else
		I_NA = 0.0;

	if (SPIKE + delta >= ONOFF_K_START && SPIKE < ONOFF_K_DURATION + ONOFF_K_START + delta)
		I_K = (ONOFF_K_E - E_POT[0]) * ONOFF_K_CONDUCTANCE;
	else
		I_K = 0.0;

	if (SPIKE + delta >= ONOFF_CA_START && SPIKE < ONOFF_CA_DURATION + ONOFF_CA_START + delta)
		I_CA = (ONOFF_CA_E - E_POT[0]) * ONOFF_CA_CONDUCTANCE;
	else
		I_CA = 0.0;

	/* calculate index for access to input current */

	for (I = 0; I <= curNumOfDendSegs; I++)
		if (L >= ICUR[I].starttime[ICUR[I].index])
			ICUR[I].index++;

	/* calculate stochastic noise current if defined */

	if (noiseDefined && Noise)
	{
		noise *= exp(-STEP / noiseDecay);
		noise += noiseevents(STEP, noiseIntensity);
		I_NOISE = noise * noiseStrength;
	}
	else I_NOISE = 0.0;

	/* calculation of temporary new soma & dendrite variables
	   COMPARTMENT (S,R) = ((S - 1) * NEURON_REGIONS) + R <----> SOMA = 0 */

	if (DENDRITE_STEMS == 0)
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + I_NA + I_K + I_CA;

		eTempA[0] = SGdSC;
		eTempB[0] = TotCur / SOMA_Cm;

	} /* End of point neuron calculations */

	else
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

/*			eNeighbors = 0;*/
/*			for (J = 1; J <= (int) DENDRITE_STEMS; J++)*/
/*				eNeighbors += E_POT[(int) ((J - 1) * DENDRITE_REGIONS) + 1];*/

		eNeighbors = E_POT[1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + eNeighbors * SOMA_Gcore +
			I_NA + I_K + I_CA;

		eTempA[0] = SGpDSmDGdSC;
		eTempB[0] = TotCur / SOMA_Cm;

		/* calculation of dendritic temporary potential factors */

		for (I = 1; I <= curNumOfDendSegs; I++)
		{
/*			if ( (I / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
			if (I == DENDRITE_REGIONS)	/* End of stem */
			{
				eNeighbors = E_POT[I - 1];
				numNeighbors = 1;
			}
			else
			{
/*				if ( ((I + DENDRITE_REGIONS - 1) / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
				if (I == 1)
					eNeighbors = E_POT[0] + E_POT[2];	  /* dendrite piece next to soma */
				else	eNeighbors = E_POT[I - 1] + E_POT[I + 1]; /** Dendrite piece in stem middle **/

				numNeighbors = 2;
			}

/*			if (I == (int) (((IStem - 1) * (int) DENDRITE_REGIONS) + IRegion))*/
			if (I == IRegion)
				iExtD = (L >= SDSTRT && L < SDSTP) * DendCur;
			else iExtD = 0;

			inpCur[I] = (int) ((L >= ICUR[I].starttime[ICUR[I].index - 1] && L < ICUR[I].starttime[ICUR[I].index]))
				* ICUR[I].current[ICUR[I].index - 1];

			/*
				I(Leak) = (Dendritepotential - Leakpotential) * Leakconductance
					==> A = DENDRITE_REST_Gm (Leakconductance)
					    B = DENDRITE_REST_Gm * DendritePotential (DLmDG)
				I(Core) = SUM( E(c) - E) * DENDRITE_Gcore, with c = neighbours of compartment
					DENDRTITE_Gcore = Conductance from dendrite to soma
					==> A = numNeighbors * DENDRITE_Gcore
					    B = eNeighbors * DENDRITE_Gcore
			*/

			tempExp = ICH[I] + IEL[I];
			SynCur += tempExp;
			tempExp += iExtD + inpCur[I] + DLmDG + eNeighbors * DENDRITE_Gcore;
			TotCur += tempExp;

			eTempA[I] = (DENDRITE_REST_Gm + numNeighbors * DENDRITE_Gcore ) / DENDRITE_Cm;
			eTempB[I] = tempExp / DENDRITE_Cm;

		} /* end of dendritic tree parameter calculations */
	}

	/* calculate new soma & dendrite potential values */

	oldEPot = E_POT[0];
	for (I = 0; I <= curNumOfDendSegs; I++)
		E_POT[I] = integrate_method(integrationMethod, E_POT[I], eTempA[I], eTempB[I], STEP);


	/* increment spike counter if a spike has occured */

	if (currentNeuron->neuronType == spiker &&
	    oldEPot < ONOFF_SPIKE_THRESHOLD && E_POT[0] > ONOFF_SPIKE_THRESHOLD)
		SpikeCounter++;
	
	/* free dynamic memory */

	free(inpCur);
	free(eTempA);
	free(eTempB);

} /* end of calculate_onoff_neuron */

/************************************************************************************************************************

 	FUNCTION	: calculate_HH_neuron(void)
 
	PURPOSE		: calculates a Hodgkin-Huxley neuron

			  The differential equations are solved in respect of the integrationMethod variable

			  A contains all multiplication factors of the differentiation variable and
			  B contains all non-multiplication factors of the differentiation variable.

			  In the case of switching from a 4-point network model to a 2-point or 1-point model
			  its necessary to relink existing extern dendrit currents from the dendrite to the soma.
			  Hence a new variable (iExtSpseudo) is needed to hold the pseudo current.

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_HH_neuron()
{
	int	I;					/* general help variables */
	unsigned int len;				/* length of allocation area */
	double  tempExp;				/* temporare exponential expression */
	double  A, B;					/* A and B value of differential equation for ion channels */
	double  alpha, beta;				/* alpha and beta forms of gate variables */
	double	gateNa, gateK;				/* gate variables for the channels */
	double  iExtS, iExtD;				/* extern currents for soma and dendrite */
	double	iExtSpseudo;				/* extern pseudo current for soma */
	double  eNeighbors;				/* potential of the neighbors of a current compartment */
	double  numNeighbors;				/* number of neighbors a current compartment has */
	double  oldEPot;				/* old soma potential, used to determine a new spike */
	double	*inpCur;				/* input current into neuron */
	double	*eTempA;				/* A value of differential equation for compartment potentials */
	double  *eTempB;				/* B value of differential equation for compartment potentials */

	/* allocate memory for input currents, potentials, ... */

	len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

	inpCur = (double *) malloc((size_t) len);

	if (! inpCur)
	{
		error(128);
		return;
	}

	eTempA = (double *) malloc((size_t) len);

	if (! eTempA)
	{
		error(128);
		return;
	}

	eTempB = (double *) malloc((size_t) len);

	if (! eTempB)
	{
		error(128);
		return;
	}

	/* calculations for active sodium channels in soma */

	alpha = calculate_form_parameter((int) HH_Form_Alpha_M, HH_AalphaM, HH_V0alphaM, HH_BalphaM, 1.0);
	beta  = calculate_form_parameter((int) HH_Form_Beta_M, HH_AbetaM, HH_V0betaM, HH_BbetaM, 0.0);

	A = alpha + beta;
	B = alpha;

	M_CONST = integrate_method(integrationMethod, M_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) HH_Form_Alpha_H, HH_AalphaH, HH_V0alphaH, HH_BalphaH, 0.0);
	beta  = calculate_form_parameter((int) HH_Form_Beta_H, HH_AbetaH, HH_V0betaH, HH_BbetaH, 1.0);

	A = alpha + beta;
	B = alpha;

	H_CONST = integrate_method(integrationMethod, H_CONST, A, B, STEP);

	gateNa = intpower(M_CONST, (int) HH_M_POWER) * intpower(H_CONST, (int) HH_H_POWER);
	I_NA = gateNa * (HH_NA_E - E_POT[0]) * HH_NA_G;

	/* calculations for active potassium channels in soma */

	alpha = calculate_form_parameter((int) HH_Form_Alpha_N, HH_AalphaN, HH_V0alphaN, HH_BalphaN, 1.0);
	beta  = calculate_form_parameter((int) HH_Form_Beta_N, HH_AbetaN, HH_V0betaN, HH_BbetaN, 0.0);

	A = alpha + beta;
	B = alpha;

	N_CONST = integrate_method(integrationMethod, N_CONST, A, B, STEP);

	gateK = intpower(N_CONST, (int) HH_N_POWER);
	I_K = gateK * (HH_K_E - E_POT[0]) * HH_K_G;

	/* set gate variables for graph windows */

	GB_I_CA1 = M_CONST / I_SCALE;
	GB_I_CA2 = H_CONST / I_SCALE;
	GB_I_KCA = N_CONST / I_SCALE;

	/* calculate index for access to input current */

	for (I = 0; I <= curNumOfDendSegs; I++)
		if (L >= ICUR[I].starttime[ICUR[I].index])
			ICUR[I].index++;

	/* calculate stochastic noise current if defined */

	if (noiseDefined && Noise)
	{
		noise *= exp(-STEP / noiseDecay);
		noise += noiseevents(STEP, noiseIntensity);
		I_NOISE = noise * noiseStrength;
	}
	else I_NOISE = 0.0;

	/* calculation of temporary new soma & dendrite variables
	   COMPARTMENT (S,R) = ((S-1) * NEURON_REGIONS) + R <----> SOMA = 0 */

	if (DENDRITE_STEMS == 0)
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + HH_SLmSG + HH_NAEmNAG * gateNa +
			HH_KEmKG * gateK;

		eTempA[0] = (SOMA_REST_Gm + HH_NA_G * gateNa + HH_K_G * gateK) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

	} /* End of point neuron calculations */
	else
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

/*		eNeighbors = 0;*/
/*		for (J = 1; J <= (int) DENDRITE_STEMS; J++)*/
/*			eNeighbors += E_POT[(int) ((J - 1) * DENDRITE_REGIONS) + 1];*/

		eNeighbors = E_POT[1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + HH_SLmSG + eNeighbors * SOMA_Gcore +
			HH_NAEmNAG * gateNa + HH_KEmKG * gateK;

		eTempA[0] = (SGpDSmDG + HH_NA_G * gateNa + HH_K_G * gateK) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

		/* calculation of dendritic temporary potential factors */

		for (I = 1; I <= curNumOfDendSegs; I++)
		{
/*			if ( (I / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
			if (I == DENDRITE_REGIONS)      /* End of stem */
			{
				eNeighbors = E_POT[I - 1];
				numNeighbors = 1;
			}
			else
			{
/*				if ( ((I + DENDRITE_REGIONS - 1) / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
				if (I == 1)
					eNeighbors = E_POT[0] + E_POT[2];	  /* dendrite piece next to soma */
				else	eNeighbors = E_POT[I - 1] + E_POT[I + 1]; /* dendrite piece in stem middle */

				numNeighbors = 2;
			}

/*			if (I == (int) (((IStem - 1) * (int) DENDRITE_REGIONS) + IRegion))*/
			if (I == IRegion)
				iExtD = (L >= SDSTRT && L < SDSTP) * DendCur;
			else iExtD = 0;

			inpCur[I] = (int) ((L >= ICUR[I].starttime[ICUR[I].index - 1] && L < ICUR[I].starttime[ICUR[I].index]))
				* ICUR[I].current[ICUR[I].index - 1];

			/*
				I(Leak) = (Dendritepotential - Leakpotential) * Leakconductance
					==> A = DENDRITE_REST_Gm (Leakconductance)
					    B = DENDRITE_REST_Gm * DendritePotential (DLmDG)
				I(Core) = SUM( E(c) - E) * DENDRITE_Gcore, with c = neighbours of compartment
					DENDRTITE_Gcore = Conductance from dendrite to soma
					==> A = numNeighbors * DENDRITE_Gcore
					    B = eNeighbors * DENDRITE_Gcore
			*/

			tempExp = ICH[I] + IEL[I];
			SynCur += tempExp;
			tempExp += iExtD + inpCur[I] + DLmDG + eNeighbors * DENDRITE_Gcore;
			TotCur += tempExp;

			eTempA[I] = (DENDRITE_REST_Gm + numNeighbors * DENDRITE_Gcore) / DENDRITE_Cm;
			eTempB[I] = tempExp / DENDRITE_Cm;

		} /* end of dendritic tree parameter calculations */
	}

	/* calculate new soma & dendrite potential values */

	oldEPot = E_POT[0];
	for (I = 0; I <= curNumOfDendSegs; I++)
		E_POT[I] = integrate_method(integrationMethod, E_POT[I], eTempA[I], eTempB[I], STEP);

	/* increment spike counter if a spike has occured */

	if (currentNeuron->neuronType == spiker &&
	    oldEPot < HH_SPIKE_THRESHOLD && E_POT[0] > HH_SPIKE_THRESHOLD)
		SpikeCounter++;
	
	/* free dynamic memory */

	free(inpCur);
	free(eTempA);
	free(eTempB);

} /* end of calculate_HH_neuron */

/************************************************************************************************************************

 	FUNCTION	: calculate_SWIM_neuron(void)
 
	PURPOSE		: calculates a SWIM neuron

			  The differential equations are solved in respect of the integrationMethod variable

			  A contains all multiplication factors of the differentiation variable and
			  B contains all non-multiplication factors of the differentiation variable.

			  In the case of switching from a 4-point network model to a 2-point or 1-point model
			  its necessary to relink existing extern soma currents from the dendrite to the soma.
			  Hence a new variable (iExtSpseudo) is needed to hold the pseudo current.

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_SWIM_neuron()
{
	int	I;					/* general help variables */
	unsigned int len;				/* length of allocation area */
	double  tempExp;				/* temporare exponential expression */
	double  A, B;					/* A and B value of differential equation for ion channels */
	double  alpha, beta;				/* alpha and beta forms of gate variables */
	double	gateNa, gateK, gateCa;			/* gate variables for the channels */
	double  iExtS, iExtD;				/* extern currents for soma and dendrite */
	double	iExtSpseudo;				/* extern pseudo current for soma */
	double  eNeighbors;				/* potential of the neighbors of a current compartment */
	double  numNeighbors;				/* number of neighbors a current compartment has */
	double  oldEPot;				/* old soma potential, used to determine a new spike */
	double	*inpCur;				/* input current into neuron */
	double	*eTempA;				/* A value of differential equation for compartment potentials */
	double  *eTempB;				/* B value of differential equation for compartment potentials */

	/* allocate memory for input currents, potentials, ... */

	len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

	inpCur = (double *) malloc((size_t) len);

	if (! inpCur)
	{
		error(128);
		return;
	}

	eTempA = (double *) malloc((size_t) len);

	if (! eTempA)
	{
		error(128);
		return;
	}

	eTempB = (double *) malloc((size_t) len);

	if (! eTempB)
	{
		error(128);
		return;
	}

	/* calculations for active sodium channels in soma */

	alpha = calculate_form_parameter((int) SWIM_Form_Alpha_M, SWIM_AalphaM, SWIM_V0alphaM, SWIM_BalphaM, 1.0);
	beta  = calculate_form_parameter((int) SWIM_Form_Beta_M, SWIM_AbetaM, SWIM_V0betaM, SWIM_BbetaM, 1.0);

	A = alpha + beta;
	B = alpha;

	M_CONST = integrate_method(integrationMethod, M_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) SWIM_Form_Alpha_H, SWIM_AalphaH, SWIM_V0alphaH, SWIM_BalphaH, 1.0);
	beta  = calculate_form_parameter((int) SWIM_Form_Beta_H, SWIM_AbetaH, SWIM_V0betaH, SWIM_BbetaH, 1.0);

	A = alpha + beta;
	B = alpha;

	H_CONST = integrate_method(integrationMethod, H_CONST, A, B, STEP);

	gateNa = intpower(M_CONST, (int) SWIM_M_POWER) * intpower(H_CONST, (int) SWIM_H_POWER);
	I_NA = gateNa * (SWIM_NA_E - E_POT[0]) * SWIM_NA_G;

	/* calculations for active potassium channels in soma */

	alpha = calculate_form_parameter((int) SWIM_Form_Alpha_N, SWIM_AalphaN, SWIM_V0alphaN, SWIM_BalphaN, 1.0);
	beta  = calculate_form_parameter((int) SWIM_Form_Beta_N, SWIM_AbetaN, SWIM_V0betaN, SWIM_BbetaN, 1.0);

	A = alpha + beta;
	B = alpha;

	N_CONST = integrate_method(integrationMethod, N_CONST, A, B, STEP);

	gateK = intpower(N_CONST, (int) SWIM_N_POWER);
	I_K = gateK * (SWIM_K_E - E_POT[0]) * SWIM_K_G;

	/* calculations for active calcium channels in soma */

	alpha = calculate_form_parameter((int) SWIM_Form_Alpha_C, SWIM_AalphaC, SWIM_V0alphaC, SWIM_BalphaC, 1.0);
	beta  = calculate_form_parameter((int) SWIM_Form_Beta_C, SWIM_AbetaC, SWIM_V0betaC, SWIM_BbetaC, 1.0);

	A = alpha + beta;
	B = alpha;

	C_CONST = integrate_method(integrationMethod, C_CONST, A, B, STEP);

	gateCa = intpower(C_CONST, (int) SWIM_C_POWER);
	I_CA = gateCa * (SWIM_CA_E - E_POT[0]) * SWIM_CA_G;

	/* set gate variables for graph windows */

	GB_I_CA1 = M_CONST / I_SCALE;
	GB_I_CA2 = H_CONST / I_SCALE;
	GB_I_KCA = N_CONST / I_SCALE;
	GB_I_A = C_CONST / I_SCALE;

	/* calculate index for access to input current */

	for (I = 0; I <= curNumOfDendSegs; I++)
		if (L >= ICUR[I].starttime[ICUR[I].index])
			ICUR[I].index++;

	/* calculate stochastic noise current if defined */

	if (noiseDefined && Noise)
	{
		noise *= exp(-STEP / noiseDecay);
		noise += noiseevents(STEP, noiseIntensity);
		I_NOISE = noise * noiseStrength;
	}
	else I_NOISE = 0.0;

	/* calculation of temporary new soma & dendrite variables
	   COMPARTMENT (S,R) = ((S-1) * NEURON_REGIONS) + R <----> SOMA = 0 */

	if (DENDRITE_STEMS == 0)
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + SWIM_NAEmNAG * gateNa +
			SWIM_KEmKG * gateK + SWIM_CAEmCAG * gateCa;

		eTempA[0] = (SOMA_REST_Gm + SWIM_NA_G * gateNa + SWIM_K_G * gateK + SWIM_CA_G * gateCa) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

	} /* End of point neuron calculations */
	else
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

/*		eNeighbors = 0;*/
/*		for (J = 1; J <= (int) DENDRITE_STEMS; J++)*/
/*			eNeighbors += E_POT[(int) ((J - 1) * DENDRITE_REGIONS) + 1];*/
		
		eNeighbors = E_POT[1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + eNeighbors * SOMA_Gcore +
			SWIM_NAEmNAG * gateNa + SWIM_KEmKG * gateK + SWIM_CAEmCAG * gateCa;

		eTempA[0] = (SGpDSmDG + SWIM_NA_G * gateNa + SWIM_K_G * gateK + SWIM_CA_G * gateCa) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

		/* calculation of dendritic temporary potential factors */

		for (I = 1; I <= curNumOfDendSegs; I++)
		{
/*			if ( (I / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
			if (I == DENDRITE_REGIONS)      /* End of stem */
			{
				eNeighbors = E_POT[I - 1];
				numNeighbors = 1;
			}
			else
			{
/*				if ( ((I + DENDRITE_REGIONS - 1) / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
				if (I == 1)
					eNeighbors = E_POT[0] + E_POT[2];	  /* dendrite piece next to soma */
				else	eNeighbors = E_POT[I - 1] + E_POT[I + 1]; /* dendrite piece in stem middle */

				numNeighbors = 2;
			}

/*			if (I == (int) (((IStem - 1) * (int) DENDRITE_REGIONS) + IRegion))*/
			if (I == IRegion)
				iExtD = (L >= SDSTRT && L < SDSTP) * DendCur;
			else iExtD = 0;

			inpCur[I] = (int) ((L >= ICUR[I].starttime[ICUR[I].index - 1] && L < ICUR[I].starttime[ICUR[I].index]))
				* ICUR[I].current[ICUR[I].index - 1];

			/*
				I(Leak) = (Dendritepotential - Leakpotential) * Leakconductance
					==> A = DENDRITE_REST_Gm (Leakconductance)
					    B = DENDRITE_REST_Gm * DendritePotential (DLmDG)
				I(Core) = SUM( E(c) - E) * DENDRITE_Gcore, with c = neighbours of compartment
					DENDRTITE_Gcore = Conductance from dendrite to soma
					==> A = numNeighbors * DENDRITE_Gcore
					    B = eNeighbors * DENDRITE_Gcore
			*/

			tempExp = ICH[I] + IEL[I];
			SynCur += tempExp;
			tempExp += iExtD + inpCur[I] + DLmDG + eNeighbors * DENDRITE_Gcore;
			TotCur += tempExp;

			eTempA[I] = (DENDRITE_REST_Gm + numNeighbors * DENDRITE_Gcore) / DENDRITE_Cm;
			eTempB[I] = tempExp / DENDRITE_Cm;

		} /* end of dendritic tree parameter calculations */
	}

	/* calculate new soma & dendrite potential values */

	oldEPot = E_POT[0];
	for (I = 0; I <= curNumOfDendSegs; I++)
		E_POT[I] = integrate_method(integrationMethod, E_POT[I], eTempA[I], eTempB[I], STEP);

	/* increment spike counter if a spike has occured */

	if (currentNeuron->neuronType == spiker &&
	    oldEPot < SWIM_SPIKE_THRESHOLD && E_POT[0] > SWIM_SPIKE_THRESHOLD)
		SpikeCounter++;
	
	/* free dynamic memory */

	free(inpCur);
	free(eTempA);
	free(eTempB);

} /* end of calculate_SWIM_neuron */

/************************************************************************************************************************

 	FUNCTION	: calculate_GB_neuron(void)
 
	PURPOSE		: calculates a Golowasch-Buchholz neuron

			  The differential equations are solved in respect of the integrationMethod variable

			  A contains all multiplication factors of the differentiation variable and
			  B contains all non-multiplication factors of the differentiation variable.

			  In the case of switching from a 4-point network model to a 2-point or 1-point model
			  its necessary to relink existing extern soma currents from the dendrite to the soma.
			  Hence a new variable (iExtSpseudo) is needed to hold the pseudo current.

	RETURNS		: nothing

************************************************************************************************************************/

void calculate_GB_neuron()
{
	int	I;					/* general help variables */
	unsigned int len;				/* length of allocation area */
	double  RdivFmul2 = 4.308507882e-5;		/* universal gas constant divided by (2 * Faraday constant) */
	double  tempExp;				/* temporare exponential expressions */
	double  A, B;					/* A and B value of differential equation for ion channels */
	double  alpha, beta;				/* alpha and beta forms of gate variables */
	double	gateNa, gateK, gateCa1, gateCa2,
		gateA, gateKCa, gateIR, gateP;		/* gate variables for the channels */
	double  iExtS, iExtD;				/* extern currents for soma and dendrite */
	double	iExtSpseudo;				/* extern pseudo current for soma */
	double  eNeighbors;				/* potential of the neighbors of a current compartment */
	double  numNeighbors;				/* number of neighbors a current compartment has */
	double  oldEPot;				/* old soma potential, used to determine a new spike */
	double	*inpCur;				/* input current into neuron */
	double	*eTempA;				/* A value of differential equation for compartment potentials */
	double  *eTempB;				/* B value of differential equation for compartment potentials */

	/* allocate memory for input currents, potentials, ... */

	len = (unsigned int) (MaxNumberOfDendritStems * (curNumOfDendSegs + 1) * sizeof(double));

	inpCur = (double *) malloc((size_t) len);

	if (! inpCur)
	{
		error(128);
		return;
	}

	eTempA = (double *) malloc((size_t) len);

	if (! eTempA)
	{
		error(128);
		return;
	}

	eTempB = (double *) malloc((size_t) len);

	if (! eTempB)
	{
		error(128);
		return;
	}

	/* calculations for active sodium channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_M, GB_AalphaM, GB_V0alphaM, GB_BalphaM, 1.0);
	beta  = calculate_form_parameter((int) GB_Form_Beta_M, GB_AbetaM, GB_V0betaM, GB_BbetaM, 0.0);

	A = alpha + beta;
	B = alpha;

	M_CONST = integrate_method(integrationMethod, M_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) GB_Form_Alpha_H, GB_AalphaH, GB_V0alphaH, GB_BalphaH, 0.0);
	beta  = calculate_form_parameter((int) GB_Form_Beta_H, GB_AbetaH, GB_V0betaH, GB_BbetaH, 1.0);

	A = alpha + beta;
	B = alpha;

	H_CONST = integrate_method(integrationMethod, H_CONST, A, B, STEP);

	gateNa = intpower(M_CONST, (int) GB_M_POWER) * intpower(H_CONST, (int) GB_H_POWER);
	I_NA = gateNa * (GB_NA_E - E_POT[0]) * GB_NA_G;

	/* calculations for active potassium channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_N, GB_AinfinityN, GB_V0infinityN, GB_BinfinityN, 1.0);
	beta  = calculate_form_parameter((int) GB_Form_Beta_N, GB_AtauN, GB_V0tauN, GB_BtauN, 1.0);

	A = 1.0 / beta;
	B = alpha / beta;

	N_CONST = integrate_method(integrationMethod, N_CONST, A, B, STEP);

	gateK = intpower(N_CONST, (int) GB_N_POWER);
	I_K = gateK * (GB_K_E - E_POT[0]) * GB_K_G;

	/* calculations for the calcium concentration */

	A = GB_K_CA;
	B = A * GB_C0_CA + GB_K_ICA * (GB_I_CA1 + GB_I_CA2);

	GB_CONC_CA = integrate_method(integrationMethod, GB_CONC_CA, A, B, STEP);

	/* calculations for the calcium equilibrium potential */

	GB_CA_E = RdivFmul2 * GB_TEMP * log(GB_CA_CA / GB_CONC_CA);
	I_CA = GB_CONC_CA;	/* use this for observing calcium equilibrium potential in graph window */

	/* calculations for first active calcium channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_C, 1.0, GB_V0c, GB_Bc, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_C, 1.0, 0.0, 1.0, GB_Tc);

	A = beta;
	B = beta * alpha;

	C_CONST = integrate_method(integrationMethod, C_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) GB_Form_Alpha_D, 1.0, GB_V0d, GB_Bd, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_D, 1.0, 0.0, 1.0, GB_Td);

	A = beta;
	B = beta * alpha;

	D_CONST = integrate_method(integrationMethod, D_CONST, A, B, STEP);

	gateCa1 = intpower(C_CONST, (int) GB_C_POWER) * intpower(D_CONST, (int) GB_D_POWER);
	GB_I_CA1 = gateCa1 * (GB_CA_E - E_POT[0]) * GB_CA1_G;

	/* calculations for second active calcium channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_E, 1.0, GB_V0e, GB_Be, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_E, 1.0, 0.0, 1.0, GB_Te);

	A = beta;
	B = beta * alpha;

	E_CONST = integrate_method(integrationMethod, E_CONST, A, B, STEP);

	gateCa2 = intpower(E_CONST, (int) GB_E_POWER);
	GB_I_CA2 = gateCa2 * (GB_CA_E - E_POT[0]) * GB_CA2_G;

	/* calculations for active A-channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_A, 1.0, GB_V0a, GB_Ba, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_A, 1.0, 0.0, 1.0, GB_Ta);

	A = beta;
	B = beta * alpha;

	A_CONST = integrate_method(integrationMethod, A_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) GB_Form_Alpha_B1, 1.0, GB_V0b1, GB_Bb1, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_B1, 1.0, 0.0, 1.0, GB_Tb1);

	A = beta;
	B = beta * alpha;

	B1_CONST = integrate_method(integrationMethod, B1_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) GB_Form_Alpha_B2, 1.0, GB_V0alphab2, GB_Balphab2, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_B2, GB_Ab2, GB_V0b2, GB_Bb2, 1.0);

	A = beta;
	B = beta * alpha;

	B2_CONST = integrate_method(integrationMethod, B2_CONST, A, B, STEP);

	GB_GAMMA = 1.0 / (1.0 + exp((E_POT[0] - GB_V0g) / GB_Bg));

	gateA = intpower(A_CONST, (int) GB_A_POWER) * (GB_GAMMA * intpower(B1_CONST, (int) GB_B1_POWER) +
		(1.0 - GB_GAMMA) * intpower(B2_CONST, (int) GB_B2_POWER));
	GB_I_A = gateA * (GB_A_E - E_POT[0]) * GB_A_G;

	/* calculations for active calcium dependend potassium channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_P, 1.0, GB_V0p1 - GB_Cca * GB_CONC_CA, GB_Bp1, 1.0);
	beta  = calculate_form_parameter((int) GB_Form_Beta_P, 1.0, GB_V0p2 - GB_Cca * GB_CONC_CA, GB_Bp2, 1.0);

	A = 1.0 / GB_Tp;
	B = A * alpha * beta * GB_CONC_CA / (GB_C1 + GB_CONC_CA);

	P_CONST = integrate_method(integrationMethod, P_CONST, A, B, STEP);

	alpha = calculate_form_parameter((int) GB_Form_Alpha_Q, GB_C2, 0.0, 1.0, GB_C3 + GB_CONC_CA);
	beta  = calculate_form_parameter((int) GB_Form_Beta_Q, 1.0, 0.0, 1.0, GB_Tq);

	A = beta;
	B = beta * alpha;

	Q_CONST = integrate_method(integrationMethod, Q_CONST, A, B, STEP);

	gateKCa = intpower(P_CONST, (int) GB_P_POWER) * intpower(Q_CONST, (int) GB_Q_POWER);
	GB_I_KCA = gateKCa * (GB_KCA_E - E_POT[0]) * GB_KCA_G;

	/* calculations for active inward rectifier channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_R, 1.0, GB_V0r, GB_Br, 1.0);
	beta  = calculate_form_parameter((int) GB_Form_Beta_R, 1.0 / GB_Atau_r, GB_V0tau_r, GB_Btau_r, 1.0);

	A = beta;
	B = beta * alpha;

	R_CONST = integrate_method(integrationMethod, R_CONST, A, B, STEP);

	gateIR = intpower(R_CONST, (int) GB_R_POWER);
	GB_I_IR = gateIR * (GB_IR_E - E_POT[0]) * GB_IR_G;

	/* calculations for proctolin channels in soma */

	alpha = calculate_form_parameter((int) GB_Form_Alpha_S, 1.0, GB_V0s, GB_Bs, 1.0);
	beta =  calculate_form_parameter((int) GB_Form_Beta_S, 1.0, 0.0, 1.0, GB_Ts);

	A = beta;
	B = beta * alpha;

	S_CONST = integrate_method(integrationMethod, S_CONST, A, B, STEP);

	gateP = intpower(S_CONST, (int) GB_S_POWER);
	GB_I_P = gateP * (GB_P_E - E_POT[0]) * GB_P_G;

	/* calculate index for access to input current */

	for (I = 0; I <= curNumOfDendSegs; I++)
		if (L >= ICUR[I].starttime[ICUR[I].index])
			ICUR[I].index++;

	/* calculate stochastic noise current if defined */

	if (noiseDefined && Noise)
	{
		noise *= exp(-STEP / noiseDecay);
		noise += noiseevents(STEP, noiseIntensity);
		I_NOISE = noise * noiseStrength;
	}
	else I_NOISE = 0.0;

	/* calculation of temporary new soma & dendrite variables
	   COMPARTMENT (S,R) = ((S-1) * NEURON_REGIONS) + R <----> SOMA = 0 */

	if (DENDRITE_STEMS == 0)
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + GB_NAEmNAG * gateNa +
			GB_KEmKG * gateK + GB_CA_E * GB_CA1_G * gateCa1 + GB_CA_E * GB_CA2_G * gateCa2 +
			GB_AEmAG * gateA + GB_KCAEmKCAG * gateKCa + GB_IREmIRG * gateIR + GB_PEmPG * gateP;

		eTempA[0] = (SOMA_REST_Gm + GB_NA_G * gateNa + GB_K_G * gateK + GB_CA1_G * gateCa1 + GB_CA2_G * gateCa2 +
			GB_A_G * gateA + GB_KCA_G * gateKCa + GB_IR_G * gateIR + GB_P_G * gateP) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

	} /* End of point neuron calculations */
	else
	{
		iExtS = (L >= SCSTRT && L < SCSTP) * SomaCur;

		iExtSpseudo = (L >= PseudoSCSTRT && L < PseudoSCSTP) * PseudoSomaCur;

		inpCur[0] = (int) ((L >= ICUR[0].starttime[ICUR[0].index - 1] && L < ICUR[0].starttime[ICUR[0].index]))
			* ICUR[0].current[ICUR[0].index - 1];

/*		eNeighbors = 0;*/
/*		for (J = 1; J <= (int) DENDRITE_STEMS; J++)*/
/*			eNeighbors += E_POT[(int) ((J - 1) * DENDRITE_REGIONS) + 1];*/

		eNeighbors = E_POT[1];

		SynCur = ICH[0] + IEL[0];
		TotCur = SynCur + iExtS + iExtSpseudo + inpCur[0] + I_NOISE + SLmSG + eNeighbors * SOMA_Gcore +
			GB_NAEmNAG * gateNa + GB_KEmKG * gateK + GB_CA_E * GB_CA1_G * gateCa1 + GB_CA_E * GB_CA2_G *
			gateCa2 + GB_AEmAG * gateA + GB_KCAEmKCAG * gateKCa + GB_IREmIRG * gateIR + GB_PEmPG * gateP;

		eTempA[0] = (SGpDSmDG + GB_NA_G * gateNa + GB_K_G * gateK + GB_CA1_G * gateCa1 + GB_CA2_G * gateCa2 +
			GB_A_G * gateA + GB_KCA_G * gateKCa + GB_IR_G * gateIR + GB_P_G * gateP) / SOMA_Cm;
		eTempB[0] = TotCur / SOMA_Cm;

		/* calculation of dendritic temporary potential factors */

		for (I = 1; I <= curNumOfDendSegs; I++)
		{
/*			if ( (I / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
			if (I == DENDRITE_REGIONS)      /* End of stem */
			{
				eNeighbors = E_POT[I - 1];
				numNeighbors = 1;
			}
			else
			{
/*				if ( ((I + DENDRITE_REGIONS - 1) / DENDRITE_STEMS) == DENDRITE_REGIONS )*/
				if (I == 1)
					eNeighbors = E_POT[0] + E_POT[2];	  /* dendrite piece next to soma */
				else	eNeighbors = E_POT[I - 1] + E_POT[I + 1]; /* dendrite piece in stem middle */

				numNeighbors = 2;
			}

/*			if (I == (int) (((IStem - 1) * (int) DENDRITE_REGIONS) + IRegion))*/
			if (I == IRegion)
				iExtD = (L >= SDSTRT && L < SDSTP) * DendCur;
			else iExtD = 0;

			inpCur[I] = (int) ((L >= ICUR[I].starttime[ICUR[I].index - 1] && L < ICUR[I].starttime[ICUR[I].index]))
				* ICUR[I].current[ICUR[I].index - 1];

			/*
				I(Leak) = (Dendritepotential - Leakpotential) * Leakconductance
					==> A = DENDRITE_REST_Gm (Leakconductance)
					    B = DENDRITE_REST_Gm * DendritePotential (DLmDG)
				I(Core) = SUM( E(c) - E) * DENDRITE_Gcore, with c = neighbours of compartment
					DENDRTITE_Gcore = Conductance from dendrite to soma
					==> A = numNeighbors * DENDRITE_Gcore
					    B = eNeighbors * DENDRITE_Gcore
			*/

			tempExp = ICH[I] + IEL[I];
			SynCur += tempExp;
			tempExp += iExtD + inpCur[I] + DLmDG + eNeighbors * DENDRITE_Gcore;
			TotCur += tempExp;

			eTempA[I] = (DENDRITE_REST_Gm + numNeighbors * DENDRITE_Gcore) / DENDRITE_Cm;
			eTempB[I] = tempExp / DENDRITE_Cm;

		} /* end of dendritic tree parameter calculations */
	}

	/* calculate new soma & dendrite potential values */

	oldEPot = E_POT[0];
	for (I = 0; I <= curNumOfDendSegs; I++)
		E_POT[I] = integrate_method(integrationMethod, E_POT[I], eTempA[I], eTempB[I], STEP);

	/* increment spike counter if a spike has occured */

	if (currentNeuron->neuronType == spiker &&
	    oldEPot < GB_SPIKE_THRESHOLD && E_POT[0] > GB_SPIKE_THRESHOLD)
		SpikeCounter++;
	
	/* free dynamic memory */

	free(inpCur);
	free(eTempA);
	free(eTempB);

} /* end of calculate_GB_neuron */
