/************************************************************************************************************************

	PROGRAM:	biosim2gnuplot

	DATE:		07/04/1993

	LAST UPDATE:	13/04/1993

	VERSION:	1.0

	FILENAME:	biosim2gnuplot.c

	PURPOSE:	this is a converter program that transforms biosim graph output files (*.asc) into files
			that can be used for gnuplot.
			After generating the data files for gnuplot the gnuplot program itself is called with the generated instruction file "gnuplot.instructions".

	USAGE:		biosim2gnuplot <biosimdatafile> <gnuplotdatafile>
			the biosimdatafile must be a biosim graph output (*.asc) file and
			the gnuplotdatafile is an arbitrarily base filename concatenated with the current object
			read in and the current object number. It contain the data in gnuplot format.
	
	LIMITATIONS:	this conversion file can only handle one parameter plottet in the biosim
			graph file. Other parameters will be ignored.

	COPYRIGHT:	(c) 1993 by Stefan Bergdoll, All rights reserved

************************************************************************************************************************/

#include "biosim.h"
#include "copying.h"

main(argc, argv)
unsigned int argc;	/* number of options in the command line */
char *argv[];		/* pointer to command line options */
{
	FILE	*inputFile,	/* biosim graph file */
		*outputFile,	/* gnuplot data file */
		*gnuFile;	/* gnuplot instructions file */
	int	fileType,	/* file type of biosim graph file */
		objectNumber;	/* object number just read in */
	char	dummy[512],	/* for not used strings read in */
		object[128],	/* object name (neuron or synapse) */
		filename[128];	/* filename for output file */
	double	time,		/* time stamp */
		stepwidth,	/* step width for time */
		oldnum,		/* old number read in */
		min,		/* minimum number */
		max;		/* maximum number */
	Boolean firstTime,	/* flag to decide if first number was read */
		checkMax;	/* flag if maximum or minimum value must be checked */

	/* check for sufficiant arguments */

	if (argc != 3)
	{
		fprintf(stderr, "\nusage: biosim2gnuplot <biosim graph file> <gnuplot data file>\n");
		exit(1);
	}

	/* open biosim graph file as input file in read mode */

	if ((inputFile = fopen(argv[1], "r")) == NULL)
	{
		fprintf(stderr, "\nBiosim input file could not be read\n");
		exit(1);
	}

	/* read in file type */

	if (fscanf(inputFile, "%i", &fileType) == 0)
	{
		fprintf(stderr, "\nBiosim input file could not be read\n");
		exit(1);
	}

	/* check if filetype is valid */

	if (fileType != OUTPUT_FILE && fileType != OUTPUT_MINMAX_FILE &&
	    fileType != S_OUTPUT_FILE && fileType != S_OUTPUT_MINMAX_FILE)
	{
		fprintf(stderr, "\nError: wrong biosim file type\n");
		exit(1);
	}

	/* read in time step width, object and object number */

	fscanf(inputFile, "%s%E", dummy, &stepwidth);
	fscanf(inputFile, "%s%i", object, &objectNumber);
	sprintf(filename, "%s.%s%i", argv[2], object, objectNumber);

	/* open gnuplot data file in write mode */

	if ((outputFile = fopen(filename, "w")) == NULL)
	{
		fprintf(stderr, "\nGnuplot data file could not be wrote\n");
		exit(1);
	}

	/* open gnuplot instruction file in write mode */

	if ((gnuFile = fopen("gnuplot.instructions", "w")) == NULL)
	{
		fprintf(stderr, "\nGnuplot instruction file could not be wrote\n");
		exit(1);
	}
	else
	{
		fprintf(gnuFile, "set data style lines\n");
		fprintf(gnuFile, "set grid\n");
	}

	/* initialization */

	time = 0.0;
	fgets(dummy, 511, inputFile); /* skip over newline char */
	fgets(dummy, 511, inputFile); /* read in variable names */
	checkMax = TRUE;
	firstTime = TRUE;

	/* read in input file until end of file is reached */

	while(! feof(inputFile))
	{
		/* read in number */

		if (! fscanf(inputFile, "%E", &min))
		{
			/* skip the rest of the line because the line contains more than one parameter */

			fgets(dummy, 511, inputFile);
			fscanf(inputFile, "%E", &min);

		}
		if (firstTime)
		{
			oldnum = min;
			firstTime = FALSE;
		}

		/* check if a new object comes in file */

		if (min == OUTPUT_FILE || min == OUTPUT_MINMAX_FILE ||
		    min == S_OUTPUT_FILE || min == S_OUTPUT_MINMAX_FILE)
		{
			/* close current gnuplot data file */

			fclose(outputFile);

			/* write instructions for gnuplot */

			fprintf(gnuFile, "plot '%s'\npause -1\n", filename);

			/* read in time step width, object and object number */

			fscanf(inputFile, "%s%E", dummy, &stepwidth);
			fscanf(inputFile, "%s%i", object, &objectNumber);
			fgets(dummy, 511, inputFile); /* skip over newline char */
			fgets(dummy, 511, inputFile); /* read in variable names */
			sprintf(filename, "%s.%s%i", argv[2], object, objectNumber);	/* generate file name */

			/* open new gnuplot data file in write mode */

			if ((outputFile = fopen(filename, "w")) == NULL)
			{
				fprintf(stderr, "\nGnuplot data file could not be wrote\n");
				exit(1);
			}

			/* initialization */

			time = 0.0;
			firstTime = TRUE;
			checkMax = TRUE;
		}
		else	/* no new object must be read in */
		{
			/* check if file contains minimum and maximum values */

			if (fileType == OUTPUT_MINMAX_FILE || fileType == S_OUTPUT_MINMAX_FILE)
			{
				/* read in maximum value */

				fscanf(inputFile, ";%E", &max);

				/* write data to gnuplot data file in */

				if (checkMax)
					if (max < oldnum)
					{
						checkMax = FALSE;
						fprintf(outputFile, "%E %E\n", time, min);
						oldnum = min;
					}
					else
					{
						fprintf(outputFile, "%E %E\n", time, max);
						oldnum = max;
					}
				else	if (min >= oldnum)
					{
						checkMax = TRUE;
						fprintf(outputFile, "%E %E\n", time, max);
						oldnum = max;
					}
					else
					{
						fprintf(outputFile, "%E %E\n", time, min);
						oldnum = min;
					}
			}
			else	fprintf(outputFile, "%E %E\n", time, min);
			time += stepwidth;	/* update time */
		}
	}

	/* write gnuplot instructions and close files */

	fprintf(gnuFile, "plot '%s'\npause -1\n", filename);
	fclose(gnuFile);
	fclose(inputFile);
	fclose(outputFile);

	/* execute gnuplot with the generated instruction file */

	system("gnuplot ./gnuplot.instructions");
}
