/************************************************************************************************************************

	PROGRAM:	BIOSIM

	FILENAME:	biosim.h

	PURPOSE:	The main header file for BIOSIM
			This file contains several DEFINES, structures and global variable declarations for BIOSIM 

************************************************************************************************************************/

#define CLK_TCK 1000		/* clock ticks per second */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifndef BATCH
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/Shell.h>
#include <X11/X10.h>

#ifdef X11R5
#include <X11/Xmu/Editres.h>
#endif

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/BulletinB.h>
#include <Xm/CascadeB.h>
#include <Xm/Command.h>
#include <Xm/CutPaste.h>
#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>
#include <Xm/DrawnB.h>
#include <Xm/FileSB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/MainW.h>
#include <Xm/MenuShell.h>
#include <Xm/MessageB.h>
#include <Xm/PanedW.h>
#include <Xm/Protocols.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/ScrolledW.h>
#include <Xm/SelectioB.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <Xm/ArrowBG.h>
#include <Xm/CascadeBG.h>
#include <Xm/LabelG.h>
#include <Xm/PushBG.h>
#include <Xm/SeparatoG.h>
#include <Xm/ToggleBG.h>
#endif

#include "text.h"
#include "parameter.h"
#include "neurvars.h"

/************************************************************************************************************************

	Global definitions for BIOSIM

************************************************************************************************************************/

#ifndef TRUE
#define TRUE	1
#define FALSE	0
#endif

#ifndef Boolean
#define Boolean	int
#endif

/******	Exit codes for BIOSIM	******/

#define GOOD	0			/* exit code for normal ending */
#define NOTGOOD 1			/* exit code for abnormal ending */

/******	Filename extensions for BIOSIM	******/

#define configurationFileExtension	"con"				/* selection mask for configuration files */
#define parameterFileExtension		"par"				/* selection mask for parameter files */
#define simulationFileExtension		"sim"				/* selection mask for simulation files */
#define neuronParameterExtension	"gen"				/* selection mask for neuron param files */
#define graphOutputExtension		"asc"				/* selection mask for graph output files */
#define deleteFileExtension		"*"				/* selection mask for delete files */
#define otherFileExtension		"*"				/* selection mask for all files */

#ifndef BATCH
/******	Graph definitions	******/

#define charWidth 			5		/* width of a character */
#define mvSpace 			40		/* indention space in pixels for y-axis of graph */
#define msSpace 			25		/* indention space in pixels for x-axis of graph */
#define textOffset			3		/* text offset for neuron number in drawing area */
#define MIN_MS_SPACE			30		/* minimal interval in pixels for inscribing x-axis */
#define MIN_MV_SPACE			15		/* minimal interval in pixels for inscribing y-axis */
#endif

/******	Mathematical definitions	******/

#define EULER				1
#define EXPEULER			2
#define TRAPEZOID			3
#define RUNGEKUTTA			4

#ifndef BATCH
/******	Neuron color description	******/
#define numActivityColors		16		/* this number always have to represent the number
							   of colors in the variable colorList */
#endif

#ifndef BATCH
/******	Neuron size descriptions	******/

#define defaultNeuronDiameter		20		/* standard neuron diameter in pixels */
#define defaultSynapseDiameter		8		/* standard synapse diameter in pixels */
#define defaultDendriteWidth		3		/* standard dendrite width in pixels for 3 compartments */
#define defaultDendriteLength		30		/* standard dendrite length in pixels for 3 compartments */
#define defaultStemLength		10		/* standard dendrite length in pixels for 1 compartment */
#define defaultStemWidth		4		/* standard dendrite width in pixels for 1 compartment */
#define defaultOwnConnectionRadius	15		/* standard radius for self connecting neurons */
#endif
#define MaxNumberOfDendritStems		1		/* max number of dendrit stems */

/******	Scaling definitions for graph output	******/

#define P_SCALE				1e2		/* conversion from and to % */
#define T_SCALE				1e3		/* conversion between ms and s */
#define E_SCALE				1e3		/* conversion between mV and V */
#define I_SCALE				1e9		/* conversion between nA and A */
#define S_SCALE				1e9		/* conversion between nS and S */
#define C_SCALE				1e12		/* conversion between pF and F */
#define CON_SCALE			1e7		/* conversion for graph rep. of current conductance of synapse */
#define MEM_SCALE			1		/* maximum value of synaptic memory */
#define STR_SCALE			1		/* maximum value of axo-axonic strength */

/******	Special character definitions	******/

#define EOS 				0x00		/* end of string */

/******	File descriptor definitions	******/

#define NEURON_COMBINED_FILE 		-2		/* file descriptor for a neuron parameter file */
#define NETWORK_FILE			100		/* file descriptor for a network file */
#define PARAMETER_FILE			200    		/* file descriptor for a network parameter file */
#define CONFIGURATION_FILE		300		/* file descriptor for a network configuration file */
#define COMMAND_FILE			1000		/* file descriptor for a input command file */
#define SELECTION_FILE			2000		/* file descriptor for a output selection file */
#define CURRENT_FILE			3000		/* file descriptor for a input current file */
#define OUTPUT_FILE			4000		/* file descriptor for a neuron graph output file */
#define OUTPUT_MINMAX_FILE		4001		/* file descriptor for a neuron graph output file min-max values */
#define S_OUTPUT_FILE			5000		/* file descriptor for a neuron graph output file */
#define S_OUTPUT_MINMAX_FILE		5001		/* file descriptor for a neuron graph output file min-max values */

/******	Buffer sizes	******/

#define DefBufferSize 			1024		/* size of buffer for reading .sim, .par or .con files	*/
#define NumberSize 			30		/* max size a number can have in bytes */

/****** Number of variables needed for neuron compartments and ion channels	******/

#define SOMA_VARS		    	5		/* number of soma variables */
#define DENDRITE_VARS		    	7		/* number of dendrite variables */
#define ONOFF_VARS		    	12		/* number of variables needed for on/off model */
#define HH_VARS				35		/* number of variables needed for HH model */
#define SWIM_VARS			47		/* number of variables needed for SWIM model */
#define GB_VARS				133		/* number of variables needed for GB model */

/****** Locations needed for neuron compartments and ion channels	******/

#define SOMA_PLACE		    	0				   /* start location for soma variables */
#define DENDRITE_PLACE		    	(SOMA_PLACE + SOMA_VARS)	   /* start location for dendrite variables */
#define ONOFF_PLACE		    	(DENDRITE_PLACE + DENDRITE_VARS)   /* start location for on/off variables */
#define HH_PLACE			(ONOFF_PLACE + ONOFF_VARS)	   /* start location for HH variables */
#define SWIM_PLACE		    	(HH_PLACE + HH_VARS)	   	   /* start location for SWIM variables */
#define GB_PLACE			(SWIM_PLACE + SWIM_VARS)	   /* start location for GB variables */
#define NEURON_LENGTH		    	(GB_PLACE + GB_VARS)	   	   /* number of neuron parameters */

#ifndef BATCH
/******	Dialog id's for general help	******/

#define DIALOG_HELP_INDEX		400
#define DIALOG_HELP_FEATURES		401
#define DIALOG_HELP_INTRO		402
#define DIALOG_HELP_TARGET		403
#define DIALOG_HELP_CELLMODEL		404
#define DIALOG_HELP_CONTEXT		405
#define DIALOG_HELP_MOUSE		406
#define DIALOG_HELP_KEYBOARD		407
#define DIALOG_HELP_GLOSSAR		408

/***** reserved keywords for inter client communication with help server *****/
#ifdef DEUTSCH
#define HELP_INDEX			"Index"
#else
#define HELP_INDEX			"Index"
#endif
#ifdef DEUTSCH
#define HELP_FEATURES			"Leistungsmerkmale von BIOSIM"
#else
#define HELP_FEATURES			"Features of BIOSIM"
#endif
#ifdef DEUTSCH
#define HELP_INTRO			"Allgemeine Vorgehensweise zur Erzeugung eines neuronalen Netzes"
#else
#define HELP_INTRO			"General steps in creating a neural net"
#endif
#ifdef DEUTSCH
#define HELP_TARGET			"Einsatzgebiete von BIOSIM"
#else
#define HELP_TARGET			"Application areas of BIOSIM"
#endif
#ifdef DEUTSCH
#define HELP_CELLMODEL			"Das Zell-Modell"
#else
#define HELP_CELLMODEL			"The cell model"
#endif
#ifdef DEUTSCH
#define HELP_CONTEXT			"Mittleres Mausklick-Men"
#else
#define HELP_CONTEXT			"Middle mouse click menu"
#endif
#ifdef DEUTSCH
#define HELP_MOUSE			"Funktionalitt der Maus"
#else
#define HELP_MOUSE			"Functionality of the mouse"
#endif
#ifdef DEUTSCH
#define HELP_KEYBOARD			"Funktionalitt der Tastatur"
#else
#define HELP_KEYBOARD			"Functionality of the keyboard"
#endif
#ifdef DEUTSCH
#define HELP_GLOSSAR			"Glossar"
#else
#define HELP_GLOSSAR			"Glossar"
#endif
#ifdef DEUTSCH
#define HELP_OPENFILE			"Datei - ffnen"
#else
#define HELP_OPENFILE			"File - Open"
#endif
#ifdef DEUTSCH
#define HELP_PRINTFILE			"Datei - Drucken"
#else
#define HELP_PRINTFILE			"File - Print"
#endif
#ifdef DEUTSCH
#define HELP_DELETEFILE			"Datei - Lschen"
#else
#define HELP_DELETEFILE			"File - Delete"
#endif
#ifdef DEUTSCH
#define HELP_SAVEAS			"Datei - Speichern als ..."
#else
#define HELP_SAVEAS			"File - Save as ..."
#endif
#ifdef DEUTSCH
#define HELP_NOISE			"Simulation - Rauschparameter"
#else
#define HELP_NOISE			"Simulation - Noise Parameter"
#endif
#ifdef DEUTSCH
#define HELP_STDDEV			"Simulation - Verteilungsparameter"
#else
#define HELP_STDDEV			"Simulation - Distribution Parameter"
#endif
#ifdef DEUTSCH
#define HELP_ONOFFMODEL			"Das Ein-/Aus-Kanal-Modell"
#else
#define HELP_ONOFFMODEL			"The ON/OFF-channel-model"
#endif
#ifdef DEUTSCH
#define HELP_HHMODEL			"Das Hodgkin-Huxley-Modell"
#else
#define HELP_HHMODEL			"The Hodgkin-Huxley-model"
#endif
#ifdef DEUTSCH
#define HELP_SWIMMODEL			"Das SWIM-Modell"
#else
#define HELP_SWIMMODEL			"The SWIM-model"
#endif
#ifdef DEUTSCH
#define HELP_GBMODEL			"Das Golowasch-Buchholz-Modell"
#else
#define HELP_GBMODEL			"The Golowasch-Buchholz-model"
#endif
#ifdef DEUTSCH
#define HELP_NEURON			"Neuronenparameter editieren"
#else
#define HELP_NEURON			"Edit Neuron Parameters"
#endif
#ifdef DEUTSCH
#define HELP_CURRENT			"Neuronen-Stromparameter editieren"
#else
#define HELP_CURRENT			"Edit Neuron Current Parameters"
#endif
#ifdef DEUTSCH
#define HELP_ADDSYNAPSE			"Synapsen editieren"
#else
#define HELP_ADDSYNAPSE			"Edit Synapses"
#endif
#ifdef DEUTSCH
#define HELP_EDITSYNAPSE		"Synapsen ndern"
#else
#define HELP_EDITSYNAPSE		"Edit synapses"
#endif
#ifdef DEUTSCH
#define HELP_SIMPARS			"Simulation - Simulationsparameter"
#else
#define HELP_SIMPARS			"Simulation - Simulation Parameters"
#endif
#ifdef DEUTSCH
#define HELP_NEWSTYPE			"Synapsen editieren"
#else
#define HELP_NEWSTYPE			"Edit Synapses"
#endif
#ifdef DEUTSCH
#define HELP_LOADGENERAL		"Neuronendateien"
#else
#define HELP_LOADGENERAL		"Neuron files"
#endif
#ifdef DEUTSCH
#define HELP_ASCII			"Optionen - Aufzeichnungsrate"
#else
#define HELP_ASCII			"Options - Ascii-Data Output Rate"
#endif
#ifdef DEUTSCH
#define HELP_SCALE			"Optionen - Skalieren"
#else
#define HELP_SCALE			"Options - Scaling"
#endif
#ifdef DEUTSCH
#define HELP_ZOOM			"Optionen - Zoomen"
#else
#define HELP_ZOOM			"Options - Zooming"
#endif
#ifdef DEUTSCH
#define HELP_SAVEGRAPH			"Datei - speichern"
#else
#define HELP_SAVEGRAPH			"File - save"
#endif
#ifdef DEUTSCH
#define HELP_AXOSYNAPSE			"Axo-axonische Synapsen editieren"
#else
#define HELP_AXOSYNAPSE			"Edit axo-axonic Synapses"
#endif
#ifdef DEUTSCH
#define HELP_LEARNPARAM			"Synapsen editieren"
#else
#define HELP_LEARNPARAM			"Edit Synapses"
#endif
#endif

#ifndef BATCH
/************************************************************************************************************************

	Application resources for BIOSIM

************************************************************************************************************************/

#define XmNprintCommand "printCommand"
#define XmCPrintCommand "PrintCommand"
#define XmNbackgroundColor "backgroundColor"
#define XmCBackgroundColor "BackgroundColor"
#define XmNrubberBandColor "rubberBandColor"
#define XmCRubberBandColor "RubberBandColor"
#define XmNneuronColor "neuronColor"
#define XmCNeuronColor "NeuronColor"
#define XmNneuronSelectionColor "neuronSelectionColor"
#define XmCNeuronSelectionColor "NeuronSelectionColor"
#define XmNneuronMovingColor "neuronMovingColor"
#define XmCNeuronMovingColor "NeuronMovingColor"
#define XmNinputNeuronColor "inputNeuronColor"
#define XmCInputNeuronColor "InputNeuronColor"
#define XmNoutputNeuronColor "outputNeuronColor"
#define XmCOutputNeuronColor "OutputNeuronColor"
#define XmNinOutNeuronColor "inOutNeuronColor"
#define XmCInOutNeuronColor "InOutNeuronColor"
#define XmNconnectionLineColor "connectionLineColor"
#define XmCConnectionLineColor "ConnectionLineColor"
#define XmNaxisColor "axisColor"
#define XmCAxisColor "AxisColor"
#define XmNgraphZeroLineColor "graphZeroLineColor"
#define XmCGraphZeroLineColor "GraphZeroLineColor"
#define XmNrestingLineColor "restingLineColor"
#define XmCRestingLineColor "RestingLineColor"
#define XmNspikeThresholdLineColor "spikeThresholdLineColor"
#define XmCSpikeThresholdLineColor "SpikeThresholdLineColor"
#define XmNsomaLineColor "somaLineColor"
#define XmCSomaLineColor "SomaLineColor"
#define XmNdendritLineColor "dendritLineColor"
#define XmCDendritLineColor "DendritLineColor"
#define XmNsynapticCurrentLineColor "synapticCurrentLineColor"
#define XmCSynapticCurrentLineColor "SynapticCurrentLineColor"
#define XmNtotalCurrentLineColor "totalCurrentLineColor"
#define XmCTotalCurrentLineColor "TotalCurrentLineColor"
#define XmNnLineColor "nLineColor"
#define XmCNLineColor "NLineColor"
#define XmNsodiumLineColor "sodiumLineColor"
#define XmCSodiumLineColor "SodiumLineColor"
#define XmNpotassiumLineColor "potassiumLineColor"
#define XmCPotassiumLineColor "PotassiumLineColor"
#define XmNcalciumLineColor "calciumLineColor"
#define XmCCalciumLineColor "CalciumLineColor"
#define XmNcalcium1LineColor "calcium1LineColor"
#define XmCCalcium1LineColor "Calcium1LineColor"
#define XmNcalcium2LineColor "calcium2LineColor"
#define XmCCalcium2LineColor "Calcium2LineColor"
#define XmNkcaLineColor "kcaLineColor"
#define XmCKcaLineColor "KcaLineColor"
#define XmNaLineColor "aLineColor"
#define XmCALineColor "ALineColor"
#define XmNirLineColor "irLineColor"
#define XmCIrLineColor "IrLineColor"
#define XmNpLineColor "pLineColor"
#define XmCPLineColor "PLineColor"
#define XmNtextColor "textColor"
#define XmCTextColor "TextColor"
#define XmNexcitationSynapseColor "excitationSynapseColor"
#define XmCExcitationSynapseColor "ExcitationSynapseColor"
#define XmNinhibitionSynapseColor "inhibitionSynapseColor"
#define XmCInhibitionSynapseColor "InhibitionSynapseColor"
#define XmNelectricalSynapseColor "electricalSynapseColor"
#define XmCElectricalSynapseColor "ElectricalSynapseColor"
#define XmNaxoSynapseColor "axoSynapseColor"
#define XmCAxoSynapseColor "AxoSynapseColor"
#define XmNotherSynapseColor "otherSynapseColor"
#define XmCOtherSynapseColor "OtherSynapseColor"
#endif

#ifndef _BIOSIM_H_
#define _BIOSIM_H_

/************************************************************************************************************************

	Structure definitions for BIOSIM

************************************************************************************************************************/

#ifndef BATCH
/******	Structure for application ressources	******/

typedef struct {
	String printCommand;
	String backgroundColor;
	String rubberBandColor;
	String neuronColor;
	String neuronSelectionColor;
	String neuronMovingColor;
	String inputNeuronColor;
	String outputNeuronColor;
	String inOutNeuronColor;
	String connectionLineColor;
	String axisColor;
	String graphZeroLineColor;
	String restingLineColor;
	String spikeThresholdLineColor;
	String somaLineColor;
	String dendritLineColor;
	String synapticCurrentLineColor;
	String totalCurrentLineColor;
	String nLineColor;
	String sodiumLineColor;
	String potassiumLineColor;
	String calciumLineColor;
	String calcium1LineColor;
	String calcium2LineColor;
	String kcaLineColor;
	String aLineColor;
	String irLineColor;
	String pLineColor;
	String textColor;
	String excitationSynapseColor;
	String inhibitionSynapseColor;
	String electricalSynapseColor;
	String axoSynapseColor;
	String otherSynapseColor;
} ResType, *ResTypePtr;
#endif

/******	Structure for process queue	******/

typedef double ProcessQueue;	/* reflects different delivery times and contains incoming synaptic conductance changes */

/******	Structure for graphs	******/

typedef struct _Graph
{
	struct  _Neuron	*neuron;		  /* Pointer to neuron which belongs the graph */
#ifndef BATCH
	Widget	formWidget;			  /* Form widget of the window containing the graph */
	Widget	dendritToggle;			  /* Dendrite Toggle of the window containing the graph */
	Widget  drawingArea;			  /* Drawing area of the window containing the graph */
#endif
	float	*soma;				  /* Array to store graph output for soma potential */
	float	*dend; 				  /* Array to store graph output for dendrit potential */
	float	*synC;  			  /* Array to store graph output for synaptic current */
	float	*totC;  			  /* Array to store graph output for total current */
	float	*iNA;	  			  /* Array to store graph output for sodium current */
	float	*iK;	  			  /* Array to store graph output for potassium current */
	float	*iCA;	  			  /* Array to store graph output for calcium current */
	float	*iCA1;  			  /* Array to store graph output for calcium1 current */
	float	*iCA2;  			  /* Array to store graph output for calcium2 current */
	float	*iKCA;  			  /* Array to store graph output for K(CA) current */
	float	*iA;	  			  /* Array to store graph output for A-current */
	float	*iIR;	  			  /* Array to store graph output for inward rectifier current */
	float	*iP;	  			  /* Array to store graph output for proctolin current */
	float	*iN;	  			  /* Array to store graph output for noise current */
	int	dendritSegment;			  /* number of dendrit segment to plot */
	unsigned int	graphSize;		  /* size of graph for determine reallocation */
#ifndef BATCH
	int	width;				  /* Width of graph */
	int	height;				  /* Height of graph */
	int	axis;				  /* flag for grid axis: 0 = no grid, 1 = x-axis, 2 = y-axis, 3 = both */
	int	gridType;			  /* flag for grid graph: 0 = no grid, 1 = dots, 2 = lines */
	char	msFormatString[10];		  /* format string for inscribing the x axis */
	char	mvFormatString[10];		  /* format string for inscribing the y axis */
#endif
	Boolean gateVariables;			  /* flag if gate variables should be drawn */
	Boolean	traceSoma;			  /* flag if soma potential should be traced */
	Boolean	traceDendrit;			  /* flag if dendrite potential should be traced */
	Boolean	traceSynapticCurrent;		  /* flag if synaptic current should be traced */
	Boolean	traceTotalCurrent;		  /* flag if total current should be traced */
	Boolean	traceNA;			  /* flag if sodium current should be traced */
	Boolean	traceK;				  /* flag if potassium current should be traced */
	Boolean	traceCA;			  /* flag if calcium current should be traced */
	Boolean	traceCA1;			  /* flag if first calcium current should be traced */
	Boolean	traceCA2;			  /* flag if second calcium current should be traced */
	Boolean	traceKCA;			  /* flag if calcium depndend potassium current should be traced */
	Boolean	traceA;				  /* flag if A-current should be traced */
	Boolean	traceIR;			  /* flag if inward rectifier current should be traced */
	Boolean	traceP;				  /* flag if proctolin current should be traced */
	Boolean	traceN;				  /* flag if noise current should be traced */
#ifndef BATCH
	Boolean	somaBound;			  /* flag if soma potential is out of bound */
	Boolean	dendBound;			  /* flag if dendrite potential is out of bound */
	Boolean	synCBound;			  /* flag if synaptic current is out of bound */
	Boolean	totCBound;			  /* flag if total current is out of bound */
	Boolean	iNABound;			  /* flag if sodium current is out of bound */
	Boolean	iKBound;			  /* flag if potassium current is out of bound */
	Boolean	iCABound;			  /* flag if calcium current should is out of bound */
	Boolean	iCA1Bound;			  /* flag if first calcium current is out of bound */
	Boolean	iCA2Bound;			  /* flag if second calcium current is out of bound */
	Boolean	iKCABound;			  /* flag if calcium depndend potassium current is out of bound */
	Boolean	iABound;			  /* flag if A-current is out of bound */
	Boolean	iIRBound;			  /* flag if inward rectifier current is out of bound */
	Boolean	iPBound;			  /* flag if proctolin current is out of bound */
	Boolean	iNBound;			  /* flag if noise current is out of bound */
#endif
	Boolean hasValues;			  /* flag that determines if the graph has values stored or not */
	double	asciiDataOutputInterval;	  /* time interval for writing out graph data during graph save */
	double	lastX;				  /* end of graph in ms */
#ifndef BATCH
	double	xScale;				  /* pixels per ms for x axis */
	double	yScale;				  /* pixels per mv for y axis */
	double	yStep;				  /* same as yScale, but no longer used */
	double	firstX;				  /* begin of graph in ms */
	double	highestY;			  /* highest value of y axis in mv */
	double	lowestY;			  /* lowest value of y axis in mv */
	double	timeInterval;			  /* time interval to inscribe x axis */
	double	voltInterval;			  /* volt interval to inscribe y axis */
	double	updateTime;			  /* time for which the graph is up to date */
#endif
} Graph;

typedef struct _SGraph
{
	struct  _Connection	*synapse;	  /* Pointer to synapse which belongs the graph */
#ifndef BATCH
	Widget	formWidget;			  /* Form widget of the window containing the graph */
	Widget  drawingArea;			  /* Drawing area of the window containing the graph */
#endif
	float	*pot;   			  /* Array to store graph output for synapse potential */
	float	*mem;   			  /* Array to store graph output for synaptic memory */
	float	*str;   			  /* Array to store graph output for synaptic current strength */
	float	*con;   			  /* Array to store graph output for synaptic current conductance */
	unsigned int	graphSize;		  /* size of graph for determine reallocation */
#ifndef BATCH
	int	width;				  /* Width of graph */
	int	height;				  /* Height of graph */
	int	axis;				  /* flag for grid axis: 0 = no grid, 1 = x-axis, 2 = y-axis, 3 = both */
	int	gridType;			  /* flag for grid graph: 0 = no grid, 1 = dots, 2 = lines */
	char	msFormatString[10];		  /* format string for inscribing the x axis */
	char	mvFormatString[10];		  /* format string for inscribing the y axis */
#endif
	Boolean	tracePot;			  /* flag if synapse potential should be traced */
	Boolean	traceMem;			  /* flag if synaptic memory should be traced */
	Boolean	traceStr;			  /* flag if synaptic current strength should be traced */
	Boolean	traceCon;			  /* flag if synaptic current conductance should be traced */
#ifndef BATCH
	Boolean	potBound;			  /* flag if synapse potential is out of bound */
	Boolean	memBound;			  /* flag if synaptic memeory is out of bound */
	Boolean	strBound;			  /* flag if synaptic current strength is out of bound */
	Boolean	conBound;			  /* flag if synaptic current conductance is out of bound */
#endif
	Boolean hasValues;			  /* flag that determines if the graph has values stored or not */
	double	asciiDataOutputInterval;	  /* time interval for writing out graph data during graph save */
	double	lastX;				  /* end of graph in ms */
#ifndef BATCH
	double	xScale;				  /* pixels per ms for x axis */
	double	yScale;				  /* pixels per mv for y axis */
	double	yStep;				  /* same as yScale, but no longer used */
	double	firstX;				  /* begin of graph in ms */
	double	highestY;			  /* highest value of y axis in mv */
	double	lowestY;			  /* lowest value of y axis in mv */
	double	timeInterval;			  /* time interval to inscribe x axis */
	double	voltInterval;			  /* volt interval to inscribe y axis */
	double	updateTime;			  /* time for which the graph is up to date */
#endif
} SGraph;

/******	Structure for input currents used by neurons	******/

typedef struct _InputCurrent
{
	double *current;				/* array of input currents */
	double *starttime;				/* array of corresponding start times of input currents */
	int index;					/* index to starttime and current array */
	unsigned int numEntries;			/* number of entries */
} InputCurrent;

/******	Structure for neurons	******/

typedef struct _Neuron {
	struct _Neuron	*next;				/* pointer to next neuron */
	struct _Neuron	*previous;			/* pointer to previous neuron */
	struct _Connection *connectionList;		/* pointer to a list of outgoing connections */
	struct _ChemicalPool *poolList;			/* pointer to a list of incoming connections */
	int		neuronId;			/* neuron identifier */
#ifndef BATCH
	unsigned int	neuronSize;			/* number of bytes allocated for the neuron */
	unsigned int	inpCurSize;			/* number of bytes allocated for the input currents to the neuron */
#endif
	Graph		*graph;				/* graph for neuron */
#ifndef BATCH
	Position	x, y;				/* position in window */
	Position	xOriginal, yOriginal;		/* original position in window */
	Region		region;				/* region for the neuron */
#endif
	enum 	typ	{ 
		spiker, nonspiker 	} neuronType;	/* type of neuron */
	Boolean		selected;			/* flag that indicats whether the neuron is selected or not */
#ifndef BATCH
	Boolean		isInputNeuron;			/* flag that indicats if neuron is shown as a input neuron */
	Boolean		isOutputNeuron;			/* flag that indicats if neuron is shown as a output neuron */
#endif
	double neuronParameters[NEURON_LENGTH];		/* neuron parameters */
	unsigned long int spikeCnt;			/* spike counter for spiking neurons */
	double onoffSpikeTimer;				/* timer for measuring a spike for on/off model */
	Boolean onoffSpikeOccuring;			/* flag if a spike has been occured for on/off model */
	Boolean noise;					/* flag if noise should be added to the neuron */
	int neuronStems;				/* number of dendritic stems of neuron */
	int neuronRegions;				/* number of dendritic regions of neuron */
	int usedDendSCLocation;				/* currently used dendritic location during computation */
	int usedDendSCStem;				/* currently used stem during computation */
	double	pseudoSomaCurrent;			/* pseudo soma current since change of network model */
	double	pseudoSCStart;				/* pseudo soma start time since change of network model */
	double	pseudoSCEnd;				/* pseudo soma end time since change of network model */
	double  synapticCurrent;			/* synaptical induced current into neuron */
	double  totalCurrent;				/* total current in the neuron */
	double	M;					/* HH, SWIM and GB parameter for sodium channel activation */
	double	H;					/* HH, SWIM and GB parameter for sodium channel inactivation */
	double	N;					/* HH, SWIM and GB parameter for potassium channel activation */
	double  A;					/* GB parameter for A-channel activation */
	double  B1;					/* GB parameter for A-channel inactivation */
	double  B2;					/* GB parameter for A-channel inactivation */
	double  C;					/* SWIM and GB parameter for first calcium channel activation */
	double  D;					/* GB parameter for first calcium channel inactivation */
	double  E;					/* GB parameter for second calcium channel activation */
	double  P;					/* GB parameter for calcium dependend potassium channel act. */
	double  Q;					/* GB parameter for calcium dependend potassium channel inact. */
	double  R;					/* GB parameter for inward rectifier activation */
	double	S;					/* GB parameter for proctolin activation */
	double	iNA;					/* sodium current */
	double	iK;					/* potassium current */
	double	iCA;					/* calcium current */
	double  iA;					/* A-channel current in GB-model */
	double  iCA1;					/* first calcium current in GB-model */
	double  iCA2;					/* second calcium current in GB-model */
	double  iKCA;					/* calcium dependend potassium current in GB-model */
	double  iIR;					/* inward rectifier current in GB-model */
	double	iP;					/* proctolin current in GB-model */
	double	iNoise;					/* noise current */
	double  conc_ca;				/* calcium concentration in GB-model */
	double  ca_e;					/* calcium equilibrium potential in GB-model */
	double  gamma;					/* rate coefficient in GB-model */
	double	somaCurrent;				/* soma current */
	double	somaSCStart;				/* start time of soma current */
	double	somaSCEnd;				/* end time of soma current */
	double	dendCurrent;				/* dendritic current */
	double	dendSCStart;				/* start time of dendritic current */
	double	dendSCEnd;				/* end time of dendritic current */
	double	*currentChemical;			/* chemical current for each compartment */
	double	*currentElectrical;			/* electrical current for each compartment */
	double	*neuronPotentials;			/* potentials for each compartment */
	InputCurrent *inputCurrent;			/* each compartment can have its own input current */
	int	dendSCLocation;				/* location of dendritic current */
	int	dendSCStem;				/* stem of dendritic current */
	double	SLmSG_const;				/* constant for faster computation */
	double	SGpDSmDG_const;				/* constant for faster computation */
	double	DLmDG_const;				/* constant for faster computation */
	double	SGdSC_const;				/* constant for faster computation */
	double	SGpDSmDGdSC_const;			/* constant for faster computation */
	double	SWIM_NAEmNAG_const;			/* constant for faster computation */
	double	SWIM_KEmKG_const;			/* constant for faster computation */
	double	SWIM_CAEmCAG_const;			/* constant for faster computation */
	double	HH_NAEmNAG_const;			/* constant for faster computation */
	double	HH_KEmKG_const;				/* constant for faster computation */
	double	GB_NAEmNAG_const;			/* constant for faster computation */
	double	GB_KEmKG_const;				/* constant for faster computation */
	double  GB_AEmAG_const;				/* constant for faster computation */
	double  GB_KCAEmKCAG_const;			/* constant for faster computation */
	double  GB_IREmIRG_const;			/* constant for faster computation */
	double  GB_PEmPG_const;				/* constant for faster computation */
} Neuron;

/******	Structure for selection list of neurons	******/

typedef struct _SelectionListNeurons {
	Neuron	*neuronPtr;				/* pointer to a neuron */
	struct _SelectionListNeurons *next;		/* pointer to next entry in selection list */
} SelectionListNeurons;

/******	Structure for synapses	******/

typedef struct _Synapse
{
	int	ionType;				/* 1 = sodium, 2 = potassium, 3 = chemical, 4 = electrical */
	double	equilPotential;				/* equilibrium potential of synapse */
} Synapse;

/******	Structure for chemical senders	******/

typedef struct _ChemicalSenderList
{
	struct _ChemicalSenderList	*prevSender;	/* predecessor */
	struct _ChemicalSenderList	*nextSender;	/* successor */
	struct _Connection		*connection;	/* pointer to connection */
} ChemicalSenderList;

/******	Structure for chemical pools	******/

typedef struct _ChemicalPool
{
	struct _ChemicalPool	*prevPool;		/* predecessor */
	struct _ChemicalPool	*nextPool;		/* successor */
	ChemicalSenderList	*senderList;		/* pointer to all senders at the same loc. and same syn. type */
	ProcessQueue		*processQueue;		/* contains time dependend conductance changes */
	double	longestDeliveryTime;			/* longest synaptic transmission time per neuron */
	double	totalConductance;			/* sum of all synaptic conductances in the pool */
	int	activeSynapses;				/* number of active synapses during competitive learning */
	int	typeOfConnection;			/* number to array of synaptic types ( 0 = sodium, ...) */
	int	currentQueuePlace;			/* points into process queue */
	int	locationOfPool;				/* compartment on which the pool is located */
	int	LTTdSS_const;				/* constant for faster computation */
} ChemicalPool;

/******	Structure for axo-axonic senders	******/

typedef struct _AxoAxonicSenderList
{
	struct _AxoAxonicSenderList	*prevSender;	/* predecessor */
	struct _AxoAxonicSenderList	*nextSender;	/* successor */
	struct _Connection		*connection;	/* pointer to connection */
	ProcessQueue            	*processQueue;  /* contains time dependend conductance changes or strength val. */
	int     			currentQueuePlace; /* points into process queue */
} AxoAxonicSenderList;

/******	Structure for connections	******/

typedef struct _Connection
{
	struct _Connection *prevConnection;		/* predecessor */
	struct _Connection *nextConnection;		/* successor */
	ChemicalPool *connectionPool;			/* pointer to connection pool */
	ChemicalSenderList *poolSenderEntry;		/* points to the sender in the receiver pool */
	AxoAxonicSenderList *axoSenderList;		/* points to the sender in the axo-axonic receiver pool */
	SGraph	*graph;					/* graph of synapse */
#ifndef BATCH
	Region	targetRegion;				/* graphical region of synapse */
#endif
	Neuron	*targetNeuron;				/* pointer to target neuron of connection */
	Neuron	*parentNeuron;				/* pointer to parent neuron of connection */
	struct _Connection *targetSynapse;		/* target synapse in case of axo-axonic connection */
	int synapseID;					/* synapse identification number */
	int targetCode;					/* neuron id of target neuron */
	int targetArea;					/* compartment number of target neuron */
	int axoSynapseNumber;				/* 0 = not axo-axonic, > 0 = number of axo-axonic connection */
	int numberOfAxoSynapses;			/* number of axo-axonic synapses to this synapse */
#ifndef BATCH
	int connectionX_N,
	    connectionX_4,
	    connectionX_2,
	    connectionX_1;				/* x-coordinates of synapse in respect of network model */
	int connectionY_N,
	    connectionY_4,
	    connectionY_2,
	    connectionY_1;				/* y-coordinates of synapse in respect of network model */
	int targetX_N,
	    targetX_4,
	    targetX_2,
	    targetX_1;					/* x-coordinate of target neuron in respect of network model */
	int targetY_N,
	    targetY_4,
	    targetY_2,
	    targetY_1;					/* y-coordinate of target neuron in respect of network model */
	int senderX_N,
	    senderX_4,
	    senderX_2,
	    senderX_1;					/* x-coordinate of sender neuron in respect of network model */
	int senderY_N,
	    senderY_4,
	    senderY_2,
	    senderY_1;					/* y-coordinate of sender neuron in respect of network model */
#endif
	int synapticType;				/* number to array of synaptic types */
	int hebbCnt;					/* counter for hebb learning stimuli */
	int stmCnt;					/* counter for short term plasticity stimuli */
	int condCnt;					/* counter for conditioned learning stimuli */
	int trainingFlag;				/* flag indicating the training states of the synapse */
	Boolean	trainingFlagSTP;			/* is synapse in a trained state for STP, 0 = no, 1 = yes */
	Boolean	trainingFlagHebb;			/* is synapse in a trained state for Hebb, 0 = no, 1 = yes */
	Boolean	trainingFlagCond;			/* is synapse in a trained state for Cond, 0 = no, 1 = yes */
	Boolean	shortTermLearningCondition;		/* condition for short term learning */
	Boolean	conditioningLearningCondition;		/* condition for conditioning learning */
	Boolean	hebbLearningCondition;			/* condition for hebbian learning */
	Boolean competitiveLearningCondition;		/* condition for competitive learning */
	Boolean competitiveDone;			/* TRUE, if competitive was done in this pool */
	Boolean selected;				/* flag if synapse is selected */
	double	potential;				/* synaptic potential */
	double	currentConductance;			/* current conductance of synapse */
	double	initialConductance;			/* initial conductance of synapse */
	double	conductance;				/* conductance of synapse can be modified by conditioning */
	double	maxConductance;				/* maximum conductance of synapse */
	double	deliveryTime;				/* transmission time of synapse */
	double	transmitterReleaseThreshold;		/* threshold for transmitter release in synapses */
	double	transmitterSatiate;			/* point where transmitter release goes into satiation */
	double	currentStrength;			/* current strength of axo-axonic connection */
	double	currentInhibitionFactor;		/* current inhibition factor of axo-axonic connection */
	double	initialInhibitionFactor;		/* initial inhibition factor of axo-axonic connection */
	double	maxInhibitionFactor;			/* maximum inhibition factor of axo-axonic connection */
	double	slope;					/* slope of learning curve */
	double	shortTermMemory;			/* short term memory duration */
	double	longTermThreshold;			/* threshold between short term and long term forgetting */
	double 	longTermMinimum;			/* minimum of long term forgetting */
	double	hebbIntervall;				/* duration of hebb intervall */
	double	hebbLearnFactor;			/* hebb learn factor */
	double	competitiveLearnFactor;			/* competitive learn factor */
	double	shortTermPlasticityLearnFactor;		/* short term plasticity learn factor */
	double	conditioningLearnFactor;		/* conditioning learn factor */
	double	synapticMemory;				/* total memory of synapse */
	double	synapticMemorySTP;			/* memory of synapse for short term plasticity */
	double	synapticMemoryHebb;			/* memory of synapse for Hebb learning */
	double	synapticMemoryCond;			/* memory of synapse for conditioned learning */
	double	hebbIntervallCounter;			/* interval counter for hebbian learning */
	double	hebbTimer;				/* timer for hebbian learning */
	double	LEXP_const;				/* constant for faster computation */
	double	SFCONST_const;				/* constant for faster computation */
	double	LFCONST_const;				/* constant for faster computation */
	double	TSmTRT_const;				/* constant for faster computation */
	int	TTdSS_const;				/* constant for faster computation */
} Connection;

/******	Structure for selection list of synapses	******/

typedef struct _SelectionListSynapses {
	Connection	*synapsePtr;			/* pointer to a synapse */
	struct _SelectionListSynapses *next;		/* pointer to next entry in selection list */
} SelectionListSynapses;

/******	Structure for simulations parameters	******/

typedef struct _SimulationParams
{
	double	length;					/* length of simulation */
	double	step;					/* time for one simulation step */
} SimulationParams;

/******	Structure for program states ******/

typedef struct _ProgramStatesData
{
	unsigned	newRead: 1;			/* flag for reading .sim, .par or .con files */
	unsigned	eofReached: 1;			/* flag for reading .sim, .par or .con files */
} ProgramStates;

/******	Structure for ascii separation characters	******/

typedef struct _AsciiDefaultsData
{
	char	separatorChar;				/* separation character in graph output files */
	char	decimalChar;				/* decimal character in graph output files */
} AsciiDefaults;

#endif

/************************************************************************************************************************

	Global variables defined in biosim.c

************************************************************************************************************************/

#ifndef BATCH
extern char *colorList[];				/* list of colors used for neuron activity */
extern unsigned long backgroundColor;			/* color for BIOSIM */
extern unsigned long rubberBandColor;			/* color for BIOSIM */
extern unsigned long neuronColor;			/* color for BIOSIM */
extern unsigned long neuronSelectionColor;		/* color for BIOSIM */
extern unsigned long neuronMovingColor;			/* color for BIOSIM */
extern unsigned long inputNeuronColor;			/* color for BIOSIM */
extern unsigned long outputNeuronColor;			/* color for BIOSIM */
extern unsigned long inOutNeuronColor;			/* color for BIOSIM */
extern unsigned long connectionLineColor;		/* color for BIOSIM */
extern unsigned long axisColor;				/* color for BIOSIM */
extern unsigned long graphZeroLineColor;		/* color for BIOSIM */
extern unsigned long restingLineColor;			/* color for BIOSIM */
extern unsigned long spikeThresholdLineColor;		/* color for BIOSIM */
extern unsigned long somaLineColor;			/* color for BIOSIM */
extern unsigned long dendritLineColor;			/* color for BIOSIM */
extern unsigned long synapticCurrentLineColor;		/* color for BIOSIM */
extern unsigned long totalCurrentLineColor;		/* color for BIOSIM */
extern unsigned long nLineColor;			/* color for BIOSIM */
extern unsigned long sodiumLineColor;			/* color for BIOSIM */
extern unsigned long potassiumLineColor;		/* color for BIOSIM */
extern unsigned long calciumLineColor;			/* color for BIOSIM */
extern unsigned long calcium1LineColor;			/* color for BIOSIM */
extern unsigned long calcium2LineColor;			/* color for BIOSIM */
extern unsigned long kcaLineColor;			/* color for BIOSIM */
extern unsigned long aLineColor;			/* color for BIOSIM */
extern unsigned long irLineColor;			/* color for BIOSIM */
extern unsigned long pLineColor;			/* color for BIOSIM */
extern unsigned long textColor;				/* color for BIOSIM */
extern unsigned long excitationSynapseColor;		/* color for BIOSIM */
extern unsigned long inhibitionSynapseColor;		/* color for BIOSIM */
extern unsigned long electricalSynapseColor;		/* color for BIOSIM */
extern unsigned long axoSynapseColor;			/* color for BIOSIM */
extern unsigned long otherSynapseColor;			/* color for BIOSIM */
extern unsigned long selectedToggleButtonColor;		/* color for BIOSIM */
extern unsigned long activityColors[];			/* colors used to show activity of neurons */
extern ResType	resData;				/* variable for application ressources */
extern Cursor cursor;					/* cursor in window */
extern Display *disp;					/* display of the application */
#endif
extern int commandMode;					/* 0 = interactiv, >0 = command driven */
#ifndef BATCH
extern XmStringCharSet defaultFont; 			/* default font, defined in the resource file */
extern XFontStruct *font;				/* font to be used */
extern char fontname[];					/* font name */
extern char geometry[];					/* geometry of biosim */
extern char helpServerPath[];				/* path name to help server */
extern char biosimDir[];				/* directory where biosim resides */
extern char biosimDataDir[];				/* data directory of biosim */
extern char unnamedFile[];				/* name of default file name */
extern char commandFile[];				/* path for command input file */
extern char selectedObjectsFile[];			/* path for selection output file */
extern char outputFile[];				/* path for output graph file */
extern char onoffChannelFile[];				/* onoff channels parameter file */
extern char swimChannelFile[];				/* swim channels parameter file */
extern char hhChannelFile[];				/* hodgkin-huxley channels parameter file */
extern char golowaschChannelFile[];			/* golowasch-buchholz channels parameter file */
extern char helpServerPath[];				/* path to help server */
#endif
extern char inputCurrentsFile[];			/* path for input currents file */
extern char currentFilename[];				/* current file name for save operations */
extern char outputFilename[];				/* name of output file for graphic data */
#ifndef BATCH
extern char programName[];				/* full name of biosim */
extern char simulationFileDirectory[];			/* path name for simulation files */
extern char parameterFileDirectory[];			/* path name for parameter files */
extern char configurationFileDirectory[];		/* path name for configuration files */
extern char neuronParameterDirectory[];			/* path name for neuron parameter files */
extern char graphOutputDirectory[];			/* path name for graph output files */
extern char deleteFileDirectory[];			/* path name for files to be deleted */
extern char otherFileDirectory[];			/* path name for other files */
extern Widget applShell;				/* application shell */
extern Widget mainWindow;				/* main window */
extern Widget drawingArea;				/* drawing area of the main window */
#endif

/************************************************************************************************************************

	Global variables defined in graphic.c

************************************************************************************************************************/

extern Neuron *neuronListHead;				/* start pointer of double chained neuron list */
extern Neuron *neuronListTail;				/* end pointer of double chained neuron list */
extern SelectionListNeurons *selectionListNeuronsHead;	/* start pointer of double chained neuron selection list */
extern SelectionListNeurons *selectionListNeuronsTail;	/* end pointer of double chained neuron selection list */
extern SelectionListSynapses *selectionListSynapsesHead;/* start pointer of double chained synapse selection list */
extern SelectionListSynapses *selectionListSynapsesTail;/* end pointer of double chained synapse selection list */
extern Neuron *currentNeuron;				/* current neuron */
extern Connection *currentSynapse;			/* current synapse */
extern int selectionListNeuronsCounter;			/* number of selected neurons in drawing area */
extern int selectionListSynapsesCounter;		/* number of selected synapses in drawing area */
extern int neuronNumber;				/* number of neurons in net */
extern int synapseNumber;				/* number of synapses in net */
extern int axoSynapsesNumber;				/* number of axo-axonic synapses in the net */
extern int numTypes;					/* number of synaptic types */
extern Synapse *synapseTypes;				/* array of synaptic types */
#ifndef BATCH
extern unsigned int neuronDiameter;			/* diameter of the graphic circle for a neuron */
extern unsigned int synapseDiameter;			/* diameter of the graphic circle for a synapse */
extern int DendriteWidth;				/* dendrite width */
extern int DendriteLength;				/* dendrite length */
extern int StemLength;					/* length of dendritic stem for 1-Point model */
extern int StemWidth;					/* width of dendrite for 1-Point model */
extern int OwnConnectionRadius;				/* radius for connection with itself */
extern double scalingFactor;				/* scaling factor for graphic objects */
#endif

/************************************************************************************************************************

	Global variables defined in calculations.c

************************************************************************************************************************/

#ifndef BATCH
extern double neuronRange;				/* working range of a neuron */
extern double topVoltage;				/* top voltage of a neuron */
extern double bottomVoltage;				/* bottom voltage of a neuron */
extern int neuronActivityInterval;			/* neuron activity interval */
extern Boolean showActivityColors;			/* option to give neurons colors durings simulation */
#endif
extern double L;					/* actual simulation time in s */
extern Boolean noiseDefined;				/* indicates if noise should be added to neurons */
extern Boolean stochasticInitialPotentials;		/* indicates if initial potentials should be stochastic */
extern Boolean stochasticSynapticStrengths;		/* indicates if synaptic strengths should be stochastic */
extern Boolean stochasticTransmissionTimes;		/* indicates if syn. transmission times should be stochastic */
extern Boolean graphSim;				/* indicates if a simulation has been run */
extern Boolean variables_initialized;			/* indicates if simulation variables are initialized */
extern Boolean resetLearningParameter;			/* indicates if learning parameter should be initialized */
extern SimulationParams simulation;			/* simulation params during simulation */
extern int integrationMethod;				/* integration method used in calculations
							   1 = Euler, 2 = Exp. Euler, 3 = Trapezoid, 4 = Runge-Kutta */
extern int simulationTime;  				/* time during simulation process */
extern int networkmodel;				/* 1 = ONOFF, 2 = SWIM, 3 = Hodgkin-Huxley, 4 = Golowasch-Buchholz */
extern Boolean complexModel;				/* TRUE if network model type is complex */
extern int networktype;					/* 1 = 1 Point, 2 = 2 Point, 3 = 4 Point, 4 = n Point */
extern int longTermAdaptiveFactor;			/* adaption factor for long term learning */

/************************************************************************************************************************

	Global variables defined in menu.c

************************************************************************************************************************/

#ifndef BATCH
extern XtWorkProcId workProcId;				/* working procedure id for run_simulation */
#endif
extern double noiseIntensity;				/* noise intensity */
extern double noiseStrength;				/* noise strength in nA */
extern double noiseDecay;				/* noise decay rate in ms */
extern double stdInitialPotential;			/* initial potential for standard deviation */
extern double stdInitialPotentialStd;			/* standard deviation of initial potential */
extern double stdSynapticStrength;			/* synaptic strength for standard deviation */
extern double stdSynapticStrengthStd;			/* standard deviation of synaptic strength */
extern double stdAxoAxonicStrength;			/* axo-axonic strength for standard deviation */
extern double stdAxoAxonicStrengthStd;			/* standard deviation of axo-axonic strength */
extern double stdSynapticTT;				/* synaptic transmission time for standard deviation */
extern double stdSynapticTTStd;				/* standard deviation of synaptic transmission time */
extern int curNumOfDendSegs;				/* current network model type */
extern Boolean autoReset;				/* reset network every time a new simulation is started */
extern Boolean showProcessingInfo;			/* show information about processing time of simulation */
extern Boolean hebbLearning;				/* flag for hebbian learning */
extern Boolean competitiveLearning;			/* flag for competitive learning */
extern Boolean conditionedLearning;			/* flag for conditioned learning */
extern Boolean plasticityLearning;			/* flag for short term plasticity learning */
#ifndef BATCH
extern Boolean boundCheckingOn;				/* flag if bound checking should be done during graph output */
extern Boolean manualWindowUpdate;			/* flag if manual window refresh should be done */
extern Boolean editNetwork;				/* flag that indicates whether network is editable or not */
extern Boolean showOutputImmediately;			/* flag for showing output immediately during simulation */
extern Boolean fileSaved;				/* flag indicating file saved status */
extern Boolean showSimulationInfo;			/* show information during simulation */
extern Widget overBiosimDialog;				/* dialog box for over biosim */
extern Widget simInfButton;				/* button in option menu */
extern Widget timeMessureButton;			/* button in option menu */
extern Widget graphDisplayButton;			/* button in option menu */
extern Widget autoResetButton;				/* button in option menu */
extern Widget neuronActivityColorsButton;		/* button in option menu */
extern Widget outputWatchButton;			/* button in option menu */
extern Widget manualUpdateButton;			/* button in option menu */
extern Widget hebbLearnButton;				/* button in learn menu */
extern Widget competitiveLearnButton;			/* button in learn menu */
extern Widget conditionedLearnButton;			/* button in learn menu */
extern Widget plasticityLearnButton;			/* button in learn menu */
extern Widget noiseButton;				/* button in option menu */
extern Widget initialPotentialsButton;			/* button in stochastic distribution menu */
extern Widget synapticStrengthButton;			/* button in stochastic distribution menu */
extern Widget synapticTransmissionTimeButton;		/* button in stochastic distribution menu */
extern Widget eulerButton;				/* button in integration methods menu */
extern Widget expEulerButton;				/* button in integration methods menu */
extern Widget trapezoidButton;				/* button in integration methods menu */
extern Widget rungeKuttaButton;				/* button in integration methods menu */
extern Widget networkOnOffButton;			/* button in network model menu */ 
extern Widget networkSWIMButton;			/* button in network model menu */ 
extern Widget networkHHButton;				/* button in network model menu */ 
extern Widget networkGBButton;				/* button in network model menu */ 
extern Widget networkComplexButton;			/* button in network model menu */ 
extern Widget networkType1Button;			/* button in network model menu */ 
extern Widget networkType2Button;			/* button in network model menu */ 
extern Widget networkType4Button;			/* button in network model menu */ 
extern Widget networkTypeNButton;			/* button in network model menu */ 
#endif

/************************************************************************************************************************

	Global variables defined in menu2.c

************************************************************************************************************************/

#ifndef BATCH
extern Widget onOffParamsBB;                          	/* mask for edit ONOFF channels */
extern Widget swimParamsBB;                           	/* mask for edit SWIM channels */
extern Widget hhParamsBB;                             	/* mask for edit HH channels */
extern Widget gbParamsBB;                             	/* mask for edit GB channels */
extern Widget editChannelsBB;				/* mask for edit channels */
extern Widget editNeuronBB;				/* mask for edit global neuron parameters */
extern Widget editCurrentBB;				/* mask for edit currents */
extern Widget addSynapseBB;				/* mask for add synapse */
extern Widget editSynapseBB;				/* mask for edit synapse */
extern Widget editSimParsBB;				/* mask for edit simulation parameters */
extern Widget newSTypeBB;				/* mask for add new synaptic type */
extern Widget simInfBB;					/* mask for show simulation information */
extern Widget newGraphForm;				/* mask for new graph window for neurons */
extern Widget newGraphSForm;				/* mask for new graph window for synapses */
extern Widget popupMenu;				/* popup menu in drawing area */
extern Widget deleteObjectsButton;			/* button in popup menu in drawing area */
extern Widget connectObjectsButton;			/* button in popup menu in drawing area */
extern Widget editChannelButton;			/* button in popup menu in drawing area */
extern Widget editNeuronButton;				/* button in popup menu in drawing area */
extern Widget editCurrentButton;			/* button in popup menu in drawing area */
extern Widget informationObjectButton;			/* button in popup menu in drawing area */
extern Widget editConnectionButton;			/* button in popup menu in drawing area */
extern Widget editAxoConnectionButton;			/* button in popup menu in drawing area */
extern Widget attachWindowButton;			/* button in popup menu in drawing area */
extern Widget attachWindowSButton;			/* button in popup menu in drawing area */
extern Widget selectNeuronsButton;			/* button in popup menu in drawing area */
extern Widget selectSynapsesButton;			/* button in popup menu in drawing area */
extern Widget selectAxoSynapsesButton;			/* button in popup menu in drawing area */
extern Widget selectGraphsButton;			/* button in popup menu in drawing area */
extern Widget showTopologyButton;			/* button in popup menu in drawing area */
extern Widget showInputNeuronsButton;			/* button in popup menu in drawing area */
extern Widget showOutputNeuronsButton;			/* button in popup menu in drawing area */
extern Widget showInputOutputNeuronsButton;		/* button in popup menu in drawing area */
extern Widget showNormalStateButton;			/* button in popup menu in drawing area */
extern Widget specifyObjectsButton;			/* button in popup menu in drawing area */
extern Widget addSynapseLearnButton;			/* button in addSynapse dialog box */
extern Widget simInfStopButton;				/* button in show simulation information */
extern Widget simInfCompletedText;			/* text field in show simulation information */
extern Widget simInfLengthText;				/* text field in show simulation information */
extern Widget editSimParsSimLenText;			/* text field in edit simulation parameter */
extern Widget editSimParsStepSizeText;			/* text field in edit simulation parameter */
extern Widget neuronActivityIntervalText;		/* text field in edit simulation parameter */
extern Widget editSimParsSimLenScale;			/* scale field in edit simulation parameter */
extern Widget editSimParsStepSizeScale;			/* scale field in edit simulation parameter */
extern Widget neuronActivityIntervalScale;		/* scale field in edit simulation parameter */
extern Widget addSynapseTargetScale;                    /* scale widget in add synapse mask */
extern Widget editSynapseLocationScale;                 /* scale widget in edit synapse mask */
extern Widget editCurrentLocationScale;                 /* scale widget in edit current mask */
extern Widget addSynapseTargetText;                     /* text widget in add synapse mask */
extern Widget editSynapseLocationText;                  /* text widget in edit synapse mask */
extern Widget editCurrentLocationText;                  /* text widget in edit current mask */
extern Widget editCurrentSomaEndScale;                  /* scale widget */
extern Widget editCurrentDendritEndScale;               /* scale widget */
extern Widget editCurrentSomaStartScale;                /* scale widget */
extern Widget editCurrentDendritStartScale;             /* scale widget */
#endif

/************************************************************************************************************************

	Global variables defined in menu3.c

************************************************************************************************************************/

extern Graph *currentGraph;				/* current neuron graph for graph switching */
extern SGraph *currentGraphS;	                    	/* current synapse graph for graph switching */
extern int graphNumber;					/* number of graphs in net */
extern int graphNumberS;				/* number of synaptic graphs in net */
extern int graphNumberAS;				/* number of axo-synaptic graphs in net */
#ifndef BATCH
extern Boolean zoomAllWindows, zoomAllSWindows;		/* flag for zooming all windows */
extern Boolean scaleAllWindows, scaleAllSWindows;	/* flag for scaling all windows */
extern Widget newGraphOptionsGridXToggle;		/* toggle button in graph window */
extern Widget newGraphOptionsGridYToggle;		/* toggle button in graph window */
extern Widget newGraphOptionsGridDotsToggle;		/* toggle button in graph window */
extern Widget newGraphOptionsGridLinesToggle;		/* toggle button in graph window */
extern Widget newGraphParameterSomaToggle;		/* toggle button in graph window */
extern Widget newGraphParameterDendritToggle;		/* toggle button in graph window */
extern Widget newGraphParameterSynapticCurrentToggle;	/* toggle button in graph window */
extern Widget newGraphParameterTotalCurrentToggle;	/* toggle button in graph window */
extern Widget newGraphParameterNToggle;			/* toggle button in graph window */
extern Widget newGraphParameterNAToggle;		/* toggle button in graph window */
extern Widget newGraphParameterKToggle;			/* toggle button in graph window */
extern Widget newGraphParameterCAToggle;		/* toggle button in graph window */
extern Widget newGraphParameterCA1Toggle;		/* toggle button in graph window */
extern Widget newGraphParameterCA2Toggle;		/* toggle button in graph window */
extern Widget newGraphParameterKCAToggle;		/* toggle button in graph window */
extern Widget newGraphParameterAToggle;			/* toggle button in graph window */
extern Widget newGraphParameterIRToggle;		/* toggle button in graph window */
extern Widget newGraphParameterPToggle;			/* toggle button in graph window */
extern Widget newGraphSOptionsGridXToggle;		/* toggle button in graph window */
extern Widget newGraphSOptionsGridYToggle;		/* toggle button in graph window */
extern Widget newGraphSOptionsGridDotsToggle;		/* toggle button in graph window */
extern Widget newGraphSOptionsGridLinesToggle;		/* toggle button in graph window */
extern Widget newGraphSParameterPotToggle;		/* toggle button in graph window */
extern Widget newGraphSParameterMemToggle;		/* toggle button in graph window */
extern Widget newGraphSParameterStrToggle;		/* toggle button in graph window */
extern Widget newGraphSParameterConToggle;		/* toggle button in graph window */
extern Widget newGraphYScalingEndScale;                	/* scale widget */
extern Widget newGraphXScalingEndScale;                	/* scale widget */
extern Widget newGraphYScalingSEndScale;               	/* scale widget */
extern Widget newGraphXScalingSEndScale;               	/* scale widget */
extern Widget newGraphYScalingStartScale;               /* scale widget */
extern Widget newGraphXScalingStartScale;               /* scale widget */
extern Widget newGraphYScalingSStartScale;              /* scale widget */
extern Widget newGraphXScalingSStartScale;              /* scale widget */
extern Widget newGraphYScalingStartText;		/* text widget */
extern Widget newGraphYScalingEndText;			/* text widget */
extern Widget newGraphXScalingStartText;		/* text widget */
extern Widget newGraphXScalingEndText;			/* text widget */
extern Widget newGraphYScalingSStartText;		/* text widget */
extern Widget newGraphYScalingSEndText;			/* text widget */
extern Widget newGraphXScalingSStartText;		/* text widget */
extern Widget newGraphXScalingSEndText;			/* text widget */
#endif

/************************************************************************************************************************

	Global variables defined in menu4.c

************************************************************************************************************************/

#ifndef BATCH
extern Widget axoSynapseBB;				/* bulletin board in axo-axonic dialog */
extern Widget learnParamBB;                             /* bulletin board in learn parameter dialog */
extern Widget axoSynapseTTText;				/* text widget in axo-axonic dialog */
extern Widget axoSynapseTTScale;			/* scale widget in axo-axonic dialog */
extern Widget axoSynapseStrText;			/* text widget in axo-axonic dialog */
extern Widget axoSynapseStrScale;			/* scale widget in axo-axonic dialog */
extern Widget axoSynapseMaxStrText;			/* text widget in axo-axonic dialog */
extern Widget axoSynapseMaxStrScale;			/* scale widget in axo-axonic dialog */
extern Widget axoSynapseTRTText;               	        /* text widget in axo-axonic dialog */
extern Widget axoSynapseTRTScale;              	        /* scale widget in axo-axonic dialog */
extern Widget axoSynapseTRSText;               	        /* text widget in axo-axonic dialog */
extern Widget axoSynapseTRSScale;              	        /* scale widget in axo-axonic dialog */
extern Widget axoSynapseLearnButton;                    /* learn parameter button in axo-axonic dialog */
#endif

/************************************************************************************************************************

	Global variables defined in fileopen.c

************************************************************************************************************************/

extern ProgramStates programStates;			/* program states for reading files */

/************************************************************************************************************************

	Global variables defined in filesave.c

************************************************************************************************************************/

extern AsciiDefaults asciiDefaults;			/* default values for graph file save */

/************************************************************************************************************************

	Global variables defined in utilities.c

************************************************************************************************************************/

extern Boolean errorOccured;				/* flag, that indicates that an error has been occured */
