/************************************************************************************************************************

	PROGRAM:	BIOSIM

	DATE:		17/10/1990

	LAST UPDATE:	28/11/1993

	VERSION:	4.3

	FILENAME:	biosim.c

	PURPOSE:	this file contains the main function of the program

	FUNCTIONS:	in order of appearance

	read_commands	- looks every second if a command file is present and execute the commands
			  the command file is generated from the supervisor
	main		- main function of the program

	COPYRIGHT:	(c) 1990 - 1993 by Stefan Bergdoll, All rights reserved

************************************************************************************************************************/

#include "copying.h"
#include "biosim.h"
#include "functions.h"

/****** global variables ******/

#ifndef BATCH
XtResource resources[] = {
	{ XmNprintCommand, XmCPrintCommand, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, printCommand), XmRString, (caddr_t) "lp " },
	{ XmNbackgroundColor, XmCBackgroundColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, backgroundColor), XmRString, (caddr_t) "gray" },
	{ XmNrubberBandColor, XmCRubberBandColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, rubberBandColor), XmRString, (caddr_t) "yellow" },
	{ XmNneuronColor, XmCNeuronColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, neuronColor), XmRString, (caddr_t) "blue" },
	{ XmNneuronSelectionColor, XmCNeuronSelectionColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, neuronSelectionColor), XmRString, (caddr_t) "orange" },
	{ XmNneuronMovingColor, XmCNeuronMovingColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, neuronMovingColor), XmRString, (caddr_t) "yellow" },
	{ XmNinputNeuronColor, XmCInputNeuronColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, inputNeuronColor), XmRString, (caddr_t) "white" },
	{ XmNoutputNeuronColor, XmCOutputNeuronColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, outputNeuronColor), XmRString, (caddr_t) "green" },
	{ XmNinOutNeuronColor, XmCInOutNeuronColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, inOutNeuronColor), XmRString, (caddr_t) "coral" },
	{ XmNaxisColor, XmCAxisColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, axisColor), XmRString, (caddr_t) "LightGrey" },
	{ XmNconnectionLineColor, XmCConnectionLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, connectionLineColor), XmRString, (caddr_t) "white" },
	{ XmNgraphZeroLineColor, XmCGraphZeroLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, graphZeroLineColor), XmRString, (caddr_t) "gray" },
	{ XmNrestingLineColor, XmCRestingLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, restingLineColor), XmRString, (caddr_t) "black" },
	{ XmNspikeThresholdLineColor, XmCSpikeThresholdLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, spikeThresholdLineColor), XmRString, (caddr_t) "sandybrown" },
	{ XmNsomaLineColor, XmCSomaLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, somaLineColor), XmRString, (caddr_t) "red" },
	{ XmNdendritLineColor, XmCDendritLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, dendritLineColor), XmRString, (caddr_t) "OrangeRed" },
	{ XmNsynapticCurrentLineColor, XmCSynapticCurrentLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, synapticCurrentLineColor), XmRString, (caddr_t) "orange" },
	{ XmNtotalCurrentLineColor, XmCTotalCurrentLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, totalCurrentLineColor), XmRString, (caddr_t) "coral" },
	{ XmNnLineColor, XmCNLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, nLineColor), XmRString, (caddr_t) "khaki" },
	{ XmNsodiumLineColor, XmCSodiumLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, sodiumLineColor), XmRString, (caddr_t) "yellow" },
	{ XmNpotassiumLineColor, XmCPotassiumLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, potassiumLineColor), XmRString, (caddr_t) "blue" },
	{ XmNcalciumLineColor, XmCCalciumLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, calciumLineColor), XmRString, (caddr_t) "turquoise" },
	{ XmNcalcium1LineColor, XmCCalcium1LineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, calcium1LineColor), XmRString, (caddr_t) "white" },
	{ XmNcalcium2LineColor, XmCCalcium2LineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, calcium2LineColor), XmRString, (caddr_t) "wheat" },
	{ XmNkcaLineColor, XmCKcaLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, kcaLineColor), XmRString, (caddr_t) "VioletRed" },
	{ XmNaLineColor, XmCALineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, aLineColor), XmRString, (caddr_t) "plum" },
	{ XmNirLineColor, XmCIrLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, irLineColor), XmRString, (caddr_t) "sienna" },
	{ XmNpLineColor, XmCPLineColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, pLineColor), XmRString, (caddr_t) "NavyBlue" },
	{ XmNtextColor, XmCTextColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, textColor), XmRString, (caddr_t) "black" },
	{ XmNexcitationSynapseColor, XmCExcitationSynapseColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, excitationSynapseColor), XmRString, (caddr_t) "yellow" },
	{ XmNinhibitionSynapseColor, XmCInhibitionSynapseColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, inhibitionSynapseColor), XmRString, (caddr_t) "coral" },
	{ XmNelectricalSynapseColor, XmCElectricalSynapseColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, electricalSynapseColor), XmRString, (caddr_t) "blue" },
	{ XmNaxoSynapseColor, XmCAxoSynapseColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, axoSynapseColor), XmRString, (caddr_t) "black" },
	{ XmNotherSynapseColor, XmCOtherSynapseColor, XmRString, sizeof(XmRString),
	XtOffset(ResTypePtr, otherSynapseColor), XmRString, (caddr_t) "white" }
};

/************************************************************************************************************************

	The following options are possible in the command line:

	printCommand		- a command string for the printer, i.e. "lp "
	backgroundColor		- the color for the background of the drawing area
	rubberBandColor		- the color for the rubber band during selecting objects or zooming in the drawing area
	neuronColor		- the color of the neurons in the drawing area
	neuronSelectionColor	- the color of the neurons in selected mode
	neuronMovingColor	- the color of the neurons during moving
	inputNeuronColor	- the color of the neurons which are shown as input neuron to a selected target neuron
	outputNeuronColor	- the color of the neurons which are shown as output neuron to a selected target neuron
	inOutNeuronColor	- the color of the neurons which are shown as input and output neuron
	connectionLineColor	- the color of the connection line between neurons
	axisColor		- the color of the axis in the graph window for neurons
	graphZeroLineColor	- the color of the zero line in the graph window for neurons
	restingLineColor	- the color of the resting potential line in the graph window for neurons
	spikeThresholdLineColor	- the color of the spike threshold line in the graph window for neurons
	somaLineColor		- the color of the soma trace line in the graph window for neurons
	dendritLineColor	- the color of the dendrite segment trace line in the graph window for neurons
	synapticCurrentLineColor- the color of the synaptic current trace line in the graph window for neurons
	totalCurrentLineColor	- the color of the total current trace line in the graph window for neurons
	nLineColor		- the color of the noie line in the graph window for neurons
	sodiumLineColor		- the color of the sodium trace line in the graph window for neurons
	potassiumLineColor	- the color of the potassium trace line in the graph window for neurons
	calciumLineColor	- the color of the calcium trace line in the graph window for neurons
	calcium1LineColor	- the color of the first calcium trace line in the graph window for neurons
	calcium2LineColor	- the color of the second calcium trace line in the graph window for neurons
	kcaLineColor		- the color of the calcium dependend potassium trace line in the graph window for neurons
	aLineColor		- the color of the A-trace line in the graph window for neurons
	irLineColor		- the color of the Inward-Rectifier-trace line in the graph window for neurons
	pLineColor		- the color of the Proctolin-trace line in the graph window for neurons
	textColor		- the color of the text in the drawing area of the main window and graph window
	excitationSynapseColor	- the color of the excitative synapses
	inhibitionSynapseColor	- the color of the inhibitive synapses
	electricalSynapseColor  - the color of the electrical synapses
	axoSynapseColor		- the color of the axo-axonic synapse
	otherSynapseColor	- the color of the other chemical synapses

************************************************************************************************************************/

XrmOptionDescRec options[] = {
	{ "-printCommand", ".printCommand", XrmoptionSepArg, (caddr_t)NULL },
	{ "-backgroundColor", ".backgroundColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-rubberBandColor", ".rubberBandColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-neuronColor", ".neuronColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-neuronSelectionColor", ".neuronSelectionColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-neuronMovingColor", ".neuronMovingColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-inputNeuronColor", ".inputNeuronColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-outputNeuronColor", ".outputNeuronColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-inOutNeuronColor", ".inOutNeuronColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-connectionLineColor", ".connectionLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-axisColor", ".axisColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-graphZeroLineColor", ".graphZeroLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-restingLineColor", ".restingLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-spikeThresholdLineColor", ".spikeThresholdLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-somaLineColor", ".somaLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-dendritLineColor", ".dendritLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-synapticCurrentLineColor", ".synapticCurrentLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-totalCurrentLineColor", ".totalCurrentLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-nLineColor", ".nLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-sodiumLineColor", ".sodiumLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-potassiumLineColor", ".potassiumLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-calciumLineColor", ".calciumLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-calcium1LineColor", ".calcium1LineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-calcium2LineColor", ".calcium2LineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-kcaLineColor", ".kcaLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-aLineColor", ".aLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-irLineColor", ".irLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-pLineColor", ".pLineColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-textColor", ".textColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-excitationSynapseColor", ".excitationSynapseColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-inhibitionSynapseColor", ".inhibitionSynapseColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-electricalSynapseColor", ".electricalSynapseColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-axoSynapseColor", ".axoSynapseColor", XrmoptionSepArg, (caddr_t)NULL },
	{ "-otherSynapseColor", ".otherSynapseColor", XrmoptionSepArg, (caddr_t)NULL },
};

char *colorList[] = {	"MidnightBlue", "NavyBlue", "CornflowerBlue", "DarkSlateBlue", \
			"MediumBlue", "SlateBlue", "BlueViolet", "blue", "MediumSlateBlue", \
			"SteelBlue", "CadetBlue", "DarkTurquoise", "MediumTurquoise", \
			"turquoise", "SkyBlue", "LightBlue"	};
unsigned long backgroundColor;				/* color for BIOSIM */
unsigned long rubberBandColor;				/* color for BIOSIM */
unsigned long neuronColor;				/* color for BIOSIM */
unsigned long neuronSelectionColor;			/* color for BIOSIM */
unsigned long neuronMovingColor;			/* color for BIOSIM */
unsigned long inputNeuronColor;				/* color for BIOSIM */
unsigned long outputNeuronColor;			/* color for BIOSIM */
unsigned long inOutNeuronColor;				/* color for BIOSIM */
unsigned long connectionLineColor;			/* color for BIOSIM */
unsigned long axisColor;				/* color for BIOSIM */
unsigned long graphZeroLineColor;			/* color for BIOSIM */
unsigned long restingLineColor;				/* color for BIOSIM */
unsigned long spikeThresholdLineColor;			/* color for BIOSIM */
unsigned long somaLineColor;				/* color for BIOSIM */
unsigned long dendritLineColor;				/* color for BIOSIM */
unsigned long synapticCurrentLineColor;			/* color for BIOSIM */
unsigned long totalCurrentLineColor;			/* color for BIOSIM */
unsigned long nLineColor;				/* color for BIOSIM */
unsigned long sodiumLineColor;				/* color for BIOSIM */
unsigned long potassiumLineColor;			/* color for BIOSIM */
unsigned long calciumLineColor;				/* color for BIOSIM */
unsigned long calcium1LineColor;			/* color for BIOSIM */
unsigned long calcium2LineColor;			/* color for BIOSIM */
unsigned long kcaLineColor;				/* color for BIOSIM */
unsigned long aLineColor;				/* color for BIOSIM */
unsigned long irLineColor;				/* color for BIOSIM */
unsigned long pLineColor;				/* color for BIOSIM */
unsigned long textColor;				/* color for BIOSIM */
unsigned long excitationSynapseColor;			/* color for BIOSIM */
unsigned long inhibitionSynapseColor;			/* color for BIOSIM */
unsigned long electricalSynapseColor;			/* color for BIOSIM */
unsigned long axoSynapseColor;				/* color for BIOSIM */
unsigned long otherSynapseColor;			/* color for BIOSIM */
unsigned long selectedToggleButtonColor;		/* color for BIOSIM */
unsigned long activityColors[numActivityColors];	/* color for neuron activity in BIOSIM */
ResType resData;					/* variable for application ressources */
Display *disp;						/* display of the application */
Cursor cursor;						/* cursor in window */
#endif
int commandMode = 0;					/* 0 = interactiv, >0 = command driven */
#ifndef BATCH
XmStringCharSet defaultFont
	= (XmStringCharSet) XmSTRING_DEFAULT_CHARSET;	/* default font, defined in the resource file */
XFontStruct *font;					/* font to be used */
char fontname[64];					/* font name */
char geometry[20];					/* geometry of biosim */
char helpServerPath[128];				/* path name to help server */
char biosimDir[128];					/* directory where biosim resides */
char biosimDataDir[128];				/* data directory of biosim */
char unnamedFile[128];					/* name of default file name */
char commandFile[128];					/* path for command input file */
char selectedObjectsFile[128];				/* path for selection output file */
char outputFile[128];					/* path for output graph file */
char onoffChannelFile[128];				/* onoff channels parameter file */
char swimChannelFile[128];				/* swim channels parameter file */
char hhChannelFile[128];				/* hodgkin-huxley channels parameter file */
char golowaschChannelFile[128];				/* golowasch-buchholz channels parameter file */
#endif
char inputCurrentsFile[128];				/* path for input currents file */
char currentFilename[128];				/* current file name for save operations */
#ifndef BATCH
char programName[128];					/* full name of biosim */
char simulationFileDirectory[128];			/* path name for simulation files */
char parameterFileDirectory[128];			/* path name for parameter files */
char configurationFileDirectory[128];			/* path name for configuration files */
char neuronParameterDirectory[128];			/* path name for neuron parameter files */
char graphOutputDirectory[128];				/* path name for graph output files */
char deleteFileDirectory[128];				/* path name for files to be deleted */
char otherFileDirectory[128];				/* path name for other files */
#endif
char outputFilename[128];				/* name of output file for graphic data */
#ifndef BATCH
Widget applShell;					/* application shell */
Widget mainWindow;					/* main window */
Widget drawingArea;					/* drawing area of the main window */
#endif

/****** functions	******/

#ifndef BATCH
/************************************************************************************************************************

 	FUNCTION	: read_commands()
 
	PURPOSE		: looks every second if a command file is present and execute the commands
			  the command file is generated from the supervisor

	RETURNS		: nothing

************************************************************************************************************************/

void read_commands()
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal 	n;			/* used as argument counter for manipulating widgets */
	int		fileType;		/* the file identification number */
	FILE		*fp;			/* pointer to the command file */
	char		inputFilename[128];	/* the filename which should be read in */
	char		command[128];		/* command string, allowed values are:
						   baue		- build a network from the input file
						   starte	- build a network and start a simulation
						   beenden	- quit the BIOSIM program */
	char		temp[128];		/* string for constructing the header line for the biosim window */
	char		*ptr;			/* used for looking for file name extension */
	Boolean		simulationRun = FALSE;	/* TRUE, if simulation is started */

	/* check if the command file can be opened */

	if ((fp = fopen(commandFile, "r")) == NULL)
	{
		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
		return;
	}

	/* read in file identification number */

	if (fscanf(fp, "%i", &fileType) == 0)
	{
		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
		return;
	}
	/* check file identification number */

	if (fileType != COMMAND_FILE)
	{
		error(100);
		if (fclose(fp) != 0)
			error(101);
		return;
	}

	/* read in input filename */

	if (fscanf(fp, "%s", inputFilename) == 0)
	{
		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
		return;
	}

	/* read in output filename */

	if (fscanf(fp, "%s", outputFilename) == 0)
	{
		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
		return;
	}

	/* if output filename is NULL then send all output of the graph traces to /dev/null */

	if (strcmp(outputFilename, "NULL") == 0)
		strcpy(outputFilename, "/dev/null");

	/* read in command to be executed */

	if (fscanf(fp, "%s", command) == 0)
	{
		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
		return;
	}

	if (fclose(fp) != 0)
	{
		error(101);
		return;
	}

	/* let the input filename be the current filename and generate the biosim header line for the window */

	strcpy(currentFilename, inputFilename);

	sprintf(temp, "Biosim %s : %s", versionString, inputFilename);
	n = 0;
	XtSetArg(args[n], XmNtitle, temp); n++;
	XtSetValues(applShell, args, n);

	/* set ptr to filename extenstion */

	ptr = strrchr(inputFilename, (int) '.');

	/* execute the read in command */

	if (strcmp(command, "baue") == 0)
	{
		commandMode = 1;	/* switch to command driven mode */

		/* check if extension is .sim, .con or .par */

		if ((*(ptr + 1) == 's') && (*(ptr + 2) == 'i') && (*(ptr + 3) == 'm'))
			read_network(inputFilename);
		else if ((*(ptr + 1) == 'c') && (*(ptr + 2) == 'o') && (*(ptr + 3) == 'n'))
			read_configuration(inputFilename);
		else if ((*(ptr + 1) == 'p') && (*(ptr + 2) == 'a') && (*(ptr + 3) == 'r'))
			read_parameter(inputFilename);
		else	/* print out an error message -> wrong filename extension */
		{
			error(102);

			/* start next check for the command file in one second */

			XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
			return;
		}
	}
	else if (strcmp(command, "starte") == 0)
	{
		commandMode = 1;	/* switch to command driven mode */

		/* check if extension is .sim, .con or .par */

		if ((*(ptr + 1) == 's') && (*(ptr + 2) == 'i') && (*(ptr + 3) == 'm'))
			read_network(inputFilename);
		else if ((*(ptr + 1) == 'p') && (*(ptr + 2) == 'a') && (*(ptr + 3) == 'r'))
			read_parameter(inputFilename);
		else	/* print out an error message -> wrong filename extension */
		{
			error(102);

			/* start next check for the command file in one second */

			XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
			return;
		}
		simulationRun = TRUE;
		XSync(disp, FALSE);
		XtAddTimeOut((int) (3 * T_SCALE), run_simulation, (caddr_t) NULL);
	}
	else if (strcmp(command, "zurcksetzen") == 0)
	{
		/* reset network without learning parameters */

		reset_network(FALSE);
	}
	else if (strcmp(command, "alles_zurcksetzen") == 0)
	{
		/* reset network including learning parameters */

		reset_network(TRUE);
	}
	else if (strcmp(command, "beenden") == 0)
	{
		/* remove command file */

		if (remove(commandFile) != 0)
			error(103);
		exit(GOOD);	/* terminate program, exit code = 0 */
	}

	if (! simulationRun)
	{
		/* switch to interactive mode again */

		commandMode = 0;

		/* remove command file */

		if (remove(commandFile) != 0)
			error(103);

		/* start next check for the command file in one second */

		XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);
	}
} /* end of read_commands */
#endif

/************************************************************************************************************************

 	FUNCTION	: main()
 
	PURPOSE		: this is the main function of the BIOSIM program

	RETURNS		: nothing

************************************************************************************************************************/

void main (argc, argv)
unsigned int argc;	/* number of options in the command line */
char *argv[];		/* pointer to command line options */
{
#ifndef BATCH
	Arg args[10];				/* argument list for manipulating widgets */
	Cardinal n;				/* used as argument counter for manipulating widgets */
	XrmDatabase applicationDB;		/* application data base */
	XrmValue value;				/* value returned by resource manager */
	XEvent event;				/* event in main application loop */
	XWindowAttributes attr;			/* attributes of a window */
	Neuron *neuron;				/* neuron pointer used in main application loop */
	Connection *synapse;			/* synapse pointer used in main application loop */
	char *str_type[20];			/* used to read resource file */
	char resourceFile[128];			/* name of resource file */
	char *buffer;				/* temporary used buffer */
#endif
	char header[128];			/* version string for biosim */
#ifndef BATCH
	int x, y;				/* x and y position of a window */
	unsigned int width, height;		/* width and height of a window */
	unsigned int oldwidth, oldheight;	/* old width and height of the a window */
	double temp;				/* temporary variable */
	XSizeHints sizeHints;			/* user prefered sizes of the top level window */
	Widget mainWindowSeparator1;		/* first main window separator */
	Widget mainWindowSeparator2;		/* second main window separator */
	Widget menuBar;				/* menu line */
	Widget hSB;				/* horizontal scroll bar */
	Widget vSB;			 	/* vertical scroll bar */
#endif

#ifndef BATCH
	/* initialize Toolkit and open display */

	applShell = XtInitialize(argv[0], "Biosim", options, XtNumber(options), &argc, argv);
	disp = XtDisplay(applShell);

	/* check if there are not matched command line options */

	if (argc != 1)
	{
		XtWarning("Biosim: bad number of parameters, exiting ...");
		exit(NOTGOOD);	/* abort program with exit code = 1 */
	}

	/* convert registers for Motif Release 1.2 */

	XmRegisterConverters();

	/* String to unit type doesn't get added !! */

	XtAddConverter ( XmRString, XmRUnitType, XmCvtStringToUnitType, NULL, 0 );

	/* read application resources */

	XtGetApplicationResources(applShell, &resData, resources, XtNumber(resources), (ArgList)NULL, 0);

	/* use Editres-Protocol on X11R5 */

#ifdef X11R5
	XtAddEventHandler(applShell, (EventMask) 0, TRUE, _XEditResCheckMessages, NULL);
#endif

	/* get biosim resources from resource file */

	buffer = getenv("HOME");
	strcpy(resourceFile, buffer);
	strcat(resourceFile, "/Biosim");

	if (! (applicationDB = XrmGetFileDatabase(resourceFile)))
	{
		/* now try default application directory */

		strcpy(resourceFile, "/usr/lib/X11/app-defaults/Biosim");
		if (! (applicationDB = XrmGetFileDatabase(resourceFile)))
		{
			fprintf(stderr, "Error: No resource file found ! Exit program ...\n");
			exit(NOTGOOD);
		}
	}

	/* get default font from resource file */

	if (XrmGetResource(applicationDB, "biosim.fontList", "Biosim.FontList", str_type, &value))
	{
		strncpy(fontname, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource fontList not found. Defaulting to fixed\n");
		strcpy(fontname, "fixed");
	}

	font = XLoadQueryFont(disp, fontname);
	if (font == NULL)
	{
		fprintf(stderr, "Font %s not loadable, trying fixed font...\n", fontname);
		font = XLoadQueryFont(disp, "fixed");
		if (font == NULL)
		{
			fprintf(stderr, "Fixed font not loadable, abandoning program...\n");
			exit(NOTGOOD);
		}
	}

	/* get biosim directories from resource file */

	if (XrmGetResource(applicationDB, "biosim.biosimDirectory", "Biosim.BiosimDirectory", str_type, &value))
	{
		strncpy(biosimDir, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimDirectory not found. Defaulting to current directory\n");
		strcpy(biosimDir, ".");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimDataDirectory", "Biosim.BiosimDataDirectory", str_type, &value))
	{
		strncpy(biosimDataDir, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimDataDirectory not found. Defaulting to ./DATA\n");
		strcpy(biosimDataDir, "./DATA");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimUnnamedFile", "Biosim.BiosimUnnamedFile", str_type, &value))
	{
		strncpy(unnamedFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimUnnamedFile not found. Defaulting to ./DATA/test.sim\n");
		strcpy(unnamedFile, "./DATA/test.sim");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimCommandFile", "Biosim.BiosimCommandFile", str_type, &value))
	{
		strncpy(commandFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimCommandFile not found. Defaulting to ./biosim.command\n");
		strcpy(commandFile, "./biosim.command");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimSelectedObjectsFile", "Biosim.BiosimSelectedObjectsFile", str_type, &value))
	{
		strncpy(selectedObjectsFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimSelectedObjectsFile not found. Defaulting to ./biosim.selection\n");
		strcpy(selectedObjectsFile, "./biosim.selection");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimInputCurrentsFile", "Biosim.BiosimInputCurrentsFile", str_type, &value))
	{
		strncpy(inputCurrentsFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimInputCurrentsFile not found. Defaulting to ./biosim.currents\n");
		strcpy(inputCurrentsFile, "./biosim.currents");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimOutputFile", "Biosim.BiosimOutputFile", str_type, &value))
	{
		strncpy(outputFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimOutputFile not found. Defaulting to ./biosim.output\n");
		strcpy(outputFile, "./biosim.output");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimOnoffChannelFile", "Biosim.BiosimOnoffChannelFile", str_type, &value))
	{
		strncpy(onoffChannelFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimOnoffChannelFile not found. Defaulting to ./ONOFF.channel\n");
		strcpy(onoffChannelFile, "./ONOFF.channel");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimSwimChannelFile", "Biosim.BiosimSwimChannelFile", str_type, &value))
	{
		strncpy(swimChannelFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimSwimChannelFile not found. Defaulting to ./SWIM.channel\n");
		strcpy(swimChannelFile, "./SWIM.channel");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimHhChannelFile", "Biosim.BiosimHhChannelFile", str_type, &value))
	{
		strncpy(hhChannelFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimHhChannelFile not found. Defaulting to ./HH.channel\n");
		strcpy(hhChannelFile, "./HH.channel");
	}

	if (XrmGetResource(applicationDB, "biosim.biosimGolowaschChannelFile", "Biosim.BiosimGolowaschChannelFile", str_type, &value))
	{
		strncpy(golowaschChannelFile, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource biosimGolowaschChannelFile not found. Defaulting to ./GB.channel\n");
		strcpy(golowaschChannelFile, "./GB.channel");
	}

	if (XrmGetResource(applicationDB, "biosim.helpServerPath", "Biosim.HelpServerPath", str_type, &value))
	{
		strncpy(helpServerPath, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource helpServerPath not found. Defaulting to ./BIN\n");
		strcpy(helpServerPath, "./BIN");
	}

	/* create needed system filenames */

	strcpy(programName, biosimDir);
	strcat(programName, "/biosim");

	strcpy(simulationFileDirectory, biosimDataDir);
	strcat(simulationFileDirectory, "/*.");
	strcat(simulationFileDirectory, simulationFileExtension);

	strcpy(parameterFileDirectory, biosimDataDir);
	strcat(parameterFileDirectory, "/*.");
	strcat(parameterFileDirectory, parameterFileExtension);

	strcpy(configurationFileDirectory, biosimDataDir);
	strcat(configurationFileDirectory, "/*.");
	strcat(configurationFileDirectory, configurationFileExtension);

	strcpy(neuronParameterDirectory, biosimDataDir);
	strcat(neuronParameterDirectory, "/*.");
	strcat(neuronParameterDirectory, neuronParameterExtension);

	strcpy(graphOutputDirectory, biosimDataDir);
	strcat(graphOutputDirectory, "/*.");
	strcat(graphOutputDirectory, graphOutputExtension);

	strcpy(deleteFileDirectory, biosimDataDir);
	strcat(deleteFileDirectory, "/*.");
	strcat(deleteFileDirectory, deleteFileExtension);

	strcpy(otherFileDirectory, biosimDataDir);
	strcat(otherFileDirectory, "/*.");
	strcat(otherFileDirectory, otherFileExtension);

	/* load colors */

	load_colors();

	/* set default number of dendritic compartments */

	curNumOfDendSegs = 3;	/* we start in the 4-point model */

	/* get default geometry of biosim from resource file */

	if (XrmGetResource(applicationDB, "biosim.geometry", "Biosim.Geometry", str_type, &value))
	{
		strncpy(geometry, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource geometry not found. Defaulting to =552x254+0+0\n");
		strcpy(geometry, "=552x254+0+0");
	}

	XParseGeometry(geometry, &x, &y, &width, &height);

	/* create main window */

	n = 0;
	XtSetArg(args[n], XmNshadowThickness, 0); n++;
	XtSetArg(args[n], XmNscrollingPolicy, XmAUTOMATIC); n++;
	XtSetArg(args[n], XmNx, x); n++;
	XtSetArg(args[n], XmNy, y); n++;
	XtSetArg(args[n], XmNwidth, width); n++;
	XtSetArg(args[n], XmNheight, height); n++;
	mainWindow = XmCreateMainWindow(applShell, "mainWindow", args, n);

	/* get horizontal and vertical scroll bar of main window */

	n = 0;
	XtSetArg(args[n], XmNverticalScrollBar, &vSB); n++;
	XtSetArg(args[n], XmNhorizontalScrollBar, &hSB); n++;
	XtGetValues(mainWindow, args, n);

	/* get main window seperators */

	mainWindowSeparator1 = XmMainWindowSep1(mainWindow);
	mainWindowSeparator2 = XmMainWindowSep2(mainWindow);

	/* manage main window */

	XtManageChild(mainWindow);

	/* create menubar in main window */

	menuBar = create_menubar(mainWindow);
	XtManageChild(menuBar);

	/* create drawing area for the main window */

	n = 0;
	XtSetArg(args[n],XmNmarginWidth, 0); n++;
	XtSetArg(args[n],XmNmarginHeight, 0); n++;
	drawingArea = XmCreateDrawingArea(mainWindow,"drawingarea", args, n);
	XtAddCallback(drawingArea, XmNinputCallback, drawing_input_CB, (caddr_t *)NULL);
	XtAddCallback(drawingArea, XmNexposeCallback, drawing_expose_CB, (caddr_t *)NULL);
	XtAddCallback(drawingArea, XmNresizeCallback, drawing_resize_CB, (caddr_t *)NULL);
	XtManageChild(drawingArea);

	/* create popup menu in drawing area */

	create_popupMenu(drawingArea);

	/* create other menus in main window */

	create_askModelPoint(mainWindow);
	create_askDendrit(mainWindow);
	create_noiseDialog(mainWindow);
	create_stddevDialog(mainWindow);
	create_onOffParams(mainWindow);
	create_swimParams(mainWindow);
	create_hhParams(mainWindow);
	create_gbParams(mainWindow);
	create_editNeuron(mainWindow);
	create_editCurrent(mainWindow);
	create_addSynapse(mainWindow);
	create_axoSynapse(mainWindow);
	create_learnParam(mainWindow);
	create_editSynapse(mainWindow);
	create_editSimPars(mainWindow);
	create_newSType(mainWindow);
	create_simInf(mainWindow);
	create_infoObject(mainWindow);
	create_asciiOutputInterval(mainWindow);
	create_newGraphXScaling(mainWindow);
	create_newGraphYScaling(mainWindow);
	create_newGraphXZoom(mainWindow);
	create_newGraphYZoom(mainWindow);
	create_asciiOutputIntervalS(mainWindow);
	create_newGraphXScalingS(mainWindow);
	create_newGraphYScalingS(mainWindow);
	create_newGraphXZoomS(mainWindow);
	create_newGraphYZoomS(mainWindow);

	/* set areas of main window */

	XmMainWindowSetAreas(mainWindow, menuBar, NULL, hSB, vSB, drawingArea);

	/* realize toplevel widgets */

	XtRealizeWidget(applShell);

	/* output all to display */

	XFlush(disp);

	/* set size hints for the window manager */

	sizeHints.x = (int) x;
	sizeHints.y = (int) y;
	sizeHints.width = (int) width;
	sizeHints.height = (int) height;
	sizeHints.flags = USPosition | USSize;
	XSetNormalHints(disp, XtWindow(applShell), &sizeHints);
#endif

	/* set version string */

	strcpy(header, "Biosim ");
	strcat(header, versionString);
	strcat(header, " : ");

#ifndef BATCH
	n = 0;
	XtSetArg(args[n], XmNtitle, header); n++;
	XtSetValues(applShell, args, n);
#endif

	/* set default values for simulation */

	LTSTOP = SIMLEN_DEF / 1.0E3;
	STEP = SIMSIZE_DEF / 1.0E3;

#ifndef BATCH
	/* set up timer for the function read_commands() */

	XtAddTimeOut((int) (1 * T_SCALE), read_commands, (caddr_t) NULL);

	/* show banner */

	XtManageChild(overBiosimDialog);

	/* main loop */

	while (TRUE)
	{
		XtNextEvent(&event);
		switch (event.type)
		{
			case VisibilityNotify:

				/* check if manual update window flag is set, if so -> do nothing */

				if (manualWindowUpdate)
					break;

				/* loop through all neurons and compare graph window with event window */

				for (neuron = neuronListHead; neuron != NULL; neuron = neuron->next)
				{
					if (neuron->graph)	/* does this neuron have a graph window ? */
						if (event.xvisibility.window ==
						    XtWindow(XtParent(neuron->graph->formWidget)))
							if (event.xvisibility.state == 0)
								draw_graph(neuron);	/* redraw graph */
					
					/* now loop through all outgoing synapses and check graph window */

					for (synapse = neuron->connectionList; synapse != NULL;
					     synapse = synapse->nextConnection)
					{
						if (synapse->graph)	/* is there a graph window ? */
							if (event.xvisibility.window ==
							    XtWindow(XtParent(synapse->graph->formWidget)))
								if (event.xvisibility.state == 0)
									draw_graphS(synapse);	/* redraw graph */
					}
				}
				XtDispatchEvent(&event);
				break;

			case ConfigureNotify:

				/* check if manual update window flag is set, if so -> do nothing */

				if (manualWindowUpdate)
					break;

				/* loop through all neurons and compare graph window with event window */

				for (neuron = neuronListHead; neuron != NULL; neuron = neuron->next)
				{
					if (neuron->graph)	/* does this neuron have a graph window ? */
						if (event.xconfigure.window ==
						    XtWindow(XtParent(neuron->graph->formWidget)))
						{
							/* get position and dimension of graph */

							XGetWindowAttributes(disp, XtWindow(neuron->graph->drawingArea),
								&attr);
							x = attr.x;
							y = attr.y;
							oldwidth = attr.width;
							oldheight = attr.height;

							XGetWindowAttributes(disp, event.xconfigure.window, &attr);
							width = attr.width;
							height = attr.height;

							if (! (oldwidth == width - x && oldheight == height - y) && width >= 100 && height >= 100)
							{
								/* resize graph */

								XResizeWindow(disp, XtWindow(neuron->graph->drawingArea),
									width - x, height - y);

								/* set new graph values */

								neuron->graph->width = (int) (width - x - mvSpace);
								neuron->graph->height = (int) (height - y - msSpace);
								neuron->graph->xScale = ((double) neuron->graph->width /
									(neuron->graph->lastX - neuron->graph->firstX));
								temp = neuron->graph->highestY - neuron->graph->lowestY;
								temp = fabs(temp);
								neuron->graph->yScale = (double) neuron->graph->height /
									temp;

								/* initialize x- and y-axis of graph */

								set_x_axis(neuron->graph);
								set_y_axis(neuron->graph);

								/* draw the graph from scratch */

								draw_graph(neuron);
							}
						}
					
					/* now loop through all outgoing synapses and check graph window */

					for (synapse = neuron->connectionList; synapse != NULL;
					     synapse = synapse->nextConnection)
					{
						if (synapse->graph)	/* is there a graph window ? */
							if (event.xconfigure.window ==
							    XtWindow(XtParent(synapse->graph->formWidget)))
							{
								/* get position and dimension of graph */

								XGetWindowAttributes(disp, XtWindow(synapse->graph->drawingArea), &attr);
								x = attr.x;
								y = attr.y;
								oldwidth = attr.width;
								oldheight = attr.height;

								XGetWindowAttributes(disp, event.xconfigure.window, &attr);
								width = attr.width;
								height = attr.height;

								if (! (oldwidth == width - x && oldheight == height - y) && width >= 100 && height >= 100)
								{
									/* resize graph */

									XResizeWindow(disp,
										XtWindow(synapse->graph->drawingArea),
										width - x, height - y);

									/* set new graph values */

									synapse->graph->width = (int) (width - x -
										mvSpace);
									synapse->graph->height = (int) (height - y -
										msSpace);
									synapse->graph->xScale = (
										(double) synapse->graph->width /
										(synapse->graph->lastX -
										synapse->graph->firstX));
									temp = synapse->graph->highestY -
										synapse->graph->lowestY;
									temp = fabs(temp);
									synapse->graph->yScale = (double)
										synapse->graph->height / temp;

									/* initialize x- and y-axis of graph */

									set_x_axisS(synapse->graph);
									set_y_axisS(synapse->graph);

									/* draw the graph from scratch */

									draw_graphS(synapse);
								}
							}
					}
				}
				XtDispatchEvent(&event);
				break;

			default:
				XtDispatchEvent(&event);
				break;
		}
	}
#else
	if (argc != 3)
	{
		fprintf(stderr, "\nusage: biosim_batch inputfile outputfile\n");
		exit(NOTGOOD);
	}

	printf("%sBatch-Version\n", header);
	commandMode = 1;			/* run in batch mode */
	strcpy(currentFilename, argv[1]);
	strcpy(inputCurrentsFile, "./biosim.currents");
	printf("Reading in filename %s\n", currentFilename);
	if (! read_network(currentFilename))
		fprintf(stderr, "Error in reading network file\n");		/* read in network file */
	strcpy(outputFilename, argv[2]);	/* filename for graph output */
	run_simulation();			/* start simulation */
#endif

} /* end of main */
